"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildKinesisStreamCWAlarms = exports.buildKinesisStream = void 0;
// Imports
const kinesis = require("@aws-cdk/aws-kinesis");
const kinesis_streams_defaults_1 = require("./kinesis-streams-defaults");
const cdk = require("@aws-cdk/core");
const utils_1 = require("./utils");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
function buildKinesisStream(scope, props) {
    if (props.existingStreamObj) {
        return props.existingStreamObj;
    }
    // Setup the stream properties
    let kinesisStreamProps;
    if (props.kinesisStreamProps) {
        // If property overrides have been provided, incorporate them and deploy
        kinesisStreamProps = utils_1.overrideProps(kinesis_streams_defaults_1.DefaultStreamProps, props.kinesisStreamProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        kinesisStreamProps = kinesis_streams_defaults_1.DefaultStreamProps;
    }
    // Create the stream and return
    return new kinesis.Stream(scope, 'KinesisStream', kinesisStreamProps);
}
exports.buildKinesisStream = buildKinesisStream;
function buildKinesisStreamCWAlarms(scope) {
    // Setup CW Alarms for KinesisStream
    const alarms = new Array();
    // Alarm if Max (GetRecords.IteratorAgeMilliseconds): >= 12 hours (50% of 24 hours default retention period)
    alarms.push(new cloudwatch.Alarm(scope, 'KinesisStreamGetRecordsIteratorAgeAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/Kinesis',
            metricName: 'GetRecords.IteratorAgeMilliseconds'
        }),
        threshold: 2592000,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.minutes(5),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Consumer Record Processing Falling Behind, there is risk for data loss due to record expiration.'
    }));
    // Alarm if Avg (ReadProvisionedThroughputExceeded): > 0
    alarms.push(new cloudwatch.Alarm(scope, 'KinesisStreamReadProvisionedThroughputExceededAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/Kinesis',
            metricName: 'ReadProvisionedThroughputExceeded'
        }),
        threshold: 0,
        evaluationPeriods: 1,
        statistic: 'Average',
        period: cdk.Duration.minutes(5),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
        alarmDescription: 'Consumer Application is Reading at a Slower Rate Than Expected.'
    }));
    return alarms;
}
exports.buildKinesisStreamCWAlarms = buildKinesisStreamCWAlarms;
//# sourceMappingURL=data:application/json;base64,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