import json

from anytree import Node

from . import AbstractNode as abs_node
from . import InnerNode as i_node
from . import utils as profiler_utils


class RootNode(abs_node.AbstractNode):
    """
    Loads the json file generated by pyinstrument profiler into the following object. Fields and structure is kept
    similar to original format. However we save children as a map with keys generated for each element for quick
    search/modifications.
    """

    def __init__(self, json_source):
        self.start_time = None
        self.duration = None
        self.sample_count = None
        self.program = None
        self.cpu_time = None
        self.root_frame = None
        self.node_key = None
        self.initialize(json_source)

    class RootFrame(abs_node.AbstractNode):
        def __init__(self, json_dict):
            self.function = None
            self.file_path_short = None
            self.file_path = None
            self.line_no = None
            self.time = None
            self.is_application_code = None
            self.children = {}
            self.initialize(json_dict)

        def initialize(self, root_frame_dict):
            self.function = root_frame_dict["function"]
            self.file_path_short = root_frame_dict["file_path_short"]
            self.file_path = root_frame_dict["file_path"]
            self.line_no = root_frame_dict["line_no"]
            self.time = root_frame_dict["time"]
            self.is_application_code = root_frame_dict["is_application_code"]
            child_json_dict = root_frame_dict["children"]
            if child_json_dict:
                for child_ele in child_json_dict:
                    node = i_node.InnerNode(child_ele)
                    self.children[node.get_key()] = node

    def initialize(self, json_source):
        """
        loads json content into object
        """
        try:
            json_dict = json.loads(json_source)
        except Exception:
            json_dict = json.load(json_source)
        self.start_time = json_dict["start_time"]
        self.duration = json_dict["duration"]
        self.sample_count = json_dict["sample_count"]
        self.program = json_dict["program"]
        self.cpu_time = json_dict["cpu_time"]
        root_frame_dict = json_dict["root_frame"]
        if root_frame_dict:
            self.root_frame = self.RootFrame(root_frame_dict)
            self.node_key = profiler_utils.combine_strings(
                self.program, self.root_frame.function, self.root_frame.file_path, self.root_frame.file_path_short,
                self.root_frame.line_no, self.root_frame.is_application_code)

    def merge(self, other_node):
        """
        Merges two json profiling files loaded in root node class object.
        """
        if other_node.get_key() is not None:
            assert self.get_key() == other_node.get_key()
            self.duration += other_node.duration
            self.sample_count = max(self.sample_count, other_node.sample_count)
            self.cpu_time += other_node.cpu_time
            self.root_frame.time += other_node.root_frame.time

            child_node_keys = self.root_frame.children.keys()

            for other_child_key, other_child_node in other_node.root_frame.children.items():
                """
                if child node key matches, combine them
                """
                if other_child_key in child_node_keys:
                    self.root_frame.children[other_child_key].merge(other_child_node)
                else:
                    """
                    if it does not match, add the other node to the current node.
                    """
                    self.root_frame.children[other_child_key] = other_child_node
                    child_node_keys = self.root_frame.children.keys()

    def get_object_dict(self):
        """
        Generates the current class object in dictionary format to be saved as a json later
        """
        obj_dict = {}
        obj_dict.update(vars(self))

        obj_dict.pop("node_key")

        """
        remove "node_key" field
        """
        root_frame = obj_dict.pop("root_frame")
        obj_dict['root_frame'] = vars(root_frame)

        """ 
        "children" should be saved as a list and not map
        """
        children_dict = obj_dict['root_frame'].pop("children")
        children_list = []
        for key, value in children_dict.items():
            children_list.append(value.get_object_dict())

        obj_dict['root_frame']['children'] = children_list

        return obj_dict

    def get_json_string(self):
        """
        Generates json string identical to the json format generated by pyinstrument profiler
        """
        obj_dict = self.get_object_dict()

        return json.dumps(obj_dict, indent=4)

    def get_tree(self):
        """
        Converts the current class object structure to a tree structure for tree visualization
        """
        total_time = self.root_frame.time
        name = self.root_frame.get_name(total_time)
        root_node = Node(name)

        for inner_child_node in self.root_frame.children.values():
            c_node_name = inner_child_node.get_name(total_time)
            c_node = Node(c_node_name, root_node)

            inner_child_node.get_tree(parent_node=c_node, total_time=total_time)

        return root_node
