from anytree import Node

from . import AbstractNode as abs_node
from . import utils as profiler_utils


class InnerNode(abs_node.AbstractNode):
    """
    Loads the children json content from the json generated by pyinstrument profiler into the following object.
    Fields and structure is kept similar to original format.
    """

    def __init__(self, json_string):
        self.function = None
        self.file_path_short = None
        self.file_path = None
        self.line_no = None
        self.time = None
        self.is_application_code = None
        self.group_id = None
        self.children = {}
        self.initialize(json_string)
        self.node_key = profiler_utils.combine_strings(
            self.function, self.file_path, self.file_path_short, self.line_no, self.is_application_code)

    def initialize(self, json_dict):
        """
        loads json content into object
        """
        self.function = json_dict["function"]
        self.file_path_short = json_dict["file_path_short"]
        self.file_path = json_dict["file_path"]
        self.line_no = json_dict["line_no"]
        self.time = json_dict["time"]
        self.is_application_code = json_dict["is_application_code"]
        try:
            if not self.is_application_code:
                self.group_id = json_dict["group_id"]
        except KeyError:
            self.group_id = ""

        child_json_dict = json_dict["children"]
        if child_json_dict:
            for child_ele in child_json_dict:
                node = InnerNode(child_ele)
                self.children[node.get_key()] = node

    def merge(self, other_node):
        """
        Merges two json profiling files loaded in root node class object.
        """
        assert self.get_key() == other_node.get_key()
        self.time += other_node.time

        child_node_keys = self.children.keys()

        for other_child_key, other_child_node in other_node.children.items():
            """
            if child node key matches, combine them
            """
            if other_child_key in child_node_keys:
                self.children[other_child_key].merge(other_child_node)
            else:
                """
                if it does not match, add the other node to the current node.
                """
                self.children[other_child_key] = other_child_node
                child_node_keys = self.children.keys()

    def get_object_dict(self):
        """
        Generates the current class object in dictionary format to be saved as a json later
        """
        obj_dict = {}
        obj_dict.update(vars(self))

        """
        remove "node_key" field
        """
        obj_dict.pop("node_key")

        """ 
        "children" should be saved as a list and not map
        """
        children_dict = obj_dict.pop("children")
        children_list = []
        for key, value in children_dict.items():
            children_list.append(value.get_object_dict())

        obj_dict['children'] = children_list

        return obj_dict

    def get_tree(self, parent_node, total_time):
        """
        Converts the current class object structure to a tree structure for tree visualization
        """
        for inner_child_node in self.children.values():
            c_node_name = inner_child_node.get_name(total_time)
            c_node = Node(c_node_name, parent_node)

            inner_child_node.get_tree(parent_node=c_node, total_time=total_time)
