"""
Upon receiving the response (which is or includes a TimeStampResp
that normally contains a TimeStampToken (TST), as defined below), the
requesting entity SHALL verify the status error returned in the
response and if no error is present it SHALL verify the various
fields contained in the TimeStampToken and the validity of the
digital signature of the TimeStampToken.  In particular, it SHALL
verify that what was time-stamped corresponds to what was requested
to be time-stamped.  The requester SHALL verify that the
TimeStampToken contains the correct certificate identifier of the
TSA, the correct data imprint and the correct hash algorithm OID.  It
SHALL then verify the timeliness of the response by verifying either
the time included in the response against a local trusted time
reference, if one is available, or the value of the nonce (large
random number with a high probability that it is generated by the
client only once) included in the response against the value included
in the request.  For more details about replay attack detection, see
the security considerations section (item 6).  If any of the
verifications above fails, the TimeStampToken SHALL be rejected.

Then, since the TSA's certificate may have been revoked, the status
of the certificate SHOULD be checked (e.g., by checking the
appropriate CRL) to verify that the certificate is still valid.

Then, the client application SHOULD check the policy field to
determine whether or not the policy under which the token was issued
is acceptable for the application.
"""
import logging
from dataclasses import dataclass
from datetime import datetime, timedelta
from typing import Optional, Union

import certifi
from asn1crypto.cms import CMSAttribute, ContentInfo
from asn1crypto.core import Sequence, SetOf
from asn1crypto.tsp import ESSCertID, ESSCertIDv2, SigningCertificate, SigningCertificateV2
from asn1crypto.x509 import Certificate
from OpenSSL.crypto import FILETYPE_ASN1, X509Store, X509StoreContext, load_certificate
from OpenSSL.crypto import verify as openssl_verify

from .algorithms import DigestAlgorithm
from .exceptions import DigestMismatchError, InvalidInput, InvalidTimeStampToken, NonceMismatchError

logger = logging.getLogger(__name__)


@dataclass(frozen=True)
class VerifyResult:
    tst_info: Sequence
    signed_attrs: Sequence


class CMSAttributes(SetOf):
    _child_spec = CMSAttribute


class TSPVerifier:
    ca_pem_file = None
    ca_path = None
    require_ess_cert_id_v2 = False

    def _verify_imprint(self, message_imprint, *, expect_message, expect_message_digest):
        hash_alg = DigestAlgorithm(message_imprint["hash_algorithm"]["algorithm"])
        hash = message_imprint["hashed_message"]
        if expect_message is not None or expect_message_digest is not None:
            if expect_message is not None:
                expect_message_digest = hash_alg.implementation(expect_message).digest()
            if hash != expect_message_digest:
                raise DigestMismatchError(f"Expected message digest {expect_message_digest}, but got {hash}")

    def _get_signing_cert_id(self, signer_info) -> ESSCertIDv2:
        for signed_attr in signer_info["signed_attrs"]:
            for signed_attr_value in signed_attr["values"]:
                if isinstance(signed_attr_value, SigningCertificateV2):
                    for cert_id in signed_attr_value["certs"]:
                        return cert_id
            for signed_attr_value in signed_attr["values"]:
                if isinstance(signed_attr_value, SigningCertificate):
                    for cert_id in signed_attr_value["certs"]:
                        return cert_id
        raise InvalidTimeStampToken("Unable to extract signing certificate ID")

    def _validate_tsa_cert(self, cert: Certificate):
        if cert.extended_key_usage_value is None:
            raise InvalidTimeStampToken("Invalid signing certificate")
        for key_usage in cert.extended_key_usage_value:  # type: ignore
            if key_usage.native == "time_stamping":
                return
        raise InvalidTimeStampToken("Signing certificate is not a timestamping certificate")

    def _get_ca_store(self):
        store = X509Store()
        ca_pem_file, ca_path = self.ca_pem_file, self.ca_path
        if ca_pem_file is None and ca_path is None:
            ca_pem_file = certifi.where()
        store.load_locations(cafile=ca_pem_file, capath=ca_path)
        return store

    def _cert_match(self, *, signing_cert_id: Union[ESSCertID, ESSCertIDv2], x509_cert) -> bool:
        if isinstance(signing_cert_id, ESSCertIDv2) and x509_cert.sha256 == signing_cert_id["cert_hash"].native:
            return True
        if self.require_ess_cert_id_v2:
            return False
        if isinstance(signing_cert_id, ESSCertID) and x509_cert.sha1 == signing_cert_id["cert_hash"].native:
            return True
        return False

    def _extract_signing_cert(self, signer_info, *, certificates):
        signing_cert_id = self._get_signing_cert_id(signer_info)
        tsa_cert, tsa_chain = None, []
        for cert in certificates:
            logger.debug("%s=>%s", cert.issuer.native["common_name"], cert.subject.native["common_name"])
            if self._cert_match(signing_cert_id=signing_cert_id, x509_cert=cert):
                self._validate_tsa_cert(cert)
                tsa_cert = load_certificate(type=FILETYPE_ASN1, buffer=cert.dump())
            else:
                tsa_chain.append(load_certificate(type=FILETYPE_ASN1, buffer=cert.dump()))
        if tsa_cert is None:
            raise InvalidTimeStampToken("Unable to locate signing certificate")

        store = self._get_ca_store()
        store_context = X509StoreContext(store=store, certificate=tsa_cert, chain=tsa_chain)
        verified_chain = store_context.get_verified_chain()
        return verified_chain[0]

    def _verify_signed_digest(self, *, tst_content, signer_info):
        for signed_attr in signer_info["signed_attrs"]:
            if signed_attr.native["type"] == "message_digest":
                digest = signed_attr.native["values"][0]
                break
        else:
            raise InvalidTimeStampToken("Expected to find message digest in signed attributes")
        digest_algorithm = DigestAlgorithm(signer_info["digest_algorithm"]["algorithm"].native)
        expect_tst_digest = digest_algorithm.implementation(bytes(tst_content)).digest()
        if digest != expect_tst_digest:
            raise DigestMismatchError(f"Expected timestamp content digest {expect_tst_digest}, but got {digest}")

    def _verify_signature(self, *, signing_cert, signer_info):
        signed_attrs = CMSAttributes(signer_info["signed_attrs"])
        string_to_sign = signed_attrs.dump()
        digest_algorithm = signer_info["digest_algorithm"]["algorithm"].native
        openssl_verify(signing_cert, signer_info["signature"].native, string_to_sign, digest_algorithm)
        return signed_attrs

    def verify(
        self,
        time_stamp_token: bytes,
        *,
        message: Optional[bytes] = None,
        message_digest: Optional[bytes] = None,
        nonce: Optional[int] = None,
    ):
        if message is not None and message_digest is not None:
            raise InvalidInput("Expected only one of message and message_digest to be set")
        tst = ContentInfo.load(time_stamp_token)
        if tst["content_type"].native != "signed_data":
            raise InvalidTimeStampToken("Expected content type to be signed_data")
        if tst["content"]["encap_content_info"]["content_type"].native != "tst_info":
            raise InvalidTimeStampToken("Expected encapsulated content type to be tst_info")
        tst_content = tst["content"]["encap_content_info"]["content"]
        if tst_content.native["version"] != "v1":
            raise InvalidTimeStampToken("Expected tst_info content version to be v1")
        if nonce and tst_content.native["nonce"] != nonce:
            raise NonceMismatchError(f"Expected nonce {nonce}, but got {tst_content.native['nonce']}")
        message_imprint = tst_content.native["message_imprint"]
        self._verify_imprint(message_imprint, expect_message=message, expect_message_digest=message_digest)
        if len(tst["content"]["signer_infos"]) != 1:
            raise InvalidTimeStampToken("Expected to find exactly one SignerInfo structure")
        signer_info = tst["content"]["signer_infos"][0]
        certificates = [cert.parse() for cert in tst["content"]["certificates"]]
        signing_cert = self._extract_signing_cert(signer_info, certificates=certificates)
        self._verify_signed_digest(tst_content=tst_content, signer_info=signer_info)
        signed_attrs = self._verify_signature(signing_cert=signing_cert, signer_info=signer_info)
        return VerifyResult(tst_info=tst_content.native, signed_attrs=signed_attrs.native)
