"use strict";
/*
 * Unless explicitly stated otherwise all files in this repository are licensed
 * under the Apache License Version 2.0.
 *
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2021 Datadog, Inc.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.redirectHandlers = void 0;
const loglevel_1 = require("loglevel");
const constants_1 = require("./constants");
/**
 * To avoid modifying code in the user's lambda handler, redirect the handler to a Datadog
 * handler that initializes the Lambda Layers and then calls the original handler.
 * 'DD_LAMBDA_HANDLER' is set to the original handler in the lambda's environment for the
 * replacement handler to find.
 *
 * Unchanged aside from parameter type
 */
function redirectHandlers(lambdas, addLayers) {
    loglevel_1.default.debug(`Wrapping Lambda function handlers with Datadog handler...`);
    lambdas.forEach((lam) => {
        const cfnFunction = lam.node.defaultChild;
        const originalHandler = cfnFunction.handler;
        lam.addEnvironment(constants_1.DD_HANDLER_ENV_VAR, originalHandler);
        const handler = getDDHandler(lam, addLayers);
        if (handler === undefined) {
            loglevel_1.default.debug("Unable to get Datadog handler");
            return;
        }
        cfnFunction.handler = handler;
    });
}
exports.redirectHandlers = redirectHandlers;
function getDDHandler(lam, addLayers) {
    const runtime = lam.runtime.name;
    const lambdaRuntime = constants_1.runtimeLookup[runtime];
    if (lambdaRuntime === undefined || lambdaRuntime === constants_1.RuntimeType.UNSUPPORTED) {
        loglevel_1.default.debug("Unsupported/undefined Lambda runtime");
        return;
    }
    switch (lambdaRuntime) {
        case constants_1.RuntimeType.NODE:
            return addLayers ? constants_1.JS_HANDLER_WITH_LAYERS : constants_1.JS_HANDLER;
        case constants_1.RuntimeType.PYTHON:
            return constants_1.PYTHON_HANDLER;
    }
}
//# sourceMappingURL=data:application/json;base64,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