"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsLoadBalancerController = void 0;
const cdk8s = require("cdk8s");
const constructs_1 = require("constructs");
/**
 * Generate aws-load-balancer-controller config yaml.
 *
 * see https://github.com/kubernetes-sigs/aws-aws-load-balancer-controller/blob/master/docs/install/v2_0_0_full.yaml
 */
class AwsLoadBalancerController extends constructs_1.Construct {
    /**
     *
     */
    constructor(scope, id, options) {
        var _a;
        super(scope, id);
        this.serviceAccountName = 'aws-load-balancer-controller';
        this.deploymentName = 'aws-load-balancer-controller';
        this.clusterName = options.clusterName;
        this.namespace = 'default';
        new cdk8s.ApiObject(this, 'aws-load-balancer-controller-crd', {
            apiVersion: 'apiextensions.k8s.io/v1beta1',
            kind: 'CustomResourceDefinition',
            metadata: {
                annotations: {
                    'controller-gen.kubebuilder.io/version': 'v0.4.0',
                },
                labels: {
                    'app.kubernetes.io/name': this.serviceAccountName,
                },
                name: 'targetgroupbindings.elbv2.k8s.aws',
            },
            spec: {
                additionalPrinterColumns: [
                    {
                        JSONPath: '.spec.serviceRef.name',
                        description: "The Kubernetes Service's name",
                        name: 'SERVICE-NAME',
                        type: 'string',
                    },
                    {
                        JSONPath: '.spec.serviceRef.port',
                        description: "The Kubernetes Service's port",
                        name: 'SERVICE-PORT',
                        type: 'string',
                    },
                    {
                        JSONPath: '.spec.targetType',
                        description: "The AWS TargetGroup's TargetType",
                        name: 'TARGET-TYPE',
                        type: 'string',
                    },
                    {
                        JSONPath: '.spec.targetGroupARN',
                        description: "The AWS TargetGroup's Amazon Resource Name",
                        name: 'ARN',
                        priority: 1,
                        type: 'string',
                    },
                    {
                        JSONPath: '.metadata.creationTimestamp',
                        name: 'AGE',
                        type: 'date',
                    },
                ],
                group: 'elbv2.k8s.aws',
                names: {
                    categories: [
                        'all',
                    ],
                    kind: 'TargetGroupBinding',
                    listKind: 'TargetGroupBindingList',
                    plural: 'targetgroupbindings',
                    singular: 'targetgroupbinding',
                },
                scope: 'Namespaced',
                subresources: {
                    status: {},
                },
                validation: {
                    openAPIV3Schema: {
                        description: 'TargetGroupBinding is the Schema for the TargetGroupBinding API',
                        properties: {
                            apiVersion: {
                                description: 'APIVersion defines the versioned schema of this representation of an object. Servers should convert recognized schemas to the latest internal value, and may reject unrecognized values. More info: https://git.k8s.io/community/contributors/devel/sig-architecture/api-conventions.md#resources',
                                type: 'string',
                            },
                            kind: {
                                description: 'Kind is a string value representing the REST resource this object represents. Servers may infer this from the endpoint the client submits requests to. Cannot be updated. In CamelCase. More info: https://git.k8s.io/community/contributors/devel/sig-architecture/api-conventions.md#types-kinds',
                                type: 'string',
                            },
                            metadata: {
                                type: 'object',
                            },
                            spec: {
                                description: 'TargetGroupBindingSpec defines the desired state of TargetGroupBinding',
                                properties: {
                                    networking: {
                                        description: 'networking provides the networking setup for ELBV2 LoadBalancer to access targets in TargetGroup.',
                                        properties: {
                                            ingress: {
                                                description: 'List of ingress rules to allow ELBV2 LoadBalancer to access targets in TargetGroup.',
                                                items: {
                                                    properties: {
                                                        from: {
                                                            description: 'List of peers which should be able to access the targets in TargetGroup. At least one NetworkingPeer should be specified.',
                                                            items: {
                                                                description: 'NetworkingPeer defines the source/destination peer for networking rules.',
                                                                properties: {
                                                                    ipBlock: {
                                                                        description: 'IPBlock defines an IPBlock peer. If specified, none of the other fields can be set.',
                                                                        properties: {
                                                                            cidr: {
                                                                                description: 'CIDR is the network CIDR. Both IPV4 or IPV6 CIDR are accepted.',
                                                                                type: 'string',
                                                                            },
                                                                        },
                                                                        required: [
                                                                            'cidr',
                                                                        ],
                                                                        type: 'object',
                                                                    },
                                                                    securityGroup: {
                                                                        description: 'SecurityGroup defines a SecurityGroup peer. If specified, none of the other fields can be set.',
                                                                        properties: {
                                                                            groupID: {
                                                                                description: 'GroupID is the EC2 SecurityGroupID.',
                                                                                type: 'string',
                                                                            },
                                                                        },
                                                                        required: [
                                                                            'groupID',
                                                                        ],
                                                                        type: 'object',
                                                                    },
                                                                },
                                                                type: 'object',
                                                            },
                                                            type: 'array',
                                                        },
                                                        ports: {
                                                            description: 'List of ports which should be made accessible on the targets in TargetGroup. If ports is empty or unspecified, it defaults to all ports with TCP.',
                                                            items: {
                                                                properties: {
                                                                    port: {
                                                                        'anyOf': [
                                                                            {
                                                                                type: 'integer',
                                                                            },
                                                                            {
                                                                                type: 'string',
                                                                            },
                                                                        ],
                                                                        'description': 'The port which traffic must match. When NodePort endpoints(instance TargetType) is used, this must be a numerical port. When Port endpoints(ip TargetType) is used, this can be either numerical or named port on pods. if port is unspecified, it defaults to all ports.',
                                                                        'x-kubernetes-int-or-string': true,
                                                                    },
                                                                    protocol: {
                                                                        description: 'The protocol which traffic must match. If protocol is unspecified, it defaults to TCP.',
                                                                        enum: [
                                                                            'TCP',
                                                                            'UDP',
                                                                        ],
                                                                        type: 'string',
                                                                    },
                                                                },
                                                                type: 'object',
                                                            },
                                                            type: 'array',
                                                        },
                                                    },
                                                    required: [
                                                        'from',
                                                        'ports',
                                                    ],
                                                    type: 'object',
                                                },
                                                type: 'array',
                                            },
                                        },
                                        type: 'object',
                                    },
                                    serviceRef: {
                                        description: 'serviceRef is a reference to a Kubernetes Service and ServicePort.',
                                        properties: {
                                            name: {
                                                description: 'Name is the name of the Service.',
                                                type: 'string',
                                            },
                                            port: {
                                                'anyOf': [
                                                    {
                                                        type: 'integer',
                                                    },
                                                    {
                                                        type: 'string',
                                                    },
                                                ],
                                                'description': 'Port is the port of the ServicePort.',
                                                'x-kubernetes-int-or-string': true,
                                            },
                                        },
                                        required: [
                                            'name',
                                            'port',
                                        ],
                                        type: 'object',
                                    },
                                    targetGroupARN: {
                                        description: 'targetGroupARN is the Amazon Resource Name (ARN) for the TargetGroup.',
                                        type: 'string',
                                    },
                                    targetType: {
                                        description: 'targetType is the TargetType of TargetGroup. If unspecified, it will be automatically inferred.',
                                        enum: [
                                            'instance',
                                            'ip',
                                        ],
                                        type: 'string',
                                    },
                                },
                                required: [
                                    'serviceRef',
                                    'targetGroupARN',
                                ],
                                type: 'object',
                            },
                            status: {
                                description: 'TargetGroupBindingStatus defines the observed state of TargetGroupBinding',
                                properties: {
                                    observedGeneration: {
                                        description: 'The generation observed by the TargetGroupBinding controller.',
                                        format: 'int64',
                                        type: 'integer',
                                    },
                                },
                                type: 'object',
                            },
                        },
                        type: 'object',
                    },
                },
                version: 'v1alpha1',
                versions: [
                    {
                        name: 'v1alpha1',
                        served: true,
                        storage: false,
                    },
                    {
                        name: 'v1beta1',
                        served: true,
                        storage: true,
                    },
                ],
            },
        });
        new cdk8s.Helm(this, 'helmawsLoadBalancerController', {
            chart: 'eks/aws-load-balancer-controller',
            releaseName: 'aws-load-balancer-controller',
            values: {
                clusterName: options.clusterName,
                serviceAccount: {
                    create: (_a = options.createServiceAccount) !== null && _a !== void 0 ? _a : true,
                    name: this.serviceAccountName,
                },
            },
        });
    }
}
exports.AwsLoadBalancerController = AwsLoadBalancerController;
//# sourceMappingURL=data:application/json;base64,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