from rich import print

from flexpasm.instructions.segments import (
	Label,
	ReadableExecutableSegment,
	ReadableWriteableSegment,
)
from flexpasm.rich_highlighter import Highlighter
from flexpasm.settings import Settings
from flexpasm.templates import MnemonicTemplate


class _ASMFileHandler:
	@staticmethod
	def save(filename: str, content: str, flag: str = "w"):
		print(f"Save content to file: {filename}")
		with open(filename, flag) as file:
			file.write(content)


class ASMProgram:
	def __init__(self, settings: Settings):
		self.settings = settings
		self._code = []
		self._code += [self._generate_start_description(), ""]

		self._prelude_warnings_count = 0
		self._post_warnings_count = 0

		self.main_res = ReadableExecutableSegment()
		self.main_rws = ReadableWriteableSegment()

		self._print_prelude_warnings()

	def _print_prelude_warnings(self):
		if self.settings.mode == "16":
			self._prelude_warnings_count += 1
			print(
				"[yellow][bold][WARNING][/bold] 16-bit mode (real address mode) is not recommended to be used[/yellow]"
			)

		if not self.settings.start_entry:
			self._prelude_warnings_count += 1
			print(
				"[yellow][bold][WARNING][/bold] Not found start_entry in Settings[/yellow]"
			)

		print(f"[bold]Total prelude warnings: {self._prelude_warnings_count}\n[/bold]")

	def _post_post_warnings(self):
		print()
		entry = [
			entry
			for entry in self._code
			if f"{self.settings.start_entry}:" in entry.strip()
		]

		if not entry:
			print("[yellow][bold][WARNING][/bold] Start Entry Not Found[/yellow]")
			self._post_warnings_count += 1

		print(f"[bold]Total post warnings: {self._post_warnings_count}[/bold]")
		print(
			f"[bold]Total warnings: {self._post_warnings_count + self._prelude_warnings_count}[/bold]"
		)

	def add_label(self, label: Label):
		print(f"[bold]Add label: {label.entry}[/bold]")
		self._code.append(f";; Label Entry {label.entry}")
		self._code.append(f"{label.generate()}\n")

	def add_instruction(self, instruction: str, indentation_level: int = 0):
		print(f"[bold]Add instruction: {instruction}[/bold]")
		Highlighter.highlight(instruction)
		if indentation_level == 0:
			indentation = ""
		else:
			indentation = "\t" * indentation_level

		self._code.append(f"{indentation}{instruction}")

	def add_template(self, template: MnemonicTemplate, merge_with_main: bool = True):
		print(f"[bold]Add template: {template.__class__.__name__}[/bold]")

		res, rws = template.generate(self.settings.mode)

		self.main_res.set_commands_for_label(template.entry, res.split("\n"))
		self.main_rws.add_command(rws)

	def _generate_start_description(self) -> str:
		title = f";; {self.settings.title}"
		author = f";; Author: {self.settings.author}"
		end = ";; Program generated by FLEXPASM (github.com/alexeev-pro/flexpasm)"
		elf_format = "ELF64" if self.settings.mode == "64" else "ELF"
		print(f"Create {elf_format} Program: {self.settings.title}")
		return f"""{";" * 120}
{author}{' ' * (118 - len(author))};;
{title}{' ' * (118 - len(title))};;
{end}{' ' * (118 - len(end))};;
{";" * 120}

format {elf_format} executable 3	   ; {elf_format} EXECUTABLE
entry {self.settings.start_entry}	 ; Set Start Entry"""

	def get_source_code(self) -> str:
		self._post_post_warnings()

		self.main_res.generate()

		self._code += self.main_res.code
		self._code += self.main_rws.code

		return "\n".join(self._code).strip()

	def save_code(self):
		_ASMFileHandler.save(self.settings.filename, self.get_source_code())
