"""Parses DDL and returns a schema with tables, columns, indexes, and constraints"""

import re
from collections import defaultdict
from copy import deepcopy
from typing import Dict, List, NamedTuple, Optional, Tuple

import sqlglot

from opendapi.defs import SqlDialect
from opendapi.logging import logger

# --- Identifier regexes (both flavors) ---------------------------------------
# Capturing: yields unquoted name via 3 alternatives
IDENT = r"(?:`([^`]+)`|\"([^\"]+)\"|([A-Za-z_][A-Za-z0-9_\$]*))"
TABLE = rf"{IDENT}(?:\s*\.\s*{IDENT})?"

# Non-capturing: same matching power, simpler group math when you don't need names
IDENT_NON_CAPTURING = r"(?:`[^`]+`|\"[^\"]+\"|[A-Za-z_][A-Za-z0-9_\$]*)"
TABLE_NON_CAPTURING = (
    rf"(?:{IDENT_NON_CAPTURING})(?:\s*\.\s*(?:{IDENT_NON_CAPTURING}))?"
)


class ParseError(Exception):
    """
    Exception raised when the DDL parser fails to parse a statement.
    """


# Define named tuples for the schema structure
class Column(NamedTuple):
    """
    Column definition.
    """

    name: str
    type: str
    is_nullable: bool
    default_value: Optional[str]
    enum_values: Optional[List[str]]


class Index(NamedTuple):
    """
    Index definition.
    """

    name: str
    columns: List[str]
    is_unique: bool
    expressions: List[str]
    index_type: Optional[str]
    where: Optional[str]


class PrimaryKey(NamedTuple):
    """
    Primary key definition.
    """

    name: str
    columns: List[str]


class ForeignKey(NamedTuple):
    """
    Foreign key definition.
    """

    name: str
    columns: List[str]
    references_table_fq_name: str
    references_table_name: str
    references_table_namespace: Optional[str]
    references_columns: List[str]
    on_delete: Optional[str]
    on_update: Optional[str]


class Check(NamedTuple):
    """
    Check constraint definition.
    """

    name: str
    columns: List[str]
    expression: str


class Table(NamedTuple):
    """
    Table definition.
    """

    name: str
    namespace: Optional[str]
    fq_name: str
    columns: List[Column]
    indexes: List[Index]
    primary_key: Optional[PrimaryKey]
    foreign_keys: List[ForeignKey]
    checks: List[Check]


def _deconflict_idx_name(
    base_name: str,
    suffix: str,
    current_idx_constraint_names: List[str],
    all_autogen_names: List[str],
    dialect: SqlDialect,
) -> str:
    """
    We will try to generate an autogenerated index name that looks similar to what the DB would generate.
    This would only help with the case where the DDL attempts to drop a constraint by the auto generated name.

    MySQL: No Truncation - Adds _counter suffix on conflict.
    Postgres: Truncates the base without _key suffix. Then checks for conflict.
              If so, adds _counter suffix while shortening the base name.
    """
    max_len = 63 if dialect is SqlDialect.POSTGRES else 64
    no_truncation = bool(dialect is SqlDialect.MYSQL)

    candidate_name = (
        f"{base_name[:(max_len - len(suffix))]}{suffix}"
        if not no_truncation
        else f"{base_name}{suffix}"
    )
    counter = 1
    while True:
        # Do not reuse a previous autogen name
        # if a constraint had an explicit name that sounds autogenerated and then dropped, that can be reused
        # Technically, postgres scopes these to schema but autogen names have table names in them - so we are good
        if (
            candidate_name not in current_idx_constraint_names
            and candidate_name not in all_autogen_names
        ):
            return candidate_name
        counter_suffix = f"_{counter}"
        # Technically postgres enforces char limit in bytes,
        # but we're not going to worry about that and assume ASCII for autogen index names
        candidate_name = (
            f"{base_name[:(max_len - len(suffix) - len(counter_suffix))]}{suffix}{counter_suffix}"
            if not no_truncation
            else f"{base_name}{suffix}{counter_suffix}"
        )
        counter += 1


def _build_idx_constraint_name(
    kind: str,
    table_info: Dict,
    column_names: List[str],
    dialect: SqlDialect,
    override_name: Optional[str] = None,
) -> str:
    """
    Builds a index or constraint name for a table.
    """
    if override_name:
        return override_name

    current_idx_constraint_names = [
        *table_info["indexes"].keys(),
        *table_info["primary_key"].keys(),
        *table_info["foreign_keys"].keys(),
        *table_info["checks"].keys(),
    ]
    all_autogen_names = table_info["all_autogen_idx_constraint_names"]

    # Primary key
    if kind == "primary_key":
        autogen_name = (
            # MySQL names the primary key "PRIMARY"
            _deconflict_idx_name(
                "PRIMARY", "", current_idx_constraint_names, all_autogen_names, dialect
            )
            if dialect is SqlDialect.MYSQL
            else _deconflict_idx_name(
                table_info["name"],
                "_pkey",
                current_idx_constraint_names,
                all_autogen_names,
                dialect,
            )
        )

    # Unique index
    elif kind == "unique":
        autogen_name = (
            # MySQL names the unique index on the first column
            _deconflict_idx_name(
                column_names[0],
                "",
                current_idx_constraint_names,
                all_autogen_names,
                dialect,
            )
            if dialect is SqlDialect.MYSQL
            else _deconflict_idx_name(
                f"{table_info['name']}_{'_'.join(column_names)}",
                "_key",
                current_idx_constraint_names,
                all_autogen_names,
                dialect,
            )
        )
    elif kind == "index":
        autogen_name = (
            # MySQL names the index on the first column
            _deconflict_idx_name(
                column_names[0],
                "",
                current_idx_constraint_names,
                all_autogen_names,
                dialect,
            )
            if dialect is SqlDialect.MYSQL
            else _deconflict_idx_name(
                f"{table_info['name']}_{'_'.join(column_names)}",
                "_idx",
                current_idx_constraint_names,
                all_autogen_names,
                dialect,
            )
        )

    # Foreign key
    elif kind == "foreign_key":
        autogen_name = (
            # MySQL foreign keys are named <table>_ibfk*_counter
            # and always have the counter suffix starting from 1
            _deconflict_idx_name(
                table_info["name"],
                "_ibfk",
                [*current_idx_constraint_names, f"{table_info['name']}_ibfk"],
                all_autogen_names,
                dialect,
            )
            if dialect is SqlDialect.MYSQL
            else _deconflict_idx_name(
                f"{table_info['name']}_{'_'.join(column_names)}",
                "_fkey",
                current_idx_constraint_names,
                all_autogen_names,
                dialect,
            )
        )

    # Check constraint
    elif kind == "check":
        autogen_name = (
            # MySQL names the check constraint <table>_chk*_counter
            # and always have the counter suffix starting from 1
            _deconflict_idx_name(
                table_info["name"],
                "_chk",
                [*current_idx_constraint_names, f"{table_info['name']}_chk"],
                all_autogen_names,
                dialect,
            )
            if dialect is SqlDialect.MYSQL
            else _deconflict_idx_name(
                f"{table_info['name']}_{'_'.join(column_names)}",
                "_check",
                current_idx_constraint_names,
                all_autogen_names,
                dialect,
            )
        )
    else:  # pragma: no cover
        raise ValueError(f"Invalid kind: {kind}")

    # Add the autogen name to the list of all autogen names
    table_info["all_autogen_idx_constraint_names"].append(autogen_name)

    return autogen_name


def _remove_comments_from_sql(sql: str) -> str:
    """
    Remove comments from a SQL string.
    """
    sql = re.sub(r"/\*.*?\*/", "", sql, flags=re.DOTALL)
    sql = re.sub(r"--.*$", "", sql, flags=re.MULTILINE)
    return sql


def _split_top_level_commas(s: str) -> List[str]:
    """
    Split by commas not inside (), backticks, or quotes (single/double). Handles doubled quotes/backticks.
    """

    parts, buf = [], []
    depth = 0
    in_sq = in_dq = in_bt = False
    i, n = 0, len(s)

    while i < n:
        ch = s[i]

        # Handle doubled-quote/backtick escapes: ''  ""  ``
        if in_sq and ch == "'" and i + 1 < n and s[i + 1] == "'":
            buf.append("''")
            i += 2
            continue
        if in_dq and ch == '"' and i + 1 < n and s[i + 1] == '"':
            buf.append('""')
            i += 2
            continue
        if in_bt and ch == "`" and i + 1 < n and s[i + 1] == "`":
            buf.append("``")
            i += 2
            continue

        if not (in_sq or in_dq or in_bt):
            if ch == "(":
                depth += 1
            elif ch == ")":
                depth = max(0, depth - 1)
            elif ch == "," and depth == 0:
                part = "".join(buf).strip()
                if part:
                    parts.append(part)
                buf = []
                i += 1
                continue

        # Toggle quote states
        if ch == "'" and not (in_dq or in_bt):
            in_sq = not in_sq
        elif ch == '"' and not (in_sq or in_bt):
            in_dq = not in_dq
        elif ch == "`" and not (in_sq or in_dq):
            in_bt = not in_bt

        buf.append(ch)
        i += 1

    tail = "".join(buf).strip()
    if tail:
        parts.append(tail)
    return parts


def normalize_multi_action_ddl(  # pylint: disable=too-many-locals
    stmt: str,
) -> List[str]:
    """
    Normalize multi-action DDL statements into single-action statements.
    Supports: ALTER TABLE, DROP, TRUNCATE, GRANT/REVOKE (multi-privilege), COMMENT ON.
    Preserves whatever quoting/backticks you provided on identifiers.
    """
    # --- Precompiled matchers ----------------------------------------------------
    stmt = _remove_comments_from_sql(stmt).strip().rstrip(";")
    alter_re = re.compile(
        rf"^ALTER\s+TABLE\s+(?P<table>{TABLE_NON_CAPTURING})\s+(?P<actions>.+)$",
        re.IGNORECASE,
    )
    drop_re = re.compile(
        r"^DROP\s+(?P<objtype>TABLE|VIEW|INDEX|SEQUENCE)\s+(?P<rest>.+)$",
        re.IGNORECASE,
    )
    truncate_re = re.compile(r"^TRUNCATE\s+(?:TABLE\s+)?(?P<tables>.+)$", re.IGNORECASE)
    grant_re = re.compile(
        r"^(?P<verb>GRANT|REVOKE)\s+"
        r"(?P<privs>.+?)\s+"
        r"ON\s+(?P<object>.+?)\s+"
        r"(?P<dir>TO|FROM)\s+(?P<users>.+?)"
        r"(?:\s+WITH\s+GRANT\s+OPTION)?"
        r"\s*;?\s*$",
        re.IGNORECASE,
    )
    comment_re = re.compile(r"^COMMENT\s+ON\s+(?P<rest>.+)$", re.IGNORECASE)

    # 1) ALTER TABLE ... <action, action, ...>
    m = alter_re.match(stmt)
    if m:
        table_name = m.group("table")
        actions = _split_top_level_commas(m.group("actions"))
        current_table_name = table_name
        resp = []
        for action in actions:
            if action.strip().startswith("RENAME TO"):
                # Special case handling for RENAME TO
                new_table_name = action.strip().split("RENAME TO")[1].strip()
                resp.append(
                    f"ALTER TABLE {current_table_name} RENAME TO {new_table_name};"
                )
                current_table_name = new_table_name
            elif action.strip().startswith("SET SCHEMA"):
                # Special case handling for SET SCHEMA
                new_schema_name = action.strip().split("SET SCHEMA")[1].strip()
                resp.append(
                    f"ALTER TABLE {current_table_name} SET SCHEMA {new_schema_name};"
                )
                current_table_name = f"{new_schema_name}.{current_table_name}"
            elif action.strip():
                resp.append(f"ALTER TABLE {current_table_name} {action.strip()};")

        return resp

    # 2) DROP <OBJECT> ...
    m = drop_re.match(stmt)
    if m:
        obj_type = m.group("objtype").upper()
        rest = m.group("rest").strip()

        # MySQL: DROP INDEX idx1, idx2 ON <table>
        if obj_type == "INDEX" and re.search(r"\s+ON\s+", rest, flags=re.IGNORECASE):
            idx_part, tbl_part = re.split(
                r"\s+ON\s+", rest, maxsplit=1, flags=re.IGNORECASE
            )
            indexes = _split_top_level_commas(idx_part)
            return [
                f"DROP INDEX {idx.strip()} ON {tbl_part.strip()};"
                for idx in indexes
                if idx.strip()
            ]

        # General multi-object drop: DROP TABLE t1, t2
        objs = _split_top_level_commas(rest)
        return [f"DROP {obj_type} {o.strip()};" for o in objs if o.strip()]

    # 3) TRUNCATE [TABLE] t1, t2
    m = truncate_re.match(stmt)
    if m:
        tables = _split_top_level_commas(m.group("tables"))
        return [f"TRUNCATE TABLE {t.strip()};" for t in tables if t.strip()]

    # 4) GRANT/REVOKE multi-privileges: GRANT a, b ON obj TO user
    m = grant_re.match(stmt)
    if m:
        verb = m.group("verb").upper()
        privs = _split_top_level_commas(m.group("privs"))
        obj = m.group("object").strip()
        direction = m.group("dir").upper()
        users = m.group("users").strip()
        with_grant = (
            "WITH GRANT OPTION"
            if re.search(r"\bWITH\s+GRANT\s+OPTION\b", m.group(0), re.IGNORECASE)
            else ""
        )
        return [
            f"{verb} {p.strip()} ON {obj} {direction} {users}{' ' + with_grant if with_grant else ''};"
            for p in privs
            if p.strip()
        ]

    # 5) COMMENT ON ... (split by top-level commas)
    m = comment_re.match(stmt)
    if m and "," in stmt:
        parts = _split_top_level_commas(m.group("rest"))
        return [f"COMMENT ON {p.strip()};" for p in parts if p.strip()]

    return [stmt + ";"]


def _remove_vendor_specific_additions_from_command(
    stmt: str,
) -> str:
    """
    Remove vendor-specific additions from command.
    """
    pattern_algorithm = re.compile(r"\s*ALGORITHM=\w+", re.IGNORECASE)
    pattern_lock = re.compile(r"\s*(,|)\s*LOCK=\w+", re.IGNORECASE)

    stmt = pattern_algorithm.sub("", stmt)
    stmt = pattern_lock.sub("", stmt)
    return stmt


def make_commands_parsable_if_possible(
    stmt: str,
) -> List[str]:
    """
    Normalize commands if possible.
    """
    new_stmts = normalize_multi_action_ddl(stmt)
    return [_remove_vendor_specific_additions_from_command(stmt) for stmt in new_stmts]


def is_generic_command_ast(expr: sqlglot.exp.DDL) -> bool:
    """
    Check if sqlglot produced a generic Command node instead of a real AST.
    This is possible when sqlglot doesn't support something, or for multi-action DDL statements.
    """
    return isinstance(expr, sqlglot.exp.Command) or expr.find(sqlglot.exp.Command)


def is_temporary_entity(stmt: sqlglot.exp.DDL) -> bool:
    """
    Check if this is a temporary entity like temporary table or view.
    """
    return stmt.find(sqlglot.exp.TemporaryProperty) or stmt.args.get("temporary", False)


def _build_fq_name(table_name: str, table_namespace: Optional[str]) -> str:
    """
    Build a fully qualified name for a table.
    """
    return f"{table_namespace}.{table_name}" if table_namespace else table_name


def _get_table_names(table_node: sqlglot.exp.Table) -> Tuple[str, Optional[str], str]:
    """
    Get the table name, namespace, and fully qualified name.

    In MySQL, database and schema are the same thing -- so most DDLs within a DB connection may not have any namespace.
    In Postgres, database and schema are different and DDLs may have schema qualification.
    """
    table_name = table_node.name.lower()
    table_namespace = (
        table_node.db.lower() if hasattr(table_node, "db") and table_node.db else None
    )
    return table_name, table_namespace, _build_fq_name(table_name, table_namespace)


def _extract_column_type_from_column_dtype(
    col_type_node: sqlglot.exp.DataType,
) -> Tuple[str, Optional[List[str]]]:
    """
    Extract column type from a column definition node.
    Returns column type and enum values if any.
    """
    col_type = col_type_node.this.name.lower()
    col_type_param_nodes = list(col_type_node.find_all(sqlglot.exp.DataTypeParam))

    # Unsupported types parse into Type.USERDEFINED with actual value going to kind
    if col_type == "userdefined":
        col_type = (str(col_type_node.args.get("kind")) or col_type).lower()

    # VARCHAR(12)
    if col_type_param_nodes:
        col_type = (
            f"{col_type}({', '.join([p.name.lower() for p in col_type_param_nodes])})"
        )

    # ARRAY[VARCHAR]
    if col_type_node.args.get("nested"):
        col_type = f"{col_type}[{','.join([c.name.lower() for c in col_type_node.args.get('values') or []])}]"

    # ENUM('a', 'b', 'c')
    if col_type == "enum":
        # Do not lower the enum values as they are case-sensitive
        enum_values = [p.name for p in col_type_node.expressions]
    else:
        enum_values = None

    return col_type, enum_values


def _process_fk_reference(reference: sqlglot.exp.Reference) -> Dict:
    """
    Process a foreign key reference node.
    Returns a dict with the following keys:
    - references_table_fq_name: the fully qualified name of the referenced table
    - references_table_name: the name of the referenced table
    - references_table_namespace: the namespace of the referenced table
    - references_columns: the columns of the referenced table
    - on_delete: the on delete action
    - on_update: the on update action
    """
    table_node = reference.find(sqlglot.exp.Table)
    table_name, table_namespace, table_fq_name = _get_table_names(table_node)
    result = {
        "references_table_fq_name": table_fq_name,
        "references_table_name": table_name,
        "references_table_namespace": table_namespace,
        "references_columns": [
            col.name.lower() for col in reference.this.args.get("expressions", [])
        ],
        "on_delete": None,
        "on_update": None,
    }
    for option in reference.args.get("options", []):
        if "on delete" in option.lower():
            result["on_delete"] = option.lower().split("on delete ")[-1].strip().lower()
        if "on update" in option.lower():
            result["on_update"] = option.lower().split("on update ")[-1].strip().lower()

    return result


def _process_table_drop(tables_by_name: Dict, dropped_table_fq_name: str) -> None:
    """
    Process a table drop and reconcile as consistent with how the DB engines handle table drop

    tables_by_name contains the current state of the tables.

    Table drops would need to be handled for foreign keys.
    - FKs with dropped tables as references would need to be dropped

    """
    tables_by_name.pop(dropped_table_fq_name, None)

    # Remove all foreign keys that reference the dropped table
    fks_and_tables_to_drop = []

    for table_fq_name, table_meta in tables_by_name.items():
        for fk_name, fk_info in table_meta["foreign_keys"].items():
            if fk_info["references_table_fq_name"] == dropped_table_fq_name:
                fks_and_tables_to_drop.append((table_fq_name, fk_name))

    for table_fq_name, fk_name in fks_and_tables_to_drop:
        del tables_by_name[table_fq_name]["foreign_keys"][fk_name]


def _process_table_rename(
    tables_by_name: Dict,
    table_fq_name: str,
    new_table_fq_name: str,
    new_table_name: str,
    new_table_namespace: Optional[str],
) -> None:
    """
    Process a table rename and reconcile as consistent with how the DB engines handle renames.

    Table renames would need to be handled for foreign keys.
    - FKs with renamed tables would need to be updated to the new table fq name
    - FKs with updated schema would need to be updated to the new table fq name

    """
    old_table_meta = tables_by_name.pop(table_fq_name)
    tables_by_name[new_table_fq_name] = {
        **old_table_meta,
        "name": new_table_name,
        "namespace": new_table_namespace,
        "fq_name": new_table_fq_name,
    }

    # Update all foreign keys that reference the renamed table
    fk_to_update = []
    for tbl_fq_name, tbl_meta in tables_by_name.items():
        for fk_name, fk_info in tbl_meta["foreign_keys"].items():
            if fk_info["references_table_fq_name"] == table_fq_name:
                fk_to_update.append((tbl_fq_name, fk_name))

    for tbl_fq_name, fk_name in fk_to_update:
        tables_by_name[tbl_fq_name]["foreign_keys"][fk_name][
            "references_table_fq_name"
        ] = new_table_fq_name
        tables_by_name[tbl_fq_name]["foreign_keys"][fk_name][
            "references_table_name"
        ] = new_table_name
        tables_by_name[tbl_fq_name]["foreign_keys"][fk_name][
            "references_table_namespace"
        ] = new_table_namespace


def _process_column_drop(
    tables_by_name: Dict, table_fq_name: str, col_name: str
) -> None:
    """
    Process a column drop and reconcile as consistent with how the DB engines handle renames/drops.

    Reconcile column renames & drops for indexes, primary key, foreign keys, and checks.
    - All entities consisting dropped columns would need to be dropped
    - All entities with renamed columns would need to be updated to the new column name
    - All FKs with dropped column as reference would need to be dropped
    """
    tables_by_name[table_fq_name]["columns"].pop(col_name, None)

    # Remove all indexes, primary key, foreign keys, and checks that reference the dropped column
    entities_to_drop = []
    for entity_key in ["indexes", "primary_key", "foreign_keys", "checks"]:
        for entity_name, entity_info in tables_by_name[table_fq_name][
            entity_key
        ].items():
            if col_name in entity_info["columns"]:
                entities_to_drop.append((entity_key, entity_name))

    for entity_key, entity_name in entities_to_drop:
        del tables_by_name[table_fq_name][entity_key][entity_name]

    # Remove all foreign keys that reference the dropped column
    fk_to_drop = []
    for tbl_fq_name, tbl_meta in tables_by_name.items():
        for fk_name, fk_info in tbl_meta["foreign_keys"].items():
            if (
                col_name in fk_info["references_columns"]
                and table_fq_name == fk_info["references_table_fq_name"]
            ):
                fk_to_drop.append((tbl_fq_name, fk_name))

    for tbl_fq_name, fk_name in fk_to_drop:
        del tables_by_name[tbl_fq_name]["foreign_keys"][fk_name]


def _process_column_rename(
    tables_by_name: Dict, table_fq_name: str, col_name: str, new_col_name: str
) -> None:
    """
    Process a column rename and reconcile as consistent with how the DB engines handle renames.

    Reconcile column renames & drops for indexes, primary key, foreign keys, and checks.
    - All entities consisting dropped columns would need to be dropped
    - All entities with renamed columns would need to be updated to the new column name
    - All FKs with renamed column as reference would need to be updated to the new column name
    """

    tables_by_name[table_fq_name]["columns"][new_col_name] = tables_by_name[
        table_fq_name
    ]["columns"].pop(col_name)

    # Replace the old column name with the new column name in place to keep ordering
    entities_to_update = []
    for entity_key in ["indexes", "primary_key", "foreign_keys", "checks"]:
        for entity_name, entity_info in tables_by_name[table_fq_name][
            entity_key
        ].items():
            if col_name in entity_info["columns"]:
                col_idx = entity_info["columns"].index(col_name)
                entities_to_update.append((entity_key, entity_name, col_idx))

    for entity_key, entity_name, col_idx in entities_to_update:
        tables_by_name[table_fq_name][entity_key][entity_name]["columns"][
            col_idx
        ] = new_col_name

    # Update all foreign keys that reference the renamed column in place to keep ordering
    fk_to_update = []
    for tbl_fq_name, tbl_meta in tables_by_name.items():
        for fk_name, fk_info in tbl_meta["foreign_keys"].items():
            if (
                col_name in fk_info["references_columns"]
                and table_fq_name == fk_info["references_table_fq_name"]
            ):
                col_idx = fk_info["references_columns"].index(col_name)
                fk_to_update.append((tbl_fq_name, fk_name, col_idx))

    for tbl_fq_name, fk_name, col_idx in fk_to_update:
        tables_by_name[tbl_fq_name]["foreign_keys"][fk_name]["references_columns"][
            col_idx
        ] = new_col_name


def _handle_table_level_constraint(
    expression: sqlglot.exp.ColumnConstraint,
    tables: Dict,
    table_fq_name: str,
    dialect: SqlDialect,
    force_constraint_name: Optional[str] = None,
) -> bool:
    """
    Handle a column constraint.
    """
    if isinstance(expression, sqlglot.exp.IndexColumnConstraint):
        # constraints are NOT in a Schema wrapper
        cols = [
            col.name.lower()
            for col_expr in expression.expressions
            for col in col_expr.find_all(sqlglot.exp.Column)
        ]
        index_name = _build_idx_constraint_name(
            "index",
            tables[table_fq_name],
            cols,
            dialect,
            force_constraint_name or expression.name,
        ).lower()
        tables[table_fq_name]["indexes"][index_name] = {
            "name": index_name,
            "columns": cols,
        }
        return True
    if isinstance(expression, sqlglot.exp.UniqueColumnConstraint):
        # constraints are in a Schema wrapper
        schema_node = expression.find(sqlglot.exp.Schema)
        cols = [
            col.name.lower()
            for col_expr in schema_node.expressions
            for col in col_expr.find_all(sqlglot.exp.Identifier)
        ]
        unique_index_name = _build_idx_constraint_name(
            "unique",
            tables[table_fq_name],
            cols,
            dialect,
            force_constraint_name or expression.name or schema_node.name,
        ).lower()
        tables[table_fq_name]["indexes"][unique_index_name] = {
            "name": unique_index_name,
            "columns": cols,
            "is_unique": True,
        }
        return True
    if isinstance(expression, sqlglot.exp.PrimaryKey):
        # constraints are NOT in a Schema wrapper
        cols = [
            col.name.lower()
            for col_expr in expression.expressions
            for col in col_expr.find_all(sqlglot.exp.Identifier)
        ]
        primary_key_name = _build_idx_constraint_name(
            "primary_key",
            tables[table_fq_name],
            cols,
            dialect,
            force_constraint_name or expression.name,
        ).lower()
        tables[table_fq_name]["primary_key"][primary_key_name] = {
            "name": primary_key_name,
            "columns": cols,
        }
        return True
    if isinstance(expression, sqlglot.exp.ForeignKey):
        # constraints are NOT in a Schema wrapper
        cols = [
            col.name.lower()
            for col_expr in expression.expressions
            for col in col_expr.find_all(sqlglot.exp.Identifier)
        ]
        foreign_key_name = _build_idx_constraint_name(
            "foreign_key",
            tables[table_fq_name],
            cols,
            dialect,
            force_constraint_name or expression.name,
        ).lower()
        tables[table_fq_name]["foreign_keys"][foreign_key_name] = {
            "name": foreign_key_name,
            "columns": cols,
            **_process_fk_reference(expression.find(sqlglot.exp.Reference)),
        }

        return True
    if isinstance(expression, sqlglot.exp.CheckColumnConstraint):
        cols = [col.name.lower() for col in expression.find_all(sqlglot.exp.Column)]
        check_name = _build_idx_constraint_name(
            "check",
            tables[table_fq_name],
            cols,
            dialect,
            force_constraint_name or expression.name,
        ).lower()
        tables[table_fq_name]["checks"][check_name] = {
            "name": check_name,
            "columns": cols,
            "expression": str(expression.this),
        }
        return True
    # Default AND NOT NULL cannot be a table level constraint
    return False


def _process_column_def(
    col_node: sqlglot.exp.ColumnDef,
    tables: Dict,
    table_fq_name: str,
    dialect: SqlDialect,
) -> None:
    """
    Process a column definition node.
    """
    col_name = col_node.name.lower()
    col_type, enum_values = _extract_column_type_from_column_dtype(
        col_node.find(sqlglot.exp.DataType)
    )
    default_value = None
    is_nullable = True

    # Process column level constraints that are declared inline with the column definition.
    for constraint in col_node.find_all(sqlglot.exp.ColumnConstraint):
        if isinstance(constraint.kind, sqlglot.exp.DefaultColumnConstraint):
            # Default value constraints don't support naming
            default_value = str(constraint.kind.this)
        if isinstance(constraint.kind, sqlglot.exp.NotNullColumnConstraint):
            # NOT NULL constraints don't support naming
            is_nullable = constraint.kind.args.get("allow_null", False)
        if isinstance(constraint.kind, sqlglot.exp.PrimaryKeyColumnConstraint):
            primary_key_name = _build_idx_constraint_name(
                "primary_key",
                tables[table_fq_name],
                [col_name],
                dialect,
                constraint.name,
            ).lower()
            tables[table_fq_name]["primary_key"][primary_key_name] = {
                "name": primary_key_name,
                "columns": [col_name],
            }
        if isinstance(constraint.kind, sqlglot.exp.UniqueColumnConstraint):
            unique_index_name = _build_idx_constraint_name(
                "unique", tables[table_fq_name], [col_name], dialect, constraint.name
            ).lower()
            tables[table_fq_name]["indexes"][unique_index_name] = {
                "name": unique_index_name,
                "columns": [col_name],
                "is_unique": True,
            }
        if isinstance(constraint.kind, sqlglot.exp.Reference):
            foreign_key_name = _build_idx_constraint_name(
                "foreign_key",
                tables[table_fq_name],
                [col_name],
                dialect,
                constraint.name,
            ).lower()
            tables[table_fq_name]["foreign_keys"][foreign_key_name] = {
                "name": foreign_key_name,
                "columns": [col_name],
                **_process_fk_reference(constraint.kind),
            }
        if isinstance(
            constraint.kind, sqlglot.exp.IndexColumnConstraint
        ):  # pragma: no cover
            # Adding INDEX to a column inline is not ansi compliant -- so alembic or liquibase will not generate this
            # But we will still capture it just to be futureproof
            index_name = _build_idx_constraint_name(
                "index", tables[table_fq_name], [col_name], dialect, constraint.name
            ).lower()
            tables[table_fq_name]["indexes"][index_name] = {
                "name": index_name,
                "columns": [col_name],
            }

    tables[table_fq_name]["columns"][col_name] = {
        "name": col_name,
        "type": col_type,
        "is_nullable": is_nullable,
        "default_value": default_value,
        "enum_values": enum_values,
    }


def _handle_create_table_like(
    stmt: sqlglot.exp.DDL,
    tables: Dict,
    table_fq_name: str,
    table_name: str,
    table_namespace: Optional[str],
    dialect: SqlDialect,
) -> bool:
    """
    Handle CREATE TABLE LIKE statement.
    Returns True if the statement was handled, False otherwise.
    """
    # Handle CREATE TABLE new LIKE old;
    from_like_node = stmt.find(sqlglot.exp.LikeProperty)
    if from_like_node:
        old_table_node = from_like_node.find(sqlglot.exp.Table)
        _, _, old_table_fq_name = _get_table_names(old_table_node)
        #  okay to error if old table does not exist
        tables[table_fq_name] = deepcopy(tables[old_table_fq_name])
        tables[table_fq_name]["namespace"] = table_namespace
        tables[table_fq_name]["name"] = table_name
        tables[table_fq_name]["fq_name"] = table_fq_name

        # MySQL copies over all the defaults, indexes, and constraints
        # Postgres does not copy over any of these unless explicitly specified
        if dialect is SqlDialect.POSTGRES:
            all_including_values = [
                str(prop.args.get("value")).lower()
                for prop in from_like_node.find_all(sqlglot.exp.Property)
                if prop.args.get("value")
            ]

            if "all" not in all_including_values:
                if "defaults" not in all_including_values:
                    for col_name in tables[table_fq_name]["columns"]:
                        tables[table_fq_name]["columns"][col_name][
                            "default_value"
                        ] = None
                if "indexes" not in all_including_values:
                    tables[table_fq_name]["indexes"] = {}
                    tables[table_fq_name]["primary_key"] = {}
                if "constraints" not in all_including_values:
                    for col_name in tables[table_fq_name]["columns"]:
                        tables[table_fq_name]["columns"][col_name]["is_nullable"] = True
                    tables[table_fq_name]["checks"] = {}
                # Other INCLUDING options we consciously ignore (no structural effect here):
                # 'comments', 'storage parameters', 'identity', 'generated', etc.
                # They're either non-structural for our schema model or require engine metadata.

        # foreign keys are not copied for both mysql and postgres
        tables[table_fq_name]["foreign_keys"] = {}
        return True
    return False


def _handle_create_table_as_select(
    stmt: sqlglot.exp.DDL,
    tables: Dict,
    table_fq_name: str,
    table_name: str,
    table_namespace: Optional[str],
) -> bool:
    """
    Handle CREATE TABLE AS SELECT statement.
    Returns True if the statement was handled, False otherwise.
    """
    # Handle CREATE TABLE new AS SELECT ... FROM old;
    as_select_nodes = list(stmt.find_all(sqlglot.exp.Select))
    # We want to handle only one SELECT statement in the CREATE TABLE statement
    if len(as_select_nodes) == 1 and (as_select_node := as_select_nodes[0]):
        from_table_node = as_select_node.find(sqlglot.exp.Table)
        _, _, from_table_fq_name = _get_table_names(from_table_node)
        is_star_select = any(
            isinstance(expr, sqlglot.exp.Star) for expr in as_select_node.expressions
        )
        if is_star_select:
            tables[table_fq_name]["columns"] = {
                col_name: {
                    "name": col_name,
                    "type": col_info["type"],
                    # nullability and default value are not copied from the source table
                    "is_nullable": True,
                    "default_value": None,
                }
                for col_name, col_info in tables[from_table_fq_name]["columns"].items()
            }
            # indices, constraints, etc. are not copied from the source table
            tables[table_fq_name]["namespace"] = table_namespace
            tables[table_fq_name]["name"] = table_name
            tables[table_fq_name]["fq_name"] = table_fq_name

        else:
            # non-star CTAS SELECT queries can be arbitrarily complex
            # and needs an engine to infer the schema
            logger.debug("NOT STAR SELECT CTAS is not supported yet: %s", stmt)
        return True
    return False


def handle_create_table(
    stmt: sqlglot.exp.DDL, tables: Dict, dialect: SqlDialect
) -> bool:
    """
    Handle CREATE TABLE statement.
    Returns True if the statement was handled, False otherwise.
    """
    if is_temporary_entity(stmt):
        # Skip temporary entities (e.g. CREATE TEMPORARY TABLE)
        return False

    # Create statement's first table is always the target table
    table_node = stmt.find(sqlglot.exp.Table)
    table_name, table_namespace, table_fq_name = _get_table_names(table_node)

    # Do not process for CREATE IF NOT EXISTS if table already exists
    if_not_exists = stmt.args.get("exists", False)
    if if_not_exists and table_fq_name in tables:
        return False

    # Handle CREATE TABLE new LIKE old;
    if _handle_create_table_like(
        stmt, tables, table_fq_name, table_name, table_namespace, dialect
    ):
        return True

    # Handle CREATE TABLE new AS SELECT ... FROM old;
    if _handle_create_table_as_select(
        stmt, tables, table_fq_name, table_name, table_namespace
    ):
        return True

    # Handle CREATE TABLE new (col1, col2, col3);
    tables[table_fq_name]["namespace"] = table_namespace
    tables[table_fq_name]["name"] = table_name
    tables[table_fq_name]["fq_name"] = table_fq_name

    for expression in stmt.this.expressions:
        if isinstance(expression, sqlglot.exp.ColumnDef):
            _process_column_def(expression, tables, table_fq_name, dialect)
        elif isinstance(expression, sqlglot.exp.Constraint):
            # Named constraints at a table level
            for sub_expr in expression.expressions:
                _handle_table_level_constraint(
                    sub_expr,
                    tables,
                    table_fq_name,
                    dialect,
                    force_constraint_name=(
                        expression.name.lower() if expression.name else None
                    ),
                )
        elif _handle_table_level_constraint(
            expression,
            tables,
            table_fq_name,
            dialect,
            force_constraint_name=None,
        ):
            # Direct Column constraints declared at the table level
            continue
        else:  # pragma: no cover
            # unlikely to happen but let's log anyway
            logger.debug(
                "No handler found for expression %s for table %s",
                expression,
                table_name,
            )
    return True


def handle_drop_table(stmt: sqlglot.exp.DDL, tables: Dict) -> bool:
    """
    Handle DROP TABLE statement.
    Returns True if the statement was handled, False otherwise.
    """
    if is_temporary_entity(stmt):
        # Skip temporary entities (e.g. DROP TEMPORARY TABLE)
        return False

    table_node = stmt.find(sqlglot.exp.Table)
    _, _, table_fq_name = _get_table_names(table_node)
    _process_table_drop(tables, table_fq_name)

    return True


def handle_alter_table(  # pylint: disable=too-many-locals
    stmt: sqlglot.exp.DDL, tables: Dict, dialect: SqlDialect
) -> bool:
    """
    Handle ALTER TABLE statement.

    Some unsupported ALTER TABLE patterns are handled manually as a Command node.
    """

    table_node = stmt.this.find(sqlglot.exp.Table)
    _, table_namespace, table_fq_name = _get_table_names(table_node)
    for action in stmt.actions:
        if isinstance(action, sqlglot.exp.AlterRename):
            # RENAME TABLE
            new_table_node = action.find(sqlglot.exp.Table)
            new_table_name, new_table_namespace, new_table_fq_name = _get_table_names(
                new_table_node
            )
            if not new_table_namespace and table_namespace:
                new_table_namespace = table_namespace
                new_table_fq_name = _build_fq_name(new_table_name, new_table_namespace)

            _process_table_rename(
                tables,
                table_fq_name,
                new_table_fq_name,
                new_table_name,
                new_table_namespace,
            )
        elif isinstance(action, sqlglot.exp.ColumnDef):
            # ADD COLUMN
            _process_column_def(action, tables, table_fq_name, dialect)
        elif isinstance(action, sqlglot.exp.Drop) and action.kind == "COLUMN":
            # DROP COLUMN
            col_name = action.find(sqlglot.exp.Column).name.lower()
            _process_column_drop(tables, table_fq_name, col_name)
        elif isinstance(action, sqlglot.exp.RenameColumn):
            # RENAME COLUMN
            old_col_node = action.this.find(sqlglot.exp.Column)
            new_col_node = action.args["to"].find(sqlglot.exp.Column)
            old_col_name = old_col_node.name.lower()
            new_col_name = new_col_node.name.lower()
            _process_column_rename(tables, table_fq_name, old_col_name, new_col_name)
        elif isinstance(action, sqlglot.exp.AlterColumn):
            # ALTER COLUMN
            # SET/DROP NOT NULL are supported
            col_name = action.this.name.lower()
            if action.args.get("allow_null") is not None:
                tables[table_fq_name]["columns"][col_name]["is_nullable"] = action.args[
                    "allow_null"
                ]
            # SET DEFAULT
            if action.args.get("default") is not None:
                tables[table_fq_name]["columns"][col_name]["default_value"] = str(
                    action.args["default"]
                )
            # DROP DEFAULT has drop=True, weirdly
            if action.args.get("drop", False):
                tables[table_fq_name]["columns"][col_name]["default_value"] = None

            # TYPE change
            if action.args.get("dtype") is not None:
                col_type, enum_values = _extract_column_type_from_column_dtype(
                    action.args["dtype"]
                )
                tables[table_fq_name]["columns"][col_name]["type"] = col_type
                tables[table_fq_name]["columns"][col_name]["enum_values"] = enum_values

        elif isinstance(action, sqlglot.exp.AddConstraint):
            # ADD CONSTRAINT, ADD PRIMARY KEY, ADD FOREIGN KEY, ADD INDEX
            for expr in action.expressions:
                # ALTER TABLE ADD INDEX does not support options that CREATE INDEX supports
                # it's limited to what CREATE TABLE ... INDEX supports
                if isinstance(expr, sqlglot.exp.Constraint):
                    # ADD CONSTRAINT, ADD UNIQUE
                    for sub_expr in expr.expressions:
                        _handle_table_level_constraint(
                            sub_expr,
                            tables,
                            table_fq_name,
                            dialect,
                            force_constraint_name=expr.name,
                        )
                else:
                    # directly invoked, e.g. ADD PRIMARY KEY
                    _handle_table_level_constraint(
                        expr,
                        tables,
                        table_fq_name,
                        dialect,
                        force_constraint_name=expr.name,
                    )
        # There is no ALTER CONSTRAINT statement that's meaningful to us
        elif isinstance(action, sqlglot.exp.Drop) and action.kind in [
            "INDEX",
            "CONSTRAINT",
        ]:
            # DROP INDEX, DROP CONSTRAINT
            name = action.find(sqlglot.exp.Identifier).name.lower()
            _remove_constraint_from_tables(name, tables, table_fq_name)
            _remove_index_from_tables(name, tables, table_fq_name)
        elif isinstance(action, sqlglot.exp.Drop) and action.kind == "FOREIGN KEY":
            # DROP FOREIGN KEY
            # not sure why this is under Drop node
            fk_name = action.find(sqlglot.exp.Identifier).name.lower()
            tables[table_fq_name]["foreign_keys"].pop(fk_name, None)
        elif isinstance(action, sqlglot.exp.Command):
            # Not fully supported actions parsed as Command nodes
            # DROP PRIMARY KEY
            if action.args.get("this") == "DROP" and "PRIMARY KEY" in action.args.get(
                "expression", ""
            ):
                tables[table_fq_name]["primary_key"] = {}

    return True


def handle_create_index(
    stmt: sqlglot.exp.DDL, tables: Dict, dialect: SqlDialect
) -> bool:
    """
    Handle CREATE INDEX statement.

    index_name, table_name, table_namespace
    column_names, uniqueness, index_type,
    includes (covering index), where
    options (partial, expression, etc.)
    """
    explicit_index_name = stmt.this.name.lower()
    is_unique_index = stmt.args.get("unique", False)
    table_node = stmt.this.find(sqlglot.exp.Table)
    table_name, _, table_fq_name = _get_table_names(table_node)
    index_params_node = stmt.this.find(sqlglot.exp.IndexParameters)

    col_names_to_expression = {
        col_expr.name.lower(): str(col_node.this)
        for col_node in index_params_node.args.get("columns", [])
        if (col_expr := col_node.find(sqlglot.exp.Column)) is not None
    }
    all_col_names = list(col_names_to_expression.keys())
    if not all_col_names:
        # Technically possible with direct expression without columns
        logger.debug(
            "No columns found for index %s on table %s", stmt.this.name, table_name
        )
        return False

    index_name = _build_idx_constraint_name(
        "unique" if is_unique_index else "index",
        tables[table_fq_name],
        all_col_names,
        dialect,
        explicit_index_name,
    )
    index_type = str(index_params_node.args.get("using") or "btree").lower()
    where_expr_node = index_params_node.find(sqlglot.exp.Where)
    tables[table_fq_name]["indexes"][index_name] = {
        "name": index_name,
        "columns": all_col_names,
        "is_unique": is_unique_index,
        # The following are only available in CREATE INDEX statements for the most part
        "expressions": {
            col_name: col_expr
            for col_name, col_expr in col_names_to_expression.items()
            if col_name != col_expr
        },
        "index_type": index_type,
        "where": str(where_expr_node.this) if where_expr_node else None,
    }
    return True


def _remove_index_from_tables(
    index_name: str,
    tables: Dict,
    table_fq_name: Optional[str] = None,
) -> None:
    """
    Remove an index from all tables.
    """
    if table_fq_name:
        tables[table_fq_name]["indexes"].pop(index_name, None)
    else:
        for table_meta in tables.values():
            if index_name in table_meta["indexes"]:
                table_meta["indexes"].pop(index_name, None)


def _remove_constraint_from_tables(
    constraint_name: str,
    tables: Dict,
    table_fq_name: Optional[str] = None,
) -> None:
    """
    Remove a constraint from all tables.
    """
    _remove_index_from_tables(constraint_name, tables, table_fq_name)
    if table_fq_name:
        tables[table_fq_name]["foreign_keys"].pop(constraint_name, None)
        tables[table_fq_name]["primary_key"].pop(constraint_name, None)
        tables[table_fq_name]["checks"].pop(constraint_name, None)
    else:  # pragma: no cover
        # unlikely to happen since DROP CONSTRAINT is always part of ALTER TABLE
        # We will just keep here for completeness
        for table_meta in tables.values():
            if constraint_name in table_meta["foreign_keys"]:
                table_meta["foreign_keys"].pop(constraint_name, None)
            if constraint_name in table_meta["primary_key"]:
                table_meta["primary_key"].pop(constraint_name, None)
            if constraint_name in table_meta["checks"]:
                table_meta["checks"].pop(constraint_name, None)


def _rename_constraint_in_tables(
    source_constraint_name: str,
    new_constraint_name: str,
    tables: Dict,
    table_fq_name: Optional[str] = None,
) -> None:
    """
    Rename a constraint in all tables.
    """
    current_constraint_info_by_table = defaultdict(dict)
    tables_to_update = [table_fq_name] if table_fq_name else list(tables.keys())
    for iter_table_fq_name in tables_to_update:
        for const_type in ["foreign_keys", "primary_key", "checks"]:
            if source_constraint_name in tables[iter_table_fq_name][const_type]:
                current_constraint_info_by_table[iter_table_fq_name][const_type] = (
                    deepcopy(
                        tables[iter_table_fq_name][const_type][source_constraint_name]
                    )
                )
                _remove_constraint_from_tables(
                    source_constraint_name, tables, iter_table_fq_name
                )

    for (
        iter_table_fq_name,
        iter_constraint_info_by_type,
    ) in current_constraint_info_by_table.items():
        for const_type, iter_constraint_info in iter_constraint_info_by_type.items():
            tables[iter_table_fq_name][const_type][new_constraint_name] = {
                **iter_constraint_info,
                "name": new_constraint_name,
            }


def _rename_index_in_tables(
    source_index_name: str,
    new_index_name: str,
    tables: Dict,
    table_fq_name: Optional[str] = None,
) -> None:
    """
    Rename an index in all tables.
    """
    current_index_info_by_table = {}
    # Remove the old index from a specific table
    tables_to_update = [table_fq_name] if table_fq_name else list(tables.keys())
    for iter_table_fq_name in tables_to_update:
        if source_index_name in tables[iter_table_fq_name]["indexes"]:
            current_index_info_by_table[iter_table_fq_name] = deepcopy(
                tables[iter_table_fq_name]["indexes"][source_index_name]
            )
        _remove_index_from_tables(source_index_name, tables, iter_table_fq_name)

    # Add the new index to the table
    for iter_table_fq_name, iter_index_info in current_index_info_by_table.items():
        tables[iter_table_fq_name]["indexes"][new_index_name] = {
            **iter_index_info,
            "name": new_index_name,
        }


def handle_alter_index(stmt: sqlglot.exp.DDL, tables: Dict) -> bool:
    """
    Handle ALTER INDEX statement.
    We will only support ALTER INDEX RENAME.
    ALTER INDEX ADD/DROP COLUMN is not valid and needs to be handled as DROP/CREATE INDEX.
    ALTER INDEX SET/DROP PROPERTY for storage parameters is not supported.

    ALTER INDEX idx SET SCHEMA is technically valid but unusual in Postgres
    but not supported by alembic and liquibase.
    """
    source_index_node = stmt.this.find(sqlglot.exp.Table)
    source_index_name, source_index_namespace, _ = _get_table_names(source_index_node)

    maybe_table_node = stmt.find(sqlglot.exp.OnProperty)
    maybe_table_fq_name = (
        _build_fq_name(maybe_table_node.name, source_index_namespace)
        if maybe_table_node
        else None
    )

    for action in stmt.actions:
        if isinstance(action, sqlglot.exp.AlterRename):
            new_index_node = action.this.find(sqlglot.exp.Table)
            new_index_name, _, _ = _get_table_names(new_index_node)
            _rename_index_in_tables(
                source_index_name,
                new_index_name,
                tables,
                maybe_table_fq_name,
            )

    return True


def handle_drop_index(stmt: sqlglot.exp.DDL, tables: Dict) -> bool:
    """
    Handle DROP INDEX statement.

    MySQL: DROP INDEX index_name ON table_name; (no schema qualification)
    Postgres: DROP INDEX schema.index_name; (no table selection)
    """
    # Weird that sqlglot adds index as a Table node :/
    index_node = stmt.find(sqlglot.exp.Table)
    index_name, index_namespace, _ = _get_table_names(index_node)

    maybe_table_node = stmt.find(sqlglot.exp.OnProperty)
    if maybe_table_node:
        # Possible only in MySQL and there's no schema expected
        # we will still use the index_namespace to find the table
        table_name = maybe_table_node.name.lower()
        table_fq_name = _build_fq_name(table_name, index_namespace)
        _remove_index_from_tables(index_name, tables, table_fq_name)
    else:
        # Handle cases without ON clause
        _remove_index_from_tables(index_name, tables)

    return True


def handle_drop_schema_database(
    stmt: sqlglot.exp.DDL, tables: Dict, dialect: SqlDialect
) -> bool:
    """
    Handle DROP SCHEMA statement or DROP DATABASE statement.
    In MySQL they are equivalent, but different in postgres
    Returns True if the statement was handled, False otherwise.
    """
    if dialect is SqlDialect.POSTGRES and stmt.kind == "DATABASE":
        # In Postgres, it's not valid to drop the current database.
        # let's make sure to not accidentally drop a schema with same name
        return False

    schema_name = stmt.find(sqlglot.exp.Identifier).name.lower()
    tables_to_remove = []
    for table_fq_name, table_meta in tables.items():
        if table_meta["namespace"] == schema_name:
            tables_to_remove.append(table_fq_name)

    for table_fq_name in tables_to_remove:
        _process_table_drop(tables, table_fq_name)

    return True


def _rewrite_alter_table_change_or_modify_column(stmt: sqlglot.exp.DDL) -> list[str]:
    """
    Rewrite ALTER TABLE CHANGE/MODIFY COLUMN into a DROP COLUMN and ADD COLUMN statement.

    -- Redefines
    ALTER TABLE orders MODIFY COLUMN price DECIMAL(10,4) NOT NULL DEFAULT 10.0

    -- Renames and Redefines
    ALTER TABLE orders CHANGE COLUMN sku product_code VARCHAR(128) NOT NULL;
    """
    stmt_str = _remove_comments_from_sql(str(stmt)).strip()

    # ALTER TABLE <table> (CHANGE|MODIFY) [COLUMN] ...
    modify_regex = re.compile(
        rf"^ALTER\s+TABLE\s+(?P<table>{TABLE})\s+"
        rf"(?P<kind>MODIFY)(?:\s+COLUMN)?\s+"
        rf"(?P<col1>{IDENT})"
        rf"\s+(?P<definition>.+)$",
    )

    change_regex = re.compile(
        rf"^ALTER\s+TABLE\s+(?P<table>{TABLE})\s+"
        rf"(?P<kind>CHANGE)(?:\s+COLUMN)?\s+"
        rf"(?P<col1>{IDENT})"
        rf"(?:\s+(?P<col2>{IDENT}))?"  # only present for CHANGE
        rf"\s+(?P<definition>.+)$",
    )

    m = modify_regex.match(stmt_str) or change_regex.match(stmt_str)
    if not m:
        return []

    kind = m.group("kind")
    table = m.group("table")
    col1 = m.group("col1")
    col2 = m.group("col2") if kind == "CHANGE" else col1
    definition = m.group("definition")

    # Preserve trailing FIRST/AFTER if present
    pos = ""
    mp = re.search(r"(?:\s+)(FIRST|AFTER\s+\S+)$", definition, re.IGNORECASE)
    if mp:
        pos = " " + mp.group(1)
        definition = definition[: mp.start()].rstrip()

    # Even though this isn't technically a valid SQL to re-add
    # an existing column, this helps override existing column definition
    # without impacting the indexes and relationships.
    split_stmts = [
        f"ALTER TABLE {table} ADD COLUMN {col2} {definition}{pos}",
    ]

    if kind.upper() == "CHANGE":
        # Rename the column first
        split_stmts.insert(
            0,
            f"ALTER TABLE {table} RENAME COLUMN {col1} TO {col2}",
        )

    return split_stmts


def _handle_command(  # pylint: disable=too-many-locals
    stmt: sqlglot.exp.Command,
    tables_by_name: Dict,
    dialect: SqlDialect,
) -> bool:
    """
    Handle a command statement for DDLs that sqlglot does not yet support
    """
    stmt_str = _remove_comments_from_sql(str(stmt)).strip()
    # ALTER TABLE <schema?.>table MODIFY COLUMN <col> <definition>;
    # ALTER TABLE <schema?.>table CHANGE COLUMN <col> <newcol> <definition>;
    if split_stmts := _rewrite_alter_table_change_or_modify_column(stmt):
        # Let's rewrite the statement into a DROP COLUMN and ADD COLUMN statement
        # and process them separately.
        for new_stmt in split_stmts:
            new_stmt_parsed = _parse_with_sqlglot(new_stmt, dialect)
            if len(new_stmt_parsed) > 0:
                _process_statement(new_stmt_parsed[0], tables_by_name, dialect)
        return True

    # ALTER TABLE <schema?.>table RENAME INDEX <old> TO <new>;
    # ALTER TABLE <schema?.>table RENAME CONSTRAINT <old> TO <new>;
    rename_index_or_constraint_match = re.compile(
        rf"^ALTER\s+TABLE\s+(?P<table>{TABLE})\s+RENAME\s+"
        rf"(?P<kind>INDEX|CONSTRAINT)\s+"
        rf"{IDENT}\s+TO\s+{IDENT}\s*;?\s*$",
        re.IGNORECASE,
    )

    # ALTER TABLE <schema?.>table SET SCHEMA <newschema>;
    set_schema_match = re.compile(
        rf"^ALTER\s+TABLE\s+"
        rf"(?:(?P<ifexists>IF\s+EXISTS)\s+)?"
        rf"(?:(?P<only>ONLY)\s+)?"
        rf"(?P<table>{TABLE})\s+SET\s+SCHEMA\s+(?P<newschema>{IDENT})\s*;?\s*$",
        re.IGNORECASE,
    )

    def _pick(triple):
        # From a 3-tuple (a,b,c) produced by IDENT, return the first non-None.
        a, b, c = triple
        return a or b or c

    def _parse_table(table_groups):
        # TABLE contributes 6 groups (two IDENTs * 3):
        # 0..2 -> schema or table (if unqualified)
        # 3..5 -> table (if qualified) else Nones
        first = _pick(table_groups[0:3])
        second = _pick(table_groups[3:6])
        if second:
            return first, second, f"{first}.{second}"
        return None, first, first

    if m := set_schema_match.match(stmt_str):
        _, table, fq_table = _parse_table(m.groups()[3:9])
        schema_triple = m.groups()[10:13]
        new_schema = _pick(schema_triple).lower()
        if new_schema and fq_table in tables_by_name:
            new_fq_table = _build_fq_name(table, new_schema)
            _process_table_rename(
                tables_by_name, fq_table, new_fq_table, table, new_schema
            )

        return True

    if m := rename_index_or_constraint_match.match(stmt_str):
        g = (
            m.groups()
        )  # includes TABLE's 6 slots + kind + old IDENT 3 + new IDENT 3 + (maybe nothing else)
        _, _, fq = _parse_table(g[1:7])
        # After TABLE (6 groups) comes named 'kind'; groups() doesn't include named separately,
        # so we get old/new from the next 3+3 slices:
        kind = m.group("kind")  # Use named group instead of g[7]
        old_name = _pick(g[8:11])
        new_name = _pick(g[11:14])

        if kind.upper() == "INDEX":
            _rename_index_in_tables(old_name, new_name, tables_by_name, fq)
            return True
        if kind.upper() == "CONSTRAINT":
            _rename_constraint_in_tables(old_name, new_name, tables_by_name, fq)
            return True

    logger.debug("No command handler found: %s", stmt)
    return False


def _process_statement(
    stmt: sqlglot.exp.DDL,
    tables_by_name: Dict,
    dialect: SqlDialect,
) -> bool:
    """
    Process a single statement.
    """
    key = stmt.key.upper() if hasattr(stmt, "key") else None
    kind = stmt.kind.upper() if hasattr(stmt, "kind") else None
    logger.debug("Attempting to process statement: %s", stmt)

    if key == "CREATE" and kind == "TABLE":
        return handle_create_table(stmt, tables_by_name, dialect)
    if key == "DROP" and kind == "TABLE":
        return handle_drop_table(stmt, tables_by_name)
    if key == "ALTER" and kind == "TABLE":
        return handle_alter_table(stmt, tables_by_name, dialect)
    if key == "CREATE" and kind == "INDEX":
        return handle_create_index(stmt, tables_by_name, dialect)
    if key == "ALTER" and kind == "INDEX":
        return handle_alter_index(stmt, tables_by_name)
    if key == "DROP" and kind == "INDEX":
        return handle_drop_index(stmt, tables_by_name)
    if key == "DROP" and kind in {"DATABASE", "SCHEMA"}:
        # Delete all the tables in the schema so far (postgres needs CASCADE)
        return handle_drop_schema_database(stmt, tables_by_name, dialect)
    if key == "COMMAND":
        # NOT fully supported -- so we will handle the important ones here
        return _handle_command(stmt, tables_by_name, dialect)

    logger.debug("No statement handler found: %s", stmt)
    return False


def _parse_with_sqlglot(sql_text: str, dialect: SqlDialect) -> List[sqlglot.exp.DDL]:
    """
    Parse DDL SQL text with sqlglot.
    """
    # Can handle errors later if needed
    try:
        return sqlglot.parse(sql_text, dialect=dialect.value)
    except sqlglot.ParseError as e:
        raise ParseError(f"Failed to parse DDL: {e}") from e


def parse_ddl(sql_text: str, dialect: SqlDialect) -> List[Table]:
    """
    Parse DDL SQL text and return a Schema namedtuple with tables, columns, indexes, and constraints.

    Args:
        sql_text: The DDL SQL text to parse
        dialect: The SQL dialect to use for parsing (default: "sqlite")

    Returns:
        List[Table]: A list of Table namedtuples containing all parsed table information

    Note:
        - Does not validate the DDLs and assumes that the DDLs are valid.
        - Does not support USE SCHEMA or similar statements.
        - Does not support VIEWS, TRIGGERS, SEQUENCES, etc.
        - Assumes that DDLs are scoped to a single database
    """

    tables_by_name = defaultdict(
        lambda: {
            "name": None,
            "namespace": None,
            "fq_name": None,
            "columns": {},
            "indexes": {},
            "primary_key": {},
            "foreign_keys": {},
            "checks": {},
            # Keep track of all autogen names to deconflict them
            "all_autogen_idx_constraint_names": [],
        }
    )

    # if sqlglot fails to parse, we must fail.
    parsed = _parse_with_sqlglot(sql_text, dialect)

    statements = []
    for stmt in parsed:
        if is_generic_command_ast(stmt):
            # sqlglot ends up creating a Command node for unsupported statements.
            # One example is a multi-action DDL statements
            # (e.g ALTER TABLE users RENAME TO users_new, ADD COLUMN new_col INT;)
            # We need to normalize them into several single-action statements.
            new_stmts = make_commands_parsable_if_possible(str(stmt))
            # Unfortunately, we need to parse each statement again to get the AST.
            # it's currently not possible to reliably split DDL file into statements without building AST.

            # We are currently letting parse errors to propagate.
            # assuming all DDLs are valid for a dialect.
            for stmt in new_stmts:
                parsed_stmts = _parse_with_sqlglot(stmt, dialect)
                statements.extend(parsed_stmts)

        elif stmt is not None:
            # stmt can be None for comments, empty strings, etc.
            statements.append(stmt)

    for stmt in statements:
        _process_statement(stmt, tables_by_name, dialect)

    # Prune out table entries that may be coming from bad DDLs
    # Even though we assume that the DDLs are valid,
    # we want to minimize failure on bad DDLs.
    # e.g. DROP INDEX idx_email ON non_existent_table
    tables_by_name = {
        table_fq_name: table_meta
        for table_fq_name, table_meta in tables_by_name.items()
        if table_meta["name"] and table_meta["columns"] and table_meta["fq_name"]
    }
    # Convert the schema to namedtuples return type

    tables = []
    for table_fq_name, table_meta in tables_by_name.items():
        # Convert columns
        columns = [
            Column(
                name=col_name,
                type=col_info["type"],
                is_nullable=col_info["is_nullable"],
                default_value=col_info.get("default_value"),
                enum_values=col_info.get("enum_values"),
            )
            for col_name, col_info in table_meta["columns"].items()
        ]

        # Convert indexes
        indexes = [
            Index(
                name=idx_name,
                columns=idx_info["columns"],
                is_unique=idx_info.get("is_unique", False),
                expressions=idx_info.get("expressions", {}),
                index_type=idx_info.get("index_type"),
                where=idx_info.get("where"),
            )
            for idx_name, idx_info in table_meta["indexes"].items()
        ]

        # Convert primary key
        first_pk = next(iter(table_meta["primary_key"].values()), None)
        primary_key = (
            PrimaryKey(
                name=first_pk["name"],
                columns=first_pk["columns"],
            )
            if first_pk
            else None
        )

        # Convert foreign keys
        # if schema changes, we may need to rebuild the fq name for the referenced table
        foreign_keys = [
            ForeignKey(
                name=fk["name"],
                columns=fk["columns"],
                references_table_fq_name=fk["references_table_fq_name"],
                references_table_name=fk["references_table_name"],
                references_table_namespace=fk["references_table_namespace"],
                references_columns=fk["references_columns"],
                on_delete=fk.get("on_delete"),
                on_update=fk.get("on_update"),
            )
            for fk in table_meta["foreign_keys"].values()
        ]

        # Convert checks
        checks = [
            Check(
                name=check["name"],
                columns=check["columns"],
                expression=check["expression"],
            )
            for check in table_meta["checks"].values()
        ]

        tables.append(
            Table(
                name=table_meta["name"],
                namespace=table_meta["namespace"],
                fq_name=table_fq_name,
                columns=columns,
                indexes=indexes,
                primary_key=primary_key,
                foreign_keys=foreign_keys,
                checks=checks,
            )
        )

    return tables
