import os
import sqlite3
from contextlib import closing
from typing import Any, Dict, List, Optional, Tuple

from specklepy.core.helpers import speckle_path_provider
from specklepy.logging.exceptions import SpeckleException
from specklepy.transports.abstract_transport import AbstractTransport


class SQLiteTransport(AbstractTransport):
    _name = "SQLite"
    _base_path: Optional[str] = None
    _root_path: Optional[str] = None
    __connection: Optional[sqlite3.Connection] = None
    app_name: str = ""
    scope: str = ""
    saved_obj_count: int = 0
    max_size: Optional[int] = None
    _current_batch: Optional[List[Tuple[str, str]]] = None
    _current_batch_size: Optional[int] = None

    def __init__(
        self,
        base_path: Optional[str] = None,
        app_name: Optional[str] = None,
        scope: Optional[str] = None,
        max_batch_size_mb: float = 10.0,
        **data: Any,
    ) -> None:
        super().__init__(**data)
        self.app_name = app_name or "Speckle"
        self.scope = scope or "Objects"
        self._base_path = base_path or self.get_base_path(self.app_name)
        self.max_size = int(max_batch_size_mb * 1000 * 1000)
        self._current_batch = []
        self._current_batch_size = 0

        try:
            os.makedirs(self._base_path, exist_ok=True)

            self._root_path = os.path.join(
                os.path.join(self._base_path, f"{self.scope}.db")
            )
            self.__initialise()
        except Exception as ex:
            raise SpeckleException(
                f"SQLiteTransport could not initialise {self.scope}.db at"
                f" {self._base_path}. Either provide a different `base_path` or use an"
                " alternative transport.",
                ex,
            )

    def __repr__(self) -> str:
        return f"SQLiteTransport(app: '{self.app_name}', scope: '{self.scope}')"

    @staticmethod
    def get_base_path(app_name):
        # # from appdirs https://github.com/ActiveState/appdirs/blob/master/appdirs.py
        # # default mac path is not the one we use (we use unix path), so using special case for this
        # system = sys.platform
        # if system.startswith("java"):
        #     import platform

        #     os_name = platform.java_ver()[3][0]
        #     if os_name.startswith("Mac"):
        #         system = "darwin"

        # if system != "darwin":
        #     return user_data_dir(appname=app_name, appauthor=False, roaming=True)

        # path = os.path.expanduser("~/.config/")
        # return os.path.join(path, app_name)

        return str(
            speckle_path_provider.user_application_data_path().joinpath(app_name)
        )

    def save_object_from_transport(
        self, id: str, source_transport: AbstractTransport
    ) -> None:
        """Adds an object from the given transport to the the local db

        Arguments:
            id {str} -- the object id
            source_transport {AbstractTransport)
            -- the transport through which the object can be found
        """
        serialized_object = source_transport.get_object(id)
        self.save_object(id, serialized_object)

    def save_object(self, id: str, serialized_object: str) -> None:
        """
        Adds an object to the current batch to be written to the db.
        If the current batch is full,
        the batch is written to the db and the current batch is reset.

        Arguments:
            id {str} -- the object id
            serialized_object {str} -- the full string representation of the object
        """
        obj_size = len(serialized_object)
        if (
            not self._current_batch
            or self._current_batch_size + obj_size < self.max_size
        ):
            self._current_batch.append((id, serialized_object))
            self._current_batch_size += obj_size
            return

        self.save_current_batch()
        self._current_batch = [(id, serialized_object)]
        self._current_batch_size = obj_size

    def save_current_batch(self) -> None:
        """Save the current batch of objects to the local db"""
        self.__check_connection()
        try:
            with closing(self.__connection.cursor()) as c:
                c.executemany(
                    "INSERT OR IGNORE INTO objects(hash, content) VALUES(?,?)",
                    self._current_batch,
                )
                self.__connection.commit()
        except Exception as ex:
            raise SpeckleException(
                "Could not save the batch of objects to the local db. Inner exception:"
                f" {ex}",
                ex,
            )

    def get_object(self, id: str) -> str or None:
        self.__check_connection()
        with closing(self.__connection.cursor()) as c:
            row = c.execute(
                "SELECT * FROM objects WHERE hash = ? LIMIT 1", (id,)
            ).fetchone()
        return row[1] if row else None

    def has_objects(self, id_list: List[str]) -> Dict[str, bool]:
        ret = {}
        self.__check_connection()
        with closing(self.__connection.cursor()) as c:
            for id in id_list:
                row = c.execute(
                    "SELECT 1 FROM objects WHERE hash = ? LIMIT 1", (id,)
                ).fetchone()
                ret[id] = bool(row)
        return ret

    def begin_write(self):
        self._object_cache = []
        self.saved_obj_count = 0

    def end_write(self):
        if self._current_batch:
            self.save_current_batch()
        self._current_batch = []
        self._current_batch_size = 0

    def copy_object_and_children(
        self, id: str, target_transport: AbstractTransport
    ) -> str:
        raise NotImplementedError

    def get_all_objects(self):
        """
        Returns all the objects in the store.
        NOTE: do not use for large collections!
        """
        self.__check_connection()
        with closing(self.__connection.cursor()) as c:
            rows = c.execute("SELECT * FROM objects").fetchall()
        return rows

    def close(self):
        """Close the connection to the database"""
        if self.__connection:
            self.__connection.close()
            self.__connection = None

    def __initialise(self) -> None:
        self.__connection = sqlite3.connect(self._root_path)
        with closing(self.__connection.cursor()) as c:
            c.execute(
                """ CREATE TABLE IF NOT EXISTS objects(
                      hash TEXT PRIMARY KEY,
                      content TEXT
                    ) WITHOUT ROWID;"""
            )
            c.execute("PRAGMA journal_mode='wal';")
            c.execute("PRAGMA count_changes=OFF;")
            c.execute("PRAGMA temp_store=MEMORY;")
            self.__connection.commit()

    def __check_connection(self):
        if not self.__connection:
            self.__connection = sqlite3.connect(self._root_path)

    def __del__(self):
        self.close()
