"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const constants = require("../../../backend/shared/constants");
const transliterator_lambda_1 = require("../../../backend/transliterator/transliterator.lambda");
jest.mock('child_process');
jest.mock('jsii-docgen');
jest.mock('jsii-rosetta/lib/commands/transliterate');
jest.mock('../../../backend/shared/code-artifact.lambda-shared');
beforeEach((done) => {
    AWSMock.setSDKInstance(AWS);
    process.env.TARGET_LANGUAGE = 'typescript';
    done();
});
afterEach((done) => {
    AWSMock.restore();
    delete process.env.TARGET_LANGUAGE;
    transliterator_lambda_1.reset();
    done();
});
describe('VPC Endpoints', () => {
    const previousEnv = process.env;
    const endpoint = 'codeartifact.d.bermuda-triangle-1.amazonaws.com';
    const apiEndpoint = 'codeartifact.api.bermuda-triangle-1.amazonaws.com';
    const domain = 'domain-name';
    const domainOwner = '123456789012';
    beforeAll(() => {
        process.env = {
            ...previousEnv,
            CODE_ARTIFACT_REPOSITORY_ENDPOINT: endpoint,
            CODE_ARTIFACT_DOMAIN_NAME: domain,
            CODE_ARTIFACT_DOMAIN_OWNER: domainOwner,
            CODE_ARTIFACT_API_ENDPOINT: apiEndpoint,
        };
    });
    afterAll(() => {
        process.env = { ...previousEnv };
    });
    test('happy path', async () => {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const forPackage = require('jsii-docgen').Documentation.forPackage;
        forPackage.mockImplementation(async (target) => {
            return new MockDocumentation(target);
        });
        // GIVEN
        const packageScope = 'scope';
        const packageName = 'package-name';
        const packageVersion = '1.2.3-dev.4';
        const s3Event = {
            Records: [{
                    awsRegion: 'bemuda-triangle-1',
                    s3: {
                        bucket: {
                            name: 'dummy-bucket',
                        },
                        object: {
                            key: `${constants.STORAGE_KEY_PREFIX}%40${packageScope}/${packageName}/v${packageVersion}${constants.ASSEMBLY_KEY_SUFFIX}`,
                            versionId: 'VersionId',
                        },
                    },
                }],
        };
        const event = {
            Records: [{
                    Sns: {
                        Message: JSON.stringify(s3Event),
                    },
                }],
        };
        const assembly = {
            targets: { python: {} },
        };
        // mock the assembly request
        mockFetchAssembly(assembly);
        // mock the file uploads
        mockPutDocs('/docs-typescript.md');
        const created = await transliterator_lambda_1.handler(event, {});
        expect(created.length).toEqual(1);
        expect(created[0].key).toEqual(`data/@${packageScope}/${packageName}/v${packageVersion}/docs-typescript.md`);
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        expect(require('../../../backend/shared/code-artifact.lambda-shared').logInWithCodeArtifact).toHaveBeenCalledWith({
            endpoint,
            domain,
            domainOwner,
            apiEndpoint,
        });
    });
});
test('uploads a file per language (scoped package)', async () => {
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const forPackage = require('jsii-docgen').Documentation.forPackage;
    forPackage.mockImplementation(async (target) => {
        return new MockDocumentation(target);
    });
    // GIVEN
    const packageScope = 'scope';
    const packageName = 'package-name';
    const packageVersion = '1.2.3-dev.4';
    const s3Event = {
        Records: [{
                awsRegion: 'bemuda-triangle-1',
                s3: {
                    bucket: {
                        name: 'dummy-bucket',
                    },
                    object: {
                        key: `${constants.STORAGE_KEY_PREFIX}%40${packageScope}/${packageName}/v${packageVersion}${constants.ASSEMBLY_KEY_SUFFIX}`,
                        versionId: 'VersionId',
                    },
                },
            }],
    };
    const event = {
        Records: [{
                Sns: {
                    Message: JSON.stringify(s3Event),
                },
            }],
    };
    const assembly = {
        targets: { python: {} },
    };
    // mock the assembly request
    mockFetchAssembly(assembly);
    // mock the file uploads
    mockPutDocs('/docs-typescript.md');
    const created = await transliterator_lambda_1.handler(event, {});
    expect(created.length).toEqual(1);
    expect(created[0].key).toEqual(`data/@${packageScope}/${packageName}/v${packageVersion}/docs-typescript.md`);
});
test('uploads a file per submodule (unscoped package)', async () => {
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const forPackage = require('jsii-docgen').Documentation.forPackage;
    forPackage.mockImplementation(async (target) => {
        return new MockDocumentation(target);
    });
    // GIVEN
    const packageName = 'package-name';
    const packageVersion = '1.2.3-dev.4';
    const s3Event = {
        Records: [{
                awsRegion: 'bemuda-triangle-1',
                s3: {
                    bucket: {
                        name: 'dummy-bucket',
                    },
                    object: {
                        key: `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.ASSEMBLY_KEY_SUFFIX}`,
                        versionId: 'VersionId',
                    },
                },
            }],
    };
    const event = {
        Records: [{
                Sns: {
                    Message: JSON.stringify(s3Event),
                },
            }],
    };
    const assembly = {
        targets: { python: {} },
        submodules: { '@scope/package-name.sub1': {}, '@scope/package-name.sub2': {} },
    };
    // mock the assembly request
    mockFetchAssembly(assembly);
    // mock the file uploads
    mockPutDocs('/docs-typescript.md', '/docs-sub1-typescript.md', '/docs-sub2-typescript.md');
    const created = await transliterator_lambda_1.handler(event, {});
    expect(created.map(({ key }) => key)).toEqual([
        `data/${packageName}/v${packageVersion}/docs-typescript.md`,
        `data/${packageName}/v${packageVersion}/docs-sub1-typescript.md`,
        `data/${packageName}/v${packageVersion}/docs-sub2-typescript.md`,
    ]);
});
describe('markers for un-supported languages', () => {
    beforeEach((done) => {
        // Switch language, as TypeScript is always supported 🙃
        process.env.TARGET_LANGUAGE = 'python';
        done();
    });
    afterEach((done) => {
        delete process.env.TARGET_LANGUAGE;
        done();
    });
    test('uploads ".not-supported" markers as relevant', async () => {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const forPackage = require('jsii-docgen').Documentation.forPackage;
        forPackage.mockImplementation(async (target) => {
            return new MockDocumentation(target);
        });
        // GIVEN
        const packageName = 'package-name';
        const packageVersion = '1.2.3-dev.4';
        const s3Event = {
            Records: [{
                    awsRegion: 'bemuda-triangle-1',
                    s3: {
                        bucket: {
                            name: 'dummy-bucket',
                        },
                        object: {
                            key: `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.ASSEMBLY_KEY_SUFFIX}`,
                            versionId: 'VersionId',
                        },
                    },
                }],
        };
        const event = {
            Records: [{
                    Sns: {
                        Message: JSON.stringify(s3Event),
                    },
                }],
        };
        const assembly = {
            targets: { phony: {} },
            submodules: { '@scope/package-name.sub1': {}, '@scope/package-name.sub2': {} },
        };
        // mock the assembly request
        mockFetchAssembly(assembly);
        // mock the file uploads
        mockPutDocs(`/docs-python.md${constants.NOT_SUPPORTED_SUFFIX}`, `/docs-sub1-python.md${constants.NOT_SUPPORTED_SUFFIX}`, `/docs-sub2-python.md${constants.NOT_SUPPORTED_SUFFIX}`);
        const created = await transliterator_lambda_1.handler(event, {});
        expect(created.map(({ key }) => key)).toEqual([
            `data/${packageName}/v${packageVersion}/docs-python.md${constants.NOT_SUPPORTED_SUFFIX}`,
            `data/${packageName}/v${packageVersion}/docs-sub1-python.md${constants.NOT_SUPPORTED_SUFFIX}`,
            `data/${packageName}/v${packageVersion}/docs-sub2-python.md${constants.NOT_SUPPORTED_SUFFIX}`,
        ]);
    });
});
class MockDocumentation {
    constructor(target) {
        this.target = target;
    }
    render() {
        return {
            render: () => `docs for ${this.target}`,
        };
    }
}
function mockFetchAssembly(response) {
    AWSMock.mock('S3', 'getObject', (request, callback) => {
        if (request.Key.endsWith(constants.ASSEMBLY_KEY_SUFFIX)) {
            callback(null, {
                Body: JSON.stringify(response),
            });
        }
        else {
            throw new Error(`Unexpected GET request: ${request.Key}`);
        }
    });
}
function mockPutDocs(...suffixes) {
    AWSMock.mock('S3', 'putObject', (request, callback) => {
        if (suffixes.filter(s => request.Key.endsWith(s)).length > 0) {
            callback(null, { VersionId: `versionId-${request.Key}` });
        }
        else {
            throw new Error(`Unexpected PUT request: ${request.Key}`);
        }
    });
}
//# sourceMappingURL=data:application/json;base64,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