"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrossRegionSupportStack = exports.CrossRegionSupportConstruct = void 0;
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const REQUIRED_ALIAS_PREFIX = 'alias/';
/**
 * A class needed to work around CodePipeline's extremely small (100 characters)
 * limit for the name/ARN of the key in the ArtifactStore.
 * Limits the length of the alias' auto-generated name to 50 characters.
 */
class AliasWithShorterGeneratedName extends kms.Alias {
    generatePhysicalName() {
        let baseName = super.generatePhysicalName();
        if (baseName.startsWith(REQUIRED_ALIAS_PREFIX)) {
            // remove the prefix, because we're taking the last characters of the name below
            baseName = baseName.substring(REQUIRED_ALIAS_PREFIX.length);
        }
        const maxLength = 50 - REQUIRED_ALIAS_PREFIX.length;
        // take the last characters, as they include the hash,
        // and so have a higher chance of not colliding
        return REQUIRED_ALIAS_PREFIX + lastNCharacters(baseName, maxLength);
    }
}
function lastNCharacters(str, n) {
    const startIndex = Math.max(str.length - n, 0);
    return str.substring(startIndex);
}
class CrossRegionSupportConstruct extends cdk.Construct {
    constructor(scope, id) {
        super(scope, id);
        const encryptionKey = new kms.Key(this, 'CrossRegionCodePipelineReplicationBucketEncryptionKey', {
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        const encryptionAlias = new AliasWithShorterGeneratedName(this, 'CrossRegionCodePipelineReplicationBucketEncryptionAlias', {
            targetKey: encryptionKey,
            aliasName: cdk.PhysicalName.GENERATE_IF_NEEDED,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        this.replicationBucket = new s3.Bucket(this, 'CrossRegionCodePipelineReplicationBucket', {
            bucketName: cdk.PhysicalName.GENERATE_IF_NEEDED,
            encryptionKey: encryptionAlias,
        });
    }
}
exports.CrossRegionSupportConstruct = CrossRegionSupportConstruct;
/**
 * A Stack containing resources required for the cross-region CodePipeline functionality to work.
 * This class is private to the aws-codepipeline package.
 */
class CrossRegionSupportStack extends cdk.Stack {
    constructor(scope, id, props) {
        super(scope, id, {
            stackName: generateStackName(props),
            env: {
                region: props.region,
                account: props.account,
            },
            synthesizer: props.synthesizer,
        });
        const crossRegionSupportConstruct = new CrossRegionSupportConstruct(this, 'Default');
        this.replicationBucket = crossRegionSupportConstruct.replicationBucket;
    }
}
exports.CrossRegionSupportStack = CrossRegionSupportStack;
function generateStackName(props) {
    return `${props.pipelineStackName}-support-${props.region}`;
}
//# sourceMappingURL=data:application/json;base64,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