"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.reset = exports.handler = void 0;
const child_process_1 = require("child_process");
const console = require("console");
const os = require("os");
const path = require("path");
const process = require("process");
const aws_sdk_1 = require("aws-sdk");
const fs = require("fs-extra");
const jsii_rosetta_1 = require("jsii-rosetta");
const transliterate_1 = require("jsii-rosetta/lib/commands/transliterate");
const clients = new Map();
const PACKAGE_KEY_REGEX = /^packages\/((?:@[^/]+\/)?[^/]+)\/v([^/]+)\/package.tgz$/;
// Capture groups:                     ┗━━━━━━━━1━━━━━━━━┛   ┗━━2━━┛
/**
 * This function receives an S3 event, and for each record, proceeds to download
 * the `.jsii` assembly the event refers to, transliterates it to Python, then
 * uploads the resulting `.jsii.python` object to S3.
 *
 * @param event   an S3 event payload
 * @param context a Lambda execution context
 *
 * @returns nothing
 */
async function handler(event, context) {
    var _a, _b;
    console.log(JSON.stringify(event, null, 2));
    const created = new Array();
    for (const record of event.Records) {
        // Key names are escaped (`@` as `%40`) in the input payload... Decode it here... We cannot use
        // `decodeURI` here because it does not undo encoding that `encodeURI` would not have done, and
        // that would not replace `@` in the position where it is in the keys... So we have to work on
        // the URI components instead.
        const inputKey = record.s3.object.key.split('/').map((comp) => decodeURIComponent(comp)).join('/');
        const [, packageName, packageVersion] = (_a = inputKey.match(PACKAGE_KEY_REGEX)) !== null && _a !== void 0 ? _a : [];
        if (packageName == null) {
            throw new Error(`Invalid object key: "${inputKey}". It was expected to match ${PACKAGE_KEY_REGEX}!`);
        }
        const client = (clients.has(record.awsRegion)
            ? clients
            : clients.set(record.awsRegion, new aws_sdk_1.S3({ region: record.awsRegion }))).get(record.awsRegion);
        console.log(`Source Bucket:  ${record.s3.bucket.name}`);
        console.log(`Source Key:     ${inputKey}`);
        console.log(`Source Version: ${record.s3.object.versionId}`);
        const object = await client.getObject({
            Bucket: record.s3.bucket.name,
            Key: inputKey,
            VersionId: record.s3.object.versionId,
        }).promise();
        const workdir = await fs.mkdtemp(path.join(os.tmpdir(), 'workdir'));
        try {
            const tarball = path.join(workdir, `${packageName.replace('@', '').replace('/', '-')}-${packageVersion}.tgz`);
            await fs.writeFile(tarball, object.Body);
            await new Promise((ok, ko) => {
                // --ignore-scripts disables lifecycle hooks, in order to prevent execution of arbitrary code
                // --no-bin-links ensures npm does not insert anything in $PATH
                const npmInstall = child_process_1.spawn('npm', ['install', '--ignore-scripts', '--no-bin-links', '--no-save', tarball], {
                    cwd: workdir,
                    env: {
                        ...process.env,
                        HOME: os.tmpdir(),
                    },
                    stdio: ['ignore', 'inherit', 'inherit'],
                });
                npmInstall.once('error', ko);
                npmInstall.once('close', (code, signal) => {
                    if (code === 0) {
                        ok();
                    }
                    else {
                        ko(`"npm install" command ${code != null ? `exited with code ${code}` : `was terminated by signal ${signal}`}`);
                    }
                });
            });
            const packageDir = path.join(workdir, 'node_modules', ...packageName.split('/'));
            await transliterate_1.transliterateAssembly([packageDir], [jsii_rosetta_1.TargetLanguage.PYTHON]);
            // Payload object key => packages/[<@scope>/]<name>/v<version>/package.tgz
            // Output object key  => packages/[<@scope>/]<name>/v<version>/assembly-python.json
            const key = inputKey.replace(/\/[^/]+$/, '/assembly-python.json');
            const response = await client.putObject({
                Bucket: record.s3.bucket.name,
                Key: key,
                Body: await fs.readFile(path.join(packageDir, '.jsii.python')),
                ContentType: 'text/json',
                Metadata: {
                    'Origin-Version-Id': (_b = record.s3.object.versionId) !== null && _b !== void 0 ? _b : 'N/A',
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            }).promise();
            created.push({
                bucket: record.s3.bucket.name,
                key,
                versionId: response.VersionId,
            });
        }
        finally {
            await fs.remove(workdir);
        }
    }
    return created;
}
exports.handler = handler;
/**
 * Visible for testing. Clears the caches so that the next execution runs clean.
 */
function reset() {
    clients.clear();
}
exports.reset = reset;
//# sourceMappingURL=data:application/json;base64,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