"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const console = require("console");
const https = require("https");
const url_1 = require("url");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const Nano = require("nano");
const shared_1 = require("../shared");
const TIMEOUT_MILLISECONDS = 10000;
const CONSTRUCT_KEYWORDS = new Set(['cdk', 'aws-cdk', 'cdk8s', 'cdktf']);
const MARKER_FILE_NAME = 'couchdb-last-transaction-id';
const NPM_REPLICA_REGISTRY_URL = 'https://replicate.npmjs.com/';
/**
 * This function triggers on a fixed schedule and reads a stream of changes frm npmjs couchdb _changes endpoint.
 * Upon invocation the function starts reading from a sequence stored in an s3 object - the `marker`.
 * If the marker fails to load (or do not exist), the stream will start from `now` - the latest change.
 * For each change:
 *  - the package version tarball will be copied from the npm registry to a stating bucket.
 *  - a message will be sent to an sqs queue
 * Currently we don't handle the function execution timeout, and accept that the last batch processed might be processed again,
 * relying on the idempotency on the consumer side.
 * npm registry API docs: https://github.com/npm/registry/blob/master/docs/REGISTRY-API.md
 * @param context a Lambda execution context
 */
async function handler(event, context) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const stagingBucket = shared_1.requireEnv('BUCKET_NAME');
    const queueUrl = shared_1.requireEnv('QUEUE_URL');
    const initialMarker = await loadLastTransactionMarker(1800000 /* @aws-cdk/cdk initial release was at 1_846_709 */);
    const config = {
        includeDocs: true,
        // pause the changes reader after each request
        wait: true,
        since: initialMarker.toFixed(),
        // `changesReader.get` stops once a response with zero changes is received, however it waits too long
        //  since we want to terminate the Lambda function we define a timeout shorter than the default
        timeout: TIMEOUT_MILLISECONDS,
        // Only items with a name
        selector: {
            name: { $gt: null },
        },
    };
    const nano = Nano(NPM_REPLICA_REGISTRY_URL);
    const db = nano.db.use('registry');
    // We need to make an explicit Promise here, because otherwise Lambda won't
    // know when it's done...
    return new Promise((ok, ko) => {
        let updatedMarker = initialMarker;
        db.changesReader.get(config)
            .on('batch', async (batch) => {
            try {
                console.log(`Received a batch of ${batch.length} element(s)`);
                const lastSeq = Math.max(...batch.map((change) => change.seq));
                // Filter out all elements that don't have a "name" in the document, as
                // these are schemas, which are not relevant to our business here.
                batch = batch.filter((item) => item.doc.name);
                console.log(`Identified ${batch.length} package update element(s)`);
                // Obtain the modified package version from the update event, and filter
                // out packages that are not of interest to us (not construct libraries).
                const versionInfos = getRelevantVersionInfos(batch);
                console.log(`Identified ${versionInfos.length} relevant package version update(s)`);
                // Process all remaining updates
                await Promise.all(versionInfos.map(processUpdatedVersion));
                // Update the transaction marker in S3.
                await saveLastTransactionMarker(lastSeq);
                updatedMarker = lastSeq;
                // If we have enough time left before timeout, proceed with the next batch, otherwise we're done here.
                if (context.getRemainingTimeInMillis() >= 120000 /* 2 minutes */) {
                    console.log('There is still time, requesting the next batch...');
                    // Note: the `resume` function is missing from the `nano` type definitions, but is there...
                    db.changesReader.resume();
                }
                else {
                    console.log('We are almost out of time, so stopping here.');
                    db.changesReader.stop();
                    ok({ initialMarker, updatedMarker });
                }
            }
            catch (err) {
                // An exception bubbled out, which means this Lambda execution has failed.
                console.error(`Unexpected error: ${err}`);
                db.changesReader.stop();
                ko(err);
            }
        })
            .once('end', () => {
            console.log('No more updates to process, exiting.');
            ok({ initialMarker, updatedMarker });
        });
    });
    //#region Last transaction marker
    /**
     * Loads the last transaction marker from S3.
     *
     * @param defaultValue the value to return in case the marker does not exist
     *
     * @returns the value of the last transaction marker.
     */
    async function loadLastTransactionMarker(defaultValue) {
        try {
            const response = await shared_1.aws.s3().getObject({
                Bucket: stagingBucket,
                Key: MARKER_FILE_NAME,
            }).promise();
            const marker = Number.parseInt(response.Body.toString('utf-8'), 10);
            console.log(`Read last transaction marker: ${marker}`);
            return marker;
        }
        catch (error) {
            if (error.code !== 'NoSuchKey') {
                throw error;
            }
            console.log(`Marker object (s3://${stagingBucket}/${MARKER_FILE_NAME}) does not exist, starting from the default (${defaultValue})`);
            return defaultValue;
        }
    }
    /**
     * Updates the last transaction marker in S3.
     *
     * @param sequence the last transaction marker value
     */
    async function saveLastTransactionMarker(sequence) {
        console.log(`Updating last transaction marker to ${sequence}`);
        return putObject(MARKER_FILE_NAME, sequence.toFixed(), { ContentType: 'text/plain' });
    }
    //#endregion
    //#region Business Logic
    async function processUpdatedVersion({ infos, modified, seq }) {
        var _a;
        try {
            // Download the tarball
            const tarball = await httpGet(infos.dist.tarball);
            // Store the tarball into the staging bucket
            // - infos.dist.tarball => https://registry.npmjs.org/<@scope>/<name>/-/<name>-<version>.tgz
            // - stagingKey         =>                     staged/<@scope>/<name>/-/<name>-<version>.tgz
            const stagingKey = `staged/${new url_1.URL(infos.dist.tarball).pathname}`.replace(/\/{2,}/g, '/');
            await putObject(stagingKey, tarball, {
                ContentType: 'application/x-gtar',
                Metadata: {
                    'Modified-At': modified.toISOString(),
                    'Origin-Integrity': infos.dist.shasum,
                    'Origin-URI': infos.dist.tarball,
                    'Sequence': seq.toFixed(),
                },
            });
            // Prepare SQS message for ingestion
            const messageBase = {
                tarballUri: `s3://${stagingBucket}/${stagingKey}`,
                metadata: {
                    dist: infos.dist.tarball,
                    seq: seq.toFixed(),
                },
                time: modified.toUTCString(),
            };
            const message = {
                ...messageBase,
                integrity: shared_1.integrity(messageBase, tarball),
            };
            // Send the SQS message out
            await shared_1.aws.sqs().sendMessage({
                MessageBody: JSON.stringify(message, null, 2),
                QueueUrl: queueUrl,
            }).promise();
        }
        catch (err) {
            // Something failed, store the payload in the problem prefix, and move on.
            console.error(`[${seq}] Failed processing ${infos.name}@${infos.version}: ${err}`);
            await putObject(`failed/${seq}`, JSON.stringify(infos, null, 2), {
                ContentType: 'text/json',
                Metadata: {
                    // User-defined metadata is limited to 2KB in size, in total. So we
                    // cap the error text to 1KB maximum, allowing up to 1KB for other
                    // attributes (which should be sufficient).
                    'Error': `${(_a = err.stack) !== null && _a !== void 0 ? _a : err}`.substring(0, 1024),
                    'Modified-At': modified.toISOString(),
                },
            });
        }
    }
    //#endregion
    //#region Asynchronous Primitives
    /**
     * Makes an HTTP GET request, and returns the resulting payload.
     *
     * @param url the URL to get.
     *
     * @returns a Buffer containing the received data.
     */
    function httpGet(url) {
        return new Promise((ok, ko) => {
            https.get(url, (response) => {
                if (response.statusCode !== 200) {
                    throw new Error(`Unsuccessful GET: ${response.statusCode} - ${response.statusMessage}`);
                }
                let body = Buffer.alloc(0);
                response.on('data', (chunk) => body = Buffer.concat([body, Buffer.from(chunk)]));
                response.once('close', () => ok(body));
                response.once('error', ko);
            });
        });
    }
    /**
     * Puts an object in the staging bucket, with standardized object metadata.
     *
     * @param key  the key for the object to be put.
     * @param body the body of the object to be put.
     * @param opts any other options to use when sending the S3 request.
     *
     * @returns the result of the S3 request.
     */
    function putObject(key, body, opts = {}) {
        return shared_1.aws.s3().putObject({
            Bucket: stagingBucket,
            Key: key,
            Body: body,
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
                ...opts.Metadata,
            },
            ...opts,
        }).promise();
    }
    //#endregion
}
exports.handler = handler;
/**
 * Obtains the `VersionInfo` corresponding to the modified version(s) in the
 * provided `Change` objects, ensures they are relevant (construct libraries),
 * and returns those only.
 *
 * @param changes the changes to be processed.
 *
 * @returns a list of `VersionInfo` objects
 */
function getRelevantVersionInfos(changes) {
    const result = new Array();
    for (const change of changes) {
        // Sometimes, there are no versions in the document. We skip those.
        if (Object.keys(change.doc.versions).length === 0) {
            console.error(`[${change.seq}] Changed document contains no 'versions': ${JSON.stringify(change, null, 2)}`);
            continue;
        }
        // Sometimes, there is no 'time' entry in the document. We skip those.
        if (change.doc.time == null) {
            console.error(`[${change.seq}] Changed document contains no 'time': ${JSON.stringify(change, null, 2)}`);
            continue;
        }
        // Get the last modification date from the change
        const sortedUpdates = Object.entries(change.doc.time)
            // Ignore the "created" and "modified" keys here
            .filter(([key]) => key !== 'created' && key !== 'modified')
            // Parse all the dates to ensure they are comparable
            .map(([version, isoDate]) => [version, new Date(isoDate)])
            // Sort by date, descending
            .sort(([, l], [, r]) => r.getTime() - l.getTime());
        let latestModified;
        for (const [version, modified] of sortedUpdates) {
            if (latestModified == null || latestModified.getTime() === modified.getTime()) {
                const infos = change.doc.versions[version];
                if (infos == null) {
                    // Could be the version in question was un-published.
                    console.log(`[${change.seq}] Could not find info for "${change.doc.name}@${version}". Was it un-published?`);
                }
                else if (isRelevantPackageVersion(infos)) {
                    result.push({ infos, modified, seq: change.seq });
                }
                else {
                    console.log(`[${change.seq}] Ignoring "${change.doc.name}@${version}" as it is not a construct library.`);
                }
                latestModified = modified;
            }
        }
    }
    return result;
    function isRelevantPackageVersion(infos) {
        var _a;
        if (infos.jsii == null) {
            return false;
        }
        return infos.name === 'construct'
            || infos.name === 'aws-cdk-lib'
            || infos.name.startsWith('@aws-cdk')
            || ((_a = infos.keywords) === null || _a === void 0 ? void 0 : _a.some((kw) => CONSTRUCT_KEYWORDS.has(kw)));
    }
}
//# sourceMappingURL=data:application/json;base64,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