"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.reset = exports.handler = exports.CATALOG_OBJECT_KEY = void 0;
const process_1 = require("process");
const zlib_1 = require("zlib");
const aws_sdk_1 = require("aws-sdk");
const semver_1 = require("semver");
const tar_stream_1 = require("tar-stream");
let s3;
exports.CATALOG_OBJECT_KEY = 'catalog.json';
const KEY_FORMAT_REGEX = /^packages\/((?:@[^/]+\/)?[^/]+)\/v([^/]+)\/.*$/;
// Capture groups:                   ┗━━━━━━━━1━━━━━━━━━┛   ┗━━2━━┛
/**
 * Regenerates the `catalog.json` object in the configured S3 bucket.
 *
 * @param event configuration for the rebuild job. In particular, the `rebuild`
 *              property can be set to `true` in order to trigger a full (i.e:
 *              non-incremental) rebuild of the object.
 * @param context the lambda context in which this execution runs.
 *
 * @returns the information about the updated S3 object.
 */
async function handler(event, context) {
    var _a;
    const BUCKET_NAME = requireEnv('BUCKET_NAME');
    const packages = new Map();
    if (!event.rebuild) {
        console.log('Loading existing catalog...');
        const data = await s3Client().getObject({ Bucket: BUCKET_NAME, Key: exports.CATALOG_OBJECT_KEY }).promise()
            .catch((err) => err.code !== 'NoSuchKey'
            ? Promise.reject(err)
            : Promise.resolve({ /* no data */}));
        if (data.Body != null) {
            const catalog = JSON.parse(data.Body.toString('utf-8'));
            for (const info of catalog.packages) {
                if (!packages.has(info.name)) {
                    packages.set(info.name, new Map());
                }
                packages.get(info.name).set(info.major, info);
            }
        }
    }
    else {
        console.log('Rebuild requested, ignoring existing catalog...');
    }
    console.log('Listing existing objects...');
    for await (const object of relevantObjects(BUCKET_NAME)) {
        // Ensure we don't already have a more recent version tracked for this package-major.
        const [, packageName, versionStr] = object.Key.match(KEY_FORMAT_REGEX);
        const version = new semver_1.SemVer(versionStr);
        const found = (_a = packages.get(packageName)) === null || _a === void 0 ? void 0 : _a.get(version.major);
        if (found != null && version.compare(found.version) <= 0) {
            console.log(`Skipping ${packageName}@${version} because it is not newer than the existing ${found.version}`);
            continue;
        }
        console.log(`Registering ${packageName}@${version}`);
        // Donwload the tarball to inspect the `package.json` data therein.
        const data = await s3Client().getObject({ Bucket: BUCKET_NAME, Key: object.Key }).promise();
        const manifest = await new Promise((ok, ko) => {
            zlib_1.gunzip(Buffer.from(data.Body), (err, tar) => {
                if (err) {
                    return ko(err);
                }
                tar_stream_1.extract()
                    .on('entry', (header, stream, next) => {
                    if (header.name !== 'package/package.json') {
                        // Not the file we are looking for, skip ahead (next run-loop tick).
                        return setImmediate(next);
                    }
                    const chunks = new Array();
                    return stream
                        .on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                        .once('end', () => {
                        ok(Buffer.concat(chunks));
                        next();
                    })
                        .resume();
                })
                    .once('finish', () => {
                    ko(new Error('Could not find package/package.json in tarball!'));
                })
                    .write(tar, (writeErr) => {
                    if (writeErr) {
                        ko(writeErr);
                    }
                });
            });
        });
        // Add the PackageInfo into the working set
        const metadata = JSON.parse(manifest.toString('utf-8'));
        const major = new semver_1.SemVer(metadata.version).major;
        if (!packages.has(metadata.name)) {
            packages.set(metadata.name, new Map());
        }
        packages.get(metadata.name).set(major, {
            author: metadata.author,
            description: metadata.description,
            keywords: metadata.keywords,
            languages: metadata.jsii.targets,
            license: metadata.license,
            major,
            metadata: metadata.jsii.metadata,
            name: metadata.name,
            time: metadata.time,
            version: metadata.version,
        });
    }
    // Build the final data package...
    console.log('Consolidating catalog...');
    const catalog = { packages: new Array(), updated: new Date().toISOString() };
    for (const majors of packages.values()) {
        for (const pkg of majors.values()) {
            catalog.packages.push(pkg);
        }
    }
    console.log(`Registered ${catalog.packages.length} package major versions`);
    // Upload the result to S3 and exit.
    return s3Client().putObject({
        Bucket: BUCKET_NAME,
        Key: exports.CATALOG_OBJECT_KEY,
        Body: JSON.stringify(catalog, null, 2),
        ContentType: 'text/json',
        Metadata: {
            'Build-Process': event.rebuild ? 'FROM_SCRATCH' : 'INCREMENTAL',
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
            'Package-Count': `${catalog.packages.length}`,
        },
    }).promise();
}
exports.handler = handler;
/**
 * A generator that asynchronously traverses the set of "interesting" objects
 * found by listing the configured S3 bucket. Those objects correspond to all
 * npm package tarballs present under the `packages/` prefix in the bucket.
 */
async function* relevantObjects(bucket) {
    var _a, _b;
    const request = { Bucket: bucket, Prefix: 'packages/' };
    do {
        const result = await s3.listObjectsV2(request).promise();
        for (const object of (_a = result.Contents) !== null && _a !== void 0 ? _a : []) {
            if (!((_b = object.Key) === null || _b === void 0 ? void 0 : _b.endsWith('/package.tgz'))) {
                continue;
            }
            yield object;
        }
        request.ContinuationToken = result.NextContinuationToken;
    } while (request.ContinuationToken != null);
}
/**
 * Reads the specified value from the environment of this process, and ensures a
 * value was provided.
 *
 * @param name the name of the environment variable to read.
 *
 * @returns the value of the environment variable, as-is.
 *
 * @throws if the environment variable had no value.
 */
function requireEnv(name) {
    const result = process_1.env[name];
    if (!result) {
        throw new Error(`Missing required environment variable: ${name}`);
    }
    return result;
}
/**
 * Visible for testing. This function ensures a new S3 client is used for the
 * next invocation.
 */
function reset() {
    s3 = undefined;
}
exports.reset = reset;
function s3Client() {
    if (s3 == null) {
        s3 = new aws_sdk_1.S3();
    }
    return s3;
}
//# sourceMappingURL=data:application/json;base64,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