"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const lambda = require("@aws-cdk/aws-lambda");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
const monitoring_1 = require("../monitoring");
const http_get_function_lambda_1 = require("../monitoring/http-get-function.lambda");
require("@aws-cdk/assert/jest");
const actions = {
    highSeverity: 'arn:aws:sns:us-east-1:123456789012:high',
    normalSeverity: 'arn:aws:sns:us-east-1:123456789012:normal',
};
test('minimal', () => {
    // GIVEN
    const stack = new core_1.Stack();
    // WHEN
    new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: actions,
        dashboardName: 'construct-hub',
    });
    // a dashboard is automatically created
    expect(stack).toHaveResource('AWS::CloudWatch::Dashboard');
});
test('watchful can be used for setting up automatic monitoring', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const fn = new lambda.Function(stack, 'Function', {
        runtime: lambda.Runtime.NODEJS_12_X,
        code: lambda.Code.fromInline('foo'),
        handler: 'index.handler',
    });
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: actions,
        dashboardName: 'construct-hub',
    });
    // WHEN
    monitoring.watchful.watchLambdaFunction('My Function', fn);
    // an alarm is automatically created for this function
    expect(stack).toHaveResource('AWS::CloudWatch::Alarm', {
        Namespace: 'AWS/Lambda',
        MetricName: 'Errors',
        //TODO: uncomment when we can use cdk-watchful 0.145.0 or above
        // AlarmActions: ['arn:aws:sns:us-east-1:123456789012:normal'],
        Dimensions: [{ Name: 'FunctionName', Value: { Ref: 'Function76856677' } }],
    });
});
test('high severity alarms trigger the correct action', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const topic = new sns.Topic(stack, 'Topic');
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: actions,
        dashboardName: 'construct-hub',
    });
    const alarm = topic.metricNumberOfNotificationsFailed().createAlarm(stack, 'Alarm', { threshold: 1, evaluationPeriods: 1 });
    // WHEN
    monitoring.addHighSeverityAlarm('My Alarm', alarm);
    // a dashboard is automatically created
    expect(stack).toHaveResource('AWS::CloudWatch::Alarm', {
        AlarmActions: ['arn:aws:sns:us-east-1:123456789012:high'],
        Dimensions: [{ Name: 'TopicName', Value: { 'Fn::GetAtt': ['TopicBFC7AF6E', 'TopicName'] } }],
    });
    expect(stack).toHaveResource('AWS::CloudWatch::Dashboard', {
        DashboardBody: {
            'Fn::Join': [
                '',
                [
                    '{"widgets":[{"type":"metric","width":24,"height":6,"x":0,"y":0,"properties":{"view":"timeSeries","title":"My Alarm","region":"',
                    { Ref: 'AWS::Region' },
                    '","annotations":{"alarms":["',
                    { 'Fn::GetAtt': ['Alarm7103F465', 'Arn'] },
                    '"]},"yAxis":{}}}]}',
                ],
            ],
        },
        DashboardName: 'construct-hub-high-severity',
    });
});
test('web canaries can ping URLs and raise high severity alarms', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: actions,
        dashboardName: 'construct-hub',
    });
    // WHEN
    monitoring.addWebCanary('Ping1', 'https://ping1');
    // THEN
    expect(stack).toHaveResource('AWS::CloudWatch::Alarm', {
        ComparisonOperator: 'GreaterThanOrEqualToThreshold',
        EvaluationPeriods: 1,
        AlarmActions: [
            'arn:aws:sns:us-east-1:123456789012:high',
        ],
        AlarmDescription: '80% error rate for https://ping1 (Ping1)',
        Metrics: [
            {
                Id: 'm1',
                Label: 'https://ping1 Errors',
                MetricStat: {
                    Metric: {
                        Dimensions: [
                            {
                                Name: 'FunctionName',
                                Value: {
                                    Ref: 'MonitoringWebCanaryPing1HttpGetFunction2403444A',
                                },
                            },
                        ],
                        MetricName: 'Errors',
                        Namespace: 'AWS/Lambda',
                    },
                    Period: 300,
                    Stat: 'Sum',
                },
                ReturnData: true,
            },
        ],
        Threshold: 4,
        TreatMissingData: 'breaching',
    });
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Environment: {
            Variables: {
                URL: 'https://ping1',
            },
        },
    });
});
describe('web canary handler', () => {
    test('web ping is successful', async () => {
        process.env.URL = 'https://amazon.com';
        await http_get_function_lambda_1.handler({});
    });
    test('web ping throws for a non-200 response', async () => {
        process.env.URL = 'https://amazon.com/not-found-please12345';
        await expect(http_get_function_lambda_1.handler({})).rejects.toThrow(/Response code 404 \(Not Found\)/);
    });
});
//# sourceMappingURL=data:application/json;base64,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