"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto_1 = require("crypto");
const events_1 = require("events");
const fs_1 = require("fs");
const path = require("path");
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const jsii_rosetta_1 = require("jsii-rosetta");
const transliterator_lambda_1 = require("../../../backend/transliterator/transliterator.lambda");
jest.mock('child_process');
jest.mock('jsii-rosetta/lib/commands/transliterate');
beforeEach((done) => {
    AWSMock.setSDKInstance(AWS);
    done();
});
afterEach((done) => {
    AWSMock.restore();
    transliterator_lambda_1.reset();
    done();
});
test('scoped package', async () => {
    // GIVEN
    const packageScope = 'scope';
    const packageName = 'package-name';
    const packageVersion = '1.2.3-dev.4';
    const payload = {
        Records: [{
                awsRegion: 'bemuda-triangle-1',
                eventVersion: '1337',
                eventSource: 's3:DummySource',
                eventName: 's3:DummyEvent',
                eventTime: '1789-07-14T00:00:00+02:00',
                userIdentity: { principalId: 'aws::principal::id' },
                requestParameters: { sourceIPAddress: '127.0.0.1' },
                responseElements: {
                    'x-amz-id-2': '456',
                    'x-amz-request-id': '123',
                },
                s3: {
                    bucket: {
                        name: 'dummy-bucket',
                        arn: 'arn:aws:s3:::dummy-bucket',
                        ownerIdentity: { principalId: 'aws::principal::id' },
                    },
                    configurationId: '42',
                    object: {
                        eTag: 'eTag',
                        key: `packages/%40${packageScope}/${packageName}/v${packageVersion}/package.tgz`,
                        sequencer: 'Seq',
                        size: 1337,
                        versionId: 'VersionId',
                    },
                    s3SchemaVersion: '1',
                },
            }],
    };
    const mockContext = {};
    const mockTarballBytes = crypto_1.randomBytes(128);
    const mockOutputAssembly = crypto_1.randomBytes(128);
    AWSMock.mock('S3', 'getObject', (request, callback) => {
        try {
            expect(request.Bucket).toBe(payload.Records[0].s3.bucket.name);
            expect(request.Key).toBe(payload.Records[0].s3.object.key.split('/').map((comp) => decodeURIComponent(comp)).join('/'));
            expect(request.VersionId).toBe(payload.Records[0].s3.object.versionId);
        }
        catch (e) {
            callback(e);
        }
        callback(null, {
            Body: mockTarballBytes,
        });
    });
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockSpawn = require('child_process').spawn;
    mockSpawn.mockImplementation((cmd, args, opts) => {
        expect(cmd).toBe('npm');
        expect(args).toContain('install');
        expect(args).toContain('--ignore-scripts'); // Ensures lifecycle hooks don't run
        expect(args).toContain('--no-bin-links'); // Ensures we don't attempt to add bin-links to $PATH
        expect(opts.cwd).toBeDefined();
        expect(opts.stdio).toEqual(['ignore', 'inherit', 'inherit']);
        const tarballPath = args[args.length - 1];
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        expect(require('fs').readFileSync(tarballPath)).toEqual(mockTarballBytes);
        return new MockChildProcess(cmd, Array.from(args), fs_1.promises.mkdir(path.join(opts.cwd, 'node_modules', `@${packageScope}`, packageName), { recursive: true }));
    });
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockTransliterateAssembly = require('jsii-rosetta/lib/commands/transliterate').transliterateAssembly;
    mockTransliterateAssembly.mockImplementation(async ([dir, ...otherDirs], languages) => {
        expect(languages).toEqual([jsii_rosetta_1.TargetLanguage.PYTHON]);
        expect(dir).toMatch(new RegExp(path.join('', 'node_modules', `@${packageScope}`, packageName) + '$'));
        expect(otherDirs).toEqual([]);
        return fs_1.promises.writeFile(path.resolve(dir, '.jsii.python'), mockOutputAssembly);
    });
    const key = payload.Records[0].s3.object.key
        // The key is not URI-encoded when it's sent to the S3 SDK...
        .split('/').map((comp) => decodeURIComponent(comp)).join('/')
        .replace(/\/package\.tgz$/, '/assembly-python.json');
    AWSMock.mock('S3', 'putObject', (request, callback) => {
        try {
            expect(request.Bucket).toBe(payload.Records[0].s3.bucket.name);
            expect(request.Key).toBe(key);
            expect(request.Body).toEqual(mockOutputAssembly);
        }
        catch (e) {
            return callback(e);
        }
        callback(null, { VersionId: 'New-VersionID' });
    });
    // WHEN
    const result = transliterator_lambda_1.handler(payload, mockContext);
    // THEN
    await expect(result).resolves.toEqual([{ bucket: payload.Records[0].s3.bucket.name, key, versionId: 'New-VersionID' }]);
    expect(mockSpawn).toHaveBeenCalled();
    expect(mockTransliterateAssembly).toHaveBeenCalled();
});
test('unscoped package', async () => {
    // GIVEN
    const packageName = 'package-name';
    const packageVersion = '1.2.3-dev.4';
    const payload = {
        Records: [{
                awsRegion: 'bemuda-triangle-1',
                eventVersion: '1337',
                eventSource: 's3:DummySource',
                eventName: 's3:DummyEvent',
                eventTime: '1789-07-14T00:00:00+02:00',
                userIdentity: { principalId: 'aws::principal::id' },
                requestParameters: { sourceIPAddress: '127.0.0.1' },
                responseElements: {
                    'x-amz-id-2': '456',
                    'x-amz-request-id': '123',
                },
                s3: {
                    bucket: {
                        name: 'dummy-bucket',
                        arn: 'arn:aws:s3:::dummy-bucket',
                        ownerIdentity: { principalId: 'aws::principal::id' },
                    },
                    configurationId: '42',
                    object: {
                        eTag: 'eTag',
                        key: `packages/${packageName}/v${packageVersion}/package.tgz`,
                        sequencer: 'Seq',
                        size: 1337,
                        versionId: 'VersionId',
                    },
                    s3SchemaVersion: '1',
                },
            }],
    };
    const mockContext = {};
    const mockTarballBytes = crypto_1.randomBytes(128);
    const mockOutputAssembly = crypto_1.randomBytes(128);
    AWSMock.mock('S3', 'getObject', (request, callback) => {
        try {
            expect(request.Bucket).toBe(payload.Records[0].s3.bucket.name);
            expect(request.Key).toBe(payload.Records[0].s3.object.key);
            expect(request.VersionId).toBe(payload.Records[0].s3.object.versionId);
        }
        catch (e) {
            callback(e);
        }
        callback(null, {
            Body: mockTarballBytes,
        });
    });
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockSpawn = require('child_process').spawn;
    mockSpawn.mockImplementation((cmd, args, opts) => {
        expect(cmd).toBe('npm');
        expect(args).toContain('install');
        expect(args).toContain('--ignore-scripts'); // Ensures lifecycle hooks don't run
        expect(args).toContain('--no-bin-links'); // Ensures we don't attempt to add bin-links to $PATH
        expect(opts.cwd).toBeDefined();
        expect(opts.stdio).toEqual(['ignore', 'inherit', 'inherit']);
        const tarballPath = args[args.length - 1];
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        expect(require('fs').readFileSync(tarballPath)).toEqual(mockTarballBytes);
        return new MockChildProcess(cmd, Array.from(args), fs_1.promises.mkdir(path.join(opts.cwd, 'node_modules', packageName), { recursive: true }));
    });
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockTransliterateAssembly = require('jsii-rosetta/lib/commands/transliterate').transliterateAssembly;
    mockTransliterateAssembly.mockImplementation(async ([dir, ...otherDirs], languages) => {
        expect(languages).toEqual([jsii_rosetta_1.TargetLanguage.PYTHON]);
        expect(dir).toMatch(new RegExp(path.join('', 'node_modules', packageName) + '$'));
        expect(otherDirs).toEqual([]);
        return fs_1.promises.writeFile(path.resolve(dir, '.jsii.python'), mockOutputAssembly);
    });
    const key = payload.Records[0].s3.object.key.replace(/\/package\.tgz$/, '/assembly-python.json');
    AWSMock.mock('S3', 'putObject', (request, callback) => {
        try {
            expect(request.Bucket).toBe(payload.Records[0].s3.bucket.name);
            expect(request.Key).toBe(key);
            expect(request.Body).toEqual(mockOutputAssembly);
        }
        catch (e) {
            return callback(e);
        }
        callback(null, { VersionId: 'New-VersionID' });
    });
    // WHEN
    const result = transliterator_lambda_1.handler(payload, mockContext);
    // THEN
    await expect(result).resolves.toEqual([{ bucket: payload.Records[0].s3.bucket.name, key, versionId: 'New-VersionID' }]);
    expect(mockSpawn).toHaveBeenCalled();
    expect(mockTransliterateAssembly).toHaveBeenCalled();
});
class MockChildProcess extends events_1.EventEmitter {
    constructor(spawnfile, spawnargs, promise) {
        super();
        this.spawnfile = spawnfile;
        this.spawnargs = spawnargs;
        this.stdin = null;
        this.stdout = null;
        this.stderr = null;
        this.stdio = [this.stdin, this.stdout, this.stderr, null, null];
        this.exitCode = 0;
        this.killed = false;
        this.signalCode = null;
        this.pid = -1; // Obviously fake
        promise.then(() => this.emit('close', this.exitCode, this.signalCode), (err) => this.emit('error', err));
    }
    get connected() {
        throw new Error('Not Implemented');
    }
    disconnect() {
        throw new Error('Not Implemented');
    }
    kill() {
        throw new Error('Not Implemented');
    }
    ref() {
        throw new Error('Not Implemented');
    }
    send() {
        throw new Error('Not Implemented');
    }
    unref() {
        throw new Error('Not Implemented');
    }
}
//# sourceMappingURL=data:application/json;base64,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