"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto_1 = require("crypto");
const stream_1 = require("stream");
const zip = require("zlib");
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const tar = require("tar-stream");
const catalog_builder_lambda_1 = require("../../../backend/catalog-builder/catalog-builder.lambda");
let mockBucketName;
beforeEach((done) => {
    process.env.BUCKET_NAME = mockBucketName = crypto_1.randomBytes(16).toString('base64');
    AWSMock.setSDKInstance(AWS);
    done();
});
afterEach((done) => {
    AWSMock.restore();
    catalog_builder_lambda_1.reset();
    process.env.BUCKET_NAME = mockBucketName = undefined;
    done();
});
test('no indexed packages', () => {
    // GIVEN
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Key).toBe(catalog_builder_lambda_1.CATALOG_OBJECT_KEY);
        }
        catch (e) {
            return cb(e);
        }
        return cb(new NoSuchKeyError());
    });
    AWSMock.mock('S3', 'listObjectsV2', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Prefix).toBe('packages/');
            expect(req.ContinuationToken).toBeUndefined();
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, {});
    });
    const mockPutObjectResult = {};
    AWSMock.mock('S3', 'putObject', (req, cb) => {
        var _a, _b;
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Key).toBe(catalog_builder_lambda_1.CATALOG_OBJECT_KEY);
            expect(req.ContentType).toBe('text/json');
            const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
            expect(body.packages).toEqual([]);
            expect(Date.parse(body.updatedAt)).toBeDefined();
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, mockPutObjectResult);
    });
    // WHEN
    const result = catalog_builder_lambda_1.handler({}, { /* context */});
    // THEN
    return expect(result).resolves.toBe(mockPutObjectResult);
});
test('initial build', () => {
    // GIVEN
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
        }
        catch (e) {
            return cb(e);
        }
        const matches = /^packages\/((?:@[^/]+\/)?[^/]+)\/v([^/]+)\/.*$/.exec(req.Key);
        if (matches != null) {
            mockNpmPackage(matches[1], matches[2]).then((pack) => cb(null, { Body: pack }), cb);
        }
        else {
            return cb(new NoSuchKeyError());
        }
    });
    const mockFirstPage = [
        { Key: 'packages/@scope/package/v1.2.3/assembly.json' },
        { Key: 'packages/@scope/package/v1.2.3/package.tgz' },
        { Key: 'packages/name/v1.2.3/assembly.json' },
        { Key: 'packages/name/v1.2.3/package.tgz' },
    ];
    const mockSecondPage = [
        { Key: 'packages/@scope/package/v1.0.0/assembly.json' },
        { Key: 'packages/@scope/package/v1.0.0/package.tgz' },
        { Key: 'packages/name/v2.0.0-pre/assembly.json' },
        { Key: 'packages/name/v2.0.0-pre/package.tgz' },
    ];
    AWSMock.mock('S3', 'listObjectsV2', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Prefix).toBe('packages/');
        }
        catch (e) {
            return cb(e);
        }
        if (req.ContinuationToken == null) {
            return cb(null, { Contents: mockFirstPage, NextContinuationToken: 'next' });
        }
        try {
            expect(req.ContinuationToken).toBe('next');
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, { Contents: mockSecondPage });
    });
    const mockPutObjectResult = {};
    AWSMock.mock('S3', 'putObject', (req, cb) => {
        var _a, _b;
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Key).toBe(catalog_builder_lambda_1.CATALOG_OBJECT_KEY);
            expect(req.ContentType).toBe('text/json');
            expect(req.Metadata).toHaveProperty('Package-Count', '3');
            const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
            expect(body.packages).toEqual([
                {
                    description: 'Package @scope/package, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    name: '@scope/package',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    name: 'name',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 2.0.0-pre',
                    languages: { foo: 'bar' },
                    major: 2,
                    name: 'name',
                    version: '2.0.0-pre',
                },
            ]);
            expect(Date.parse(body.updatedAt)).toBeDefined();
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, mockPutObjectResult);
    });
    // WHEN
    const result = catalog_builder_lambda_1.handler({}, { /* context */});
    // THEN
    return expect(result).resolves.toBe(mockPutObjectResult);
});
test('incremental build', () => {
    // GIVEN
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
        }
        catch (e) {
            return cb(e);
        }
        const matches = /^packages\/((?:@[^/]+\/)?[^/]+)\/v([^/]+)\/.*$/.exec(req.Key);
        if (matches != null) {
            mockNpmPackage(matches[1], matches[2]).then((pack) => cb(null, { Body: pack }), cb);
        }
        else if (req.Key === catalog_builder_lambda_1.CATALOG_OBJECT_KEY) {
            return cb(null, {
                Body: JSON.stringify({
                    packages: [
                        {
                            description: 'Package @scope/package, version 2.3.4',
                            languages: { foo: 'bar' },
                            major: 2,
                            name: '@scope/package',
                            version: '2.3.4',
                        },
                        {
                            description: 'Package name, version 1.0.0',
                            languages: { foo: 'bar' },
                            major: 1,
                            name: 'name',
                            version: '1.0.0',
                        },
                        {
                            description: 'Package name, version 2.0.0-pre.10',
                            languages: { foo: 'bar' },
                            major: 2,
                            name: 'name',
                            version: '2.0.0-pre.10',
                        },
                    ],
                    updatedAt: new Date().toISOString(),
                }, null, 2),
            });
        }
        else {
            return cb(new NoSuchKeyError());
        }
    });
    const mockFirstPage = [
        { Key: 'packages/@scope/package/v1.2.3/package.tgz' },
        { Key: 'packages/name/v1.2.3/package.tgz' },
    ];
    const mockSecondPage = [
        { Key: 'packages/@scope/package/v2.0.5/package.tgz' },
        { Key: 'packages/name/v2.0.0-pre.1/package.tgz' },
    ];
    AWSMock.mock('S3', 'listObjectsV2', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Prefix).toBe('packages/');
        }
        catch (e) {
            return cb(e);
        }
        if (req.ContinuationToken == null) {
            return cb(null, { Contents: mockFirstPage, NextContinuationToken: 'next' });
        }
        try {
            expect(req.ContinuationToken).toBe('next');
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, { Contents: mockSecondPage });
    });
    const mockPutObjectResult = {};
    AWSMock.mock('S3', 'putObject', (req, cb) => {
        var _a, _b;
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Key).toBe(catalog_builder_lambda_1.CATALOG_OBJECT_KEY);
            expect(req.ContentType).toBe('text/json');
            expect(req.Metadata).toHaveProperty('Package-Count', '4');
            const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
            expect(body.packages).toEqual([
                {
                    description: 'Package @scope/package, version 2.3.4',
                    languages: { foo: 'bar' },
                    major: 2,
                    name: '@scope/package',
                    version: '2.3.4',
                },
                {
                    description: 'Package @scope/package, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    name: '@scope/package',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    name: 'name',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 2.0.0-pre.10',
                    languages: { foo: 'bar' },
                    major: 2,
                    name: 'name',
                    version: '2.0.0-pre.10',
                },
            ]);
            expect(Date.parse(body.updatedAt)).toBeDefined();
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, mockPutObjectResult);
    });
    // WHEN
    const result = catalog_builder_lambda_1.handler({}, { /* context */});
    // THEN
    return expect(result).resolves.toBe(mockPutObjectResult);
});
class NoSuchKeyError extends Error {
    constructor() {
        super(...arguments);
        this.code = 'NoSuchKey';
        this.time = new Date();
    }
}
function mockNpmPackage(name, version) {
    const packageJson = {
        name,
        version,
        description: `Package ${name}, version ${version}`,
        jsii: {
            targets: { foo: 'bar' },
        },
    };
    const tarball = tar.pack();
    tarball.entry({ name: 'package/ignore-me.txt' }, 'Ignore Me!');
    tarball.entry({ name: 'package/package.json' }, JSON.stringify(packageJson, null, 2));
    tarball.finalize();
    const gzip = zip.createGzip();
    tarball.pipe(gzip);
    const passthrough = new stream_1.PassThrough();
    gzip.pipe(passthrough);
    return new Promise((ok) => {
        const chunks = new Array();
        passthrough.on('data', (chunk) => chunks.push(Buffer.from(chunk)));
        passthrough.once('end', () => {
            ok(Buffer.concat(chunks));
        });
    });
}
//# sourceMappingURL=data:application/json;base64,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