import terrascript.core as core


@core.resource(type="aws_elastic_beanstalk_application_version", namespace="elastic_beanstalk")
class ApplicationVersion(core.Resource):
    """
    (Required) Name of the Beanstalk Application the version is associated with.
    """

    application: str | core.StringOut = core.attr(str)

    """
    ARN assigned by AWS for this Elastic Beanstalk Application.
    """
    arn: str | core.StringOut = core.attr(str, computed=True)

    """
    (Required) S3 bucket that contains the Application Version source bundle.
    """
    bucket: str | core.StringOut = core.attr(str)

    """
    (Optional) Short description of the Application Version.
    """
    description: str | core.StringOut | None = core.attr(str, default=None)

    """
    (Optional) On delete, force an Application Version to be deleted when it may be in use by multiple E
    lastic Beanstalk Environments.
    """
    force_delete: bool | core.BoolOut | None = core.attr(bool, default=None)

    id: str | core.StringOut = core.attr(str, computed=True)

    """
    (Required) S3 object that is the Application Version source bundle.
    """
    key: str | core.StringOut = core.attr(str)

    """
    (Required) Unique name for the this Application Version.
    """
    name: str | core.StringOut = core.attr(str)

    """
    (Optional) Key-value map of tags for the Elastic Beanstalk Application Version. If configured with a
    provider [`default_tags` configuration block](https://registry.terraform.io/providers/hashicorp/aws
    /latest/docs#default_tags-configuration-block) present, tags with matching keys will overwrite those
    defined at the provider-level.
    """
    tags: dict[str, str] | core.MapOut[core.StringOut] | None = core.attr(
        str, default=None, kind=core.Kind.map
    )

    """
    Map of tags assigned to the resource, including those inherited from the provider [`default_tags` co
    nfiguration block](https://registry.terraform.io/providers/hashicorp/aws/latest/docs#default_tags-co
    nfiguration-block).
    """
    tags_all: dict[str, str] | core.MapOut[core.StringOut] | None = core.attr(
        str, default=None, computed=True, kind=core.Kind.map
    )

    def __init__(
        self,
        resource_name: str,
        *,
        application: str | core.StringOut,
        bucket: str | core.StringOut,
        key: str | core.StringOut,
        name: str | core.StringOut,
        description: str | core.StringOut | None = None,
        force_delete: bool | core.BoolOut | None = None,
        tags: dict[str, str] | core.MapOut[core.StringOut] | None = None,
        tags_all: dict[str, str] | core.MapOut[core.StringOut] | None = None,
        depends_on: list[str] | core.ArrayOut[core.StringOut] | None = None,
        provider: str | core.StringOut | None = None,
        lifecycle: core.Lifecycle | None = None,
    ):
        super().__init__(
            name=resource_name,
            args=ApplicationVersion.Args(
                application=application,
                bucket=bucket,
                key=key,
                name=name,
                description=description,
                force_delete=force_delete,
                tags=tags,
                tags_all=tags_all,
                depends_on=depends_on,
                provider=provider,
                lifecycle=lifecycle,
            ),
        )

    @core.schema_args
    class Args(core.Resource.Args):
        application: str | core.StringOut = core.arg()

        bucket: str | core.StringOut = core.arg()

        description: str | core.StringOut | None = core.arg(default=None)

        force_delete: bool | core.BoolOut | None = core.arg(default=None)

        key: str | core.StringOut = core.arg()

        name: str | core.StringOut = core.arg()

        tags: dict[str, str] | core.MapOut[core.StringOut] | None = core.arg(default=None)

        tags_all: dict[str, str] | core.MapOut[core.StringOut] | None = core.arg(default=None)
