import os
import py_compile
import sys
import shutil
import tempfile
from ..scaffoldy import CodeProcessor


def scaffold_file(file_path, global_variables):
    destination_dir = os.path.dirname(file_path)
    destination_path = os.path.join(destination_dir, os.path.basename(file_path))
    os.makedirs(destination_dir, exist_ok=True)

    with open(file_path, 'r') as f:
        content = f.read()

    with open(destination_path, 'w') as f:
        f.write(CodeProcessor.process_code(content, global_variables))

    return destination_path, file_path, content


def compile_file(path: str, output_path: str, scaffoldy=True, global_variables=None):
    if global_variables is None:
        global_variables = {}

    if scaffoldy:
        path, old_content = scaffold_file(path, global_variables)

    if os.path.isdir(output_path) and not os.path.isfile(output_path):
        filename = os.path.basename(path)
        output_path = os.path.join(output_path, filename)

    pyc_path = f'{output_path}c'
    py_compile.compile(path, cfile=pyc_path, optimize=0)
    print(f"Compiled file: {path} to {pyc_path}")


def compile_directory(path: str, output_dir: str, rename_init: str = '__init__.py', scaffoldy=True, global_variables=None):
    os.makedirs(output_dir, exist_ok=True)

    if global_variables is None:
        global_variables = {}

    for root, _, files in os.walk(path):
        for file in files:
            file_path = os.path.join(root, file)

            if scaffoldy:
                file_path, old_path, old_content = scaffold_file(file_path, global_variables)

            rel_path = os.path.relpath(file_path, path)
            output_file = os.path.join(output_dir, rel_path)

            os.makedirs(os.path.dirname(output_file), exist_ok=True)

            if file.endswith('.py'):
                output_file += 'c'

                if file == '__init__.py':
                    output_file = output_file.replace('__init__.py', 'second_init.py')

                py_compile.compile(file_path, cfile=output_file, optimize=0)
                print(f"Compiled file: {file_path} to {output_file}")
            else:
                shutil.copy(file_path, output_file)
                print(f"Copied file: {file_path} to {output_file}")

            if scaffoldy:
                with open(old_path, 'w') as f:
                    f.write(old_content)

def compile(path: str,
            output_dir: str = 'compiled_files',  # Directory name to store compiled package
            output_file: str = 'compiled_file.py',  # File name to store compiled python file
            rename_init: str = '__init__.py',
            syspath: str = None,
            scaffoldy: bool = True,
            global_variables: dict | None = None):

    if global_variables is None:
        global_variables = {}

    if syspath:
        sys.path.insert(1, syspath)

    if os.path.isfile(path):
        compile_file(path, output_dir, scaffoldy=scaffoldy, global_variables=global_variables)
    elif os.path.isdir(path):
        compile_directory(path, output_dir, rename_init=rename_init, scaffoldy=scaffoldy, global_variables=global_variables)
    else:
        raise Exception(f"The path {path} is neither a file nor a directory or does not exist.")
