import sys
import subprocess
import argparse
import ast
import numpy as np
from math import log10, floor
from itertools import groupby
from collections import defaultdict
import matplotlib.cm
import matplotlib.pyplot as plt


def input_parser():

    description_text = ("Runs a bedfile simulator to test muMerge.")
    
    parser = argparse.ArgumentParser(description=description_text)

    parser.add_argument(
        '-f',
        type=str,
        help=("Output file basename (full path, sans extension). WARNING: will "
            "overwrite any existing file)"),
        required=True
    )
    parser.add_argument(
        '-o',
        type=int,
        metavar='+int',
        help=('The number of bases by which to offset the experiment. This '
            'must be a positive integer.'),
        default=0
    )
    parser.add_argument(
        '-r',
        type=int,
        metavar='+int',
        help='The number of replicates per condition',
        default=2
    )
    parser.add_argument(
        '-N',
        type=int,
        metavar='+int',
        help=("The number of iid simulations for the specified set of mu, "
        "replicates, and conditions."),
        default=10
    )
    parser.add_argument(
        '-w',
        type=float,
        metavar='float',
        help=("width ratio (sigma:1/2-width bed region) for converting "
        "between bed region widths and probability distribution sigma. Input "
        "flag for muMerge ('-w')."),
        default=1.0
    )
    parser.add_argument(
        '-a',
        metavar='m n p l',
        action='append',
        type=float,
        nargs=4,
        help=("Parameters for a single site in condition A. Parameter order "
        "is mu, n, p, and l. May specify multiple sites with mutiple '-a' "
        "flags.")
    )
    parser.add_argument(
        '-b',
        metavar='m n p l',
        action='append',
        type=float,
        nargs=4,
        help=("Parameters for a single site in condition B. Parameter order "
        "is mu, n, p, and l. May specify multiple sites with mutiple '-b' "
        "flags.")
    )
    parser.add_argument(
        '-c',
        metavar='m n p l',
        action='append',
        type=float,
        nargs=4,
        help=("Parameters for a single site in condition C. Parameter order "
        "is mu, n, p, and l. May specify multiple sites with mutiple '-c' "
        "flags.")
    )

    args = parser.parse_args()
    
    # Cast mu, n and l as integers
    if args.a != None:
        a = [(int(m[0]), int(m[1]), m[2], int(m[3])) for m in args.a]
    else:
        a = None

    if args.b != None:
        b = [(int(m[0]), int(m[1]), m[2], int(m[3])) for m in args.b]
    else:
        b = None

    if args.c != None:
        c = [(int(m[0]), int(m[1]), m[2], int(m[3])) for m in args.c]
    else:
        c = None

    params = {
        'basename': args.f,
        'offset': args.o,
        'reps': args.r,
        'N': args.N,
        'width': args.w,
        'A': a,
        'B': b,
        'C': c
    }

    return params


###############################################################################
def mu_generator(npl = (10000, 0.5, 100), rep=2,cond=1,N=100):
    '''
    Input: distribution parameters (n, p, l), # of replicates, # of conditions, 
    # of simulations
    Output: dictionary of simulated sites in the same format as that generated 
    by tfit_dict(). key = simulation id, value = dict of sites

    Simulates tfit calls for a given number of experimental conditions and 
    replicates per condition. Number of simulations is specified by N. 
    
    The mu value is pulled from a binomial distribution (parameters n, p) and 
    the tfit call width is pulled from a poisson distribution (parameter l) 
    
    The output dictionary is in the same format as that output by tfit_dict() 
    so that the output of mu_generator() can be input into mumerge for the 
    purpose of benchmarking its results.

    i.e. {T01: {(reg_start, reg_stop): [(start1, stop1, cov, id), ...]}, ...}

    Note: Instead of representing chromosomes, the top-level key in the output 
    represent the simulation. This way the lower level "region" keys do not
    conflict with one another.

    NOTE: THIS IS DEPRICATED. I DON'T ACTUALLY USE IT IN THE PIPELINE.
    '''
    #n_sites = int(rep * cond * N)
    n, p, l = npl
    print("mu +/- stdev: 0 +/-", np.sqrt(n*p*(1-p)))
    print("width mean (and variance):", l) 
    mu_dict = defaultdict(dict)
    
    order = floor(log10(N))

    # Initialize dictionary in same format as that generated by tfit_dict()
    for i in range(N):
        sites = []
        for r in range(rep):
            for c in range(cond):
                mu = np.random.binomial(n, p) - int(n/2)
                sig = np.random.poisson(l)
                sampid = '_'.join([str(c), str(r)])
                sites.append((mu-sig, mu+sig, 0, sampid))
        
        start, stop, cov, ids = zip(*sites)
        region = (int(min(start)), int(max(stop)))

        zi = str(i).zfill(order+1)
        mu_dict[''.join(['T', zi])] = {region: sites}
    
    return dict(mu_dict)


###############################################################################
def bedfile_generator(
    filename, 
    mu_npl=[(0, 10000, 0.5, 100)], 
    N=100, 
    offset=0):
    '''
    Inputs: bedfile name (complete path string), list of mu positions and 
    their binomial and poisson parameters (mu_npl), number of simulations (N)
    Outputs: Generates bedfile and returns 0

    This is an alternative to mu_generator. It produces equivalent sets of 
    sites, but writes them to an actual bed file instead. These can then be 
    input into mumerge as standard inputs.

    Note: The `offset` variable is there to prevent the resulting bedfiles 
    from containing negative intergers, in the event that these will be input 
    into bedtools (which does not permit negative integers)

    mu_npl = [(mu1, n1, p1, l1), ...]

    '''

    order = floor(log10(N))
    mu_npl = sorted(mu_npl, key=lambda x: x[0])

    with open(filename, 'w') as f:

        for i in range(N):
            # Define test# string (replacement for chromosome name)
            zi = str(i+1).zfill(order+1)
            tnum = ''.join(['T', zi])

            for site in mu_npl:
                    mu = site[0]
                    n = site[1]
                    p = site[2]
                    l = site[3]

                    pos = np.random.binomial(n, p) - int(n/2) + mu + offset
                    sig = np.random.poisson(l)

                    start = str(pos-sig)
                    stop = str(pos+sig)

                    bedline = "\t".join([tnum, start, stop])+"\n"
                    f.write(bedline)
        
    return 0


###############################################################################
def exp_simulator(
    basename, 
    mu_npl=[[(0, 10000, 0.5, 100)]],
    rep=2, 
    N=100, 
    offset=10000):
    '''
    This function simulates the entire set of bedfiles corresponding to the 
    given number of conditions and replicates. It also generates the 
    experimental info file that's required as an input to mumerge. This file 
    has extension '.input'.
    '''

    print("Your bedfiles will have a {} base offset.\n".format(str(offset)))

    # Check that BinomialVar + PoissonVar < offset + min(mus)
    #cummvar = npl[0] * npl[1] * (1-npl[1]) + npl[2]

    #if offset < cummvar - min(mus):
    #    print("Warning: bedtools does not allow for negative integers.\n"
    #        "You may need to increase the offset to avoid some simulated\n"
    #        "bed regions from ending up negative. Check your results.\n")    

    Ncond = len(mu_npl)

    with open(basename+'.input', 'w') as finput, \
        open(basename+'.info', 'w') as finfo:
        
        # write header for input file
        finput.write('#file\tsampid\tgroup\n')

        # write header for info file
        finfo.write("Simulation:\n")
        finfo.write("Basename: {}\n".format(basename))
        finfo.write("Mu values are offset by: {}\n".format(str(offset)))
        finfo.write("Number cond, rep: {}, {}\n".format(str(Ncond), str(rep)))
        finfo.write("Sampling params (mu, n, p, l): {}\n".format(str(mu_npl)))
        finfo.write("Number of simulations (N): {}\n".format(str(N)))
        finfo.write("muMerge input file: {}\n".format(basename+'.input'))
        finfo.write("\n")
        finfo.write("Bedfiles:\n")

        for c, cond in enumerate(mu_npl):
            for r in range(rep):

                # sample string for bedfile name and sample id for input file
                samp = '_'.join(['c'+str(c), 'r'+str(r), 'N'+str(N)])
                sampid = '_'.join([str(c), str(r)])

                bedfile = basename + '_' + samp + '.bed'

                # Generate bedfile (function defined above)
                bedfile_generator(
                    bedfile, 
                    mu_npl=cond,
                    N=N, 
                    offset=offset
                )

                # write sample info to input file
                line = '\t'.join([bedfile, sampid, str(c)])
                finput.write(line+'\n')

                finfo.write(bedfile+'\n')
        
    return 0

###############################################################################
def test_groupings(mu_dict):
    '''
    Input: dictionary of bed region tuples (start, stop, cov, id), generated 
    by mu_generator()
    Output: List of tuples, each of which contains all the sample IDs for a 
    group

    Extracts all sample IDs from the site dictionary generated by 
    mu_generator() and groups them 

    Each id must be of the form '#_#' where the first integer indicates the 
    group number and the second integer indicates the replicate number

    e.g. [(1, 2, 0, '0_0'), (2, 4, 0, '0_1'), (1, 8, 0, '1_0')] --> 
    [('0_0', '0_1'), ('1_0')]
    '''
    sampid = set()
    for region in mu_dict.values():
        for sites in region.values():
            tmp = set([m[-1] for m in sites])
            sampid = sampid.union(tmp)
    
    sampid = sorted(list(sampid))

    groups = groupby(sampid, key = lambda s: s.split('_')[0])
    groups = [sorted(list(g)) for k, g in groups]

    return sorted(groups)
###############################################################################
def list2tups(int_list):
    '''
    Input: sorted list of unique integers
    Output: sorted list of integer tuples

    Converts a list of sorted integers to a list of tuples (start, stop) 
    indicating the values for contiguous series of integers.

    e.g. [2,3,4,5,6,9,10,11] --> [(2,6), (9,10)]
    '''
    tup_list = []
    diffs = [(i-e, e) for i, e in enumerate(int_list)]

    index, ints = zip(*diffs)

    for i in set(index):
        group = [e[1] for e in diffs if e[0] == i]
        tup_list.append((min(group), max(group)))

    return sorted(tup_list)

###############################################################################
def tup_union(sites):
    '''
    Input: List of tuples (start, stop, cov, id) indicating bed regions
    Output: List of tuples (start, stop, 0, 'union') indicating the union of 
    the input tuples

    Calculates a list of tuples, in the same format as the input tuples, which 
    are the union of the input list.

    e.g. [(1, 10, 0, 'id1), (8, 20, 0, 'id2')] --> [(1, 20, 0, 'union')]
    '''
    union = []
    for site in sites:
        union.extend(list(range(site[0], site[1]+1)))

    union = sorted(list(set(union)))

    if union != []:
        final_tups = list2tups(union)
    else:
        return [(0, 0, 0, 'union')]

    starts, stops = list(zip(*final_tups))
    covs = [0 for i in range(len(final_tups))]
    ids = ['union' for i in range(len(final_tups))]

    final_tups = list(zip(starts, stops, covs, ids))

    return final_tups


###############################################################################
def tup_intersect(sites):
    '''
    Input: List of tuples (start, stop, cov, id) indicating bed regions
    Output: List of tuples (start, stop, 0, 'intersect') indicating the union 
    of the input tuples

    Calculates a list of tuples, in the same format as the input tuples, which 
    are the intersection of the input list.

    e.g. [(1, 10, 0, 'id1), (8, 20, 0, 'id2')] --> [(8, 10, 0, 'intersect')] 
    '''
    intersect = [list(range(s[0], s[1]+1)) for s in sites]
    intersect = set(intersect[0]).intersection(*intersect[1:])
    intersect = sorted(list(intersect))
    
    if intersect != []:
        final_tups = list2tups(intersect)
    else:
        return [(0, 0, 0, 'intersect')]

    starts, stops = list(zip(*final_tups))
    covs = [0 for i in range(len(final_tups))]
    ids = ['intersect' for i in range(len(final_tups))]

    final_tups = list(zip(starts, stops, covs, ids))

    return final_tups

###############################################################################
def line_maker(sites, union=True, intersect=True):
    
    starts, stops, cov, sampids = zip(*sites)
    num = len(sites)
    #print(sampids)

    cmap = matplotlib.cm.get_cmap('tab10')
    index = {sampid: i for i, sampid in enumerate(sorted(sampids))}
    xmin = min(starts)
    xmax = max(stops)
    #print(index)
    
    for site in sites:
        idx = index[site[3]]

        cidx = int(site[3][0])
        line = plt.Line2D((site[0] ,site[1]), (idx+1, idx+1), 
                            lw=4.5, color=cmap(cidx))

        plt.gca().add_line(line)
        plt.scatter((site[0]+site[1])/2, idx+1, color=cmap(cidx))
    
    plt.xlim([xmin-1, xmax+1])
    plt.ylim([0, num+1])

    if union == True:
        union_tup = tup_union(sites)
        pos = 0.3
        for un in union_tup:
            line = plt.Line2D((un[0] ,un[1]), (pos, pos), lw=4.5, color='k')
            plt.gca().add_line(line)
            plt.scatter((un[0]+un[1])/2, pos, color='k')
        plt.ylim([-1, num+1])

    if intersect == True:
        inter_tup = tup_intersect(sites)
        pos = -0.3
        for it in inter_tup:
            line = plt.Line2D((it[0] ,it[1]), (pos, pos), lw=4.5, color='k')
            plt.gca().add_line(line)
            plt.scatter((it[0]+it[1])/2, pos, color='k')
        plt.ylim([-1, num+1])

    plt.show()


###############################################################################
def stats_computer(mumerge_file, merge_file, intersect_file):
    # ast.literal_eval(string)
    return 0

###############################################################################
# Run main
###############################################################################
if __name__ == '__main__':
    params = input_parser()

    basename = params['basename']
    conditions = [params['A'], params['B'], params['C']]
    rep = params['reps']
    N = params['N']
    offset = params['offset']
    width = params['width']

    mu_npl = [c for c in conditions if c != None]
    print(mu_npl)

    ## Generate the simulated bedfiles
    exp_simulator(
        basename, 
        mu_npl=mu_npl,
        rep=rep, 
        N=N,
        offset=offset
    )

    ## Run muMerge
    mumerge = '/mnt/c/Users/Jacob/Dropbox/0DOWELL/muMerge/mumerge/mumerge.py'
    input = basename + '.input'
    output = basename
    cmd = ['python3', mumerge, '-i', input, '-o', output, '-w', str(width)]
    print("cmd: ", ' '.join(cmd), '\n')

    subprocess.run(
        cmd, 
        check=True, 
        stdout=subprocess.PIPE, 
        universal_newlines=True
    )

    ## Generate list of input bedfiles
    with open(input, 'r') as f:
        header = f.readline()

        bedfiles = []
        for line in f:
            bedfile = line.strip().split('\t')[0].strip()
            bedfiles.append(bedfile)

    # Run bedtools intersect
    intersectfile = basename + '_BEDTOOLS_INTERSECT.bed'
    bi = 'bedtools intersect'
    si = '-a stdin'
    cmd = (
        bi, '-a', bedfiles[0], '-b', bedfiles[1], 
        *(' '.join(['|', bi, si, '-b', f]) for f in bedfiles[2:]), 
        '>', intersectfile
    )

    cmd = ' '.join(cmd)

    subprocess.run(
        cmd,
        universal_newlines=True,
        shell=True
    )

    stats_computer(
        basename + '_MUMERGE.bed', 
        basename + '_BEDTOOLS_MERGE.bed',
        basename + '_BEDTOOLS_INTERSECT.bed'    
    )
    
    sys.exit(0)