from typing import Any
import httpx
import logging
from mcp.server.fastmcp import FastMCP
from .material_interpre_tool import MaterialInterpretation, BaseMaterial


logging.basicConfig(level=logging.DEBUG)  # 输出所有级别日志（包括 DEBUG 细节）

# 初始化 MCP 服务的代码...
# Initialize FastMCP server
mcp = FastMCP("weather")

# Constants
NWS_API_BASE = "https://api.weather.gov"
USER_AGENT = "weather-app/1.0"

async def make_nws_request(url: str) -> dict[str, Any] | None:
    """Make a request to the NWS API with proper error handling."""
    headers = {
        "User-Agent": USER_AGENT,
        "Accept": "application/geo+json"
    }
    async with httpx.AsyncClient() as client:
        try:
            response = await client.get(url, headers=headers, timeout=30.0)
            response.raise_for_status()
            return response.json()
        except Exception:
            return None

def format_alert(feature: dict) -> str:
    """Format an alert feature into a readable string."""
    props = feature["properties"]
    return f"""
Event: {props.get('event', 'Unknown')}
Area: {props.get('areaDesc', 'Unknown')}
Severity: {props.get('severity', 'Unknown')}
Description: {props.get('description', 'No description available')}
Instructions: {props.get('instruction', 'No specific instructions provided')}
"""

@mcp.resource("test://hello")
def hello() -> str:
    """简单的测试资源"""
    return "Hello, World!"


@mcp.prompt()
def introduce_china_province(province: str) -> str:
    """介绍中国省份

    参数:
    province: 省份名称
    """
    return f"""
    请介绍这个省份：{province}

    要求介绍以下内容：
    1. 历史沿革
    2. 人文地理、风俗习惯
    3. 经济发展状况
    4. 旅游建议
    """

@mcp.prompt()
def hello_prompt() -> str:
    """介绍中国省份

    参数:
    province: 省份名称
    """
    return f"""
你好，我是你的好朋友，请介绍中国的一个省份。
"""

# @mcp.prompt()
# def debug_code(code: str, error_message: str) -> list[base.Message]:
#     """调试代码的对话式提示模板
#
#     参数:
#     code: 需要调试的代码
#     error_message: 错误信息
#     """
#     return [
#         base.SystemMessage("你是一位专业的代码调试助手。请仔细分析用户提供的代码和错误信息，找出问题所在并提供修复方案。"),
#         base.UserMessage("我的代码有问题，请帮我修复："),
#         base.UserMessage(f"```\n{code}\n```"),
#         base.UserMessage(f"错误信息：\n{error_message}"),
#         base.AssistantMessage("我会帮你分析这段代码和错误信息。首先让我理解问题所在..."),
#     ]

@mcp.tool()
async def get_alerts(state: str) -> str:
    """Get weather alerts for a US state.

    Args:
        state: Two-letter US state code (e.g. CA, NY)
    """
    url = f"{NWS_API_BASE}/alerts/active/area/{state}"
    data = await make_nws_request(url)

    if not data or "features" not in data:
        return "Unable to fetch alerts or no alerts found."

    if not data["features"]:
        return "No active alerts for this state."

    alerts = [format_alert(feature) for feature in data["features"]]
    return "\n---\n".join(alerts)


@mcp.tool()
async def get_forecast(latitude: float, longitude: float) -> str:
    """Get weather forecast for a location.

    Args:
        latitude: Latitude of the location
        longitude: Longitude of the location
    """
    # First get the forecast grid endpoint
    points_url = f"{NWS_API_BASE}/points/{latitude},{longitude}"
    points_data = await make_nws_request(points_url)

    if not points_data:
        return "Unable to fetch forecast data for this location."

    # Get the forecast URL from the points response
    forecast_url = points_data["properties"]["forecast"]
    forecast_data = await make_nws_request(forecast_url)

    if not forecast_data:
        return "Unable to fetch detailed forecast."

    # Format the periods into a readable forecast
    periods = forecast_data["properties"]["periods"]
    forecasts = []
    for period in periods[:5]:  # Only show next 5 periods
        forecast = f"""
{period['name']}:
Temperature: {period['temperature']}°{period['temperatureUnit']}
Wind: {period['windSpeed']} {period['windDirection']}
Forecast: {period['detailedForecast']}
"""
        forecasts.append(forecast)

    return "\n---\n".join(forecasts)

@mcp.tool()
async def material_interpretation(md_content: str, doc_type: str) -> dict:
    """
    解析文档内容，返回图片链接、表格内容和纯文本内容。
    参数:
    md_content: 文档的Markdown内容
    doc_type: 文档类型，支持doc、sheet、offline_doc、offline_sheet、offline_pdf
    """
    # print("MCP 接收的参数类型：", type(md_content), type(doc_type))  # 确认是否为 str
    # print("MCP 接收的参数内容：", md_content,'\n', doc_type)  # 打印部分内容以确认

    material = BaseMaterial()
    material.md_content = md_content
    material.type = doc_type
    material,html_text = MaterialInterpretation.material_interpretation(material)
    interpretation_result = {
        "img_url_list": material.img_url_list,
        "table_list": material.table_list,
        "table_content": material.table_content,
        "net_content": material.net_content,
        "html_text": html_text
    }
    return interpretation_result


def main():
    # Initialize and run the server
    mcp.run(transport='stdio')
    # mcp.run(transport='sse')

if __name__ == "__main__":
    main()