/*
    Copyright 2013-2021 Stéphane De Mita, Mathieu Siol

    This file is part of the EggLib library.

    EggLib is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    EggLib is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with EggLib.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef EGGLIB_STIRLING_HPP
#define EGGLIB_STIRLING_HPP

namespace egglib {

   /** \brief Maximal n values for pre-computed Stirling numbers
    *
    * \ingroup diversity
    *
    * Header: <egglib-cpp/stirling.hpp>
    *
    */
    const unsigned int MAX_N_STIRLING = 1000;

   /** \brief Size of the Stirling numbers table
    *
    * \ingroup diversity
    *
    * Header: <egglib-cpp/stirling.hpp>
    *
    */
    const unsigned int NUM_STIRLING = 500499;

   /** \brief Array of log(|S(n,k)|) (Stirling numbers of the 1st kind)
    *
    * \ingroup diversity
    *
    * The values must be accessed using the stirling_table() function.
    *
    * Header: <egglib-cpp/stirling.hpp>
    *
    */
    const double STIRLING_TABLE[500499] = {
          0.00000000000000000000000000000000000000000000000000,    0.00000000000000000000000000000000000000000000000000,    0.69314718055994528622676398299518041312694549560547,    1.09861228866810956006361266190651804208755493164062,    0.00000000000000000000000000000000000000000000000000,    1.79175946922805495731267910741735249757766723632812,    2.39789527279837066942036472028121352195739746093750,    1.79175946922805495731267910741735249757766723632812,    0.00000000000000000000000000000000000000000000000000,    3.17805383034794575181081199843902140855789184570312,
          3.91202300542814551675974144018255174160003662109375,    3.55534806148941306958022323669865727424621582031250,    2.30258509299404545700440394284669309854507446289062,    0.00000000000000000000000000000000000000000000000000,    4.78749174278204581156614949577488005161285400390625,    5.61312810638807047070031330804340541362762451171875,    5.41610040220442012781632001860998570919036865234375,    4.44265125649031578092262861900962889194488525390625,    2.70805020110221006390816000930499285459518432617188,    0.00000000000000000000000000000000000000000000000000,
          6.57925121201010121296803845325484871864318847656250,    7.47533923656673682245354939368553459644317626953125,    7.39264752072162334428639951511286199092864990234375,    6.59987049921283652764714133809320628643035888671875,    5.16478597392351357342477058409713208675384521484375,    3.04452243772342301397770825133193284273147583007812,    0.00000000000000000000000000000000000000000000000000,    8.52516136106541289052529464242979884147644042968750,    9.47792177272095948126207076711580157279968261718750,    9.48280727862148253848317835945636034011840820312500,
          8.82010864450860765373363392427563667297363281250000,    7.58069975222456271524151816265657544136047363281250,    5.77455154554440852621155499946326017379760742187500,    3.33220451017520380432301863038446754217147827148438,    0.00000000000000000000000000000000000000000000000000,   10.60460290274524908227249397896230220794677734375000,   11.60444665743768588583861856022849678993225097656250,   11.67949019915174524442136316793039441108703613281250,   11.11667774591167301423411117866635322570800781250000,   10.01900134874846770571821252815425395965576171875000,
          8.41980084540758788591574557358399033546447753906250,    6.30261897574490514983835964812897145748138427734375,    3.58351893845610991462535821483470499515533447265625,    0.00000000000000000000000000000000000000000000000000,   12.80182748008146909057813900290057063102722167968750,   13.84173955069362982328584621427580714225769042968750,   13.97481934044915519166352169122546911239624023437500,   13.49210458471167051186512253480032086372375488281250,   12.50367410776239296410494716838002204895019531250000,   11.05521397685372342323262273566797375679016113281250,
          9.15377002048778720677546516526490449905395507812500,    6.76849321164862960387154089403338730335235595703125,    3.80666248977031962397177267121151089668273925781250,    0.00000000000000000000000000000000000000000000000000,   15.10441257307551587985017249593511223793029785156250,   16.17906280234009841478837188333272933959960937500000,   16.36132226083242002800943737383931875228881835937500,   15.94487257714962247234780079452320933341979980468750,   15.04425304508133010017445485573261976242065429687500,   13.71243077279557986969393823528662323951721191406250,
         11.96891257009258424659492447972297668457031250000000,    9.80642583969299685975329339271411299705505371093750,    7.18538701558041648098651421605609357357025146484375,    4.00733318523247028508649236755445599555969238281250,    0.00000000000000000000000000000000000000000000000000,   17.50230784587388299655685841571539640426635742187500,   18.60752406215685894608213857281953096389770507812500,   18.83224704189173692725489672739058732986450195312500,   18.47192575917449630651390179991722106933593750000000,   17.64405912405780441076785791665315628051757812500000,
         16.40624274027299733802465198095887899398803710937500,   14.78536404704599860338021244388073682785034179687500,   12.78667523330221733601774758426472544670104980468750,   10.39421250459511547603597136912867426872253417968750,    7.56268124672188424284513530437834560871124267578125,    4.18965474202642518264383397763594985008239746093750,    0.00000000000000000000000000000000000000000000000000,   19.98721449566188823609991231933236122131347656250000,   21.11965177383164515845237474422901868820190429687500,   21.38159357159117135438464174512773752212524414062500,
         21.06969893336112065185261599253863096237182617187500,   20.30350934577748844844791165087372064590454101562500,   19.14372720276590200683131115511059761047363281250000,   17.62195593527849979409438674338161945343017578125000,   15.75088453888297834737386438064277172088623046875000,   13.52711222906207311211801425088196992874145507812500,   10.92899136938359916371155122760683298110961914062500,    7.90728360942634722619004605803638696670532226562500,    4.35670882668959169592426405870355665683746337890625,    0.00000000000000000000000000000000000000000000000000,
         22.55216385312342097790860862005501985549926757812500,   23.70908711039376726148475427180528640747070312500000,   24.00405362616994864310981938615441322326660156250000,   23.73456398601356198696521460078656673431396484375000,   23.02161233537662354819985921494662761688232421875000,   21.92807392812313338481544633395969867706298828125000,   20.48873665589271553244543611072003841400146484375000,   18.72105547028363048411847557872533798217773437500000,   16.62910043867348974799824645742774009704589843750000,   14.20381119577785078433862508973106741905212402343750,
         11.41961428583207016629330610157921910285949707031250,    8.22443157322115681040486379060894250869750976562500,    4.51085950651684974843647069064900279045104980468750,    0.00000000000000000000000000000000000000000000000000,   25.19122118273867982907177065499126911163330078125000,   26.37035677943880074280968983657658100128173828125000,   26.69492759120939950889805913902819156646728515625000,   26.46302421605349763922276906669139862060546875000000,   25.79669817218381311363373242784291505813598632812500,   24.76039374998585884668500511907041072845458984375000,
         23.39114125959440926294519158545881509780883789062500,   21.70962493775274637641814479138702154159545898437500,   19.72471139007860685410378209780901670455932617187500,   17.43483291894815678801933245267719030380249023437500,   14.82701946541049409233892220072448253631591796875000,   11.87287005777658421834530599880963563919067382812500,    8.51819269174932003352296305820345878601074218750000,    4.65396035015752307373304574866779148578643798828125,    0.00000000000000000000000000000000000000000000000000,   27.89927138384089033706914051435887813568115234375000,
         29.09870258866740755365754012018442153930664062500000,   29.45004186576593241397858946584165096282958984375000,   29.25179368847570415823611256200820207595825195312500,   28.62679317168984383101815183181315660476684570312500,   27.64064674761720041828993998933583498001098632812500,   26.33201653220129756505230034235864877700805664062500,   24.72386969556349356480495771393179893493652343750000,   22.82832980271033918029388587456196546554565429687500,   20.64864545353245972592048929072916507720947265625000,   18.17935387355446508195200294721871614456176757812500,
         15.40466861724804914501874009147286415100097656250000,   12.29408352285288685834530042484402656555175781250000,    8.79179002431936140737889218144118785858154296875000,    4.78749174278204581156614949577488005161285400390625,    0.00000000000000000000000000000000000000000000000000,   30.67186010608067192606540629640221595764160156250000,   31.88995147079046077465136477258056402206420898437500,   32.26567480848004265681083779782056808471679687500000,   32.09782250074237452963643590919673442840576171875000,   31.50981839436538223253592150285840034484863281250000,
         30.56814747870018322828400414437055587768554687500000,   29.31269103074890836069243960082530975341796875000000,   27.76808488544434538880523177795112133026123046875000,   25.94876074538337462627168861217796802520751953125000,   23.86124218028181687145661271642893552780151367187500,   21.50489604525327536066470202058553695678710937500000,   18.87145333651287870679880143143236637115478515625000,   15.94303034379194095038201339775696396827697753906250,   12.68751065240157238633855740772560238838195800781250,    9.04782144247840669493143650470301508903503417968750,
          4.91265488573605146882528060814365744590759277343750,    0.00000000000000000000000000000000000000000000000000,   33.50507345013689075585716636851429939270019531250000,   34.74041483243328798380389343947172164916992187500000,   35.13849297269772620211369940079748630523681640625000,   34.99829649835805867041926831007003784179687500000000,   34.44369608511274805096036288887262344360351562500000,   33.54185203914951785009179729968309402465820312500000,   32.33357629312130399057423346675932407379150390625000,   30.84475756873706231431242485996335744857788085937500,
         29.09151931826866288588462339248508214950561523437500,   27.08270975744059683165687602013349533081054687500000,   24.82098293875825234522380924317985773086547851562500,   22.30290880864446378950560756493359804153442382812500,   19.51812837466962236021572607569396495819091796875000,   16.44715346858215809788816841319203376770019531250000,   13.05660811633886098093171312939375638961791992187500,    9.28841190739633759676507906988263130187988281250000,    5.03043792139243528538372629554942250251770019531250,    0.00000000000000000000000000000000000000000000000000,
         36.39544520803305260869819903746247291564941406250000,   37.64680950690201655106648104265332221984863281250000,   38.06549713681206270621260046027600765228271484375000,   37.95062531392323990075965411961078643798828125000000,   37.42640619044607319665374234318733215332031250000000,   36.56052626778021874542901059612631797790527343750000,   35.39452064414920329227243200875818729400634765625000,   33.95523971280862696175972814671695232391357421875000,   32.26008106287038401660538511350750923156738281250000,   30.31959986669619766530558990780264139175415039062500,
         28.13878609257274732158293772954493761062622070312500,   25.71750547878793824452259286772459745407104492187500,   23.05024469514868812325403268914669752120971679687500,   20.12504563914381350286930683068931102752685546875000,   16.92116491737721872823385638184845447540283203125000,   13.40422252951331749670771387172862887382507324218750,    9.51532194150602350646295235492289066314697265625000,    5.14166355650265938237453156034462153911590576171875,    0.00000000000000000000000000000000000000000000000000,   39.33988418719949464730234467424452304840087890625000,
         40.60619487159935658837639493867754936218261718750000,   41.04397692209792580797511618584394454956054687500000,   40.95242562207472047930423286743462085723876953125000,   40.45601508269213297808164497837424278259277343750000,   39.62284637497892703095203614793717861175537109375000,   38.49502494078013370426560868509113788604736328125000,   37.10015299226995466597145423293113708496093750000000,   35.45660943899668637868671794421970844268798828125000,   33.57623766052751079769222997128963470458984375000000,   31.46574501452405314694260596297681331634521484375000,
         29.12734111449512752756163536105304956436157226562500,   26.55881464045983975097442453261464834213256835937500,   23.75305810840614739731790905352681875228881835937500,   20.69686220890359606983111007139086723327636718750000,   17.36848282806641208253495278768241405487060546875000,   13.73272659648535309884209709707647562026977539062500,    9.73002662417436958719463291345164179801940917968750,    5.24702407216048616334092002944089472293853759765625,    0.00000000000000000000000000000000000000000000000000,   42.33561646075348505746660521253943443298339843750000,
         43.61592223600509043990314239636063575744628906250000,   44.07147267486532626890038955025374889373779296875000,   44.00150305070374656679632607847452163696289062500000,   43.53068855867851283392155892215669155120849609375000,   42.72745982987970592148485593497753143310546875000000,   41.63437777871399703144561499357223510742187500000000,   40.27964153963598903374077053740620613098144531250000,   38.68240008268229956911454792134463787078857421875000,   36.85549117855037337676549213938415050506591796875000,   34.80692008821306870913758757524192333221435546875000,
         32.54062121481194225225408445112407207489013671875000,   30.05673289694862759802163054700940847396850585937500,   27.35145427175673660258325980976223945617675781250000,   24.41643038765600692840962437912821769714355468750000,   21.23745429205159140906289394479244947433471679687500,   17.79197106692788565851515159010887145996093750000000,   14.04411914870954269929370639147236943244934082031250,    9.93377424514100049179887719219550490379333496093750,    5.34710753071746847098211219417862594127655029296875,    0.00000000000000000000000000000000000000000000000000,
         45.38013889847690052192774601280689239501953125000000,   46.67359366918137197899341117590665817260742187500000,   47.14574339476168773899189545772969722747802734375000,   47.09583443520344303578895051032304763793945312500000,   46.64869589280279882359536713920533657073974609375000,   45.87302224355800461808030377142131328582763671875000,   44.81174199959985315899757551960647106170654296875000,   43.49353451283003124672177364118397235870361328125000,   41.93816267220695692685694666579365730285644531250000,   40.15924649473485175121822976507246494293212890625000,
         38.16579467623877519599773222580552101135253906250000,   35.96304642795854533687816001474857330322265625000000,   33.55287027586938819467832217924296855926513671875000,   30.93381932030465364391602633986622095108032226562500,   28.10084872263947630699476576410233974456787109375000,   25.04460876247229705882091366220265626907348632812500,   21.75008407949862743180347024463117122650146484375000,   18.19405358190607557844487018883228302001953125000000,   14.34010002397601901691359671531245112419128417968750,   10.12763060418342142554593010572716593742370605468750,
          5.44241771052179323930886312155053019523620605468750,    0.00000000000000000000000000000000000000000000000000,   48.47118135183522014131085597909986972808837890625000,   49.77702817861749906569457380101084709167480468750000,   50.26473966173448815197843941859900951385498046875000,   50.23355119651508715605814359150826930999755859375000,   49.80840884198275375638331752270460128784179687500000,   49.05821945069417466811501071788370609283447265625000,   48.02621087269368160832527792081236839294433593750000,   46.74145317210659555939855636097490787506103515625000,
         45.22420534426344573830647277645766735076904296875000,   43.48871272309951052648102631792426109313964843750000,   41.54477471337104788062788429670035839080810546875000,   39.39864042443844738272673566825687885284423828125000,   37.05348824433968246694348636083304882049560546875000,   34.50960549219450257396601955406367778778076171875000,   31.76430456348428066348787979222834110260009765625000,   28.81154854384632457708903530146926641464233398437500,   25.64118155097661855279511655680835247039794921875000,   22.23752394463698678350738191511482000350952148437500,
         18.57680056375032506821298738941550254821777343750000,   14.62212703894445375851773860631510615348815917968750,   10.31251298183904907546093454584479331970214843750000,    5.53338948872752034446875768480822443962097167968750,    0.00000000000000000000000000000000000000000000000000,   51.60667556776436981635924894362688064575195312500000,   52.92423367922251742356820614077150821685791015625000,   53.42658068888480471514412784017622470855712890625000,   53.41292415315608366199739975854754447937011718750000,   53.00829788038121392901302897371351718902587890625000,
         52.28178032340417047407754580490291118621826171875000,   51.27684492969051888167086872272193431854248046875000,   50.02288275845655363127661985345184803009033203125000,   48.54055841531200599092699121683835983276367187500000,   46.84462403347706072054279502481222152709960937500000,   44.94551227694757500330524635501205921173095703125000,   42.85026903179655022313454537652432918548583984375000,   40.56308733009982603334719897247850894927978515625000,   38.08556813954893982554494868963956832885742187500000,   35.41676114721133927787377615459263324737548828125000,
         32.55298920327509648586783441714942455291748046875000,   29.48741134205532432588370284065604209899902343750000,   26.20920918150484624220553087070584297180175781250000,   22.70215052853717097036678751464933156967163085937500,   18.94199440150813629202275478746742010116577148437500,   14.89145997272803789712725119898095726966857910156250,   10.48921654632878031065956747625023126602172851562500,    5.62040086571714958552092866739258170127868652343750,    0.00000000000000000000000000000000000000000000000000,   54.78472939811231867679452989250421524047851562500000,
         56.11338357052742509267773129977285861968994140625000,   56.62953478488006453517300542443990707397460937500000,   56.63234983531140898094236035831272602081298828125000,   56.24692700876394013675962924025952816009521484375000,   55.54248372313685422341222874820232391357421875000000,   54.56269620821814214650657959282398223876953125000000,   53.33722146882329440131798037327826023101806640625000,   51.88705911305611806483284453861415386199951171875000,   50.22737694321461532354078372009098529815673828125000,   48.36912189676129258941728039644658565521240234375000,
         46.31997883084883937954145949333906173706054687500000,   44.08494202284973084715602453798055648803710937500000,   41.66663065044546954140969319269061088562011718750000,   39.06541094359918986356206005439162254333496093750000,   36.27934542794610450755499186925590038299560546875000,   33.30395489099369399355055065825581550598144531250000,   30.13173754493892886330286273732781410217285156250000,   26.75132388420002627071880851872265338897705078125000,   23.14601712361858076860698929522186517715454101562500,   19.29118088742821868208920932374894618988037109375000,
         15.14919496881186766756854922277852892875671386718750,   10.65843513400151998382625606609508395195007324218750,    5.70378247465620091816163039766252040863037109375000,    0.00000000000000000000000000000000000000000000000000,   58.00360522298051790812678518705070018768310546875000,   59.34279701679830054672493133693933486938476562500000,   59.87200264186405007649227627553045749664306640625000,   59.89033824856868903907525236718356609344482421875000,   59.52294793485293666890356689691543579101562500000000,   58.83916172765358965079940389841794967651367187500000,
         57.88282416610123704003854072652757167816162109375000,   56.68381420446904428445122903212904930114746093750000,   55.26341059972745028971985448151826858520507812500000,   53.63712564266494808862262289039790630340576171875000,   51.81632899097704836322009214200079441070556640625000,   49.80922516537165734007430728524923324584960937500000,   47.62145163211961573779262835159897804260253906250000,   45.25643218066751671813108259811997413635253906250000,   42.71555398382125190437363926321268081665039062500000,   39.99819836131052142036423902027308940887451171875000,
         37.10162765432940545906603801995515823364257812500000,   34.02070341298527011986152501776814460754394531250000,   30.74737407630700047889149573165923357009887695312500,   27.26980670747942525622420362196862697601318359375000,   23.57091013300485471404499548953026533126831054687500,   19.62570947559078149424749426543712615966796875000000,   15.39629176114856967672039900207892060279846191406250,   10.82077778474336682279499655123800039291381835937500,    5.78382518232973730221146979602053761482238769531250,    0.00000000000000000000000000000000000000000000000000,
         61.26170176100199427082770853303372859954833984375000,   62.61092222911620552849853993393480777740478515625000,   63.15250297356904241041775094345211982727050781250000,   63.18550197946974833485000999644398689270019531250000,   62.83509409343812279757912619970738887786865234375000,   62.17070049513960583453808794729411602020263671875000,   61.23630601363684178295443416573107242584228515625000,   60.06197600969331062970013590529561042785644531250000,   58.66922366400799404573263018392026424407958984375000,   57.07384938270628538248274708166718482971191406250000,
         55.28757397930835537636085064150393009185791015625000,   53.31902979282546084505156613886356353759765625000000,   51.17437794325170585807427414692938327789306640625000,   48.85768859426022459047089796513319015502929687500000,   46.37115611661459269043916719965636730194091796875000,   43.71518490355872188501962227746844291687011718750000,   40.88835786828455098884660401381552219390869140625000,   37.88727961483525064068089704960584640502929687500000,   34.70626319870576281800822471268475055694580078125000,   31.33679472695389733871706994250416755676269531250000,
         27.76664782700810008009284501895308494567871093750000,   23.97839365517675958017207449302077293395996093750000,   19.94676529704839040846309217158704996109008789062500,   15.63359545235987724254300701431930065155029296875000,   10.97678203321994772068137535825371742248535156250000,    5.86078622346586541880242293700575828552246093750000,    0.00000000000000000000000000000000000000000000000000,   64.55753862700633760596247157081961631774902343750000,   65.91632220123122465338383335620164871215820312500000,   66.46966011654444628220517188310623168945312500000000,
         66.51654651432716036651981994509696960449218750000000,   66.18217477706018314620450837537646293640136718750000,   65.53603964436459250464395154267549514770507812500000,   64.62224326756359005230478942394256591796875000000000,   63.47100842143646559634362347424030303955078125000000,   62.10404650406656656969062169082462787628173828125000,   60.53740083880049382969446014612913131713867187500000,   58.78308667227681638678404851816594600677490234375000,   56.85009233138179496336306328885257244110107421875000,   54.74501053714862308652300271205604076385498046875000,
         52.47243812721318079184129601344466209411621093750000,   50.03521829906497231377215939573943614959716796875000,   47.43456466468025212179782101884484291076660156250000,   44.67008489715775709782974445261061191558837890625000,   41.73970559572425287342412048019468784332275390625000,   38.63948408282612234643238480202853679656982421875000,   35.36327212502468597676852368749678134918212890625000,   31.90216329760050939512439072132110595703125000000000,   28.24359434682028791030461434274911880493164062500000,   24.36984508499229562517029989976435899734497070312500,
         20.25539488392918485715199494734406471252441406250000,   15.86185410434085873987442028010264039039611816406250,   11.12692469164966624362023139838129281997680664062500,    5.93489419561958797544320987071841955184936523437500,    0.00000000000000000000000000000000000000000000000000,   67.88974313718154007801786065101623535156250000000000,   69.25766246644221268979890737682580947875976562500000,   69.82219327850117451816913671791553497314453125000000,   69.88226164014815822156378999352455139160156250000000,   69.56306952761026707321434514597058296203613281250000,
         68.93417072325132721744012087583541870117187500000000,   68.03976572977153125520999310538172721862792968750000,   66.91021088025685514821816468611359596252441406250000,   65.56738619192370265409408602863550186157226562500000,   64.02754126689633551450242521241307258605957031250000,   62.30294026736833501445289584808051586151123046875000,   60.40287054222142870685274829156696796417236328125000,   58.33428439344049820647342130541801452636718750000000,   56.10221312896663192759660887531936168670654296875000,   53.71002895046569136638936470262706279754638671875000,
         51.15959621687447622662148205563426017761230468750000,   48.45133342267858722607343224808573722839355468750000,   45.58419327966113598904485115781426429748535156250000,   42.55555625846063350081749376840889453887939453125000,   39.36101938412972600644934573210775852203369140625000,   35.99404275089456461955705890432000160217285156250000,   32.44538380312447145570331485942006111145019531250000,   28.70218860222941970050669624470174312591552734375000,   24.74648382921512634879945835564285516738891601562500,   20.55252703431303018533071735873818397521972656250000,
         16.08173307344127422879864752758294343948364257812500,   11.27163067207156821325497730867937207221984863281250,    6.00635315960173254978826662409119307994842529296875,    0.00000000000000000000000000000000000000000000000000,   71.25703896716800045396666973829269409179687500000000,   72.63370053090636702108895406126976013183593750000000,   73.20890717493473687227378832176327705383300781250000,   73.28151380293857641845534089952707290649414062500000,   72.97672286572839084328734315931797027587890625000000,   72.36413514073848318730597384274005889892578125000000,
         71.48803370407732415969803696498274803161621093750000,   70.37888866316198743788845604285597801208496093750000,   69.05872424365685446900897659361362457275390625000000,   67.54396633218178180868562776595354080200195312500000,   65.84709107516597725862084189429879188537597656250000,   63.97763910292145794755924725905060768127441406250000,   61.94286515854278007964239805005490779876708984375000,   59.74816286168499601672010612674057483673095703125000,   57.39734110232781461036211112514138221740722656250000,   54.89279507155118409400529344566166400909423828125000,
         52.23559553659279686144145671278238296508789062500000,   49.42550731851333978283946635201573371887207031250000,   46.46093809840998289928393205627799034118652343750000,   43.33880899376163853276011650450527667999267578125000,   40.05432617580115106648008804768323898315429687500000,   36.60061432244048518214185605756938457489013671875000,   32.96814082108437560236779972910881042480468750000000,   29.14379915677050547628823551349341869354248046875000,   25.10939468197566171170365123543888330459594726562500,   20.83898988233641702549903129693120718002319335937500,
         16.29382678856890009910785011015832424163818359375000,   11.41128026120802729792558238841593265533447265625000,    6.07534603108868420662247444852255284786224365234375,    0.00000000000000000000000000000000000000000000000000,   74.65823634883017234642466064542531967163085937500000,   76.04327670674668127048789756372570991516113281250000,   76.62868384118301889884605770930647850036621093750000,   76.71323930915947642006358364596962928771972656250000,   76.42213939027354285826731938868761062622070312500000,   75.82502180799261282118095550686120986938476562500000,
         74.96623900722144639985344838351011276245117187500000,   73.87635834500080989073467208072543144226074218750000,   72.57752796149394214353378629311919212341308593750000,   71.08632525306541083409683778882026672363281250000000,   69.41540808677841312146483687683939933776855468750000,   67.57453323794116784029029076918959617614746093750000,   65.57121199015253409925207961350679397583007812500000,   63.41114300749877230600759503431618213653564453125000,   61.09849960642369381957905716262757778167724609375000,   58.63611540936364718845652532763779163360595703125000,
         56.02559347073407280959145282395184040069580078125000,   53.26735211261332381127431290224194526672363281250000,   50.36061217253009658634255174547433853149414062500000,   47.30332287908046851043764036148786544799804687500000,   44.09201527672816212088946485891938209533691406250000,   40.72156080212543116658707731403410434722900390625000,   37.18479467529379434154179762117564678192138671875000,   33.47193223678446116764462203718721866607666015625000,   29.56964611363936512589134508743882179260253906250000,   25.45954701389248953091737348586320877075195312500000,
         21.11552497481400791912165004760026931762695312500000,   16.49866850154838715525329462252557277679443359375000,   11.54621516283486926113255321979522705078125000000000,    6.14203740558735589161187817808240652084350585937500,    0.00000000000000000000000000000000000000000000000000,   78.09222355331530707189813256263732910156250000000000,   79.48530612119706972862331895157694816589355468750000,   80.08047544415299512365891132503747940063476562500000,   80.17643826798935435817838879302144050598144531250000,   79.89837925339830349003023002296686172485351562500000,
         79.31596465008766472237766720354557037353515625000000,   78.47360515800318125911871902644634246826171875000000,   77.40195149144817321484879357740283012390136718750000,   76.12325871043370284496631938964128494262695312500000,   74.65423509149840697318722959607839584350585937500000,   73.00769519923711925457610050216317176818847656250000,   71.19358150116410399732558289542794227600097656250000,   69.21962453730705533416767138987779617309570312500000,   67.09178199799187325425009476020932197570800781250000,   64.81453426355633951061463449150323867797851562500000,
         62.39108104756412842561985598877072334289550781250000,   59.82346522550864165168604813516139984130859375000000,   57.11263857986522651799532468430697917938232421875000,   54.25847644947475600929465144872665405273437500000000,   51.25974207977869667729464708827435970306396484375000,   48.11399537232597367619746364653110504150390625000000,   44.81743328836570583462162176147103309631347656250000,   41.36463837909511198631662409752607345581054687500000,   37.74819432370156135903016547672450542449951171875000,   33.95809605443867695839799125678837299346923828125000,
         29.98082195550112416526644665282219648361206054687500,   25.79781064679782431880994408857077360153198242187500,   21.38279896449897421462083002552390098571777343750000,   16.69673841531048452679897309280931949615478515625000,   11.67674355019962284529810858657583594322204589843750,    6.20657592672492697261077410075813531875610351562500,    0.00000000000000000000000000000000000000000000000000,   81.55795945611502872907294658944010734558105468750000,   82.95877171950603212735586566850543022155761718750000,   83.56329794793551002385356696322560310363769531250000,
         83.67016918389559521074261283501982688903808593750000,   83.40455400114970530012215021997690200805664062500000,   82.83614009280709922222740715369582176208496093750000,   82.00938701158511889843794051557779312133789062500000,   80.95501708019070008504058932885527610778808593750000,   79.69537795192172779934480786323547363281250000000000,   78.24729147862693423576274653896689414978027343750000,   76.62370806324686611787910806015133857727050781250000,   74.83473066744471680067363195121288299560546875000000,   72.88827849832905769744684221222996711730957031250000,
         70.79053087961518997417442733421921730041503906250000,   68.54622914918445530929602682590484619140625000000000,   66.15888167663302965593175031244754791259765625000000,   63.63089875786128857271251035854220390319824218750000,   60.96367311869395422263551154173910617828369140625000,   58.15761451528646119868426467292010784149169921875000,   55.21214151754609389399774954654276371002197265625000,   52.12562875757548397359641967341303825378417968750000,   48.89530267694680532031270558945834636688232421875000,   45.51707189920833229734853375703096389770507812500000,
         41.98526791896242826851448626257479190826416015625000,   38.29225443747849055853293975815176963806152343750000,   34.42783252854449216329157934524118900299072265625000,   30.37830883306902407525740272831171751022338867187500,   26.12496908049998722844975418411195278167724609375000,   21.64141338946163628520480415318161249160766601562500,   16.88847050377182412717047554906457662582397460937500,   11.80314431707397915261026355437934398651123046875000,    6.26909628370626137439103331416845321655273437500000,    0.00000000000000000000000000000000000000000000000000,
         85.05446701758151561989507172256708145141601562500000,   86.46271811212152158532262546941637992858886718750000,   87.07622551186230452913150656968355178833007812500000,   87.19354412014068600456084823235869407653808593750000,   86.93982276084726379394851392135024070739746093750000,   86.38476459145913111115078208968043327331542968750000,   85.57287002575753831479232758283615112304687500000000,   84.53492300463381070585455745458602905273437500000000,   83.29335162364570521731366170570254325866699218750000,   81.86507669595648906124552013352513313293457031250000,
         80.26316694517430505584343336522579193115234375000000,   78.49786480573729363641177769750356674194335937500000,   76.57725282716369008539913920685648918151855468750000,   74.50770127634146433592832181602716445922851562500000,   72.29417497408384463142283493652939796447753906250000,   69.94044477239999935136438580229878425598144531250000,   67.44923089643381786117970477789640426635742187500000,   64.82229456742162199134327238425612449645996093750000,   62.06048740153762821591953979805111885070800781250000,   59.16376317457965683388465549796819686889648437500000,
         56.13115252688215406351446290500462055206298828125000,   52.96069722769611587409599451348185539245605468750000,   49.64933590572876198621088406071066856384277343750000,   46.19272659194265884252672549337148666381835937500000,   42.58498121545226666739836218766868114471435546875000,   38.81826998744234202831648872233927249908447265625000,   34.88222256893048722758976509794592857360839843750000,   30.76299300813268899901231634430587291717529296875000,   26.44173058623874084105409565381705760955810546875000,   21.89191290337301865065455785952508449554443359375000,
         17.07425826854226613704668125137686729431152343750000,   11.92567067537568092916444584261626005172729492187500,    6.32972090552269595065126850386150181293487548828125,    0.00000000000000000000000000000000000000000000000000,   88.58082754219768162329273764044046401977539062500000,   89.99624614281191270492854528129100799560546875000000,   90.61838551938728869572514668107032775878906250000000,   90.74572436393545160626672441139817237854003906250000,   90.50338875213580536183144431561231613159179687500000,   89.96109224324446529408305650576949119567871093750000,
         89.16336929031054125971422763541340827941894531250000,   88.14105691513083229438052512705326080322265625000000,   86.91665329193797617790551157668232917785644531250000,   85.50716577723738964778021909296512603759765625000000,   83.92576660361174845093046315014362335205078125000000,   82.18282000989781010957813123241066932678222656250000,   80.28655074898534849126008339226245880126953125000000,   78.24349461251236448333656881004571914672851562500000,   76.05880913800655207523959688842296600341796875000000,   73.73649012274188407900510355830192565917968750000000,
         71.27952149535717296657821862027049064636230468750000,   68.68997543738657896028598770499229431152343750000000,   65.96907293965692531401145970448851585388183593750000,   63.11721039440163139033757033757865428924560546875000,   60.13395430375965133862337097525596618652343750000000,   57.01800301804500747948623029515147209167480468750000,   53.76711099715346620087075280025601387023925781250000,   50.37796672238651041197954327799379825592041015625000,   46.84600905493756073383337934501469135284423828125000,   43.16515678731587968286476098001003265380859375000000,
         39.32740903510698160516767529770731925964355468750000,   35.32224315240772938295776839368045330047607421875000,   31.13567699764150731311929121147841215133666992187500,   26.74873756809244440546535770408809185028076171875000,   22.13479224269801903801635489799082279205322265625000,   17.25445962556946710719785187393426895141601562500000,   12.04455321636548248420695017557591199874877929687500,    6.38856140554563012301514390856027603149414062500000,    0.00000000000000000000000000000000000000000000000000,   92.13617560368707870566140627488493919372558593750000,
         93.55850807534345392468821955844759941101074218750000,   94.18895415239842350274557247757911682128906250000000,   94.32591653291837019423837773501873016357421875000000,   94.09449609691775151532056042924523353576660156250000,   93.56441250768156692174670752137899398803710937500000,   92.78022841173536505721131106838583946228027343750000,   91.77282658262254244618816301226615905761718750000000,   90.56476638858657679520547389984130859375000000000000,   89.17313111743304432366130640730261802673339843750000,   87.61118390869376071350416168570518493652343750000000,
         85.88939585358512829316168790683150291442871093750000,   84.01611613040915926831075921654701232910156250000000,   81.99802482919233170832740142941474914550781250000000,   79.84044671495625777879467932507395744323730468750000,   77.54757170576104385872895363718271255493164062500000,   75.12260984802266250426328042522072792053222656250000,   72.56789801687115470940625527873635292053222656250000,   69.88496900779283294014021521434187889099121093750000,   67.07458931411819946788455126807093620300292968750000,   64.13676868437454459126456640660762786865234375000000,
         61.07074188086367882988270139321684837341308593750000,   57.87492042816951709482964361086487770080566406250000,   54.54680906640215454217468504793941974639892578125000,   51.08287748722440113624543300829827785491943359375000,   47.47837175479987337212151032872498035430908203125000,   43.72703987075435350106999976560473442077636718750000,   39.82072891832329020189717994071543216705322265625000,   35.74878031016709201139747165143489837646484375000000,   31.49708984624441043820297636557370424270629882812500,   27.04657450796479523091875307727605104446411132812500,
         22.37050215692115173737875011283904314041137695312500,   17.42940107506579749951924895867705345153808593750000,   12.16000252867564590530946588842198252677917480468750,    6.44571981938557847513493470614776015281677246093750,    0.00000000000000000000000000000000000000000000000000,   95.71969454214320194296305999159812927246093750000000,   97.14870331326581265329878078773617744445800781250000,   97.78715243889806174593104515224695205688476562500000,   97.93336907050775153038557618856430053710937500000000,   97.71242690328722346748691052198410034179687500000000,
         97.19404804817381204884441103786230087280273437500000,   96.42281831804091041249193949624896049499511718750000,   95.42965991957666460621112491935491561889648437500000,   94.23718576199689778150059282779693603515625000000000,   92.86254594801935979830886935815215110778808593750000,   91.31908374038168574315932346507906913757324218750000,   89.61736434972041820401500444859266281127929687500000,   87.76584632546664010988024529069662094116210937500000,   85.77133619657423935223050648346543312072753906250000,   83.63930466311214217967062722891569137573242187500000,
         81.37411022031852780855842866003513336181640625000000,   78.97915816256335119760478846728801727294921875000000,   76.45701243461775220566778443753719329833984375000000,   73.80947133735675436128076398745179176330566406250000,   71.03761386722534609816648298874497413635253906250000,   68.14182047955428345176187576726078987121582031250000,   65.12176971206255871038592886179685592651367187500000,   61.97640996637099419785954523831605911254882812500000,   58.70390345849102686770493164658546447753906250000000,   55.30153650445678437108654179610311985015869140625000,
         51.76558632547992289119065389968454837799072265625000,   48.09112848791075833787544979713857173919677734375000,   44.27175922511926131619475199840962886810302734375000,   40.29918991780197501384463976137340068817138671875000,   36.16264013702785007353668333962559700012207031250000,   31.84789586440932041000451135914772748947143554687500,   27.33577474447129418422264279797673225402832031250000,   22.59945448198369177816857700236141681671142578125000,   17.59938127739835067586682271212339401245117187500000,   12.27221144803421815083765977760776877403259277343750,
          6.50128967054038930939441343070939183235168457031250,    0.00000000000000000000000000000000000000000000000000,   99.33061245478742762315960135310888290405273437500000,  100.76607458111499227015883661806583404541015625000000,  101.41224271298958115039567928761243820190429687500000,  101.56736908452450052209314890205860137939453125000000,  101.35649859944737727346364408731460571289062500000000,  100.84935270019583697376219788566231727600097656250000,  100.09053602901597912477882346138358116149902343750000,   99.11100475713618607187527231872081756591796875000000,
         97.93341871358994410456944024190306663513183593750000,   96.57498694643116721181286266073584556579589843750000,   95.04912356472731005396781256422400474548339843750000,   93.36647699238534414689638651907444000244140625000000,   91.53560232138792684963846113532781600952148437500000,   89.56341738911896754871122539043426513671875000000000,   87.45552090943009204693225910887122154235839843750000,   85.21641861142234120052307844161987304687500000000000,   82.84968545139749096506420755758881568908691406250000,   80.35808153817612264901981689035892486572265625000000,
         77.74363302359832061938504921272397041320800781250000,   75.00768508176601301329355919733643531799316406250000,   72.15093125414391295180394081398844718933105468750000,   69.17342129448753951237449655309319496154785156250000,   66.07454782749888977377850096672773361206054687500000,   62.85301034208216464094221009872853755950927734375000,   59.50675298209747410282943747006356716156005859375000,   56.03286990753691299005367909558117389678955078125000,   52.42746812396200084549491293728351593017578125000000,   48.68547168374693967507482739165425300598144531250000,
         44.80034134775664966809927136637270450592041015625000,   40.76366686393770777385725523345172405242919921875000,   36.56455817524123830253302003256976604461669921875000,   32.18870210020171640508124255575239658355712890625000,   27.61682628577782239176485745701938867568969726562500,   22.82202650155302237067189707886427640914916992187500,   17.76467413377885051772864244412630796432495117187500,   12.38135699919111409883498708950355648994445800781250,    6.55535689181066505426542789791710674762725830078125,    0.00000000000000000000000000000000000000000000000000,
        102.96819861451380972994229523465037345886230468750000,  104.40990450658989630028372630476951599121093750000000,  105.06352543522561404643056448549032211303710937500000,  105.22723948941890625974338036030530929565429687500000,  105.02606149497351850641280179843306541442871093750000,  104.52970956656903922521451022475957870483398437500000,  103.78280342340126196631899802014231681823730468750000,  102.81632845128079623009398346766829490661621093750000,  101.65298564730224484264908824115991592407226562500000,  100.31003618098353058485372457653284072875976562500000,
         98.80095698786452373951760819181799888610839843750000,   97.13647031360386563392239622771739959716796875000000,   95.32521679701969219422608148306608200073242187500000,   93.37421268394949436242313822731375694274902343750000,   91.28916951362585052720532985404133796691894531250000,   89.07472228008462877824058523401618003845214843750000,   86.73459421849990746977709932252764701843261718750000,   84.27171598219983650324138579890131950378417968750000,   81.68831063984288221035967580974102020263671875000000,   78.98595186422038239015819272026419639587402343750000,
         76.16559993620829516203230014070868492126464843750000,   73.22761818637208364179969066753983497619628906250000,   70.17177088681530960911914007738232612609863281250000,   66.99720213097977250527037540450692176818847656250000,   63.70239367475384995032072765752673149108886718750000,   60.28509780895057446059581707231700420379638671875000,   56.74223874984321724923574947752058506011962890625000,   53.06977223408789257064199773594737052917480468750000,   49.26248706326847326408824301324784755706787109375000,   45.31372256680093357772420858964323997497558593750000,
         41.21495904788659458972688298672437667846679687500000,   36.95520745395425876722583780065178871154785156250000,   32.52006475951479558261780766770243644714355468750000,   27.89017681741872678458094014786183834075927734375000,   23.03856471299929253859772870782762765884399414062500,   17.92553145190450791801595187280327081680297851562500,   12.48760207925867504741290758829563856124877929687500,    6.60800062529608656092250384972430765628814697265625,    0.00000000000000000000000000000000000000000000000000,  106.63176026064346046950959134846925735473632812500000,
        108.07951255250175393030076520517468452453613281250000,  108.74033632895842060861468780785799026489257812500000,  108.91233641756282679580181138589978218078613281250000,  108.72049654841738686172902816906571388244628906250000,  108.23452923706909700740652624517679214477539062500000,  107.49906602451872572601132560521364212036132812500000,  106.54511737172768448544957209378480911254882812500000,  105.39542042843575586630322504788637161254882812500000,  104.06728254375889264338184148073196411132812500000000,  102.57423651486949722766439663246273994445800781250000,
        100.92707028176351968795643188059329986572265625000000,   99.13450055453316167586308438330888748168945312500000,   97.20363092475068356179690454155206680297851562500000,   95.14027280270872211076493840664625167846679687500000,   92.94917521423970185878715710714459419250488281250000,   90.63419166715557651059498311951756477355957031250000,   88.19840194476688566282973624765872955322265625000000,   85.64420038620168895704409806057810783386230468750000,   82.97335820575109721630724379792809486389160156250000,   80.18706472476874580479488940909504890441894531250000,
         77.28595048648234921984112588688731193542480468750000,   74.27009375586489170473214471712708473205566406250000,   71.13901064169255050728679634630680084228515625000000,   67.89162783151411417748022358864545822143554687500000,   64.52623552044335042410239111632108688354492187500000,   61.04041631789233690597029635682702064514160156250000,   57.43094340764545790989359375089406967163085937500000,   53.69363748946369696568581275641918182373046875000000,   49.82316612769349717382283415645360946655273437500000,   45.81275938393036284423942561261355876922607421875000,
         41.65379872752811962755004060454666614532470703125000,   37.33520541005668746947776526212692260742187500000000,   32.84249474806727420173046994023025035858154296875000,   28.15623803558136373226261639501899480819702148437500,   23.24938809313934484634955879300832748413085937500000,   18.08218526194988484689929464366286993026733398437500,   12.59109692273355207703389169182628393173217773437500,    6.65929391968363759701787785161286592483520507812500,    0.00000000000000000000000000000000000000000000000000,  110.32063971475739094785240013152360916137695312500000,
        111.77425225493449545410840073600411415100097656250000,  112.44204379467159071737114572897553443908691406250000,  112.62204686949330323386675445362925529479980468750000,  112.43921332197031404120934894308447837829589843750000,  111.96324812759600320077879587188363075256347656250000,  111.23879181879664201915147714316844940185546875000000,  110.29687631326726204861188307404518127441406250000000,  109.16027052467939029156696051359176635742187500000000,  107.84632278775131908332696184515953063964843750000000,  106.36861568697344182510278187692165374755859375000000,
        104.73799579076423071910539874807000160217285156250000,  102.96324767408772515864257002249360084533691406250000,  101.05155273570926510728895664215087890625000000000000,   99.00881114317633091559400781989097595214843750000000,   96.83987295773953007937961956486105918884277343750000,   94.54870669163550189750822028145194053649902343750000,   92.13852322233748282087617553770542144775390625000000,   89.61186671971144335202552611008286476135253906250000,   86.97068027153106584137276513502001762390136718750000,   84.21635126366159340705053182318806648254394531250000,
         81.34973973667183599900454282760620117187500000000000,   78.37119156986059920200204942375421524047851562500000,   75.28053722008560555423173354938626289367675781250000,   72.07707570605218450054962886497378349304199218750000,   68.75954243671876042753865476697683334350585937500000,   65.32605817906991774179914500564336776733398437500000,   61.77405473830069837504197494126856327056884765625000,   58.10017046783498528839118080213665962219238281250000,   54.30010501773291764493478694930672645568847656250000,   50.36841685537314816656362381763756275177001953125000,
         46.29823736463859518153185490518808364868164062500000,   42.08085846212114944364657276310026645660400390625000,   37.70511987293237865515038720332086086273193359375000,   33.15646247594444417927661561407148838043212890625000,   28.41538941226311010268545942381024360656738281250000,   23.45479094153210652962116000708192586898803710937500,   18.23484983658462965649960096925497055053710937500000,   12.69198038133320061149333923822268843650817871093750,    6.70930434025829836031107333838008344173431396484375,    0.00000000000000000000000000000000000000000000000000,
        114.03421178146169268075027503073215484619140625000000,  115.49350873039638543104956625029444694519042968750000,  116.16804656958889552242908393964171409606933593750000,  116.35578657680875380719953682273626327514648437500000,  116.18164810561199828953249379992485046386718750000000,  115.71532693295803539967891993001103401184082031250000,  115.00147011653243112050404306501150131225585937500000,  114.07112785885168193544814130291342735290527343750000,  112.94709698464136238271748879924416542053222656250000,  111.64676225749649063345714239403605461120605468750000,
        110.18375073059114299667271552607417106628417968750000,  108.56896143132088639049470657482743263244628906250000,  106.81123965925374363905575592070817947387695312500000,  104.91783635465496615779557032510638236999511718750000,  102.89473085568745602813578443601727485656738281250000,  100.74686310561446589417755603790283203125000000000000,   98.47830359169124392337835161015391349792480468750000,   96.09237898948062195358943426981568336486816406250000,   93.59176524002876362828828860074281692504882812500000,   90.97855584445318299913196824491024017333984375000000,
         88.25431056742904445400199620053172111511230468750000,   85.42008795546466615178360370919108390808105468750000,   82.47646377330303835151426028460264205932617187500000,   79.42353643535656715357617940753698348999023437500000,   76.26091961358754645061708288267254829406738281250000,   72.98772132078988761350046843290328979492187500000000,   69.60250778254022918645205209031701087951660156250000,   66.10324918220183576522686053067445755004882812500000,   62.48724269420567623001261381432414054870605468750000,   58.75100580351832491032837424427270889282226562500000,
         54.89012922835880914362860494293272495269775390625000,   50.89907290930269567752475268207490444183349609375000,   46.77087881578651007430380559526383876800537109375000,   42.49675746505901940963667584583163261413574218750000,   38.06547426167018244314022012986242771148681640625000,   33.46240203973819404836831381544470787048339843750000,   28.66798147958512998911828617565333843231201171875000,   23.65504536533591206648452498484402894973754882812500,   18.38372345931104590022187039721757173538208007812500,   12.79038104605596792850974452449008822441101074218750,
          6.75809450442773052003531120135448873043060302734375,    0.00000000000000000000000000000000000000000000000000,  117.77188139974505531881732167676091194152832031250000,  119.23669642003666524487925926223397254943847656250000,  119.91777160433275639661587774753570556640625000000000,  120.11299805470980572863481938838958740234375000000000,  119.94726219480934048533526947721838951110839843750000,  119.49024918670430395195580786094069480895996093750000,  118.78661046058377337431011255830526351928710937500000,  117.86741171604134592598711606115102767944335937500000,
        116.75547429609400751360226422548294067382812500000000,  115.46821538203036539016466122120618820190429687500000,  114.01930182151915005306364037096500396728515625000000,  112.41967969262935866936459206044673919677734375000000,  110.67824877999414923124277265742421150207519531250000,  108.80232236921092692227830411866307258605957031250000,  106.79795065801386044768150895833969116210937500000000,  104.67015384769811703336017671972513198852539062500000,  102.42309321598659721530566457659006118774414062500000,  100.06019818085174222233035834506154060363769531250000,
         97.58426113605924001603852957487106323242187500000000,   94.99750791726553700300428317859768867492675781250000,   92.30164919113671828654332784935832023620605468750000,   89.49791630985923518437630264088511466979980468750000,   86.58708391942691662279685260728001594543457031250000,   83.56948065162023908669652882963418960571289062500000,   80.44498843108701180426578503102064132690429687500000,   77.21303018758834468826535157859325408935546875000000,   73.87254498723982010233157780021429061889648437500000,   70.42194868534818397165508940815925598144531250000000,
         66.85907702756016135481331730261445045471191406250000,   63.18110649515215726523820194415748119354248046875000,   59.38444580096968650195776717737317085266113281250000,   55.46458728240330060543783474713563919067382812500000,   51.41590159938099446890191757120192050933837890625000,   47.23134944496703724325925577431917190551757812500000,   42.90206712915444597911118762567639350891113281250000,   38.41675211638797549085211358033120632171630859375000,   33.76071487687092087526252726092934608459472656250000,   28.91433870526184790605839225463569164276123046875000,
         23.85040345868698707931798708159476518630981445312500,   18.52898997786421375622012419626116752624511718750000,   12.88641823425290944271637272322550415992736816406250,    6.80572255341698539865546990768052637577056884765625,    0.00000000000000000000000000000000000000000000000000,  121.53308151543863857568794628605246543884277343750000,  123.00325704343613608671148540452122688293457031250000,  123.69067213218903589222463779151439666748046875000000,  123.89314882401878037399001186713576316833496093750000,  123.73554030736524111944163450971245765686035156250000,
        123.28751992120558611532032955437898635864257812500000,  122.59374158607398896947415778413414955139160156250000,  121.68528404266113795983983436599373817443847656250000,  120.58499015620384398062014952301979064941406250000000,  119.31030598355029326285148272290825843811035156250000,  117.87493404482295034085836960002779960632324218750000,  116.28986270993469531731534516438841819763183593750000,  114.56404077410924458035879069939255714416503906250000,  112.70483757596034024572873022407293319702148437500000,  110.71836693380481619897182099521160125732421875000000,
        108.60972096066504377631645184010267257690429687500000,  106.38314207133923616765969200059771537780761718750000,  104.04215121684583778005617205053567886352539062500000,  101.58964416684675313717889366671442985534667968750000,   99.02796375489810998260509222745895385742187500000000,   96.35895345509749176926561631262302398681640625000000,   93.58399593473784250363678438588976860046386718750000,   90.70403900871916391679405933246016502380371093750000,   87.71961051131717113094055093824863433837890625000000,   84.63082287018917782006610650569200515747070312500000,
         81.43736752301830961187079083174467086791992187500000,   78.13849868199022807857545558363199234008789062500000,   74.73300524945624090378260007128119468688964843750000,   71.21916882986467101090966025367379188537597656250000,   67.59470464539501222134276758879423141479492187500000,   63.85668055981133761633827816694974899291992187500000,   60.00140704643910538607087801210582256317138671875000,   56.02428729154834741166268941015005111694335937500000,   51.91961079774520015917005366645753383636474609375000,   47.68026416053742622125355410389602184295654296875000,
         43.29731585113643177464837208390235900878906250000000,   38.75940106386912020752788521349430084228515625000000,   34.05177297028762950503733009099960327148437500000000,   29.15476201893021368505287682637572288513183593750000,   24.04109922063762638799744308926165103912353515625000,   18.67082017330343290950622758828103542327880859375000,   12.98020286074839724221874348586425185203552246093750,    6.85224256905187800725798297207802534103393554687500,    0.00000000000000000000000000000000000000000000000000,  125.31727114935689826324960449710488319396972656250000,
        126.79265773821531126941408729180693626403808593750000,  127.48622590974368051774945342913269996643066406250000,  127.69572978496634618750249501317739486694335937500000,  127.54598912643260177901538554579019546508789062500000,  127.10666442119855901182745583355426788330078125000000,  126.42241043235119946075428742915391921997070312500000,  125.52431677321050074169761501252651214599609375000000,  124.43524517847181698471104027703404426574707031250000,  123.17266744333912242836959194391965866088867187500000,  121.75031811348966925834247376769781112670898437500000,
        120.17922364863811424129380611702799797058105468750000,  118.46837703321760670860385289415717124938964843750000,  116.62519813448837169289618032053112983703613281250000,  114.65585805854794898550608195364475250244140625000000,  112.56551355781735423988720867782831192016601562500000,  110.35847981005390749942307593300938606262207031250000,  108.03835962246603230596520006656646728515625000000000,  105.60814091125330094200762687250971794128417968750000,  103.07027041385639165582688292488455772399902343750000,  100.42670906026503985231101978570222854614257812500000,
         97.67897272531180874466372188180685043334960937500000,   94.82816089080233723507262766361236572265625000000000,   91.87497487106125504396914038807153701782226562500000,   88.81972657290590689171949634328484535217285156250000,   85.66233818424916535150259733200073242187500000000000,   82.40233264711058325246995082125067710876464843750000,   79.03881420993985784662072546780109405517578125000000,   75.57043770523115711057471344247460365295410156250000,   71.99536437833842228428693488240242004394531250000000,   68.31120098369963500317680882290005683898925781250000,
         64.51491728244980095041682943701744079589843750000000,   60.60273472227629554254235699772834777832031250000000,   56.56997544746483441713280626572668552398681640625000,   52.41085496934783094502563471905887126922607421875000,   48.11819214282444789887449587695300579071044921875000,   43.68299325987392478509718785062432289123535156250000,   39.09383630048893820685407263226807117462158203125000,   34.33592166847988380595779744908213615417480468750000,   29.38953103909457453823961259331554174423217773437500,   24.22735024845097839829577424097806215286254882812500,
         18.80937297044620137853598862420767545700073242187500,   13.07183820898510973051998007576912641525268554687500,    6.89770494312863569064120383700355887413024902343750,    0.00000000000000000000000000000000000000000000000000,  129.12393363912721611086453776806592941284179687500000,  130.60438936485934391384944319725036621093750000000000,  131.30393361038915145400096662342548370361328125000000,  131.52025372715993967176473233848810195922851562500000,  131.37813595800301413873967248946428298950195312500000,  130.94722706418789925919554661959409713745117187500000,
        130.27218120714545079863455612212419509887695312500000,  129.38409695434128821034391876310110092163085937500000,  128.30585255534487032491597346961498260498046875000000,  127.05494275743465948380617192015051841735839843750000,  125.64513089550477786815463332459330558776855468750000,  124.08747779645949549376382492482662200927734375000000,  122.39101637269315858702611876651644706726074218750000,  120.56321215150005343730299500748515129089355468750000,  118.61028796406132812535361154004931449890136718750000,  116.53745883783015813150996109470725059509277343750000,
        114.34910541486256363441498251631855964660644531250000,  112.04890396222454285179992439225316047668457031250000,  109.63992484686433215301804011687636375427246093750000,  107.12470746446948055563552770763635635375976562500000,  104.50531709279678693746973294764757156372070312500000,  101.78338745036130319476797012612223625183105468750000,   98.96015156748373442496813368052244186401367187500000,   96.03646272697001506912783952429890632629394531250000,   93.01280658384901300905767129734158515930175781250000,   89.88930504482105732222407823428511619567871093750000,
         86.66571201723567696717509534209966659545898437500000,   83.34140067326865164432092569768428802490234375000000,   79.91534136696490975282358704134821891784667968750000,   76.38606873106159866892994614318013191223144531250000,   72.75163568835748151286679785698652267456054687500000,   69.00955102336577340338408248499035835266113281250000,   65.15669559323198711808799998834729194641113281250000,   61.18920991547661714093919727019965648651123046875000,   57.10234224615479092790337745100259780883789062500000,   52.89024045117506034330290276557207107543945312500000,
         48.54566129432795662523858482018113136291503906250000,   44.05955393501750450013787485659122467041015625000000,   39.42044366150012280058945179916918277740478515625000,   34.61348217508560054511690395884215831756591796875000,   29.61890604234817558904069301206618547439575195312500,   24.40935923713887945041278726421296596527099609375000,   18.94479651122276209207484498620033264160156250000000,   13.16142061565785503773895470658317208290100097656250,    6.94215670569946929902016563573852181434631347656250,    0.00000000000000000000000000000000000000000000000000,
        132.95257503561629164323676377534866333007812500000000,  134.43796495883810848681605421006679534912109375000000,  135.14331735453160376891901250928640365600585937500000,  135.36625396199082160819671116769313812255859375000000,  135.23152749235279657114006113260984420776367187500000,  134.80877024138152364685083739459514617919921875000000,  134.14263450197452698375855106860399246215820312500000,  133.26422609528944462908839341253042221069335937500000,  132.19643769103768704553658608347177505493164062500000,  130.95678450751370291982311755418777465820312500000000,
        129.55905578862564198061591014266014099121093750000000,  128.01434342038632507865258958190679550170898437500000,  126.33171646443102531520707998424768447875976562500000,  124.51868180207580394380784127861261367797851562500000,  122.58150908464496353644790360704064369201660156250000,  120.52546602200679615179979009553790092468261718750000,  118.35499233118862605351750971749424934387207031250000,  116.07383041939264955999533412978053092956542968750000,  113.68512469003437104220211040228605270385742187500000,  111.19149748582549364073202013969421386718750000000000,
        108.59510717327455608938180375844240188598632812500000,  105.89769219477510375782003393396735191345214843750000,  103.10060375504289709169825073331594467163085937500000,  100.20482897790303411511558806523680686950683593750000,   97.21100574696610863156820414587855339050292968750000,   94.11942994957105668163421796634793281555175781250000,   90.93005542049513678648509085178375244140625000000000,   87.64248648531652463589125545695424079895019531250000,   84.25596259199559767694154288619756698608398437500000,   80.76933404938095861780311679467558860778808593750000,
         77.18102730849162185222667176276445388793945312500000,   73.48899745093362412262649741023778915405273437500000,   69.69066447524114948919304879382252693176269531250000,   65.78282841686940685121953720226883888244628906250000,   61.76155600507625820227985968813300132751464843750000,   57.62202794284049645057166344486176967620849609375000,   53.35833009046255170915173948742449283599853515625000,   48.96316215848698760737534030340611934661865234375000,   44.42742068833225488333482644520699977874755859375000,   39.73958233444804477585421409457921981811523437500000,
         34.88475375356848928731778869405388832092285156250000,   29.84312970991117452967955614440143108367919921875000,   24.58731531128054825785511638969182968139648437500000,   19.07722910917899739047243201639503240585327148437500,   13.24904008023083257228336151456460356712341308593750,    6.98564181763920810652734871837310492992401123046875,    0.00000000000000000000000000000000000000000000000000,  136.80272263732635451560781802982091903686523437500000,  138.29291831437924997771915514022111892700195312500000,  139.00391936232793455019418615847826004028320312500000,
        139.23328306533610998485528398305177688598632812500000,  139.10572865998648239838075824081897735595703125000000,  138.69087335317266251877299509942531585693359375000000,  138.03336645724726849948638118803501129150390625000000,  137.16431953732362103437480982393026351928710937500000,  136.10663781569988373121304903179407119750976562500000,  134.87785476701279208100459072738885879516601562500000,  133.49178297403813076016376726329326629638671875000000,  131.95954243363377145215054042637348175048828125000000,  130.29023499536035046730830799788236618041992187500000,
        128.49140507336005612160079181194305419921875000000000,  126.56936479861477096164890099316835403442382812500000,  124.52942963007352261683990946039557456970214843750000,  122.37609274342860032902535749599337577819824218750000,  120.11315627670221317657706094905734062194824218750000,  117.74383133301998327624460216611623764038085937500000,  115.27081477318711222324054688215255737304687500000000,  112.69634832854320904971245909109711647033691406250000,  110.02226389525885963394102873280644416809082031250000,  107.25001772289888890554721001535654067993164062500000,
        104.38071539357579808893206063657999038696289062500000,  101.41512888436604100661497795954346656799316406250000,   98.35370653678140229203563649207353591918945312500000,   95.19657636874478612298844382166862487792968750000000,   91.94354281580874044266238342970609664916992187500000,   88.59407664452780295505363028496503829956054687500000,   85.14729740769071497652475954964756965637207031250000,   81.60194736913432222991104936227202415466308593750000,   77.95635526364411305166868260130286216735839843750000,   74.20838750140154616019572131335735321044921875000000,
         70.35538336455316255069192266091704368591308593750000,   66.39406919733733047905843704938888549804687500000000,   62.32044426558674388161307433620095252990722656250000,   58.12962734967926792251091683283448219299316406250000,   53.81564733368846020766795845702290534973144531250000,   49.37115138177968276522733503952622413635253906250000,   44.78698746826473353621622663922607898712158203125000,   40.05158726525881007773932651616632938385009765625000,   35.15001568531794617911145905964076519012451171875000,   30.06242868108142118899195338599383831024169921875000,
         24.76139521116294517355527204927057027816772460937500,   19.20680010058742226419781218282878398895263671875000,   13.33478080901900675314664113102480769157409667968750,    7.02820143205800462737897760234773159027099609375000,    0.00000000000000000000000000000000000000000000000000,  140.67392364823427897135843522846698760986328125000000,  142.16880268620320748595986515283584594726562500000000,  142.88530071650168906671751756221055984497070312500000,  143.12091171980523540696594864130020141601562500000000,  143.00032157357003370634629391133785247802734375000000,
        142.59313187355303398362593725323677062988281250000000,  141.94398797512380383523122873157262802124023437500000,  141.08400584490360074596537742763757705688476562500000,  140.03610158943189389901817776262760162353515625000000,  138.81782495825729029093054123222827911376953125000000,  137.44300957375159555340360384434461593627929687500000,  135.92280090883124898937239777296781539916992187500000,  134.26633060215394266379007603973150253295898437500000,  132.48117719902370481577236205339431762695312500000000,  130.57369145613478167433640919625759124755859375000000,
        128.54923221405539379702531732618808746337890625000000,  126.41234115130095005952171050012111663818359375000000,  124.16687450141505166811839444562792778015136718750000,  121.81610364173009486421506153419613838195800781250000,  119.36279260046758565749769331887364387512207031250000,  116.80925803280143782103550620377063751220703125000000,  114.15741555422360420379845891147851943969726562500000,  111.40881517978864678752870531752705574035644531250000,  108.56466781216629158279829425737261772155761718750000,  105.62586413189328027328883763402700424194335937500000,
        102.59298679304122003941301954910159111022949218750000,   99.46631646438356710859807208180427551269531250000000,   96.24583194191284007956710411235690116882324218750000,   92.93120426259910971111821709200739860534667968750000,   89.52178444353792485799203859642148017883300781250000,   86.01658412528556141296576242893934249877929687500000,   82.41424797681732172804913716390728950500488281250000,   78.71301617276223794306133640930056571960449218750000,   74.91067450919838677236839430406689643859863281250000,   71.00448867132186592243670020252466201782226562500000,
         66.99111762703208228231233078986406326293945312500000,   62.86649880019569991418393328785896301269531250000000,   58.62569406978440866851087776012718677520751953125000,   54.26267984305790292864912771619856357574462890625000,   49.77005478187675180379301309585571289062500000000000,   45.13862193869116623545778566040098667144775390625000,   40.35677129796720663534870254807174205780029296875000,   35.40952901360976312616912764497101306915283203125000,   30.27701493869775362099971971474587917327880859375000,   24.93176435197359808171313488855957984924316406250000,
         19.33363060532829535986820701509714126586914062500000,   13.41872170208896619669758365489542484283447265625000,    7.06987412845857221554979332722723484039306640625000,    0.00000000000000000000000000000000000000000000000000,  144.56574394634486679933615960180759429931640625000000,  146.06518959720094130716461222618818283081054687500000,  146.78704022426623510000354144722223281860351562500000,  147.02872764699259278131648898124694824218750000000000,  146.91490454819870592473307624459266662597656250000000,  146.51515647821929633209947496652603149414062500000000,
        145.87412397795776541897794231772422790527343750000000,  145.02292622021701617995859123766422271728515625000000,  143.98448870261756837862776592373847961425781250000000,  142.77637567304179810889763757586479187011718750000000,  141.41243973160592872773122508078813552856445312500000,  139.90384946655598241704865358769893646240234375000000,  138.25976362277978637393971439450979232788085937500000,  136.48779183954525251465383917093276977539062500000000,  134.59432006644897228397894650697708129882812500000000,  132.58474664607692261597549077123403549194335937500000,
        130.46365737089894309974624775350093841552734375000000,  128.23495759607345689801150001585483551025390625000000,  125.90197332244368055853556143119931221008300781250000,  123.46752930690303173832944594323635101318359375000000,  120.93400976630447019033454125747084617614746093750000,  118.30340558444281384709029225632548332214355468750000,  115.57735079859945415137190138921141624450683593750000,  112.75715034485608612158102914690971374511718750000000,  109.84380046255925833520450396463274955749511718750000,  106.83800272215354709715029457584023475646972656250000,
        103.74017229610355173008429119363427162170410156250000,  100.55044080354905133845022646710276603698730468750000,   97.26865379782087472904095193371176719665527343750000,   93.89436270803930995043629081919789314270019531250000,   90.42681076810957563338888576254248619079589843750000,   86.86491214168410124329966492950916290283203125000000,   83.20722304572196037497633369639515876770019531250000,   79.45190314015965782346029300242662429809570312500000,   75.59666471574516322107228916138410568237304687500000,   71.63870616597952789561531972140073776245117187500000,
         67.57462469517182057643367443233728408813476562500000,   63.40030089850934302830864908173680305480957031250000,   59.11074424577082453424736740998923778533935546875000,   54.69988270490430437575923861004412174224853515625000,   50.16027007469638476777618052437901496887207031250000,   45.48266777490444212617148878052830696105957031250000,   40.65542708280097627948634908534586429595947265625000,   35.66353810098577525877772131934762001037597656250000,   30.48708704798434609983814880251884460449218750000000,   25.09857777202318729337093827780336141586303710937500,
         19.45783420878760594519008009228855371475219726562500,   13.50093679004417701605689217103645205497741699218750,    7.11069612297882791551728587364777922630310058593750,    0.00000000000000000000000000000000000000000000000000,  148.47776695177304873141110874712467193603515625000000,  149.98166774147284741047769784927368164062500000000000,  150.70873336866512204323953483253717422485351562500000,  150.95633462125837809253425803035497665405273437500000,  150.84909119310790970303060021251440048217773437500000,  150.45657223151152948048547841608524322509765625000000,
        149.82341271001959626119059976190328598022460937500000,  148.98073394199616359401261433959007263183593750000000,  147.95146947745786292216507717967033386230468750000000,  146.75319646648389948495605494827032089233398437500000,  145.39978463382701079353864770382642745971679687500000,  143.90242356271937751444056630134582519531250000000000,  142.27029669778480069908255245536565780639648437500000,  140.51104205286284809517383109778165817260742187500000,  138.63107770360389281449897680431604385375976562500000,  136.63583803737571997771738097071647644042968750000000,
        134.52994906061928759299917146563529968261718750000000,  132.31736084403902964368171524256467819213867187500000,  130.00144902342171349118871148675680160522460937500000,  127.58509342102857431200391147285699844360351562500000,  125.07073936571732986067218007519841194152832031250000,  122.46044563788728964937035925686359405517578125000000,  119.75592183764112519384070765227079391479492187500000,  116.95855718379507948156970087438821792602539062500000,  114.06944218076974095765763195231556892395019531250000,  111.08938416484535594008775660768151283264160156250000,
        108.01891741070821240100485738366842269897460937500000,  104.85830820872619995043351082131266593933105468750000,  101.60755508699250526660762261599302291870117187500000,   98.26638412868874183914158493280410766601562500000000,   94.83423910522689936897222651168704032897949218750000,   91.31026588831348078656446887180209159851074218750000,   87.69329029474481274064601166173815727233886718750000,   83.98178812346036181679664878174662590026855468750000,   80.17384561816544419343699701130390167236328125000000,   76.26710786022917432092071976512670516967773437500000,
         72.25871155578469995361956534907221794128417968750000,   68.14519715119237730505119543522596359252929687500000,   63.92239289821348080522511736489832401275634765625000,   59.58525988858333022335500572808086872100830078125000,   55.12768128435615011539994156919419765472412109375000,   50.54216930508160743329426622949540615081787109375000,   45.81944671337863894677866483107209205627441406250000,   40.94782878215723087578226113691926002502441406250000,   35.91227202355151604251659591682255268096923828125000,   30.69283126703487241115908545907586812973022460937500,
         25.26198098367542144160324824042618274688720703125000,   19.57951757441557205652316042687743902206420898437500,   13.58149562676142885209173982730135321617126464843750,    7.15070145759252628891999847837723791599273681640625,    0.00000000000000000000000000000000000000000000000000,  152.40959258449734647911100182682275772094726562500000,  153.91784197339185880082368385046720504760742187500000,  154.64999134075051756553875748068094253540039062500000,  154.90335155748746842618857044726610183715820312500000,  154.80250956861920030860346741974353790283203125000000,
        154.41701782768726047834206838160753250122070312500000,  153.79150508015649734261387493461370468139648437500000,  152.95709382895054773143783677369356155395507812500000,  151.93672447436148331689764745533466339111328125000000,  150.74798563190657318955345544964075088500976562500000,  149.40476248193905917105439584702253341674804687500000,  147.91826369384494910264038480818271636962890625000000,  146.29769524803592162243148777633905410766601562500000,  144.55072109162489368827664293348789215087890625000000,  142.68378867866894665894506033509969711303710937500000,
        140.70236535119141763061634264886379241943359375000000,  138.61111385294381648236594628542661666870117187500000,  136.41402505380042953220254275947809219360351562500000,  134.11451980444931564306898508220911026000976562500000,  131.71552799193514715625497046858072280883789062500000,  129.21955038339851284945325460284948348999023437500000,  126.62870719707242983531614299863576889038085937500000,  123.94477621618784723978023976087570190429687500000000,  121.16922247582216698447155067697167396545410156250000,  118.30322098851021905829838942736387252807617187500000,
        115.34767355702017255225655389949679374694824218750000,  112.30322040268688965625187847763299942016601562500000,  109.17024708109900643648870754987001419067382812500000,  105.94888693908798416032368550077080726623535156250000,  102.63901916857159335449978243559598922729492187500000,   99.24026231711071943664137506857514381408691406250000,   95.75196290552229072545742383226752281188964843750000,   92.17317856097477601906575728207826614379882812500000,   88.50265477630688337740139104425907135009765625000000,   84.73879402092080681541119702160358428955078125000000,
         80.87961540923890879639657214283943176269531250000000,   76.92270240943055625848501222208142280578613281250000,   72.86513503850621020774269709363579750061035156250000,   68.70340146339805187380989082157611846923828125000000,   64.43328161744592819104582304134964942932128906250000,   60.04969184213697275254162377677857875823974609375000,   55.54647377231179916634573601186275482177734375000000,   50.91610101909744656722978106699883937835693359375000,   46.14926038643901051727880258113145828247070312500000,   41.23423359969314816453334060497581958770751953125000,
         36.15594582230254872001751209609210491180419921875000,   30.89442254459304137981234816834330558776855468750000,   25.42211073873270876788410532753914594650268554687500,   19.69878099524238734829850727692246437072753906250000,   13.66046364330519935492702643387019634246826171875000,    7.18992217074580786118076503043994307518005371093750,    0.00000000000000000000000000000000000000000000000000,  156.36083630307877001541783101856708526611328125000000,  157.87333237442643962822330649942159652709960937500000,  158.61044014498455112516239751130342483520507812500000,
        158.86941166608858111430890858173370361328125000000000,  158.77480140272371045284671708941459655761718750000000,  158.39614488238208878101431764662265777587890625000000,  157.77806404305439968993596266955137252807617187500000,  156.95168172773389869689708575606346130371093750000000,  155.93994410589138510658813174813985824584960937500000,  154.76044996286765353943337686359882354736328125000000,  153.42709842835088807078136596828699111938476562500000,  151.95111553570615114949760027229785919189453125000000,  150.34172785071504563347843941301107406616210937500000,
        148.60662305662063431554997805505990982055664062500000,  146.75227551763771316473139449954032897949218750000000,  144.78418276059352365336962975561618804931640625000000,  142.70704115789419574866769835352897644042968750000000,  140.52487888628493806209007743746042251586914062500000,  138.24115808164847862826718483120203018188476562500000,  135.85885426430579059342562686651945114135742187500000,  133.38051862915662582054210361093282699584960937500000,  130.80832714986709675031306687742471694946289062500000,  128.14411932639811197987000923603773117065429687500000,
        125.38942862365152564052550587803125381469726562500000,  122.54550608980864012664824258536100387573242187500000,  119.61333823181091418064170284196734428405761718750000,  116.59365991348883540013048332184553146362304687500000,  113.48696279574841128123807720839977264404296875000000,  110.29349963423476310708792880177497863769531250000000,  107.01328457002128402564267162233591079711914062500000,  103.64608937825306611557607538998126983642578125000000,  100.19143546453939563889434793964028358459472656250000,   96.64858120476476699423074023798108100891113281250000,
         93.01650399370960542455577524378895759582519531250000,   89.29387607907344204249966423958539962768554687500000,   85.47903287895171331456367624923586845397949218750000,   81.56993196677370860925293527543544769287109375000000,   77.56410018859351396258716704323887825012207031250000,   73.45856534430167528171295998618006706237792968750000,   69.24976734045840487397072138264775276184082031250000,   64.93344141433505001259618438780307769775390625000000,   60.50446243085967523711588000878691673278808593750000,   55.95663346388602832348624360747635364532470703125000,
         51.28239221035740058596275048330426216125488281250000,   46.47239196845403341740166069939732551574707031250000,   41.51488315414674445946729974821209907531738281250000,   36.39476162875001108432115870527923107147216796875000,   31.09202541859960433612286578863859176635742187500000,   25.57909571840242790585762122645974159240722656250000,   19.81571889151167908948991680517792701721191406250000,   13.73790246753689636705075827194377779960632324218750,    7.22838845157360410098590364214032888412475585937500,    0.00000000000000000000000000000000000000000000000000,
        160.33112821663090130641649011522531509399414062500000,  161.84777339039467847214837092906236648559570312500000,  162.58971977109138151718070730566978454589843750000000,  162.85416166921015701518626883625984191894531250000000,  162.76562136225109611586958635598421096801757812500000,  162.39361727043490191135788336396217346191406250000000,  161.78276401681651464059541467577219009399414062500000,  160.96418402506040479238436091691255569458007812500000,  159.96082826022055201065086293965578079223632812500000,  158.79030450715470124123385176062583923339843750000000,
        157.46652448293775705678854137659072875976562500000000,  156.00073002792899501400825101882219314575195312500000,  154.40216653140885227912804111838340759277343750000000,  152.67854343064311706257285550236701965332031250000000,  150.83635977713106512965168803930282592773437500000000,  148.88114079305486825433035846799612045288085937500000,  146.81761369193151267609209753572940826416015625000000,  144.64984083378894297311489935964345932006835937500000,  142.38132213467682163354766089469194412231445312500000,  140.01507480753801360151555854827165603637695312500000,
        137.55369603403320866163994651287794113159179687500000,  134.99941252445015038574638310819864273071289062500000,  132.35411980579493729237583465874195098876953125000000,  129.61941329995539717856445349752902984619140625000000,  126.79661269859796846048993756994605064392089843750000,  123.88678073525559852896549273282289505004882812500000,  120.89073714935049963514757109805941581726074218750000,  117.80906839881733105812600115314126014709472656250000,  114.64213348447003681940259411931037902832031250000000,  111.39006608321615487966482760384678840637207031250000,
        108.05277303510986541823513107374310493469238281250000,  104.62992907916060403294977732002735137939453125000000,  101.12096757305954497496713884174823760986328125000000,   97.52506674949152909448457648977637290954589843750000,   93.84113084030251172862335806712508201599121093750000,   90.06776511783604632910282816737890243530273437500000,   86.20324352952687263496045488864183425903320312500000,   82.24546709195828952942974865436553955078125000000000,   78.19191049475780630473309429362416267395019531250000,   74.03955333399031246699451003223657608032226562500000,
         69.78479087292568294742522994056344032287597656250000,   65.42331692162184708649874664843082427978515625000000,   60.94996783023165676240751054137945175170898437500000,   56.35851080176858829418051755055785179138183593750000,   51.64135006812428230205114232376217842102050781250000,   46.78910765638913460406911326572299003601074218750000,   41.79000471647589165513636544346809387207031250000000,   36.62890967972570166466539376415312290191650390625000,   31.28579482713213977262967091519385576248168945312500,   25.73305715659640569015209621284157037734985351562500,
         19.93042026063065463858947623521089553833007812500000,   13.81387021333434006464813137426972389221191406250000,    7.26612877955645064531609023106284439563751220703125,    0.00000000000000000000000000000000000000000000000000,  164.32011226319517049887508619576692581176757812500000,  165.84081303263323547980689909309148788452148437500000,  166.58748342632233629956317599862813949584960937500000,  166.85726107277977803278190549463033676147460937500000,  166.77463637406063412527146283537149429321289062500000,  166.40911050656399083891301415860652923583984375000000,
        165.80529033465057864304981194436550140380859375000000,  164.99429718337262329441728070378303527832031250000000,  163.99908593547169743942504283040761947631835937500000,  162.83727231652923705951252486556768417358398437500000,  161.52277939733903622254729270935058593750000000000000,  160.06686341485382740756904240697622299194335937500000,  158.47878698694194099516607820987701416015625000000000,  156.76627951274198835562856402248144149780273437500000,  154.93586272434379225160228088498115539550781250000000,  152.99308729615296442716498859226703643798828125000000,
        150.94270877644100892212009057402610778808593750000000,  148.78882090670461479930963832885026931762695312500000,  146.53495824740713260325719602406024932861328125000000,  144.18417618862250151323678437620401382446289062500000,  141.73911394904675375983060803264379501342773437500000,  139.20204452666993688580987509340047836303710937500000,  136.57491444976957950530049856752157211303710937500000,  133.85937540137922496796818450093269348144531250000000,  131.05680923834009377060283441096544265747070312500000,  128.16834752372710681811440736055374145507812500000000,
        125.19488639058542389648209791630506515502929687500000,  122.13709732302113764035311760380864143371582031250000,  118.99543425515170724793279077857732772827148437500000,  115.77013723282655632829118985682725906372070312500000,  112.46123274472924435940512921661138534545898437500000,  109.06853069789089261121262097731232643127441406250000,  105.59161787793958353631751378998160362243652343750000,  102.02984758627862049706891411915421485900878906250000,   98.38232497276568722099909791722893714904785156250000,   94.64788736790467282844474539160728454589843750000000,
         90.82507864191831004063715226948261260986328125000000,   86.91211624900223853273928398266434669494628906250000,   82.90684910846181310262181796133518218994140625000000,   78.80670376120725961754942545667290687561035156250000,   74.60861521145027097645652247592806816101074218750000,   70.30893734334057398882578127086162567138671875000000,   65.90332549720086774414085084572434425354003906250000,   61.38658019458972603388247080147266387939453125000000,   56.75243521315276495897705899551510810852050781250000,   51.99326355102085273074408178217709064483642578125000,
         47.09965800438733651844813721254467964172363281250000,   42.05981232635262045960189425386488437652587890625000,   36.85856923422820585756198852322995662689208984375000,   31.47587684180334832717562676407396793365478515625000,   25.88410940415274197334838390816003084182739257812500,   20.04296908481914840649551479145884513854980468750000,   13.88842174282511976457499258685857057571411132812500,    7.30317005123680029754495990346185863018035888671875,    0.00000000000000000000000000000000000000000000000000,  168.32744544842765321845945436507463455200195312500000,
        169.85211213727404810924781486392021179199218750000000,  170.60339682274099004644085653126239776611328125000000,  170.87838148952599226504389662295579910278320312500000,  170.80152499212707084552675951272249221801757812500000,  170.44231116566288619651459157466888427734375000000000,  169.84533872854666469720541499555110931396484375000000,  169.04172729931212870724266394972801208496093750000000,  168.05443488586553257846389897167682647705078125000000,  166.90108419518054461150313727557659149169921875000000,  165.59560853240347455539449583739042282104492187500000,
        164.14927725108600498060695827007293701171875000000000,  162.57136875101795681075600441545248031616210937500000,  160.86963076934441119192342739552259445190429687500000,  159.05060590308283963167923502624034881591796875000000,  157.11986825279316803971596527844667434692382812500000,  155.08219944219882791003328748047351837158203125000000,  152.94172207426004206354264169931411743164062500000000,  150.70200254040386766973824705928564071655273437500000,  148.36613126207376467391441110521554946899414062500000,  145.93678597083328440930927172303199768066406250000000,
        143.41628199507184149297245312482118606567382812500000,  140.80661240875909356873307842761278152465820312500000,  138.10948012441224364010849967598915100097656250000000,  135.32632346297233993936970364302396774291992187500000,  132.45833633406942908550263382494449615478515625000000,  129.50648386312542470477637834846973419189453125000000,  126.47151407466694195136369671672582626342773437500000,  123.35396606183580558990797726437449455261230468750000,  120.15417492445118341493071056902408599853515625000000,  116.87227363010329383996577234938740730285644531250000,
        113.50819183497853259723342489451169967651367187500000,  110.06165158489291400201182113960385322570800781250000,  106.53215969391986561731755500659346580505371093750000,  102.91899645873554902664182009175419807434082031250000,   99.22120020001601403691893210634589195251464843750000,   95.43754691296858538862579734995961189270019531250000,   91.56652403659258254720043623819947242736816406250000,   87.60629698547569432776072062551975250244140625000000,   83.55466658393766010703984647989273071289062500000000,   79.40901483118875603395281359553337097167968750000000,
         75.16623539921447161304968176409602165222167968750000,   70.82264374634713988143630558624863624572753906250000,   66.37385942551006223766307812184095382690429687500000,   61.81464957159139572695494280196726322174072265625000,   57.13871676487859474491415312513709068298339843750000,   52.33840480689521257318119751289486885070800781250000,   47.40427912936282695000045350752770900726318359375000,   42.32450780189157057975535281002521514892578125000000,   37.08390940346051678488947800360620021820068359375000,   31.66240933235810217638572794385254383087158203125000,
         26.03236044057927500716687063686549663543701171875000,   20.15344470114374431091164296958595514297485351562500,   13.96160890460070369556433433899655938148498535156250,    7.33953769540767453349872084800153970718383789062500,    0.00000000000000000000000000000000000000000000000000,  172.35279713916278865326603408902883529663085937500000,  173.88134367744416408640972804278135299682617187500000,  174.63713751470089619033387862145900726318359375000000,  174.91720600863371259947598446160554885864257812500000,  174.84597680678464826087292749434709548950195312500000,
        174.49291633975002469014725647866725921630859375000000,  173.90261484293259286459942813962697982788085937500000,  173.10618968414681262402154970914125442504882812500000,  172.12660128025288486242061480879783630371093750000000,  170.98147844919000704067002516239881515502929687500000,  169.68476371316995710003538988530635833740234375000000,  168.24773837864668735164741519838571548461914062500000,  166.67969531263113935892761219292879104614257812500000,  164.98839911589595885743619874119758605957031250000000,  163.18041160401497791099245660006999969482421875000000,
        161.26132846949391819180164020508527755737304687500000,  159.23595536996000987528532277792692184448242187500000,  157.10844149746876041717769112437963485717773437500000,  154.88238254326674336880387272685766220092773437500000,  152.56090113713619871305127162486314773559570312500000,  150.14671036943977355804236140102148056030273437500000,  147.64216436754429651045938953757286071777343750000000,  145.04929878743274684893549419939517974853515625000000,  142.36986330988898430405242834240198135375976562500000,  139.60534768328022892092121765017509460449218750000000,
        136.75700245823952627688413485884666442871093750000000,  133.82585526556050581348245032131671905517578125000000,  130.81272326532271677024255041033029556274414062500000,  127.71822222066522556360723683610558509826660156250000,  124.54277250812276633951114490628242492675781250000000,  121.28660225650982340539485448971390724182128906250000,  117.94974769896852251349628204479813575744628906250000,  114.53205072036622880204959074035286903381347656250000,  111.03315347761113685010059271007776260375976562500000,  107.45248985622738757683691801503300666809082031250000,
        103.78927339409466412689653225243091583251953125000000,  100.04248114177345030384458368644118309020996093750000,   96.21083272373917338882165495306253433227539062500000,   92.29276359563664300367236137390136718750000000000000,   88.28639112948070533093414269387722015380859375000000,   84.18947165681427691197313833981752395629882812500000,   79.99934589036598708844394423067569732666015625000000,   75.71286911914658901423536008223891258239746093750000,   71.32632105427616409087931970134377479553222656250000,   66.83528789975949280233180616050958633422851562500000,
         62.23450562864097435067378683015704154968261718750000,   57.51764765805486945282609667629003524780273437500000,   52.67703045660666560934259905479848384857177734375000,   47.70319380232522377127679646946489810943603515625000,   42.58428165466008152861832058988511562347412109375000,   37.30508990375398781225158018060028553009033203125000,   31.84552257008018472106414264999330043792724609375000,   26.17791233907299996985784673597663640975952148437500,   20.26192213802825392576778540387749671936035156250000,   14.03348075050448606759800895815715193748474121093750,
          7.37525577800975451481235722894780337810516357421875,    0.00000000000000000000000000000000000000000000000000,  176.39584840699734513691510073840618133544921875000000,  177.92819212374803328202688135206699371337890625000000,  178.68839428218674925119557883590459823608398437500000,  178.97342860811670561815844848752021789550781250000000,  178.90769189274257655597466509789228439331054687500000,  178.56063312884779747946595307439565658569335937500000,  177.97683377639995683239249046891927719116210937500000,  177.18740846524815424345433712005615234375000000000000,
        176.21531937333844552995287813246250152587890625000000,  175.07820063878060068418562877923250198364257812500000,  173.79000307493024024552141781896352767944335937500000,  172.36201888140345772626460529863834381103515625000000,  170.80355419549579210070078261196613311767578125000000,  169.12238914036359460624225903302431106567382812500000,  167.32510325420099661641870625317096710205078125000000,  165.41731215732281157215766143053770065307617187500000,  163.40384369222329041804186999797821044921875000000000,  161.28887158598823248212283942848443984985351562500000,
        159.07601854664414986473275348544120788574218750000000,  156.76843687198874022215022705495357513427734375000000,  154.36887218019441547767200972884893417358398437500000,  151.87971423658706271453411318361759185791015625000000,  149.30303774163206753655686043202877044677734375000000,  146.64063517529947944240120705217123031616210937500000,  143.89404324736395324180193711072206497192382812500000,  141.06456410849787630468199495226144790649414062500000,  138.15328218546440552927379030734300613403320312500000,  135.16107728342100813279103022068738937377929687500000,
        132.08863442749682803878386039286851882934570312500000,  128.93645077906452911520318593829870223999023437500000,  125.70483984831136581306054722517728805541992187500000,  122.39393312527927548671868862584233283996582031250000,  119.00367915951183306333405198529362678527832031250000,  115.53384002761600868325331248342990875244140625000000,  111.98398503228628442229819484055042266845703125000000,  108.35348136893382786638539982959628105163574218750000,  104.64148136892872287262434838339686393737792968750000,  100.84690577113006781928561395034193992614746093750000,
         96.96842227155428872720221988856792449951171875000000,   93.00441833442161509992729406803846359252929687500000,   88.95296688671369622625206829980015754699707031250000,   84.81178301813433506595174549147486686706542968750000,   80.57816910024880030505300965160131454467773437500000,   76.24894471407620244463032577186822891235351562500000,   71.82035625866318184762349119409918785095214843750000,   67.28795881083283347834367305040359497070312500000000,   62.64645921312869347730156732723116874694824218750000,   57.88950358056546718898971448652446269989013671875000,
         53.00938275714135272664861986413598060607910156250000,   47.99661243822436063055647537112236022949218750000000,   42.83931392045720798478214419446885585784912109375000,   37.52226174083150311844292446039617061614990234375000,   32.02533977665402176171483006328344345092773437500000,   26.32086168984642071677626518066972494125366210937500,   20.36847242182166795032571826595813035964965820312500,   14.10408373326595210528466850519180297851562500000000,    7.41034709782102396502523333765566349029541015625000,    0.00000000000000000000000000000000000000000000000000,
        180.45629141754378110817924607545137405395507812500000,  181.99235284887160446487541776150465011596679687500000,  182.75686655612790332270378712564706802368164062500000,  183.04675360637543235498014837503433227539062500000000,  182.98638029279857164510758593678474426269531250000000,  182.64517816328924482149886898696422576904296875000000,  182.06771964970366184388694819062948226928710937500000,  181.28511620767898193662404082715511322021484375000000,  180.32033118969599172487505711615085601806640625000000,  179.19100333470606756236520595848560333251953125000000,
        177.91109090323442387671093456447124481201171875000000,  176.49189602145105482122744433581829071044921875000000,  174.94273700533631199505180120468139648437500000000000,  173.27140827805067146982764825224876403808593750000000,  171.48450573849984834851056803017854690551757812500000,  169.58766342184105724300025030970573425292968750000000,  167.58572967708295209376956336200237274169921875000000,  165.48290090530448992467427160590887069702148437500000,  163.28282476701608061375736724585294723510742187500000,  160.98868093619071828470623586326837539672851562500000,
        158.60324501085662518562457989901304244995117187500000,  156.12893955664551981499243993312120437622070312500000,  153.56787515162926638367935083806514739990234375000000,  150.92188353225421337810985278338193893432617187500000,  148.19254439601760964251297991722822189331054687500000,  145.38120702349883117676654364913702011108398437500000,  142.48900759277492511500895489007234573364257812500000,  139.51688284135178719225223176181316375732421875000000,  136.46558056286323790118331089615821838378906250000000,  133.33566729277876561354787554591894149780273437500000,
        130.12753342829023495141882449388504028320312500000000,  126.84139593421166125608579022809863090515136718750000,  123.47729870262040208217513281852006912231445312500000,  120.03511055352947778374073095619678497314453125000000,  116.51452078190071404151240130886435508728027343750000,  112.91503206736709330471057910472154617309570312500000,  109.23595046092540883364563342183828353881835937500000,  105.47637203987333975874207681044936180114746093750000,  101.63516566820375430779677117243409156799316406250000,   97.71095110044366549573169322684407234191894531250000,
         93.70207140239205045872949995100498199462890625000000,   89.60655830277653421944705769419670104980468750000000,   85.42208859095924822213419247418642044067382812500000,   81.14592896880689920635632006451487541198730468750000,   76.77486574123962270732590695843100547790527343750000,   72.30511421396633409131027292460203170776367187500000,   67.73220036519177256195689551532268524169921875000000,   63.05080376541316411476145731285214424133300781250000,   58.25454493343946182903891894966363906860351562500000,   53.33569065746209503231511916965246200561523437500000,
         48.28473399546154354311511269770562648773193359375000,   43.08977491501639178750338032841682434082031250000000,   37.73556783279963866561956820078194141387939453125000,   32.20197762429902610392673523165285587310791015625000,   26.46129998617045231412703287787735462188720703125000,   20.47316285656608414456059108488261699676513671875000,   14.17346188697689512991928495466709136962890625000000,    7.44483327389219340375348110683262348175048828125000,    0.00000000000000000000000000000000000000000000000000,  184.53382886144947860884713008999824523925781250000000,
        186.07353157257051634587696753442287445068359375000000,  186.84226388218351644354697782546281814575195312500000,  187.13689514975109773331496398895978927612304687500000,  187.08176153445856471080332994461059570312500000000000,  186.74627715515720183248049579560756683349609375000000,  186.17500519834686656395206227898597717285156250000000,  185.39905355494337868549337144941091537475585937500000,  184.44138622051866605033865198493003845214843750000000,  183.31964587979575753706740215420722961425781250000000,  182.04779747014276836125645786523818969726562500000000,
        180.63715216128690599362016655504703521728515625000000,  179.09703945072621422696101944893598556518554687500000,  177.43526694561910517222713679075241088867187500000000,  175.65844566337929677501961123198270797729492187500000,  173.77222667575929904160147998481988906860351562500000,  171.78147731167695155818364582955837249755859375000000,  169.69041495633848626312101259827613830566406250000000,  167.50271035188484347600024193525314331054687500000000,  165.22156847572418314484821166843175888061523437500000,  162.84979260657885902219277340918779373168945312500000,
        160.38983555603866193450812716037034988403320312500000,  157.84384093650822933341260068118572235107421875000000,  155.21367656906454612908419221639633178710937500000000,  152.50096159179938126726483460515737533569335937500000,  149.70708843756710848538205027580261230468750000000000,  146.83324056208360275377344805747270584106445312500000,  143.88040658735459942363377194851636886596679687500000,  140.84939135989995406816888134926557540893554687500000,  137.74082429317738274221483152359724044799804687500000,  134.55516525825208873357041738927364349365234375000000,
        131.29270819815735649171983823180198669433593750000000,  127.95358256337433999760833103209733963012695312500000,  124.53775259333917801995994523167610168457031250000000,  121.04501439729246214938029879704117774963378906250000,  117.47499071261952963141084183007478713989257812500000,  113.82712313518403846046567196026444435119628906250000,  110.10066151819404467460117302834987640380859375000000,  106.29465011642325578122836304828524589538574218750000,  102.40790990114561509471968747675418853759765625000000,   98.43901627399232268089690478518605232238769531250000,
         94.38627114507808357757312478497624397277832031250000,   90.24766798266018952290323795750737190246582031250000,   86.02084794380454013662529177963733673095703125000000,   81.70304448941719499543978599831461906433105468750000,   77.29101286413862226254423148930072784423828125000000,   72.78093930620227069994143676012754440307617187500000,   68.16832255114304928156343521550297737121582031250000,   63.44781660099669551300394232384860515594482421875000,   58.61301794499881623323744861409068107604980468750000,   53.65617075878611075268054264597594738006591796875000,
         48.56774679480705003697948995977640151977539062500000,   43.33582592264399124815099639818072319030761718750000,   37.94514357834668061286720330826938152313232421875000,   32.37554669228203607644900330342352390289306640625000,   26.59931397700855271182263095397502183914184570312500,   20.57605727972871889619455032516270875930786132812500,   14.24165699216704972229763370705768465995788574218750,    7.47873482556787472219639312243089079856872558593750,    0.00000000000000000000000000000000000000000000000000,  188.62817342367159767491102684289216995239257812500000,
        190.17144384367756515530345495790243148803710937500000,  190.94430541983993521171214524656534194946289062500000,  191.24357673318976935661339666694402694702148437500000,  191.19356417692293348409293685108423233032226562500000,  190.86366447674467394790553953498601913452148437500000,  190.29843138816812597724492661654949188232421875000000,  189.52896888795353902423812542110681533813476562500000,  188.57824113293003165381378494203090667724609375000000,  187.46389415640385323058580979704856872558593750000000,  186.19989886857518968099611811339855194091796875000000,
        184.79757467494945899488811846822500228881835937500000,  183.26626134221334041285444982349872589111328125000000,  181.61377864092756340141932014375925064086914062500000,  179.84675157197705175349256023764610290527343750000000,  177.97084698582074224759708158671855926513671875000000,  175.99094979994455911764816846698522567749023437500000,  173.91129684602304905638447962701320648193359375000000,  171.73558024853693382283381652086973190307617187500000,  169.46702840936879397304437588900327682495117187500000,  167.10847020816515851038275286555290222167968750000000,
        164.66238639717252567606919910758733749389648437500000,  162.13095106338400341883243527263402938842773437500000,  159.51606526446155953635752666741609573364257812500000,  156.81938440300345405375992413610219955444335937500000,  154.04234051322046639143081847578287124633789062500000,  151.18616034743354248348623514175415039062500000000000,  148.25187993540711772766371723264455795288085937500000,  145.24035612592416555344243533909320831298828125000000,  142.15227549229277315134822856634855270385742187500000,  138.98816088104746313547366298735141754150390625000000,
        135.74837579821175381766806822270154953002929687500000,  132.43312675419457491443608887493610382080078125000000,  129.04246362195783603965537622570991516113281250000000,  125.57627799940907209474971750751137733459472656250000,  122.03429950219290844870556611567735671997070312500000,  118.41608984317751662729278905317187309265136718750000,  114.72103447542923504443024285137653350830078125000000,  110.94833148077312046098086284473538398742675781250000,  107.09697726891566560425417264923453330993652343750000,  103.16574850271665297896106494590640068054199218750000,
         99.15317946971717333326523657888174057006835937500000,   95.05753385851983239263063296675682067871093750000000,   90.87676954163198672631551744416356086730957031250000,   86.60849446949369223602843703702092170715332031250000,   82.24991107503400655787118012085556983947753906250000,   77.79774556584990818919322919100522994995117187500000,   73.24815696621158167545218020677566528320312500000000,   68.59661847030471903963189106434583663940429687500000,   63.83776007622838477573168347589671611785888671875000,   58.96515568493205705635773483663797378540039062500000,
         53.97102818951867675423272885382175445556640625000000,   48.84582926625738963366529787890613079071044921875000,   43.57761982482389839788083918392658233642578125000000,   38.15111737583747242297249613329768180847167968750000,   32.54615188429955452420472283847630023956298828125000,   26.73498598965462136334281240124255418777465820312500,   20.67721629633507163248395954724401235580444335937500,   14.30870872702998042313993209972977638244628906250000,    7.51207124583546637097697384888306260108947753906250,    0.00000000000000000000000000000000000000000000000000,
        192.73904728784486906079109758138656616210937500000000,  194.28581455609520389771205373108386993408203125000000,  195.06271947396868426949367858469486236572265625000000,  195.36653075140236524021020159125328063964843750000000,  195.32152538612726289102283772081136703491210937500000,  194.99708276409995733047253452241420745849609375000000,  194.43774705245007794474076945334672927856445312500000,  193.67461800128691606914799194782972335815429687500000,  192.73065949159223464448587037622928619384765625000000,  191.62352036029457735821779351681470870971679687500000,
        190.36717684623897639539791271090507507324218750000000,  188.97295585072762946765578817576169967651367187500000,  187.45020657368255001529178116470575332641601562500000,  185.80676001423270804480125661939382553100585937500000,  184.04925411785731625968765001744031906127929687500000,  182.18337036361626246616651769727468490600585937500000,  180.21400998709498253447236493229866027832031250000000,  178.14542786447532307647634297609329223632812500000000,  175.98133595586358524087700061500072479248046875000000,  173.72498438053654012946935836225748062133789062500000,
        171.37922573334967069058620836585760116577148437500000,  168.94656662153727211261866614222526550292968750000000,  166.42920929619421599454653915017843246459960937500000,  163.82908548723773378696932923048734664916992187500000,  161.14788400965059622649278026074171066284179687500000,  158.38707331926988786108267959207296371459960937500000,  155.54791991083413904561894014477729797363281250000000,  152.63150323788622131360170897096395492553710937500000,  149.63872767206544267537537962198257446289062500000000,  146.57033189348632618020928930491209030151367187500000,
        143.42689600380919046074268408119678497314453125000000,  140.20884657162761754989332985132932662963867187500000,  136.91645975020142600442341063171625137329101562500000,  133.54986254584341054396645631641149520874023437500000,  130.10903225766537616436835378408432006835937500000000,  126.59379405251348771344055421650409698486328125000000,  123.00381657942854474185878643766045570373535156250000,  119.33860546223081655625719577074050903320312500000000,  115.59749443258995427186164306476712226867675781250000,  111.77963377384004672876471886411309242248535156250000,
        107.88397563075153584577492438256740570068359375000000,  103.90925559275045486629096558317542076110839843750000,   99.85396976394611101568443700671195983886718750000000,   95.71634627291443564445216907188296318054199218750000,   91.49430981909894455839094007387757301330566406250000,   87.18543735653898352211399469524621963500976562500000,   82.78690231086834216966963140293955802917480468750000,   78.29540370378288116626208648085594177246093750000000,   73.70707504470784954264672705903649330139160156250000,   69.01736554895470021619985345751047134399414062500000,
         64.22088265006202334461704595014452934265136718750000,   59.31117898893005246918619377538561820983886718750000,   54.28045740381261197171625099144876003265380859375000,   49.11915063132736491979812853969633579254150390625000,   43.81530167497216865513109951280057430267333984375000,   38.35361109831713832818422815762460231781005859375000,   32.71389281069037480165206943638622760772705078125000,   26.86839422538715993482583144214004278182983398437500,   20.77669749365673368401985499076545238494873046875000,   14.37465480617088253723068191902711987495422363281250,
          7.54486106865845762570188526296988129615783691406250,    0.00000000000000000000000000000000000000000000000000,  196.86618167288997938157990574836730957031250000000000,  198.41637749603344786919478792697191238403320312500000,  199.19724305626300520088989287614822387695312500000000,  199.50549807814823566332051996141672134399414062500000,  199.46539053572897159938293043524026870727539062500000,  199.14628254387434935779310762882232666015625000000000,  198.59270854916701409820234403014183044433593750000000,  197.83576379583075777190970256924629211425781250000000,
        196.89841149228573158325161784887313842773437500000000,  195.79830278132462240137101616710424423217773437500000,  194.54941864031229670217726379632949829101562500000000,  193.16309278759032963534991722553968429565429687500000,  191.64868308925477435877837706357240676879882812500000,  190.01403091541740764114365447312593460083007812500000,  188.26578620374414185789646580815315246582031250000000,  186.40964400853292204374156426638364791870117187500000,  184.45052072127006681512284558266401290893554687500000,  182.39268798196627585639362223446369171142578125000000,
        180.23987617568684527213918045163154602050781250000000,  177.99535558489918685154407285153865814208984375000000,  175.66200080609340261617035139352083206176757812500000,  173.24234241010486812228919006884098052978515625000000,  170.73860872147557188327482435852289199829101562500000,  168.15275982748269711919419933110475540161132812500000,  165.48651538724413967429427430033683776855468750000000,  162.74137742259947003731213044375181198120117187500000,  159.91864898783148873917525634169578552246093750000000,  157.01944940324099775352806318551301956176757812500000,
        154.04472657677999336556240450590848922729492187500000,  150.99526681363479951869521755725145339965820312500000,  147.87170241542659709921281319111585617065429687500000,  144.67451729103677848797815386205911636352539062500000,  141.40405073437872829344996716827154159545898437500000,  138.06049946640581538304104469716548919677734375000000,  134.64391798575385905678558629006147384643554687500000,  131.15421722161715933907544240355491638183593750000000,  127.59116143086403383222204865887761116027832031250000,  123.95436322604210488407261436805129051208496093750000,
        120.24327655845070239593042060732841491699218750000000,  116.45718740681036251771729439496994018554687500000000,  112.59520183203196097565523814409971237182617187500000,  108.65623094520643121541070286184549331665039062500000,  104.63897218956367396458517760038375854492187500000000,  100.54188614411776825363631360232830047607421875000000,   96.36316779719932412717753322795033454895019531250000,   92.10071088272844974653708050027489662170410156250000,   87.75206337653933985620824387297034263610839843750000,   83.31437154580666515357734169811010360717773437500000,
         78.78430892330040080651087919250130653381347656250000,   74.15798506408117418686742894351482391357421875000000,   69.43082664210292875850427662953734397888183593750000,   64.59741985284470899841835489496588706970214843750000,   59.65129730321912404633621918037533760070800781250000,   54.58464291163927128991417703218758106231689453125000,   49.38787152737584307260476634837687015533447265625000,   44.04900922479327363134871120564639568328857421875000,   38.55274052884838908994424855336546897888183593750000,   32.87886413897891912938575842417776584625244140625000,
         26.99961303080477748039811558555811643600463867187500,   20.87455563836077132577884185593575239181518554687500,   14.43953110809193951524775911821052432060241699218750,    7.57712193087667884583424893207848072052001953125000,    0.00000000000000000000000000000000000000000000000000,  201.00931639928151639651332516223192214965820312500000,  202.56287491801239752930996473878622055053710937500000,  203.34762147421349709475180134177207946777343750000000,  203.66022767148587035990203730762004852294921875000000,  203.62491283211593895430269185453653335571289062500000,
        203.31102188183416501487954519689083099365234375000000,  202.76307943707996628290857188403606414794921875000000,  202.01217598694211119436658918857574462890625000000000,  201.08127370708882608596468344330787658691406250000000,  199.98802559114764676451159175485372543334960937500000,  198.74641681381626767688430845737457275390625000000000,  197.36778728710478958419116679579019546508789062500000,  195.86150283848141384623886551707983016967773437500000,  194.23541442117300448444439098238945007324218750000000,  192.49618309054281439784972462803125381469726562500000,
        190.64951650978511565881490241736173629760742187500000,  188.70034516129081225699337664991617202758789062500000,  186.65295627687891055757063440978527069091796875000000,  184.51109737752014439138292800635099411010742187500000,  182.27805749100124899086949881166219711303710937500000,  179.95673165499576384718238841742277145385742187500000,  177.54967268490926812773977871984243392944335937500000,  175.05913308254724825019366107881069183349609375000000,  172.48709919765076392650371417403221130371093750000000,  169.83531921480653181788511574268341064453125000000000,
        167.10532615023782909702276811003684997558593750000000,  164.29845675912358160530857276171445846557617187500000,  161.41586704292367926427687052637338638305664062500000,  158.45854488641569446372159291058778762817382812500000,  155.42732023107438976694538723677396774291992187500000,  152.32287309470621039508841931819915771484375000000000,  149.14573966945025063068896997720003128051757812500000,  145.89631666588110192606109194457530975341796875000000,  142.57486401582340818094962742179632186889648437500000,  139.18150599727817962047993205487728118896484375000000,
        135.71623079876943052113347221165895462036132812500000,  132.17888849490054781199432909488677978515625000000000,  128.56918735745188087093993090093135833740234375000000,  124.88668837425699109644483542069792747497558593750000,  121.13079778821239074204640928655862808227539062500000,  117.30075739719920591141999466344714164733886718750000,  113.39563226733467615758854662999510765075683593750000,  109.41429539993629305172362364828586578369140625000000,  105.35540874730789084878779249265789985656738281250000,  101.21739978032093176807393319904804229736328125000000,
         96.99843255198693725560588063672184944152832031250000,   92.69637184647744732046703575178980827331542968750000,   88.30873850801222602058260235935449600219726562500000,   83.83265334019553449707018444314599037170410156250000,   79.26476594540986297943163663148880004882812500000000,   74.60116336005776815909484867006540298461914062500000,   69.83725104154669338640815112739801406860351562500000,   64.96759517177785880903684301301836967468261718750000,   59.98570945720562264114050776697695255279541015625000,   54.88375994732040652479554410092532634735107421875000,
         49.65214457978910189694943255744874477386474609375000,   44.27887340705581920019540120847523212432861328125000,   38.74861576009666208619819371961057186126708984375000,   33.04115591584941569180955411866307258605957031250000,   27.12871314720586823909798113163560628890991210937500,   20.97084285781292223305172228720039129257202148437500,   14.50337179249498120725547778420150279998779296875000,    7.60887062919125956028665314079262316226959228515625,    0.00000000000000000000000000000000000000000000000000,  205.16819948264117101643932983279228210449218750000000,
        206.72505714738284154918801505118608474731445312500000,  207.51360794550006971803668420761823654174804687500000,  207.83047620303014468845503870397806167602539062500000,  207.79985296167942010470142122358083724975585937500000,  207.49106605153028226595779415220022201538085937500000,  206.94863016947678602264204528182744979858398437500000,  206.20363082728562176271225325763225555419921875000000,  205.27902884075530209884163923561573028564453125000000,  204.19247863805637166478845756500959396362304687500000,  202.95796909440363720023015048354864120483398437500000,
        201.58684574229661734534602146595716476440429687500000,  200.08848172214067062668618746101856231689453125000000,  198.47073684545082983277097810059785842895507812500000,  196.74028248115041606070008128881454467773437500000000,  194.90283801342874880901945289224386215209960937500000,  192.96334703804163268614502158015966415405273437500000,  190.92611130416577225332730449736118316650390625000000,  188.79489428859488953094114549458026885986328125000000,  186.57300246844579305616207420825958251953125000000000,  184.26334989868982461302948649972677230834960937500000,
        181.86851007359877030467032454907894134521484375000000,  179.39075794859309098683297634124755859375000000000000,  176.83210423561067159425874706357717514038085937500000,  174.19432354618049885175423696637153625488281250000000,  171.47897756899959631482488475739955902099609375000000,  168.68743418561288649470952805131673812866210937500000,  165.82088321736677016815519891679286956787109375000000,  162.88034933789114688806876074522733688354492187500000,  159.86670256330566530778014566749334335327148437500000,  156.78066663685382309267879463732242584228515625000000,
        153.62282554836062331560242455452680587768554687500000,  150.39362836638548515111324377357959747314453125000000,  147.09339250811856913969677407294511795043945312500000,  143.72230552576203876924409996718168258666992187500000,  140.28042544572224414878292009234428405761718750000000,  136.76767965612620514548325445502996444702148437500000,  133.18386229678753807093016803264617919921875000000000,  129.52863006155024550025700591504573822021484375000000,  125.80149627343256213407585164532065391540527343750000,  122.00182303518077731041557854041457176208496093750000,
        118.12881118793188761628698557615280151367187500000000,  114.18148772366942012013169005513191223144531250000000,  110.15869018621965835791343124583363533020019531250000,  106.05904745115716991676890756934881210327148437500000,  101.88095608359026300604455173015594482421875000000000,   97.62255121462807494481239700689911842346191406250000,   93.28167052309700579826312605291604995727539062500000,   88.85580941445880398532608523964881896972656250000000,   84.34206478539425688722985796630382537841796875000000,   79.73706374184195055931922979652881622314453125000000,
         75.03687212471146494863205589354038238525390625000000,   70.23687539780686961421451997011899948120117187500000,   65.33162086154061398701742291450500488281250000000000,   60.31460437147668329771477147005498409271240234375000,   55.17797508265883976719123893417418003082275390625000,   49.91211492717250308714938000775873661041259765625000,   44.50501877905385583744646282866597175598144531250000,   38.94134156163371329739675275050103664398193359375000,   33.20085386330248411468346603214740753173828125000000,   27.25576194011159714136738330125808715820312500000000,
         21.06560880703935367819212842732667922973632812500000,   14.56620940836222466430172062246128916740417480468750,    7.64012317269536289643383497605100274085998535156250,    0.00000000000000000000000000000000000000000000000000,  209.34258675253684600647829938679933547973632812500000,  210.90268220732451709409360773861408233642578125000000,  211.69496323586861308285733684897422790527343750000000,  212.01600770942718554579187184572219848632812500000000,  211.98997875874201213264313992112874984741210937500000,  211.68618722173644641770806629210710525512695312500000,
        211.14913780443595214819652028381824493408203125000000,  210.40991084354817530766013078391551971435546875000000,  209.49146549787076310167321935296058654785156250000000,  208.41145724899459423795633483678102493286132812500000,  207.18387821612915900004736613482236862182617187500000,  205.82007902464400217468210030347108840942382812500000,  204.32943953056701502646319568157196044921875000000000,  202.71982773598210769705474376678466796875000000000000,  200.99792458287737417776952497661113739013671875000000,  199.16946035938261161390983033925294876098632812500000,
        197.23939087593225849559530615806579589843750000000000,  195.21203141241488765444955788552761077880859375000000,  193.09116031923159084726648870855569839477539062500000,  190.88010033606636284275737125426530838012695312500000,  188.58178323340592896784073673188686370849609375000000,  186.19880175544267331133596599102020263671875000000000,  183.73345174106020749604795128107070922851562500000000,  181.18776653681143784524465445429086685180664062500000,  178.56354527740336379793006926774978637695312500000000,  175.86237622235552180427475832402706146240234375000000,
        173.08565605486421645764494314789772033691406250000000,  170.23460583910448917777102906256914138793945312500000,  167.31028417399835461765178479254245758056640625000000,  164.31359796025967057175876107066869735717773437500000,  161.24531110303271930206392426043748855590820312500000,  158.10605139735002921952400356531143188476562500000000,  154.89631578259786692797206342220306396484375000000000,  151.61647410119749679324741009622812271118164062500000,  148.26677145270278401767427567392587661743164062500000,  144.84732919499697345599997788667678833007812500000000,
        141.35814460713390872115269303321838378906250000000000,  137.79908919168255465592665132135152816772460937500000,  134.16990555630610515436273999512195587158203125000000,  130.47020277269695043287356384098529815673828125000000,  126.69945006355037264711427269503474235534667968750000,  122.85696861211195596297329757362604141235351562500000,  118.94192122027023117425414966419339179992675781250000,  114.95329945524480308449710719287395477294921875000000,  110.88990781488155334955081343650817871093750000000000,  106.75034429792293622085708193480968475341796875000000,
        102.53297657483273042089422233402729034423828125000000,   98.23591269708948914285429054871201515197753906250000,   93.85696492905040599907806608825922012329101562500000,   89.39360479222312960700946860015392303466796875000000,   84.84290670859067517994844820350408554077148437500000,   80.20147660920913779136753873899579048156738281250000,   75.46536036093925758905243128538131713867187500000000,   70.62992456466737678510980913415551185607910156250000,   65.68969868756971663970034569501876831054687500000000,   60.63816170756138035358162596821784973144531250000000,
         55.46744679010180334444157779216766357421875000000000,   50.16792070411608506219636183232069015502929687500000,   44.72756393054010004561860114336013793945312500000000,   39.13101771804297612789014237932860851287841796875000,   33.35803965144101113082797382958233356475830078125000,   27.38082361080068238834428484551608562469482421875000,   21.15890082268743910276498354505747556686401367187500,   14.62807499367177754834301595110446214675903320312500,    7.67089483136211658376168998074717819690704345703125,    0.00000000000000000000000000000000000000000000000000,
        213.53224149456323743834218475967645645141601562500000,  215.09551546846731184814416337758302688598632812500000,  215.89145531873600702965632081031799316406250000000000,  216.21659326441664461526670493185520172119140625000000,  216.19506489266873927590495441108942031860351562500000,  215.89616416137820920084777753800153732299804687500000,  215.36438573056955192441819235682487487792968750000000,  214.63080458626643576280912384390830993652343750000000,  213.71837796035640621994389221072196960449218750000000,  212.64476203870236759030376560986042022705078125000000,
        211.42395176463077177686500363051891326904296875000000,  210.06730237018155094119720160961151123046875000000000,  208.58419987613072521526191849261522293090820312500000,  206.98251985923499773889489006251096725463867187500000,  205.26895215172947928294888697564601898193359375000000,  203.44923719274143536495103035122156143188476562500000,  201.52834217993600418594724033027887344360351562500000,  199.51059501645156046834017615765333175659179687500000,  197.39978793216721442149719223380088806152343750000000,  195.19925884067987453818204812705516815185546875000000,
        192.91195603564560201448330190032720565795898437500000,  190.54049020451856222280184738337993621826171875000000,  188.08717663641274953079118859022855758666992187500000,  185.55406973855298247144673950970172882080078125000000,  182.94299143787202410749159753322601318359375000000000,  180.25555465793607368141238112002611160278320312500000,  177.49318277923825348807440605014562606811523437500000,  174.65712578163089574445621110498905181884765625000000,  171.74847361006553114748385269194841384887695312500000,  168.76816718428781882721523288637399673461914062500000,
        165.71700737946738968275894876569509506225585937500000,  162.59566223064038581469503697007894515991210937500000,  159.40467255401267721026670187711715698242187500000000,  156.14445612867328350148454774171113967895507812500000,  152.81531054009775516533409245312213897705078125000000,  149.41741474960042523889569565653800964355468750000000,  145.95082941964838596504705492407083511352539062500000,  142.41549599194084407827176619321107864379882812500000,  138.81123448172638745745643973350524902343750000000000,  135.13773991628977455548010766506195068359375000000000,
        131.39457730599627893752767704427242279052734375000000,  127.58117499050722187803330598399043083190917968750000,  123.69681614797727320365083869546651840209960937500000,  119.74062818757610671127622481435537338256835937500000,  115.71156966065098004037281498312950134277343750000000,  111.60841421654419036713079549372196197509765625000000,  107.42973098604400661315594334155321121215820312500000,  103.17386058516323998901498271152377128601074218750000,   98.83888567469507790974603267386555671691894531250000,   94.42259465753473079985269578173756599426269531250000,
         89.92243660178675668248615693300962448120117187500000,   85.33546477473471725261333631351590156555175781250000,   80.65826515253507977831759490072727203369140625000000,   75.88686475731763891872105887159705162048339843750000,   71.01661237402150561592861777171492576599121093750000,   66.04202060862495216042589163407683372497558593750000,   60.95655246512596647789905546233057975769042968750000,   55.75232596075725410855739028193056583404541015625000,   50.41969348558957619843567954376339912414550781250000,   44.94662185949776755933271488174796104431152343750000,
         39.31773934057130759356368798762559890747070312500000,   33.51279115006582287605851888656616210937500000000000,   27.50395939152244295655691530555486679077148437500000,   21.25076406518220295538412756286561489105224609375000,   14.68899816751311959706072229892015457153320312500000,    7.70120018085744550973004152183420956134796142578125,    0.00000000000000000000000000000000000000000000000000,  217.73693411395422003806743305176496505737304687500000,  219.30332931944721508443763013929128646850585937500000,  220.10285905492037272779271006584167480468750000000000,
        220.43201066999043291616544593125581741333007812500000,  220.41489257281057234649779275059700012207031250000000,  220.12078196077359848459309432655572891235351562500000,  219.59416340727415217770612798631191253662109375000000,  218.86610639204280914782430045306682586669921875000000,  217.95956597488932970918540377169847488403320312500000,  216.89219872590825843872153200209140777587890625000000,  215.67800202603976345017144922167062759399414062500000,  214.32833526550831493295845575630664825439453125000000,  212.85259012121963451136252842843532562255859375000000,
        211.25864917581409940794401336461305618286132812500000,  209.55321052132040904325549490749835968017578125000000,  207.74202405304345120384823530912399291992187500000000,  205.83006759291416187807044479995965957641601562500000,  203.82168083141087322474049869924783706665039062500000,  201.72066896321229023669729940593242645263671875000000,  199.53038407549044563893403392285108566284179687500000,  197.25378989103225535473029594868421554565429687500000,  194.89351384350476337203872390091419219970703125000000,  192.45188936145947877776052337139844894409179687500000,
        189.93099047590442296495893970131874084472656250000000,  187.33266032880459306397824548184871673583984375000000,  184.65853477391578962851781398057937622070312500000000,  181.91006197964546231560234446078538894653320312500000,  179.08851873481870597970555536448955535888671875000000,  176.19502400113881890320044476538896560668945312500000,  173.23055013618952102660841774195432662963867187500000,  170.19593211784393815833027474582195281982421875000000,  167.09187502765780664049088954925537109375000000000000,  163.91895999197808464487025048583745956420898437500000,
        160.67764873119872959250642452389001846313476562500000,  157.36828682690457981152576394379138946533203125000000,  153.99110578125669235305394977331161499023437500000000,  150.54622391102085998682014178484678268432617187500000,  147.03364608852126593774300999939441680908203125000000,  143.45326231204467148927506059408187866210937500000000,  139.80484505737530298574711196124553680419921875000000,  136.08804532866497538634575903415679931640625000000000,  132.30238728896210886887274682521820068359375000000000,  128.44726130629283034068066626787185668945312500000000,
        124.52191519747981374166556634008884429931640625000000,  120.52544338531026824057335034012794494628906250000000,  116.45677360037498715428228024393320083618164062500000,  112.31465065020491067571128951385617256164550781250000,  108.09761663580076174184796400368213653564453125000000,  103.80398680578643677563377423211932182312011718750000,   99.43181998152472544916236074641346931457519531250000,   94.97888213336671014985768124461174011230468750000000,   90.44260119448784962514764629304409027099609375000000,   85.82001049603020703671063529327511787414550781250000,
         81.10767718723481323195301229134202003479003906250000,   76.30161049122082772555586416274309158325195312500000,   71.39714234784534596656158100813627243041992187500000,   66.38876940451808650323073379695415496826171875000000,   61.26993953164059547589204157702624797821044921875000,   56.03275638154826054915247368626296520233154296875000,   50.66755869657605160227831220254302024841308593750000,   45.16230031875093686721811536699533462524414062500000,   39.50159715477530397720329347066581249237060546875000,   33.66518266102752221513583208434283733367919921875000,
         27.62522772587713504321982327383011579513549804687500,   21.34124165014831930875516263768076896667480468750000,   14.74900721528734059972975956043228507041931152343750,    7.73105314400712639866242170683108270168304443359375,    0.00000000000000000000000000000000000000000000000000,  221.95644181913033321507100481539964675903320312500000,  223.52590285685673165971820708364248275756835937500000,  224.32895589103372913086786866188049316406250000000000,  224.66204416528444198775105178356170654296875000000000,  224.64924927004054211465700063854455947875976562500000,
        224.35983176809710926136176567524671554565429687500000,  223.83826611858316368852683808654546737670898437500000,  223.11561615746120423864340409636497497558593750000000,  222.21483454980781857557303737848997116088867187500000,  221.15357795644121097211609594523906707763671875000000,  219.94584583984956793756282422691583633422851562500000,  218.60300133434594727077637799084186553955078125000000,  217.13444130285532196467102039605379104614257812500000,  215.54805480800212080794153735041618347167968750000000,  213.85054761877614737386466003954410552978515625000000,
        212.04767844462779180503275711089372634887695312500000,  210.14443502726535939473251346498727798461914062500000,  208.14516807337824388923763763159513473510742187500000,  206.05369489957229234278202056884765625000000000000000,  203.87338084592607856393442489206790924072265625000000,  201.60720405882489103532861918210983276367187500000000,  199.25780761957267372963542584329843521118164062500000,  196.82754189512056086641678120940923690795898437500000,  194.31849922593937662895768880844116210937500000000000,  191.73254252900102301282458938658237457275390625000000,
        189.07132900825510546383156906813383102416992187500000,  186.33632988365067717495549004524946212768554687500000,  183.52884684133564974217733833938837051391601562500000,  180.65002575100891135662095621228218078613281250000000,  177.70086807695102493198646698147058486938476562500000,  174.68224031684667352237738668918609619140625000000000,  171.59488172989745180530007928609848022460937500000000,  168.43941055768451064977853093296289443969726562500000,  165.21632889392014931218000128865242004394531250000000,  161.92602631973002758059010375291109085083007812500000,
        158.56878238719576756921014748513698577880859375000000,  155.14476800376615983623196370899677276611328125000000,  151.65404574231871492884238250553607940673828125000000,  148.09656907478324683324899524450302124023437500000000,  144.47218050006694056719425134360790252685546875000000,  140.78060850824164162986562587320804595947265625000000,  137.02146329113597289506287779659032821655273437500000,  133.19423107294144870138552505522966384887695312500000,  129.29826689110367965440673287957906723022460937500000,  125.33278560495612907743634423241019248962402343750000,
        121.29685084371537584502220852300524711608886718750000,  117.18936152177516873962304089218378067016601562500000,  113.00903544104389197855198290199041366577148437500000,  108.75438935795934014549857238307595252990722656250000,  104.42371470330185445618553785607218742370605468750000,  100.01504788632833253814169438555836677551269531250000,   95.52613376183262516860850155353546142578125000000000,   90.95438034522899783951288554817438125610351562500000,   86.29680215820111754965182626619935035705566406250000,   81.54994856757554089199402369558811187744140625000000,
         76.70981196718160788350360235199332237243652343750000,   71.77170835334860043985827360302209854125976562500000,   66.73011925422696322129922918975353240966796875000000,   61.57847818899867320396879222244024276733398437500000,   56.30887517532237751538559678010642528533935546875000,   50.91163599016295648880259250290691852569580078125000,   45.37470213609145730515592731535434722900390625000000,   39.68267776634955623649148037657141685485839843750000,   33.81528513307560501743864733725786209106445312500000,   27.74468443569633535616958397440612316131591796875000,
         21.43037477005623259174171835184097290039062500000000,   14.80812916760596742449251905782148241996765136718750,    7.76046702921342035352836319361813366413116455078125,    0.00000000000000000000000000000000000000000000000000,  226.19054832372756891345488838851451873779296875000000,  227.76302159318444751079368870705366134643554687500000,  228.56953357520058034424437209963798522949218750000000,  228.90648415195497022978088352829217910766601562500000,  228.89792845374506669031688943505287170410156250000000,  228.61311054006384324566170107573270797729492187500000,
        228.09649473977921729783702176064252853393554687500000,  227.37913912405292649054899811744689941406250000000000,  226.48399376107877856156846974045038223266601562500000,  225.42871513310495856785564683377742767333984375000000,  224.22730445589803593975375406444072723388671875000000,  222.89112822516673872996761929243803024291992187500000,  221.42958805488945017714286223053932189941406250000000,  219.85057900088517612857685890048742294311523437500000,  218.16081396965063277093577198684215545654296875000000,  216.36605989077287404143135063350200653076171875000000,
        214.47131377437281685160996858030557632446289062500000,  212.48093663098168804026499856263399124145507812500000,  210.39875712128267082334787119179964065551757812500000,  208.22815298960594532218237873166799545288085937500000,  205.97211588025152195768896490335464477539062500000000,  203.63330351224752234884363133460283279418945312500000,  201.21408208851798349314776714891195297241210937500000,  198.71656105455295460160414222627878189086914062500000,  196.14262178498290722927777096629142761230468750000000,  193.49394139122054525614657904952764511108398437500000,
        190.77201256233203707779466640204191207885742187500000,  187.97816014322646083201107103377580642700195312500000,  185.11355499796715662341739516705274581909179687500000,  182.17922558697654267234611324965953826904296875000000,  179.17606759497041934992012102156877517700195312500000,  176.10485187440571053230087272822856903076171875000000,  172.96623091185011844572727568447589874267578125000000,  169.76074397816461214461014606058597564697265625000000,  166.48882108486512265699275303632020950317382812500000,  163.15078583630977959728625137358903884887695312500000,
        159.74685723870857145811896771192550659179687500000000,  156.27715050095255833184637594968080520629882812500000,  152.74167683768337155925109982490539550781250000000000,  149.14034226073587774408224504441022872924804687500000,  145.47294531997806643630610778927803039550781250000000,  141.73917372745361831221089232712984085083007812500000,  137.93859976825848434600629843771457672119140625000000,  134.07067436614164535058080218732357025146484375000000,  130.13471962937632042667246423661708831787109375000000,  126.12991965036921726550644962117075920104980468750000,
        122.05530926724372875469271093606948852539062500000000,  117.90976041245460237405495718121528625488281250000000,  113.69196556571880307728861225768923759460449218750000,  109.40041768678784706025908235460519790649414062500000,  105.03338581436796062007488217204809188842773437500000,  100.58888526114562012025999138131737709045410156250000,   96.06464098216881097869190853089094161987304687500000,   91.45804220050399635510984808206558227539062500000000,   86.76608567168742069952713791280984878540039062500000,   81.98530394873533566624246304854750633239746093750000,
         77.11167349669233317399630323052406311035156250000000,   72.14049520668223181019129697233438491821289062500000,   67.06623626904367085899139055982232093811035156250000,   61.88231658108347232882806565612554550170898437500000,   56.58081320732238594928276143036782741546630859375000,   51.15203959696599156359297921881079673767089843750000,   45.58392551031717943033072515390813350677490234375000,   39.86106390709504410096997162327170372009277343750000,   33.96316636076443984393335995264351367950439453125000,   27.86238287561853255169808107893913984298706054687500,
         21.51820280695241649482341017574071884155273437500000,   14.86638987343959072973120782990008592605590820312500,    7.78945456608667186060301901306957006454467773437500,    0.00000000000000000000000000000000000000000000000000,  230.43904356577695580199360847473144531250000000000000,  232.01447718145826115687668789178133010864257812500000,  232.82438588887657715531531721353530883789062500000000,  233.16512693489360685816791374236345291137695312500000,  233.16072934322181708921561948955059051513671875000000,  232.88042080590486193614196963608264923095703125000000,
        232.36865551598117463072412647306919097900390625000000,  231.65648567369760257861344143748283386230468750000000,  230.76685856691315734678937587887048721313476562500000,  229.71743025213660871486354153603315353393554687500000,  228.52220339555978512180445250123739242553710937500000,  227.19254750030816580874670762568712234497070312500000,  225.73786852856900964070518966764211654663085937500000,  224.16606707828393041381787043064832687377929687500000,  222.48386269357752098585478961467742919921875000000000,  220.69702997392596444115042686462402343750000000000000,
        218.81057459484077298839110881090164184570312500000000,  216.82886721171882982162060216069221496582031250000000,  214.75574711046050424556597135961055755615234375000000,  212.59460365621268351787875872105360031127929687500000,  210.34844113769150908410665579140186309814453125000000,  208.01993098173505813974770717322826385498046875000000,  205.61145421360774321328790392726659774780273437500000,  203.12513627810298544318357016891241073608398437500000,  200.56287580014168270281516015529632568359375000000000,  197.92636847863772686650918330997228622436523437500000,
        195.21712702669802297350543085485696792602539062500000,  192.43649786345383745356230065226554870605468750000000,  189.58567510685801948966400232166051864624023437500000,  186.66571229810591603381908498704433441162109375000000,  183.67753219680582787987077608704566955566406250000000,  180.62193491444222104291839059442281723022460937500000,  177.49960459685178193467436358332633972167968750000000,  174.31111482057644934684503823518753051757812500000000,  171.05693283022839068507892079651355743408203125000000,  167.73742271225293620773300062865018844604492187500000,
        164.35284757301522517991543281823396682739257812500000,  160.90337076460519938336801715195178985595703125000000,  157.38905617900667266439995728433132171630859375000000,  153.80986760927737577731022611260414123535156250000000,  150.16566715416431065932556521147489547729492187500000,  146.45621261912756949641334358602762222290039062500000,  142.68115384097021092202339787036180496215820312500000,  138.84002783389090041055169422179460525512695312500000,  134.93225262022625088320637587457895278930664062500000,  130.95711956744290205278957728296518325805664062500000,
        126.91378400146496119305083993822336196899414062500000,  122.80125380169505433514132164418697357177734375000000,  118.61837560032893179595703259110450744628906250000000,  114.36381810113935841854981845244765281677246093750000,  110.03605189144262510581029346212744712829589843750000,  105.63332493198927863886638078838586807250976562500000,  101.15363265338230291945365024730563163757324218750000,   96.59468123509761028344655642285943031311035156250000,   91.95384215000325411892845295369625091552734375000000,   87.22809535500215361025766469538211822509765625000000,
         82.41395748878110794066742528229951858520507812500000,   77.50738992495899992718477733433246612548828125000000,   72.50367922999444658671563956886529922485351562500000,   67.39727898490353652505291393026709556579589843750000,   62.18159614584966021766376798041164875030517578125000,   56.84869546106403248586502741090953350067138671875000,   51.38887864845980146810688893310725688934326171875000,   45.79006428532745331949627143330872058868408203125000,   40.03683466278366864798954338766634464263916015625000,   34.10889116881615734655497362837195396423339843750000,
         27.97837407643326912420889129862189292907714843750000,   21.60476343704682378188408620189875364303588867187500,   14.92381406801200149914166104281321167945861816406250,    7.81802793853072852670038628275506198406219482421875,    0.00000000000000000000000000000000000000000000000000,  234.70172344281823484379856381565332412719726562500000,  236.28006715541511084666126407682895660400390625000000,  237.09331239364712473616236820816993713378906250000000,  237.43777447722973761301545891910791397094726562500000,  237.43745667252096609445288777351379394531250000000000,
        237.16157044377663964951352681964635848999023437500000,  236.65455985197479549242416396737098693847656250000000,  235.94747113387896320091385859996080398559570312500000,  235.06324863062621943754493258893489837646484375000000,  234.01954774605442821666656527668237686157226562500000,  232.83037231719569604138087015599012374877929687500000,  231.50709452690406919828092213720083236694335937500000,  230.05912431212874480479513294994831085205078125000000,  228.49436739452863776023150421679019927978515625000000,  226.81954949213741201674565672874450683593750000000000,
        225.04045236401265128733939491212368011474609375000000,  223.16208979209994822667795233428478240966796875000000,  221.18884146628533926559612154960632324218750000000000,  219.12455663243775916271260939538478851318359375000000,  216.97263555226371067874424625188112258911132812500000,  214.73609437078013684185862075537443161010742187500000,  212.41761736504818713910935912281274795532226562500000,  210.01959944916393396852072328329086303710937500000000,  207.54418105041310127489850856363773345947265625000000,  204.99327693546456430340185761451721191406250000000000,
        202.36860018083999079863133374601602554321289062500000,  199.67168220145816803778870962560176849365234375000000,  196.90388954355267969731357879936695098876953125000000,  194.06643799257773252975312061607837677001953125000000,  191.16040442834730583854252472519874572753906250000000,  188.18673676846304942955612204968929290771484375000000,  185.14626226989921065069211181253194808959960937500000,  182.03969440225134235333825927227735519409179687500000,  178.86763846084576812245359178632497787475585937500000,  175.63059605083640235534403473138809204101562500000000,
        172.32896854245782947145926300436258316040039062500000,  168.96305957111118800639815162867307662963867187500000,  165.53307663261657012299110647290945053100585937500000,  162.03913180268284577323356643319129943847656250000000,  158.48124158947354089832515455782413482666015625000000,  154.85932590821215626419871114194393157958984375000000,  151.17320614620558671958860941231250762939453125000000,  147.42260226455468341555388178676366806030273437500000,  143.60712885813723005412612110376358032226562500000000,  139.72629006696143960652989335358142852783203125000000,
        135.77947319817116067497408948838710784912109375000000,  131.76594087688397394231287762522697448730468750000000,  127.68482149307202178079023724421858787536621093750000,  123.53509764738657850102754309773445129394531250000000,  119.31559221641983015160803915932774543762207031250000,  115.02495155076837818342028185725212097167968750000000,  110.66162517804764320317190140485763549804687500000000,  106.22384119424589243863010779023170471191406250000000,  101.70957627084591479160735616460442543029785156250000,   97.11651885276644691202818648889660835266113281250000,
         92.44202362912662351845938246697187423706054687500000,   87.68305465667685894004534929990768432617187500000000,   82.83611349619282293588184984400868415832519531250000,   77.89714720952217419380758656188845634460449218750000,   72.86142876611249619145382894203066825866699218750000,   67.72339881760095181562064681202173233032226562500000,   62.47645201511152635021062451414763927459716796875000,   57.11264138634879117262244108133018016815185546875000,   51.62225747652304619350616121664643287658691406250000,   45.99320820420142297280108323320746421813964843750000,
         40.21006568449613638449591235257685184478759765625000,   34.25252158319872108904746710322797298431396484375000,   28.09270687815953948529568151570856571197509765625000,   21.69009272785346809087059227749705314636230468750000,   14.98042543588644726071379409404471516609191894531250,    7.84619881549742448356710156076587736606597900390625,    0.00000000000000000000000000000000000000000000000000,  238.97838956183431946556083858013153076171875000000000,  240.55959468411938928511517588049173355102539062500000,  241.37611819197485374388634227216243743896484375000000,
        241.72423416865382250762195326387882232666015625000000,  241.72792046784090302935510408133268356323242187500000,  241.45637246881048554314475040882825851440429687500000,  240.95402411260721464714151807129383087158203125000000,  240.25191559224958837148733437061309814453125000000000,  239.37298815135463314618391450494527816772460937500000,  238.33489633268948182376334443688392639160156250000000,  237.15164488586950142234854865819215774536132812500000,  235.83460836988976438988174777477979660034179687500000,  234.39320034995628816432144958525896072387695312500000,
        232.83533128295761116532958112657070159912109375000000,  231.16773263020584749938279855996370315551757812500000,  229.39619283653999559646763373166322708129882812500000,  227.52573327161272231933253351598978042602539062500000,  225.56074209373281291846069507300853729248046875000000,  223.50507789229752120263583492487668991088867187500000,  221.36215115510668738352251239120960235595703125000000,  219.13498915469583039339340757578611373901367187500000,  216.82628822628097964297921862453222274780273437500000,  214.43845631172811749820539262145757675170898437500000,
        211.97364788423317349952412769198417663574218750000000,  209.43379283266932588958297856152057647705078125000000,  206.82062050018183185784437227994203567504882812500000,  204.13567979141606656412477605044841766357421875000000,  201.38035605550342665992502588778734207153320312500000,  198.55588529649168094692868180572986602783203125000000,  195.66336614480147204631066415458917617797851562500000,  192.70376993239383978107071015983819961547851562500000,  189.67794914348115753455203957855701446533203125000000,  186.58664445663998776581138372421264648437500000000000,
        183.43049054933078423346159979701042175292968750000000,  180.21002079930204331503773573786020278930664062500000,  176.92567098705225703270116355270147323608398437500000,  173.57778207782149593185749836266040802001953125000000,  170.16660213920664546094485558569431304931640625000000,  166.69228743039707296702545136213302612304687500000000,  163.15490268032203857728745788335800170898437500000000,  159.55442055388985522768052760511636734008789062500000,  155.89072028722094387376273516565561294555664062500000,  152.16358545355330988968489691615104675292968750000000,
        148.37270080048060094668471720069646835327148437500000,  144.51764807539140633707575034350156784057617187500000,  140.59790072822534057195298373699188232421875000000000,  136.61281734744966342987027019262313842773437500000000,  132.56163364456367048660467844456434249877929687500000,  128.44345275188501886987069156020879745483398437500000,  124.25723353441379970263369614258408546447753906250000,  120.00177653445764747175417141988873481750488281250000,  115.67570706081900766548642423003911972045898437500000,  111.27745479334204503629734972491860389709472656250000,
        106.80522908503328949336719233542680740356445312500000,  102.25698888816192777539981761947274208068847656250000,   97.63040587849907581130537437275052070617675781250000,   92.92281885891885906403331318870186805725097656250000,   88.13117682151853671257413225248456001281738281250000,   83.25196702795190617507614661008119583129882812500000,   78.28112295513381013734033331274986267089843750000000,   73.21390465467479202743561472743749618530273437500000,   68.04474048420993881336471531540155410766601562500000,   62.76701338489920090069063007831573486328125000000000,
         57.37276522185943861131818266585469245910644531250000,   51.85227589127070757513138232752680778503417968750000,   46.19344314499032577714388025924563407897949218750000,   40.38082938485241157877680961973965167999267578125000,   34.39411699005245282023679465055465698242187500000000,   28.20542805372858907730915234424173831939697265625000,   21.77422522851167485669066081754863262176513671875000,   15.03624666964690526071990461787208914756774902343750,    7.87397837960449997751766204601153731346130371093750,    0.00000000000000000000000000000000000000000000000000,
        243.26884900298270508756104391068220138549804687500000,  244.85286834003895251044014003127813339233398437500000,  245.67261370095059191953623667359352111816406250000000,  246.02431860617141978764266241341829299926757812500000,  246.03193583665844812458090018481016159057617187500000,  245.76464483206623867772577796131372451782226562500000,  245.26686943311023014757665805518627166748046875000000,  244.56964371998853380318905692547559738159179687500000,  243.69590570225369674517423845827579498291015625000000,  242.66330887018978046398842707276344299316406250000000,
        241.48585864730839034564269240945577621459960937500000,  240.17493168727969532483257353305816650390625000000000,  238.73994486178312968149839434772729873657226562500000,  237.18881300188601812806155066937208175659179687500000,  235.52827291186838465364417061209678649902343750000000,  233.76411928197009615360002499073743820190429687500000,  231.90138058760800277013913728296756744384765625000000,  229.94445292991005658223002683371305465698242187500000,  227.89720366987009469994518440216779708862304687500000,  225.76305289989448965570772998034954071044921875000000,
        223.54503834519425709004281088709831237792968750000000,  221.24586766653462177600886207073926925659179687500000,  218.86796103812207547889556735754013061523437500000000,  216.41348611502795051819703076034784317016601562500000,  213.88438696910466774170345161110162734985351562500000,  211.28240818821538482552568893879652023315429687500000,  208.60911505362761886317457538098096847534179687500000,  205.86591050337412411863624583929777145385742187500000,  203.05404943416237983910832554101943969726562500000000,  200.17465077654108540627930779010057449340820312500000,
        197.22870768738738433967228047549724578857421875000000,  194.21709613320933840441284701228141784667968750000000,  191.14058208211324085823434870690107345581054687500000,  187.99982747781314174062572419643402099609375000000000,  184.79539513298527708684559911489486694335937500000000,  181.52775264950574296562990639358758926391601562500000,  178.19727544805851948694908060133457183837890625000000,  174.80424896801309841976035386323928833007812500000000,  171.34887007933545532978314440697431564331054687500000,  167.83124773077670965903962496668100357055664062500000,
        164.25140284193716411209607031196355819702148437500000,  160.60926743034875130433647427707910537719726562500000,  156.90468294778182212212414015084505081176757812500000,  153.13739778184904594127146992832422256469726562500000,  149.30706385885252984735416248440742492675781250000000,  145.41323226076266905693046282976865768432617187500000,  141.45534774206851125200046226382255554199218750000000,  137.43274199953197012291639111936092376708984375000000,  133.34462550769714539455890189856290817260742187500000,  129.19007768280727077581104822456836700439453125000000,
        124.96803507411374312141560949385166168212890625000000,  120.67727719969985855641425587236881256103515625000000,  116.31640953726829934566922020167112350463867187500000,  111.88384303951669096477417042478919029235839843750000,  107.37776935529323907303478335961699485778808593750000,  102.79613068204677972516947193071246147155761718750000,   98.13658282295004653406067518517374992370605468750000,   93.39644952923354992435633903369307518005371093750000,   88.57266550628513357423798879608511924743652343750000,   83.66170444267811490135500207543373107910156250000000,
         78.65948690881640459338086657226085662841796875000000,   73.56126067313989835838583530858159065246582031250000,   68.36144239368530861611361615359783172607421875000000,   63.05340385895166122054433799348771572113037109375000,   57.62917629453848178400221513584256172180175781250000,   52.07902943903809500625357031822204589843750000000000,   46.39085133978255015563263441435992717742919921875000,   40.54919512041342244401675998233258724212646484375000,   34.53373428348680818089633248746395111083984375000000,   28.31658242405636372041044523939490318298339843750000,
         21.85719405385460234469974238891154527664184570312500,   15.09129952453842271609119052300229668617248535156250,    7.90137735379261485491042549256235361099243164062500,    0.00000000000000000000000000000000000000000000000000,  247.57291409618687794136349111795425415039062500000000,  249.15970187967008087071008048951625823974609375000000,  249.98261443817756344287772662937641143798828125000000,  250.33784538646918349513725843280553817749023437500000,  250.34932276783723636981449089944362640380859375000000,  250.08621022970993408307549543678760528564453125000000,
        249.59292153876162956294137984514236450195312500000000,  248.90048460346804404252907261252403259277343750000000,  248.03183407581602182290225755423307418823242187500000,  247.00462221778084881407266948372125625610351562500000,  245.83285490606155576642777305096387863159179687500000,  244.52791062786479869828326627612113952636718750000000,  243.09920926227351856141467578709125518798828125000000,  241.55466967867769767508434597402811050415039062500000,  239.90103365183503569824097212404012680053710937500000,  238.14410170764156760014884639531373977661132812500000,
        236.28890897901885637111263349652290344238281250000000,  234.33985902131692569128063041716814041137695312500000,  232.30082743584725335495022591203451156616210937500000,  230.17524334280741982183826621621847152709960937500000,  227.96615429436138811070122756063938140869140625000000,  225.67627859828093050964525900781154632568359375000000,  223.30804792525435686911805532872676849365234375000000,  220.86364231392229839912033639848232269287109375000000,  218.34501915252357662211579736322164535522460937500000,  215.75393733212820279732113704085350036621093750000000,
        213.09197748666872485046042129397392272949218750000000,  210.36055902812509543764463160187005996704101562500000,  207.56095453019426599894359242171049118041992187500000,  204.69430189609903436576132662594318389892578125000000,  201.76161465576888076611794531345367431640625000000000,  198.76379066729793976264772936701774597167968750000000,  195.70161944221882777128485031425952911376953125000000,  192.57578826998064869258087128400802612304687500000000,  189.38688728132302685480681248009204864501953125000000,  186.13541356092383693976444192230701446533203125000000,
        182.82177439518312667132704518735408782958984375000000,  179.44628972006671574490610510110855102539062500000000,  176.00919381558651366503909230232238769531250000000000,  172.51063627693480384550639428198337554931640625000000,  168.95068227682827455282676964998245239257812500000000,  165.32931211862677400858956389129161834716796875000000,  161.64642006467775559030997101217508316040039062500000,  157.90181240848914967500604689121246337890625000000000,  154.09520474209219287331507075577974319458007812500000,  150.22621835056247618922498077154159545898437500000000,
        146.29437564321412423851143103092908859252929687500000,  142.29909450433601136865036096423864364624023437500000,  138.23968141404955645157315302640199661254882812500000,  134.11532315003731241631612647324800491333007812500000,  129.92507683098543225241883192211389541625976562500000,  125.66785799916451082935964222997426986694335937500000,  121.34242635792023179419629741460084915161132812500000,  116.94736867335015517710417043417692184448242187500000,  112.48107820876981577384867705404758453369140625000000,  107.94172987226893667411786736920475959777832031250000,
        103.32725000209191534850106108933687210083007812500000,   98.63527936253674965882964897900819778442382812500000,   93.86312743030754290884942747652530670166015625000000,   89.00771534934516182602237677201628684997558593750000,   84.06550391282912926271819742396473884582519531250000,   79.03240141862450229837122606113553047180175781250000,   73.90364394574758932776603614911437034606933593750000,   68.67363700931848313757654977962374687194824218750000,   63.33574176765708330094639677554368972778320312500000,   57.88197929773001959574685315601527690887451171875000,
         52.30260964219688446519285207614302635192871093750000,   46.58551157844708257016463903710246086120605468750000,   40.71522936140831916418392211198806762695312500000000,   34.67142800317036233082035323604941368103027343750000,   28.42621296521530283030188002157956361770629882812500,   21.93903096273751529565743112470954656600952148437500,   15.14560486939596550826081511331722140312194824218750,    7.92840602618053402750319946790114045143127441406250,    0.00000000000000000000000000000000000000000000000000,  251.89040220972319161774066742509603500366210937500000,
        253.47991403587317904566589277237653732299804687500000,  254.30594081898740910219203215092420578002929687500000,  254.66463690913639084101305343210697174072265625000000,  254.67990594201378939942514989525079727172851562500000,  254.42089592178263046662323176860809326171875000000000,  253.93201057333303083396458532661199569702148437500000,  253.24427158377244495568447746336460113525390625000000,  252.38061013596529846836347132921218872070312500000000,  251.35867710206792935423436574637889862060546875000000,  250.19247860779034908773610368371009826660156250000000,
        248.89339473143732561766228172928094863891601562500000,  247.47084808131850763857073616236448287963867187500000,  245.93276125245489538428955711424350738525390625000000,  244.28588064315260908188065513968467712402343750000000,  242.53601223334135283948853611946105957031250000000000,  240.68819739607533847447484731674194335937500000000000,  238.74684668622543881610909011214971542358398437500000,  236.71584345132927751365059521049261093139648437500000,  234.59862530336891950355493463575839996337890625000000,  232.39824904054557919153012335300445556640625000000000,
        230.11744298833301058948563877493143081665039062500000,  227.75864963220925574205466546118259429931640625000000,  225.32406065574051012845302466303110122680664062500000,  222.81564596276922429751721210777759552001953125000000,  220.23517787876656370826822239905595779418945312500000,  217.58425144683238272591552231460809707641601562500000,  214.86430152714598307284177280962467193603515625000000,  212.07661725381927908529178239405155181884765625000000,  209.22235428560708214718033559620380401611328125000000,  206.30254519669591672936803661286830902099609375000000,
        203.31810828367633803281933069229125976562500000000000,  200.26985500967361986113246530294418334960937500000000,  197.15849626273495687200920656323432922363281250000000,  193.98464757019240778390667401254177093505859375000000,  190.74883338178130998130654916167259216308593750000000,  187.45149051022679032030282542109489440917968750000000,  184.09297079760671067560906521975994110107421875000000,  180.67354305809945458349830005317926406860351562500000,  177.19339433195233368678600527346134185791015625000000,  173.65263047100592075366876088082790374755859375000000,
        170.05127606230047376811853609979152679443359375000000,  166.38927368264114647899987176060676574707031250000000,  162.66648246297324931219918653368949890136718750000000,  158.88267592646147363666386809200048446655273437500000,  155.03753904765804350063262972980737686157226562500000,  151.13066446135806586426042485982179641723632812500000,  147.16154772778617143558221869170665740966796875000000,  143.12958153453280374378664419054985046386718750000000,  139.03404868371828229101083707064390182495117187500000,  134.87411367332623512993450276553630828857421875000000,
        130.64881263198952865423052571713924407958984375000000,  126.35704130330378802682389505207538604736328125000000,  121.99754069426627722805278608575463294982910156250000,  117.56887989609839451077277772128582000732421875000000,  113.06943544516187216686375904828310012817382812500000,  108.49736640349118488302337937057018280029296875000000,  103.85058408299165932930918643251061439514160156250000,   99.12671498539562264795677037909626960754394531250000,   94.32305503738152197001909371465444564819335937500000,   89.43651249841106221083464333787560462951660156250000,
         84.46353589956316909592715092003345489501953125000000,   79.40002185926918798486440209671854972839355468750000,   74.24119532319610925696906633675098419189453125000000,   68.98145118545373577489954186603426933288574218750000,   63.61414046452168946643723757006227970123291015625000,   58.13127454987095887872783350758254528045654296875000,   52.52310422231938957793317968025803565979003906250000,   46.77749939832509085135825444012880325317382812500000,   40.87899584983048129060989595018327236175537109375000,   34.80725046254883636720478534698486328125000000000000,
         28.53436090834784977232629898935556411743164062500000,   22.01976643108995190800669661257416009902954101562500,   15.19918273416026899269581917906180024147033691406250,    7.95507427326269578315987018868327140808105468750000,    0.00000000000000000000000000000000000000000000000000,  256.22113555000953510898398235440254211425781250000000,  257.81332832114952680058195255696773529052734375000000,  258.64241796424937547271838411688804626464843750000000,  259.00452019004450221473234705626964569091796875000000,  259.02351455161488047451712191104888916015625000000000,
        258.76853355997445760294795036315917968750000000000000,  258.28397093581054377864347770810127258300781250000000,  257.60084210364010459670680575072765350341796875000000,  256.74207467659562098560854792594909667968750000000000,  255.72531798866407370951492339372634887695312500000000,  254.56457822560798831545980647206306457519531250000000,  253.27123683161471490166150033473968505859375000000000,  251.85471888528394401873811148107051849365234375000000,  250.32295041589992479202919639647006988525390625000000,  248.68268212190224630830925889313220977783203125000000,
        246.93972508148118549797800369560718536376953125000000,  245.09912651881418810262402985244989395141601562500000,  243.16530356468214790766069199889898300170898437500000,  241.14214685282621530859614722430706024169921875000000,  239.03310198834253696986706927418708801269531250000000,  236.84123447549117713606392499059438705444335937500000,  234.56928207306285116828803438693284988403320312500000,  232.21969744897063492317101918160915374755859375000000,  229.79468324731632833390904124826192855834960937500000,  227.29622114650354092191264498978853225708007812500000,
        224.72609610347842590272193774580955505371093750000000,  222.08591669977820970416360069066286087036132812500000,  219.37713229854941232588316779583692550659179687500000,  216.60104756700116013234946876764297485351562500000000,  213.75883480142033477022778242826461791992187500000000,  210.85154440180943424820725340396165847778320312500000,  207.88011377327234185941051691770553588867187500000000,  204.84537487634855779106146655976772308349609375000000,  201.74806060484874592475534882396459579467773437500000,  198.58881013463786757711204700171947479248046875000000,
        195.36817335818506080613587982952594757080078125000000,  192.08661449600688797545444685965776443481445312500000,  188.74451495616969509683258365839719772338867187500000,  185.34217549585238771214790176600217819213867187500000,  181.87981772384324585800641216337680816650390625000000,  178.35758496912788473309774417430162429809570312500000,  174.77554252788024768960895016789436340332031250000000,  171.13367728869681627656973432749509811401367187500000,  167.43189672335410023151780478656291961669921875000000,  163.67002721723477520754386205226182937622070312500000,
        159.84781169934188937986618839204311370849609375000000,  155.96490651591767573336255736649036407470703125000000,  152.02087747339604106855404097586870193481445312500000,  148.01519495488415145700855646282434463500976562500000,  143.94722798849042533220199402421712875366210937500000,  139.81623711417134359180636238306760787963867187500000,  135.62136585647868969317642040550708770751953125000000,  131.36163056114219216397032141685485839843750000000000,  127.03590829039035270398017019033432006835937500000000,  122.64292239058809741436562035232782363891601562500000,
        118.18122523955847213983361143618822097778320312500000,  113.64917754051988652008731150999665260314941406250000,  109.04492334147738574756658636033535003662109375000000,  104.36635970351584035142877837643027305603027343750000,   99.61109958955756837895023636519908905029296875000000,   94.77642605252016494432609761133790016174316406250000,   89.85923510001538261349196545779705047607421875000000,   84.85596359349891315559943905100226402282714843750000,   79.76249702744844682911207200959324836730957031250000,   74.57404973552523586022289237007498741149902343750000,
         69.28500648063405265020264778286218643188476562500000,   63.88870860204522728054143954068422317504882812500000,   58.37715823534425396701408317312598228454589843750000,   52.74059730806160217753131291829049587249755859375000,   46.96688726101798749823501566424965858459472656250000,   41.04055574684662843765181605704128742218017578125000,   34.94125186844733832458587130531668663024902343750000,   28.64106583290392293861259531695395708084106445312500,   22.09942972011261730358455679379403591156005859375000,   15.25205235425157468398538185283541679382324218750000,
          7.98139158158006889465241329162381589412689208984375,    0.00000000000000000000000000000000000000000000000000,  260.56494097186322278503212146461009979248046875000000,  262.15977284114728718122933059930801391601562500000000,  262.99187551809063734253868460655212402343750000000000,  263.35732668424026314824004657566547393798828125000000,  263.37998212990595447990926913917064666748046875000000,  263.12895902385923818656010553240776062011718750000000,  262.64864112490784009423805400729179382324218750000000,  261.97003756142515840110718272626399993896484375000000,
        261.11607228624200160993495956063270568847656250000000,  260.10439295893291955508175306022167205810546875000000,  258.94900565037727346862084232270717620849609375000000,  257.66129296130696957334293983876705169677734375000000,  256.25068219941380220916471444070339202880859375000000,  254.72510255652969135553576052188873291015625000000000,  253.09130872947130797001591417938470840454101562500000,  251.35511656270580260752467438578605651855468750000000,  249.52157876860366059190710075199604034423828125000000,  247.59511865880179470877919811755418777465820312500000,
        245.57963372447818528598872944712638854980468750000000,  243.47857709938350012635055463761091232299804687500000,  241.29502249131803637283155694603919982910156250000000,  239.03171654904940623964648693799972534179687500000000,  236.69112153558501177030848339200019836425781250000000,  234.27545042059364277520217001438140869140625000000000,  231.78669597032572369244007859379053115844726562500000,  229.22665503008275322827103082090616226196289062500000,  226.59694891504858560438151471316814422607421875000000,  223.89904061892633535535424016416072845458984375000000,
        221.13424939526930756983347237110137939453125000000000,  218.30376314919720925900037400424480438232421875000000,  215.40864898726945853013603482395410537719726562500000,  212.44986220351160000063828192651271820068359375000000,  209.42825392483911173258093185722827911376953125000000,  206.34457759567760604113573208451271057128906250000000,  203.19949444669623517256695777177810668945312500000000,  199.99357806421238592520239762961864471435546875000000,  196.72731815344522487976064439862966537475585937500000,  193.40112356920585057196149136871099472045898437500000,
        190.01532467089006672722462099045515060424804687500000,  186.57017504404569763210020028054714202880859375000000,  183.06585261771505201977561227977275848388671875000000,  179.50246019469011571345617994666099548339843750000000,  175.88002540030876730270392727106809616088867187500000,  172.19850004404051446726953145116567611694335937500000,  168.45775887643668511373107321560382843017578125000000,  164.65759771161864932764729019254446029663085937500000,  160.79773087185699864676280412822961807250976562500000,  156.87778789539055424029356800019741058349609375000000,
        152.89730943076668268076900858432054519653320312500000,  148.85574221979888420719362329691648483276367187500000,  144.75243304565339030887116678059101104736328125000000,  140.58662149117949979881814215332269668579101562500000,  136.35743131351631518555223010480403900146484375000000,  132.06386019174036050571885425597429275512695312500000,  127.70476754143751918491034302860498428344726562500000,  123.27886000888820206000673351809382438659667968750000,  118.78467415145405539078637957572937011718750000000000,  114.22055567041446977327723288908600807189941406250000,
        109.58463437449029242998221889138221740722656250000000,  104.87479379697255410519574070349335670471191406250000,  100.08863403755064780398242874071002006530761718750000,   95.22342590736083423053059959784150123596191406250000,   90.27605375402805520934634841978549957275390625000000,   85.24294332428360121411969885230064392089843750000000,   80.11996950944454454202059423550963401794433593750000,   74.90233652044970824590564006939530372619628906250000,   69.58441944913451493448519613593816757202148437500000,   64.15955038869975624038488604128360748291015625000000,
         58.61972262895280749717130674980580806732177734375000,   52.95516962900543944670062046498060226440429687500000,   47.15374471731232830506996833719313144683837890625000,   41.19996777037496826778806280344724655151367187500000,   35.07348043274275539715745253488421440124511718750000,   28.74636575373075331185646064113825559616088867187500,   22.17804894000130033759887737687677145004272460937500,   15.30423221204736172751381673151627182960510253906250,    8.00736706798332953383123822277411818504333496093750,    0.00000000000000000000000000000000000000000000000000,
        264.92164979855277806564117781817913055419921875000000,  266.51908011774150963901774957776069641113281250000000,  267.35414747489858200424350798130035400390625000000000,  267.72289211775506601043161936104297637939453125000000,  267.74914638851612380676669999957084655761718750000000,  267.50201226508289664707263000309467315673828125000000,  267.02586359092407519710832275450229644775390625000000,  266.35170317169951204050448723137378692626953125000000,  265.50245121858125685321283526718616485595703125000000,  264.49575359163816301588667556643486022949218750000000,
        263.34561608486325212652445770800113677978515625000000,  262.06342226084831281696096993982791900634765625000000,  260.65860143155026662498130463063716888427734375000000,  259.13908569776594958966597914695739746093750000000000,  257.51163347290429328495520167052745819091796875000000,  255.78206505764782718870264943689107894897460937500000,  253.95543831363951881030516233295202255249023437500000,  252.03618236458285650769539643079042434692382812500000,  250.02820115904074782520183362066745758056640625000000,  247.93495492635088339739013463258743286132812500000000,
        245.75952510966413910864503122866153717041015625000000,  243.50466674194771599104569759219884872436523437500000,  241.17285113509811367293877992779016494750976562500000,  238.76630099447632460396562237292528152465820312500000,  236.28701953695772886021586600691080093383789062500000,  233.73681480747143268672516569495201110839843750000000,  231.11732010992076880029344465583562850952148437500000,  228.43001126214531382174754980951547622680664062500000,  225.67622123015738111462269444018602371215820312500000,  222.85715257981726722391613293439149856567382812500000,
        219.97388809432172251945303287357091903686523437500000,  217.02739983624323372168873902410268783569335937500000,  214.01855687825863583384489174932241439819335937500000,  210.94813188343007936964568216353654861450195312500000,  207.81680668121535404679889325052499771118164062500000,  204.62517695725054522881691809743642807006835937500000,  201.37375615183020727272378280758857727050781250000000,  198.06297864273307141047553159296512603759765625000000,  194.69320227168782366788946092128753662109375000000000,  191.26471025962175076529092621058225631713867187500000,
        187.77771254329582006903365254402160644531250000000000,  184.23234655451071262177720200270414352416992187500000,  180.62867745234262883968767710030078887939453125000000,  176.96669780844678143694181926548480987548828125000000,  173.24632673497492874048475641757249832153320312500000,  169.46740843371324558574997354298830032348632812500000,  165.62971013324718683179526124149560928344726562500000,  161.73291936784247013747517485171556472778320312500000,  157.77664053674303090701869223266839981079101562500000,  153.76039066506984909210586920380592346191406250000000,
        149.68359426661234579114534426480531692504882812500000,  145.54557718346634942463424522429704666137695312500000,  141.34555924628583056801289785653352737426757812500000,  137.08264556001142864261055365204811096191406250000000,  132.75581617082039542765414807945489883422851562500000,  128.36391380729355660150758922100067138671875000000000,  123.90562930770872185348707716912031173706054687500000,  119.37948423936633446373889455571770668029785156250000,  114.78381007559505633253138512372970581054687500000000,  110.11672310814465447492693783715367317199707031250000,
        105.37609401741633519122842699289321899414062500000000,  100.55951067121020514605334028601646423339843750000000,   95.66423223014334098479594103991985321044921875000000,   90.68713193618391699146741302683949470520019531250000,   85.62462494159305492757994215935468673706054687500000,   80.47257602329973735777457477524876594543457031250000,   75.22617972917137763033679220825433731079101562500000,   69.87980191265357632346422178670763969421386718750000,   64.42676582854751643480994971469044685363769531250000,   58.85905630533506638357721385546028614044189453125000,
         53.16689869658461731205534306354820728302001953125000,   47.33813856118504759251663926988840103149414062500000,   41.35728832360954498881255858577787876129150390625000,   35.20398247672923019990776083432137966156005859375000,   28.85029720249533369269556715153157711029052734375000,   22.25565110954534731035892036743462085723876953125000,   15.35574007568796162104263203218579292297363281250000,    8.03300949859666779673261771677061915397644042968750,    0.00000000000000000000000000000000000000000000000000,  269.29109765101981111001805402338504791259765625000000,
        270.89108692108101195117342285811901092529296875000000,  271.72907201502107454871293157339096069335937500000000,  272.10105632777873552186065353453159332275390625000000,  272.13084906292283449147362262010574340820312500000000,  271.88753715903527563568786717951297760009765625000000,  271.41548459452798169877496547996997833251953125000000,  270.74568783213675260412856005132198333740234375000000,  269.90106326847831041959580034017562866210937500000000,  268.89925484929699450731277465820312500000000000000000,  267.75426794163496424516779370605945587158203125000000,
        266.47748688879511291816015727818012237548828125000000,  265.07834279729792115176678635179996490478515625000000,  263.56477044007039012285531498491764068603515625000000,  261.94353168376778739911969751119613647460937500000000,  260.22045099545056245915475301444530487060546875000000,  258.40059106924582010833546519279479980468750000000000,  256.48838649632142505652154795825481414794921875000000,  254.48774730898941243140143342316150665283203125000000,  252.40214042795227555870951618999242782592773437500000,  250.23465459502628505106258671730756759643554687500000,
        247.98805275602848041671677492558956146240234375000000,  245.66481476318878662823408376425504684448242187500000,  243.26717250889782917511183768510818481445312500000000,  240.79713906859589656050957273691892623901367187500000,  238.25653304766504447798070032149553298950195312500000,  235.64699904824763621036254335194826126098632812500000,  232.97002496581242780848697293549776077270507812500000,  230.22695667098054173038690350949764251708984375000000,  227.41901051516890674975002184510231018066406250000000,  224.54728400893586126585432793945074081420898437500000,
        221.61276495239926020985876675695180892944335937500000,  218.61633924263060180237516760826110839843750000000000,  215.55879753980170221439038868993520736694335937500000,  212.44084093934020529559347778558731079101562500000000,  209.26308576940849093261931557208299636840820312500000,  206.02606761012762603968440089374780654907226562500000,  202.73024461194916057138470932841300964355468750000000,  199.37600017453661394029040820896625518798828125000000,  195.96364503373428078703000210225582122802734375000000,  192.49341879210160755064862314611673355102539062500000,
        188.96549091760502392389753367751836776733398437500000,  185.37996122497676765306096058338880538940429687500000,  181.73685984461081943663884885609149932861328125000000,  178.03614667433535601048788521438837051391601562500000,  174.27771029964085869323753286153078079223632812500000,  170.46136635760473154732608236372470855712890625000000,  166.58685530845755806694796774536371231079101562500000,  162.65383956603247384009591769427061080932617187500000,  158.66189992370331651727610733360052108764648437500000,  154.61053119519033316464629024267196655273437500000000,
        150.49913696896996384566591586917638778686523437500000,  146.32702334989542691801034379750490188598632812500000,  142.09339153062526861504011321812868118286132812500000,  137.79732899670531764968472998589277267456054687500000,  133.43779912015975241956766694784164428710937500000000,  129.01362883380912194297707173973321914672851562500000,  124.52349399754146475061133969575166702270507812500000,  119.96590196184330068263079738244414329528808593750000,  115.33917069370721719678840599954128265380859375000000,  110.64140364215685963245050515979528427124023437500000,
        105.87045926542492679800488986074924468994140625000000,  101.02391379009597471849701832979917526245117187500000,   96.09901528004114368286536773666739463806152343750000,   91.09262639129865135600994108244776725769042968750000,   86.00115216993204114714899333193898200988769531250000,   80.82044773765885281591181410476565361022949218750000,   75.54569841150407683016965165734291076660156250000000,   70.17126121376526270978501997888088226318359375000000,   64.69045094488943448141071712598204612731933593750000,   59.09524433452057223803421948105096817016601562500000,
         53.37585897311469551596019300632178783416748046875000,   47.52013297374635669712006347253918647766113281250000,   41.51257161519684757422510301694273948669433593750000,   35.33280252874343574376325705088675022125244140625000,   28.95289530387653798015890060923993587493896484375000,   22.33226221191718963154926314018666744232177734375000,   15.40659303541406011106573714641854166984558105468750,    8.05832730658095641729232738725841045379638671875000,    0.00000000000000000000000000000000000000000000000000,  273.67312428569368876196676865220069885253906250000000,
        275.27563411004166482598520815372467041015625000000000,  276.11649134862744858764926902949810028076171875000000,  276.49166311068609047651989385485649108886718750000000,  276.52493576541922948308638297021389007568359375000000,  276.28538136356633003742899745702743530273437500000000,  275.81735407207696653131279163062572479248046875000000,  275.15184399634432566017494536936283111572265625000000,  274.31176365330628641459043137729167938232421875000000,  273.31475496904010924481553956866264343261718750000000,  272.17482274460257940518204122781753540039062500000000,
        270.90335193537504210326005704700946807861328125000000,  269.50977524672845220266026444733142852783203125000000,  268.00202990237107769644353538751602172851562500000000,  266.38688097590005554593517445027828216552734375000000,  264.67015682959595324064139276742935180664062500000000,  262.85692469370758317381842061877250671386718750000000,  260.95162430456736046835430897772312164306640625000000,  258.95817142893292839289642870426177978515625000000000,  256.88003930118958351158653385937213897705078125000000,  254.72032355408768466986657585948705673217773437500000,
        252.48179460654330341640161350369453430175781250000000,  250.16694037707432585193600971251726150512695312500000,  247.77800143416124001305433921515941619873046875000000,  245.31700016102684003271861001849174499511718750000000,  242.78576512955189059539407026022672653198242187500000,  240.18595159924231552395212929695844650268554687500000,  237.51905885118245009834936354309320449829101562500000,  234.78644491270779326441697776317596435546875000000000,  231.98933911168165877825231291353702545166015625000000,  229.12885280969166501563449855893850326538085937500000,
        226.20598859407547820410400163382291793823242187500000,  223.22164815433526996457658242434263229370117187500000,  220.17663902550114585210394579917192459106445312500000,  217.07168034662788613786688074469566345214843750000000,  213.90740775483189395345107186585664749145507812500000,  210.68437751257050649655866436660289764404296875000000,  207.40306994707125909371825400739908218383789062500000,  204.06389226503546296953572891652584075927734375000000,  200.66718079226509985346638131886720657348632812500000,  197.21320267613154442187806125730276107788085937500000,
        193.70215707835694729510578326880931854248046875000000,  190.13417587602765479459776543080806732177734375000000,  186.50932387976351378711115103214979171752929687500000,  182.82759856921785512895439751446247100830078125000000,  179.08892933728088792122434824705123901367187500000000,  175.29317622520292729859647806733846664428710937500000,  171.44012812101823328703176230192184448242187500000000,  167.52950038276787836366565898060798645019531250000000,  163.56093183566889592839288525283336639404296875000000,  159.53398107803187144781986717134714126586914062500000,
        155.44812201375066251785028725862503051757812500000000,  151.30273850875468610865937080234289169311523437500000,  147.09711804386071776207245420664548873901367187500000,  142.83044420560483445115096401423215866088867187500000,  138.50178781801119498595653567463159561157226562500000,  134.11009646937571915259468369185924530029296875000000,  129.65418212560058464077883400022983551025390625000000,  125.13270644070557580107561079785227775573730468750000,  120.54416326929387537347793113440275192260742187500000,  115.88685774562161157064110739156603813171386718750000,
        111.15888110609824934726930223405361175537109375000000,  106.35808017688543714029947295784950256347656250000000,  101.48202009658032807237759698182344436645507812500000,   96.52793835151852874787437031045556068420410156250000,   91.49268749959139768179738894104957580566406250000000,   86.37266293937696559623873326927423477172851562500000,   81.16371056916929660474124830216169357299804687500000,   75.86100688197355168540525482967495918273925781250000,   70.45890045258593659127654973417520523071289062500000,   64.95069798920738435299426782876253128051757812500000,
         59.32836846471359137922263471409678459167480468750000,   53.58212202985502869978517992421984672546386718750000,   47.69978965789965030808161827735602855682373046875000,   41.66586977170755545785141293890774250030517578125000,   35.45998341556551025632870732806622982025146484375000,   29.05419384692510931245124083943665027618408203125000,   22.40790724694135960248786432202905416488647460937500,   15.45680753762208325952087761834263801574707031250000,    8.08332860878637404766777763143181800842285156250000,    0.00000000000000000000000000000000000000000000000000,
        278.06757344036617496385588310658931732177734375000000,  279.67256648056712720062932930886745452880859375000000,  280.51625156722883502879994921386241912841796875000000,  280.89456007744064436337794177234172821044921875000000,  280.93125584511921033481485210359096527099609375000000,  280.69539618433890382220852188766002655029296875000000,  280.23132550710573696051142178475856781005859375000000,  279.57002755232679191976785659790039062500000000000000,  278.73441089928280689491657540202140808105468750000000,  277.74211535778727011347655206918716430664062500000000,
        276.60714503407695019632228650152683258056640625000000,  275.34088533856248659503762610256671905517578125000000,  273.95277039270115437830099835991859436035156250000000,  272.45073966396796549815917387604713439941406250000000,  270.84156120236377773835556581616401672363281250000000,  269.13106701150456956384005025029182434082031250000000,  267.32432858027095790021121501922607421875000000000000,  265.42579048844885392099968157708644866943359375000000,  263.43937391037900397350313141942024230957031250000000,  261.36855804089992716399137862026691436767578125000000,
        259.21644502260824083350598812103271484375000000000000,  256.98581233681221647202619351446628570556640625000000,  254.67915552595118811041174922138452529907226562500000,  252.29872335823620232986286282539367675781250000000000,  249.84654701167534085470833815634250640869140625000000,  247.32446447202687522803898900747299194335937500000000,  244.73414106055568595365912187844514846801757812500000,  242.07708680159856839964049868285655975341796875000000,  239.35467118584506351908203214406967163085937500000000,  236.56813576848745128700102213770151138305664062500000,
        233.71860495191805284775909967720508575439453125000000,  230.80709523334306254582770634442567825317382812500000,  227.83452314343577427280251868069171905517578125000000,  224.80171205926163224830816034227609634399414062500000,  221.70939804045974597102031111717224121093750000000000,  218.55823481002633457137562800198793411254882812500000,  215.34879797850513227785995695739984512329101562500000,  212.08158859178095667630259413272142410278320312500000,  208.75703606711135762452613562345504760742187500000000,  205.37550056881400450947694480419158935546875000000000,
        201.93727486360450029678759165108203887939453125000000,  198.44258568549847154827148187905550003051757812500000,  194.89159463108083514271129388362169265747070312500000,  191.28439859747774676179687958210706710815429687500000,  187.62102976726205838531313929706811904907226562500000,  183.90145513650284669893153477460145950317382812500000,  180.12557557397096275053627323359251022338867187500000,  176.29322439085694895766209810972213745117187500000000,  172.40416539093851611141872126609086990356445312500000,  168.45809036060151697711262386292219161987304687500000,
        164.45461594605961863635457120835781097412109375000000,  160.39327985102050888599478639662265777587890625000000,  156.27353627127700974597246386110782623291015625000000,  152.09475046244406826190243009477853775024414062500000,  147.85619231226326064643217250704765319824218750000000,  143.55702875816595565083844121545553207397460937500000,  139.19631485227608891364070586860179901123046875000000,  134.77298322724814738649001810699701309204101562500000,  130.28583165387621534136997070163488388061523437500000,  125.73350830057252380811405600979924201965332031250000,
        121.11449419902437796281446935608983039855957031250000,  116.42708228026641847918654093518853187561035156250000,  111.66935215762704558528639608994126319885253906250000,  106.83913957789107485041313339024782180786132812500000,  101.93399911037280958225892391055822372436523437500000,   96.95115815117593172089982545003294944763183593750000,   91.88745961830215946974931284785270690917968750000000,   86.73928969420003909363003913313150405883789062500000,   81.50248546015217243621009401977062225341796875000000,   76.17221496840009820061823120340704917907714843750000,
         70.74281870797517512983176857233047485351562500000000,   65.20759563654776513885735766962170600891113281250000,   59.55850729329466020089967059902846813201904296875000,   53.78575669494663458181094028986990451812744140625000,   47.87716796442848732340280548669397830963134765625000,   41.81723294298905813093369943089783191680908203125000,   35.58556634806614482613440486602485179901123046875000,   29.15422535195474651459335291292518377304077148437500,   22.48261028010639250851454562507569789886474609375000,   15.50639941680735489626385970041155815124511718750000,
          8.10802122137674530222284374758601188659667968750000,    0.00000000000000000000000000000000000000000000000000,  282.47429268763033860523137263953685760498046875000000,  284.08173262141559689553105272352695465087890625000000,  284.92820250239526558289071545004844665527343750000000,  285.30959851593496523491921834647655487060546875000000,  285.34966225458549615723313763737678527832031250000000,  285.11743644643701145469094626605510711669921875000000,  284.65725580764262758748373016715049743652343750000000,  284.00009770628372507417225278913974761962890625000000,
        283.16886673257727125019300729036331176757812500000000,  282.18120049155402284668525680899620056152343750000000,  281.05110227523545063377241604030132293701171875000000,  279.78995780274215121607994660735130310058593750000000,  278.40720244085207468742737546563148498535156250000000,  276.91077770707283889350946992635726928710937500000000,  275.30745441187411870487267151474952697753906250000000,  273.60306796231105863625998608767986297607421875000000,  271.80269384586409842086140997707843780517578125000000,  269.91078120309032328805187717080116271972656250000000,
        267.93125630977328910375945270061492919921875000000000,  265.86760399053008541159215383231639862060546875000000,  263.72293254126742567677865736186504364013671875000000,  261.50002612170999327645404264330863952636718750000000,  259.20138748497151937044691294431686401367187500000000,  256.82927315438070081654586829245090484619140625000000,  254.38572262437421045433438848704099655151367187500000,  251.87258277980544107776950113475322723388671875000000,  249.29152844947469702674425207078456878662109375000000,  246.64407980392181229944981168955564498901367187500000,
        243.93161715352007945512013975530862808227539062500000,  241.15539358624309329570678528398275375366210937500000,  238.31654579508821711897326167672872543334960937500000,  235.41610337591652069022529758512973785400390625000000,  232.45499682231414340094488579779863357543945312500000,  229.43406440128848089443636126816272735595703125000000,  226.35405805947132762412365991622209548950195312500000,  223.21564848198568142834119498729705810546875000000000,  220.01942940372993007258628495037555694580078125000000,  216.76592125438574498730304185301065444946289062500000,
        213.45557420308054474844539072364568710327148437500000,  210.08877065563720520913193468004465103149414062500000,  206.66582724617958888302382547408342361450195312500000,  203.18699635508744449907680973410606384277343750000000,  199.65246717654954977660963777452707290649414062500000,  196.06236635093685549691144842654466629028320312500000,  192.41675816964101386474794708192348480224609375000000,  188.71564435264579628892533946782350540161132812500000,  184.95896339168362487725971732288599014282226562500000,  181.14658944413022823027858976274728775024414062500000,
        177.27833075455146172316744923591613769531250000000000,  173.35392757174591338298341725021600723266601562500000,  169.37304951888637560841743834316730499267578125000000,  165.33529236255174055258976295590400695800781250000000,  161.24017411255218235055508557707071304321289062500000,  157.08713036785806593798042740672826766967773437500000,  152.87550880383739126955333631485700607299804687500000,  148.60456267133596952589869033545255661010742187500000,  144.27344314751482556857808958739042282104492187500000,  139.88119033994439632806461304426193237304687500000000,
        135.42672269675125562571338377892971038818359375000000,  130.90882451322562474160804413259029388427734375000000,  126.32613114452033187262713909149169921875000000000000,  121.67711142833672965934965759515762329101562500000000,  116.96004668146827043528901413083076477050781250000000,  112.17300544633327774590725312009453773498535156250000,  107.31381290854881171981105580925941467285156250000000,  102.38001355498026612167450366541743278503417968750000,   97.36882514931097887256328249350190162658691406250000,   92.27708140058582841902534710243344306945800781250000,
         87.10115968113373696724011097103357315063476562500000,   81.83688863809956615114060696214437484741210937500000,   76.47942824433363284697406925261020660400390625000000,   71.02311124447119539127015741541981697082519531250000,   65.46122916839085803530906559899449348449707031250000,   59.78573642694107803663428057916462421417236328125000,   53.98682919199541174748446792364120483398437500000000,   48.05232501015797907939486321993172168731689453125000,   41.96670940093316204411166836507618427276611328125000,   35.70959100152934695415751775726675987243652343750000,
         29.25302113329610875780417700298130512237548828125000,   22.55639448856013018485100474208593368530273437500000,   15.55538392555032700670381018426269292831420898437500,    8.13241267450090532520334818400442600250244140625000,    0.00000000000000000000000000000000000000000000000000,  286.89313329542699193552834913134574890136718750000000,  288.50298477686567366617964580655097961425781250000000,  289.35219759123936000833055004477500915527343750000000,  289.73663325985796745953848585486412048339843750000000,  289.78001142269806678086752071976661682128906250000000,
        289.55136037187452302532619796693325042724609375000000,  289.09500518903286092609050683677196502685546875000000,  288.44191687146286540155415423214435577392578125000000,  287.61499597495765101484721526503562927246093750000000,  286.63187781871164361291448585689067840576171875000000,  285.50656476987796850153245031833648681640625000000000,  284.25044271991595223880722187459468841552734375000000,  282.87294812128652665705885738134384155273437500000000,  281.38202436011198415144463069736957550048828125000000,  279.78444480493448054403415881097316741943359375000000,
        278.08604804316740910508087836205959320068359375000000,  276.29191331702520528779132291674613952636718750000000,  274.40649406276168065232923254370689392089843750000000,  272.43372137061902549248770810663700103759765625000000,  270.37708538515067857588292099535465240478515625000000,  268.23970022168782634253147989511489868164062500000000,  266.02435635902958210863289423286914825439453125000000,  263.73356337452162279078038409352302551269531250000000,  261.36958513118173641487373970448970794677734375000000,  258.93446899332207067345734685659408569335937500000000,
        256.43007026480069043827825225889682769775390625000000,  253.85807276561459389085939619690179824829101562500000,  251.22000625688002628521644510328769683837890625000000,  248.51726127033492730333819054067134857177734375000000,  245.75110178191053478258254472166299819946289062500000,  242.92267607961076691935886628925800323486328125000000,  240.03302610679108397562231402844190597534179687500000,  237.08309550785571673259255476295948028564453125000000,  234.07373656068605782820668537169694900512695312500000,  231.00571614606650427958811633288860321044921875000000,
        227.87972087696553558089362923055887222290039062500000,  224.69636148824417887226445600390434265136718750000000,  221.45617656905645276310679037123918533325195312500000,  218.15963570498763601790415123105049133300781250000000,  214.80714208416566179948858916759490966796875000000000,  211.39903461063263989672122988849878311157226562500000,  207.93558955874786420281452592462301254272460937500000,  204.41702179395434768593986518681049346923828125000000,  200.84348557757920161748188547790050506591796875000000,  197.21507496620094457284722011536359786987304687500000,
        193.53182380926887162786442786455154418945312500000000,  189.79370534188464603175816591829061508178710937500000,  186.00063136274047792539931833744049072265625000000000,  182.15245097992675482601043768227100372314453125000000,  178.24894889942970621632412075996398925781250000000000,  174.28984322236331649946805555373430252075195312500000,  170.27478270698554752016207203269004821777343750000000,  166.20334343994585424297838471829891204833984375000000,  162.07502484750025928406103048473596572875976562500000,  157.88924496098806571353634353727102279663085937500000,
        153.64533483088808907268685288727283477783203125000000,  149.34253195921323253969603683799505233764648437500000,  144.97997258947535215156676713377237319946289062500000,  140.55668265512213110923767089843750000000000000000000,  136.07156713871285091954632662236690521240234375000000,  131.52339753177508896442304830998182296752929687500000,  126.91079700455884449183940887451171875000000000000000,  122.23222278921812744556518737226724624633789062500000,  117.48594513996343380313192028552293777465820312500000,  112.67002204602650294873456005007028579711914062500000,
        107.78226861823027604714297922328114509582519531250000,  102.82021971836611839989927830174565315246582031250000,   97.78108390821516593405249295756220817565917968750000,   92.66168609348122231494926381856203079223632812500000,   87.45839521887255330057087121531367301940917968750000,   82.16703185841191725558019243180751800537109375000000,   76.78274824658741692928742850199341773986816406250000,   71.29986970605307305959286168217658996582031250000000,   65.71168064395742192118632374331355094909667968750000,   60.01012863168813282754854299128055572509765625000000,
         54.18540327000241774157984764315187931060791015625000,   48.22531578878136571120194275863468647003173828125000,   42.11434563214704951406019972637295722961425781250000,   35.83209559104329144929579342715442180633544921875000,   29.35061135821709754623043409083038568496704101562500,   22.62928220430862324974441435188055038452148437500000,   15.60377576268800225989252794533967971801757812500000,    8.15651022607996623037251993082463741302490234375000,    0.00000000000000000000000000000000000000000000000000,  291.32395009427028753634658642113208770751953125000000,
        292.93617871596740087625221349298954010009765625000000,  293.78809374826568046046304516494274139404296875000000,  294.17552256370743180013960227370262145996093750000000,  294.22216313340157967104460112750530242919921875000000,  293.99702946267257175350096076726913452148437500000000,  293.54443706196917673878488130867481231689453125000000,  292.89535056180505989686935208737850189208984375000000,  292.07266644375516762011102400720119476318359375000000,  291.09401766702802660802262835204601287841796875000000,  289.97340557135851213388377800583839416503906250000000,
        288.72221609340334680382511578500270843505859375000000,  287.34988662199833697741269133985042572021484375000000,  285.86436224189526456029852852225303649902343750000000,  284.27241868987658790501882322132587432861328125000000,  282.57989752437572406051913276314735412597656250000000,  280.79188151346102131356019526720046997070312500000000,  278.91282814031796988274436444044113159179687500000000,  276.94667304039393229686538688838481903076171875000000,  274.89691138759798150204005651175975799560546875000000,  272.76666280372364781214855611324310302734375000000000,
        270.55872375003326624209876172244548797607421875000000,  268.27561026636601582140428945422172546386718750000000,  265.91959316686649117400520481169223785400390625000000,  263.49272726840877112408634275197982788085937500000000,  260.99687584561814901462639681994915008544921875000000,  258.43373122808566222374793142080307006835937500000000,  255.80483224980201839571236632764339447021484375000000,  253.11157910700941897630400490015745162963867187500000,  250.35524606415728499086981173604726791381835937500000,  247.53699235841105519284610636532306671142578125000000,
        244.65787158408866730496811214834451675415039062500000,  241.71883978439615248134941793978214263916015625000000,  238.72076243520697858002677094191312789916992187500000,  235.66442047166506768007820937782526016235351562500000,  232.55051548107400094522745348513126373291015625000000,  229.37967416335541770422423724085092544555664062500000,  226.15245214216750468949612695723772048950195312500000,  222.86933719469317338734981603920459747314453125000000,  219.53075195545420683629345148801803588867187500000000,  216.13705613874458322243299335241317749023437500000000,
        212.68854831497745294655032921582460403442382812500000,  209.18546726805726621023495681583881378173828125000000,  205.62799295353462980529002379626035690307617187500000,  202.01624707052647522687038872390985488891601562500000,  198.35029325397715638246154412627220153808593750000000,  194.63013688758923080968088470399379730224609375000000,  190.85572453147835858544567599892616271972656250000000,  187.02694295210645236693380866199731826782226562500000,  183.14361773511495812272187322378158569335937500000000,  179.20551145407961257660645060241222381591796875000000,
        175.21232135967807153065223246812820434570312500000000,  171.16367654397836872703919652849435806274414062500000,  167.05913452312671552135725505650043487548828125000000,  162.89817716815727521861845161765813827514648437500000,  158.68020589733043834712589159607887268066406250000000,  154.40453602354219242442923132330179214477539062500000,  150.07039012588225546096509788185358047485351562500000,  145.67689028397475681231298949569463729858398437500000,  141.22304897547428481630049645900726318359375000000000,  136.70775838851909611548762768507003784179687500000000,
        132.12977783866867298456782009452581405639648437500000,  127.48771889917766486632899614050984382629394531250000,  122.78002774780827621725620701909065246582031250000000,  118.00496409343558923410455463454127311706542968750000,  113.16057585802731466628756606951355934143066406250000,  108.24466853455903958547423826530575752258300781250000,  103.25476778985984083192306570708751678466796875000000,   98.18807338903819470488087972626090049743652343750000,   93.04140181658844710455014137551188468933105468750000,   87.81111395026417199005663860589265823364257812500000,
         82.49302263166202919819625094532966613769531250000000,   77.08227267900535650824167532846331596374511718750000,   71.57318229772583606518310261890292167663574218750000,   65.95902906082051231351215392351150512695312500000000,   60.23175397368112271578866057097911834716796875000000,   54.38154032528238701615919126197695732116699218750000,   48.39619327489222655458434019237756729125976562500000,   42.26018642497389521395234623923897743225097656250000,   35.95311694231843091529299272224307060241699218750000,   29.44702510228719560814170108642429113388061523437500,
         22.70129495482021653174342645797878503799438476562500,   15.65158909980081958224218396935611963272094726562500,    8.18032087477368463623861316591501235961914062500000,    0.00000000000000000000000000000000000000000000000000,  295.76660135076065216708229854702949523925781250000000,  297.38117360795297372533241286873817443847656250000000,  298.23575124321592966225580312311649322509765625000000,  298.62612798359344878917909227311611175537109375000000,  298.67598041000036346304113976657390594482421875000000,  298.45430838919656935104285366833209991455078125000000,
        298.00541792544925101537955924868583679199218750000000,  297.36026729013462954753777012228965759277343750000000,  296.54174885594011357170529663562774658203125000000000,  295.56749315432688263172167353332042694091796875000000,  294.45150040257777845909004099667072296142578125000000,  293.20515646397683440227410756051540374755859375000000,  291.83789952402327116942615248262882232666015625000000,  290.35767620673567535050096921622753143310546875000000,  288.77126443897464014298748224973678588867187500000000,  287.08450855361536468990379944443702697753906250000000,
        285.30249462960586015469743870198726654052734375000000,  283.42968396342274672861094586551189422607421875000000,  281.47001648289199238206492736935615539550781250000000,  279.42699211852936969080474227666854858398437500000000,  277.30373570497778246135567314922809600830078125000000,  275.10304937035169814407709054648876190185546875000000,  272.82745527804354424006305634975433349609375000000000,  270.47923082952610229767742566764354705810546875000000,  268.06043790384097746937186457216739654541015625000000,  265.57294732743014264997327700257301330566406250000000,
        263.01845948977114630906726233661174774169921875000000,  260.39852181480193848983617499470710754394531250000000,  257.71454364436789319370291195809841156005859375000000,  254.96780897348710936967108864337205886840820312500000,  252.15948738805460038747696671634912490844726562500000,  249.29064348659844085887016262859106063842773437500000,  246.36224501376256057483260519802570343017578125000000,  243.37516989063152550443192012608051300048828125000000,  240.33021229312174682490876875817775726318359375000000,  237.22808790242720533569809049367904663085937500000000,
        234.06943842941669231549894902855157852172851562500000,  230.85483549679045722768933046609163284301757812500000,  227.58478394784015108598396182060241699218750000000000,  224.25972463813590707104594912379980087280273437500000,  220.88003675585943597070581745356321334838867187500000,  217.44603970738759812775242608040571212768554687500000,  213.95799459676388210027653258293867111206054687500000,  210.41610532059729621323640458285808563232421875000000,  206.82051929346070551218872424215078353881835937500000,  203.17132781281600273359799757599830627441406250000000,
        199.46856606668629297018924262374639511108398437500000,  195.71221278154962419648654758930206298828125000000000,  191.90218950207008674624375998973846435546875000000000,  188.03835948813016898384375963360071182250976562500000,  184.12052620798732505136285908520221710205078125000000,  180.14843139902532698215509299188852310180664062500000,  176.12175265924875589007569942623376846313476562500000,  172.04010052306207967376394663006067276000976562500000,  167.90301496359819566350779496133327484130859375000000,  163.70996125043134838961123023182153701782226562500000,
        159.46032507530719612987013533711433410644531250000000,  155.15340683875129457192088011652231216430664062500000,  150.78841496603510563545569311827421188354492187500000,  146.36445809060595024675421882420778274536132812500000,  141.88053590488803479274793062359094619750976562500000,  137.33552842984215658361790701746940612792968750000000,  132.72818339244631147266773041337728500366210937500000,  128.05710131962248965464823413640260696411132812500000,  123.32071785152214715708396397531032562255859375000000,  118.51728263716502453917200909927487373352050781250000,
        113.64483398778190803568577393889427185058593750000000,  108.70116820821435510424635140225291252136230468750000,  103.68380217517999142273765755817294120788574218750000,   98.58992723888606235504994401708245277404785156250000,   93.41635182294075434583646710962057113647460937500000,   88.15942907851247412054362939670681953430175781250000,   82.81496443655802863759163301438093185424804687500000,   77.37809560349809601120796287432312965393066406250000,   71.84313395583731676197203341871500015258789062500000,   66.20335050561548939640488242730498313903808593750000,
         60.45067995130366256262277602218091487884521484375000,   54.57529951595670070219057379290461540222167968750000,   48.56500852171622995001598610542714595794677734375000,   42.40427495127190837820307933725416660308837890625000,   36.07269055826190395919184084050357341766357421875000,   29.54229040144035778325815044809132814407348632812500,   22.77245350121966538381457212381064891815185546875000,   15.69883760613441303632953349733725190162658691406250,    8.20385137218387860968960012542083859443664550781250,    0.00000000000000000000000000000000000000000000000000,
        300.22094864701415417584939859807491302490234375000000,  301.83783190344871627530665136873722076416015625000000,  302.69503358456194064274313859641551971435546875000000,  303.08831426350195670238463208079338073730468750000000,  303.14132940468692822832963429391384124755859375000000,  302.92306488346201831518555991351604461669921875000000,  302.47781726439620797464158385992050170898437500000000,  301.83653847066130992971011437475681304931640625000000,  301.02211673610867137540481053292751312255859375000000,  300.05218010260580285830656066536903381347656250000000,
        298.94072757692561026487965136766433715820312500000000,  297.69914483837470697835669852793216705322265625000000,  296.33687073832766145642381161451339721679687500000000,  294.86185329058480419917032122611999511718750000000000,  293.28087244477347894644481129944324493408203125000000,  291.59977512349070138952811248600482940673828125000000,  289.82365051450727833071141503751277923583984375000000,  287.95696350798829143968760035932064056396484375000000,  286.00365808654902366470196284353733062744140625000000,  283.96723868108716715141781605780124664306640625000000,
        281.85083506339833547826856374740600585937500000000000,  279.65725473226126496228971518576145172119140625000000,  277.38902565674555944497114978730678558349609375000000,  275.04843148470905589419999159872531890869140625000000,  272.63754079178113443049369379878044128417968750000000,  270.15823156423431328221340663731098175048828125000000,  267.61221183105328691453905776143074035644531250000000,  265.00103715512739199766656383872032165527343750000000,  262.32612553981221026333514600992202758789062500000000,  259.58877019073457859121845103800296783447265625000000,
        256.79015048360457740272977389395236968994140625000000,  253.93134141984629081889579538255929946899414062500000,  251.01332179798038168883067555725574493408203125000000,  248.03698128619481622081366367638111114501953125000000,  245.00312654771209963655564934015274047851562500000000,  241.91248654339642598642967641353607177734375000000000,  238.76571711403215658720000647008419036865234375000000,  235.56340492670381081552477553486824035644531250000000,  232.30607085484342633208143524825572967529296875000000,  228.99417284910853709334332961589097976684570312500000,
        225.62810834578272078942973166704177856445312500000000,  222.20821625043060976167907938361167907714843750000000,  218.73477852675867438847490120679140090942382812500000,  215.20802141374986149457981809973716735839843750000000,  211.62811628792798046561074443161487579345703125000000,  207.99518018186992662776901852339506149291992187500000,  204.30927596464022144573391415178775787353515625000000,  200.57041218451456643379060551524162292480468750000000,  196.77854256902958240971202030777931213378906250000000,  192.93356517188533416629070416092872619628906250000000,
        189.03532115036637151206377893686294555664062500000000,  185.08359315055437832597817759960889816284179687500000,  181.07810327046061615874350536614656448364257812500000,  177.01851056306048803890007548034191131591796875000000,  172.90440803175809492131520528346300125122070312500000,  168.73531905965978694439399987459182739257812500000000,  164.51069320071630386337346863001585006713867187500000,  160.22990124468677208824374247342348098754882812500000,  155.89222944818706650949025060981512069702148437500000,  151.49687279977453613355464767664670944213867187500000,
        147.04292715671013525025045964866876602172851562500000,  142.52938005289072975756425876170396804809570312500000,  137.95509992897541451384313404560089111328125000000000,  133.31882347354016360441164579242467880249023437500000,  128.61914068349767603649524971842765808105468750000000,  123.85447714643477468143828446045517921447753906250000,  119.02307290763157254787074634805321693420410156250000,  114.12295709690444311945611843839287757873535156250000,  109.15191723544037927240424323827028274536132812500000,  104.10746179126388710756145883351564407348632812500000,
         98.98677405966820685989659978076815605163574218750000,   93.78665474342469110524689313024282455444335937500000,   88.50344958859773214499000459909439086914062500000000,   83.13295691967536527045012917369604110717773437500000,   77.67030761929311211133608594536781311035156250000000,   72.10980650795748658765660366043448448181152343750000,   66.44471829557279818345705280080437660217285156250000,   60.66697161930957804543140809983015060424804687500000,   54.76673786955765166339915595017373561859130859375000,   48.73181075299549291912626358680427074432373046875000,
         42.54665284332629937580350087955594062805175781250000,   36.19085068160985230178994243033230304718017578125000,   29.63643430096999153988690522965043783187866210937500,   22.84277787424191785703442292287945747375488281250000,   15.74553447206599088303846656344830989837646484375000,    8.22710823434814564336647890741005539894104003906250,    0.00000000000000000000000000000000000000000000000000,  304.68685676566866504799691028892993927001953125000000,  306.30601922115448587646824307739734649658203125000000,  307.16580740832563378717168234288692474365234375000000,
        307.56194922671011227066628634929656982421875000000000,  307.61807929301443209624267183244228363037109375000000,  307.40316963714036546662100590765476226806640625000000,  306.96150745169524043376441113650798797607421875000000,  306.32403832557474743225611746311187744140625000000000,  305.51364632819513644790276885032653808593750000000000,  304.54795695546300748901558108627796173095703125000000,  303.44096792206289592286339029669761657714843750000000,  302.20406462012755355317494831979274749755859375000000,  300.84668644442217555479146540164947509765625000000000,
        299.37678265823706169612705707550048828125000000000000,  297.80113507675213213587994687259197235107421875000000,  296.12559303859779902268201112747192382812500000000000,  294.35524865031976560203474946320056915283203125000000,  292.49457018921350481832632794976234436035156250000000,  290.54750546924140053306473419070243835449218750000000,  288.51756318078884078204282559454441070556640625000000,  286.40787777371286892957868985831737518310546875000000,  284.22126183925456643919460475444793701171875000000000,  281.96024885377079272075206972658634185791015625000000,
        279.62712839168057143979240208864212036132812500000000,  277.22397538252550930337747558951377868652343750000000,  274.75267460528090168736525811254978179931640625000000,  272.21494133506269008648814633488655090332031250000000,  269.61233885207946059381356462836265563964843750000000,  266.94629336906729122347314842045307159423828125000000,  264.21810681713873236731160432100296020507812500000000,  261.42896784092067719029728323221206665039062500000000,  258.57996128496074561553541570901870727539062500000000,  255.67207639955532272324489895254373550415039062500000,
        252.70621395171164635939931031316518783569335937500000,  249.68319239318375934999494347721338272094726562500000,  246.60375321042346286048996262252330780029296875000000,  243.46856555934019183951022569090127944946289062500000,  240.27823026984160037500259932130575180053710937500000,  237.03328329034960120225150603801012039184570312500000,  233.73419863018213504801678936928510665893554687500000,  230.38139084733526829040783923119306564331054687500000,  226.97521712037470820177986752241849899291992187500000,  223.51597893551888773799873888492584228515625000000000,
        220.00392341329657597270852420479059219360351562500000,  216.43924429316714963533740956336259841918945312500000,  212.82208258900635655663791112601757049560546875000000,  209.15252692322428629267960786819458007812500000000000,  205.43061354233853421646927017718553543090820312500000,  201.65632601193223649715946521610021591186523437500000,  197.82959458394614671306044328957796096801757812500000,  193.95029522403436317290470469743013381958007812500000,  190.01824828110028420269372873008251190185546875000000,  186.03321677494315622425347100943326950073242187500000,
        181.99490427097947531365207396447658538818359375000000,  177.90295230300748130503052379935979843139648437500000,  173.75693729565719536367396358400583267211914062500000,  169.55636692713045476921251974999904632568359375000000,  165.30067585961131726435269229114055633544921875000000,  160.98922074870128540169389452785253524780273437500000,  156.62127442361540374804462771862745285034179687500000,  152.19601910562568036766606383025646209716796875000000,  147.71253850197962265156093053519725799560546875000000,  143.16980857442135288692952599376440048217773437500000,
        138.56668673301359717697778251022100448608398437500000,  133.90189914380465552312671206891536712646484375000000,  129.17402575831857802768354304134845733642578125000000,  124.38148256728963758632744429633021354675292968750000,  119.52250044119735150616179453209042549133300781250000,  114.59509973255900661115447292104363441467285156250000,  109.59705955997901583032216876745223999023437500000000,  104.52588034244629966451611835509538650512695312500000,   99.37873766007542997158452635630965232849121093750000,   94.15242481598284030042123049497604370117187500000000,
         88.84328045501453630095056723803281784057617187500000,   83.44709608293483427132741780951619148254394531250000,   77.95899603126284205245610792189836502075195312500000,   72.37327882308005655431770719587802886962890625000000,   66.68320311153763668698957189917564392089843750000000,   60.88069170553291087344405241310596466064453125000000,   54.95591038423597041173707111738622188568115234375000,   48.89664744944057872544362908229231834411621093750000,   42.68736026623795254408832988701760768890380859375000,   36.30763035389462345392530551180243492126464843750000,
         29.72948290167051865751091099809855222702026367187500,   22.91228740810151265350214089266955852508544921875000,   15.79169243121611643232427013572305440902709960937500,    8.25009775257284339033958531217649579048156738281250,    0.00000000000000000000000000000000000000000000000000,  309.16419358014690033087390474975109100341796875000000,  310.78560423968019676976837217807769775390625000000000,  311.64794237192364789734710939228534698486328125000000,  312.04690367206541168343392200767993927001953125000000,  312.10610217303963054291671141982078552246093750000000,
        311.89449620400904450434609316289424896240234375000000,  311.45636365441458792702178470790386199951171875000000,  310.82264379552498212433420121669769287109375000000000,  310.01621651073031671330682002007961273193359375000000,  309.05470469868771488108905032277107238769531250000000,  307.95210470643678490887396037578582763671875000000000,  306.71980154263428630656562745571136474609375000000000,  305.36723503068486706979456357657909393310546875000000,  303.90235555164031211461406201124191284179687500000000,  302.33194663842277805088087916374206542968750000000000,
        300.66185988227863390420679934322834014892578125000000,  298.89719012965696265382575802505016326904296875000000,  297.04240885055759235910954885184764862060546875000000,  295.10146747999641547721694223582744598388671875000000,  293.07787874118923809874104335904121398925781250000000,  290.97478151837123050427180714905261993408203125000000,  288.79499323371805985516402870416641235351562500000000,  286.54105259052965948285418562591075897216796875000000,  284.21525478950070464634336531162261962890625000000000,  281.81968079257620729549671523272991180419921875000000,
        279.35622182725251150259282439947128295898437500000000,  276.82660004629707373169367201626300811767578125000000,  274.23238605264282341522630304098129272460937500000000,  271.57501384567109425915987230837345123291015625000000,  268.85579362884874399242107756435871124267578125000000,  266.07592282968317931590718217194080352783203125000000,  263.23649561411241393216187134385108947753906250000000,  260.33851112367113955770037136971950531005859375000000,  257.38288062138070699802483431994915008544921875000000,  254.37043369859406993782613426446914672851562500000000,
        251.30192366797689373925095424056053161621093750000000,  248.17803224592583433150139171630144119262695312500000,  244.99937360986595535905507858842611312866210937500000,  241.76649790116891836078139021992683410644531250000000,  238.47989423221599736280040815472602844238281250000000,  235.13999324587180694834387395530939102172851562500000,  231.74716926692423157874145545065402984619140625000000,  228.30174207755163706679013557732105255126953125000000,  224.80397834233502862844034098088741302490234375000000,  221.25409270251961402209417428821325302124023437500000,
        217.65224855396152747744054067879915237426757812500000,  213.99855851831503628091013524681329727172851562500000,  210.29308461237471306048973929136991500854492187500000,  206.53583811595996166943223215639591217041015625000000,  202.72677913418579009885434061288833618164062500000000,  198.86581584527144173080159816890954971313476562500000,  194.95280342006844875868409872055053710937500000000000,  190.98754259408320876900688745081424713134765625000000,  186.96977786676018240541452541947364807128906250000000,  182.89919529597696623568481300026178359985351562500000,
        178.77541984783425732530304230749607086181640625000000,  174.59801225260903834168857429176568984985351562500000,  170.36646530679666966534568928182125091552734375000000,  166.08019954802318807196570560336112976074218750000000,  161.73855821365714291459880769252777099609375000000000,  157.34080137439045188330055680125951766967773437500000,  152.88609910986244244668341707438230514526367187500000,  148.37352356318984902827651239931583404541015625000000,  143.80203967320431956977699883282184600830078125000000,  139.17049433480596576373500283807516098022460937500000,
        134.47760367571856932045193389058113098144531250000000,  129.72193805739340177751728333532810211181640625000000,  124.90190430227519868822128046303987503051757812500000,  120.01572450980098949457897106185555458068847656250000,  115.06141063592102113943838048726320266723632812500000,  110.03673375599126416091166902333498001098632812500000,  104.93918657939234151399432448670268058776855468750000,   99.76593729194853210628934903070330619812011718750000,   94.51377210072692491849011275917291641235351562500000,   89.17902283683295650007494259625673294067382812500000,
         83.75747445972072569020383525639772415161132812500000,   78.24424500812206417776906164363026618957519531250000,   72.63362695284256176364579005166888236999511718750000,   66.91887312308433877205970929935574531555175781250000,   61.09190072070310861818143166601657867431640625000000,   55.14287012402302678992782603017985820770263671875000,   49.05956443013140955144990584813058376312255859375000,   42.82643598610447810415280400775372982025146484375000,   36.42306147100136826111338450573384761810302734375000,   29.82146140332267592043535842094570398330688476562500,
         22.98100077242089156470683519728481769561767578125000,   15.83732378129868401117619214346632361412048339843750,    8.27282600365040110546033247373998165130615234375000,    0.00000000000000000000000000000000000000000000000000,  313.65282994987904885419993661344051361083984375000000,  315.27645859424882246457855217158794403076171875000000,  316.14131105275822619660175405442714691162109375000000,  316.54305127486071569364867173135280609130859375000000,  316.60527296888324144674697890877723693847656250000000,  316.39692090660980738903163000941276550292968750000000,
        315.96226374399344649646081961691379547119140625000000,  315.33223445379394433984998613595962524414062500000000,  314.52970871547859132988378405570983886718750000000000,  313.57230678387577427201904356479644775390625000000000,  312.47402356842400195091613568365573883056640625000000,  311.24624360442322767994483001530170440673828125000000,  309.89840703637321439600782468914985656738281250000000,  308.43846523934143988299183547496795654296875000000000,  306.87320332494834929093485698103904724121093750000000,  305.20847498320989643616485409438610076904296875000000,
        303.44937763201789948652731254696846008300781250000000,  301.60038575094449697644449770450592041015625000000000,  299.66545419799746241551474668085575103759765625000000,  297.64809951580110691793379373848438262939453125000000,  295.55146479359746081172488629817962646484375000000000,  293.37837203844247824235935695469379425048828125000000,  291.13136491696593566302908584475517272949218750000000,  288.81274397394992092813481576740741729736328125000000,  286.42459590181738349201623350381851196289062500000000,  283.96881805361090300721116364002227783203125000000000,
        281.44713911425213836992043070495128631591796875000000,  278.86113663973253551375819370150566101074218750000000,  276.21225202041011925757629796862602233886718750000000,  273.50180330839560838285251520574092864990234375000000,  270.73099626005705431452952325344085693359375000000000,  267.90093387587421602802351117134094238281250000000000,  265.01262466614178947565960697829723358154296875000000,  262.06698982867328595602884888648986816406250000000000,  259.06486949098427885473938658833503723144531250000000,  256.00702814243800276017282158136367797851562500000000,
        252.89415936000449391940492205321788787841796875000000,  249.72688991348704234951583202928304672241210937500000,  246.50578332143487614303012378513813018798828125000000,  243.23134291681805052576237358152866363525390625000000,  239.90401447136699175644025672227144241333007812500000,  236.52418841886830591647594701498746871948242187500000,  233.09220171032620783080346882343292236328125000000000,  229.60833932748860775063803885132074356079101562500000,  226.07283547557938163663493469357490539550781250000000,  222.48587447099083647117367945611476898193359375000000,
        218.84759133502527106429624836891889572143554687500000,  215.15807210039056940331647638231515884399414062500000,  211.41735383293021754980145487934350967407226562500000,  207.62542436688917746323568280786275863647460937500000,  203.78222174776371389270934741944074630737304687500000,  199.88763337233913830459641758352518081665039062500000,  195.94149481075604057878081221133470535278320312500000,  191.94358829021669521353032905608415603637695312500000,  187.89364081408504603132314514368772506713867187500000,  183.79132188344632936605194117873907089233398437500000,
        179.63624078043679332949977833777666091918945312500000,  175.42794336353148310081451199948787689208984375000000,  171.16590831411917861260008066892623901367187500000000,  166.84954276061921518703456968069076538085937500000000,  162.47817719050379992040689103305339813232421875000000,  158.05105954108373111921537201851606369018554687500000,  153.56734833576572896163270343095064163208007812500000,  149.02610470231945782870752736926078796386718750000000,  144.42628307166540935213561169803142547607421875000000,  139.76672030733450924344651866704225540161132812500000,
        135.04612295365311069872404914349317550659179687500000,  130.26305221019379132485482841730117797851562500000000,  125.41590613451904800967895425856113433837890625000000,  120.50289843542202561366138979792594909667968750000000,  115.52203303130272615817375481128692626953125000000000,  110.47107329339371517562540248036384582519531250000000,  105.34750454206762526609963970258831977844238281250000,  100.14848787218919312635989626869559288024902343750000,   94.87080268199179045041091740131378173828125000000000,   89.51077426100330569624929921701550483703613281250000,
         84.06418128045744708742859074845910072326660156250000,   78.52613573121900003570772241801023483276367187500000,   72.89092426440275573895632987841963768005371093750000,   67.15179410628373091185494558885693550109863281250000,   61.30065706184898743913436192087829113006591796875000,   55.32766830856248674308517365716397762298583984375000,   49.22060592921747712580327061004936695098876953125000,   42.96391743428392828718642704188823699951171875000000,   36.53717483554812872625916497781872749328613281250000,   29.91239414570399901549535570666193962097167968750000,
         23.04893600234970918450017052236944437026977539062500,   15.88244040379450439104402903467416763305664062500000,    8.29529885950245926551360753364861011505126953125000,    0.00000000000000000000000000000000000000000000000000,  318.15263962020935650798492133617401123046875000000000,  319.77845677799541590502485632896423339843750000000000,  320.64578885129054697245010174810886383056640625000000,  321.05026849205347616589278914034366607666015625000000,  321.11546933846864249062491580843925476074218750000000,  320.91032274689274572665453888475894927978515625000000,
        320.47908821019211700331652536988258361816406250000000,  319.85269242397447442272095941007137298583984375000000,  319.05400684929367116637877188622951507568359375000000,  318.10064905493885589748970232903957366943359375000000,  317.00661244800227223095134831964969635009765625000000,  315.78328100653072851855540648102760314941406250000000,  314.44009509529655588266905397176742553710937500000000,  312.98500696708492796460632234811782836914062500000000,  311.42480318134823846776271238923072814941406250000000,  309.76533938195285600158968009054660797119140625000000,
        308.01171539947785049662343226373195648193359375000000,  306.16840855045921898636152036488056182861328125000000,  304.23937692922669384643086232244968414306640625000000,  302.22814069670619119278853759169578552246093750000000,  300.13784693108232204394880682229995727539062500000000,  297.97132199261181995098013430833816528320312500000000,  295.73111426317046834810753352940082550048828125000000,  293.41952936621925118743092752993106842041015625000000,  291.03865944087408479390433058142662048339843750000000,  288.59040766237171737884636968374252319335937500000000,
        286.07650892353228755382588133215904235839843750000000,  283.49854738675333010178292170166969299316406250000000,  280.85797146265889523419900797307491302490234375000000,  278.15610665538594048484810627996921539306640625000000,  275.39416662558750203970703296363353729248046875000000,  272.57326275347423916173283942043781280517578125000000,  269.69441243052733625518158078193664550781250000000000,  266.75854626620730414288118481636047363281250000000000,  263.76651436235545133968116715550422668457031250000000,  260.71909178102896476048044860363006591796875000000000,
        257.61698330972905068847467191517353057861328125000000,  254.46082761023495777408243156969547271728515625000000,  251.25120082268051646678941324353218078613281250000000,  247.98861968442915326704678591340780258178710937500000,  244.67354421320780488713353406637907028198242187500000,  241.30637999542975080657924991101026535034179687500000,  237.88748011335547971611958928406238555908203125000000,  234.41714673844120397916412912309169769287109375000000,  230.89563241269931381793867330998182296752929687500000,  227.32314103496406687554554082453250885009765625000000,
        223.69982856447117569587135221809148788452148437500000,  220.02580344999068984179757535457611083984375000000000,  216.30112678878558085671102162450551986694335937500000,  212.52581221579060866133659146726131439208984375000000,  208.69982551951903815279365517199039459228515625000000,  204.82308397719722847796219866722822189331054687500000,  200.89545539739168589221662841737270355224609375000000,  196.91675685380528193491045385599136352539062500000000,  192.88675308884273817966459318995475769042968750000000,  188.80515455981438321941823232918977737426757812500000,
        184.67161509407051767084340099245309829711914062500000,  180.48572911169873123071738518774509429931640625000000,  176.24702836537454686549608595669269561767578125000000,  171.95497813616830740102159325033426284790039062500000,  167.60897281109822642974904738366603851318359375000000,  163.20833075238081733004946727305650711059570312500000,  158.75228834887198559044918511062860488891601562500000,  154.23999311608127982253790833055973052978515625000000,  149.67049568100816259175189770758152008056640625000000,  145.04274045005317361756169702857732772827148437500000,
        140.35555470992352411485626362264156341552734375000000,  135.60763584938007397795445285737514495849609375000000,  130.79753630917966233937477227300405502319335937500000,  125.92364576207401682950148824602365493774414062500000,  120.98416988491752022127911914139986038208007812500000,  115.97710489738837225104362005367875099182128906250000,  110.90020678691325883846729993820190429687500000000000,  105.75095378791790778905124170705676078796386718750000,  100.52650019126583913475769804790616035461425781250000,   95.22361885827386629443935817107558250427246093750000,
         89.83862879474730789297609589993953704833984375000000,   84.36730262839505201100109843537211418151855468750000,   78.80474653458504974423703970387578010559082031250000,   73.14524156555745548757840879261493682861328125000000,   67.38202955463604837405000580474734306335449218750000,   61.50701710973608982158111757598817348480224609375000,   55.51035439769251667030403041280806064605712890625000,   49.37981466823948295541413244791328907012939453125000,   43.09984076800836305665143299847841262817382812500000,   36.65000020630499477647390449419617652893066406250000,
         30.00230464729168744497656007297337055206298828125000,   23.11611052699668533705334994010627269744873046875000,   15.92705378252730774590872897533699870109558105468750,    8.31752199628716937240824336186051368713378906250000,    0.00000000000000000000000000000000000000000000000000,  322.66349912672620803277823142707347869873046875000000,  324.29147604761010370566509664058685302734375000000000,  325.16125389835281112027587369084358215332031250000000,  325.56843447159400284363073296844959259033203125000000,  325.63657158521681367346900515258312225341796875000000,
        325.43458332063636362363467924296855926513671875000000,  325.00672007861226120439823716878890991210937500000000,  324.38390230098423216986702755093574523925781250000000,  323.58899721904214175083325244486331939697265625000000,  322.63961967738032399211078882217407226562500000000000,  321.54976152085140483904979191720485687255859375000000,  320.33080609193314103322336450219154357910156250000000,  318.99219388112231854393030516803264617919921875000000,  317.54187790957644210720900446176528930664062500000000,  315.98664606134929044856107793748378753662109375000000,
        314.33235579757075583984260447323322296142578125000000,  312.58410921180819741493905894458293914794921875000000,  310.74638629476794449146836996078491210937500000000000,  308.82314820104795671795727685093879699707031250000000,  306.81791852023980027297511696815490722656250000000000,  304.73384811579154529681545682251453399658203125000000,  302.57376748284900713770184665918350219726562500000000,  300.34022948487790927174501121044158935546875000000000,  298.03554457418607626095763407647609710693359375000000,  295.66181006961926414078334346413612365722656250000000,
        293.22093468343399536024662666022777557373046875000000,  290.71465921174933555448660627007484436035156250000000,  288.14457409798893650076934136450290679931640625000000,  285.51213442537277842347975820302963256835937500000000,  282.81867277842883368066395632922649383544921875000000,  280.06541032463519513839855790138244628906250000000000,  277.25346639858770458886283449828624725341796875000000,  274.38386681743691042356658726930618286132812500000000,  271.45755111406759851888637058436870574951171875000000,  268.47537884090354509680764749646186828613281250000000,
        265.43813507030569098787964321672916412353515625000000,  262.34653519579035219067009165883064270019531250000000,  259.20122912059571262943791225552558898925781250000000,  256.00280490559561030750046484172344207763671875000000,  252.75179193654199139018601272255182266235351562500000,  249.44866366057823370283585973083972930908203125000000,  246.09383993351525532489176839590072631835937500000000,  242.68768901216108702101337257772684097290039062500000,  239.23052921979453344647481571882963180541992187500000,  235.72263030746071876819769386202096939086914062500000,
        232.16421452896636878904246259480714797973632812500000,  228.55545744312311740031873341649770736694335937500000,  224.89648845280012778857781086117029190063476562500000,  221.18739108659403314049995969980955123901367187500000,  217.42820302530284948261396493762731552124023437500000,  213.61891587180656415512203238904476165771484375000000,  209.75947465931352553525357507169246673583984375000000,  205.84977708913424976344686001539230346679687500000000,  201.88967248508450325061858166009187698364257812500000,  197.87896044718198140799358952790498733520507812500000,
        193.81738918235399182776745874434709548950195312500000,  189.70465348425088336625776719301939010620117187500000,  185.54039232778018231329042464494705200195312500000000,  181.32418603639624166135035920888185501098632812500000,  177.05555297121020430495264008641242980957031250000000,  172.73394568025801731891988310962915420532226562500000,  168.35874643330507183236477430909872055053710937500000,  163.92926205177323595307825598865747451782226562500000,  159.44471792395873421810392756015062332153320312500000,  154.90425107163412121735746040940284729003906250000000,
        150.30690210402482875906571280211210250854492187500000,  145.65160585718240326968953013420104980468750000000000,  140.93718046846646529957070015370845794677734375000000,  136.16231457379751645930809900164604187011718750000000,  131.32555223486664885967911686748266220092773437500000,  126.42527509801416840673482511192560195922851562500000,  121.45968114669298643093497958034276962280273437500000,  116.42675922189940251882944721728563308715820312500000,  111.32425823005061715775809716433286666870117187500000,  106.14964960633108148613246157765388488769531250000000,
        100.90008110927897178044077008962631225585937500000000,   95.57231932091896453584922710433602333068847656250000,   90.16267720780886918419128051027655601501464843750000,   84.66692158629201969688438111916184425354003906250000,   79.08015303685374419728759676218032836914062500000000,   73.39664722264194551826221868395805358886718750000000,   67.60964078364054330450017005205154418945312500000000,   61.71103532074636177640059031546115875244140625000000,   55.69097617122967847080872161313891410827636718750000,   49.53723192436941502592162578366696834564208984375000,
         43.23424092759342585168269579298794269561767578125000,   36.76156634485101903919712640345096588134765625000000,   30.09121564181195296328041877131909132003784179687500,   23.18254119628629439375799847766757011413574218750000,   15.97117502121477983223485352937132120132446289062500,    8.33950090300594482073393010068684816360473632812500,    0.00000000000000000000000000000000000000000000000000,  327.18528770377520231704693287611007690429687500000000,  328.81539633308801739985938183963298797607421875000000,  329.68758696646989392320392653346061706542968750000000,
        330.09743096564437792039825581014156341552734375000000,  330.16846257348839799306006170809268951416015625000000,  329.96958673544924067755346186459064483642578125000000,  329.54504483160667405172716826200485229492187500000000,  328.92575107525067323877010494470596313476562500000000,  328.13456845945171380662941373884677886962890625000000,  327.18910907021773937231046147644519805908203125000000,  326.10336313478882175331818871200084686279296875000000,  324.88871328696569662497495301067829132080078125000000,  323.55460005428051317721838131546974182128906250000000,
        322.10897712341488841047976166009902954101562500000000,  320.55863358692693054763367399573326110839843750000000,  318.90942859440798429204733110964298248291015625000000,  317.16646636117172874946845695376396179199218750000000,  315.33422939846258259422029368579387664794921875000000,  313.41668175499728477007010951638221740722656250000000,  311.41735027009116265617194585502147674560546875000000,  309.33938940031424635890289209783077239990234375000000,  307.18563356983287349066813476383686065673828125000000,  304.95863990346589389446307905018329620361328125000000,
        302.66072344700904750425252132117748260498046875000000,  300.29398644769162274315021932125091552734375000000000,  297.86034288647329049126710742712020874023437500000000,  295.36153917638296206860104575753211975097656250000000,  292.79917173617542403007973916828632354736328125000000,  290.17470199529526553305913694202899932861328125000000,  287.48946927009114915563259273767471313476562500000000,  284.74470186241018154760240577161312103271484375000000,  281.94152666302335319414851255714893341064453125000000,  279.08097748871716703433776274323463439941406250000000,
        276.16400233965106281175394542515277862548828125000000,  273.19146973002557388099376112222671508789062500000000,  270.16417421822347932902630418539047241210937500000000,  267.08284124088731914525851607322692871093750000000000,  263.94813133773482149990741163492202758789062500000000,  260.76064383943219127104384824633598327636718750000000,  257.52092007887432600909960456192493438720703125000000,  254.22944617624119700849405489861965179443359375000000,  250.88665543981122141303785610944032669067382812500000,  247.49293041738249598893162328749895095825195312500000,
        244.04860462703985035659570712596178054809570312500000,  240.55396399068627033557277172803878784179687500000000,  237.00924798907288959526340477168560028076171875000000,  233.41465055286124652411672286689281463623046875000000,  229.77032070042017153355118352919816970825195312500000,  226.07636292948819800585624761879444122314453125000000,  222.33283736642474082145781721919775009155273437500000,  218.53975967344521791346778627485036849975585937500000,  214.69710071089551206569012720137834548950195312500000,  210.80478594818723081516509409993886947631835937500000,
        206.86269461339207964556408114731311798095703125000000,  202.87065856758687232286320067942142486572265625000000,  198.82846088572981102515768725425004959106445312500000,  194.73583412101257295034884009510278701782226562500000,  190.59245822410483128805935848504304885864257812500000,  186.39795808231025375789613462984561920166015625000000,  182.15190063614190307816897984594106674194335937500000,  177.85379152191833895813033450394868850708007812500000,  173.50307117830593028884322848170995712280273437500000,  169.09911034182127309577481355518102645874023437500000,
        164.64120484055442261706048157066106796264648437500000,  160.12856957599288421079108957201242446899414062500000,  155.56033155878051843501452822238206863403320312500000,  150.93552183417176593138719908893108367919921875000000,  146.25306609499600085655401926487684249877929687500000,  141.51177373165853623504517599940299987792968750000000,  136.71032500667391218485136050730943679809570312500000,  131.84725596076668807654641568660736083984375000000000,  126.92094055211693159890273818746209144592285156250000,  121.92956939054344900341675383970141410827636718750000,
        116.87112424089708895280637079849839210510253906250000,  111.74334721504310152795369504019618034362792968750000,  106.54370322036373863738845102488994598388671875000000,  101.26933374046976155113952700048685073852539062500000,   95.91699932235235337429912760853767395019531250000000,   90.48300712527391453932068543508648872375488281250000,   84.96311837462816640709206694737076759338378906250000,   79.35242826561018603115371661260724067687988281250000,   73.64520727170499014846427598968148231506347656250000,   67.83468702943545736161468084901571273803710937500000,
         61.91276431357678688982559833675622940063476562500000,   55.86957980427764880460017593577504158020019531250000,   49.69289759484234991759876720607280731201171875000000,   43.36715169047077722552785417065024375915527343750000,   36.87190105965209596661225077696144580841064453125000,   30.17914911277804534961433091666549444198608398437500,   23.24824430634381755567119398619979619979858398437500,   16.01481486006183629910992749501019716262817382812500,    8.36124088964234957188637054059654474258422851562500,    0.00000000000000000000000000000000000000000000000000,
        331.71788719692847280384739860892295837402343750000000,  333.35010015136504080146551132202148437500000000000000,  334.22467138497614769221399910748004913330078125000000,  334.63714224747980097163235768675804138183593750000000,  334.71102764757648628801689483225345611572265625000000,  334.51521953216752081061713397502899169921875000000000,  334.09395033240832617593696340918540954589843750000000,  333.47812805991321738474653102457523345947265625000000,  332.69061146374690451921196654438972473144531250000000,  331.74900984043688367819413542747497558593750000000000,
        330.66731174844966290038428269326686859130859375000000,  329.45689904466360076185083016753196716308593750000000,  328.12721221043403829753515310585498809814453125000000,  326.68620550119186418669414706528186798095703125000000,  325.14066910857400216627866029739379882812500000000000,  323.49646374910855683992849662899971008300781250000000,  321.75869562651803335029399022459983825683593750000000,  319.93184962751098510125302709639072418212890625000000,  318.01989253802116763836238533258438110351562500000000,  316.02635427812447233009152114391326904296875000000000,
        313.95439271612940501654520630836486816406250000000000,  311.80684601094844765611924231052398681640625000000000,  309.58627534100924094673246145248413085937500000000000,  307.29500012369976502668578177690505981445312500000000,  304.93512729779814662833814509212970733642578125000000,  302.50857586030264201326644979417324066162109375000000,  300.01709757165139080825611017644405364990234375000000,  297.46229453847399781807325780391693115234375000000000,  294.84563422977896607335424050688743591308593750000000,  292.16846236648672174851526506245136260986328125000000,
        289.43201403544384220367646776139736175537109375000000,  286.63742331040953104093205183744430541992187500000000,  283.78573160892568694180226884782314300537109375000000,  280.87789497177709563402459025382995605468750000000000,  277.91479041822327644695178605616092681884765625000000,  274.89722150333818717626854777336120605468750000000000,  271.82592318212277859856840223073959350585937500000000,  268.70156606743415750315762124955654144287109375000000,  265.52476015433222755746101029217243194580078125000000,  262.29605807151540375343756750226020812988281250000000,
        259.01595791059031625991337932646274566650390625000000,  255.68490567558140469373029191046953201293945312500000,  252.30329738802703332112287171185016632080078125000000,  248.87148087696044740368961356580257415771484375000000,  245.38975727784406899445457383990287780761718750000000,  241.85838225993336436658864840865135192871093750000000,  238.27756699746169033460319042205810546875000000000000,  234.64747889633574118306569289416074752807617187500000,  230.96824208461387684110377449542284011840820312500000,  227.23993767181434577651089057326316833496093750000000,
        223.46260377898667570661928039044141769409179687500000,  219.63623533839404444734100252389907836914062500000000,  215.76078365852512774836213793605566024780273437500000,  211.83615574689426352961163502186536788940429687500000,  207.86221337961850963438337203115224838256835937500000,  203.83877190297889114845020230859518051147460937500000,  199.76559874797547422531351912766695022583007812500000,  195.64241163414138213738624472171068191528320312500000,  191.46887643343822560382250230759382247924804687500000,  187.24460465872394365760555956512689590454101562500000,
        182.96915053383901295092073269188404083251953125000000,  178.64200759349913028017908800393342971801757812500000,  174.26260475055607912509003654122352600097656250000000,  169.83030175531590089121891651302576065063476562500000,  165.34438395588577463968249503523111343383789062500000,  160.80405624917173668109171558171510696411132812500000,  156.20843608813061109685804694890975952148437500000000,  151.55654538082956150901736691594123840332031250000000,  146.84730107894318962280522100627422332763671875000000,  142.07950420504701583013229537755250930786132812500000,
        137.25182700605373042890278156846761703491210937500000,  132.36279783969177969993324950337409973144531250000000,  127.41078329548059855369501747190952301025390625000000,  122.39396691188613885969971306622028350830078125000000,  117.31032366382773091117996955290436744689941406250000,  112.15758913982506328466115519404411315917968750000000,  106.93322197663373174236767226830124855041503906250000,  101.63435762698320274921570671722292900085449218750000,   96.25775083457982361778704216703772544860839843750000,   90.79970317161181014853355009108781814575195312500000,
         85.25597048192911131536675384268164634704589843750000,   79.62164277468718864838592708110809326171875000000000,   73.89098552341550885103060863912105560302734375000000,   68.05722554190819550967717077583074569702148437500000,   62.11225495110399918985422118566930294036865234375000,   56.04620993835897024837322533130645751953125000000000,   49.84685025783261380638577975332736968994140625000000,   43.49860572225311017291460302658379077911376953125000,   36.98103124772926975083464640192687511444091796875000,   30.26612632614848052980960346758365631103515625000000,
         23.31323562350451794600303401239216327667236328125000,   16.05798369145809800784263643436133861541748046875000,    8.38274709486331381924628658453002572059631347656250,    0.00000000000000000000000000000000000000000000000000,  336.26118197919845442811492830514907836914062500000000,  337.89547252363365714700194075703620910644531250000000,  338.77239295872652746766107156872749328613281250000000,  339.18745503188193879395839758217334747314453125000000,  339.26415455406771570778801105916500091552734375000000,  339.07137060947746931560686789453029632568359375000000,
        338.65332675231883285960066132247447967529296875000000,  338.04092482089725990590522997081279754638671875000000,  337.25701931694618451729184016585350036621093750000000,  336.31921671986731325887376442551612854003906250000000,  335.24150387212102941703051328659057617187500000000000,  334.03526178996139606169890612363815307617187500000000,  332.70993083047562777210259810090065002441406250000000,  331.27346572675298830290557816624641418457031250000000,  329.73265766632482609566068276762962341308593750000000,  328.09336881794126838940428569912910461425781250000000,
        326.36070724778898011209093965590000152587890625000000,  324.53916008096899759038933552801609039306640625000000,  322.63269669237223524760338477790355682373046875000000,  320.64484992319046341435750946402549743652343750000000,  318.57878088212845568705233745276927947998046875000000,  316.43733127938247662314097397029399871826171875000000,  314.22306615088677972380537539720535278320312500000000,  311.93830907626005455313134007155895233154296875000000,  309.58517146249573670502286404371261596679687500000000,  307.16557708451091457391157746315002441406250000000000,
        304.68128279633430111061898060142993927001953125000000,  302.13389612193014954755199141800403594970703125000000,  299.52489028147266481028054840862751007080078125000000,  296.85561709293864396386197768151760101318359375000000,  294.12731810014599886926589533686637878417968750000000,  291.34113420975990038641612045466899871826171875000000,  288.49811406623774701074580661952495574951171875000000,  285.59922135150844724194030277431011199951171875000000,  282.64534116268748675793176516890525817871093750000000,  279.63728559431115172628778964281082153320312500000000,
        276.57579862993384267610963433980941772460937500000000,  273.46156043034477534092729911208152770996093750000000,  270.29519109124817077827174216508865356445312500000000,  267.07725393136649927328107878565788269042968750000000,  263.80825836203774770183372311294078826904296875000000,  260.48866238108951165486359968781471252441406250000000,  257.11887472676738752852543257176876068115234375000000,  253.69925672151089202088769525289535522460937500000000,  250.23012383020952142942405771464109420776367187500000,  246.71174695306592639099108055233955383300781250000000,
        243.14435346920177494212111923843622207641601562500000,  239.52812804355372122699918691068887710571289062500000,  235.86321320632083597956807352602481842041015625000000,  232.14970971115295128583966288715600967407226562500000,  228.38767667533667804491415154188871383666992187500000,  224.57713150236688193217560183256864547729492187500000,  220.71804958441421717907360289245843887329101562500000,  216.81036377924556290963664650917053222656250000000000,  212.85396365304686128183675464242696762084960937500000,  208.84869447725287727735121734440326690673828125000000,
        204.79435596382072048982081469148397445678710937500000,  200.69070071927782805687456857413053512573242187500000,  196.53743239321462965563114266842603683471679687500000,  192.33420349151839445767109282314777374267578125000000,  188.08061281837558453844394534826278686523437500000000,  183.77620250367687049219966866075992584228515625000000,  179.42045456364900246626348234713077545166015625000000,  175.01278693195030200513429008424282073974609375000000,  170.55254888563001713919220492243766784667968750000000,  166.03901577466544381422863807529211044311523437500000,
        161.47138294446773443269194103777408599853515625000000,  156.84875871675330927246250212192535400390625000000000,  152.17015626414928419762873090803623199462890625000000,  147.43448417599540789524326100945472717285156250000000,  142.64053546455187415631371550261974334716796875000000,  137.78697469882402515395369846373796463012695312500000,  132.87232287278084186255000531673431396484375000000000,  127.89493950931168342322052922099828720092773437500000,  122.85300136150414118674234487116336822509765625000000,  117.74447688532390543514338787645101547241210937500000,
        112.56709540290235338488855632022023200988769531250000,  107.31830952420564528893009992316365242004394531250000,  101.99524890263087684161291690543293952941894531250000,   96.59466269862947740421077469363808631896972656250000,   91.11284710653818308401241665706038475036621093750000,   85.54555278773817406090529402717947959899902343750000,   79.88786475488416272128233686089515686035156250000000,   74.13404366212087381882156478241086006164550781250000,   68.27731167264379052994627272710204124450683593750000,   62.30955641773519460002717096358537673950195312500000,
         56.22090974864519097309312201105058193206787109375000,   49.99912723000716852084224228747189044952392578125000,   43.62863462502506450846340158022940158843994140625000,   37.08898293407364832319217384792864322662353515625000,   30.35216786122682464110766886733472347259521484375000,   23.37753040703549345380451995879411697387695312500000,   16.10069157483706092648390040267258882522583007812500,    8.40402449331059919757080933777615427970886230468750,    0.00000000000000000000000000000000000000000000000000,  340.81505887079902095138095319271087646484375000000000,
        342.45140089614352518765372224152088165283203125000000,  343.33063989021451334338053129613399505615234375000000,  343.74825839884101696952711790800094604492187500000000,  343.82773336739802516603958792984485626220703125000000,  343.63793115159955959825310856103897094726562500000000,  343.22306650080565759708406403660774230957031250000000,  342.61403510972263575240503996610641479492187500000000,  341.83368723169593295097001828253269195556640625000000,  340.89962650437530555791454389691352844238281250000000,  339.82583801064782846879097633063793182373046875000000,
        338.62370186668795213336125016212463378906250000000000,  337.30265823201631292249658145010471343994140625000000,  335.87066223160996969454572536051273345947265625000000,  334.33450595155619566867244429886341094970703125000000,  332.70005290448716550599783658981323242187500000000000,  330.97241290003040603551198728382587432861328125000000,  329.15607517209491561516188085079193115234375000000000,  327.25501154435067974191042594611644744873046875000000,  325.27275762817220083888969384133815765380859375000000,  323.21247761069673742895247414708137512207031250000000,
        321.07701658003549027853296138346195220947265625000000,  318.86894324438492276385659351944923400878906250000000,  316.59058514791576044444809667766094207763671875000000,  314.24405795507374250519205816090106964111328125000000,  311.83128999410797632663161493837833404541015625000000,  309.35404297339380264020292088389396667480468750000000,  306.81392957939681309653678908944129943847656250000000,  304.21242851199946244378224946558475494384765625000000,  301.55089739700326845195377245545387268066406250000000,  298.83058392692191773676313459873199462890625000000000,
        296.05263551260424037536722607910633087158203125000000,  293.21810767470407199652981944382190704345703125000000,  290.32797136186889019882073625922203063964843750000000,  287.38311934904953659497550688683986663818359375000000,  284.38437184254229350699461065232753753662109375000000,  281.33248139676487653559888713061809539794921875000000,  278.22813723020470888513955287635326385498046875000000,  275.07196901360418905824190005660057067871093750000000,  271.86455019158927370881428942084312438964843750000000,  268.60640088910253098219982348382472991943359375000000,
        265.29799044575537436685408465564250946044921875000000,  261.93973961425382412926410324871540069580078125000000,  258.53202245312559170997701585292816162109375000000000,  255.07516793887705830456980038434267044067382812500000,  251.56946131827376689216180238872766494750976562500000,  248.01514521753097142209298908710479736328125000000000,  244.41242052170767351526592392474412918090820312500000,  240.76144703442423633532598614692687988281250000000000,  237.06234392508267205812444444745779037475585937500000,  233.31518996798985199347953312098979949951171875000000,
        229.52002357509559260506648570299148559570312500000000,  225.67684262139636075517046265304088592529296875000000,  221.78560405935462540583102963864803314208984375000000,  217.84622331588113297584641259163618087768554687500000,  213.85857346244651466804498340934514999389648437500000,  209.82248414565640359796816483139991760253906250000000,  205.73774026204782217064348515123128890991210937500000,  201.60408035684412197952042333781719207763671875000000,  197.42119472181155970247345976531505584716796875000000,  193.18872316205110450937354471534490585327148437500000,
        188.90625239534190882295661140233278274536132812500000,  184.57331304030512342251313384622335433959960937500000,  180.18937614088963528047315776348114013671875000000000,  175.75384916412579627831291873008012771606445312500000,  171.26607139529130563460057601332664489746093750000000,  166.72530863897114272731414530426263809204101562500000,  162.13074711519701054385222960263490676879882812500000,  157.48148641587761176197091117501258850097656250000000,  152.77653135672065332073543686419725418090820312500000,  148.01478252196082507907703984528779983520507812500000,
        143.19502525096638123613956850022077560424804687500000,  138.31591675381369554997945670038461685180664062500000,  133.37597096249632500075676944106817245483398437500000,  128.37354061901297086478734854608774185180664062500000,  123.30679596182632451473182300105690956115722656250000,  118.17369918469107403780071763321757316589355468750000,  112.97197358698225855277996743097901344299316406250000,  107.69906598322940283196658128872513771057128906250000,  102.35210044733918266501859761774539947509765625000000,   96.92782076555083392577216727659106254577636718750000,
         91.42251795325204000164376338943839073181152343750000,   85.83193767872874957447493216022849082946777343750000,   80.15116013854706977781461318954825401306152343750000,   74.37444133944464397245610598474740982055664062500000,   68.49499895805196558740135515108704566955566406250000,   62.50471629254098360206626239232718944549560546875000,   56.39372100753936933870136272162199020385742187500000,   50.14976462097180842647503595799207687377929687500000,   43.75726898303924627953165327198803424835205078125000,   37.19578130895282441770177683793008327484130859375000,
         30.43729363991555914026321261189877986907958984375000,   23.44114343065212224814786168280988931655883789062500,   16.14294825075005590520049736369401216506958007812500,    8.42507790250843058288410247769206762313842773437500,    0.00000000000000000000000000000000000000000000000000,  345.37940706226680731560918502509593963623046875000000,  347.01777506430630637623835355043411254882812500000000,  347.89930270491993269388331100344657897949218750000000,  348.31944372039862400924903340637683868408203125000000,  348.40165641844180299813160672783851623535156250000000,
        348.21479455888106713246088474988937377929687500000000,  347.80306415836543010300374589860439300537109375000000,  347.19735479891704699184629134833812713623046875000000,  346.42051248652688855145242996513843536376953125000000,  345.49013799527449464221717789769172668457031250000000,  344.42021460832688717346172779798507690429687500000000,  343.22212148629591865756083279848098754882812500000000,  341.90529852232464236294617876410484313964843750000000,  340.47770115249238642718410119414329528808593750000000,  338.94612226958059864045935682952404022216796875000000,
        337.31642662773771235151798464357852935791015625000000,  335.59372566749181032719206996262073516845703125000000,  333.78251060898662672116188332438468933105468750000000,  331.88675559205847775956499390304088592529296875000000,  329.90999885547967096499633044004440307617187500000000,  327.85540751162488959380425512790679931640625000000000,  325.72582986258049686512094922363758087158203125000000,  323.52383811369873001240193843841552734375000000000000,  321.25176358692112899007042869925498962402343750000000,  318.91172600509958101611118763685226440429687500000000,
        316.50565803783086948897107504308223724365234375000000,  314.03532602215307178994407877326011657714843750000000,  311.50234756680066539047402329742908477783203125000000,  308.90820659563615890874643810093402862548828125000000,  306.25426627001502311031799763441085815429687500000000,  303.54178014117462680587777867913246154785156250000000,  300.77190181519472389481961727142333984375000000000000,  297.94569335958198053049272857606410980224609375000000,  295.06413263841267280440661124885082244873046875000000,  292.12811972952096084554796107113361358642578125000000,
        289.13848255045638779847649857401847839355468750000000,  286.09598179834705433677299879491329193115234375000000,  283.00131529127025942216278053820133209228515625000000,  279.85512178438438013472477905452251434326171875000000,  276.65798432224988800953724421560764312744140625000000,  273.41043317895531572503386996686458587646484375000000,  270.11294842945278560364386066794395446777343750000000,  266.76596218859305054138530977070331573486328125000000,  263.36986054846698834808194078505039215087890625000000,  259.92498523961882028743275441229343414306640625000000,
        256.43163503732091612619115039706230163574218750000000,  252.89006693026621519493346568197011947631835937500000,  249.30049706562286360167490784078836441040039062500000,  245.66310148131833557272329926490783691406250000000000,  241.97801663359223311999812722206115722656250000000000,  238.24533972520742963752127252519130706787109375000000,  234.46512883717662134586134925484657287597656250000000,  230.63740286437914051020925398916006088256835937500000,  226.76214125295888379696407355368137359619140625000000,  222.83928353484424178532208316028118133544921875000000,
        218.86872865205478433381358627229928970336914062500000,  214.85033406059045546498964540660381317138671875000000,  210.78391460055985362487263046205043792724609375000000,  206.66924111571063349401811137795448303222656250000000,  202.50603880157385106031142640858888626098632812500000,  198.29398525690180576930288225412368774414062500000000,  194.03270820782228156531346030533313751220703125000000,  189.72178286796122392843244597315788269042968750000000,  185.36072889047835587916779331862926483154296875000000,  180.94900685922863203813903965055942535400390625000000,
        176.48601425573806977808999363332986831665039062500000,  171.97108082590736444217327516525983810424804687500000,  167.40346325471932686923537403345108032226562500000000,  162.78233903794935599762538913637399673461914062500000,  158.10679941592513841897016391158103942871093750000000,  153.37584120438509671657811850309371948242187500000000,  148.58835731961573856096947565674781799316406250000000,  143.74312574681783871710649691522121429443359375000000,  138.83879663868071929755387827754020690917968750000000,  133.87387715073103322538372594863176345825195312500000,
        128.84671351460949040301784407347440719604492187500000,  123.75546971068567358997825067490339279174804687500000,  118.59810191393374623203271767124533653259277343750000,  113.37232763213236808041983749717473983764648437500000,  108.07558810403165239222289528697729110717773437500000,  102.70500203291376806191692594438791275024414062500000,   97.25730802953870579585782252252101898193359375000000,   91.72879211955613243389962008222937583923339843750000,   86.11519515841287386592739494517445564270019531250000,   80.41159269841443801851710304617881774902343750000000,
         74.61223626278130893751949770376086235046386718750000,   68.71033919798696842917706817388534545898437500000000,   62.69778061844367300636804429814219474792480468750000,   56.56468414484636753059021430090069770812988281250000,   50.29879738480939010969450464472174644470214843750000,   43.88453840598276656237430870532989501953125000000000,   37.30145076324267705558668239973485469818115234375000,   30.52152295442808593861627741716802120208740234375000,   23.50408900290508995567506644874811172485351562500000,   16.18476315420433309100189944729208946228027343750000,
          8.44591198941127352384228288428857922554016113281250,    0.00000000000000000000000000000000000000000000000000,  349.95411804077019723990815691649913787841796875000000,  351.59448709993392867545480839908123016357421875000000,  352.47827417972143848601263016462326049804687500000000,  352.90090459047058857322554104030132293701171875000000,  352.98581822598112012201454490423202514648437500000000,  352.80185638115437996020773425698280334472656250000000,  352.39321641202047885599313303828239440917968750000000,  351.79078181991127394212526269257068634033203125000000,
        351.01739436642372993446770124137401580810546875000000,  350.09065194285875577406841330230236053466796875000000,  349.02453599571146014568512327969074249267578125000000,  347.83042467826487609272589907050132751464843750000000,  346.51775755267846079732407815754413604736328125000000,  345.09449029002115594266797415912151336669921875000000,  343.56741650304218183009652420878410339355468750000000,  341.94240209064207647315924987196922302246093750000000,  340.22456001778630252374568954110145568847656250000000,  338.41838337485137344629038125276565551757812500000000,
        336.52784849231284169945865869522094726562500000000000,  334.55649610118541659176116809248924255371093750000000,  332.50749609409052709452225826680660247802734375000000,  330.38369983196741941355867311358451843261718750000000,  328.18768285168914644600590690970420837402343750000000,  325.92178007636215397724299691617488861083984375000000,  323.58811509913027748552849516272544860839843750000000,  321.18862473070026908317231573164463043212890625000000,  318.72507972371857931648264639079570770263671875000000,  316.19910238254033174598589539527893066406250000000000,
        313.61218161390246450537233613431453704833984375000000,  310.96568585819358077060314826667308807373046875000000,  308.26087425238148398420889861881732940673828125000000,  305.49890630714372719012317247688770294189453125000000,  302.68085032728168926041689701378345489501953125000000,  299.80769076239994319621473550796508789062500000000000,  296.88033464141261674740235321223735809326171875000000,  293.89961721769623181899078190326690673828125000000000,  290.86630693014308235433418303728103637695312500000000,  287.78110976785990260395919904112815856933593750000000,
        284.64467311193288878712337464094161987304687500000000,  281.45758911588177397788967937231063842773437500000000,  278.22039767664222154053277336061000823974609375000000,  274.93358903973967244382947683334350585937500000000000,  271.59760607543751120829256251454353332519531250000000,  268.21284625680300450767390429973602294921875000000000,  264.77966336563821414529229514300823211669921875000000,  261.29836894790361156992730684578418731689453125000000,  257.76923353648788861391949467360973358154296875000000,  254.19248765583910198984085582196712493896484375000000,
        250.56832261997735145087062846869230270385742187500000,  246.89689113267513675964437425136566162109375000000000,  243.17830769605649265940883196890354156494140625000000,  239.41264883146175179717829450964927673339843750000000,  235.59995311409878127051342744380235671997070312500000,  231.74022102069596940054907463490962982177734375000000,  227.83341458703864645940484479069709777832031250000000,  223.87945686984875237612868659198284149169921875000000,  219.87823120490091355350159574300050735473632812500000,  215.82958025049441630471847020089626312255859375000000,
        211.73330480234238848424865864217281341552734375000000,  207.58916236251675968560448382049798965454101562500000,  203.39686544119658151430485304445028305053710937500000,  199.15607956548996071433066390454769134521484375000000,  194.86642096438674798264401033520698547363281250000000,  190.52745389277131948801979888230562210083007812500000,  186.13868755015087685933394823223352432250976562500000,  181.69957254105449351300194393843412399291992187500000,  177.20949681356134419729642104357481002807617187500000,  172.66778099966521153874055016785860061645507812500000,
        168.07367306556582775556307751685380935668945312500000,  163.42634216072022468324576038867235183715820312500000,  158.72487153054990471900964621454477310180664062500000,  153.96825032771900509942497592419385910034179687500000,  149.15536411904085412061249371618032455444335937500000,  144.28498383685274575327639468014240264892578125000000,  139.35575286173826725644175894558429718017578125000000,  134.36617184307289107891847379505634307861328125000000,  129.31458075846413180443050805479288101196289062500000,  124.19913757342074234202300431206822395324707031250000,
        119.01779267510669058083294657990336418151855468750000,  113.76825799918009352040826342999935150146484375000000,  108.44796941730376715895545203238725662231445312500000,  103.05404046070171375504287425428628921508789062500000,   97.58320475370602764542127260938286781311035156250000,   92.03174351233117533865879522636532783508300781250000,   86.39539295086575521054328419268131256103515625000000,   80.66922414110332795189606258645653724670410156250000,   74.84748427901939749062876217067241668701171875000000,   68.92338253015057603079185355454683303833007812500000,
         62.88879396771366003804359934292733669281005859375000,   56.73383830474838163127060397528111934661865234375000,   50.44625936889454465017479378730058670043945312500000,   44.01047156996793319194694049656391143798828125000000,   37.40601492190872079390828730538487434387207031250000,   30.60487449355546729634625080507248640060424804687500,   23.56638098650839552306024415884166955947875976562500,   16.22614542731091091809503268450498580932617187500000,    8.46653127661400972669980546925216913223266601562500,    0.00000000000000000000000000000000000000000000000000,
        354.53908551944078908491064794361591339111328125000000,  356.18143128144959064229624345898628234863281250000000,  357.06744927421704005610081367194652557373046875000000,  357.49253675749974945574649609625339508056640625000000,  357.58011543091345174616435542702674865722656250000000,  357.39901425372431731375399976968765258789062500000000,  356.99342199332096470243413932621479034423828125000000,  356.39421610330089151830179616808891296386718750000000,  355.62423410560199954488780349493026733398437500000000,  354.70107099196678746011457405984401702880859375000000,
        353.63870633825024469842901453375816345214843750000000,  352.44851724212117005663458257913589477539062500000000,  351.13994287409025218948954716324806213378906250000000,  349.72093906848698452449752949178218841552734375000000,  348.19830007612091549162869341671466827392578125000000,  346.57789284913559413325856439769268035888671875000000,  344.86483177617270712289609946310520172119140625000000,  343.06361170800130366842495277523994445800781250000000,  341.17821104685214095297851599752902984619140625000000,  339.21217288797492983576375991106033325195312500000000,
        337.16866976693017932120710611343383789062500000000000,  335.05055595663895928737474605441093444824218750000000,  332.86041016871928377440781332552433013916015625000000,  330.60057076034524925489677116274833679199218750000000,  328.27316501704547135886969044804573059082031250000000,  325.88013370135951163319987244904041290283203125000000,  323.42325178026129606223548762500286102294921875000000,  320.90414603973101748124463483691215515136718750000000,  318.32431014188023254973813891410827636718750000000000,  315.68511756425147041227319277822971343994140625000000,
        312.98783277231706279053469188511371612548828125000000,  310.23362090771468047023518010973930358886718750000000,  307.42355622131628933857427909970283508300781250000000,  304.55862943815299104244331829249858856201171875000000,  301.63975420781781622281414456665515899658203125000000,  298.66777276724792500317562371492385864257812500000000,  295.64346092124492315633688122034072875976562500000000,  292.56753242860082764309481717646121978759765625000000,  289.44064286739939007020439021289348602294921875000000,  286.26339304128646290337201207876205444335937500000000,
        283.03633197874745519584394060075283050537109375000000,  279.75995956928153418630245141685009002685546875000000,  276.43472887351515510090393945574760437011718750000000,  273.06104813849532320091384463012218475341796875000000,  269.63928254444653020982514135539531707763671875000000,  266.16975570500341063961968757212162017822265625000000,  262.65275093921098914506728760898113250732421875000000,  259.08851233030759431130718439817428588867187500000000,  255.47724558338020983683236408978700637817382812500000,  251.81911869133395498465688433498144149780273437500000,
        248.11426241617323285026941448450088500976562500000000,  244.36277059030450686805124860256910324096679687500000,  240.56470024037062671595776919275522232055664062500000,  236.72007153397967726959905121475458145141601562500000,  232.82886754753425861963478382676839828491210937500000,  228.89103385116177946656534913927316665649414062500000,  224.90647790443546227834303863346576690673828125000000,  220.87506825410207511595217511057853698730468750000000,  216.79663352234152284836454782634973526000976562500000,  212.67096117109511510534503031522035598754882812500000,
        208.49779602463738115147862117737531661987304687500000,  204.27683852872911529630073346197605133056640625000000,  200.00774272025731193025421816855669021606445312500000,  195.69011387609594976311200298368930816650390625000000,  191.32350580382777138765959534794092178344726562500000,  186.90741772972600642788165714591741561889648437500000,  182.44129073071988500487350393086671829223632812500000,  177.92450364659720207782811485230922698974609375000000,  173.35636839596506320049229543656110763549804687500000,  168.73612460389375655722687952220439910888671875000000,
        164.06293342992637462884886190295219421386718750000000,  159.33587046121590447000926360487937927246093750000000,  154.55391750558183616703900042921304702758789062500000,  149.71595308143426450442348141223192214965820312500000,  144.82074135330503850127570331096649169921875000000000,  139.86691919977474185543542262166738510131835937500000,  134.85298102018867894003051333129405975341796875000000,  129.77726078115861696460342500358819961547851562500000,  124.63791066411651797807280672714114189147949218750000,  119.43287548771370154554460896179080009460449218750000,
        114.15986182400891379984386730939149856567382812500000,  108.81630037598034732582163996994495391845703125000000,  103.39929969168814238855702569708228111267089843750000,   97.90558858898864968978159595280885696411132812500000,   92.33144364579747787047381279990077018737792968750000,   86.67259659885480971297511132434010505676269531250000,   80.92411419558091267845156835392117500305175781250000,   75.08023945379824226620257832109928131103515625000000,   69.13417750054733801334805320948362350463867187500000,   63.07779950400833968160441145300865173339843750000000,
         56.90122139978729620679587242193520069122314453125000,   50.59218336015609196465447894297540187835693359375000,   44.13509625638918265622123726643621921539306640625000,   37.50949667575216039949737023562192916870117187500000,   30.68736636757735070091257512103766202926635742187500,   23.62803281667364885265669727232307195663452148437500,   16.26710393128465881318334140814840793609619140625000,    8.48694014824521580919736152281984686851501464843750,    0.00000000000000000000000000000000000000000000000000,  359.13420536957539752620505169034004211425781250000000,
        360.77850402692257603121106512844562530517578125000000,  361.66672506480887250290834344923496246337890625000000,  362.09423805979878352445666678249835968017578125000000,  362.18444673306106551535776816308498382568359375000000,  362.00616783585832081371336244046688079833984375000000,  361.60358161873443805234273895621299743652343750000000,  361.00755952136415771747124381363391876220703125000000,  360.24093483239596480416366830468177795410156250000000,  359.32129962949233004110283218324184417724609375000000,  358.26263158668899677650188095867633819580078125000000,
        357.07630670101548275852110236883163452148437500000000,  355.77176369436529057566076517105102539062500000000000,  354.35695849671361656874069012701511383056640625000000,  352.83868591954689009071444161236286163330078125000000,  351.22281388167704108127509243786334991455078125000000,  349.51445810001285963153350166976451873779296875000000,  347.71811508165876603015931323170661926269531250000000,  345.83776518794888943375553935766220092773437500000000,  343.87695375706306322172167710959911346435546875000000,  341.83885583739350977339199744164943695068359375000000,
        339.72632847470026717928703874349594116210937500000000,  337.54195340686180770717328414320945739746093750000000,  335.28807226691606047097593545913696289062500000000000,  332.96681586440990940900519490242004394531250000000000,  330.58012873467436065766378305852413177490234375000000,  328.12978986871473807696020230650901794433593750000000,  325.61743033193698693139594979584217071533203125000000,  323.04454832700065480821649543941020965576171875000000,  320.41252214034767575867590494453907012939453125000000,  317.72262132338732953940052539110183715820312500000000,
        314.97601639086269642575643956661224365234375000000000,  312.17378726550180090271169319748878479003906250000000,  309.31693065600796899161650799214839935302734375000000,  306.40636652206239887163974344730377197265625000000000,  303.44294375331026003550505265593528747558593750000000,  300.42744516777776198068750090897083282470703125000000,  297.36059191769902554369764402508735656738281250000000,  294.24304737644985152655863203108310699462890625000000,  291.07542056853560552553972229361534118652343750000000,  287.85826919484424024631152860820293426513671875000000,
        284.59210229725727003824431449174880981445312500000000,  281.27738259988933577915304340422153472900390625000000,  277.91452855845943759049987420439720153808593750000000,  274.50391614437535281467717140913009643554687500000000,  271.04588038588190102018415927886962890625000000000000,  267.54071668495157609868329018354415893554687500000000,  263.98868192537042887124698609113693237304687500000000,  260.38999538461104066300322301685810089111328125000000,  256.74483945950390761936432681977748870849609375000000,  253.05336021336191265618253964930772781372070312500000,
        249.31566775001394375976815354079008102416992187500000,  245.53183641812188398034777492284774780273437500000000,  241.70190484713347700562735553830862045288085937500000,  237.82587581422393441243912093341350555419921875000000,  233.90371593955390494556922931224107742309570312500000,  229.93535520507177238869189750403165817260742187500000,  225.92068628987507850069960113614797592163085937500000,  221.85956371275233323103748261928558349609375000000000,  217.75180276990562333594425581395626068115234375000000,  213.59717825292847237506066448986530303955078125000000,
        209.39542292880267382315651047974824905395507812500000,  205.14622575988869357388466596603393554687500000000000,  200.84922983749120817265065852552652359008789062500000,  196.50402999744540011306526139378547668457031250000000,  192.11017008010725248823291622102260589599609375000000,  187.66713978991603539725474547594785690307617187500000,  183.17437110104708608560031279921531677246093750000000,  178.63123414522320331343507859855890274047851562500000,  174.03703250503889421452186070382595062255859375000000,  169.39099782057365928267245180904865264892578125000000,
        164.69228359784116832997824531048536300659179687500000,  159.93995808371414568682666867971420288085937500000000,  155.13299604200744852278148755431175231933593750000000,  150.27026922756644466971920337527990341186523437500000,  145.35053530701068780217610765248537063598632812500000,  140.37242491283831213877419941127300262451171875000000,  135.33442643721065223871846683323383331298828125000000,  130.23486806634446111274883151054382324218750000000000,  125.07189641671713786763575626537203788757324218750000,  119.84345094682049648326938040554523468017578125000000,
        114.54723306335567656333296326920390129089355468750000,  109.18066848935610835269471863284707069396972656250000,  103.74086096952221680567163275554776191711425781250000,   98.22453468662877185124671086668968200683593750000000,   92.62796174396579829135589534416794776916503906250000,   86.94686955673249428855342557653784751892089843750000,   81.17632069694172969320788979530334472656250000000000,   75.31055414658234781200008001178503036499023437500000,   69.34277113024094774118566419929265975952148437500000,   63.26483904117018397528227069415152072906494140625000,
         57.06687016204084272885665996000170707702636718750000,   50.73660112894553009255105280317366123199462890625000,   44.25843938877834204959071939811110496520996093750000,   37.61191821152752368107030633836984634399414062500000,   30.76901613190027617861233011353760957717895507812500,   23.68905751851142582609099918045103549957275390625000,   16.30764725783599189412598207127302885055541992187500,    8.50714285556273530630733148427680134773254394531250,    0.00000000000000000000000000000000000000000000000000,  363.73937555556352663188590668141841888427734375000000,
        365.38560382978408824783400632441043853759765625000000,  366.27600068141197198201552964746952056884765625000000,  366.70590836344950957936816848814487457275390625000000,  366.79871283045690688595641404390335083007812500000000,  366.62321875165764595294604077935218811035156250000000,  366.22359793230958757703774608671665191650390625000000,  365.63071583273347187059698626399040222167968750000000,  364.86740151616169214321416802704334259033203125000000,  363.95124413375748417820432223379611968994140625000000,  362.89621942916517127741826698184013366699218750000000,
        361.71370225680522025868413038551807403564453125000000,  360.41313083645496817553066648542881011962890625000000,  359.00246112998456737841479480266571044921875000000000,  357.48848843642349493165966123342514038085937500000000,  355.87708155931346709621720947325229644775390625000000,  354.17335745344905717502115294337272644042968750000000,  352.38181418364445107727078720927238464355468750000000,  350.50643396353495973016833886504173278808593750000000,  348.55076425921572536026360467076301574707031250000000,  346.51798250855597416375530883669853210449218750000000,
        344.41094839825643703079549595713615417480468750000000,  342.23224655173788733009132556617259979248046875000000,  339.98422172802418117498746141791343688964843750000000,  337.66900810123763676529051735997200012207031250000000,  335.28855381002648528010467998683452606201171875000000,  332.84464168939177852735156193375587463378906250000000,  330.33890689297294329662690870463848114013671875000000,  327.77285196096681829658336937427520751953125000000000,  325.14785977315096943129901774227619171142578125000000,  322.46520473794572581027750857174396514892578125000000,
        319.72606250001797434379113838076591491699218750000000,  316.93151839553399895521579310297966003417968750000000,  314.08257484213714860743493773043155670166015625000000,  311.18015781736551161884563043713569641113281250000000,  308.22512255254042656815727241337299346923828125000000,  305.21825854765154417691519483923912048339843750000000,  302.16029399531163335268502123653888702392578125000000,  299.05189968759270868758903816342353820800781250000000,  295.89369246782570144205237738788127899169921875000000,  292.68623827972771778149763122200965881347656250000000,
        289.43005485812847155102645047008991241455078125000000,  286.12561409877099549703416414558887481689453125000000,  282.77334413891924214112805202603340148925781250000000,  279.37363117561955050405231304466724395751953125000000,  275.92682104426489786419551819562911987304687500000000,  272.43322057648140344099374487996101379394531250000000,  268.89309875317576370434835553169250488281250000000000,  265.30668766577633732595131732523441314697265625000000,  261.67418329618055850005475804209709167480468750000000,  257.99574612363517189805861562490463256835937500000000,
        254.27150156466049679693242069333791732788085937500000,  250.50154025014319358888315036892890930175781250000000,  246.68591814181101540270901750773191452026367187500000,  242.82465648843592020966752897948026657104492187500000,  238.91774162023745020633214153349399566650390625000000,  234.96512457804331575061951298266649246215820312500000,  230.96672057176090220309561118483543395996093750000000,  226.92240826057809499616269022226333618164062500000000,  222.83202884499249307737045455724000930786132812500000,  218.69538495820540902059292420744895935058593750000000,
        214.51223934154512562599848024547100067138671875000000,  210.28231328532010024900955613702535629272460937500000,  206.00528481275213721346517559140920639038085937500000,  201.68078658028383642886183224618434906005859375000000,  197.30840346244829675015353132039308547973632812500000,  192.88766978345412894668697845190763473510742187500000,  188.41806615044822592608397826552391052246093750000000,  183.89901583478928159820497967302799224853515625000000,  179.32988063723428240336943417787551879882812500000000,  174.70995616024242735875304788351058959960937500000000,
        170.03846639503782967040024232119321823120117187500000,  165.31455751185572466965822968631982803344726562500000,  160.53729071790235138905700296163558959960937500000000,  155.70563401761054933558625634759664535522460937500000,  150.81845267194802318044821731746196746826171875000000,  145.87449810534565131092676892876625061035156250000000,  140.87239494687696605978999286890029907226562500000000,  135.81062581193995697503851260989904403686523437500000,  130.68751332530607101034547667950391769409179687500000,  125.50119874668619956992188235744833946228027343750000,
        120.24961637249606383193167857825756072998046875000000,  114.93046263266730022678530076518654823303222656250000,  109.54115844994714734639273956418037414550781250000000,  104.07880293693064288618188584223389625549316406250000,   98.54011580465001429729454685002565383911132812500000,   92.92136483764817000974289840087294578552246093750000,   87.21827327842520105605217395350337028503417968750000,   81.42589966578677262987184803932905197143554687500000,   75.53847908181548120865045348182320594787597656250000,   69.54920897864269591082120314240455627441406250000000,
         63.44995309898558133454571361653506755828857421875000,   57.23082019166583478408938390202820301055908203125000,   50.87954347065894467050384264439344406127929687500000,   44.38052706778059786074663861654698848724365234375000,   37.71330104053095766403203015215694904327392578125000,   30.84984080950061624548652616795152425765991210937500,   23.74946772355596991133097617421299219131469726562500,   16.34778373999086298340444045607000589370727539062500,    8.52714352226940519585696165449917316436767578125000,    0.00000000000000000000000000000000000000000000000000,
        368.35449607240474279024056158959865570068359375000000,  370.00263119709302372939419001340866088867187500000000,  370.89517724666001186051289550960063934326171875000000,  371.32744950263435157467029057443141937255859375000000,  371.42281636098624630903941579163074493408203125000000,  371.25007053319785654821316711604595184326171875000000,  370.85337545050845164951169863343238830566406250000000,  370.26359062912069930462166666984558105468750000000000,  369.50354091610790874256053939461708068847656250000000,  368.59081252567074216131004504859447479248046875000000,
        367.53937924492799993458902463316917419433593750000000,  366.36061474658652059588348492980003356933593750000000,  365.06395669806602199969347566366195678710937500000000,  363.65736103301117054797941818833351135253906250000000,  362.14762346885584065603325143456459045410156250000000,  360.54061361629118209748412482440471649169921875000000,  358.84144958234219302539713680744171142578125000000000,  357.05463089601119008875684812664985656738281250000000,  355.18414152192895016924012452363967895507812500000000,  353.23353094499731241739937104284763336181640625000000,
        351.20597887554623639516648836433887481689453125000000,  349.10434751611353476619115099310874938964843750000000,  346.93122424222576682950602844357490539550781250000000,  344.68895679681452293152688071131706237792968750000000,  342.37968256748979456460801884531974792480468750000000,  340.00535313569270101652364246547222137451171875000000,  337.56775500997514427581336349248886108398437500000000,  335.06852725129897407896351069211959838867187500000000,  332.50917654541399315348826348781585693359375000000000,  329.89109016170311861060326918959617614746093750000000,
        327.21554714937832386567606590688228607177734375000000,  324.48372805350459202600177377462387084960937500000000,  321.69672337995558564216480590403079986572265625000000,  318.85554099639119840503553859889507293701171875000000,  315.96111262300820499149267561733722686767578125000000,  313.01429954014247414306737482547760009765625000000000,  310.01589761831661462565534748136997222900390625000000,  306.96664175889043235656572505831718444824218750000000,  303.86720981922479722925345413386821746826171875000000,  300.71822608455948966366122476756572723388671875000000,
        297.52026433910731384457903914153575897216796875000000,  294.27385058079283908227807842195034027099609375000000,  290.97946541729100999873480759561061859130859375000000,  287.63754617530463519869954325258731842041015625000000,  284.24848875016027704987209290266036987304687500000000,  280.81264921863709105309681035578250885009765625000000,  277.33034523434730544977355748414993286132812500000000,  273.80185722185103713854914531111717224121093750000000,  270.22742938292259395893779583275318145751953125000000,  266.60727052592380914575187489390373229980468750000000,
        262.94155472701288545067654922604560852050781250000000,  259.23042182987205706012900918722152709960937500000000,  255.47397778873417450995475519448518753051757812500000,  251.67229485767319374645012430846691131591796875000000,  247.82541162736507089903170708566904067993164062500000,  243.93333290878567254367226269096136093139648437500000,  239.99602946154737992401351220905780792236328125000000,  236.01343756275636565078457351773977279663085937500000,  231.98545841034950854009366594254970550537109375000000,  227.91195735280581402548705227673053741455078125000000,
        223.79276293486901749929529614746570587158203125000000,  219.62766574640710359744844026863574981689453125000000,  215.41641705871018075413303449749946594238281250000000,  211.15872722930410532171663362532854080200195312500000,  206.85426385264116788675892166793346405029296875000000,  202.50264962970595661317929625511169433593750000000000,  198.10345992449325080997368786484003067016601562500000,  193.65621996930653381241427268832921981811523437500000,  189.16040167365315483039012178778648376464843750000000,  184.61541998290402943894150666892528533935546875000000,
        180.02062872247032032646529842168092727661132812500000,  175.37531585056626681762281805276870727539062500000000,  170.67869802707826920595834963023662567138671875000000,  165.92991438685439220535045024007558822631835937500000,  161.12801938184477990034793037921190261840820312500000,  156.27197452658438692196796182543039321899414062500000,  151.36063884369320931000402197241783142089843750000000,  146.39275775788607347749348264187574386596679687500000,  141.36695012505538215918932110071182250976562500000000,  136.28169300261578200661460869014263153076171875000000,
        131.13530366191918119511683471500873565673828125000000,  125.92591820383761103130382252857089042663574218750000,  120.65146595114960348382737720385193824768066406250000,  115.30963853653256023790163453668355941772460937500000,  109.89785225356442310840066056698560714721679687500000,  104.41320174594186198646639240905642509460449218750000,   98.85240240870618322333029936999082565307617187500000,   93.21171785637308460081840166822075843811035156250000,   87.48686730082651763495960040017962455749511718750000,   81.67290538347937456364888930693268775939941406250000,
         75.76406341639813035726547241210937500000000000000000,   69.75353520354632053113164147362112998962402343750000,   63.63318095609095337295002536848187446594238281250000,   57.39310600296963826849605538882315158843994140625000,   51.02104024524901859649617108516395092010498046875000,   44.50138460436525633667770307511091232299804687500000,   37.81366602575173629929849994368851184844970703125000,   30.92985691224392041931423591449856758117675781250000,   23.80927568546584893738327082246541976928710937500000,   16.38752146237308693343948107212781906127929687500000,
          8.54694614956558496032812399789690971374511718750000,    0.00000000000000000000000000000000000000000000000000,  372.97946888568901613325579091906547546386718750000000,  374.62948859022407077645766548812389373779296875000000,  375.52415781748493373015662655234336853027343750000000,  375.95876522228098792766104452311992645263671875000000,  376.05666184627119719152688048779964447021484375000000,  375.88662856582936910854186862707138061523437500000000,  375.49282050910596808535046875476837158203125000000000,  374.90609128400473082365351729094982147216796875000000,
        374.14926153196955738167162053287029266357421875000000,  373.23991452159492610007873736321926116943359375000000,  372.19202205962159268892719410359859466552734375000000,  371.01695660062534898315789178013801574707031250000000,  369.72415521248672121146228164434432983398437500000000,  368.32157374391846360595081932842731475830078125000000,  366.81600826537743387234513647854328155517578125000000,  365.21332912122329616977367550134658813476562500000000,  363.51865548950166839858866296708583831787109375000000,  361.73648827468088029490900225937366485595703125000000,
        359.87081309624869618346565403044223785400390625000000,  357.92518135435517478981637395918369293212890625000000,  355.90277492072772247411194257438182830810546875000000,  353.80645839501710270269541069865226745605468750000000,  351.63882177825053076958283782005310058593750000000000,  349.40221566250210116777452640235424041748046875000000,  347.09878050560939755087019875645637512207031250000000,  344.73047117966285668444470502436161041259765625000000,  342.29907770529291610728250816464424133300781250000000,  339.80624287948393202896113507449626922607421875000000,
        337.25347735185420106063247658312320709228515625000000,  334.64217258870507976098451763391494750976562500000000,  331.97361207566922303158207796514034271240234375000000,  329.24898104140504528913879767060279846191406250000000,  326.46937493143065012191073037683963775634765625000000,  323.63580681919819426184403710067272186279296875000000,  320.74921390818485633644741028547286987304687500000000,  317.81046325212201963950064964592456817626953125000000,  314.82035679901429148230818100273609161376953125000000,  311.77963584717673484192346222698688507080078125000000,
        308.68898498729117818584199994802474975585937500000000,  305.54903559278545799315907061100006103515625000000000,  302.36036891115907110361149534583091735839843750000000,  299.12351880082468369437265209853649139404296875000000,  295.83897415127904650944401510059833526611328125000000,  292.50718101872740817270823754370212554931640625000000,  289.12854450444450549184693954885005950927734375000000,  285.70343039902451209854916669428348541259765625000000,  282.23216661210369693435495719313621520996093750000000,  278.71504440403924718339112587273120880126953125000000,
        275.15231943330547892401227727532386779785156250000000,  271.54421263095002814225153997540473937988281250000000,  267.89091091128017296796315349638462066650390625000000,  264.19256772596588689339114353060722351074218750000000,  260.44930346691313616247498430311679840087890625000000,  256.66120572152681234001647680997848510742187500000000,  252.82832938232309061277192085981369018554687500000000,  248.95069661121786452895321417599916458129882812500000,  245.02829665718780915995012037456035614013671875000000,  241.06108552433107661272515542805194854736328125000000,
        237.04898548561516236077295616269111633300781250000000,  232.99188443574794860069232527166604995727539062500000,  228.88963507460368873580591753125190734863281250000000,  224.74205391043108193116495385766029357910156250000000,  220.54892006960582762076228391379117965698242187500000,  216.30997389690560339658986777067184448242187500000000,  212.02491532709720445382117759436368942260742187500000,  207.69340200494011128284910228103399276733398437500000,  203.31504712641421406260633375495672225952148437500000,  198.88941696892382537953380960971117019653320312500000,
        194.41602807224003868213912937790155410766601562500000,  189.89434402479349728309898637235164642333984375000000,  185.32377180133426008978858590126037597656250000000000,  180.70365758757699836678511928766965866088867187500000,  176.03328201477921766127110458910465240478515625000000,  171.31185471166182310298609081655740737915039062500000,  166.53850806188711430877447128295898437500000000000000,  161.71229003143378122331341728568077087402343750000000,  156.83215590027847952114825602620840072631835937500000,  151.89695869498373781425470951944589614868164062500000,
        146.90543807061757775045407470315694808959960937500000,  141.85620732850449599027342628687620162963867187500000,  136.74773817594169145195337478071451187133789062500000,  131.57834272863604496706102509051561355590820312500000,  126.34615211691279057504289085045456886291503906250000,  121.04909086928766726032336009666323661804199218750000,  115.68484599216559161050099646672606468200683593750000,  110.25082931303229827335599111393094062805175781250000,  104.74413116231296783098514424636960029602050781250000,   99.16146276765798006636032368987798690795898437500000,
         93.49908371552335495380248175933957099914550781250000,   87.75270932288037784019252285361289978027343750000000,   81.91739046353247033493971684947609901428222656250000,   75.98735480371439621194440405815839767456054687500000,   69.95579261810809157395851798355579376220703125000000,   63.81456070019986270835943287238478660583496093750000,   57.55376106815934633686993038281798362731933593750000,   51.16112041475433613868517568334937095642089843750000,   44.62103655137700286559265805408358573913574218750000,   37.91303340767260721122511313296854496002197265625000,
         31.00908046114757965483477164525538682937622070312500,   23.86849329494927118844316282775253057479858398437500,   16.42686827098079405118369322735816240310668945312500,    8.56655462095396025290483521530404686927795410156250,    0.00000000000000000000000000000000000000000000000000,  377.61419787391866975667653605341911315917968750000000,  379.26608036786336697332444600760936737060546875000000,  380.16284732895638853733544237911701202392578125000000,  380.59976112291172967161401174962520599365234375000000,  380.70015563769277378014521673321723937988281250000000,
        380.53280003553913957148324698209762573242187500000000,  380.14184121206181998786632902920246124267578125000000,  379.55812690319220337187289260327816009521484375000000,  378.80447355644309936906211078166961669921875000000000,  377.89846148785346713339095003902912139892578125000000,  376.85406050206808004077174700796604156494140625000000,  375.68264180163612309115706011652946472167968750000000,  374.39364181059386282868217676877975463867187500000000,  372.99501623922600401783711276948451995849609375000000,  371.49356144916765742891584523022174835205078125000000,
        369.89514844882404531745123676955699920654296875000000,  368.20489741023595797742018476128578186035156250000000,  366.42731052913234179868595674633979797363281250000000,  364.56637498857980972388759255409240722656250000000000,  362.62564400563735489413375034928321838378906250000000,  360.60830150796232373977545648813247680664062500000000,  358.51721437958724436612101271748542785644531250000000,  356.35497512684838739005499519407749176025390625000000,  354.12393706305954310664674267172813415527343750000000,  351.82624358036656531112384982407093048095703125000000,
        349.46385269721753275007358752191066741943359375000000,  347.03855779325147068448131904006004333496093750000000,  344.55200523916840893434709869325160980224609375000000,  342.00570947639943142348784022033214569091796875000000,  339.40106598579302499274490401148796081542968750000000,  336.73936249609033666274626739323139190673828125000000,  334.02178871460040454621776007115840911865234375000000,  331.24944480915644362539751455187797546386718750000000,  328.42334882845801757866865955293178558349609375000000,  325.54444321459345701441634446382522583007812500000000,
        322.61360053490108157348004169762134552001953125000000,  319.63162853886802849956438876688480377197265625000000,  316.59927462836026279546786099672317504882812500000000,  313.51722981526074818248162046074867248535156250000000,  310.38613222891137866099597886204719543457031250000000,  307.20657022609174191529746167361736297607421875000000,  303.97908514822620418271981179714202880859375000000000,  300.70417376377679374854778870940208435058593750000000,  297.38229042810593227841309271752834320068359375000000,  294.01384898827944880395079962909221649169921875000000,
        290.59922445616689401504117995500564575195312500000000,  287.13875446966193294429103843867778778076171875000000,  283.63274055877178625451051630079746246337890625000000,  280.08144923064031672765850089490413665771484375000000,  276.48511288519006257047294639050960540771484375000000,  272.84393057094234791293274611234664916992187500000000,  269.15806858864488049221108667552471160888671875000000,  265.42766094856176550820237025618553161621093750000000,  261.65280968562223051776527427136898040771484375000000,  257.83358503504081227220012806355953216552734375000000,
        253.97002546948868939580279402434825897216796875000000,  250.06213759737553914419549982994794845581054687500000,  246.10989592026129457735805772244930267333984375000000,  242.11324244583281029008503537625074386596679687500000,  238.07208615120842409851320553570985794067382812500000,  233.98630228954485232861770782619714736938476562500000,  229.85573153096828491470660082995891571044921875000000,  225.68017892669348611889290623366832733154296875000000,  221.45941268277022118127206340432167053222656250000000,  217.19316272714735305271460674703121185302734375000000,
        212.88111905058707407079054974019527435302734375000000,  208.52292979830392027906782459467649459838867187500000,  204.11819908493080788502993527799844741821289062500000,  199.66648450037897077891102526336908340454101562500000,  195.16729426818827164424874354153871536254882812500000,  190.62008401083045328050502575933933258056640625000000,  186.02425306784689951200562063604593276977539062500000,  181.37914030231800666115304920822381973266601562500000,  176.68401931850129926715453621000051498413085937500000,  171.93809299794810385719756595790386199951171875000000,
        167.14048724222345754242269322276115417480468750000000,  162.29024378648645665634830947965383529663085937500000,  157.38631191826652866438962519168853759765625000000000,  152.42753889796651378674141597002744674682617187500000,  147.41265882945666021441866178065538406372070312500000,  142.34027966720225322205806151032447814941406250000000,  137.20886796600686352576303761452436447143554687500000,  132.01673087407999673814629204571247100830078125000000,  126.76199473043588739074039040133357048034667968750000,  121.44257944017618910947930999100208282470703125000000,
        116.05616754638265319954371079802513122558593750000000,  110.60016656595300332810438703745603561401367187500000,  105.07166266452287572974455542862415313720703125000000,   99.46736304420575436324725160375237464904785156250000,   93.78352339899232958941865945234894752502441406250000,   88.01585528061933416665851837024092674255371093750000,   82.15940591936393389005388598889112472534179687500000,   76.20839945441792906422051601111888885498046875000000,   70.15602274495699930412229150533676147460937500000000,   63.99412927581229837414866778999567031860351562500000,
         57.71281785890800364313690806739032268524169921875000,   51.29981207896422290559712564572691917419433593750000,   44.73950673352631923762601218186318874359130859375000,   38.01142282879888512070465367287397384643554687500000,   31.08752700564899029700427490752190351486206054687500,   23.92713209395965279213669418822973966598510742187500,   16.46583178248656764708357513882219791412353515625000,    8.58597270681106294887285912409424781799316406250000,    0.00000000000000000000000000000000000000000000000000,  382.25858877306001204487984068691730499267578125000000,
        383.91231273119825573303387500345706939697265625000000,  384.81115254027236005640588700771331787109375000000000,  385.25034460759007970409584231674671173095703125000000,  385.35320586444890977873001247644424438476562500000000,  385.18849387827646069126785732805728912353515625000000,  384.80034738227573143376503139734268188476562500000000,  384.21960827716901576422969810664653778076171875000000,  383.46908882930921436127391643822193145751953125000000,  382.56636639680783673611585982143878936767578125000000,  381.52540876249338452907977625727653503417968750000000,
        380.35758584536085891159018501639366149902343750000000,  379.07233338400146749336272478103637695312500000000000,  377.67760689979814969774452038109302520751953125000000,  376.18020298704851711590890772640705108642578125000000,  374.58599325221484832582063972949981689453125000000000,  372.90009878824713496214826591312885284423828125000000,  371.12702300218273876453167758882045745849609375000000,  369.27075455396425240905955433845520019531250000000000,  367.33484838413193074302398599684238433837890625000000,  365.32249037606857200444210320711135864257812500000000,
        363.23654959109069295664085075259208679199218750000000,  361.07962092667918341248878277838230133056640625000000,  358.85406029592797949590021744370460510253906250000000,  356.56201389626056652559782378375530242919921875000000,  354.20544275555567992341821081936359405517578125000000,  351.78614346726442363433307036757469177246093750000000,  349.30576582191650913955527357757091522216796875000000,  346.76582788971040827163960784673690795898437500000000,  344.16772899331584767423919402062892913818359375000000,  341.51276092159997688213479705154895782470703125000000,
        338.80211766665075856508337892591953277587890625000000,  336.03690391316177965563838370144367218017578125000000,  333.21814246727944919257424771785736083984375000000000,  330.34678077872428048067376948893070220947265625000000,  327.42369668337062194041209295392036437988281250000000,  324.44970347202797711361199617385864257812500000000000,  321.42555437376904592383652925491333007812500000000000,  318.35194652795229330877191387116909027099609375000000,  315.22952450741490793006960302591323852539062500000000,  312.05888344566363912235829047858715057373046875000000,
        308.84057181286675586306955665349960327148437500000000,  305.57509387873051309725269675254821777343750000000000,  302.26291189468844322618679143488407135009765625000000,  298.90444802303699134427006356418132781982421875000000,  295.50008603656084460453712381422519683837890625000000,  292.05017280868025864037917926907539367675781250000000,  288.55501961110553565958980470895767211914062500000000,  285.01490323333251808435306884348392486572265625000000,  281.43006693596709055782412178814411163330078125000000,  277.80072124778132547362474724650382995605468750000000,
        274.12704461452062787429895251989364624023437500000000,  270.40918390576007368508726358413696289062500000000000,  266.64725478450804985186550766229629516601562500000000,  262.84134194274651008527143858373165130615234375000000,  258.99149920464225260730017907917499542236328125000000,  255.09774949774066499230684712529182434082031250000000,  251.16008469102530398231465369462966918945312500000000,  247.17846529726961080086766742169857025146484375000000,  243.15282003559352119737013708800077438354492187500000,  239.08304524852451322658453136682510375976562500000000,
        234.96900416612891149270581081509590148925781250000000,  230.81052600787197093268332537263631820678710937500000,  226.60740491074707847474201116710901260375976562500000,  222.35939866982695889419119339436292648315429687500000,  218.06622727466890410141786560416221618652343750000000,  213.72757122187763911824731621891260147094726562500000,  209.34306958049518243569764308631420135498046875000000,  204.91231778263377805160416755825281143188476562500000,  200.43486510675501222067396156489849090576171875000000,  195.91021181504009973650681786239147186279296875000000,
        191.33780589918072223554190713912248611450195312500000,  186.71703938034946190782648045569658279418945312500000,  182.04724409873722379415994510054588317871093750000000,  177.32768691539510541588242631405591964721679687500000,  172.55756423360011808654235210269689559936523437500000,  167.73599572778746846779540646821260452270507812500000,  162.86201714423137332232727203518152236938476562500000,  157.93457200774290072331496048718690872192382812500000,  152.95250203085282691972679458558559417724609375000000,  147.91453597378750828283955343067646026611328125000000,
        142.81927664162870428299356717616319656372070312500000,  137.66518562469184416841017082333564758300781250000000,  132.45056528278843188672908581793308258056640625000000,  127.17353733433954232623364077880978584289550781250000,  121.83201722385638277046382427215576171875000000000000,  116.42368318648094316358765354380011558532714843750000,  110.94593857688872162725601810961961746215820312500000,  105.39586553766825716138555435463786125183105468750000,   99.77016738088300940034969244152307510375976562500000,   94.06509603763305449319886974990367889404296875000000,
         88.27635941840418354331632144749164581298828125000000,   82.39900122864005993505998048931360244750976562500000,   76.42724219417240760776621755212545394897460937500000,   70.35426586760701184175559319555759429931640625000000,   64.17192252955634046429622685536742210388183593750000,   57.87030788586724838751251809298992156982421875000000,   51.43714250932921316916690557263791561126708984375000,   44.85681827591083958850504131987690925598144531250000,   38.10885335699092024697165470570325851440429687500000,   31.16521164193728665736671246122568845748901367187500,
         23.98520328920380606518847343977540731430053710937500,   16.50441939308883831927232677116990089416503906250000,    8.60520406873894927457513404078781604766845703125000,    0.00000000000000000000000000000000000000000000000000,  386.91254912321750225601135753095149993896484375000000,  388.56809367119785747490823268890380859375000000000000,  389.46898198279984626424266025424003601074218750000000,  389.91042483086829406602191738784313201904296875000000,  390.01572238355430499723297543823719024658203125000000,  389.85362073115317116389633156359195709228515625000000,
        389.46825051413946994216530583798885345458984375000000,  388.89044783516197867356822825968265533447265625000000,  388.14302079316922799989697523415088653564453125000000,  387.24354378444024860073113813996315002441406250000000,  386.20598255213883476244518533349037170410156250000000,  385.04170570240148663287982344627380371093750000000000,  383.76014824931763769200188107788562774658203125000000,  382.36926547774049822692177258431911468505859375000000,  380.87585415925127563241403549909591674804687500000000,  379.28578643580470952656469307839870452880859375000000,
        377.60418425188731816888321191072463989257812500000000,  375.83555214989922887980355881154537200927734375000000,  373.98388018426618373268865980207920074462890625000000,  372.05272493020589763546013273298740386962890625000000,  370.04527413157580895131104625761508941650390625000000,  367.96439892517929592941072769463062286376953125000000,  365.81269649114364028719137422740459442138671875000000,  363.59252522693913078910554759204387664794921875000000,  361.30603401270531094269244931638240814208984375000000,  358.95518675573589462146628648042678833007812500000000,
        356.54178312548259555114782415330410003662109375000000,  354.06747618631135310351965017616748809814453125000000,  351.53378748258040786822675727307796478271484375000000,  348.94212001507986542492290027439594268798828125000000,  346.29376945947905142020317725837230682373046875000000,  343.58993390911695087197585962712764739990234375000000,  340.83172237117491931712720543146133422851562500000000,  338.02016220333086948812706395983695983886718750000000,  335.15620564471237230463884770870208740234375000000000,  332.24073556835890030924929305911064147949218750000000,
        329.27457056096812948453589342534542083740234375000000,  326.25846941832185166276758536696434020996093750000000,  323.19313513059654496828443370759487152099609375000000,  320.07921842010694035707274451851844787597656250000000,  316.91732088439511016986216418445110321044921875000000,  313.70799778956524050954612903296947479248046875000000,  310.45176055206110277140396647155284881591796875000000,  307.14907894144215561027522198855876922607421875000000,  303.80038303193811088931397534906864166259765625000000,  300.40606492649061465272097848355770111083984375000000,
        296.96648027349908716132631525397300720214843750000000,  293.48194959347006260941270738840103149414062500000000,  289.95275943013780306500848382711410522460937500000000,  286.37916333831645943064359016716480255126953125000000,  282.76138271868796891794772818684577941894531250000000,  279.09960750789093708590371534228324890136718750000000,  275.39399673059728002044721506536006927490234375000000,  271.64467891871970550710102543234825134277343750000000,  267.85175240144133113062707707285881042480468750000000,  264.01528546837823796522570773959159851074218750000000,
        260.13531640684226431403658352792263031005859375000000,  256.21185341283813841073424555361270904541015625000000,  252.24487437408791379311878699809312820434570312500000,  248.23432652198502523788192775100469589233398437500000,  244.18012594792753588990308344364166259765625000000000,  240.08215697792300602486648131161928176879882812500000,  235.94027139766606637749646324664354324340820312500000,  231.75428751842542851591133512556552886962890625000000,  227.52398907199250288613256998360157012939453125000000,  223.24912392058786281268112361431121826171875000000000,
        218.92940256492838102531095501035451889038085937500000,  214.56449643055270826152991503477096557617187500000000,  210.15403590888976737005577888339757919311523437500000,  205.69760812532095428650791291147470474243164062500000,  201.19475440148943334861542098224163055419921875000000,  196.64496737316693497632513754069805145263671875000000,  192.04768771788596382066316436976194381713867187500000,  187.40230043798635506391292437911033630371093750000000,  182.70813063436395395910949446260929107666015625000000,  177.96443869356875211451551876962184906005859375000000,
        173.17041479538912085445190314203500747680664062500000,  168.32517262889146536508633289486169815063476562500000,  163.42774218102798045038070995360612869262695312500000,  158.47706143202000816927466075867414474487304687500000,  153.47196675392791576086892746388912200927734375000000,  148.41118176066390788037097081542015075683593750000000,  143.29330429579172800913511309772729873657226562500000,  138.11679116410394385638937819749116897583007812500000,  132.87994010760303353890776634216308593750000000000000,  127.58086838681686003837967291474342346191406250000000,
        122.21748714093068599595426348969340324401855468750000,  116.78747044539836963394918711856007575988769531250000,  111.28821763430569546926562907174229621887207031250000,  105.71680696257524800785176921635866165161132812500000,  100.06993798169339981996017741039395332336425781250000,   94.34385898375521151137945707887411117553710937500000,   88.53427435659386901534162461757659912109375000000000,   82.63622439441139988502982305362820625305175781250000,   76.64392651852801918721524998545646667480468750000000,   70.55056107933145881361269857734441757202148437500000,
         64.34797525330191092507448047399520874023437500000000,   58.02626173624712890841692569665610790252685546875000,   51.57313818121982507136635831557214260101318359375000,   44.97299363115313752814472536556422710418701171875000,   38.20534350766911302343942224979400634765625000000000,   31.24214903040271096301694342400878667831420898437500,   24.04271776500229762518756615463644266128540039062500,   16.54263828694029569987833383493125438690185546875000,    8.62425226370964459476908814394846558570861816406250,    0.00000000000000000000000000000000000000000000000000,
        391.57598821732960914232535287737846374511718750000000,  393.23333291788560472923563793301582336425781250000000,  394.13624591006851005658973008394241333007812500000000,  394.57991264964152833272237330675125122070312500000000,  394.68761673169234427405172027647495269775390625000000,  394.52809288543272714377962984144687652587890625000000,  394.14546372780671390501083806157112121582031250000000,  393.57055960083749823752441443502902984619140625000000,  392.82618445072642998638912104070186614990234375000000,  391.92990970938132022638455964624881744384765625000000,
        390.89569906420479128428269177675247192382812500000000,  389.73491978125963441925705410540103912353515625000000,  388.45700611347177755305892787873744964599609375000000,  387.06991306421730314468732103705406188964843750000000,  385.58043752993893349412246607244014739990234375000000,  383.99445212874775279487948864698410034179687500000000,  382.31707959079426473181229084730148315429687500000000,  380.55282552167329868098022416234016418457031250000000,  378.70568129196283280180068686604499816894531250000000,  376.77920502711157269004615955054759979248046875000000,
        374.77658624086637928485288284718990325927734375000000,  372.70069804870973939614486880600452423095703125000000,  370.55413981024872782654711045324802398681640625000000,  368.33927229762042543370625935494899749755859375000000,  366.05824695719957162509672343730926513671875000000000,  363.71303045217064209282398223876953125000000000000000,  361.30542539710569371891324408352375030517578125000000,  358.83708799161610158989788033068180084228515625000000,  356.30954310752110814064508303999900817871093750000000,  353.72419726848272603092482313513755798339843750000000,
        351.08234987268798477089148946106433868408203125000000,  348.38520294086907824748777784407138824462890625000000,  345.63386961867752233956707641482353210449218750000000,  342.82938162050021446702885441482067108154296875000000,  339.97269576854154138345620594918727874755859375000000,  337.06469975439767949865199625492095947265625000000000,  334.10621722892818752370658330619335174560546875000000,  331.09801230886051826018956489861011505126953125000000,  328.04079357438479291886324062943458557128906250000000,  324.93521762035004485369427129626274108886718750000000,
        321.78189221404943509696749970316886901855468750000000,  318.58137910458174246741691604256629943847656250000000,  315.33419652208658590097911655902862548828125000000000,  312.04082139952339502997347153723239898681640625000000,  308.70169134490419082794687710702419281005859375000000,  305.31720638783616550426813773810863494873046875000000,  301.88773052075725900067482143640518188476562500000000,  298.41359305225029174835071898996829986572265625000000,  294.89508978721852372473222203552722930908203125000000,  291.33248404641670958881149999797344207763671875000000,
        287.72600753581582466722466051578521728515625000000000,  284.07586107446849155167001299560070037841796875000000,  280.38221518790783193253446370363235473632812500000000,  276.64521057261225678303162567317485809326171875000000,  272.86495843567263364093378186225891113281250000000000,  269.04154071247552337808883748948574066162109375000000,  265.17501016394504631534800864756107330322265625000000,  261.26539035363560969926766119897365570068359375000000,  257.31267550371819652355043217539787292480468750000000,  253.31683022762786094972398132085800170898437500000000,
        249.27778913581616393457807134836912155151367187500000,  245.19545630964631754977745003998279571533203125000000,  241.06970463696205797532456927001476287841796875000000,  236.90037500120789104585128370672464370727539062500000,  232.68727531415063936037768144160509109497070312500000,  228.43017938019718826581083703786134719848632812500000,  224.12882557797567528723448049277067184448242187500000,  219.78291534217720482047297991812229156494140625000000,  215.39211142557115863382932730019092559814453125000000,  210.95603591751324756842222996056079864501953125000000,
        206.47426799104781025562260765582323074340820312500000,  201.94634134572234529514389578253030776977539062500000,  197.37174130730466004024492576718330383300781250000000,  192.74990153849861940216214861720800399780273437500000,  188.08020030620977536273130681365728378295898437500000,  183.36195624055696384857583325356245040893554687500000,  178.59442350819588796184689272195100784301757812500000,  173.77678630701657880308630410581827163696289062500000,  168.90815257011587391389184631407260894775390625000000,  163.98754674309614642879751045256853103637695312500000,
        159.01390146884068599320016801357269287109375000000000,  153.98604797612557604225003160536289215087890625000000,  148.90270492027684667846187949180603027343750000000000,  143.76246536217325910911313258111476898193359375000000,  138.56378149154670609277673065662384033203125000000000,  133.30494659516892852479941211640834808349609375000000,  127.98407363082201015913597075268626213073730468750000,  122.59906958050444814034563023597002029418945312500000,  117.14760450150630788357375422492623329162597656250000,  111.62707384260080800686409929767251014709472656250000,
        106.03455210041731504588824464008212089538574218750000,  100.36673518965424989346502115949988365173339843750000,   94.61986788190704089629434747621417045593261718750000,   88.78965115585855016888672253116965293884277343750000,   82.87112200323139177271514199674129486083984375000000,   76.85849464510278039597324095666408538818359375000000,   70.74494632964839979649696033447980880737304687500000,   64.52232122517676771167316474020481109619140625000000,   58.18070910957542452024426893331110477447509765625000,   51.70782480462922592323593562468886375427246093750000,
         45.08805460523486630108891404233872890472412109375000,   38.30091126495638320648140506818890571594238281250000,   31.31835341225409763410425512120127677917480468750000,   24.09968609553894935970674850977957248687744140625000,   16.58049544417732334977699792943894863128662109375000,    8.64312074801402729917754186317324638366699218750000,    0.00000000000000000000000000000000000000000000000000,  396.24881705179149093964952044188976287841796875000000,  397.90794189150989268455305136740207672119140625000000,  398.81285624962657720971037633717060089111328125000000,
        399.25872057581977969675790518522262573242187500000000,  399.36880207883399407364777289330959320068359375000000,  399.21182424122736165372771210968494415283203125000000,  398.83190172510273896477883681654930114746093750000000,  398.25985914956407896170276217162609100341796875000000,  397.51849632354725372351822443306446075439453125000000,  396.62538171332170122695970349013805389404296875000000,  395.59447693607404517024406231939792633056640625000000,  394.43714789254164543308434076607227325439453125000000,  393.16282804007954609915032051503658294677734375000000,
        391.77947205816633413633098825812339782714843750000000,  390.29387691847239238995825871825218200683593750000000,  388.71191565897430564291425980627536773681640625000000,  387.03871173291685181538923643529415130615234375000000,  385.27877174048444430809468030929565429687500000000000,  383.43608829390694836547481827437877655029296875000000,  381.51422098852555109260720200836658477783203125000000,  379.51636102178804321738425642251968383789062500000000,  377.44538339574864949099719524383544921875000000000000,  375.30388955134782236200408078730106353759765625000000,
        373.09424253103878754700417630374431610107421875000000,  370.81859623666872494140989147126674652099609375000000,  368.47891996989022800335078500211238861083984375000000,  366.07701916602684377721743658185005187988281250000000,  363.61455302829313041002023965120315551757812500000000,  361.09304961668868827473488636314868927001953125000000,  358.51391883042236941037117503583431243896484375000000,  355.87846363437955687913927249610424041748046875000000,  353.18788981187913122994359582662582397460937500000000,  350.44331447270644730451749637722969055175781250000000,
        347.64577350349946982532856054604053497314453125000000,  344.79622811431158879713620990514755249023437500000000,  341.89557060859038983835489489138126373291015625000000,  338.94462948240180821812828071415424346923828125000000,  335.94417394137030896672513335943222045898437500000000,  332.89491790963666062452830374240875244140625000000000,  329.79752359350044343955232761800289154052734375000000,  326.65260465280186963354935869574546813964843750000000,  323.46072902510627500305417925119400024414062500000000,  320.22242144107940475805662572383880615234375000000000,
        316.93816566382452037942130118608474731445312500000000,  313.60840648020877097224001772701740264892578125000000,  310.23355146816442129420465789735317230224609375000000,  306.81397256049780253306380473077297210693359375000000,  303.35000742275894936028635129332542419433593750000000,  299.84196066014118287057499401271343231201171875000000,  296.29010486611946362245362251996994018554687500000000,  292.69468152354397716408129781484603881835937500000000,  289.05590176712530592340044677257537841796875000000000,  285.37394701465029811515705659985542297363281250000000,
        281.64896947280544736713636666536331176757812500000000,  277.88109252213524769103969447314739227294921875000000,  274.07041098439276538556441664695739746093750000000000,  270.21699127433083731375518254935741424560546875000000,  266.32087143679950713703874498605728149414062500000000,  262.38206106885075996615341864526271820068359375000000,  258.40054112537103492286405526101589202880859375000000,  254.37626360554619964204903226345777511596679687500000,  250.30915111619458457425935193896293640136718750000000,  246.19909630664355404405796434730291366577148437500000,
        242.04596116835665498001617379486560821533203125000000,  237.84957619090340585898957215249538421630859375000000,  233.60973936406338680171757005155086517333984375000000,  229.32621501383155759867804590612649917602539062500000,  224.99873245778579189391166437417268753051757812500000,  220.62698446262953666519024409353733062744140625000000,  216.21062548365677002948359586298465728759765625000000,  211.74926966230978564453835133463144302368164062500000,  207.24248855379599376647092867642641067504882812500000,  202.68980855176056365962722338736057281494140625000000,
        198.09070797109549744163814466446638107299804687500000,  193.44461374288150068423419725149869918823242187500000,  188.75089766692363468791882041841745376586914062500000,  184.00887215699458465678617358207702636718750000000000,  179.21778540127769474565866403281688690185546875000000,  174.37681584500290910000330768525600433349609375000000,  169.48506588311556697590276598930358886718750000000000,  164.54155462697241318892338313162326812744140625000000,  159.54520957916349743754835799336433410644531250000000,  154.49485701277285443211439996957778930664062500000000,
        149.38921080324340096012747380882501602172851562500000,  144.22685939910635966043628286570310592651367187500000,  139.00625053749124049318197648972272872924804687500000,  133.72567320497137188795022666454315185546875000000000,  128.38323620461153495853068307042121887207031250000000,  122.97684250264138938746327767148613929748535156250000,  117.50415827336183838269789703190326690673828125000000,  111.96257520950769048795336857438087463378906250000000,  106.34916417310925851325009716674685478210449218750000,  100.66061756049096231890871422365307807922363281250000,
         94.89317673616267256875289604067802429199218750000000,   89.04253937833415477598464349284768104553222656250000,   83.10373928043503610751940868794918060302734375000000,   77.07098756322643851035536499693989753723144531250000,   70.93745846855618708559632068499922752380371093750000,   64.69499324860622380128916120156645774841308593750000,   58.33367885174150302418638602830469608306884765625000,   51.84122735340913834534148918464779853820800781250000,   45.20202238210156053810351295396685600280761718750000,   38.39557410181857477482481044717133045196533203125000,
         31.39383862535166969109923229552805423736572265625000,   24.15611855653395068088684638496488332748413085937500,   16.61799764857290284680857439525425434112548828125000,    8.66181288102617941149219404906034469604492187500000,    0.00000000000000000000000000000000000000000000000000,  400.93094827891576414913288317620754241943359375000000,  402.59183365552564737299690023064613342285156250000000,  403.49872655667371645904495380818843841552734375000000,  403.94676273073645234035211615264415740966796875000000,  404.05919318354420965988538227975368499755859375000000,
        403.90473026382676380308112129569053649902343750000000,  403.52748074700355118693551048636436462402343750000000,  402.95826356717253702299785800278186798095703125000000,  402.21987441224052872712491080164909362792968750000000,  401.32987878275406501416000537574291229248046875000000,  400.30223621276689982551033608615398406982421875000000,  399.14831121428551341523416340351104736328125000000000,  397.87753641681052840795018710196018218994140625000000,  396.49786613588520367557066492736339569091796875000000,  395.01609737140495326457312330603599548339843750000000,
        393.43810352779507866216590628027915954589843750000000,  391.76900872194181602026219479739665985107421875000000,  390.01332048337712876673322170972824096679687500000000,  388.17503259509732060905662365257740020751953125000000,  386.25770604587251000339165329933166503906250000000000,  384.26453363480959524167701601982116699218750000000000,  382.19839216285743077605729922652244567871093750000000,  380.06188505887354267542832531034946441650390625000000,  377.85737753632275826021214015781879425048828125000000,  375.58702584714404792975983582437038421630859375000000,
        373.25280181977507254487136378884315490722656250000000,  370.85651359204177879291819408535957336425781250000000,  368.39982324564738291883259080350399017333984375000000,  365.88426189645571184883010573685169219970703125000000,  363.31124267932932525582145899534225463867187500000000,  360.68207197796829177605104632675647735595703125000000,  357.99795918194843125093029811978340148925781250000000,  355.26002519991430972368107177317142486572265625000000,  352.46930991598867421998875215649604797363281250000000,  349.62677874321894933018484152853488922119140625000000,
        346.73332840130967724689980968832969665527343750000000,  343.78979202448857677154592238366603851318359375000000,  340.79694368800647907846723683178424835205078125000000,  337.75550242761289609916275367140769958496093750000000,  334.66613581472199712152360007166862487792968750000000,  331.52946314038223363240831531584262847900390625000000,  328.34605825318294591852463781833648681640625000000000,  325.11645208956451824633404612541198730468750000000000,  321.84113492939496836697799153625965118408203125000000,  318.52055840494330141154932789504528045654296875000000,
        315.15513728735413678805343806743621826171875000000000,  311.74525107128738454775884747505187988281250000000000,  308.29124537542554662650218233466148376464843750000000,  304.79343317398661383776925504207611083984375000000000,  301.25209587214033035706961527466773986816406250000000,  297.66748423625585928675718605518341064453125000000000,  294.03981918815975404868368059396743774414062500000000,  290.36929247101119244689471088349819183349609375000000,  286.65606719297841209481703117489814758300781250000000,  282.90027825358811242040246725082397460937500000000000,
        279.10203265639830760846962220966815948486328125000000,  275.26140971048374694873928092420101165771484375000000,  271.37846112210831961419899016618728637695312500000000,  267.45321097685803124477388337254524230957031250000000,  263.48565561141339230744051747024059295654296875000000,  259.47576337301887861030991189181804656982421875000000,  255.42347426354729122977005317807197570800781250000000,  251.32869946383098636033537331968545913696289062500000,  247.19132073261235404970648232847452163696289062500000,  243.01118967303506224197917617857456207275390625000000,
        238.78812685800994586315937340259552001953125000000000,  234.52192080401920293297735042870044708251953125000000,  230.21232678091973866685293614864349365234375000000000,  225.85906544302230258836061693727970123291015625000000,  221.46182126409436818903486710041761398315429687500000,  217.02024075588388996038702316582202911376953125000000,  212.53393044620059981753001920878887176513671875000000,  208.00245458839904699743783567100763320922851562500000,  203.42533256915118045071722008287906646728515625000000,  198.80203597548862148869375232607126235961914062500000,
        194.13198527502112256115651689469814300537109375000000,  189.41454605471076888534298632293939590454101562500000,  184.64902475324026909220265224575996398925781250000000,  179.83466380940038220614951569586992263793945312500000,  174.97063613342734811340051237493753433227539062500000,  170.05603878907396619979408569633960723876953125000000,  165.08988575035678536551131401211023330688476562500000,  160.07109956703141051548300310969352722167968750000000,  154.99850173506499118047941010445356369018554687500000,  149.87080152023165169339335989207029342651367187500000,
        144.68658292105544660444138571619987487792968750000000,  139.44428937698251047550002112984657287597656250000000,  134.14220572230715333716943860054016113281250000000000,  128.77843674669000506582960952073335647583007812500000,  123.35088153566566404606419382616877555847167968750000,  117.85720250972293854374584043398499488830566406250000,  112.29478772915892648143199039623141288757324218750000,  106.66070453972874076953303301706910133361816406250000,  100.95164193271006070062867365777492523193359375000000,   95.16383797412071032795211067423224449157714843750000,
         89.29298714580305329491238808259367942810058593750000,   83.33412014274276202741020824760198593139648437500000,   77.28144508119378031096857739612460136413574218750000,   71.12813328864854156563524156808853149414062500000000,   64.86602318948986578561743954196572303771972656250000,   58.48519898742245004541473463177680969238281250000000,   51.97337009312223443657785537652671337127685546875000,   45.31491754710777541959032532759010791778564453125000,   38.48934899925911423679281142540276050567626953125000,   31.46861811929914054530854627955704927444458007812500,
         24.21202513637280517855288053397089242935180664062500,   16.65515149483403334329523204360157251358032226562500,    8.68033192879341797265624336432665586471557617187500,    0.00000000000000000000000000000000000000000000000000,  405.62229616114490227118949405848979949951171875000000,  407.28492287130353588509024120867252349853515625000000,  408.19377196938921770197339355945587158203125000000000,  408.64395480121305581633350811898708343505859375000000,  408.75870634989956897697993554174900054931640625000000,  408.60672794158637088912655599415302276611328125000000,
        408.23211853261392434433219023048877716064453125000000,  407.66569141014934984923456795513629913330078125000000,  406.93023815799449494079453870654106140136718750000000,  406.04332131198918887093896046280860900878906250000000,  405.01889831157905064173974096775054931640625000000000,  403.86833225837040117767173796892166137695312500000000,  402.60105492372690605407115072011947631835937500000000,  401.22502022146687750137061811983585357666015625000000,  399.74702513519139301934046670794486999511718750000000,  398.17294338507218753875349648296833038330078125000000,
        396.50789969512771904192049987614154815673828125000000,  394.75640246217108142445795238018035888671875000000000,  392.92244657249261763354297727346420288085937500000000,  391.00959433548587185214273631572723388671875000000000,  389.02104007378699179753311909735202789306640625000000,  386.95966230374193628449575044214725494384765625000000,  384.82806635316796928236726671457290649414062500000000,  382.62861951199232635190128348767757415771484375000000,  380.36348028193390291562536731362342834472656250000000,  378.03462291193454802851192653179168701171875000000000,
        375.64385812983090318084578029811382293701171875000000,  373.19285077683730378339532762765884399414062500000000,  370.68313489890761047718115150928497314453125000000000,  368.11612673364231795858358964323997497558593750000000,  365.49313594311723818464088253676891326904296875000000,  362.81537537478226340681430883705615997314453125000000,  360.08396957935377713511115871369838714599609375000000,  357.29996227274335751644684933125972747802734375000000,  354.46432289583839292390621267259120941162109375000000,  351.57795239938690201597637496888637542724609375000000,
        348.64168835985162786528235301375389099121093750000000,  345.65630951475708343423320911824703216552734375000000,  342.62253979190421659950516186654567718505859375000000,  339.54105189521266083829686976969242095947265625000000,  336.41247050035360643960302695631980895996093750000000,  333.23737510536858508203295059502124786376953125000000,  330.01630257481139096853439696133136749267578125000000,  326.74974941035821984769427217543125152587890625000000,  323.43817377610821495181880891323089599609375000000000,  320.08199730278414563144906423985958099365234375000000,
        316.68160669161562736917403526604175567626953125000000,  313.23735513574155220339889638125896453857421875000000,  309.74956357441789123186026699841022491455078125000000,  306.21852179309945540808257646858692169189453125000000,  302.64448938051270943105919286608695983886718750000000,  299.02769655211091048840899020433425903320312500000000,  295.36834484776068165956530719995498657226562500000000,  291.66660771011390806961571797728538513183593750000000,  287.92263094884253860072931274771690368652343750000000,  284.13653309473266972418059594929218292236328125000000,
        280.30840564651879276425461284816265106201171875000000,  276.43831321227622765945852734148502349853515625000000,  272.52629354615186230148538015782833099365234375000000,  268.57235748018570120621006935834884643554687500000000,  264.57648874994026755302911624312400817871093750000000,  260.53864371158738322264980524778366088867187500000000,  256.45875094698726570641156286001205444335937500000000,  252.33671075210932599475199822336435317993164062500000,  248.17239450286112401045102160423994064331054687500000,  243.96564389098898573138285428285598754882812500000000,
        239.71627002115630489242903422564268112182617187500000,  235.42405235855736123085080180317163467407226562500000,  231.08873751444309618818806484341621398925781250000000,  226.71003785467036095724324695765972137451171875000000,  222.28762991377283242400153540074825286865234375000000,  217.82115259401729190358310006558895111083984375000000,  213.31020512535911848317482508718967437744140625000000,  208.75434475803297118545742705464363098144531250000000,  204.15308415456561874634644482284784317016601562500000,  199.50588844210182060123770497739315032958984375000000,
        194.81217187886721831091563217341899871826171875000000,  190.07129408007381243805866688489913940429687500000000,  185.28255573823807367261906620115041732788085937500000,  180.44519376027506041282322257757186889648437500000000,  175.55837572824316339392680674791336059570312500000000,  170.62119357147096820881415624171495437622070312500000,  165.63265631396225785465503577142953872680664062500000,  160.59168173108716359820391517132520675659179687500000,  155.49708671178927943401504307985305786132812500000000,  150.34757607439811977201316040009260177612304687500000,
        145.14172952223788115588831715285778045654296875000000,  139.87798634488345328463765326887369155883789062500000,  134.55462736555998048970650415867567062377929687500000,  129.16975349549838369966892059892416000366210937500000,  123.72126006861975611172965727746486663818359375000000,  118.20680587510945258600258966907858848571777343750000,  112.62377546106198167308320989832282066345214843750000,  106.96923276919906697912665549665689468383789062500000,  101.23986349426274955476401373744010925292968750000000,   95.43190250780502026373142143711447715759277343750000,
         89.54104119507309178516152314841747283935546875000000,   83.56230724834352940888493321835994720458984375000000,   77.48990587126409934626281028613448143005371093750000,   71.31700556523017553445242810994386672973632812500000,   65.03544201162098659096955088898539543151855468750000,   58.63529675098297389013168867677450180053710937500000,   52.10427660758891477144061354920268058776855468750000,   45.42676010936752817315209540538489818572998046875000,   38.58225246462089330634626094251871109008789062500000,   31.54270496983637883658957434818148612976074218750000,
         24.26741554672135592340964649338275194168090820312500,   16.69196339556320651809073751792311668395996093750000,    8.69868106746161373621362145058810710906982421875000,    0.00000000000000000000000000000000000000000000000000,  410.32277652693727532096090726554393768310546875000000,  411.98712575448780626174993813037872314453125000000000,  412.89790916587895708289579488337039947509765625000000,  413.35021399720636736674350686371326446533203125000000,  413.46725938594596527764224447309970855712890625000000,  413.31773574530564019369194284081459045410156250000000,
        412.94573427958107458835002034902572631835937500000000,  412.38206266720214898668928071856498718261718750000000,  411.64950840541649768056231550872325897216796875000000,  410.76563106739729391847504302859306335449218750000000,  409.74438598785820886405417695641517639160156250000000,  408.59713483796895161503925919532775878906250000000000,  407.33330850256083977001253515481948852539062500000000,  405.96086045805890307747176848351955413818359375000000,  404.48658762959632895217509940266609191894531250000000,  402.91636400495286807199590839445590972900390625000000,
        401.25531486155171023710863664746284484863281250000000,  399.50794940442131064628483727574348449707031250000000,  397.67826355889764045059564523398876190185546875000000,  395.76982088564670902997022494673728942871093750000000,  393.78581715639876392742735333740711212158203125000000,  391.72913252334291200895677320659160614013671875000000,  389.60237412850557348065194673836231231689453125000000,  387.40791124821129187694168649613857269287109375000000,  385.14790453842431361408671364188194274902343750000000,  382.82433056739768062470830045640468597412109375000000,
        380.43900254590710119373397901654243469238281250000000,  377.99358796147339489834848791360855102539062500000000,  375.48962367051746014112723059952259063720703125000000,  372.92852888701742131161154247820377349853515625000000,  370.31161641797490347016719169914722442626953125000000,  367.64010242778635983995627611875534057617187500000000,  364.91511496040914153127232566475868225097656250000000,  362.13770140634466088158660568296909332275390625000000,  359.30883506824460482675931416451930999755859375000000,  356.42942095239556010710657574236392974853515625000000,
        353.50030089195723803641158156096935272216796875000000,  350.52225809049917870652279816567897796630859375000000,  347.49602116024055931120528839528560638427734375000000,  344.42226771778786087452317588031291961669921875000000,  341.30162759058214305696310475468635559082031250000000,  338.13468567930317476566415280103683471679687500000000,  334.92198451483250210003461688756942749023437500000000,  331.66402654279283979121828451752662658691406250000000,  328.36127616396845496637979522347450256347656250000000,  325.01416155491085646644933149218559265136718750000000,
        321.62307628961838190662092529237270355224609375000000,  318.18838078024469950833008624613285064697265625000000,  314.71040355225858320409315638244152069091796875000000,  311.18944236727253382923663593828678131103515625000000,  307.62576520482713249293738044798374176025390625000000,  304.01961111271378968012868426740169525146484375000000,  300.37119093389770796420634724199771881103515625000000,  296.68068791673533723951550200581550598144531250000000,  292.94825821393760634236969053745269775390625000000000,  289.17403127457828304613940417766571044921875000000000,
        285.35811013237656652563600800931453704833984375000000,  281.50057159246335913849179632961750030517578125000000,  277.60146631785448789742076769471168518066406250000000,  273.66081881589082058781059458851814270019531250000000,  269.67862732393695068822125904262065887451171875000000,  265.65486359264497195908916182816028594970703125000000,  261.58947256407265058442135341465473175048828125000000,  257.48237194086800627701450139284133911132812500000000,  253.33345164158185980340931564569473266601562500000000,  249.14257313591988918233255390077829360961914062500000,
        244.90956865236785233719274401664733886718750000000000,  240.63424024909082277190464083105325698852539062500000,  236.31635873727981334013747982680797576904296875000000,  231.95566244415792311883706133812665939331054687500000,  227.55185580060663141921395435929298400878906250000000,  223.10460773577790405397536233067512512207031250000000,  218.61354985803274075806257314980030059814453125000000,  214.07827439801172886291169561445713043212890625000000,  209.49833188547333406859252136200666427612304687500000,  204.87322852659659133678360376507043838500976562500000,
        200.20242324255789867493149358779191970825195312500000,  195.48532432313112394695053808391094207763671875000000,  190.72128564054798971483251079916954040527343750000000,  185.90960235852847404203203041106462478637695312500000,  181.04950605878687497352075297385454177856445312500000,  176.14015919183702862937934696674346923828125000000000,  171.18064873978124751374707557260990142822265625000000,  166.16997895493523174081929028034210205078125000000000,  161.10706300825791004172060638666152954101562500000000,  155.99071334377896391742979176342487335205078125000000,
        150.81963048708067276493238750845193862915039062500000,  145.59238999399374847598664928227663040161132812500000,  140.30742714533002413190843071788549423217773437500000,  134.96301888812223523927968926727771759033203125000000,  129.55726238415900297695770859718322753906250000000000,  124.08804933916576374031137675046920776367187500000000,  118.55303503117227137408917769789695739746093750000000,  112.94960060522812739236542256549000740051269531250000,  107.27480670945153917728021042421460151672363281250000,  101.52533584599680693827394861727952957153320312500000,
         95.69741979164680856229097116738557815551757812500000,   89.78674693071555168444319861009716987609863281250000,   83.78834204460105183898122049868106842041015625000000,   77.69640751253515986718412023037672042846679687500000,   71.50410909454595298484491650015115737915039062500000,   65.20327981044752618799975607544183731079101562500000,   58.78399861593461395159465610049664974212646484375000,   52.23396982420134548874557367525994777679443359375000,   45.53756952307085725806246045976877212524414062500000,   38.67430054904463077036780305206775665283203125000000,
         31.61611189257115483997040428221225738525390625000000,   24.32229923265497362194764718879014253616333007812500,   16.72843958790243590328827849589288234710693359375000,    8.71686338654480508125743654090911149978637695312500,    0.00000000000000000000000000000000000000000000000000,  415.03230672824963676248444244265556335449218750000000,  416.69836003292857640190050005912780761718750000000000,  417.61105632266799148055724799633026123046875000000000,  418.06545901096734496604767628014087677001953125000000,  418.18477156362729374450282193720340728759765625000000,
        418.03767358903235162870259955525398254394531250000000,  417.66824860588087631185771897435188293457031250000000,  417.10729872213880753406556323170661926269531250000000,  416.37760736662045246703200973570346832275390625000000,  415.49673115282462276809383183717727661132812500000000,  414.47862330187507495793397538363933563232421875000000,  413.33464403600629566426505334675312042236328125000000,  412.07422332690055100101744756102561950683593750000000,  410.70531417992447131837252527475357055664062500000000,  409.23471342178629583941074088215827941894531250000000,
        407.66829526215929035970475524663925170898437500000000,  406.01118548077317882416537031531333923339843750000000,  404.26789403464249517128337174654006958007812500000000,  402.44241782694831499611609615385532379150390625000000,  400.53832160354284042114159092307090759277343750000000,  398.55880251430454563887906260788440704345703125000000,  396.50674227143684902330278418958187103271484375000000,  394.38474975039588343861396424472332000732421875000000,  392.19519612806885788813815452158451080322265625000000,  389.94024412363540932346950285136699676513671875000000,
        387.62187252826362282576155848801136016845703125000000,  385.24189693370163922736537642776966094970703125000000,  382.80198736600721076683839783072471618652343750000000,  380.30368337823210822534747421741485595703125000000000,  377.74840704053571016629575751721858978271484375000000,  375.13747417796230365638621151447296142578125000000000,  372.47210413792697636381490156054496765136718750000000,  369.75342831626221595797687768936157226562500000000000,  366.98249762882551294751465320587158203125000000000000,  364.16028908246249784497194923460483551025390625000000,
        361.28771157257938284601550549268722534179687500000000,  358.36561101320978650619508698582649230957031250000000,  355.39477488813525951627525500953197479248046875000000,  352.37593629749034107589977793395519256591796875000000,  349.30977756267913036936079151928424835205078125000000,  346.19693344285514058356056921184062957763671875000000,  343.03799400826017063081962987780570983886718750000000,  339.83350720908055109248380176723003387451171875000000,  336.58398117290352047348278574645519256591796875000000,  333.28988625915224019991001114249229431152343750000000,
        329.95165689488868565604207105934619903564453125000000,  326.56969321296674024779349565505981445312500000000000,  323.14436251059942151186987757682800292968750000000000,  319.67600054388140051742084324359893798828125000000000,  316.16491267161933365059667266905307769775390625000000,  312.61137485991014273167820647358894348144531250000000,  309.01563455721668560727266594767570495605468750000000,  305.37791144819635746898711659014225006103515625000000,  301.69839809319000778486952185630798339843750000000000,  297.97726045906227909654262475669384002685546875000000,
        294.21463834596727338066557422280311584472656250000000,  290.41064571357253498717909678816795349121093750000000,  286.56537090929759870050475001335144042968750000000000,  282.67887680018367291268077678978443145751953125000000,  278.75120080909761099974275566637516021728515625000000,  274.78235485506701252234051935374736785888671875000000,  270.77232519663192533698747865855693817138671875000000,  266.72107217615604213278857059776782989501953125000000,  262.62852986206524974477360956370830535888671875000000,  258.49460558493836970228585414588451385498046875000000,
        254.31917936225536891470255795866250991821289062500000,  250.10210320538638484322291333228349685668945312500000,  245.84320030104890975053422152996063232421875000000000,  241.54226405795063215009577106684446334838867187500000,  237.19905700762589617625053506344556808471679687500000,  232.81330954652813147731649223715066909790039062500000,  228.38471850420572195616841781884431838989257812500000,  223.91294551980445248773321509361267089843750000000000,  219.39761520612873368918371852487325668334960937500000,  214.83831307696627277437073644250631332397460937500000,
        210.23458320922324560342531185597181320190429687500000,  205.58592560648395419775624759495258331298828125000000,  200.89179322473026445550203789025545120239257812500000,  196.15158861390253264289640355855226516723632812500000,  191.36466012047708318277727812528610229492187500000000,  186.53029758591341646933869924396276473999023437500000,  181.64772746322532270824012812227010726928710937500000,  176.71610725845246747667260933667421340942382812500000,  171.73451918467330301609763409942388534545898437500000,  166.70196289237227915691619273275136947631835937500000,
        161.61734711009393095082486979663372039794921875000000,  156.47947999154317244574485812336206436157226562500000,  151.28705791715722739354532677680253982543945312500000,  146.03865243628155212718411348760128021240234375000000,  140.73269495574513143765216227620840072631835937500000,  135.36745867528304643201408907771110534667968750000000,  129.94103713057018012477783486247062683105468750000000,  124.45131851719821725055226124823093414306640625000000,  118.89595471411600158262444892898201942443847656250000,  113.27232357367810777759586926549673080444335937500000,
        107.57748255327011577264784136787056922912597656250000,  101.80811106208086869173712329939007759094238281250000,   95.96043787771421307297714520245790481567382812500000,   90.03014847531213149522955063730478286743164062500000,   84.01226481351733355040778405964374542236328125000000,   77.90098653181075860629789531230926513671875000000000,   71.68947673022908873008418595418334007263183593750000,   65.36956584526718927463662112131714820861816406250000,   58.93133032303405371976623428054153919219970703125000,   52.36247203807272398989880457520484924316406250000000,
         45.64736470782330002293747384101152420043945312500000,   38.76550886413011198783351574093103408813476562500000,   31.68885125608607111757919483352452516555786132812500,   24.37668538232852455394095159135758876800537109375000,   16.76458613987696466551824414636939764022827148437500,    8.73488189204748266547539969906210899353027343750000,    0.00000000000000000000000000000000000000000000000000,  419.75080559954471937089692801237106323242187500000000,  421.41854490611882511075236834585666656494140625000000,  422.33313307467142294626682996749877929687500000000000,
        422.78960997764517060204525478184223175048828125000000,  422.91116358012175169278634712100028991699218750000000,  422.76646279223024293969501741230487823486328125000000,  422.39958351291875260358210653066635131835937500000000,  421.84132231800680301603279076516628265380859375000000,  421.11445858651262597049935720860958099365234375000000,  420.23654597613824535073945298790931701660156250000000,  419.22153558674744999734684824943542480468750000000000,  418.08078617458738790446659550070762634277343750000000,  416.82372677325327003927668556571006774902343750000000,
        415.45830988528052785113686695694923400878906250000000,  413.99133220109007424980518408119678497314453125000000,  412.42866810882651407155208289623260498046875000000000,  410.77544384191475046463892795145511627197265625000000,  409.03617005580946397458319552242755889892578125000000,  407.21484457321821537334471940994262695312500000000000,  405.31503326217989524593576788902282714843750000000000,  403.33993458307435275855823419988155364990234375000000,  401.29243173580960046820109710097312927246093750000000,  399.17513525224319437256781384348869323730468750000000,
        396.99041812798412820484372787177562713623046875000000,  394.74044505864941356776398606598377227783203125000000,  392.42719696644854820988257415592670440673828125000000,  390.05249172694834669528063386678695678710937500000000,  387.61800180209422705956967547535896301269531250000000,  385.12526933318144983786623924970626831054687500000000,  382.57571913215286940612713806331157684326171875000000,  379.97066992138741170492721721529960632324218750000000,  377.31134410396788325670058839023113250732421875000000,  374.59887629324521185480989515781402587890625000000000,
        371.83432078867275549782789312303066253662109375000000,  369.01865815169247753146919421851634979248046875000000,  366.15280100892402970202965661883354187011718750000000,  363.23759918854506167917861603200435638427734375000000,  360.27384427843708181171678006649017333984375000000000,  357.26227368054827593368827365338802337646484375000000,  354.20357422432743987883441150188446044921875000000000,  351.09838539251734346180455759167671203613281250000000,  347.94730220464595049634226597845554351806640625000000,  344.75087779692347567106480710208415985107421875000000,
        341.50962573168652625099639408290386199951171875000000,  338.22402206483479858434293419122695922851562500000000,  334.89450719572448633698513731360435485839843750000000,  331.52148752058667469100328162312507629394531250000000,  328.10533690762969172283192165195941925048828125000000,  324.64639800947571757205878384411334991455078125000000,  321.14498342640399641823023557662963867187500000000000,  317.60137673197641561273485422134399414062500000000000,  314.01583337094825765234418213367462158203125000000000,  310.38858143788763754855608567595481872558593750000000,
        306.71982234360410757290082983672618865966796875000000,  303.00973137529183532024035230278968811035156250000000,  299.25845815520153792022028937935829162597656250000000,  295.46612700164735088037559762597084045410156250000000,  291.63283719521132297813892364501953125000000000000000,  287.75866315210782886424567550420761108398437500000000,  283.84365450580366996291559189558029174804687500000000,  279.88783609713675559760304167866706848144531250000000,  275.89120787232138809486059471964836120605468750000000,  271.85374468736307562721776776015758514404296875000000,
        267.77539601650556733147823251783847808837890625000000,  263.65608556138892026865505613386631011962890625000000,  259.49571075658923291484825313091278076171875000000000,  255.29414216611542087775887921452522277832031250000000,  251.05122276424151550600072368979454040527343750000000,  246.76676709271751519736426416784524917602539062500000,  242.44056028491189636042690835893154144287109375000000,  238.07235694574342232954222708940505981445312500000000,  233.66187987433130501813138835132122039794921875000000,  229.20881861406962798355380073189735412597656250000000,
        224.71282781225949065628810785710811614990234375000000,  220.17352536843137045252660755068063735961914062500000,  215.59049034697332558607740793377161026000976562500000,  210.96326062552890334700350649654865264892578125000000,  206.29133024570523957663681358098983764648437500000000,  201.57414642675891514045360963791608810424804687500000,  196.81110619587860810497659258544445037841796875000000,  192.00155258018449444534780923277139663696289062500000,  187.14477029524485374167852569371461868286132812500000,  182.23998085231767163350014016032218933105468750000000,
        177.28633699105029108977760188281536102294921875000000,  172.28291632523837506596464663743972778320312500000000,  167.22871406542108729809115175157785415649414062500000,  162.12263465221079172806639689952135086059570312500000,  156.96348209648726879095192998647689819335937500000000,  151.74994877445229235490842256695032119750976562500000,  146.48060236365137143366155214607715606689453125000000,  141.15387052573410642253293190151453018188476562500000,  135.76802283637849200204072985798120498657226562500000,  130.32114932312276778247905895113945007324218750000000,
        124.81113478441795905382605269551277160644531250000000,  119.23562780840310892926936503499746322631835937500000,  113.59200305853269696854113135486841201782226562500000,  107.87731490100861719838576391339302062988281250000000,  102.08823974757359565046499483287334442138671875000000,   96.22100346834530171236110618337988853454589843750000,   90.27128871735092729977623093873262405395507812500000,   84.23411471507910164291388355195522308349609375000000,   78.10367844257402225593978073447942733764648437500000,   71.87314041806722286764852469787001609802246093750000,
         65.53432856994268718153762165457010269165039062500000,   59.07731690709534433381122653372585773468017578125000,   52.48980493508563682780732051469385623931884765625000,   45.75616406806170743948314338922500610351562500000000,   38.85589259784338622694122022949159145355224609375000,   31.76093509445448503925035765860229730606079101562500,   24.43058293621172794019003049470484256744384765625000,   16.80040895645484155807025672402232885360717773437500,    8.75273950944748868607803160557523369789123535156250,    0.00000000000000000000000000000000000000000000000000,
        424.47819341825709216209361329674720764160156250000000,  426.14760100606827109004370868206024169921875000000000,  427.06406047657668523243046365678310394287109375000000,  427.52258843727310022586607374250888824462890625000000,  427.64635752052441830528550781309604644775390625000000,  427.50402604325233824056340381503105163574218750000000,  427.13966234988993164733983576297760009765625000000000,  426.58405752243845654447795823216438293457031250000000,  425.85998690922531295655062422156333923339843750000000,  424.98500121685606245591770857572555541992187500000000,
        423.97304941737786521116504445672035217285156250000000,  422.83548878536055326549103483557701110839843750000000,  421.58174739295850486087147146463394165039062500000000,  420.21977720989110594018711708486080169677734375000000,  418.75637475441118340313551016151905059814453125000000,  417.19741455188068357529118657112121582031250000000000,  415.54802324316233352874405682086944580078125000000000,  413.81271213114325746573740616440773010253906250000000,  411.99547990246628614841029047966003417968750000000000,  410.09989348727782498826854862272739410400390625000000,
        408.12915259193209749355446547269821166992187500000000,  406.08614183505966366283246316015720367431640625000000,  403.97347333143551395551185123622417449951171875000000,  401.79352181732355120402644388377666473388671875000000,  399.54845388201340483647072687745094299316406250000000,  397.24025249115214819539687596261501312255859375000000,  394.87073771151057144379592500627040863037109375000000,  392.44158434309946414941805414855480194091796875000000,  389.95433701220304101298097521066665649414062500000000,  387.41042316361102848532027564942836761474609375000000,
        384.81116430213842249941080808639526367187500000000000,  382.15778576536962418686016462743282318115234375000000,  379.45142525640335406933445483446121215820312500000000,  376.69314032354577648220583796501159667968750000000000,  373.88391494071942133814445696771144866943359375000000,  371.02466531583098685587174259126186370849609375000000,  368.11624503299088928542914800345897674560546875000000,  365.15944961716854777478147298097610473632812500000000,  362.15502059575044313532998785376548767089843750000000,  359.10364911988335734349675476551055908203125000000000,
        356.00597919891765741340350359678268432617187500000000,  352.86261059333048706321278586983680725097656250000000,  349.67410140487947955989511683583259582519531250000000,  346.44097039718155883747385814785957336425781250000000,  343.16369907522124549359432421624660491943359375000000,  339.84273354832015456850058399140834808349609375000000,  336.47848619771389166999142616987228393554687500000000,  333.07133716697984482379979453980922698974609375000000,  329.62163569106400018426938913762569427490234375000000,  326.12970127748758386587724089622497558593750000000000,
        322.59582475143088231561705470085144042968750000000000,  319.02026917473159528526593931019306182861328125000000,  315.40327064737692808193969540297985076904296875000000,  311.74503899875691104170982725918292999267578125000000,  308.04575837477801769637153483927249908447265625000000,  304.30558772586624627365381456911563873291015625000000,  300.52466119990674542350461706519126892089843750000000,  296.70308844325438712985487654805183410644531250000000,  292.84095481208652245186385698616504669189453125000000,  288.93832149553799126806552521884441375732421875000000,
        284.99522555125440703704953193664550781250000000000000,  281.01167985319801800869754515588283538818359375000000,  276.98767295073264449456473812460899353027343750000000,  272.92316883718848430362413637340068817138671875000000,  268.81810662524264898820547387003898620605468750000000,  264.67240012553895667224423959851264953613281250000000,  260.48593732398762767843436449766159057617187500000000,  256.25857975211670236603822559118270874023437500000000,  251.99016174366786913196847308427095413208007812500000,  247.68048956931698967309785075485706329345703125000000,
        243.32934043992040074044780340045690536499023437500000,  238.93646136701292448378808330744504928588867187500000,  234.50156786736440039931039791554212570190429687500000,  230.02434249619102502038003876805305480957031250000000,  225.50443319105542627767135854810476303100585937500000,  220.94145140549895245385414455085992813110351562500000,  216.33497000793869347035069949924945831298828125000000,  211.68452091721937335933034773916006088256835937500000,  206.98959244129312651239160913974046707153320312500000,  202.24962627963228101179993245750665664672851562500000,
        197.46401414293831066970597021281719207763671875000000,  192.63209393521535162108193617314100265502929687500000,  187.75314543296124725202389527112245559692382812500000,  182.82638538364128066859848331660032272338867187500000,  177.85096193013774268365523312240839004516601562500000,  172.82594824873976335766201373189687728881835937500000,  167.75033526441831099873525090515613555908203125000000,  162.62302327725234363242634572088718414306640625000000,  157.44281229610882633096480276435613632202148437500000,  152.20839082754594073776388540863990783691406250000000,
        146.91832280602253035795001778751611709594726562500000,  141.57103227116292032405908685177564620971679687500000,  136.16478529248149698105407878756523132324218750000000,  130.69766850229251531345653347671031951904296875000000,  125.16756341009984510037611471489071846008300781250000,  119.57211541695284040542901493608951568603515625000000,  113.90869609688263608404668048024177551269531250000000,  108.17435682035706179249245906248688697814941406250000,  102.36577109329820700622803997248411178588867187500000,   96.47916196632979790592798963189125061035156250000000,
         90.51020935690259250350209185853600502014160156250000,   84.45392982860454367255442775785923004150390625000000,   78.30451778217063463216618401929736137390136718750000,   72.05513122917871271511103259399533271789550781250000,   65.69759566221854640843957895413041114807128906250000,   59.22198272258621898345154477283358573913574218750000,   52.61598961389913853281541378237307071685791015625000,   45.86398551159616232553162262775003910064697265625000,   38.94546652971069988780072890222072601318359375000000,   31.83237511919714179953189159277826547622680664062500,
         24.48400059591320854224250069819390773773193359375000,   16.83591378533750315682482323609292507171630859375000,    8.77043908654689019499528512824326753616333007812500,    0.00000000000000000000000000000000000000000000000000,  429.21439186665151055422029457986354827880859375000000,  430.88545035955235107394400984048843383789062500000000,  431.80376096557671417031087912619113922119140625000000,  432.26431729807580950364354066550731658935546875000000,  432.39027682181784939530189149081707000732421875000000,  432.25028736406363805144792422652244567871093750000000,
        431.88840977934495413137483410537242889404296875000000,  431.33542969415373136143898591399192810058593750000000,  430.61411844565373030491173267364501953125000000000000,  429.74202379481846492126351222395896911621093750000000,  428.73309258036596247620764188468456268310546875000000,  427.59868058078228614249383099377155303955078125000000,  426.34821488492201524422853253781795501708984375000000,  424.98964690139422373249544762074947357177734375000000,  423.52977294227736138054751791059970855712890625000000,  421.97446763094944799377117305994033813476562500000000,
        420.32885797168142971713677980005741119384765625000000,  418.59745586618964807712472975254058837890625000000000,  416.78426081204230513321817852556705474853515625000000,  414.89284074417844294657697901129722595214843750000000,  412.92639655335159432070213370025157928466796875000000,  410.88781421108058111713035032153129577636718750000000,  408.77970734492663495984743349254131317138671875000000,  406.60445235730765034531941637396812438964843750000000,  404.36421765221416535496246069669723510742187500000000,  402.06098815515787237018230371177196502685546875000000,
        399.69658603578363909036852419376373291015625000000000,  397.27268833889866073150187730789184570312500000000000,  394.79084207736156031387508846819400787353515625000000,  392.25247722501643465875531546771526336669921875000000,  389.65891795968690303197945468127727508544921875000000,  387.01139243811246615223353728652000427246093750000000,  384.31104133156230773238348774611949920654296875000000,  381.55892530904691284376895055174827575683593750000000,  378.75603162187780981184914708137512207031250000000000,  375.90327991681067487661493942141532897949218750000000,
        373.00152738366455196228343993425369262695312500000000,  370.05157332600828112845192663371562957763671875000000,  367.05416322939731799124274402856826782226562500000000,  364.00999239006137031537946313619613647460937500000000,  360.91970915738750136370072141289710998535156250000000,  357.78391783561073680175468325614929199218750000000000,  354.60318128350303368279128335416316986083984375000000,  351.37802324530122177748125977814197540283203125000000,  348.10893044143233510112622752785682678222656250000000,  344.79635444362907037429977208375930786132812500000000,
        341.44071335564927949235425330698490142822265625000000,  338.04239331792194889203528873622417449951171875000000,  334.60174985195368435597629286348819732666015625000000,  331.11910905817507000392652116715908050537109375000000,  327.59476867903140373528003692626953125000000000000000,  324.02899903748021870342199690639972686767578125000000,  320.42204385960781110043171793222427368164062500000000,  316.77412098878807000801316462457180023193359375000000,  313.08542299765207417294732294976711273193359375000000,  309.35611770308941004259395413100719451904296875000000,
        305.58634858854429694474674761295318603515625000000000,  301.77623513698313217901159077882766723632812500000000,  297.92587307707441368620493449270725250244140625000000,  294.03533454433221550061716698110103607177734375000000,  290.10466815820291230920702219009399414062500000000000,  286.13389901532275416684569790959358215332031250000000,  282.12302859841878444058238528668880462646484375000000,  278.07203459955849211837630718946456909179687500000000,  273.98087065566170394959044642746448516845703125000000,  269.84946599335501105088042095303535461425781250000000,
        265.67772497936380204919259995222091674804687500000000,  261.46552657167660527193220332264900207519531250000000,  257.21272366567131939518731087446212768554687500000000,  252.91914232822873032091592904180288314819335937500000,  248.58458091156646219133108388632535934448242187500000,  244.20880903705969444672518875449895858764648437500000,  239.79156643765287526548490859568119049072265625000000,  235.33256164556041767355054616928100585937500000000000,  230.83147050975298952835146337747573852539062500000000,  226.28793452517351170172332786023616790771484375000000,
        221.70155895264429091184865683317184448242187500000000,  217.07191070492362428012711461633443832397460937500000,  212.39851597023462659308279398828744888305664062500000,  207.68085753967673667830240447074174880981445312500000,  202.91837179906960386688297148793935775756835937500000,  198.11044533874033390929980669170618057250976562500000,  193.25641112627570805670984555035829544067382812500000,  188.35554417694939388638886157423257827758789062500000,  183.40705664394866403199557680636644363403320312500000,  178.41009223505918157570704352110624313354492187500000,
        173.36371984333723617055511567741632461547851562500000,  168.26692625548463411178090609610080718994140625000000,  163.11860777176337933269678615033626556396484375000000,  157.91756053352739286310679744929075241088867187500000,  152.66246930631589862059627193957567214965820312500000,  147.35189440456980491944705136120319366455078125000000,  141.98425636369930202818068210035562515258789062500000,  136.55781785988858700875425711274147033691406250000000,  131.07066223834502238787536043673753738403320312500000,  125.52066782327158023235824657604098320007324218750000,
        119.90547692803357904267613776028156280517578125000000,  114.22245813261970681651291670277714729309082031250000,  108.46865990332288731679000193253159523010253906250000,  102.64075292817278750590048730373382568359375000000000,   96.73495752277551673614652827382087707519531250000000,   90.74695094919901805496920133009552955627441406250000,   84.67174719220014367238036356866359710693359375000000,   78.50353814730011947631282964721322059631347656250000,   72.23547939168560105827054940164089202880859375000000,   65.85939405171487237566907424479722976684570312500000,
         59.36535146807393203971514594741165637969970703125000,   52.74104660697059898666338995099067687988281250000000,   45.97084646732484003450736054219305515289306640625000,   39.03424504533712280363033642061054706573486328125000,   31.90318273070921861744864145293831825256347656250000,   24.53694683261503328708386106882244348526000976562500,   16.87110622249553770757302117999643087387084960937500,    8.78798339619780044529306906042620539665222167968750,    0.00000000000000000000000000000000000000000000000000,  433.95932399501481313563999719917774200439453125000000,
        435.63201635167536096560070291161537170410156250000000,  436.55215832539551001900690607726573944091796875000000,  437.01472080104252881938009522855281829833984375000000,  437.14284623807611751544754952192306518554687500000000,  437.00517207616132964176358655095100402832031250000000,  436.64575174391205791835091076791286468505859375000000,  436.09536545057312650897074490785598754882812500000000,  435.37678054205088074013474397361278533935546875000000,  434.50754183986157386243576183915138244628906250000000,  433.50159404486043968063313513994216918945312500000000,
        432.37029142625158328883117064833641052246093750000000,  431.12306006914354838954750448465347290039062500000000,  429.76785079433926739511662162840366363525390625000000,  428.31145967551054809518973343074321746826171875000000,  426.75976139679420384709374047815799713134765625000000,  425.11788328395033431661431677639484405517578125000000,  423.39033779119750988684245385229587554931640625000000,  421.58112517646554806560743600130081176757812500000000,  419.69381432478854776491061784327030181884765625000000,  417.73160725253950431579141877591609954833984375000000,
        415.69739122127089103742036968469619750976562500000000,  413.59378130437005438579944893717765808105468750000000,  411.42315549928133577850530855357646942138671875000000,  409.18768394931015563997789286077022552490234375000000,  406.88935346006934423712664283812046051025390625000000,  404.52998821979412014115951023995876312255859375000000,  402.11126742911716291928314603865146636962890625000000,  399.63474039362267831165809184312820434570312500000000,  397.10183951726747864086064510047435760498046875000000,  394.51389154661137581570073962211608886718750000000000,
        391.87212734768257860196172259747982025146484375000000,  389.17769044417121904189116321504116058349609375000000,  386.43164450384489327916526235640048980712890625000000,  383.63497992691435456436011008918285369873046875000000,  380.78861966357754909040522761642932891845703125000000,  377.89342436663184798817383125424385070800781250000000,  374.95019696775034390157088637351989746093750000000000,  371.95968675191710417493595741689205169677734375000000,  368.92259299293982621748000383377075195312500000000000,  365.83956820340836202376522123813629150390625000000000,
        362.71122104453962720072013325989246368408203125000000,  359.53811893473636018825345672667026519775390625000000,  356.32079039014172394672641530632972717285156250000000,  353.05972712579563221879652701318264007568359375000000,  349.75538594204004994026036001741886138916015625000000,  346.40819041745038475710316561162471771240234375000000,  343.01853242668335042253602296113967895507812500000000,  339.58677349915490140119800344109535217285156250000000,  336.11324603231486207732814364135265350341796875000000,  332.59825437142103510268498212099075317382812500000000,
        329.04207576608388308159192092716693878173828125000000,  325.44496121241792252476443536579608917236328125000000,  321.80713618835881106861052103340625762939453125000000,  318.12880128856807004922302439808845520019531250000000,  314.41013276431812073496985249221324920654296875000000,  310.65128297281250979722244665026664733886718750000000,  306.85238073953286175310495309531688690185546875000000,  303.01353163639691956632304936647415161132812500000000,  299.13481817774970750178908929228782653808593750000000,  295.21629993547611547910491935908794403076171875000000,
        291.25801357380908029881538823246955871582031250000000,  287.25997280369767850061180070042610168457031250000000,  283.22216825588725441775750368833541870117187500000000,  279.14456727112951739400159567594528198242187500000000,  275.02711360518037508882116526365280151367187500000000,  270.86972704543859435943886637687683105468750000000000,  266.67230293521271278223139233887195587158203125000000,  262.43471160067048231212538667023181915283203125000000,  258.15679767449194059736328199505805969238281250000000,  253.83837930910522118210792541503906250000000000000000,
        249.47924727110267895113793201744556427001953125000000,  245.07916390698312625318067148327827453613281250000000,  240.63786196871495803861762396991252899169921875000000,  236.15504328571807945991167798638343811035156250000000,  231.63037726767265667149331420660018920898437500000000,  227.06349922001555796668981201946735382080078125000000,  222.45400845101298159534053411334753036499023437500000,  217.80146614579862784921715501695871353149414062500000,  213.10539297863954288914101198315620422363281250000000,  208.36526642978310519538354128599166870117187500000000,
        203.58051776738352600659709423780441284179687500000000,  198.75052864797225993243046104907989501953125000000000,  193.87462728044934578974789474159479141235351562500000,  188.95208408826820800641144160181283950805664062500000,  183.98210679190052019293943885713815689086914062500000,  178.96383481820660676930856425315141677856445312500000,  173.89633292421089549861790146678686141967773437500000,  168.77858389896803714691486675292253494262695312500000,  163.60948017733215920088696293532848358154296875000000,  158.38781416168137639033375307917594909667968750000000,
        153.11226699952035801288729999214410781860351562500000,  147.78139550300230098400788847357034683227539062500000,  142.39361681607843479469011072069406509399414062500000,  136.94719032964451344014378264546394348144531250000000,  131.44019620537466153109562583267688751220703125000000,  125.87050968150643370790930930525064468383789062500000,  120.23577007903433866431441856548190116882324218750000,  114.53334307539843450740590924397110939025878906250000,  108.76027432058174326812149956822395324707031250000000,  102.91323176913684278588334564119577407836914062500000,
         96.98843308278686947687674546614289283752441406250000,   90.98155294622938527027145028114318847656250000000000,   84.88760284043050319269241299480199813842773437500000,   78.70077222790672522023669444024562835693359375000000,   72.41421432096451837878703372552990913391113281250000,   66.01974994666942109233787050470709800720214843750000,   59.50744620958226960283354856073856353759765625000000,   52.86499590064471476580365560948848724365234375000000,   46.07676390216576578495732974261045455932617187500000,   39.12224215028557949835885665379464626312255859375000,
         31.97336902918570444853685330599546432495117187500000,   24.58942989513817423130603856407105922698974609375000,   16.90599171746302431529329624027013778686523437500000,    8.80537513890966927476711134659126400947570800781250,    0.00000000000000000000000000000000000000000000000000,  438.71291418612116785880061797797679901123046875000000,  440.38722369069176920675090514123439788818359375000000,  441.30917765154947574046673253178596496582031250000000,  441.77372448571009044826496392488479614257812500000000,  441.90399180684971724986098706722259521484375000000000,
        441.76860676764096069746301509439945220947265625000000,  441.41161543412670198449632152915000915527343750000000,  440.86379263649405402247793972492218017578125000000000,  440.14790174963866320467786863446235656738281250000000,  439.28148466245261261065024882555007934570312500000000,  438.27848393431304430123418569564819335937500000000000,  437.15025231308311504108132794499397277832031250000000,  435.90621486101264281387557275593280792236328125000000,  434.55432178591473757478524930775165557861328125000000,  433.10136889250270542106591165065765380859375000000000,
        431.55323089025591798417735844850540161132812500000000,  429.91503538650454174785409122705459594726562500000000,  428.19129534380056156805949285626411437988281250000000,  426.38601173218864914815640076994895935058593750000000,  424.50275433458040197365335188806056976318359375000000,  422.54472623683659548987634479999542236328125000000000,  420.51481593050993978977203369140625000000000000000000,  418.41563987085652343012043274939060211181640625000000,  416.24957758241220062700449489057064056396484375000000,  414.01880087580184408579953014850616455078125000000000,
        411.72529836057793772852164693176746368408203125000000,  409.37089616310629480722127482295036315917968750000000,  406.95727555493414229204063303768634796142578125000000,  404.48598804483208368765190243721008300781250000000000,  401.95846837250451244472060352563858032226562500000000,  399.37604575383585370218497700989246368408203125000000,  396.73995365944011837200378067791461944580078125000000,  394.05133835516807039311970584094524383544921875000000,  391.31126639143235479423310607671737670898437500000000,  388.52073119506115972399129532277584075927734375000000,
        385.68065889089643860643263906240463256835937500000000,  382.79191345902097509679151698946952819824218750000000,  379.85530131621646887651877477765083312988281250000000,  376.87157539615327550563961267471313476562500000000000,  373.84143879124684417547541670501232147216796875000000,  370.76554800957006818862282671034336090087890625000000,  367.64451589228650618679239414632320404052734375000000,  364.47891423046576164779253304004669189453125000000000,  361.26927611459723266307264566421508789062500000000000,  358.01609804545398674235912039875984191894531250000000,
        354.71984183100164500501705333590507507324218750000000,  351.38093629068413292770856060087680816650390625000000,  347.99977878554227572749368846416473388671875000000000,  344.57673659014648137599579058587551116943359375000000,  341.11214812019045439228648319840431213378906250000000,  337.60632402773694593633990734815597534179687500000000,  334.05954817448531457557692192494869232177734375000000,  330.47207849200594864669255912303924560546875000000000,  326.84414773662467723625013604760169982910156250000000,  323.17596414551599082187749445438385009765625000000000,
        319.46771199955043130103149451315402984619140625000000,  315.71955209752337623285711742937564849853515625000000,  311.93162214554922684328630566596984863281250000000000,  308.10403706462022910272935405373573303222656250000000,  304.23688921859485390086774714291095733642578125000000,  300.33024856417654291362850926816463470458984375000000,  296.38416272376383631126373074948787689208984375000000,  292.39865698138487459800671786069869995117187500000000,  288.37373420125862821805640123784542083740234375000000,  284.30937466784854450452257879078388214111328125000000,
        280.20553584557131898691295646131038665771484375000000,  276.06215205558976322208764031529426574707031250000000,  271.87913406633737167794606648385524749755859375000000,  267.65636859357988441843190230429172515869140625000000,  263.39371770490083690674509853124618530273437500000000,  259.09101812248559326690156012773513793945312500000000,  254.74808041694728899528854526579380035400390625000000,  250.36468808367203564557712525129318237304687500000000,  245.94059649171927617317123804241418838500976562500000,  241.47553169367319014781969599425792694091796875000000,
        236.96918908295279493358975742012262344360351562500000,  232.42123188290733537542109843343496322631835937500000,  227.83128944948327898600837215781211853027343750000000,  223.19895536628385457333934027701616287231445312500000,  218.52378530734898731680004857480525970458984375000000,  213.80529463886111329884442966431379318237304687500000,  209.04295572607932740538672078400850296020507812500000,  204.23619490595231695806432981044054031372070312500000,  199.38438907883264050724392291158437728881835937500000,  194.48686186422875721291347872465848922729492187500000,
        189.54287925523121316473407205194234848022460937500000,  184.55164469366681601059099193662405014038085937500000,  179.51229347257469726173439994454383850097656250000000,  174.42388635347643344175594393163919448852539062500000,  169.28540226209912589183659292757511138916015625000000,  164.09572989634000350633868947625160217285156250000000,  158.85365804250281485110463108867406845092773437500000,  153.55786434770814707917452324181795120239257812500000,  148.20690223449943800915207248181104660034179687500000,  142.79918556333660717427846975624561309814453125000000,
        137.33297054332894049366586841642856597900390625000000,  131.80633425188364071800606325268745422363281250000000,  126.21714893651943611985188908874988555908203125000000,  120.56305101728936790550505975261330604553222656250000,  114.84140335688759648746781749650835990905761718750000,  109.04924887334249206105596385896205902099609375000000,  103.18325286880595115235337289050221443176269531250000,   97.23963042907443821150081930682063102722167968750000,   91.21405373646106795604282524436712265014648437500000,   85.10153184029756801010080380365252494812011718750000,
         78.89625183955580212114000460132956504821777343750000,   72.59136464855144765806471696123480796813964843750000,   66.17868885949438606530748074874281883239746093750000,   59.64828940291742753743164939805865287780761718750000,   52.98785695435908849049155833199620246887207031250000,   46.18175433724655221112698200158774852752685546875000,   39.20947148334985143947051255963742733001708984375000,   32.04294482507131647253117989748716354370117187500000,   24.64145781765809672947398212272673845291137695312500,   16.94057557840288552597485249862074851989746093750000,
          8.82261694534417451052377145970240235328674316406250,    0.00000000000000000000000000000000000000000000000000,  443.47508812091894014884019270539283752441406250000000,  445.15099837403153060222393833100795745849609375000000,  446.07474531779371318407356739044189453125000000000000,  446.54125515710643412603531032800674438476562500000000,  446.67364081668233666277956217527389526367187500000000,  446.54051926136264683009358122944831848144531250000000,  446.18592925732400544802658259868621826171875000000000,  445.64064029379005660302937030792236328125000000000000,
        444.92741179519521210750099271535873413085937500000000,  444.06378272525171269080601632595062255859375000000000,  443.06369349910426080896286293864250183105468750000000,  441.93849533228927839445532299578189849853515625000000,  440.69761224634572727154591120779514312744140625000000,  439.34899381234475868041045032441616058349609375000000,  437.89943553708030776761006563901901245117187500000000,  436.35481212170356002388871274888515472412109375000000,  434.72025141709121953681460581719875335693359375000000,  433.00026685200589326996123418211936950683593750000000,
        431.19886006255700294786947779357433319091796875000000,  429.31960167966798280758666805922985076904296875000000,  427.36569580506466081715188920497894287109375000000000,  425.34003210294008567871060222387313842773437500000000,  423.24522834930587578128324821591377258300781250000000,  421.08366553087631700691417790949344635009765625000000,  418.85751705681099110734066925942897796630859375000000,  416.56877326784729120845440775156021118164062500000000,  414.21926215163819051667815074324607849121093750000000,  411.81066696957071826545870862901210784912109375000000,
        409.34454134813404380111023783683776855468750000000000,  406.82232227273607350070960819721221923828125000000000,  404.24534133375959754630457609891891479492187500000000,  401.61483450657163984942599199712276458740234375000000,  398.93195069409460984388715587556362152099609375000000,  396.19775921877032942575169727206230163574218750000000,  393.41325641760550979597610421478748321533203125000000,  390.57937146750140300355269573628902435302734375000000,  387.69697154674770445126341655850410461425781250000000,  384.76686642128203175161615945398807525634765625000000,
        381.78981253022305963895632885396480560302734375000000,  378.76651663362650879207649268209934234619140625000000,  375.69763907586741424893261864781379699707031250000000,  372.58379671013960887648863717913627624511718750000000,  369.42556552295980054623214527964591979980468750000000,  366.22348299202656107809161767363548278808593750000000,  362.97805020612383941625012084841728210449218750000000,  359.68973377180788020268664695322513580322265625000000,  356.35896752826141664627357386052608489990234375000000,  352.98615408882454858030541799962520599365234375000000,
        349.57166622524971444363472983241081237792968750000000,  346.11584810859807248561992309987545013427734375000000,  342.61901641884702485185698606073856353759765625000000,  339.08146133366824415134033188223838806152343750000000,  335.50344740542021781948278658092021942138671875000000,  331.88521433414848615939263254404067993164062500000000,  328.22697764327529057482024654746055603027343750000000,  324.52892926366263282034196890890598297119140625000000,  320.79123803082831045685452409088611602783203125000000,  317.01405009927117362167336978018283843994140625000000,
        313.19748927709832742038997821509838104248046875000000,  309.34165728343339196726446971297264099121093750000000,  305.44663393041025756247108802199363708496093750000000,  301.51247723090585850513889454305171966552734375000000,  297.53922343253179860766977071762084960937500000000000,  293.52688697777580273395869880914688110351562500000000,  289.47546038954976665991125628352165222167968750000000,  285.38491408075418576117954216897487640380859375000000,  281.25519608579287478278274647891521453857421875000000,  277.08623171126265560815227217972278594970703125000000,
        272.87792310228167025343282148241996765136718750000000,  268.63014872009699729460407979786396026611328125000000,  264.34276272570889432245166972279548645019531250000000,  260.01559426325314916539355181157588958740234375000000,  255.64844663576346306399500463157892227172851562500000,  251.24109636468199369119247421622276306152343750000000,  246.79329212305398755233909469097852706909179687500000,  242.30475353071378208369424100965261459350585937500000,  237.77516979788728690436983015388250350952148437500000,  233.20419820146298661711625754833221435546875000000000,
        228.59146237565127535162901040166616439819335937500000,  223.93655039579027743457118049263954162597656250000000,  219.23901263057044275228690821677446365356445312500000,  214.49835933383124597639834973961114883422851562500000,  209.71405794218640039616730064153671264648437500000000,  204.88553003888407033628027420490980148315429687500000,  200.01214793728522067794983740895986557006835937500000,  195.09323082886078282172093167901039123535156250000000,  190.12804043031002265706774778664112091064453125000000,  185.11577605182347383561136666685342788696289062500000,
        180.05556899305656770593486726284027099609375000000000,  174.94647615425921571841172408312559127807617187500000,  169.78747272619702357587812002748250961303710937500000,  164.57744379263019141035329084843397140502929687500000,  159.31517464135961859028611797839403152465820312500000,  153.99933953172387646191054955124855041503906250000000,  148.62848860455110866496397648006677627563476562500000,  143.20103254024121497423038817942142486572265625000000,  137.71522446531528771629382390528917312622070312500000,  132.16913846809441679397423285990953445434570312500000,
        126.56064389674433812160714296624064445495605468750000,  120.88737435811798093254765262827277183532714843750000,  115.14668998445958436604996677488088607788085937500000,  109.33563104286197642522893147543072700500488281250000,  103.45086026097727938122261548414826393127441406250000,   97.48859022360750259394990280270576477050781250000000,   91.44449068278584036306710913777351379394531250000000,   85.31356832561949943283252650871872901916503906250000,   79.09000795437644626417750259861350059509277343750000,   72.76695824977153392865147907286882400512695312500000,
         66.33623563121051347479806281626224517822265625000000,   59.78790291501693587861154810525476932525634765625000,   53.10964871901241224350087577477097511291503906250000,   46.28583386339090566252707503736019134521484375000000,   39.29594632925297759129534824751317501068115234375000,   32.11192064905957721521190251223742961883544921875000,   24.69303842708856677745643537491559982299804687500000,   16.97486297695513712824322283267974853515625000000000,    8.83971137870347511977797694271430373191833496093750,    0.00000000000000000000000000000000000000000000000000,
        448.24577274538461324482341296970844268798828125000000,  449.92326765547841205261647701263427734375000000000000,  450.84878894370132229596492834389209747314453125000000,  451.31724085380500355313415639102458953857421875000000,  451.45172177571157590136863291263580322265625000000000,  451.32083858417070132418302819132804870605468750000000,  450.96862280755112806218676269054412841796875000000000,  450.42583863209029004792682826519012451171875000000000,  449.71524155257998245360795408487319946289062500000000,  448.85436761556326246136450208723545074462890625000000,
        447.85715509000584688692470081150531768798828125000000,  446.73495364914299443626077845692634582519531250000000,  445.49718625714228892320534214377403259277343750000000,  444.15180182593536528656841255724430084228515625000000,  442.70559553694414489655173383653163909912109375000000,  441.16444205097548092453507706522941589355468750000000,  439.53346942622744109030463732779026031494140625000000,  437.81719151749064167233882471919059753417968750000000,  436.01961058297376894188346341252326965332031250000000,  434.14429805397554673618287779390811920166015625000000,
        432.19445899684393452844233252108097076416015625000000,  430.17298419358695582559448666870594024658203125000000,  428.08249268255349306855350732803344726562500000000000,  425.92536685058706780182546935975551605224609375000000,  423.70378163963630413491046056151390075683593750000000,  421.41972905209428290618234314024448394775390625000000,  419.07503886347848265359061770141124725341796875000000,  416.67139624756947569039766676723957061767578125000000,  414.21035686695557842540438286960124969482421875000000,  411.69335986678470362676307559013366699218750000000000,
        409.12173912144237419852288439869880676269531250000000,  406.49673301581003670435165986418724060058593750000000,  403.81949298967009553962270729243755340576171875000000,  401.09109103205685187276685610413551330566406250000000,  398.31252627921872999650076963007450103759765625000000,  395.48473084338093030964955687522888183593750000000000,  392.60857497818108186038443818688392639160156250000000,  389.68487166937882193451514467597007751464843750000000,  386.71438072535181618150090798735618591308593750000000,  383.69781243033742157422238960862159729003906250000000,
        380.63583081383859507695888169109821319580078125000000,  377.52905658170612923640874214470386505126953125000000,  374.37806974780647806255728937685489654541015625000000,  371.18341199965846044506179168820381164550781250000000,  367.94558882676005850953515619039535522460937500000000,  364.66507143638631305293529294431209564208984375000000,  361.34229847828453330293996259570121765136718750000000,  357.97767759683080157628864981234073638916015625000000,  354.57158682674912597576621919870376586914062500000000,  351.12437584637677900900598615407943725585937500000000,
        347.63636710061626899914699606597423553466796875000000,  344.10785680411441944670514203608036041259765625000000,  340.53911583379999683529604226350784301757812500000000,  336.93039051867390298866666853427886962890625000000000,  333.28190333364403841187595389783382415771484375000000,  329.59385350321161922693136148154735565185546875000000,  325.86641751992692661588080227375030517578125000000000,  322.09974958172472270234720781445503234863281250000000,  318.29398195150304218259407207369804382324218750000000,  314.44922524161876253856462426483631134033203125000000,
        310.56556862531891738399281166493892669677734375000000,  306.64307997650479364892817102372646331787109375000000,  302.68180593862086880108108744025230407714843750000000,  298.68177192286765375683899037539958953857421875000000,  294.64298203534264075642568059265613555908203125000000,  290.56541893211101523775141686201095581054687500000000,  286.44904360058734482663567177951335906982421875000000,  282.29379506495723717307555489242076873779296875000000,  278.09959001268072142920573242008686065673828125000000,  273.86632233837508465512655675411224365234375000000000,
        269.59386260056982109745149500668048858642578125000000,  265.28205738593743490127963013947010040283203125000000,  260.93072857461874036744120530784130096435546875000000,  256.53967249915814363703248091042041778564453125000000,  252.10865898831423237425042316317558288574218750000000,  247.63743028559574099745077546685934066772460937500000,  243.12569983074502033559838309884071350097656250000000,  238.57315089052167422778438776731491088867187500000000,  233.97943502297096074471483007073402404785156250000000,  229.34417035683389940459164790809154510498046875000000,
        224.66693966480215749470517039299011230468750000000000,  219.94728820583810602329322136938571929931640625000000,  215.18472130766610916907666251063346862792968750000000,  210.37870165564874014307861216366291046142578125000000,  205.52864624841481600014958530664443969726562500000000,  200.63392297358609539514873176813125610351562500000000,  195.69384674846949678794771898537874221801757812500000,  190.70767516028766408453520853072404861450195312500000,  185.67460352794191180691996123641729354858398437500000,  180.59375929184813003303133882582187652587890625000000,
        175.46419561926697383569262456148862838745117187500000,  170.28488408874173387630435172468423843383789062500000,  165.05470628738927985068585257977247238159179687500000,  159.77244411703506443700462114065885543823242187500000,  154.43676855705786010730662383139133453369140625000000,  149.04622656993231544220179785043001174926757812500000,  143.59922575512931075536471325904130935668945312500000,  138.09401625169772387380362488329410552978515625000000,  132.52866925017573862533026840537786483764648437500000,  126.90105128705698689373093657195568084716796875000000,
        121.20879324023185574787930818274617195129394531250000,  115.44925259245727033885486889630556106567382812500000,  109.61946703773671174531045835465192794799804687500000,  103.71609680410192311228456674143671989440917968750000,   97.73535204741388326965534361079335212707519531250000,   91.67290015878617737143940757960081100463867187500000,   85.52374552989424216775660170242190361022949218750000,   79.28207073064571375198283931240439414978027343750000,   72.94102227016117012681206688284873962402343750000000,   66.49241445481732171174371615052223205566406250000000,
         59.92630804437246183624665718525648117065429687500000,   53.23038965453887527701226645149290561676025390625000,   46.38901815593818156457928125746548175811767578125000,   39.38167963080071842796314740553498268127441406250000,   32.18030676166417691774768172763288021087646484375000,   24.74417935015061686954140895977616310119628906250000,   17.00885895287903437633758585434406995773315429687500,    8.85666093701724754794213367858901619911193847656250,    0.00000000000000000000000000000000000000000000000000,  453.02489623849612598860403522849082946777343750000000,
        454.70396001345312697594636119902133941650390625000000,  455.63123736333000124432146549224853515625000000000000,  456.10161081704404750780668109655380249023437500000000,  456.23816438131001405054121278226375579833984375000000,  456.10949493712445246274000965058803558349609375000000,  455.75962683645724382586195133626461029052734375000000,  455.21931900040090113179758191108703613281250000000000,  454.51132301516031475330237299203872680664062500000000,  453.65317201864468188432510942220687866210937500000000,  452.65880213245105778696597553789615631103515625000000,
        451.53956147849339686217717826366424560546875000000000,  450.30487194803350803340435959398746490478515625000000,  448.96268177274919253250118345022201538085937500000000,  447.51978578266727026857552118599414825439453125000000,  445.98205856780356270974152721464633941650390625000000,  444.35462835915831192323821596801280975341796875000000,  442.64200939920925748083391226828098297119140625000000,  440.84820452627616305107949301600456237792968750000000,  438.97678592651254803058691322803497314453125000000000,  437.03095958190328929049428552389144897460937500000000,
        435.01361733984924740070709958672523498535156250000000,  432.92737944517421055934391915798187255859375000000000,  430.77462962551538794286898337304592132568359375000000,  428.55754429274526273729861713945865631103515625000000,  426.27811704443791995799983851611614227294921875000000,  423.93817937379111526752240024507045745849609375000000,  421.53941829296604737464804202318191528320312500000000,  419.08339142266947874304605647921562194824218750000000,  416.57153998568441011229879222810268402099609375000000,  414.00520005399357614805921912193298339843750000000000,
        411.38561233109231807247851975262165069580078125000000,  408.71393069801388264750130474567413330078125000000000,  405.99122970983239611086901277303695678710937500000000,  403.21851119628666992866783402860164642333984375000000,  400.39671009369880039230338297784328460693359375000000,  397.52669961404978948849020525813102722167968750000000,  394.60929583981260293512605130672454833984375000000000,  391.64526181905802104665781371295452117919921875000000,  388.63531122379856697079958394169807434082031250000000,  385.58011162500508817174704745411872863769531250000000,
        382.48028742982171479525277391076087951660156250000000,  379.33642251991437888136715628206729888916015625000000,  376.14906262435869166438351385295391082763671875000000,  372.91871745582170660782139748334884643554687500000000,  369.64586263485352901625446975231170654296875000000000,  366.33094142375551882651052437722682952880859375000000,  362.97436628863539453959674574434757232666015625000000,  359.57652030580186419683741405606269836425781250000000,  356.13775842654075631799059920012950897216796875000000,  352.65840861247829707281198352575302124023437500000000,
        349.13877285214209678088082000613212585449218750000000,  345.57912806793439131070044822990894317626953125000000,  341.97972692149932072425144724547863006591796875000000,  338.34079852437645286045153625309467315673828125000000,  334.66254905985834966486436314880847930908203125000000,  330.94516232109430120544857345521450042724609375000000,  327.18880016968739710137015208601951599121093750000000,  323.39360291830377036603749729692935943603515625000000,  319.55968964013914046518038958311080932617187500000000,  315.68715840745409195733373053371906280517578125000000,
        311.77608646079210075185983441770076751708984375000000,  307.82653030991474452093825675547122955322265625000000,  303.83852576692527236446039751172065734863281250000000,  299.81208791149407488774158991873264312744140625000000,  295.74721098753354908694745972752571105957031250000000,  291.64386823009760973945958539843559265136718750000000,  287.50201162068009352879016660153865814208984375000000,  283.32157156845988765780930407345294952392578125000000,  279.10245651436770231157424859702587127685546875000000,  274.84455245412448221031809225678443908691406250000000,
        270.54772237561030578945064917206764221191406250000000,  266.21180560504603818117175251245498657226562500000000,  261.83661705549764064926421269774436950683593750000000,  257.42194637011846225504996255040168762207031250000000,  252.96755695130656249602907337248325347900390625000000,  248.47318486554397054533183109015226364135742187500000,  243.93853761206668195882230065762996673583984375000000,  239.36329274165044012079306412488222122192382812500000,  234.74709630963448603324650321155786514282226562500000,  230.08956114478550603053008671849966049194335937500000,
        225.39026491265158824717218521982431411743164062500000,  220.64874794858096151983772870153188705444335937500000,  215.86451083146815221880387980490922927856445312500000,  211.03701166440129100010381080210208892822265625000000,  206.16566302254159381845965981483459472656250000000000,  201.24982852154863621763070113956928253173828125000000,  196.28881895138744084761128760874271392822265625000000,  191.28188791006169822139781899750232696533203125000000,  186.22822685924194274775800295174121856689453125000000,  181.12695950830433844203071203082799911499023437500000,
        175.97713541417996907512133475393056869506835937500000,  170.77772266060694050793244969099760055541992187500000,  165.52759945051246859293314628303050994873046875000000,  160.22554440749752302508568391203880310058593750000000,  154.87022533427418125029362272471189498901367187500000,  149.46018611402840292612381745129823684692382812500000,  143.99383136035419283871306106448173522949218750000000,  138.46940831606963229205575771629810333251953125000000,  132.88498536155131546365737449377775192260742187500000,  127.23842630580061552336701424792408943176269531250000,
        121.52735937865233495358552318066358566284179687500000,  115.74913949116836420216714031994342803955078125000000,  109.90080183909019240218185586854815483093261718750000,  103.97900422283210275509190978482365608215332031250000,   97.97995443862554054703650763258337974548339843750000,   91.89931758340999579104391159489750862121582031250000,   85.73209581772692899903631769120693206787109375000000,   79.47246954108578620434855110943317413330078125000000,   73.11358315074495806129561970010399818420410156250000,   66.64724889765444970635144272819161415100097656250000,
         60.06352554057443882129518897272646427154541015625000,   53.35009774672955984442523913457989692687988281250000,   46.49132248893025831648628809489309787750244140625000,   39.46668400051785141613436280749738216400146484375000,   32.24811316238441349923959933221340179443359375000000,   24.79488802014267889717302750796079635620117187500000,   17.04256841849957382351021806243807077407836914062500,    8.87346805533363003348767961142584681510925292968750,    0.00000000000000000000000000000000000000000000000000,  457.81238798127816380656440742313861846923828125000000,
        459.49300512035460997140035033226013183593750000000000,  460.42202059492944954399717971682548522949218750000000,  460.89429546086671507509890943765640258789062500000000,  461.03289949072251374673214741051197052001953125000000,  460.90641966669812745749368332326412200927734375000000,  460.55887322512273840402485802769660949707031250000000,  460.02101385963089796860003843903541564941406250000000,  459.31558926910395257436903193593025207519531250000000,  458.46012969183829000030527822673320770263671875000000,  457.46856910158265918653341941535472869873046875000000,
        456.35225406080854781976086087524890899658203125000000,  455.12060537340306609621620737016201019287109375000000,  453.78157057089146064754459075629711151123046875000000,  452.34194410723773671634262427687644958496093750000000,  450.80760047270950963138602674007415771484375000000000,  449.18366803820936183910816907882690429687500000000000,  447.47466139730977374711073935031890869140625000000000,  445.68458392833059633630909956991672515869140625000000,  443.81700852876832641413784585893154144287109375000000,  441.87514204940202944271732121706008911132812500000000,
        439.86187735288467592909000813961029052734375000000000,  437.77983583707668913120869547128677368164062500000000,  435.63140251364734467642847448587417602539062500000000,  433.41875520425770673682563938200473785400390625000000,  431.14388903808281838792026974260807037353515625000000,  428.80863715888438036927254870533943176269531250000000,  426.41468834644149410451063886284828186035156250000000,  423.96360210504252563623595051467418670654296875000000,  421.45682165664942431249073706567287445068359375000000,  418.89568518830321863788412883877754211425781250000000,
        416.28143563531148174661211669445037841796875000000000,  413.61522922869414742308435961604118347167968750000000,  410.89814299362149085936835035681724548339843750000000,  408.13118135246247675240738317370414733886718750000000,  405.31528195959987215246655978262424468994140625000000,  402.45132087386707553378073498606681823730468750000000,  399.54011715720133679496939294040203094482421875000000,  396.58243697402997440804028883576393127441406250000000,  393.57899725436277549306396394968032836914062500000000,  390.53046897403379489333019591867923736572265625000000,
        387.43748009763515938175260089337825775146484375000000,  384.30061822309556873733527027070522308349609375000000,  381.12043296133384728818782605230808258056640625000000,  377.89743807976992684416472911834716796875000000000000,  374.63211343453986046370118856430053710937500000000000,  371.32490671291998296510428190231323242187500000000000,  367.97623500461060075394925661385059356689453125000000,  364.58648621808009693268104456365108489990234375000000,  361.15602035606178787929820828139781951904296875000000,  357.68517066246965896425535902380943298339843750000000,
        354.17424465140783240713062696158885955810546875000000,  350.62352502756039029918611049652099609375000000000000,  347.03327050602530334799666889011859893798828125000000,  343.40371653857363298811833374202251434326171875000000,  339.73507595235150802182033658027648925781250000000000,  336.02753950617960754243540577590465545654296875000000,  332.28127636881964690473978407680988311767578125000000,  328.49643452286591127631254494190216064453125000000000,  324.67314109726038395820069126784801483154296875000000,  320.81150263081673301712726242840290069580078125000000,
        316.91160526855765056097880005836486816406250000000000,  312.97351489211888520003412850201129913330078125000000,  308.99727718493261363619240000844001770019531250000000,  304.98291763237602935987524688243865966796875000000000,  300.93044145654289422964211553335189819335937500000000,  296.83983348475749153294600546360015869140625000000000,  292.71105795040142538709915243089199066162109375000000,  288.54405822404442005790770053863525390625000000000000,  284.33875647226193450478604063391685485839843750000000,  280.09505324086575228648143820464611053466796875000000,
        275.81282695856378950338694266974925994873046875000000,  271.49193335628706336137838661670684814453125000000000,  267.13220479655689132414408959448337554931640625000000,  262.73344950630291805282467976212501525878906250000000,  258.29545070545765383940306492149829864501953125000000,  253.81796562242254822194809094071388244628906250000000,  249.30072438609937535147764720022678375244140625000000,  244.74342878256672406678262632340192794799804687500000,  240.14575086262635750244953669607639312744140625000000,  235.50733138428432766886544413864612579345703125000000,
        230.82777807171882500369974877685308456420898437500000,  226.10666366933622839496820233762264251708984375000000,  221.34352376604849155228293966501951217651367187500000,  216.53785436079360238181834574788808822631835937500000,  211.68910913543794549696031026542186737060546875000000,  206.79669639535717351463972590863704681396484375000000,  201.85997563097910756368946749716997146606445312500000,  196.87825364509663472745160106569528579711914062500000,  191.85078018046846182187437079846858978271484375000000,  186.77674296965258804448239970952272415161132812500000,
        181.65526211356677777075674384832382202148437500000000,  176.48538367615412880695657804608345031738281250000000,  171.26607235872853607361321337521076202392578125000000,  165.99620308770820997779082972556352615356445312500000,  160.67455131169512583255709614604711532592773437500000,  155.29978175573432963574305176734924316406250000000000,  149.87043531871196933025203179568052291870117187500000,  144.38491371952579811477335169911384582519531250000000,  138.84146139232561267817800398916006088256835937500000,  133.23814399144939102370699401944875717163085937500000,
        127.57282267925839391864428762346506118774414062500000,  121.84312311527088468210422433912754058837890625000000,  116.04639771362975864121835911646485328674316406250000,  110.17967924376762312022037804126739501953125000000000,  104.23962314774475146350596332922577857971191406250000,   98.22243492886190097124199382960796356201171875000000,   92.12377745413712659683369565755128860473632812500000,   85.93865071489629769985185703262686729431152343750000,   79.66123299994035278359660878777503967285156250000000,   73.28466665222676112989574903622269630432128906250000,
         66.80076192280607472184783546254038810729980468750000,   60.19957562302333542447740910574793815612792968750000,   53.46879052333919446482468629255890846252441406250000,   46.59276174869774678199974005110561847686767578125000,   39.55097173179354541616703500039875507354736328125000,   32.31534959848518440139741869643330574035644531250000,   24.84517168342687654103428940288722515106201171875000,   17.07599616296814204474685539025813341140747070312500,    8.89013510781884086497939279070124030113220214843750,    0.00000000000000000000000000000000000000000000000000,
        462.60817852687489448726410046219825744628906250000000,  464.29033381291606019658502191305160522460937500000000,  465.22106981164756689395289868116378784179687500000000,  465.69522634324027876573381945490837097167968750000000,  465.83585909266076896528829820454120635986328125000000,  465.71154523691285476161283440887928009033203125000000,  465.36629495679045476208557374775409698486328125000000,  464.83085675598749730852432548999786376953125000000000,  464.12797446750533936210558749735355377197265625000000,  463.27517543949710443484946154057979583740234375000000,
        462.28639149804968155876849777996540069580078125000000,  461.17296763891903310650377534329891204833984375000000,  459.94432356515602577928802929818630218505859375000000,  458.60840608938661944193881936371326446533203125000000,  457.17200926613043066026875749230384826660156250000000,  455.64100745836310579761629924178123474121093750000000,  454.02052914552780293888645246624946594238281250000000,  452.31508923735930238763103261590003967285156250000000,  450.52869161385052620971691794693470001220703125000000,  448.66490984223651139473076909780502319335937500000000,
        446.72695159728243652352830395102500915527343750000000,  444.71771070891770705202361568808555603027343750000000,  442.63980967690258694346994161605834960937500000000000,  440.49563474261873352588736452162265777587890625000000,  438.28736507997041371709201484918594360351562500000000,  436.01699728889764173800358548760414123535156250000000,  433.68636609951670379814459010958671569824218750000000,  431.29716199154108835500665009021759033203125000000000,  428.85094628156480212055612355470657348632812500000000,  426.34916411572538663676823489367961883544921875000000,
        423.79315571724106348483473993837833404541015625000000,  421.18416617030402449017856270074844360351562500000000,  418.52335396876293316381634213030338287353515625000000,  415.81179851629264021539711393415927886962890625000000,  413.05050673164623731281608343124389648437500000000000,  410.24041888612748607556568458676338195800781250000000,  407.38241377912811458372743800282478332519531250000000,  404.47731334031846017751377075910568237304687500000000,  401.52588673300994059900403954088687896728515625000000,  398.52885402166532458068104460835456848144531250000000,
        395.48688945700973818020429462194442749023437500000000,  392.40062442429717748382245190441608428955078125000000,  389.27065009370318193759885616600513458251953125000000,  386.09751980629380341270007193088531494140625000000000,  382.88175122437831987554091028869152069091796875000000,  379.62382827112122640755842439830303192138671875000000,  376.32420288095175919806933961808681488037109375000000,  372.98329657945839699095813557505607604980468750000000,  369.60150190901151745492825284600257873535156250000000,  366.17918371425469103996874764561653137207031250000000,
        362.71668029978246750033576972782611846923828125000000,  359.21430447073885261488612741231918334960937500000000,  355.67234446568994599147117696702480316162109375000000,  352.09106478990531741146696731448173522949218750000000,  348.47070695611154178550350479781627655029296875000000,  344.81149013882622966775670647621154785156250000000000,  341.11361174752539682231144979596138000488281250000000,  337.37724792312712907005334272980690002441406250000000,  333.60255396157293716896674595773220062255859375000000,  329.78966466764325105032185092568397521972656250000000,
        325.93869464154715842596488073468208312988281250000000,  322.04973850026397030887892469763755798339843750000000,  318.12287103508100472026853822171688079833984375000000,  314.15814730625947959197219461202621459960937500000000,  310.15560267525512472275295294821262359619140625000000,  306.11525277442444803455146029591560363769531250000000,  302.03709341364600504675763659179210662841796875000000,  297.92110042277158754586707800626754760742187500000000,  293.76722942829513840479194186627864837646484375000000,  289.57541556206427912911749444901943206787109375000000,
        285.34557309926981361059006303548812866210937500000000,  281.07759502230555881396867334842681884765625000000000,  276.77135250639310015685623511672019958496093750000000,  272.42669432210107061109738424420356750488281250000000,  268.04344614903209276235429570078849792480468750000000,  263.62140979399873685906641185283660888671875000000000,  259.16036230593289246826316229999065399169921875000000,  254.66005497854843042659922502934932708740234375000000,  250.12021223038422590434493031352758407592773437500000,  245.54053035024563200749980751425027847290039062500000,
        240.92067609421263796320999972522258758544921875000000,  236.26028511822872246739279944449663162231445312500000,  231.55896022777480425247631501406431198120117187500000,  226.81626942318723649805178865790367126464843750000000,  222.03174371571327583296806551516056060791015625000000,  217.20487468528941121803654823452234268188476562500000,  212.33511174614829997153719887137413024902343750000000,  207.42185908052027798476046882569789886474609375000000,  202.46447219368454284449398983269929885864257812500000,  197.46225403515219909422739874571561813354492187500000,
        192.41445062047495184742729179561138153076171875000000,  187.32024607560234130687604192644357681274414062500000,  182.17875701026079582334205042570829391479492187500000,  176.98902610771457943883433472365140914916992187500000,  171.75001479446473240386694669723510742187500000000000,  166.46059482357944148134265560656785964965820312500000,  161.11953856759672021325968671590089797973632812500000,  155.72550776882130207923182751983404159545898437500000,  150.27704043296023428410990163683891296386718750000000,  144.77253547171943637295044027268886566162109375000000,
        139.21023459464745997138379607349634170532226562500000,  133.58820081084155617645592428743839263916015625000000,  127.90429271370950914388231467455625534057617187500000,  122.15613346717844933664309792220592498779296875000000,  116.34107306037634543827152810990810394287109375000000,  110.45614190564292300678062019869685173034667968750000,  104.49799315333726212884357664734125137329101562500000,   98.46283007803762643561640288680791854858398437500000,   92.34631337871580569753859890624880790710449218750000,   86.14344093712999494982796022668480873107910156250000,
         79.84838898889341862741275690495967864990234375000000,   73.45429787815044164744904264807701110839843750000000,   66.95297590959708600166777614504098892211914062500000,   60.33447799884993401065003126859664916992187500000000,   53.58648506951453072133517707698047161102294921875000,   46.69335044687598923474070033989846706390380859375000,   39.63455480956042720208643004298210144042968750000000,   32.38202557341085707776073832064867019653320312500000,   24.89503740564568090576358372345566749572753906250000,   17.10914685634659093693699105642735958099365234375000,
          8.90666440977005180457126698456704616546630859375000,    0.00000000000000000000000000000000000000000000000000,  467.41219957160819831187836825847625732421875000000000,  469.09587806353431460593128576874732971191406250000000,  470.02831731319429309223778545856475830078125000000000,  470.50433613811486566191888414323329925537109375000000,  470.64697627981547611852874979376792907714843750000000,  470.52480520236088068486424162983894348144531250000000,  470.18182609046277775632916018366813659667968750000000,  469.64878229520684271847130730748176574707031250000000,
        468.94841380531255481400876305997371673583984375000000,  468.09824508867438908055191859602928161621093750000000,  467.11220582452739336076774634420871734619140625000000,  466.00163943543890354703762568533420562744140625000000,  464.77596451111566011604736559092998504638671875000000,  463.44312712762928185838973149657249450683593750000000,  462.00992091789362348208669573068618774414062500000000,  460.48222009139129795585176907479763031005859375000000,  458.86515320620515012706164270639419555664062500000000,  457.16323545487767887607333250343799591064453125000000,
        455.38047118244173816492548212409019470214843750000000,  453.52043458607926140757626853883266448974609375000000,  451.58633412166631160289398394525051116943359375000000,  449.58106454049055855648475699126720428466796875000000,  447.50724939525963463893276639282703399658203125000000,  445.36727610506630981035414151847362518310546875000000,  443.16332514096995964791858568787574768066406250000000,  440.89739451544312487385468557476997375488281250000000,  438.57132048350410968851065263152122497558593750000000,  436.18679516003521712264046072959899902343750000000000,
        433.74538160574911671574227511882781982421875000000000,  431.24852681922556030258419923484325408935546875000000,  428.69757298444022808325826190412044525146484375000000,  426.09376725520763784516020677983760833740234375000000,  423.43827030492820995277725160121917724609375000000000,  420.73216382830202064724289812147617340087890625000000,  417.97645714857861776181380264461040496826171875000000,  415.17209305746342806742177344858646392822265625000000,  412.31995299351308403856819495558738708496093750000000,  409.42086164760138444762560538947582244873046875000000,
        406.47559106997431399577180854976177215576171875000000,  403.48486434187111626670230180025100708007812500000000,  400.44935886517316703248070552945137023925781250000000,  397.36970931564616194009431637823581695556640625000000,  394.24651029875974472815869376063346862792968750000000,  391.08031874155477680687909014523029327392578125000000,  387.87165604938644491994637064635753631591796875000000,  384.62101005244625184786855243146419525146484375000000,  381.32883676362820324357016943395137786865234375000000,  377.99556196646386752036050893366336822509765625000000,
        374.62158264940489971195347607135772705078125000000000,  371.20726830063767920364625751972198486328125000000000,  367.75296207579566498679923824965953826904296875000000,  364.25898184935709878118359483778476715087890625000000,  360.72562115913990510307485237717628479003906250000000,  357.15315005209617993386927992105484008789062500000000,  353.54181583854131076805060729384422302246093750000000,  349.89184376100797635444905608892440795898437500000000,  346.20343758306842119054635986685752868652343750000000,  342.47678010270817594573600217700004577636718750000000,
        338.71203359414386113712680526077747344970703125000000,  334.90934018134669258870417252182960510253906250000000,  331.06882214594963897980051115155220031738281250000000,  327.19058217167003022041171789169311523437500000000000,  323.27470352686583510148921050131320953369140625000000,  319.32125018634900470715365372598171234130859375000000,  315.33026689310008805477991700172424316406250000000000,  311.30177916005868610227480530738830566406250000000000,  307.23579321168961087096249684691429138183593750000000,  303.13229586455162234415183775126934051513671875000000,
        298.99125434559977065873681567609310150146484375000000,  294.81261604644316776102641597390174865722656250000000,  290.59630821123721489129820838570594787597656250000000,  286.34223755530916832867660559713840484619140625000000,  282.05028981099013662969809956848621368408203125000000,  277.72032919643896775596658699214458465576171875000000,  273.35219780248667120758909732103347778320312500000000,  268.94571489168805555891594849526882171630859375000000,  264.50067610281695351659436710178852081298828125000000,  260.01685255297780940964003093540668487548828125000000,
        255.49398982828461157623678445816040039062500000000000,  250.93180685267338958510663360357284545898437500000000,  246.32999462280960756288550328463315963745117187500000,  241.68821479520823913844651542603969573974609375000000,  237.00609810953233136388007551431655883789062500000000,  232.28324262953250922691950108855962753295898437500000,  227.51921178014634961073170416057109832763671875000000,  222.71353215581490303520695306360721588134765625000000,  217.86569107096863717742962762713432312011718750000000,  212.97513381875776872220740187913179397583007812500000,
        208.04126059826478467584820464253425598144531250000000,  203.06342306342833126109326258301734924316406250000000,  198.04092043843655801538261584937572479248046875000000,  192.97299513406184701125312130898237228393554687500000,  187.85882778683895821814076043665409088134765625000000,  182.69753162754116715404961723834276199340820312500000,  177.48814606629727563813503365963697433471679687500000,  172.22962935788839899942104239016771316528320312500000,  166.92085018090418202518776524811983108520507812500000,  161.56057792668420347581559326499700546264648437500000,
        156.14747144585430760344024747610092163085937500000000,  150.68006593838967432930076029151678085327148437500000,  145.15675759281617729357094503939151763916015625000000,  139.57578547482543740443361457437276840209960937500000,  133.93521002590918556052201893180608749389648437500000,  128.23288734519698550684552174061536788940429687500000,  122.46643817288058642134274123236536979675292968750000,  116.63321014224217719856824260205030441284179687500000,  110.73023137513611402482638368383049964904785156250000,  104.75415279438453808324993588030338287353515625000000,
         98.70117550767585612447874154895544052124023437500000,   92.56695810554232650702033424749970436096191406250000,   86.34649641765517458225076552480459213256835937500000,   80.03396468188897472373355412855744361877441406250000,   73.62250129708239398951263865455985069274902343750000,   67.10391267322702901765296701341867446899414062500000,   60.46825188008445906007182202301919460296630859375000,   53.70319804257814411130311782471835613250732421875000,   46.79310273287913446438324172049760818481445312500000,   39.71744492053056774238939397037029266357421875000000,
         32.44815035485109433466277550905942916870117187500000,   24.94449207768224496817310864571481943130493164062500,   17.14202505352340111244302534032613039016723632812500,    8.92305821954572842003017285605892539024353027343750,    0.00000000000000000000000000000000000000000000000000,  472.22438392698057896268437616527080535888671875000000,  473.90957095253219222286134026944637298583984375000000,  474.84369649842437866027466952800750732421875000000000,  475.32155860838372518628602847456932067871093750000000,  475.46618522225037395401159301400184631347656250000000,
        475.34613418208914481510873883962631225585937500000000,  475.00540173533113375015091150999069213867187500000000,  474.47472611758792027103481814265251159667968750000000,  473.77684349502590066549601033329963684082031250000000,  472.92927546554824402846861630678176879882812500000000,  471.94594956293292398186167702078819274902343750000000,  470.83820763084014515698072500526905059814453125000000,  469.61546713402691466399119235575199127197265625000000,  468.28567339539068825615686364471912384033203125000000,  466.85561960523671132250456139445304870605468750000000,
        465.33117979462878111007739789783954620361328125000000,  463.71748257177642926762928254902362823486328125000000,  462.01904338017544660033308900892734527587890625000000,  460.23986699487630858129705302417278289794921875000000,  458.38352820493867056939052417874336242675781250000000,  456.45323620631194216912263073027133941650390625000000,  454.45188662767776577311451546847820281982421875000000,  452.38210402781476204836508259177207946777343750000000,  450.24627695372754487834754399955272674560546875000000,  448.04658712087558569692191667854785919189453125000000,
        445.78503389850203575406339950859546661376953125000000,  443.46345500768995862017618492245674133300781250000000,  441.08354413649254865958937443792819976806640625000000,  438.64686602448193752934457734227180480957031250000000,  436.15486945404501284428988583385944366455078125000000,  433.60889849777248628015513531863689422607421875000000,  431.01020230330794902329216711223125457763671875000000,  428.35994364399863343351171351969242095947265625000000,  425.65920642197806955664418637752532958984375000000000,  422.90900227722232784799416549503803253173828125000000,
        420.11027642968247164390049874782562255859375000000000,  417.26391286031201843798044137656688690185546875000000,  414.37073891956441684669698588550090789794921875000000,  411.43152943787578124101855792105197906494140625000000,  408.44701040111226575390901416540145874023437500000000,  405.41786224444848585335421375930309295654296875000000,  402.34472281025244910779292695224285125732421875000000,  399.22819000897396790605853311717510223388671875000000,  396.06882421652250059196376241743564605712890625000000,  392.86715043698285398932057432830333709716796875000000,
        389.62366025559396121025201864540576934814453125000000,  386.33881360358554957201704382896423339843750000000000,  383.01304035362346667170641012489795684814453125000000,  379.64674176218147749750642105937004089355468750000000,  376.24029177306033489003311842679977416992187500000000,  372.79403819446281431737588718533515930175781250000000,  369.30830376046304763804073445498943328857421875000000,  365.78338708633299347638967446982860565185546875000000,  362.21956352599011097481707111001014709472656250000000,  358.61708593876653594634262844920158386230468750000000,
        354.97618537176418840317637659609317779541015625000000,  351.29707166322009470604825764894485473632812500000000,  347.57993397155627235406427644193172454833984375000000,  343.82494123410697284271009266376495361328125000000000,  340.03224255889705318622873164713382720947265625000000,  336.20196755227289031608961522579193115234375000000000,  332.33422658465860877186059951782226562500000000000000,  328.42911099620715731361997313797473907470703125000000,  324.48669324364504973345901817083358764648437500000000,  320.50702698914722077461192384362220764160156250000000,
        316.49014713163211354185477830469608306884765625000000,  312.43606978042288346841814927756786346435546875000000,  308.34479217077080193121219053864479064941406250000000,  304.21629252028378687100484967231750488281250000000000,  300.05052982482686729781562462449073791503906250000000,  295.84744359196776031240005977451801300048828125000000,  291.60695350951300497399643063545227050781250000000000,  287.32895904611285686769406311213970184326171875000000,  283.01333898029781721561448648571968078613281250000000,  278.65995085363516636789427138864994049072265625000000,
        274.26863034294268572921282611787319183349609375000000,  269.83919054566530348893138580024242401123046875000000,  265.37142117157770826452178880572319030761718750000000,  260.86508763291629975356045179069042205810546875000000,  256.31993002383302382440888322889804840087890625000000,  251.73566197867694427259266376495361328125000000000000,  247.11196939701824248913908377289772033691406250000000,  242.44850902148124305313103832304477691650390625000000,  237.74490685231231168472731951624155044555664062500000,  233.00075638010389411647338420152664184570312500000000,
        228.21561661515818286716239526867866516113281250000000,  223.38900988851415263525268528610467910766601562500000,  218.52041939555971339359530247747898101806640625000000,  213.60928644827566813546582125127315521240234375000000,  208.65500739632358317976468242704868316650390625000000,  203.65693017018210753121820744127035140991210937500000,  198.61435039106896738303476013243198394775390625000000,  193.52650698209833990404149517416954040527343750000000,  188.39257720255852746049640700221061706542968750000000,  183.21167101174577851452340837568044662475585937500000,
        177.98282464968119143122748937457799911499023437500000,  172.70499329823496736935339868068695068359375000000000,  167.37704265632319788892345968633890151977539062500000,  161.99773922509001522485050372779369354248046875000000,  156.56573905087319076301355380564928054809570312500000,  151.07957461187331205110240262001752853393554687500000,  145.53763945412779889920784626156091690063476562500000,  139.93817007705564492425764910876750946044921875000000,  134.27922442916883483121637254953384399414062500000000,  128.55865618712695663816703017801046371459960937500000,
        122.77408373650922612796421162784099578857421875000000,  116.92285242131485745176178170368075370788574218750000,  111.00198813703377709316555410623550415039062500000000,  105.00813964074190209885273361578583717346191406250000,   98.93750593280378780036699026823043823242187500000000,   92.78574355275341645210573915392160415649414062500000,   86.54784633358595158370007993653416633605957031250000,   80.21798656890754841697344090789556503295898437500000,   73.78930076386527048271091189235448837280273437500000,   67.25359348358485078733792761340737342834472656250000,
         60.60091600011197243702554260380566120147705078125000,   53.81894568619996732650179183110594749450683593750000,   46.89203240585931808936948073096573352813720703125000,   39.79965346301028716879955027252435684204101562500000,   32.51373298247591492327046580612659454345703125000000,   24.99354242137701476167421787977218627929687500000000,   17.17463519797020055079883604776114225387573242187500,    8.93931874041750873516321007627993822097778320312500,    0.00000000000000000000000000000000000000000000000000,  477.04466549258563645707909017801284790039062500000000,
        478.73134664131737281422829255461692810058593750000000,  479.66714183880219479760853573679924011230468750000000,  480.14682857970871054931194521486759185791015625000000,  480.29342114164154509126092307269573211669921875000000,  480.17546783430674395276582799851894378662109375000000,  479.83695802600510660340660251677036285400390625000000,  479.30862487379863523528911173343658447265625000000000,  478.61320074313800887466641142964363098144531250000000,  477.76820437255463502879138104617595672607421875000000,  476.78756115231271905940957367420196533203125000000000,
        475.68261134215839547323412261903285980224609375000000,  474.46277127114717586664482951164245605468750000000000,  473.13598549336461474013049155473709106445312500000000,  471.70904673660476191798807121813297271728515625000000,  470.18782882979814985446864739060401916503906250000000,  468.57746040408881071925861760973930358886718750000000,  466.88245712349635141436010599136352539062500000000000,  465.10682415959877289424184709787368774414062500000000,  463.25413685690307374898111447691917419433593750000000,  461.32760511214075904717901721596717834472656250000000,
        459.33012538928176127228653058409690856933593750000000,  457.26432320827234434545971453189849853515625000000000,  455.13258819632142149202991276979446411132812500000000,  452.93710326274981525784824043512344360351562500000000,  450.67986908015592462106724269688129425048828125000000,  448.36272477933243862935341894626617431640625000000000,  445.98736556213111725810449570417404174804687500000000,  443.55535778450172301745624281466007232666015625000000,  441.06815194693916737378458492457866668701171875000000,  438.52709394161337286277557723224163055419921875000000,
        435.93343483748822109191678464412689208984375000000000,  433.28833943172833187418291345238685607910156250000000,  430.59289375398674337702686898410320281982421875000000,  427.84811167708954826593981124460697174072265625000000,  425.05494076120044155686628073453903198242187500000000,  422.21426743726971153591875918209552764892578125000000,  419.32692161833648469837498851120471954345703125000000,  416.39368081319247494320734404027462005615234375000000,  413.41527380539037039852701127529144287109375000000000,  410.39238395106519874389050528407096862792968750000000,
        407.32565214115300022967858240008354187011718750000000,  404.21567946701418350130552425980567932128906250000000,  401.06302962296217629045713692903518676757812500000000,  397.86823107456370962609071284532546997070312500000000,  394.63177901765584465465508401393890380859375000000000,  391.35413714969917009511846117675304412841796875000000,  388.03573927224601902707945555448532104492187500000000,  384.67699074087209965000511147081851959228515625000000,  381.27826977682735787311685271561145782470703125000000,  377.83992865285546258746762759983539581298828125000000,
        374.36229476406180083358776755630970001220703125000000,  370.84567159334409325310843996703624725341796875000000,  367.29033957970159463002346456050872802734375000000000,  363.69655689668485365473316051065921783447265625000000,  360.06456014731531922734575346112251281738281250000000,  356.39456498097530356972129084169864654541015625000000,  352.68676663702149198797997087240219116210937500000000,  348.94134041920773370293318293988704681396484375000000,  345.15844210439058770134579390287399291992187500000000,  341.33820828843175831934786401689052581787109375000000,
        337.48075667169302960246568545699119567871093750000000,  333.58618628603483102779136970639228820800781250000000,  329.65457766476902179419994354248046875000000000000000,  325.68599295657708125872886739671230316162109375000000,  321.68047598397640740586211904883384704589843750000000,  317.63805224649695446714758872985839843750000000000000,  313.55872886830871948404819704592227935791015625000000,  309.44249448961460302598425187170505523681640625000000,  305.28931910068484967268886975944042205810546875000000,  301.09915381695367386782891117036342620849609375000000,
        296.87193059311567822078359313309192657470703125000000,  292.60756187364813740714453160762786865234375000000000,  288.30594017662707528870669193565845489501953125000000,  283.96693760710240894695743918418884277343750000000000,  279.59040529562906840510549955070018768310546875000000,  275.17617275681146793431253172457218170166015625000000,  270.72404716189203099929727613925933837890625000000000,  266.23381251847945350164081901311874389648437500000000,  261.70522874945908142763073556125164031982421875000000,  257.13803066192087953822920098900794982910156250000000,
        252.53192679556082111957948654890060424804687500000000,  247.88659813842176049547560978680849075317382812500000,  243.20169669599931694392580538988113403320312500000000,  238.47684389759751866222359240055084228515625000000000,  233.71162882132017557523795403540134429931640625000000,  228.90560621614807246260170359164476394653320312500000,  224.05829429609482872365333605557680130004882812500000,  219.16917227733566164715739432722330093383789062500000,  214.23767762432862582500092685222625732421875000000000,  209.26320296511786978044256102293729782104492187500000,
        204.24509262900031103526998776942491531372070312500000,  199.18263875127280471133417449891567230224609375000000,  194.07507687949080832368053961545228958129882812500000,  188.92158100310578561220609117299318313598632812500000,  183.72125791290068264061119407415390014648437500000000,  178.47314077753605943144066259264945983886718750000000,  173.17618180071653455343039240688085556030273437500000,  167.82924379262988168193260207772254943847656250000000,  162.43109045156006686738692224025726318359375000000000,  156.98037510346023282181704416871070861816406250000000,
        151.47562758539589822248672135174274444580078125000000,  145.91523887845019658016099128872156143188476562500000,  140.29744299034649657187401317059993743896484375000000,  134.62029544837827188530354760587215423583984375000000,  128.88164757581239427963737398386001586914062500000000,  123.07911547013425490604277001693844795227050781250000,  117.21004225013831501200911588966846466064453125000000,  111.27145164669937571488844696432352066040039062500000,  105.25999031067297551089723128825426101684570312500000,   99.17185519250217851094930665567517280578613281250000,
         93.00270083609591154072404606267809867858886718750000,   86.74751913120684321256703697144985198974609375000000,   80.40048047875134784590045455843210220336914062500000,   73.95471953998898584359267260879278182983398437500000,   67.40203908328525983506551710888743400573730468750000,   60.73248862944941350860972306691110134124755859375000,   53.93374384398658349937250022776424884796142578125000,   46.99015292617613681613875087350606918334960937500000,   39.88119155631441259401981369592249393463134765625000,   32.57878227535601922681962605565786361694335937500000,
         25.04219499501245849160113721154630184173583984375000,   17.20698162534634079179340915288776159286499023437500,    8.95544812234739140421879710629582405090332031250000,    0.00000000000000000000000000000000000000000000000000,  481.87297922988790332965436391532421112060546875000000,  483.56114034640046384083689190447330474853515625000000,  484.49858885271322606058674864470958709716796875000000,  484.98008191517737941467203199863433837890625000000000,  485.12862028633105637709377333521842956542968750000000,  485.01274283188479330419795587658882141113281250000000,
        484.67643209851109986630035564303398132324218750000000,  484.15041620142517331260023638606071472167968750000000,  483.45742372728801683479105122387409210205078125000000,  482.61497056620311241204035468399524688720703125000000,  481.63697996737715811832458712160587310791015625000000,  480.53479060230716868318268097937107086181640625000000,  479.31781765440427989233285188674926757812500000000000,  477.99400489423567250923952087759971618652343750000000,  476.57014456822622605614014901220798492431640625000000,  475.05211028061131628419389016926288604736328125000000,
        473.44503065953165332757635042071342468261718750000000,  471.75342156045894626004155725240707397460937500000000,  469.98128851946518125259899534285068511962890625000000,  468.13220740163308164483169093728065490722656250000000,  466.20938876684704155195504426956176757812500000000000,  464.21572987402663557077175937592983245849609375000000,  462.15385716125859971725731156766414642333984375000000,  460.02616129023755320304189808666706085205078125000000,  457.83482631571325782715575769543647766113281250000000,  455.58185416245248688937863335013389587402343750000000,
        453.26908531696000181909766979515552520751953125000000,  450.89821643800667061441345140337944030761718750000000,  448.47081543807263415146735496819019317626953125000000,  445.98833447284761177797918207943439483642578125000000,  443.45212118798843903277884237468242645263671875000000,  440.86342850438387586109456606209278106689453125000000,  438.22342317017779578236513771116733551025390625000000,  435.53319326611006090388400480151176452636718750000000,  432.79375481766351185797248035669326782226562500000000,  430.00605764107956474617822095751762390136718750000000,
        427.17099052903353140209219418466091156005859375000000,  424.28938586452784420544048771262168884277343750000000,  421.36202373750955985087784938514232635498046875000000,  418.38963562719089850361342541873455047607421875000000,  415.37290770354741198389092460274696350097656250000000,  412.31248379358231659352895803749561309814453125000000,  409.20896805137476803793106228113174438476562500000000,  406.06292736542519605791312642395496368408203125000000,  402.87489353217887355640414170920848846435546875000000,  399.64536522069272450607968494296073913574218750000000,
        396.37480975008560335481888614594936370849609375000000,  393.06366469857579204472131095826625823974609375000000,  389.71233936048321311318431980907917022705078125000000,  386.32121606548309955542208626866340637207031250000000,  382.89065137259683524462161585688591003417968750000000,  379.42097714983924561238382011651992797851562500000000,  375.91250154908101421824540011584758758544921875000000,  372.36550988449096166732488200068473815917968750000000,  368.78026542187433278741082176566123962402343750000000,  365.15701008529816817826940678060054779052734375000000,
        361.49596508656793503178050741553306579589843750000000,  357.79733148238631201820680871605873107910156250000000,  354.06129066335910238194628618657588958740234375000000,  350.28800477841372185139334760606288909912109375000000,  346.47761709764517945586703717708587646484375000000000,  342.63025231609702814239426515996456146240234375000000,  338.74601680051335961252334527671337127685546875000000,  334.82499878065095799684058874845504760742187500000000,  330.86726848631798247879487462341785430908203125000000,  326.87287823089468474790919572114944458007812500000000,
        322.84186244169109158974606543779373168945312500000000,  318.77423763710004323002067394554615020751953125000000,  314.67000235010237929600407369434833526611328125000000,  310.52913699727469065692275762557983398437500000000000,  306.35160369202696983847999945282936096191406250000000,  302.13734600035780886173597536981105804443359375000000,  297.88628863694344772738986648619174957275390625000000,  293.59833709887800523574696853756904602050781250000000,  289.27337723383340062355273403227329254150390625000000,  284.91127473881516607434605248272418975830078125000000,
        280.51187458502948857130832038819789886474609375000000,  276.07500036364427842272561974823474884033203125000000,  271.60045354640806181123480200767517089843750000000000,  267.08801265416093428939348086714744567871093750000000,  262.53743232522344897006405517458915710449218750000000,  257.94844227444775697222212329506874084472656250000000,  253.32074613233987747662467882037162780761718750000000,  248.65402015207683916742098517715930938720703125000000,  243.94791177040332286196644417941570281982421875000000,  239.20203800625790790945757180452346801757812500000000,
        234.41598367848072825836425181478261947631835937500000,  229.58929942102255949976097326725721359252929687500000,  224.72149947061916464008390903472900390625000000000000,  219.81205919780066437851928640156984329223632812500000,  214.86041234722989656802383251488208770751953125000000,  209.86594794753838755241304170340299606323242187500000,  204.82800684382044664744171313941478729248046875000000,  199.74587779748350158115499652922153472900390625000000,  194.61879308786834030797763261944055557250976562500000,  189.44592353749004587371018715202808380126953125000000,
        184.22637286730449091010086704045534133911132812500000,  178.95917126929799678691779263317584991455078125000000,  173.64326805989813351516204420477151870727539062500000,  168.27752324784469806218112353235483169555664062500000,  162.86069781241175746799854096025228500366210937500000,  157.39144243975613335351226851344108581542968750000000,  151.86828440329276190823293291032314300537109375000000,  146.28961219367951684944273438304662704467773437500000,  140.65365739865899286087369546294212341308593750000000,  134.95847319333842051491956226527690887451171875000000,
        129.20190861406712201642221771180629730224609375000000,  123.38157753427276475122198462486267089843750000000000,  117.49482090925344834886345779523253440856933593750000,  111.53866036475515954862203216180205345153808593750000,  105.50974050277702076527930330485105514526367187500000,   99.40425627917660733601223910227417945861816406250000,   93.21786029563523356955556664615869522094726562500000,   86.94554255020673849685408640652894973754882812500000,   80.58147160088773830466379877179861068725585937500000,   74.11878031312294012877828208729624748229980468750000,
         67.54926970496502747209888184443116188049316406250000,   60.86298759087757304087062948383390903472900390625000,   54.04760797251675796815106878057122230529785156250000,   47.08747742640035482963867252692580223083496093750000,   39.96207004979940080602318630553781986236572265625000,   32.64330683908367802814609603956341743469238281250000,   25.09045619857711884037598792929202318191528320312500,   17.23906856695888123454096785280853509902954101562500,    8.97144846369383230921812355518341064453125000000000,    0.00000000000000000000000000000000000000000000000000,
        486.70926113683941593990311957895755767822265625000000,  488.39888831423922965768724679946899414062500000000000,  489.33797408058893552151857875287532806396484375000000,  489.82125549075800563514349050819873809814453125000000,  489.97171990716100253848708234727382659912109375000000,  489.85789683861833054834278300404548645019531250000000,  489.52376206702996341846301220357418060302734375000000,  489.00003870223645208170637488365173339843750000000000,  488.30945157410383217211347073316574096679687500000000,  487.46951373554946940203080885112285614013671875000000,
        486.49414629766016560097341425716876983642578125000000,  485.39468633998200175483361817896366119384765625000000,  484.18054789110425417675287462770938873291015625000000,  482.85967392425379784981487318873405456542968750000000,  481.43885618662091019359650090336799621582031250000000,  479.92396803627951840098830871284008026123046875000000,  478.32013807362358193131512962281703948974609375000000,  476.63188231779548686972702853381633758544921875000000,  474.86320663871521219334681518375873565673828125000000,  473.01768738865280283789616078138351440429687500000000,
        471.09853575459806052094791084527969360351562500000000,  469.10864975176320967875653877854347229003906250000000,  467.05065669513277271107654087245464324951171875000000,  464.92694823706023043996538035571575164794921875000000,  462.73970953129429517503012903034687042236328125000000,  460.49094370570179535206989385187625885009765625000000,  458.18249255074209713711752556264400482177734375000000,  455.81605412759279261081246659159660339355468750000000,  453.39319784791894107911502942442893981933593750000000,  450.91537746233456118716276250779628753662109375000000,
        448.38394230668461659661261364817619323730468750000000,  445.80014708733790484984638169407844543457031250000000,  443.16516043369847466237843036651611328125000000000000,  440.48007240445770094083854928612709045410156250000000,  437.74590110104838913684943690896034240722656250000000,  434.96359851534310791976167820394039154052734375000000,  432.13405571737257560016587376594543457031250000000000,  429.25810747161301605956396088004112243652343750000000,  426.33653635634215106620104052126407623291015625000000,  423.37007644904235803551273420453071594238281250000000,
        420.35941663132672374558751471340656280517578125000000,  417.30520355898090656410204246640205383300781250000000,  414.20804433614756590031902305781841278076171875000000,  411.06850892717744727633544243872165679931640625000000,  407.88713233504108757188078016042709350585937500000000,  404.66441657128410724908462725579738616943359375000000,  401.40083243918451216813991777598857879638671875000000,  398.09682114894002324945176951587200164794921875000000,  394.75279578128663615643745288252830505371093750000000,  391.36914261386550606403034180402755737304687500000000,
        387.94622232285524887629435397684574127197265625000000,  384.48437107082582997463759966194629669189453125000000,  380.98390149041216545811039395630359649658203125000000,  377.44510357221753338308189995586872100830078125000000,  373.86824546431148519332055002450942993164062500000000,  370.25357418976830103929387405514717102050781250000000,  366.60131628787138424740987829864025115966796875000000,  362.91167838387997335303225554525852203369140625000000,  359.18484769159834968377253971993923187255859375000000,  355.42099245239546689845155924558639526367187500000000,
        351.62026231377967633306980133056640625000000000000000,  347.78278865013811582684866152703762054443359375000000,  343.90868482778830639290390536189079284667968750000000,  339.99804641605663846348761580884456634521484375000000,  336.05095134568784942530328407883644104003906250000000,  332.06746001549765878735342994332313537597656250000000,  328.04761534779436260578222572803497314453125000000000,  323.99144279272292124005616642534732818603515625000000,  319.89895028130445098213385790586471557617187500000000,  315.77012812656386131493491120636463165283203125000000,
        311.60494887174922951089683920145034790039062500000000,  307.40336708423501477227546274662017822265625000000000,  303.16531909327619587202207185328006744384765625000000,  298.89072266932197408095817081630229949951171875000000,  294.57947664210519178595859557390213012695312500000000,  290.23146045418917537972447462379932403564453125000000,  285.84653364606390368862776085734367370605468750000000,  281.42453526823476295248838141560554504394531250000000,  276.96528321501898517453810200095176696777343750000000,  272.46857347395064152806298807263374328613281250000000,
        267.93417928377482439827872440218925476074218750000000,  263.36185019296328846394317224621772766113281250000000,  258.75131100949027995739015750586986541748046875000000,  254.10226063123397466370079200714826583862304687500000,  249.41437074478804447608126793056726455688476562500000,  244.68728437863174463018367532640695571899414062500000,  239.92061429447551290650153532624244689941406250000000,  235.11394119810259439873334486037492752075195312500000,  230.26681174810011043518898077309131622314453125000000,  225.37873633741568824007117655128240585327148437500000,
        220.44918661858662289887433871626853942871093750000000,  215.47759273861308315645146649330854415893554687500000,  210.46334024362352010939503088593482971191406250000000,  205.40576660647388962388504296541213989257812500000000,  200.30415732196152589494886342436075210571289062500000,  195.15774150405061959645536262542009353637695312500000,  189.96568690694587644429702777415513992309570312500000,  184.72709427640430135397764388471841812133789062500000,  179.44099091857049188547534868121147155761718750000000,  174.10632334982091151687200181186199188232421875000000,
        168.72194886124538015792495571076869964599609375000000,  163.28662579364581119989452417939901351928710937500000,  157.79900227081807884133013430982828140258789062500000,  152.25760307700841167388716712594032287597656250000000,  146.66081428411828824209806043654680252075195312500000,  141.00686512889745927168405614793300628662109375000000,  135.29380650069924740819260478019714355468750000000000,  129.51948521294977467732678633183240890502929687500000,  123.68151297668715926647564629092812538146972656250000,  117.77722864316037032494932645931839942932128906250000,
        111.80365179031261391173757147043943405151367187500000,  105.75742502658594901276956079527735710144042968750000,   99.63474136661426427963306196033954620361328125000000,   93.43125152136030919791664928197860717773437500000000,   87.14194364691559258062625303864479064941406250000000,   80.76098450639707948539580684155225753784179687500000,   74.28150521585035903626703657209873199462890625000000,   67.69530508787536859927058685570955276489257812500000,   60.99243027395942107204973581247031688690185546875000,   54.16055315385020918483860441483557224273681640625000,
         47.18401872187435230898699956014752388000488281250000,   40.04229953153377863372952560894191265106201171875000,   32.70731507260853732077521272003650665283203125000000,   25.13833227881958265470530022867023944854736328125000,   17.27090015308485604350607900414615869522094726562500,    8.98732181285012288185498618986457586288452148437500,    0.00000000000000000000000000000000000000000000000000,  491.55344822329794851611950434744358062744140625000000,  493.24452779687590009416453540325164794921875000000000,  494.18523506081334062400856055319309234619140625000000,
        494.67028717152260242073680274188518524169921875000000,  494.82265823405873561569023877382278442382812500000000,  494.71086848622002207775949500501155853271484375000000,  494.37888700134647024242440238595008850097656250000000,  493.85743192013615043833851814270019531250000000000000,  493.16922433770633915628422982990741729736328125000000,  492.33177448130322773067746311426162719726562500000000,  491.35900132728130529358168132603168487548828125000000,  490.26224036013184104376705363392829895019531250000000,  489.05090444517043124506017193198204040527343750000000,
        487.73293574529998295474797487258911132812500000000000,  486.31512549155530678035574965178966522216796875000000,  484.80334677542379040460218675434589385986328125000000,  483.20272814594625288009410724043846130371093750000000,  481.51778575938323001537355594336986541748046875000000,  479.75252579017728749022353440523147583007812500000000,  477.91052504581000448524719104170799255371093750000000,  475.99499530583432260755216702818870544433593750000000,  474.00883530469894822090282104909420013427734375000000,  471.95467319474249734412296675145626068115234375000000,
        469.83490157694996014470234513282775878906250000000000,  467.65170665954411788334255106747150421142578125000000,  465.40709272643806571068125776946544647216796875000000,  463.10290282241726345091592520475387573242187500000000,  460.74083635880339215873391367495059967041015625000000,  458.32246419147696769869071431457996368408203125000000,  455.84924160821577743263333104550838470458984375000000,  453.32251957440377054808777756989002227783203125000000,  450.74355451824476403999142348766326904296875000000000,  448.11351688364118217577924951910972595214843750000000,
        445.43349863722482950834091752767562866210937500000000,  442.70451988297429579688468948006629943847656250000000,  439.92753471143794286035699769854545593261718750000000,  437.10343638932511112216161563992500305175781250000000,  434.23306197800269501385628245770931243896484375000000,  431.31719645539038765491568483412265777587890625000000,  428.35657640423130487761227414011955261230468750000000,  425.35189332021366226399550214409828186035156250000000,  422.30379658554068100784206762909889221191406250000000,  419.21289614698252989910542964935302734375000000000000,
        416.07976493194274780762498266994953155517578125000000,  412.90494103144698101459653116762638092041015625000000,  409.68892967505036040165578015148639678955078125000000,  406.43220501934052890646853484213352203369140625000000,  403.13521176888394847992458380758762359619140625000000,  399.79836664604050611160346306860446929931640625000000,  396.42205972398926405730890110135078430175781250000000,  393.00665563551211789672379381954669952392578125000000,  389.55249466852393425142508931457996368408203125000000,  386.05989375798480978119187057018280029296875000000000,
        382.52914738264263405653764493763446807861328125000000,  378.96052837401947499529342167079448699951171875000000,  375.35428864413319161030813120305538177490234375000000,  371.71065983763736539913224987685680389404296875000000,  368.02985391333567122273962013423442840576171875000000,  364.31206365937828195455949753522872924804687500000000,  360.55746314586178868921706452965736389160156250000000,  356.76620811802047228411538526415824890136718750000000,  352.93843633270989812444895505905151367187500000000000,  349.07426784042962708554114215075969696044921875000000,
        345.17380521471471865879721008241176605224609375000000,  341.23713373032273921126034110784530639648437500000000,  337.26432149126765125402016565203666687011718750000000,  333.25541950938423951811273582279682159423828125000000,  329.21046173374463705840753391385078430175781250000000,  325.12946503089779071160592138767242431640625000000000,  321.01242911553987369188689626753330230712890625000000,  316.85933643086087840856635011732578277587890625000000,  312.67015197743631915727746672928333282470703125000000,  308.44482308913643464620690792798995971679687500000000,
        304.18327915411077810858841985464096069335937500000000,  299.88543127845770186468143947422504425048828125000000,  295.55117188970501729272655211389064788818359375000000,  291.18037427670310535177122801542282104492187500000000,  286.77289206194751614020788110792636871337890625000000,  282.32855860170695905253523960709571838378906250000000,  277.84718630861118526809150353074073791503906250000000,  273.32856589054216556178289465606212615966796875000000,  268.77246549875940218043979257345199584960937500000000,  264.17862977714366934378631412982940673828125000000000,
        259.54677880325516525772400200366973876953125000000000,  254.87660691053287109753000549972057342529296875000000,  250.16778137938243276039429474622011184692382812500000,  245.41994098306813043564034160226583480834960937500000,  240.63269437219571500463644042611122131347656250000000,  235.80561827907808947202283889055252075195312500000000,  230.93825552035113446436298545449972152709960937500000,  226.03011277275265911157475784420967102050781250000000,  221.08065809288825676048872992396354675292968750000000,  216.08931814693616502154327463358640670776367187500000,
        211.05547511041982033930253237485885620117187500000000,  205.97846319117226698836020659655332565307617187500000,  200.85756472015725648816442117094993591308593750000000,  195.69200574452898422350699547678232192993164062500000,  190.48095104475268612986837979406118392944335937500000,  185.22349848216344980755820870399475097656250000000000,  179.91867256423674348297936376184225082397460937500000,  174.56541709104422466225514654070138931274414062500000,  169.16258671651351619402703363448381423950195312500000,  163.70893722036140616182819940149784088134765625000000,
        158.20311423845802778487268369644880294799804687500000,  152.64364013750440562944277189671993255615234375000000,  147.02889863958989735692739486694335937500000000000000,  141.35711669686131131129513960331678390502929687500000,  135.62634297687137063803675118833780288696289062500000,  129.83442213175143820080847945064306259155273437500000,  123.97896376955866060143307549878954887390136718750000,  118.05730469478167776742338901385664939880371093750000,  112.06646249282394478541391436010599136352539062500000,  106.00307783189707322435424430295825004577636718750000,
         99.86334183688623511443438474088907241821289062500000,   93.64290337773904582263639895245432853698730468750000,   87.33674881659258915078680729493498802185058593750000,   80.93904316806872145662055118009448051452636718750000,   74.44291584364370351067918818444013595581054687500000,   67.84016449380459334861370734870433807373046875000000,   61.12083364897434734075432061217725276947021484375000,   54.27259410753481461142655462026596069335937500000000,   47.27978932085054708522875444032251834869384765625000,   40.12189033662316006711989757604897022247314453125000,
         32.77081517480197447866885340772569179534912109375000,   25.18582933410230140225394279696047306060791015625000,   17.30248041616238552364848146680742502212524414062500,    9.00307016981826180312964424956589937210083007812500,    0.00000000000000000000000000000000000000000000000000,  496.40547848721757873136084526777267456054687500000000,  498.09799702833714718508417718112468719482421875000000,  499.04031030638020638434682041406631469726562500000000,  499.52711578860731833628960885107517242431640625000000,  499.68137445334451740563963539898395538330078125000000,
        499.57159735201901185064343735575675964355468750000000,  499.24174690498347217726404778659343719482421875000000,  498.72253631977793020269018597900867462158203125000000,  498.03668297885189986118348315358161926269531250000000,  497.20169429554516682401299476623535156250000000000000,  496.23148711528926924074767157435417175292968750000000,  495.13739532498061635124031454324722290039062500000000,  493.92883061889602913652197457849979400634765625000000,  492.61373433742829774928395636379718780517578125000000,  491.19889717943198093053069896996021270751953125000000,
        489.69019195037526515079662203788757324218750000000000,  488.09274712542094221134902909398078918457031250000000,  486.41107897256580372413736768066883087158203125000000,  484.64919394270714292360935360193252563476562500000000,  482.81066926790771276500890962779521942138671875000000,  480.89871728717565702027059160172939300537109375000000,  478.91623741866430918889818713068962097167968750000000,  476.86585861413993825408397242426872253417968750000000,  474.74997438290489526480087079107761383056640625000000,  472.57077194494382865741499699652194976806640625000000,
        470.33025669507907196020823903381824493408203125000000,  468.03027288481865753055899403989315032958984375000000,  465.67252122550428339309291914105415344238281250000000,  463.25857396452067860082024708390235900878906250000000,  460.78988787143219951758510433137416839599609375000000,  458.26781548302983537723775953054428100585937500000000,  455.69361488836398166313301771879196166992187500000000,  453.06845828187869074099580757319927215576171875000000,  450.39343947109824739527539350092411041259765625000000,  447.66958049227196170249953866004943847656250000000000,
        444.89783746097577932232525199651718139648437500000000,  442.07910576341663499988499097526073455810546875000000,  439.21422467696532976333401165902614593505859375000000,  436.30398149440554789180168882012367248535156250000000,  433.34911521487072150193853303790092468261718750000000,  430.35031985494470063713379204273223876953125000000000,  427.30824742552766792869078926742076873779296875000000,  424.22351061350349255008040927350521087646484375000000,  421.09668520175290495899389497935771942138671875000000,  417.92831225642834169775596819818019866943359375000000,
        414.71890010650071189957088790833950042724609375000000,  411.46892613727214893515338189899921417236328125000000,  408.17883841671834943554131314158439636230468750000000,  404.84905717110899558974779210984706878662109375000000,  401.47997612427070635021664202213287353515625000000000,  398.07196371306798710065777413547039031982421875000000,  394.62536419011871657858137041330337524414062500000000,  391.14049862341420293887495063245296478271484375000000,  387.61766580132871240493841469287872314453125000000000,  384.05714305047172274498734623193740844726562500000000,
        380.45918697292086108063813298940658569335937500000000,  376.82403410856784375937422737479209899902343750000000,  373.15190152758799513321719132363796234130859375000000,  369.44298735740306938168941996991634368896484375000000,  365.69747124792490922118304297327995300292968750000000,  361.91551477834343586437171325087547302246093750000000,  358.09726180824048924478120170533657073974609375000000,  354.24283877537089892939548008143901824951171875000000,  350.35235494203703865423449315130710601806640625000000,  346.42590259160039067865000106394290924072265625000000,
        342.46355717630382287097745575010776519775390625000000,  338.46537741722733017013524658977985382080078125000000,  334.43140535685614622707362286746501922607421875000000,  330.36166636440168531407834962010383605957031250000000,  326.25616909367988682788563892245292663574218750000000,  322.11490539300780255871359258890151977539062500000000,  317.93785016622979355815914459526538848876953125000000,  313.72496118362170136606437154114246368408203125000000,  309.47617884103675578444381244480609893798828125000000,  305.19142586525123306273599155247211456298828125000000,
        300.87060696303103668469702824950218200683593750000000,  296.51360841096357034984976053237915039062500000000000,  292.12029758258182710051187314093112945556640625000000,  287.69052240873099890450248494744300842285156250000000,  283.22411076649183314657420851290225982666015625000000,  278.72086979125873540397151373326778411865234375000000,  274.18058510576594244412262924015522003173828125000000,  269.60301995894417359522776678204536437988281250000000,  264.98791426645038882270455360412597656250000000000000,  260.33498354352673231915105134248733520507812500000000,
        255.64391771947902043393696658313274383544921875000000,  250.91437982148980267993465531617403030395507812500000,  246.14600451364972855117230210453271865844726562500000,  241.33839647496716906971414573490619659423828125000000,  236.49112859762115590456232894212007522583007812500000,  231.60373998380146076669916510581970214843750000000000,  226.67573371602671272739826235920190811157226562500000,  221.70657437174369874810508918017148971557617187500000,  216.69568524814141596834815572947263717651367187500000,  211.64244525729063184371625538915395736694335937500000,
        206.54618544471736640844028443098068237304687500000000,  201.40618507605958598105644341558218002319335937500000,  196.22166722617575373988074716180562973022460937500000,  190.99179379251401655892550479620695114135742187500000,  185.71565983910736008510866668075323104858398437500000,  180.39228715845737838208151515573263168334960937500000,  175.02061691476936289291188586503267288208007812500000,  169.59950120214807611773721873760223388671875000000000,  164.12769331361425884097116068005561828613281250000000,  158.60383646869121321287821047008037567138671875000000,
        153.02645068543588990905846003443002700805664062500000,  147.39391740247347684089618269354104995727539062500000,  141.70446135126073272658686619251966476440429687500000,  135.95612903913877289596712216734886169433593750000000,  130.14676301631496357913420069962739944458007812500000,  124.27397084511731861766747897490859031677246093750000,  118.33508733850131022791174473240971565246582031250000,  112.32712814262268352649698499590158462524414062500000,  106.24673203690389300390961579978466033935546875000000,  100.09008830615209717507241293787956237792968750000000,
         93.85284402727597807825077325105667114257812500000000,   87.52998381481195622200175421312451362609863281250000,   81.11567097968661244067334337159991264343261718750000,   74.60303327211769897076010238379240036010742187500000,   67.98386672236310346306709107011556625366210937500000,   61.24821428029649439395143417641520500183105468750000,   54.38374520213631058140890672802925109863281250000000,   47.37480143422789780061066267080605030059814453125000,   40.20085255520627498526664567179977893829345703125000,   32.83381515076280265930108726024627685546875000000000,
         25.23295331906475524874622351489961147308349609375000,   17.33381329385676394849724601954221725463867187500000,    9.01869548772134344005735329119488596916198730468750,    0.00000000000000000000000000000000000000000000000000,  501.26529089157929774955846369266510009765625000000000,  502.95923520176580723273218609392642974853515625000000,  503.90313928227232054268824867904186248779296875000000,  504.39168111688138651516055688261985778808593750000000,  504.54780868573323004966368898749351501464843750000000,  504.44002393733688904831069521605968475341796875000000,
        504.11228269399475721002090722322463989257812500000000,  503.59529326581809982599224895238876342773437500000000,  502.91176934468927584021002985537052154541015625000000,  502.07921554203494451940059661865234375000000000000000,  501.11154657656692279488197527825832366943359375000000,  500.02009473557905039342585951089859008789062500000000,  498.81427053519553282967535778880119323730468750000000,  497.50201448186874131351942196488380432128906250000000,  496.09011672710204265968059189617633819580078125000000,  494.58444977185507696049171499907970428466796875000000,
        492.99014199591846363546210341155529022216796875000000,  491.31170975475828299749991856515407562255859375000000,  489.55315974885729701782111078500747680664062500000000,  487.71806960551100473821861669421195983886718750000000,  485.80965219144070488255238160490989685058593750000000,  483.83080757442547792379627935588359832763671875000000,  481.78416546927275021516834385693073272705078125000000,  479.67212025492017346550710499286651611328125000000000,  477.49686012212788455144618637859821319580078125000000,  475.26039153331430497928522527217864990234375000000000,
        472.96455990103328304030583240091800689697265625000000,  470.61106718855666031231521628797054290771484375000000,  468.20148698420894106675405055284500122070312500000000,  465.73727748622985700421850197017192840576171875000000,  463.21979274707587137527298182249069213867187500000000,  460.65029245817839864685083739459514617919921875000000,  458.02995050322903125561424531042575836181640625000000,  455.35986246640692343135015107691287994384765625000000,  452.64105224892483647636254318058490753173828125000000,  449.87447792087209563760552555322647094726562500000000,
        447.06103691408316080924123525619506835937500000000000,  444.20157064454576811840524896979331970214843750000000,  441.29686863882966463279444724321365356445312500000000,  438.34767222750190285296412184834480285644531250000000,  435.35467785900533499443554319441318511962890625000000,  432.31854007960151875522569753229618072509765625000000,  429.23987421842150524753378704190254211425781250000000,  426.11925881117292647104477509856224060058593750000000,  422.95723779143014553483226336538791656494140625000000,  419.75432247453090894850902259349822998046875000000000,
        416.51099335578379623257205821573734283447265625000000,  413.22770174187098746187984943389892578125000000000000,  409.90487123191070395478163845837116241455078125000000,  406.54289906256695985575788654386997222900390625000000,  403.14215732980534312446252442896366119384765625000000,  399.70299409833978643291629850864410400390625000000000,  396.22573440846889525346341542899608612060546875000000,  392.71068118882169528660597279667854309082031250000000,  389.15811608250231756755965761840343475341796875000000,  385.56830019321313329783151857554912567138671875000000,
        381.94147475713276662645512260496616363525390625000000,  378.27786174561231291590956971049308776855468750000000,  374.57766440311110045513487420976161956787109375000000,  370.84106772422364883823320269584655761718750000000000,  367.06823887312725673837121576070785522460937500000000,  363.25932754830660087463911622762680053710937500000000,  359.41446629497852427448378875851631164550781250000000,  355.53377076723688787751598283648490905761718750000000,  351.61733994155895288713509216904640197753906250000000,  347.66525628296272998341009952127933502197265625000000,
        343.67758586476065829629078507423400878906250000000000,  339.65437844252858212712453678250312805175781250000000,  335.59566748258629331758129410445690155029296875000000,  331.50147014496587871690280735492706298828125000000000,  327.37178722052306056866655126214027404785156250000000,  323.20660302152077747450675815343856811523437500000000,  319.00588522467421626060968264937400817871093750000000,  314.76958466529731595073826611042022705078125000000000,  310.49763508081605323241092264652252197265625000000000,  306.18995280151631277476553805172443389892578125000000,
        301.84643638596639902971219271421432495117187500000000,  297.46696619808494688186328858137130737304687500000000,  293.05140392231317036930704489350318908691406250000000,  288.59959201278121554423705674707889556884765625000000,  284.11135307172560260369209572672843933105468750000000,  279.58648915170579130062833428382873535156250000000000,  275.02478097536555878832587040960788726806640625000000,  270.42598706557851073739584535360336303710937500000000,  265.78984277778198475061799399554729461669921875000000,  261.11605922511813560049631632864475250244140625000000,
        256.40432208564033089714939706027507781982421875000000,  251.65429027926788307922834064811468124389648437500000,  246.86559450034724250144790858030319213867187500000000,  242.03783558955075250196387059986591339111328125000000,  237.17058272635637194980517961084842681884765625000000,  232.26337142042896743987512309104204177856445312500000,  227.31570127677383652553544379770755767822265625000000,  222.32703350544713316594425123184919357299804687500000,  217.29678814173877299253945238888263702392578125000000,  212.22434093692262990771268960088491439819335937500000,
        207.10901987266751689276134129613637924194335937500000,  201.95010124374420001913676969707012176513671875000000,  196.74680524338378972970531322062015533447265625000000,  191.49829097308494851858995389193296432495117187500000,  186.20365078322350882444879971444606781005859375000000,  180.86190383171688722541148308664560317993164062500000,  175.47198872419642157183261588215827941894531250000000,  170.03275506928741833689855411648750305175781250000000,  164.54295374484954095350985880941152572631835937500000,  159.00122562291767280839849263429641723632812500000000,
        153.40608843921131665410939604043960571289062500000000,  147.75592141276442248454259242862462997436523437500000,  142.04894711589400912998826242983341217041015625000000,  136.28320995506280155495915096253156661987304687500000,  130.45655043576940101957006845623254776000976562500000,  124.56657412980479193720384500920772552490234375000000,  118.61061391184897217954130610451102256774902343750000,  112.58568354021799962083605350926518440246582031250000,  106.48841995518364456074777990579605102539062500000000,  100.31501064941998890844843117520213127136230468750000,
         94.06110095312008922974200686439871788024902343750000,   87.72167377799003418203938053920865058898925781250000,   81.29089077454324296923005022108554840087890625000000,   74.76187807359475812063465127721428871154785156250000,   68.12643012566128675189247587695717811584472656250000,   61.37458833924332424203385016880929470062255859375000,   54.49402046631306717472398304380476474761962890625000,   47.46906698490547427127239643596112728118896484375000,   40.27919604013747800763667328283190727233886718750000,   32.89632281787648082627129042521119117736816406250000,
         25.27971004910482477612276852596551179885864257812500,   17.36490263200737160786957247182726860046386718750000,    9.03419967425730874310829676687717437744140625000000,    0.00000000000000000000000000000000000000000000000000,  506.13282534203489149149390868842601776123046875000000,  507.82818244725797285354929044842720031738281250000000,  508.77366238353488370194099843502044677734375000000000,  509.26392385329955914130550809204578399658203125000000,  509.42190196500462207040982320904731750488281250000000,  509.31608964651638871146133169531822204589843750000000,
        508.99043617639222247817087918519973754882812500000000,  508.47564500278286914181080646812915802001953125000000,  507.79442614910948350370745174586772918701171875000000,  506.96428143708635616349056363105773925781250000000000,  505.99912346327784007371519692242145538330078125000000,  504.91028291386970749954343773424625396728515625000000,  503.70716912033606149634579196572303771972656250000000,  502.39772174447773522842908278107643127441406250000000,  500.98873037608717595503549091517925262451171875000000,  499.48606719402482667646836489439010620117187500000000,
        497.89486046219684567404328845441341400146484375000000,  496.21962660033398151426808908581733703613281250000000,  494.46437253277741774581954814493656158447265625000000,  492.63267625392910531445522792637348175048828125000000,  490.72775112778430184334865771234035491943359375000000,  488.75249783905235290148993954062461853027343750000000,  486.70954683066401003088685683906078338623046875000000,  484.60129331406415076344273984432220458984375000000000,  482.42992641144616072779172100126743316650390625000000,  480.19745361124830651533557102084159851074218750000000,
        477.90572144322760550494422204792499542236328125000000,  475.55643307643168782306020148098468780517578125000000,  473.15116339160249481210485100746154785156250000000000,  470.69137196469824857558705843985080718994140625000000,  468.17841431037169286355492658913135528564453125000000,  465.61355166636639069110970012843608856201171875000000,  462.99795954685623655677773058414459228515625000000000,  460.33273525110496393608627840876579284667968750000000,  457.61890448079708448858582414686679840087890625000000,  454.85742719299287273315712809562683105468750000000000,
        452.04920279442228547850390896201133728027343750000000,  449.19507476561813064108719117939472198486328125000000,  446.29583478936120854996261186897754669189453125000000,  443.35222644639912914499291218817234039306640625000000,  440.36494853191356924071442335844039916992187500000000,  437.33465803833774998565786518156528472900390625000000,  434.26197284357152739175944589078426361083984375000000,  431.14747413814774290585773997008800506591796875000000,  427.99170862028671535881585441529750823974609375000000,  424.79519048386930535343708470463752746582031250000000,
        421.55840322104916140233399346470832824707031250000000,  418.28180125840174241602653637528419494628906250000000,  414.96581144309084265842102468013763427734375000000000,  411.61083439346236900746589526534080505371093750000000,  408.21724572668324526603100821375846862792968750000000,  404.78539717449746149213751778006553649902343750000000,  401.31561759682318779596243984997272491455078125000000,  397.80821390174401130934711545705795288085937500000000,  394.26347187941456695625674910843372344970703125000000,  390.68165695649975077685667201876640319824218750000000,
        387.06301487696435970065067522227764129638671875000000,  383.40777231432133476118906401097774505615234375000000,  379.71613741981127532199025154113769531250000000000000,  375.98830031042064092616783455014228820800781250000000,  372.22443350012650853386730886995792388916015625000000,  368.42469227729583280961378477513790130615234375000000,  364.58921503073378289627726189792156219482421875000000,  360.71812352648464639059966430068016052246093750000000,  356.81152313711947954288916662335395812988281250000000,  352.86950302489924524707021191716194152832031250000000,
        348.89213627987260224472265690565109252929687500000000,  344.87948001365248273941688239574432373046875000000000,  340.83157540930568529802258126437664031982421875000000,  336.74844772748849663912551477551460266113281250000000,  332.63010626865559515863424167037010192871093750000000,  328.47654429086537675175350159406661987304687500000000,  324.28773888238816880402737297117710113525390625000000,  320.06365078800047285767504945397377014160156250000000,  315.80422418750367796747013926506042480468750000000000,  311.50938642464524264141800813376903533935546875000000,
        307.17904768422727102006319910287857055664062500000000,  302.81310061476995088014518842101097106933593750000000,  298.41141989363177344785071909427642822265625000000000,  293.97386173098550443683052435517311096191406250000000,  289.50026330848277211771346628665924072265625000000000,  284.99044214781372375000501051545143127441406250000000,  280.44419540366175169765483587980270385742187500000000,  275.86129907475742584210820496082305908203125000000000,  271.24150712583110589548596180975437164306640625000000,  266.58455051223182863395777530968189239501953125000000,
        261.89013609779874514060793444514274597167968750000000,  257.15794545521259806264424696564674377441406250000000,  252.38763353648184306621260475367307662963867187500000,  247.57882719939593130220600869506597518920898437500000,  242.73112357365272373499465174973011016845703125000000,  237.84408824788161496144311968237161636352539062500000,  232.91725325586222083984466735273599624633789062500000,  227.95011483679093089449452236294746398925781250000000,  222.94213094036197730929416138678789138793945312500000,  217.89271844256271037920669186860322952270507812500000,
        212.80125003226294211344793438911437988281250000000000,  207.66705072167758316936669871211051940917968750000000,  202.48939392532517445033590774983167648315429687500000,  197.26749704182580558153858873993158340454101562500000,  192.00051646032380858741817064583301544189453125000000,  186.68754189787955510837491601705551147460937500000000,  181.32758995507219879073090851306915283203125000000000,  175.91959675325753664765215944498777389526367187500000,  170.46240948707298912268015556037425994873046875000000,  164.95477668803331994240579660981893539428710937500000,
        159.39533694695094823146064300090074539184570312500000,  153.78260578104143974087492097169160842895507812500000,  148.11496025125953224232944194227457046508789062500000,  142.39062083007848968918551690876483917236328125000000,  136.60762988026218067716399673372507095336914062500000,  130.76382591775754349328053649514913558959960937500000,  124.85681257704241886585805332288146018981933593750000,  118.88392084589659702942299190908670425415039062500000,  112.84216264440311761063640005886554718017578125000000,  106.72817312159703817542322212830185890197753906250000,
        100.53813802431268698001076700165867805480957031250000,   94.26770098076868009684403659775853157043457031250000,   87.91184324309453756995935691520571708679199218750000,   81.46472484321883200664160540327429771423339843750000,   74.91947033301511282843421213328838348388671875000000,   68.26787262240895870490930974483489990234375000000000,   61.49997161641962861722277011722326278686523437500000,   54.60343359945726859905335004441440105438232421875000,   47.56259761677100073029578197747468948364257812500000,   40.35693041437038885987931280396878719329833984375000,
         32.95834581163928334035517764277756214141845703125000,   25.32610520468686488015919167082756757736206054687500,   17.39575218746092133415004354901611804962158203125000,    9.04958459309678886484107351861894130706787109375000,    0.00000000000000000000000000000000000000000000000000,  511.00802266523595562830450944602489471435546875000000,  512.70477981037686276977183297276496887207031250000000,  513.65182091401743491587694734334945678710937500000000,  514.14378559590988970739999786019325256347656250000000,  514.30359621731656716292491182684898376464843750000000,
        514.19973676657787109434138983488082885742187500000000,  513.87615003218456877220887690782546997070312500000000,  513.36353463552825360238784924149513244628906250000000,  512.68459695366675532568478956818580627441406250000000,  511.85683603099153060611570253968238830566406250000000,  510.89416234683648099235142581164836883544921875000000,  509.80790498524748954878305085003376007080078125000000,  508.60747208713581812844495289027690887451171875000000,  507.30080245962204799070605076849460601806640625000000,  505.89468511720053811586694791913032531738281250000000,
        504.39499189989760452590417116880416870117187500000000,  502.80685093615807090827729552984237670898437500000000,  501.13477868778579704667208716273307800292968750000000,  499.38278227834274503038614057004451751708984375000000,  497.55444004237591570927179418504238128662109375000000,  495.65296581195724456847528927028179168701171875000000,  493.68126085735070773807819932699203491210937500000000,  491.64195631609209158341400325298309326171875000000000,  489.53744819648699149183812551200389862060546875000000,  487.36992651438606571900891140103340148925781250000000,
        485.14139974432606550180935300886631011962890625000000,  482.85371549117087397462455555796623229980468750000000,  480.50857808543224791719694621860980987548828125000000,  478.10756365369115883368067443370819091796875000000000,  475.65213310071635532949585467576980590820312500000000,  473.14364335204845701809972524642944335937500000000000,  470.58335713795298715922399424016475677490234375000000,  467.97245154672253875105525366961956024169921875000000,  465.31202553366932761491625569760799407958984375000000,  462.60310653912955558553221635520458221435546875000000,
        459.84665634241207499144366011023521423339843750000000,  457.04357625738612114218994975090026855468750000000000,  454.19471175819944619433954358100891113281250000000000,  451.30085660958707194367889314889907836914062500000000,  448.36275656473054596062866039574146270751953125000000,  445.38111268413661036902340129017829895019531250000000,  442.35658432114013294267351739108562469482421875000000,  439.28979181307886392460204660892486572265625000000000,  436.18131891170111202882253564894199371337890625000000,  433.03171498174725684293662197887897491455078125000000,
        429.84149699274814793170662596821784973144531250000000,  426.61115132576946962217334657907485961914062500000000,  423.34113541401438851607963442802429199218750000000000,  420.03187923378135337770800106227397918701171875000000,  416.68378666020237233169609680771827697753906250000000,  413.29723670040249317025882191956043243408203125000000,  409.87258461517239993554539978504180908203125000000000,  406.41016293890527322218986228108406066894531250000000,  402.91028240637638191401492804288864135742187500000000,  399.37323279391921460046432912349700927734375000000000,
        395.79928368164934227024787105619907379150390625000000,  392.18868514259077073802473023533821105957031250000000,  388.54166836385530814368394203484058380126953125000000,  384.85844620439286245527910068631172180175781250000000,  381.13921369326880039807292632758617401123046875000000,  377.38414847191563694650540128350257873535156250000000,  373.59341118334367592979106120765209197998046875000000,  369.76714581087651367852231487631797790527343750000000,  365.90547996858794022045913152396678924560546875000000,  362.00852514525695369229651987552642822265625000000000,
        358.07637690332165902873384766280651092529296875000000,  354.10911503399341881959117017686367034912109375000000,  350.10680366938606766780139878392219543457031250000000,  346.06949135222123459243448451161384582519531250000000,  341.99721106338137133207055740058422088623046875000000,  337.88998020729258087158086709678173065185546875000000,  333.74780055483466867372044362127780914306640625000000,  329.57065814317729746107943356037139892578125000000000,  325.35852313164332372252829372882843017578125000000000,  321.11134961238082041745656169950962066650390625000000,
        316.82907537429514377436134964227676391601562500000000,  312.51162161833525487963925115764141082763671875000000,  308.15889262184816743683768436312675476074218750000000,  303.77077534929884450320969335734844207763671875000000,  299.34713900619846072004293091595172882080078125000000,  294.88783453258230338178691454231739044189453125000000,  290.39269403181992856843862682580947875976562500000000,  285.86153012991655941732460632920265197753906250000000,  281.29413525976394794270163401961326599121093750000000,  276.69028086400379606857313774526119232177734375000000,
        272.04971650926881920895539224147796630859375000000000,  267.37216890353420239989645779132843017578125000000000,  262.65734080713701814602245576679706573486328125000000,  257.90490982666153740865411236882209777832031250000000,  253.11452707931934469343104865401983261108398437500000,  248.28581571363397983986942563205957412719726562500000,  243.41836927011351576766173820942640304565429687500000,  238.51174986311315251441556029021739959716796875000000,  233.56548616216875302598054986447095870971679687500000,  228.57907114763588651840109378099441528320312500000000,
        223.55195961138485927222063764929771423339843750000000,  218.48356536843678554760117549449205398559570312500000,  213.37325813960620735088014043867588043212890625000000,  208.22036005821664161885564681142568588256835937500000,  203.02414174549988956641755066812038421630859375000000,  197.78381788901106119737960398197174072265625000000000,  192.49854224583441464346833527088165283203125000000000,  187.16740197691484581810072995722293853759765625000000,  181.78941119974740558973280712962150573730468750000000,  176.36350362286086124186113011091947555541992187500000,
        170.88852409567942913781735114753246307373046875000000,  165.36321886959871108047082088887691497802734375000000,  159.78622431800312142513575963675975799560546875000000,  154.15605380107851374305028002709150314331054687500000,  148.47108228095990511974378023296594619750976562500000,  142.72952818742089675652096047997474670410156250000000,  136.92943189464898523510782979428768157958984375000000,  131.06862998223022032107110135257244110107421875000000,  125.14472419867199448617611778900027275085449218750000,  119.15504369442921017707703867927193641662597656250000,
        113.09659859923530689229664858430624008178710937500000,  106.96602231715236541731428587809205055236816406250000,  100.75949889388733993200730765238404273986816406250000,   94.47267029891021650200855219736695289611816406250000,   88.10051616657509043761820066720247268676757812500000,   81.63719495066045794828824000433087348937988281250000,   75.07582966322281947668670909479260444641113281250000,   68.40821171146355084147216985002160072326660156250000,   61.62437953358053022157037048600614070892333984375000,   54.71199798192282059972058050334453582763671875000000,
         47.65540470334138944963342510163784027099609375000000,   40.43406507805645588859988492913544178009033203125000,   33.01989159125830042285087984055280685424804687500000,   25.37214433548440695176395820453763008117675781250000,   17.42636563079627975980656628962606191635131835937500,    9.06485206522757636093956534750759601593017578125000,    0.00000000000000000000000000000000000000000000000000,  515.89082458782240792061202228069305419921875000000000,  517.58896923131931089301360771059989929199218750000000,  518.53755706575725525908637791872024536132812500000000,
        519.03120882349344356043729931116104125976562500000000,  519.19283424113734781712992116808891296386718750000000,  519.09090844748061499558389186859130859375000000000000,  518.76936779400443811027798801660537719726562500000000,  518.25890611027000431931810453534126281738281250000000,  517.58222614905025693587958812713623046875000000000000,  516.75682418997416789352428168058395385742187500000000,  515.79660860038359260215656831860542297363281250000000,  514.71290686160000404925085604190826416015625000000000,  513.51512591861330747633473947644233703613281250000000,
        512.21120371448375863110413774847984313964843750000000,  510.80792867555538805390824563801288604736328125000000,  509.31117228710076005882001481950283050537109375000000,  507.72606252341842036912566982209682464599609375000000,  506.05711586715989369622548110783100128173828125000000,  504.30833961750875005236594006419181823730468750000000,  502.48331242330772283821715973317623138427734375000000,  500.58524855669230646526557393372058868408203125000000,  498.61704984336478219120181165635585784912109375000000,  496.58134808328185272330301813781261444091796875000000,
        494.48054004737599598229280672967433929443359375000000,  492.31681660887267071302630938589572906494140625000000,  490.09218719006241826718905940651893615722656250000000,  487.80850043048394582001492381095886230468750000000000,  485.46746177955668599679484032094478607177734375000000,  483.07064856497208893415518105030059814453125000000000,  480.61952297335238881714758463203907012939453125000000,  478.11544329187324819940840825438499450683593750000000,  475.55967369169826497454778291285037994384765625000000,  472.95339278115898196119815111160278320312500000000000,
        470.29770111498692131135612726211547851562500000000000,  467.59362781288854193917359225451946258544921875000000,  464.84213641437469277661875821650028228759765625000000,  462.04413007552170711278449743986129760742187500000000,  459.20045619614097631711047142744064331054687500000000,  456.31191055181062665724311955273151397705078125000000,  453.37924099372054342893534339964389801025390625000000,  450.40315076979817376923165284097194671630859375000000,  447.38430151271882095898035913705825805664062500000000,  444.32331593385009682606323622167110443115234375000000,
        441.22078025669532053143484517931938171386718750000000,  438.07724641878314741916256025433540344238281250000000,  434.89323406705432262242538854479789733886718750000000,  431.66923236848521128194988705217838287353515625000000,  428.40570165487184794983477331697940826416015625000000,  425.10307491828558568158769048750400543212890625000000,  421.76175917164232487266417592763900756835937500000000,  418.38213668704344172510900534689426422119140625000000,  414.96456612300141841842560097575187683105468750000000,  411.50938355032349136308766901493072509765625000000000,
        408.01690338525810375358560122549533843994140625000000,  404.48741923748514182079816237092018127441406250000000,  400.92120467963201235761516727507114410400390625000000,  397.31851394420675660512642934918403625488281250000000,  393.67958255313277504683355800807476043701171875000000,  390.00462788444826855993596836924552917480468750000000,  386.29384968017018309183185920119285583496093750000000,  382.54743049882097238878486678004264831542968750000000,  378.76553611565907431213418021798133850097656250000000,  374.94831587323960775393061339855194091796875000000000,
        371.09590298454969570229877717792987823486328125000000,  367.20841479061050449672620743513107299804687500000000,  363.28595297410942066562711261212825775146484375000000,  359.32860373031365952556370757520198822021484375000000,  355.33643789622499298275215551257133483886718750000000,  351.30951103864629203599179163575172424316406250000000,  347.24786350155750369594898074865341186523437500000000,  343.15152041292344620160292834043502807617187500000000,  339.02049165078398118566838093101978302001953125000000,  334.85477176820177191984839737415313720703125000000000,
        330.65433987635907442381721921265125274658203125000000,  326.41915948480448150803567841649055480957031250000000,  322.14917829754324429814005270600318908691406250000000,  317.84432796333970827618031762540340423583984375000000,  313.50452377825376970577053725719451904296875000000000,  309.12966433805610222407267428934574127197265625000000,  304.71963113776018872158601880073547363281250000000000,  300.27428811505620842581265605986118316650390625000000,  295.79348113393803032522555440664291381835937500000000,  291.27703740425897649402031674981117248535156250000000,
        286.72476483233185717836022377014160156250000000000000,  282.13645129699261815403588116168975830078125000000000,  277.51186384475522572756744921207427978515625000000000,  272.85074779678757295187097042798995971679687500000000,  268.15282575941392906315741129219532012939453125000000,  263.41779652867023742146557196974754333496093750000000,  258.64533387808580755518050864338874816894531250000000,  253.83508521729717699599859770387411117553710937500000,  248.98667010727947968007356394082307815551757812500000,  244.09967861586096660175826400518417358398437500000000,
        239.17366949470201120675483252853155136108398437500000,  234.20816815600284144238685257732868194580078125000000,  229.20266442375842075307446066290140151977539062500000,  224.15661003029578068890259601175785064697265625000000,  219.06941582396484591299667954444885253906250000000000,  213.94044864803535688224656041711568832397460937500000,  208.76902784385478639705979730933904647827148437500000,  203.55442132286580658728780690580606460571289062500000,  198.29584114180590859177755191922187805175781250000000,  192.99243850285395751598116476088762283325195312500000,
        187.64329808504982111116987653076648712158203125000000,  182.24743159421038285472604911774396896362304687500000,  176.80377039477116341004148125648498535156250000000000,  171.31115705712909402791410684585571289062500000000000,  165.76833561631661950741545297205448150634765625000000,  160.17394028972711339520174078643321990966796875000000,  154.52648233974011304781015496701002120971679687500000,  148.82433468677979249150666873902082443237304687500000,  143.06571377300878111782367341220378875732421875000000,  137.24865803719600876320328097790479660034179687500000,
        131.37100217387728662288282066583633422851562500000000,  125.43034609513388488721830071881413459777832031250000,  119.42401716194913774415908847004175186157226562500000,  113.34902375994134615666553145274519920349121093750000,  107.20199759288354357522621285170316696166992187500000,  100.97912104855356574262259528040885925292968750000000,   94.67603447944703987104730913415551185607910156250000,   88.28771594255137244999787071719765663146972656250000,   81.80832235259407525518327020108699798583984375000000,   75.23097521965677003663586219772696495056152343750000,
         68.54746448485262533267814433202147483825683593750000,   61.74782715503572916304619866423308849334716796875000,   54.81972668485899902179880882613360881805419921875000,   47.74749935607119510905249626375734806060791015625000,   40.51060921537164460914937080815434455871582031250000,   33.08096744503755815003387397155165672302246093750000,   25.41783286436508859651439706794917583465576171875000,   17.45674654894576605101974564604461193084716796875000,    9.08000387024817889880523580359295010566711425781250,    0.00000000000000000000000000000000000000000000000000,
        520.78117371604423624376067891716957092285156250000000,  522.48069352470884041395038366317749023437500000000000,  523.43081389898270572302863001823425292968750000000000,  523.92613687581297199358232319355010986328125000000000,  524.08955968777399903046898543834686279296875000000000,  523.98954868296686981921084225177764892578125000000000,  523.67003382830205282516544684767723083496093750000000,  523.16170419616412345931166782975196838378906250000000,  522.48725893708740386500721797347068786621093750000000,  521.66419157865254874195670709013938903808593750000000,
        520.70640838174938380689127370715141296386718750000000,  519.62523522481114923721179366111755371093750000000000,  518.43007785207316828746115788817405700683593750000000,  517.12887333377227605524240061640739440917968750000000,  515.72840949594933590560685843229293823242187500000000,  514.23455745398098315490642562508583068847656250000000,  512.65244501018446499074343591928482055664062500000000,  510.98658864775501342592178843915462493896484375000000,  509.24099581889055343708605505526065826416015625000000,  507.41924546194013601052574813365936279296875000000000,
        505.52455226222053852325188927352428436279296875000000,  503.55981857195678230709745548665523529052734375000000,  501.52767682261605841631535440683364868164062500000000,  499.43052451487136522700893692672252655029296875000000,  497.27055334446436063444707542657852172851562500000000,  495.04977364459819000330753624439239501953125000000000,  492.77003505064033106464194133877754211425781250000000,  490.43304409003377486442332156002521514892578125000000,  488.04037924861466990478220395743846893310546875000000,  485.59350394975854214862920343875885009765625000000000,
        483.09377779498009886083309538662433624267578125000000,  480.54246634677838301286101341247558593750000000000000,  477.94074968161618244266719557344913482666015625000000,  475.28972989930309722694801166653633117675781250000000,  472.59043774204582177844713442027568817138671875000000,  469.84383845005334023881005123257637023925781250000000,  467.05083695935650212049949914216995239257812500000000,  464.21228253030670884982100687921047210693359375000000,  461.32897288119619361168588511645793914794921875000000,  458.40165788994596596239716745913028717041015625000000,
        455.43104291732390720426337793469429016113281250000000,  452.41779179729638826756854541599750518798828125000000,  449.36252953356438410992268472909927368164062500000000,  446.26584473585069190448848530650138854980468750000000,  443.12829182489230106511968187987804412841796875000000,  439.95039303119546048037591390311717987060546875000000,  436.73264020930122342178947292268276214599609375000000,  433.47549648649697928703972138464450836181640625000000,  430.17939776249744454617029987275600433349609375000000,  426.84475407455158801894867792725563049316406250000000,
        423.47195084065009496043785475194454193115234375000000,  420.06134999196513035712996497750282287597656250000000,  416.61329100431692040729103609919548034667968750000000,  413.12809183729405049234628677368164062500000000000000,  409.60604978863494807228562422096729278564453125000000,  406.04744227058137084895861335098743438720703125000000,  402.45252751412402858477435074746608734130859375000000,  398.82154520636316874515614472329616546630859375000000,  395.15471706558116693486226722598075866699218750000000,  391.45224735807153138011926785111427307128906250000000,
        387.71432336026720122390543110668659210205078125000000,  383.94111576926007955989916808903217315673828125000000,  380.13277906439390108062070794403553009033203125000000,  376.28945182223668552978779189288616180419921875000000,  372.41125698689234013727400451898574829101562500000000,  368.49830209728952468140050768852233886718750000000000,  364.55067947278342899153358303010463714599609375000000,  360.56846635811854184794356115162372589111328125000000,  356.55172502852900606740149669349193572998046875000000,  352.50050285548280726288794539868831634521484375000000,
        348.41483233332053259800886735320091247558593750000000,  344.29473106677659188790130428969860076904296875000000,  340.14020171911357692806632257997989654541015625000000,  335.95123192033599934802623465657234191894531250000000,  331.72779413467657150249578990042209625244140625000000,  327.46984548626579680785653181374073028564453125000000,  323.17732754159794694714946672320365905761718750000000,  318.85016604708772547382977791130542755126953125000000,  314.48827061967188001290196552872657775878906250000000,  310.09153438804071356571512296795845031738281250000000,
        305.65983358167920869163936004042625427246093750000000,  301.19302706445387229905463755130767822265625000000000,  296.69095580898834896288462914526462554931640625000000,  292.15344230752106113868649117648601531982421875000000,  287.58028991432121301841107197105884552001953125000000,  282.97128211404623243652167730033397674560546875000000,  278.32618170963496595504693686962127685546875000000000,  273.64472992243628368669305928051471710205078125000000,  268.92664539624996677957824431359767913818359375000000,  264.17162309578117174169165082275867462158203125000000,
        259.37933308865734716164297424256801605224609375000000,  254.54941919859123800051747821271419525146484375000000,  249.68149751545598746815812774002552032470703125000000,  244.77515474591788802172231953591108322143554687500000,  239.82994638579143042989016976207494735717773437500000,  234.84539469236446507238724734634160995483398437500000,  229.82098643149694794374227058142423629760742187500000,  224.75617037021444843958306591957807540893554687500000,  219.65035448065498258074512705206871032714843750000000,  214.50290281540924297587480396032333374023437500000000,
        209.31313200729840673375292681157588958740234375000000,  204.08030733817713553435169160366058349609375000000000,  198.80363831107527516905975062400102615356445312500000,  193.48227364743340217501099687069654464721679687500000,  188.11529561575048319355119019746780395507812500000000,  182.70171357885837437606824096292257308959960937500000,  177.24045662324624572647735476493835449218750000000000,  171.73036510400160636891087051481008529663085937500000,  166.17018090119387352388002909719944000244140625000000,  160.55853613541313507084851153194904327392578125000000,
        154.89394002830687213645433075726032257080078125000000,  149.17476351364368269969418179243803024291992187500000,  143.39922109910028780177526641637086868286132812500000,  137.56534933930623765263590030372142791748046875000000,  131.67098109326082067127572372555732727050781250000000,  125.71371448444274676603527041152119636535644531250000,  119.69087513056928173682535998523235321044921875000000,  113.59946971779952207270980579778552055358886718750000,  107.43612829278848153080616611987352371215820312500000,  101.19703162713258848270925227552652359008789062500000,
         94.87781849673616818563459673896431922912597656250000,   88.47346542029390548123046755790710449218750000000000,   81.97812781130024006870371522381901741027832031250000,   75.38492571447420687036355957388877868652343750000000,   68.68564764029524383204261539503931999206542968750000,   61.87032919861621138579721446149051189422607421875000,   54.92663247966801520760782295837998390197753906250000,   47.83889243234438737317759660072624683380126953125000,   40.58657180108358630832299240864813327789306640625000,   33.14158049555993557078181765973567962646484375000000,
         25.46317609122495895235260832123458385467529296875000,   17.48689844771765677933217375539243221282958984375000,    9.09504174761271855231825611554086208343505859375000,    0.00000000000000000000000000000000000000000000000000,  525.67901351599505233025411143898963928222656250000000,  527.37989635999269921740051358938217163085937500000000,  528.33153532271114727336680516600608825683593750000000,  528.82851393444559562340145930647850036621093750000000,  528.99371704247528214182239025831222534179687500000000,  528.89560229196649743244051933288574218750000000000000,
        528.57809331708699573937337845563888549804687500000000,  528.07187446741761505109025165438652038574218750000000,  527.39964131325973539787810295820236206054687500000000,  526.57888464299730912898667156696319580078125000000000,  525.62350861688810255145654082298278808593750000000000,  524.54483751071222741302335634827613830566406250000000,  523.35227586361463636421831324696540832519531250000000,  522.05375986483170436258660629391670227050781250000000,  520.65607672861494847893482074141502380371093750000000,  519.16509718604254430829314514994621276855468750000000,
        517.58594885042771238659042865037918090820312500000000,  515.92314818608440418756799772381782531738281250000000,  514.18070277656295274937292560935020446777343750000000,  512.36219182594288668042281642556190490722656250000000,  510.47083040691865107874036766588687896728515625000000,  508.50952137046903089867555536329746246337890625000000,  506.48089774987641931147663854062557220458984375000000,  504.38735774395394173552631400525569915771484375000000,  502.23109383745929790165973827242851257324218750000000,  500.01411723910115370017592795193195343017578125000000,
        497.73827854274230730879935435950756072998046875000000,  495.40528531455879601708147674798965454101562500000000,  493.01671715724546629644464701414108276367187500000000,  490.57403868760025034134741872549057006835937500000000,  488.07861077604405863894498907029628753662109375000000,  485.53170032881092765819630585610866546630859375000000,  482.93448884065321635716827586293220520019531250000000,  480.28807990429424989997642114758491516113281250000000,  477.59350582986365907345316372811794281005859375000000,  474.85173350117958079863456077873706817626953125000000,
        472.06366957452098631620174273848533630371093750000000,  469.23016510833912207090179435908794403076171875000000,  466.35201969834173496565199457108974456787109375000000,  463.42998518088916171109303832054138183593750000000000,  460.46476895816044816456269472837448120117187500000000,  457.45703699069065351068275049328804016113281250000000,  454.40741649633139331854181364178657531738281250000000,  451.31649838920321826662984676659107208251953125000000,  448.18483948759893564783851616084575653076171875000000,  445.01296451589956859606900252401828765869140625000000,
        441.80136792226227271385141648352146148681640625000000,  438.55051553102163097719312645494937896728515625000000,  435.26084604634132801947998814284801483154296875000000,  431.93277242158603712596232071518898010253906250000000,  428.56668310710170999300316907465457916259765625000000,  425.16294318755444692214950919151306152343750000000000,  421.72189541864048578645451925694942474365234375000000,  418.24386117181626332239829935133457183837890625000000,  414.72914129467835664399899542331695556640625000000000,  411.17801689373055751275387592613697052001953125000000,
        407.59075004548742526822024956345558166503906250000000,  403.96758444116653663513716310262680053710937500000000,  400.30874596960364897313411347568035125732421875000000,  396.61444324247162285246304236352443695068359375000000,  392.88486806538793416621047072112560272216796875000000,  389.12019585804966936848359182476997375488281250000000,  385.32058602612795539243961684405803680419921875000000,  381.48618228728406620575697161257266998291015625000000,  377.61711295332946747294045053422451019287109375000000,  373.71349117023368080481304787099361419677734375000000,
        369.77541511739252655388554558157920837402343750000000,  365.80296816728719022648874670267105102539062500000000,  361.79621900640159992690314538776874542236328125000000,  357.75522171800696469290414825081825256347656250000000,  353.68001582717653263898682780563831329345703125000000,  349.57062630814380099764093756675720214843750000000000,  345.42706355387429084657924249768257141113281250000000,  341.24932330747213882204960100352764129638671875000000,  337.03738655478770169793278910219669342041015625000000,  332.79121937733248159929644316434860229492187500000000,
        328.51077276432897633640095591545104980468750000000000,  324.19598238243503374178544618189334869384765625000000,  319.84676830137061642744811251759529113769531250000000,  315.46303467333848402631701901555061340332031250000000,  311.04466936376968533295439556241035461425781250000000,  306.59154353052025498982402496039867401123046875000000,  302.10351114821060036774724721908569335937500000000000,  297.58040847390560657004243694245815277099609375000000,  293.02205344979159917784272693097591400146484375000000,  288.42824503788875745158293284475803375244140625000000,
        283.79876248114942427491769194602966308593750000000000,  279.13336448450655780106899328529834747314453125000000,  274.43178830854310490394709631800651550292968750000000,  269.69374876743449931382201611995697021484375000000000,  264.91893712163994223374174907803535461425781250000000,  260.10701985447121842298656702041625976562500000000000,  255.25763732010059925414680037647485733032226562500000,  250.37040224875659077952150255441665649414062500000000,  245.44489809273397895594825968146324157714843750000000,  240.48067719436835432134103029966354370117187500000000,
        235.47725875420627517087268643081188201904296875000000,  230.43412657416214983641111757606267929077148437500000,  225.35072654636866218424984253942966461181640625000000,  220.22646385356873111049935687333345413208007812500000,  215.06069984107656978267186786979436874389648437500000,  209.85274851334244772260717581957578659057617187500000,  204.60187259969870865461416542530059814453125000000000,  199.30727912359196807301486842334270477294921875000000,  193.96811439704731583333341404795646667480468750000000,  188.58345834667596818690071813762187957763671875000000,
        183.15231805843291112978477030992507934570312500000000,  177.67362040454113980558759067207574844360351562500000,  172.14620358614263295748969539999961853027343750000000,  166.56880738749637771434208843857049942016601562500000,  160.94006188942915969164459966123104095458984375000000,  155.25847432787853108493436593562364578247070312500000,  149.52241370305043233202013652771711349487304687500000,  143.73009263938379831415659282356500625610351562500000,  137.87954585685221786661713849753141403198242187500000,  131.96860442671254531887825578451156616210937500000000,
        125.99486473001798003679141402244567871093750000000000,  119.95565068584799917061900487169623374938964843750000,  113.84796732404652175318915396928787231445312500000000,  107.66844307587214757404581177979707717895507812500000,  101.41325713709747446955589111894369125366210937500000,   95.07804674608483708198036765679717063903808593750000,   88.65778692103002356361685087904334068298339843750000,   82.14663161078328812436666339635848999023437500000000,   75.53769943013283238997246371582150459289550781250000,   68.82277749324478577364061493426561355590820312500000,
         61.99190004622335692374690552242100238800048828125000,   55.03272784710361520410515367984771728515625000000000,   47.92959454316364542592054931446909904479980468750000,   40.66196160687092486796245793811976909637451171875000,   33.20173770467420126806246116757392883300781250000000,   25.50817919667891331414466549176722764968872070312500,   17.51682475422429874356566870119422674179077148437500,    9.10996739782939535245986917288973927497863769531250,    0.00000000000000000000000000000000000000000000000000,  530.58428829443346330663189291954040527343750000000000,
        532.28652224242091506312135607004165649414062500000000,  533.23966607592012678651371970772743225097656250000000,  533.73828500418130715843290090560913085937500000000000,  533.90525160608751775725977495312690734863281250000000,  533.80901490054270652763079851865768432617187500000000,  533.49349224019476878311252221465110778808593750000000,  532.98936328591128130938159301877021789550781250000000,  532.31932004971304195350967347621917724609375000000000,  531.50085059376363005867460742592811584472656250000000,  530.54785698376963409828022122383117675781250000000000,
        529.47166189346796727477340027689933776855468750000000,  528.28166865305036026256857439875602722167968750000000,  526.98581256313252652034861966967582702636718750000000,  525.59088021532443235628306865692138671875000000000000,  524.10274194271107717213453724980354309082031250000000,  522.52652515335125826823059469461441040039062500000000,  520.86674627409263393928995355963706970214843750000000,  519.12741299907986558537231758236885070800781250000000,  517.31210477531294600339606404304504394531250000000000,  515.42403703809145554259885102510452270507812500000000,
        513.46611311047354320180602371692657470703125000000000,  511.44096659902174906164873391389846801757812500000000,  509.35099637031646579998778179287910461425781250000000,  507.19839566592793289601104333996772766113281250000000,  504.98517653603164490050403401255607604980468750000000,  502.71319049709450155205558985471725463867187500000000,  500.38414611625609040856943465769290924072265625000000,  497.99962407338472303308662958443164825439453125000000,  495.56109013705588495213305577635765075683593750000000,  493.06990640293861360987648367881774902343750000000000,
        490.52734107527049900454585440456867218017578125000000,  487.93457701921892066820873878896236419677734375000000,  485.29271927032726807738072238862514495849609375000000,  482.60280165425109544230508618056774139404296875000000,  479.86579264362586627612472511827945709228515625000000,  477.08260055769102336853393353521823883056640625000000,  474.25407819310646573285339400172233581542968750000000,  471.38102696038362182662240229547023773193359375000000,  468.46420058886360493488609790802001953125000000000000,  465.50430845369538701561396010220050811767578125000000,
        462.50201857041389530422748066484928131103515625000000,  459.45796029616786881888401694595813751220703125000000,  456.37272677117056218776269815862178802490234375000000,  453.24687712933302918827394023537635803222656250000000,  450.08093850315003692230675369501113891601562500000000,  446.87540784460207987649482674896717071533203125000000,  443.63075358102543077620794065296649932861328125000000,  440.34741712249763168074423447251319885253906250000000,  437.02581423521843362323124893009662628173828125000000,  433.66633629358977941592456772923469543457031250000000,
        430.26935142216029817063827067613601684570312500000000,  426.83520553726305024611065164208412170410156250000000,  423.36422329701554190251044929027557373046875000000000,  419.85670896733375911935581825673580169677734375000000,  416.31294721071998310435446910560131072998046875000000,  412.73320380380016558774514123797416687011718750000000,  409.11772628889264069584896788001060485839843750000000,  405.46674456427263066871091723442077636718750000000000,  401.78047141724903212889330461621284484863281250000000,  398.05910300367673926302813924849033355712890625000000,
        394.30281927708415423694532364606857299804687500000000,  390.51178437019569855692679993808269500732421875000000,  386.68614693126107795251300558447837829589843750000000,  382.82604041726943933099391870200634002685546875000000,  378.93158334581556800912949256598949432373046875000000,  375.00287950709667939008795656263828277587890625000000,  371.04001813724659086801693774759769439697265625000000,  367.04307405395672958547947928309440612792968750000000,  363.01210775508593542326707392930984497070312500000000,  358.94716548072204886921099387109279632568359375000000,
        354.84827923892294165852945297956466674804687500000000,  350.71546679513107847014907747507095336914062500000000,  346.54873162502320838029845617711544036865234375000000,  342.34806283031639395630918443202972412109375000000000,  338.11343501680937606579391285777091979980468750000000,  333.84480813368185181388980709016323089599609375000000,  329.54212727280781791705521754920482635498046875000000,  325.20532242655355048555065877735614776611328125000000,  320.83430820222793045104481279850006103515625000000000,  316.42898349102034671886940486729145050048828125000000,
        311.98923108890539879212155938148498535156250000000000,  307.51491726659531877885456196963787078857421875000000,  303.00589128518714687743340618908405303955078125000000,  298.46198485366505792626412585377693176269531250000000,  293.88301152387583670133608393371105194091796875000000,  289.26876601798574029089650139212608337402343750000000,  284.61902348273576990322908386588096618652343750000000,  279.93353866403379015537211671471595764160156250000000,  275.21204499452733216458000242710113525390625000000000,  270.45425358578512486928957514464855194091796875000000,
        265.65985211554226452790317125618457794189453125000000,  260.82850359911577697857865132391452789306640625000000,  255.95984503253521324950270354747772216796875000000000,  251.05348589311694240677752532064914703369140625000000,  246.10900648109566191124031320214271545410156250000000,  241.12595608344540210055129136890172958374023437500000,  236.10385093810910461797902826219797134399414062500000,  231.04217197341412770583701785653829574584960937500000,  225.94036229336882115603657439351081848144531250000000,  220.79782437467682143505953717976808547973632812500000,
        215.61391693548605985597532708197832107543945312500000,  210.38795142889799194563238415867090225219726562500000,  205.11918810580496597140154335647821426391601562500000,  199.80683158135218491224804893136024475097656250000000,  194.45002582676303859443578403443098068237304687500000,  189.04784849283061021196772344410419464111328125000000,  183.59930445227632844762410968542098999023437500000000,  178.10331842438498028968751896172761917114257812500000,  172.55872651547073814981558825820684432983398437500000,  166.96426647098837747762445360422134399414062500000000,
        161.31856638699082395760342478752136230468750000000000,  155.62013156676695757596462499350309371948242187500000,  149.86732912817743113009782973676919937133789062500000,  144.05836986187676984627614729106426239013671875000000,  138.19128670094769972820358816534280776977539062500000,  132.26390897505365273900679312646389007568359375000000,  126.27383136742314206912851659581065177917480468750000,  120.21837614161528051681671058759093284606933593750000,  114.09454671285463689400785369798541069030761718750000,  107.89896993733580643493041861802339553833007812500000,
        101.62782347403248195405467413365840911865234375000000,   95.27674306153507188810181105509400367736816406250000,   88.84070225410629007001261925324797630310058593750000,   82.31385357136143454681587172672152519226074218750000,   75.68931423245646783470874652266502380371093750000000,   68.95886998847549875790718942880630493164062500000000,   62.11255375397949762827920494601130485534667968750000,   55.13802498602691315454649156890809535980224609375000,   48.01961606055083819910578313283622264862060546875000,   40.73678720740608838468688190914690494537353515625000,
         33.26144587829575272053261869587004184722900390625000,   25.55284724561373366213956614956259727478027343750000,   17.54652881922003970771584135945886373519897460937500,    9.12478248361453481152238964568823575973510742187500,    0.00000000000000000000000000000000000000000000000000,  535.49694318016952365724137052893638610839843750000000,  537.20051649458503106870921328663825988769531250000000,  538.15515170927221788588212803006172180175781250000000,  538.65539589496370354027021676301956176757812500000000,  538.82410947724292782368138432502746582031250000000000,
        538.72973292435983694304013624787330627441406250000000,  538.41617735806448763469234108924865722656250000000000,  537.91411778431529455701820552349090576171875000000000,  537.24624267874571614811429753899574279785156250000000,  536.43003739038340427214279770851135253906250000000000,  535.47940189670964628021465614438056945800781250000000,  534.40565727038119803182780742645263671875000000000000,  533.21820562922016506490763276815414428710937500000000,  531.92498137813299763365648686885833740234375000000000,  530.53277047584072079189354553818702697753906250000000,
        529.04744284441221680026501417160034179687500000000000,  527.47412567114156445313710719347000122070312500000000,  525.81733532762564209406264126300811767578125000000000,  524.08107959870858394424431025981903076171875000000000,  522.26893815242465279879979789257049560546875000000000,  520.38412676288953662151470780372619628906250000000000,  518.42954919961266568861901760101318359375000000000000,  516.40783961500983423320576548576354980468750000000000,  514.32139751422732842911500483751296997070312500000000,  512.17241686468344141758279874920845031738281250000000,
        509.96291052528744103256030939519405364990234375000000,  507.69473090059619835301418788731098175048828125000000,  505.36958752239308978460030630230903625488281250000000,  502.98906210956226914277067407965660095214843750000000,  500.55462154241928374176495708525180816650390625000000,  498.06762909991539345355704426765441894531250000000000,  495.52935424033989875169936567544937133789062500000000,  492.94098115327420828180038370192050933837890625000000,  490.30361626895978588436264544725418090820312500000000,  487.61829487825627893471391871571540832519531250000000,
        484.88598699000857550345244817435741424560546875000000,  482.10760253142922238112078048288822174072265625000000,  479.28399597991773362082312814891338348388671875000000,  476.41597050072954289134941063821315765380859375000000,  473.50428165341725161852082237601280212402343750000000,  470.54964072049460810376331210136413574218750000000000,  467.55271770391794916577055118978023529052734375000000,  464.51414402843573725476744584739208221435546875000000,  461.43451498537962152113323099911212921142578125000000,  458.31439194586056373736937530338764190673828125000000,
        455.15430436844314954214496538043022155761718750000000,  451.95475162306900074327131733298301696777343750000000,  448.71620465018895629327744245529174804687500000000000,  445.43910747165938346370239742100238800048828125000000,  442.12387856789473516982980072498321533203125000000000,  438.77091213399251046212157234549522399902343750000000,  435.38057922600836491255904547870159149169921875000000,  431.95322880722841318856808356940746307373046875000000,  428.48918870312348872175789438188076019287109375000000,  424.98876647265723249802249483764171600341796875000000,
        421.45225020272931715226150117814540863037109375000000,  417.87990923175442503634258173406124114990234375000000,  414.27199480768302919386769644916057586669921875000000,  410.62874068515907310938928276300430297851562500000000,  406.95036366596235666293068788945674896240234375000000,  403.23706408639276332905865274369716644287109375000000,  399.48902625481588302136515267193317413330078125000000,  395.70641884219060102623188868165016174316406250000000,  391.88939522803775616921484470367431640625000000000000,  388.03809380397780159910325892269611358642578125000000,
        384.15263823666174403115292079746723175048828125000000,  380.23313769163439701515017077326774597167968750000000,  376.27968701940631035540718585252761840820312500000000,  372.29236690475732984850765205919742584228515625000000,  368.27124398005781813481007702648639678955078125000000,  364.21637090316232843179022893309593200683593750000000,  360.12778640020337661553639918565750122070312500000000,  356.00551527339570156982517801225185394287109375000000,  351.84956837373380267308675684034824371337890625000000,  347.65994253824641191386035643517971038818359375000000,
        343.43662049124139912237296812236309051513671875000000,  339.17957070873592329007806256413459777832031250000000,  334.88874724502386470703640952706336975097656250000000,  330.56408952006660229017143137753009796142578125000000,  326.20552206611881729259039275348186492919921875000000,  321.81295423169893865633639506995677947998046875000000,  317.38627984069063359129359014332294464111328125000000,  312.92537680400585031748050823807716369628906250000000,  308.43010668084815506517770700156688690185546875000000,  303.90031418618485758997849188745021820068359375000000,
        299.33582664055137456671218387782573699951171875000000,  294.73645335777280251932097598910331726074218750000000,  290.10198496558052738691912963986396789550781250000000,  285.43219265341377877120976336300373077392578125000000,  280.72682734091665679443394765257835388183593750000000,  275.98561875975275370365125127136707305908203125000000,  271.20827444034216568979900330305099487304687500000000,  266.39447859395551176930894143879413604736328125000000,  261.54389087925358126085484400391578674316406250000000,  256.65614504079860580532113090157508850097656250000000,
        251.73084740525126790089416317641735076904296875000000,  246.76757521885056689825432840734720230102539062500000,  241.76587480729529033851576969027519226074218750000000,  236.72525953623397754199686460196971893310546875000000,  231.64520754712762595772801432758569717407226562500000,  226.52515923916999440734798554331064224243164062500000,  221.36451446309101243059558328241109848022460937500000,  216.16262938685144945338834077119827270507812500000000,  210.91881298624440432831761427223682403564453125000000,  205.63232310496380250697256997227668762207031250000000,
        200.30236201842808441142551600933074951171875000000000,  194.92807142309078471953398548066616058349609375000000,  189.50852675753364451338711660355329513549804687500000,  184.04273074253700315239257179200649261474609375000000,  178.52960600353063114198448602110147476196289062500000,  172.96798660897394483981770463287830352783203125000000,  167.35660832047463486560445744544267654418945312500000,  161.69409730234164612738823052495718002319335937500000,  155.97895697640214507373457308858633041381835937500000,  150.20955262759520110193989239633083343505859375000000,
        144.38409326052763503867026884108781814575195312500000,  138.50061006751238323886354919523000717163085937500000,  132.55693068119322219899913761764764785766601562500000,  126.55064813006516999394079903140664100646972656250000,  120.47908306383729382105229888111352920532226562500000,  114.33923732342242374215857125818729400634765625000000,  108.12773622895194591819745255634188652038574218750000,  101.84075594034729306258668657392263412475585937500000,   95.47393073296986187870061257854104042053222656250000,   89.02223273253645174918347038328647613525390625000000,
         82.47981306370434140262659639120101928710937500000000,   75.83978758320748170262959320098161697387695312500000,   69.09394071123293201708293054252862930297851562500000,   62.23230406199772346553800161927938461303710937500000,   55.24253582183489186263614101335406303405761718750000,   48.10896712467176428162929369136691093444824218750000,   40.81105698621195898567748372443020343780517578125000,   33.32071167102950681737638660706579685211181640625000,   25.59718519060980312929132196586579084396362304687500,   17.57601391935301293756310769822448492050170898437500,
          9.13948863100423025684904132504016160964965820312500,    0.00000000000000000000000000000000000000000000000000,  540.41692410599762297351844608783721923828125000000000,  542.12182523849662629800150170922279357910156250000000,  543.07793856737157511815894395112991333007812500000000,  543.57979320435583758808206766843795776367187500000000,  543.75023753506377488520229235291481018066406250000000,  543.65770355165284399845404550433158874511718750000000,  543.34609619500383814738597720861434936523437500000000,  542.84608584968214017862919718027114868164062500000000,
        542.18035747675764923769747838377952575683593750000000,  541.36639372530328273569466546177864074707031250000000,  540.41809249112679935933556407690048217773437500000000,  539.34677324710219181724824011325836181640625000000000,  538.16183689569095349725103005766868591308593750000000,  536.87121693950405187933938577771186828613281250000000,  535.48169869470257253851741552352905273437500000000000,  533.99915165995957977429497987031936645507812500000000,  532.42870278699842856440227478742599487304687500000000,  530.77486837514720718900207430124282836914062500000000,
        529.04165628087810091528808698058128356933593750000000,  527.23264637225577189383329823613166809082031250000000,  525.35105473936437192605808377265930175781250000000000,  523.39978557353481392055982723832130432128906250000000,  521.38147354666864430328132584691047668457031250000000,  519.29851877439728013996500521898269653320312500000000,  517.15311592020327680074842646718025207519531250000000,  514.94727862024490150361089035868644714355468750000000,  512.68286013396914313489105552434921264648437500000000,  510.36157092286788383717066608369350433349609375000000,
        507.98499370814016629083198495209217071533203125000000,  505.55459644333768665092065930366516113281250000000000,  503.07174355034084101134794764220714569091796875000000,  500.53770569923597122397040948271751403808593750000000,  497.95366835980235009628813713788986206054687500000000,  495.32073931073517769618774764239788055419921875000000,  492.63995525975178679800592362880706787109375000000000,  489.91228770137735182288452051579952239990234375000000,  487.13864811799771814548876136541366577148437500000000,  484.31989261258644319241284392774105072021484375000000,
        481.45682604750822974892798811197280883789062500000000,  478.55020575231299062579637393355369567871093750000000,  475.60074485396597765429760329425334930419921875000000,  472.60911527510444329891470260918140411376953125000000,  469.57595043937209311479819007217884063720703125000000,  466.50184771740305222920142114162445068359375000000000,  463.38737064242246788126067258417606353759765625000000,  460.23305092054130227552377618849277496337890625000000,  457.03939025751992630830500274896621704101562500000000,  453.80686202096836723285377956926822662353515625000000,
        450.53591275454618880758062005043029785156250000000000,  447.22696355866469275497365742921829223632812500000000,  443.88041135041754614576348103582859039306640625000000,  440.49663001393207650835392996668815612792968750000000,  437.07597145100129409911460243165493011474609375000000,  433.61876654069817504932871088385581970214843750000000,  430.12532601566175571861094795167446136474609375000000,  426.59594126185669438200420700013637542724609375000000,  423.03088504782715517649194225668907165527343750000000,  419.43041218877743858683970756828784942626953125000000,
        415.79476015019901069535990245640277862548828125000000,  412.12414959522129720426164567470550537109375000000000,  408.41878487937543695807107724249362945556640625000000,  404.67885449602590597351081669330596923828125000000000,  400.90453147532753064297139644622802734375000000000000,  397.09597373921116059136693365871906280517578125000000,  393.25332441457084087232942692935466766357421875000000,  389.37671210652820263931062072515487670898437500000000,  385.46625113336938284192001447081565856933593750000000,  381.52204172449182806303724646568298339843750000000000,
        377.54417018245351300720358267426490783691406250000000,  373.53270900998415982030564919114112854003906250000000,  369.48771700259669614752056077122688293457031250000000,  365.40923930721993428960558958351612091064453125000000,  361.29730744706165523894014768302440643310546875000000,  357.15193931270050597959198057651519775390625000000000,  352.97313911919474094247561879456043243408203125000000,  348.76089732877977667158120311796665191650390625000000,  344.51519053850654472626047208905220031738281250000000,  340.23598133194434467441169545054435729980468750000000,
        335.92321809382957553680171258747577667236328125000000,  331.57683478628814555122517049312591552734375000000000,  327.19675068498514747261651791632175445556640625000000,  322.78287007326235880100284703075885772705078125000000,  318.33508189200085780612425878643989562988281250000000,  313.85325934259844871121458709239959716796875000000000,  309.33725944006084773718612268567085266113281250000000,  304.78692251277794866837211884558200836181640625000000,  300.20207164507735342340311035513877868652343750000000,  295.58251205810813644347945228219032287597656250000000,
        290.92803042400561253089108504354953765869140625000000,  286.23839410760041346293292008340358734130859375000000,  281.51335032915909550865762867033481597900390625000000,  276.75262524075640158116584643721580505371093750000000,  271.95592290786271405522711575031280517578125000000000,  267.12292418656397785525768995285034179687500000000000,  262.25328548548407070484245195984840393066406250000000,  257.34663739992004138912307098507881164550781250000000,  252.40258320388883817031455691903829574584960937500000,  247.42069718366934694131487049162387847900390625000000,
        242.40052279394566880910133477300405502319335937500000,  237.34157061474536476453067734837532043457031250000000,  232.24331608392725456724292598664760589599609375000000,  227.10519697589202792187279555946588516235351562500000,  221.92661059233248010968964081257581710815429687500000,  216.70691062502140766810043714940547943115234375000000,  211.44540364364536344510270282626152038574218750000000,  206.14134515323584651014243718236684799194335937500000,  200.79393515547917559160850942134857177734375000000000,  195.40231313562946979800472036004066467285156250000000,
        189.96555238131460896511271130293607711791992187500000,  184.48265352042309928037866484373807907104492187500000,  178.95253714146988954780681524425745010375976562500000,  173.37403532998476407556154299527406692504882812500000,  167.74588191672722814473672769963741302490234375000000,  162.06670118541916281174053438007831573486328125000000,  156.33499472582062139736081007868051528930664062500000,  150.54912603765751555329188704490661621093750000000000,  144.70730238558243740953912492841482162475585937500000,  138.80755326568569785194995347410440444946289062500000,
        132.84770465665661731691216118633747100830078125000000,  126.82534797390188430199486901983618736267089843750000,  120.73780229356358972836460452526807785034179687500000,  114.58206792121980299725692020729184150695800781250000,  108.35476867866219663483207114040851593017578125000000,  102.05207926328034773177932947874069213867187500000000,   95.66963252257185956750618061050772666931152343750000,   89.20239918796315237159433308988809585571289062500000,   82.64452902234300779582554241642355918884277343750000,   75.98913655218812834846175974234938621520996093750000,
         69.22800489796809131348709343001246452331542968750000,   62.35116440378802593613727367483079433441162109375000,   55.34627201457611533896852051839232444763183593750000,   48.19765765069715968138552852906286716461181640625000,   40.88477914130245238766292459331452846527099609375000,   33.37954159062267223134767846204340457916259765625000,   25.64119787523724625089016626589000225067138671875000,   17.60528325933446680551242025103420019149780273437500,    9.15408743042538297629562293877825140953063964843750,    0.00000000000000000000000000000000000000000000000000,
        545.34417779115483426721766591072082519531250000000000,  547.05039537818515782419126480817794799804687500000000,  548.00797377153355682821711525321006774902343750000000,  548.51142430050958864740096032619476318359375000000000,  548.68358342236035696259932592511177062988281250000000,  548.59287472668313512258464470505714416503906250000000,  548.28319702292947113164700567722320556640625000000000,  547.78521610749851333821425214409828186035156250000000,  547.12161344864387046982301399111747741699218750000000,  546.30986900874927414406556636095046997070312500000000,
        545.36387860871116117778001353144645690917968750000000,  544.29496012323295417445478960871696472167968750000000,  543.11251323682631664269138127565383911132812500000000,  541.82447054370095429476350545883178710937500000000000,  540.43761670833578136807773262262344360351562500000000,  538.95782079424088806263171136379241943359375000000000,  537.39020950343729055020958185195922851562500000000000,  535.73929904669648749404586851596832275390625000000000,  534.00909733383537059125956147909164428710937500000000,  532.20318441278880072786705568432807922363281250000000,
        530.32477666766044421819970011711120605468750000000000,  528.37677868792900426342384889721870422363281250000000,  526.36182563962483982322737574577331542968750000000000,  524.28231822185432520200265571475028991699218750000000,  522.14045176551144322729669511318206787109375000000000,  519.93824065370893094950588420033454895019531250000000,  517.67753896883982633880805224180221557617187500000000,  515.36005806849084365239832550287246704101562500000000,  512.98738164086591950763249769806861877441406250000000,  510.56097867571190818125614896416664123535156250000000,
        508.08221469902389344497350975871086120605468750000000,  505.55236155204897841031197458505630493164062500000000,  502.97260594224945862151798792183399200439453125000000,  500.34405695231890831564669497311115264892578125000000,  497.66775266036756875109858810901641845703125000000000,  494.94466599805093665054300799965858459472656250000000,  492.17570995221024077181937173008918762207031250000000,  489.36174219841842614187044091522693634033203125000000,  486.50356924082058185376808978617191314697265625000000,  483.60195012117776514060096815228462219238281250000000,
        480.65759975055050290393410250544548034667968750000000,  477.67119190921147264816681854426860809326171875000000,  474.64336195383492622568155638873577117919921875000000,  471.57470926553617118770489469170570373535156250000000,  468.46579946772936864363146014511585235595703125000000,  465.31716643888324824729352258145809173583984375000000,  462.12931414195639945319271646440029144287109375000000,  458.90271828948374377432628534734249114990234375000000,  455.63782786088648890654440037906169891357421875000000,  452.33506648651513160075410269200801849365234375000000,
        448.99483371116406260625808499753475189208984375000000,  445.61750614825979255329002626240253448486328125000000,  442.20343853459672800454427488148212432861328125000000,  438.75296469433766333168023265898227691650390625000000,  435.26639841998303381842561066150665283203125000000000,  431.74403427713019709699437953531742095947265625000000,  428.18614833906309513622545637190341949462890625000000,  424.59299885652649209077935665845870971679687500000000,  420.96482686742893974951584823429584503173828125000000,  417.30185675067838246832252480089664459228515625000000,
        413.60429672786887067559291608631610870361328125000000,  409.87233931610438730785972438752651214599609375000000,  406.10616173485414037713780999183654785156250000000000,  402.30592626937863087732694111764430999755859375000000,  398.47178059294469676387961953878402709960937500000000,  394.60385804974845314063713885843753814697265625000000,  390.70227790019504254814819432795047760009765625000000,  386.76714552992729068137123249471187591552734375000000,  382.79855262375770053040469065308570861816406250000000,  378.79657730543215166107984259724617004394531250000000,
        374.76128424393851901186280883848667144775390625000000,  370.69272472686367336791590787470340728759765625000000,  366.59093670110030416253721341490745544433593750000000,  362.45594478100463220471283420920372009277343750000000,  358.28776022390354683011537417769432067871093750000000,  354.08638087265165950157097540795803070068359375000000,  349.85179106472708099317969754338264465332031250000000,  345.58396150714634131873026490211486816406250000000000,  341.28284911625314634875394403934478759765625000000000,  336.94839682120164070511236786842346191406250000000000,
        332.58053332970729343287530355155467987060546875000000,  328.17917285436624297290109097957611083984375000000000,  323.74421479755727659721742384135723114013671875000000,  319.27554339262218263684189878404140472412109375000000,  314.77302729867307107269880361855030059814453125000000,  310.23651914599116707904613576829433441162109375000000,  305.66585502855446065950673073530197143554687500000000,  301.06085393975575925651355646550655364990234375000000,  296.42131714683671361854067072272300720214843750000000,  291.74702749896283648922690190374851226806640625000000,
        287.03774866317763780898530967533588409423828125000000,  282.29322428170280545600689947605133056640625000000000,  277.51317704316323897728580050170421600341796875000000,  272.69730765930177085465402342379093170166015625000000,  267.84529373758351766809937544167041778564453125000000,  262.95678853874352398634073324501514434814453125000000,  258.03141960677379529442987404763698577880859375000000,  253.06878725703333543606277089565992355346679687500000,  248.06846290605321314615139272063970565795898437500000,  243.02998722412959864414006005972623825073242187500000,
        237.95286808888826612928824033588171005249023437500000,  232.83657831456386588797613512724637985229492187500000,  227.68055312765852704615099355578422546386718750000000,  222.48418735478634289393085055053234100341796875000000,  217.24683228269464052573312073945999145507812500000000,  211.96779214346125286283495370298624038696289062500000,  206.64632016941300207690801471471786499023437500000000,  201.28161415203879869295633397996425628662109375000000,  195.87281142661629473877837881445884704589843750000000,  190.41898318883508522958436515182256698608398437500000,
        184.91912803059955194839858449995517730712890625000000,  179.37216455840342632654937915503978729248046875000000,  173.77692292781523519806796684861183166503906250000000,  168.13213508987337263533845543861389160156250000000000,  162.43642349707863559160614386200904846191406250000000,  156.68828795480428084374580066651105880737304687500000,  150.88609022362948053341824561357498168945312500000000,  145.02803587277335850558301899582147598266601562500000,  139.11215274514316320164653006941080093383789062500000,  133.13626520709402711872826330363750457763671875000000,
        127.09796310120317741620965534821152687072753906250000,  120.99456396899921628573792986571788787841796875000000,  114.82306661842602579781669192016124725341796875000000,  108.58009340943368670195923186838626861572265625000000,  102.26181761222348143292038002982735633850097656250000,   95.86387068066370886754157254472374916076660156250000,   89.38122198505942606061580590903759002685546875000000,   82.80801995867568621179088950157165527343750000000000,   76.13737782889202776459569577127695083618164062500000,   69.36107744667370411661977414041757583618164062500000,
         62.46914791531599320251189055852591991424560546875000,   55.44924496676749470225331606343388557434082031250000,   48.28569733541180397651260136626660823822021484375000,   40.95796169061654978804654092527925968170166015625000,   33.43794200225490698130670352838933467864990234375000,   25.68489003723207986240595346316695213317871093750000,   17.63433997402933783860135008580982685089111328125000,    9.16858043772795028303335129749029874801635742187500,    0.00000000000000000000000000000000000000000000000000,  550.27865172428562345885438844561576843261718750000000,
        551.98617458279875336302211508154869079589843750000000,  552.94520520304934052546741440892219543457031250000000,  553.45023730562309083325089886784553527832031250000000,  553.62409552930739664589054882526397705078125000000000,  553.53519513365995408094022423028945922851562500000000,  553.22742884556225817505037412047386169433593750000000,  552.73145790618184491904685273766517639160156250000000,  552.06996031261917323718080297112464904785156250000000,  551.26041335390800668392330408096313476562500000000000,  550.31671078299143573531182482838630676269531250000000,
        549.25016887830997802666388452053070068359375000000000,  548.07018610421869198034983128309249877929687500000000,  546.78469414087646782718366011977195739746093750000000,  545.40047699248088974854908883571624755859375000000000,  543.92340327619774598133517429232597351074218750000000,  542.35859943085608847468392923474311828613281250000000,  540.71058156308254183386452496051788330078125000000000,  538.98335761850830749608576297760009765625000000000000,  537.18050780558519363694358617067337036132812500000000,  535.30524878134667687845649197697639465332031250000000,
        533.36048551065800893411505967378616333007812500000000,  531.34885362929105667717522010207176208496093750000000,  529.27275439383799948700470849871635437011718750000000,  527.13438377503223364328732714056968688964843750000000,  524.93575687378540806093951687216758728027343750000000,  522.67872856469125508738216012716293334960937500000000,  520.36501106907996927475323900580406188964843750000000,  517.99618900817927169555332511663436889648437500000000,  515.57373237228614470950560644268989562988281250000000,  513.09900775416497253900161013007164001464843750000000,
        510.57328812713018351132632233202457427978515625000000,  507.99776139543513409080333076417446136474609375000000,  505.37353790302211109519703313708305358886718750000000,  502.70165705372238562631537206470966339111328125000000,  499.98309316965662674192572012543678283691406250000000,  497.21876069338611614512046799063682556152343750000000,  494.40951882219286517283762805163860321044921875000000,  491.55617564886790660239057615399360656738281250000000,  488.65949187190540214942302554845809936523437500000000,  485.72018412853554991670534946024417877197265625000000,
        482.73892799618005255979369394481182098388671875000000,  479.71636070137594742845976725220680236816406250000000,  476.65308356974094294855603948235511779785156250000000,  473.54966424594920226809335872530937194824218750000000,  470.40663870880047170430771075189113616943359375000000,  467.22451310316785111353965476155281066894531250000000,  464.00376540780052891932427883148193359375000000000000,  460.74484695556202495936304330825805664062500000000000,  457.44818382062004502586205489933490753173828125000000,  454.11417808533587958663702011108398437500000000000000,
        450.74320899806735951642622239887714385986328125000000,  447.33563403176952988360426388680934906005859375000000,  443.89178985212430461615440435707569122314453125000000,  440.41199320291929097948013804852962493896484375000000,  436.89654171551148920116247609257698059082031250000000,  433.34571464843645571818342432379722595214843750000000,  429.75977356253588368417695164680480957031250000000000,  426.13896293637111511998227797448635101318359375000000,  422.48351072614940449057030491530895233154296875000000,  418.79362887390999503622879274189472198486328125000000,
        415.06951376728335389998392201960086822509765625000000,  411.31134665375094527917099185287952423095703125000000,  407.51929401198054847554885782301425933837890625000000,  403.69350788249226980042294599115848541259765625000000,  399.83412615961816527487826533615589141845703125000000,  395.94127284645020381503854878246784210205078125000000,  392.01505827421959793355199508368968963623046875000000,  388.05557928731820993561996147036552429199218750000000,  384.06291939495287124373135156929492950439453125000000,  380.03714889021233602761640213429927825927734375000000,
        375.97832493712809309727163054049015045166015625000000,  371.88649162611125120747601613402366638183593750000000,  367.76167999795961804920807480812072753906250000000000,  363.60390803643571189240901730954647064208984375000000,  359.41318062922687204263638705015182495117187500000000,  355.18948949690519611976924352347850799560546875000000,  350.93281308930420436809072270989418029785156250000000,  346.64311644852313065712223760783672332763671875000000,  342.32035103755487170928972773253917694091796875000000,  337.96445453330051122975419275462627410888671875000000,
        333.57535058249334269930841401219367980957031250000000,  329.15294851878604731609812006354331970214843750000000,  324.69714303897308127488940954208374023437500000000000,  320.20781383600422032031929120421409606933593750000000,  315.68482518610164788697147741913795471191406250000000,  311.12802548691252013668417930603027343750000000000000,  306.53724674320420717776869423687458038330078125000000,  301.91230399613414192572236061096191406250000000000000,  297.25299469159671161833102814853191375732421875000000,  292.55909798254475617795833386480808258056640625000000,
        287.83037395950606196493026800453662872314453125000000,  283.06656280273728043539449572563171386718750000000000,  278.26738384857844721409492194652557373046875000000000,  273.43253456155355252121808007359504699707031250000000,  268.56168940260187127933022566139698028564453125000000,  263.65449858247887959805666469037532806396484375000000,  258.71058668780784728369326330721378326416015625000000,  253.72955116545438158937031403183937072753906250000000,  248.71096064878241804763092659413814544677734375000000,  243.65435310687485070957336574792861938476562500000000,
        238.55923379489075841775047592818737030029296875000000,  233.42507298029354956270253751426935195922851562500000,  228.25130341560500824016344267874956130981445312500000,  223.03731752348406303099181968718767166137695312500000,  217.78246425411276732120313681662082672119140625000000,  212.48604556788160380165209062397480010986328125000000,  207.14731248791204620829375926405191421508789062500000,  201.76546065668432561324152629822492599487304687500000,  196.33962531848212051954760681837797164916992187500000,  190.86887563393230493602459318935871124267578125000000,
        185.35220821381759037649317178875207901000976562500000,  179.78853973554853951100085396319627761840820312500000,  174.17669847583019304693152662366628646850585937500000,  168.51541455531753399554872885346412658691406250000000,  162.80330864294276693726715166121721267700195312500000,  157.03887880573202551204303745180368423461914062500000,  151.22048510961241163386148400604724884033203125000000,  145.34633147138390540931141003966331481933593750000000,  139.41444412236626249068649485707283020019531250000000,  133.42264585681553512586106080561876296997070312500000,
        127.36852498340954298328142613172531127929687500000000,  121.24939754674143443935463437810540199279785156250000,  115.06226089359704189973854226991534233093261718750000,  108.80373595740705638945655664429068565368652343750000,  102.46999461539809317400795407593250274658203125000000,   96.05666696095789802711806260049343109130859375000000,   89.55872103539516615455795545130968093872070312500000,   82.97030397349213615143526112660765647888183593750000,   76.28452773372562489839765476062893867492675781250000,   69.49317292684030178406828781589865684509277343750000,
         62.58626744372917016789870103821158409118652343750000,   55.55146583092522405422641895711421966552734375000000,   48.37309566358265300323182600550353527069091796875000,   41.03061247725477755921019706875085830688476562500000,   33.49591913267298082246270496398210525512695312500000,   25.72826631155743015710868348833173513412475585937500,   17.66318713047139254967987653799355030059814453125000,    9.18296917518004995883984520332887768745422363281250,    0.00000000000000000000000000000000000000000000000000,  555.22029414689484383416129276156425476074218750000000,
        556.92911127018714978476054966449737548828125000000000,  557.88958148692745453445240855216979980468750000000000,  558.39618107986768791306531056761741638183593750000000,  558.57172297758245349541539326310157775878906250000000,  558.48461418111628518090583384037017822265625000000000,  558.17874138306410713994409888982772827148437500000000,  557.68476130200417628657305613160133361816406250000000,  557.02534848545769818883854895830154418945312500000000,  556.21797756250805377931101247668266296386718750000000,  555.27654022528554378368426114320755004882812500000000,
        554.21235115815500194003107026219367980957031250000000,  553.03480760346974420826882123947143554687500000000000,  551.75184032212212059675948694348335266113281250000000,  550.37023264992865279054967686533927917480468750000000,  548.89585274708747419936116784811019897460937500000000,  547.33382677636086555139627307653427124023437500000000,  545.68867072531077155872480943799018859863281250000000,  543.96439255857239913893863558769226074218750000000000,  542.16457262653159432375105097889900207519531250000000,  540.29242783888480516907293349504470825195312500000000,
        538.35086351399695558939129114151000976562500000000000,  536.34251573391770762100350111722946166992187500000000,  534.26978628771644252992700785398483276367187500000000,  532.13487175942452722665620967745780944824218750000000,  529.93978793968653917545452713966369628906250000000000,  527.68639046569785477913683280348777770996093750000000,  525.37639239138752600410953164100646972656250000000000,  523.01137923829332976310979574918746948242187500000000,  520.59282196295225730864331126213073730468750000000000,  518.12208818895373951818328350782394409179687500000000,
        515.60045198405987321166321635246276855468750000000000,  513.02910240997084656555671244859695434570312500000000,  510.40915103075445813374244607985019683837890625000000,  507.74163853300018445224850438535213470458984375000000,  505.02754058442531004402553662657737731933593750000000,  502.26777303646383643354056403040885925292968750000000,  499.46319655920234481527586467564105987548828125000000,  496.61462078303020462044514715671539306640625000000000,  493.72280800989324234251398593187332153320312500000000,  490.78847654757663576674531213939189910888671875000000,
        487.81230371259744060807861387729644775390625000000000,  484.79492854074919705453794449567794799804687500000000,  481.73695423887153310715802945196628570556640625000000,  478.63895040681393311388092115521430969238281250000000,  475.50145505467907014462980441749095916748046875000000,  472.32497643713361412665108218789100646972656250000000,  469.10999472376903440817841328680515289306640625000000,  465.85696352209686210699146613478660583496093750000000,  462.56631126770355422195279970765113830566406250000000,  459.23844249432056585646932944655418395996093750000000,
        455.87373899503216989614884369075298309326171875000000,  452.47256088451877076295204460620880126953125000000000,  449.03524757107658160748542286455631256103515625000000,  445.56211864614903106485144235193729400634765625000000,  442.05347469822055472832289524376392364501953125000000,  438.50959805714995809466927312314510345458984375000000,  434.93075347433517663375823758542537689208984375000000,  431.31718874349650150179513730108737945556640625000000,  427.66913526632839648300432600080966949462890625000000,  423.98680856679004591569537296891212463378906250000000,
        420.27040875737702663172967731952667236328125000000000,  416.52012096032797217048937454819679260253906250000000,  412.73611568637574009699164889752864837646484375000000,  408.91854917333233743192977271974086761474609375000000,  405.06756368650968624933739192783832550048828125000000,  401.18328778271234114072285592555999755859375000000000,  397.26583653929378670000005513429641723632812500000000,  393.31531174953556728723924607038497924804687500000000,  389.33180208539852173998951911926269531250000000000000,  385.31538322848683719712425954639911651611328125000000,
        381.26611796987327807073597796261310577392578125000000,  377.18405627924607870227191597223281860351562500000000,  373.06923534365097339104977436363697052001953125000000,  368.92167957592391758225858211517333984375000000000000,  364.74140059272667713230475783348083496093750000000000,  360.52839716191482466456363908946514129638671875000000,  356.28265511878186089234077371656894683837890625000000,  352.00414725052996800513938069343566894531250000000000,  347.69283314811696072865743190050125122070312500000000,  343.34865902441868001915281638503074645996093750000000,
        338.97155749742046282335650175809860229492187500000000,  334.56144733691184001145302318036556243896484375000000,  330.11823317289707802046905271708965301513671875000000,  325.64180516365337325623841024935245513916015625000000,  321.13203862105717689701123163104057312011718750000000,  316.58879359045721457732724957168102264404296875000000,  312.01191438199703043210320174694061279296875000000000,  307.40122904986446883413009345531463623046875000000000,  302.75654881547586683154804632067680358886718750000000,  298.07766743007255172415170818567276000976562500000000,
        293.36436047160532325506210327148437500000000000000000,  288.61638457010730007823440246284008026123046875000000,  283.83347655497874484353815205395221710205078125000000,  279.01535251672919457632815465331077575683593750000000,  274.16170677470643113338155671954154968261718750000000,  269.27221074118250498941051773726940155029296875000000,  264.34651167082074607606045901775360107421875000000000,  259.38423128299308473287965171039104461669921875000000,  254.38496424260696926467062439769506454467773437500000,  249.34827648298994517972460016608238220214843750000000,
        244.27370335190428818350483197718858718872070312500000,  239.16074755885404101718449965119361877441406250000000,  234.00887689840988059586379677057266235351562500000000,  228.81752172019841395922412630170583724975585937500000,  223.58607211134645353922678623348474502563476562500000,  218.31387475135599629538774024695158004760742187500000,  213.00022939239462971272587310522794723510742187500000,  207.64438490953344285117054823786020278930664062500000,  202.24553485519371065493032801896333694458007812500000,  196.80281243950889802363235503435134887695312500000000,
        191.31528484287491664872504770755767822265625000000000,  185.78194674786084306106204167008399963378906250000000,  180.20171295386313659037114121019840240478515625000000,  174.57340990803271552067599259316921234130859375000000,  168.89576594826638711310806684195995330810546875000000,  163.16740000594097637076629325747489929199218750000000,  157.38680845420435616688337177038192749023437500000000,  151.55234970732090005185455083847045898437500000000000,  145.66222607124271348766342271119356155395507812500000,  139.71446220591354858697741292417049407958984375000000,
        133.70687937239736697847547475248575210571289062500000,  127.63706438312846103144693188369274139404296875000000,  121.50233182221880667839286616072058677673339843750000,  115.29967761059728559303039219230413436889648437500000,  109.02572128936820661238016327843070030212402343750000,  102.67663337591227445955155417323112487792968750000000,   96.24804263524217162739660125225782394409179687500000,   89.73491581079218804006814025342464447021484375000000,   83.13139876903764502458216156810522079467773437500000,   76.43060222881841525577328866347670555114746093750000,
         69.62430558904858912683266680687665939331054687500000,   62.70253555576574910901399562135338783264160156250000,   55.65294551682239898582338355481624603271484375000000,   48.45986191409642174221517052501440048217773437500000,   41.10273917452668257510595140047371387481689453125000,   33.55347907417651498462873860262334346771240234375000,   25.77133123335494957473201793618500232696533203125000,   17.69182772980622786462845397181808948516845703125000,    9.19725513242752690246106794802471995353698730468750,    0.00000000000000000000000000000000000000000000000000,
        560.16905403727298562444048002362251281738281250000000,  561.87915459095029291347600519657135009765625000000000,  562.84105197609596871188841760158538818359375000000000,  563.34920520576747549057472497224807739257812500000000,  563.52641560494805617054225876927375793457031250000000,  563.44108198671779064170550554990768432617187500000000,  563.13708505709951168682891875505447387695312500000000,  562.64507704443144575634505599737167358398437500000000,  561.98772906813292138394899666309356689453125000000000,  561.18251311078938670107163488864898681640625000000000,
        560.24331881102580155129544436931610107421875000000000,  559.18145926158263137040194123983383178710937500000000,  558.00633048130896440852666273713111877441406250000000,  556.72586230702881948673166334629058837890625000000000,  555.34683739855313433508854359388351440429687500000000,  553.87512344902222594100749120116233825683593750000000,  552.31584633284512619866291061043739318847656250000000,  550.67352190423684987763408571481704711914062500000000,  548.95215813071547472645761445164680480957031250000000,  547.15533548675591646315297111868858337402343750000000,
        545.28627111523792336811311542987823486328125000000000,  543.34787066697413138172123581171035766601562500000000,  541.34277064771470122650498524308204650878906250000000,  539.27337335493234604655299335718154907226562500000000,  537.14187596040449079737300053238868713378906250000000,  534.95029491748198324785334989428520202636718750000000,  532.70048659745953045785427093505859375000000000000000,  530.39416485687286240136018022894859313964843750000000,  528.03291608606946283543948084115982055664062500000000,  525.61821217478996004501823335886001586914062500000000,
        523.15142174284358134173089638352394104003906250000000,  520.63381991622668465424794703722000122070312500000000,  518.06659687621947796287713572382926940917968750000000,  515.45086536744383920449763536453247070312500000000000,  512.78766731791517941019264981150627136230468750000000,  510.07797969779119284794433042407035827636718750000000,  507.32271972233058932033600285649299621582031250000000,  504.52274948741103344218572601675987243652343750000000,  501.67888011196083652976085431873798370361328125000000,  498.79187545018589844403322786092758178710937500000000,
        495.86245542701027488874387927353382110595703125000000,  492.89129904230566125988843850791454315185546875000000,  489.87904708295076261492795310914516448974609375000000,  486.82630457628994236074504442512989044189453125000000,  483.73364301396378550634835846722126007080078125000000,  480.60160237119515613812836818397045135498046875000000,  477.43069294332400431812857277691364288330078125000000,  474.22139701857639693116652779281139373779296875000000,  470.97417040365735374507494270801544189453125000000000,  467.68944381670036136711132712662220001220703125000000,
        464.36762416033474210053100250661373138427734375000000,  461.00909568610467204052838496863842010498046875000000,  457.61422106014498467629891820251941680908203125000000,  454.18334233886804440771811641752719879150390625000000,  450.71678186240808372531319037079811096191406250000000,  447.21484307268889324404881335794925689697265625000000,  443.67781126220614851263235323131084442138671875000000,  440.10595425893387755422736518085002899169921875000000,  436.49952305216118020325666293501853942871093750000000,  432.85875236352882211576797999441623687744140625000000,
        429.18386116705875110710621811449527740478515625000000,  425.47505316154331467259908095002174377441406250000000,  421.73251719827555916708661243319511413574218750000000,  417.95642766675882739946246147155761718750000000000000,  414.14694484071713986850227229297161102294921875000000,  410.30421518644425304955802857875823974609375000000000,  406.42837163526661470314138568937778472900390625000000,  402.51953382165197581343818455934524536132812500000000,  398.57780828827333152730716392397880554199218750000000,  394.60328865912492801726330071687698364257812500000000,
        390.59605578158959815482376143336296081542968750000000,  386.55617783816722976553137414157390594482421875000000,  382.48371042839238498345366679131984710693359375000000,  378.37869662129037351405713707208633422851562500000000,  374.24116697855157553931348957121372222900390625000000,  370.07113954842532166367163881659507751464843750000000,  365.86861983016808608226710930466651916503906250000000,  361.63360070870010076760081574320793151855468750000000,  357.36606235894697647381690330803394317626953125000000,  353.06597211915720890829106792807579040527343750000000,
        348.73328433228698486345820128917694091796875000000000,  344.36794015434247739904094487428665161132812500000000,  339.96986732834562872085371054708957672119140625000000,  335.53897992235425817852956242859363555908203125000000,  331.07517802971227638408890925347805023193359375000000,  326.57834742942213779315352439880371093750000000000000,  322.04835920423039397064712829887866973876953125000000,  317.48506931367245442743296734988689422607421875000000,  312.88831811895198597994749434292316436767578125000000,  308.25792985610650021044421009719371795654296875000000,
        303.59371205344331201558816246688365936279296875000000,  298.89545488870089684496633708477020263671875000000000,  294.16293048079137406602967530488967895507812500000000,  289.39589211030602200480643659830093383789062500000000,  284.59407336218941964034456759691238403320312500000000,  279.75718718311134125542594119906425476074218750000000,  274.88492484505184165755053982138633728027343750000000,  269.97695480545519330917159095406532287597656250000000,  265.03292145296330772907822392880916595458984375000000,  260.05244372618705028799013234674930572509765625000000,
        255.03511359116293988336110487580299377441406250000000,  249.98049436103281095711281523108482360839843750000000,  244.88811883900930865820555482059717178344726562500000,  239.75748726278087019636586774140596389770507812500000,  234.58806502507249547306855674833059310913085937500000,  229.37928014100182849688280839473009109497070312500000,  224.13052042801317043085873592644929885864257812500000,  218.84113035835858340760751161724328994750976562500000,  213.51040753710421427058463450521230697631835937500000,  208.13759875018817524505720939487218856811523437500000,
        202.72189551678516750143899116665124893188476562500000,  197.26242906767967610903724562376737594604492187500000,  191.75826465591529768062173388898372650146484375000000,  186.20839508688791852364374790340662002563476562500000,  180.61173333126129136871895752847194671630859375000000,  174.96710405423064571550639811903238296508789062500000,  169.27323385692156421100662555545568466186523437500000,  163.52873997759894564296700991690158843994140625000000,  157.73211713849704551648756023496389389038085937500000,  151.88172214376473334596084896475076675415039062500000,
        145.97575572869428128797153476625680923461914062500000,  140.01224102073791755174170248210430145263671875000000,  133.98899778540121019432262983173131942749023437500000,  127.90361137530773305570619413629174232482910156250000,  121.75339494936851281181589001789689064025878906250000,  115.53534303682815220781776588410139083862304687500000,  109.24607381957356722068652743473649024963378906250000,  102.88175648722575772353593492880463600158691406250000,   96.43801850752555537837906740605831146240234375000000,   89.90982535619043858332588570192456245422363281250000,
         83.29132166063681097512017004191875457763671875000000,   76.57561692844019773929176153615117073059082031250000,   69.75448937421414541404374176636338233947753906250000,   62.81796454585938249692844692617654800415039062500000,   55.75369469848499193176394328474998474121093750000000,   48.54600516587672132118314038962125778198242187500000,   41.17434929081743888445998891256749629974365234375000,   33.61062778846140020050370367243885993957519531250000,   25.81408924079096323112025856971740722656250000000000,   17.72026470916520324294651800300925970077514648437500,
          9.21143976741948300457352161174640059471130371093750,    0.00000000000000000000000000000000000000000000000000,  565.12488109487424026156077161431312561035156250000000,  566.83625441293906987993977963924407958984375000000000,  567.79956673604829120449721813201904296875000000000000,  568.30925997301710594911128282546997070312500000000000,  568.48812395026448029966559261083602905273437500000000,  568.40454936249409456650028005242347717285156250000000,  568.10241097630864715029019862413406372070312500000000,  567.61235656186215692287078127264976501464843750000000,
        566.95705383184736092516686767339706420898437500000000,  566.15397213584662949870107695460319519042968750000000,  565.21699906644380462239496409893035888671875000000000,  564.15744612745231734152184799313545227050781250000000,  562.98470811304150629439391195774078369140625000000000,  561.70671393155760142690269276499748229980468750000000,  560.33024555963561397220473736524581909179687500000000,  558.86117021377799574111122637987136840820312500000000,  557.30461346831395985645940527319908142089843750000000,  555.66509103044143103034002706408500671386718750000000,
        553.94661085509858367004198953509330749511718750000000,  552.15275352371122608019504696130752563476562500000000,  550.28673639361511504830559715628623962402343750000000,  548.35146542781876632943749427795410156250000000000000,  546.34957753404296454391442239284515380859375000000000,  544.28347549498607804707717150449752807617187500000000,  542.15535704556384644092759117484092712402343750000000,  539.96723927580319468688685446977615356445312500000000,  537.72097926364381237362977117300033569335937500000000,  535.41829163933937252295436337590217590332031250000000,
        533.06076363170450349571183323860168457031250000000000,  530.64986803186559427558677271008491516113281250000000,  528.18697442252550899866037070751190185546875000000000,  525.67335895304495352320373058319091796875000000000000,  523.11021288782797000749269500374794006347656250000000,  520.49865011396047975722467526793479919433593750000000,  517.83971376110662276914808899164199829101562500000000,  515.13438206034152244683355093002319335937500000000000,  512.38357354741663129971129819750785827636718750000000,  509.58815169878886308651999570429325103759765625000000,
        506.74892907476009895617607980966567993164062500000000,  503.86667103259509303825325332581996917724609375000000,  500.94209906303285606554709374904632568359375000000000,  497.97589379575998691507265903055667877197265625000000,  494.96869771288328365699271671473979949951171875000000,  491.92111760397193620519828982651233673095703125000000,  488.83372679163841212357510812580585479736328125000000,  485.70706715274513953772839158773422241210937500000000,  482.54165095703211818545241840183734893798828125000000,  479.33796254215388898956007324159145355224609375000000,
        476.09645984172072985529666766524314880371093750000000,  472.81757578088195259624626487493515014648437500000000,  469.50171955222083397529786452651023864746093750000000,  466.14927778320077322860015556216239929199218750000000,  462.76061560507889680593507364392280578613281250000000,  459.33607763205105811721296049654483795166015625000000,  455.87598885838542628334835171699523925781250000000000,  452.38065548042413865914568305015563964843750000000000,  448.85036564955822768752113915979862213134765625000000,  445.28539016160061692062299698591232299804687500000000,
        441.68598308738063451528432779014110565185546875000000,  438.05238234884870962559944018721580505371093750000000,  434.38481024450442191664478741586208343505859375000000,  430.68347392753696567524457350373268127441406250000000,  426.94856583968550012286868877708911895751953125000000,  423.18026410348176113984663970768451690673828125000000,  419.37873287523001408771960996091365814208984375000000,  415.54412266079043547506444156169891357421875000000000,  411.67657059597877378109842538833618164062500000000000,  407.77620069315344153437763452529907226562500000000000,
        403.84312405533938772350666113197803497314453125000000,  399.87743905903533914170111529529094696044921875000000,  395.87923150665295679573318921029567718505859375000000,  391.84857474935455456943600438535213470458984375000000,  387.78552978087896008219104260206222534179687500000000,  383.69014530277479479991598054766654968261718750000000,  379.56245776129298974410630762577056884765625000000000,  375.40249135602698515867814421653747558593750000000000,  371.21025802022398920598789118230342864990234375000000,  366.98575737252537010135711170732975006103515625000000,
        362.72897663972526061115786433219909667968750000000000,  358.43989054996217191728646866977214813232421875000000,  354.11846119557770862229517661035060882568359375000000,  349.76463786468480066105257719755172729492187500000000,  345.37835684028709692938718944787979125976562500000000,  340.95954116557345514593180269002914428710937500000000,  336.50810037377988237494719214737415313720703125000000,  332.02393018075673580824513919651508331298828125000000,  327.50691213810341650969348847866058349609375000000000,  322.95691324442702807573368772864341735839843750000000,
        318.37378551194677811508881859481334686279296875000000,  313.75736548529158653764170594513416290283203125000000,  309.10747370891880336785106919705867767333984375000000,  304.42391413911758490939973853528499603271484375000000,  299.70647349602967324244673363864421844482421875000000,  294.95492055052636715117841958999633789062500000000000,  290.16900534010449064226122573018074035644531250000000,  285.34845830719251580376294441521167755126953125000000,  280.49298935237959540245356038212776184082031250000000,  275.60228679406981200372683815658092498779296875000000,
        270.67601622490411727994796819984912872314453125000000,  265.71381925394899781167623586952686309814453125000000,  260.71531212209856676054187119007110595703125000000000,  255.68008417632762530047330074012279510498046875000000,  250.60769618632309629902010783553123474121093750000000,  245.49767848454854402007185854017734527587890625000000,  240.34952890788628110385616309940814971923828125000000,  235.16271051556557836192951072007417678833007812500000,  229.93664905400956399716960731893777847290039062500000,  224.67073013437604345199360977858304977416992187500000,
        219.36429608275551572660333476960659027099609375000000,  214.01664241599814886285457760095596313476562500000000,  208.62701388769048094218305777758359909057617187500000,  203.19460003853171770060725975781679153442382812500000,  197.71853017280707831559993792325258255004882812500000,  192.19786766721966841942048631608486175537109375000000,  186.63160349924632441798166837543249130249023437500000,  181.01864885838980967491806950420141220092773437500000,  175.35782667385078070765302982181310653686523437500000,  169.64786185440181043304619379341602325439453125000000,
        163.88736998813612899539293721318244934082031250000000,  158.07484418789789515358279459178447723388671875000000,  152.20863968788546571886399760842323303222656250000000,  146.28695569159333444986259564757347106933593750000000,  140.30781383159396114024275448173284530639648437500000,  134.26903241424622592603554949164390563964843750000000,  128.16819536762247366823430638760328292846679687500000,  122.00261445958547312784503446891903877258300781250000,  115.76928286078494068078725831583142280578613281250000,  109.46481742595753416935622226446866989135742187500000,
        103.08538604804924432301049819216132164001464843750000,   96.62661492766855531044711824506521224975585937500000,   90.08346830204658317597932182252407073974609375000000,   83.45008958789625808094569947570562362670898437500000,   76.71958710904209510772489011287689208984375000000000,   69.88373792249930716025119181722402572631835937500000,   62.93256644395306409478507703170180320739746093750000,   55.85372382093763832244803779758512973785400390625000,   48.63153430359000140015268698334693908691406250000000,   41.24545017428128801384445978328585624694824218750000,
         33.66737111032676210697900387458503246307373046875000,   25.85654467780191723136340442579239606857299804687500,   17.74850094347318574250493838917464017868041992187500,    9.22552450730122153288448316743597388267517089843750,    0.00000000000000000000000000000000000000000000000000,  570.08772572513419163442449644207954406738281250000000,  571.80036130618759671051520854234695434570312500000000,  572.76507652991824670607456937432289123535156250000000,  573.27629636372114418918499723076820373535156250000000,  573.45679923891657381318509578704833984375000000000000,
        573.37496780047558786463923752307891845703125000000000,  573.07467092217723347857827320694923400878906250000000,  572.58655194775201380252838134765625000000000000000000,  571.93327520443597222765674814581871032714843750000000,  571.13230742233645287342369556427001953125000000000000,  570.19753415560342091339407488703727722167968750000000,  569.14026532205514286033576354384422302246093750000000,  567.96989449031309504789533093571662902832031250000000,  566.69434963621131373656680807471275329589843750000000,  565.32041204647146059869555756449699401855468750000000,
        563.85394845186408474546624347567558288574218750000000,  562.30008411544986302033066749572753906250000000000000,  560.66333458432279712724266573786735534667968750000000,  558.94770778600980065675685182213783264160156250000000,  557.15678439242526565067237243056297302246093750000000,  555.29378195735421286372002214193344116210937500000000,  553.36160673651568231434794142842292785644531250000000,  551.36289601868224963254760950803756713867187500000000,  549.30005304945700572716305032372474670410156250000000,  547.17527610319120867643505334854125976562500000000000,
        544.99058288151161377754760906100273132324218750000000,  542.74783114255080818111309781670570373535156250000000,  540.44873626244566366949584335088729858398437500000000,  538.09488627924565662397071719169616699218750000000000,  535.68775485480784936953568831086158752441406250000000,  533.22871250262585363088874146342277526855468750000000,  530.71903636183856178831774741411209106445312500000000,  528.15991874486473989236401394009590148925781250000000,  525.55247464457863770803669467568397521972656250000000,  522.89774835400055508216610178351402282714843750000000,
        520.19671932515905155014479532837867736816406250000000,  517.45030737260219666495686396956443786621093750000000,  514.65937730987320719577837735414505004882812500000000,  511.82474309328620165615575388073921203613281250000000,  508.94717153585992264197557233273983001708984375000000,  506.02738564481705907383002340793609619140625000000000,  503.06606762821326128687360323965549468994140625000000,  500.06386160972982679595588706433773040771484375000000,  497.02137608519711875487701036036014556884765625000000,  493.93918614981828341115033254027366638183593750000000,
        490.81783552118190527835395187139511108398437500000000,  487.65783837985827631200663745403289794921875000000000,  484.45968104657293906711856834590435028076171875000000,  481.22382351255606636186712421476840972900390625000000,  477.95070083760924717353191226720809936523437500000000,  474.64072442866739720557234250009059906005859375000000,  471.29428321010129820933798328042030334472656250000000,  467.91174469568625227111624553799629211425781250000000,  464.49345597100847271576640196144580841064453125000000,  461.03974459407959329837467521429061889648437500000000,
        457.55091942104809277225285768508911132812500000000000,  454.02727136312682887364644557237625122070312500000000,  450.46907408017608531736186705529689788818359375000000,  446.87658461578058677332592196762561798095703125000000,  443.25004397812620027252705767750740051269531250000000,  439.58967767050995689714909531176090240478515625000000,  435.89569617489121355902170762419700622558593750000000,  432.16829539151507333372137509286403656005859375000000,  428.40765703729545066380524076521396636962890625000000,  424.61394900533809959597419947385787963867187500000000,
        420.78732568770249145018169656395912170410156250000000,  416.92792826324466659571044147014617919921875000000000,  413.03588495214881959327613003551959991455078125000000,  409.11131123853670032985974103212356567382812500000000,  405.15431006234251753994612954556941986083984375000000,  401.16497198144878666425938718020915985107421875000000,  397.14337530490257677229237742722034454345703125000000,  393.08958619785585142381023615598678588867187500000000,  389.00365875871273146913154050707817077636718750000000,  384.88563506880331033244146965444087982177734375000000,
        380.73554521474750345078064128756523132324218750000000,  376.55340728351649204341811127960681915283203125000000,  372.33922733004055771743878722190856933593750000000000,  368.09299931705515973590081557631492614746093750000000,  363.81470502671277245099190622568130493164062500000000,  359.50431394331991441504214890301227569580078125000000,  355.16178310638173343249945901334285736083984375000000,  350.78705693295074752313666976988315582275390625000000,  346.38006700807773086125962436199188232421875000000000,  341.94073184195053727307822555303573608398437500000000,
        337.46895659207609696750296279788017272949218750000000,  332.96463274861145009708707220852375030517578125000000,  328.42763778067455859854817390441894531250000000000000,  323.85783474116431079892208799719810485839843750000000,  319.25507182728551924810744822025299072265625000000000,  314.61918189360136466348194517195224761962890625000000,  309.94998191402135034877574071288108825683593750000000,  305.24727238866603329370263963937759399414062500000000,  300.51083669102354178903624415397644042968750000000000,  295.74044035021933041207375936210155487060546875000000,
        290.93583026254481183059397153556346893310546875000000,  286.09673382562255028460640460252761840820312500000000,  281.22285798770565179438563063740730285644531250000000,  276.31388820360132285713916644454002380371093750000000,  271.36948728754930471041006967425346374511718750000000,  266.38929415204233919212128967046737670898437500000000,  261.37292242002592956851003691554069519042968750000000,  256.31995889610425365390256047248840332031250000000000,  251.22996188027161679201526567339897155761718750000000,  246.10245930521492141451744828373193740844726562500000,
        240.93694667532432163170597050338983535766601562500000,  235.73288478211301821829692926257848739624023437500000,  230.48969716667193097237031906843185424804687500000000,  225.20676729492814160948910284787416458129882812500000,  219.88343540566444289652281440794467926025390625000000,  214.51899498426575974008301272988319396972656250000000,  209.11268880670840530910936649888753890991210937500000,  203.66370448803695580863859504461288452148437500000000,  198.17116945702119323868828359991312026977539062500000,  192.63414526325084352720296010375022888183593750000000,
        187.05162110382900664262706413865089416503906250000000,  181.42250643303296442354621831327676773071289062500000,  175.74562248846217471509589813649654388427734375000000,  170.01969252945241350971627980470657348632812500000000,  164.24333053542520133305515628308057785034179687500000,  158.41502804997679731968673877418041229248046875000000,  152.53313877619490313009009696543216705322265625000000,  146.59586042336644595707184635102748870849609375000000,  140.60121316557493287291435990482568740844726562500000,  134.54701388527161043384694494307041168212890625000000,
        128.43084512011046172119677066802978515625000000000000,  122.25001727997592126939707668498158454895019531250000,  116.00152220897122390397271374240517616271972656250000,  109.68197546574916145800671074539422988891601562500000,  103.28754367670235581044835271313786506652832031250000,   96.81385180451999872275337111204862594604492187500000,   90.25586287628536297233949881047010421752929687500000,   83.60771912550468698555050650611519813537597656250000,   76.86252771893764190735964803025126457214355468750000,   70.01206458190668513452692423015832901000976562500000,
         63.04635302303470467677470878697931766510009765625000,   55.95304310670965008966959430836141109466552734375000,   48.71645802314940709720758604817092418670654296875000,   41.31604901736921675592384417541325092315673828125000,   33.72371475125132889161250204779207706451416015625000,   25.89870179674335304298438131809234619140625000000000,   17.77653924719296796297385299112647771835327148437500,    9.23951074927596138763874478172510862350463867187500,    0.00000000000000000000000000000000000000000000000000,  575.05753902471019500808324664831161499023437500000000,
        576.77142652826614721561782062053680419921875000000000,  577.73753280396761056181276217103004455566406250000000,  578.25026603804064961877884343266487121582031250000000,  578.43239336864132837945362553000450134277343750000000,  578.35228945872347594558959826827049255371093750000000,  578.05381733528884069528430700302124023437500000000000,  577.56761594711326779361115768551826477050781250000000,  576.91634625713300010829698294401168823242187500000000,  576.11747238953398664307314902544021606445312500000000,  575.18487786777268411242403090000152587890625000000000,
        574.12987102682291151722893118858337402343750000000000,  572.96184420918348223494831472635269165039062500000000,  571.68872445449767383252037689089775085449218750000000,  570.31729235324962701270123943686485290527343750000000,  568.85341414184904351714067161083221435546875000000000,  567.30221476141173297946806997060775756835937500000000,  565.66820958640198568900814279913902282714843750000000,  563.95540650270652349718147888779640197753906250000000,  562.16738625691357356117805466055870056152343750000000,  560.30736658194143728906055912375450134277343750000000,
        558.37825400746726245415629819035530090332031250000000,  556.38268618317351865698583424091339111328125000000000,  554.32306679606915622571250423789024353027343750000000,  552.20159463710342606646008789539337158203125000000000,  550.02028799533843539393274113535881042480468750000000,  547.78100528360846510622650384902954101562500000000000,  545.48546259710838057799264788627624511718750000000000,  543.13524875494954358146060258150100708007812500000000,  540.73183826017941555619472637772560119628906250000000,  538.27660252614793989778263494372367858886718750000000,
        535.77081964941271507996134459972381591796875000000000,  533.21568295658607894438318908214569091796875000000000,  530.61230851100617655902169644832611083984375000000000,  527.96174173217445968475658446550369262695312500000000,  525.26496325459652325662318617105484008789062500000000,  522.52289413148218955029733479022979736328125000000000,  519.73640047160722588159842416644096374511718750000000,  516.90629758365741963643813505768775939941406250000000,  514.03335369090746098663657903671264648437500000000000,  511.11829326963055564192472957074642181396484375000000,
        508.16180005680081421814975328743457794189453125000000,  505.16451976611620011681225150823593139648437500000000,  502.12706254590875687426887452602386474609375000000000,  499.05000520791031703993212431669235229492187500000000,  495.93389325196130812400951981544494628906250000000000,  492.77924270846091303610592149198055267333984375000000,  489.58654181755366607831092551350593566894531250000000,  486.35625256165360497107030823826789855957031250000000,  483.08881206585562040345394052565097808837890625000000,  479.78463387901246051114867441356182098388671875000000,
        476.44410914673403567576315253973007202148437500000000,  473.06760768623860258230706676840782165527343750000000,  469.65547897183876102644717320799827575683593750000000,  466.20805303883980741375125944614410400390625000000000,  462.72564131275180443481076508760452270507812500000000,  459.20853736994587279696133919060230255126953125000000,  455.65701763520712574973003938794136047363281250000000,  452.07134202103696907215635292232036590576171875000000,  448.45175451302708324874402023851871490478515625000000,  444.79848370515486521981074474751949310302734375000000,
        441.11174328842884051482542417943477630615234375000000,  437.39173249593397940770955756306648254394531250000000,  433.63863650698863239085767418146133422851562500000000,  429.85262681281756158568896353244781494140625000000000,  426.03386154586661405119230039417743682861328125000000,  422.18248577463168658141512423753738403320312500000000,  418.29863176564157356551731936633586883544921875000000,  414.38241921401981926464941352605819702148437500000000,  410.43395544385106177287525497376918792724609375000000,  406.45333557939159163652220740914344787597656250000000,
        402.44064268798786088154884055256843566894531250000000,  398.39594789540120700621628202497959136962890625000000,  394.31931047407442747498862445354461669921875000000000,  390.21077790472475044225575402379035949707031250000000,  386.07038591149370176935917697846889495849609375000000,  381.89815847073606391859357245266437530517578125000000,  377.69410779338227257539983838796615600585937500000000,  373.45823428065546067955438047647476196289062500000000,  369.19052645277469082429888658225536346435546875000000,  364.89096085011539116749190725386142730712890625000000,
        360.55950190613521044724620878696441650390625000000000,  356.19610179120246584716369397938251495361328125000000,  351.80070022627910475421231240034103393554687500000000,  347.37322426521831175705301575362682342529296875000000,  342.91358804422634420916438102722167968750000000000000,  338.42169249681029441489954479038715362548828125000000,  333.89742503228796977055026218295097351074218750000000,  329.34065917566192638332722708582878112792968750000000,  324.75125416636183217633515596389770507812500000000000,  320.12905451302685833070427179336547851562500000000000,
        315.47388950112849670404102653264999389648437500000000,  310.78557264982214292103890329599380493164062500000000,  306.06390111394904351982404477894306182861328125000000,  301.30865502658690502357785589993000030517578125000000,  296.51959677695396067065303213894367218017578125000000,  291.69647021779849183076294139027595520019531250000000,  286.83899979563483384481514804065227508544921875000000,  281.94688959631423585960874333977699279785156250000000,  277.01982229740576713084010407328605651855468750000000,  272.05745801770757452686666510999202728271484375000000,
        267.05943305286626809902372770011425018310546875000000,  262.02535848453067046648357063531875610351562500000000,  256.95481864865877241754787974059581756591796875000000,  251.84736944648778944610967300832271575927734375000000,  246.70253647920654316294530872255563735961914062500000,  241.51981298445824108966917265206575393676757812500000,  236.29865754936901112159830518066883087158203125000000,  231.03849157072116327071853447705507278442382812500000,  225.73869642803359170102339703589677810668945312500000,  220.39861032950182107015280053019523620605468750000000,
        215.01752478375757959838665556162595748901367187500000,  209.59468064195922920589509885758161544799804687500000,  204.12926364445300464467436540871858596801757812500000,  198.62039939369341823294234927743673324584960937500000,  193.06714765967620905939838849008083343505859375000000,  187.46849590504083948871993925422430038452148437500000,  181.82335189320806989599077496677637100219726562500000,  176.13053521306838433702068869024515151977539062500000,  170.38876751599750036803015973418951034545898437500000,  164.59666121286500128917396068572998046875000000000000,
        158.75270631683713418169645592570304870605468750000000,  152.85525503745992637050221674144268035888671875000000,  146.90250362618280632887035608291625976562500000000000,  140.89247083381908964838657993823289871215820312500000,  134.82297215302574500128685031086206436157226562500000,  128.69158876408914693456608802080154418945312500000000,  122.49562975094531225295213516801595687866210937500000,  116.23208566219918225215224083513021469116210937500000,  109.89757079052373001104569993913173675537109375000000,  103.48825052495402587737771682441234588623046875000000,
         96.99974861858181895968300523236393928527832031250000,   90.42702691582290697169810300692915916442871093750000,   83.76422649364718608921975828707218170166015625000000,   77.00445338763947233928774949163198471069335937500000,   70.13948241656767379481607349589467048645019531250000,   63.15933580640611211265422753058373928070068359375000,   56.05166256211168729350902140140533447265625000000000,   48.80078483702503433505626162514090538024902343750000,   41.38615286119776470741271623410284519195556640625000,   33.77966430284470789047190919518470764160156250000000,
         25.94056476094642604834916710387915372848510742187500,   17.80438237600894169077037076931446790695190429687500,    9.25339986143662862616565689677372574806213378906250,    0.00000000000000000000000000000000000000000000000000,  580.03427276713080118497600778937339782714843750000000,  581.74940201003755646524950861930847167968750000000000,  582.71688767347507109661819413304328918457031250000000,  583.23112132023402409686241298913955688476562500000000,  583.41485889574198608897859230637550354003906250000000,  583.33646714773760777461575344204902648925781250000000,
        583.03980330194997350190533325076103210449218750000000,  582.55550194337422453827457502484321594238281250000000,  581.90622069168864527455298230051994323730468750000000,  581.10942107873017903330037370324134826660156250000000,  580.17898460512219571683090180158615112304687500000000,  579.12621802635203493991866707801818847656250000000000,  577.96051245849844235635828226804733276367187500000000,  576.68979400167529547616140916943550109863281250000000,  575.32084254419874014274682849645614624023437500000000,  573.85952381993286053329939022660255432128906250000000,
        572.31096243786396371433511376380920410156250000000000,  570.67967358783494091767352074384689331054687500000000,  568.96966510044296683190623298287391662597656250000000,  567.18451778175426625239197164773941040039062500000000,  565.32744952716598163533490151166915893554687500000000,  563.40136712226410509174456819891929626464843750000000,  561.40890855824318350641988217830657958984375000000000,  559.35247794280473954131593927741050720214843750000000,  557.23427456148942837899085134267807006835937500000000,  555.05631726750311827345285564661026000976562500000000,
        552.82046510380996551248244941234588623046875000000000,  550.52843485880362095485907047986984252929687500000000,  548.18181610549936522147618234157562255859375000000000,  545.78208415966025768284453079104423522949218750000000,  543.33061130467660859721945598721504211425781250000000,  540.82867656333826289483113214373588562011718750000000,  538.27747424386132024665130302309989929199218750000000,  535.67812144601350610173540189862251281738281250000000,  533.03166468025722224410856142640113830566406250000000,  530.33908572651955637411447241902351379394531250000000,
        527.60130683802776729862671345472335815429687500000000,  524.81919537849989865208044648170471191406250000000000,  521.99356796699510141479549929499626159667968750000000,  519.12519419326861225272295996546745300292968750000000,  516.21479995702043197525199502706527709960937500000000,  513.26307047659190629929071292281150817871093750000000,  510.27065300613645604244084097445011138916015625000000,  507.23815929482691444718511775135993957519531250000000,  504.16616781706704841781174764037132263183593750000000,  501.05522579879550448822556063532829284667968750000000,
        497.90585106168009588145650923252105712890625000000000,  494.71853370420126339013222604990005493164062500000000,  491.49373763622816113638691604137420654296875000000000,  488.23190198164030562111292965710163116455078125000000,  484.93344236178029404982225969433784484863281250000000,  481.59875207099815952460630796849727630615234375000000,  478.22820315422490011769696138799190521240234375000000,  474.82214739536567549293977208435535430908203125000000,  471.38091722429913943415158428251743316650390625000000,  467.90482654939404483229736797511577606201171875000000,
        464.39417152168437041837023571133613586425781250000000,  460.84923123616800921809044666588306427001953125000000,  457.27026837509424694871995598077774047851562500000000,  453.65752979757797902493621222674846649169921875000000,  450.01124707940510916159837506711483001708984375000000,  446.33163700647588711944990791380405426025390625000000,  442.61890202495544599514687433838844299316406250000000,  438.87323065086371798315667547285556793212890625000000,  435.09479784153148784753284417092800140380859375000000,  431.28376533107393697719089686870574951171875000000000,
        427.44028193178155561327002942562103271484375000000000,  423.56448380309603862770018167793750762939453125000000,  419.65649468963130175325204618275165557861328125000000,  415.71642612949852946258033625781536102294921875000000,  411.74437763401476786384591832756996154785156250000000,  407.74043683970188567400327883660793304443359375000000,  403.70467963331952887529041618108749389648437500000000,  399.63717025052028475329279899597167968750000000000000,  395.53796134856798971668467856943607330322265625000000,  391.40709405341027604663395322859287261962890625000000,
        387.24459798125838005944387987256050109863281250000000,  383.05049123468103289269492961466312408447265625000000,  378.82478037307930662791477516293525695800781250000000,  374.56746035726519039599224925041198730468750000000000,  370.27851446771580867789452895522117614746093750000000,  365.95791419592666215976350940763950347900390625000000,  361.60561910812413088933681137859821319580078125000000,  357.22157668043024614235037006437778472900390625000000,  352.80572210439578384466585703194141387939453125000000,  348.35797806162327105994336307048797607421875000000000,
        343.87825446599651968426769599318504333496093750000000,  339.36644817181013422668911516666412353515625000000000,  334.82244264584403481421759352087974548339843750000000,  330.24610760116217988979769870638847351074218750000000,  325.63729859011436928994953632354736328125000000000000,  320.99585655369060077646281570196151733398437500000000,  316.32160732400933511598850600421428680419921875000000,  311.61436107630765945941675454378128051757812500000000,  306.87391172633931546442909166216850280761718750000000,  302.10003626856098435382591560482978820800781250000000,
        297.29249404989803906573797576129436492919921875000000,  292.45102597320709492123569361865520477294921875000000,  287.57535362378439458552747964859008789062500000000000,  282.66517831139475447344011627137660980224609375000000,  277.72018001928722696902696043252944946289062500000000,  272.74001625050590291721164248883724212646484375000000,  267.72432076046464999308227561414241790771484375000000,  262.67270216320230247220024466514587402343750000000000,  257.58474239692998253303812816739082336425781250000000,  252.45999503237129601984634064137935638427734375000000,
        247.29798340492814645585895050317049026489257812500000,  242.09819854879299327876651659607887268066406250000000,  236.86009690769660096520965453237295150756835937500000,  231.58309779290416940966679248958826065063476562500000,  226.26658055421700055376277305185794830322265625000000,  220.90988142392635040778259281069040298461914062500000,  215.51228998667477299022721126675605773925781250000000,  210.07304521973173905280418694019317626953125000000000,  204.59133103791876351351675111800432205200195312500000,  199.06627126486873180510883685201406478881835937500000,
        193.49692393686845548472774680703878402709960937500000,  187.88227482643850407839636318385601043701171875000000,  182.22123004901163767499383538961410522460937500000000,  176.51260758622333923995029181241989135742187500000000,  170.75512752158672924451821018010377883911132812500000,  164.94740073621565557004942093044519424438476562500000,  159.08791575039430199467460624873638153076171875000000,  153.17502331647648361467872746288776397705078125000000,  147.20691826327305307131609879434108734130859375000000,  141.18161795242110656545264646410942077636718750000000,
        135.09693651981513085047481581568717956542968750000000,  128.95045382038551906589418649673461914062500000000000,  122.73947764315009578695025993511080741882324218750000,  116.46099727130213352666032733395695686340332031250000,  110.11162576071362195762048941105604171752929687500000,  103.68752729136735979409422725439071655273437500000000,   97.18432443422189237480779411271214485168457031250000,   90.59697787768020305065874708816409111022949218750000,   83.91962756805044421071215765550732612609863281250000,   77.14537843486588997166109038516879081726074218750000,
         70.26600421473899871216417523100972175598144531250000,   63.27152607469659528760530520230531692504882812500000,   56.14959198329246703451644862070679664611816406250000,   48.88452307936869090099207824096083641052246093750000,   41.45576859976566908017048262991011142730712890625000,   33.83522524017870836132715339772403240203857421875000,   25.98213764718583718149602646008133888244628906250000,   17.83203302845255322495177097152918577194213867187500,    9.26719318356896515354037546785548329353332519531250,    0.00000000000000000000000000000000000000000000000000,
        585.01787938883899187203496694564819335937500000000000,  586.73424034180538910732138901948928833007812500000000,  587.70309390900877133390167728066444396972656250000000,  588.21881518507814234908437356352806091308593750000000,  588.40414902167640320840291678905487060546875000000000,  588.32745431723265028267633169889450073242187500000000,  588.03258254117122305615339428186416625976562500000000,  587.55016394558811043680179864168167114257812500000000,  586.90285282782667763967765495181083679199218750000000,  586.10810814095566456671804189682006835937500000000000,
        585.17980937073980385321192443370819091796875000000000,  584.12926169673085041722515597939491271972656250000000,  582.96585500855348982440773397684097290039062500000000,  581.69751446377608772309031337499618530273437500000000,  580.33101924299296570097794756293296813964843750000000,  578.87223456976289526210166513919830322265625000000000,  577.32628471122745850152568891644477844238281250000000,  575.69768466112907390197506174445152282714843750000000,  573.99044218167932740470860153436660766601562500000000,  572.20813812382107244047801941633224487304687500000000,
        570.35399052940999808924971148371696472167968750000000,  568.43090642256402134080417454242706298828125000000000,  566.44152411730533458467107266187667846679687500000000,  564.38824812206917158619035035371780395507812500000000,  562.27327819577465106704039499163627624511718750000000,  560.09863373331779712316347286105155944824218750000000,  557.86617438410053182451520115137100219726562500000000,  555.57761760478160795173607766628265380859375000000000,  553.23455369609371246042428538203239440917968750000000,  550.83845875906024502910440787672996520996093750000000,
        548.39070591836605217395117506384849548339843750000000,  545.89257509296953685407061129808425903320312500000000,  543.34526154127661357051692903041839599609375000000000,  540.74988336668627653125440701842308044433593750000000,  538.10748813639509080530842766165733337402343750000000,  535.41905874005306031904183328151702880859375000000000,  532.68551859368471923517063260078430175781250000000000,  529.90773627715293514484073966741561889648437500000000,  527.08652967945772616076283156871795654296875000000000,  524.22266971470367025176528841257095336914062500000000,
        521.31688366211835727881407365202903747558593750000000,  518.36985817567097001301590353250503540039062500000000,  515.38224200231070426525548100471496582031250000000000,  512.35464844238845216750632971525192260742187500000000,  509.28765758122307261146488599479198455810546875000000,  506.18181831690469607565319165587425231933593750000000,  503.03765020613121805581613443791866302490234375000000,  499.85564514707880334754008799791336059570312500000000,  496.63626891591309231444029137492179870605468750000000,  493.37996257149762868721154518425464630126953125000000,
        490.08714374108888023329200223088264465332031250000000,  486.75820779828467266270308755338191986083984375000000,  483.39352894316868969326606020331382751464843750000000,  479.99346119345062788852374069392681121826171875000000,  476.55833929439364737845608033239841461181640625000000,  473.08847955445241950656054541468620300292968750000000,  469.58418061277109245565952733159065246582031250000000,  466.04572414401718560839071869850158691406250000000000,  462.47337550543062434371677227318286895751953125000000,  458.86738433043723262017010711133480072021484375000000,
        455.22798507270721302120364271104335784912109375000000,  451.55539750412106059229699894785881042480468750000000,  447.84982716972899652319028973579406738281250000000000,  444.11146580245605264281039126217365264892578125000000,  440.34049170000128015090012922883033752441406250000000,  436.53707006610318330785958096385002136230468750000000,  432.70135331809734680064138956367969512939453125000000,  428.83348136246240756008774042129516601562500000000000,  424.93358183984133802368887700140476226806640625000000,  421.00177034083179705703514628112316131591796875000000,
        417.03815059366030482124187983572483062744140625000000,  413.04281462468503605123260058462619781494140625000000,  409.01584289251400150533299893140792846679687500000000,  404.95730439637463859980925917625427246093750000000000,  400.86725675922434675158001482486724853515625000000000,  396.74574628595382819185033440589904785156250000000000,  392.59280799689332752677728421986103057861328125000000,  388.40846563670100977105903439223766326904296875000000,  384.19273165857407548173796385526657104492187500000000,  379.94560718358741269184974953532218933105468750000000,
        375.66708193482742217383929528295993804931640625000000,  371.35713414584165548149030655622482299804687500000000,  367.01573044278148927332949824631214141845703125000000,  362.64282569945419254509033635258674621582031250000000,  358.23836286434044495763373561203479766845703125000000,  353.80227275845470558124361559748649597167968750000000,  349.33447384273887337258202023804187774658203125000000,  344.83487195347538545320276170969009399414062500000000,  340.30336000398369833419565111398696899414062500000000,  335.73981765062154636325431056320667266845703125000000,
        331.14411092084469601104501634836196899414062500000000,  326.51609180078270355807035230100154876708984375000000,  321.85559777946031090323231182992458343505859375000000,  317.16245134642605307817575521767139434814453125000000,  312.43645943913998053176328539848327636718750000000000,  307.67741283600855695112841203808784484863281250000000,  302.88508549043439188608317635953426361083984375000000,  298.05923380065650007963995449244976043701171875000000,  293.19959580948551547407987527549266815185546875000000,  288.30589032727249332310748286545276641845703125000000,
        283.37781597057318094812217168509960174560546875000000,  278.41505010796447550092125311493873596191406250000000,  273.41724770331137506218510679900646209716796875000000,  268.38404004544452163827372714877128601074218750000000,  263.31503335165638191028847359120845794677734375000000,  258.20980723061984463129192590713500976562500000000000,  253.06791298822261637724295724183320999145507812500000,  247.88887175734325296616589184850454330444335937500000,  242.67217242968357027166348416358232498168945312500000,  237.41726936434000094777729827910661697387695312500000,
        232.12357984372167152287147473543882369995117187500000,  226.79048124256652840813330840319395065307617187500000,  221.41730786999769975409435573965311050415039062500000,  216.00334743757062483382469508796930313110351562500000,  210.54783709781332845523138530552387237548828125000000,  205.04995898749078264700074214488267898559570312500000,  199.50883519727457837689144071191549301147460937500000,  193.92352207406329966943303588777780532836914062500000,  188.29300374310392385268642101436853408813476562500000,  182.61618471327250290414667688310146331787109375000000,
        176.89188139902489638188853859901428222656250000000000,  171.11881235478645635339489672333002090454101562500000,  165.29558696944241091841831803321838378906250000000000,  159.42069230672402113668795209378004074096679687500000,  153.49247769697228704899316653609275817871093750000000,  147.50913658043410237041825894266366958618164062500000,  141.46868496258360892170458100736141204833984375000000,  135.36893565454619192678364925086498260498046875000000,  129.20746721690878189292561728507280349731445312500000,  122.98158617384042656794918002560734748840332031250000,
        116.68828057228476779982884181663393974304199218750000,  110.32416225960169242625852348282933235168457031250000,  103.88539423417012130812508985400199890136718750000000,   97.36759791145441056414711056277155876159667968750000,   90.76573284970422150763624813407659530639648437500000,   84.07393788967448244875413365662097930908203125000000,   77.28531687923161541675654007121920585632324218750000,   70.39164249651960858500387985259294509887695312500000,   63.38293487263200631787185557186603546142578125000000,   56.24684096208483907730624196119606494903564453125000,
         48.96768091096095076864003203809261322021484375000000,   41.52490298402464929949928773567080497741699218750000,   33.89040292500379081275241333059966564178466796875000,   26.02342444806282983904566208366304636001586914062500,   17.85949384747200596734728605952113866806030273437500,    9.28089202792712697487331752199679613113403320312500,    0.00000000000000000000000000000000000000000000000000,  590.00831197561785756988683715462684631347656250000000,  591.72589475983852480567293241620063781738281250000000,  592.69610492307447202620096504688262939453125000000000,
        593.21330124465680455614347010850906372070312500000000,  593.40021758000636964425211772322654724121093750000000,  593.32520504326669197325827553868293762207031250000000,  593.03210939199607310001738369464874267578125000000000,  592.55155657598072593827964738011360168457031250000000,  591.90619759102833086217287927865982055664062500000000,  591.11348882502511514758225530385971069335937500000000,  590.18730775695257761981338262557983398437500000000000,  589.13895799416070531151490285992622375488281250000000,  587.97782820003612869186326861381530761718750000000000,
        586.71184258689288526511518284678459167480468750000000,  585.34777962240866600041044875979423522949218750000000,  583.89150401248321031744126230478286743164062500000000,  582.34813967314767069183290004730224609375000000000000,  580.72220139105615999142173677682876586914062500000000,  579.01769684746909661043900996446609497070312500000000,  577.23820692417416466923896223306655883789062500000000,  575.38694979407114260538946837186813354492187500000000,  573.46683270307960356149123981595039367675781250000000,  571.48049427004877998115262016654014587402343750000000,
        569.43033938491009848803514614701271057128906250000000,  567.31856825951069822622230276465415954589843750000000,  565.14720080878475982899544760584831237792968750000000,  562.91809726572341787687037140130996704101562500000000,  560.63297573120303241012152284383773803710937500000000,  558.29342720841793834551936015486717224121093750000000,  555.90092855717318798269843682646751403808593750000000,  553.45685371572585609101224690675735473632812500000000,  550.96248347021332847361918538808822631835937500000000,  548.41901399894459245842881500720977783203125000000000,
        545.82756437732996346312575042247772216796875000000000,  543.18918319631234226108063012361526489257812500000000,  540.50485442086301191011443734169006347656250000000000,  537.77550259394297427206765860319137573242187500000000,  535.00199747418923834629822522401809692382812500000000,  532.18515818160540220560505986213684082031250000000000,  529.32575691408214879629667848348617553710937500000000,  526.42452228812169323646230623126029968261718750000000,  523.48214234930787824850995093584060668945312500000000,  520.49926729153992255305638536810874938964843750000000,
        517.47651191858642505394527688622474670410156250000000,  514.41445787692441626859363168478012084960937500000000,  511.31365568495050411002011969685554504394531250000000,  508.17462658036254197213565930724143981933593750000000,  504.99786420471480141713982447981834411621093750000000,  501.78383614175493221409851685166358947753906250000000,  498.53298532410173038442735560238361358642578125000000,  495.24573132105814465830917470157146453857421875000000,  491.92247151883037759034777991473674774169921875000000,  488.56358220310306705869152210652828216552734375000000,
        485.16941955277309261873597279191017150878906250000000,  481.74032055264592599996831268072128295898437500000000,  478.27660383202209004593896679580211639404296875000000,  474.77857043533373371246852912008762359619140625000000,  471.24650453031853203356149606406688690185546875000000,  467.68067405861927454679971560835838317871093750000000,  464.08133133317255669680889695882797241210937500000000,  460.44871358628029156534466892480850219726562500000000,  456.78304347183973277424229308962821960449218750000000,  453.08452952483708031650166958570480346679687500000000,
        449.35336658087192063248949125409126281738281250000000,  445.58973615818098323870799504220485687255859375000000,  441.79380680435502881664433516561985015869140625000000,  437.96573440969723378657363355159759521484375000000000,  434.10566248894355112497578375041484832763671875000000,  430.21372243286049297239514999091625213623046875000000,  426.29003373104313823205302469432353973388671875000000,  422.33470416706052219524281099438667297363281250000000,  418.34782998692963928988319821655750274658203125000000,  414.32949604174291380331851541996002197265625000000000,
        410.27977590512836059133405797183513641357421875000000,  406.19873196607898080401355400681495666503906250000000,  402.08641549755299138269037939608097076416015625000000,  397.94286670111478088074363768100738525390625000000000,  393.76811472775398215162567794322967529296875000000000,  389.56217767489454217866295948624610900878906250000000,  385.32506255947316731180762872099876403808593750000000,  381.05676526683652127758250571787357330322265625000000,  376.75727047507331235465244390070438385009765625000000,  372.42655155425677548919338732957839965820312500000000,
        368.06457043992929811793146654963493347167968750000000,  363.67127748000973497255472466349601745605468750000000,  359.24661125414087337048840709030628204345703125000000,  354.79049836432290021548396907746791839599609375000000,  350.30285319549233236102736555039882659912109375000000,  345.78357764450464628680492751300334930419921875000000,  341.23256081575931375482468865811824798583984375000000,  336.64967868146362661718740127980709075927734375000000,  332.03479370426879313527024351060390472412109375000000,  327.38775441971387181183672510087490081787109375000000,
        322.70839497558961284084944054484367370605468750000000,  317.99653462496536349135567434132099151611328125000000,  313.25197716921616120089311152696609497070312500000000,  308.47451034692159055339288897812366485595703125000000,  303.66390516399138732595019973814487457275390625000000,  298.81991515978029383404646068811416625976562500000000,  293.94227560328420167934382334351539611816406250000000,  289.03070261274564245468354783952236175537109375000000,  284.08489219111993406841065734624862670898437500000000,  279.10451916884881029545795172452926635742187500000000,
        274.08923604423114284145412966609001159667968750000000,  269.03867171034232796955620869994163513183593750000000,  263.95243005590305074292700737714767456054687500000000,  258.83008842569228136198944412171840667724609375000000,  253.67119592399313887653988786041736602783203125000000,  248.47527154208989941253094002604484558105468750000000,  243.24180208792458302013983484357595443725585937500000,  237.97023989259093923465115949511528015136718750000000,  232.66000026426789304423436988145112991333007812500000,  227.31045865533866390251205302774906158447265625000000,
        221.92094750263365199316467624157667160034179687500000,  216.49075269374270646949298679828643798828125000000000,  211.01910960389528781888657249510288238525390625000000,  205.50519863763580019622168038040399551391601562500000,  199.94814019697153639754105824977159500122070312500000,  194.34698898223516039251990150660276412963867187500000,  188.70072751280881107049935963004827499389648437500000,  183.00825873106549579460988752543926239013671875000000,  177.26839752303513364495302084833383560180664062500000,  171.47986095156178976139926817268133163452148437500000,
        165.64125694961123258508450817316770553588867187500000,  159.75107115952172875950054731220006942749023437500000,  153.80765152367703763047757092863321304321289062500000,  147.80919012675502699494245462119579315185546875000000,  141.75370165004218847570882644504308700561523437500000,  135.63899761088990203461435157805681228637695312500000,  129.46265530559395529053290374577045440673828125000000,  123.22198002262005900320218643173575401306152343750000,  116.91395860093501823939732275903224945068359375000000,  110.53520170681916567900771042332053184509277343750000,
        104.08187118367118273454252630472183227539062500000000,   97.54958731730579302166006527841091156005859375000000,   90.93330856091318992184824310243129730224609375000000,   84.22717267406557084541418589651584625244140625000000,   77.42428244663554437465791124850511550903320312500000,   70.51640952129932315983751323074102401733398437500000,   63.49357301556926813645986840128898620605468750000000,   56.34341889165001049377679009921848773956298828125000,   49.05026632398770658483044826425611972808837890625000,   41.59356262581032126490754308179020881652832031250000,
         33.94520260885519746807403862476348876953125000000000,   26.06442907430672306645647040568292140960693359375000,   17.88676742194837743227253668010234832763671875000000,    9.29449767998290532489136239746585488319396972656250,    0.00000000000000000000000000000000000000000000000000,  595.00552424938189233216689899563789367675781250000000,  596.72431913326010999298887327313423156738281250000000,  597.69587475712069135624915361404418945312500000000000,  598.21453373550286869431147351861000061035156250000000,  598.40301902369549225113587453961372375488281250000000,
        598.32967401571318077913019806146621704101562500000000,  598.03833880116405907756416127085685729980468750000000,  597.55963505782312950032064691185951232910156250000000,  596.91621050063531583873555064201354980468750000000000,  596.12551896588684030575677752494812011718750000000000,  595.20143593394504932803101837635040283203125000000000,  594.15526344386375967587810009717941284179687500000000,  592.99638893324322452826891094446182250976562500000000,  591.73273566672753531747730448842048645019531250000000,  590.37108139422628028114559128880500793457031250000000,
        588.91729029701389208639739081263542175292968750000000,  587.37648593117637574323453009128570556640625000000000,  585.75318286576191439962713047862052917480468750000000,  584.05138868902201920718653127551078796386718750000000,  582.27468430010276279062964022159576416015625000000000,  580.42628798811722390382783487439155578613281250000000,  578.50910720467459213978145271539688110351562500000000,  576.52578085610502967028878629207611083984375000000000,  574.47871419529269587656017392873764038085937500000000,  572.37010786729115352500230073928833007812500000000000,
        570.20198228619051405985374003648757934570312500000000,  567.97619824653042996942531317472457885742187500000000,  565.69447447020684194285422563552856445312500000000000,  563.35840263851127929228823632001876831054687500000000,  560.96946034448671980499057099223136901855468750000000,  558.52902231322138959512813016772270202636718750000000,  556.03837017007003851176705211400985717773437500000000,  553.49870098403471274650655686855316162109375000000000,  550.91113477204817172605544328689575195312500000000000,  548.27672111699382639926625415682792663574218750000000,
        545.59644502600338000775082036852836608886718750000000,  542.87123213441361713194055482745170593261718750000000,  540.10195334362663288629846647381782531738281250000000,  537.28942896714056587370578199625015258789062500000000,  534.43443244756736021372489631175994873046875000000000,  531.53769369800068034237483516335487365722656250000000,  528.59990211327351516956696286797523498535156250000000,  525.62170929011722364521119743585586547851562500000000,  522.60373148977328128239605575799942016601562500000000,  519.54655187202513388911029323935508728027343750000000,
        516.45072252573095283878501504659652709960937500000000,  513.31676631766117679944727569818496704101562500000000,  510.14517857864120742306113243103027343750000000000000,  506.93642864360981548088602721691131591796875000000000,  503.69096126015716663459897972643375396728515625000000,  500.40919787833831833268050104379653930664062500000000,  497.09153783303861473541473969817161560058593750000000,  493.73835942884494443205767311155796051025390625000000,  490.35002093623347718676086515188217163085937500000000,  486.92686150688200541480910032987594604492187500000000,
        483.46920201504332226249971427023410797119140625000000,  479.97734583114942097381572239100933074951171875000000,  476.45157953313997722943895496428012847900390625000000,  472.89217356041706352698383852839469909667968750000000,  469.29938281479854822464403696358203887939453125000000,  465.67344721237759586074389517307281494140625000000000,  462.01459218977703358177677728235721588134765625000000,  458.32302916791923053096979856491088867187500000000000,  454.59895597609380502035492099821567535400390625000000,  450.84255723881051380885764956474304199218750000000000,
        447.05400472764989672214142046868801116943359375000000,  443.23345768008050526987062767148017883300781250000000,  439.38106308698803559309453703463077545166015625000000,  435.49695595045449181270669214427471160888671875000000,  431.58125951313917312290868721902370452880859375000000,  427.63408546043797286984045058488845825195312500000000,  423.65553409643410986973321996629238128662109375000000,  419.64569449450249294386594556272029876708984375000000,  415.60464462328394574797130189836025238037109375000000,  411.53245144860994741975446231663227081298828125000000,
        407.42917101182609940224210731685161590576171875000000,  403.29484848483440373456687666475772857666015625000000,  399.12951820205023523158160969614982604980468750000000,  394.93320366934659659818862564861774444580078125000000,  390.70591754993330368961323983967304229736328125000000,  386.44766162699750111642060801386833190917968750000000,  382.15842674280264645858551375567913055419921875000000,  377.83819271381503313023131340742111206054687500000000,  373.48692822129231672079185955226421356201171875000000,  369.10459067662515053598326630890369415283203125000000,
        364.69112606057723269259440712630748748779296875000000,  360.24646873540802971547236666083335876464843750000000,  355.77054122869350294422474689781665802001953125000000,  351.26325398747712824842892587184906005859375000000000,  346.72450510118233069078996777534484863281250000000000,  342.15417999150270134123275056481361389160156250000000,  337.55215106724330098586506210267543792724609375000000,  332.91827734182697895448654890060424804687500000000000,  328.25240401088211683600093238055706024169921875000000,  323.55436198700635941349901258945465087890625000000000,
        318.82396738843436878596548922359943389892578125000000,  314.06102097792995664349291473627090454101562500000000,  309.26530754776257481353241018950939178466796875000000,  304.43659524610848166048526763916015625000000000000000,  299.57463483962754935419070534408092498779296875000000,  294.67915890629757313945447094738483428955078125000000,  289.74988095182249026038334704935550689697265625000000,  284.78649444205643703753594309091567993164062500000000,  279.78867174288291153061436489224433898925781250000000,  274.75606295782756660628365352749824523925781250000000,
        269.68829465235120323995943181216716766357421875000000,  264.58496845221401372327818535268306732177734375000000,  259.44565950150064281842787750065326690673828125000000,  254.26991476378799461599555797874927520751953125000000,  249.05725114746692838707531336694955825805664062500000,  243.80715343332229849693248979747295379638671875000000,  238.51907197904310464764421340078115463256835937500000,  233.19242017125944244071433786302804946899414062500000,  227.82657159084877207533281762152910232543945312500000,  222.42085685144462559037492610514163970947265625000000,
        216.97456006408924622519407421350479125976562500000000,  211.48691487252546039599110372364521026611328125000000,  205.95709999335056750169314909726381301879882812500000,  200.38423418270707543342723511159420013427734375000000,  194.76737053574831293190072756260633468627929687500000,  189.10549000602375713242508936673402786254882812500000,  183.39749400813613533500756602734327316284179687500000,  177.64219593717547240885323844850063323974609375000000,  171.83831140069398202285810839384794235229492187500000,  165.98444691087863134271174203604459762573242187500000,
        160.07908672271153704969037789851427078247070312500000,  154.12057742359584722180443350225687026977539062500000,  148.10710977459797277333564125001430511474609375000000,  142.03669716379454257548786699771881103515625000000000,  135.90714984479839699815784115344285964965820312500000,  129.71604387874330654994992073625326156616210937500000,  123.46068334664838062053604517132043838500976562500000,  117.13805390692044738898403011262416839599609375000000,  110.74476507136887448723427951335906982421875000000000,  104.27697755424176762062415946274995803833007812500000,
         97.73031053678369062254205346107482910156250000000000,   91.09972139148172232125943992286920547485351562500000,   84.37934682038462597120087593793869018554687500000000,   77.56228857835847634305537212640047073364257812500000,   70.64031729495071942892536753788590431213378906250000,   63.60345109580612898980689351446926593780517578125000,   56.43933497192802661857058410532772541046142578125000,   49.13228714665342522494029253721237182617187500000000,   41.66175400163900377492609550245106220245361328125000,   33.99962943605348186792980413883924484252929687500000,
         26.10515535699833122862401069141924381256103515625000,   17.91385628816042796529472980182617902755737304687500,    9.30801139914962760713024181313812732696533203125000,    0.00000000000000000000000000000000000000000000000000,  600.00947055532742524519562721252441406250000000000000,  601.72946795129053043638123199343681335449218750000000,  602.70235806889490959292743355035781860351562500000000,  603.22246750608621823630528524518013000488281250000000,  603.41250841274552385584684088826179504394531250000000,  603.34081652606391799054108560085296630859375000000000,
        603.05122631109895792178576812148094177246093750000000,  602.57435520362196257337927818298339843750000000000000,  601.93284765825967497221427038311958312988281250000000,  601.14415497327229331858688965439796447753906250000000,  600.22215063866644868539879098534584045410156250000000,  599.17813512926602470543002709746360778808593750000000,  598.02149465756122026505181565880775451660156250000000,  596.76015153839000504376599565148353576660156250000000,  595.40088279936867365904618054628372192382812500000000,  593.94955209055228806391824036836624145507812500000000,
        592.41128259965876168280374258756637573242187500000000,  590.79058866806235528201796114444732666015625000000000,  589.09147777943644541664980351924896240234375000000000,  587.31753083732189679722068831324577331542968750000000,  585.47196623277238813898293301463127136230468750000000,  583.55769160756756264163414016366004943847656250000000,  581.57734613880302276811562478542327880859375000000000,  579.53333542443397163879126310348510742187500000000000,  577.42786052370115612575318664312362670898437500000000,  575.26294232970383291103644296526908874511718750000000,
        573.04044217726618626329582184553146362304687500000000,  570.76207938691811705211875960230827331542968750000000,  568.42944629453893412573961541056632995605468750000000,  566.04402120176155221997760236263275146484375000000000,  563.60717959470412097289226949214935302734375000000000,  561.12020391096143612230662256479263305664062500000000,  558.58429208205029681266751140356063842773437500000000,  556.00056503701728161104256287217140197753906250000000,  553.37007332001348913763649761676788330078125000000000,  550.69380294835605127445887774229049682617187500000000,
        547.97268061643978853680891916155815124511718750000000,  545.20757833372499590041115880012512207031250000000000,  542.39931757106069198925979435443878173828125000000000,  539.54867297814371340791694819927215576171875000000000,  536.65637572547643685538787394762039184570312500000000,  533.72311651634822737833019345998764038085937500000000,  530.74954830785304693563375622034072875976562500000000,  527.73628877449118590448051691055297851562500000000000,  524.68392254331581625592662021517753601074218750000000,  521.59300322570891239593038335442543029785156250000000,
        518.46405526758906034956453368067741394042968750000000,  515.29757563704822587169473990797996520996093750000000,  512.09403536603736029064748436212539672851562500000000,  508.85388096065918261956539936363697052001953125000000,  505.57753569287245909436023794114589691162109375000000,  502.26540078488608287443639710545539855957031250000000,  498.91785649620186404717969708144664764404296875000000,  495.53526312212176208049640990793704986572265625000000,  492.11796191153496238257503136992454528808593750000000,  488.66627591092708371434127911925315856933593750000000,
        485.18051074078886131246690638363361358642578125000000,  481.66095530992828344096778891980648040771484375000000,  478.10788247259552008472383022308349609375000000000000,  474.52154963280469246456050314009189605712890625000000,  470.90219929977081392280524596571922302246093750000000,  467.25005959796476417977828532457351684570312500000000,  463.56534473491728931549005210399627685546875000000000,  459.84825542957366906193783506751060485839843750000000,  456.09897930369919549775659106671810150146484375000000,  452.31769123856724945653695613145828247070312500000000,
        448.50455369891886903133126907050609588623046875000000,  444.65971702595811620994936674833297729492187500000000,  440.78331970094637881629751063883304595947265625000000,  436.87548858077218483231263235211372375488281250000000,  432.93633910669876740939798764884471893310546875000000,  428.96597548733495841588592156767845153808593750000000,  424.96449085672134060587268322706222534179687500000000,  420.93196740828608426454593427479267120361328125000000,  416.86847650528909525746712461113929748535156250000000,  412.77407876824457844122662208974361419677734375000000,
        408.64882413969053231994621455669403076171875000000000,  404.49275192655193222890375182032585144042968750000000,  400.30589082022510183378471992909908294677734375000000,  396.08825889439668799241189844906330108642578125000000,  391.83986358048741749371401965618133544921875000000000,  387.56070162049741156806703656911849975585937500000000,  383.25075899690318692591972649097442626953125000000000,  378.91001083913204183772904798388481140136718750000000,  374.53842130601066173767321743071079254150390625000000,  370.13594344344164710491895675659179687500000000000000,
        365.70251901642120628821430727839469909667968750000000,  361.23807831435135540232295170426368713378906250000000,  356.74253992843443938909331336617469787597656250000000,  352.21581049975640098637086339294910430908203125000000,  347.65778443646587447801721282303333282470703125000000,  343.06834359824364355517900548875331878662109375000000,  338.44735694601604336639866232872009277343750000000000,  333.79468015460616925338399596512317657470703125000000,  329.11015518572361315818852744996547698974609375000000,  324.39360981836995279081747867166996002197265625000000,
        319.64485713337347760898410342633724212646484375000000,  314.86369494835946625244105234742164611816406250000000,  310.04990519900394474461791105568408966064453125000000,  305.20325326189805537069332785904407501220703125000000,  300.32348721376388311909977346658706665039062500000000,  295.41033702109245950850890949368476867675781250000000,  290.46351365351142703730147331953048706054687500000000,  285.48270811331502727625775150954723358154296875000000,  280.46759037258647140333778224885463714599609375000000,  275.41780820818530628457665443420410156250000000000000,
        270.33298592353662570531014353036880493164062500000000,  265.21272294460823104600422084331512451171875000000000,  260.05659227565837454676511697471141815185546875000000,  254.86413879822890748982899822294712066650390625000000,  249.63487739439142387709580361843109130859375000000000,  244.36829087234374924264557193964719772338867187500000,  239.06382766902459025004645809531211853027343750000000,  233.72089930033803284459281712770462036132812500000000,  228.33887752472628562827594578266143798828125000000000,  222.91709118001941192233061883598566055297851562500000,
        217.45482264650016190898895729333162307739257812500000,  211.95130388067491367110051214694976806640625000000000,  206.40571195397083670286519918590784072875976562500000,  200.81716401803032567841000854969024658203125000000000,  195.18471160283831977722002193331718444824218750000000,  189.50733413482362266222480684518814086914062500000000,  183.78393153828463368881784845143556594848632812500000,  178.01331575364145010098582133650779724121093750000000,  172.19420096827590782595507334917783737182617187500000,  166.32519230761448625344200991094112396240234375000000,
        160.40477267224179058757727034389972686767578125000000,  154.43128732652039047934522386640310287475585937500000,  148.40292573886586069420445710420608520507812500000000,  142.31770003416386316530406475067138671875000000000000,  136.17341923140153880922298412770032882690429687500000,  129.96765818479147469588497187942266464233398437500000,  123.69771979530774785871471976861357688903808593750000,  117.36058856739077782549429684877395629882812500000000,  110.95287288419393689764547161757946014404296875000000,  104.47073235587994588513538474217057228088378906250000,
         97.90978508346560715835948940366506576538085937500000,   91.26498738238062458094645990058779716491699218750000,   84.53047492012436237018846441060304641723632812500000,   77.69934843888242426146462094038724899291992187500000,   70.76337757677444528781052213162183761596679687500000,   63.71257948867533826842191047035157680511474609375000,   56.53459821490265824195375898852944374084472656250000,   49.21375104763756525017015519551932811737060546875000,   41.72948345637584566247824113816022872924804687500000,   34.05368844660353033759747631847858428955078125000000,
         26.14560704971838944743467436637729406356811523437500,   17.94076293120013687598657270427793264389038085937500,    9.32143441948176842970497091300785541534423828125000,    0.00000000000000000000000000000000000000000000000000,  605.02010584942365767346927896142005920410156250000000,  606.74129631082917057938175275921821594238281250000000,  607.71551012013401305011939257383346557617187500000000,  608.23705800463130799471400678157806396484375000000000,  608.42864140215908719255821779370307922363281250000000,  608.35858845555242169211851432919502258300781250000000,
        608.07072804820870715047931298613548278808593750000000,  607.59567340361491005751304328441619873046875000000000,  606.95606573649183701490983366966247558593750000000000,  606.16935382063286397169576957821846008300781250000000,  605.24940916401533286261837929487228393554687500000000,  604.20753068144995268085040152072906494140625000000000,  603.05310336120533065695781260728836059570312500000000,  601.79404856644237042928580194711685180664062500000000,  600.43714259827243040490429848432540893554687500000000,  598.98824856929195448174141347408294677734375000000000,
        597.45248929082220001873793080449104309082031250000000,  595.83437886692809115629643201828002929687500000000000,  594.13792466560130378638859838247299194335937500000000,  592.36670758231662148318719118833541870117187500000000,  590.52394609633108757407171651721000671386718750000000,  588.61254802464543445239542052149772644042968750000000,  586.63515279900673249358078464865684509277343750000000,  584.59416634519902800093404948711395263671875000000000,  582.49179011830119634396396577358245849609375000000000,  580.33004547098244074732065200805664062500000000000000,
        578.11079425782975249603623524308204650878906250000000,  575.83575637640353761526057496666908264160156250000000,  573.50652479447887799324234947562217712402343750000000,  571.12457849849056401581037789583206176757812500000000,  568.69129371068459022353636100888252258300781250000000,  566.20795365486173977842554450035095214843750000000000,  563.67575709786399329459527507424354553222656250000000,  561.09582585247778752091107890009880065917968750000000,  558.46921139453218074777396395802497863769531250000000,  555.79690072069047346303705126047134399414062500000000,
        553.07982155227568910049740225076675415039062500000000,  550.31884697334521661105100065469741821289062500000000,  547.51479957725848635163856670260429382324218750000000,  544.66845518452964824973605573177337646484375000000000,  541.78054618531757569144247099757194519042968750000000,  538.85176455207601975416764616966247558593750000000000,  535.88276456136736669577658176422119140625000000000000,  532.87416525838602865405846387147903442382812500000000,  529.82655269315034729515900835394859313964843750000000,  526.74048195344630585168488323688507080078125000000000,
        523.61647901632227330992463976144790649414062500000000,  520.45504243713776304502971470355987548828125000000000,  517.25664489278074142930563539266586303710937500000000,  514.02173459361995355720864608883857727050781250000000,  510.75073657699584828151273541152477264404296875000000,  507.44405389353227064930251799523830413818359375000000,  504.10206869623323200357845053076744079589843750000000,  500.72514324118480999459279701113700866699218750000000,  497.31362080768275291120517067611217498779296875000000,  493.86782654473529419192345812916755676269531250000000,
        490.38806825012540002717287279665470123291015625000000,  486.87463708754415847579366527497768402099609375000000,  483.32780824671323216534801758825778961181640625000000,  479.74784155089116666204063221812248229980468750000000,  476.13498201569126422327826730906963348388671875000000,  472.48946036272553783419425599277019500732421875000000,  468.81149349122000558054423891007900238037109375000000,  465.10128491041388087978702969849109649658203125000000,  461.35902513526070833904668688774108886718750000000000,  457.58489204767818137042922899127006530761718750000000,
        453.77905122535275950212962925434112548828125000000000,  449.94165623988470770200365222990512847900390625000000,  446.07284892585596480785170570015907287597656250000000,  442.17275962222134921830729581415653228759765625000000,  438.24150738725080600488581694662570953369140625000000,  434.27920018809447810781421139836311340332031250000000,  430.28593506589402295503532513976097106933593750000000,  426.26179827722552317936788313090801239013671875000000,  422.20686541252945289670606143772602081298828125000000,  418.12120149205702546169050037860870361328125000000000,
        414.00486103974327534160693176090717315673828125000000,  409.85788813530081142744165845215320587158203125000000,  405.68031644471426488962606526911258697509765625000000,  401.47216922920250681272591464221477508544921875000000,  397.23345933260560514099779538810253143310546875000000,  392.96418914703644986730068922042846679687500000000000,  388.66435055652635810474748723208904266357421875000000,  384.33392485827340578907751478254795074462890625000000,  379.97288266097882569738430902361869812011718750000000,  375.58118375963306334597291424870491027832031250000000,
        371.15877698597222433818387798964977264404296875000000,  366.70560003368740353835164569318294525146484375000000,  362.22157925731266914226580411195755004882812500000000,  357.70662944355387935502221807837486267089843750000000,  353.16065355363997468884917907416820526123046875000000,  348.58354243508358649705769494175910949707031250000000,  343.97517450102373004483524709939956665039062500000000,  339.33541537508631336095277220010757446289062500000000,  334.66411749943813447316642850637435913085937500000000,  329.96111970342008135048672556877136230468750000000000,
        325.22624672982078664063010364770889282226562500000000,  320.45930871549120411145850084722042083740234375000000,  315.66010062259283586172387003898620605468750000000000,  310.82840161631600039982004091143608093261718750000000,  305.96397438438441440666792914271354675292968750000000,  301.06656439307585060305427759885787963867187500000000,  296.13589907382043975303531624376773834228515625000000,  291.17168693367494824997265823185443878173828125000000,  286.17361658209767938387813046574592590332031250000000,  281.14135566544536004585097543895244598388671875000000,
        276.07454969945808898046379908919334411621093750000000,  270.97282078866248866688692942261695861816406250000000,  265.83576622007154810489737428724765777587890625000000,  260.66295691675895795924589037895202636718750000000000,  255.45393573477775817082147113978862762451171875000000,  250.20821558442264631594298407435417175292968750000000,  244.92527735393233001559565309435129165649414062500000,  239.60456761029220729142252821475267410278320312500000,  234.24549604772576572031539399176836013793945312500000,  228.84743264960914643779688049107789993286132812500000,
        223.40970452373332477691292297095060348510742187500000,  217.93159236384883570281090214848518371582031250000000,  212.41232648198021593088924419134855270385742187500000,  206.85108234572766150449751876294612884521484375000000,  201.24697554222328221840143669396638870239257812500000,  195.59905607497509549830283503979444503784179687500000,  189.90630188073740214349527377635240554809570312500000,  184.16761142975482812289556022733449935913085937500000,  178.38179524288008792609616648405790328979492187500000,  172.54756612132473492238204926252365112304687500000000,
        166.66352783669555037704412825405597686767578125000000,  160.72816196710223834998032543808221817016601562500000,  154.73981248481055672527872957289218902587890625000000,  148.69666759558762691995070781558752059936523437500000,  142.59673819022492580188554711639881134033203125000000,  136.43783208130909656574658583849668502807617187500000,  130.21752294351796308546909131109714508056640625000000,  123.93311252435918845549167599529027938842773437500000,  117.58158420010725819793151458725333213806152343750000,  111.15954525031065713847056031227111816406250000000000,
        104.66315420537536340361839393153786659240722656250000,   98.08802810972269980993587523698806762695312500000000,   91.42912224468592796711163828149437904357910156250000,   84.68057126552821500808931887149810791015625000000000,   77.83547492344320062329643405973911285400390625000000,   70.88560188620826352234871592372655868530273437500000,   63.82096835843208282312843948602676391601562500000000,   56.62921744968808468456700211390852928161621093750000,   49.29466554040063641650704084895551204681396484375000,   41.79675720677947481362934922799468040466308593750000,
         34.10738457899631015379782184027135372161865234375000,   26.18578783062404369275100179947912693023681640625000,   17.96748978634092352990592189598828554153442382812500,    9.33476795035123352306527522159740328788757324218750,    0.00000000000000000000000000000000000000000000000000,  610.03738568623862192907836288213729858398437500000000,  611.75975990436609208700247108936309814453125000000000,  612.73528676458249719871673732995986938476562500000000,  613.25826126725678477669134736061096191406250000000000,  613.45137423021719769167248159646987915039062500000000,
        613.38294626358720051939599215984344482421875000000000,  613.09680071149102786876028403639793395996093750000000,  612.62354661456288340559694916009902954101562500000000,  611.98582196789664067182457074522972106933593750000000,  611.20107303435793255630414932966232299804687500000000,  610.28316934829592810274334624409675598144531250000000,  609.24340826886555078090168535709381103515625000000000,  608.09117356120486874715425074100494384765625000000000,  606.83438563517938746372237801551818847656250000000000,  605.47982006148231448605656623840332031250000000000000,
        604.03333940935203827393706887960433959960937500000000,  602.50006610606214962899684906005859375000000000000000,  600.88451400914641453709919005632400512695312500000000,  599.19069036026075991685502231121063232421875000000000,  597.42217603483300081279594451189041137695312500000000,  595.58218958710006063483888283371925354003906250000000,  593.67363899488145762006752192974090576171875000000000,  591.69916392904099211591528728604316711425781250000000,  589.66117062655905556312063708901405334472656250000000,  587.56186092064911008492344990372657775878906250000000,
        585.40325660479015823511872440576553344726562500000000,  583.18722003352252158947521820664405822753906250000000,  580.91547166058194306970108300447463989257812500000000,  578.58960506373364296450745314359664916992187500000000,  576.21109989124818184791365638375282287597656250000000,  573.78133307746099944779416546225547790527343750000000,  571.30158860724873193248640745878219604492187500000000,  568.77306605653484439244493842124938964843750000000000,  566.19688809446222421684069558978080749511718750000000,  563.57410709998748643556609749794006347656250000000000,
        560.90571101936598097381647676229476928710937500000000,  558.19262856985710641311015933752059936523437500000000,  555.43573387784522310539614409208297729492187500000000,  552.63585062560696314903907477855682373046875000000000,  549.79375576951133552938699722290039062500000000000000,  546.91018288299119376461021602153778076171875000000000,  543.98582516980661694105947390198707580566406250000000,  541.02133818659604003187268972396850585937500000000000,  538.01734230825923077645711600780487060546875000000000,  534.97442496512758225435391068458557128906250000000000,
        531.89314267700342497846577316522598266601562500000000,  528.77402290586644539871485903859138488769531250000000,  525.61756574625189841754036024212837219238281250000000,  522.42424546991514944238588213920593261718750000000000,  519.19451193935094579501310363411903381347656250000000,  515.92879190297173863655189052224159240722656250000000,  512.62749018323529526242054998874664306640625000000000,  509.29099076768295617512194439768791198730468750000000,  505.91965781171893468126654624938964843750000000000000,  502.51383656095276819542050361633300781250000000000000,
        499.07385420006227150224731303751468658447265625000000,  495.60002063436667185669648461043834686279296875000000,  492.09262920962936505020479671657085418701171875000000,  488.55195737501651365164434537291526794433593750000000,  484.97826729361446496113785542547702789306640625000000,  481.37180640444472601302550174295902252197265625000000,  477.73280793950021916316472925245761871337890625000000,  474.06149139896052702169981785118579864501953125000000,  470.35806298741181308287195861339569091796875000000000,  466.62271601360237127664731815457344055175781250000000,
        462.85563125599571776547236368060111999511718750000000,  459.05697729614553281862754374742507934570312500000000,  455.22691082169302490001427941024303436279296875000000,  451.36557690059112246672157198190689086914062500000000,  447.47310922797572629860951565206050872802734375000000,  443.54963034693531653829268179833889007568359375000000,  439.59525184427633348605013452470302581787109375000000,  435.61007452223378777489415369927883148193359375000000,  431.59418854694439460217836312949657440185546875000000,  427.54767357436782049262546934187412261962890625000000,
        423.47059885422373781693750061094760894775390625000000,  419.36302331239221530267968773841857910156250000000000,  415.22499561211481022837688215076923370361328125000000,  411.05655419422362228942802175879478454589843750000000,  406.85772729651637291681254282593727111816406250000000,  402.62853295229035666125128045678138732910156250000000,  398.36897896793948348204139620065689086914062500000000,  394.07906287940977563266642391681671142578125000000000,  389.75877188719857713294913992285728454589843750000000,  385.40808276946700061671435832977294921875000000000000,
        381.02696177271712940637371502816677093505859375000000,  376.61536447935918658913578838109970092773437500000000,  372.17323565136246088513871654868125915527343750000000,  367.70050904904206845458247698843479156494140625000000,  363.19710722388299473095685243606567382812500000000000,  358.66294128413920816456084139645099639892578125000000,  354.09791063176788838973152451217174530029296875000000,  349.50190266906486158404732123017311096191406250000000,  344.87479247315593511302722617983818054199218750000000,  340.21644243626184334061690606176853179931640625000000,
        335.52670186939673158121877349913120269775390625000000,  330.80540656687145428804797120392322540283203125000000,  326.05237832864764868645579554140567779541015625000000,  321.26742443723173892067279666662216186523437500000000,  316.45033708538920791397686116397380828857421875000000,  311.60089275050364676644676364958286285400390625000000,  306.71885151088713428180199116468429565429687500000000,  301.80395629876272778346901759505271911621093750000000,  296.85593208396994668873958289623260498046875000000000,  291.87448498168492960758158005774021148681640625000000,
        286.85930127657070443092379719018936157226562500000000,  281.81004635477256670128554105758666992187500000000000,  276.72636353401782116634421981871128082275390625000000,  271.60787278074184314391459338366985321044921875000000,  266.45416930161621849038056097924709320068359375000000,  261.26482199504823711322387680411338806152343750000000,  256.03937174611706950599909760057926177978515625000000,  250.77732954594230818656797055155038833618164062500000,  245.47817441357500456433626823127269744873046875000000,  240.14135109506784715449612122029066085815429687500000,
        234.76626751030877926496032159775495529174804687500000,  229.35229191334894949250156059861183166503906250000000,  223.89874972614481407617859076708555221557617187500000,  218.40491999864727290514565538614988327026367187500000,  212.87003143972077623402583412826061248779296875000000,  207.29325795310742819310689810663461685180664062500000,  201.67371360010045577837445307523012161254882812500000,  196.01044689515734376072941813617944717407226562500000,  190.30243432158925998010090552270412445068359375000000,  184.54857293067104251349519472569227218627929687500000,
        178.74767185767001365093165077269077301025390625000000,  172.89844255055055555203580297529697418212890625000000,  166.99948745900519497809000313282012939453125000000000,  161.04928686959578953974414616823196411132812500000000,  155.04618349247729724993405397981405258178710937500000,  148.98836429984942242299439385533332824707031250000000,  142.87383897661911191789840813726186752319335937500000,  136.70041415634392478750669397413730621337890625000000,  130.46566236073070399470452684909105300903320312500000,  124.16688420960704775097838137298822402954101562500000,
        117.80106197611853247053659288212656974792480468750000,  111.36480186052378371641680132597684860229492187500000,  104.85426133709104590252536581829190254211425781250000,   98.26505641659409207022690679877996444702148437500000,   91.59214136857033849992149043828248977661132812500000,   84.82964985772291299781500129029154777526855468750000,   77.97068066532695240766770439222455024719238281250000,   71.00700150930943266303074778988957405090332031250000,   63.92862766394294027350042597390711307525634765625000,   56.72320132744462739537993911653757095336914062500000,
         49.37503798734592663777220877818763256072998046875000,   41.86358134492816418514848919585347175598144531250000,   34.16072267291709607661687186919152736663818359375000,   26.22570130445622638148961414117366075515747070312500,   17.99403924036046120704668282996863126754760742187500,    9.34801317710125445614721684250980615615844726562500,    0.00000000000000000000000000000000000000000000000000,  615.06126620708482732879929244518280029296875000000000,  616.78481500821237659692997112870216369628906250000000,  617.76164443632296752184629440307617187500000000000000,
        618.28603390642467729776399210095405578613281250000000,  618.48066370706283123581670224666595458984375000000000,  618.41384697648402379854815080761909484863281250000000,  618.12940156142974501563003286719322204589843750000000,  617.65793234882823981024557724595069885253906250000000,  617.02207413428789095632964745163917541503906250000000,  616.23927068326236167195020243525505065917968750000000,  615.32338956493481418874580413103103637695312500000000,  614.28572658729603972460608929395675659179687500000000,  613.13566429363379484129836782813072204589843750000000,
        611.88112213914143922011135146021842956542968750000000,  610.52887496046537307847756892442703247070312500000000,  609.08478477791209115821402519941329956054687500000000,  607.55397362741939559782622382044792175292968750000000,  605.94095511116393026895821094512939453125000000000000,  604.24973633424076524534029886126518249511718750000000,  602.48389814051267876493511721491813659667968750000000,  600.64665914646639066631905734539031982421875000000000,  598.74092747686211168911540880799293518066406250000000,  596.76934302670019860670436173677444458007812500000000,
        594.73431232811367408430669456720352172851562500000000,  592.63803757536447847087401896715164184570312500000000,  590.48254098463200989499455317854881286621093750000000,  588.26968539128506563429255038499832153320312500000000,  586.00119178509783068875549361109733581542968750000000,  583.67865433267309072107309475541114807128906250000000,  581.30355332194108086696360260248184204101562500000000,  578.87726637611513069714419543743133544921875000000000,  576.40107821688411604554858058691024780273437500000000,  573.87618920391605570330284535884857177734375000000000,
        571.30372283627798424276988953351974487304687500000000,  568.68473236849501972756115719676017761230468750000000,  566.02020666770204115891829133033752441406250000000000,  563.31107541719200071383966132998466491699218750000000,  560.55821375454468125099083408713340759277343750000000,  557.76244641855498684890335425734519958496093750000000,  554.92455146773306751128984615206718444824218750000000,  552.04526362371120740135665982961654663085937500000000,  549.12527728506745461345417425036430358886718750000000,  546.16524925055978201271500438451766967773437500000000,
        543.16580118530885101790772750973701477050781250000000,  540.12752185888371059263590723276138305664062500000000,  537.05096918036849729105597361922264099121093750000000,  533.93667205220970117807155475020408630371093750000000,  530.78513206184709360968554392457008361816406250000000,  527.59682502774387558019952848553657531738281250000000,  524.37220241438512857712339609861373901367187500000000,  521.11169262905434607091592624783515930175781250000000,  517.81570221167498857539612799882888793945312500000000,  514.48461692768944431009003892540931701660156250000000,
        511.11880277280243944915127940475940704345703125000000,  507.71860689742169370219926349818706512451171875000000,  504.28435845775408097324543632566928863525390625000000,  500.81636939975663835866726003587245941162109375000000,  497.31493518146595533835352398455142974853515625000000,  493.78033543864063403816544450819492340087890625000000,  490.21283459812849514491972513496875762939453125000000,  486.61268244290505435856175608932971954345703125000000,  482.98011463231790685313171707093715667724609375000000,  479.31535318070535822698730044066905975341796875000000,
        475.61860689722618644736940041184425354003906250000000,  471.89007178944416409649420529603958129882812500000000,  468.12993143294528408659971319139003753662109375000000,  464.33835730902274008258245885372161865234375000000000,  460.51550911225234585799626074731349945068359375000000,  456.66153502957587306809728033840656280517578125000000,  452.77657199233470919352839700877666473388671875000000,  448.86074590252599136874778196215629577636718750000000,  444.91417183440029248231439851224422454833984375000000,  440.93695421237777054557227529585361480712890625000000,
        436.92918696612582607485819607973098754882812500000000,  432.89095366351460825171670876443386077880859375000000,  428.82232762205006793010397814214229583740234375000000,  424.72337199926880657585570588707923889160156250000000,  420.59413986247011507657589390873908996582031250000000,  416.43467423805611815623706206679344177246093750000000,  412.24500814064469977893168106675148010253906250000000,  408.02516458201932891824981197714805603027343750000000,  403.77515655987633635959355160593986511230468750000000,  399.49498702622815926588373258709907531738281250000000,
        395.18464883521534147803322412073612213134765625000000,  390.84412466997252977307653054594993591308593750000000,  386.47338694808354375709313899278640747070312500000000,  382.07239770504253328908816911280155181884765625000000,  377.64110845501693347614491358399391174316406250000000,  373.17946002807860850225551985204219818115234375000000,  368.68738238292797859685379080474376678466796875000000,  364.16479439399063267046585679054260253906250000000000,  359.61160361160057163942838087677955627441406250000000,  355.02770599381108240777393803000450134277343750000000,
        350.41298560818046325948671437799930572509765625000000,  345.76731430166898917377693578600883483886718750000000,  341.09055133655004965476109646260738372802734375000000,  336.38254298997975411111838184297084808349609375000000,  331.64312211458189949553343467414379119873046875000000,  326.87210765708306325905141420662403106689453125000000,  322.06930413167316373801440931856632232666015625000000,  317.23450104436187757528387010097503662109375000000000,  312.36747226414485112400143407285213470458984375000000,  307.46797533627653820076375268399715423583984375000000,
        302.53575073236061143688857555389404296875000000000000,  297.57052103130274645081954076886177062988281250000000,  292.57199002440660251522785983979701995849609375000000,  287.53984173702360749302897602319717407226562500000000,  282.47373935816369794338243082165718078613281250000000,  277.37332406831910702749155461788177490234375000000000,  272.23821375441912095993757247924804687500000000000000,  267.06800159928388893604278564453125000000000000000000,  261.86225453114275296684354543685913085937500000000000,  256.62051151667668591471738182008266448974609375000000,
        251.34228167857631319748179521411657333374023437500000,  246.02704221570107279148942325264215469360351562500000,  240.67423610049229409924009814858436584472656250000000,  235.28326952422028739420056808739900588989257812500000,  229.85350905579332447814522311091423034667968750000000,  224.38427847404412318610411603003740310668945312500000,  218.87485522642450064267904963344335556030273437500000,  213.32446645858718170529755298048257827758789062500000,  207.73228454906754336661833804100751876831054687500000,  202.09742207072676478674111422151327133178710937500000,
        196.41892608518435281439451500773429870605468750000000,  190.69577165737382529187016189098358154296875000000000,  184.92685445356431728214374743402004241943359375000000,  179.11098225634341929435322526842355728149414062500000,  173.24686519231678971664223354309797286987304687500000,  167.33310442017324248809018172323703765869140625000000,  161.36817896489668555659591220319271087646484375000000,  155.35043030359659610439848620444536209106445312500000,  149.27804420309965394153550732880830764770507812500000,  143.14902916978473967901663854718208312988281250000000,
        136.96119068472961544102872721850872039794921875000000,  130.71210014244235253499937243759632110595703125000000,  124.39905706009358254959806799888610839843750000000000,  118.01904263200167122249695239588618278503417968750000,  111.56866200274116351920383749529719352722167968750000,  105.04407161337769593956181779503822326660156250000000,   98.44088646332559733309608418494462966918945312500000,   91.75405983199031823005498154088854789733886718750000,   84.97772441457652803364908322691917419433593750000000,   78.10497804292107559831492835655808448791503906250000,
         71.12758750501929227993969107046723365783691406250000,   64.03556716418444239025120623409748077392578125000000,   56.81655832613037659939436707645654678344726562500000,   49.45487560384262337720429059118032455444335937500000,   41.92996184153214755951921688392758369445800781250000,   34.21370747186406191531204967759549617767333984375000,   26.26535100448071347045697621069848537445068359375000,   18.02041363281987429445507586933672428131103515625000,    9.36117126167876634212916542310267686843872070312500,    0.00000000000000000000000000000000000000000000000000,
        620.09170412847731768124504014849662780761718750000000,  621.81641847103867348778294399380683898925781250000000,  622.79454013841336745826993137598037719726562500000000,  623.32033309969142464979086071252822875976562500000000,  623.51646720358053244126494973897933959960937500000000,  623.45124817648968473804416134953498840332031250000000,  623.16848840917896268365439027547836303710937500000000,  622.69878866373164782999083399772644042968750000000000,  622.06478055627508183533791452646255493164062500000000,  621.28390536833740043221041560173034667968750000000000,
        620.37002871245420010382076725363731384277343750000000,  619.33444485006634749879594892263412475585937500000000,  618.18653510407341400423320010304450988769531250000000,  616.93421797384871752001345157623291015625000000000000,  615.58426755863683865754865109920501708984375000000000,  614.14254532454856416734401136636734008789062500000000,  612.61417290924384815298253670334815979003906250000000,  611.00366365109800881327828392386436462402343750000000,  609.31502450883249366597738116979598999023437500000000,  607.55183628366955872479593381285667419433593750000000,
        605.71731764208959702955326065421104431152343750000000,  603.81437684241802799078868702054023742675781250000000,  601.84565398934648783324519172310829162597656250000000,  599.81355589467875688569620251655578613281250000000000,  597.72028509723406841658288612961769104003906250000000,  595.56786421840763523505302146077156066894531250000000,  593.35815655592887196689844131469726562500000000000000,  591.09288361616233942186227068305015563964843750000000,  588.77364013411829546384979039430618286132812500000000,  586.40190701596895905822748318314552307128906250000000,
        583.97906255138718734087888151407241821289062500000000,  581.50639217544051007280359044671058654785156250000000,  578.98509700707529646024340763688087463378906250000000,  576.41630134975866894819773733615875244140625000000000,  573.80105930697982330457307398319244384765625000000000,  571.14036063903574813593877479434013366699218750000000,  568.43513596639229490392608568072319030761718750000000,  565.68626140778508215589681640267372131347656250000000,  562.89456272726886254531564190983772277832031250000000,  560.06081905297708090074593201279640197753906250000000,
        557.18576622091700301098171621561050415039062500000000,  554.27009978930664146901108324527740478515625000000000,  551.31447776243840053211897611618041992187500000000000,  548.31952305760648869181750342249870300292968750000000,  545.28582574404708793736062943935394287109375000000000,  542.21394507897025505371857434511184692382812500000000,  539.10441136247948179516242817044258117675781250000000,  535.95772763038303310167975723743438720703125000000000,  532.77437120151228100439766421914100646972656250000000,  529.55479509411952676600776612758636474609375000000000,
        526.29942932416349776758579537272453308105468750000000,  523.00868209677457798534305766224861145019531250000000,  519.68294090087385939114028587937355041503906250000000,  516.32257351577811732568079605698585510253906250000000,  512.92792893762668882118305191397666931152343750000000,  509.49933823259283371953642927110195159912109375000000,  506.03711532308653886502725072205066680908203125000000,  502.54155771247638995191664434969425201416015625000000,  499.01294715327378526126267388463020324707031250000000,  495.45155026319730495742987841367721557617187500000000,
        491.85761909307262840229668654501438140869140625000000,  488.23139165011286877415841445326805114746093750000000,  484.57309237975471205572830513119697570800781250000000,  480.88293260890208102864562533795833587646484375000000,  477.16111095312891166031477041542530059814453125000000,  473.40781369013433277359581552445888519287109375000000,  469.62321510149894265850889496505260467529296875000000,  465.80747778457816821173764765262603759765625000000000,  461.96075293616951285002869553864002227783203125000000,  458.08318060941246585571207106113433837890625000000000,
        454.17488994521261247427901253104209899902343750000000,  450.23599937933084902397240512073040008544921875000000,  446.26661682613803350250236690044403076171875000000000,  442.26683983990182014167658053338527679443359375000000,  438.23675575435214568642550148069858551025390625000000,  434.17644180115189556090626865625381469726562500000000,  430.08596520779224192665424197912216186523437500000000,  425.96538327532289258670061826705932617187500000000000,  421.81474343622636524742119945585727691650390625000000,  417.63408329264564144978066906332969665527343750000000,
        413.42343063507439637760398909449577331542968750000000,  409.18280344152361749365809373557567596435546875000000,  404.91220985707775525952456519007682800292968750000000,  400.61164815365663116608629934489727020263671875000000,  396.28110666969729436459601856768131256103515625000000,  391.92056372936451680288882926106452941894531250000000,  387.52998754079186483068042434751987457275390625000000,  383.10933607274148471333319321274757385253906250000000,  378.65855690894858298634062521159648895263671875000000,  374.17758707929181127838091924786567687988281250000000,
        369.66635286679206728877034038305282592773437500000000,  365.12476958929460124636534601449966430664062500000000,  360.55274135453078088175971060991287231445312500000000,  355.95016078708039231059956364333629608154296875000000,  351.31690872556492877265554852783679962158203125000000,  346.65285388819165746099315583705902099609375000000000,  341.95785250453735670816968195140361785888671875000000,  337.23174791120089821561123244464397430419921875000000,  332.47437010867031403904547914862632751464843750000000,  327.68553527642558265142724849283695220947265625000000,
        322.86504524294252860272536054253578186035156250000000,  318.01268690685731144185410812497138977050781250000000,  313.12823160509577746779541485011577606201171875000000,  308.21143442325580963370157405734062194824218750000000,  303.26203344294538055692100897431373596191406250000000,  298.27974892011224028465221635997295379638671875000000,  293.26428238764020761664141900837421417236328125000000,  288.21531567461499889759579673409461975097656250000000,  283.13250983365981028327951207756996154785156250000000,  278.01550396658865338395116850733757019042968750000000,
        272.86391393728717957856133580207824707031250000000000,  267.67733095918623575926176272332668304443359375000000,  262.45532004288691041438141837716102600097656250000000,  257.19741828739319089436321519315242767333984375000000,  251.90313299593853457736258860677480697631835937500000,  246.57193959448861164673871826380491256713867187500000,  241.20327932756831046390288975089788436889648437500000,  235.79655670199036876510945148766040802001953125000000,  230.35113664420930490450700744986534118652343750000000,  224.86634133121376066810626070946455001831054687500000,
        219.34144664788442469216533936560153961181640625000000,  213.77567821529376601574767846614122390747070312500000,  208.16820692415822691145876888185739517211914062500000,  202.51814389510059299936983734369277954101562500000000,  196.82453477194968627372873015701770782470703125000000,  191.08635323520783799722266849130392074584960937500000,  185.30249359902674655131704639643430709838867187500000,  179.47176232518597771559143438935279846191406250000000,  173.59286824982666530559072270989418029785156250000000,  167.66441127058780580227903556078672409057617187500000,
        161.68486917992515827791066840291023254394531250000000,  155.65258225008210501982830464839935302734375000000000,  149.56573506985384369727398734539747238159179687500000,  143.42233499362629345341701991856098175048828125000000,  137.22018637575544630635704379528760910034179687500000,  130.95685950856017143451026640832424163818359375000000,  124.62965283084240297739597735926508903503417968750000,  118.23554648168612857261905446648597717285156250000000,  111.77114457290423388258204795420169830322265625000000,  105.23260253463551805452880216762423515319824218750000,
         98.61553437658660925535514252260327339172363281250000,   91.91489240908083502290537580847740173339843750000000,   85.12480837829305357900011586025357246398925781250000,   78.23837918652935741192777641117572784423828125000000,   71.24737071121920450877951225265860557556152343750000,   64.14179642355878741000196896493434906005859375000000,   56.90929675509519114484646706841886043548583984375000,   49.53418546211587170091661391779780387878417968750000,   41.99590454913670356518196058459579944610595703125000,   34.26634362568062641685173730365931987762451171875000,
         26.30474039436543876036012079566717147827148437500000,   18.04661525730104543185916554648429155349731445312500,    9.37424334324611940871818660525605082511901855468750,    0.00000000000000000000000000000000000000000000000000,  625.12865673089095253089908510446548461914062500000000,  626.85452770271103872801177203655242919921875000000000,  627.83393143181717732659308239817619323730468750000000,  628.36111657874789671041071414947509765625000000000000,  628.55874264056149058887967839837074279785156250000000,  628.49510799108657010947354137897491455078125000000000,
        628.21401960601906466763466596603393554687500000000000,  627.74607415117657183145638555288314819335937500000000,  627.11390008307034804602153599262237548828125000000000,  626.33493621275567875272827222943305969238281250000000,  625.42304620468871689809020608663558959960937500000000,  624.38952277849011807120405137538909912109375000000000,  623.24374603830460728204343467950820922851562500000000,  621.99363352675652549805818125605583190917968750000000,  620.64595860259260007296688854694366455078125000000000,  619.20658217276331924949772655963897705078125000000000,
        617.68062547004137741168960928916931152343750000000000,  616.07260156092024772078730165958404541015625000000000,  614.38651724833073330955812707543373107910156250000000,  612.62595328020495344389928504824638366699218750000000,  610.79412836121548480150522664189338684082031250000000,  608.89395087036109543987549841403961181640625000000000,  606.92806110809146957763005048036575317382812500000000,  604.89886615094064836739562451839447021484375000000000,  602.80856886636445324256783351302146911621093750000000,  600.65919226408914255443960428237915039062500000000000,
        598.45260008636762449896195903420448303222656250000000,  596.19051433736797207529889419674873352050781250000000,  593.87453030077278981480048969388008117675781250000000,  591.50612948030436655244557186961174011230468750000000,  589.08669081043785809015389531850814819335937500000000,  586.61750041698428503877948969602584838867187500000000,  584.09976015453469244675943627953529357910156250000000,  581.53459510630557360855164006352424621582031250000000,  578.92306019905709035811014473438262939453125000000000,  576.26614605948668668133905157446861267089843750000000,
        573.56478421737108419620199128985404968261718750000000,  570.81985174360659129888517782092094421386718750000000,  568.03217539734259844408370554447174072265625000000000,  565.20253534496077918447554111480712890625000000000000,  562.33166850421673643722897395491600036621093750000000,  559.42027155904213486792286857962608337402343750000000,  556.46900368399008129927096888422966003417968750000000,  553.47848901185352588072419166564941406250000000000000,  550.44931887340419507381739094853401184082031250000000,  547.38205383432887174421921372413635253906250000000000,
        544.27722555115701652539428323507308959960937500000000,  541.13533846518510017631342634558677673339843750000000,  537.95687135101024978212080895900726318359375000000000,  534.74227873424979406991042196750640869140625000000000,  531.49199219125432591681601479649543762207031250000000,  528.20642154210884200438158586621284484863281250000000,  524.88595594690070811338955536484718322753906250000000,  521.53096491408621204755036160349845886230468750000000,  518.14179922879861805995460599660873413085937500000000,  514.71879180806536169257014989852905273437500000000000,
        511.26225848914242533282958902418613433837890625000000,  507.77249875650392141324118711054325103759765625000000,  504.24979641243282912910217419266700744628906250000000,  500.69442019564041856938274577260017395019531250000000,  497.10662435187560959093389101326465606689453125000000,  493.48664916007805913977790623903274536132812500000000,  489.83472141726036852560355328023433685302734375000000,  486.15105488497914620893425308167934417724609375000000,  482.43585069996049696783302351832389831542968750000000,  478.68929775118186853433144278824329376220703125000000,
        474.91157302547384233548655174672603607177734375000000,  471.10284192349155318879638798534870147705078125000000,  467.26325854770777823432581499218940734863281250000000,  463.39296596390181548485998064279556274414062500000000,  459.49209643745592757113627158105373382568359375000000,  455.56077164561827430588891729712486267089843750000000,  451.59910286675398083389154635369777679443359375000000,  447.60719114747496405470883473753929138183593750000000,  443.58512744841959829500410705804824829101562500000000,  439.53299276933705641567939892411231994628906250000000,
        435.45085825402441059850389137864112854003906250000000,  431.33878527556112203455995768308639526367187500000000,  427.19682550218476535519585013389587402343750000000000,  423.02502094405593879855587147176265716552734375000000,  418.82340398106697421098942868411540985107421875000000,  414.59199737175129030219977721571922302246093750000000,  410.33081424326138630931382067501544952392578125000000,  406.03985806228695309982867911458015441894531250000000,  401.71912258668868389577255584299564361572265625000000,  397.36859179752650561567861586809158325195312500000000,
        392.98823981105908842437202110886573791503906250000000,  388.57803077018525073071941733360290527343750000000000,  384.13791871468674798961728811264038085937500000000000,  379.66784742951460884796688333153724670410156250000000,  375.16775027023533084502560086548328399658203125000000,  370.63754996461727841960964724421501159667968750000000,  366.07715838919256157169002108275890350341796875000000,  361.48647631947198988200398162007331848144531250000000,  356.86539315231613045398262329399585723876953125000000,  352.21378659877723293902818113565444946289062500000000,
        347.53152234551697574715944938361644744873046875000000,  342.81845368267465801181970164179801940917968750000000,  338.07442109580284750336431898176670074462890625000000,  333.29925181920339127827901393175125122070312500000000,  328.49275934767450735307647846639156341552734375000000,  323.65474290332372220291290432214736938476562500000000,  318.78498685369675058609573170542716979980468750000000,  313.88326007701692788032232783734798431396484375000000,  308.94931526981622482708189636468887329101562500000000,  303.98288819165105678621330298483371734619140625000000,
        298.98369684093319165185675956308841705322265625000000,  293.95144055514259662231779657304286956787109375000000,  288.88579902781907549069728702306747436523437500000000,  283.78643123372802392623270861804485321044921875000000,  278.65297425244051510162535123527050018310546875000000,  273.48504197923443825857248157262802124023437500000000,  268.28222371067550966472481377422809600830078125000000,  263.04408259043304951774189248681068420410156250000000,  257.77015389878192763717379420995712280273437500000000,  252.45994316677288793471234384924173355102539062500000,
        247.11292409315021245674870442599058151245117187500000,  241.72853623866041061774012632668018341064453125000000,  236.30618246832645468202827032655477523803710937500000,  230.84522610740805248497053980827331542968750000000000,  225.34498777095754462607146706432104110717773437500000,  219.80474181989646353940770495682954788208007812500000,  214.22371238808568705280777066946029663085937500000000,  208.60106891459776079500443302094936370849609375000000,  202.93592110284654950191907119005918502807617187500000,  197.22731321279897542808612342923879623413085937500000,
        191.47421757339739656345045659691095352172851562500000,  185.67552717853121180269226897507905960083007812500000,  179.83004720004987575521226972341537475585937500000000,  173.93648521356826108785753604024648666381835937500000,  167.99343988470997146578156389296054840087890625000000,  161.99938780156693951539637055248022079467773437500000,  155.95266805884347149913082830607891082763671875000000,  149.85146409382420529254886787384748458862304687500000,  143.69378213464617033423564862459897994995117187500000,  137.47742543393977143750817049294710159301757812500000,
        131.19996320610968609798874240368604660034179687500000,  124.85869283516917960241698892787098884582519531250000,  118.45059342787770617633213987573981285095214843750000,  111.97226808554998456202156376093626022338867187500000,  105.41987124903778294537914916872978210449218750000000,   98.78901595937824708926200401037931442260742187500000,   92.07465357826977481181529583409428596496582031250000,   85.27091492275388873167685233056545257568359375000000,   78.37089598496078224343364126980304718017578125000000,   71.36636175058568198892317013815045356750488281250000,
         64.24732481703388486948824720457196235656738281250000,   57.00142475952341669653833378106355667114257812500000,   49.61297449500900569319128408096730709075927734375000,   42.06141520522021437500370666384696960449218750000000,   34.31863569300499250402936013415455818176269531250000,   26.34387286999657007413588871713727712631225585937500,   18.07264636260362067332607693970203399658203125000000,    9.38723053877292912261509627569466829299926757812500,    0.00000000000000000000000000000000000000000000000000,  630.17208184781020463560707867145538330078125000000000,
        631.89910066341599303996190428733825683593750000000000,  632.87977642461987670685630291700363159179687500000000,  633.40834261874192634422797709703445434570312500000000,  633.60744847814578406541841104626655578613281250000000,  633.54538508256825934950029477477073669433593750000000,  633.26595403308158438449027016758918762207031250000000,  632.79974792753614565299358218908309936523437500000000,  632.16939208255098492372781038284301757812500000000000,  631.39232285212381157180061563849449157714843750000000,  630.48240196124254453025059774518013000488281250000000,
        629.45092059254670857626479119062423706054687500000000,  628.30725763322050170245347544550895690917968750000000,  627.05932966841874076635576784610748291015625000000000,  625.71390931354369513428537175059318542480468750000000,  624.27685691170484005851903930306434631347656250000000,  622.75329328449709009873913601040840148925781250000000,  621.14773121880170947406440973281860351562500000000000,  619.46417735272189020179212093353271484375000000000000,  617.70621237065859077119966968894004821777343750000000,  615.87705500411095727031351998448371887207031250000000,
        613.97961374032468029326992109417915344238281250000000,  612.01652906206550142087507992982864379882812500000000,  609.99020829617779781983699649572372436523437500000000,  607.90285462338056277076248079538345336914062500000000,  605.75649142542351910378783941268920898437500000000000,  603.55298287185075878369389101862907409667968750000000,  601.29405144648376335680950433015823364257812500000000,  598.98129296260640330729074776172637939453125000000000,  596.61618950149829743168083950877189636230468750000000,  594.20012062151090503903105854988098144531250000000000,
        591.73437311732641319395042955875396728515625000000000,  589.22014955634745092538651078939437866210937500000000,  586.65857577772658260073512792587280273437500000000000,  584.05070750667698575853137299418449401855468750000000,  581.39753621044906140014063566923141479492187500000000,  578.69999430122584271884988993406295776367187500000000,  575.95895977406973997858585789799690246582031250000000,  573.17526035410628537647426128387451171875000000000000,  570.34967721568523302266839891672134399414062500000000,  567.48294832682813648716546595096588134765625000000000,
        564.57577146445316884637577459216117858886718750000000,  561.62880693935403542127460241317749023437500000000000,  558.64268006446047820645617321133613586425781250000000,  555.61798339532208501623244956135749816894531250000000,  552.55527876789233232557307928800582885742187500000000,  549.45509915540446854720357805490493774414062500000000,  546.31795036334244741738075390458106994628906250000000,  543.14431257912383443908765912055969238281250000000000,  539.93464179106615574710303917527198791503906250000000,  536.68937108945067393506178632378578186035156250000000,
        533.40891186097883291949983686208724975585937500000000,  530.09365488659909715352114289999008178710937500000000,  526.74397135154367788345552980899810791015625000000000,  523.36021377541999299864983186125755310058593750000000,  519.94271686932711418194230645895004272460937500000000,  516.49179832621109653700841590762138366699218750000000,  513.00775955000312933407258242368698120117187500000000,  509.49088632849230862120748497545719146728515625000000,  505.94144945436363514090771786868572235107421875000000,  502.35970529837550202501006424427032470703125000000000,
        498.74589633823251233479822985827922821044921875000000,  495.10025164635152350456337444484233856201171875000000,  491.42298733938764598860871046781539916992187500000000,  487.71430699209685144523973576724529266357421875000000,  483.97440201784741020674118772149085998535156250000000,  480.20345201785727340393350459635257720947265625000000,  476.40162510101771431436645798385143280029296875000000,  472.56907817597078746985062025487422943115234375000000,  468.70595721693035784483072347939014434814453125000000,  464.81239750457348236523102968931198120117187500000000,
        460.88852384318090571468928828835487365722656250000000,  456.93445075506673447307548485696315765380859375000000,  452.95028265320974014684907160699367523193359375000000,  448.93611399288005259222700260579586029052734375000000,  444.89202940294222798911505378782749176025390625000000,  440.81810379740937833048519678413867950439453125000000,  436.71440246772363025229424238204956054687500000000000,  432.58098115613904610654572024941444396972656250000000,  428.41788611049162227573106065392494201660156250000000,  424.22515412054826811072416603565216064453125000000000,
        420.00281253603662889872794039547443389892578125000000,  415.75087926637002055940683931112289428710937500000000,  411.46936276198886162092094309628009796142578125000000,  407.15826197715318812697660177946090698242187500000000,  402.81756631392556755599798634648323059082031250000000,  398.44725554699027725291671231389045715332031250000000,  394.04729972885485267397598363459110260009765625000000,  389.61765907487688309629447758197784423828125000000000,  385.15828382745041835732990875840187072753906250000000,  380.66911409856970749387983232736587524414062500000000,
        376.15007968986463993132929317653179168701171875000000,  371.60109988906850730927544645965099334716796875000000,  367.02208324173466280626598745584487915039062500000000,  362.41292729686153961665695533156394958496093750000000,  357.77351832491359573396039195358753204345703125000000,  353.10373100653833944306825287640094757080078125000000,  348.40342809006995139498030766844749450683593750000000,  343.67246001568105384649243205785751342773437500000000,  338.91066450378917807029210962355136871337890625000000,  334.11786610503753536249860189855098724365234375000000,
        329.29387570885234026718535460531711578369140625000000,  324.43849000722127584595000371336936950683593750000000,  319.55149090993324989540269598364830017089843750000000,  314.63264490706723108814912848174571990966796875000000,  309.68170237400192945642629638314247131347656250000000,  304.69839681363379213507869280874729156494140625000000,  299.68244402982503515886492095887660980224609375000000,  294.63354122534303769498364999890327453613281250000000,  289.55136601668300500023178756237030029296875000000000,  284.43557535716087159016751684248447418212890625000000,
        279.28580435851466745589277707040309906005859375000000,  274.10166499991356658938457258045673370361328125000000,  268.88274471173019719572039321064949035644531250000000,  263.62860481962616177042946219444274902343750000000000,  258.33877883239853190389112569391727447509765625000000,  253.01277055456557718571275472640991210937500000000000,  247.65005200176631205977173522114753723144531250000000,  242.25006109361569883731135632842779159545898437500000,  236.81219909458539518709585536271333694458007812500000,  231.33582776862800756134674884378910064697265625000000,
        225.82026620745227774023078382015228271484375000000000,  220.26478728537028928258223459124565124511718750000000,  214.66861368518820540884917136281728744506835937500000,  209.03091342934627050453855190426111221313476562500000,  203.35079483796070576318015810102224349975585937500000,  197.62730081999109188473084941506385803222656250000000,  191.85940238465835250281088519841432571411132812500000,  186.04599123645155600570433307439088821411132812500000,  180.18587128721273415976611431688070297241210937500000,  174.27774888104963224577659275382757186889648437500000,
        168.32022147972000425397709477692842483520507812500000,  162.31176449426470753678586333990097045898437500000000,  156.25071586835565540241077542304992675781250000000000,  150.13525791349738369717670138925313949584960937500000,  143.96339575656082843124750070273876190185546875000000,  137.73293157271274367303703911602497100830078125000000,  131.44143352201095353848359081894159317016601562500000,  125.08619795657746465167292626574635505676269531250000,  118.66420297309856834999663988128304481506347656250000,  112.17205068401891310259088641032576560974121093750000,
        105.60589456192977309001435060054063796997070312500000,   98.96134669964479257942002732306718826293945312500000,   92.23335753012294446762098232284188270568847656250000,   85.41605696061644437122595263645052909851074218750000,   78.50254009190081205815658904612064361572265625000000,   71.48457103625293029836029745638370513916015625000000,   64.35216153511485970284411450847983360290527343750000,   57.09295032473114162030469742603600025177001953125000,   49.69124949962295545446977484971284866333007812500000,   42.12649943519138417968861176632344722747802734375000,
         34.37058814364018388687327387742698192596435546875000,   26.38275176123573828590451739728450775146484375000000,   18.09850915390336467680754140019416809082031250000000,    9.40013394360883758338331972481682896614074707031250,    0.00000000000000000000000000000000000000000000000000,  635.22193785505965024640318006277084350585937500000000,  636.95009585306399912951746955513954162597656250000000,  637.93203376152087002992630004882812500000000000000000,  638.46197002787232577247777953743934631347656250000000,  638.66254370553292574186343699693679809570312500000000,
        638.60203863787955924635753035545349121093750000000000,  638.32425109133100704639218747615814208984375000000000,  637.85976962379095311916898936033248901367187500000000,  637.23121643156639493099646642804145812988281250000000,  636.45602542497283593547763302922248840332031250000000,  635.54805639817891460552345961332321166992187500000000,  634.51859900178249063174007460474967956542968750000000,  633.37703090795525895373430103063583374023437500000000,  632.13126774385989392612827941775321960449218750000000,  630.78808137894463925476884469389915466308593750000000,
        629.35333158807156905822921544313430786132812500000000,  627.83213877567209237895440310239791870117187500000000,  626.22901544162084519484778866171836853027343750000000,  624.54796805052092167898081243038177490234375000000000,  622.79257721339308773167431354522705078125000000000000,  620.96606167761774486280046403408050537109375000000000,  619.07133002670639143616426736116409301757812500000000,  617.11102291277268250269116833806037902832031250000000,  615.08754789905003690364537760615348815917968750000000,  613.00310846467459668929222971200942993164062500000000,
        610.85972834766482719714986160397529602050781250000000,  608.65927212820622571598505601286888122558593750000000,  606.40346275223271277354797348380088806152343750000000,  604.09389654420056103845126926898956298828125000000000,  601.73205614362177584553137421607971191406250000000000,  599.31932171249991370132192969322204589843750000000000,  596.85698069325064807344460859894752502441406250000000,  594.34623634402305469848215579986572265625000000000000,  591.78821523689350669883424416184425354003906250000000,  589.18397387155016531323781237006187438964843750000000,
        586.53450453082746207655873149633407592773437500000000,  583.84074048332649908843450248241424560546875000000000,  581.10356062124014897563029080629348754882812500000000,  578.32379360755419384076958522200584411621093750000000,  575.50222159535496757598593831062316894531250000000000,  572.63958357254409747838508337736129760742187500000000,  569.73657837744508469768334180116653442382812500000000,  566.79386742427243461861507967114448547363281250000000,  563.81207717198549289605580270290374755859375000000000,  560.79180136546858648216584697365760803222656250000000,
        557.73360307410712266573682427406311035156250000000000,  554.63801654955580033856676891446113586425781250000000,  551.50554892169566301163285970687866210937500000000000,  548.33668174939896289288299158215522766113281250000000,  545.13187244067705705674597993493080139160156250000000,  541.89155555502020433777943253517150878906250000000000,  538.61614399922973461798392236232757568359375000000000,  535.30603012672440854657907038927078247070312500000000,  531.96158674915875508304452523589134216308593750000000,  528.58316806820391775545431300997734069824218750000000,
        525.17111053446501500729937106370925903320312500000000,  521.72573363975368465617066249251365661621093750000000,  518.24734064826191115571418777108192443847656250000000,  514.73621927159717870381427928805351257324218750000000,  511.19264229211609062986099161207675933837890625000000,  507.61686813853202693280763924121856689453125000000000,  504.00914141736694773499039001762866973876953125000000,  500.36969340344791135066770948469638824462890625000000,  496.69874249232606189252692274749279022216796875000000,  492.99649461720264298492111265659332275390625000000000,
        489.26314363268659235473023727536201477050781250000000,  485.49887166746827915631001815199851989746093750000000,  481.70384944778589897396159358322620391845703125000000,  477.87823659336191894908552058041095733642578125000000,  474.02218188731461623319773934781551361083984375000000,  470.13582352138706710320548154413700103759765625000000,  466.21928931768826487314072437584400177001953125000000,  462.27269692800496159179601818323135375976562500000000,  458.29615401161652243899879977107048034667968750000000,  454.28975839242730216938070952892303466796875000000000,
        450.25359819612089040674618445336818695068359375000000,  446.18775196793706072639906778931617736816406250000000,  442.09228877157238457584753632545471191406250000000000,  437.96726826961298684182111173868179321289062500000000,  433.81274078581475350802065804600715637207031250000000,  429.62874734946001353819156065583229064941406250000000,  425.41531972193155297645716927945613861083984375000000,  421.17248040555966781539609655737876892089843750000000,  416.90024263471315180140663869678974151611328125000000,  412.59861034901820175946340896189212799072265625000000,
        408.26757814850168415432563051581382751464843750000000,  403.90713123036744036653544753789901733398437500000000,  399.51724530701920912179048173129558563232421875000000,  395.09788650484978234089794568717479705810546875000000,  390.64901124321323777621728368103504180908203125000000,  386.17056609289011248620226979255676269531250000000000,  381.66248761324243332637706771492958068847656250000000,  377.12470216713188619905849918723106384277343750000000,  372.55712571254383647101349197328090667724609375000000,  367.95966356971536015407764352858066558837890625000000,
        363.33221016241168399574235081672668457031250000000000,  358.67464873182342444124515168368816375732421875000000,  353.98685102137051217141561210155487060546875000000000,  349.26867693049081253775511868298053741455078125000000,  344.51997413526356695001595653593540191650390625000000,  339.74057767346192804325255565345287322998046875000000,  334.93030949134470120043260976672172546386718750000000,  330.08897794918021872945246286690235137939453125000000,  325.21637728213670470722718164324760437011718750000000,  320.31228701277200343611184507608413696289062500000000,
        315.37647131090250240958994254469871520996093750000000,  310.40867829611511297116521745920181274414062500000000,  305.40863927760341312023228965699672698974609375000000,  300.37606792534319311016588471829891204833984375000000,  295.31065936586310272105038166046142578125000000000000,  290.21208919499554212961811572313308715820312500000000,  285.08001239899130041521857492625713348388671875000000,  279.91406217422940017058863304555416107177734375000000,  274.71384863441795687322155572474002838134765625000000,  269.47895739263714176559005863964557647705078125000000,
        264.20894800376953526210854761302471160888671875000000,  258.90335225076239566988078877329826354980468750000000,  253.56167225569609513513569254428148269653320312500000,  248.18337839373049291680217720568180084228515625000000,  242.76790698456696304674551356583833694458007812500000,  237.31465773199394675430085044354200363159179687500000,  231.82299087723049524356611073017120361328125000000000,  226.29222402597281416092300787568092346191406250000000,  220.72162860206225332149188034236431121826171875000000,  215.11042587224420685743098147213459014892578125000000,
        209.45778247622075696199317462742328643798828125000000,  203.76280538364832750630739610642194747924804687500000,  198.02453618430138249095762148499488830566406250000000,  192.24194459852560612489469349384307861328125000000000,  186.41392107131696320720948278903961181640625000000000,  180.53926828351416133955353870987892150878906250000000,  174.61669137585278122060117311775684356689453125000000,  168.64478663352318221768655348569154739379882812500000,  162.62202831700784599888720549643039703369140625000000,  156.54675324466327879235905129462480545043945312500000,
        150.41714262718343775304674636572599411010742187500000,  144.23120051442222688820038456469774246215820312500000,  137.98672802763749700716289225965738296508789062500000,  131.68129229542535085784038528800010681152343750000000,  125.31218866025592717505787732079625129699707031250000,  118.87639423035875552159268409013748168945312500000000,  112.37051015032351131139876088127493858337402343750000,  105.79068894491587116135633550584316253662109375000000,   99.13254177860054028315062168985605239868164062500000,   92.39101817493047974494402296841144561767578125000000,
         85.56024715017942128270078683272004127502441406250000,   78.63332293207396617162885377183556556701660156250000,   71.60200877729039348196238279342651367187500000000000,   64.45631558865339627573121106252074241638183593750000,   57.18388128032373884934713714756071567535400390625000,   49.76901714083764005636112415231764316558837890625000,   42.19116275528942594519321573898196220397949218750000,   34.42220536084752069427850074134767055511474609375000,   26.42138033362067162101993744727224111557006835937500,   18.12420579387325147990850382484495639801025390625000,
          9.41295463203789850581415521446615457534790039062500,    0.00000000000000000000000000000000000000000000000000,  640.27818366040810360573232173919677734375000000000000,  642.00747230096419571054866537451744079589843750000000,  642.99066261359337204339681193232536315917968750000000,  643.52195813724756590090692043304443359375000000000000,  643.72398783095150065491907298564910888671875000000000,  643.66502835871142451651394367218017578125000000000000,  643.38887069179804711893666535615921020507812500000000,  642.92609937591157631686655804514884948730468750000000,
        642.29933350648491341416956856846809387207031250000000,  641.52600456348227453418076038360595703125000000000000,  640.61997041893346249707974493503570556640625000000000,  639.59251919642827033385401591658592224121093750000000,  638.45302735521966042142594233155250549316406250000000,  637.20940956414403899543685838580131530761718750000000,  635.86843694431388485099887475371360778808593750000000,  634.43596869819748462759889662265777587890625000000000,  632.91712480737044188572326675057411193847656250000000,  631.31641747762864724791143089532852172851562500000000,
        629.63785299175242471392266452312469482421875000000000,  627.88501187791064239718252792954444885253906250000000,  626.06111288882243570697028189897537231445312500000000,  624.16906469271361856954172253608703613281250000000000,  622.21150809853065766219515353441238403320312500000000,  620.19085089245641029265243560075759887695312500000000,  618.10929683770280007593100890517234802246093750000000,  615.96887001333584521489683538675308227539062500000000,  613.77143539409075856383424252271652221679687500000000,  611.51871637106046364351641386747360229492187500000000,
        609.21230976205720253346953541040420532226562500000000,  606.85369874614673335599945858120918273925781250000000,  604.44426406943239271640777587890625000000000000000000,  601.98529380163063251529820263385772705078125000000000,  599.47799187030830125877400860190391540527343750000000,  596.92348555822479738708352670073509216308593750000000,  594.32283211636990927217993885278701782226562500000000,  591.67702461903365929174469783902168273925781250000000,  588.98699716612736665410920977592468261718750000000000,  586.25362952085845336114289239048957824707031250000000,
        583.47775125691771336278179660439491271972656250000000,  580.66014547790030064788879826664924621582031250000000,  577.80155216225227832183009013533592224121093750000000,  574.90267117921962380933109670877456665039062500000000,  571.96416501476505800383165478706359863281250000000000,  568.98666124097201191034400835633277893066406250000000,  565.97075475787198683974565938115119934082031250000000,  562.91700983276393799314973875880241394042968750000000,  559.82596195881785661185858771204948425292968750000000,  556.69811955196189501293702051043510437011718750000000,
        553.53396550266927533812122419476509094238281250000000,  550.33395859722008935932535678148269653320312500000000,  547.09853482124913170991931110620498657226562500000000,  543.82810855688205720070982351899147033691406250000000,  540.52307368344099813839420676231384277343750000000000,  537.18380459056299969233805313706398010253906250000000,  533.81065711158248632273171097040176391601562500000000,  530.40396938415676686418009921908378601074218750000000,  526.96406264435552202485268935561180114746093750000000,  523.49124195976662576867965981364250183105468750000000,
        519.98579690658198160235770046710968017578125000000000,  516.44800219510568695113761350512504577636718750000000,  512.87811824766833979083457961678504943847656250000000,  509.27639173252202908770414069294929504394531250000000,  505.64305605692442213694448582828044891357421875000000,  501.97833182229794601880712434649467468261718750000000,  498.28242724405771468809689395129680633544921875000000,  494.55553853844031664266367442905902862548828125000000,  490.79785027843121270052506588399410247802734375000000,  487.00953572067498953401809558272361755371093750000000,
        483.19075710506029963653418235480785369873046875000000,  479.34166592849703647516435012221336364746093750000000,  475.46240319424111930857179686427116394042968750000000,  471.55309963797867567336652427911758422851562500000000,  467.61387593174350740810041315853595733642578125000000,  463.64484286661848955191089771687984466552734375000000,  459.64610151505627300139167346060276031494140625000000,  455.61774337354273711753194220364093780517578125000000,  451.55985048622937938489485532045364379882812500000000,  447.47249555006010268698446452617645263671875000000000,
        443.35574200182753656918066553771495819091796875000000,  439.20964408750620577848167158663272857666015625000000,  435.03424691412266156476107425987720489501953125000000,  430.82958648433975668012863025069236755371093750000000,  426.59568971385044733324320986866950988769531250000000,  422.33257443159385502440272830426692962646484375000000,  418.04024936272560353245353326201438903808593750000000,  413.71871409418997700413456186652183532714843750000000,  409.36795902265862423519138246774673461914062500000000,  404.98796528451271115045528858900070190429687500000000,
        400.57870466745526982776937074959278106689453125000000,  396.14013950324749657738720998167991638183593750000000,  391.67222254096157030289759859442710876464843750000000,  387.17489680003870944346999749541282653808593750000000,  382.64809540232863582787103950977325439453125000000000,  378.09174138216491201092139817774295806884765625000000,  373.50574747340237991011235862970352172851562500000000,  368.89001587219769362491206265985965728759765625000000,  364.24443797416319057447253726422786712646484375000000,  359.56889408435210953030036762356758117675781250000000,
        354.86325309834916197360143996775150299072265625000000,  350.12737215253139311244012787938117980957031250000000,  345.36109624133882789465133100748062133789062500000000,  340.56425779913934093201532959938049316406250000000000,  335.73667624398734687929390929639339447021484375000000,  330.87815748026088158439961262047290802001953125000000,  325.98849335680250760560738854110240936279296875000000,  321.06746107679032320447731763124465942382812500000000,  316.11482255510935601705568842589855194091796875000000,  311.13032371848305501771392300724983215332031250000000,
        306.11369374303745871657156385481357574462890625000000,  301.06464422330844854513998143374919891357421875000000,  295.98286826594176091020926833152770996093750000000000,  290.86803950046567024401156231760978698730468750000000,  285.71981099851518592913635075092315673828125000000000,  280.53781409173393512901384383440017700195312500000000,  275.32165707724664116540225222706794738769531250000000,  270.07092379804697657164069823920726776123046875000000,  264.78517208384454306724364869296550750732421875000000,  259.46393203581021680292906239628791809082031250000000,
        254.10670413619186547293793410062789916992187500000000,  248.71295716086802940480993129312992095947265625000000,  243.28212586947518047963967546820640563964843750000000,  237.81360844367296181189885828644037246704101562500000,  232.30676363925962846224138047546148300170898437500000,  226.76090761203954571101348847150802612304687500000000,  221.17531037035985264083137735724449157714843750000000,  215.54919179878237400771467946469783782958984375000000,  209.88171718709222091092669870704412460327148437500000,  204.17199218629232859711919445544481277465820312500000,
        198.41905709780243682871514465659856796264648437500000,  192.62188038298498327094421256333589553833007812500000,  186.77935125633209167972381692379713058471679687500000,  180.89027119579967006757215131074190139770507812500000,  174.95334416603532190492842346429824829101562500000000,  168.96716530214101226192724425345659255981445312500000,  162.93020773974450321475160308182239532470703125000000,  156.84080719684322957618860527873039245605468750000000,  150.69714380755689830948540475219488143920898437500000,  144.49722056826468019608000759035348892211914062500000,
        138.23883756918854714967892505228519439697265625000000,  131.91956092969036262729787267744541168212890625000000,  125.53668500419325937400572001934051513671875000000000,  119.08718593347381897729064803570508956909179687500000,  112.56766391469056998175801709294319152832031250000000,  105.97427054464739626382652204483747482299804687500000,   99.30261607878297525076050078496336936950683593750000,   92.54764915004477643378777429461479187011718750000000,   85.70349790202382678216963540762662887573242187500000,   78.76325570720564428484067320823669433593750000000000,
         71.71868498400223757016647141426801681518554687500000,   64.55979581350128171379765262827277183532714843750000,   57.27422530421912938436435069888830184936523437500000,   49.84628395471982997833038098178803920745849609375000,   42.25541057539106759577407501637935638427734375000000,   34.47349164356656103791465284302830696105957031250000,   26.45976179001144146241131238639354705810546875000000,   18.14973840376880076519228168763220310211181640625000,    9.42569365781532830794731125934049487113952636718750,    0.00000000000000000000000000000000000000000000000000,
        645.34077869343502698029624298214912414550781250000000,  647.07118955575947438774164766073226928710937500000000,  648.05562266830156659125350415706634521484375000000000,  648.58826679099934153782669454813003540039062500000000,  648.79174087188209796295268461108207702636718750000000,  648.73431445184326094022253528237342834472656250000000,  648.45977324605394187528872862458229064941406250000000,  647.99869781547931779641658067703247070312500000000000,  647.37370417397210076160263270139694213867187500000000,  646.60222138442736650176811963319778442382812500000000,
        645.69810540544551713537657633423805236816406250000000,  644.67264283872839314426528289914131164550781250000000,  643.53520893284132853295886889100074768066406250000000,  642.29371739814052943984279409050941467285156250000000,  640.95493860523720286437310278415679931640625000000000,  639.52473118031036847241921350359916687011718750000000,  638.00821467666844455379759892821311950683593750000000,  636.40990099926534639962483197450637817382812500000000,  634.73379624107258223375538364052772521972656250000000,  632.98348083829716870241099968552589416503906250000000,
        631.16217353884223939530784264206886291503906250000000,  629.27278308450786425964906811714172363281250000000000,  627.31795042899670988845173269510269165039062500000000,  625.30008356846155948005616664886474609375000000000000,  623.22138653633624016947578638792037963867187500000000,  621.08388373802404203161131590604782104492187500000000,  618.88944052725616984389489516615867614746093750000000,  616.63978072389397766528418287634849548339843750000000,  614.33650162188143895036773756146430969238281250000000,  611.98108692177549983171047642827033996582031250000000,
        609.57491793487577069754479452967643737792968750000000,  607.11928333844298322219401597976684570312500000000000,  604.61538770884033056063344702124595642089843750000000,  602.06435901800898591318400576710700988769531250000000,  599.46725524583973765402333810925483703613281250000000,  596.82507023475795904232654720544815063476562500000000,  594.13873889172066355968127027153968811035156250000000,  591.40914182571475521399406716227531433105468750000000,  588.63710949490189250354887917637825012207031250000000,  585.82342592612167209153994917869567871093750000000000,
        582.96883206003440136555582284927368164062500000000000,  580.07402876737671704177046194672584533691406250000000,  577.13967957528666374855674803256988525390625000000000,  574.16641313721561346028465777635574340820312500000000,  571.15482547535714274999918416142463684082031250000000,  568.10548202066183876013383269309997558593750000000000,  565.01891947222588896693196147680282592773437500000000,  561.89564749505257168493699282407760620117187500000000,  558.73615027280402500764466822147369384765625000000000,  555.54088793011499092244775965809822082519531250000000,
        552.31029783728422444255556911230087280273437500000000,  549.04479580864460785960545763373374938964843750000000,  545.74477720459469765046378597617149353027343750000000,  542.41061794613960955757647752761840820312500000000000,  539.04267544979177273489767685532569885253906250000000,  535.64128948981704070320120081305503845214843750000000,  532.20678299404744393541477620601654052734375000000000,  528.73946277882066624442813917994499206542968750000000,  525.23962022800969862146303057670593261718750000000000,  521.70753192059589764539850875735282897949218750000000,
        518.14346021076994475151877850294113159179687500000000,  514.54765376414263755577849224209785461425781250000000,  510.92034805328387392364675179123878479003906250000000,  507.26176581547753130507771857082843780517578125000000,  503.57211747529890999430790543556213378906250000000000,  499.85160153435305119273834861814975738525390625000000,  496.10040493028134278574725612998008728027343750000000,  492.31870336693145873141475021839141845703125000000000,  488.50666161739349035997292958199977874755859375000000,  484.66443380143095964740496128797531127929687500000000,
        480.79216363867709560508956201374530792236328125000000,  476.88998467882021259356406517326831817626953125000000,  472.95802050986895892492611892521381378173828125000000,  468.99638494546394440476433373987674713134765625000000,  465.00518219208936443465063348412513732910156250000000,  460.98450699692858734124456532299518585205078125000000,  456.93444477700990091761923395097255706787109375000000,  452.85507173019192350693629123270511627197265625000000,  448.74645492844967975543113425374031066894531250000000,  444.60865239383463176636723801493644714355468750000000,
        440.44171315740015870687784627079963684082031250000000,  436.24567730130257814380456693470478057861328125000000,  432.02057598420771000746753998100757598876953125000000,  427.76643145005613178000203333795070648193359375000000,  423.48325702016103377900435589253902435302734375000000,  419.17105706853396895894547924399375915527343750000000,  414.82982698025432455324335023760795593261718750000000,  410.45955309261620413963100872933864593505859375000000,  406.06021261870176886077388189733028411865234375000000,  401.63177355294249082362512126564979553222656250000000,
        397.17419455813768536245333962142467498779296875000000,  392.68742483330186132661765441298484802246093750000000,  388.17140396160874615816283039748668670654296875000000,  383.62606173758933891804190352559089660644531250000000,  379.05131797262214377042255364358425140380859375000000,  374.44708227762419028294971212744712829589843750000000,  369.81325382171036153522436507046222686767578125000000,  365.14972106543649488230585120618343353271484375000000,  360.45636146707244051867746748030185699462890625000000,  355.73304116016555553869693540036678314208984375000000,
        350.97961460045024750797892920672893524169921875000000,  346.19592417993180788471363484859466552734375000000000,  341.38179980571896976471180096268653869628906250000000,  336.53705844089682841513422317802906036376953125000000,  331.66150360441542943590320646762847900390625000000000,  326.75492482661246640418539755046367645263671875000000,  321.81709705658789744120440445840358734130859375000000,  316.84778001719536177915870212018489837646484375000000,  311.84671750290243608105811290442943572998046875000000,  306.81363661518736307698418386280536651611328125000000,
        301.74824692947589710456668399274349212646484375000000,  296.65023958686424521147273480892181396484375000000000,  291.51928630300159284161054529249668121337890625000000,  286.35503828550713478762190788984298706054687500000000,  281.15712505014374755774042569100856781005859375000000,  275.92515312463638110784813761711120605468750000000000,  270.65870462747648161894176155328750610351562500000000,  265.35733570725318486438482068479061126708984375000000,  260.02057482594585735569125972688198089599609375000000,  254.64792086714777497036266140639781951904296875000000,
        249.23884104728563215758185833692550659179687500000000,  243.79276860446668706572381779551506042480468750000000,  238.30910023551629706162202637642621994018554687500000,  232.78719324691476799671363551169633865356445312500000,  227.22636237953395266231382265686988830566406250000000,  221.62587626008712504699360579252243041992187500000000,  215.98495342375741756768547929823398590087890625000000,  210.30275784220336277030583005398511886596679687500000,  204.57839387858919621976383496075868606567382812500000,  198.81090057585694808039988856762647628784179687500000,
        192.99924516536029273083840962499380111694335937500000,  187.14231565919371291784045752137899398803710937500000,  181.23891235970049251591262873262166976928710937500000,  175.28773808190578620269661769270896911621093750000000,  169.28738683651346264014136977493762969970703125000000,  163.23633065923985441258992068469524383544921875000000,  157.13290419194720470841275528073310852050781250000000,  150.97528651571082036753068678081035614013671875000000,  144.76147959629619776933395769447088241577148437500000,  138.48928251510514542133023496717214584350585937500000,
        132.15626040385876649452256970107555389404296875000000,  125.75970664992577496832382166758179664611816406250000,  119.29659644704271670434536645188927650451660156250000,  112.76352906479014848173392238095402717590332031250000,  106.15665519132265615098731359466910362243652343750000,   99.47158419184324884554371237754821777343750000000000,   92.70326382697975020619196584448218345642089843750000,   85.84582138543858320645085768774151802062988281250000,   78.89234940179105137758597265928983688354492187500000,   71.83460947305594856970856199041008949279785156250000,
         64.66261087501423787671228637918829917907714843750000,   57.36398992654183359718444989994168281555175781250000,   49.92305635182199807786673773080110549926757812500000,   42.31924820172792323091925936751067638397216796875000,   34.52445120856434357392572565004229545593261718750000,   26.49789927218434826272641657851636409759521484375000,   18.17510906447898122451078961603343486785888671875000,    9.43835205468725213506786531070247292518615722656250,    0.00000000000000000000000000000000000000000000000000,  650.40968289565523718920303508639335632324218750000000,
        652.14120767561632874276256188750267028808593750000000,  653.12687411977037754695629701018333435058593750000000,  653.66085633664363285788567736744880676269531250000000,  653.86576334552160005841869860887527465820312500000000,  653.80985761972499403782421723008155822753906250000000,  653.53691965692087251227349042892456054687500000000000,  653.07752606053645649808458983898162841796875000000000,  652.45428978199231551116099581122398376464843750000000,  651.68463748034685067977989092469215393066406250000000,  650.78242320950198518403340131044387817382812500000000,
        649.75893205447425771126290783286094665527343750000000,  648.62353805550162633153377100825309753417968750000000,  647.38415396447703642479609698057174682617187500000000,  646.04754939955296322295907884836196899414062500000000,  644.61958240696333177766064181923866271972656250000000,  643.10537210660652363003464415669441223144531250000000,  641.50943009612899459170876070857048034667968750000000,  639.83576227102980737981852144002914428710937500000000,  638.08794896679307839804096147418022155761718750000000,  636.26920891672307334374636411666870117187500000000000,
        634.38245092545071202039252966642379760742187500000000,  632.43031607977763997041620314121246337890625000000000,  630.41521257329168292926624417304992675781250000000000,  628.33934469626228747074492275714874267578125000000000,  626.20473716621165749529609456658363342285156250000000,  624.01325570083031379908788949251174926757812500000000,  621.76662453289907261932967230677604675292968750000000,  619.46644141583635700953891500830650329589843750000000,  617.11419055422356905182823538780212402343750000000000,  614.71125380627370304864598438143730163574218750000000,
        612.25892043768510575318941846489906311035156250000000,  609.75839565367641625925898551940917968750000000000000,  607.21080809457896521053044125437736511230468750000000,  604.61721644752583415538538247346878051757812500000000,  601.97861530053216938540572300553321838378906250000000,  599.29594034414571979141328483819961547851562500000000,  596.57007300874363409093348309397697448730468750000000,  593.80184461160808950808132067322731018066406250000000,  590.99204007648017977771814912557601928710937500000000,  588.14140127886901154852239415049552917480468750000000,
        585.25063006257721553993178531527519226074218750000000,  582.32039096639857689297059550881385803222656250000000,  579.35131369449595695186872035264968872070312500000000,  576.34399535939121506089577451348304748535156250000000,  573.29900252262768844957463443279266357421875000000000,  570.21687305489695063442923128604888916015625000000000,  567.09811783462282619439065456390380859375000000000000,  563.94322230162265441322233527898788452148437500000000,  560.75264788041567953769117593765258789062500000000000,  557.52683328599550804938189685344696044921875000000000,
        554.26619572336903729592449963092803955078125000000000,  550.97113199084435564145678654313087463378906250000000,  547.64201949592074925021734088659286499023437500000000,  544.27921719162964109273161739110946655273437500000000,  540.88306644031672476558014750480651855468750000000000,  537.45389181109237597411265596747398376464843750000000,  533.99200181650940066901966929435729980468750000000000,  530.49768959343884944246383383870124816894531250000000,  526.97123353260121803032234311103820800781250000000000,  523.41289786074230505619198083877563476562500000000000,
        519.82293317904293417086591944098472595214843750000000,  516.20157696098215183155843988060951232910156250000000,  512.54905401255746255628764629364013671875000000000000,  508.86557689746621235826751217246055603027343750000000,  505.15134632960115368405240587890148162841796875000000,  501.40655153497391438577324151992797851562500000000000,  497.63137058497108000665321014821529388427734375000000,  493.82597070265654792819987051188945770263671875000000,  489.99050854365947316182428039610385894775390625000000,  486.12513045303023773158201947808265686035156250000000,
        482.22997269930226593714905902743339538574218750000000,  478.30516168686324363079620525240898132324218750000000,  474.35081414761936002832953818142414093017578125000000,  470.36703731282119633760885335505008697509765625000000,  466.35392906581461147652589716017246246337890625000000,  462.31157807638157919427612796425819396972656250000000,  458.24006391724333298043347895145416259765625000000000,  454.13945716320802148402435705065727233886718750000000,  450.00981947336345001531299203634262084960937500000000,  445.85120365663277652856777422130107879638671875000000,
        441.66365372093315500023891218006610870361328125000000,  437.44720490610205843040603213012218475341796875000000,  433.20188370067756977732642553746700286865234375000000,  428.92770784254770433108205907046794891357421875000000,  424.62468630340606523532187566161155700683593750000000,  420.29281925687689636106370016932487487792968750000000,  415.93209803009466440926189534366130828857421875000000,  411.54250503844411923637380823493003845214843750000000,  407.12401370308384684904012829065322875976562500000000,  402.67658835079237178433686494827270507812500000000000,
        398.20018409558173289042315445840358734130859375000000,  393.69474670143296179958269931375980377197265625000000,  389.16021242540006142007769085466861724853515625000000,  384.59650784022426250885473564267158508300781250000000,  380.00354963547891884445562027394771575927734375000000,  375.38124439614091443218057975172996520996093750000000,  370.72948835734013073306414298713207244873046875000000,  366.04816713389158167046844027936458587646484375000000,  361.33715542304264545236947014927864074707031250000000,  356.59631667868598015047609806060791015625000000000000,
        351.82550275508327786155859939754009246826171875000000,  347.02455351791769544433918781578540802001953125000000,  342.19329642024177928760764189064502716064453125000000,  337.33154604060251813280046917498111724853515625000000,  332.43910358031246232712874189019203186035156250000000,  327.51575631647693853665259666740894317626953125000000,  322.56127700698948501667473465204238891601562500000000,  317.57542324325254412542562931776046752929687500000000,  312.55793674587096120376372709870338439941406250000000,  307.50854259797915801755152642726898193359375000000000,
        302.42694841020141893750405870378017425537109375000000,  297.31284341048529995532589964568614959716796875000000,  292.16589745117732945800526067614555358886718750000000,  286.98575992471262452454539015889167785644531250000000,  281.77205857813487455132417380809783935546875000000000,  276.52439821533226904648472554981708526611328125000000,  271.24235927432567905270843766629695892333984375000000,  265.92549626514818328359979204833507537841796875000000,  260.57333605174613921917625702917575836181640625000000,  255.18537595886857616278575733304023742675781250000000,
        249.76108168200744330533780157566070556640625000000000,  244.29988497501707911396806593984365463256835937500000,  238.80118108597397963421826716512441635131835937500000,  233.26432590698271951623610220849514007568359375000000,  227.68863279782632957903842907398939132690429687500000,  222.07336903637295222324610222131013870239257812500000,  216.41775184020153233177552465349435806274414062500000,  210.72094389364366406880435533821582794189453125000000,  204.98204830188686287328891921788454055786132812500000,  199.20010287835538065337459556758403778076171875000000,
        193.37407365248611768038244917988777160644531250000000,  187.50284746123233503567462321370840072631835937500000,  181.58522345777669215749483555555343627929687500000000,  175.61990333319900514652545098215341567993164062500000,  169.60547999873563185246894136071205139160156250000000,  163.54042441440284960663120727986097335815429687500000,  157.42307016944531028457276988774538040161132812500000,  151.25159531474312757381994742900133132934570312500000,  145.02400080765229972712404560297727584838867187500000,  138.73808474233649690177117008715867996215820312500000,
        132.39141128385762158359284512698650360107421875000000,  125.98127287293213782959355739876627922058105468750000,  119.50464377609917221434443490579724311828613281250000,  112.95812235466350159640569472685456275939941406250000,  106.33785840691071200581063749268651008605957031250000,   99.63946042608412767549452837556600570678710937500000,   92.85787531828043483983492478728294372558593750000000,   85.98722953463894214110041502863168716430664062500000,   79.02061478867867094777466263622045516967773437500000,   71.94979187244648244359268574044108390808105468750000,
         64.76476927241181158478866564109921455383300781250000,   57.45318253339292624559675459749996662139892578125000,   49.99934062037610260631481651216745376586914062500000,   42.38268083951754050531235407106578350067138671875000,   34.57508819251654585968935862183570861816406250000000,   26.53579586237549747806951927486807107925415039062500,   18.20031981754399552642098569776862859725952148437500,    9.45093083689411272985125833656638860702514648437500,    0.00000000000000000000000000000000000000000000000000,  655.48485671088906201475765556097030639648437500000000,
        657.21748721865935749519849196076393127441406250000000,  658.20437765929693796351784840226173400878906250000000,  658.73968761568198715394828468561172485351562500000000,  658.94601625948621403949800878763198852539062500000000,  658.89161905129071783449035137891769409179687500000000,  658.62027130940953156823525205254554748535156250000000,  658.16254570665842038579285144805908203125000000000000,  657.54105215102856618614168837666511535644531250000000,  656.77321491092016003676690161228179931640625000000000,  655.87288614428666733147110790014266967773437500000000,
        654.85134942473655428329948335886001586914062500000000,  653.71797758666320987686049193143844604492187500000000,  652.48068242367696711880853399634361267089843750000000,  651.14623279971135616506217047572135925292968750000000,  649.72048617762936828512465581297874450683593750000000,  648.20856123903593015711521729826927185058593750000000,  646.61496926808854368573520332574844360351562500000000,  644.94371595546101616491796448826789855957031250000000,  643.19838152748752690968103706836700439453125000000000,  641.38218469344894856476457789540290832519531250000000,
        639.49803431044495027890661731362342834472656250000000,  637.54857158712377440679119899868965148925781250000000,  635.53620490239893570105778053402900695800781250000000,  633.46313879044021177833201363682746887207031250000000,  631.33139826714307218935573473572731018066406250000000,  629.14284939961737563862698152661323547363281250000000,  626.89921681823636845365399494767189025878906250000000,  624.60209871978099727130029350519180297851562500000000,  622.25297979596109598787734284996986389160156250000000,  619.85324243421837309142574667930603027343750000000000,
        617.40417647020751701347762718796730041503906250000000,  614.90698771871336703043198212981224060058593750000000,  612.36280546834791493893135339021682739257812500000000,  609.77268909254075879289302974939346313476562500000000,  607.13763390309509304643142968416213989257812500000000,  604.45857635147172004508320242166519165039062500000000,  601.73639866586017888039350509643554687500000000000000,  598.97193299815910449979128316044807434082031250000000,  596.16596514355433100718073546886444091796875000000000,  593.31923788596191116084810346364974975585937500000000,
        590.43245401479146039491752162575721740722656250000000,  587.50627905198086864402284845709800720214843750000000,  584.54134372280304887681268155574798583984375000000000,  581.53824619937495299382135272026062011718750000000000,  578.49755414192622993141412734985351562500000000000000,  575.41980655961731372372014448046684265136718750000000,  572.30551550989741826924728229641914367675781250000000,  569.15516765302379553759237751364707946777343750000000,  565.96922567630883804667973890900611877441406250000000,  562.74812960091651348193408921360969543457031250000000,
        559.49229798250667045067530125379562377929687500000000,  556.20212901571551356028066948056221008300781250000000,  552.87800155132299551041796803474426269531250000000000,  549.52027603396061294915853068232536315917968750000000,  546.12929536735327928909100592136383056640625000000000,  542.70538571332156152493553236126899719238281250000000,  539.24885723011016125383321195840835571289062500000000,  535.76000475501791697752196341753005981445312500000000,  532.23910843578562435141066089272499084472656250000000,  528.68643431474367844202788546681404113769531250000000,
        525.10223486930522085458505898714065551757812500000000,  521.48674951203975069802254438400268554687500000000000,  517.84020505322655480995308607816696166992187500000000,  514.16281612850866622466128319501876831054687500000000,  510.45478559400135054602287709712982177734375000000000,  506.71630489098015459603630006313323974609375000000000,  502.94755438206010467183659784495830535888671875000000,  499.14870366059102479994180612266063690185546875000000,  495.31991183481716234382474794983863830566406250000000,  491.46132778819594477681675925850868225097656250000000,
        487.57309041712460384587757289409637451171875000000000,  483.65532884719226558445370756089687347412109375000000,  479.70816262895539239252684637904167175292968750000000,  475.73170191411963969585485756397247314453125000000000,  471.72604761290966735032270662486553192138671875000000,  467.69129153330885628747637383639812469482421875000000,  463.62751650276078407841851003468036651611328125000000,  459.53479647283802478341385722160339355468750000000000,  455.41319660729971019463846459984779357910156250000000,  451.26277335388243727720691822469234466552734375000000,
        447.08357450009242484156857244670391082763671875000000,  442.87563921319178916746750473976135253906250000000000,  438.63899806450092455634148791432380676269531250000000,  434.37367303806519203135394491255283355712890625000000,  430.07967752366437252931063994765281677246093750000000,  425.75701629406972870128811337053775787353515625000000,  421.40568546638075986265903338789939880371093750000000,  417.02567244720017924919375218451023101806640625000000,  412.61695586132600510609336197376251220703125000000000,  408.17950546356098584510618820786476135253906250000000,
        403.71328203315613336599199101328849792480468750000000,  399.21823725031350704739452339708805084228515625000000,  394.69431355408306671961327083408832550048828125000000,  390.14144398088416210157447494566440582275390625000000,  385.55955198277587214761297218501567840576171875000000,  380.94855122448353768049855716526508331298828125000000,  376.30834535806144458547350950539112091064453125000000,  371.63882777393223477702122181653976440429687500000000,  366.93988132689452186241396702826023101806640625000000,  362.21137803552039713395060971379280090332031250000000,
        357.45317875318352207614225335419178009033203125000000,  352.66513280875267355440882965922355651855468750000000,  347.84707761475999632239108905196189880371093750000000,  342.99883824060208326045540161430835723876953125000000,  338.12022694804835509785334579646587371826171875000000,  333.21104268601771991598070599138736724853515625000000,  328.27107054122677709528943523764610290527343750000000,  323.30008114091549487056909129023551940917968750000000,  318.29783000340216858603525906801223754882812500000000,  313.26405683170764859823975712060928344726562500000000,
        308.19848474490623857491300441324710845947265625000000,  303.10081944119775698709418065845966339111328125000000,  297.97074828593417805677745491266250610351562500000000,  292.80793931696877052672789432108402252197265625000000,  287.61204015869236627622740343213081359863281250000000,  282.38267683497122106928145512938499450683593750000000,  277.11945246986743995876167900860309600830078125000000,  271.82194586347611675591906532645225524902343750000000,  266.48970992841333327305619604885578155517578125000000,  261.12226997038357012570486404001712799072265625000000,
        255.71912179379054919081681873649358749389648437500000,  250.27972961044974908872973173856735229492187500000000,  244.80352372603076105406216811388731002807617187500000,  239.28989797478647005846141837537288665771484375000000,  233.73820686827292547604884020984172821044921875000000,  228.14776241795709665893809869885444641113281250000000,  222.51783058462095254981250036507844924926757812500000,  216.84762729902365663292584940791130065917968750000000,  211.13631398801700811418413650244474411010742187500000,  205.38299252775740910692547913640737533569335937500000,
        199.58669953022919685281522106379270553588867187500000,  193.74639985019535970423021353781223297119140625000000,  187.86097917590646488861239049583673477172851562500000,  181.92923553704878258940880186855792999267578125000000,  175.94986952567469984387571457773447036743164062500000,  169.92147297775196079783199820667505264282226562500000,  163.84251580110239387977344449609518051147460937500000,  157.71133055519050003567826934158802032470703125000000,  151.52609428289389370547723956406116485595703125000000,  145.28480695472981665261613670736551284790039062500000,
        138.98526569859458845712651964277029037475585937500000,  132.62503373328229372418718412518501281738281250000000,  126.20140257268930383816041285172104835510253906250000,  119.71134557544934295947314240038394927978515625000000,  113.15146021336177284410950960591435432434082031250000,  106.51789541310941444862692151218652725219726562500000,   99.80625881375495112024509580805897712707519531250000,   93.01149648417212745243887184187769889831542968750000,   86.12773405478593247153185075148940086364746093750000,   79.14806243447551992176158819347620010375976562500000,
         72.06424162630233354320807848125696182250976562500000,   64.86627934299859532529808348044753074645996093750000,   57.54181037050043556746459216810762882232666015625000,   50.07514292938650868336480925790965557098388671875000,   42.44571359551160583123419201001524925231933593750000,   34.62540665402316619747580261901021003723144531250000,   26.57345458477588451273732061963528394699096679687500,   18.22537266614117612562040449120104312896728515625000,    9.46343099965834433362488198326900601387023925781250,    0.00000000000000000000000000000000000000000000000000,
        660.56626107587351270922226831316947937011718750000000,  662.29998923364382790168747305870056152343750000000000,  663.28809446609795941185439005494117736816406250000000,  663.82472195443403961689909920096397399902343750000000,  664.03246110274073998880339786410331726074218750000000,  663.97956041299801199784269556403160095214843750000000,  663.70979006187826598761603236198425292968750000000000,  663.25371881824423780926736071705818176269531250000000,  662.63395356551370696251979097723960876464843750000000,  661.86791619455277668748749420046806335449218750000000,
        660.96945697612875392223941162228584289550781250000000,  659.94985797778974756511161103844642639160156250000000,  658.81849083068402705976041033864021301269531250000000,  657.58326637047423446347238495945930480957031250000000,  656.25095270530471225356450304388999938964843750000000,  654.82740671146154909365577623248100280761718750000000,  653.31774662761984018288785591721534729003906250000000,  651.72648341854119280469603836536407470703125000000000,  650.05762256301954948867205530405044555664062500000000,  648.31474417013384936581132933497428894042968750000000,
        646.50106691606140429939841851592063903808593750000000,  644.61949970037562707148026674985885620117187500000000,  642.67268384270596470742020756006240844726562500000000,  640.66302789559563279908616095781326293945312500000000,  638.59273662461021103808889165520668029785156250000000,  636.46383533072958016418851912021636962890625000000000,  634.27819041642032971139997243881225585937500000000000,  632.03752689482155346922809258103370666503906250000000,  629.74344339049059726676205173134803771972656250000000,  627.39742506591926485270960256457328796386718750000000,
        625.00085482066708664206089451909065246582031250000000,  622.55502304246465428150258958339691162109375000000000,  620.06113613700119913119124248623847961425781250000000,  617.52032402171562353032641112804412841796875000000000,  614.93364673607220538542605936527252197265625000000000,  612.30210029457521159201860427856445312500000000000000,  609.62662188766626059077680110931396484375000000000000,  606.90809451854840972373494878411293029785156250000000,  604.14735115004668841720558702945709228515625000000000,  601.34517842418267719040159136056900024414062500000000,
        598.50232000772393803345039486885070800781250000000000,  595.61947960915426847350317984819412231445312500000000,  592.69732370601059301407076418399810791015625000000000,  589.73648401608386393490945920348167419433593750000000,  586.73755974140829039242817088961601257324218750000000,  583.70111961009661172283813357353210449218750000000000,  580.62770373780335830815602093935012817382812500000000,  577.51782532781430745671968907117843627929687500000000,  574.37197222637064442096743732690811157226562500000000,  571.19060834780646018771221861243247985839843750000000,
        567.97417498231118315743515267968177795410156250000000,  564.72309199762241860298672690987586975097656250000000,  561.43775894463544773316243663430213928222656250000000,  558.11855607578252147504827007651329040527343750000000,  554.76584528403918739059008657932281494140625000000000,  551.37997096954939024726627394556999206542968750000000,  547.96126084010268186830217018723487854003906250000000,  544.51002665103078470565378665924072265625000000000000,  541.02656488950060520437546074390411376953125000000000,  537.51115740766908857040107250213623046875000000000000,
        533.96407200869919051910983398556709289550781250000000,  530.38556298923458598437719047069549560546875000000000,  526.77587164156807375547941774129867553710937500000000,  523.13522671841190003760857507586479187011718750000000,  519.46384486289616688736714422702789306640625000000000,  515.76193100615591902169398963451385498046875000000000,  512.02967873464001513639232143759727478027343750000000,  508.26727062906104492867598310112953186035156250000000,  504.47487857672075506343389861285686492919921875000000,  500.65266405876815269948565401136875152587890625000000,
        496.80077841379545589006738737225532531738281250000000,  492.91936307903250735762412659823894500732421875000000,  489.00854981026873247174080461263656616210937500000000,  485.06846088151354479123256169259548187255859375000000,  481.09920926529400730942143127322196960449218750000000,  477.10089879438424986801692284643650054931640625000000,  473.07362430566769262441084720194339752197265625000000,  469.01747176674126649231766350567340850830078125000000,  464.93251838578538581714383326470851898193359375000000,  460.81883270514612149781896732747554779052734375000000,
        456.67647467899342927921679802238941192626953125000000,  452.50549573535164427084964700043201446533203125000000,  448.30593882272125938470708206295967102050781250000000,  444.07783844144427121136686764657497406005859375000000,  439.82122065989489101411891169846057891845703125000000,  435.53610311550886535769677720963954925537109375000000,  431.22249500059575666455202735960483551025390625000000,  426.88039703281043557581142522394657135009765625000000,  422.50980141008642476663226261734962463378906250000000,  418.11069174976495332884951494634151458740234375000000,
        413.68304301157434110791655257344245910644531250000000,  409.22682140403890116431284695863723754882812500000000,  404.74198427381253395651583559811115264892578125000000,  400.22847997734481850784504786133766174316406250000000,  395.68624773419622897563385777175426483154296875000000,  391.11521746121752585167996585369110107421875000000000,  386.51530958670247173358802683651447296142578125000000,  381.88643484350728840581723488867282867431640625000000,  377.22849404000356798860593698918819427490234375000000,  372.54137780759509723793598823249340057373046875000000,
        367.82496632337625896980171091854572296142578125000000,  363.07912900634590869231033138930797576904296875000000,  358.30372418540582657442428171634674072265625000000000,  353.49859873717116443003760650753974914550781250000000,  348.66358769139299056405434384942054748535156250000000,  343.79851380154292428414919413626194000244140625000000,  338.90318707782807905459776520729064941406250000000000,  333.97740427958933651098050177097320556640625000000000,  329.02094836368087271694093942642211914062500000000000,  324.03358788503021514770807698369026184082031250000000,
        319.01507634512489630651543848216533660888671875000000,  313.96515148366097491816617548465728759765625000000000,  308.88353450800605060067027807235717773437500000000000,  303.76992925446563731384230777621269226074218750000000,  298.62402127458329914588830433785915374755859375000000,  293.44547683883706667984370142221450805664062500000000,  288.23394184909403747951728291809558868408203125000000,  282.98904065003364394215168431401252746582031250000000,  277.71037472841686621904955245554447174072265625000000,  272.39752128753252691240049898624420166015625000000000,
        267.05003168235236898908624425530433654785156250000000,  261.66742969881966018874663859605789184570312500000000,  256.24920965823343976808246225118637084960937500000000,  250.79483432478363624795747455209493637084960937500000,  245.30373259086368875614425633102655410766601562500000,  239.77529691071373463273630477488040924072265625000000,  234.20888044809825601078046020120382308959960937500000,  228.60379389791160065215080976486206054687500000000000,  222.95930193461893509265792090445756912231445312500000,  217.27461923199228976955055259168148040771484375000000,
        211.54890598833486592411645688116550445556640625000000,  205.78126287883605982642620801925659179687500000000000,  199.97072534126954224120709113776683807373046875000000,  194.11625708214972974019474349915981292724609375000000,  188.21674266667591268742398824542760848999023437500000,  182.27097902594326228609133977442979812622070312500000,  176.27766567716338386162533424794673919677734375000000,  170.23539340452916235335578676313161849975585937500000,  164.14263108649379319103900343179702758789062500000000,  157.99771027492286634696938563138246536254882812500000,
        151.79880702625118260584713425487279891967773437500000,  145.54392034511721476519596762955188751220703125000000,  139.23084641353065649127529468387365341186523437500000,  132.85714752383969994298240635544061660766601562500000,  126.42011428240255099808564409613609313964843750000000,  119.91671915870797704428696306422352790832519531250000,  113.34355875330656715505028842017054557800292968750000,  106.69678113904802785327774472534656524658203125000000,   99.97199311811337452127190772444009780883789062500000,   93.16413993899723777758481446653604507446289062500000,
         86.26734642781394768462632782757282257080078125000000,   79.27470270478065117458754684776067733764648437500000,   72.17796799953936215388239361345767974853515625000000,   64.96714926625232067181059392169117927551269531250000,   57.62988054675484761446568882092833518981933593750000,   50.15046933162569331443592091090977191925048828125000,   42.50835148046414246891799848526716232299804687500000,   34.67541057556123007543646963313221931457519531250000,   26.61087840697986095506166748236864805221557617187500,   18.25026957604021049519360531121492385864257812500000,
          9.47585351965690136921693920157849788665771484375000,    0.00000000000000000000000000000000000000000000000000,  665.65385741110594608471728861331939697265625000000000,  667.38867525085993293032515794038772583007812500000000,  668.37798619828424762090435251593589782714843750000000,  668.91592115509718041721498593688011169433593750000000,  669.12505983675316656444920226931571960449218750000000,  669.07364384008633351186290383338928222656250000000000,  668.80543823740561037993757054209709167480468750000000,  668.35100792001412628451362252235412597656250000000000,
        667.73295676546626964409369975328445434570312500000000,  666.96870430015837882820051163434982299804687500000000,  666.07209891644527033349731937050819396972656250000000,  665.05442118122607553232228383421897888183593750000000,  663.92504152511264692293480038642883300781250000000000,  662.69186982630287729989504441618919372558593750000000,  661.36167343576312305231112986803054809570312500000000,  659.94030864020703575079096481204032897949218750000000,  658.43289323098133536404930055141448974609375000000000,  656.84393784780786518240347504615783691406250000000000,
        655.17744775083292552153579890727996826171875000000000,  653.43700292408391305798431858420372009277343750000000,  651.62582200188580827671103179454803466796875000000000,  649.74681391664285001752432435750961303710937500000000,  647.80262008847648758091963827610015869140625000000000,  645.79564923225666461803484708070755004882812500000000,  643.72810633285803305625449866056442260742187500000000,  641.60201696349372468830551952123641967773437500000000,  639.41924784838420237065292894840240478515625000000000,  637.18152436909076641313731670379638671875000000000000,
        634.89044556286580700543709099292755126953125000000000,  632.54749704716425640071975067257881164550781250000000,  630.15406221710634326882427558302879333496093750000000,  627.71143199519383415463380515575408935546875000000000,  625.22081335996347206673817709088325500488281250000000,  622.68333683885623486276017501950263977050781250000000,  620.10006311776692200510296970605850219726562500000000,  617.47198889350045192259130999445915222167968750000000,  614.80005207426358992961468175053596496582031250000000,  612.08513641622164414002327248454093933105468750000000,
        609.32807567021586692135315388441085815429687500000000,  606.52965730131006694136885926127433776855468750000000,  603.69062583441836977726779878139495849609375000000000,  600.81168587145293713547289371490478515625000000000000,  597.89350481893166033842135220766067504882812500000000,  594.93671535953956208686577156186103820800781250000000,  591.94191769656151791423326358199119567871093750000000,  588.90968159624503641680348664522171020507812500000000,  585.84054824986969833844341337680816650390625000000000,  582.73503197452043878001859411597251892089843750000000,
        579.59362176917466058512218296527862548828125000000000,  576.41678274067669462965568527579307556152343750000000,  573.20495741241711584734730422496795654296875000000000,  569.95856692701602241868386045098304748535156250000000,  566.67801215300471540103899314999580383300781250000000,  563.36367470435448012722190469503402709960937500000000,  560.01591788071345945354551076889038085937500000000000,  556.63508753534699735610047355294227600097656250000000,  553.22151287701535693486221134662628173828125000000000,  549.77550721135855837928829714655876159667968750000000,
        546.29736862677100361906923353672027587890625000000000,  542.78738062923309826146578416228294372558593750000000,  539.24581273010301174508640542626380920410156250000000,  535.67292099047278952639317139983177185058593750000000,  532.06894852532593631622148677706718444824218750000000,  528.43412597041356093541253358125686645507812500000000,  524.76867191447604454879183322191238403320312500000000,  521.07279329918378607544582337141036987304687500000000,  517.34668578893069934565573930740356445312500000000000,  513.59053411241256981156766414642333984375000000000000,
        509.80451237772928152480744756758213043212890625000000,  505.98878436257814428245183080434799194335937500000000,  502.14350378095451787885394878685474395751953125000000,  498.26881452762802382494555786252021789550781250000000,  494.36485090153735200146911665797233581542968750000000,  490.43173780912462689229869283735752105712890625000000,  486.46959094852263660868629813194274902343750000000000,  482.47851697540329496405320242047309875488281250000000,  478.45861365120373420722899027168750762939453125000000,  474.40996997435621551630902104079723358154296875000000,
        470.33266629506414346906240098178386688232421875000000,  466.22677441408842469172668643295764923095703125000000,  462.09235766593297967119724489748477935791015625000000,  457.92947098674562766973394900560379028320312500000000,  453.73816096718036305901478044688701629638671875000000,  449.51846589040093249423080123960971832275390625000000,  445.27041575533684181209537200629711151123046875000000,  440.99403228523812003913917578756809234619140625000000,  436.68932892151036639916128478944301605224609375000000,  432.35631080274310988897923380136489868164062500000000,
        427.99497472877919790334999561309814453125000000000000,  423.60530910960488881755736656486988067626953125000000,  419.18729389876699542583082802593708038330078125000000,  414.74090051095373610223759897053241729736328125000000,  410.26609172329500552223180420696735382080078125000000,  405.76282155986058342023170553147792816162109375000000,  401.23103515874623781201080419123172760009765625000000,  396.67066862104798019572626799345016479492187500000000,  392.08164884092468582821311429142951965332031250000000,  387.46389331584515502981957979500293731689453125000000,
        382.81730993599933299265103414654731750488281250000000,  378.14179675172880479294690303504467010498046875000000,  373.43724171769599706749431788921356201171875000000000,  368.70352241235963219878613017499446868896484375000000,  363.94050573115998759021749719977378845214843750000000,  359.14804755163527261174749583005905151367187500000000,  354.32599236848665213983622379601001739501953125000000,  349.47417289638559623199398629367351531982421875000000,  344.59240963806570334782009012997150421142578125000000,  339.68051041496022435239865444600582122802734375000000,
        334.73826985733177252768655307590961456298828125000000,  329.76546885048725243905209936201572418212890625000000,  324.76187393327029440115438774228096008300781250000000,  319.72723664457271297578699886798858642578125000000000,  314.66129281309559928558883257210254669189453125000000,  309.56376178500767082368838600814342498779296875000000,  304.43434558348485552414786070585250854492187500000000,  299.27272799335895570038701407611370086669921875000000,  294.07857356323250996865681372582912445068359375000000,  288.85152651641965348972007632255554199218750000000000,
        283.59120956091783227748237550258636474609375000000000,  278.29722258728759243240347132086753845214843750000000,  272.96914124176572613578173331916332244873046875000000,  267.60651536014199791679857298731803894042968750000000,  262.20886724582186388943227939307689666748046875000000,  256.77568977303354813557234592735767364501953125000000,  251.30644429323334065884409938007593154907226562500000,  245.80055831933324839155829977244138717651367187500000,  240.25742295830198713701975066214799880981445312500000,  234.67639005784062078419083263725042343139648437500000,
        229.05676902702552411028591450303792953491210937500000,  223.39782328382301557212485931813716888427734375000000,  217.69876627393517765085562132298946380615234375000000,  211.95875699516693657642463222146034240722656250000000,  206.17689494895651591832574922591447830200195312500000,  200.35221442527929980315093416720628738403320312500000,  194.48367800803970339984516613185405731201171875000000,  188.57016916427866703998006414622068405151367187500000,  182.61048375067480264988262206315994262695312500000000,  176.60332023308097859626286663115024566650390625000000,
        170.54726836672935519345628563314676284790039062500000,  164.44079602287450825315318070352077484130859375000000,  158.28223376733188842990784905850887298583984375000000,  152.06975669104275539211812429130077362060546875000000,  145.80136285313702160237880889326333999633789062500000,  139.47484750954947685386287048459053039550781250000000,  133.08777204545432937266014050692319869995117187500000,  126.63742617842238757930317660793662071228027343750000,  120.12078150704458323616563575342297554016113281250000,  113.53443377838303263160923961549997329711914062500000,
        106.87453022874422003951622173190116882324218750000000,  100.13667684026242454820021521300077438354492187500000,   93.31581805744809798852656967937946319580078125000000,   86.40607791807431681263551581650972366333007812500000,   79.40054576925386697894282406195998191833496093750000,   72.29098008236840655627020169049501419067382812500000,   65.06738706778361347460304386913776397705078125000000,   57.71740003763392223845585249364376068115234375000000,   50.22532576653645719488849863409996032714843750000000,   42.57059941152299131772451801225543022155761718750000,
         34.72510386537676652096706675365567207336425781250000,   26.64807024138869095963855215813964605331420898437500,   18.27501247652881133376467914786189794540405273437500,    9.48819935547919968144015001598745584487915039062500,    0.00000000000000000000000000000000000000000000000000,  670.74760761191271285497350618243217468261718750000000,  672.48350727325782827392686158418655395507812500000000,  673.47401498405849906703224405646324157714843750000000,  674.01324748702256783872144296765327453613281250000000,  674.22377488686095148295862600207328796386718750000000,
        674.17383192804527425323612987995147705078125000000000,  673.90717861537166299967793747782707214355468750000000,  673.45437598871342288475716486573219299316406250000000,  672.83802493832467916945461183786392211914062500000000,  672.07554263913687009335262700915336608886718750000000,  671.18077561387121932057198137044906616210937500000000,  670.16500293424928713648114353418350219726562500000000,  669.03759383315855302498675882816314697265625000000000,  667.80645723195561913598794490098953247070312500000000,  666.47835972321240660676266998052597045898437500000000,
        665.05915700127707168576307594776153564453125000000000,  663.55396640599838065099902451038360595703125000000000,  661.96729824666624608653364703059196472167968750000000,  660.30315755828723922604694962501525878906250000000000,  658.56512419233854416233953088521957397460937500000000,  656.75641673286281729815527796745300292968750000000000,  654.87994413579269803449278697371482849121093750000000,  652.93834791161020802974235266447067260742187500000000,  650.93403692659148873644880950450897216796875000000000,  648.86921637323439426836557686328887939453125000000000,
        646.74591208455444757419172674417495727539062500000000,  644.56599109336752917442936450242996215820312500000000,  642.33117913577643776079639792442321777343750000000000,  640.04307564713315059634624049067497253417968750000000,  637.70316668454620412376243621110916137695312500000000,  635.31283612266929594625253230333328247070312500000000,  632.87337540202838681580033153295516967773437500000000,  630.38599205652838008973048999905586242675781250000000,  627.85181720539435445971321314573287963867187500000000,  625.27191216197934409137815237045288085937500000000000,
        622.64727428564776801067637279629707336425781250000000,  619.97884218184469773404998704791069030761718750000000,  617.26750033836458442237926647067070007324218750000000,  614.51408327190392810734920203685760498046875000000000,  611.71937924755707172153051942586898803710937500000000,  608.88413362449557553190970793366432189941406250000000,  606.00905187326600298547418788075447082519531250000000,  603.09480230363851660513319075107574462890625000000000,  600.14201853649569784465711563825607299804687500000000,  597.15130174867579171404941007494926452636718750000000,
        594.12322271582502253295388072729110717773437500000000,  591.05832367503501245664665475487709045410156250000000,  587.95712002625862169225001707673072814941406250000000,  584.82010188911465320416027680039405822753906250000000,  581.64773552965471026254817843437194824218750000000000,  578.44046466990675980923697352409362792968750000000000,  575.19871169149996603664476424455642700195312500000000,  571.92287874336011554987635463476181030273437500000000,  568.61334876232922397321090102195739746093750000000000,  565.27048641457236044516321271657943725585937500000000,
        561.89463896476536319823935627937316894531250000000000,  558.48613707930280725122429430484771728515625000000000,  555.04529556909767507022479549050331115722656250000000,  551.57241407695948964828858152031898498535156250000000,  548.06777771401687004981795325875282287597656250000000,  544.53165764919936009391676634550094604492187500000000,  540.96431165537933338782750070095062255859375000000000,  537.36598461541996130108600482344627380371093750000000,  533.73690899105042717565083876252174377441406250000000,  530.07730525720046443893807008862495422363281250000000,
        526.38738230417322938592405989766120910644531250000000,  522.66733780979882340034237131476402282714843750000000,  518.91735858350455146137392148375511169433593750000000,  515.13762088405076156050199642777442932128906250000000,  511.32829071250733932174625806510448455810546875000000,  507.48952408189575180585961788892745971679687500000000,  503.62146726477317315584514290094375610351562500000000,  499.72425701991517144051613286137580871582031250000000,  495.79802079912690260243834927678108215332031250000000,  491.84287693510810868247062899172306060791015625000000,
        487.85893481119552461677812971174716949462890625000000,  483.84629501371148307953262701630592346191406250000000,  479.80504946756087747417041100561618804931640625000000,  475.73528155563633390556788071990013122558593750000000,  471.63706622251311273430474102497100830078125000000000,  467.51047006284272811171831563115119934082031250000000,  463.35555139478253749985015019774436950683593750000000,  459.17236031873181900664349086582660675048828125000000,  454.96093876157789281933219172060489654541015625000000,  450.72132050659229207667522132396697998046875000000000,
        446.45353120905360810866113752126693725585937500000000,  442.15758839760968612608849070966243743896484375000000,  437.83350146133039970663958229124546051025390625000000,  433.48127162233618037134874612092971801757812500000000,  429.10089189382534868855145759880542755126953125000000,  424.69234702325456964899785816669464111328125000000000,  420.25561342035820189266814850270748138427734375000000,  415.79065906962250664946623146533966064453125000000000,  411.29744342675212465110234916210174560546875000000000,  406.77591729858892222182475961744785308837890625000000,
        402.22602270585878159181447699666023254394531250000000,  397.64769272803050625952892005443572998046875000000000,  393.04085132947352576593402773141860961914062500000000,  388.40541316599779975149431265890598297119140625000000,  383.74128337074392902650288306176662445068359375000000,  379.04835731826688061119057238101959228515625000000000,  374.32652036552309482431155629456043243408203125000000,  369.57564756831845897977473214268684387207031250000000,  364.79560337161194638611050322651863098144531250000000,  359.98624127188759302953258156776428222656250000000000,
        355.14740344960551965414197184145450592041015625000000,  350.27892036951641330233542248606681823730468750000000,  345.38061034637706825378700159490108489990234375000000,  340.45227907332025552022969350218772888183593750000000,  335.49371910982006284029921516776084899902343750000000,  330.50470932584011052313144318759441375732421875000000,  325.48501429835107501276070252060890197753906250000000,  320.43438365595585537448641844093799591064453125000000,  315.35255136684577337291557341814041137695312500000000,  310.23923496473338445866829715669155120849609375000000,
        305.09413470674053314724005758762359619140625000000000,  299.91693265646671306967618875205516815185546875000000,  294.70729168458950653075589798390865325927734375000000,  289.46485437835531229211483150720596313476562500000000,  284.18924185016192041075555607676506042480468750000000,  278.88005243410492539624101482331752777099609375000000,  273.53686025781382795685203745961189270019531250000000,  268.15921367510253503496642224490642547607421875000000,  262.74663354285547711697290651500225067138671875000000,  257.29861132310554694413440302014350891113281250000000,
        251.81460698835451239574467763304710388183593750000000,  246.29404670475850025468389503657817840576171875000000,  240.73632026372789027846010867506265640258789062500000,  235.14077822764019742862728890031576156616210937500000,  229.50672874955739644065033644437789916992187500000000,  223.83343401985041509760776534676551818847656250000000,  218.12010628418789792704046703875064849853515625000000,  212.36590336707917003877810202538967132568359375000000,  206.56992362261070184104028157889842987060546875000000,  200.73120021858551353943767026066780090332031250000000,
        194.84869464117875281772285234183073043823242187500000,  188.92128928343552729529619682580232620239257812500000,  182.94777895108819620872964151203632354736328125000000,  176.92686108143360002031840849667787551879882812500000,  170.85712442290216017681814264506101608276367187500000,  164.73703586108769059137557633221149444580078125000000,  158.56492499669411699869669973850250244140625000000000,  152.33896597552910634476575069129467010498046875000000,  146.05715593101552940424880944192409515380859375000000,  139.71728921227540354266238864511251449584960937500000,
        133.31692631605000087802181951701641082763671875000000,  126.85335608936040330263494979590177536010742187500000,  120.32354927765085506052855635061860084533691406250000,  113.72410079177585373599868034943938255310058593750000,  107.05115704832476808405772317200899124145507812500000,  100.30032322577150694087322335690259933471679687500000,   93.46654298060333587727654958143830299377441406250000,   86.54393957780145285596518078818917274475097656250000,   79.52560160652537035730347270146012306213378906250000,   72.40328679466189498725725570693612098693847656250000,
         65.16700062317235619957500603049993515014648437500000,   57.80437568852106267058843513950705528259277343750000,   50.29971806304411074961535632610321044921875000000000,   42.63246221454715367826793226413428783416748046875000,   34.77449035931838494661860750056803226470947265625000,   26.68503294657087465679978777188807725906372070312500,   18.29960326130991887794152717106044292449951171875000,    9.50046944807101567675999831408262252807617187500000,    0.00000000000000000000000000000000000000000000000000,  675.84747403973688051337376236915588378906250000000000,
        677.58444776779276708111865445971488952636718750000000,  678.57614341312410033424384891986846923828125000000000,  679.11666367820464529359014704823493957519531250000000,  679.32856913384910058084642514586448669433593750000000,  679.28008772428995598602341488003730773925781250000000,  679.01497442323989162105135619640350341796875000000000,  678.56378644501148755807662382721900939941406250000000,  677.94912171097530517727136611938476562500000000000000,  677.18839505753942376031773164868354797363281250000000,  676.29545114657389603962656110525131225585937500000000,
        675.28156756014789152686716988682746887207031250000000,  674.15611233633387655572732910513877868652343750000000,  672.92699344040522646537283435463905334472656250000000,  671.60097670548850601335288956761360168457031250000000,  670.18391723096624446043279021978378295898437500000000,  668.68093190123863678309135138988494873046875000000000,  667.09653069001512903923867270350456237792968750000000,  665.43471840093445734964916482567787170410156250000000,  663.69907474571198235935298725962638854980468750000000,  661.89281824998465708631556481122970581054687500000000,
        660.01885788423658141255145892500877380371093750000000,  658.07983523952793802891392260789871215820312500000000,  656.07815932298478855955181643366813659667968750000000,  654.01603552342874081659829244017601013183593750000000,  651.89548992165418894728645682334899902343750000000000,  649.71838984633825475611956790089607238769531250000000,  647.48646137469256700569530948996543884277343750000000,  645.20130432604037196142598986625671386718750000000000,  642.86440518232348040328361093997955322265625000000000,  640.47714828221432981081306934356689453125000000000000,
        638.04082556804826253937790170311927795410156250000000,  635.55664511217878498428035527467727661132812500000000,  633.02573860797849647497059777379035949707031250000000,  630.44916797789414886210579425096511840820312500000000,  627.82793122474288338707992807030677795410156250000000,  625.16296763134221237123711034655570983886718750000000,  622.45516239647281508950982242822647094726562500000000,  619.70535078124567007762379944324493408203125000000000,  616.91432182852440746501088142395019531250000000000000,  614.08282170863242299674311652779579162597656250000000,
        611.21155673677401409804588183760643005371093750000000,  608.30119610109545646992046386003494262695312500000000,  605.35237433487066027737455442547798156738281250000000,  602.36569356172299194440711289644241333007812500000000,  599.34172553893154145043808966875076293945312500000000,  596.28101352059991313581122085452079772949218750000000,  593.18407395967528827895876020193099975585937500000000,  590.05139806542899805208435282111167907714843750000000,  586.88345323097121308819623664021492004394531250000000,  583.68068434361316576541867107152938842773437500000000,
        580.44351498938078748324187472462654113769531250000000,  577.17234856167419820849318057298660278320312500000000,  573.86756928292106749722734093666076660156250000000000,  570.52954314709404570749029517173767089843750000000000,  567.15861879008321011497173458337783813476562500000000,  563.75512829416891236178344115614891052246093750000000,  560.31938793216556859988486394286155700683593750000000,  556.85169885622701713145943358540534973144531250000000,  553.35234773578520162118365988135337829589843750000000,  549.82160734863464313093572854995727539062500000000000,
        546.25973712877373600349528715014457702636718750000000,  542.66698367424908155953744426369667053222656250000000,  539.04358121792859037668677046895027160644531250000000,  535.38975206384498051193077117204666137695312500000000,  531.70570699148743187834043055772781372070312500000000,  527.99164563019553497724700719118118286132812500000000,  524.24775680559571355843218043446540832519531250000000,  520.47421885983442280121380463242530822753906250000000,  516.67119994719587339204736053943634033203125000000000,  512.83885830653002813050989061594009399414062500000000,
        508.97734251178502518087043426930904388427734375000000,  505.08679170180187156802276149392127990722656250000000,  501.16733579041766688533243723213672637939453125000000,  497.21909565781163564679445698857307434082031250000000,  493.24218332392939601049874909222126007080078125000000,  489.23670210472869257500860840082168579101562500000000,  485.20274675190239577204920351505279541015625000000000,  481.14040357665408009779639542102813720703125000000000,  477.04975055802492533985059708356857299804687500000000,  472.93085743619866434528375975787639617919921875000000,
        468.78378579114195190413738600909709930419921875000000,  464.60858910687176148712751455605030059814453125000000,  460.40531282157741088667535223066806793212890625000000,  456.17399436376177845886559225618839263916015625000000,  451.91466317450635870045516639947891235351562500000000,  447.62734071590352868952322751283645629882812500000000,  443.31204046563806286940234713256359100341796875000000,  438.96876789764235127222491428256034851074218750000000,  434.59752044868423581647220999002456665039062500000000,  430.19828747068640950601547956466674804687500000000000,
        425.77105016851066920935409143567085266113281250000000,  421.31578152287204375170404091477394104003906250000000,  416.83244619798028907098341733217239379882812500000000,  412.32100043342876460883417166769504547119140625000000,  407.78139191977498967389692552387714385986328125000000,  403.21355965717322078489814884960651397705078125000000,  398.61743379632952155589009635150432586669921875000000,  393.99293546095367446469026617705821990966796875000000,  389.33997655077877197982161305844783782958984375000000,  384.65845952410541030985768884420394897460937500000000,
        379.94827715870451356749981641769409179687500000000000,  375.20931228977775617750012315809726715087890625000000,  370.44143752352454157517058774828910827636718750000000,  365.64451492470254834188381209969520568847656250000000,  360.81839567638616017575259320437908172607421875000000,  355.96291970992484721136861480772495269775390625000000,  351.07791530288329795439494773745536804199218750000000,  346.16319864248890780800138600170612335205078125000000,  341.21857335183830173264141194522380828857421875000000,  336.24382997579652965214336290955543518066406250000000,
        331.23874542317128089052857831120491027832031250000000,  326.20308236134417256835149601101875305175781250000000,  321.13658855909187650468084029853343963623046875000000,  316.03899617281723521955427713692188262939453125000000,  310.91002097083116950670955702662467956542968750000000,  305.74936148966054361153510399162769317626953125000000,  300.55669811560289872431894764304161071777343750000000,  295.33169208387738535748212598264217376708984375000000,  290.07398438672612428490538150072097778320312500000000,  284.78319458066437164234230294823646545410156250000000,
        279.45891948175022889699903316795825958251953125000000,  274.10073173619451836202642880380153656005859375000000,  268.70817825183559079960105009377002716064453125000000,  263.28077847389619137175031937658786773681640625000000,  257.81802248597705329302698373794555664062500000000000,  252.31936891433548453278490342199802398681640625000000,  246.78424261007012319169007241725921630859375000000000,  241.21203207975815985264489427208900451660156250000000,  235.60208663024306474653712939471006393432617187500000,  229.95371318746188649129180703312158584594726562500000,
        224.26617274221320030846982263028621673583984375000000,  218.53867636732192636372928973287343978881835937500000,  212.77038074038830473000416532158851623535156250000000,  206.96038309376046981924446299672126770019531250000000,  201.10771549793773260716989170759916305541992187500000,  195.21133836551661033809068612754344940185546875000000,  189.27013303900506002719339448958635330200195312500000,  183.28289329598064227866416331380605697631835937500000,  177.24831556733212778453889768570661544799804687500000,  171.16498761621534185906057246029376983642578125000000,
        165.03137536349140646052546799182891845703125000000000,  158.84580746510243898228509351611137390136718750000000,  152.60645714151397100977192167192697525024414062500000,  146.31132061969427127223752904683351516723632812500000,  139.95819136068402599448745604604482650756835937500000,  133.54462899101997663819929584860801696777343750000000,  127.06792150491541804058215348049998283386230468750000,  120.52503881193987922415544744580984115600585937500000,  113.91257500355764875621389364823698997497558593750000,  107.22667569301881940191378816962242126464843750000000,
        100.46294527108960892292088828980922698974609375000000,   93.61632662177530050939822103828191757202148437500000,   86.68094225240835726253862958401441574096679687500000,   79.64988000895269237844331655651330947875976562500000,   72.51489689018487183602701406925916671752929687500000,   65.26599766168519067832676228135824203491210937500000,   57.89081421792108272939003654755651950836181640625000,   50.37365194228191001002414850518107414245605468750000,   42.69394462635293052699125837534666061401367187500000,   34.82357382261459122219093842431902885437011718750000,
         26.72176932858081244148706900887191295623779296875000,   18.32404378937151889772394497413188219070434570312500,    9.51266472116483363663519412511959671974182128906250,    0.00000000000000000000000000000000000000000000000000,  680.95341951363741372915683314204216003417968750000000,  682.69145965697714473208179697394371032714843750000000,  683.68433452830572605307679623365402221679687500000000,  684.22613290697415777685819193720817565917968750000000,  684.43940590572844939742935821413993835449218750000000,  684.39237472003435414080740883946418762207031250000000,
        684.12878932853516289469553157687187194824218750000000,  683.67920314559421512967674061655998229980468750000000,  683.06621114196730104595189914107322692871093750000000,  682.30722582841667644970584660768508911132812500000000,  681.41609001474262186093255877494812011718750000000000,  680.40407979893791434733429923653602600097656250000000,  679.28056202726145329506834968924522399902343750000000,  678.05344370978741608269046992063522338867187500000000,  676.72948991930513784609502181410789489746093750000000,  675.31455515781715348566649481654167175292968750000000,
        673.81375585053422128112288191914558410644531250000000,  672.23160163066961558797629550099372863769531250000000,  670.57209706452226782857906073331832885742187500000000,  668.83882171710854436241788789629936218261718750000000,  667.03499404783167392452014610171318054199218750000000,  665.16352303306553039874415844678878784179687500000000,  663.22705033499869387014769017696380615234375000000000,  661.22798509140466194367036223411560058593750000000000,  659.16853287650076254067244008183479309082031250000000,  657.05072000722725533705670386552810668945312500000000,
        654.87641409579964602016843855381011962890625000000000,  652.64734154753466555121121928095817565917968750000000,  650.36510255204996155953267589211463928222656250000000,  648.03118400177038438414456322789192199707031250000000,  645.64697068436476001807022839784622192382812500000000,  643.21375502827766013069776818156242370605468750000000,  640.73274562792767028440721333026885986328125000000000,  638.20507473375880636012880131602287292480468750000000,  635.63180485953159859491279348731040954589843750000000,  633.01393463301860720093827694654464721679687500000000,
        630.35240399517988407751545310020446777343750000000000,  627.64809883580210225773043930530548095703125000000000,  624.90185513966184771561529487371444702148437500000000,  622.11446270585224738169927150011062622070312500000000,  619.28666849349497169896494597196578979492187500000000,  616.41917963925982348882826045155525207519531250000000,  613.51266618561169252643594518303871154785156250000000,  610.56776355326394423173042014241218566894531250000000,  607.58507478674698631948558613657951354980468750000000,  604.56517259813756481889868155121803283691406250000000,
        601.50860123072209262318210676312446594238281250000000,  598.41587816158209989225724712014198303222656250000000,  595.28749565971190804702928289771080017089843750000000,  592.12392221423795035661896690726280212402343750000000,  588.92560384555486052704509347677230834960937500000000,  585.69296531068368949490832164883613586425781250000000,  582.42641121284123073564842343330383300781250000000000,  579.12632702407779561326606199145317077636718750000000,  575.79308002884658890252467244863510131835937500000000,  572.42702019550404202163917943835258483886718750000000,
        569.02848098198285242688143625855445861816406250000000,  565.59778008121418224618537351489067077636718750000000,  562.13522011129111888294573873281478881835937500000000,  558.64108925484640622016740962862968444824218750000000,  555.11566185166270770423579961061477661132812500000000,  551.55919894812927850580308586359024047851562500000000,  547.97194880679273865098366513848304748535156250000000,  544.35414737893677283864235505461692810058593750000000,  540.70601874283170218404848128557205200195312500000000,  537.02777551004135148104978725314140319824218750000000,
        533.31961920194419235485838726162910461425781250000000,  529.58174059841564940143143758177757263183593750000000,  525.81432006043439741915790364146232604980468750000000,  522.01752782820267384522594511508941650390625000000000,  518.19152429622079125692835077643394470214843750000000,  514.33646026661028827220434322953224182128906250000000,  510.45247718186203655932331457734107971191406250000000,  506.53970733805533654958708211779594421386718750000000,  502.59827407950035649264464154839515686035156250000000,  498.62829197564678906928747892379760742187500000000000,
        494.62986698101553884043823927640914916992187500000000,  490.60309657882191913813585415482521057128906250000000,  486.54806990888050677313003689050674438476562500000000,  482.46486788030523484849254600703716278076171875000000,  478.35356326944923921473673544824123382568359375000000,  474.21422080345934091383242048323154449462890625000000,  470.04689722975638233037898316979408264160156250000000,  465.85164137169141440608655102550983428955078125000000,  461.62849417056435186168528161942958831787109375000000,  457.37748871413674578434438444674015045166015625000000,
        453.09865025170790886477334424853324890136718750000000,  448.79199619576837676504510454833507537841796875000000,  444.45753611018625406359205953776836395263671875000000,  440.09527168482225079060299322009086608886718750000000,  435.70519669641220161793171428143978118896484375000000,  431.28729695549139933063997887074947357177734375000000,  426.84155023907476333988597616553306579589843750000000,  422.36792620874030035338364541530609130859375000000000,  417.86638631369385166181018576025962829589843750000000,  413.33688367832064614049158990383148193359375000000000,
        408.77936297365346263177343644201755523681640625000000,  404.19376027210222446228726767003536224365234375000000,  399.58000288470390160000533796846866607666015625000000,  394.93800918005581479519605636596679687500000000000000,  390.26768838398987782056792639195919036865234375000000,  385.56894035893691352612222544848918914794921875000000,  380.84165536180353228701278567314147949218750000000000,  376.08571377905275312514277175068855285644531250000000,  371.30098583752828744763974100351333618164062500000000,  366.48733128940222059100051410496234893798828125000000,
        361.64459906944239264703355729579925537109375000000000,  356.77262692259603227284969761967658996582031250000000,  351.87124099966314361154218204319477081298828125000000,  346.94025541858110273096826858818531036376953125000000,  341.97947178856577465921873226761817932128906250000000,  336.98867869403733266153722070157527923583984375000000,  331.96765113490829435249906964600086212158203125000000,  326.91614991941190737634315155446529388427734375000000,  321.83392100519853329387842677533626556396484375000000,  316.72069478391637176173389889299869537353515625000000,
        311.57618530391340527785359881818294525146484375000000,  306.40008942503260414014221169054508209228515625000000,  301.19208589871686854166910052299499511718750000000000,  295.95183436577178781590191647410392761230468750000000,  290.67897426313459163793595507740974426269531250000000,  285.37312362984783931096899323165416717529296875000000,  280.03387780110369931207969784736633300781250000000000,  274.66080797767830290467827580869197845458984375000000,  269.25345965627758459959295578300952911376953125000000,  263.81135090421014410821953788399696350097656250000000,
        258.33397045933946856166585348546504974365234375000000,  252.82077563336201819765847176313400268554687500000000,  247.27118999302948054719308856874704360961914062500000,  241.68460078986140615597832947969436645507812500000000,  236.06035610404302360620931722223758697509765625000000,  230.39776166239627741560980211943387985229492187500000,  224.69607728332391616277163848280906677246093750000000,  218.95451289318100407399469986557960510253906250000000,  213.17222404825992043697624467313289642333984375000000,  207.34830688402698228856024798005819320678710937500000,
        201.48179239781663341091189067810773849487304687500000,  195.57163995209404561137489508837461471557617187500000,  189.61672986161030962648510467261075973510742187500000,  183.61585489792528846919594798237085342407226562500000,  177.56771050703611081189592368900775909423828125000000,  171.47088348774161659093806520104408264160156250000000,  165.32383881650991952483309432864189147949218750000000,  159.12490422430289527255808934569358825683593750000000,  152.87225202548663105517334770411252975463867187500000,  146.56387755929682725763996131718158721923828125000000,
        140.19757341691192209509608801454305648803710937500000,  133.77089837239708458582754246890544891357421875000000,  127.28113958442099828971549868583679199218750000000000,  120.72526614348724649516952922567725181579589843750000,  114.09987133803933545550535200163722038269042968750000,  107.40109999393249040622322354465723037719726562500000,  100.62455572975838435922923963516950607299804687500000,   93.76518067217557472758926451206207275390625000000000,   86.81709658561773323981469729915261268615722656250000,   79.77339058723028131225873949006199836730957031250000,
         72.62581896069544029614917235448956489562988281250000,   65.36438576987848136923275887966156005859375000000000,   57.97672222057722279942026943899691104888916015625000,   50.44713302023303924670472042635083198547363281250000,   42.75505129689135941362110315822064876556396484375000,   34.87235795159689644151512766256928443908691406250000,   26.75828214223731649212822958361357450485229492187500,   18.34833588583000718585935828741639852523803710937500,    9.52478608169717766429585026344284415245056152343750,    0.00000000000000000000000000000000000000000000000000,
        686.06540730199401423305971547961235046386718750000000,  687.80450631063888522476190701127052307128906250000000,  688.79855181736843405815307050943374633789062500000000,  689.34161879389239402371458709239959716796875000000000,  689.55624896971153248159680515527725219726562500000000,  689.51065684235891239950433373451232910156250000000000,  689.24858743101231084438040852546691894531250000000000,  688.80059037544117472862126305699348449707031250000000,  688.18925771390877343947067856788635253906250000000000,  687.43199964434359117149142548441886901855468750000000,
        686.54265713325276010436937212944030761718750000000000,  685.53250480017482004768680781126022338867187500000000,  684.41090830264283795258961617946624755859375000000000,  683.18577369653814912453526630997657775878906250000000,  681.86386529357071140111656859517097473144531250000000,  680.45103699612673153751529753208160400390625000000000,  678.95240476668891460576560348272323608398437500000000,  677.37247789328330327407456934452056884765625000000000,  675.71526069914125400828197598457336425781250000000000,  673.98433259590831312380032613873481750488281250000000,
        672.18291196920995389518793672323226928710937500000000,  670.31390779295190895936684682965278625488281250000000,  668.37996179132198903971584513783454895019531250000000,  666.38348322287913561012828722596168518066406250000000,  664.32667783666602190351113677024841308593750000000000,  662.21157217451241194794420152902603149414062500000000,  660.04003412024451336037600412964820861816406250000000,  657.81379039469516101235058158636093139648437500000000,  655.53444154453222836309578269720077514648437500000000,  653.20347485876789050962543115019798278808593750000000,
        650.82227555951772046682890504598617553710937500000000,  648.39213654613126891490537673234939575195312500000000,  645.91426691922504232934443280100822448730468750000000,  643.38979946977690360654378309845924377441406250000000,  640.81979728564340348384575918316841125488281250000000,  638.20525960164195566903799772262573242187500000000000,  635.54712699825643085205228999257087707519531250000000,  632.84628603693727200152352452278137207031250000000000,  630.10357340604184628318762406706809997558593750000000,  627.31977964004579462198307737708091735839843750000000,
        624.49565246523854966653743758797645568847656250000000,  621.63189981731670741282869130373001098632812500000000,  618.72919256979162128118332475423812866210937500000000,  615.78816700668380690331105142831802368164062500000000,  612.80942706840835398907074704766273498535156250000000,  609.79354639589598718885099515318870544433593750000000,  606.74107019471750845696078613400459289550781250000000,  603.65251693820130185486050322651863098144531250000000,  600.52837992614900031185243278741836547851562500000000,  597.36912871372055633401032537221908569335937500000000,
        594.17521042330270120146451517939567565917968750000000,  590.94705095066433386818971484899520874023437500000000,  587.68505607539202628686325624585151672363281250000000,  584.38961248445889395952690392732620239257812500000000,  581.06108871679487037908984348177909851074218750000000,  577.69983603585762921284185722470283508300781250000000,  574.30618923644715323462150990962982177734375000000000,  570.88046739134347262734081596136093139648437500000000,  567.42297454276058488176204264163970947265625000000000,  563.93400034309422608203021809458732604980468750000000,
        560.41382064898357384663540869951248168945312500000000,  556.86269807230178230383899062871932983398437500000000,  553.28088249133270437596365809440612792968750000000000,  549.66861152506612597790081053972244262695312500000000,  546.02611097326030176191125065088272094726562500000000,  542.35359522466444559540832415223121643066406250000000,  538.65126763556042988057015463709831237792968750000000,  534.91932088057990313245682045817375183105468750000000,  531.15793727756204134493600577116012573242187500000000,  527.36728908805150695116026327013969421386718750000000,
        523.54753879488259826757712289690971374511718750000000,  519.69883935815312270278809592127799987792968750000000,  515.82133445076738098578061908483505249023437500000000,  511.91515867461322386589017696678638458251953125000000,  507.98043775832451274254708550870418548583984375000000,  504.01728873748993464687373489141464233398437500000000,  500.02582011807254502855357713997364044189453125000000,  496.00613202372215937430155463516712188720703125000000,  491.95831632758353180179256014525890350341796875000000,  487.88245676912890758103458210825920104980468750000000,
        483.77862905647361912997439503669738769531250000000000,  479.64690095456819562969030812382698059082031250000000,  475.48733235959502962941769510507583618164062500000000,  471.29997535984006162834702990949153900146484375000000,  467.08487428324781376431928947567939758300781250000000,  462.84206573181347721401834860444068908691406250000000,  458.57157860290891449039918370544910430908203125000000,  454.27343409758168490952812135219573974609375000000000,  449.94764571581504242203664034605026245117187500000000,  445.59421923867608938962803222239017486572265625000000,
        441.21315269722680341146769933402538299560546875000000,  436.80443632801285502864629961550235748291015625000000,  432.36805251488573276219540275633335113525390625000000,  427.90397571685315369904856197535991668701171875000000,  423.41217238158742475206963717937469482421875000000000,  418.89260084415366236498812213540077209472656250000000,  414.34521121045059999232762493193149566650390625000000,  409.76994522477622240330674685537815093994140625000000,  405.16673612085423883399926126003265380859375000000000,  400.53550845556532067348598502576351165771484375000000,
        395.87617792453676202057977207005023956298828125000000,  391.18865115863724213340901769697666168212890625000000,  386.47282550031650316668674349784851074218750000000000,  381.72858875860316629768931306898593902587890625000000,  376.95581894144288526149466633796691894531250000000000,  372.15438396391130027041072025895118713378906250000000,  367.32414133067129569099051877856254577636718750000000,  362.46493779086574704706436023116111755371093750000000,  357.57660896343554668419528752565383911132812500000000,  352.65897893063078072373173199594020843505859375000000,
        347.71185979723020409437594935297966003417968750000000,  342.73505121270983408976462669670581817626953125000000,  337.72833985328293238126207143068313598632812500000000,  332.69149886038599106541369110345840454101562500000000,  327.62428723178271638971637003123760223388671875000000,  322.52644916101019134657690301537513732910156250000000,  317.39771332038071705028414726257324218750000000000000,  312.23779208217098357636132277548313140869140625000000,  307.04638067196862039054394699633121490478515625000000,  301.82315624738851056463317945599555969238281250000000,
        296.56777689450416346517158672213554382324218750000000,  291.27988053334041751440963707864284515380859375000000,  285.95908372262215380033012479543685913085937500000000,  280.60498035264305372038506902754306793212890625000000,  275.21714021357172441639704629778861999511718750000000,  269.79510742471376261164550669491291046142578125000000,  264.33839870814358619099948555231094360351562500000000,  258.84650148765626909153070300817489624023437500000000,  253.31887179108366581203881651163101196289062500000000,  247.75493193059236318731564097106456756591796875000000,
        242.15406793150634712219471111893653869628906250000000,  236.51562667534923889434139709919691085815429687500000,  230.83891271699152980545477475970983505249023437500000,  225.12318472880161834837053902447223663330078125000000,  219.36765151625354519637767225503921508789062500000000,  213.57146753917641035513952374458312988281250000000000,  207.73372786028212999553943518549203872680664062500000,  201.85346242717608333805401343852281570434570312500000,  195.92962957496129661194572690874338150024414062500000,  189.96110861275838033179752528667449951171875000000000,
        183.94669132761529795061505865305662155151367187500000,  177.88507220154491506036720238626003265380859375000000,  171.77483708931879391457187011837959289550781250000000,  165.61445004278368742234306409955024719238281250000000,  159.40223788715434238838497549295425415039062500000000,  153.13637204941014147152600344270467758178710937500000,  146.81484699926406278791546355932950973510742187500000,  140.43545447575644402604666538536548614501953125000000,  133.99575241773527523037046194076538085937500000000000,  127.49302716512485744715377222746610641479492187500000,
        120.92424700572387052943668095394968986511230468750000,  114.28600444089128984614944783970713615417480468750000,  107.57444352461270398180204210802912712097167968750000,  100.78516711843259656689042458310723304748535156250000,   93.91311660640518255149800097569823265075683593750000,   86.95241302443497488638968206942081451416015625000000,   79.89614277485740956308291060850024223327636718750000,   72.73606143991931105574622051790356636047363281250000,   65.46217239509114449447224615141749382019042968750000,   58.06210617049298150504910154268145561218261718750000,
         50.52016681029209621556219644844532012939453125000000,   42.81578679135949982992315199226140975952148437500000,   34.92084637537065816559334052726626396179199218750000,   26.79457409236348297554286546073853969573974609375000,   18.37248134274812727539938350673764944076538085937500,    9.53683442021335281424398999661207199096679687500000,    0.00000000000000000000000000000000000000000000000000,  691.18340111441068529529729858040809631347656250000000,  692.92355153787525523512158542871475219726562500000000,  693.91875920503412089601624757051467895507812500000000,
        694.46308539383755942253628745675086975097656250000000,  694.67906252437921921227825805544853210449218750000000,  694.63489844646505844139028340578079223632812500000000,  694.37433325500774117244873195886611938476562500000000,  693.92791284028407972073182463645935058593750000000000,  693.31822632604121281474363058805465698242187500000000,  692.56268161011701067764079198241233825683593750000000,  691.67511782449685142637463286519050598144531250000000,  690.66680811592766531248344108462333679199218750000000,  689.54711695638616220094263553619384765625000000000000,
        688.32394944868224229139741510152816772460937500000000,  687.00406914284963022510055452585220336914062500000000,  685.59332933959171896276529878377914428710937500000000,  684.09684553531872097664745524525642395019531250000000,  682.51912666839314169919816777110099792480468750000000,  680.86417681348746100411517545580863952636718750000000,  679.13557522245957898121559992432594299316406250000000,  677.33654019988705385912908241152763366699218750000000,  675.46998070914048639679094776511192321777343750000000,  673.53853852758879838802386075258255004882812500000000,
        671.54462302504020954074803739786148071289062500000000,  669.49044011513854002259904518723487854003906250000000,  667.37801655370935804967302829027175903320312500000000,  665.20922048464194631378632038831710815429687500000000,  662.98577893209596822998719289898872375488281250000000,  660.70929278696280562144238501787185668945312500000000,  658.38124972138268731214338913559913635253906250000000,  656.00303537782542662171181291341781616210937500000000,  653.57594311181696866697166115045547485351562500000000,  651.10118251480150775023503229022026062011718750000000,
        648.57988690227307415625546127557754516601562500000000,  646.01311991950728952360805124044418334960937500000000,  643.40188139101917386142304167151451110839843750000000,  640.74711251878613893495639786124229431152343750000000,  638.04970051719283219426870346069335937500000000000000,  635.31048275873206421238137409090995788574218750000000,  632.53025049308132565784035250544548034667968750000000,  629.70975219276078860275447368621826171875000000000000,  626.84969657077931515232194215059280395507812500000000,  623.95075530917756623239256441593170166015625000000000,
        621.01356553193704712612088769674301147460937500000000,  618.03873205115621658478630706667900085449218750000000,  615.02682941153136653156252577900886535644531250000000,  611.97840375491318809508811682462692260742187500000000,  608.89397452392040577251464128494262695312500000000000,  605.77403602121910353162093088030815124511718750000000,  602.61905883903614267183002084493637084960937500000000,  599.42949117171770012646447867155075073242187500000000,  596.20576002264306225697509944438934326171875000000000,  592.94827231547731116734212264418601989746093750000000,
        589.65741591862672521529020741581916809082031250000000,  586.33356059075913435663096606731414794921875000000000,  582.97705885439108897116966545581817626953125000000000,  579.58824680378882021614117547869682312011718750000000,  576.16744485276126397366169840097427368164062500000000,  572.71495842734282177843851968646049499511718750000000,  569.23107860784273270837729796767234802246093750000000,  565.71608272428829877753742039203643798828125000000000,  562.17023490887606840260559692978858947753906250000000,  558.59378660869333543814718723297119140625000000000000,
        554.98697706164580267795827239751815795898437500000000,  551.35003373824577010964276269078254699707031250000000,  547.68317275165327373542822897434234619140625000000000,  543.98659923814170724654104560613632202148437500000000,  540.26050770994061167584732174873352050781250000000000,  536.50508238223471835226519033312797546386718750000000,  532.72049747591984214523108676075935363769531250000000,  528.90691749756842909846454858779907226562500000000000,  525.06449749791920567076886072754859924316406250000000,  521.19338331007372744352323934435844421386718750000000,
        517.29371176847462265868671238422393798828125000000000,  513.36561090962857178965350612998008728027343750000000,  509.40920015543963472737232223153114318847656250000000,  505.42459047993168042012257501482963562011718750000000,  501.41188456005272655602311715483665466308593750000000,  497.37117691117498452513245865702629089355468750000000,  493.30255400783380537177436053752899169921875000000000,  489.20609439017835029517300426959991455078125000000000,  485.08186875654195091556175611913204193115234375000000,  480.92994004247935890816734172403812408447265625000000,
        476.75036348655635265458840876817703247070312500000000,  472.54318668312299678291310556232929229736328125000000,  468.30844962224392702410113997757434844970703125000000,  464.04618471690554315500776283442974090576171875000000,  459.75641681756695788862998597323894500732421875000000,  455.43916321406720726372441276907920837402343750000000,  451.09443362484859108008095063269138336181640625000000,  446.72223017340257911200751550495624542236328125000000,  442.32254735178850069132749922573566436767578125000000,  437.89537197102163190720602869987487792968750000000000,
        433.44068309806692695929086767137050628662109375000000,  428.95845197911518198452540673315525054931640625000000,  424.44864194875691509878379292786121368408203125000000,  419.91120832460023848398122936487197875976562500000000,  415.34609828681072940526064485311508178710937500000000,  410.75325074197451158397598192095756530761718750000000,  406.13259617060748496442101895809173583984375000000000,  401.48405645754365878019598312675952911376953125000000,  396.80754470434629865849274210631847381591796875000000,  392.10296502277901709021534770727157592773437500000000,
        387.37021230826661621904349885880947113037109375000000,  382.60917199215236905729398131370544433593750000000000,  377.81971977142467267185566015541553497314453125000000,  373.00172131444043088777107186615467071533203125000000,  368.15503194100847395020537078380584716796875000000000,  363.27949627501760687664500437676906585693359375000000,  358.37494786759276621523895300924777984619140625000000,  353.44120878854158718240796588361263275146484375000000,  348.47808918360209418096928857266902923583984375000000,  343.48538679472625290145515464246273040771484375000000,
        338.46288644031801595701836049556732177734375000000000,  333.41035945199541856709402054548263549804687500000000,  328.32756306404536417176132090389728546142578125000000,  323.21423975129062000632984563708305358886718750000000,  318.07011651058024881422170437872409820556640625000000,  312.89490408052949987904867157340049743652343750000000,  307.68829609347875475577893666923046112060546875000000,  302.44996815287913705105893313884735107421875000000000,  297.17957682844740929795079864561557769775390625000000,  291.87675856043426847463706508278846740722656250000000,
        286.54112846319810614659218117594718933105468750000000,  281.17227901694570846302667632699012756347656250000000,  275.76977863495494602830149233341217041015625000000000,  270.33317009179603473967290483415126800537109375000000,  264.86196879596349162966362200677394866943359375000000,  259.35566088786617910955101251602172851562500000000000,  253.81370114121935444018163252621889114379882812500000,  248.23551064245305042277323082089424133300781250000000,  242.62047421867944763107516337186098098754882812500000,  236.96793757991142115315597038716077804565429687500000,
        231.27720413541700850146298762410879135131835937500000,  225.54753143710706808633403852581977844238281250000000,  219.77812719440743194354581646621227264404296875000000,  213.96814479479971282671613153070211410522460937500000,  208.11667825166600209740863647311925888061523437500000,  202.22275648564101402371306903660297393798828125000000,  196.28533682658118664221547078341245651245117187500000,  190.30329759947201750946987885981798171997070312500000,  184.27542962774688817262358497828245162963867187500000,  178.20042644975401913143286947160959243774414062500000,
        172.07687299599965058405359741300344467163085937500000,  165.90323241293293676790199242532253265380859375000000,  159.67783063872482784972817171365022659301757812500000,  153.39883823116883831971790641546249389648437500000000,  147.06424880817058920001727528870105743408203125000000,  140.67185327387795723552699200809001922607421875000000,  134.21920874871355522373050916939973831176757812500000,  127.70360077021013012199546210467815399169921875000000,  121.12199683938963801210775272920727729797363281250000,  114.47098868604378196778270648792386054992675781250000,
        107.74671960740818121848860755562782287597656250000000,  100.94479172271500999613635940477252006530761718750000,   94.06014568777627005147223826497793197631835937500000,   87.08690182396863121994101675227284431457519531250000,   80.01814583246944323491334216669201850891113281250000,   72.84563260740333134890533983707427978515625000000000,   65.55936484883110892951663117855787277221679687500000,   58.14697242386231579303057515062391757965087890625000,   50.59275872574903587519656866788864135742187500000000,   42.87615559224796868420526152476668357849121093750000,
         34.96904265743559392376482719555497169494628906250000,   26.83064783498926075822055281605571508407592773437500,   18.39648191992835180030851915944367647171020507812500,    9.54881061126006791539566620485857129096984863281250,    0.00000000000000000000000000000000000000000000000000,  696.30736509381404175655916333198547363281250000000000,  698.04855957920028686203295364975929260253906250000000,  699.04492104518715223093749955296516418457031250000000,  699.59049718827827746281400322914123535156250000000000,  699.80781119203129492234438657760620117187500000000000,
        699.76506430811241443734616041183471679687500000000000,  699.50599174197031970834359526634216308593750000000000,  699.06113565924090380576672032475471496582031250000000,  698.45308228698377206455916166305541992187500000000000,  697.69923723562146733456756919622421264648437500000000,  696.81343781137809401116101071238517761230468750000000,  695.80695569391025401273509487509727478027343750000000,  694.68915417288485514291096478700637817382812500000000,  693.46793739927045407966943457722663879394531250000000,  692.15006816096592956455424427986145019531250000000000,
        690.74139915508919784770114347338676452636718750000000,  689.24704540881919001549249514937400817871093750000000,  687.67151550658638825552770867943763732910156250000000,  686.01881326923933102079899981617927551269531250000000,  684.29251778267700956348562613129615783691406250000000,  682.49584726342311569169396534562110900878906250000000,  680.63171065652704783133231103420257568359375000000000,  678.70274978401846510678296908736228942871093750000000,  676.71137411773520398128312081098556518554687500000000,  674.65978972602931662549963220953941345214843750000000,
        672.55002356817840336589142680168151855468750000000000,  670.38394403695519940811209380626678466796875000000000,  668.16327844803993230016203597187995910644531250000000,  665.88962802412447672395501285791397094726562500000000,  663.56448080743757600430399179458618164062500000000000,  661.18922284715085879724938422441482543945312500000000,  658.76514794069635172490961849689483642578125000000000,  656.29346615545557597215520218014717102050781250000000,  653.77531131591922530787996947765350341796875000000000,  651.21174760862606945011066272854804992675781250000000,
        648.60377543098479691252578049898147583007812500000000,  645.95233658900178852491080760955810546875000000000000,  643.25831893185602439189096912741661071777343750000000,  640.52256049734455700672697275876998901367187500000000,  637.74585323080611942714313045144081115722656250000000,  634.92894633072251053818035870790481567382812500000000,  632.07254926639564018842065706849098205566406250000000,  629.17733450660284688638057559728622436523437500000000,  626.24393999269693722453666850924491882324218750000000,  623.27297138504513895895797759294509887695312500000000,
        620.26500410784376526862615719437599182128906250000000,  617.22058521407439002359751611948013305664062500000000,  614.14023508958405273006064817309379577636718750000000,  611.02444901289334211469395086169242858886718750000000,  607.87369858530144028918584808707237243652343750000000,  604.68843304409938355092890560626983642578125000000000,  601.46908047019815057865343987941741943359375000000000,  598.21604890015976252470863983035087585449218750000000,  594.92972735149191976233851164579391479492187500000000,  591.61048676907091703469632193446159362792968750000000,
        588.25868089969571883557364344596862792968750000000000,  584.87464710102096887567313387989997863769531250000000,  581.45870709044993418501690030097961425781250000000000,  578.01116763898414774303091689944267272949218750000000,  574.53232121451549119228729978203773498535156250000000,  571.02244657857954734936356544494628906250000000000000,  567.48180934020058430178323760628700256347656250000000,  563.91066247008052414457779377698898315429687500000000,  560.30924677808229716902133077383041381835937500000000,  556.67779135665637113561388105154037475585937500000000,
        553.01651399261606911750277504324913024902343750000000,  549.32562154942843335447832942008972167968750000000000,  545.60531032198707634961465373635292053222656250000000,  541.85576636564337604795582592487335205078125000000000,  538.07716580110854920349083840847015380859375000000000,  534.26967509668236289144260808825492858886718750000000,  530.43345132912929784652078524231910705566406250000000,  526.56864242439530698902672156691551208496093750000000,  522.67538737924394354195101186633110046386718750000000,  518.75381646478410857525886967778205871582031250000000,
        514.80405141276764879876282066106796264648437500000000,  510.82620558544016375890350900590419769287109375000000,  506.82038412965141560562187805771827697753906250000000,  502.78668411584993691576528362929821014404296875000000,  498.72519466251731046213535591959953308105468750000000,  494.63599704652756372524891048669815063476562500000000,  490.51916479985533214858151040971279144287109375000000,  486.37476379299397422073525376617908477783203125000000,  482.20285230538735277150408364832401275634765625000000,  478.00348108312408612619037739932537078857421875000000,
        473.77669338408753674229956232011318206787109375000000,  469.52252501070296375473844818770885467529296875000000,  465.24100433037239099576254375278949737548828125000000,  460.93215228363413871193188242614269256591796875000000,  456.59598238003633241532952524721622467041015625000000,  452.23250068165856419000192545354366302490234375000000,  447.84170577416745118171093054115772247314453125000000,  443.42358872523601576176588423550128936767578125000000,  438.97813303010423169325804337859153747558593750000000,  434.50531454400027087103808298707008361816406250000000,
        430.00510140108337964193196967244148254394531250000000,  425.47745391950809334957739338278770446777343750000000,  420.92232449214259304426377639174461364746093750000000,  416.33965746240721728099742904305458068847656250000000,  411.72938898462098222807981073856353759765625000000000,  407.09144686816881630875286646187305450439453125000000,  402.42575040471274405717849731445312500000000000000000,  397.73221017757947493009851314127445220947265625000000,  393.01072785235413675763993524014949798583984375000000,  388.26119594760132258670637384057044982910156250000000,
        383.48349758451092839095508679747581481933593750000000,  378.67750621413676981319440528750419616699218750000000,  373.84308532074652475785114802420139312744140625000000,  368.98008809964085230603814125061035156250000000000000,  364.08835710762036796950269490480422973632812500000000,  359.16772388407815697064506821334362030029296875000000,  354.21800854047455686668399721384048461914062500000000,  349.23901931570071610622107982635498046875000000000000,  344.23055209456032343950937502086162567138671875000000,  339.19238988628512743161991238594055175781250000000000,
        334.12430225964732244392507709562778472900390625000000,  329.02604473083613356720888987183570861816406250000000,  323.89735809981391412293305620551109313964843750000000,  318.73796773035741125568165443837642669677734375000000,  313.54758276841096176212886348366737365722656250000000,  308.32589529271467654325533658266067504882812500000000,  303.07257939091385878782602958381175994873046875000000,  297.78729015349057362982421182096004486083984375000000,  292.46966257685653545195236802101135253906250000000000,  287.11931036579949250153731554746627807617187500000000,
        281.73582462414094607083825394511222839355468750000000,  276.31877242091837842963286675512790679931640625000000,  270.86769521760641055152518674731254577636718750000000,  265.38210713978787680389359593391418457031250000000000,  259.86149307421970888754003681242465972900390625000000,  254.30530656933606792335922364145517349243164062500000,  248.71296751380114642415719572454690933227539062500000,  243.08385956365353308683552313596010208129882812500000,  237.41732728373204963645548559725284576416015625000000,  231.71267296326720952492905780673027038574218750000000,
        225.96915305853332256447174586355686187744140625000000,  220.18597420701249234298302326351404190063476562500000,  214.36228874725233595199824776500463485717773437500000,  208.49718966605263403835124336183071136474609375000000,  202.58970487918230674040387384593486785888671875000000,  196.63879073273645303743251133710145950317382812500000,  190.64332458845257178836618550121784210205078125000000,  184.60209632645921828952850773930549621582031250000000,  178.51379856119245914669591002166271209716796875000000,  172.37701531810736810257367324084043502807617187500000,
        166.19020885694970957047189585864543914794921875000000,  159.95170424703866274285246618092060089111328125000000,  153.65967119468837154272478073835372924804687500000000,  147.31210248323412770332652144134044647216796875000000,  140.90678819871484961367968935519456863403320312500000,  134.44128465947247264011821243911981582641601562500000,  127.91287661656824070632865186780691146850585937500000,  121.31853079975689979619346559047698974609375000000000,  114.65483818237493096603429876267910003662109375000000,  107.91794131963530389839434064924716949462890625000000,
        101.10344160281267988921172218397259712219238281250000,   94.20627897347107193581905448809266090393066406250000,   87.22057305210415734109119512140750885009765625000000,   80.13940885203751918197667691856622695922851562500000,   72.95454059225205867278418736532330513000488281250000,   65.65597031005952999294095207005739212036132812500000,   58.23132722191139976075646700337529182434082031250000,   50.66491408219838632476239581592381000518798828125000,   42.93616210132696409118580049835145473480224609375000,   35.01695029725775043516478035598993301391601562500000,
         26.86650597851807020788328372873365879058837890625000,   18.42033934568260633568570483475923538208007812500000,    9.56071551376638595343138149473816156387329101562500,    0.00000000000000000000000000000000000000000000000000,  701.43726380873704329133033752441406250000000000000000,  703.17949509887705517030553892254829406738281250000000,  704.17700211326405224099289625883102416992187500000000,  704.72381907773058173916069790720939636230468750000000,  704.94246001121791778132319450378417968750000000000000,  704.90111961623210845573339611291885375976562500000000,
        704.64352824316699752671411260962486267089843750000000,  704.20022435761927681596716865897178649902343750000000,  703.59379130764523324614856392145156860351562500000000,  702.84163242885597355780191719532012939453125000000000,  701.95758321046309902158100157976150512695312500000000,  700.95291387076758837793022394180297851562500000000000,  699.83698652044517984904814511537551879882812500000000,  698.61770435995822481345385313034057617187500000000000,  697.30182941474356539401924237608909606933593750000000,  695.89521377658786605024943128228187561035156250000000,
        694.40297200045824865810573101043701171875000000000000,  692.82961231278409286460373550653457641601562500000000,  691.17913827554355066240532323718070983886718750000000,  689.45512880274213785014580935239791870117187500000000,  687.66080201609679534158203750848770141601562500000000,  685.79906683482124662987189367413520812988281250000000,  683.87256511737268738215789198875427246093750000000000,  681.88370642870279425551416352391242980957031250000000,  679.83469698229987443482968956232070922851562500000000,  677.72756393068686975311720743775367736816406250000000,
        675.56417590469288825261173769831657409667968750000000,  673.34626050008432684990111738443374633789062500000000,  671.07541925931684545503230765461921691894531250000000,  668.75314058207538892020238563418388366699218750000000,  666.38081091100298181117977946996688842773437500000000,  663.95972447161034324381034821271896362304687500000000,  661.49109179278661940770689398050308227539062500000000,  658.97604719298078634892590343952178955078125000000000,  656.41565538433997062384150922298431396484375000000000,  653.81091732088520984689239412546157836914062500000000,
        651.16277539573172816744772717356681823730468750000000,  648.47211807528026383806718513369560241699218750000000,  645.73978404439208134135697036981582641601562500000000,  642.96656592514239036972867324948310852050781250000000,  640.15321362234737989638233557343482971191406250000000,  637.30043734125035825854865834116935729980468750000000,  634.40891031627018037397647276520729064941406250000000,  631.47927128426658782700542360544204711914062500000000,  628.51212673121824536792701110243797302246093750000000,  625.50805293734390488680219277739524841308593750000000,
        622.46759784243010926729766651988029479980468750000000,  619.39128275034738635440589860081672668457031250000000,  616.27960388935514401964610442519187927246093750000000,  613.13303384276446195144671946763992309570312500000000,  609.95202286276924041885649785399436950683593750000000,  606.73700007875163464632350951433181762695312500000000,  603.48837461005018667492549866437911987304687500000000,  600.20653659205106578156119212508201599121093750000000,  596.89185812346806869754800572991371154785156250000000,  593.54469414181676256703212857246398925781250000000000,
        590.16538323332849813596112653613090515136718750000000,  586.75424838288870432734256610274314880371093750000000,  583.31159766900066188100026920437812805175781250000000,  579.83772490825435852457303553819656372070312500000000,  576.33291025333380730444332584738731384277343750000000,  572.79742074818307173700304701924324035644531250000000,  569.23151084359631113329669460654258728027343750000000,  565.63542287617917736497474834322929382324218750000000,  562.00938751333683285338338464498519897460937500000000,  558.35362416669715912576066330075263977050781250000000,
        554.66834137614000610483344644308090209960937500000000,  550.95373716640403927158331498503684997558593750000000,  547.20999937805208901409059762954711914062500000000000,  543.43730597441162899485789239406585693359375000000000,  539.63582532595523844065610319375991821289062500000000,  535.80571647344322627759538590908050537109375000000000,  531.94712937103338390443241223692893981933593750000000,  528.06020511043948317819740623235702514648437500000000,  524.14507612712202444527065381407737731933593750000000,  520.20186638939458134700544178485870361328125000000000,
        516.23069157124132289027329534292221069335937500000000,  512.23165920955932506331009790301322937011718750000000,  508.20486884646282987887389026582241058349609375000000,  504.15041215721430489793419837951660156250000000000000,  500.06837306428235478961141780018806457519531250000000,  495.95882783796099602113827131688594818115234375000000,  491.82184518392762129224138334393501281738281250000000,  487.65748631805860213717096485197544097900390625000000,  483.46580502876730633943225257098674774169921875000000,  479.24684772707723823259584605693817138671875000000000,
        475.00065348459094138888758607208728790283203125000000,  470.72725405946641785703832283616065979003906250000000,  466.42667391046262537201982922852039337158203125000000,  462.09893019906661493223509751260280609130859375000000,  457.74403277966683845079387538135051727294921875000000,  453.36198417768639501446159556508064270019531250000000,  448.95277955554001891869120299816131591796875000000000,  444.51640666622802200436126440763473510742187500000000,  440.05284579432480995819787494838237762451171875000000,  435.56206968406860369213973172008991241455078125000000,
        431.04404345419533228778163902461528778076171875000000,  426.49872449910503746650647372007369995117187500000000,  421.92606237587926898413570597767829895019531250000000,  417.32599867660405834612902253866195678710937500000000,  412.69846688537603540680720470845699310302734375000000,  408.04339221929262748744804412126541137695312500000000,  403.36069145264099233827437274158000946044921875000000,  398.65027272340898889524396508932113647460937500000000,  393.91203532113956953253364190459251403808593750000000,  389.14586945504163395526120439171791076660156250000000,
        384.35165600114828521327581256628036499023437500000000,  379.52926622718234739295439794659614562988281250000000,  374.67856149364348539165803231298923492431640625000000,  369.79939292946613704771152697503566741943359375000000,  364.89160108042364072389318607747554779052734375000000,  359.95501552824964619503589347004890441894531250000000,  354.98945447822910637114546261727809906005859375000000,  349.99472431276103634445462375879287719726562500000000,  344.97061910811743246085825376212596893310546875000000,  339.91692011131004846902214922010898590087890625000000,
        334.83339517362543347189784981310367584228515625000000,  329.71979813698993666548631154000759124755859375000000,  324.57586816887823033539461903274059295654296875000000,  319.40132904096770971591467969119548797607421875000000,  314.19588834616109807029715739190578460693359375000000,  308.95923664793957641450106166303157806396484375000000,  303.69104655524841973601724021136760711669921875000000,  298.39097171525406793080037459731101989746093750000000,  293.05864571530986495417892001569271087646484375000000,  287.69368088431957630746182985603809356689453125000000,
        282.29566698235487365309381857514381408691406250000000,  276.86416976583842597392504103481769561767578125000000,  271.39872941380377824316383339464664459228515625000000,  265.89885879864283424467430450022220611572265625000000,  260.36404158228299365873681381344795227050781250000000,  254.79373011583501806853746529668569564819335937500000,  249.18734311732313813081418629735708236694335937500000,  243.54426309803685057886468712240457534790039062500000,  237.86383350319403007233631797134876251220703125000000,  232.14535552679700458611478097736835479736328125000000,
        226.38808455357678894870332442224025726318359375000000,  220.59122617247422226682829204946756362915039062500000,  214.75393169583972508007718715816736221313476562500000,  208.87529310598512211072375066578388214111328125000000,  202.95433733528867037421150598675012588500976562500000,  196.99001976696112592435383703559637069702148437500000,  190.98121681979202435286424588412046432495117187500000,  184.92671745034806463081622496247291564941406250000000,  178.82521336835799274922464974224567413330078125000000,  172.67528771291102884788415394723415374755859375000000,
        166.47540187523253507606568746268749237060546875000000,  160.22388007348766336690459866076707839965820312500000,  153.91889117973988732046564109623432159423828125000000,  147.55842715952942967305716592818498611450195312500000,  141.14027729712270797790552023798227310180664062500000,  134.66199712469315841190109495073556900024414062500000,  128.12087062233237588770862203091382980346679687500000,  121.51386376363224428587272996082901954650878906250000,  114.83756678019398123069549910724163055419921875000000,  108.08812149955546999535727081820368766784667968750000,
        101.26112859902097795838926685974001884460449218750000,   94.35152731954445926021435298025608062744140625000000,   87.35343659403605443003471009433269500732421875000000,   80.25994076094146123523387359455227851867675781250000,   73.06279337675189822221000213176012039184570312500000,   65.75199582837615253083640709519386291503906250000000,   58.31517669365525335933853057213127613067626953125000,   50.73663809987642991927714319899678230285644531250000,   42.99581064157309384654581663198769092559814453125000,   35.06457273179467648560603265650570392608642578125000,
         26.90215108485879014210695459041744470596313476562500,   18.44405531757917060531326569616794586181640625000000,    9.57254997141338925814579852158203721046447753906250,    0.00000000000000000000000000000000000000000000000000,  706.57306224578735509567195549607276916503906250000000,  708.31632317743083149252925068140029907226562500000000,  709.31496759882338665192946791648864746093750000000000,  709.86301637439009937224909663200378417968750000000000,  710.08297442944422073196619749069213867187500000000000,  710.04302996571243511425564065575599670410156250000000,
        709.78690851255487359594553709030151367187500000000000,  709.34514485988665910554118454456329345703125000000000,  708.74031949429763699299655854701995849609375000000000,  707.98983348912008750630775466561317443847656250000000,  707.10752052528710009937640279531478881835937500000000,  706.10464936551022674393607303500175476074218750000000,  704.99058094485940273443702608346939086914062500000000,  703.77321751472516098147025331854820251464843750000000,  702.45932033788164972065715119242668151855468750000000,  701.05474089918766367190983146429061889648437500000000,
        699.56459327859101904323324561119079589843750000000000,  697.99338534064179384586168453097343444824218750000000,  696.34512038361140184861142188310623168945312500000000,  694.62337714390503151662414893507957458496093750000000,  692.83137364192475615709554404020309448242187500000000,  690.97201876379483564960537478327751159667968750000000,  689.04795439644476573448628187179565429687500000000000,  687.06159018931680293462704867124557495117187500000000,  685.01513249177276065893238410353660583496093750000000,  682.91060863969846650434192270040512084960937500000000,
        680.74988749149383693293202668428421020507812500000000,  678.53469691193402013595914468169212341308593750000000,  676.26663875157657912495778873562812805175781250000000,  673.94720175531904260424198582768440246582031250000000,  671.57777274645491161209065467119216918945312500000000,  669.15964636517196595377754420042037963867187500000000,  666.69403358788088098663138225674629211425781250000000,  664.18206921241460349847329780459403991699218750000000,  661.62481846135642626904882490634918212890625000000000,  659.02328282956136717984918504953384399414062500000000,
        656.37840528085712321626488119363784790039062500000000,  653.69107488184010890108766034245491027832031250000000,  650.96213094676409127714578062295913696289062500000000,  648.19236675610909514944069087505340576171875000000000,  645.38253290201453182817203924059867858886718750000000,  642.53334030595749482017708942294120788574218750000000,  639.64546294757144551113015040755271911621093750000000,  636.71954033805684503022348508238792419433593750000000,  633.75617976707326306495815515518188476562500000000000,  630.75595834814168938464717939496040344238281250000000,
        627.71942488431750462041236460208892822265625000000000,  624.64710157311299099092138931155204772949218750000000,  621.53948556726902552327373996376991271972656250000000,  618.39705040594321872049476951360702514648437500000000,  615.22024732912484523694729432463645935058593750000000,  612.00950648658101727050961926579475402832031250000000,  608.76523805132342204160522669553756713867187500000000,  605.48783324645557968324283137917518615722656250000000,  602.17766529326854652026668190956115722656250000000000,  598.83509028758703607309143990278244018554687500000000,
        595.46044801061816542642191052436828613281250000000000,  592.05406267988519175560213625431060791015625000000000,  588.61624364524925567820901051163673400878906250000000,  585.14728603450214450276689603924751281738281250000000,  581.64747135256448018481023609638214111328125000000000,  578.11706803791128095326712355017662048339843750000000,  574.55633197949589430209016427397727966308593750000000,  570.96550699711781362566398456692695617675781250000000,  567.34482528789919797418406233191490173339843750000000,  563.69450784127479892049450427293777465820312500000000,
        560.01476482467717232793802395462989807128906250000000,  556.30579594188691316958284005522727966308593750000000,  552.56779076583927690080599859356880187988281250000000,  548.80092904750301840977044776082038879394531250000000,  545.00538100230471627583028748631477355957031250000000,  541.18130757542849096353165805339813232421875000000000,  537.32886068719551531103206798434257507324218750000000,  533.44818345962073635746492072939872741699218750000000,  529.53941042512940384767716750502586364746093750000000,  525.60266771832971244293730705976486206054687500000000,
        521.63807325164316353038884699344635009765625000000000,  517.64573687551637704018503427505493164062500000000000,  513.62576052386054925591452047228813171386718750000000,  509.57823834529460782505339011549949645996093750000000,  505.50325682070325683525879867374897003173828125000000,  501.40089486755636016823700629174709320068359375000000,  497.27122393138125744371791370213031768798828125000000,  493.11430806471997811968321911990642547607421875000000,  488.93020399385341079323552548885345458984375000000000,  484.71896117352008559464593417942523956298828125000000,
        480.48062182981021805971977300941944122314453125000000,  476.21522099136569750044145621359348297119140625000000,  471.92278650896872704834095202386379241943359375000000,  467.60333906355634780993568710982799530029296875000000,  463.25689216264964898073230870068073272705078125000000,  458.88345212513922888319939374923706054687500000000000,  454.48301805432191713407519273459911346435546875000000,  450.05558179903295012991293333470821380615234375000000,  445.60112790266981619424768723547458648681640625000000,  441.11963353985015601210761815309524536132812500000000,
        436.61106844039341012830846011638641357421875000000000,  432.07539480025832290266407653689384460449218750000000,  427.51256717900906778595526702702045440673828125000000,  422.92253238331852571718627586960792541503906250000000,  418.30522933595131007677991874516010284423828125000000,  413.66058892959398463062825612723827362060546875000000,  408.98853386482369387522339820861816406250000000000000,  404.28897847142167165657156147062778472900390625000000,  399.56182851214498441549949347972869873046875000000000,  394.80698096797226526177837513387203216552734375000000,
        390.02432380372681564040249213576316833496093750000000,  385.21373571286324022366898134350776672363281250000000,  380.37508584007031231521978043019771575927734375000000,  375.50823348019747527359868399798870086669921875000000,  370.61302775185095015331171452999114990234375000000000,  365.68930724382681773931835778057575225830078125000000,  360.73689963234971855854382738471031188964843750000000,  355.75562126686293140664929524064064025878906250000000,  350.74527672186923155095428228378295898437500000000000,  345.70565831204169171542162075638771057128906250000000,
        340.63654556751305335637880489230155944824218750000000,  335.53770466589998022755025885999202728271484375000000,  330.40888781722105704830028116703033447265625000000000,  325.24983259741833307998604141175746917724609375000000,  320.06026122568204073104425333440303802490234375000000,  314.83987978019820275221718475222587585449218750000000,  309.58837734627843474299879744648933410644531250000000,  304.30542509007159424072597175836563110351562500000000,  298.99067525019376034833840094506740570068359375000000,  293.64376003861144681650330312550067901611328125000000,
        288.26429044096562392951454967260360717773437500000000,  282.85185490518972528661834076046943664550781250000000,  277.40601790573219886937295086681842803955078125000000,  271.92631836889296437220764346420764923095703125000000,  266.41226794268226285566925071179866790771484375000000,  260.86334909214309618619154207408428192138671875000000,  255.27901299817631297628395259380340576171875000000000,  249.65867723447868797848059330135583877563476562500000,  244.00172319313185198552673682570457458496093750000000,  238.30749322453067406968330033123493194580078125000000,
        232.57528745153143745483248494565486907958984375000000,  226.80436021071341201604809612035751342773437500000000,  220.99391606520228492627211380749940872192382812500000,  215.14310532323534630450012627989053726196289062500000,  209.25101898410136413986037950962781906127929687500000,  203.31668301765500928013352677226066589355468750000000,  197.33905186451323743312968872487545013427734375000000,  191.31700102025109799797064624726772308349609375000000,  185.24931853706908668755204416811466217041015625000000,  179.13469523866484678364940918982028961181640625000000,
        172.97171339593600691841857042163610458374023437500000,  166.75883354927668733580503612756729125976562500000000,  160.49437908291841381469566840678453445434570312500000,  154.17651805144089394161710515618324279785156250000000,  147.80324161891772405397205147892236709594726562500000,  141.37233828374732524935097899287939071655273437500000,  134.88136280742864414605719503015279769897460937500000,  128.32759841418049973071902059018611907958984375000000,  121.70801033614459640830318676307797431945800781250000,  115.01918807752727502702327910810708999633789062500000,
        108.25727275217141709617862943559885025024414062500000,  101.41786433704187686544173629954457283020019531250000,   94.49590138577534048636152874678373336791992187500000,   87.48550215666371343559148954227566719055175781250000,   80.37975032592034096978750312700867652893066406250000,   73.17039879988662676169042242690920829772949218750000,   65.84744832710975970258004963397979736328125000000000,   58.39852685857216130216329474933445453643798828125000,   50.80793590592884356738068163394927978515625000000000,   43.05510545903898389497044263407588005065917968750000,
         35.11191333697544081360319978557527065277099609375000,   26.93758567052428176680223259609192609786987304687500,   18.46763150316755641711097268853336572647094726562500,    9.58431481299297516329716017935425043106079101562500,    0.00000000000000000000000000000000000000000000000000,  711.71472580228987681039143353700637817382812500000000,  713.45900930433822395571041852235794067382812500000000,  714.45878309828651708812685683369636535644531250000000,  715.00805479493703842308605089783668518066406250000000,  715.22932029604521630972158163785934448242187500000000,
        715.19076135035084007540717720985412597656250000000000,  714.93609869982049076497787609696388244628906250000000,  714.49586348280058700765948742628097534179687500000000,  713.89263334180714082322083413600921630859375000000000,  713.14380710035368338139960542321205139160156250000000,  712.26321663981127585429931059479713439941406250000000,  711.26212927309495626104762777686119079589843750000000,  710.14990476311879774584667757153511047363281250000000,  708.93444441372980691085103899240493774414062500000000,  707.62250872495894782332470640540122985839843750000000,
        706.21994857328479611169314011931419372558593750000000,  704.73187756099321177316596731543540954589843750000000,  703.16280318706071739143226295709609985351562500000000,  701.51672848141822669276734814047813415527343750000000,  699.79723199738316452567232772707939147949218750000000,  698.00753164777052006684243679046630859375000000000000,  696.15053627861266249965410679578781127929687500000000,  694.22888779762297417619265615940093994140625000000000,  692.24499593039422506990376859903335571289062500000000,  690.20106715319639079098124057054519653320312500000000,
        688.09912897570916356926318258047103881835937500000000,  685.94105047374728201248217374086380004882812500000000,  683.72855977036044805572601035237312316894531250000000,  681.46325901290651927411090582609176635742187500000000,  679.14663727963034034473821520805358886718750000000000,  676.78008176204684787080623209476470947265625000000000,  674.36488750202977371372980996966361999511718750000000,  671.90226590995428068708861246705055236816406250000000,  669.39335224890817244158824905753135681152343750000000,  666.83921223720437865267740562558174133300781250000000,
        664.24084789523772087704855948686599731445312500000000,  661.59920274165733644622378051280975341796875000000000,  658.91516642675446746579837054014205932617187500000000,  656.18957887705118992016650736331939697265625000000000,  653.42323401367013957496965304017066955566406250000000,  650.61688309765668236650526523590087890625000000000000,  647.77123774763208530202973634004592895507812500000000,  644.88697266866188329004216939210891723632812500000000,  641.96472812579122546594589948654174804687500000000000,  639.00511219112831895472481846809387207031250000000000,
        636.00870278950435476872371509671211242675781250000000,  632.97604956446537016745423898100852966308593750000000,  629.90767558357413236080901697278022766113281250000000,  626.80407889961963974201353266835212707519531250000000,  623.66573398229968461237149313092231750488281250000000,  620.49309303318739239330170676112174987792968750000000,  617.28658719528277742938371375203132629394531250000000,  614.04662766714261579181766137480735778808593750000000,  610.77360673044586292235180735588073730468750000000000,  607.46789869886129054066259413957595825195312500000000,
        604.12986079522613636072492226958274841308593750000000,  600.75983396328206254111137241125106811523437500000000,  597.35814361955704043793957680463790893554687500000000,  593.92510035039356353081529960036277770996093750000000,  590.46100055861325017758645117282867431640625000000000,  586.96612706384678403992438688874244689941406250000000,  583.44074966016228245280217379331588745117187500000000,  579.88512563425683765672147274017333984375000000000000,  576.29950024716617917874827980995178222656250000000000,  572.68410718215693577803904190659523010253906250000000,
        569.03916896120961155247641727328300476074218750000000,  565.36489733227961096417857334017753601074218750000000,  561.66149362930900679202750325202941894531250000000000,  557.92914910678132400789763778448104858398437500000000,  554.16804525044835827429778873920440673828125000000000,  550.37835406569683982525020837783813476562500000000000,  546.56023834489792534441221505403518676757812500000000,  542.71385191494664468336850404739379882812500000000000,  538.83933986609383737231837585568428039550781250000000,  534.93683876306420188484480604529380798339843750000000,
        531.00647683935824261425295844674110412597656250000000,  527.04837417555233969324035570025444030761718750000000,  523.06264286232715221558464691042900085449218750000000,  519.04938714887884998461231589317321777343750000000000,  515.00870357730070736579364165663719177246093750000000,  510.94068110345540389971574768424034118652343750000000,  506.84540120479800862085539847612380981445312500000000,  502.72293797554971206409391015768051147460937500000000,  498.57335820957217720206244848668575286865234375000000,  494.39672147123553713754517957568168640136718750000000,
        490.19308015452480731255491264164447784423828125000000,  485.96247953058212942778482101857662200927734375000000,  481.70495778383292417856864631175994873046875000000000,  477.42054603680026048095896840095520019531250000000000,  473.10926836366434145020321011543273925781250000000000,  468.77114179258006743111764080822467803955078125000000,  464.40617629671999111451441422104835510253906250000000,  460.01437477396490294268005527555942535400390625000000,  455.59573301511670706531731411814689636230468750000000,  451.15023966046209125124732963740825653076171875000000,
        446.67787614446643829069216735661029815673828125000000,  442.17861662832524416444357484579086303710937500000000,  437.65242792005005867395084351301193237304687500000000,  433.09926938170713128783972933888435363769531250000000,  428.51909282336981732441927306354045867919921875000000,  423.91184238328355604608077555894851684570312500000000,  419.27745439367311064415844157338142395019531250000000,  414.61585723155263849548646248877048492431640625000000,  409.92697115381940875522559508681297302246093750000000,  405.21070811582893611557665280997753143310546875000000,
        400.46697157255869115033419802784919738769531250000000,  395.69565626136699165726895444095134735107421875000000,  390.89664796524488110662787221372127532958984375000000,  386.06982325534102074016118422150611877441406250000000,  381.21504921140439137161592952907085418701171875000000,  376.33218311864942506872466765344142913818359375000000,  371.42107213938203358338796533644199371337890625000000,  366.48155295754986582323908805847167968750000000000000,  361.51345139418151575227966532111167907714843750000000,  356.51658199145424532616743817925453186035156250000000,
        351.49074756288797516390332020819187164306640625000000,  346.43573870687845328575349412858486175537109375000000,  341.35133328047709255770314484834671020507812500000000,  336.23729582996770659519825130701065063476562500000000,  331.09337697439855219272430986166000366210937500000000,  325.91931273777447586326161399483680725097656250000000,  320.71482382510691877541830763220787048339843750000000,  315.47961483693870832212269306182861328125000000000000,  310.21337341629987349733710289001464843750000000000000,  304.91576932129282795358449220657348632812500000000000,
        299.58645341564016462143626995384693145751953125000000,  294.22505656852854372118599712848663330078125000000000,  288.83118845393352103201323188841342926025390625000000,  283.40443623827746932875015772879123687744140625000000,  277.94436314372882179668522439897060394287109375000000,  272.45050687265018041216535493731498718261718750000000,  266.92237787660286585378344170749187469482421875000000,  261.35945745084649161071865819394588470458984375000000,  255.76119563237270426725444849580526351928710937500000,  250.12700887608031052877777256071567535400390625000000,
        244.45627747962984699370281305164098739624023437500000,  238.74834272266411971941124647855758666992187500000000,  233.00250368027414538119046483188867568969726562500000,  227.21801366360301699387491680681705474853515625000000,  221.39407623203507569087378215044736862182617187500000,  215.52984071114934749857638962566852569580078125000000,  209.62439713806892882530519273132085800170898437500000,  203.67677054040581197114079259335994720458984375000000,  197.68591443590725020840181969106197357177734375000000,  191.65070341612036486367287579923868179321289062500000,
        185.56992464754677030214224942028522491455078125000000,  179.44226808601865741366054862737655639648437500000000,  173.26631515192261190350109245628118515014648437500000,  167.04052555203361407620832324028015136718750000000000,  160.76322185340833925693004857748746871948242187500000,  154.43257130946332722487568389624357223510742187500000,  148.04656429870217948518984485417604446411132812500000,  141.60298854914171329255623277276754379272460937500000,  135.09939806669595441235287580639123916625976562500000,  128.53307533441633836446271743625402450561523437500000,
        121.90098485732762867428391473367810249328613281250000,  115.19971542621435389719408703967928886413574218750000,  108.42540745484862441117002163082361221313476562500000,  101.57366023314207836847344879060983657836914062500000,   94.63941164037278497289662482216954231262207031250000,   87.61677927285565203874284634366631507873535156250000,   80.49884615690527311926416587084531784057617187500000,   73.27736456074842408270342275500297546386718750000000,   65.94233460631660648232355015352368354797363281250000,   58.48138362919895882896526018157601356506347656250000,
         50.87881253661137748167675454169511795043945312500000,   43.11405072466776999817739124409854412078857421875000,   35.15897542913707951584001420997083187103271484375000,   26.97281220769766818534662888851016759872436523437500,   18.49106954068216523978662735316902399063110351562500,    9.59601085275616760839056951226666569709777832031250,    0.00000000000000000000000000000000000000000000000000,  716.86222027910343967960216104984283447265625000000000,  718.60751937088548402243759483098983764648437500000000,  719.60841460785024992219405248761177062988281250000000,
        720.15890045350670334300957620143890380859375000000000,  720.38146385522384207433788105845451354980468750000000,  720.34428015596790828567463904619216918945312500000000,  720.09106534357454165728995576500892639160156250000000,  719.65234692869489663280546665191650390625000000000000,  719.05069972701824099203804507851600646972656250000000,  718.30352032462531042256159707903861999511718750000000,  717.42463881202309039508691057562828063964843750000000,  716.42532105814711940183769911527633666992187500000000,  715.31492565726478005672106519341468811035156250000000,
        714.10135296729674792004516348242759704589843750000000,  712.79136272556638687092345207929611206054687500000000,  711.39080519885817466274602338671684265136718750000000,  709.90479350931013868830632418394088745117187500000000,  708.33783478681004908139584586024284362792968750000000,  706.69393178850839376536896452307701110839843750000000,  704.97666287935862783342599868774414062500000000000000,  703.18924585854358610959025099873542785644531250000000,  701.33458952524426877062069252133369445800781250000000,  699.41533580052714569319505244493484497070312500000000,
        697.43389447806737280188826844096183776855468750000000,  695.39247215236582633224315941333770751953125000000000,  693.29309649762933531746966764330863952636718750000000,  691.13763679724706889828667044639587402343750000000000,  688.92782142214309715200215578079223632812500000000000,  686.66525280552070853445911779999732971191406250000000,  684.35142034746934314171085134148597717285156250000000,  681.98771159567695576697587966918945312500000000000000,  679.57542198110695608193054795265197753906250000000000,  677.11576333495440849219448864459991455078125000000000,
        674.60987137186361906060483306646347045898437500000000,  672.05881229161821011075517162680625915527343750000000,  669.46358862532576949888607487082481384277343750000000,  666.82514443105173995718359947204589843750000000000000,  664.14436992678690785396611317992210388183593750000000,  661.42210563472531248407904058694839477539062500000000,  658.65914609941842172702308744192123413085937500000000,  655.85624323297179216751828789710998535156250000000000,  653.01410933265503899747272953391075134277343750000000,  650.13341980980294465553015470504760742187500000000000,
        647.21481566345391911454498767852783203125000000000000,  644.25890572760454233502969145774841308593750000000000,  641.26626871710277555393986403942108154296875000000000,  638.23745509393586416990729048848152160644531250000000,  635.17298877288510539074195548892021179199218750000000,  632.07336868314519051637034863233566284179687500000000,  628.93907020047231526405084878206253051757812500000000,  625.77054646267117732350016012787818908691406250000000,  622.56822957971996856940677389502525329589843750000000,  619.33253174852802658278960734605789184570312500000000,
        616.06384628118155433185165748000144958496093750000000,  612.76254855454783410095842555165290832519531250000000,  609.42899688824252280028304085135459899902343750000000,  606.06353335721121311507886275649070739746093750000000,  602.66648454451296856859698891639709472656250000000000,  599.23816223930805335839977487921714782714843750000000,  595.77886408454185129812685772776603698730468750000000,  592.28887417835528594878269359469413757324218750000000,  588.76846363285653751518111675977706909179687500000000,  585.21789109352221203153021633625030517578125000000000,
        581.63740322218518485897220671176910400390625000000000,  578.02723514627416534494841471314430236816406250000000,  574.38761087672128269332461059093475341796875000000000,  570.71874369672195825842209160327911376953125000000000,  567.02083652332851215760456398129463195800781250000000,  563.29408224367227830953197553753852844238281250000000,  559.53866402744517927203560248017311096191406250000000,  555.75475561711721184110501781105995178222656250000000,  551.94252159723419026704505085945129394531250000000000,  548.10211764401174150407314300537109375000000000000000,
        544.23369075633047486917348578572273254394531250000000,  540.33737946913311134267132729291915893554687500000000,  536.41331405013147559657227247953414916992187500000000,  532.46161668064041805337183177471160888671875000000000,  528.48240162127729036001255735754966735839843750000000,  524.47577536319374758022604510188102722167968750000000,  520.44183676543207184295170009136199951171875000000000,  516.38067717893818553420715034008026123046875000000000,  512.29238055770144910638919100165367126464843750000000,  508.17702355743398356935358606278896331787109375000000,
        504.03467562214962072175694629549980163574218750000000,  499.86539905894994717527879402041435241699218750000000,  495.66924910127744396959315054118633270263671875000000,  491.44627396084666770548210479319095611572265625000000,  487.19651486842064969096099957823753356933593750000000,  482.92000610355302114840014837682247161865234375000000,  478.61677501337481999144074507057666778564453125000000,  474.28684202045832307703676633536815643310546875000000,  469.93022061975034375791437923908233642578125000000000,  465.54691736452110717436880804598331451416015625000000,
        461.13693184123235369042959064245223999023437500000000,  456.70025663318313036143081262707710266113281250000000,  452.23687727274466396920615807175636291503906250000000,  447.74677218194869965373072773218154907226562500000000,  443.22991260114451961271697655320167541503906250000000,  438.68626250538682143087498843669891357421875000000000,  434.11577850816189538818434812128543853759765625000000,  429.51840975200207140005659312009811401367187500000000,  424.89409778547644691570894792675971984863281250000000,  420.24277642597843396288226358592510223388671875000000,
        415.56437160766176930337678641080856323242187500000000,  410.85880121379653928670450113713741302490234375000000,  406.12597489273730388958938419818878173828125000000000,  401.36579385660019170245504938066005706787109375000000,  396.57815066165170492240577004849910736083984375000000,  391.76292896929913922576815821230411529541015625000000,  386.92000328645599438459612429141998291015625000000000,  382.04923868392415897687897086143493652343750000000000,  377.15049049128901970107108354568481445312500000000000,  372.22360396666368842488736845552921295166015625000000,
        367.26841393943993807624792680144309997558593750000000,  362.28474442400573707345756702125072479248046875000000,  357.27240820216655947660910896956920623779296875000000,  352.23120637176202762930188328027725219726562500000000,  347.16092785869000181264709681272506713867187500000000,  342.06134888923969583629514090716838836669921875000000,  336.93223241928234301667544059455394744873046875000000,  331.77332751647469422096037305891513824462890625000000,  326.58436869117633705172920599579811096191406250000000,  321.36507517127751043517491780221462249755859375000000,
        316.11515011555030696399626322090625762939453125000000,  310.83427975947824961622245609760284423828125000000000,  305.52213248675968770839972421526908874511718750000000,  300.17835781881643697488470934331417083740234375000000,  294.80258531363881502329604700207710266113281250000000,  289.39442336415078216305118985474109649658203125000000,  283.95345788494466887641465291380882263183593750000000,  278.47925087469241134385811164975166320800781250000000,  272.97133883973947376944124698638916015625000000000000,  267.42923106228732876843423582613468170166015625000000,
        261.85240769410108896408928558230400085449218750000000,  256.24031765378111913378234021365642547607421875000000,  250.59237630220380310674954671412706375122070312500000,  244.90796286666849823632219340652227401733398437500000,  239.18641757943618131321272812783718109130859375000000,  233.42703849053759768139570951461791992187500000000000,  227.62907790774343652628886047750711441040039062500000,  221.79173840814152640632528346031904220581054687500000,  215.91416835550032260471198242157697677612304687500000,  209.99545684504869313968811184167861938476562500000000,
        204.03462798187086946199997328221797943115234375000000,  198.03063438002186558151151984930038452148437500000000,  191.98234974567949961965496186167001724243164062500000,  185.88856037780337260301166679710149765014648437500000,  179.74795538203241562769107986241579055786132812500000,  173.55911534544605956398299895226955413818359375000000,  167.32049915795090555548085831105709075927734375000000,  161.03042858574147544459265191107988357543945312500000,  154.68707009696001364318362902849912643432617187500000,  148.28841330001915821412694640457630157470703125000000,
        141.83224516763627320870000403374433517456054687500000,  135.31611896483858004103240091353654861450195312500000,  128.73731644783632077633228618651628494262695312500000,  122.09280140850360396598262013867497444152832031250000,  115.37916193782064055994851514697074890136718750000000,  108.59253776276827352376130875200033187866210937500000,  101.72852749915716685791267082095146179199218750000000,   94.78206836454170058914314722642302513122558593750000,   87.74727730558716132236440898850560188293457031250000,   80.61723671073849573076586239039897918701171875000000,
         73.38369822184783686225273413583636283874511718750000,   66.03666134569050427671754732728004455566406250000000,   58.56375281364979201725873281247913837432861328125000,   50.94927293942582480212877271696925163269042968750000,   43.17265053605443370088323717936873435974121093750000,   35.20576226641909300951738259755074977874755859375000,   27.00783312526749213589027931448072195053100585937500,   18.51437103972542885799157375004142522811889648437500,    9.60763889075128574290829419624060392379760742187500,    0.00000000000000000000000000000000000000000000000000,
        722.01551187360121275560231879353523254394531250000000,  723.76181966319268212828319519758224487304687500000000,  724.76382851656012462626677006483078002929687500000000,  725.31551985482315103581640869379043579101562500000000,  725.53937173925021397735690698027610778808593750000000,  725.50355315367846742446999996900558471679687500000000,  725.25177536470459926931653171777725219726562500000000,  724.81456227891828802967211231589317321777343750000000,  724.21448590228578723326791077852249145507812500000000,  723.46894059576675317657645791769027709960937500000000,
        722.59175466767987927596550434827804565429687500000000,  721.59419254881970573478611186146736145019531250000000,  720.48561166837157543341163545846939086914062500000000,  719.27391144003308909304905682802200317382812500000000,  717.96585083856473374908091500401496887207031250000000,  716.56727951987465985439484938979148864746093750000000,  715.08331012362123146886005997657775878906250000000000,  713.51844940725675314752152189612388610839843750000000,  711.87669985090030877472599968314170837402343750000000,  710.16163962606833592872135341167449951171875000000000,
        708.37648641248551939497701823711395263671875000000000,  706.52414895595597954525146633386611938476562500000000,  704.60726918371676674723858013749122619628906250000000,  702.62825694972070778021588921546936035156250000000000,  700.58931895829880431847414001822471618652343750000000,  698.49248303920956004731124266982078552246093750000000,  696.33961867388347855012398213148117065429687500000000,  694.13245447103747665096307173371315002441406250000000,  691.87259313910180935636162757873535156250000000000000,  689.56152438885771971399663016200065612792968750000000,
        687.20063611248315282864496111869812011718750000000000,  684.79122411781747814529808238148689270019531250000000,  682.33450064412807023472851142287254333496093750000000,  679.83160184433211270516039803624153137207031250000000,  677.28359438585675889044068753719329833984375000000000,  674.69148129614529807440703734755516052246093750000000,  672.05620715774227846850408241152763366699218750000000,  669.37866274083125972538255155086517333984375000000000,  666.65968914718837368127424269914627075195312500000000,  663.90008152810833053081296384334564208984375000000000,
        661.10059242946113045036327093839645385742187500000000,  658.26193480924177947599673643708229064941406250000000,  655.38478476648651849245652556419372558593750000000000,  652.46978401499575284105958417057991027832031250000000,  649.51754213073911614628741517663002014160156250000000,  646.52863859796173073846148326992988586425781250000000,  643.50362467574211677856510505080223083496093750000000,  640.44302510397710648248903453350067138671875000000000,  637.34733966538431104709161445498466491699218750000000,  634.21704461808883479534415528178215026855468750000000,
        631.05259401159980825468664988875389099121093750000000,  627.85442089747891714068828150629997253417968750000000,  624.62293844469195391866378486156463623046875000000000,  621.35854096850164296483853831887245178222656250000000,  618.06160488076943693158682435750961303710937500000000,  614.73248956867462311493000015616416931152343750000000,  611.37153820809965054650092497467994689941406250000000,  607.97907851727188699442194774746894836425781250000000,  604.55542345566618678276427090167999267578125000000000,  601.10087187265867214591708034276962280273437500000000,
        597.61570910996726979647064581513404846191406250000000,  594.10020756151322984806029126048088073730468750000000,  590.55462719397530690912390127778053283691406250000000,  586.97921603099428011773852631449699401855468750000000,  583.37421060369888436980545520782470703125000000000000,  579.73983636996842960797948762774467468261718750000000,  576.07630810462217141321161761879920959472656250000000,  572.38383026251972296449821442365646362304687500000000,  568.66259731636989727121545001864433288574218750000000,  564.91279407088393327285302802920341491699218750000000,
        561.13459595475467267533531412482261657714843750000000,  557.32816929180830811674240976572036743164062500000000,  553.49367155255163197580259293317794799804687500000000,  549.63125158722505148034542798995971679687500000000000,  545.74104984136727125587640330195426940917968750000000,  541.82319855480568548955488950014114379882812500000000,  537.87782194489716403040802106261253356933593750000000,  533.90503637476535914174746721982955932617187500000000,  529.90495050720676317723700776696205139160156250000000,  525.87766544486976272310130298137664794921875000000000,
        521.82327485724522375676315277814865112304687500000000,  517.74186509494938945863395929336547851562500000000000,  513.63351529172325626859674230217933654785156250000000,  509.49829745451768303610151633620262145996093750000000,  505.33627654198727441325900144875049591064453125000000,  501.14751053166327210419694893062114715576171875000000,  496.93205047603305501979775726795196533203125000000000,  492.68994054770746515714563429355621337890625000000000,  488.42121807381494136279798112809658050537109375000000,  484.12591355971744633279740810394287109375000000000000,
        479.80405070210315443546278402209281921386718750000000,  475.45564639146738272756920196115970611572265625000000,  471.08071070395311608081101439893245697021484375000000,  466.67924688247910580685129389166831970214843750000000,  462.25125130704168441297952085733413696289062500000000,  457.79671345403272653129533864557743072509765625000000,  453.31561584437076817266643047332763671875000000000000,  448.80793398019523010589182376861572265625000000000000,  444.27363626982685218536062166094779968261718750000000,  439.71268394064378526309155859053134918212890625000000,
        435.12503093947083243619999848306179046630859375000000,  430.51062382002055528573691844940185546875000000000000,  425.86940161686516148620285093784332275390625000000000,  421.20129570535038965317653492093086242675781250000000,  416.50622964679524784514796920120716094970703125000000,  411.78411901824102869795751757919788360595703125000000,  407.03487122593401181802619248628616333007812500000000,  402.25838530163326822730596177279949188232421875000000,  397.45455168073766571978922002017498016357421875000000,  392.62325196111868308435077778995037078857421875000000,
        387.76435864142484888361650519073009490966796875000000,  382.87773483749521119534620083868503570556640625000000,  377.96323397537275923241395503282546997070312500000000,  373.02069945924966987149673514068126678466796875000000,  368.04996431249770694194012321531772613525390625000000,  363.05085078973894496812135912477970123291015625000000,  358.02316995769109553293674252927303314208984375000000,  352.96672124227478661850909702479839324951171875000000,  347.88129193919166937121190130710601806640625000000000,  342.76665668487174798428895883262157440185546875000000,
        337.62257688433595603783032856881618499755859375000000,  332.44880009212516824845806695520877838134765625000000,  327.24505934199572720899595879018306732177734375000000,  322.01107242057389612455153837800025939941406250000000,  316.74654107958127724486985243856906890869140625000000,  311.45115018058277200907468795776367187500000000000000,  306.12456676545122036259272135794162750244140625000000,  300.76643904487747249731910414993762969970703125000000,  295.37639529625607792695518583059310913085937500000000,  289.95404266112734603666467592120170593261718750000000,
        284.49896583102446356861037202179431915283203125000000,  279.01072560903082830918719992041587829589843750000000,  273.48885733255190189083805307745933532714843750000000,  267.93286914070620241545839235186576843261718750000000,  262.34224006727100686475750990211963653564453125000000,  256.71641793721983049181289970874786376953125000000000,  251.05481704145563526253681629896163940429687500000000,  245.35681556027631700089841615408658981323242187500000,  239.62175270125564452428079675883054733276367187500000,  233.84892551141783201273938175290822982788085937500000,
        228.03758531659607911024068016558885574340820312500000,  222.18693373241995914213475771248340606689453125000000,  216.29611818110927856650960166007280349731445312500000,  210.36422683570447134115966036915779113769531250000000,  204.39028289793057524548203218728303909301757812500000,  198.37323809680017916434735525399446487426757812500000,  192.31196527127011108859733212739229202270507812500000,  186.20524987042196585207420866936445236206054687500000,  180.05178016689706055331043899059295654296875000000000,  173.85013593121047392742184456437826156616210937500000,
        167.59877525270530895795673131942749023437500000000000,  161.29601911259521784813841804862022399902343750000000,  154.94003320921976296631328295916318893432617187500000,  148.52880639597543677155044861137866973876953125000000,  142.06012490497113276433083228766918182373046875000000,  135.53154127466655154421459883451461791992187500000000,  128.94033654839023483873461373150348663330078125000000,  122.28347381847626706985465716570615768432617187500000,  115.55754048937366462723730364814400672912597656250000,  108.75867561421782170327787753194570541381835937500000,
        101.88247714734707471961883129552006721496582031250000,   94.92388165691333767881587846204638481140136718750000,   87.87700545195538381904043490067124366760253906250000,   80.73493029478268567800114396959543228149414062500000,   73.48940721232644079918827628716826438903808593750000,   66.13043510738752672750706551596522331237792968750000,   58.64564011806122323378076544031500816345214843750000,   51.01932197519366951610209071077406406402587890625000,   43.23090891915575184611952863633632659912109375000000,   35.25227705011732126649803831242024898529052734375000,
         27.04265080983283908722114574629813432693481445312500,   18.53753758193118272856736439280211925506591796875000,    9.61919971315236210784860304556787014007568359375000,    0.00000000000000000000000000000000000000000000000000,  727.17456717281584133161231875419616699218750000000000,  728.92187685539863650774350389838218688964843750000000,  729.92499159954445531184319406747817993164062500000000,  730.47787988748837051389273256063461303710937500000000,  730.70301096181503908155718818306922912597656250000000,  730.66854749331650964450091123580932617187500000000000,
        730.41819605987575414474122226238250732421875000000000,  729.98247698742011380090843886137008666992187500000000,  729.38395948915285771363414824008941650390625000000000,  728.64003571314788132440298795700073242187500000000000,  727.76453219418942808260908350348472595214843750000000,  726.76871193078670785325812175869941711425781250000000,  725.66193119066088002000469714403152465820312500000000,  724.45208844506930745410500094294548034667968750000000,  723.14594190646005245071137323975563049316406250000000,  721.74934061880992430815240368247032165527343750000000,
        720.26739673711119849031092599034309387207031250000000,  718.70461664319930150668369606137275695800781250000000,  717.06500253609021910961018875241279602050781250000000,  715.35213238898734289250569418072700500488281250000000,  713.56922375654210100037744268774986267089843750000000,  711.71918532488166420080233365297317504882812500000000,  709.80465902047126292018219828605651855468750000000000,  707.82805474998986028367653489112854003906250000000000,  705.79157931946531334688188508152961730957031250000000,  703.69726070551428165344987064599990844726562500000000,
        701.54696857836904655414400622248649597167968750000000,  699.34243177476889741228660568594932556152343750000000,  697.08525326807466626632958650588989257812500000000000,  694.77692306894573448516894131898880004882812500000000,  692.41882940271864299575099721550941467285156250000000,  690.01226844226300727314082905650138854980468750000000,  687.55845282255586425890214741230010986328125000000000,  685.05851912190178154560271650552749633789062500000000,  682.51353446196185359440278261899948120117187500000000,  679.92450235257069834915455430746078491210937500000000,
        677.29236788626656107226153835654258728027343750000000,  674.61802237038807561475550755858421325683593750000000,  671.90230747069279004790587350726127624511718750000000,  669.14601892904227042890852317214012145996093750000000,  666.34990990830374357756227254867553710937500000000000,  663.51469400982546176237519830465316772460937500000000,  660.64104800235077163961250334978103637695312500000000,  657.72961429580777803494129329919815063476562500000000,  654.78100318884548869391437619924545288085937500000000,  651.79579491513095490518026053905487060546875000000000,
        648.77454151015854222350753843784332275390625000000000,  645.71776851753827486390946432948112487792968750000000,  642.62597655135834884276846423745155334472656250000000,  639.49964272918282404134515672922134399414062500000000,  636.33922198849109008733648806810379028320312500000000,  633.14514829785900928982300683856010437011718750000000,  629.91783577287390016863355413079261779785156250000000,  626.65767970564138522604480385780334472656250000000000,  623.36505751574975420226110145449638366699218750000000,  620.04032962970347853115526959300041198730468750000000,
        616.68384029507387822377495467662811279296875000000000,  613.29591833495862829295219853520393371582031250000000,  609.87687784775437194184632971882820129394531250000000,  606.42701885673614015104249119758605957031250000000000,  602.94662791347877828229684382677078247070312500000000,  599.43597865875744901131838560104370117187500000000000,  595.89533234420082408178132027387619018554687500000000,  592.32493831765623326646164059638977050781250000000000,  588.72503447494068495871033519506454467773437500000000,  585.09584768039599111943971365690231323242187500000000,
        581.43759415844056093192193657159805297851562500000000,  577.75047985810431327990954741835594177246093750000000,  574.03470079235069079004460945725440979003906250000000,  570.29044335382195640704594552516937255859375000000000,  566.51788460849468265223549678921699523925781250000000,  562.71719256859785218694014474749565124511718750000000,  558.88852644601763586251763626933097839355468750000000,  555.03203688730934572959085926413536071777343750000000,  551.14786619132257783348904922604560852050781250000000,  547.23614851036245454452000558376312255859375000000000,
        543.29701003571801720681833103299140930175781250000000,  539.33056916830889804259641095995903015136718750000000,  535.33693667513205127761466428637504577636718750000000,  531.31621583211858705908525735139846801757812500000000,  527.26850255394890609750291332602500915527343750000000,  523.19388551131589792930753901600837707519531250000000,  519.09244623606969071261119097471237182617187500000000,  514.96425921462468977551907300949096679687500000000000,  510.80939196996274631601409055292606353759765625000000,  506.62790513251422908069798722863197326660156250000000,
        502.41985250016040254195104353129863739013671875000000,  498.18528108755043604105594567954540252685546875000000,  493.92423116488788537026266567409038543701171875000000,  489.63673628630021994467824697494506835937500000000000,  485.32282330786244983755750581622123718261718750000000,  480.98251239530799239219049923121929168701171875000000,  476.61581702141842242781422100961208343505859375000000,  472.22274395304413019403000362217426300048828125000000,  467.80329322766772293107351288199424743652343750000000,  463.35745811937880489495000801980495452880859375000000,
        458.88522509408960559085244312882423400878906250000000,  454.38657375377272273908602073788642883300781250000000,  449.86147676945989815067150630056858062744140625000000,  445.30989980269180250616045668721199035644531250000000,  440.73180141505758911080192774534225463867187500000000,  436.12713296541210183931980282068252563476562500000000,  431.49583849429882320691831409931182861328125000000000,  426.83785459504872505931416526436805725097656250000000,  422.15311027095896179162082262337207794189453125000000,  417.44152677788571281780605204403400421142578125000000,
        412.70301745150948136142687872052192687988281250000000,  407.93748751844771049945848062634468078613281250000000,  403.14483389030294802068965509533882141113281250000000,  398.32494493963179138518171384930610656738281250000000,  393.47770025671769644759478978812694549560546875000000,  388.60297038590704232774442061781883239746093750000000,  383.70061654014176610871800221502780914306640625000000,  378.77049029217397446700488217175006866455078125000000,  373.81243324079144940697005949914455413818359375000000,  368.82627665020135054874117486178874969482421875000000,
        363.81184106052563720368198119103908538818359375000000,  358.76893586713737249738187529146671295166015625000000,  353.69735886632247456873301416635513305664062500000000,  348.59689576447260606073541566729545593261718750000000,  343.46731964770418699117726646363735198974609375000000,  338.30839040844767851012875325977802276611328125000000,  333.11985412515440430070157162845134735107421875000000,  327.90144239081951127445790916681289672851562500000000,  322.65287158551086577062960714101791381835937500000000,  317.37384208751319647490163333714008331298828125000000,
        312.06403741703729792789090424776077270507812500000000,  306.72312330568684046738781034946441650390625000000000,  301.35074668400915243182680569589138031005859375000000,  295.94653457845794264358119107782840728759765625000000,  290.51009290794831940729636698961257934570312500000000,  285.04100516885011984413722530007362365722656250000000,  279.53883099572317405545618385076522827148437500000000,  274.00310458329852281167404726147651672363281250000000,  268.43333295310702624192344956099987030029296875000000,  262.82899404569150192401139065623283386230468750000000,
        257.18953461643695845850743353366851806640625000000000,  251.51436790962210920952202286571264266967773437500000,  245.80287108122774952789768576622009277343750000000000,  240.05438233618420440507179591804742813110351562500000,  234.26819773993440776393981650471687316894531250000000,  228.44356765720274893283203709870576858520507812500000,  222.57969276241314560138562228530645370483398437500000,  216.67571955593356847202812787145376205444335937500000,  210.73073530777568862504267599433660507202148437500000,  204.74376233494652410627168137580156326293945312500000,
        198.71375149955613892416295129805803298950195312500000,  192.63957479099613578910066280514001846313476562500000,  186.52001682565756368603615555912256240844726562500000,  180.35376505991908402393164578825235366821289062500000,  174.13939846402860212037921883165836334228515625000000,  167.87537434263950331114756409078836441040039062500000,  161.56001290744873699622985441237688064575195312500000,  155.19147910206089591156342066824436187744140625000000,  148.76776103954009045082784723490476608276367187500000,  142.28664422569929115525155793875455856323242187500000,
        135.74568048638323602972377557307481765747070312500000,  129.14215016564290294809325132519006729125976562500000,  122.47301566953906615253799827769398689270019531250000,  115.73486372892888596197735751047730445861816406250000,  108.92383273572482949020923115313053131103515625000000,  102.03551999510803227622091071680188179016113281250000,   95.06486143784513842547312378883361816406250000000000,   88.00597274707671147098153596743941307067871093750000,   80.85193507042471594559174263849854469299316406250000,   73.59449883107559742256853496655821800231933593750000,
         66.22366233876819308079575421288609504699707031250000,   58.72705114896612599295622203499078750610351562500000,   51.08896442006944482727703871205449104309082031250000,   43.28882982995075678900320781394839286804199218750000,   35.29852292599867524813816999085247516632080078125000,   27.07726760667942755844705970957875251770019531250000,   18.56057072160891863177312188781797885894775390625000,    9.63069409257809638802427798509597778320312500000000,    0.00000000000000000000000000000000000000000000000000,  732.33935314673931316065136343240737915039062500000000,
        734.08765800300068349315552040934562683105468750000000,  735.09187101140253162157023325562477111816406250000000,  735.64594781742562190629541873931884765625000000000000,  735.87234891153411808772943913936614990234375000000000,  735.83923069700688301963964477181434631347656250000000,  735.59029509517847600363893434405326843261718750000000,  735.15605887447986788174603134393692016601562500000000,  734.55908847216767298959894105792045593261718750000000,  733.81677383558997007639845833182334899902343750000000,  732.94293973462447411293396726250648498535156250000000,
        731.94884774136744454153813421726226806640625000000000,  730.84385296574134827096713706851005554199218750000000,  729.63585293842311330081429332494735717773437500000000,  728.33160510989955582772381603717803955078125000000000,  726.93695791128243399725761264562606811523437500000000,  725.45702301085270846670027822256088256835937500000000,  723.89630641180428938241675496101379394531250000000000,  722.25881002815356168866856023669242858886718750000000,  720.54811163010288055374985560774803161621093750000000,  718.76742864181926506716990843415260314941406250000000,
        716.91966968367091794789303094148635864257812500000000,  715.00747667463929246878251433372497558593750000000000,  713.03325956682317610102472826838493347167968750000000,  710.99922526007242140622111037373542785644531250000000,  708.90740186943901335325790569186210632324218750000000,  706.75965924500326309498632326722145080566406250000000,  704.55772644205376309400890022516250610351562500000000,  702.30320668889828539249720051884651184082031250000000,  699.99759028558560203236993402242660522460937500000000,  697.64226577962176634173374623060226440429687500000000,
        695.23852969741471952147549018263816833496093750000000,  692.78759505765765425167046487331390380859375000000000,  690.29059885154470066481735557317733764648437500000000,  687.74860864196114107471657916903495788574218750000000,  685.16262840760703056730562821030616760253906250000000,  682.53360373696352780825691297650337219238281250000000,  679.86242645993979749619029462337493896484375000000000,  677.14993879114490482606925070285797119140625000000000,  674.39693704732019341463455930352210998535156250000000,  671.60417499207369473879225552082061767578125000000000,
        668.77236685326488441205583512783050537109375000000000,  665.90219005190169809793587774038314819335937500000000,  662.99428767597942169231828302145004272460937500000000,  660.04927072812972710380563512444496154785156250000000,  657.06772017208982106240000575780868530273437500000000,  654.05018879974045376002322882413864135742187500000000,  650.99720293767643397586653009057044982910156250000000,  647.90926400990599631768418475985527038574218750000000,  644.78684997123286848363932222127914428710937500000000,  641.63041662413320409541483968496322631835937500000000,
        638.44039883042080418817931786179542541503906250000000,  635.21721162769642887724330648779869079589843750000000,  631.96125125943569855735404416918754577636718750000000,  628.67289612658419173385482281446456909179687500000000,  625.35250766767046570748789235949516296386718750000000,  622.00043117368397815880598500370979309082031250000000,  618.61699654331403053220128640532493591308593750000000,  615.20251898355138564511435106396675109863281250000000,  611.75729966014910132798831909894943237304687500000000,  608.28162630198028182348934933543205261230468750000000,
        604.77577376292811095481738448143005371093750000000000,  601.24000454458723652351181954145431518554687500000000,  597.67456928273611538315890356898307800292968750000000,  594.07970720025582522794138640165328979492187500000000,  590.45564652891755486052716150879859924316406250000000,  586.80260490223281522048637270927429199218750000000000,  583.12078972135736876225564628839492797851562500000000,  579.41039849585342835780465975403785705566406250000000,  575.67161916095290052908239886164665222167968750000000,  571.90463037280960634234361350536346435546875000000000,
        568.10960178309881030145334079861640930175781250000000,  564.28669429419301195594016462564468383789062500000000,  560.43606029603415663586929440498352050781250000000000,  556.55784388572237730841152369976043701171875000000000,  552.65218107074099407327594235539436340332031250000000,  548.71919995665871283563319593667984008789062500000000,  544.75902092006776911148335784673690795898437500000000,  540.77175676744229804171482101082801818847656250000000,  536.75751288053834286984056234359741210937500000000000,  532.71638734889017996465554460883140563964843750000000,
        528.64847108989988555549643933773040771484375000000000,  524.55384795696465971559518948197364807128906250000000,  520.43259483603196713374927639961242675781250000000000,  516.28478173092582892422797158360481262207031250000000,  512.11047183773973756615305319428443908691406250000000,  507.90972160854920502970344386994838714599609375000000,  503.68258080465250259294407442212104797363281250000000,  499.42909253950909942432190291583538055419921875000000,  495.14929331150324287591502070426940917968750000000000,  490.84321302662203834188403561711311340332031250000000,
        486.51087501109935828935704194009304046630859375000000,  482.15229601403603965081856586039066314697265625000000,  477.76748619997192690789233893156051635742187500000000,  473.35644913134223088491125963628292083740234375000000,  468.91918174071548719439306296408176422119140625000000,  464.45567429266702674794942140579223632812500000000000,  459.96591033510225088321021758019924163818359375000000,  455.44986663979960894721443764865398406982421875000000,  450.90751313190008886522264219820499420166015625000000,  446.33881280802108904026681557297706604003906250000000,
        441.74372164262513251742348074913024902343750000000000,  437.12218848221891676075756549835205078125000000000000,  432.47415492690498695083078928291797637939453125000000,  427.79955519874550873282714746892452239990234375000000,  423.09831599633525911485776305198669433593750000000000,  418.37035633491058206345769576728343963623046875000000,  413.61558737124619256064761430025100708007812500000000,  408.83391221250826674804557114839553833007812500000000,  404.02522570814517166581936180591583251953125000000000,  399.18941422379714367707492783665657043457031250000000,
        394.32635539610100749996490776538848876953125000000000,  389.43591786714625868626171723008155822753906250000000,  384.51796099720917254671803675591945648193359375000000,  379.57233455424795920407632365822792053222656250000000,  374.59887837848253866468439809978008270263671875000000,  369.59742202020294143949286080896854400634765625000000,  364.56778434875650418689474463462829589843750000000000,  359.50977313043858885066583752632141113281250000000000,  354.42318457276979870584909804165363311767578125000000,  349.30780283236094874155241996049880981445312500000000,
        344.16339948325907016624114476144313812255859375000000,  338.98973294231501540707540698349475860595703125000000,  333.78654784771759977957117371261119842529296875000000,  328.55357438639191514084814116358757019042968750000000,  323.29052756544660951476544141769409179687500000000000,  317.99710642228041024281992577016353607177734375000000,  312.67299316729298652717261575162410736083984375000000,  307.31785225239298142696497961878776550292968750000000,  301.93132935762645274735405109822750091552734375000000,  296.51305028725238344122772105038166046142578125000000,
        291.06261976544431036018067970871925354003906250000000,  285.57962012046198196912882849574089050292968750000000,  280.06360984459581686678575351834297180175781250000000,  274.51412201538636281838989816606044769287109375000000,  268.93066256151939796836813911795616149902343750000000,  263.31270835433099364308873191475868225097656250000000,  257.65970510295625217622728087007999420166015625000000,  251.97106502772405178802728187292814254760742187500000,  246.24616428233198917041590902954339981079101562500000,  240.48434009048250459272821899503469467163085937500000,
        234.68488755685638125214609317481517791748046875000000,  228.84705610531281649855372961610555648803710937500000,  222.97004548875776208660681731998920440673828125000000,  217.05300130485835552462958730757236480712890625000000,  211.09500993923003875352151226252317428588867187500000,  205.09509284229332592985883820801973342895507812500000,  199.05220002690333558348356746137142181396484375000000,  192.96520265006589056611119303852319717407226562500000,  186.83288451220877846026269253343343734741210937500000,  180.65393226973745299801521468907594680786132812500000,
        174.42692410849898010383185464888811111450195312500000,  168.15031656391388992233260069042444229125976562500000,  161.82242909322289392548555042594671249389648437500000,  155.44142589997247227984189521521329879760742187500000,  149.00529437120007969497237354516983032226562500000000,  142.51181930036861444932583253830671310424804687500000,  135.95855181430553670907102059572935104370117187500000,  129.34277157104435218570870347321033477783203125000000,  122.66144030330569592024403391405940055847167968750000,  115.91114408097128318786417366936802864074707031250000,
        109.08802064703972689585498301312327384948730468750000,  102.18766666954627453378634527325630187988281250000000,   95.20501745359480594288470456376671791076660156250000,   88.13418806787004200486990157514810562133789062500000,   80.96825905647706633772031636908650398254394531250000,   73.69898024976451722523052012547850608825683593750000,   66.31634937506025551101629389449954032897949218750000,   58.80799141559892007080634357407689094543457031250000,   51.15820496749601886676828144118189811706542968750000,   43.34641715605328471383472788147628307342529296875000,
         35.34450298557815983713226160034537315368652343750000,   27.11168582072775024016664247028529644012451171875000,   18.58347198636961294937464117538183927536010742187500,    9.64212278840171954641391494078561663627624511718750,    0.00000000000000000000000000000000000000000000000000,  737.50983714177743877371540293097496032714843750000000,  739.25913053634815241821343079209327697753906250000000,  740.26443427974288624682230874896049499511718750000000,  740.81969128147056835587136447429656982421875000000000,  741.04735334559882176108658313751220703125000000000000,
        741.01557065288352532661519944667816162109375000000000,  740.76804049991949341347208246588706970214843750000000,  740.33527612057605438167229294776916503906250000000000,  739.73984119283829841151600703597068786621093750000000,  738.99912347541169310716213658452033996582031250000000,  738.12694598186851635546190664172172546386718750000000,  737.13456886377673527022125199437141418457031250000000,  736.03134607697143110272008925676345825195312500000000,  734.82517421348222796950722113251686096191406250000000,  733.52280996228114418045151978731155395507812500000000,
        732.13010114079816048615612089633941650390625000000000,  730.65215892869196068204473704099655151367187500000000,  729.09348894764355009101564064621925354003906250000000,  727.45809282293930664309300482273101806640625000000000,  725.74954811727604919724399223923683166503906250000000,  723.97107211912157254118937999010086059570312500000000,  722.12557337721204930858220905065536499023437500000000,  720.21569379655727516365004703402519226074218750000000,  718.24384336760556379886111244559288024902343750000000,  716.21222907640265020745573565363883972167968750000000,
        714.12287916827517619822174310684204101562500000000000,  711.97766366447297059494303539395332336425781250000000,  709.77831182964814615843351930379867553710937500000000,  707.52642713737395752104930579662322998046875000000000,  705.22350016691439122951123863458633422851562500000000,  702.87091977728334768471540883183479309082031250000000,  700.46998283727884881955105811357498168945312500000000,  698.02190273767371309077134355902671813964843750000000,  695.52781687042488556471653282642364501953125000000000,  692.98879322701930050243390724062919616699218750000000,
        690.40583624190048794844187796115875244140625000000000,  687.77989198586067232099594548344612121582031250000000,  685.11185279723076746449805796146392822265625000000000,  682.40256142479586287663551047444343566894531250000000,  679.65281474496237024141009896993637084960937500000000,  676.86336710631314872443908825516700744628906250000000,  674.03493334688801041920669376850128173828125000000000,  671.16819152304788076435215771198272705078125000000000,  668.26378538334813583787763491272926330566406250000000,  665.32232661628142977860989049077033996582031250000000,
        662.34439689690111663367133587598800659179687500000000,  659.33054975406730591203086078166961669921875000000000,  656.28131227727976693131495267152786254882812500000000,  653.19718667968948011548491194844245910644531250000000,  650.07865173184382001636549830436706542968750000000000,  646.92616407897298813622910529375076293945312500000000,  643.74015945311452924215700477361679077148437500000000,  640.52105379006741259217960759997367858886718750000000,  637.26924426003142798435874283313751220703125000000000,  633.98511021980220903060398995876312255859375000000000,
        630.66901409352885821135714650154113769531250000000000,  627.32130218828626766480738297104835510253906250000000,  623.94230545005382282397476956248283386230468750000000,  620.53234016510884885065024718642234802246093750000000,  617.09170861132702157192397862672805786132812500000000,  613.62069966343233318184502422809600830078125000000000,  610.11958935583186303119873628020286560058593750000000,  606.58864140631624195520998910069465637207031250000000,  603.02810770358792069600895047187805175781250000000000,  599.43822876129388532717712223529815673828125000000000,
        595.81923414098935154470382258296012878417968750000000,  592.17134284622522955032764002680778503417968750000000,  588.49476368975774676073342561721801757812500000000000,  584.78969563568387002305826172232627868652343750000000,  581.05632811815110017050756141543388366699218750000000,  577.29484133813184598693624138832092285156250000000000,  573.50540653962457326997537165880203247070312500000000,  569.68818626651489012147067114710807800292968750000000,  565.84333460122252290602773427963256835937500000000000,  561.97099738615509068040410056710243225097656250000000,
        558.07131242889909117366187274456024169921875000000000,  554.14440969198926723038312047719955444335937500000000,  550.19041146802430830575758591294288635253906250000000,  546.20943254081885243067517876625061035156250000000000,  542.20158033321831680950708687305450439453125000000000,  538.16695504214067113935016095638275146484375000000000,  534.10564976134935477602994069457054138183593750000000,  530.01775059240992504783207550644874572753906250000000,  525.90333674422970489104045554995536804199218750000000,  521.76248062153354112524539232254028320312500000000000,
        517.59524790258296889078337699174880981445312500000000,  513.40169760640094409609446302056312561035156250000000,  509.18188214972570904137683100998401641845703125000000,  504.93584739387404169974615797400474548339843750000000,  500.66363268165827093980624340474605560302734375000000,  496.36527086446005796460667625069618225097656250000000,  492.04078831953012240774114616215229034423828125000000,  487.69020495754278954336768947541713714599609375000000,  483.31353422039956058142706751823425292968750000000000,  478.91078306923805030237417668104171752929687500000000,
        474.48195196256455119510064832866191864013671875000000,  470.02703482439096660527866333723068237304687500000000,  465.54601900221757659892318770289421081542968750000000,  461.03888521466205929755233228206634521484375000000000,  456.50560748849449055342120118439197540283203125000000,  451.94615308479160376009531319141387939453125000000000,  447.36048241388056112555204890668392181396484375000000,  442.74854893869064653699751943349838256835937500000000,  438.11029906608166584192076697945594787597656250000000,  433.44567202566105379446526058018207550048828125000000,
        428.75459973554262660400127060711383819580078125000000,  424.03700665443687967126606963574886322021484375000000,  419.29280961939190319753834046423435211181640625000000,  414.52191766843060349856386892497539520263671875000000,  409.72423184724789280153345316648483276367187500000000,  404.89964499904203876212704926729202270507812500000000,  400.04804153645716269238619133830070495605468750000000,  395.16929719450837410477106459438800811767578125000000,  390.26327876323972532190964557230472564697265625000000,  385.32984379873960278928279876708984375000000000000000,
        380.36884031099026515221339650452136993408203125000000,  375.38010642687305562503752298653125762939453125000000,  370.36347002646868986630579456686973571777343750000000,  365.31874835059954875760013237595558166503906250000000,  360.24574757733637397905113175511360168457031250000000,  355.14426236494733757353969849646091461181640625000000,  350.01407535849097030222765170037746429443359375000000,  344.85495665694031686143716797232627868652343750000000,  339.66666323737888433242915198206901550292968750000000,  334.44893833240979574839002452790737152099609375000000,
        329.20151075647305560778477229177951812744140625000000,  323.92409417625543710528290830552577972412109375000000,  318.61638631979911906455527059733867645263671875000000,  313.27806811825394106563180685043334960937500000000000,  307.90880277346332150045782327651977539062500000000000,  302.50823474370565691060619428753852844238281250000000,  297.07598863891701057582395151257514953613281250000000,  291.61166801557089911511866375803947448730468750000000,  286.11485406005954246211331337690353393554687500000000,  280.58510414787713216355768963694572448730468750000000,
        275.02195026410527134430594742298126220703125000000000,  269.42489726860185328405350446701049804687500000000000,  263.79342098682445794111117720603942871093750000000000,  258.12696610432288935044198296964168548583984375000000,  252.42494383950014480433310382068157196044921875000000,  246.68672936517654648014286067336797714233398437500000,  240.91165894463654240098549053072929382324218750000000,  235.09902674203331685021112207323312759399414062500000,  229.24808126003955521809984929859638214111328125000000,  223.35802134918563410792557988315820693969726562500000,
        217.42799172306183663749834522604942321777343750000000,  211.45707790101070600030652713030576705932617187500000,  205.44430048450567483087070286273956298828125000000000,  199.38860865431854563212255015969276428222656250000000,  193.28887275178874460834776982665061950683593750000000,  187.14387577766271419932309072464704513549804687500000,  180.95230360423167326189286541193723678588867187500000,  174.71273364839143482640793081372976303100585937500000,  168.42362169138306171589647419750690460205078125000000,  162.08328645066114859218942001461982727050781250000000,
        155.68989140401276927150320261716842651367187500000000,  149.24142322638809332602249924093484878540039062500000,  142.73566601249098084736033342778682708740234375000000,  136.17017020338545307822641916573047637939453125000000,  129.54221478401461808971362188458442687988281250000000,  122.84876082636914418344531441107392311096191406250000,  116.08639375165843432569090509787201881408691406250000,  109.25125066597276202173816272988915443420410156250000,  102.33892761191825115929532330483198165893554687500000,   95.34435928037279950331139843910932540893554687500000,
         88.26166013673041277343145338818430900573730468750000,   81.08391013248088086129428120329976081848144531250000,   73.80285851578082656487822532653808593750000000000000,   66.40850244194461993174627423286437988281250000000000,   58.88846633213444192733732052147388458251953125000000,   51.22704823010369068470026832073926925659179687500000,   43.40367471827841683307269704528152942657470703125000,   35.39022026735924697504742653109133243560791015625000,   27.14590771745411501569833490066230297088623046875000,   18.60624287773371676735223445575684309005737304687500,
          9.65348654705203479409192368621006608009338378906250,    0.00000000000000000000000000000000000000000000000000,  742.68598687435121519229141995310783386230468750000000,  744.43626225427999543171608820557594299316406250000000,  745.44264929886696791072608903050422668457031250000000,  745.99907828110644913977012038230895996093750000000000,  746.22799238356992646004073321819305419921875000000000,  746.19753560894594102137489244341850280761718750000000,  745.95140066054773342330008745193481445312500000000000,  745.52009726039023007615469396114349365234375000000000,
        744.92618634372024644108023494482040405273437500000000,  744.18705349260562798008322715759277343750000000000000,  743.31651997288690836285240948200225830078125000000000,  742.32584452149899334472138434648513793945312500000000,  741.22437994394317684054840356111526489257812500000000,  740.02002189560278111457591876387596130371093750000000,  738.71952630447435694804880768060684204101562500000000,  737.32874037360079455538652837276458740234375000000000,  735.85277479223782393091823905706405639648437500000000,  734.29613479782733520551118999719619750976562500000000,
        732.66282172335797895357245579361915588378906250000000,  730.95641291969161557062761858105659484863281250000000,  729.18012553457117519428720697760581970214843750000000,  727.33686803943078302836511284112930297851562500000000,  725.42928231903590585716301575303077697753906250000000,  723.45977839534214126615552231669425964355468750000000,  721.43056333320510020712390542030334472656250000000000,  719.34366550031927545205689966678619384765625000000000,  717.20095508068970957538112998008728027343750000000000,  715.00416153942580876901047304272651672363281250000000,
        712.75488858597407215711427852511405944824218750000000,  710.45462706894466009543975815176963806152343750000000,  708.10476614851404519868083298206329345703125000000000,  705.70660302505280014884192496538162231445312500000000,  703.26135145012142402265453711152076721191406250000000,  700.77014920467104275303427129983901977539062500000000,  698.23406469654275952052557840943336486816406250000000,  695.65410280318860714032780379056930541992187500000000,  693.03121006448509433539584279060363769531250000000000,  690.36627931345765318837948143482208251953125000000000,
        687.66015381882812107505742460489273071289062500000000,  684.91363100190915247367229312658309936523437500000000,  682.12746578096539451507851481437683105468750000000000,  679.30237358837780448084231466054916381835937500000000,  676.43903309946153967757709324359893798828125000000000,  673.53808870635589300945866852998733520507812500000000,  670.60015276584681487292982637882232666015625000000000,  667.62580764612539496738463640213012695312500000000000,  664.61560759422366118087666109204292297363281250000000,  661.57008044308906846708850935101509094238281250000000,
        658.48972917488765688176499679684638977050781250000000,  655.37503335508847612800309434533119201660156250000000,  652.22645045013632625341415405273437500000000000000000,  649.04441704000964818987995386123657226562500000000000,  645.82934993565231707179918885231018066406250000000000,  642.58164721013724829390412196516990661621093750000000,  639.30168915142962760000955313444137573242187500000000,  635.98983914375855874823173508048057556152343750000000,  632.64644448384933639317750930786132812500000000000000,  629.27183713760859973262995481491088867187500000000000,
        625.86633444227027212036773562431335449218750000000000,  622.43023975849769158230628818273544311523437500000000,  618.96384307648406775115290656685829162597656250000000,  615.46742157968787978461477905511856079101562500000000,  611.94124016948785538261290639638900756835937500000000,  608.38555195371702666307101026177406311035156250000000,  604.80059870176000913488678634166717529296875000000000,  601.18661126863617027993313968181610107421875000000000,  597.54380999027046073024393990635871887207031250000000,  593.87240505194597517402144148945808410644531250000000,
        590.17259683175086593109881505370140075683593750000000,  586.44457622066590829490451142191886901855468750000000,  582.68852492079133753577480092644691467285156250000000,  578.90461572307265214476501569151878356933593750000000,  575.09301276576968575682258233428001403808593750000000,  571.25387177479183264949824661016464233398437500000000,  567.38734028692965694062877446413040161132812500000000,  563.49355785691568598849698901176452636718750000000000,  559.57265624916203705652151256799697875976562500000000,  555.62475961494658349693054333329200744628906250000000,
        551.64998465574547026335494592785835266113281250000000,  547.64844077334407757007284089922904968261718750000000,  543.62023020729623112856643274426460266113281250000000,  539.56544816024552346789278090000152587890625000000000,  535.48418291156633586069801822304725646972656250000000,  531.37651591973610720742726698517799377441406250000000,  527.24252191379684973071562126278877258300781250000000,  523.08226897422616730182198807597160339355468750000000,  518.89581860348971531493589282035827636718750000000000,  514.68322578650895593455061316490173339843750000000000,
        510.44453904123969323336496017873287200927734375000000,  506.17980045951452439112472347915172576904296875000000,  501.88904573827227295623742975294589996337890625000000,  497.57230420125421233024098910391330718994140625000000,  493.22959881121619218902196735143661499023437500000000,  488.86094617266746809036703780293464660644531250000000,  484.46635652511270109243923798203468322753906250000000,  480.04583372673783969730720855295658111572265625000000,  475.59937522844279556011315435171127319335937500000000,  471.12697203808869517160928808152675628662109375000000,
        466.62860867478786985884653404355049133300781250000000,  462.10426311302489921217784285545349121093750000000000,  457.55390671635808530481881462037563323974609375000000,  452.97750416040383925064816139638423919677734375000000,  448.37501334476468173306784592568874359130859375000000,  443.74638529351057059102458879351615905761718750000000,  439.09156404377404214756097644567489624023437500000000,  434.41048652196258217372815124690532684326171875000000,  429.70308240703553792627644725143909454345703125000000,  424.96927398022683064482407644391059875488281250000000,
        420.20897596052685685208416543900966644287109375000000,  415.42209532516494618903379887342453002929687500000000,  410.60853111424694361630827188491821289062500000000000,  405.76817421862057244652532972395420074462890625000000,  400.90090714993834808410611003637313842773437500000000,  396.00660379178549419521004892885684967041015625000000,  391.08512913061929339164635166525840759277343750000000,  386.13633896513943000172730535268783569335937500000000,  381.16007959256398862635251134634017944335937500000000,  376.15618747012564426768221892416477203369140625000000,
        371.12448884992898001655703410506248474121093750000000,  366.06479938510761940051452256739139556884765625000000,  360.97692370500459446702734567224979400634765625000000,  355.86065495684874804283026605844497680664062500000000,  350.71577431112655176548287272453308105468750000000000,  345.54205042753488896778435446321964263916015625000000,  340.33923887805207186829647980630397796630859375000000,  335.10708152326668596288072876632213592529296875000000,  329.84530583765746314384159632027149200439453125000000,  324.55362417900573746010195463895797729492187500000000,
        319.23173299654541779091232456266880035400390625000000,  313.87931197179381115347496233880519866943359375000000,  308.49602308525055605059606023132801055908203125000000,  303.08150960128631368206697516143321990966796875000000,  297.63539496254418281750986352562904357910156250000000,  292.15728158402907865820452570915222167968750000000000,  286.64674953572784943389706313610076904296875000000000,  281.10335510105909406775026582181453704833984375000000,  275.52662919665255003565107472240924835205078125000000,  269.91607563685607829029322601854801177978515625000000,
        264.27116922390365516548627056181430816650390625000000,  258.59135364177370774996234104037284851074218750000000,  252.87603912833995423170563299208879470825195312500000,  247.12459989634621138065995182842016220092773437500000,  241.33637126888433499516395386308431625366210937500000,  235.51064648924986499878286849707365036010742187500000,  229.64667315806269698441610671579837799072265625000000,  223.74364924209325522497238125652074813842773437500000,  217.80071858897005654398526530712842941284179687500000,  211.81696586939406756755488459020853042602539062500000,
        205.79141085305553815487655811011791229248046875000000,  199.72300190535457886653603054583072662353515625000000,  193.61060856823874587462341878563165664672851562500000,  187.45301305862471963337156921625137329101562500000000,  181.24890048013187993092287797480821609497070312500000,  174.99684749575126829768123570829629898071289062500000,  168.69530914720760961245105136185884475708007812500000,  162.34260342646140884426131378859281539916992187500000,  155.93689309947316701254749204963445663452148437500000,  149.47616414269037932172068394720554351806640625000000,
        142.95819996530548223745427094399929046630859375000000,  136.38055033553985140315489843487739562988281250000000,  129.74049357785034430889936629682779312133789062500000,  123.03499011579502564472932135686278343200683593750000,  116.26062473391061757865827530622482299804687500000000,  109.41353391309016274135501589626073837280273437500000,  102.48931308194191558413876919075846672058105468750000,   95.48289632827750494925567181780934333801269531250000,   88.38839752509643687972129555419087409973144531250000,   81.19889604191381238251778995618224143981933593750000,
         73.90614055508659419047035044059157371520996093750000,   66.50012765806718562089372426271438598632812500000000,   58.96848121986283786100102588534355163574218750000000,   51.29549874155518551788190961815416812896728515625000,   43.46060627216424876451128511689603328704833984375000,   35.43567775803915509413855033926665782928466796875000,   27.17993552378560195847967406734824180603027343750000,   18.62888487172196505525789689272642135620117187500000,    9.66478610230596757446619449183344841003417968750000,    0.00000000000000000000000000000000000000000000000000,
        747.86777042464336773264221847057342529296875000000000,  749.61902131790861858462449163198471069335937500000000,  750.62648432359469552466180175542831420898437500000000,  751.18407717633988340821815654635429382324218750000000,  751.41423450130889705178560689091682434082031250000000,  751.38509416705380772327771410346031188964843750000000,  751.14034431471816333214519545435905456542968750000000,  750.71049117694383312482386827468872070312500000000000,  750.11809296263220403488958254456520080566406250000000,  749.38053308914129502227297052741050720214843750000000,
        748.51163108312266558641567826271057128906250000000000,  747.52264427278237235441338270902633666992187500000000,  746.42292431708074218477122485637664794921875000000000,  745.22036593682128113869111984968185424804687500000000,  743.92172429965137325780233368277549743652343750000000,  742.53284599362916651443811133503913879394531250000000,  741.05884121595113356306683272123336791992187500000000,  739.50421481723560646059922873973846435546875000000000,  737.87296783476017481007147580385208129882812500000000,  736.16867740339205283817136660218238830566406250000000,
        734.39456052530636043229606002569198608398437500000000,  732.55352558916138150379993021488189697265625000000000,  730.64821445341317485144827514886856079101562500000000,  728.68103716490179522224934771656990051269531250000000,  726.65420086014046319178305566310882568359375000000000,  724.56973402152709695656085386872291564941406250000000,  722.42950698766571804299019277095794677734375000000000,  720.23524941548396327561931684613227844238281250000000,  717.98856524119139521644683554768562316894531250000000,  715.69094557316714144690195098519325256347656250000000,
        713.34377986271226745884632691740989685058593750000000,  710.94836563127080353297060355544090270996093750000000,  708.50591698023197295697173103690147399902343750000000,  706.01757206811919331812532618641853332519531250000000,  703.48439970723939040908589959144592285156250000000000,  700.90740520569374893966596573591232299804687500000000,  698.28753555960713583772303536534309387207031250000000,  695.62568408337608616420766338706016540527343750000000,  692.92269455184373327938374131917953491210937500000000,  690.17936491690772982110502198338508605957031250000000,
        687.39645065167792381544131785631179809570312500000000,  684.57466776751175530080217868089675903320312500000000,  681.71469554277177849144209176301956176757812500000000,  678.81717899671855320775648579001426696777343750000000,  675.88273113739649033959722146391868591308593750000000,  672.91193500851011322083650156855583190917968750000000,  669.90534555703231944789877161383628845214843750000000,  666.86349134050124121131375432014465332031250000000000,  663.78687609059340957173844799399375915527343750000000,  660.67598014752672952454304322600364685058593750000000,
        657.53126177809724595135776326060295104980468750000000,  654.35315838864460147306090220808982849121093750000000,  651.14208764293709919002139940857887268066406250000000,  647.89844849383166547340806573629379272460937500000000,  644.62262213657686515944078564643859863281250000000000,  641.31497289076901324733626097440719604492187500000000,  637.97584901721222649939591065049171447753906250000000,  634.60558347527614841965259984135627746582031250000000,  631.20449462575948018638882786035537719726562500000000,  627.77288688375688252563122659921646118164062500000000,
        624.31105132556945136457215994596481323242187500000000,  620.81926625330129354551900178194046020507812500000000,  617.29779772042297736334148794412612915039062500000000,  613.74690002126726540154777467250823974609375000000000,  610.16681614714150327927200123667716979980468750000000,  606.55777821147955819469643756747245788574218750000000,  602.92000784624156040081288665533065795898437500000000,  599.25371657155221782886656001210212707519531250000000,  595.55910614039919437345815822482109069824218750000000,  591.83636886003512245224555954337120056152343750000000,
        588.08568789158800882432842627167701721191406250000000,  584.30723752924404834629967808723449707031250000000000,  580.50118346024510174174793064594268798828125000000000,  576.66768300683577308518579229712486267089843750000000,  572.80688535118724757921881973743438720703125000000000,  568.91893174423898926761467009782791137695312500000000,  565.00395569930958572513191029429435729980468750000000,  561.06208317125458506779978051781654357910156250000000,  557.09343272187106776982545852661132812500000000000000,  553.09811567219344397017266601324081420898437500000000,
        549.07623624225004732579691335558891296386718750000000,  545.02789167880473542027175426483154296875000000000000,  540.95317237154881695460062474012374877929687500000000,  536.85216195815928585943765938282012939453125000000000,  532.72493741859511828806716948747634887695312500000000,  528.57156915895598103816155344247817993164062500000000,  524.39212108518859167816117405891418457031250000000000,  520.18665066688367915048729628324508666992187500000000,  515.95520899136954540153965353965759277343750000000000,  511.69784080827184880035929381847381591796875000000000,
        507.41458456467029236591770313680171966552734375000000,  503.10547243095061276108026504516601562500000000000000,  498.77053031741581889946246519684791564941406250000000,  494.40977788168299866811139509081840515136718750000000,  490.02322852686194210036774165928363800048828125000000,  485.61088939047539270177367143332958221435546875000000,  481.17276132404595045954920351505279541015625000000000,  476.70883886324003242407343350350856781005859375000000,  472.21911018842337171008693985641002655029296875000000,  467.70355707544388224050635471940040588378906250000000,
        463.16215483642014305587508715689182281494140625000000,  458.59487225027356771533959545195102691650390625000000,  454.00167148269832750884233973920345306396484375000000,  449.38250799521932776769972406327724456787109375000000,  444.73733044294095861914684064686298370361328125000000,  440.06608056053852351396926678717136383056640625000000,  435.36869303598905389662832021713256835937500000000000,  430.64509537148143181184423156082630157470703125000000,  425.89520773088111127435695379972457885742187500000000,  421.11894277305685818646452389657497406005859375000000,
        416.31620547030513534991769120097160339355468750000000,  411.48689291102181186943198554217815399169921875000000,  406.63089408568811222721706144511699676513671875000000,  401.74808965513750536047155037522315979003906250000000,  396.83835169996496006206143647432327270507812500000000,  391.90154344982209977388265542685985565185546875000000,  386.93751899121366477629635483026504516601562500000000,  381.94612295226602327602449804544448852539062500000000,  376.92719016277982291285297833383083343505859375000000,  371.88054528770305751095293089747428894042968750000000,
        366.80600243196221299513126723468303680419921875000000,  361.70336471436991132577531971037387847900390625000000,  356.57242380808105508549488149583339691162109375000000,  351.41295944479315949138253927230834960937500000000000,  346.22473887957488614119938574731349945068359375000000,  341.00751631285692155870492570102214813232421875000000,  335.76103226572359972124104388058185577392578125000000,  330.48501290419574161205673590302467346191406250000000,  325.17916930768473093849024735391139984130859375000000,  319.84319667622122551620122976601123809814453125000000,
        314.47677347039933692940394394099712371826171875000000,  309.07956047722319681270164437592029571533203125000000,  303.65119979417409012967254966497421264648437500000000,  298.19131372282242864457657560706138610839843750000000,  292.69950356215571218854165636003017425537109375000000,  287.17534829046576305700000375509262084960937500000000,  281.61840312309203682161751203238964080810546875000000,  276.02819793152025340532418340444564819335937500000000,  270.40423550723329526590532623231410980224609375000000,  264.74598965124567939710686914622783660888671875000000,
        259.05290306735196281806565821170806884765625000000000,  253.32438503368805982063349802047014236450195312500000,  247.55980882313767210689547937363386154174804687500000,  241.75850883826134918308525811880826950073242187500000,  235.91977742062218226237746421247720718383789062500000,  230.04286128739454397873487323522567749023437500000000,  224.12695753969589418375107925385236740112304687500000,  218.17120917681643277319381013512611389160156250000000,  212.17470003797160416070255450904369354248046875000000,  206.13644907777245407487498596310615539550781250000000,
        200.05540386251450968302378896623849868774414062500000,  193.93043315059787801146740093827247619628906250000000,  187.76031839054471106464916374534368515014648437500000,  181.54374393234206763736438006162643432617187500000000,  175.27928569973252592717471998184919357299804687500000,  168.96539800920996299282705876976251602172851562500000,  162.60039814116703382751438766717910766601562500000000,  156.18244816331596780401014257222414016723632812500000,  149.70953336684289070035447366535663604736328125000000,  143.17943648834358327803784050047397613525390625000000,
        136.58970663579498250328470021486282348632812500000000,  129.93762148545914669739431701600551605224609375000000,  123.22014082445537042076466605067253112792968750000000,  116.43384881235333239146712003275752067565917968750000,  109.57488131627455629768519429489970207214355468750000,  102.63883316198375439398660091683268547058105468750000,   95.62063784511718722569639794528484344482421875000000,   88.51440865691535009318613447248935699462890625000000,   81.31322439530592305345635395497083663940429687500000,   74.00883317499280167339748004451394081115722656250000,
         66.59123103747903371640859404578804969787597656250000,   59.04804130930253336373425554484128952026367187500000,   51.36356095833816226559065398760139942169189453125000,   43.51721550945060101867056800983846187591552734375000,   35.48087839367994433814601507037878036499023437500000,   27.21377142896977119335133465938270092010498046875000,   18.65139941942952717113257676828652620315551757812500,    9.67602217557289456806302041513845324516296386718750,    0.00000000000000000000000000000000000000000000000000,  753.05515623048404449946247041225433349609375000000000,
        754.80737624454025080922292545437812805175781250000000,  755.81590796322848291310947388410568237304687500000000,  756.37465667971309812855906784534454345703125000000000,  756.60604852504513928579399362206459045410156250000000,  756.57821527705357311788247898221015930175781250000000,  756.33484054548387121030827984213829040527343750000000,  755.90642709586256842158036306500434875488281250000000,  755.31553042699908928625518456101417541503906250000000,  754.57953180339086429739836603403091430664062500000000,  753.71224902101255338493501767516136169433593750000000,
        752.72493800525069218565477058291435241699218750000000,  751.62694927235406794352456927299499511718750000000000,  750.42617661067572498723166063427925109863281250000000,  749.12937442822305911249713972210884094238281250000000,  747.74238869757471093180356547236442565917968750000000,  746.27032912233187289530178532004356384277343750000000,  744.71770016383936763304518535733222961425781250000000,  743.08850256040136628143955022096633911132812500000000,  741.38631322689400349190691486001014709472656250000000,  739.61434901525842633418506011366844177246093750000000,
        737.77551822609052578627597540616989135742187500000000,  735.87246268567309925856534391641616821289062500000000,  733.90759245932156318303896114230155944824218750000000,  731.88311474827708025259198620915412902832031250000000,  729.80105814221428772725630551576614379882812500000000,  727.66329312642619697726331651210784912109375000000000,  725.47154954127938708552392199635505676269531250000000,  723.22743154090733241901034489274024963378906250000000,  720.93243048416491092211799696087837219238281250000000,  718.58793610373447791062062606215476989746093750000000,
        716.19524623194354262523120269179344177246093750000000,  713.75557530937078354327240958809852600097656250000000,  711.27006186102494211809244006872177124023437500000000,  708.73977509213852954417234286665916442871093750000000,  706.16572072946485150168882682919502258300781250000000,  703.54884621291580515389796346426010131835937500000000,  700.89004532532931079913396388292312622070312500000000,  698.19016233425963946501724421977996826171875000000000,  695.44999570829145341122057288885116577148437500000000,  692.67030146098284149047685787081718444824218750000000,
        689.85179616776042621495435014367103576660156250000000,  686.99515969460242104105418547987937927246093750000000,  684.10103767192185841850005090236663818359375000000000,  681.17004374249961529130814597010612487792968750000000,  678.20276160846367474732687696814537048339843750000000,  675.19974689905177456239471212029457092285156250000000,  672.16152887811210803192807361483573913574218750000000,  669.08861200792807721882127225399017333984375000000000,  665.98147738391855909867445006966590881347656250000000,  662.84058405301630045869387686252593994140625000000000,
        659.66637022701888781739398837089538574218750000000000,  656.45925440090240954305045306682586669921875000000000,  653.21963638495117265847511589527130126953125000000000,  649.94789825857458254176890477538108825683593750000000,  646.64440525281816007918678224086761474609375000000000,  643.30950656782101759745273739099502563476562500000000,  639.94353613081500498083187267184257507324218750000000,  636.54681329967149849835550412535667419433593750000000,  633.11964351649794480181299149990081787109375000000000,  629.66231891532152076251804828643798828125000000000000,
        626.17511888750652815360808745026588439941406250000000,  622.65831060818561581982066854834556579589843750000000,  619.11214952667353372817160561680793762207031250000000,  615.53687982354745145130436867475509643554687500000000,  611.93273483682105506886728107929229736328125000000000,  608.29993745941908400709507986903190612792968750000000,  604.63870050995114979741629213094711303710937500000000,  600.94922707860018817882519215345382690429687500000000,  597.23171084978196176962228491902351379394531250000000,  593.48633640307400582969421520829200744628906250000000,
        589.71327949378701305249705910682678222656250000000000,  585.91270731442250507825519889593124389648437500000000,  582.08477873815331804507877677679061889648437500000000,  578.22964454536213452229276299476623535156250000000000,  574.34744763417904778179945424199104309082031250000000,  570.43832321587672140594804659485816955566406250000000,  566.50239899590383174654562026262283325195312500000000,  562.53979534126506223401520401239395141601562500000000,  558.55062543489054860401665791869163513183593750000000,  554.53499541758060331630986183881759643554687500000000,
        550.49300451804674594313837587833404541015625000000000,  546.42474517152606949821347370743751525878906250000000,  542.33030312739242617681156843900680541992187500000000,  538.20975754614107700035674497485160827636718750000000,  534.06318108608365946565754711627960205078125000000000,  529.89063998004405675601447001099586486816406250000000,  525.69219410231096389907179400324821472167968750000000,  521.46789702606224636838305741548538208007812500000000,  517.21779607144173951382981613278388977050781250000000,  512.94193234443332585215102881193161010742187500000000,
        508.64034076664319172778050415217876434326171875000000,  504.31305009606705880287336185574531555175781250000000,  499.96008293888752405109698884189128875732421875000000,  495.58145575231213797451346181333065032958984375000000,  491.17717883843096160489949397742748260498046875000000,  486.74725632903925998107297345995903015136718750000000,  482.29168616133620162145234644412994384765625000000000,  477.81046004437865803993190638720989227294921875000000,  473.30356341612980486388551071286201477050781250000000,  468.77097539091090538931894116103649139404296875000000,
        464.21266869702094481908716261386871337890625000000000,  459.62860960425518896954599767923355102539062500000000,  455.01875784100576538548921234905719757080078125000000,  450.38306650058785862711374647915363311767578125000000,  445.72148193638582824860350228846073150634765625000000,  441.03394364536279681487940251827239990234375000000000,  436.32038413942632359976414591073989868164062500000000,  431.58072880408036553490092046558856964111328125000000,  426.81489574373574669152731075882911682128906250000000,  422.02279561297962118260329589247703552246093750000000,
        417.20433143303375800314825028181076049804687500000000,  412.35939839254814387459191493690013885498046875000000,  407.48788363179102134381537325680255889892578125000000,  402.58966600919768552557798102498054504394531250000000,  397.66461584913565729948459193110466003417968750000000,  392.71259466962686701663187704980373382568359375000000,  387.73345488863679975111153908073902130126953125000000,  382.72703950739975198303000070154666900634765625000000,  377.69318176908802797697717323899269104003906250000000,  372.63170479095873588448739610612392425537109375000000,
        367.54242116791311900669825263321399688720703125000000,  362.42513254518411258686683140695095062255859375000000,  357.27962915762128659480367787182331085205078125000000,  352.10568933276726966141723096370697021484375000000000,  346.90307895460620102312532253563404083251953125000000,  341.67155088451818301109597086906433105468750000000000,  336.41084433557369948175619356334209442138671875000000,  331.12068419585858691789326258003711700439453125000000,  325.80078029600701938761631026864051818847656250000000,  320.45082661554380365487304516136646270751953125000000,
        315.07050042197641914754058234393596649169921875000000,  309.65946133582036736697773449122905731201171875000000,  304.21735031387714798256638459861278533935546875000000,  298.74378854208475786435883492231369018554687500000000,  293.23837622811294068014831282198429107666015625000000,  287.70069128254471024774829857051372528076171875000000,  282.13028787593844981529400683939456939697265625000000,  276.52669485727085429971339181065559387207031250000000,  270.88941401715464962762780487537384033203125000000000,  265.21791817676307800866197794675827026367187500000000,
        259.51164908048917823180090636014938354492187500000000,  253.77001506693866872410580981522798538208007812500000,  247.99238848878778185280680190771818161010742187500000,  242.17810284718248681201657745987176895141601562500000,  236.32644960055239380380953662097454071044921875000000,  230.43667460072538233362138271331787109375000000000000,  224.50797410078138227618183009326457977294921875000000,  218.53949026882062867116474080830812454223632812500000,  212.53030612926963272002467419952154159545898437500000,  206.47943983791995492538262624293565750122070312500000,
        200.38583817779976925521623343229293823242187500000000,  194.24836913918986169846903067082166671752929687500000,  188.06581341725049583146756049245595932006835937500000,  181.83685462298609536446747370064258575439453125000000,  175.56006795516961460634774994105100631713867187500000,  169.23390701898387078472296707332134246826171875000000,  162.85668839682568886928493157029151916503906250000000,  156.42657347139416401660128030925989151000976562500000,  149.94154686152276667598925996571779251098632812500000,  143.39939064380283184618747327476739883422851562500000,
        136.79765327825256804317177738994359970092773437500000,  130.13361180492748303549888078123331069946289062500000,  123.40422538620815373633377021178603172302246093750000,  116.60607756811738511260045925155282020568847656250000,  109.73530361515422271168063161894679069519042968750000,  102.78749776112555025520123308524489402770996093750000,   95.75759292012263301785424118861556053161621093750000,   88.63970181200917863861832302063703536987304687500000,   81.42690267326709374628990190103650093078613281250000,   74.11094306685500043840875150635838508605957031250000,
         66.68181849200750832551420899108052253723144531250000,   59.12715174225350267533940495923161506652832031250000,   51.43123926150724400940816849470138549804687500000000,   43.57350605951608457644397276453673839569091796875000,   35.52582506084678470870130695402622222900390625000000,   27.24741758541995295672677457332611083984375000000000,   18.67378794758410265330894617363810539245605468750000,    9.68719547617101994774202466942369937896728515625000,    0.00000000000000000000000000000000000000000000000000,  758.24811308137429932685336098074913024902343750000000,
        760.00129590173230553773464635014533996582031250000000,  761.01088917565050451230490580201148986816406250000000,  761.57078585044780538737541064620018005371093750000000,  761.80340362557194566761609166860580444335937500000000,  761.77686823103397273371228948235511779785156250000000,  761.53485877561547567893285304307937622070312500000000,  761.10787457976607583987060934305191040039062500000000,  760.51846844831686667021131142973899841308593750000000,  759.78401950467537062650080770254135131835937500000000,  758.91834382261879454745212569832801818847656250000000,
        757.93269593062825606466503813862800598144531250000000,  756.83642520610396786651108413934707641601562500000000,  755.63742450713493781222496181726455688476562500000000,  754.34244748287937909481115639209747314453125000000000,  752.95733949003965790325310081243515014648437500000000,  751.48720973720116944605251774191856384277343750000000,  749.93656229411567437637131661176681518554687500000000,  748.30939759699583646579412743449211120605468750000000,  746.60929233688796102796914055943489074707031250000000,  744.83946321100256682257167994976043701171875000000000,
        743.00281842676986343576572835445404052734375000000000,  741.10199977262936954502947628498077392578125000000000,  739.13941732616717672499362379312515258789062500000000,  737.11727834664043257362209260463714599609375000000000,  735.03761152380059229471953585743904113769531250000000,  732.90228748195931984810158610343933105468750000000000,  730.71303623679432348581030964851379394531250000000000,  728.47146215178418060531839728355407714843750000000000,  726.17905682724097005120711401104927062988281250000000,  723.83721026777061524626333266496658325195312500000000,
        721.44722060669050733849871903657913208007812500000000,  719.01030261344146765623008832335472106933593750000000,  716.52759516875175904715433716773986816406250000000000,  714.00016785957529918960062786936759948730468750000000,  711.42902681966972977534169331192970275878906250000000,  708.81511992063701654842589050531387329101562500000000,  706.15934140120259598916163668036460876464843750000000,  703.46253600861530230758944526314735412597656250000000,  700.72550271465695459482958540320396423339843750000000,  697.94899805936097436642739921808242797851562500000000,
        695.13373916775537963985698297619819641113281250000000,  692.28040647846114552521612495183944702148437500000000,  689.38964621755053485685493797063827514648437500000000,  686.46207264651388868514914065599441528320312500000000,  683.49827010932665416476083919405937194824218750000000,  680.49879490034936679876409471035003662109375000000000,  677.46417697201582086563576012849807739257812500000000,  674.39492149889019856345839798450469970703125000000000,  671.29151031264518678653985261917114257812500000000000,  668.15440322076130996720166876912117004394531250000000,
        664.98403922024090206832624971866607666015625000000000,  661.78083761632740333880065008997917175292968750000000,  658.54519905508050214848481118679046630859375000000000,  655.27750647768107228330336511135101318359375000000000,  651.97812600346901490411255508661270141601562500000000,  648.64740774797041922283824533224105834960937500000000,  645.28568658150641113024903461337089538574218750000000,  641.89328283339330027956748381257057189941406250000000,  638.47050294623238642088836058974266052246093750000000,  635.01764008433360686467494815587997436523437500000000,
        631.53497469991532398125855252146720886230468750000000,  628.02277506036375598341692239046096801757812500000000,  624.48129773952427967742551118135452270507812500000000,  620.91078807570443132135551422834396362304687500000000,  617.31148059882514189666835591197013854980468750000000,  613.68359942892118397139711305499076843261718750000000,  610.02735864799819864856544882059097290039062500000000,  606.34296264705915291415294632315635681152343750000000,  602.63060644996198789158370345830917358398437500000000,  598.89047601560798739228630438446998596191406250000000,
        595.12274851983806911448482424020767211914062500000000,  591.32759261828221042378572747111320495605468750000000,  587.50516869130387931363657116889953613281250000000000,  583.65562907207618081883992999792098999023437500000000,  579.77911825873695761401904746890068054199218750000000,  575.87577311148311309807468205690383911132812500000000,  571.94572303539132462901761755347251892089843750000000,  567.98909014967671282647643238306045532226562500000000,  564.00598944403975565364817157387733459472656250000000,  559.99652892268829873501090332865715026855468750000000,
        555.96080973656705737084848806262016296386718750000000,  551.89892630427561925898771733045578002929687500000000,  547.81096642210309255460742861032485961914062500000000,  543.69701136356820825312752276659011840820312500000000,  539.55713596880400473310146480798721313476562500000000,  535.39140872409166149736847728490829467773437500000000,  531.19989183180484815238742157816886901855468750000000,  526.98264127099071174598066136240959167480468750000000,  522.73970684877906478504883125424385070800781250000000,  518.47113224277518384042195975780487060546875000000000,
        514.17695503456059213931439444422721862792968750000000,  509.85720673439141137350816279649734497070312500000000,  505.51191279715396831306861713528633117675781250000000,  501.14109262960351998117403127253055572509765625000000,  496.74475958888268678492750041186809539794921875000000,  492.32292097228258853647275827825069427490234375000000,  487.87557799817852810519980266690254211425781250000000,  483.40272577803887088521150872111320495605468750000000,  478.90435327937342435689060948789119720458984375000000,  474.38044327945164013726753182709217071533203125000000,
        469.83097230958605905470903962850570678710937500000000,  465.25591058973964209144469350576400756835937500000000,  460.65522195317527121005696244537830352783203125000000,  456.02886376082562946976395323872566223144531250000000,  451.37678680501699091109912842512130737304687500000000,  446.69893520213480542224715463817119598388671875000000,  441.99524627376791841015801765024662017822265625000000,  437.26565041581744708309997804462909698486328125000000,  432.51007095499488741552340798079967498779296875000000,  427.72842399207513608416775241494178771972656250000000,
        422.92061823120172903145430609583854675292968750000000,  418.08655479446616709537920542061328887939453125000000,  413.22612702090691527700982987880706787109375000000000,  408.33922024898129166103899478912353515625000000000000,  403.42571158147137566629680804908275604248046875000000,  398.48546963167575540865072980523109436035156250000000,  393.51835424962524712100275792181491851806640625000000,  388.52421622692935443410533480346202850341796875000000,  383.50289697871795624450896866619586944580078125000000,  378.45422820098497140861582010984420776367187500000000,
        373.37803150146368125206208787858486175537109375000000,  368.27411800196597368994844146072864532470703125000000,  363.14228790989932349475566297769546508789062500000000,  357.98233005642805437673814594745635986328125000000000,  352.79402139847013586404500529170036315917968750000000,  347.57712648140920919104246422648429870605468750000000,  342.33139685905217675099265761673450469970703125000000,  337.05657046696620682268985547125339508056640625000000,  331.75237094488261391234118491411209106445312500000000,  326.41850690334399587300140410661697387695312500000000,
        321.05467112919427563610952347517013549804687500000000,  315.66053972385014958490501157939434051513671875000000,  310.23577116753693871942232362926006317138671875000000,  304.78000530180588611983694136142730712890625000000000,  299.29286222165183062315918505191802978515625000000000,  293.77394106740308643566095270216464996337890625000000,  288.22281870522169811010826379060745239257812500000000,  282.63904828351036258027306757867336273193359375000000,  277.02215765072230624355142936110496520996093750000000,  271.37164761796901757406885735690593719482421875000000,
        265.68699004735583457659231498837471008300781250000000,  259.96762574407290458111674524843692779541015625000000,  254.21296212684006832205341197550296783447265625000000,  248.42237064723428829893236979842185974121093750000000,  242.59518392357787774926691781729459762573242187500000,  236.73069254925795235067198518663644790649414062500000,  230.82814152836351695441408082842826843261718750000000,  224.88672628307776335532253142446279525756835937500000,  218.90558816700004740596341434866189956665039062500000,  212.88380940602039004261314403265714645385742187500000,
        206.82040737294087762165872845798730850219726562500000,  200.71432808294360938816680572926998138427734375000000,  194.56443877321584068340598605573177337646484375000000,  188.36951940019901030609617009758949279785156250000000,  182.12825285018715248952503316104412078857421875000000,  175.83921361089699075819225981831550598144531250000000,  169.50085458976610652825911529362201690673828125000000,  163.11149168442446466542605776339769363403320312500000,  156.66928560546074322701315395534038543701171875000000,  150.17222031194225451145030092447996139526367187500000,
        143.61807723273574310951516963541507720947265625000000,  137.00440419188333862621220760047435760498046875000000,  130.32847760492845168300846125930547714233398437500000,  123.58725602092788165009551448747515678405761718750000,  116.77732238350124305270583136007189750671386718750000,  109.89481136540565842096839332953095436096191406250000,  102.93531661911539742959575960412621498107910156250000,   95.89377048755406462987593840807676315307617187500000,   88.76428512934533898715017130598425865173339843750000,   81.53993822942840097312000580132007598876953125000000,
         74.21247680869279861326504033058881759643554687500000,   66.77189583356040714079426834359765052795410156250000,   59.20581757379277831887520733289420604705810546875000,   51.49853795837712056027157814241945743560791015625000,   43.62948149077487869362812489271163940429687500000000,   35.57052059771437058088849880732595920562744140625000,   27.28087610953702224492189998272806406021118164062500,   18.69605185908849165343781351111829280853271484375000,    9.69830670159609020686275471234694123268127441406250,    0.00000000000000000000000000000000000000000000000000,
        763.44661011264020089583937078714370727539062500000000,  765.20074950148136849747970700263977050781250000000000,  766.21139726155024618492461740970611572265625000000000,  766.77243408871868268761318176984786987304687500000000,  767.00626931257215801451820880174636840820312500000000,  766.98102265770728536153910681605339050292968750000000,  766.74036876204581858473829925060272216796875000000000,  766.31480352277901602064957842230796813964843750000000,  765.72687706673605134710669517517089843750000000000000,  764.99396638792711655696621164679527282714843750000000,
        764.12988584637662370369071140885353088378906250000000,  763.14588857957653544872300699353218078613281250000000,  762.05132282997249149048002436757087707519531250000000,  760.85408052763239084015367552638053894042968750000000,  759.56091456373121673095738515257835388183593750000000,  758.17766967879322237422456964850425720214843750000000,  756.70945458507776493206620216369628906250000000000000,  755.16077295855041029426502063870429992675781250000000,  753.53562493035303759825183078646659851074218750000000,  751.83758696401525867258897051215171813964843750000000,
        750.06987559768538176285801455378532409667968750000000,  748.23539894069995170866604894399642944335937500000000,  746.33679873817231964494567364454269409179687500000000,  744.37648507395294927846407517790794372558593750000000,  742.35666525881242705509066581726074218750000000000000,  740.27936807559933640732197090983390808105468750000000,  738.14646428020535040559479966759681701660156250000000,  735.95968405573159998311894014477729797363281250000000,  733.72063196667897955194348469376564025878906250000000,  731.43079984606015386816579848527908325195312500000000,
        729.09157796122383388137677684426307678222656250000000,  726.70426473687132329359883442521095275878906250000000,  724.27007526127647452085511758923530578613281250000000,  721.79014876043436288455268368124961853027343750000000,  719.26555519213991374272154644131660461425781250000000,  716.69730108584406025329371914267539978027343750000000,  714.08633473309225792036158964037895202636718750000000,  711.43355081630932090774876996874809265136718750000000,  708.73979454980019454524153843522071838378906250000000,  706.00586539544701736303977668285369873046875000000000,
        703.23252040619490799144841730594635009765625000000000,  700.42047724263136387889971956610679626464843750000000,  697.57041690148741963639622554183006286621093750000000,  694.68298618946062106260797008872032165527343750000000,  691.75879997120034659019438549876213073730468750000000,  688.79844321644861793174641206860542297363281250000000,  685.80247286806218198762508109211921691894531250000000,  682.77141954987064309534616768360137939453125000000000,  679.70578913094914241810329258441925048828125000000000,  676.60606416085533965087961405515670776367187500000000,
        673.47270518862956123484764248132705688476562500000000,  670.30615197685176553932251408696174621582031250000000,  667.10682462074282739195041358470916748046875000000000,  663.87512458116361813154071569442749023437500000000000,  660.61143563938094303011894226074218750000000000000000,  657.31612478060833382187411189079284667968750000000000,  653.98954301257435872685164213180541992187500000000000,  650.63202612471286556683480739593505859375000000000000,  647.24389539298488216445548459887504577636718750000000,  643.82545823483189906255574896931648254394531250000000,
        640.37700881830335220001870766282081604003906250000000,  636.89882862900367399561218917369842529296875000000000,  633.39118699814491719735087826848030090332031250000000,  629.85434159467206427507335320115089416503906250000000,  626.28853888415108031040290370583534240722656250000000,  622.69401455685044766141800209879875183105468750000000,  619.07099392722352604323532432317733764648437500000000,  615.41969230679831071029184386134147644042968750000000,  611.74031535229221390181919559836387634277343750000000,  608.03305939061203844175906851887702941894531250000000,
        604.29811172224640358763281255960464477539062500000000,  600.53565090442430118855554610490798950195312500000000,  596.74584701529477115400368347764015197753906250000000,  592.92886190026615622628014534711837768554687500000000,  589.08484940155017284268978983163833618164062500000000,  585.21395557185746838513296097517013549804687500000000,  581.31631887311345963098574429750442504882812500000000,  577.39207036098014214076101779937744140625000000000000,  573.44133385590578200208256021142005920410156250000000,  569.46422610135232389438897371292114257812500000000000,
        565.46085690979725768556818366050720214843750000000000,  561.43132929704415801097638905048370361328125000000000,  557.37573960532949968182947486639022827148437500000000,  553.29417761566423905605915933847427368164062500000000,  549.18672664979840192245319485664367675781250000000000,  545.05346366216372189228422939777374267578125000000000,  540.89445932209912371035898104310035705566406250000000,  536.70977808663383257226087152957916259765625000000000,  532.49947826406264539400581270456314086914062500000000,  528.26361206851242968696169555187225341796875000000000,
        524.00222566566912973939906805753707885742187500000000,  519.71535920979874845215817913413047790527343750000000,  515.40304687216507772973272949457168579101562500000000,  511.06531686091773281077621504664421081542968750000000,  506.70219143249056514832773245871067047119140625000000,  502.31368689452148146301624365150928497314453125000000,  497.89981360027570644888328388333320617675781250000000,  493.46057593452081846407963894307613372802734375000000,  488.99597229077357951609883457422256469726562500000000,  484.50599503980436111305607482790946960449218750000000,
        479.99063048925546581813250668346881866455078125000000,  475.44985883419275296546402387320995330810546875000000,  470.88365409837643937862594611942768096923828125000000,  466.29198406600130510923918336629867553710937500000000,  461.67481020361503851745510473847389221191406250000000,  457.03208757188724575826199725270271301269531250000000,  452.36376472685293492759228684008121490478515625000000,  447.66978361021330101721105165779590606689453125000000,  442.95007942822428503859555348753929138183593750000000,  438.20458051865114157408243045210838317871093750000000,
        433.43320820521086034204927273094654083251953125000000,  428.63587663886045220351661555469036102294921875000000,  423.81249262522436538347392342984676361083984375000000,  418.96295543737937805417459458112716674804687500000000,  414.08715661313613054517190903425216674804687500000000,  409.18497973586823945879586972296237945556640625000000,  404.25630019784540536420536227524280548095703125000000,  399.30098494491926430782768875360488891601562500000000,  394.31889220129653494950616732239723205566406250000000,  389.30987117300230693217599764466285705566406250000000,
        384.27376172849670865616644732654094696044921875000000,  379.21039405474721206701360642910003662109375000000000,  374.11958828688449330002185888588428497314453125000000,  369.00115410937138449298799969255924224853515625000000,  363.85489032639617335007642395794391632080078125000000,  358.68058439895469291514018550515174865722656250000000,  353.47801194581109029968501999974250793457031250000000,  348.24693620521321690830518491566181182861328125000000,  342.98710745389303156116511672735214233398437500000000,  337.69826237948274183509056456387042999267578125000000,
        332.38012340203317762643564492464065551757812500000000,  327.03239793980873173495638184249401092529296875000000,  321.65477761395732159144245088100433349609375000000000,  316.24693738599216885631904006004333496093750000000000,  310.80853462126754038763465359807014465332031250000000,  305.33920807076339087871019728481769561767578125000000,  299.83857676249755286335130222141742706298828125000000,  294.30623879273491638741688802838325500488281250000000,  288.74177000583250674026203341782093048095703125000000,  283.14472255001351186365354806184768676757812500000000,
        277.51462329456740008026827126741409301757812500000000,  271.85097209186881173081928864121437072753906250000000,  266.15323986514533771696733310818672180175781250000000,  260.42086650002005399073823355138301849365234375000000,  254.65325851442659654821909498423337936401367187500000,  248.84978647742573798495868686586618423461914062500000,  243.00978214259899345961457584053277969360351562500000,  237.13253525589104242499161045998334884643554687500000,  231.21728999078516153531381860375404357910156250000000,  225.26324095524955737346317619085311889648437500000000,
        219.26952870462787359429057687520980834960937500000000,  213.23523468209660336469823960214853286743164062500000,  207.15937549288281616099993698298931121826171875000000,  201.04089639934144884136912878602743148803710937500000,  194.87866390020278117845009546726942062377929687500000,  188.67145722745507896433991845697164535522460937500000,  182.41795855658918412700586486607789993286132812500000,  176.11674167782521749359148088842630386352539062500000,  169.76625881407835549907758831977844238281250000000000,  163.36482519110865041511715389788150787353515625000000,
        156.91060085997486339692841283977031707763671875000000,  150.40156913225234802666818723082542419433593750000000,  143.83551080106045105821976903825998306274414062500000,  137.20997306615419120134902186691761016845703125000000,  130.52223172997440769904642365872859954833984375000000,  123.76924473939254767174134030938148498535156250000000,  116.94759444650057389480934944003820419311523437500000,  110.05341494293384130287449806928634643554687500000000,  103.08229931020643732608732534572482109069824218750000,   96.02917933020614782435586676001548767089843750000000,
         88.88816661021500920014659641310572624206542968750000,   81.65233829330098558330064406618475914001464843750000,   74.31344086773576407267682952806353569030761718750000,   66.86146877636566898672754177823662757873535156250000,   59.28404377421412618787144310772418975830078125000000,   51.56546128416840701902401633560657501220703125000000,   43.68514531203470596665283665060997009277343750000000,   35.61496779514258292920203530229628086090087890625000,   27.31414908250831530267532798461616039276123046875000,   18.71819253354813028522585227619856595993041992187500,
          9.70935653778267493407838628627359867095947265625000,    0.00000000000000000000000000000000000000000000000000,  768.65061679971688590740086510777473449707031250000000,  770.40570659453885582479415461421012878417968750000000,  771.41740185877938529301900416612625122070312500000000,  771.97957113005259088822640478610992431640625000000000,  772.21461542906592967483447864651679992675781250000000,  772.19064851691325657156994566321372985839843750000000,  771.95134059043209617811953648924827575683593750000000,  771.52718414515982203738531097769737243652343750000000,
        770.94072664576253828272456303238868713378906250000000,  770.20934296846508004819042980670928955078125000000000,  769.34684576795120847236830741167068481445312500000000,  768.36448679663840266584884375333786010742187500000000,  767.27161316594117579370504245162010192871093750000000,  766.07611588020120052533457055687904357910156250000000,  764.78474707354939710057806223630905151367187500000000,  763.40335087012056192179443314671516418457031250000000,  761.93703548464452524058287963271141052246093750000000,  760.39030419722791975800646468997001647949218750000000,
        758.76715683109705423703417181968688964843750000000000,  757.07116961872554838919313624501228332519531250000000,  755.30555893502469189115799963474273681640625000000000,  753.47323278648082123254425823688507080078125000000000,  751.57683286957785639970097690820693969726562500000000,  749.61876926861748415831243619322776794433593750000000,  747.60124933958206838724436238408088684082031250000000,  745.52630195165136228752089664340019226074218750000000,  743.39579798508066232898272573947906494140625000000000,  741.21146778274248845264082774519920349121093750000000,
        738.97491610207953272038139402866363525390625000000000,  736.68763500030615887226304039359092712402343750000000,  734.35101499859797513636294752359390258789062500000000,  731.96635480371310222835745662450790405273437500000000,  729.53486981301807645650114864110946655273437500000000,  727.05769958762255100737093016505241394042968750000000,  724.53591444559594947349978610873222351074218750000000,  721.97052130109716472361469641327857971191406250000000,  719.36246885420723629067651927471160888671875000000000,  716.71265221921305510477395728230476379394531250000000,
        714.02191706520284242287743836641311645507812500000000,  711.29106333144318341510370373725891113281250000000000,  708.52084857062106948433211073279380798339843750000000,  705.71199096525106142507866024971008300781250000000000,  702.86517205606821789842797443270683288574218750000000,  699.98103921580297992477426305413246154785156250000000,  697.06020789717331354040652513504028320312500000000000,  694.10326368008406916487729176878929138183593750000000,  691.11076413975717969151446595788002014160156250000000,  688.08324055474429314926965162158012390136718750000000,
        685.02119947139726718887686729431152343750000000000000,  681.92512413934639425860950723290443420410156250000000,  678.79547583078317529725609347224235534667968750000000,  675.63269505483947341417660936713218688964843750000000,  672.43720267705043625028338283300399780273437500000000,  669.20940095275386738649103790521621704101562500000000,  665.94967448229567708040121942758560180664062500000000,  662.65839109504690895846579223871231079101562500000000,  659.33590266848818828293588012456893920898437500000000,  655.98254588795373365428531542420387268066406250000000,
        652.59864295204556583485100418329238891601562500000000,  649.18450222822002615430392324924468994140625000000000,  645.74041886258703470957698300480842590332031250000000,  642.26667534757143585011363029479980468750000000000000,  638.76354205072038894286379218101501464843750000000000,  635.23127770762869204190792515873908996582031250000000,  631.67012988167016374063678085803985595703125000000000,  628.08033539296798153372947126626968383789062500000000,  624.46212071881654992466792464256286621093750000000000,  620.81570236755578662268817424774169921875000000000000,
        617.14128722772693436127156019210815429687500000000000,  613.43907289416290495864814147353172302246093750000000,  609.70924797253007909603184089064598083496093750000000,  605.95199236369364825804950669407844543457031250000000,  602.16747752916467106842901557683944702148437500000000,  598.35586673877230623475043103098869323730468750000000,  594.51731530160714100929908454418182373046875000000000,  590.65197078118751505826367065310478210449218750000000,  586.75997319571945354255149140954017639160156250000000,  582.84145520424215192178962752223014831542968750000000,
        578.89654227938160602207062765955924987792968750000000,  574.92535286737245314725441858172416687011718750000000,  570.92799853594226533459732308983802795410156250000000,  566.90458411060387788893422111868858337402343750000000,  562.85520779984597083966946229338645935058593750000000,  558.77996130966528198769083246588706970214843750000000,  554.67892994783994708996033295989036560058593750000000,  550.55219271829889748914865776896476745605468750000000,  546.39982240590632045496022328734397888183593750000000,  542.22188565193857812118949368596076965332031250000000,
        538.01844302049994439585134387016296386718750000000000,  533.78954905608395620220107957720756530761718750000000,  529.53525233246034531475743278861045837402343750000000,  525.25559549303056883218232542276382446289062500000000,  520.95061528276744411414256319403648376464843750000000,  516.62034257182369856309378519654273986816406250000000,  512.26480237086389024625532329082489013671875000000000,  507.88401383814488099233130924403667449951171875000000,  503.47799027834196294861612841486930847167968750000000,  499.04673913308641886033001355826854705810546875000000,
        494.59026196315295464955852366983890533447265625000000,  490.10855442220258737506810575723648071289062500000000,  485.60160622195883206586586311459541320800781250000000,  481.06940108865893535039504058659076690673828125000000,  476.51191671059461896220454946160316467285156250000000,  471.92912467651592578476993367075920104980468750000000,  467.32099040464140671247150748968124389648437500000000,  462.68747306197587931819725781679153442382812500000000,  458.02852547359981372210313566029071807861328125000000,  453.34409402154915369465015828609466552734375000000000,
        448.63411853286004316032631322741508483886718750000000,  443.89853215630489557952387258410453796386718750000000,  439.13726122729065082239685580134391784667968750000000,  434.35022512033742714265827089548110961914062500000000,  429.53733608848983749339822679758071899414062500000000,  424.69849908894985901497420854866504669189453125000000,  419.83361159414499752529081888496875762939453125000000,  414.94256338736749967210926115512847900390625000000000,  410.02523634203106439599650911986827850341796875000000,  405.08150418349850951926782727241516113281250000000000,
        400.11123223232493728573899716138839721679687500000000,  395.11427712764896114094881340861320495605468750000000,  390.09048652933057610425748862326145172119140625000000,  385.03969879729737613160978071391582489013671875000000,  379.96174264639716966485138982534408569335937500000000,  374.85643677488428693322930485010147094726562500000000,  369.72358946446479421865660697221755981445312500000000,  364.56299814961147376379813067615032196044921875000000,  359.37444895360960117614013142883777618408203125000000,  354.15771618852159008383750915527343750000000000000000,
        348.91256181594553709146566689014434814453125000000000,  343.63873486509487520379479974508285522460937500000000,  338.33597080432917891812394373118877410888671875000000,  333.00399086182079599893768317997455596923828125000000,  327.64250129052976490129367448389530181884765625000000,  322.25119257208456247099093161523342132568359375000000,  316.82973855350360281590837985277175903320312500000000,  311.37779550993894872590317390859127044677734375000000,  305.89500112575518642188399098813533782958984375000000,  300.38097338526171142802922986447811126708984375000000,
        294.83530936326627625021501444280147552490234375000000,  289.25758390428785560288815759122371673583984375000000,  283.64734817772074393360526300966739654541015625000000,  278.00412809444623007948393933475017547607421875000000,  272.32742256828373683674726635217666625976562500000000,  266.61670160321023104188498109579086303710937500000000,  260.87140418437348898805794306099414825439453125000000,  255.09093594749600697468849830329418182373046875000000,  249.27466659719723907073785085231065750122070312500000,  243.42192703991037205923930741846561431884765625000000,
        237.53200619126323545060586184263229370117187500000000,  231.60414741080779776893905363976955413818359375000000,  225.63754450853537036891793832182884216308593750000000,  219.63133725734991230638115666806697845458984375000000,  213.58460633312176923936931416392326354980468750000000,  207.49636758851553963722835760563611984252929687500000,  201.36556554768876026173529680818319320678710937500000,  195.19106598517402062498149462044239044189453125000000,  188.97164742240750001656124368309974670410156250000000,  182.70599133763010968323214910924434661865234375000000,
        176.39267083678240055633068550378084182739257812500000,  170.03013747114809461891127284616231918334960937500000,  163.61670580719203371700132265686988830566406250000000,  157.15053524871183299183030612766742706298828125000000,  150.62960847176239553846244234591722488403320312500000,  144.05170564539923816482769325375556945800781250000000,  137.41437335649450801611237693578004837036132812500000,  130.71488680551985339661769103258848190307617187500000,  123.95020334803170669601968256756663322448730468750000,  117.11690475520951792987034423276782035827636718750000,
        110.21112454793444612732855603098869323730468750000000,  103.22845524688750629138667136430740356445312500000000,   96.16382808281436211927939439192414283752441406250000,   89.01135412132227031634101876989006996154785156250000,   81.76410997305471539675636449828743934631347656250000,   74.41384160289827320866606896743178367614746093750000,   66.95054293914860465974925318732857704162597656250000,   59.36183523091384017789096105843782424926757812500000,   51.63201340360784286076523130759596824645996093750000,   43.74050097381712731703373719938099384307861328125000,
         35.65916939772250060514124925248324871063232421875000,   27.34723855108457613027894694823771715164184570312500,   18.74021132778410958508175099268555641174316406250000,    9.72034565935827110649825044674798846244812011718750,    0.00000000000000000000000000000000000000000000000000,  773.86010295255834989802679046988487243652343750000000,  775.61613706485206876095617190003395080566406250000000,  776.62887293682945255568483844399452209472656250000000,  777.19216703984841387864435091614723205566406250000000,  777.42841214598001897684298455715179443359375000000000,
        777.40571609424068810767494142055511474609375000000000,  777.16774466983747515769209712743759155273437500000000,  776.74498698804507057502632960677146911621093750000000,  776.15998786706927603518124669790267944335937500000000,  775.43012007688275843975134193897247314453125000000000,  774.56919457520314153953222557902336120605468750000000,  773.58846173528843337408034130930900573730468750000000,  772.49726754147047813603421673178672790527343750000000,  771.30350207470951318100560456514358520507812500000000,  770.01391671310102537972852587699890136718750000000000,
        768.63435496426620829879539087414741516113281250000000,  767.16992454430396719544660300016403198242187500000000,  765.62512833550601953902514651417732238769531250000000,  764.00396585046792097273282706737518310546875000000000,  762.31001308720817633002297952771186828613281250000000,  760.54648625338143119734013453125953674316406250000000,  758.71629324802916016778908669948577880859375000000000,  756.82207571387971256626769900321960449218750000000000,  754.86624373005554389237659052014350891113281250000000,  752.85100469164615333284018561244010925292968750000000,
        750.77838754760205119964666664600372314453125000000000,  748.65026329554143558198120445013046264648437500000000,  746.46836243068219118868000805377960205078125000000000,  744.23428989556543911021435633301734924316406250000000,  741.94953796335425977304112166166305541992187500000000,  739.61549740039356493070954456925392150878906250000000,  737.23346718643858821451431140303611755371093750000000,  734.80466301848889543180121108889579772949218750000000,  732.33022478290695289615541696548461914062500000000000,  729.81122314777223891724133864045143127441406250000000,
        727.24866540128118685970548540353775024414062500000000,  724.64350064096902315213810652494430541992187500000000,  721.99662440148779296578140929341316223144531250000000,  719.30888279478904223651625216007232666015625000000000,  716.58107622517377421900164335966110229492187500000000,  713.81396273228210702654905617237091064453125000000000,  711.00826100731876522331731393933296203613281250000000,  708.16465312132766030117636546492576599121093750000000,  705.28378699890618008794263005256652832031250000000000,  702.36627866619301130413077771663665771484375000000000,
        699.41271429811251891806023195385932922363281250000000,  696.42365208659953168535139411687850952148437500000000,  693.39962394874942219757940620183944702148437500000000,  690.34113709147300141921732574701309204101562500000000,  687.24867544719757006532745435833930969238281250000000,  684.12270099341401419223984703421592712402343750000000,  680.96365496735973010800080373883247375488281250000000,  677.77195898582317568070720881223678588867187500000000,  674.54801607892193260340718552470207214355468750000000,  671.29221164572493307787226513028144836425781250000000,
        668.00491433872332436294527724385261535644531250000000,  664.68647688340479362523183226585388183593750000000000,  661.33723683852531394222751259803771972656250000000000,  657.95751730208951357781188562512397766113281250000000,  654.54762756753939356713090091943740844726562500000000,  651.10786373419591654965188354253768920898437500000000,  647.63850927560247328074183315038681030273437500000000,  644.13983556905282057414297014474868774414062500000000,  640.61210238928049420792376622557640075683593750000000,  637.05555836899623045610496774315834045410156250000000,
        633.47044142870856830995762720704078674316406250000000,  629.85697917804077405889984220266342163085937500000000,  626.21538929054997879575239494442939758300781250000000,  622.54587985387274784443434327840805053710937500000000,  618.84864969686134372750530019402503967285156250000000,  615.12388869522089862584834918379783630371093750000000,  611.37177805702822297462262213230133056640625000000000,  607.59249058938894449966028332710266113281250000000000,  603.78619094738257899734890088438987731933593750000000,  599.95303586634042858349857851862907409667968750000000,
        596.09317437841571063472656533122062683105468750000000,  592.20674801431664491246920078992843627929687500000000,  588.29389099099921622837428003549575805664062500000000,  584.35473038604766315984306856989860534667968750000000,  580.38938629940241753502050414681434631347656250000000,  576.39797200304064972442574799060821533203125000000000,  572.38059407915432075242279097437858581542968750000000,  568.33735254732425801194040104746818542480468750000000,  564.26834098113806703622685745358467102050781250000000,  560.17364661465626340941525995731353759765625000000000,
        556.05335043909190062549896538257598876953125000000000,  551.90752729002417709125438705086708068847656250000000,  547.73624592543501421459950506687164306640625000000000,  543.53956909482076298445463180541992187500000000000000,  539.31755359959572615480283275246620178222656250000000,  535.07025034497428350732661783695220947265625000000000,  530.79770438348873540235217660665512084960937500000000,  526.49995495026428216078784316778182983398437500000000,  522.17703549015050157322548329830169677734375000000000,  517.82897367677537658892106264829635620117187500000000,
        513.45579142355973090161569416522979736328125000000000,  509.05750488670560116588603705167770385742187500000000,  504.63412446013722956195124424993991851806640625000000,  500.18565476235210098820971325039863586425781250000000,  495.71209461510551363971899263560771942138671875000000,  491.21343701382562585422419942915439605712890625000000,  486.68966908962369188884622417390346527099609375000000,  482.14077206273515230350312776863574981689453125000000,  477.56672118719382069684797897934913635253906250000000,  472.96748568650684774183901026844978332519531250000000,
        468.34302868006471953776781447231769561767578125000000,  463.69330709998024531159899197518825531005859375000000,  459.01827159801456446075462736189365386962890625000000,  454.31786644220170501284883357584476470947265625000000,  449.59202940274133197817718610167503356933593750000000,  444.84069162667901764507405459880828857421875000000000,  440.06377750084146782683092169463634490966796875000000,  435.26120450243831783154746517539024353027343750000000,  430.43288303667947047870256938040256500244140625000000,  425.57871626069203330189338885247707366943359375000000,
        420.69859989294616298138862475752830505371093750000000,  415.79242200732232959126122295856475830078125000000000,  410.86006281086332592167309485375881195068359375000000,  405.90139440415924809713033027946949005126953125000000,  400.91628052320942288133664987981319427490234375000000,  395.90457626148838699009502306580543518066406250000000,  390.86612777081353442554245702922344207763671875000000,  385.80077193947221303460537455976009368896484375000000,  380.70833604590501408893032930791378021240234375000000,  375.58863738606862625601934269070625305175781250000000,
        370.44148287240312811263720504939556121826171875000000,  365.26666860211059884022688493132591247558593750000000,  360.06397939220551052130758762359619140625000000000000,  354.83318827852241383880027569830417633056640625000000,  349.57405597555407439358532428741455078125000000000000,  344.28633029364613094003289006650447845458984375000000,  338.96974550967672712431522086262702941894531250000000,  333.62402168690238113413215614855289459228515625000000,  328.24886393914465543275582604110240936279296875000000,  322.84396163391096479244879446923732757568359375000000,
        317.40898752838353402694338001310825347900390625000000,  311.94359683145825101746595464646816253662109375000000,  306.44742618414335311172180809080600738525390625000000,  300.92009254963653575032367371022701263427734375000000,  295.36119200324640132748754695057868957519531250000000,  289.77029841099528084669145755469799041748046875000000,  284.14696198419557049419381655752658843994140625000000,  278.49070769549354054106515832245349884033203125000000,  272.80103353977210645098239183425903320312500000000000,  267.07740862084142463572788983583450317382812500000000,
        261.31927104194068078868440352380275726318359375000000,  255.52602557464854271529475226998329162597656250000000,  249.69704107672802706474612932652235031127929687500000,  243.83164762458218888241390231996774673461914062500000,  237.92913332018918026733445003628730773925781250000000,  231.98874072540073143500194419175386428833007812500000,  226.00966286804015226152841933071613311767578125000000,  219.99103875397244678424613084644079208374023437500000,  213.93194830676776518885162658989429473876953125000000,  207.83140664115128970479418057948350906372070312500000,
        201.68835755733644532483594957739114761352539062500000,  195.50166611955276607659470755606889724731445312500000,  189.27011015223158096887345891445875167846679687500000,  182.99237044957502007491711992770433425903320312500000,  176.66701944612901797881931997835636138916015625000000,  170.29250803411508741191937588155269622802734375000000,  163.86715013296588949742726981639862060546875000000000,  157.38910451118394462355354335159063339233398437500000,  150.85635322098193000783794559538364410400390625000000,  144.26667581875062751350924372673034667968750000000000,
        137.61761828960743514471687376499176025390625000000000,  130.90645524292008872180304024368524551391601562500000,  124.13014345354029899226588895544409751892089843750000,  117.28526412209772900041571119800209999084472656250000,  110.36795020884171947272989200428128242492675781250000,  103.37379368350919150998379336670041084289550781250000,   96.29772523536605888239137129858136177062988281250000,   89.13385539778704469426884315907955169677734375000000,   81.87526025821954078764974838122725486755371093750000,   74.51368526718555074239702662453055381774902343750000,
         67.03912384724876005748228635638952255249023437500000,   59.43919675022435455957747763022780418395996093750000,   51.69819841248430947189262951724231243133544921875000,   43.79555186964149271489077364094555377960205078125000,   35.70312810479354936887830263003706932067871093750000,   27.38014652833554407607152825221419334411621093750000,   18.76210957633216835915845877025276422500610351562500,    9.73127472989046005125146621139720082283020019531250,    0.00000000000000000000000000000000000000000000000000,  779.07503871016729135590139776468276977539062500000000,
        780.83201112412484690139535814523696899414062500000000,  781.84578079142897877318318933248519897460937500000000,  782.41019220801729261438595131039619445800781250000000,  782.64762995683372537314426153898239135742187500000000,  782.62619599576566997711779549717903137207031250000000,  782.38955172752616817888338118791580200195312500000000,  781.96818290830651676515117287635803222656250000000000,  781.38463172542014945065602660179138183593750000000000,  780.65626885404151380498660728335380554199218750000000,  779.79690356326000255648978054523468017578125000000000,
        778.81778485308495874051004648208618164062500000000000,  777.72825758473891255562193691730499267578125000000000,  776.53621091819218236196320503950119018554687500000000,  775.24839547657961702498141676187515258789062500000000,  773.87065415096731157973408699035644531250000000000000,  772.40809415782132418826222419738769531250000000000000,  770.86521797977400183299323543906211853027343750000000,  769.24602481619967875303700566291809082031250000000000,  767.55409042740006952953990548849105834960937500000000,  765.79263084990031984489178285002708435058593750000000,
        763.96455387086211885616648942232131958007812500000000,  762.07250107429911167855607345700263977050781250000000,  760.11888252870505766622954979538917541503906250000000,  758.10590566236021459189942106604576110839843750000000,  756.03559949762313863175222650170326232910156250000000,  753.90983514268441467720549553632736206054687500000000,  751.73034323789886457234388217329978942871093750000000,  749.49872890329197616665624082088470458984375000000000,  747.21648461996073820046149194240570068359375000000000,  744.88500139101142849540337920188903808593750000000000,
        742.50557846039555442985147237777709960937500000000000,  740.07943181555594946985365822911262512207031250000000,  737.60770165852773061487823724746704101562500000000000,  735.09145899742793517361860722303390502929687500000000,  732.53171148412400270899524912238121032714843750000000,  729.92940860284011250769253820180892944335937500000000,  727.28544629742384586279513314366340637207031250000000,  724.60067111111152371449861675500869750976562500000000,  721.87588390124221859878161922097206115722656250000000,  719.11184318198797882359940558671951293945312500000000,
        716.30926814038866723421961069107055664062500000000000,  713.46884136449727975559653714299201965332031250000000,  710.59121131702454476908314973115921020507812500000000,  707.67699458330901052249828353524208068847656250000000,  704.72677791859541684971190989017486572265625000000000,  701.74112011633849306235788390040397644042968750000000,  698.72055371647945776203414425253868103027343750000000,  695.66558657026655509980628266930580139160156250000000,  692.57670327616472150111803784966468811035156250000000,  689.45436649964972275483887642621994018554687500000000,
        686.29901818817609182588057592511177062988281250000000,  683.11108069130398234847234562039375305175781250000000,  679.89095779483636761142406612634658813476562500000000,  676.63903567683678375033196061849594116210937500000000,  673.35568379253015791618963703513145446777343750000000,  670.04125569434449971595313400030136108398437500000000,  666.69608979268593884626170620322227478027343750000000,  663.32051006245694679819280281662940979003906250000000,  659.91482669982121933571761474013328552246093750000000,  656.47933673325849213142646476626396179199218750000000,
        653.01432459255784124252386391162872314453125000000000,  649.52006263903592753194971010088920593261718750000000,  645.99681165995605169882765039801597595214843750000000,  642.44482132983443989360239356756210327148437500000000,  638.86433064107518475793767720460891723632812500000000,  635.25556830614232239895500242710113525390625000000000,  631.61875313328027914394624531269073486328125000000000,  627.95409437760861237620702013373374938964843750000000,  624.26179206925507969572208821773529052734375000000000,  620.54203732004032190161524340510368347167968750000000,
        616.79501261009886547981295734643936157226562500000000,  613.02089205569211571855703368782997131347656250000000,  609.21984165936760291515383869409561157226562500000000,  605.39201954351335643877973780035972595214843750000000,  601.53757616826680987287545576691627502441406250000000,  597.65665453465226164553314447402954101562500000000000,  593.74939037375247607997152954339981079101562500000000,  589.81591232263747315300861373543739318847656250000000,  585.85634208772205511195352301001548767089843750000000,  581.87079459615551968454383313655853271484375000000000,
        577.85937813579789690265897661447525024414062500000000,  573.82219448428156738373218104243278503417968750000000,  569.75933902761494209698867052793502807617187500000000,  565.67090086873622567509301006793975830078125000000000,  561.55696292638776867534033954143524169921875000000000,  557.41760202464126905397279188036918640136718750000000,  553.25288897336531590553931891918182373046875000000000,  549.06288863989823312294902279973030090332031250000000,  544.84766001214904918015236034989356994628906250000000,  540.60725625332315757987089455127716064453125000000000,
        536.34172474843683176004560664296150207519531250000000,  532.05110714275519967486616224050521850585937500000000,  527.73543937226065736467717215418815612792968750000000,  523.39475168623107492749113589525222778320312500000000,  519.02906866197724866651697084307670593261718750000000,  514.63840921176608844689326360821723937988281250000000,  510.22278658192487910127965733408927917480468750000000,  505.78220834409677308940445072948932647705078125000000,  501.31667637858993202826241031289100646972656250000000,  496.82618684973459721732069738209247589111328125000000,
        492.31073017313360651314724236726760864257812500000000,  487.77029097466282792083802632987499237060546875000000,  483.20484804104660270240856334567070007324218750000000,  478.61437426180367538108839653432369232177734375000000,  473.99883656232270823238650336861610412597656250000000,  469.35819582779373604353168047964572906494140625000000,  464.69240681768519607430789619684219360351562500000000,  460.00141807041438823944190517067909240722656250000000,  455.28517179782107859864481724798679351806640625000000,  450.54360376900501705677015706896781921386718750000000,
        445.77664318304334756248863413929939270019531250000000,  440.98421253004977415912435390055179595947265625000000,  436.16622743998271971577196381986141204833984375000000,  431.32259651854712956264847889542579650878906250000000,  426.45322116946954338345676660537719726562500000000000,  421.55799540235267386378836818039417266845703125000000,  416.63680562523785511075402610003948211669921875000000,  411.68953042091521865586400963366031646728515625000000,  406.71604030592749268180341459810733795166015625000000,  401.71619747110713660731562413275241851806640625000000,
        396.68985550237238157933461479842662811279296875000000,  391.63685908037638228051946498453617095947265625000000,  386.55704365746470330122974701225757598876953125000000,  381.45023511023629225746844895184040069580078125000000,  376.31624936582755935887689702212810516357421875000000,  371.15489199984472179494332522153854370117187500000000,  365.96595780364663141881464980542659759521484375000000,  360.74923031843877652136143296957015991210937500000000,  355.50448133336050204889033921062946319580078125000000,  350.23147034443883285348420031368732452392578125000000,
        344.92994397093139014032203704118728637695312500000000,  339.59963532518651163627509959042072296142578125000000,  334.24026333170058933319523930549621582031250000000000,  328.85153199054468586837174370884895324707031250000000,  323.43312957975354038353543728590011596679687500000000,  317.98472779060972470688284374773502349853515625000000,  312.50598078900219434217433445155620574951171875000000,  306.99652419517030921269906684756278991699218750000000,  301.45597397314895715680904686450958251953125000000000,  295.88392522008001606081961654126644134521484375000000,
        290.27995084422781246757949702441692352294921875000000,  284.64360011898730817847535945475101470947265625000000,  278.97439709838022281473968178033828735351562500000000,  273.27183887742864953906973823904991149902343750000000,  267.53539367833485584924346767365932464599609375000000,  261.76449874048910260171396657824516296386718750000000,  255.95855798890357846175902523100376129150390625000000,  250.11693945159558438717795070260763168334960937500000,  244.23897239159813921105524059385061264038085937500000,  238.32394411346368201520817819982767105102539062500000,
        232.37109639714574882418673951178789138793945312500000,  226.37962150369378377945395186543464660644531250000000,  220.34865768693302356950880493968725204467773437500000,  214.27728413275028174211911391466856002807617187500000,  208.16451523217813246446894481778144836425781250000000,  202.00929407537432780372910201549530029296875000000000,  195.81048502980678449603146873414516448974609375000000,  189.56686523610702010955719742923974990844726562500000,  183.27711481731773801584495231509208679199218750000000,  176.93980554915376046665187459439039230346679687500000,
        170.55338767703184998936194460839033126831054687500000,  164.11617448531322338567406404763460159301757812500000,  157.62632411887818761897506192326545715332031250000000,  151.08181801749111627941601909697055816650390625000000,  144.48043513600097753624140750616788864135742187500000,  137.81972086863083859498146921396255493164062500000000,  131.09694924424988471400865819305181503295898437500000,  124.30907646736301330747664906084537506103515625000000,  117.45268317816778846918168710544705390930175781250000,  110.52390178616610683093313127756118774414062500000000,
        103.51832371980869140770664671435952186584472656250000,   96.43087913631946150871954159811139106750488281250000,   89.25567804606471611350571038201451301574707031250000,   81.98579602231201590711862081661820411682128906250000,   74.61297801003330221192300086840987205505371093750000,   67.12721693467851480363606242462992668151855468750000,   59.51613305919742913374648196622729301452636718750000,   51.76402033916220091214199783280491828918457031250000,   43.85030133727378398589280550368130207061767578125000,   35.74684657143295396508619887754321098327636718750000,
         27.41287499438494634773633151780813932418823242187500,   18.78388859192806847886458854191005229949951171875000,    9.74214440212736398905235546408221125602722167968750,    0.00000000000000000000000000000000000000000000000000,  784.29539453524557757191359996795654296875000000000000,  786.05329930649736525083426386117935180664062500000000,  787.06809603925864848861237987875938415527343750000000,  787.63361734373665967723354697227478027343750000000000,  787.87223967253896717011230066418647766113281250000000,  787.85205914290224882279289886355400085449218750000000,
        787.61673280386776241357438266277313232421875000000000,  787.19674307351499464857624843716621398925781250000000,  786.61462952369925005768891423940658569335937500000000,  785.88776074616987443732796236872673034667968750000000,  785.02994432968921501014847308397293090820312500000000,  784.05242790692329890589462593197822570800781250000000,  782.96455521998041149345226585865020751953125000000000,  781.77421451027692000934621319174766540527343750000000,  780.48815564712504055933095514774322509765625000000000,  779.11222090507499160594306886196136474609375000000000,
        777.65151700005105794843984767794609069824218750000000,  776.11054601329112756502581760287284851074218750000000,  774.49330682847767093335278332233428955078125000000000,  772.80337496506604111345950514078140258789062500000000,  771.04396628472147767752176150679588317871093750000000,  769.21798845844398329063551500439643859863281250000000,  767.32808300673718804318923503160476684570312500000000,  765.37665998218892582372063770890235900878906250000000,  763.36592684053641733044059947133064270019531250000000,  761.29791267137704835477052256464958190917968750000000,
        759.17448868688404672866454347968101501464843750000000,  756.99738566555276975122978910803794860839843750000000,  754.76820889749888010555878281593322753906250000000000,  752.48845106397084236959926784038543701171875000000000,  750.15950339666744639544049277901649475097656250000000,  747.78266539518813260656315833330154418945312500000000,  745.35915332848912839835975319147109985351562500000000,  742.89010770496963687037350609898567199707031250000000,  740.37659986309517989866435527801513671875000000000000,  737.81963780833041255391435697674751281738281250000000,
        735.22017140112825472897384315729141235351562500000000,  732.57909698368007411772850900888442993164062500000000,  729.89726151925674457743298262357711791992187500000000,  727.17546630658080175635404884815216064453125000000000,  724.41447032229029900918249040842056274414062500000000,  721.61499323677287520695244893431663513183593750000000,  718.77771814217419432679889723658561706542968750000000,  715.90329402596091767918551340699195861816406250000000,  712.99233801886339279008097946643829345703125000000000,  710.04543744217460243817185983061790466308593750000000,
        707.06315167612217464920831844210624694824218750000000,  704.04601386825493136711884289979934692382812500000000,  700.99453249841701563127571716904640197753906250000000,  697.90919281484957537031732499599456787109375000000000,  694.79045815421523002441972494125366210937500000000000,  691.63877115683442298177396878600120544433593750000000,  688.45455488711604630225338041782379150390625000000000,  685.23821386803479072113987058401107788085937500000000,  681.99013503752235010324511677026748657226562500000000,  678.71068863377911384304752573370933532714843750000000,
        675.40022901575957803288474678993225097656250000000000,  672.05909542442577730980701744556427001953125000000000,  668.68761268978028056153561919927597045898437500000000,  665.28609188817983977060066536068916320800781250000000,  661.85483095397512443014420568943023681640625000000000,  658.39411524912577533541480079293251037597656250000000,  654.90421809407791897683637216687202453613281250000000,  651.38540126287921339098829776048660278320312500000000,  647.83791544522318872623145580291748046875000000000000,  644.26200067785885039484128355979919433593750000000000,
        640.65788674758277920773252844810485839843750000000000,  637.02579356782132435910170897841453552246093750000000,  633.36593153063074623787542805075645446777343750000000,  629.67850183578491396474419161677360534667968750000000,  625.96369679846145572810200974345207214355468750000000,  622.22170013691368239960866048932075500488281250000000,  618.45268724139043570175999775528907775878906250000000,  614.65682542545482647255994379520416259765625000000000,  610.83427416075630844716215506196022033691406250000000,  606.98518529621856032463256269693374633789062500000000,
        603.10970326251947426499100401997566223144531250000000,  599.20796526267031367751769721508026123046875000000000,  595.28010144942493298003682866692543029785156250000000,  591.32623509018935692438390105962753295898437500000000,  587.34648272004233149345964193344116210937500000000000,  583.34095428342391187470639124512672424316406250000000,  579.30975326499674338265322148799896240234375000000000,  575.25297681013967121543828397989273071289062500000000,  571.17071583548897706350544467568397521972656250000000,  567.06305512990229544811882078647613525390625000000000,
        562.93007344618001752678537741303443908691406250000000,  558.77184358384624829341191798448562622070312500000000,  554.58843246325386644457466900348663330078125000000000,  550.37990119124845023179659619927406311035156250000000,  546.14630511859252237627515569329261779785156250000000,  541.88769388932428228144999593496322631835937500000000,  537.60411148219498045364161953330039978027343750000000,  533.29559624430021358421072363853454589843750000000000,  528.96218091699643082392867654561996459960937500000000,  524.60389265416415582876652479171752929687500000000000,
        520.22075303285384961782256141304969787597656250000000,  515.81277805632817035075277090072631835937500000000000,  511.37997814947931374263134784996509552001953125000000,  506.92235814658511117158923298120498657226562500000000,  502.43991727133038693864364176988601684570312500000000,  497.93