#cython: language_level=3

from obitools3.apps.progress cimport ProgressBar  # @UnresolvedImport
from obitools3.dms import DMS
from obitools3.dms.view.view cimport View, Line_selection
from obitools3.uri.decode import open_uri
from obitools3.apps.optiongroups import addMinimalInputOption, addTaxonomyOption, addMinimalOutputOption, addNoProgressBarOption
from obitools3.dms.view import RollbackException
from functools import reduce
from obitools3.apps.config import logger
from obitools3.utils cimport tobytes, str2bytes, tostr
from io import BufferedWriter
from obitools3.dms.capi.obiview cimport COUNT_COLUMN
from obitools3.dms.capi.obitypes cimport OBI_INT

import time
import math 
import sys

from obitools3.dms.column.column cimport Column
from obitools3.dms.capi.obitypes cimport OBI_STR
import statistics

from cpython.exc cimport PyErr_CheckSignals

 
__title__="Normalize counts across samples"

 
def addOptions(parser):
    
    addMinimalInputOption(parser)
    addMinimalOutputOption(parser)
    addNoProgressBarOption(parser)

    group = parser.add_argument_group('obi normalize specific options')

    group.add_argument('--sample-tag', '-s',
                       action="store", 
                       dest="normalize:sample-tag-name",
                       metavar="<SAMPLE TAG NAME>",
                       type=str,
                       help="Name of the tag where merged sample count informations are kept (default: 'MERGED_sample' (generated by obi uniq)).")

    group.add_argument('--normalized-sample-tag', '-n',
                       action="store", 
                       dest="normalize:normalized-sample-tag-name",
                       metavar="<NORMALIZED SAMPLE TAG NAME>",
                       type=str,
                       default='NORM_MERGED_sample',
                       help="Name of the tag where normalized merged sample count informations should be kept (default: 'NORM_MERGED_sample').")


def run(config):
     
    DMS.obi_atexit()
    
    logger("info", "obi normalize")

    # Open the input
    input = open_uri(config['obi']['inputURI'])
    if input is None:
        raise Exception("Could not read input view")
    i_dms = input[0]
    i_view = input[1]
    i_view_name = input[1].name

    # Open the output: only the DMS, as the output view is going to be created by cloning the input view
    # (could eventually be done via an open_uri() argument)
    output = open_uri(config['obi']['outputURI'],
                      input=False,
                      dms_only=True)
    if output is None:
        raise Exception("Could not create output view")
    o_dms = output[0]
    output_0 = output[0]
    o_view_name = output[1]

    # stdout output: create temporary view
    if type(output_0)==BufferedWriter:
        o_dms = i_dms
        i=0
        o_view_name = b"temp"
        while o_view_name in i_dms: # Making sure view name is unique in output DMS
            o_view_name = o_view_name+b"_"+str2bytes(str(i))
            i+=1
        imported_view_name = o_view_name

    # If the input and output DMS are not the same, import the input view in the output DMS before cloning it to modify it
    # (could be the other way around: clone and modify in the input DMS then import the new view in the output DMS)
    if i_dms != o_dms:
        imported_view_name = i_view_name
        i=0
        while imported_view_name in o_dms:  # Making sure view name is unique in output DMS
            imported_view_name = i_view_name+b"_"+str2bytes(str(i))
            i+=1
        View.import_view(i_dms.full_path[:-7], o_dms.full_path[:-7], i_view_name, imported_view_name)
        i_view = o_dms[imported_view_name]

    # Clone output view from input view
    o_view = i_view.clone(o_view_name)
    if o_view is None:
        raise Exception("Couldn't create output view")
    i_view.close()

    # Initialize the progress bar
    if config['obi']['noprogressbar'] == False:
        pb = ProgressBar(len(o_view), config)
    else:
        pb = None

    try:

        if 'sample-tag-name' in config['normalize'] and config['normalize']['sample-tag-name'] is not None:
            count_col = o_view[config['normalize']['sample-tag-name']]
            #print('oui', repr(count_col))
        else:
            try:
                count_col = o_view['MERGED_sample'] # TODO macro
            except:
                try:
                    count_col = o_view[COUNT_COLUMN]
                except:
                    raise Exception
        
        Column.new_column(o_view,
                          config['normalize']['normalized-sample-tag-name'],
                          OBI_INT,
                          nb_elements_per_line=count_col.nb_elements_per_line,
                          elements_names=count_col.elements_names,
                          dict_column=True,
                          alias=config['normalize']['normalized-sample-tag-name']
                         )
        
        norm_count_col = o_view[config['normalize']['normalized-sample-tag-name']]
                
        i = 0
        sample_dict = {}
        for i in range(len(o_view)):
            PyErr_CheckSignals()
            if pb is not None:
                pb(i)
            #print('\n', count_col[i].is_NA(),'\n')
            if not count_col[i].is_NA():
                for sample in count_col[i]:
                    if sample not in sample_dict:
                        sample_dict[sample] = 0
                    sample_dict[sample] += count_col[i][sample]
            i+=1
        
        ref_sample = min(sample_dict, key=sample_dict.get)
        ref_sample_count = sample_dict[ref_sample]
        
        print('\n', ref_sample, ref_sample_count, '\n')
        
        i = 0
        for i in range(len(o_view)):
            PyErr_CheckSignals()
            if pb is not None:
                pb(i)
            o_view[i]['NORM_COUNT'] = 0
            if not count_col[i].is_NA():
                for sample in count_col[i]:
                    #print(count_col)
                    #print(sample_dict)
                    #print("count_col[i][sample] / (sample_dict[sample] / ref_sample_count)", count_col[i][sample], sample_dict[sample], ref_sample_count)
                    norm_count_col[i][sample] = round(count_col[i][sample] / (sample_dict[sample] / ref_sample_count))
                    o_view['NORM_COUNT'][i] += norm_count_col[i][sample]
                    if norm_count_col[i][sample] == 0:
                        norm_count_col[i][sample] = None
                i+=1

    except Exception, e:
        raise RollbackException("obi normalize error, rollbacking view: "+str(e), o_view)
        
    if pb is not None:
        pb(i, force=True)
        print("", file=sys.stderr)
    
    # Save command config in View and DMS comments
    command_line = " ".join(sys.argv[1:])
    input_dms_name=[input[0].name]
    input_view_name=[i_view_name]
    if 'taxoURI' in config['obi'] and config['obi']['taxoURI'] is not None:
        input_dms_name.append(config['obi']['taxoURI'].split("/")[-3])
        input_view_name.append("taxonomy/"+config['obi']['taxoURI'].split("/")[-1])
    o_view.write_config(config, "normalize", command_line, input_dms_name=input_dms_name, input_view_name=input_view_name)
    o_dms.record_command_line(command_line)

    #print("\n\nOutput view:\n````````````", file=sys.stderr)
    #print(repr(o_view), file=sys.stderr)

    # stdout output: write to buffer
    if type(output_0)==BufferedWriter:
        logger("info", "Printing to output...")
        o_view.print_to_output(output_0, noprogressbar=config['obi']['noprogressbar'])
        o_view.close()

    # If the input and the output DMS are different or if stdout output, delete the temporary imported view used to create the final view
    if i_dms != o_dms or type(output_0)==BufferedWriter:
        View.delete_view(o_dms, imported_view_name)
        o_dms.close(force=True)
    i_dms.close(force=True)

    logger("info", "Done.")
