import collections as _collections
import weakref as _weakref
import sys

from vulkan._vulkancache import ffi


_weakkey_dict = _weakref.WeakKeyDictionary()
PY3 = sys.version_info >= (3, 0)


class ProcedureNotFoundError(Exception):
    pass


class ExtensionNotSupportedError(Exception):
    pass


def _cstr(x):
    if not isinstance(x, ffi.CData):
        return x

    t = ffi.typeof(x)
    if 'item' not in dir(t) or t.item.cname != 'char':
        return x

    if PY3:
        return ffi.string(x).decode('ascii')
    else:
        return ffi.string(x)


class StrWrap(object):
    """Wrap a FFI Cdata object

    This class is a proxy class which auto-convert FFI string to Python
    string. It must be used only on object containing string data.
    Original CFFI string can always be accessed by prefixing the property with
    an underscore.
    """
    def __init__(self, obj):
        self.obj = obj

    def __setattr__(self, key, value):
        if key == 'obj':
            return super(StrWrap, self).__setattr__(key, value)

        setattr(self.obj, key, value)

    def __getattr__(self, key):
        try:
            attr = getattr(self.obj, key)
        except AttributeError as origin_exc:
            # Remove the first underscore if exists
            if key.startswith('_'):
                try:
                    return getattr(self.obj, key[1:])
                except AttributeError:
                    raise origin_exc
            raise origin_exc

        return _cstr(attr)


def _cast_ptr2(x, _type):
    if isinstance(x, ffi.CData):
        if (_type.item == ffi.typeof(x) or
            (_type.item.cname == 'void' and ffi.typeof(x).kind in
             ['struct', 'union'])):
            return ffi.addressof(x), x
        return x, x

    if isinstance(x, _collections.abc.Iterable):
        if _type.item.kind == 'pointer':
            ptrs = [_cast_ptr(i, _type.item) for i in x]
            ret = ffi.new(_type.item.cname+'[]', [i for i, _ in ptrs])
            _weakkey_dict[ret] = tuple(i for _, i in ptrs if i != ffi.NULL)
        else:
            ret = ffi.new(_type.item.cname+'[]', x)

        return ret, ret

    return ffi.cast(_type, x), x


def _cast_ptr3(x, _type):
    if isinstance(x, str):
        x = x.encode('ascii')
    return _cast_ptr2(x, _type)


_cast_ptr = _cast_ptr3 if PY3 else _cast_ptr2


# Load SDK
_lib_names = ('libvulkan.so.1', 'vulkan-1.dll', 'libvulkan.dylib')
for name in _lib_names:
    try:
        lib = ffi.dlopen(name)
        break
    except OSError:
        pass
else:
    raise OSError('Cannot find Vulkan SDK version. Please ensure that it is '
                  'installed and that the <sdk_root>/<version>/lib/ folder is '
                  'in the library path')


VK_ATTACHMENT_LOAD_OP_LOAD = 0
VK_ATTACHMENT_LOAD_OP_CLEAR = 1
VK_ATTACHMENT_LOAD_OP_DONT_CARE = 2
VK_ATTACHMENT_LOAD_OP__BEGIN_RANGE = 0
VK_ATTACHMENT_LOAD_OP__END_RANGE = 2
VK_ATTACHMENT_LOAD_OP__RANGE_SIZE = 3
VK_ATTACHMENT_LOAD_OP__MAX_ENUM = 2147483647
VK_ATTACHMENT_STORE_OP_STORE = 0
VK_ATTACHMENT_STORE_OP_DONT_CARE = 1
VK_ATTACHMENT_STORE_OP__BEGIN_RANGE = 0
VK_ATTACHMENT_STORE_OP__END_RANGE = 1
VK_ATTACHMENT_STORE_OP__RANGE_SIZE = 2
VK_ATTACHMENT_STORE_OP__MAX_ENUM = 2147483647
VK_BLEND_FACTOR_ZERO = 0
VK_BLEND_FACTOR_ONE = 1
VK_BLEND_FACTOR_SRC_COLOR = 2
VK_BLEND_FACTOR_ONE_MINUS_SRC_COLOR = 3
VK_BLEND_FACTOR_DST_COLOR = 4
VK_BLEND_FACTOR_ONE_MINUS_DST_COLOR = 5
VK_BLEND_FACTOR_SRC_ALPHA = 6
VK_BLEND_FACTOR_ONE_MINUS_SRC_ALPHA = 7
VK_BLEND_FACTOR_DST_ALPHA = 8
VK_BLEND_FACTOR_ONE_MINUS_DST_ALPHA = 9
VK_BLEND_FACTOR_CONSTANT_COLOR = 10
VK_BLEND_FACTOR_ONE_MINUS_CONSTANT_COLOR = 11
VK_BLEND_FACTOR_CONSTANT_ALPHA = 12
VK_BLEND_FACTOR_ONE_MINUS_CONSTANT_ALPHA = 13
VK_BLEND_FACTOR_SRC_ALPHA_SATURATE = 14
VK_BLEND_FACTOR_SRC1_COLOR = 15
VK_BLEND_FACTOR_ONE_MINUS_SRC1_COLOR = 16
VK_BLEND_FACTOR_SRC1_ALPHA = 17
VK_BLEND_FACTOR_ONE_MINUS_SRC1_ALPHA = 18
VK_BLEND_FACTOR__BEGIN_RANGE = 0
VK_BLEND_FACTOR__END_RANGE = 18
VK_BLEND_FACTOR__RANGE_SIZE = 19
VK_BLEND_FACTOR__MAX_ENUM = 2147483647
VK_BLEND_OP_ADD = 0
VK_BLEND_OP_SUBTRACT = 1
VK_BLEND_OP_REVERSE_SUBTRACT = 2
VK_BLEND_OP_MIN = 3
VK_BLEND_OP_MAX = 4
VK_BLEND_OP__BEGIN_RANGE = 0
VK_BLEND_OP__END_RANGE = 4
VK_BLEND_OP__RANGE_SIZE = 5
VK_BLEND_OP__MAX_ENUM = 2147483647
VK_BORDER_COLOR_FLOAT_TRANSPARENT_BLACK = 0
VK_BORDER_COLOR_INT_TRANSPARENT_BLACK = 1
VK_BORDER_COLOR_FLOAT_OPAQUE_BLACK = 2
VK_BORDER_COLOR_INT_OPAQUE_BLACK = 3
VK_BORDER_COLOR_FLOAT_OPAQUE_WHITE = 4
VK_BORDER_COLOR_INT_OPAQUE_WHITE = 5
VK_BORDER_COLOR__BEGIN_RANGE = 0
VK_BORDER_COLOR__END_RANGE = 5
VK_BORDER_COLOR__RANGE_SIZE = 6
VK_BORDER_COLOR__MAX_ENUM = 2147483647
VK_PIPELINE_CACHE_HEADER_VERSION_ONE = 1
VK_PIPELINE_CACHE_HEADER_VERSION__BEGIN_RANGE = 1
VK_PIPELINE_CACHE_HEADER_VERSION__END_RANGE = 1
VK_PIPELINE_CACHE_HEADER_VERSION__RANGE_SIZE = 1
VK_PIPELINE_CACHE_HEADER_VERSION__MAX_ENUM = 2147483647
VK_BUFFER_CREATE_SPARSE_BINDING_BIT = 0x00000001
VK_BUFFER_CREATE_SPARSE_RESIDENCY_BIT = 0x00000002
VK_BUFFER_CREATE_SPARSE_ALIASED_BIT = 0x00000004
VK_BUFFER_CREATE_FLAG_BITS__MAX_ENUM = 2147483647
VK_BUFFER_USAGE_TRANSFER_SRC_BIT = 0x00000001
VK_BUFFER_USAGE_TRANSFER_DST_BIT = 0x00000002
VK_BUFFER_USAGE_UNIFORM_TEXEL_BUFFER_BIT = 0x00000004
VK_BUFFER_USAGE_STORAGE_TEXEL_BUFFER_BIT = 0x00000008
VK_BUFFER_USAGE_UNIFORM_BUFFER_BIT = 0x00000010
VK_BUFFER_USAGE_STORAGE_BUFFER_BIT = 0x00000020
VK_BUFFER_USAGE_INDEX_BUFFER_BIT = 0x00000040
VK_BUFFER_USAGE_VERTEX_BUFFER_BIT = 0x00000080
VK_BUFFER_USAGE_INDIRECT_BUFFER_BIT = 0x00000100
VK_BUFFER_USAGE_FLAG_BITS__MAX_ENUM = 2147483647
VK_COLOR_COMPONENT_R_BIT = 0x00000001
VK_COLOR_COMPONENT_G_BIT = 0x00000002
VK_COLOR_COMPONENT_B_BIT = 0x00000004
VK_COLOR_COMPONENT_A_BIT = 0x00000008
VK_COLOR_COMPONENT_FLAG_BITS__MAX_ENUM = 2147483647
VK_COMPONENT_SWIZZLE_IDENTITY = 0
VK_COMPONENT_SWIZZLE_ZERO = 1
VK_COMPONENT_SWIZZLE_ONE = 2
VK_COMPONENT_SWIZZLE_R = 3
VK_COMPONENT_SWIZZLE_G = 4
VK_COMPONENT_SWIZZLE_B = 5
VK_COMPONENT_SWIZZLE_A = 6
VK_COMPONENT_SWIZZLE__BEGIN_RANGE = 0
VK_COMPONENT_SWIZZLE__END_RANGE = 6
VK_COMPONENT_SWIZZLE__RANGE_SIZE = 7
VK_COMPONENT_SWIZZLE__MAX_ENUM = 2147483647
VK_COMMAND_POOL_CREATE_TRANSIENT_BIT = 0x00000001
VK_COMMAND_POOL_CREATE_RESET_COMMAND_BUFFER_BIT = 0x00000002
VK_COMMAND_POOL_CREATE_FLAG_BITS__MAX_ENUM = 2147483647
VK_COMMAND_POOL_RESET_RELEASE_RESOURCES_BIT = 0x00000001
VK_COMMAND_POOL_RESET_FLAG_BITS__MAX_ENUM = 2147483647
VK_COMMAND_BUFFER_RESET_RELEASE_RESOURCES_BIT = 0x00000001
VK_COMMAND_BUFFER_RESET_FLAG_BITS__MAX_ENUM = 2147483647
VK_COMMAND_BUFFER_LEVEL_PRIMARY = 0
VK_COMMAND_BUFFER_LEVEL_SECONDARY = 1
VK_COMMAND_BUFFER_LEVEL__BEGIN_RANGE = 0
VK_COMMAND_BUFFER_LEVEL__END_RANGE = 1
VK_COMMAND_BUFFER_LEVEL__RANGE_SIZE = 2
VK_COMMAND_BUFFER_LEVEL__MAX_ENUM = 2147483647
VK_COMMAND_BUFFER_USAGE_ONE_TIME_SUBMIT_BIT = 0x00000001
VK_COMMAND_BUFFER_USAGE_RENDER_PASS_CONTINUE_BIT = 0x00000002
VK_COMMAND_BUFFER_USAGE_SIMULTANEOUS_USE_BIT = 0x00000004
VK_COMMAND_BUFFER_USAGE_FLAG_BITS__MAX_ENUM = 2147483647
VK_COMPARE_OP_NEVER = 0
VK_COMPARE_OP_LESS = 1
VK_COMPARE_OP_EQUAL = 2
VK_COMPARE_OP_LESS_OR_EQUAL = 3
VK_COMPARE_OP_GREATER = 4
VK_COMPARE_OP_NOT_EQUAL = 5
VK_COMPARE_OP_GREATER_OR_EQUAL = 6
VK_COMPARE_OP_ALWAYS = 7
VK_COMPARE_OP__BEGIN_RANGE = 0
VK_COMPARE_OP__END_RANGE = 7
VK_COMPARE_OP__RANGE_SIZE = 8
VK_COMPARE_OP__MAX_ENUM = 2147483647
VK_CULL_MODE_NONE = 0
VK_CULL_MODE_FRONT_BIT = 0x00000001
VK_CULL_MODE_BACK_BIT = 0x00000002
VK_CULL_MODE_FRONT_AND_BACK = 0x00000003
VK_CULL_MODE_FLAG_BITS__MAX_ENUM = 2147483647
VK_DESCRIPTOR_TYPE_SAMPLER = 0
VK_DESCRIPTOR_TYPE_COMBINED_IMAGE_SAMPLER = 1
VK_DESCRIPTOR_TYPE_SAMPLED_IMAGE = 2
VK_DESCRIPTOR_TYPE_STORAGE_IMAGE = 3
VK_DESCRIPTOR_TYPE_UNIFORM_TEXEL_BUFFER = 4
VK_DESCRIPTOR_TYPE_STORAGE_TEXEL_BUFFER = 5
VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER = 6
VK_DESCRIPTOR_TYPE_STORAGE_BUFFER = 7
VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER_DYNAMIC = 8
VK_DESCRIPTOR_TYPE_STORAGE_BUFFER_DYNAMIC = 9
VK_DESCRIPTOR_TYPE_INPUT_ATTACHMENT = 10
VK_DESCRIPTOR_TYPE__BEGIN_RANGE = 0
VK_DESCRIPTOR_TYPE__END_RANGE = 10
VK_DESCRIPTOR_TYPE__RANGE_SIZE = 11
VK_DESCRIPTOR_TYPE__MAX_ENUM = 2147483647
VK_DYNAMIC_STATE_VIEWPORT = 0
VK_DYNAMIC_STATE_SCISSOR = 1
VK_DYNAMIC_STATE_LINE_WIDTH = 2
VK_DYNAMIC_STATE_DEPTH_BIAS = 3
VK_DYNAMIC_STATE_BLEND_CONSTANTS = 4
VK_DYNAMIC_STATE_DEPTH_BOUNDS = 5
VK_DYNAMIC_STATE_STENCIL_COMPARE_MASK = 6
VK_DYNAMIC_STATE_STENCIL_WRITE_MASK = 7
VK_DYNAMIC_STATE_STENCIL_REFERENCE = 8
VK_DYNAMIC_STATE__BEGIN_RANGE = 0
VK_DYNAMIC_STATE__END_RANGE = 8
VK_DYNAMIC_STATE__RANGE_SIZE = 9
VK_DYNAMIC_STATE__MAX_ENUM = 2147483647
VK_FENCE_CREATE_SIGNALED_BIT = 0x00000001
VK_FENCE_CREATE_FLAG_BITS__MAX_ENUM = 2147483647
VK_POLYGON_MODE_FILL = 0
VK_POLYGON_MODE_LINE = 1
VK_POLYGON_MODE_POINT = 2
VK_POLYGON_MODE__BEGIN_RANGE = 0
VK_POLYGON_MODE__END_RANGE = 2
VK_POLYGON_MODE__RANGE_SIZE = 3
VK_POLYGON_MODE__MAX_ENUM = 2147483647
VK_FORMAT_UNDEFINED = 0
VK_FORMAT_R4G4_UNORM_PACK8 = 1
VK_FORMAT_R4G4B4A4_UNORM_PACK16 = 2
VK_FORMAT_B4G4R4A4_UNORM_PACK16 = 3
VK_FORMAT_R5G6B5_UNORM_PACK16 = 4
VK_FORMAT_B5G6R5_UNORM_PACK16 = 5
VK_FORMAT_R5G5B5A1_UNORM_PACK16 = 6
VK_FORMAT_B5G5R5A1_UNORM_PACK16 = 7
VK_FORMAT_A1R5G5B5_UNORM_PACK16 = 8
VK_FORMAT_R8_UNORM = 9
VK_FORMAT_R8_SNORM = 10
VK_FORMAT_R8_USCALED = 11
VK_FORMAT_R8_SSCALED = 12
VK_FORMAT_R8_UINT = 13
VK_FORMAT_R8_SINT = 14
VK_FORMAT_R8_SRGB = 15
VK_FORMAT_R8G8_UNORM = 16
VK_FORMAT_R8G8_SNORM = 17
VK_FORMAT_R8G8_USCALED = 18
VK_FORMAT_R8G8_SSCALED = 19
VK_FORMAT_R8G8_UINT = 20
VK_FORMAT_R8G8_SINT = 21
VK_FORMAT_R8G8_SRGB = 22
VK_FORMAT_R8G8B8_UNORM = 23
VK_FORMAT_R8G8B8_SNORM = 24
VK_FORMAT_R8G8B8_USCALED = 25
VK_FORMAT_R8G8B8_SSCALED = 26
VK_FORMAT_R8G8B8_UINT = 27
VK_FORMAT_R8G8B8_SINT = 28
VK_FORMAT_R8G8B8_SRGB = 29
VK_FORMAT_B8G8R8_UNORM = 30
VK_FORMAT_B8G8R8_SNORM = 31
VK_FORMAT_B8G8R8_USCALED = 32
VK_FORMAT_B8G8R8_SSCALED = 33
VK_FORMAT_B8G8R8_UINT = 34
VK_FORMAT_B8G8R8_SINT = 35
VK_FORMAT_B8G8R8_SRGB = 36
VK_FORMAT_R8G8B8A8_UNORM = 37
VK_FORMAT_R8G8B8A8_SNORM = 38
VK_FORMAT_R8G8B8A8_USCALED = 39
VK_FORMAT_R8G8B8A8_SSCALED = 40
VK_FORMAT_R8G8B8A8_UINT = 41
VK_FORMAT_R8G8B8A8_SINT = 42
VK_FORMAT_R8G8B8A8_SRGB = 43
VK_FORMAT_B8G8R8A8_UNORM = 44
VK_FORMAT_B8G8R8A8_SNORM = 45
VK_FORMAT_B8G8R8A8_USCALED = 46
VK_FORMAT_B8G8R8A8_SSCALED = 47
VK_FORMAT_B8G8R8A8_UINT = 48
VK_FORMAT_B8G8R8A8_SINT = 49
VK_FORMAT_B8G8R8A8_SRGB = 50
VK_FORMAT_A8B8G8R8_UNORM_PACK32 = 51
VK_FORMAT_A8B8G8R8_SNORM_PACK32 = 52
VK_FORMAT_A8B8G8R8_USCALED_PACK32 = 53
VK_FORMAT_A8B8G8R8_SSCALED_PACK32 = 54
VK_FORMAT_A8B8G8R8_UINT_PACK32 = 55
VK_FORMAT_A8B8G8R8_SINT_PACK32 = 56
VK_FORMAT_A8B8G8R8_SRGB_PACK32 = 57
VK_FORMAT_A2R10G10B10_UNORM_PACK32 = 58
VK_FORMAT_A2R10G10B10_SNORM_PACK32 = 59
VK_FORMAT_A2R10G10B10_USCALED_PACK32 = 60
VK_FORMAT_A2R10G10B10_SSCALED_PACK32 = 61
VK_FORMAT_A2R10G10B10_UINT_PACK32 = 62
VK_FORMAT_A2R10G10B10_SINT_PACK32 = 63
VK_FORMAT_A2B10G10R10_UNORM_PACK32 = 64
VK_FORMAT_A2B10G10R10_SNORM_PACK32 = 65
VK_FORMAT_A2B10G10R10_USCALED_PACK32 = 66
VK_FORMAT_A2B10G10R10_SSCALED_PACK32 = 67
VK_FORMAT_A2B10G10R10_UINT_PACK32 = 68
VK_FORMAT_A2B10G10R10_SINT_PACK32 = 69
VK_FORMAT_R16_UNORM = 70
VK_FORMAT_R16_SNORM = 71
VK_FORMAT_R16_USCALED = 72
VK_FORMAT_R16_SSCALED = 73
VK_FORMAT_R16_UINT = 74
VK_FORMAT_R16_SINT = 75
VK_FORMAT_R16_SFLOAT = 76
VK_FORMAT_R16G16_UNORM = 77
VK_FORMAT_R16G16_SNORM = 78
VK_FORMAT_R16G16_USCALED = 79
VK_FORMAT_R16G16_SSCALED = 80
VK_FORMAT_R16G16_UINT = 81
VK_FORMAT_R16G16_SINT = 82
VK_FORMAT_R16G16_SFLOAT = 83
VK_FORMAT_R16G16B16_UNORM = 84
VK_FORMAT_R16G16B16_SNORM = 85
VK_FORMAT_R16G16B16_USCALED = 86
VK_FORMAT_R16G16B16_SSCALED = 87
VK_FORMAT_R16G16B16_UINT = 88
VK_FORMAT_R16G16B16_SINT = 89
VK_FORMAT_R16G16B16_SFLOAT = 90
VK_FORMAT_R16G16B16A16_UNORM = 91
VK_FORMAT_R16G16B16A16_SNORM = 92
VK_FORMAT_R16G16B16A16_USCALED = 93
VK_FORMAT_R16G16B16A16_SSCALED = 94
VK_FORMAT_R16G16B16A16_UINT = 95
VK_FORMAT_R16G16B16A16_SINT = 96
VK_FORMAT_R16G16B16A16_SFLOAT = 97
VK_FORMAT_R32_UINT = 98
VK_FORMAT_R32_SINT = 99
VK_FORMAT_R32_SFLOAT = 100
VK_FORMAT_R32G32_UINT = 101
VK_FORMAT_R32G32_SINT = 102
VK_FORMAT_R32G32_SFLOAT = 103
VK_FORMAT_R32G32B32_UINT = 104
VK_FORMAT_R32G32B32_SINT = 105
VK_FORMAT_R32G32B32_SFLOAT = 106
VK_FORMAT_R32G32B32A32_UINT = 107
VK_FORMAT_R32G32B32A32_SINT = 108
VK_FORMAT_R32G32B32A32_SFLOAT = 109
VK_FORMAT_R64_UINT = 110
VK_FORMAT_R64_SINT = 111
VK_FORMAT_R64_SFLOAT = 112
VK_FORMAT_R64G64_UINT = 113
VK_FORMAT_R64G64_SINT = 114
VK_FORMAT_R64G64_SFLOAT = 115
VK_FORMAT_R64G64B64_UINT = 116
VK_FORMAT_R64G64B64_SINT = 117
VK_FORMAT_R64G64B64_SFLOAT = 118
VK_FORMAT_R64G64B64A64_UINT = 119
VK_FORMAT_R64G64B64A64_SINT = 120
VK_FORMAT_R64G64B64A64_SFLOAT = 121
VK_FORMAT_B10G11R11_UFLOAT_PACK32 = 122
VK_FORMAT_E5B9G9R9_UFLOAT_PACK32 = 123
VK_FORMAT_D16_UNORM = 124
VK_FORMAT_X8_D24_UNORM_PACK32 = 125
VK_FORMAT_D32_SFLOAT = 126
VK_FORMAT_S8_UINT = 127
VK_FORMAT_D16_UNORM_S8_UINT = 128
VK_FORMAT_D24_UNORM_S8_UINT = 129
VK_FORMAT_D32_SFLOAT_S8_UINT = 130
VK_FORMAT_BC1_RGB_UNORM_BLOCK = 131
VK_FORMAT_BC1_RGB_SRGB_BLOCK = 132
VK_FORMAT_BC1_RGBA_UNORM_BLOCK = 133
VK_FORMAT_BC1_RGBA_SRGB_BLOCK = 134
VK_FORMAT_BC2_UNORM_BLOCK = 135
VK_FORMAT_BC2_SRGB_BLOCK = 136
VK_FORMAT_BC3_UNORM_BLOCK = 137
VK_FORMAT_BC3_SRGB_BLOCK = 138
VK_FORMAT_BC4_UNORM_BLOCK = 139
VK_FORMAT_BC4_SNORM_BLOCK = 140
VK_FORMAT_BC5_UNORM_BLOCK = 141
VK_FORMAT_BC5_SNORM_BLOCK = 142
VK_FORMAT_BC6H_UFLOAT_BLOCK = 143
VK_FORMAT_BC6H_SFLOAT_BLOCK = 144
VK_FORMAT_BC7_UNORM_BLOCK = 145
VK_FORMAT_BC7_SRGB_BLOCK = 146
VK_FORMAT_ETC2_R8G8B8_UNORM_BLOCK = 147
VK_FORMAT_ETC2_R8G8B8_SRGB_BLOCK = 148
VK_FORMAT_ETC2_R8G8B8A1_UNORM_BLOCK = 149
VK_FORMAT_ETC2_R8G8B8A1_SRGB_BLOCK = 150
VK_FORMAT_ETC2_R8G8B8A8_UNORM_BLOCK = 151
VK_FORMAT_ETC2_R8G8B8A8_SRGB_BLOCK = 152
VK_FORMAT_EAC_R11_UNORM_BLOCK = 153
VK_FORMAT_EAC_R11_SNORM_BLOCK = 154
VK_FORMAT_EAC_R11G11_UNORM_BLOCK = 155
VK_FORMAT_EAC_R11G11_SNORM_BLOCK = 156
VK_FORMAT_ASTC_4x4_UNORM_BLOCK = 157
VK_FORMAT_ASTC_4x4_SRGB_BLOCK = 158
VK_FORMAT_ASTC_5x4_UNORM_BLOCK = 159
VK_FORMAT_ASTC_5x4_SRGB_BLOCK = 160
VK_FORMAT_ASTC_5x5_UNORM_BLOCK = 161
VK_FORMAT_ASTC_5x5_SRGB_BLOCK = 162
VK_FORMAT_ASTC_6x5_UNORM_BLOCK = 163
VK_FORMAT_ASTC_6x5_SRGB_BLOCK = 164
VK_FORMAT_ASTC_6x6_UNORM_BLOCK = 165
VK_FORMAT_ASTC_6x6_SRGB_BLOCK = 166
VK_FORMAT_ASTC_8x5_UNORM_BLOCK = 167
VK_FORMAT_ASTC_8x5_SRGB_BLOCK = 168
VK_FORMAT_ASTC_8x6_UNORM_BLOCK = 169
VK_FORMAT_ASTC_8x6_SRGB_BLOCK = 170
VK_FORMAT_ASTC_8x8_UNORM_BLOCK = 171
VK_FORMAT_ASTC_8x8_SRGB_BLOCK = 172
VK_FORMAT_ASTC_10x5_UNORM_BLOCK = 173
VK_FORMAT_ASTC_10x5_SRGB_BLOCK = 174
VK_FORMAT_ASTC_10x6_UNORM_BLOCK = 175
VK_FORMAT_ASTC_10x6_SRGB_BLOCK = 176
VK_FORMAT_ASTC_10x8_UNORM_BLOCK = 177
VK_FORMAT_ASTC_10x8_SRGB_BLOCK = 178
VK_FORMAT_ASTC_10x10_UNORM_BLOCK = 179
VK_FORMAT_ASTC_10x10_SRGB_BLOCK = 180
VK_FORMAT_ASTC_12x10_UNORM_BLOCK = 181
VK_FORMAT_ASTC_12x10_SRGB_BLOCK = 182
VK_FORMAT_ASTC_12x12_UNORM_BLOCK = 183
VK_FORMAT_ASTC_12x12_SRGB_BLOCK = 184
VK_FORMAT__BEGIN_RANGE = 0
VK_FORMAT__END_RANGE = 184
VK_FORMAT__RANGE_SIZE = 185
VK_FORMAT__MAX_ENUM = 2147483647
VK_FORMAT_G8B8G8R8_422_UNORM = 1000156000
VK_FORMAT_B8G8R8G8_422_UNORM = 1000156001
VK_FORMAT_G8_B8_R8_3PLANE_420_UNORM = 1000156002
VK_FORMAT_G8_B8R8_2PLANE_420_UNORM = 1000156003
VK_FORMAT_G8_B8_R8_3PLANE_422_UNORM = 1000156004
VK_FORMAT_G8_B8R8_2PLANE_422_UNORM = 1000156005
VK_FORMAT_G8_B8_R8_3PLANE_444_UNORM = 1000156006
VK_FORMAT_R10X6_UNORM_PACK16 = 1000156007
VK_FORMAT_R10X6G10X6_UNORM_2PACK16 = 1000156008
VK_FORMAT_R10X6G10X6B10X6A10X6_UNORM_4PACK16 = 1000156009
VK_FORMAT_G10X6B10X6G10X6R10X6_422_UNORM_4PACK16 = 1000156010
VK_FORMAT_B10X6G10X6R10X6G10X6_422_UNORM_4PACK16 = 1000156011
VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_420_UNORM_3PACK16 = 1000156012
VK_FORMAT_G10X6_B10X6R10X6_2PLANE_420_UNORM_3PACK16 = 1000156013
VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_422_UNORM_3PACK16 = 1000156014
VK_FORMAT_G10X6_B10X6R10X6_2PLANE_422_UNORM_3PACK16 = 1000156015
VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_444_UNORM_3PACK16 = 1000156016
VK_FORMAT_R12X4_UNORM_PACK16 = 1000156017
VK_FORMAT_R12X4G12X4_UNORM_2PACK16 = 1000156018
VK_FORMAT_R12X4G12X4B12X4A12X4_UNORM_4PACK16 = 1000156019
VK_FORMAT_G12X4B12X4G12X4R12X4_422_UNORM_4PACK16 = 1000156020
VK_FORMAT_B12X4G12X4R12X4G12X4_422_UNORM_4PACK16 = 1000156021
VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_420_UNORM_3PACK16 = 1000156022
VK_FORMAT_G12X4_B12X4R12X4_2PLANE_420_UNORM_3PACK16 = 1000156023
VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_422_UNORM_3PACK16 = 1000156024
VK_FORMAT_G12X4_B12X4R12X4_2PLANE_422_UNORM_3PACK16 = 1000156025
VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_444_UNORM_3PACK16 = 1000156026
VK_FORMAT_G16B16G16R16_422_UNORM = 1000156027
VK_FORMAT_B16G16R16G16_422_UNORM = 1000156028
VK_FORMAT_G16_B16_R16_3PLANE_420_UNORM = 1000156029
VK_FORMAT_G16_B16R16_2PLANE_420_UNORM = 1000156030
VK_FORMAT_G16_B16_R16_3PLANE_422_UNORM = 1000156031
VK_FORMAT_G16_B16R16_2PLANE_422_UNORM = 1000156032
VK_FORMAT_G16_B16_R16_3PLANE_444_UNORM = 1000156033
VK_FORMAT_FEATURE_SAMPLED_IMAGE_BIT = 0x00000001
VK_FORMAT_FEATURE_STORAGE_IMAGE_BIT = 0x00000002
VK_FORMAT_FEATURE_STORAGE_IMAGE_ATOMIC_BIT = 0x00000004
VK_FORMAT_FEATURE_UNIFORM_TEXEL_BUFFER_BIT = 0x00000008
VK_FORMAT_FEATURE_STORAGE_TEXEL_BUFFER_BIT = 0x00000010
VK_FORMAT_FEATURE_STORAGE_TEXEL_BUFFER_ATOMIC_BIT = 0x00000020
VK_FORMAT_FEATURE_VERTEX_BUFFER_BIT = 0x00000040
VK_FORMAT_FEATURE_COLOR_ATTACHMENT_BIT = 0x00000080
VK_FORMAT_FEATURE_COLOR_ATTACHMENT_BLEND_BIT = 0x00000100
VK_FORMAT_FEATURE_DEPTH_STENCIL_ATTACHMENT_BIT = 0x00000200
VK_FORMAT_FEATURE_BLIT_SRC_BIT = 0x00000400
VK_FORMAT_FEATURE_BLIT_DST_BIT = 0x00000800
VK_FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_LINEAR_BIT = 0x00001000
VK_FORMAT_FEATURE_FLAG_BITS__MAX_ENUM = 2147483647
VK_FRONT_FACE_COUNTER_CLOCKWISE = 0
VK_FRONT_FACE_CLOCKWISE = 1
VK_FRONT_FACE__BEGIN_RANGE = 0
VK_FRONT_FACE__END_RANGE = 1
VK_FRONT_FACE__RANGE_SIZE = 2
VK_FRONT_FACE__MAX_ENUM = 2147483647
VK_IMAGE_ASPECT_COLOR_BIT = 0x00000001
VK_IMAGE_ASPECT_DEPTH_BIT = 0x00000002
VK_IMAGE_ASPECT_STENCIL_BIT = 0x00000004
VK_IMAGE_ASPECT_METADATA_BIT = 0x00000008
VK_IMAGE_ASPECT_FLAG_BITS__MAX_ENUM = 2147483647
VK_IMAGE_CREATE_SPARSE_BINDING_BIT = 0x00000001
VK_IMAGE_CREATE_SPARSE_RESIDENCY_BIT = 0x00000002
VK_IMAGE_CREATE_SPARSE_ALIASED_BIT = 0x00000004
VK_IMAGE_CREATE_MUTABLE_FORMAT_BIT = 0x00000008
VK_IMAGE_CREATE_CUBE_COMPATIBLE_BIT = 0x00000010
VK_IMAGE_CREATE_FLAG_BITS__MAX_ENUM = 2147483647
VK_IMAGE_LAYOUT_UNDEFINED = 0
VK_IMAGE_LAYOUT_GENERAL = 1
VK_IMAGE_LAYOUT_COLOR_ATTACHMENT_OPTIMAL = 2
VK_IMAGE_LAYOUT_DEPTH_STENCIL_ATTACHMENT_OPTIMAL = 3
VK_IMAGE_LAYOUT_DEPTH_STENCIL_READ_ONLY_OPTIMAL = 4
VK_IMAGE_LAYOUT_SHADER_READ_ONLY_OPTIMAL = 5
VK_IMAGE_LAYOUT_TRANSFER_SRC_OPTIMAL = 6
VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL = 7
VK_IMAGE_LAYOUT_PREINITIALIZED = 8
VK_IMAGE_LAYOUT__BEGIN_RANGE = 0
VK_IMAGE_LAYOUT__END_RANGE = 8
VK_IMAGE_LAYOUT__RANGE_SIZE = 9
VK_IMAGE_LAYOUT__MAX_ENUM = 2147483647
VK_IMAGE_LAYOUT_DEPTH_READ_ONLY_STENCIL_ATTACHMENT_OPTIMAL = 1000117000
VK_IMAGE_LAYOUT_DEPTH_ATTACHMENT_STENCIL_READ_ONLY_OPTIMAL = 1000117001
VK_IMAGE_TILING_OPTIMAL = 0
VK_IMAGE_TILING_LINEAR = 1
VK_IMAGE_TILING__BEGIN_RANGE = 0
VK_IMAGE_TILING__END_RANGE = 1
VK_IMAGE_TILING__RANGE_SIZE = 2
VK_IMAGE_TILING__MAX_ENUM = 2147483647
VK_IMAGE_TYPE_1D = 0
VK_IMAGE_TYPE_2D = 1
VK_IMAGE_TYPE_3D = 2
VK_IMAGE_TYPE__BEGIN_RANGE = 0
VK_IMAGE_TYPE__END_RANGE = 2
VK_IMAGE_TYPE__RANGE_SIZE = 3
VK_IMAGE_TYPE__MAX_ENUM = 2147483647
VK_IMAGE_USAGE_TRANSFER_SRC_BIT = 0x00000001
VK_IMAGE_USAGE_TRANSFER_DST_BIT = 0x00000002
VK_IMAGE_USAGE_SAMPLED_BIT = 0x00000004
VK_IMAGE_USAGE_STORAGE_BIT = 0x00000008
VK_IMAGE_USAGE_COLOR_ATTACHMENT_BIT = 0x00000010
VK_IMAGE_USAGE_DEPTH_STENCIL_ATTACHMENT_BIT = 0x00000020
VK_IMAGE_USAGE_TRANSIENT_ATTACHMENT_BIT = 0x00000040
VK_IMAGE_USAGE_INPUT_ATTACHMENT_BIT = 0x00000080
VK_IMAGE_USAGE_FLAG_BITS__MAX_ENUM = 2147483647
VK_IMAGE_VIEW_TYPE_1D = 0
VK_IMAGE_VIEW_TYPE_2D = 1
VK_IMAGE_VIEW_TYPE_3D = 2
VK_IMAGE_VIEW_TYPE_CUBE = 3
VK_IMAGE_VIEW_TYPE_1D_ARRAY = 4
VK_IMAGE_VIEW_TYPE_2D_ARRAY = 5
VK_IMAGE_VIEW_TYPE_CUBE_ARRAY = 6
VK_IMAGE_VIEW_TYPE__BEGIN_RANGE = 0
VK_IMAGE_VIEW_TYPE__END_RANGE = 6
VK_IMAGE_VIEW_TYPE__RANGE_SIZE = 7
VK_IMAGE_VIEW_TYPE__MAX_ENUM = 2147483647
VK_SHARING_MODE_EXCLUSIVE = 0
VK_SHARING_MODE_CONCURRENT = 1
VK_SHARING_MODE__BEGIN_RANGE = 0
VK_SHARING_MODE__END_RANGE = 1
VK_SHARING_MODE__RANGE_SIZE = 2
VK_SHARING_MODE__MAX_ENUM = 2147483647
VK_INDEX_TYPE_UINT16 = 0
VK_INDEX_TYPE_UINT32 = 1
VK_INDEX_TYPE__BEGIN_RANGE = 0
VK_INDEX_TYPE__END_RANGE = 1
VK_INDEX_TYPE__RANGE_SIZE = 2
VK_INDEX_TYPE__MAX_ENUM = 2147483647
VK_LOGIC_OP_CLEAR = 0
VK_LOGIC_OP_AND = 1
VK_LOGIC_OP_AND_REVERSE = 2
VK_LOGIC_OP_COPY = 3
VK_LOGIC_OP_AND_INVERTED = 4
VK_LOGIC_OP_NO_OP = 5
VK_LOGIC_OP_XOR = 6
VK_LOGIC_OP_OR = 7
VK_LOGIC_OP_NOR = 8
VK_LOGIC_OP_EQUIVALENT = 9
VK_LOGIC_OP_INVERT = 10
VK_LOGIC_OP_OR_REVERSE = 11
VK_LOGIC_OP_COPY_INVERTED = 12
VK_LOGIC_OP_OR_INVERTED = 13
VK_LOGIC_OP_NAND = 14
VK_LOGIC_OP_SET = 15
VK_LOGIC_OP__BEGIN_RANGE = 0
VK_LOGIC_OP__END_RANGE = 15
VK_LOGIC_OP__RANGE_SIZE = 16
VK_LOGIC_OP__MAX_ENUM = 2147483647
VK_MEMORY_HEAP_DEVICE_LOCAL_BIT = 0x00000001
VK_MEMORY_HEAP_FLAG_BITS__MAX_ENUM = 2147483647
VK_ACCESS_INDIRECT_COMMAND_READ_BIT = 0x00000001
VK_ACCESS_INDEX_READ_BIT = 0x00000002
VK_ACCESS_VERTEX_ATTRIBUTE_READ_BIT = 0x00000004
VK_ACCESS_UNIFORM_READ_BIT = 0x00000008
VK_ACCESS_INPUT_ATTACHMENT_READ_BIT = 0x00000010
VK_ACCESS_SHADER_READ_BIT = 0x00000020
VK_ACCESS_SHADER_WRITE_BIT = 0x00000040
VK_ACCESS_COLOR_ATTACHMENT_READ_BIT = 0x00000080
VK_ACCESS_COLOR_ATTACHMENT_WRITE_BIT = 0x00000100
VK_ACCESS_DEPTH_STENCIL_ATTACHMENT_READ_BIT = 0x00000200
VK_ACCESS_DEPTH_STENCIL_ATTACHMENT_WRITE_BIT = 0x00000400
VK_ACCESS_TRANSFER_READ_BIT = 0x00000800
VK_ACCESS_TRANSFER_WRITE_BIT = 0x00001000
VK_ACCESS_HOST_READ_BIT = 0x00002000
VK_ACCESS_HOST_WRITE_BIT = 0x00004000
VK_ACCESS_MEMORY_READ_BIT = 0x00008000
VK_ACCESS_MEMORY_WRITE_BIT = 0x00010000
VK_ACCESS_FLAG_BITS__MAX_ENUM = 2147483647
VK_MEMORY_PROPERTY_DEVICE_LOCAL_BIT = 0x00000001
VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT = 0x00000002
VK_MEMORY_PROPERTY_HOST_COHERENT_BIT = 0x00000004
VK_MEMORY_PROPERTY_HOST_CACHED_BIT = 0x00000008
VK_MEMORY_PROPERTY_LAZILY_ALLOCATED_BIT = 0x00000010
VK_MEMORY_PROPERTY_FLAG_BITS__MAX_ENUM = 2147483647
VK_PHYSICAL_DEVICE_TYPE_OTHER = 0
VK_PHYSICAL_DEVICE_TYPE_INTEGRATED_GPU = 1
VK_PHYSICAL_DEVICE_TYPE_DISCRETE_GPU = 2
VK_PHYSICAL_DEVICE_TYPE_VIRTUAL_GPU = 3
VK_PHYSICAL_DEVICE_TYPE_CPU = 4
VK_PHYSICAL_DEVICE_TYPE__BEGIN_RANGE = 0
VK_PHYSICAL_DEVICE_TYPE__END_RANGE = 4
VK_PHYSICAL_DEVICE_TYPE__RANGE_SIZE = 5
VK_PHYSICAL_DEVICE_TYPE__MAX_ENUM = 2147483647
VK_PIPELINE_BIND_POINT_GRAPHICS = 0
VK_PIPELINE_BIND_POINT_COMPUTE = 1
VK_PIPELINE_BIND_POINT__BEGIN_RANGE = 0
VK_PIPELINE_BIND_POINT__END_RANGE = 1
VK_PIPELINE_BIND_POINT__RANGE_SIZE = 2
VK_PIPELINE_BIND_POINT__MAX_ENUM = 2147483647
VK_PIPELINE_CREATE_DISABLE_OPTIMIZATION_BIT = 0x00000001
VK_PIPELINE_CREATE_ALLOW_DERIVATIVES_BIT = 0x00000002
VK_PIPELINE_CREATE_DERIVATIVE_BIT = 0x00000004
VK_PIPELINE_CREATE_FLAG_BITS__MAX_ENUM = 2147483647
VK_PRIMITIVE_TOPOLOGY_POINT_LIST = 0
VK_PRIMITIVE_TOPOLOGY_LINE_LIST = 1
VK_PRIMITIVE_TOPOLOGY_LINE_STRIP = 2
VK_PRIMITIVE_TOPOLOGY_TRIANGLE_LIST = 3
VK_PRIMITIVE_TOPOLOGY_TRIANGLE_STRIP = 4
VK_PRIMITIVE_TOPOLOGY_TRIANGLE_FAN = 5
VK_PRIMITIVE_TOPOLOGY_LINE_LIST_WITH_ADJACENCY = 6
VK_PRIMITIVE_TOPOLOGY_LINE_STRIP_WITH_ADJACENCY = 7
VK_PRIMITIVE_TOPOLOGY_TRIANGLE_LIST_WITH_ADJACENCY = 8
VK_PRIMITIVE_TOPOLOGY_TRIANGLE_STRIP_WITH_ADJACENCY = 9
VK_PRIMITIVE_TOPOLOGY_PATCH_LIST = 10
VK_PRIMITIVE_TOPOLOGY__BEGIN_RANGE = 0
VK_PRIMITIVE_TOPOLOGY__END_RANGE = 10
VK_PRIMITIVE_TOPOLOGY__RANGE_SIZE = 11
VK_PRIMITIVE_TOPOLOGY__MAX_ENUM = 2147483647
VK_QUERY_CONTROL_PRECISE_BIT = 0x00000001
VK_QUERY_CONTROL_FLAG_BITS__MAX_ENUM = 2147483647
VK_QUERY_PIPELINE_STATISTIC_INPUT_ASSEMBLY_VERTICES_BIT = 0x00000001
VK_QUERY_PIPELINE_STATISTIC_INPUT_ASSEMBLY_PRIMITIVES_BIT = 0x00000002
VK_QUERY_PIPELINE_STATISTIC_VERTEX_SHADER_INVOCATIONS_BIT = 0x00000004
VK_QUERY_PIPELINE_STATISTIC_GEOMETRY_SHADER_INVOCATIONS_BIT = 0x00000008
VK_QUERY_PIPELINE_STATISTIC_GEOMETRY_SHADER_PRIMITIVES_BIT = 0x00000010
VK_QUERY_PIPELINE_STATISTIC_CLIPPING_INVOCATIONS_BIT = 0x00000020
VK_QUERY_PIPELINE_STATISTIC_CLIPPING_PRIMITIVES_BIT = 0x00000040
VK_QUERY_PIPELINE_STATISTIC_FRAGMENT_SHADER_INVOCATIONS_BIT = 0x00000080
VK_QUERY_PIPELINE_STATISTIC_TESSELLATION_CONTROL_SHADER_PATCHES_BIT = 0x00000100
VK_QUERY_PIPELINE_STATISTIC_TESSELLATION_EVALUATION_SHADER_INVOCATIONS_BIT = 0x00000200
VK_QUERY_PIPELINE_STATISTIC_COMPUTE_SHADER_INVOCATIONS_BIT = 0x00000400
VK_QUERY_PIPELINE_STATISTIC_FLAG_BITS__MAX_ENUM = 2147483647
VK_QUERY_RESULT_64_BIT = 0x00000001
VK_QUERY_RESULT_WAIT_BIT = 0x00000002
VK_QUERY_RESULT_WITH_AVAILABILITY_BIT = 0x00000004
VK_QUERY_RESULT_PARTIAL_BIT = 0x00000008
VK_QUERY_RESULT_FLAG_BITS__MAX_ENUM = 2147483647
VK_QUERY_TYPE_OCCLUSION = 0
VK_QUERY_TYPE_PIPELINE_STATISTICS = 1
VK_QUERY_TYPE_TIMESTAMP = 2
VK_QUERY_TYPE__BEGIN_RANGE = 0
VK_QUERY_TYPE__END_RANGE = 2
VK_QUERY_TYPE__RANGE_SIZE = 3
VK_QUERY_TYPE__MAX_ENUM = 2147483647
VK_QUEUE_GRAPHICS_BIT = 0x00000001
VK_QUEUE_COMPUTE_BIT = 0x00000002
VK_QUEUE_TRANSFER_BIT = 0x00000004
VK_QUEUE_SPARSE_BINDING_BIT = 0x00000008
VK_QUEUE_FLAG_BITS__MAX_ENUM = 2147483647
VK_SUBPASS_CONTENTS_INLINE = 0
VK_SUBPASS_CONTENTS_SECONDARY_COMMAND_BUFFERS = 1
VK_SUBPASS_CONTENTS__BEGIN_RANGE = 0
VK_SUBPASS_CONTENTS__END_RANGE = 1
VK_SUBPASS_CONTENTS__RANGE_SIZE = 2
VK_SUBPASS_CONTENTS__MAX_ENUM = 2147483647
VK_SUCCESS = 0
VK_NOT_READY = 1
VK_TIMEOUT = 2
VK_EVENT_SET = 3
VK_EVENT_RESET = 4
VK_INCOMPLETE = 5
VK_ERROR_OUT_OF_HOST_MEMORY = -1
VK_ERROR_OUT_OF_DEVICE_MEMORY = -2
VK_ERROR_INITIALIZATION_FAILED = -3
VK_ERROR_DEVICE_LOST = -4
VK_ERROR_MEMORY_MAP_FAILED = -5
VK_ERROR_LAYER_NOT_PRESENT = -6
VK_ERROR_EXTENSION_NOT_PRESENT = -7
VK_ERROR_FEATURE_NOT_PRESENT = -8
VK_ERROR_INCOMPATIBLE_DRIVER = -9
VK_ERROR_TOO_MANY_OBJECTS = -10
VK_ERROR_FORMAT_NOT_SUPPORTED = -11
VK_ERROR_FRAGMENTED_POOL = -12
VK_RESULT__BEGIN_RANGE = -12
VK_RESULT__END_RANGE = 5
VK_RESULT__RANGE_SIZE = 18
VK_RESULT__MAX_ENUM = 2147483647
VK_ERROR_OUT_OF_POOL_MEMORY = 1000069000
VK_ERROR_INVALID_EXTERNAL_HANDLE = 1000072003
VK_ERROR_SURFACE_LOST_KHR = -1000000000
VK_ERROR_NATIVE_WINDOW_IN_USE_KHR = -1000000001
VK_SUBOPTIMAL_KHR = 1000001003
VK_ERROR_OUT_OF_DATE_KHR = -1000001004
VK_ERROR_INCOMPATIBLE_DISPLAY_KHR = -1000003001
VK_ERROR_VALIDATION_FAILED_EXT = -1000011001
VK_ERROR_INVALID_SHADER_NV = -1000012000
VK_ERROR_OUT_OF_POOL_MEMORY_KHR = None
VK_ERROR_INVALID_EXTERNAL_HANDLE_KHR = None
VK_ERROR_INVALID_DRM_FORMAT_MODIFIER_PLANE_LAYOUT_EXT = -1000158000
VK_ERROR_FRAGMENTATION_EXT = -1000161000
VK_ERROR_NOT_PERMITTED_EXT = -1000174001
VK_ERROR_INVALID_DEVICE_ADDRESS_EXT = -1000244000
VK_SHADER_STAGE_VERTEX_BIT = 0x00000001
VK_SHADER_STAGE_TESSELLATION_CONTROL_BIT = 0x00000002
VK_SHADER_STAGE_TESSELLATION_EVALUATION_BIT = 0x00000004
VK_SHADER_STAGE_GEOMETRY_BIT = 0x00000008
VK_SHADER_STAGE_FRAGMENT_BIT = 0x00000010
VK_SHADER_STAGE_COMPUTE_BIT = 0x00000020
VK_SHADER_STAGE_ALL_GRAPHICS = 0x0000001F
VK_SHADER_STAGE_ALL = 0x7FFFFFFF
VK_SHADER_STAGE_FLAG_BITS__MAX_ENUM = 2147483647
VK_SPARSE_MEMORY_BIND_METADATA_BIT = 0x00000001
VK_SPARSE_MEMORY_BIND_FLAG_BITS__MAX_ENUM = 2147483647
VK_STENCIL_FACE_FRONT_BIT = 0x00000001
VK_STENCIL_FACE_BACK_BIT = 0x00000002
VK_STENCIL_FRONT_AND_BACK = 0x00000003
VK_STENCIL_FACE_FLAG_BITS__MAX_ENUM = 2147483647
VK_STENCIL_OP_KEEP = 0
VK_STENCIL_OP_ZERO = 1
VK_STENCIL_OP_REPLACE = 2
VK_STENCIL_OP_INCREMENT_AND_CLAMP = 3
VK_STENCIL_OP_DECREMENT_AND_CLAMP = 4
VK_STENCIL_OP_INVERT = 5
VK_STENCIL_OP_INCREMENT_AND_WRAP = 6
VK_STENCIL_OP_DECREMENT_AND_WRAP = 7
VK_STENCIL_OP__BEGIN_RANGE = 0
VK_STENCIL_OP__END_RANGE = 7
VK_STENCIL_OP__RANGE_SIZE = 8
VK_STENCIL_OP__MAX_ENUM = 2147483647
VK_STRUCTURE_TYPE_APPLICATION_INFO = 0
VK_STRUCTURE_TYPE_INSTANCE_CREATE_INFO = 1
VK_STRUCTURE_TYPE_DEVICE_QUEUE_CREATE_INFO = 2
VK_STRUCTURE_TYPE_DEVICE_CREATE_INFO = 3
VK_STRUCTURE_TYPE_SUBMIT_INFO = 4
VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_INFO = 5
VK_STRUCTURE_TYPE_MAPPED_MEMORY_RANGE = 6
VK_STRUCTURE_TYPE_BIND_SPARSE_INFO = 7
VK_STRUCTURE_TYPE_FENCE_CREATE_INFO = 8
VK_STRUCTURE_TYPE_SEMAPHORE_CREATE_INFO = 9
VK_STRUCTURE_TYPE_EVENT_CREATE_INFO = 10
VK_STRUCTURE_TYPE_QUERY_POOL_CREATE_INFO = 11
VK_STRUCTURE_TYPE_BUFFER_CREATE_INFO = 12
VK_STRUCTURE_TYPE_BUFFER_VIEW_CREATE_INFO = 13
VK_STRUCTURE_TYPE_IMAGE_CREATE_INFO = 14
VK_STRUCTURE_TYPE_IMAGE_VIEW_CREATE_INFO = 15
VK_STRUCTURE_TYPE_SHADER_MODULE_CREATE_INFO = 16
VK_STRUCTURE_TYPE_PIPELINE_CACHE_CREATE_INFO = 17
VK_STRUCTURE_TYPE_PIPELINE_SHADER_STAGE_CREATE_INFO = 18
VK_STRUCTURE_TYPE_PIPELINE_VERTEX_INPUT_STATE_CREATE_INFO = 19
VK_STRUCTURE_TYPE_PIPELINE_INPUT_ASSEMBLY_STATE_CREATE_INFO = 20
VK_STRUCTURE_TYPE_PIPELINE_TESSELLATION_STATE_CREATE_INFO = 21
VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_STATE_CREATE_INFO = 22
VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_CREATE_INFO = 23
VK_STRUCTURE_TYPE_PIPELINE_MULTISAMPLE_STATE_CREATE_INFO = 24
VK_STRUCTURE_TYPE_PIPELINE_DEPTH_STENCIL_STATE_CREATE_INFO = 25
VK_STRUCTURE_TYPE_PIPELINE_COLOR_BLEND_STATE_CREATE_INFO = 26
VK_STRUCTURE_TYPE_PIPELINE_DYNAMIC_STATE_CREATE_INFO = 27
VK_STRUCTURE_TYPE_GRAPHICS_PIPELINE_CREATE_INFO = 28
VK_STRUCTURE_TYPE_COMPUTE_PIPELINE_CREATE_INFO = 29
VK_STRUCTURE_TYPE_PIPELINE_LAYOUT_CREATE_INFO = 30
VK_STRUCTURE_TYPE_SAMPLER_CREATE_INFO = 31
VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_CREATE_INFO = 32
VK_STRUCTURE_TYPE_DESCRIPTOR_POOL_CREATE_INFO = 33
VK_STRUCTURE_TYPE_DESCRIPTOR_SET_ALLOCATE_INFO = 34
VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET = 35
VK_STRUCTURE_TYPE_COPY_DESCRIPTOR_SET = 36
VK_STRUCTURE_TYPE_FRAMEBUFFER_CREATE_INFO = 37
VK_STRUCTURE_TYPE_RENDER_PASS_CREATE_INFO = 38
VK_STRUCTURE_TYPE_COMMAND_POOL_CREATE_INFO = 39
VK_STRUCTURE_TYPE_COMMAND_BUFFER_ALLOCATE_INFO = 40
VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_INFO = 41
VK_STRUCTURE_TYPE_COMMAND_BUFFER_BEGIN_INFO = 42
VK_STRUCTURE_TYPE_RENDER_PASS_BEGIN_INFO = 43
VK_STRUCTURE_TYPE_BUFFER_MEMORY_BARRIER = 44
VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER = 45
VK_STRUCTURE_TYPE_MEMORY_BARRIER = 46
VK_STRUCTURE_TYPE_LOADER_INSTANCE_CREATE_INFO = 47
VK_STRUCTURE_TYPE_LOADER_DEVICE_CREATE_INFO = 48
VK_STRUCTURE_TYPE__BEGIN_RANGE = 0
VK_STRUCTURE_TYPE__END_RANGE = 48
VK_STRUCTURE_TYPE__RANGE_SIZE = 49
VK_STRUCTURE_TYPE__MAX_ENUM = 2147483647
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBGROUP_PROPERTIES = 1000094000
VK_STRUCTURE_TYPE_BIND_BUFFER_MEMORY_INFO = 1000157000
VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_INFO = 1000157001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_16BIT_STORAGE_FEATURES = 1000083000
VK_STRUCTURE_TYPE_MEMORY_DEDICATED_REQUIREMENTS = 1000127000
VK_STRUCTURE_TYPE_MEMORY_DEDICATED_ALLOCATE_INFO = 1000127001
VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_FLAGS_INFO = 1000060000
VK_STRUCTURE_TYPE_DEVICE_GROUP_RENDER_PASS_BEGIN_INFO = 1000060003
VK_STRUCTURE_TYPE_DEVICE_GROUP_COMMAND_BUFFER_BEGIN_INFO = 1000060004
VK_STRUCTURE_TYPE_DEVICE_GROUP_SUBMIT_INFO = 1000060005
VK_STRUCTURE_TYPE_DEVICE_GROUP_BIND_SPARSE_INFO = 1000060006
VK_STRUCTURE_TYPE_BIND_BUFFER_MEMORY_DEVICE_GROUP_INFO = 1000060013
VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_DEVICE_GROUP_INFO = 1000060014
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_GROUP_PROPERTIES = 1000070000
VK_STRUCTURE_TYPE_DEVICE_GROUP_DEVICE_CREATE_INFO = 1000070001
VK_STRUCTURE_TYPE_BUFFER_MEMORY_REQUIREMENTS_INFO_2 = 1000146000
VK_STRUCTURE_TYPE_IMAGE_MEMORY_REQUIREMENTS_INFO_2 = 1000146001
VK_STRUCTURE_TYPE_IMAGE_SPARSE_MEMORY_REQUIREMENTS_INFO_2 = 1000146002
VK_STRUCTURE_TYPE_MEMORY_REQUIREMENTS_2 = 1000146003
VK_STRUCTURE_TYPE_SPARSE_IMAGE_MEMORY_REQUIREMENTS_2 = 1000146004
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FEATURES_2 = 1000059000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROPERTIES_2 = 1000059001
VK_STRUCTURE_TYPE_FORMAT_PROPERTIES_2 = 1000059002
VK_STRUCTURE_TYPE_IMAGE_FORMAT_PROPERTIES_2 = 1000059003
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_FORMAT_INFO_2 = 1000059004
VK_STRUCTURE_TYPE_QUEUE_FAMILY_PROPERTIES_2 = 1000059005
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_PROPERTIES_2 = 1000059006
VK_STRUCTURE_TYPE_SPARSE_IMAGE_FORMAT_PROPERTIES_2 = 1000059007
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SPARSE_IMAGE_FORMAT_INFO_2 = 1000059008
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_POINT_CLIPPING_PROPERTIES = 1000117000
VK_STRUCTURE_TYPE_RENDER_PASS_INPUT_ATTACHMENT_ASPECT_CREATE_INFO = 1000117001
VK_STRUCTURE_TYPE_IMAGE_VIEW_USAGE_CREATE_INFO = 1000117002
VK_STRUCTURE_TYPE_PIPELINE_TESSELLATION_DOMAIN_ORIGIN_STATE_CREATE_INFO = 1000117003
VK_STRUCTURE_TYPE_RENDER_PASS_MULTIVIEW_CREATE_INFO = 1000053000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_FEATURES = 1000053001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_PROPERTIES = 1000053002
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VARIABLE_POINTER_FEATURES = 1000120000
VK_STRUCTURE_TYPE_PROTECTED_SUBMIT_INFO = 1000145000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROTECTED_MEMORY_FEATURES = 1000145001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROTECTED_MEMORY_PROPERTIES = 1000145002
VK_STRUCTURE_TYPE_DEVICE_QUEUE_INFO_2 = 1000145003
VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_CREATE_INFO = 1000156000
VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_INFO = 1000156001
VK_STRUCTURE_TYPE_BIND_IMAGE_PLANE_MEMORY_INFO = 1000156002
VK_STRUCTURE_TYPE_IMAGE_PLANE_MEMORY_REQUIREMENTS_INFO = 1000156003
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLER_YCBCR_CONVERSION_FEATURES = 1000156004
VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_IMAGE_FORMAT_PROPERTIES = 1000156005
VK_STRUCTURE_TYPE_DESCRIPTOR_UPDATE_TEMPLATE_CREATE_INFO = 1000085000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_IMAGE_FORMAT_INFO = 1000071000
VK_STRUCTURE_TYPE_EXTERNAL_IMAGE_FORMAT_PROPERTIES = 1000071001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_BUFFER_INFO = 1000071002
VK_STRUCTURE_TYPE_EXTERNAL_BUFFER_PROPERTIES = 1000071003
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ID_PROPERTIES = 1000071004
VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_BUFFER_CREATE_INFO = 1000072000
VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_IMAGE_CREATE_INFO = 1000072001
VK_STRUCTURE_TYPE_EXPORT_MEMORY_ALLOCATE_INFO = 1000072002
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_FENCE_INFO = 1000112000
VK_STRUCTURE_TYPE_EXTERNAL_FENCE_PROPERTIES = 1000112001
VK_STRUCTURE_TYPE_EXPORT_FENCE_CREATE_INFO = 1000113000
VK_STRUCTURE_TYPE_EXPORT_SEMAPHORE_CREATE_INFO = 1000077000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_SEMAPHORE_INFO = 1000076000
VK_STRUCTURE_TYPE_EXTERNAL_SEMAPHORE_PROPERTIES = 1000076001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_3_PROPERTIES = 1000168000
VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_SUPPORT = 1000168001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_DRAW_PARAMETER_FEATURES = 1000063000
VK_SYSTEM_ALLOCATION_SCOPE_COMMAND = 0
VK_SYSTEM_ALLOCATION_SCOPE_OBJECT = 1
VK_SYSTEM_ALLOCATION_SCOPE_CACHE = 2
VK_SYSTEM_ALLOCATION_SCOPE_DEVICE = 3
VK_SYSTEM_ALLOCATION_SCOPE_INSTANCE = 4
VK_SYSTEM_ALLOCATION_SCOPE__BEGIN_RANGE = 0
VK_SYSTEM_ALLOCATION_SCOPE__END_RANGE = 4
VK_SYSTEM_ALLOCATION_SCOPE__RANGE_SIZE = 5
VK_SYSTEM_ALLOCATION_SCOPE__MAX_ENUM = 2147483647
VK_INTERNAL_ALLOCATION_TYPE_EXECUTABLE = 0
VK_INTERNAL_ALLOCATION_TYPE__BEGIN_RANGE = 0
VK_INTERNAL_ALLOCATION_TYPE__END_RANGE = 0
VK_INTERNAL_ALLOCATION_TYPE__RANGE_SIZE = 1
VK_INTERNAL_ALLOCATION_TYPE__MAX_ENUM = 2147483647
VK_SAMPLER_ADDRESS_MODE_REPEAT = 0
VK_SAMPLER_ADDRESS_MODE_MIRRORED_REPEAT = 1
VK_SAMPLER_ADDRESS_MODE_CLAMP_TO_EDGE = 2
VK_SAMPLER_ADDRESS_MODE_CLAMP_TO_BORDER = 3
VK_SAMPLER_ADDRESS_MODE__BEGIN_RANGE = 0
VK_SAMPLER_ADDRESS_MODE__END_RANGE = 3
VK_SAMPLER_ADDRESS_MODE__RANGE_SIZE = 4
VK_SAMPLER_ADDRESS_MODE__MAX_ENUM = 2147483647
VK_FILTER_NEAREST = 0
VK_FILTER_LINEAR = 1
VK_FILTER__BEGIN_RANGE = 0
VK_FILTER__END_RANGE = 1
VK_FILTER__RANGE_SIZE = 2
VK_FILTER__MAX_ENUM = 2147483647
VK_SAMPLER_MIPMAP_MODE_NEAREST = 0
VK_SAMPLER_MIPMAP_MODE_LINEAR = 1
VK_SAMPLER_MIPMAP_MODE__BEGIN_RANGE = 0
VK_SAMPLER_MIPMAP_MODE__END_RANGE = 1
VK_SAMPLER_MIPMAP_MODE__RANGE_SIZE = 2
VK_SAMPLER_MIPMAP_MODE__MAX_ENUM = 2147483647
VK_VERTEX_INPUT_RATE_VERTEX = 0
VK_VERTEX_INPUT_RATE_INSTANCE = 1
VK_VERTEX_INPUT_RATE__BEGIN_RANGE = 0
VK_VERTEX_INPUT_RATE__END_RANGE = 1
VK_VERTEX_INPUT_RATE__RANGE_SIZE = 2
VK_VERTEX_INPUT_RATE__MAX_ENUM = 2147483647
VK_PIPELINE_STAGE_TOP_OF_PIPE_BIT = 0x00000001
VK_PIPELINE_STAGE_DRAW_INDIRECT_BIT = 0x00000002
VK_PIPELINE_STAGE_VERTEX_INPUT_BIT = 0x00000004
VK_PIPELINE_STAGE_VERTEX_SHADER_BIT = 0x00000008
VK_PIPELINE_STAGE_TESSELLATION_CONTROL_SHADER_BIT = 0x00000010
VK_PIPELINE_STAGE_TESSELLATION_EVALUATION_SHADER_BIT = 0x00000020
VK_PIPELINE_STAGE_GEOMETRY_SHADER_BIT = 0x00000040
VK_PIPELINE_STAGE_FRAGMENT_SHADER_BIT = 0x00000080
VK_PIPELINE_STAGE_EARLY_FRAGMENT_TESTS_BIT = 0x00000100
VK_PIPELINE_STAGE_LATE_FRAGMENT_TESTS_BIT = 0x00000200
VK_PIPELINE_STAGE_COLOR_ATTACHMENT_OUTPUT_BIT = 0x00000400
VK_PIPELINE_STAGE_COMPUTE_SHADER_BIT = 0x00000800
VK_PIPELINE_STAGE_TRANSFER_BIT = 0x00001000
VK_PIPELINE_STAGE_BOTTOM_OF_PIPE_BIT = 0x00002000
VK_PIPELINE_STAGE_HOST_BIT = 0x00004000
VK_PIPELINE_STAGE_ALL_GRAPHICS_BIT = 0x00008000
VK_PIPELINE_STAGE_ALL_COMMANDS_BIT = 0x00010000
VK_PIPELINE_STAGE_FLAG_BITS__MAX_ENUM = 2147483647
VK_SPARSE_IMAGE_FORMAT_SINGLE_MIPTAIL_BIT = 0x00000001
VK_SPARSE_IMAGE_FORMAT_ALIGNED_MIP_SIZE_BIT = 0x00000002
VK_SPARSE_IMAGE_FORMAT_NONSTANDARD_BLOCK_SIZE_BIT = 0x00000004
VK_SPARSE_IMAGE_FORMAT_FLAG_BITS__MAX_ENUM = 2147483647
VK_SAMPLE_COUNT_1_BIT = 0x00000001
VK_SAMPLE_COUNT_2_BIT = 0x00000002
VK_SAMPLE_COUNT_4_BIT = 0x00000004
VK_SAMPLE_COUNT_8_BIT = 0x00000008
VK_SAMPLE_COUNT_16_BIT = 0x00000010
VK_SAMPLE_COUNT_32_BIT = 0x00000020
VK_SAMPLE_COUNT_64_BIT = 0x00000040
VK_SAMPLE_COUNT_FLAG_BITS__MAX_ENUM = 2147483647
VK_ATTACHMENT_DESCRIPTION_MAY_ALIAS_BIT = 0x00000001
VK_ATTACHMENT_DESCRIPTION_FLAG_BITS__MAX_ENUM = 2147483647
VK_DESCRIPTOR_POOL_CREATE_FREE_DESCRIPTOR_SET_BIT = 0x00000001
VK_DESCRIPTOR_POOL_CREATE_FLAG_BITS__MAX_ENUM = 2147483647
VK_DEPENDENCY_BY_REGION_BIT = 0x00000001
VK_DEPENDENCY_FLAG_BITS__MAX_ENUM = 2147483647
VK_OBJECT_TYPE_UNKNOWN = 0
VK_OBJECT_TYPE_INSTANCE = 1
VK_OBJECT_TYPE_PHYSICAL_DEVICE = 2
VK_OBJECT_TYPE_DEVICE = 3
VK_OBJECT_TYPE_QUEUE = 4
VK_OBJECT_TYPE_SEMAPHORE = 5
VK_OBJECT_TYPE_COMMAND_BUFFER = 6
VK_OBJECT_TYPE_FENCE = 7
VK_OBJECT_TYPE_DEVICE_MEMORY = 8
VK_OBJECT_TYPE_BUFFER = 9
VK_OBJECT_TYPE_IMAGE = 10
VK_OBJECT_TYPE_EVENT = 11
VK_OBJECT_TYPE_QUERY_POOL = 12
VK_OBJECT_TYPE_BUFFER_VIEW = 13
VK_OBJECT_TYPE_IMAGE_VIEW = 14
VK_OBJECT_TYPE_SHADER_MODULE = 15
VK_OBJECT_TYPE_PIPELINE_CACHE = 16
VK_OBJECT_TYPE_PIPELINE_LAYOUT = 17
VK_OBJECT_TYPE_RENDER_PASS = 18
VK_OBJECT_TYPE_PIPELINE = 19
VK_OBJECT_TYPE_DESCRIPTOR_SET_LAYOUT = 20
VK_OBJECT_TYPE_SAMPLER = 21
VK_OBJECT_TYPE_DESCRIPTOR_POOL = 22
VK_OBJECT_TYPE_DESCRIPTOR_SET = 23
VK_OBJECT_TYPE_FRAMEBUFFER = 24
VK_OBJECT_TYPE_COMMAND_POOL = 25
VK_OBJECT_TYPE__BEGIN_RANGE = 0
VK_OBJECT_TYPE__END_RANGE = 25
VK_OBJECT_TYPE__RANGE_SIZE = 26
VK_OBJECT_TYPE__MAX_ENUM = 2147483647
VK_OBJECT_TYPE_SAMPLER_YCBCR_CONVERSION = 1000156000
VK_OBJECT_TYPE_DESCRIPTOR_UPDATE_TEMPLATE = 1000085000
VK_DESCRIPTOR_BINDING_UPDATE_AFTER_BIND_BIT_EXT = 0x00000001
VK_DESCRIPTOR_BINDING_UPDATE_UNUSED_WHILE_PENDING_BIT_EXT = 0x00000002
VK_DESCRIPTOR_BINDING_PARTIALLY_BOUND_BIT_EXT = 0x00000004
VK_DESCRIPTOR_BINDING_VARIABLE_DESCRIPTOR_COUNT_BIT_EXT = 0x00000008
VK_DESCRIPTOR_BINDING_FLAG_BITS_EXT__MAX_ENUM = 2147483647
VK_CONDITIONAL_RENDERING_INVERTED_BIT_EXT = 0x00000001
VK_CONDITIONAL_RENDERING_FLAG_BITS_EXT__MAX_ENUM = 2147483647
VK_INDIRECT_COMMANDS_LAYOUT_USAGE_UNORDERED_SEQUENCES_BIT_NVX = 0x00000001
VK_INDIRECT_COMMANDS_LAYOUT_USAGE_SPARSE_SEQUENCES_BIT_NVX = 0x00000002
VK_INDIRECT_COMMANDS_LAYOUT_USAGE_EMPTY_EXECUTIONS_BIT_NVX = 0x00000004
VK_INDIRECT_COMMANDS_LAYOUT_USAGE_INDEXED_SEQUENCES_BIT_NVX = 0x00000008
VK_INDIRECT_COMMANDS_LAYOUT_USAGE_FLAG_BITS_NVX__MAX_ENUM = 2147483647
VK_INDIRECT_COMMANDS_TOKEN_TYPE_PIPELINE_NVX = 0
VK_INDIRECT_COMMANDS_TOKEN_TYPE_DESCRIPTOR_SET_NVX = 1
VK_INDIRECT_COMMANDS_TOKEN_TYPE_INDEX_BUFFER_NVX = 2
VK_INDIRECT_COMMANDS_TOKEN_TYPE_VERTEX_BUFFER_NVX = 3
VK_INDIRECT_COMMANDS_TOKEN_TYPE_PUSH_CONSTANT_NVX = 4
VK_INDIRECT_COMMANDS_TOKEN_TYPE_DRAW_INDEXED_NVX = 5
VK_INDIRECT_COMMANDS_TOKEN_TYPE_DRAW_NVX = 6
VK_INDIRECT_COMMANDS_TOKEN_TYPE_DISPATCH_NVX = 7
VK_INDIRECT_COMMANDS_TOKEN_TYPE_NVX__BEGIN_RANGE = 0
VK_INDIRECT_COMMANDS_TOKEN_TYPE_NVX__END_RANGE = 7
VK_INDIRECT_COMMANDS_TOKEN_TYPE_NVX__RANGE_SIZE = 8
VK_INDIRECT_COMMANDS_TOKEN_TYPE_NVX__MAX_ENUM = 2147483647
VK_OBJECT_ENTRY_USAGE_GRAPHICS_BIT_NVX = 0x00000001
VK_OBJECT_ENTRY_USAGE_COMPUTE_BIT_NVX = 0x00000002
VK_OBJECT_ENTRY_USAGE_FLAG_BITS_NVX__MAX_ENUM = 2147483647
VK_OBJECT_ENTRY_TYPE_DESCRIPTOR_SET_NVX = 0
VK_OBJECT_ENTRY_TYPE_PIPELINE_NVX = 1
VK_OBJECT_ENTRY_TYPE_INDEX_BUFFER_NVX = 2
VK_OBJECT_ENTRY_TYPE_VERTEX_BUFFER_NVX = 3
VK_OBJECT_ENTRY_TYPE_PUSH_CONSTANT_NVX = 4
VK_OBJECT_ENTRY_TYPE_NVX__BEGIN_RANGE = 0
VK_OBJECT_ENTRY_TYPE_NVX__END_RANGE = 4
VK_OBJECT_ENTRY_TYPE_NVX__RANGE_SIZE = 5
VK_OBJECT_ENTRY_TYPE_NVX__MAX_ENUM = 2147483647
VK_DESCRIPTOR_UPDATE_TEMPLATE_TYPE_DESCRIPTOR_SET = 0
VK_DESCRIPTOR_UPDATE_TEMPLATE_TYPE__BEGIN_RANGE = 0
VK_DESCRIPTOR_UPDATE_TEMPLATE_TYPE__END_RANGE = 0
VK_DESCRIPTOR_UPDATE_TEMPLATE_TYPE__RANGE_SIZE = 1
VK_DESCRIPTOR_UPDATE_TEMPLATE_TYPE__MAX_ENUM = 2147483647
VK_VIEWPORT_COORDINATE_SWIZZLE_POSITIVE_X_NV = 0
VK_VIEWPORT_COORDINATE_SWIZZLE_NEGATIVE_X_NV = 1
VK_VIEWPORT_COORDINATE_SWIZZLE_POSITIVE_Y_NV = 2
VK_VIEWPORT_COORDINATE_SWIZZLE_NEGATIVE_Y_NV = 3
VK_VIEWPORT_COORDINATE_SWIZZLE_POSITIVE_Z_NV = 4
VK_VIEWPORT_COORDINATE_SWIZZLE_NEGATIVE_Z_NV = 5
VK_VIEWPORT_COORDINATE_SWIZZLE_POSITIVE_W_NV = 6
VK_VIEWPORT_COORDINATE_SWIZZLE_NEGATIVE_W_NV = 7
VK_VIEWPORT_COORDINATE_SWIZZLE_NV__BEGIN_RANGE = 0
VK_VIEWPORT_COORDINATE_SWIZZLE_NV__END_RANGE = 7
VK_VIEWPORT_COORDINATE_SWIZZLE_NV__RANGE_SIZE = 8
VK_VIEWPORT_COORDINATE_SWIZZLE_NV__MAX_ENUM = 2147483647
VK_DISCARD_RECTANGLE_MODE_INCLUSIVE_EXT = 0
VK_DISCARD_RECTANGLE_MODE_EXCLUSIVE_EXT = 1
VK_DISCARD_RECTANGLE_MODE_EXT__BEGIN_RANGE = 0
VK_DISCARD_RECTANGLE_MODE_EXT__END_RANGE = 1
VK_DISCARD_RECTANGLE_MODE_EXT__RANGE_SIZE = 2
VK_DISCARD_RECTANGLE_MODE_EXT__MAX_ENUM = 2147483647
VK_POINT_CLIPPING_BEHAVIOR_ALL_CLIP_PLANES = 0
VK_POINT_CLIPPING_BEHAVIOR_USER_CLIP_PLANES_ONLY = 1
VK_POINT_CLIPPING_BEHAVIOR__BEGIN_RANGE = 0
VK_POINT_CLIPPING_BEHAVIOR__END_RANGE = 1
VK_POINT_CLIPPING_BEHAVIOR__RANGE_SIZE = 2
VK_POINT_CLIPPING_BEHAVIOR__MAX_ENUM = 2147483647
VK_COVERAGE_MODULATION_MODE_NONE_NV = 0
VK_COVERAGE_MODULATION_MODE_RGB_NV = 1
VK_COVERAGE_MODULATION_MODE_ALPHA_NV = 2
VK_COVERAGE_MODULATION_MODE_RGBA_NV = 3
VK_COVERAGE_MODULATION_MODE_NV__BEGIN_RANGE = 0
VK_COVERAGE_MODULATION_MODE_NV__END_RANGE = 3
VK_COVERAGE_MODULATION_MODE_NV__RANGE_SIZE = 4
VK_COVERAGE_MODULATION_MODE_NV__MAX_ENUM = 2147483647
VK_VALIDATION_CACHE_HEADER_VERSION_ONE_EXT = 1
VK_VALIDATION_CACHE_HEADER_VERSION_EXT__BEGIN_RANGE = 1
VK_VALIDATION_CACHE_HEADER_VERSION_EXT__END_RANGE = 1
VK_VALIDATION_CACHE_HEADER_VERSION_EXT__RANGE_SIZE = 1
VK_VALIDATION_CACHE_HEADER_VERSION_EXT__MAX_ENUM = 2147483647
VK_SHADER_INFO_TYPE_STATISTICS_AMD = 0
VK_SHADER_INFO_TYPE_BINARY_AMD = 1
VK_SHADER_INFO_TYPE_DISASSEMBLY_AMD = 2
VK_SHADER_INFO_TYPE_AMD__BEGIN_RANGE = 0
VK_SHADER_INFO_TYPE_AMD__END_RANGE = 2
VK_SHADER_INFO_TYPE_AMD__RANGE_SIZE = 3
VK_SHADER_INFO_TYPE_AMD__MAX_ENUM = 2147483647
VK_QUEUE_GLOBAL_PRIORITY_LOW_EXT = 128
VK_QUEUE_GLOBAL_PRIORITY_MEDIUM_EXT = 256
VK_QUEUE_GLOBAL_PRIORITY_HIGH_EXT = 512
VK_QUEUE_GLOBAL_PRIORITY_REALTIME_EXT = 1024
VK_QUEUE_GLOBAL_PRIORITY_EXT__BEGIN_RANGE = 128
VK_QUEUE_GLOBAL_PRIORITY_EXT__END_RANGE = 1024
VK_QUEUE_GLOBAL_PRIORITY_EXT__RANGE_SIZE = 897
VK_QUEUE_GLOBAL_PRIORITY_EXT__MAX_ENUM = 2147483647
VK_TIME_DOMAIN_DEVICE_EXT = 0
VK_TIME_DOMAIN_CLOCK_MONOTONIC_EXT = 1
VK_TIME_DOMAIN_CLOCK_MONOTONIC_RAW_EXT = 2
VK_TIME_DOMAIN_QUERY_PERFORMANCE_COUNTER_EXT = 3
VK_TIME_DOMAIN_EXT__BEGIN_RANGE = 0
VK_TIME_DOMAIN_EXT__END_RANGE = 3
VK_TIME_DOMAIN_EXT__RANGE_SIZE = 4
VK_TIME_DOMAIN_EXT__MAX_ENUM = 2147483647
VK_CONSERVATIVE_RASTERIZATION_MODE_DISABLED_EXT = 0
VK_CONSERVATIVE_RASTERIZATION_MODE_OVERESTIMATE_EXT = 1
VK_CONSERVATIVE_RASTERIZATION_MODE_UNDERESTIMATE_EXT = 2
VK_CONSERVATIVE_RASTERIZATION_MODE_EXT__BEGIN_RANGE = 0
VK_CONSERVATIVE_RASTERIZATION_MODE_EXT__END_RANGE = 2
VK_CONSERVATIVE_RASTERIZATION_MODE_EXT__RANGE_SIZE = 3
VK_CONSERVATIVE_RASTERIZATION_MODE_EXT__MAX_ENUM = 2147483647
VK_RESOLVE_MODE_NONE_KHR = 0
VK_RESOLVE_MODE_SAMPLE_ZERO_BIT_KHR = 0x00000001
VK_RESOLVE_MODE_AVERAGE_BIT_KHR = 0x00000002
VK_RESOLVE_MODE_MIN_BIT_KHR = 0x00000004
VK_RESOLVE_MODE_MAX_BIT_KHR = 0x00000008
VK_RESOLVE_MODE_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_GEOMETRY_OPAQUE_BIT_NV = 0x00000001
VK_GEOMETRY_NO_DUPLICATE_ANY_HIT_INVOCATION_BIT_NV = 0x00000002
VK_GEOMETRY_FLAG_BITS_NV__MAX_ENUM = 2147483647
VK_GEOMETRY_INSTANCE_TRIANGLE_CULL_DISABLE_BIT_NV = 0x00000001
VK_GEOMETRY_INSTANCE_TRIANGLE_FRONT_COUNTERCLOCKWISE_BIT_NV = 0x00000002
VK_GEOMETRY_INSTANCE_FORCE_OPAQUE_BIT_NV = 0x00000004
VK_GEOMETRY_INSTANCE_FORCE_NO_OPAQUE_BIT_NV = 0x00000008
VK_GEOMETRY_INSTANCE_FLAG_BITS_NV__MAX_ENUM = 2147483647
VK_BUILD_ACCELERATION_STRUCTURE_ALLOW_UPDATE_BIT_NV = 0x00000001
VK_BUILD_ACCELERATION_STRUCTURE_ALLOW_COMPACTION_BIT_NV = 0x00000002
VK_BUILD_ACCELERATION_STRUCTURE_PREFER_FAST_TRACE_BIT_NV = 0x00000004
VK_BUILD_ACCELERATION_STRUCTURE_PREFER_FAST_BUILD_BIT_NV = 0x00000008
VK_BUILD_ACCELERATION_STRUCTURE_LOW_MEMORY_BIT_NV = 0x00000010
VK_BUILD_ACCELERATION_STRUCTURE_FLAG_BITS_NV__MAX_ENUM = 2147483647
VK_COPY_ACCELERATION_STRUCTURE_MODE_CLONE_NV = 0
VK_COPY_ACCELERATION_STRUCTURE_MODE_COMPACT_NV = 1
VK_COPY_ACCELERATION_STRUCTURE_MODE_NV__BEGIN_RANGE = 0
VK_COPY_ACCELERATION_STRUCTURE_MODE_NV__END_RANGE = 1
VK_COPY_ACCELERATION_STRUCTURE_MODE_NV__RANGE_SIZE = 2
VK_COPY_ACCELERATION_STRUCTURE_MODE_NV__MAX_ENUM = 2147483647
VK_ACCELERATION_STRUCTURE_TYPE_TOP_LEVEL_NV = 0
VK_ACCELERATION_STRUCTURE_TYPE_BOTTOM_LEVEL_NV = 1
VK_ACCELERATION_STRUCTURE_TYPE_NV__BEGIN_RANGE = 0
VK_ACCELERATION_STRUCTURE_TYPE_NV__END_RANGE = 1
VK_ACCELERATION_STRUCTURE_TYPE_NV__RANGE_SIZE = 2
VK_ACCELERATION_STRUCTURE_TYPE_NV__MAX_ENUM = 2147483647
VK_GEOMETRY_TYPE_TRIANGLES_NV = 0
VK_GEOMETRY_TYPE_AABBS_NV = 1
VK_GEOMETRY_TYPE_NV__BEGIN_RANGE = 0
VK_GEOMETRY_TYPE_NV__END_RANGE = 1
VK_GEOMETRY_TYPE_NV__RANGE_SIZE = 2
VK_GEOMETRY_TYPE_NV__MAX_ENUM = 2147483647
VK_RAY_TRACING_SHADER_GROUP_TYPE_GENERAL_NV = 0
VK_RAY_TRACING_SHADER_GROUP_TYPE_TRIANGLES_HIT_GROUP_NV = 1
VK_RAY_TRACING_SHADER_GROUP_TYPE_PROCEDURAL_HIT_GROUP_NV = 2
VK_RAY_TRACING_SHADER_GROUP_TYPE_NV__BEGIN_RANGE = 0
VK_RAY_TRACING_SHADER_GROUP_TYPE_NV__END_RANGE = 2
VK_RAY_TRACING_SHADER_GROUP_TYPE_NV__RANGE_SIZE = 3
VK_RAY_TRACING_SHADER_GROUP_TYPE_NV__MAX_ENUM = 2147483647
VK_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_TYPE_OBJECT_NV = 0
VK_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_TYPE_BUILD_SCRATCH_NV = 1
VK_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_TYPE_UPDATE_SCRATCH_NV = 2
VK_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_TYPE_NV__BEGIN_RANGE = 0
VK_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_TYPE_NV__END_RANGE = 2
VK_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_TYPE_NV__RANGE_SIZE = 3
VK_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_TYPE_NV__MAX_ENUM = 2147483647
VK_MEMORY_OVERALLOCATION_BEHAVIOR_DEFAULT_AMD = 0
VK_MEMORY_OVERALLOCATION_BEHAVIOR_ALLOWED_AMD = 1
VK_MEMORY_OVERALLOCATION_BEHAVIOR_DISALLOWED_AMD = 2
VK_MEMORY_OVERALLOCATION_BEHAVIOR_AMD__BEGIN_RANGE = 0
VK_MEMORY_OVERALLOCATION_BEHAVIOR_AMD__END_RANGE = 2
VK_MEMORY_OVERALLOCATION_BEHAVIOR_AMD__RANGE_SIZE = 3
VK_MEMORY_OVERALLOCATION_BEHAVIOR_AMD__MAX_ENUM = 2147483647
VK_COLOR_SPACE_SRGB_NONLINEAR_KHR = 0
VK_COLORSPACE_SRGB_NONLINEAR_KHR = 0
VK_COLOR_SPACE_KHR__BEGIN_RANGE = 0
VK_COLOR_SPACE_KHR__END_RANGE = 0
VK_COLOR_SPACE_KHR__RANGE_SIZE = 1
VK_COLOR_SPACE_KHR__MAX_ENUM = 2147483647
VK_COMPOSITE_ALPHA_OPAQUE_BIT_KHR = 0x00000001
VK_COMPOSITE_ALPHA_PRE_MULTIPLIED_BIT_KHR = 0x00000002
VK_COMPOSITE_ALPHA_POST_MULTIPLIED_BIT_KHR = 0x00000004
VK_COMPOSITE_ALPHA_INHERIT_BIT_KHR = 0x00000008
VK_COMPOSITE_ALPHA_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_DISPLAY_PLANE_ALPHA_OPAQUE_BIT_KHR = 0x00000001
VK_DISPLAY_PLANE_ALPHA_GLOBAL_BIT_KHR = 0x00000002
VK_DISPLAY_PLANE_ALPHA_PER_PIXEL_BIT_KHR = 0x00000004
VK_DISPLAY_PLANE_ALPHA_PER_PIXEL_PREMULTIPLIED_BIT_KHR = 0x00000008
VK_DISPLAY_PLANE_ALPHA_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_PRESENT_MODE_IMMEDIATE_KHR = 0
VK_PRESENT_MODE_MAILBOX_KHR = 1
VK_PRESENT_MODE_FIFO_KHR = 2
VK_PRESENT_MODE_FIFO_RELAXED_KHR = 3
VK_PRESENT_MODE_KHR__BEGIN_RANGE = 0
VK_PRESENT_MODE_KHR__END_RANGE = 3
VK_PRESENT_MODE_KHR__RANGE_SIZE = 4
VK_PRESENT_MODE_KHR__MAX_ENUM = 2147483647
VK_SURFACE_TRANSFORM_IDENTITY_BIT_KHR = 0x00000001
VK_SURFACE_TRANSFORM_ROTATE_90_BIT_KHR = 0x00000002
VK_SURFACE_TRANSFORM_ROTATE_180_BIT_KHR = 0x00000004
VK_SURFACE_TRANSFORM_ROTATE_270_BIT_KHR = 0x00000008
VK_SURFACE_TRANSFORM_HORIZONTAL_MIRROR_BIT_KHR = 0x00000010
VK_SURFACE_TRANSFORM_HORIZONTAL_MIRROR_ROTATE_90_BIT_KHR = 0x00000020
VK_SURFACE_TRANSFORM_HORIZONTAL_MIRROR_ROTATE_180_BIT_KHR = 0x00000040
VK_SURFACE_TRANSFORM_HORIZONTAL_MIRROR_ROTATE_270_BIT_KHR = 0x00000080
VK_SURFACE_TRANSFORM_INHERIT_BIT_KHR = 0x00000100
VK_SURFACE_TRANSFORM_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_DEBUG_REPORT_INFORMATION_BIT_EXT = 0x00000001
VK_DEBUG_REPORT_WARNING_BIT_EXT = 0x00000002
VK_DEBUG_REPORT_PERFORMANCE_WARNING_BIT_EXT = 0x00000004
VK_DEBUG_REPORT_ERROR_BIT_EXT = 0x00000008
VK_DEBUG_REPORT_DEBUG_BIT_EXT = 0x00000010
VK_DEBUG_REPORT_FLAG_BITS_EXT__MAX_ENUM = 2147483647
VK_DEBUG_REPORT_OBJECT_TYPE_UNKNOWN_EXT = 0
VK_DEBUG_REPORT_OBJECT_TYPE_INSTANCE_EXT = 1
VK_DEBUG_REPORT_OBJECT_TYPE_PHYSICAL_DEVICE_EXT = 2
VK_DEBUG_REPORT_OBJECT_TYPE_DEVICE_EXT = 3
VK_DEBUG_REPORT_OBJECT_TYPE_QUEUE_EXT = 4
VK_DEBUG_REPORT_OBJECT_TYPE_SEMAPHORE_EXT = 5
VK_DEBUG_REPORT_OBJECT_TYPE_COMMAND_BUFFER_EXT = 6
VK_DEBUG_REPORT_OBJECT_TYPE_FENCE_EXT = 7
VK_DEBUG_REPORT_OBJECT_TYPE_DEVICE_MEMORY_EXT = 8
VK_DEBUG_REPORT_OBJECT_TYPE_BUFFER_EXT = 9
VK_DEBUG_REPORT_OBJECT_TYPE_IMAGE_EXT = 10
VK_DEBUG_REPORT_OBJECT_TYPE_EVENT_EXT = 11
VK_DEBUG_REPORT_OBJECT_TYPE_QUERY_POOL_EXT = 12
VK_DEBUG_REPORT_OBJECT_TYPE_BUFFER_VIEW_EXT = 13
VK_DEBUG_REPORT_OBJECT_TYPE_IMAGE_VIEW_EXT = 14
VK_DEBUG_REPORT_OBJECT_TYPE_SHADER_MODULE_EXT = 15
VK_DEBUG_REPORT_OBJECT_TYPE_PIPELINE_CACHE_EXT = 16
VK_DEBUG_REPORT_OBJECT_TYPE_PIPELINE_LAYOUT_EXT = 17
VK_DEBUG_REPORT_OBJECT_TYPE_RENDER_PASS_EXT = 18
VK_DEBUG_REPORT_OBJECT_TYPE_PIPELINE_EXT = 19
VK_DEBUG_REPORT_OBJECT_TYPE_DESCRIPTOR_SET_LAYOUT_EXT = 20
VK_DEBUG_REPORT_OBJECT_TYPE_SAMPLER_EXT = 21
VK_DEBUG_REPORT_OBJECT_TYPE_DESCRIPTOR_POOL_EXT = 22
VK_DEBUG_REPORT_OBJECT_TYPE_DESCRIPTOR_SET_EXT = 23
VK_DEBUG_REPORT_OBJECT_TYPE_FRAMEBUFFER_EXT = 24
VK_DEBUG_REPORT_OBJECT_TYPE_COMMAND_POOL_EXT = 25
VK_DEBUG_REPORT_OBJECT_TYPE_SURFACE_KHR_EXT = 26
VK_DEBUG_REPORT_OBJECT_TYPE_SWAPCHAIN_KHR_EXT = 27
VK_DEBUG_REPORT_OBJECT_TYPE_DEBUG_REPORT_CALLBACK_EXT_EXT = 28
VK_DEBUG_REPORT_OBJECT_TYPE_DEBUG_REPORT_EXT = 28
VK_DEBUG_REPORT_OBJECT_TYPE_DISPLAY_KHR_EXT = 29
VK_DEBUG_REPORT_OBJECT_TYPE_DISPLAY_MODE_KHR_EXT = 30
VK_DEBUG_REPORT_OBJECT_TYPE_OBJECT_TABLE_NVX_EXT = 31
VK_DEBUG_REPORT_OBJECT_TYPE_INDIRECT_COMMANDS_LAYOUT_NVX_EXT = 32
VK_DEBUG_REPORT_OBJECT_TYPE_VALIDATION_CACHE_EXT_EXT = 33
VK_DEBUG_REPORT_OBJECT_TYPE_VALIDATION_CACHE_EXT = 33
VK_DEBUG_REPORT_OBJECT_TYPE_EXT__BEGIN_RANGE = 0
VK_DEBUG_REPORT_OBJECT_TYPE_EXT__END_RANGE = 33
VK_DEBUG_REPORT_OBJECT_TYPE_EXT__RANGE_SIZE = 34
VK_DEBUG_REPORT_OBJECT_TYPE_EXT__MAX_ENUM = 2147483647
VK_RASTERIZATION_ORDER_STRICT_AMD = 0
VK_RASTERIZATION_ORDER_RELAXED_AMD = 1
VK_RASTERIZATION_ORDER_AMD__BEGIN_RANGE = 0
VK_RASTERIZATION_ORDER_AMD__END_RANGE = 1
VK_RASTERIZATION_ORDER_AMD__RANGE_SIZE = 2
VK_RASTERIZATION_ORDER_AMD__MAX_ENUM = 2147483647
VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32_BIT_NV = 0x00000001
VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32_KMT_BIT_NV = 0x00000002
VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_IMAGE_BIT_NV = 0x00000004
VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_IMAGE_KMT_BIT_NV = 0x00000008
VK_EXTERNAL_MEMORY_HANDLE_TYPE_FLAG_BITS_NV__MAX_ENUM = 2147483647
VK_EXTERNAL_MEMORY_FEATURE_DEDICATED_ONLY_BIT_NV = 0x00000001
VK_EXTERNAL_MEMORY_FEATURE_EXPORTABLE_BIT_NV = 0x00000002
VK_EXTERNAL_MEMORY_FEATURE_IMPORTABLE_BIT_NV = 0x00000004
VK_EXTERNAL_MEMORY_FEATURE_FLAG_BITS_NV__MAX_ENUM = 2147483647
VK_VALIDATION_CHECK_ALL_EXT = 0
VK_VALIDATION_CHECK_SHADERS_EXT = 1
VK_VALIDATION_CHECK_EXT__BEGIN_RANGE = 0
VK_VALIDATION_CHECK_EXT__END_RANGE = 1
VK_VALIDATION_CHECK_EXT__RANGE_SIZE = 2
VK_VALIDATION_CHECK_EXT__MAX_ENUM = 2147483647
VK_VALIDATION_FEATURE_ENABLE_GPU_ASSISTED_EXT = 0
VK_VALIDATION_FEATURE_ENABLE_GPU_ASSISTED_RESERVE_BINDING_SLOT_EXT = 1
VK_VALIDATION_FEATURE_ENABLE_EXT__BEGIN_RANGE = 0
VK_VALIDATION_FEATURE_ENABLE_EXT__END_RANGE = 1
VK_VALIDATION_FEATURE_ENABLE_EXT__RANGE_SIZE = 2
VK_VALIDATION_FEATURE_ENABLE_EXT__MAX_ENUM = 2147483647
VK_VALIDATION_FEATURE_DISABLE_ALL_EXT = 0
VK_VALIDATION_FEATURE_DISABLE_SHADERS_EXT = 1
VK_VALIDATION_FEATURE_DISABLE_THREAD_SAFETY_EXT = 2
VK_VALIDATION_FEATURE_DISABLE_API_PARAMETERS_EXT = 3
VK_VALIDATION_FEATURE_DISABLE_OBJECT_LIFETIMES_EXT = 4
VK_VALIDATION_FEATURE_DISABLE_CORE_CHECKS_EXT = 5
VK_VALIDATION_FEATURE_DISABLE_UNIQUE_HANDLES_EXT = 6
VK_VALIDATION_FEATURE_DISABLE_EXT__BEGIN_RANGE = 0
VK_VALIDATION_FEATURE_DISABLE_EXT__END_RANGE = 6
VK_VALIDATION_FEATURE_DISABLE_EXT__RANGE_SIZE = 7
VK_VALIDATION_FEATURE_DISABLE_EXT__MAX_ENUM = 2147483647
VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_FD_BIT = 0x00000001
VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32_BIT = 0x00000002
VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32_KMT_BIT = 0x00000004
VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_TEXTURE_BIT = 0x00000008
VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_TEXTURE_KMT_BIT = 0x00000010
VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D12_HEAP_BIT = 0x00000020
VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D12_RESOURCE_BIT = 0x00000040
VK_EXTERNAL_MEMORY_HANDLE_TYPE_FLAG_BITS__MAX_ENUM = 2147483647
VK_EXTERNAL_MEMORY_FEATURE_DEDICATED_ONLY_BIT = 0x00000001
VK_EXTERNAL_MEMORY_FEATURE_EXPORTABLE_BIT = 0x00000002
VK_EXTERNAL_MEMORY_FEATURE_IMPORTABLE_BIT = 0x00000004
VK_EXTERNAL_MEMORY_FEATURE_FLAG_BITS__MAX_ENUM = 2147483647
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_FD_BIT = 0x00000001
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32_BIT = 0x00000002
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32_KMT_BIT = 0x00000004
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D12_FENCE_BIT = 0x00000008
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_SYNC_FD_BIT = 0x00000010
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_FLAG_BITS__MAX_ENUM = 2147483647
VK_EXTERNAL_SEMAPHORE_FEATURE_EXPORTABLE_BIT = 0x00000001
VK_EXTERNAL_SEMAPHORE_FEATURE_IMPORTABLE_BIT = 0x00000002
VK_EXTERNAL_SEMAPHORE_FEATURE_FLAG_BITS__MAX_ENUM = 2147483647
VK_SEMAPHORE_IMPORT_TEMPORARY_BIT = 0x00000001
VK_SEMAPHORE_IMPORT_FLAG_BITS__MAX_ENUM = 2147483647
VK_EXTERNAL_FENCE_HANDLE_TYPE_OPAQUE_FD_BIT = 0x00000001
VK_EXTERNAL_FENCE_HANDLE_TYPE_OPAQUE_WIN32_BIT = 0x00000002
VK_EXTERNAL_FENCE_HANDLE_TYPE_OPAQUE_WIN32_KMT_BIT = 0x00000004
VK_EXTERNAL_FENCE_HANDLE_TYPE_SYNC_FD_BIT = 0x00000008
VK_EXTERNAL_FENCE_HANDLE_TYPE_FLAG_BITS__MAX_ENUM = 2147483647
VK_EXTERNAL_FENCE_FEATURE_EXPORTABLE_BIT = 0x00000001
VK_EXTERNAL_FENCE_FEATURE_IMPORTABLE_BIT = 0x00000002
VK_EXTERNAL_FENCE_FEATURE_FLAG_BITS__MAX_ENUM = 2147483647
VK_FENCE_IMPORT_TEMPORARY_BIT = 0x00000001
VK_FENCE_IMPORT_FLAG_BITS__MAX_ENUM = 2147483647
VK_SURFACE_COUNTER_VBLANK_EXT = 0x00000001
VK_SURFACE_COUNTER_FLAG_BITS_EXT__MAX_ENUM = 2147483647
VK_DISPLAY_POWER_STATE_OFF_EXT = 0
VK_DISPLAY_POWER_STATE_SUSPEND_EXT = 1
VK_DISPLAY_POWER_STATE_ON_EXT = 2
VK_DISPLAY_POWER_STATE_EXT__BEGIN_RANGE = 0
VK_DISPLAY_POWER_STATE_EXT__END_RANGE = 2
VK_DISPLAY_POWER_STATE_EXT__RANGE_SIZE = 3
VK_DISPLAY_POWER_STATE_EXT__MAX_ENUM = 2147483647
VK_DEVICE_EVENT_TYPE_DISPLAY_HOTPLUG_EXT = 0
VK_DEVICE_EVENT_TYPE_EXT__BEGIN_RANGE = 0
VK_DEVICE_EVENT_TYPE_EXT__END_RANGE = 0
VK_DEVICE_EVENT_TYPE_EXT__RANGE_SIZE = 1
VK_DEVICE_EVENT_TYPE_EXT__MAX_ENUM = 2147483647
VK_DISPLAY_EVENT_TYPE_FIRST_PIXEL_OUT_EXT = 0
VK_DISPLAY_EVENT_TYPE_EXT__BEGIN_RANGE = 0
VK_DISPLAY_EVENT_TYPE_EXT__END_RANGE = 0
VK_DISPLAY_EVENT_TYPE_EXT__RANGE_SIZE = 1
VK_DISPLAY_EVENT_TYPE_EXT__MAX_ENUM = 2147483647
VK_PEER_MEMORY_FEATURE_COPY_SRC_BIT = 0x00000001
VK_PEER_MEMORY_FEATURE_COPY_DST_BIT = 0x00000002
VK_PEER_MEMORY_FEATURE_GENERIC_SRC_BIT = 0x00000004
VK_PEER_MEMORY_FEATURE_GENERIC_DST_BIT = 0x00000008
VK_PEER_MEMORY_FEATURE_FLAG_BITS__MAX_ENUM = 2147483647
VK_MEMORY_ALLOCATE_DEVICE_MASK_BIT = 0x00000001
VK_MEMORY_ALLOCATE_FLAG_BITS__MAX_ENUM = 2147483647
VK_DEVICE_GROUP_PRESENT_MODE_LOCAL_BIT_KHR = 0x00000001
VK_DEVICE_GROUP_PRESENT_MODE_REMOTE_BIT_KHR = 0x00000002
VK_DEVICE_GROUP_PRESENT_MODE_SUM_BIT_KHR = 0x00000004
VK_DEVICE_GROUP_PRESENT_MODE_LOCAL_MULTI_DEVICE_BIT_KHR = 0x00000008
VK_DEVICE_GROUP_PRESENT_MODE_FLAG_BITS_KHR__MAX_ENUM = 2147483647
VK_SUBGROUP_FEATURE_BASIC_BIT = 0x00000001
VK_SUBGROUP_FEATURE_VOTE_BIT = 0x00000002
VK_SUBGROUP_FEATURE_ARITHMETIC_BIT = 0x00000004
VK_SUBGROUP_FEATURE_BALLOT_BIT = 0x00000008
VK_SUBGROUP_FEATURE_SHUFFLE_BIT = 0x00000010
VK_SUBGROUP_FEATURE_SHUFFLE_RELATIVE_BIT = 0x00000020
VK_SUBGROUP_FEATURE_CLUSTERED_BIT = 0x00000040
VK_SUBGROUP_FEATURE_QUAD_BIT = 0x00000080
VK_SUBGROUP_FEATURE_FLAG_BITS__MAX_ENUM = 2147483647
VK_TESSELLATION_DOMAIN_ORIGIN_UPPER_LEFT = 0
VK_TESSELLATION_DOMAIN_ORIGIN_LOWER_LEFT = 1
VK_TESSELLATION_DOMAIN_ORIGIN__BEGIN_RANGE = 0
VK_TESSELLATION_DOMAIN_ORIGIN__END_RANGE = 1
VK_TESSELLATION_DOMAIN_ORIGIN__RANGE_SIZE = 2
VK_TESSELLATION_DOMAIN_ORIGIN__MAX_ENUM = 2147483647
VK_SAMPLER_YCBCR_MODEL_CONVERSION_RGB_IDENTITY = 0
VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_IDENTITY = 1
VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_709 = 2
VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_601 = 3
VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_2020 = 4
VK_SAMPLER_YCBCR_MODEL_CONVERSION__BEGIN_RANGE = 0
VK_SAMPLER_YCBCR_MODEL_CONVERSION__END_RANGE = 4
VK_SAMPLER_YCBCR_MODEL_CONVERSION__RANGE_SIZE = 5
VK_SAMPLER_YCBCR_MODEL_CONVERSION__MAX_ENUM = 2147483647
VK_SAMPLER_YCBCR_RANGE_ITU_FULL = 0
VK_SAMPLER_YCBCR_RANGE_ITU_NARROW = 1
VK_SAMPLER_YCBCR_RANGE__BEGIN_RANGE = 0
VK_SAMPLER_YCBCR_RANGE__END_RANGE = 1
VK_SAMPLER_YCBCR_RANGE__RANGE_SIZE = 2
VK_SAMPLER_YCBCR_RANGE__MAX_ENUM = 2147483647
VK_CHROMA_LOCATION_COSITED_EVEN = 0
VK_CHROMA_LOCATION_MIDPOINT = 1
VK_CHROMA_LOCATION__BEGIN_RANGE = 0
VK_CHROMA_LOCATION__END_RANGE = 1
VK_CHROMA_LOCATION__RANGE_SIZE = 2
VK_CHROMA_LOCATION__MAX_ENUM = 2147483647
VK_SAMPLER_REDUCTION_MODE_WEIGHTED_AVERAGE_EXT = 0
VK_SAMPLER_REDUCTION_MODE_MIN_EXT = 1
VK_SAMPLER_REDUCTION_MODE_MAX_EXT = 2
VK_SAMPLER_REDUCTION_MODE_EXT__BEGIN_RANGE = 0
VK_SAMPLER_REDUCTION_MODE_EXT__END_RANGE = 2
VK_SAMPLER_REDUCTION_MODE_EXT__RANGE_SIZE = 3
VK_SAMPLER_REDUCTION_MODE_EXT__MAX_ENUM = 2147483647
VK_BLEND_OVERLAP_UNCORRELATED_EXT = 0
VK_BLEND_OVERLAP_DISJOINT_EXT = 1
VK_BLEND_OVERLAP_CONJOINT_EXT = 2
VK_BLEND_OVERLAP_EXT__BEGIN_RANGE = 0
VK_BLEND_OVERLAP_EXT__END_RANGE = 2
VK_BLEND_OVERLAP_EXT__RANGE_SIZE = 3
VK_BLEND_OVERLAP_EXT__MAX_ENUM = 2147483647
VK_DEBUG_UTILS_MESSAGE_SEVERITY_VERBOSE_BIT_EXT = 0x00000001
VK_DEBUG_UTILS_MESSAGE_SEVERITY_INFO_BIT_EXT = 0x00000010
VK_DEBUG_UTILS_MESSAGE_SEVERITY_WARNING_BIT_EXT = 0x00000100
VK_DEBUG_UTILS_MESSAGE_SEVERITY_ERROR_BIT_EXT = 0x00001000
VK_DEBUG_UTILS_MESSAGE_SEVERITY_FLAG_BITS_EXT__MAX_ENUM = 2147483647
VK_DEBUG_UTILS_MESSAGE_TYPE_GENERAL_BIT_EXT = 0x00000001
VK_DEBUG_UTILS_MESSAGE_TYPE_VALIDATION_BIT_EXT = 0x00000002
VK_DEBUG_UTILS_MESSAGE_TYPE_PERFORMANCE_BIT_EXT = 0x00000004
VK_DEBUG_UTILS_MESSAGE_TYPE_FLAG_BITS_EXT__MAX_ENUM = 2147483647
VK_VENDOR_ID_VIV = 0x10001
VK_VENDOR_ID_VSI = 0x10002
VK_VENDOR_ID_KAZAN = 0x10003
VK_VENDOR_ID__BEGIN_RANGE = 65537
VK_VENDOR_ID__END_RANGE = 65539
VK_VENDOR_ID__RANGE_SIZE = 3
VK_VENDOR_ID__MAX_ENUM = 2147483647
VK_DRIVER_ID_AMD_PROPRIETARY_KHR = 1
VK_DRIVER_ID_AMD_OPEN_SOURCE_KHR = 2
VK_DRIVER_ID_MESA_RADV_KHR = 3
VK_DRIVER_ID_NVIDIA_PROPRIETARY_KHR = 4
VK_DRIVER_ID_INTEL_PROPRIETARY_WINDOWS_KHR = 5
VK_DRIVER_ID_INTEL_OPEN_SOURCE_MESA_KHR = 6
VK_DRIVER_ID_IMAGINATION_PROPRIETARY_KHR = 7
VK_DRIVER_ID_QUALCOMM_PROPRIETARY_KHR = 8
VK_DRIVER_ID_ARM_PROPRIETARY_KHR = 9
VK_DRIVER_ID_GOOGLE_PASTEL_KHR = 10
VK_DRIVER_ID_KHR__BEGIN_RANGE = 1
VK_DRIVER_ID_KHR__END_RANGE = 10
VK_DRIVER_ID_KHR__RANGE_SIZE = 10
VK_DRIVER_ID_KHR__MAX_ENUM = 2147483647
VK_SHADING_RATE_PALETTE_ENTRY_NO_INVOCATIONS_NV = 0
VK_SHADING_RATE_PALETTE_ENTRY_16_INVOCATIONS_PER_PIXEL_NV = 1
VK_SHADING_RATE_PALETTE_ENTRY_8_INVOCATIONS_PER_PIXEL_NV = 2
VK_SHADING_RATE_PALETTE_ENTRY_4_INVOCATIONS_PER_PIXEL_NV = 3
VK_SHADING_RATE_PALETTE_ENTRY_2_INVOCATIONS_PER_PIXEL_NV = 4
VK_SHADING_RATE_PALETTE_ENTRY_1_INVOCATION_PER_PIXEL_NV = 5
VK_SHADING_RATE_PALETTE_ENTRY_1_INVOCATION_PER_2X1_PIXELS_NV = 6
VK_SHADING_RATE_PALETTE_ENTRY_1_INVOCATION_PER_1X2_PIXELS_NV = 7
VK_SHADING_RATE_PALETTE_ENTRY_1_INVOCATION_PER_2X2_PIXELS_NV = 8
VK_SHADING_RATE_PALETTE_ENTRY_1_INVOCATION_PER_4X2_PIXELS_NV = 9
VK_SHADING_RATE_PALETTE_ENTRY_1_INVOCATION_PER_2X4_PIXELS_NV = 10
VK_SHADING_RATE_PALETTE_ENTRY_1_INVOCATION_PER_4X4_PIXELS_NV = 11
VK_SHADING_RATE_PALETTE_ENTRY_NV__BEGIN_RANGE = 0
VK_SHADING_RATE_PALETTE_ENTRY_NV__END_RANGE = 11
VK_SHADING_RATE_PALETTE_ENTRY_NV__RANGE_SIZE = 12
VK_SHADING_RATE_PALETTE_ENTRY_NV__MAX_ENUM = 2147483647
VK_COARSE_SAMPLE_ORDER_TYPE_DEFAULT_NV = 0
VK_COARSE_SAMPLE_ORDER_TYPE_CUSTOM_NV = 1
VK_COARSE_SAMPLE_ORDER_TYPE_PIXEL_MAJOR_NV = 2
VK_COARSE_SAMPLE_ORDER_TYPE_SAMPLE_MAJOR_NV = 3
VK_COARSE_SAMPLE_ORDER_TYPE_NV__BEGIN_RANGE = 0
VK_COARSE_SAMPLE_ORDER_TYPE_NV__END_RANGE = 3
VK_COARSE_SAMPLE_ORDER_TYPE_NV__RANGE_SIZE = 4
VK_COARSE_SAMPLE_ORDER_TYPE_NV__MAX_ENUM = 2147483647

def VK_MAKE_VERSION(major, minor, patch):
    return (((major) << 22) | ((minor) << 12) | (patch))


def VK_VERSION_MAJOR(version):
    return version >> 22


def VK_VERSION_MINOR(version):
    return (version >> 12) & 0x3ff


def VK_VERSION_PATCH(version):
    return version & 0xfff


VK_API_VERSION = VK_MAKE_VERSION(1, 0, 0)
VK_API_VERSION_1_0 = VK_MAKE_VERSION(1, 0, 0)
VK_NULL_HANDLE = 0
_UINT64_MAX = ffi.new('unsigned long long int*', 18446744073709551615)
UINT64_MAX = _UINT64_MAX[0]


VK_MAX_PHYSICAL_DEVICE_NAME_SIZE = 256
VK_UUID_SIZE = 16
VK_LUID_SIZE = 8
VK_MAX_EXTENSION_NAME_SIZE = 256
VK_MAX_DESCRIPTION_SIZE = 256
VK_MAX_MEMORY_TYPES = 32
VK_MAX_MEMORY_HEAPS = 16
VK_LOD_CLAMP_NONE = 1000.0
VK_REMAINING_MIP_LEVELS = 4294967295
VK_REMAINING_ARRAY_LAYERS = 4294967295
VK_WHOLE_SIZE = -1
VK_ATTACHMENT_UNUSED = 4294967295
VK_TRUE = 1
VK_FALSE = 0
VK_QUEUE_FAMILY_IGNORED = 4294967295
VK_QUEUE_FAMILY_EXTERNAL = 4294967294
VK_QUEUE_FAMILY_FOREIGN_EXT = 4294967293
VK_SUBPASS_EXTERNAL = 4294967295
VK_MAX_DEVICE_GROUP_SIZE = 32
VK_MAX_DRIVER_NAME_SIZE_KHR = 256
VK_MAX_DRIVER_INFO_SIZE_KHR = 256
VK_SHADER_UNUSED_NV = 4294967295
VK_KHR_surface = 1
VK_KHR_SURFACE_SPEC_VERSION = 25
VK_KHR_SURFACE_EXTENSION_NAME = "VK_KHR_surface"
VK_OBJECT_TYPE_SURFACE_KHR = 1000000000
VK_KHR_swapchain = 1
VK_KHR_SWAPCHAIN_SPEC_VERSION = 70
VK_KHR_SWAPCHAIN_EXTENSION_NAME = "VK_KHR_swapchain"
VK_STRUCTURE_TYPE_SWAPCHAIN_CREATE_INFO_KHR = 1000001000
VK_STRUCTURE_TYPE_PRESENT_INFO_KHR = 1000001001
VK_IMAGE_LAYOUT_PRESENT_SRC_KHR = 1000001002
VK_OBJECT_TYPE_SWAPCHAIN_KHR = 1000001000
VK_STRUCTURE_TYPE_DEVICE_GROUP_PRESENT_CAPABILITIES_KHR = 1000001007
VK_STRUCTURE_TYPE_IMAGE_SWAPCHAIN_CREATE_INFO_KHR = 1000001008
VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_SWAPCHAIN_INFO_KHR = 1000001009
VK_STRUCTURE_TYPE_ACQUIRE_NEXT_IMAGE_INFO_KHR = 1000001010
VK_STRUCTURE_TYPE_DEVICE_GROUP_PRESENT_INFO_KHR = 1000001011
VK_STRUCTURE_TYPE_DEVICE_GROUP_SWAPCHAIN_CREATE_INFO_KHR = 1000001012
VK_SWAPCHAIN_CREATE_SPLIT_INSTANCE_BIND_REGIONS_BIT_KHR = 0x00000001
VK_SWAPCHAIN_CREATE_PROTECTED_BIT_KHR = 0x00000002
VK_KHR_display = 1
VK_KHR_DISPLAY_SPEC_VERSION = 21
VK_KHR_DISPLAY_EXTENSION_NAME = "VK_KHR_display"
VK_STRUCTURE_TYPE_DISPLAY_MODE_CREATE_INFO_KHR = 1000002000
VK_STRUCTURE_TYPE_DISPLAY_SURFACE_CREATE_INFO_KHR = 1000002001
VK_OBJECT_TYPE_DISPLAY_KHR = 1000002000
VK_OBJECT_TYPE_DISPLAY_MODE_KHR = 1000002001
VK_KHR_display_swapchain = 1
VK_KHR_DISPLAY_SWAPCHAIN_SPEC_VERSION = 9
VK_KHR_DISPLAY_SWAPCHAIN_EXTENSION_NAME = "VK_KHR_display_swapchain"
VK_STRUCTURE_TYPE_DISPLAY_PRESENT_INFO_KHR = 1000003000
VK_KHR_xlib_surface = 1
VK_KHR_XLIB_SURFACE_SPEC_VERSION = 6
VK_KHR_XLIB_SURFACE_EXTENSION_NAME = "VK_KHR_xlib_surface"
VK_STRUCTURE_TYPE_XLIB_SURFACE_CREATE_INFO_KHR = 1000004000
VK_KHR_xcb_surface = 1
VK_KHR_XCB_SURFACE_SPEC_VERSION = 6
VK_KHR_XCB_SURFACE_EXTENSION_NAME = "VK_KHR_xcb_surface"
VK_STRUCTURE_TYPE_XCB_SURFACE_CREATE_INFO_KHR = 1000005000
VK_KHR_wayland_surface = 1
VK_KHR_WAYLAND_SURFACE_SPEC_VERSION = 6
VK_KHR_WAYLAND_SURFACE_EXTENSION_NAME = "VK_KHR_wayland_surface"
VK_STRUCTURE_TYPE_WAYLAND_SURFACE_CREATE_INFO_KHR = 1000006000
VK_KHR_mir_surface = 1
VK_KHR_MIR_SURFACE_SPEC_VERSION = 4
VK_KHR_MIR_SURFACE_EXTENSION_NAME = "VK_KHR_mir_surface"
VK_KHR_android_surface = 1
VK_KHR_ANDROID_SURFACE_SPEC_VERSION = 6
VK_KHR_ANDROID_SURFACE_EXTENSION_NAME = "VK_KHR_android_surface"
VK_STRUCTURE_TYPE_ANDROID_SURFACE_CREATE_INFO_KHR = 1000008000
VK_KHR_win32_surface = 1
VK_KHR_WIN32_SURFACE_SPEC_VERSION = 6
VK_KHR_WIN32_SURFACE_EXTENSION_NAME = "VK_KHR_win32_surface"
VK_STRUCTURE_TYPE_WIN32_SURFACE_CREATE_INFO_KHR = 1000009000
VK_ANDROID_native_buffer = 1
VK_ANDROID_NATIVE_BUFFER_SPEC_VERSION = 5
VK_ANDROID_NATIVE_BUFFER_NUMBER = 11
VK_ANDROID_NATIVE_BUFFER_NAME = "VK_ANDROID_native_buffer"
VK_STRUCTURE_TYPE_NATIVE_BUFFER_ANDROID = 1000010000
VK_EXT_debug_report = 1
VK_EXT_DEBUG_REPORT_SPEC_VERSION = 9
VK_EXT_DEBUG_REPORT_EXTENSION_NAME = "VK_EXT_debug_report"
VK_STRUCTURE_TYPE_DEBUG_REPORT_CALLBACK_CREATE_INFO_EXT = 1000011000
VK_STRUCTURE_TYPE_DEBUG_REPORT_CREATE_INFO_EXT = None
VK_OBJECT_TYPE_DEBUG_REPORT_CALLBACK_EXT = 1000011000
VK_DEBUG_REPORT_OBJECT_TYPE_SAMPLER_YCBCR_CONVERSION_EXT = 1000011000
VK_DEBUG_REPORT_OBJECT_TYPE_DESCRIPTOR_UPDATE_TEMPLATE_EXT = 1000011000
VK_NV_glsl_shader = 1
VK_NV_GLSL_SHADER_SPEC_VERSION = 1
VK_NV_GLSL_SHADER_EXTENSION_NAME = "VK_NV_glsl_shader"
VK_EXT_depth_range_unrestricted = 1
VK_EXT_DEPTH_RANGE_UNRESTRICTED_SPEC_VERSION = 1
VK_EXT_DEPTH_RANGE_UNRESTRICTED_EXTENSION_NAME = "VK_EXT_depth_range_unrestricted"
VK_KHR_sampler_mirror_clamp_to_edge = 1
VK_KHR_SAMPLER_MIRROR_CLAMP_TO_EDGE_SPEC_VERSION = 1
VK_KHR_SAMPLER_MIRROR_CLAMP_TO_EDGE_EXTENSION_NAME = "VK_KHR_sampler_mirror_clamp_to_edge"
VK_SAMPLER_ADDRESS_MODE_MIRROR_CLAMP_TO_EDGE = 4
VK_IMG_filter_cubic = 1
VK_IMG_FILTER_CUBIC_SPEC_VERSION = 1
VK_IMG_FILTER_CUBIC_EXTENSION_NAME = "VK_IMG_filter_cubic"
VK_FILTER_CUBIC_EXT = 1000015000
VK_FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_CUBIC_BIT_EXT = 0x00002000
VK_FILTER_CUBIC_IMG = None
VK_FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_CUBIC_BIT_IMG = None
VK_AMD_extension_17 = 1
VK_AMD_EXTENSION_17_SPEC_VERSION = 0
VK_AMD_EXTENSION_17_EXTENSION_NAME = "VK_AMD_extension_17"
VK_AMD_extension_18 = 1
VK_AMD_EXTENSION_18_SPEC_VERSION = 0
VK_AMD_EXTENSION_18_EXTENSION_NAME = "VK_AMD_extension_18"
VK_AMD_rasterization_order = 1
VK_AMD_RASTERIZATION_ORDER_SPEC_VERSION = 1
VK_AMD_RASTERIZATION_ORDER_EXTENSION_NAME = "VK_AMD_rasterization_order"
VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_RASTERIZATION_ORDER_AMD = 1000018000
VK_AMD_extension_20 = 1
VK_AMD_EXTENSION_20_SPEC_VERSION = 0
VK_AMD_EXTENSION_20_EXTENSION_NAME = "VK_AMD_extension_20"
VK_AMD_shader_trinary_minmax = 1
VK_AMD_SHADER_TRINARY_MINMAX_SPEC_VERSION = 1
VK_AMD_SHADER_TRINARY_MINMAX_EXTENSION_NAME = "VK_AMD_shader_trinary_minmax"
VK_AMD_shader_explicit_vertex_parameter = 1
VK_AMD_SHADER_EXPLICIT_VERTEX_PARAMETER_SPEC_VERSION = 1
VK_AMD_SHADER_EXPLICIT_VERTEX_PARAMETER_EXTENSION_NAME = "VK_AMD_shader_explicit_vertex_parameter"
VK_EXT_debug_marker = 1
VK_EXT_DEBUG_MARKER_SPEC_VERSION = 4
VK_EXT_DEBUG_MARKER_EXTENSION_NAME = "VK_EXT_debug_marker"
VK_STRUCTURE_TYPE_DEBUG_MARKER_OBJECT_NAME_INFO_EXT = 1000022000
VK_STRUCTURE_TYPE_DEBUG_MARKER_OBJECT_TAG_INFO_EXT = 1000022001
VK_STRUCTURE_TYPE_DEBUG_MARKER_MARKER_INFO_EXT = 1000022002
VK_AMD_extension_24 = 1
VK_AMD_EXTENSION_24_SPEC_VERSION = 0
VK_AMD_EXTENSION_24_EXTENSION_NAME = "VK_AMD_extension_24"
VK_QUEUE_RESERVED_6_BIT_KHR = 0x00000040
VK_PIPELINE_STAGE_RESERVED_27_BIT_KHR = 0x08000000
VK_ACCESS_RESERVED_30_BIT_KHR = 0x40000000
VK_ACCESS_RESERVED_31_BIT_KHR = 0x80000000
VK_BUFFER_USAGE_RESERVED_15_BIT_KHR = 0x00008000
VK_BUFFER_USAGE_RESERVED_16_BIT_KHR = 0x00010000
VK_IMAGE_USAGE_RESERVED_13_BIT_KHR = 0x00002000
VK_IMAGE_USAGE_RESERVED_14_BIT_KHR = 0x00004000
VK_IMAGE_USAGE_RESERVED_15_BIT_KHR = 0x00008000
VK_FORMAT_FEATURE_RESERVED_27_BIT_KHR = 0x08000000
VK_FORMAT_FEATURE_RESERVED_28_BIT_KHR = 0x10000000
VK_QUERY_TYPE_RESERVED_8 = 1000023008
VK_AMD_extension_25 = 1
VK_AMD_EXTENSION_25_SPEC_VERSION = 0
VK_AMD_EXTENSION_25_EXTENSION_NAME = "VK_AMD_extension_25"
VK_QUEUE_RESERVED_5_BIT_KHR = 0x00000020
VK_PIPELINE_STAGE_RESERVED_26_BIT_KHR = 0x04000000
VK_ACCESS_RESERVED_28_BIT_KHR = 0x10000000
VK_ACCESS_RESERVED_29_BIT_KHR = 0x20000000
VK_BUFFER_USAGE_RESERVED_13_BIT_KHR = 0x00002000
VK_BUFFER_USAGE_RESERVED_14_BIT_KHR = 0x00004000
VK_IMAGE_USAGE_RESERVED_10_BIT_KHR = 0x00000400
VK_IMAGE_USAGE_RESERVED_11_BIT_KHR = 0x00000800
VK_IMAGE_USAGE_RESERVED_12_BIT_KHR = 0x00001000
VK_FORMAT_FEATURE_RESERVED_25_BIT_KHR = 0x02000000
VK_FORMAT_FEATURE_RESERVED_26_BIT_KHR = 0x04000000
VK_QUERY_TYPE_RESERVED_4 = 1000024004
VK_AMD_gcn_shader = 1
VK_AMD_GCN_SHADER_SPEC_VERSION = 1
VK_AMD_GCN_SHADER_EXTENSION_NAME = "VK_AMD_gcn_shader"
VK_NV_dedicated_allocation = 1
VK_NV_DEDICATED_ALLOCATION_SPEC_VERSION = 1
VK_NV_DEDICATED_ALLOCATION_EXTENSION_NAME = "VK_NV_dedicated_allocation"
VK_STRUCTURE_TYPE_DEDICATED_ALLOCATION_IMAGE_CREATE_INFO_NV = 1000026000
VK_STRUCTURE_TYPE_DEDICATED_ALLOCATION_BUFFER_CREATE_INFO_NV = 1000026001
VK_STRUCTURE_TYPE_DEDICATED_ALLOCATION_MEMORY_ALLOCATE_INFO_NV = 1000026002
VK_EXT_extension_28 = 1
VK_EXT_EXTENSION_28_SPEC_VERSION = 0
VK_EXT_EXTENSION_28_EXTENSION_NAME = "VK_NV_extension_28"
VK_EXT_transform_feedback = 1
VK_EXT_TRANSFORM_FEEDBACK_SPEC_VERSION = 1
VK_EXT_TRANSFORM_FEEDBACK_EXTENSION_NAME = "VK_EXT_transform_feedback"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TRANSFORM_FEEDBACK_FEATURES_EXT = 1000028000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TRANSFORM_FEEDBACK_PROPERTIES_EXT = 1000028001
VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_STREAM_CREATE_INFO_EXT = 1000028002
VK_QUERY_TYPE_TRANSFORM_FEEDBACK_STREAM_EXT = 1000028004
VK_BUFFER_USAGE_TRANSFORM_FEEDBACK_BUFFER_BIT_EXT = 0x00000800
VK_BUFFER_USAGE_TRANSFORM_FEEDBACK_COUNTER_BUFFER_BIT_EXT = 0x00001000
VK_ACCESS_TRANSFORM_FEEDBACK_WRITE_BIT_EXT = 0x02000000
VK_ACCESS_TRANSFORM_FEEDBACK_COUNTER_READ_BIT_EXT = 0x04000000
VK_ACCESS_TRANSFORM_FEEDBACK_COUNTER_WRITE_BIT_EXT = 0x08000000
VK_PIPELINE_STAGE_TRANSFORM_FEEDBACK_BIT_EXT = 0x01000000
VK_NVX_extension_30 = 1
VK_NVX_EXTENSION_30_SPEC_VERSION = 0
VK_NVX_EXTENSION_30_EXTENSION_NAME = "VK_NVX_extension_30"
VK_NVX_extension_31 = 1
VK_NVX_EXTENSION_31_SPEC_VERSION = 0
VK_NVX_EXTENSION_31_EXTENSION_NAME = "VK_NVX_extension_31"
VK_AMD_extension_32 = 1
VK_AMD_EXTENSION_32_SPEC_VERSION = 0
VK_AMD_EXTENSION_32_EXTENSION_NAME = "VK_AMD_extension_32"
VK_AMD_extension_33 = 1
VK_AMD_EXTENSION_33_SPEC_VERSION = 0
VK_AMD_EXTENSION_33_EXTENSION_NAME = "VK_AMD_extension_33"
VK_AMD_draw_indirect_count = 1
VK_AMD_DRAW_INDIRECT_COUNT_SPEC_VERSION = 1
VK_AMD_DRAW_INDIRECT_COUNT_EXTENSION_NAME = "VK_AMD_draw_indirect_count"
VK_AMD_extension_35 = 1
VK_AMD_EXTENSION_35_SPEC_VERSION = 0
VK_AMD_EXTENSION_35_EXTENSION_NAME = "VK_AMD_extension_35"
VK_AMD_negative_viewport_height = 1
VK_AMD_NEGATIVE_VIEWPORT_HEIGHT_SPEC_VERSION = 1
VK_AMD_NEGATIVE_VIEWPORT_HEIGHT_EXTENSION_NAME = "VK_AMD_negative_viewport_height"
VK_AMD_gpu_shader_half_float = 1
VK_AMD_GPU_SHADER_HALF_FLOAT_SPEC_VERSION = 1
VK_AMD_GPU_SHADER_HALF_FLOAT_EXTENSION_NAME = "VK_AMD_gpu_shader_half_float"
VK_AMD_shader_ballot = 1
VK_AMD_SHADER_BALLOT_SPEC_VERSION = 1
VK_AMD_SHADER_BALLOT_EXTENSION_NAME = "VK_AMD_shader_ballot"
VK_AMD_extension_39 = 1
VK_AMD_EXTENSION_39_SPEC_VERSION = 0
VK_AMD_EXTENSION_39_EXTENSION_NAME = "VK_AMD_extension_39"
VK_AMD_extension_40 = 1
VK_AMD_EXTENSION_40_SPEC_VERSION = 0
VK_AMD_EXTENSION_40_EXTENSION_NAME = "VK_AMD_extension_40"
VK_AMD_extension_41 = 1
VK_AMD_EXTENSION_41_SPEC_VERSION = 0
VK_AMD_EXTENSION_41_EXTENSION_NAME = "VK_AMD_extension_41"
VK_AMD_texture_gather_bias_lod = 1
VK_AMD_TEXTURE_GATHER_BIAS_LOD_SPEC_VERSION = 1
VK_AMD_TEXTURE_GATHER_BIAS_LOD_EXTENSION_NAME = "VK_AMD_texture_gather_bias_lod"
VK_STRUCTURE_TYPE_TEXTURE_LOD_GATHER_FORMAT_PROPERTIES_AMD = 1000041000
VK_AMD_shader_info = 1
VK_AMD_SHADER_INFO_SPEC_VERSION = 1
VK_AMD_SHADER_INFO_EXTENSION_NAME = "VK_AMD_shader_info"
VK_AMD_extension_44 = 1
VK_AMD_EXTENSION_44_SPEC_VERSION = 0
VK_AMD_EXTENSION_44_EXTENSION_NAME = "VK_AMD_extension_44"
VK_AMD_extension_45 = 1
VK_AMD_EXTENSION_45_SPEC_VERSION = 0
VK_AMD_EXTENSION_45_EXTENSION_NAME = "VK_AMD_extension_45"
VK_AMD_extension_46 = 1
VK_AMD_EXTENSION_46_SPEC_VERSION = 0
VK_AMD_EXTENSION_46_EXTENSION_NAME = "VK_AMD_extension_46"
VK_AMD_shader_image_load_store_lod = 1
VK_AMD_SHADER_IMAGE_LOAD_STORE_LOD_SPEC_VERSION = 1
VK_AMD_SHADER_IMAGE_LOAD_STORE_LOD_EXTENSION_NAME = "VK_AMD_shader_image_load_store_lod"
VK_NVX_extension_48 = 1
VK_NVX_EXTENSION_48_SPEC_VERSION = 0
VK_NVX_EXTENSION_48_EXTENSION_NAME = "VK_NVX_extension_48"
VK_GOOGLE_extension_49 = 1
VK_GOOGLE_EXTENSION_49_SPEC_VERSION = 0
VK_GOOGLE_EXTENSION_49_EXTENSION_NAME = "VK_GOOGLE_extension_49"
VK_GOOGLE_extension_50 = 1
VK_GOOGLE_EXTENSION_50_SPEC_VERSION = 0
VK_GOOGLE_EXTENSION_50_EXTENSION_NAME = "VK_GOOGLE_extension_50"
VK_NV_corner_sampled_image = 1
VK_NV_CORNER_SAMPLED_IMAGE_SPEC_VERSION = 2
VK_NV_CORNER_SAMPLED_IMAGE_EXTENSION_NAME = "VK_NV_corner_sampled_image"
VK_IMAGE_CREATE_CORNER_SAMPLED_BIT_NV = 0x00002000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CORNER_SAMPLED_IMAGE_FEATURES_NV = 1000050000
VK_NVX_extension_52 = 1
VK_NVX_EXTENSION_52_SPEC_VERSION = 0
VK_NVX_EXTENSION_52_EXTENSION_NAME = "VK_NVX_extension_52"
VK_NV_extension_53 = 1
VK_NV_EXTENSION_53_SPEC_VERSION = 0
VK_NV_EXTENSION_53_EXTENSION_NAME = "VK_NV_extension_53"
VK_KHR_multiview = 1
VK_KHR_MULTIVIEW_SPEC_VERSION = 1
VK_KHR_MULTIVIEW_EXTENSION_NAME = "VK_KHR_multiview"
VK_STRUCTURE_TYPE_RENDER_PASS_MULTIVIEW_CREATE_INFO_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_FEATURES_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_PROPERTIES_KHR = None
VK_DEPENDENCY_VIEW_LOCAL_BIT_KHR = None
VK_IMG_format_pvrtc = 1
VK_IMG_FORMAT_PVRTC_SPEC_VERSION = 1
VK_IMG_FORMAT_PVRTC_EXTENSION_NAME = "VK_IMG_format_pvrtc"
VK_FORMAT_PVRTC1_2BPP_UNORM_BLOCK_IMG = 1000054000
VK_FORMAT_PVRTC1_4BPP_UNORM_BLOCK_IMG = 1000054001
VK_FORMAT_PVRTC2_2BPP_UNORM_BLOCK_IMG = 1000054002
VK_FORMAT_PVRTC2_4BPP_UNORM_BLOCK_IMG = 1000054003
VK_FORMAT_PVRTC1_2BPP_SRGB_BLOCK_IMG = 1000054004
VK_FORMAT_PVRTC1_4BPP_SRGB_BLOCK_IMG = 1000054005
VK_FORMAT_PVRTC2_2BPP_SRGB_BLOCK_IMG = 1000054006
VK_FORMAT_PVRTC2_4BPP_SRGB_BLOCK_IMG = 1000054007
VK_NV_external_memory_capabilities = 1
VK_NV_EXTERNAL_MEMORY_CAPABILITIES_SPEC_VERSION = 1
VK_NV_EXTERNAL_MEMORY_CAPABILITIES_EXTENSION_NAME = "VK_NV_external_memory_capabilities"
VK_NV_external_memory = 1
VK_NV_EXTERNAL_MEMORY_SPEC_VERSION = 1
VK_NV_EXTERNAL_MEMORY_EXTENSION_NAME = "VK_NV_external_memory"
VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_IMAGE_CREATE_INFO_NV = 1000056000
VK_STRUCTURE_TYPE_EXPORT_MEMORY_ALLOCATE_INFO_NV = 1000056001
VK_NV_external_memory_win32 = 1
VK_NV_EXTERNAL_MEMORY_WIN32_SPEC_VERSION = 1
VK_NV_EXTERNAL_MEMORY_WIN32_EXTENSION_NAME = "VK_NV_external_memory_win32"
VK_STRUCTURE_TYPE_IMPORT_MEMORY_WIN32_HANDLE_INFO_NV = 1000057000
VK_STRUCTURE_TYPE_EXPORT_MEMORY_WIN32_HANDLE_INFO_NV = 1000057001
VK_NV_win32_keyed_mutex = 1
VK_NV_WIN32_KEYED_MUTEX_SPEC_VERSION = 1
VK_NV_WIN32_KEYED_MUTEX_EXTENSION_NAME = "VK_NV_win32_keyed_mutex"
VK_STRUCTURE_TYPE_WIN32_KEYED_MUTEX_ACQUIRE_RELEASE_INFO_NV = 1000058000
VK_KHR_get_physical_device_properties2 = 1
VK_KHR_GET_PHYSICAL_DEVICE_PROPERTIES_2_SPEC_VERSION = 1
VK_KHR_GET_PHYSICAL_DEVICE_PROPERTIES_2_EXTENSION_NAME = "VK_KHR_get_physical_device_properties2"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FEATURES_2_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROPERTIES_2_KHR = None
VK_STRUCTURE_TYPE_FORMAT_PROPERTIES_2_KHR = None
VK_STRUCTURE_TYPE_IMAGE_FORMAT_PROPERTIES_2_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_FORMAT_INFO_2_KHR = None
VK_STRUCTURE_TYPE_QUEUE_FAMILY_PROPERTIES_2_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_PROPERTIES_2_KHR = None
VK_STRUCTURE_TYPE_SPARSE_IMAGE_FORMAT_PROPERTIES_2_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SPARSE_IMAGE_FORMAT_INFO_2_KHR = None
VK_KHR_device_group = 1
VK_KHR_DEVICE_GROUP_SPEC_VERSION = 3
VK_KHR_DEVICE_GROUP_EXTENSION_NAME = "VK_KHR_device_group"
VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_FLAGS_INFO_KHR = None
VK_STRUCTURE_TYPE_DEVICE_GROUP_RENDER_PASS_BEGIN_INFO_KHR = None
VK_STRUCTURE_TYPE_DEVICE_GROUP_COMMAND_BUFFER_BEGIN_INFO_KHR = None
VK_STRUCTURE_TYPE_DEVICE_GROUP_SUBMIT_INFO_KHR = None
VK_STRUCTURE_TYPE_DEVICE_GROUP_BIND_SPARSE_INFO_KHR = None
VK_PEER_MEMORY_FEATURE_COPY_SRC_BIT_KHR = None
VK_PEER_MEMORY_FEATURE_COPY_DST_BIT_KHR = None
VK_PEER_MEMORY_FEATURE_GENERIC_SRC_BIT_KHR = None
VK_PEER_MEMORY_FEATURE_GENERIC_DST_BIT_KHR = None
VK_MEMORY_ALLOCATE_DEVICE_MASK_BIT_KHR = None
VK_PIPELINE_CREATE_VIEW_INDEX_FROM_DEVICE_INDEX_BIT_KHR = None
VK_PIPELINE_CREATE_DISPATCH_BASE_KHR = None
VK_DEPENDENCY_DEVICE_GROUP_BIT_KHR = None
VK_STRUCTURE_TYPE_BIND_BUFFER_MEMORY_DEVICE_GROUP_INFO_KHR = None
VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_DEVICE_GROUP_INFO_KHR = None
VK_IMAGE_CREATE_SPLIT_INSTANCE_BIND_REGIONS_BIT_KHR = None
VK_EXT_validation_flags = 1
VK_EXT_VALIDATION_FLAGS_SPEC_VERSION = 1
VK_EXT_VALIDATION_FLAGS_EXTENSION_NAME = "VK_EXT_validation_flags"
VK_STRUCTURE_TYPE_VALIDATION_FLAGS_EXT = 1000061000
VK_NN_vi_surface = 1
VK_NN_VI_SURFACE_SPEC_VERSION = 1
VK_NN_VI_SURFACE_EXTENSION_NAME = "VK_NN_vi_surface"
VK_STRUCTURE_TYPE_VI_SURFACE_CREATE_INFO_NN = 1000062000
VK_KHR_shader_draw_parameters = 1
VK_KHR_SHADER_DRAW_PARAMETERS_SPEC_VERSION = 1
VK_KHR_SHADER_DRAW_PARAMETERS_EXTENSION_NAME = "VK_KHR_shader_draw_parameters"
VK_EXT_shader_subgroup_ballot = 1
VK_EXT_SHADER_SUBGROUP_BALLOT_SPEC_VERSION = 1
VK_EXT_SHADER_SUBGROUP_BALLOT_EXTENSION_NAME = "VK_EXT_shader_subgroup_ballot"
VK_EXT_shader_subgroup_vote = 1
VK_EXT_SHADER_SUBGROUP_VOTE_SPEC_VERSION = 1
VK_EXT_SHADER_SUBGROUP_VOTE_EXTENSION_NAME = "VK_EXT_shader_subgroup_vote"
VK_ARM_extension_01 = 1
VK_ARM_EXTENSION_01_SPEC_VERSION = 0
VK_ARM_EXTENSION_01_EXTENSION_NAME = "VK_ARM_extension_01"
VK_EXT_astc_decode_mode = 1
VK_EXT_ASTC_DECODE_MODE_SPEC_VERSION = 1
VK_EXT_ASTC_DECODE_MODE_EXTENSION_NAME = "VK_EXT_astc_decode_mode"
VK_STRUCTURE_TYPE_IMAGE_VIEW_ASTC_DECODE_MODE_EXT = 1000067000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ASTC_DECODE_FEATURES_EXT = 1000067001
VK_IMG_extension_69 = 1
VK_IMG_EXTENSION_69_SPEC_VERSION = 0
VK_IMG_EXTENSION_69_EXTENSION_NAME = "VK_IMG_extension_69"
VK_KHR_maintenance1 = 1
VK_KHR_MAINTENANCE1_SPEC_VERSION = 2
VK_KHR_MAINTENANCE1_EXTENSION_NAME = "VK_KHR_maintenance1"
VK_FORMAT_FEATURE_TRANSFER_SRC_BIT_KHR = None
VK_FORMAT_FEATURE_TRANSFER_DST_BIT_KHR = None
VK_IMAGE_CREATE_2D_ARRAY_COMPATIBLE_BIT_KHR = None
VK_KHR_device_group_creation = 1
VK_KHR_DEVICE_GROUP_CREATION_SPEC_VERSION = 1
VK_KHR_DEVICE_GROUP_CREATION_EXTENSION_NAME = "VK_KHR_device_group_creation"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_GROUP_PROPERTIES_KHR = None
VK_STRUCTURE_TYPE_DEVICE_GROUP_DEVICE_CREATE_INFO_KHR = None
VK_MAX_DEVICE_GROUP_SIZE_KHR = None
VK_MEMORY_HEAP_MULTI_INSTANCE_BIT_KHR = None
VK_KHR_external_memory_capabilities = 1
VK_KHR_EXTERNAL_MEMORY_CAPABILITIES_SPEC_VERSION = 1
VK_KHR_EXTERNAL_MEMORY_CAPABILITIES_EXTENSION_NAME = "VK_KHR_external_memory_capabilities"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_IMAGE_FORMAT_INFO_KHR = None
VK_STRUCTURE_TYPE_EXTERNAL_IMAGE_FORMAT_PROPERTIES_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_BUFFER_INFO_KHR = None
VK_STRUCTURE_TYPE_EXTERNAL_BUFFER_PROPERTIES_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ID_PROPERTIES_KHR = None
VK_LUID_SIZE_KHR = None
VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_FD_BIT_KHR = None
VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32_BIT_KHR = None
VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32_KMT_BIT_KHR = None
VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_TEXTURE_BIT_KHR = None
VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_TEXTURE_KMT_BIT_KHR = None
VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D12_HEAP_BIT_KHR = None
VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D12_RESOURCE_BIT_KHR = None
VK_EXTERNAL_MEMORY_FEATURE_DEDICATED_ONLY_BIT_KHR = None
VK_EXTERNAL_MEMORY_FEATURE_EXPORTABLE_BIT_KHR = None
VK_EXTERNAL_MEMORY_FEATURE_IMPORTABLE_BIT_KHR = None
VK_KHR_external_memory = 1
VK_KHR_EXTERNAL_MEMORY_SPEC_VERSION = 1
VK_KHR_EXTERNAL_MEMORY_EXTENSION_NAME = "VK_KHR_external_memory"
VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_BUFFER_CREATE_INFO_KHR = None
VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_IMAGE_CREATE_INFO_KHR = None
VK_STRUCTURE_TYPE_EXPORT_MEMORY_ALLOCATE_INFO_KHR = None
VK_QUEUE_FAMILY_EXTERNAL_KHR = None
VK_KHR_external_memory_win32 = 1
VK_KHR_EXTERNAL_MEMORY_WIN32_SPEC_VERSION = 1
VK_KHR_EXTERNAL_MEMORY_WIN32_EXTENSION_NAME = "VK_KHR_external_memory_win32"
VK_STRUCTURE_TYPE_IMPORT_MEMORY_WIN32_HANDLE_INFO_KHR = 1000073000
VK_STRUCTURE_TYPE_EXPORT_MEMORY_WIN32_HANDLE_INFO_KHR = 1000073001
VK_STRUCTURE_TYPE_MEMORY_WIN32_HANDLE_PROPERTIES_KHR = 1000073002
VK_STRUCTURE_TYPE_MEMORY_GET_WIN32_HANDLE_INFO_KHR = 1000073003
VK_KHR_external_memory_fd = 1
VK_KHR_EXTERNAL_MEMORY_FD_SPEC_VERSION = 1
VK_KHR_EXTERNAL_MEMORY_FD_EXTENSION_NAME = "VK_KHR_external_memory_fd"
VK_STRUCTURE_TYPE_IMPORT_MEMORY_FD_INFO_KHR = 1000074000
VK_STRUCTURE_TYPE_MEMORY_FD_PROPERTIES_KHR = 1000074001
VK_STRUCTURE_TYPE_MEMORY_GET_FD_INFO_KHR = 1000074002
VK_KHR_win32_keyed_mutex = 1
VK_KHR_WIN32_KEYED_MUTEX_SPEC_VERSION = 1
VK_KHR_WIN32_KEYED_MUTEX_EXTENSION_NAME = "VK_KHR_win32_keyed_mutex"
VK_STRUCTURE_TYPE_WIN32_KEYED_MUTEX_ACQUIRE_RELEASE_INFO_KHR = 1000075000
VK_KHR_external_semaphore_capabilities = 1
VK_KHR_EXTERNAL_SEMAPHORE_CAPABILITIES_SPEC_VERSION = 1
VK_KHR_EXTERNAL_SEMAPHORE_CAPABILITIES_EXTENSION_NAME = "VK_KHR_external_semaphore_capabilities"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_SEMAPHORE_INFO_KHR = None
VK_STRUCTURE_TYPE_EXTERNAL_SEMAPHORE_PROPERTIES_KHR = None
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_FD_BIT_KHR = None
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32_BIT_KHR = None
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32_KMT_BIT_KHR = None
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D12_FENCE_BIT_KHR = None
VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_SYNC_FD_BIT_KHR = None
VK_EXTERNAL_SEMAPHORE_FEATURE_EXPORTABLE_BIT_KHR = None
VK_EXTERNAL_SEMAPHORE_FEATURE_IMPORTABLE_BIT_KHR = None
VK_KHR_external_semaphore = 1
VK_KHR_EXTERNAL_SEMAPHORE_SPEC_VERSION = 1
VK_KHR_EXTERNAL_SEMAPHORE_EXTENSION_NAME = "VK_KHR_external_semaphore"
VK_STRUCTURE_TYPE_EXPORT_SEMAPHORE_CREATE_INFO_KHR = None
VK_SEMAPHORE_IMPORT_TEMPORARY_BIT_KHR = None
VK_KHR_external_semaphore_win32 = 1
VK_KHR_EXTERNAL_SEMAPHORE_WIN32_SPEC_VERSION = 1
VK_KHR_EXTERNAL_SEMAPHORE_WIN32_EXTENSION_NAME = "VK_KHR_external_semaphore_win32"
VK_STRUCTURE_TYPE_IMPORT_SEMAPHORE_WIN32_HANDLE_INFO_KHR = 1000078000
VK_STRUCTURE_TYPE_EXPORT_SEMAPHORE_WIN32_HANDLE_INFO_KHR = 1000078001
VK_STRUCTURE_TYPE_D3D12_FENCE_SUBMIT_INFO_KHR = 1000078002
VK_STRUCTURE_TYPE_SEMAPHORE_GET_WIN32_HANDLE_INFO_KHR = 1000078003
VK_KHR_external_semaphore_fd = 1
VK_KHR_EXTERNAL_SEMAPHORE_FD_SPEC_VERSION = 1
VK_KHR_EXTERNAL_SEMAPHORE_FD_EXTENSION_NAME = "VK_KHR_external_semaphore_fd"
VK_STRUCTURE_TYPE_IMPORT_SEMAPHORE_FD_INFO_KHR = 1000079000
VK_STRUCTURE_TYPE_SEMAPHORE_GET_FD_INFO_KHR = 1000079001
VK_KHR_push_descriptor = 1
VK_KHR_PUSH_DESCRIPTOR_SPEC_VERSION = 2
VK_KHR_PUSH_DESCRIPTOR_EXTENSION_NAME = "VK_KHR_push_descriptor"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PUSH_DESCRIPTOR_PROPERTIES_KHR = 1000080000
VK_DESCRIPTOR_SET_LAYOUT_CREATE_PUSH_DESCRIPTOR_BIT_KHR = 0x00000001
VK_DESCRIPTOR_UPDATE_TEMPLATE_TYPE_PUSH_DESCRIPTORS_KHR = 1
VK_EXT_conditional_rendering = 1
VK_EXT_CONDITIONAL_RENDERING_SPEC_VERSION = 1
VK_EXT_CONDITIONAL_RENDERING_EXTENSION_NAME = "VK_EXT_conditional_rendering"
VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_CONDITIONAL_RENDERING_INFO_EXT = 1000081000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CONDITIONAL_RENDERING_FEATURES_EXT = 1000081001
VK_STRUCTURE_TYPE_CONDITIONAL_RENDERING_BEGIN_INFO_EXT = 1000081002
VK_ACCESS_CONDITIONAL_RENDERING_READ_BIT_EXT = 0x00100000
VK_BUFFER_USAGE_CONDITIONAL_RENDERING_BIT_EXT = 0x00000200
VK_PIPELINE_STAGE_CONDITIONAL_RENDERING_BIT_EXT = 0x00040000
VK_KHR_shader_float16_int8 = 1
VK_KHR_SHADER_FLOAT16_INT8_SPEC_VERSION = 1
VK_KHR_SHADER_FLOAT16_INT8_EXTENSION_NAME = "VK_KHR_shader_float16_int8"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FLOAT16_INT8_FEATURES_KHR = 1000082000
VK_KHR_16bit_storage = 1
VK_KHR_16BIT_STORAGE_SPEC_VERSION = 1
VK_KHR_16BIT_STORAGE_EXTENSION_NAME = "VK_KHR_16bit_storage"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_16BIT_STORAGE_FEATURES_KHR = None
VK_KHR_incremental_present = 1
VK_KHR_INCREMENTAL_PRESENT_SPEC_VERSION = 1
VK_KHR_INCREMENTAL_PRESENT_EXTENSION_NAME = "VK_KHR_incremental_present"
VK_STRUCTURE_TYPE_PRESENT_REGIONS_KHR = 1000084000
VK_KHR_descriptor_update_template = 1
VK_KHR_DESCRIPTOR_UPDATE_TEMPLATE_SPEC_VERSION = 1
VK_KHR_DESCRIPTOR_UPDATE_TEMPLATE_EXTENSION_NAME = "VK_KHR_descriptor_update_template"
VK_STRUCTURE_TYPE_DESCRIPTOR_UPDATE_TEMPLATE_CREATE_INFO_KHR = None
VK_OBJECT_TYPE_DESCRIPTOR_UPDATE_TEMPLATE_KHR = None
VK_DESCRIPTOR_UPDATE_TEMPLATE_TYPE_DESCRIPTOR_SET_KHR = None
VK_DEBUG_REPORT_OBJECT_TYPE_DESCRIPTOR_UPDATE_TEMPLATE_KHR_EXT = None
VK_NVX_device_generated_commands = 1
VK_NVX_DEVICE_GENERATED_COMMANDS_SPEC_VERSION = 3
VK_NVX_DEVICE_GENERATED_COMMANDS_EXTENSION_NAME = "VK_NVX_device_generated_commands"
VK_STRUCTURE_TYPE_OBJECT_TABLE_CREATE_INFO_NVX = 1000086000
VK_STRUCTURE_TYPE_INDIRECT_COMMANDS_LAYOUT_CREATE_INFO_NVX = 1000086001
VK_STRUCTURE_TYPE_CMD_PROCESS_COMMANDS_INFO_NVX = 1000086002
VK_STRUCTURE_TYPE_CMD_RESERVE_SPACE_FOR_COMMANDS_INFO_NVX = 1000086003
VK_STRUCTURE_TYPE_DEVICE_GENERATED_COMMANDS_LIMITS_NVX = 1000086004
VK_STRUCTURE_TYPE_DEVICE_GENERATED_COMMANDS_FEATURES_NVX = 1000086005
VK_PIPELINE_STAGE_COMMAND_PROCESS_BIT_NVX = 0x00020000
VK_ACCESS_COMMAND_PROCESS_READ_BIT_NVX = 0x00020000
VK_ACCESS_COMMAND_PROCESS_WRITE_BIT_NVX = 0x00040000
VK_OBJECT_TYPE_OBJECT_TABLE_NVX = 1000086000
VK_OBJECT_TYPE_INDIRECT_COMMANDS_LAYOUT_NVX = 1000086001
VK_NV_clip_space_w_scaling = 1
VK_NV_CLIP_SPACE_W_SCALING_SPEC_VERSION = 1
VK_NV_CLIP_SPACE_W_SCALING_EXTENSION_NAME = "VK_NV_clip_space_w_scaling"
VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_W_SCALING_STATE_CREATE_INFO_NV = 1000087000
VK_DYNAMIC_STATE_VIEWPORT_W_SCALING_NV = 1000087000
VK_EXT_direct_mode_display = 1
VK_EXT_DIRECT_MODE_DISPLAY_SPEC_VERSION = 1
VK_EXT_DIRECT_MODE_DISPLAY_EXTENSION_NAME = "VK_EXT_direct_mode_display"
VK_EXT_acquire_xlib_display = 1
VK_EXT_ACQUIRE_XLIB_DISPLAY_SPEC_VERSION = 1
VK_EXT_ACQUIRE_XLIB_DISPLAY_EXTENSION_NAME = "VK_EXT_acquire_xlib_display"
VK_EXT_display_surface_counter = 1
VK_EXT_DISPLAY_SURFACE_COUNTER_SPEC_VERSION = 1
VK_EXT_DISPLAY_SURFACE_COUNTER_EXTENSION_NAME = "VK_EXT_display_surface_counter"
VK_STRUCTURE_TYPE_SURFACE_CAPABILITIES_2_EXT = 1000090000
VK_STRUCTURE_TYPE_SURFACE_CAPABILITIES2_EXT = None
VK_EXT_display_control = 1
VK_EXT_DISPLAY_CONTROL_SPEC_VERSION = 1
VK_EXT_DISPLAY_CONTROL_EXTENSION_NAME = "VK_EXT_display_control"
VK_STRUCTURE_TYPE_DISPLAY_POWER_INFO_EXT = 1000091000
VK_STRUCTURE_TYPE_DEVICE_EVENT_INFO_EXT = 1000091001
VK_STRUCTURE_TYPE_DISPLAY_EVENT_INFO_EXT = 1000091002
VK_STRUCTURE_TYPE_SWAPCHAIN_COUNTER_CREATE_INFO_EXT = 1000091003
VK_GOOGLE_display_timing = 1
VK_GOOGLE_DISPLAY_TIMING_SPEC_VERSION = 1
VK_GOOGLE_DISPLAY_TIMING_EXTENSION_NAME = "VK_GOOGLE_display_timing"
VK_STRUCTURE_TYPE_PRESENT_TIMES_INFO_GOOGLE = 1000092000
VK_NV_sample_mask_override_coverage = 1
VK_NV_SAMPLE_MASK_OVERRIDE_COVERAGE_SPEC_VERSION = 1
VK_NV_SAMPLE_MASK_OVERRIDE_COVERAGE_EXTENSION_NAME = "VK_NV_sample_mask_override_coverage"
VK_NV_geometry_shader_passthrough = 1
VK_NV_GEOMETRY_SHADER_PASSTHROUGH_SPEC_VERSION = 1
VK_NV_GEOMETRY_SHADER_PASSTHROUGH_EXTENSION_NAME = "VK_NV_geometry_shader_passthrough"
VK_NV_viewport_array2 = 1
VK_NV_VIEWPORT_ARRAY2_SPEC_VERSION = 1
VK_NV_VIEWPORT_ARRAY2_EXTENSION_NAME = "VK_NV_viewport_array2"
VK_NVX_multiview_per_view_attributes = 1
VK_NVX_MULTIVIEW_PER_VIEW_ATTRIBUTES_SPEC_VERSION = 1
VK_NVX_MULTIVIEW_PER_VIEW_ATTRIBUTES_EXTENSION_NAME = "VK_NVX_multiview_per_view_attributes"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_PER_VIEW_ATTRIBUTES_PROPERTIES_NVX = 1000097000
VK_SUBPASS_DESCRIPTION_PER_VIEW_ATTRIBUTES_BIT_NVX = 0x00000001
VK_SUBPASS_DESCRIPTION_PER_VIEW_POSITION_X_ONLY_BIT_NVX = 0x00000002
VK_NV_viewport_swizzle = 1
VK_NV_VIEWPORT_SWIZZLE_SPEC_VERSION = 1
VK_NV_VIEWPORT_SWIZZLE_EXTENSION_NAME = "VK_NV_viewport_swizzle"
VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_SWIZZLE_STATE_CREATE_INFO_NV = 1000098000
VK_EXT_discard_rectangles = 1
VK_EXT_DISCARD_RECTANGLES_SPEC_VERSION = 1
VK_EXT_DISCARD_RECTANGLES_EXTENSION_NAME = "VK_EXT_discard_rectangles"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DISCARD_RECTANGLE_PROPERTIES_EXT = 1000099000
VK_STRUCTURE_TYPE_PIPELINE_DISCARD_RECTANGLE_STATE_CREATE_INFO_EXT = 1000099001
VK_DYNAMIC_STATE_DISCARD_RECTANGLE_EXT = 1000099000
VK_NV_extension_101 = 1
VK_NV_EXTENSION_101_SPEC_VERSION = 0
VK_NV_EXTENSION_101_EXTENSION_NAME = "VK_NV_extension_101"
VK_EXT_conservative_rasterization = 1
VK_EXT_CONSERVATIVE_RASTERIZATION_SPEC_VERSION = 1
VK_EXT_CONSERVATIVE_RASTERIZATION_EXTENSION_NAME = "VK_EXT_conservative_rasterization"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CONSERVATIVE_RASTERIZATION_PROPERTIES_EXT = 1000101000
VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_CONSERVATIVE_STATE_CREATE_INFO_EXT = 1000101001
VK_NV_extension_103 = 1
VK_NV_EXTENSION_103_SPEC_VERSION = 0
VK_NV_EXTENSION_103_EXTENSION_NAME = "VK_NV_extension_103"
VK_NV_extension_104 = 1
VK_NV_EXTENSION_104_SPEC_VERSION = 0
VK_NV_EXTENSION_104_EXTENSION_NAME = "VK_NV_extension_104"
VK_EXT_swapchain_colorspace = 1
VK_EXT_SWAPCHAIN_COLOR_SPACE_SPEC_VERSION = 3
VK_EXT_SWAPCHAIN_COLOR_SPACE_EXTENSION_NAME = "VK_EXT_swapchain_colorspace"
VK_COLOR_SPACE_DISPLAY_P3_NONLINEAR_EXT = 1000104001
VK_COLOR_SPACE_EXTENDED_SRGB_LINEAR_EXT = 1000104002
VK_COLOR_SPACE_DCI_P3_LINEAR_EXT = 1000104003
VK_COLOR_SPACE_DCI_P3_NONLINEAR_EXT = 1000104004
VK_COLOR_SPACE_BT709_LINEAR_EXT = 1000104005
VK_COLOR_SPACE_BT709_NONLINEAR_EXT = 1000104006
VK_COLOR_SPACE_BT2020_LINEAR_EXT = 1000104007
VK_COLOR_SPACE_HDR10_ST2084_EXT = 1000104008
VK_COLOR_SPACE_DOLBYVISION_EXT = 1000104009
VK_COLOR_SPACE_HDR10_HLG_EXT = 1000104010
VK_COLOR_SPACE_ADOBERGB_LINEAR_EXT = 1000104011
VK_COLOR_SPACE_ADOBERGB_NONLINEAR_EXT = 1000104012
VK_COLOR_SPACE_PASS_THROUGH_EXT = 1000104013
VK_COLOR_SPACE_EXTENDED_SRGB_NONLINEAR_EXT = 1000104014
VK_EXT_hdr_metadata = 1
VK_EXT_HDR_METADATA_SPEC_VERSION = 1
VK_EXT_HDR_METADATA_EXTENSION_NAME = "VK_EXT_hdr_metadata"
VK_STRUCTURE_TYPE_HDR_METADATA_EXT = 1000105000
VK_IMG_extension_107 = 1
VK_IMG_EXTENSION_107_SPEC_VERSION = 0
VK_IMG_EXTENSION_107_EXTENSION_NAME = "VK_IMG_extension_107"
VK_IMG_extension_108 = 1
VK_IMG_EXTENSION_108_SPEC_VERSION = 0
VK_IMG_EXTENSION_108_EXTENSION_NAME = "VK_IMG_extension_108"
VK_IMG_extension_109 = 1
VK_IMG_EXTENSION_109_SPEC_VERSION = 0
VK_IMG_EXTENSION_109_EXTENSION_NAME = "VK_IMG_extension_109"
VK_KHR_create_renderpass2 = 1
VK_KHR_CREATE_RENDERPASS_2_SPEC_VERSION = 1
VK_KHR_CREATE_RENDERPASS_2_EXTENSION_NAME = "VK_KHR_create_renderpass2"
VK_STRUCTURE_TYPE_ATTACHMENT_DESCRIPTION_2_KHR = 1000109000
VK_STRUCTURE_TYPE_ATTACHMENT_REFERENCE_2_KHR = 1000109001
VK_STRUCTURE_TYPE_SUBPASS_DESCRIPTION_2_KHR = 1000109002
VK_STRUCTURE_TYPE_SUBPASS_DEPENDENCY_2_KHR = 1000109003
VK_STRUCTURE_TYPE_RENDER_PASS_CREATE_INFO_2_KHR = 1000109004
VK_STRUCTURE_TYPE_SUBPASS_BEGIN_INFO_KHR = 1000109005
VK_STRUCTURE_TYPE_SUBPASS_END_INFO_KHR = 1000109006
VK_IMG_extension_111 = 1
VK_IMG_EXTENSION_111_SPEC_VERSION = 0
VK_IMG_EXTENSION_111_EXTENSION_NAME = "VK_IMG_extension_111"
VK_KHR_shared_presentable_image = 1
VK_KHR_SHARED_PRESENTABLE_IMAGE_SPEC_VERSION = 1
VK_KHR_SHARED_PRESENTABLE_IMAGE_EXTENSION_NAME = "VK_KHR_shared_presentable_image"
VK_STRUCTURE_TYPE_SHARED_PRESENT_SURFACE_CAPABILITIES_KHR = 1000111000
VK_PRESENT_MODE_SHARED_DEMAND_REFRESH_KHR = 1000111000
VK_PRESENT_MODE_SHARED_CONTINUOUS_REFRESH_KHR = 1000111001
VK_IMAGE_LAYOUT_SHARED_PRESENT_KHR = 1000111000
VK_KHR_external_fence_capabilities = 1
VK_KHR_EXTERNAL_FENCE_CAPABILITIES_SPEC_VERSION = 1
VK_KHR_EXTERNAL_FENCE_CAPABILITIES_EXTENSION_NAME = "VK_KHR_external_fence_capabilities"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_FENCE_INFO_KHR = None
VK_STRUCTURE_TYPE_EXTERNAL_FENCE_PROPERTIES_KHR = None
VK_EXTERNAL_FENCE_HANDLE_TYPE_OPAQUE_FD_BIT_KHR = None
VK_EXTERNAL_FENCE_HANDLE_TYPE_OPAQUE_WIN32_BIT_KHR = None
VK_EXTERNAL_FENCE_HANDLE_TYPE_OPAQUE_WIN32_KMT_BIT_KHR = None
VK_EXTERNAL_FENCE_HANDLE_TYPE_SYNC_FD_BIT_KHR = None
VK_EXTERNAL_FENCE_FEATURE_EXPORTABLE_BIT_KHR = None
VK_EXTERNAL_FENCE_FEATURE_IMPORTABLE_BIT_KHR = None
VK_KHR_external_fence = 1
VK_KHR_EXTERNAL_FENCE_SPEC_VERSION = 1
VK_KHR_EXTERNAL_FENCE_EXTENSION_NAME = "VK_KHR_external_fence"
VK_STRUCTURE_TYPE_EXPORT_FENCE_CREATE_INFO_KHR = None
VK_FENCE_IMPORT_TEMPORARY_BIT_KHR = None
VK_KHR_external_fence_win32 = 1
VK_KHR_EXTERNAL_FENCE_WIN32_SPEC_VERSION = 1
VK_KHR_EXTERNAL_FENCE_WIN32_EXTENSION_NAME = "VK_KHR_external_fence_win32"
VK_STRUCTURE_TYPE_IMPORT_FENCE_WIN32_HANDLE_INFO_KHR = 1000114000
VK_STRUCTURE_TYPE_EXPORT_FENCE_WIN32_HANDLE_INFO_KHR = 1000114001
VK_STRUCTURE_TYPE_FENCE_GET_WIN32_HANDLE_INFO_KHR = 1000114002
VK_KHR_external_fence_fd = 1
VK_KHR_EXTERNAL_FENCE_FD_SPEC_VERSION = 1
VK_KHR_EXTERNAL_FENCE_FD_EXTENSION_NAME = "VK_KHR_external_fence_fd"
VK_STRUCTURE_TYPE_IMPORT_FENCE_FD_INFO_KHR = 1000115000
VK_STRUCTURE_TYPE_FENCE_GET_FD_INFO_KHR = 1000115001
VK_KHR_extension_117 = 1
VK_KHR_EXTENSION_117_SPEC_VERSION = 0
VK_KHR_EXTENSION_117_EXTENSION_NAME = "VK_KHR_extension_117"
VK_KHR_maintenance2 = 1
VK_KHR_MAINTENANCE2_SPEC_VERSION = 1
VK_KHR_MAINTENANCE2_EXTENSION_NAME = "VK_KHR_maintenance2"
VK_IMAGE_CREATE_BLOCK_TEXEL_VIEW_COMPATIBLE_BIT_KHR = None
VK_IMAGE_CREATE_EXTENDED_USAGE_BIT_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_POINT_CLIPPING_PROPERTIES_KHR = None
VK_STRUCTURE_TYPE_RENDER_PASS_INPUT_ATTACHMENT_ASPECT_CREATE_INFO_KHR = None
VK_STRUCTURE_TYPE_IMAGE_VIEW_USAGE_CREATE_INFO_KHR = None
VK_STRUCTURE_TYPE_PIPELINE_TESSELLATION_DOMAIN_ORIGIN_STATE_CREATE_INFO_KHR = None
VK_IMAGE_LAYOUT_DEPTH_READ_ONLY_STENCIL_ATTACHMENT_OPTIMAL_KHR = None
VK_IMAGE_LAYOUT_DEPTH_ATTACHMENT_STENCIL_READ_ONLY_OPTIMAL_KHR = None
VK_POINT_CLIPPING_BEHAVIOR_ALL_CLIP_PLANES_KHR = None
VK_POINT_CLIPPING_BEHAVIOR_USER_CLIP_PLANES_ONLY_KHR = None
VK_TESSELLATION_DOMAIN_ORIGIN_UPPER_LEFT_KHR = None
VK_TESSELLATION_DOMAIN_ORIGIN_LOWER_LEFT_KHR = None
VK_KHR_extension_119 = 1
VK_KHR_EXTENSION_119_SPEC_VERSION = 0
VK_KHR_EXTENSION_119_EXTENSION_NAME = "VK_KHR_extension_119"
VK_KHR_get_surface_capabilities2 = 1
VK_KHR_GET_SURFACE_CAPABILITIES_2_SPEC_VERSION = 1
VK_KHR_GET_SURFACE_CAPABILITIES_2_EXTENSION_NAME = "VK_KHR_get_surface_capabilities2"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SURFACE_INFO_2_KHR = 1000119000
VK_STRUCTURE_TYPE_SURFACE_CAPABILITIES_2_KHR = 1000119001
VK_STRUCTURE_TYPE_SURFACE_FORMAT_2_KHR = 1000119002
VK_KHR_variable_pointers = 1
VK_KHR_VARIABLE_POINTERS_SPEC_VERSION = 1
VK_KHR_VARIABLE_POINTERS_EXTENSION_NAME = "VK_KHR_variable_pointers"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VARIABLE_POINTER_FEATURES_KHR = None
VK_KHR_get_display_properties2 = 1
VK_KHR_GET_DISPLAY_PROPERTIES_2_SPEC_VERSION = 1
VK_KHR_GET_DISPLAY_PROPERTIES_2_EXTENSION_NAME = "VK_KHR_get_display_properties2"
VK_STRUCTURE_TYPE_DISPLAY_PROPERTIES_2_KHR = 1000121000
VK_STRUCTURE_TYPE_DISPLAY_PLANE_PROPERTIES_2_KHR = 1000121001
VK_STRUCTURE_TYPE_DISPLAY_MODE_PROPERTIES_2_KHR = 1000121002
VK_STRUCTURE_TYPE_DISPLAY_PLANE_INFO_2_KHR = 1000121003
VK_STRUCTURE_TYPE_DISPLAY_PLANE_CAPABILITIES_2_KHR = 1000121004
VK_MVK_ios_surface = 1
VK_MVK_IOS_SURFACE_SPEC_VERSION = 2
VK_MVK_IOS_SURFACE_EXTENSION_NAME = "VK_MVK_ios_surface"
VK_STRUCTURE_TYPE_IOS_SURFACE_CREATE_INFO_MVK = 1000122000
VK_MVK_macos_surface = 1
VK_MVK_MACOS_SURFACE_SPEC_VERSION = 2
VK_MVK_MACOS_SURFACE_EXTENSION_NAME = "VK_MVK_macos_surface"
VK_STRUCTURE_TYPE_MACOS_SURFACE_CREATE_INFO_MVK = 1000123000
VK_MVK_moltenvk = 1
VK_MVK_MOLTENVK_SPEC_VERSION = 0
VK_MVK_MOLTENVK_EXTENSION_NAME = "VK_MVK_moltenvk"
VK_EXT_external_memory_dma_buf = 1
VK_EXT_EXTERNAL_MEMORY_DMA_BUF_SPEC_VERSION = 1
VK_EXT_EXTERNAL_MEMORY_DMA_BUF_EXTENSION_NAME = "VK_EXT_external_memory_dma_buf"
VK_EXTERNAL_MEMORY_HANDLE_TYPE_DMA_BUF_BIT_EXT = 0x00000200
VK_EXT_queue_family_foreign = 1
VK_EXT_QUEUE_FAMILY_FOREIGN_SPEC_VERSION = 1
VK_EXT_QUEUE_FAMILY_FOREIGN_EXTENSION_NAME = "VK_EXT_queue_family_foreign"
VK_KHR_dedicated_allocation = 1
VK_KHR_DEDICATED_ALLOCATION_SPEC_VERSION = 3
VK_KHR_DEDICATED_ALLOCATION_EXTENSION_NAME = "VK_KHR_dedicated_allocation"
VK_STRUCTURE_TYPE_MEMORY_DEDICATED_REQUIREMENTS_KHR = None
VK_STRUCTURE_TYPE_MEMORY_DEDICATED_ALLOCATE_INFO_KHR = None
VK_EXT_debug_utils = 1
VK_EXT_DEBUG_UTILS_SPEC_VERSION = 1
VK_EXT_DEBUG_UTILS_EXTENSION_NAME = "VK_EXT_debug_utils"
VK_STRUCTURE_TYPE_DEBUG_UTILS_OBJECT_NAME_INFO_EXT = 1000128000
VK_STRUCTURE_TYPE_DEBUG_UTILS_OBJECT_TAG_INFO_EXT = 1000128001
VK_STRUCTURE_TYPE_DEBUG_UTILS_LABEL_EXT = 1000128002
VK_STRUCTURE_TYPE_DEBUG_UTILS_MESSENGER_CALLBACK_DATA_EXT = 1000128003
VK_STRUCTURE_TYPE_DEBUG_UTILS_MESSENGER_CREATE_INFO_EXT = 1000128004
VK_OBJECT_TYPE_DEBUG_UTILS_MESSENGER_EXT = 1000128000
VK_ANDROID_external_memory_android_hardware_buffer = 1
VK_ANDROID_EXTERNAL_MEMORY_ANDROID_HARDWARE_BUFFER_SPEC_VERSION = 3
VK_ANDROID_EXTERNAL_MEMORY_ANDROID_HARDWARE_BUFFER_EXTENSION_NAME = "VK_ANDROID_external_memory_android_hardware_buffer"
VK_EXTERNAL_MEMORY_HANDLE_TYPE_ANDROID_HARDWARE_BUFFER_BIT_ANDROID = 0x00000400
VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_USAGE_ANDROID = 1000129000
VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_PROPERTIES_ANDROID = 1000129001
VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_FORMAT_PROPERTIES_ANDROID = 1000129002
VK_STRUCTURE_TYPE_IMPORT_ANDROID_HARDWARE_BUFFER_INFO_ANDROID = 1000129003
VK_STRUCTURE_TYPE_MEMORY_GET_ANDROID_HARDWARE_BUFFER_INFO_ANDROID = 1000129004
VK_STRUCTURE_TYPE_EXTERNAL_FORMAT_ANDROID = 1000129005
VK_EXT_sampler_filter_minmax = 1
VK_EXT_SAMPLER_FILTER_MINMAX_SPEC_VERSION = 1
VK_EXT_SAMPLER_FILTER_MINMAX_EXTENSION_NAME = "VK_EXT_sampler_filter_minmax"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLER_FILTER_MINMAX_PROPERTIES_EXT = 1000130000
VK_STRUCTURE_TYPE_SAMPLER_REDUCTION_MODE_CREATE_INFO_EXT = 1000130001
VK_FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_MINMAX_BIT_EXT = 0x00010000
VK_KHR_storage_buffer_storage_class = 1
VK_KHR_STORAGE_BUFFER_STORAGE_CLASS_SPEC_VERSION = 1
VK_KHR_STORAGE_BUFFER_STORAGE_CLASS_EXTENSION_NAME = "VK_KHR_storage_buffer_storage_class"
VK_AMD_gpu_shader_int16 = 1
VK_AMD_GPU_SHADER_INT16_SPEC_VERSION = 1
VK_AMD_GPU_SHADER_INT16_EXTENSION_NAME = "VK_AMD_gpu_shader_int16"
VK_AMD_extension_134 = 1
VK_AMD_EXTENSION_134_SPEC_VERSION = 0
VK_AMD_EXTENSION_134_EXTENSION_NAME = "VK_AMD_extension_134"
VK_AMD_extension_135 = 1
VK_AMD_EXTENSION_135_SPEC_VERSION = 0
VK_AMD_EXTENSION_135_EXTENSION_NAME = "VK_AMD_extension_135"
VK_AMD_extension_136 = 1
VK_AMD_EXTENSION_136_SPEC_VERSION = 0
VK_AMD_EXTENSION_136_EXTENSION_NAME = "VK_AMD_extension_136"
VK_AMD_mixed_attachment_samples = 1
VK_AMD_MIXED_ATTACHMENT_SAMPLES_SPEC_VERSION = 1
VK_AMD_MIXED_ATTACHMENT_SAMPLES_EXTENSION_NAME = "VK_AMD_mixed_attachment_samples"
VK_AMD_shader_fragment_mask = 1
VK_AMD_SHADER_FRAGMENT_MASK_SPEC_VERSION = 1
VK_AMD_SHADER_FRAGMENT_MASK_EXTENSION_NAME = "VK_AMD_shader_fragment_mask"
VK_EXT_inline_uniform_block = 1
VK_EXT_INLINE_UNIFORM_BLOCK_SPEC_VERSION = 1
VK_EXT_INLINE_UNIFORM_BLOCK_EXTENSION_NAME = "VK_EXT_inline_uniform_block"
VK_DESCRIPTOR_TYPE_INLINE_UNIFORM_BLOCK_EXT = 1000138000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INLINE_UNIFORM_BLOCK_FEATURES_EXT = 1000138000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INLINE_UNIFORM_BLOCK_PROPERTIES_EXT = 1000138001
VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET_INLINE_UNIFORM_BLOCK_EXT = 1000138002
VK_STRUCTURE_TYPE_DESCRIPTOR_POOL_INLINE_UNIFORM_BLOCK_CREATE_INFO_EXT = 1000138003
VK_AMD_extension_140 = 1
VK_AMD_EXTENSION_140_SPEC_VERSION = 0
VK_AMD_EXTENSION_140_EXTENSION_NAME = "VK_AMD_extension_140"
VK_EXT_shader_stencil_export = 1
VK_EXT_SHADER_STENCIL_EXPORT_SPEC_VERSION = 1
VK_EXT_SHADER_STENCIL_EXPORT_EXTENSION_NAME = "VK_EXT_shader_stencil_export"
VK_AMD_extension_142 = 1
VK_AMD_EXTENSION_142_SPEC_VERSION = 0
VK_AMD_EXTENSION_142_EXTENSION_NAME = "VK_AMD_extension_142"
VK_AMD_extension_143 = 1
VK_AMD_EXTENSION_143_SPEC_VERSION = 0
VK_AMD_EXTENSION_143_EXTENSION_NAME = "VK_AMD_extension_143"
VK_EXT_sample_locations = 1
VK_EXT_SAMPLE_LOCATIONS_SPEC_VERSION = 1
VK_EXT_SAMPLE_LOCATIONS_EXTENSION_NAME = "VK_EXT_sample_locations"
VK_IMAGE_CREATE_SAMPLE_LOCATIONS_COMPATIBLE_DEPTH_BIT_EXT = 0x00001000
VK_STRUCTURE_TYPE_SAMPLE_LOCATIONS_INFO_EXT = 1000143000
VK_STRUCTURE_TYPE_RENDER_PASS_SAMPLE_LOCATIONS_BEGIN_INFO_EXT = 1000143001
VK_STRUCTURE_TYPE_PIPELINE_SAMPLE_LOCATIONS_STATE_CREATE_INFO_EXT = 1000143002
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLE_LOCATIONS_PROPERTIES_EXT = 1000143003
VK_STRUCTURE_TYPE_MULTISAMPLE_PROPERTIES_EXT = 1000143004
VK_DYNAMIC_STATE_SAMPLE_LOCATIONS_EXT = 1000143000
VK_KHR_relaxed_block_layout = 1
VK_KHR_RELAXED_BLOCK_LAYOUT_SPEC_VERSION = 1
VK_KHR_RELAXED_BLOCK_LAYOUT_EXTENSION_NAME = "VK_KHR_relaxed_block_layout"
VK_KHR_get_memory_requirements2 = 1
VK_KHR_GET_MEMORY_REQUIREMENTS_2_SPEC_VERSION = 1
VK_KHR_GET_MEMORY_REQUIREMENTS_2_EXTENSION_NAME = "VK_KHR_get_memory_requirements2"
VK_STRUCTURE_TYPE_BUFFER_MEMORY_REQUIREMENTS_INFO_2_KHR = None
VK_STRUCTURE_TYPE_IMAGE_MEMORY_REQUIREMENTS_INFO_2_KHR = None
VK_STRUCTURE_TYPE_IMAGE_SPARSE_MEMORY_REQUIREMENTS_INFO_2_KHR = None
VK_STRUCTURE_TYPE_MEMORY_REQUIREMENTS_2_KHR = None
VK_STRUCTURE_TYPE_SPARSE_IMAGE_MEMORY_REQUIREMENTS_2_KHR = None
VK_KHR_image_format_list = 1
VK_KHR_IMAGE_FORMAT_LIST_SPEC_VERSION = 1
VK_KHR_IMAGE_FORMAT_LIST_EXTENSION_NAME = "VK_KHR_image_format_list"
VK_STRUCTURE_TYPE_IMAGE_FORMAT_LIST_CREATE_INFO_KHR = 1000147000
VK_EXT_blend_operation_advanced = 1
VK_EXT_BLEND_OPERATION_ADVANCED_SPEC_VERSION = 2
VK_EXT_BLEND_OPERATION_ADVANCED_EXTENSION_NAME = "VK_EXT_blend_operation_advanced"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BLEND_OPERATION_ADVANCED_FEATURES_EXT = 1000148000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BLEND_OPERATION_ADVANCED_PROPERTIES_EXT = 1000148001
VK_STRUCTURE_TYPE_PIPELINE_COLOR_BLEND_ADVANCED_STATE_CREATE_INFO_EXT = 1000148002
VK_BLEND_OP_ZERO_EXT = 1000148000
VK_BLEND_OP_SRC_EXT = 1000148001
VK_BLEND_OP_DST_EXT = 1000148002
VK_BLEND_OP_SRC_OVER_EXT = 1000148003
VK_BLEND_OP_DST_OVER_EXT = 1000148004
VK_BLEND_OP_SRC_IN_EXT = 1000148005
VK_BLEND_OP_DST_IN_EXT = 1000148006
VK_BLEND_OP_SRC_OUT_EXT = 1000148007
VK_BLEND_OP_DST_OUT_EXT = 1000148008
VK_BLEND_OP_SRC_ATOP_EXT = 1000148009
VK_BLEND_OP_DST_ATOP_EXT = 1000148010
VK_BLEND_OP_XOR_EXT = 1000148011
VK_BLEND_OP_MULTIPLY_EXT = 1000148012
VK_BLEND_OP_SCREEN_EXT = 1000148013
VK_BLEND_OP_OVERLAY_EXT = 1000148014
VK_BLEND_OP_DARKEN_EXT = 1000148015
VK_BLEND_OP_LIGHTEN_EXT = 1000148016
VK_BLEND_OP_COLORDODGE_EXT = 1000148017
VK_BLEND_OP_COLORBURN_EXT = 1000148018
VK_BLEND_OP_HARDLIGHT_EXT = 1000148019
VK_BLEND_OP_SOFTLIGHT_EXT = 1000148020
VK_BLEND_OP_DIFFERENCE_EXT = 1000148021
VK_BLEND_OP_EXCLUSION_EXT = 1000148022
VK_BLEND_OP_INVERT_EXT = 1000148023
VK_BLEND_OP_INVERT_RGB_EXT = 1000148024
VK_BLEND_OP_LINEARDODGE_EXT = 1000148025
VK_BLEND_OP_LINEARBURN_EXT = 1000148026
VK_BLEND_OP_VIVIDLIGHT_EXT = 1000148027
VK_BLEND_OP_LINEARLIGHT_EXT = 1000148028
VK_BLEND_OP_PINLIGHT_EXT = 1000148029
VK_BLEND_OP_HARDMIX_EXT = 1000148030
VK_BLEND_OP_HSL_HUE_EXT = 1000148031
VK_BLEND_OP_HSL_SATURATION_EXT = 1000148032
VK_BLEND_OP_HSL_COLOR_EXT = 1000148033
VK_BLEND_OP_HSL_LUMINOSITY_EXT = 1000148034
VK_BLEND_OP_PLUS_EXT = 1000148035
VK_BLEND_OP_PLUS_CLAMPED_EXT = 1000148036
VK_BLEND_OP_PLUS_CLAMPED_ALPHA_EXT = 1000148037
VK_BLEND_OP_PLUS_DARKER_EXT = 1000148038
VK_BLEND_OP_MINUS_EXT = 1000148039
VK_BLEND_OP_MINUS_CLAMPED_EXT = 1000148040
VK_BLEND_OP_CONTRAST_EXT = 1000148041
VK_BLEND_OP_INVERT_OVG_EXT = 1000148042
VK_BLEND_OP_RED_EXT = 1000148043
VK_BLEND_OP_GREEN_EXT = 1000148044
VK_BLEND_OP_BLUE_EXT = 1000148045
VK_ACCESS_COLOR_ATTACHMENT_READ_NONCOHERENT_BIT_EXT = 0x00080000
VK_NV_fragment_coverage_to_color = 1
VK_NV_FRAGMENT_COVERAGE_TO_COLOR_SPEC_VERSION = 1
VK_NV_FRAGMENT_COVERAGE_TO_COLOR_EXTENSION_NAME = "VK_NV_fragment_coverage_to_color"
VK_STRUCTURE_TYPE_PIPELINE_COVERAGE_TO_COLOR_STATE_CREATE_INFO_NV = 1000149000
VK_NV_extension_151 = 1
VK_NV_EXTENSION_151_SPEC_VERSION = 0
VK_NV_EXTENSION_151_EXTENSION_NAME = "VK_NV_extension_151"
VK_NV_extension_152 = 1
VK_NV_EXTENSION_152_SPEC_VERSION = 0
VK_NV_EXTENSION_152_EXTENSION_NAME = "VK_NV_extension_152"
VK_NV_framebuffer_mixed_samples = 1
VK_NV_FRAMEBUFFER_MIXED_SAMPLES_SPEC_VERSION = 1
VK_NV_FRAMEBUFFER_MIXED_SAMPLES_EXTENSION_NAME = "VK_NV_framebuffer_mixed_samples"
VK_STRUCTURE_TYPE_PIPELINE_COVERAGE_MODULATION_STATE_CREATE_INFO_NV = 1000152000
VK_NV_fill_rectangle = 1
VK_NV_FILL_RECTANGLE_SPEC_VERSION = 1
VK_NV_FILL_RECTANGLE_EXTENSION_NAME = "VK_NV_fill_rectangle"
VK_POLYGON_MODE_FILL_RECTANGLE_NV = 1000153000
VK_NV_extension_155 = 1
VK_NV_EXTENSION_155_SPEC_VERSION = 0
VK_NV_EXTENSION_155_EXTENSION_NAME = "VK_NV_extension_155"
VK_EXT_post_depth_coverage = 1
VK_EXT_POST_DEPTH_COVERAGE_SPEC_VERSION = 1
VK_EXT_POST_DEPTH_COVERAGE_EXTENSION_NAME = "VK_EXT_post_depth_coverage"
VK_KHR_sampler_ycbcr_conversion = 1
VK_KHR_SAMPLER_YCBCR_CONVERSION_SPEC_VERSION = 1
VK_KHR_SAMPLER_YCBCR_CONVERSION_EXTENSION_NAME = "VK_KHR_sampler_ycbcr_conversion"
VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_CREATE_INFO_KHR = None
VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_INFO_KHR = None
VK_STRUCTURE_TYPE_BIND_IMAGE_PLANE_MEMORY_INFO_KHR = None
VK_STRUCTURE_TYPE_IMAGE_PLANE_MEMORY_REQUIREMENTS_INFO_KHR = None
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLER_YCBCR_CONVERSION_FEATURES_KHR = None
VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_IMAGE_FORMAT_PROPERTIES_KHR = None
VK_DEBUG_REPORT_OBJECT_TYPE_SAMPLER_YCBCR_CONVERSION_KHR_EXT = None
VK_OBJECT_TYPE_SAMPLER_YCBCR_CONVERSION_KHR = None
VK_FORMAT_G8B8G8R8_422_UNORM_KHR = None
VK_FORMAT_B8G8R8G8_422_UNORM_KHR = None
VK_FORMAT_G8_B8_R8_3PLANE_420_UNORM_KHR = None
VK_FORMAT_G8_B8R8_2PLANE_420_UNORM_KHR = None
VK_FORMAT_G8_B8_R8_3PLANE_422_UNORM_KHR = None
VK_FORMAT_G8_B8R8_2PLANE_422_UNORM_KHR = None
VK_FORMAT_G8_B8_R8_3PLANE_444_UNORM_KHR = None
VK_FORMAT_R10X6_UNORM_PACK16_KHR = None
VK_FORMAT_R10X6G10X6_UNORM_2PACK16_KHR = None
VK_FORMAT_R10X6G10X6B10X6A10X6_UNORM_4PACK16_KHR = None
VK_FORMAT_G10X6B10X6G10X6R10X6_422_UNORM_4PACK16_KHR = None
VK_FORMAT_B10X6G10X6R10X6G10X6_422_UNORM_4PACK16_KHR = None
VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_420_UNORM_3PACK16_KHR = None
VK_FORMAT_G10X6_B10X6R10X6_2PLANE_420_UNORM_3PACK16_KHR = None
VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_422_UNORM_3PACK16_KHR = None
VK_FORMAT_G10X6_B10X6R10X6_2PLANE_422_UNORM_3PACK16_KHR = None
VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_444_UNORM_3PACK16_KHR = None
VK_FORMAT_R12X4_UNORM_PACK16_KHR = None
VK_FORMAT_R12X4G12X4_UNORM_2PACK16_KHR = None
VK_FORMAT_R12X4G12X4B12X4A12X4_UNORM_4PACK16_KHR = None
VK_FORMAT_G12X4B12X4G12X4R12X4_422_UNORM_4PACK16_KHR = None
VK_FORMAT_B12X4G12X4R12X4G12X4_422_UNORM_4PACK16_KHR = None
VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_420_UNORM_3PACK16_KHR = None
VK_FORMAT_G12X4_B12X4R12X4_2PLANE_420_UNORM_3PACK16_KHR = None
VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_422_UNORM_3PACK16_KHR = None
VK_FORMAT_G12X4_B12X4R12X4_2PLANE_422_UNORM_3PACK16_KHR = None
VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_444_UNORM_3PACK16_KHR = None
VK_FORMAT_G16B16G16R16_422_UNORM_KHR = None
VK_FORMAT_B16G16R16G16_422_UNORM_KHR = None
VK_FORMAT_G16_B16_R16_3PLANE_420_UNORM_KHR = None
VK_FORMAT_G16_B16R16_2PLANE_420_UNORM_KHR = None
VK_FORMAT_G16_B16_R16_3PLANE_422_UNORM_KHR = None
VK_FORMAT_G16_B16R16_2PLANE_422_UNORM_KHR = None
VK_FORMAT_G16_B16_R16_3PLANE_444_UNORM_KHR = None
VK_IMAGE_ASPECT_PLANE_0_BIT_KHR = None
VK_IMAGE_ASPECT_PLANE_1_BIT_KHR = None
VK_IMAGE_ASPECT_PLANE_2_BIT_KHR = None
VK_IMAGE_CREATE_DISJOINT_BIT_KHR = None
VK_FORMAT_FEATURE_MIDPOINT_CHROMA_SAMPLES_BIT_KHR = None
VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_LINEAR_FILTER_BIT_KHR = None
VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_SEPARATE_RECONSTRUCTION_FILTER_BIT_KHR = None
VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_CHROMA_RECONSTRUCTION_EXPLICIT_BIT_KHR = None
VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_CHROMA_RECONSTRUCTION_EXPLICIT_FORCEABLE_BIT_KHR = None
VK_FORMAT_FEATURE_DISJOINT_BIT_KHR = None
VK_FORMAT_FEATURE_COSITED_CHROMA_SAMPLES_BIT_KHR = None
VK_SAMPLER_YCBCR_MODEL_CONVERSION_RGB_IDENTITY_KHR = None
VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_IDENTITY_KHR = None
VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_709_KHR = None
VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_601_KHR = None
VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_2020_KHR = None
VK_SAMPLER_YCBCR_RANGE_ITU_FULL_KHR = None
VK_SAMPLER_YCBCR_RANGE_ITU_NARROW_KHR = None
VK_CHROMA_LOCATION_COSITED_EVEN_KHR = None
VK_CHROMA_LOCATION_MIDPOINT_KHR = None
VK_KHR_bind_memory2 = 1
VK_KHR_BIND_MEMORY_2_SPEC_VERSION = 1
VK_KHR_BIND_MEMORY_2_EXTENSION_NAME = "VK_KHR_bind_memory2"
VK_STRUCTURE_TYPE_BIND_BUFFER_MEMORY_INFO_KHR = None
VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_INFO_KHR = None
VK_IMAGE_CREATE_ALIAS_BIT_KHR = None
VK_EXT_image_drm_format_modifier = 1
VK_EXT_IMAGE_DRM_FORMAT_MODIFIER_SPEC_VERSION = 1
VK_EXT_IMAGE_DRM_FORMAT_MODIFIER_EXTENSION_NAME = "VK_EXT_image_drm_format_modifier"
VK_STRUCTURE_TYPE_DRM_FORMAT_MODIFIER_PROPERTIES_LIST_EXT = 1000158000
VK_STRUCTURE_TYPE_DRM_FORMAT_MODIFIER_PROPERTIES_EXT = 1000158001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_DRM_FORMAT_MODIFIER_INFO_EXT = 1000158002
VK_STRUCTURE_TYPE_IMAGE_DRM_FORMAT_MODIFIER_LIST_CREATE_INFO_EXT = 1000158003
VK_STRUCTURE_TYPE_IMAGE_DRM_FORMAT_MODIFIER_EXPLICIT_CREATE_INFO_EXT = 1000158004
VK_STRUCTURE_TYPE_IMAGE_DRM_FORMAT_MODIFIER_PROPERTIES_EXT = 1000158005
VK_IMAGE_TILING_DRM_FORMAT_MODIFIER_EXT = 1000158000
VK_IMAGE_ASPECT_MEMORY_PLANE_0_BIT_EXT = 0x00000080
VK_IMAGE_ASPECT_MEMORY_PLANE_1_BIT_EXT = 0x00000100
VK_IMAGE_ASPECT_MEMORY_PLANE_2_BIT_EXT = 0x00000200
VK_IMAGE_ASPECT_MEMORY_PLANE_3_BIT_EXT = 0x00000400
VK_EXT_extension_160 = 1
VK_EXT_EXTENSION_160_SPEC_VERSION = 0
VK_EXT_EXTENSION_160_EXTENSION_NAME = "VK_EXT_extension_160"
VK_EXT_validation_cache = 1
VK_EXT_VALIDATION_CACHE_SPEC_VERSION = 1
VK_EXT_VALIDATION_CACHE_EXTENSION_NAME = "VK_EXT_validation_cache"
VK_STRUCTURE_TYPE_VALIDATION_CACHE_CREATE_INFO_EXT = 1000160000
VK_STRUCTURE_TYPE_SHADER_MODULE_VALIDATION_CACHE_CREATE_INFO_EXT = 1000160001
VK_OBJECT_TYPE_VALIDATION_CACHE_EXT = 1000160000
VK_EXT_descriptor_indexing = 1
VK_EXT_DESCRIPTOR_INDEXING_SPEC_VERSION = 2
VK_EXT_DESCRIPTOR_INDEXING_EXTENSION_NAME = "VK_EXT_descriptor_indexing"
VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_BINDING_FLAGS_CREATE_INFO_EXT = 1000161000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_INDEXING_FEATURES_EXT = 1000161001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_INDEXING_PROPERTIES_EXT = 1000161002
VK_STRUCTURE_TYPE_DESCRIPTOR_SET_VARIABLE_DESCRIPTOR_COUNT_ALLOCATE_INFO_EXT = 1000161003
VK_STRUCTURE_TYPE_DESCRIPTOR_SET_VARIABLE_DESCRIPTOR_COUNT_LAYOUT_SUPPORT_EXT = 1000161004
VK_DESCRIPTOR_POOL_CREATE_UPDATE_AFTER_BIND_BIT_EXT = 0x00000002
VK_DESCRIPTOR_SET_LAYOUT_CREATE_UPDATE_AFTER_BIND_POOL_BIT_EXT = 0x00000002
VK_EXT_shader_viewport_index_layer = 1
VK_EXT_SHADER_VIEWPORT_INDEX_LAYER_SPEC_VERSION = 1
VK_EXT_SHADER_VIEWPORT_INDEX_LAYER_EXTENSION_NAME = "VK_EXT_shader_viewport_index_layer"
VK_NV_extension_164 = 1
VK_EXT_EXTENSION_164_SPEC_VERSION = 0
VK_EXT_EXTENSION_164_EXTENSION_NAME = "VK_NV_extension_164"
VK_NV_shading_rate_image = 1
VK_NV_SHADING_RATE_IMAGE_SPEC_VERSION = 3
VK_NV_SHADING_RATE_IMAGE_EXTENSION_NAME = "VK_NV_shading_rate_image"
VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_SHADING_RATE_IMAGE_STATE_CREATE_INFO_NV = 1000164000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADING_RATE_IMAGE_FEATURES_NV = 1000164001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADING_RATE_IMAGE_PROPERTIES_NV = 1000164002
VK_IMAGE_LAYOUT_SHADING_RATE_OPTIMAL_NV = 1000164003
VK_DYNAMIC_STATE_VIEWPORT_SHADING_RATE_PALETTE_NV = 1000164004
VK_ACCESS_SHADING_RATE_IMAGE_READ_BIT_NV = 0x00800000
VK_IMAGE_USAGE_SHADING_RATE_IMAGE_BIT_NV = 0x00000100
VK_PIPELINE_STAGE_SHADING_RATE_IMAGE_BIT_NV = 0x00400000
VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_COARSE_SAMPLE_ORDER_STATE_CREATE_INFO_NV = 1000164005
VK_DYNAMIC_STATE_VIEWPORT_COARSE_SAMPLE_ORDER_NV = 1000164006
VK_NV_ray_tracing = 1
VK_NV_RAY_TRACING_SPEC_VERSION = 3
VK_NV_RAY_TRACING_EXTENSION_NAME = "VK_NV_ray_tracing"
VK_STRUCTURE_TYPE_RAY_TRACING_PIPELINE_CREATE_INFO_NV = 1000165000
VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_CREATE_INFO_NV = 1000165001
VK_STRUCTURE_TYPE_GEOMETRY_NV = 1000165003
VK_STRUCTURE_TYPE_GEOMETRY_TRIANGLES_NV = 1000165004
VK_STRUCTURE_TYPE_GEOMETRY_AABB_NV = 1000165005
VK_STRUCTURE_TYPE_BIND_ACCELERATION_STRUCTURE_MEMORY_INFO_NV = 1000165006
VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET_ACCELERATION_STRUCTURE_NV = 1000165007
VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_INFO_NV = 1000165008
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_TRACING_PROPERTIES_NV = 1000165009
VK_STRUCTURE_TYPE_RAY_TRACING_SHADER_GROUP_CREATE_INFO_NV = 1000165011
VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_INFO_NV = 1000165012
VK_SHADER_STAGE_RAYGEN_BIT_NV = 0x00000100
VK_SHADER_STAGE_ANY_HIT_BIT_NV = 0x00000200
VK_SHADER_STAGE_CLOSEST_HIT_BIT_NV = 0x00000400
VK_SHADER_STAGE_MISS_BIT_NV = 0x00000800
VK_SHADER_STAGE_INTERSECTION_BIT_NV = 0x00001000
VK_SHADER_STAGE_CALLABLE_BIT_NV = 0x00002000
VK_PIPELINE_STAGE_RAY_TRACING_SHADER_BIT_NV = 0x00200000
VK_PIPELINE_STAGE_ACCELERATION_STRUCTURE_BUILD_BIT_NV = 0x02000000
VK_BUFFER_USAGE_RAY_TRACING_BIT_NV = 0x00000400
VK_PIPELINE_BIND_POINT_RAY_TRACING_NV = 1000165000
VK_DESCRIPTOR_TYPE_ACCELERATION_STRUCTURE_NV = 1000165000
VK_ACCESS_ACCELERATION_STRUCTURE_READ_BIT_NV = 0x00200000
VK_ACCESS_ACCELERATION_STRUCTURE_WRITE_BIT_NV = 0x00400000
VK_QUERY_TYPE_ACCELERATION_STRUCTURE_COMPACTED_SIZE_NV = 1000165000
VK_PIPELINE_CREATE_DEFER_COMPILE_BIT_NV = 0x00000020
VK_OBJECT_TYPE_ACCELERATION_STRUCTURE_NV = 1000165000
VK_DEBUG_REPORT_OBJECT_TYPE_ACCELERATION_STRUCTURE_NV_EXT = 1000165000
VK_INDEX_TYPE_NONE_NV = 1000165000
VK_NV_representative_fragment_test = 1
VK_NV_REPRESENTATIVE_FRAGMENT_TEST_SPEC_VERSION = 1
VK_NV_REPRESENTATIVE_FRAGMENT_TEST_EXTENSION_NAME = "VK_NV_representative_fragment_test"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_REPRESENTATIVE_FRAGMENT_TEST_FEATURES_NV = 1000166000
VK_STRUCTURE_TYPE_PIPELINE_REPRESENTATIVE_FRAGMENT_TEST_STATE_CREATE_INFO_NV = 1000166001
VK_NV_extension_168 = 1
VK_EXT_EXTENSION_168_SPEC_VERSION = 0
VK_EXT_EXTENSION_168_EXTENSION_NAME = "VK_NV_extension_168"
VK_KHR_maintenance3 = 1
VK_KHR_MAINTENANCE3_SPEC_VERSION = 1
VK_KHR_MAINTENANCE3_EXTENSION_NAME = "VK_KHR_maintenance3"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_3_PROPERTIES_KHR = None
VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_SUPPORT_KHR = None
VK_KHR_draw_indirect_count = 1
VK_KHR_DRAW_INDIRECT_COUNT_SPEC_VERSION = 1
VK_KHR_DRAW_INDIRECT_COUNT_EXTENSION_NAME = "VK_KHR_draw_indirect_count"
VK_EXT_filter_cubic = 1
VK_EXT_FILTER_CUBIC_SPEC_VERSION = 1
VK_EXT_FILTER_CUBIC_EXTENSION_NAME = "VK_EXT_filter_cubic"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_VIEW_IMAGE_FORMAT_INFO_EXT = 1000170000
VK_STRUCTURE_TYPE_FILTER_CUBIC_IMAGE_VIEW_IMAGE_FORMAT_PROPERTIES_EXT = 1000170001
VK_QCOM_extension_172 = 1
VK_QCOM_extension_172_SPEC_VERSION = 0
VK_QCOM_extension_172_EXTENSION_NAME = "VK_QCOM_extension_172"
VK_QCOM_extension_173 = 1
VK_QCOM_extension_173_SPEC_VERSION = 0
VK_QCOM_extension_173_EXTENSION_NAME = "VK_QCOM_extension_173"
VK_QCOM_extension_174 = 1
VK_QCOM_extension_174_SPEC_VERSION = 0
VK_QCOM_extension_174_EXTENSION_NAME = "VK_QCOM_extension_174"
VK_EXT_global_priority = 1
VK_EXT_GLOBAL_PRIORITY_SPEC_VERSION = 2
VK_EXT_GLOBAL_PRIORITY_EXTENSION_NAME = "VK_EXT_global_priority"
VK_STRUCTURE_TYPE_DEVICE_QUEUE_GLOBAL_PRIORITY_CREATE_INFO_EXT = 1000174000
VK_EXT_extension_176 = 1
VK_KHR_EXTENSION_176_SPEC_VERSION = 0
VK_KHR_EXTENSION_176_EXTENSION_NAME = "VK_KHR_extension_176"
VK_EXT_extension_177 = 1
VK_KHR_EXTENSION_177_SPEC_VERSION = 0
VK_KHR_EXTENSION_177_EXTENSION_NAME = "VK_KHR_extension_177"
VK_KHR_8bit_storage = 1
VK_KHR_8BIT_STORAGE_SPEC_VERSION = 1
VK_KHR_8BIT_STORAGE_EXTENSION_NAME = "VK_KHR_8bit_storage"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_8BIT_STORAGE_FEATURES_KHR = 1000177000
VK_EXT_external_memory_host = 1
VK_EXT_EXTERNAL_MEMORY_HOST_SPEC_VERSION = 1
VK_EXT_EXTERNAL_MEMORY_HOST_EXTENSION_NAME = "VK_EXT_external_memory_host"
VK_STRUCTURE_TYPE_IMPORT_MEMORY_HOST_POINTER_INFO_EXT = 1000178000
VK_STRUCTURE_TYPE_MEMORY_HOST_POINTER_PROPERTIES_EXT = 1000178001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_MEMORY_HOST_PROPERTIES_EXT = 1000178002
VK_EXTERNAL_MEMORY_HANDLE_TYPE_HOST_ALLOCATION_BIT_EXT = 0x00000080
VK_EXTERNAL_MEMORY_HANDLE_TYPE_HOST_MAPPED_FOREIGN_MEMORY_BIT_EXT = 0x00000100
VK_AMD_buffer_marker = 1
VK_AMD_BUFFER_MARKER_SPEC_VERSION = 1
VK_AMD_BUFFER_MARKER_EXTENSION_NAME = "VK_AMD_buffer_marker"
VK_KHR_shader_atomic_int64 = 1
VK_KHR_SHADER_ATOMIC_INT64_SPEC_VERSION = 1
VK_KHR_SHADER_ATOMIC_INT64_EXTENSION_NAME = "VK_KHR_shader_atomic_int64"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_ATOMIC_INT64_FEATURES_KHR = 1000180000
VK_AMD_extension_182 = 1
VK_KHR_EXTENSION_182_SPEC_VERSION = 0
VK_KHR_EXTENSION_182_EXTENSION_NAME = "VK_AMD_extension_182"
VK_AMD_extension_183 = 1
VK_KHR_EXTENSION_183_SPEC_VERSION = 0
VK_KHR_EXTENSION_183_EXTENSION_NAME = "VK_AMD_extension_183"
VK_AMD_extension_184 = 1
VK_KHR_EXTENSION_184_SPEC_VERSION = 0
VK_KHR_EXTENSION_184_EXTENSION_NAME = "VK_AMD_extension_184"
VK_EXT_calibrated_timestamps = 1
VK_EXT_CALIBRATED_TIMESTAMPS_SPEC_VERSION = 1
VK_EXT_CALIBRATED_TIMESTAMPS_EXTENSION_NAME = "VK_EXT_calibrated_timestamps"
VK_STRUCTURE_TYPE_CALIBRATED_TIMESTAMP_INFO_EXT = 1000184000
VK_AMD_shader_core_properties = 1
VK_AMD_SHADER_CORE_PROPERTIES_SPEC_VERSION = 1
VK_AMD_SHADER_CORE_PROPERTIES_EXTENSION_NAME = "VK_AMD_shader_core_properties"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_CORE_PROPERTIES_AMD = 1000185000
VK_AMD_extension_187 = 1
VK_KHR_EXTENSION_187_SPEC_VERSION = 0
VK_KHR_EXTENSION_187_EXTENSION_NAME = "VK_AMD_extension_187"
VK_AMD_extension_188 = 1
VK_KHR_EXTENSION_188_SPEC_VERSION = 0
VK_KHR_EXTENSION_188_EXTENSION_NAME = "VK_AMD_extension_188"
VK_AMD_extension_189 = 1
VK_KHR_EXTENSION_189_SPEC_VERSION = 0
VK_KHR_EXTENSION_189_EXTENSION_NAME = "VK_AMD_extension_189"
VK_AMD_memory_overallocation_behavior = 1
VK_AMD_MEMORY_OVERALLOCATION_BEHAVIOR_SPEC_VERSION = 1
VK_AMD_MEMORY_OVERALLOCATION_BEHAVIOR_EXTENSION_NAME = "VK_AMD_memory_overallocation_behavior"
VK_STRUCTURE_TYPE_DEVICE_MEMORY_OVERALLOCATION_CREATE_INFO_AMD = 1000189000
VK_EXT_vertex_attribute_divisor = 1
VK_EXT_VERTEX_ATTRIBUTE_DIVISOR_SPEC_VERSION = 3
VK_EXT_VERTEX_ATTRIBUTE_DIVISOR_EXTENSION_NAME = "VK_EXT_vertex_attribute_divisor"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VERTEX_ATTRIBUTE_DIVISOR_PROPERTIES_EXT = 1000190000
VK_STRUCTURE_TYPE_PIPELINE_VERTEX_INPUT_DIVISOR_STATE_CREATE_INFO_EXT = 1000190001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VERTEX_ATTRIBUTE_DIVISOR_FEATURES_EXT = 1000190002
VK_GOOGLE_extension_192 = 1
VK_GOOGLE_EXTENSION_192_SPEC_VERSION = 0
VK_GOOGLE_EXTENSION_192_EXTENSION_NAME = "VK_GOOGLE_extension_192"
VK_GOOGLE_extension_193 = 1
VK_GOOGLE_EXTENSION_193_SPEC_VERSION = 0
VK_GOOGLE_EXTENSION_193_EXTENSION_NAME = "VK_GOOGLE_extension_193"
VK_GOOGLE_extension_194 = 1
VK_GOOGLE_EXTENSION_194_SPEC_VERSION = 0
VK_GOOGLE_EXTENSION_194_EXTENSION_NAME = "VK_GOOGLE_extension_194"
VK_GOOGLE_extension_195 = 1
VK_GOOGLE_EXTENSION_195_SPEC_VERSION = 0
VK_GOOGLE_EXTENSION_195_EXTENSION_NAME = "VK_GOOGLE_extension_195"
VK_GOOGLE_extension_196 = 1
VK_GOOGLE_EXTENSION_196_SPEC_VERSION = 0
VK_GOOGLE_EXTENSION_196_EXTENSION_NAME = "VK_GOOGLE_extension_196"
VK_KHR_driver_properties = 1
VK_KHR_DRIVER_PROPERTIES_SPEC_VERSION = 1
VK_KHR_DRIVER_PROPERTIES_EXTENSION_NAME = "VK_KHR_driver_properties"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DRIVER_PROPERTIES_KHR = 1000196000
VK_KHR_shader_float_controls = 1
VK_KHR_SHADER_FLOAT_CONTROLS_SPEC_VERSION = 1
VK_KHR_SHADER_FLOAT_CONTROLS_EXTENSION_NAME = "VK_KHR_shader_float_controls"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FLOAT_CONTROLS_PROPERTIES_KHR = 1000197000
VK_NV_shader_subgroup_partitioned = 1
VK_NV_SHADER_SUBGROUP_PARTITIONED_SPEC_VERSION = 1
VK_NV_SHADER_SUBGROUP_PARTITIONED_EXTENSION_NAME = "VK_NV_shader_subgroup_partitioned"
VK_SUBGROUP_FEATURE_PARTITIONED_BIT_NV = 0x00000100
VK_KHR_depth_stencil_resolve = 1
VK_KHR_DEPTH_STENCIL_RESOLVE_SPEC_VERSION = 1
VK_KHR_DEPTH_STENCIL_RESOLVE_EXTENSION_NAME = "VK_KHR_depth_stencil_resolve"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEPTH_STENCIL_RESOLVE_PROPERTIES_KHR = 1000199000
VK_STRUCTURE_TYPE_SUBPASS_DESCRIPTION_DEPTH_STENCIL_RESOLVE_KHR = 1000199001
VK_KHR_swapchain_mutable_format = 1
VK_KHR_SWAPCHAIN_MUTABLE_FORMAT_SPEC_VERSION = 1
VK_KHR_SWAPCHAIN_MUTABLE_FORMAT_EXTENSION_NAME = "VK_KHR_swapchain_mutable_format"
VK_SWAPCHAIN_CREATE_MUTABLE_FORMAT_BIT_KHR = 0x00000004
VK_NV_compute_shader_derivatives = 1
VK_NV_COMPUTE_SHADER_DERIVATIVES_SPEC_VERSION = 1
VK_NV_COMPUTE_SHADER_DERIVATIVES_EXTENSION_NAME = "VK_NV_compute_shader_derivatives"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COMPUTE_SHADER_DERIVATIVES_FEATURES_NV = 1000201000
VK_NV_mesh_shader = 1
VK_NV_MESH_SHADER_SPEC_VERSION = 1
VK_NV_MESH_SHADER_EXTENSION_NAME = "VK_NV_mesh_shader"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MESH_SHADER_FEATURES_NV = 1000202000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MESH_SHADER_PROPERTIES_NV = 1000202001
VK_SHADER_STAGE_TASK_BIT_NV = 0x00000040
VK_SHADER_STAGE_MESH_BIT_NV = 0x00000080
VK_PIPELINE_STAGE_TASK_SHADER_BIT_NV = 0x00080000
VK_PIPELINE_STAGE_MESH_SHADER_BIT_NV = 0x00100000
VK_NV_fragment_shader_barycentric = 1
VK_NV_FRAGMENT_SHADER_BARYCENTRIC_SPEC_VERSION = 1
VK_NV_FRAGMENT_SHADER_BARYCENTRIC_EXTENSION_NAME = "VK_NV_fragment_shader_barycentric"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADER_BARYCENTRIC_FEATURES_NV = 1000203000
VK_NV_shader_image_footprint = 1
VK_NV_SHADER_IMAGE_FOOTPRINT_SPEC_VERSION = 1
VK_NV_SHADER_IMAGE_FOOTPRINT_EXTENSION_NAME = "VK_NV_shader_image_footprint"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_IMAGE_FOOTPRINT_FEATURES_NV = 1000204000
VK_NV_scissor_exclusive = 1
VK_NV_SCISSOR_EXCLUSIVE_SPEC_VERSION = 1
VK_NV_SCISSOR_EXCLUSIVE_EXTENSION_NAME = "VK_NV_scissor_exclusive"
VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_EXCLUSIVE_SCISSOR_STATE_CREATE_INFO_NV = 1000205000
VK_DYNAMIC_STATE_EXCLUSIVE_SCISSOR_NV = 1000205001
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXCLUSIVE_SCISSOR_FEATURES_NV = 1000205002
VK_NV_device_diagnostic_checkpoints = 1
VK_NV_DEVICE_DIAGNOSTIC_CHECKPOINTS_SPEC_VERSION = 2
VK_NV_DEVICE_DIAGNOSTIC_CHECKPOINTS_EXTENSION_NAME = "VK_NV_device_diagnostic_checkpoints"
VK_STRUCTURE_TYPE_CHECKPOINT_DATA_NV = 1000206000
VK_STRUCTURE_TYPE_QUEUE_FAMILY_CHECKPOINT_PROPERTIES_NV = 1000206001
VK_KHR_extension_208 = 1
VK_KHR_EXTENSION_208_SPEC_VERSION = 0
VK_KHR_EXTENSION_208_EXTENSION_NAME = "VK_KHR_extension_208"
VK_KHR_extension_209 = 1
VK_KHR_EXTENSION_209_SPEC_VERSION = 0
VK_KHR_EXTENSION_209_EXTENSION_NAME = "VK_KHR_extension_209"
VK_INTEL_extension_210 = 1
VK_KHR_EXTENSION_210_SPEC_VERSION = 0
VK_KHR_EXTENSION_210_EXTENSION_NAME = "VK_KHR_extension_210"
VK_INTEL_extension_211 = 1
VK_KHR_EXTENSION_211_SPEC_VERSION = 0
VK_KHR_EXTENSION_211_EXTENSION_NAME = "VK_KHR_extension_211"
VK_KHR_vulkan_memory_model = 1
VK_KHR_VULKAN_MEMORY_MODEL_SPEC_VERSION = 3
VK_KHR_VULKAN_MEMORY_MODEL_EXTENSION_NAME = "VK_KHR_vulkan_memory_model"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_MEMORY_MODEL_FEATURES_KHR = 1000211000
VK_EXT_pci_bus_info = 1
VK_EXT_PCI_BUS_INFO_SPEC_VERSION = 2
VK_EXT_PCI_BUS_INFO_EXTENSION_NAME = "VK_EXT_pci_bus_info"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PCI_BUS_INFO_PROPERTIES_EXT = 1000212000
VK_AMD_extension_214 = 1
VK_KHR_EXTENSION_214_SPEC_VERSION = 0
VK_KHR_EXTENSION_214_EXTENSION_NAME = "VK_KHR_extension_214"
VK_FUCHSIA_imagepipe_surface = 1
VK_FUCHSIA_IMAGEPIPE_SURFACE_SPEC_VERSION = 1
VK_FUCHSIA_IMAGEPIPE_SURFACE_EXTENSION_NAME = "VK_FUCHSIA_imagepipe_surface"
VK_STRUCTURE_TYPE_IMAGEPIPE_SURFACE_CREATE_INFO_FUCHSIA = 1000214000
VK_GOOGLE_extension_216 = 1
VK_KHR_EXTENSION_216_SPEC_VERSION = 0
VK_KHR_EXTENSION_216_EXTENSION_NAME = "VK_KHR_extension_216"
VK_GOOGLE_extension_217 = 1
VK_KHR_EXTENSION_217_SPEC_VERSION = 0
VK_KHR_EXTENSION_217_EXTENSION_NAME = "VK_KHR_extension_217"
VK_EXT_macos_ios_window = 1
VK_EXT_MACOS_IOS_WINDOW_SPEC_VERSION = 0
VK_EXT_MACOS_IOS_WINDOW_EXTENSION_NAME = "VK_EXT_macos_ios_window"
VK_EXT_fragment_density_map = 1
VK_EXT_FRAGMENT_DENSITY_MAP_SPEC_VERSION = 1
VK_EXT_FRAGMENT_DENSITY_MAP_EXTENSION_NAME = "VK_EXT_fragment_density_map"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_FEATURES_EXT = 1000218000
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_PROPERTIES_EXT = 1000218001
VK_STRUCTURE_TYPE_RENDER_PASS_FRAGMENT_DENSITY_MAP_CREATE_INFO_EXT = 1000218002
VK_IMAGE_CREATE_SUBSAMPLED_BIT_EXT = 0x00004000
VK_IMAGE_LAYOUT_FRAGMENT_DENSITY_MAP_OPTIMAL_EXT = 1000218000
VK_ACCESS_FRAGMENT_DENSITY_MAP_READ_BIT_EXT = 0x01000000
VK_FORMAT_FEATURE_FRAGMENT_DENSITY_MAP_BIT_EXT = 0x01000000
VK_IMAGE_USAGE_FRAGMENT_DENSITY_MAP_BIT_EXT = 0x00000200
VK_IMAGE_VIEW_CREATE_FRAGMENT_DENSITY_MAP_DYNAMIC_BIT_EXT = 0x00000001
VK_PIPELINE_STAGE_FRAGMENT_DENSITY_PROCESS_BIT_EXT = 0x00800000
VK_SAMPLER_CREATE_SUBSAMPLED_BIT_EXT = 0x00000001
VK_SAMPLER_CREATE_SUBSAMPLED_COARSE_RECONSTRUCTION_BIT_EXT = 0x00000002
VK_EXT_extension_220 = 1
VK_EXT_EXTENSION_220_SPEC_VERSION = 0
VK_EXT_EXTENSION_220_EXTENSION_NAME = "VK_EXT_extension_220"
VK_KHR_extension_221 = 1
VK_KHR_EXTENSION_221_SPEC_VERSION = 0
VK_KHR_EXTENSION_221_EXTENSION_NAME = "VK_KHR_extension_221"
VK_RENDER_PASS_CREATE_RESERVED_0_BIT_KHR = 0x00000001
VK_EXT_scalar_block_layout = 1
VK_EXT_SCALAR_BLOCK_LAYOUT_SPEC_VERSION = 1
VK_EXT_SCALAR_BLOCK_LAYOUT_EXTENSION_NAME = "VK_EXT_scalar_block_layout"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SCALAR_BLOCK_LAYOUT_FEATURES_EXT = 1000221000
VK_EXT_extension_223 = 1
VK_EXT_EXTENSION_223_SPEC_VERSION = 0
VK_EXT_EXTENSION_223_EXTENSION_NAME = "VK_EXT_extension_223"
VK_GOOGLE_hlsl_functionality1 = 1
VK_GOOGLE_HLSL_FUNCTIONALITY1_SPEC_VERSION = 1
VK_GOOGLE_HLSL_FUNCTIONALITY1_EXTENSION_NAME = "VK_GOOGLE_hlsl_functionality1"
VK_GOOGLE_decorate_string = 1
VK_GOOGLE_DECORATE_STRING_SPEC_VERSION = 1
VK_GOOGLE_DECORATE_STRING_EXTENSION_NAME = "VK_GOOGLE_decorate_string"
VK_AMD_extension_226 = 1
VK_AMD_EXTENSION_226_SPEC_VERSION = 0
VK_AMD_EXTENSION_226_EXTENSION_NAME = "VK_AMD_extension_226"
VK_AMD_extension_227 = 1
VK_AMD_EXTENSION_227_SPEC_VERSION = 0
VK_AMD_EXTENSION_227_EXTENSION_NAME = "VK_AMD_extension_227"
VK_AMD_extension_228 = 1
VK_AMD_EXTENSION_228_SPEC_VERSION = 0
VK_AMD_EXTENSION_228_EXTENSION_NAME = "VK_AMD_extension_228"
VK_AMD_extension_229 = 1
VK_AMD_EXTENSION_229_SPEC_VERSION = 0
VK_AMD_EXTENSION_229_EXTENSION_NAME = "VK_AMD_extension_229"
VK_AMD_extension_230 = 1
VK_AMD_EXTENSION_230_SPEC_VERSION = 0
VK_AMD_EXTENSION_230_EXTENSION_NAME = "VK_AMD_extension_230"
VK_AMD_extension_231 = 1
VK_AMD_EXTENSION_231_SPEC_VERSION = 0
VK_AMD_EXTENSION_231_EXTENSION_NAME = "VK_AMD_extension_231"
VK_AMD_extension_232 = 1
VK_AMD_EXTENSION_232_SPEC_VERSION = 0
VK_AMD_EXTENSION_232_EXTENSION_NAME = "VK_AMD_extension_232"
VK_AMD_extension_233 = 1
VK_AMD_EXTENSION_233_SPEC_VERSION = 0
VK_AMD_EXTENSION_233_EXTENSION_NAME = "VK_AMD_extension_233"
VK_AMD_extension_234 = 1
VK_AMD_EXTENSION_234_SPEC_VERSION = 0
VK_AMD_EXTENSION_234_EXTENSION_NAME = "VK_AMD_extension_234"
VK_AMD_extension_235 = 1
VK_AMD_EXTENSION_235_SPEC_VERSION = 0
VK_AMD_EXTENSION_235_EXTENSION_NAME = "VK_AMD_extension_235"
VK_AMD_extension_236 = 1
VK_AMD_EXTENSION_236_SPEC_VERSION = 0
VK_AMD_EXTENSION_236_EXTENSION_NAME = "VK_AMD_extension_236"
VK_KHR_extension_237 = 1
VK_KHR_EXTENSION_237_SPEC_VERSION = 0
VK_KHR_EXTENSION_237_EXTENSION_NAME = "VK_KHR_extension_237"
VK_EXT_memory_budget = 1
VK_EXT_MEMORY_BUDGET_SPEC_VERSION = 1
VK_EXT_MEMORY_BUDGET_EXTENSION_NAME = "VK_EXT_memory_budget"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_BUDGET_PROPERTIES_EXT = 1000237000
VK_EXT_memory_priority = 1
VK_EXT_MEMORY_PRIORITY_SPEC_VERSION = 1
VK_EXT_MEMORY_PRIORITY_EXTENSION_NAME = "VK_EXT_memory_priority"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_PRIORITY_FEATURES_EXT = 1000238000
VK_STRUCTURE_TYPE_MEMORY_PRIORITY_ALLOCATE_INFO_EXT = 1000238001
VK_KHR_extension_240 = 1
VK_KHR_EXTENSION_240_SPEC_VERSION = 0
VK_KHR_EXTENSION_240_EXTENSION_NAME = "VK_KHR_extension_240"
VK_NV_dedicated_allocation_image_aliasing = 1
VK_NV_DEDICATED_ALLOCATION_IMAGE_ALIASING_SPEC_VERSION = 1
VK_NV_DEDICATED_ALLOCATION_IMAGE_ALIASING_EXTENSION_NAME = "VK_NV_dedicated_allocation_image_aliasing"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEDICATED_ALLOCATION_IMAGE_ALIASING_FEATURES_NV = 1000240000
VK_NV_extension_242 = 1
VK_NV_EXTENSION_242_SPEC_VERSION = 0
VK_NV_EXTENSION_242_EXTENSION_NAME = "VK_NV_extension_242"
VK_INTEL_extension_243 = 1
VK_INTEL_EXTENSION_243_SPEC_VERSION = 0
VK_INTEL_EXTENSION_243_EXTENSION_NAME = "VK_INTEL_extension_243"
VK_MESA_extension_244 = 1
VK_MESA_EXTENSION_244_SPEC_VERSION = 0
VK_MESA_EXTENSION_244_EXTENSION_NAME = "VK_MESA_extension_244"
VK_EXT_buffer_device_address = 1
VK_EXT_BUFFER_DEVICE_ADDRESS_SPEC_VERSION = 2
VK_EXT_BUFFER_DEVICE_ADDRESS_EXTENSION_NAME = "VK_EXT_buffer_device_address"
VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BUFFER_ADDRESS_FEATURES_EXT = 1000244000
VK_STRUCTURE_TYPE_BUFFER_DEVICE_ADDRESS_INFO_EXT = 1000244001
VK_STRUCTURE_TYPE_BUFFER_DEVICE_ADDRESS_CREATE_INFO_EXT = 1000244002
VK_BUFFER_USAGE_SHADER_DEVICE_ADDRESS_BIT_EXT = 0x00020000
VK_BUFFER_CREATE_DEVICE_ADDRESS_CAPTURE_REPLAY_BIT_EXT = 0x00000010
VK_EXT_extension_246 = 1
VK_EXT_EXTENSION_246_SPEC_VERSION = 0
VK_EXT_EXTENSION_246_EXTENSION_NAME = "VK_EXT_extension_246"
VK_EXT_separate_stencil_usage = 1
VK_EXT_SEPARATE_STENCIL_USAGE_SPEC_VERSION = 1
VK_EXT_SEPARATE_STENCIL_USAGE_EXTENSION_NAME = "VK_EXT_separate_stencil_usage"
VK_STRUCTURE_TYPE_IMAGE_STENCIL_USAGE_CREATE_INFO_EXT = 1000246000
VK_EXT_validation_features = 1
VK_EXT_VALIDATION_FEATURES_SPEC_VERSION = 1
VK_EXT_VALIDATION_FEATURES_EXTENSION_NAME = "VK_EXT_validation_features"
VK_STRUCTURE_TYPE_VALIDATION_FEATURES_EXT = 1000247000
VK_KHR_extension_249 = 1
VK_KHR_EXTENSION_249_SPEC_VERSION = 0
VK_KHR_EXTENSION_249_EXTENSION_NAME = "VK_KHR_extension_249"
VK_NV_extension_250 = 1
VK_NV_EXTENSION_250_SPEC_VERSION = 0
VK_NV_EXTENSION_250_EXTENSION_NAME = "VK_NV_extension_250"
VK_NV_extension_251 = 1
VK_NV_EXTENSION_251_SPEC_VERSION = 0
VK_NV_EXTENSION_251_EXTENSION_NAME = "VK_NV_extension_251"
VK_EXT_extension_252 = 1
VK_NV_EXTENSION_252_SPEC_VERSION = 0
VK_NV_EXTENSION_252_EXTENSION_NAME = "VK_NV_extension_252"
VK_EXT_extension_253 = 1
VK_NV_EXTENSION_253_SPEC_VERSION = 0
VK_NV_EXTENSION_253_EXTENSION_NAME = "VK_NV_extension_253"
VK_EXT_extension_254 = 1
VK_EXT_EXTENSION_254_SPEC_VERSION = 1
VK_EXT_EXTENSION_254_EXTENSION_NAME = "VK_EXT_extension_254"
VK_EXT_extension_255 = 1
VK_EXT_EXTENSION_255_SPEC_VERSION = 0
VK_EXT_EXTENSION_255_EXTENSION_NAME = "VK_EXT_extension_255"
VK_EXT_extension_256 = 1
VK_EXT_EXTENSION_256_SPEC_VERSION = 0
VK_EXT_EXTENSION_256_EXTENSION_NAME = "VK_EXT_extension_256"




class VkException(Exception):
    pass


class VkError(Exception):
    pass


class VkNotReady(VkException):
    pass
class VkTimeout(VkException):
    pass
class VkEventSet(VkException):
    pass
class VkEventReset(VkException):
    pass
class VkIncomplete(VkException):
    pass
class VkSuboptimalKhr(VkException):
    pass

class VkErrorOutOfHostMemory(VkError):
    pass
class VkErrorOutOfDeviceMemory(VkError):
    pass
class VkErrorInitializationFailed(VkError):
    pass
class VkErrorDeviceLost(VkError):
    pass
class VkErrorMemoryMapFailed(VkError):
    pass
class VkErrorLayerNotPresent(VkError):
    pass
class VkErrorExtensionNotPresent(VkError):
    pass
class VkErrorFeatureNotPresent(VkError):
    pass
class VkErrorIncompatibleDriver(VkError):
    pass
class VkErrorTooManyObjects(VkError):
    pass
class VkErrorFormatNotSupported(VkError):
    pass
class VkErrorFragmentedPool(VkError):
    pass
class VkErrorOutOfPoolMemory(VkError):
    pass
class VkErrorInvalidExternalHandle(VkError):
    pass
class VkErrorSurfaceLostKhr(VkError):
    pass
class VkErrorNativeWindowInUseKhr(VkError):
    pass
class VkErrorOutOfDateKhr(VkError):
    pass
class VkErrorIncompatibleDisplayKhr(VkError):
    pass
class VkErrorInvalidShaderNv(VkError):
    pass
class VkErrorInvalidExternalHandleKhr(VkError):
    pass
class VkErrorFragmentationExt(VkError):
    pass
class VkErrorInvalidDeviceAddressExt(VkError):
    pass

exception_codes = {
    1:VkNotReady,
    2:VkTimeout,
    3:VkEventSet,
    4:VkEventReset,
    5:VkIncomplete,
    1000001003:VkSuboptimalKhr,

    -1:VkErrorOutOfHostMemory,
    -2:VkErrorOutOfDeviceMemory,
    -3:VkErrorInitializationFailed,
    -4:VkErrorDeviceLost,
    -5:VkErrorMemoryMapFailed,
    -6:VkErrorLayerNotPresent,
    -7:VkErrorExtensionNotPresent,
    -8:VkErrorFeatureNotPresent,
    -9:VkErrorIncompatibleDriver,
    -10:VkErrorTooManyObjects,
    -11:VkErrorFormatNotSupported,
    -12:VkErrorFragmentedPool,
    1000069000:VkErrorOutOfPoolMemory,
    1000072003:VkErrorInvalidExternalHandle,
    -1000000000:VkErrorSurfaceLostKhr,
    -1000000001:VkErrorNativeWindowInUseKhr,
    -1000001004:VkErrorOutOfDateKhr,
    -1000003001:VkErrorIncompatibleDisplayKhr,
    -1000012000:VkErrorInvalidShaderNv,
    None:VkErrorInvalidExternalHandleKhr,
    -1000161000:VkErrorFragmentationExt,
    -1000244000:VkErrorInvalidDeviceAddressExt,
}

_internal_vkInternalAllocationNotification = None

@ffi.callback('PFN_vkInternalAllocationNotification')
def _external_vkInternalAllocationNotification(*args):
    return _internal_vkInternalAllocationNotification(*[_cstr(x) for x in args])
_internal_vkInternalFreeNotification = None

@ffi.callback('PFN_vkInternalFreeNotification')
def _external_vkInternalFreeNotification(*args):
    return _internal_vkInternalFreeNotification(*[_cstr(x) for x in args])
_internal_vkReallocationFunction = None

@ffi.callback('PFN_vkReallocationFunction')
def _external_vkReallocationFunction(*args):
    return _internal_vkReallocationFunction(*[_cstr(x) for x in args])
_internal_vkAllocationFunction = None

@ffi.callback('PFN_vkAllocationFunction')
def _external_vkAllocationFunction(*args):
    return _internal_vkAllocationFunction(*[_cstr(x) for x in args])
_internal_vkFreeFunction = None

@ffi.callback('PFN_vkFreeFunction')
def _external_vkFreeFunction(*args):
    return _internal_vkFreeFunction(*[_cstr(x) for x in args])
_internal_vkVoidFunction = None

@ffi.callback('PFN_vkVoidFunction')
def _external_vkVoidFunction(*args):
    return _internal_vkVoidFunction(*[_cstr(x) for x in args])
_internal_vkDebugReportCallbackEXT = None

@ffi.callback('PFN_vkDebugReportCallbackEXT')
def _external_vkDebugReportCallbackEXT(*args):
    return _internal_vkDebugReportCallbackEXT(*[_cstr(x) for x in args])
_internal_vkDebugUtilsMessengerCallbackEXT = None

@ffi.callback('PFN_vkDebugUtilsMessengerCallbackEXT')
def _external_vkDebugUtilsMessengerCallbackEXT(*args):
    return _internal_vkDebugUtilsMessengerCallbackEXT(*[_cstr(x) for x in args])


def _get_pfn_name(struct_name):
    if struct_name == 'VkAllocationCallbacks':
        return 'vkInternalAllocationNotification'
    if struct_name == 'VkAllocationCallbacks':
        return 'vkInternalFreeNotification'
    if struct_name == 'VkAllocationCallbacks':
        return 'vkReallocationFunction'
    if struct_name == 'VkAllocationCallbacks':
        return 'vkAllocationFunction'
    if struct_name == 'VkAllocationCallbacks':
        return 'vkFreeFunction'
    if struct_name == 'VkAllocationCallbacks':
        return 'vkVoidFunction'
    if struct_name == 'VkDebugReportCallbackCreateInfoEXT':
        return 'vkDebugReportCallbackEXT'
    if struct_name == 'VkDebugUtilsMessengerCreateInfoEXT':
        return 'vkDebugUtilsMessengerCallbackEXT'


def _new(ctype, **kwargs):
    _type = ffi.typeof(ctype)

    # keep only valued kwargs
    kwargs = {k: kwargs[k] for k in kwargs if kwargs[k]}

    # cast pointer
    ptrs = {}
    pfns = {}
    pcs = {}
    for k, v in kwargs.items():
        # convert tuple pair to dict
        ktype = dict(_type.fields)[k].type

        if k == 'pCode':
            pcs[k] = ffi.cast('uint32_t*', ffi.from_buffer(v))
        elif k.startswith('pfn'):
            pfn_name = _get_pfn_name(ctype)
            mod = sys.modules[__name__]
            setattr(mod, '_internal_' + pfn_name, v)
            pfns[k] = getattr(mod, '_external_' + pfn_name)
        elif ktype.kind == 'pointer':
            ptrs[k] = _cast_ptr(v, ktype)

    # init object
    init = dict(kwargs,  **{k: v for k, (v, _) in ptrs.items()})
    init.update(pfns)
    init.update(pcs)

    ret = ffi.new(_type.cname + '*', init)[0]

    # reference created pointer in the object
    _weakkey_dict[ret] = [v for _, v in ptrs.values() if v != ffi.NULL]
    if pcs:
        _weakkey_dict[ret].extend([x for x in pcs.values()])

    return ret



def VkBaseOutStructure(sType=None,pNext=None,):

    return _new('VkBaseOutStructure', sType=sType,pNext=pNext)

def VkBaseInStructure(sType=None,pNext=None,):

    return _new('VkBaseInStructure', sType=sType,pNext=pNext)

def VkOffset2D(x=None,y=None,):

    return _new('VkOffset2D', x=x,y=y)

def VkOffset3D(x=None,y=None,z=None,):

    return _new('VkOffset3D', x=x,y=y,z=z)

def VkExtent2D(width=None,height=None,):

    return _new('VkExtent2D', width=width,height=height)

def VkExtent3D(width=None,height=None,depth=None,):

    return _new('VkExtent3D', width=width,height=height,depth=depth)

def VkViewport(x=None,y=None,width=None,height=None,minDepth=None,maxDepth=None,):

    return _new('VkViewport', x=x,y=y,width=width,height=height,minDepth=minDepth,maxDepth=maxDepth)

def VkRect2D(offset=None,extent=None,):

    return _new('VkRect2D', offset=offset,extent=extent)

def VkClearRect(rect=None,baseArrayLayer=None,layerCount=None,):

    return _new('VkClearRect', rect=rect,baseArrayLayer=baseArrayLayer,layerCount=layerCount)

def VkComponentMapping(r=None,g=None,b=None,a=None,):

    return _new('VkComponentMapping', r=r,g=g,b=b,a=a)

def VkPhysicalDeviceProperties(apiVersion=None,driverVersion=None,vendorID=None,deviceID=None,deviceType=None,deviceName=None,pipelineCacheUUID=None,limits=None,sparseProperties=None,):

    return _new('VkPhysicalDeviceProperties', apiVersion=apiVersion,driverVersion=driverVersion,vendorID=vendorID,deviceID=deviceID,deviceType=deviceType,deviceName=deviceName,pipelineCacheUUID=pipelineCacheUUID,limits=limits,sparseProperties=sparseProperties)

def VkExtensionProperties(extensionName=None,specVersion=None,):

    return _new('VkExtensionProperties', extensionName=extensionName,specVersion=specVersion)

def VkLayerProperties(layerName=None,specVersion=None,implementationVersion=None,description=None,):

    return _new('VkLayerProperties', layerName=layerName,specVersion=specVersion,implementationVersion=implementationVersion,description=description)

def VkApplicationInfo(sType=VK_STRUCTURE_TYPE_APPLICATION_INFO,pNext=None,pApplicationName=None,applicationVersion=None,pEngineName=None,engineVersion=None,apiVersion=None,):

    return _new('VkApplicationInfo', sType=sType,pNext=pNext,pApplicationName=pApplicationName,applicationVersion=applicationVersion,pEngineName=pEngineName,engineVersion=engineVersion,apiVersion=apiVersion)

def VkAllocationCallbacks(pUserData=None,pfnAllocation=None,pfnReallocation=None,pfnFree=None,pfnInternalAllocation=None,pfnInternalFree=None,):

    return _new('VkAllocationCallbacks', pUserData=pUserData,pfnAllocation=pfnAllocation,pfnReallocation=pfnReallocation,pfnFree=pfnFree,pfnInternalAllocation=pfnInternalAllocation,pfnInternalFree=pfnInternalFree)

def VkDeviceQueueCreateInfo(sType=VK_STRUCTURE_TYPE_DEVICE_QUEUE_CREATE_INFO,pNext=None,flags=None,queueFamilyIndex=None,queueCount=None,pQueuePriorities=None,):
    if queueCount is None and pQueuePriorities is not None:
        queueCount = len(pQueuePriorities)

    return _new('VkDeviceQueueCreateInfo', sType=sType,pNext=pNext,flags=flags,queueFamilyIndex=queueFamilyIndex,queueCount=queueCount,pQueuePriorities=pQueuePriorities)

def VkDeviceCreateInfo(sType=VK_STRUCTURE_TYPE_DEVICE_CREATE_INFO,pNext=None,flags=None,queueCreateInfoCount=None,pQueueCreateInfos=None,enabledLayerCount=None,ppEnabledLayerNames=None,enabledExtensionCount=None,ppEnabledExtensionNames=None,pEnabledFeatures=None,):
    if queueCreateInfoCount is None and pQueueCreateInfos is not None:
        queueCreateInfoCount = len(pQueueCreateInfos)
    if enabledLayerCount is None and ppEnabledLayerNames is not None:
        enabledLayerCount = len(ppEnabledLayerNames)
    if enabledExtensionCount is None and ppEnabledExtensionNames is not None:
        enabledExtensionCount = len(ppEnabledExtensionNames)

    return _new('VkDeviceCreateInfo', sType=sType,pNext=pNext,flags=flags,queueCreateInfoCount=queueCreateInfoCount,pQueueCreateInfos=pQueueCreateInfos,enabledLayerCount=enabledLayerCount,ppEnabledLayerNames=ppEnabledLayerNames,enabledExtensionCount=enabledExtensionCount,ppEnabledExtensionNames=ppEnabledExtensionNames,pEnabledFeatures=pEnabledFeatures)

def VkInstanceCreateInfo(sType=VK_STRUCTURE_TYPE_INSTANCE_CREATE_INFO,pNext=None,flags=None,pApplicationInfo=None,enabledLayerCount=None,ppEnabledLayerNames=None,enabledExtensionCount=None,ppEnabledExtensionNames=None,):
    if enabledLayerCount is None and ppEnabledLayerNames is not None:
        enabledLayerCount = len(ppEnabledLayerNames)
    if enabledExtensionCount is None and ppEnabledExtensionNames is not None:
        enabledExtensionCount = len(ppEnabledExtensionNames)

    return _new('VkInstanceCreateInfo', sType=sType,pNext=pNext,flags=flags,pApplicationInfo=pApplicationInfo,enabledLayerCount=enabledLayerCount,ppEnabledLayerNames=ppEnabledLayerNames,enabledExtensionCount=enabledExtensionCount,ppEnabledExtensionNames=ppEnabledExtensionNames)

def VkQueueFamilyProperties(queueFlags=None,queueCount=None,timestampValidBits=None,minImageTransferGranularity=None,):

    return _new('VkQueueFamilyProperties', queueFlags=queueFlags,queueCount=queueCount,timestampValidBits=timestampValidBits,minImageTransferGranularity=minImageTransferGranularity)

def VkPhysicalDeviceMemoryProperties(memoryTypeCount=None,memoryTypes=None,memoryHeapCount=None,memoryHeaps=None,):

    return _new('VkPhysicalDeviceMemoryProperties', memoryTypeCount=memoryTypeCount,memoryTypes=memoryTypes,memoryHeapCount=memoryHeapCount,memoryHeaps=memoryHeaps)

def VkMemoryAllocateInfo(sType=VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_INFO,pNext=None,allocationSize=None,memoryTypeIndex=None,):

    return _new('VkMemoryAllocateInfo', sType=sType,pNext=pNext,allocationSize=allocationSize,memoryTypeIndex=memoryTypeIndex)

def VkMemoryRequirements(size=None,alignment=None,memoryTypeBits=None,):

    return _new('VkMemoryRequirements', size=size,alignment=alignment,memoryTypeBits=memoryTypeBits)

def VkSparseImageFormatProperties(aspectMask=None,imageGranularity=None,flags=None,):

    return _new('VkSparseImageFormatProperties', aspectMask=aspectMask,imageGranularity=imageGranularity,flags=flags)

def VkSparseImageMemoryRequirements(formatProperties=None,imageMipTailFirstLod=None,imageMipTailSize=None,imageMipTailOffset=None,imageMipTailStride=None,):

    return _new('VkSparseImageMemoryRequirements', formatProperties=formatProperties,imageMipTailFirstLod=imageMipTailFirstLod,imageMipTailSize=imageMipTailSize,imageMipTailOffset=imageMipTailOffset,imageMipTailStride=imageMipTailStride)

def VkMemoryType(propertyFlags=None,heapIndex=None,):

    return _new('VkMemoryType', propertyFlags=propertyFlags,heapIndex=heapIndex)

def VkMemoryHeap(size=None,flags=None,):

    return _new('VkMemoryHeap', size=size,flags=flags)

def VkMappedMemoryRange(sType=VK_STRUCTURE_TYPE_MAPPED_MEMORY_RANGE,pNext=None,memory=None,offset=None,size=None,):

    return _new('VkMappedMemoryRange', sType=sType,pNext=pNext,memory=memory,offset=offset,size=size)

def VkFormatProperties(linearTilingFeatures=None,optimalTilingFeatures=None,bufferFeatures=None,):

    return _new('VkFormatProperties', linearTilingFeatures=linearTilingFeatures,optimalTilingFeatures=optimalTilingFeatures,bufferFeatures=bufferFeatures)

def VkImageFormatProperties(maxExtent=None,maxMipLevels=None,maxArrayLayers=None,sampleCounts=None,maxResourceSize=None,):

    return _new('VkImageFormatProperties', maxExtent=maxExtent,maxMipLevels=maxMipLevels,maxArrayLayers=maxArrayLayers,sampleCounts=sampleCounts,maxResourceSize=maxResourceSize)

def VkDescriptorBufferInfo(buffer=None,offset=None,range=None,):

    return _new('VkDescriptorBufferInfo', buffer=buffer,offset=offset,range=range)

def VkDescriptorImageInfo(sampler=None,imageView=None,imageLayout=None,):

    return _new('VkDescriptorImageInfo', sampler=sampler,imageView=imageView,imageLayout=imageLayout)

def VkWriteDescriptorSet(sType=VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET,pNext=None,dstSet=None,dstBinding=None,dstArrayElement=None,descriptorCount=None,descriptorType=None,pImageInfo=None,pBufferInfo=None,pTexelBufferView=None,):
    if descriptorCount is None and pImageInfo is not None:
        descriptorCount = len(pImageInfo)
    if descriptorCount is None and pBufferInfo is not None:
        descriptorCount = len(pBufferInfo)
    if descriptorCount is None and pTexelBufferView is not None:
        descriptorCount = len(pTexelBufferView)

    return _new('VkWriteDescriptorSet', sType=sType,pNext=pNext,dstSet=dstSet,dstBinding=dstBinding,dstArrayElement=dstArrayElement,descriptorCount=descriptorCount,descriptorType=descriptorType,pImageInfo=pImageInfo,pBufferInfo=pBufferInfo,pTexelBufferView=pTexelBufferView)

def VkCopyDescriptorSet(sType=VK_STRUCTURE_TYPE_COPY_DESCRIPTOR_SET,pNext=None,srcSet=None,srcBinding=None,srcArrayElement=None,dstSet=None,dstBinding=None,dstArrayElement=None,descriptorCount=None,):

    return _new('VkCopyDescriptorSet', sType=sType,pNext=pNext,srcSet=srcSet,srcBinding=srcBinding,srcArrayElement=srcArrayElement,dstSet=dstSet,dstBinding=dstBinding,dstArrayElement=dstArrayElement,descriptorCount=descriptorCount)

def VkBufferCreateInfo(sType=VK_STRUCTURE_TYPE_BUFFER_CREATE_INFO,pNext=None,flags=None,size=None,usage=None,sharingMode=None,queueFamilyIndexCount=None,pQueueFamilyIndices=None,):
    if queueFamilyIndexCount is None and pQueueFamilyIndices is not None:
        queueFamilyIndexCount = len(pQueueFamilyIndices)

    return _new('VkBufferCreateInfo', sType=sType,pNext=pNext,flags=flags,size=size,usage=usage,sharingMode=sharingMode,queueFamilyIndexCount=queueFamilyIndexCount,pQueueFamilyIndices=pQueueFamilyIndices)

def VkBufferViewCreateInfo(sType=VK_STRUCTURE_TYPE_BUFFER_VIEW_CREATE_INFO,pNext=None,flags=None,buffer=None,format=None,offset=None,range=None,):

    return _new('VkBufferViewCreateInfo', sType=sType,pNext=pNext,flags=flags,buffer=buffer,format=format,offset=offset,range=range)

def VkImageSubresource(aspectMask=None,mipLevel=None,arrayLayer=None,):

    return _new('VkImageSubresource', aspectMask=aspectMask,mipLevel=mipLevel,arrayLayer=arrayLayer)

def VkImageSubresourceLayers(aspectMask=None,mipLevel=None,baseArrayLayer=None,layerCount=None,):

    return _new('VkImageSubresourceLayers', aspectMask=aspectMask,mipLevel=mipLevel,baseArrayLayer=baseArrayLayer,layerCount=layerCount)

def VkImageSubresourceRange(aspectMask=None,baseMipLevel=None,levelCount=None,baseArrayLayer=None,layerCount=None,):

    return _new('VkImageSubresourceRange', aspectMask=aspectMask,baseMipLevel=baseMipLevel,levelCount=levelCount,baseArrayLayer=baseArrayLayer,layerCount=layerCount)

def VkMemoryBarrier(sType=VK_STRUCTURE_TYPE_MEMORY_BARRIER,pNext=None,srcAccessMask=None,dstAccessMask=None,):

    return _new('VkMemoryBarrier', sType=sType,pNext=pNext,srcAccessMask=srcAccessMask,dstAccessMask=dstAccessMask)

def VkBufferMemoryBarrier(sType=VK_STRUCTURE_TYPE_BUFFER_MEMORY_BARRIER,pNext=None,srcAccessMask=None,dstAccessMask=None,srcQueueFamilyIndex=None,dstQueueFamilyIndex=None,buffer=None,offset=None,size=None,):

    return _new('VkBufferMemoryBarrier', sType=sType,pNext=pNext,srcAccessMask=srcAccessMask,dstAccessMask=dstAccessMask,srcQueueFamilyIndex=srcQueueFamilyIndex,dstQueueFamilyIndex=dstQueueFamilyIndex,buffer=buffer,offset=offset,size=size)

def VkImageMemoryBarrier(sType=VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER,pNext=None,srcAccessMask=None,dstAccessMask=None,oldLayout=None,newLayout=None,srcQueueFamilyIndex=None,dstQueueFamilyIndex=None,image=None,subresourceRange=None,):

    return _new('VkImageMemoryBarrier', sType=sType,pNext=pNext,srcAccessMask=srcAccessMask,dstAccessMask=dstAccessMask,oldLayout=oldLayout,newLayout=newLayout,srcQueueFamilyIndex=srcQueueFamilyIndex,dstQueueFamilyIndex=dstQueueFamilyIndex,image=image,subresourceRange=subresourceRange)

def VkImageCreateInfo(sType=VK_STRUCTURE_TYPE_IMAGE_CREATE_INFO,pNext=None,flags=None,imageType=None,format=None,extent=None,mipLevels=None,arrayLayers=None,samples=None,tiling=None,usage=None,sharingMode=None,queueFamilyIndexCount=None,pQueueFamilyIndices=None,initialLayout=None,):
    if queueFamilyIndexCount is None and pQueueFamilyIndices is not None:
        queueFamilyIndexCount = len(pQueueFamilyIndices)

    return _new('VkImageCreateInfo', sType=sType,pNext=pNext,flags=flags,imageType=imageType,format=format,extent=extent,mipLevels=mipLevels,arrayLayers=arrayLayers,samples=samples,tiling=tiling,usage=usage,sharingMode=sharingMode,queueFamilyIndexCount=queueFamilyIndexCount,pQueueFamilyIndices=pQueueFamilyIndices,initialLayout=initialLayout)

def VkSubresourceLayout(offset=None,size=None,rowPitch=None,arrayPitch=None,depthPitch=None,):

    return _new('VkSubresourceLayout', offset=offset,size=size,rowPitch=rowPitch,arrayPitch=arrayPitch,depthPitch=depthPitch)

def VkImageViewCreateInfo(sType=VK_STRUCTURE_TYPE_IMAGE_VIEW_CREATE_INFO,pNext=None,flags=None,image=None,viewType=None,format=None,components=None,subresourceRange=None,):

    return _new('VkImageViewCreateInfo', sType=sType,pNext=pNext,flags=flags,image=image,viewType=viewType,format=format,components=components,subresourceRange=subresourceRange)

def VkBufferCopy(srcOffset=None,dstOffset=None,size=None,):

    return _new('VkBufferCopy', srcOffset=srcOffset,dstOffset=dstOffset,size=size)

def VkSparseMemoryBind(resourceOffset=None,size=None,memory=None,memoryOffset=None,flags=None,):

    return _new('VkSparseMemoryBind', resourceOffset=resourceOffset,size=size,memory=memory,memoryOffset=memoryOffset,flags=flags)

def VkSparseImageMemoryBind(subresource=None,offset=None,extent=None,memory=None,memoryOffset=None,flags=None,):

    return _new('VkSparseImageMemoryBind', subresource=subresource,offset=offset,extent=extent,memory=memory,memoryOffset=memoryOffset,flags=flags)

def VkSparseBufferMemoryBindInfo(buffer=None,bindCount=None,pBinds=None,):
    if bindCount is None and pBinds is not None:
        bindCount = len(pBinds)

    return _new('VkSparseBufferMemoryBindInfo', buffer=buffer,bindCount=bindCount,pBinds=pBinds)

def VkSparseImageOpaqueMemoryBindInfo(image=None,bindCount=None,pBinds=None,):
    if bindCount is None and pBinds is not None:
        bindCount = len(pBinds)

    return _new('VkSparseImageOpaqueMemoryBindInfo', image=image,bindCount=bindCount,pBinds=pBinds)

def VkSparseImageMemoryBindInfo(image=None,bindCount=None,pBinds=None,):
    if bindCount is None and pBinds is not None:
        bindCount = len(pBinds)

    return _new('VkSparseImageMemoryBindInfo', image=image,bindCount=bindCount,pBinds=pBinds)

def VkBindSparseInfo(sType=VK_STRUCTURE_TYPE_BIND_SPARSE_INFO,pNext=None,waitSemaphoreCount=None,pWaitSemaphores=None,bufferBindCount=None,pBufferBinds=None,imageOpaqueBindCount=None,pImageOpaqueBinds=None,imageBindCount=None,pImageBinds=None,signalSemaphoreCount=None,pSignalSemaphores=None,):
    if waitSemaphoreCount is None and pWaitSemaphores is not None:
        waitSemaphoreCount = len(pWaitSemaphores)
    if bufferBindCount is None and pBufferBinds is not None:
        bufferBindCount = len(pBufferBinds)
    if imageOpaqueBindCount is None and pImageOpaqueBinds is not None:
        imageOpaqueBindCount = len(pImageOpaqueBinds)
    if imageBindCount is None and pImageBinds is not None:
        imageBindCount = len(pImageBinds)
    if signalSemaphoreCount is None and pSignalSemaphores is not None:
        signalSemaphoreCount = len(pSignalSemaphores)

    return _new('VkBindSparseInfo', sType=sType,pNext=pNext,waitSemaphoreCount=waitSemaphoreCount,pWaitSemaphores=pWaitSemaphores,bufferBindCount=bufferBindCount,pBufferBinds=pBufferBinds,imageOpaqueBindCount=imageOpaqueBindCount,pImageOpaqueBinds=pImageOpaqueBinds,imageBindCount=imageBindCount,pImageBinds=pImageBinds,signalSemaphoreCount=signalSemaphoreCount,pSignalSemaphores=pSignalSemaphores)

def VkImageCopy(srcSubresource=None,srcOffset=None,dstSubresource=None,dstOffset=None,extent=None,):

    return _new('VkImageCopy', srcSubresource=srcSubresource,srcOffset=srcOffset,dstSubresource=dstSubresource,dstOffset=dstOffset,extent=extent)

def VkImageBlit(srcSubresource=None,srcOffsets=None,dstSubresource=None,dstOffsets=None,):

    return _new('VkImageBlit', srcSubresource=srcSubresource,srcOffsets=srcOffsets,dstSubresource=dstSubresource,dstOffsets=dstOffsets)

def VkBufferImageCopy(bufferOffset=None,bufferRowLength=None,bufferImageHeight=None,imageSubresource=None,imageOffset=None,imageExtent=None,):

    return _new('VkBufferImageCopy', bufferOffset=bufferOffset,bufferRowLength=bufferRowLength,bufferImageHeight=bufferImageHeight,imageSubresource=imageSubresource,imageOffset=imageOffset,imageExtent=imageExtent)

def VkImageResolve(srcSubresource=None,srcOffset=None,dstSubresource=None,dstOffset=None,extent=None,):

    return _new('VkImageResolve', srcSubresource=srcSubresource,srcOffset=srcOffset,dstSubresource=dstSubresource,dstOffset=dstOffset,extent=extent)

def VkShaderModuleCreateInfo(sType=VK_STRUCTURE_TYPE_SHADER_MODULE_CREATE_INFO,pNext=None,flags=None,codeSize=None,pCode=None,):

    return _new('VkShaderModuleCreateInfo', sType=sType,pNext=pNext,flags=flags,codeSize=codeSize,pCode=pCode)

def VkDescriptorSetLayoutBinding(binding=None,descriptorType=None,descriptorCount=None,stageFlags=None,pImmutableSamplers=None,):
    if descriptorCount is None and pImmutableSamplers is not None:
        descriptorCount = len(pImmutableSamplers)

    return _new('VkDescriptorSetLayoutBinding', binding=binding,descriptorType=descriptorType,descriptorCount=descriptorCount,stageFlags=stageFlags,pImmutableSamplers=pImmutableSamplers)

def VkDescriptorSetLayoutCreateInfo(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_CREATE_INFO,pNext=None,flags=None,bindingCount=None,pBindings=None,):
    if bindingCount is None and pBindings is not None:
        bindingCount = len(pBindings)

    return _new('VkDescriptorSetLayoutCreateInfo', sType=sType,pNext=pNext,flags=flags,bindingCount=bindingCount,pBindings=pBindings)

def VkDescriptorPoolSize(type=None,descriptorCount=None,):

    return _new('VkDescriptorPoolSize', type=type,descriptorCount=descriptorCount)

def VkDescriptorPoolCreateInfo(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_POOL_CREATE_INFO,pNext=None,flags=None,maxSets=None,poolSizeCount=None,pPoolSizes=None,):
    if poolSizeCount is None and pPoolSizes is not None:
        poolSizeCount = len(pPoolSizes)

    return _new('VkDescriptorPoolCreateInfo', sType=sType,pNext=pNext,flags=flags,maxSets=maxSets,poolSizeCount=poolSizeCount,pPoolSizes=pPoolSizes)

def VkDescriptorSetAllocateInfo(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_SET_ALLOCATE_INFO,pNext=None,descriptorPool=None,descriptorSetCount=None,pSetLayouts=None,):
    if descriptorSetCount is None and pSetLayouts is not None:
        descriptorSetCount = len(pSetLayouts)

    return _new('VkDescriptorSetAllocateInfo', sType=sType,pNext=pNext,descriptorPool=descriptorPool,descriptorSetCount=descriptorSetCount,pSetLayouts=pSetLayouts)

def VkSpecializationMapEntry(constantID=None,offset=None,size=None,):

    return _new('VkSpecializationMapEntry', constantID=constantID,offset=offset,size=size)

def VkSpecializationInfo(mapEntryCount=None,pMapEntries=None,dataSize=None,pData=None,):
    if mapEntryCount is None and pMapEntries is not None:
        mapEntryCount = len(pMapEntries)
    if dataSize is None and pData is not None:
        dataSize = len(pData)

    return _new('VkSpecializationInfo', mapEntryCount=mapEntryCount,pMapEntries=pMapEntries,dataSize=dataSize,pData=pData)

def VkPipelineShaderStageCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_SHADER_STAGE_CREATE_INFO,pNext=None,flags=None,stage=None,module=None,pName=None,pSpecializationInfo=None,):

    return _new('VkPipelineShaderStageCreateInfo', sType=sType,pNext=pNext,flags=flags,stage=stage,module=module,pName=pName,pSpecializationInfo=pSpecializationInfo)

def VkComputePipelineCreateInfo(sType=VK_STRUCTURE_TYPE_COMPUTE_PIPELINE_CREATE_INFO,pNext=None,flags=None,stage=None,layout=None,basePipelineHandle=None,basePipelineIndex=None,):

    return _new('VkComputePipelineCreateInfo', sType=sType,pNext=pNext,flags=flags,stage=stage,layout=layout,basePipelineHandle=basePipelineHandle,basePipelineIndex=basePipelineIndex)

def VkVertexInputBindingDescription(binding=None,stride=None,inputRate=None,):

    return _new('VkVertexInputBindingDescription', binding=binding,stride=stride,inputRate=inputRate)

def VkVertexInputAttributeDescription(location=None,binding=None,format=None,offset=None,):

    return _new('VkVertexInputAttributeDescription', location=location,binding=binding,format=format,offset=offset)

def VkPipelineVertexInputStateCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_VERTEX_INPUT_STATE_CREATE_INFO,pNext=None,flags=None,vertexBindingDescriptionCount=None,pVertexBindingDescriptions=None,vertexAttributeDescriptionCount=None,pVertexAttributeDescriptions=None,):
    if vertexBindingDescriptionCount is None and pVertexBindingDescriptions is not None:
        vertexBindingDescriptionCount = len(pVertexBindingDescriptions)
    if vertexAttributeDescriptionCount is None and pVertexAttributeDescriptions is not None:
        vertexAttributeDescriptionCount = len(pVertexAttributeDescriptions)

    return _new('VkPipelineVertexInputStateCreateInfo', sType=sType,pNext=pNext,flags=flags,vertexBindingDescriptionCount=vertexBindingDescriptionCount,pVertexBindingDescriptions=pVertexBindingDescriptions,vertexAttributeDescriptionCount=vertexAttributeDescriptionCount,pVertexAttributeDescriptions=pVertexAttributeDescriptions)

def VkPipelineInputAssemblyStateCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_INPUT_ASSEMBLY_STATE_CREATE_INFO,pNext=None,flags=None,topology=None,primitiveRestartEnable=None,):

    return _new('VkPipelineInputAssemblyStateCreateInfo', sType=sType,pNext=pNext,flags=flags,topology=topology,primitiveRestartEnable=primitiveRestartEnable)

def VkPipelineTessellationStateCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_TESSELLATION_STATE_CREATE_INFO,pNext=None,flags=None,patchControlPoints=None,):

    return _new('VkPipelineTessellationStateCreateInfo', sType=sType,pNext=pNext,flags=flags,patchControlPoints=patchControlPoints)

def VkPipelineViewportStateCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_STATE_CREATE_INFO,pNext=None,flags=None,viewportCount=None,pViewports=None,scissorCount=None,pScissors=None,):
    if viewportCount is None and pViewports is not None:
        viewportCount = len(pViewports)
    if scissorCount is None and pScissors is not None:
        scissorCount = len(pScissors)

    return _new('VkPipelineViewportStateCreateInfo', sType=sType,pNext=pNext,flags=flags,viewportCount=viewportCount,pViewports=pViewports,scissorCount=scissorCount,pScissors=pScissors)

def VkPipelineRasterizationStateCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_CREATE_INFO,pNext=None,flags=None,depthClampEnable=None,rasterizerDiscardEnable=None,polygonMode=None,cullMode=None,frontFace=None,depthBiasEnable=None,depthBiasConstantFactor=None,depthBiasClamp=None,depthBiasSlopeFactor=None,lineWidth=None,):

    return _new('VkPipelineRasterizationStateCreateInfo', sType=sType,pNext=pNext,flags=flags,depthClampEnable=depthClampEnable,rasterizerDiscardEnable=rasterizerDiscardEnable,polygonMode=polygonMode,cullMode=cullMode,frontFace=frontFace,depthBiasEnable=depthBiasEnable,depthBiasConstantFactor=depthBiasConstantFactor,depthBiasClamp=depthBiasClamp,depthBiasSlopeFactor=depthBiasSlopeFactor,lineWidth=lineWidth)

def VkPipelineMultisampleStateCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_MULTISAMPLE_STATE_CREATE_INFO,pNext=None,flags=None,rasterizationSamples=None,sampleShadingEnable=None,minSampleShading=None,pSampleMask=None,alphaToCoverageEnable=None,alphaToOneEnable=None,):

    return _new('VkPipelineMultisampleStateCreateInfo', sType=sType,pNext=pNext,flags=flags,rasterizationSamples=rasterizationSamples,sampleShadingEnable=sampleShadingEnable,minSampleShading=minSampleShading,pSampleMask=pSampleMask,alphaToCoverageEnable=alphaToCoverageEnable,alphaToOneEnable=alphaToOneEnable)

def VkPipelineColorBlendAttachmentState(blendEnable=None,srcColorBlendFactor=None,dstColorBlendFactor=None,colorBlendOp=None,srcAlphaBlendFactor=None,dstAlphaBlendFactor=None,alphaBlendOp=None,colorWriteMask=None,):

    return _new('VkPipelineColorBlendAttachmentState', blendEnable=blendEnable,srcColorBlendFactor=srcColorBlendFactor,dstColorBlendFactor=dstColorBlendFactor,colorBlendOp=colorBlendOp,srcAlphaBlendFactor=srcAlphaBlendFactor,dstAlphaBlendFactor=dstAlphaBlendFactor,alphaBlendOp=alphaBlendOp,colorWriteMask=colorWriteMask)

def VkPipelineColorBlendStateCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_COLOR_BLEND_STATE_CREATE_INFO,pNext=None,flags=None,logicOpEnable=None,logicOp=None,attachmentCount=None,pAttachments=None,blendConstants=None,):
    if attachmentCount is None and pAttachments is not None:
        attachmentCount = len(pAttachments)

    return _new('VkPipelineColorBlendStateCreateInfo', sType=sType,pNext=pNext,flags=flags,logicOpEnable=logicOpEnable,logicOp=logicOp,attachmentCount=attachmentCount,pAttachments=pAttachments,blendConstants=blendConstants)

def VkPipelineDynamicStateCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_DYNAMIC_STATE_CREATE_INFO,pNext=None,flags=None,dynamicStateCount=None,pDynamicStates=None,):
    if dynamicStateCount is None and pDynamicStates is not None:
        dynamicStateCount = len(pDynamicStates)

    return _new('VkPipelineDynamicStateCreateInfo', sType=sType,pNext=pNext,flags=flags,dynamicStateCount=dynamicStateCount,pDynamicStates=pDynamicStates)

def VkStencilOpState(failOp=None,passOp=None,depthFailOp=None,compareOp=None,compareMask=None,writeMask=None,reference=None,):

    return _new('VkStencilOpState', failOp=failOp,passOp=passOp,depthFailOp=depthFailOp,compareOp=compareOp,compareMask=compareMask,writeMask=writeMask,reference=reference)

def VkPipelineDepthStencilStateCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_DEPTH_STENCIL_STATE_CREATE_INFO,pNext=None,flags=None,depthTestEnable=None,depthWriteEnable=None,depthCompareOp=None,depthBoundsTestEnable=None,stencilTestEnable=None,front=None,back=None,minDepthBounds=None,maxDepthBounds=None,):

    return _new('VkPipelineDepthStencilStateCreateInfo', sType=sType,pNext=pNext,flags=flags,depthTestEnable=depthTestEnable,depthWriteEnable=depthWriteEnable,depthCompareOp=depthCompareOp,depthBoundsTestEnable=depthBoundsTestEnable,stencilTestEnable=stencilTestEnable,front=front,back=back,minDepthBounds=minDepthBounds,maxDepthBounds=maxDepthBounds)

def VkGraphicsPipelineCreateInfo(sType=VK_STRUCTURE_TYPE_GRAPHICS_PIPELINE_CREATE_INFO,pNext=None,flags=None,stageCount=None,pStages=None,pVertexInputState=None,pInputAssemblyState=None,pTessellationState=None,pViewportState=None,pRasterizationState=None,pMultisampleState=None,pDepthStencilState=None,pColorBlendState=None,pDynamicState=None,layout=None,renderPass=None,subpass=None,basePipelineHandle=None,basePipelineIndex=None,):
    if stageCount is None and pStages is not None:
        stageCount = len(pStages)

    return _new('VkGraphicsPipelineCreateInfo', sType=sType,pNext=pNext,flags=flags,stageCount=stageCount,pStages=pStages,pVertexInputState=pVertexInputState,pInputAssemblyState=pInputAssemblyState,pTessellationState=pTessellationState,pViewportState=pViewportState,pRasterizationState=pRasterizationState,pMultisampleState=pMultisampleState,pDepthStencilState=pDepthStencilState,pColorBlendState=pColorBlendState,pDynamicState=pDynamicState,layout=layout,renderPass=renderPass,subpass=subpass,basePipelineHandle=basePipelineHandle,basePipelineIndex=basePipelineIndex)

def VkPipelineCacheCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_CACHE_CREATE_INFO,pNext=None,flags=None,initialDataSize=None,pInitialData=None,):
    if initialDataSize is None and pInitialData is not None:
        initialDataSize = len(pInitialData)

    return _new('VkPipelineCacheCreateInfo', sType=sType,pNext=pNext,flags=flags,initialDataSize=initialDataSize,pInitialData=pInitialData)

def VkPushConstantRange(stageFlags=None,offset=None,size=None,):

    return _new('VkPushConstantRange', stageFlags=stageFlags,offset=offset,size=size)

def VkPipelineLayoutCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_LAYOUT_CREATE_INFO,pNext=None,flags=None,setLayoutCount=None,pSetLayouts=None,pushConstantRangeCount=None,pPushConstantRanges=None,):
    if setLayoutCount is None and pSetLayouts is not None:
        setLayoutCount = len(pSetLayouts)
    if pushConstantRangeCount is None and pPushConstantRanges is not None:
        pushConstantRangeCount = len(pPushConstantRanges)

    return _new('VkPipelineLayoutCreateInfo', sType=sType,pNext=pNext,flags=flags,setLayoutCount=setLayoutCount,pSetLayouts=pSetLayouts,pushConstantRangeCount=pushConstantRangeCount,pPushConstantRanges=pPushConstantRanges)

def VkSamplerCreateInfo(sType=VK_STRUCTURE_TYPE_SAMPLER_CREATE_INFO,pNext=None,flags=None,magFilter=None,minFilter=None,mipmapMode=None,addressModeU=None,addressModeV=None,addressModeW=None,mipLodBias=None,anisotropyEnable=None,maxAnisotropy=None,compareEnable=None,compareOp=None,minLod=None,maxLod=None,borderColor=None,unnormalizedCoordinates=None,):

    return _new('VkSamplerCreateInfo', sType=sType,pNext=pNext,flags=flags,magFilter=magFilter,minFilter=minFilter,mipmapMode=mipmapMode,addressModeU=addressModeU,addressModeV=addressModeV,addressModeW=addressModeW,mipLodBias=mipLodBias,anisotropyEnable=anisotropyEnable,maxAnisotropy=maxAnisotropy,compareEnable=compareEnable,compareOp=compareOp,minLod=minLod,maxLod=maxLod,borderColor=borderColor,unnormalizedCoordinates=unnormalizedCoordinates)

def VkCommandPoolCreateInfo(sType=VK_STRUCTURE_TYPE_COMMAND_POOL_CREATE_INFO,pNext=None,flags=None,queueFamilyIndex=None,):

    return _new('VkCommandPoolCreateInfo', sType=sType,pNext=pNext,flags=flags,queueFamilyIndex=queueFamilyIndex)

def VkCommandBufferAllocateInfo(sType=VK_STRUCTURE_TYPE_COMMAND_BUFFER_ALLOCATE_INFO,pNext=None,commandPool=None,level=None,commandBufferCount=None,):

    return _new('VkCommandBufferAllocateInfo', sType=sType,pNext=pNext,commandPool=commandPool,level=level,commandBufferCount=commandBufferCount)

def VkCommandBufferInheritanceInfo(sType=VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_INFO,pNext=None,renderPass=None,subpass=None,framebuffer=None,occlusionQueryEnable=None,queryFlags=None,pipelineStatistics=None,):

    return _new('VkCommandBufferInheritanceInfo', sType=sType,pNext=pNext,renderPass=renderPass,subpass=subpass,framebuffer=framebuffer,occlusionQueryEnable=occlusionQueryEnable,queryFlags=queryFlags,pipelineStatistics=pipelineStatistics)

def VkCommandBufferBeginInfo(sType=VK_STRUCTURE_TYPE_COMMAND_BUFFER_BEGIN_INFO,pNext=None,flags=None,pInheritanceInfo=None,):

    return _new('VkCommandBufferBeginInfo', sType=sType,pNext=pNext,flags=flags,pInheritanceInfo=pInheritanceInfo)

def VkRenderPassBeginInfo(sType=VK_STRUCTURE_TYPE_RENDER_PASS_BEGIN_INFO,pNext=None,renderPass=None,framebuffer=None,renderArea=None,clearValueCount=None,pClearValues=None,):
    if clearValueCount is None and pClearValues is not None:
        clearValueCount = len(pClearValues)

    return _new('VkRenderPassBeginInfo', sType=sType,pNext=pNext,renderPass=renderPass,framebuffer=framebuffer,renderArea=renderArea,clearValueCount=clearValueCount,pClearValues=pClearValues)

def VkClearColorValue(float32=None,int32=None,uint32=None,):

    return _new('VkClearColorValue', float32=float32,int32=int32,uint32=uint32)

def VkClearDepthStencilValue(depth=None,stencil=None,):

    return _new('VkClearDepthStencilValue', depth=depth,stencil=stencil)

def VkClearValue(color=None,depthStencil=None,):

    return _new('VkClearValue', color=color,depthStencil=depthStencil)

def VkClearAttachment(aspectMask=None,colorAttachment=None,clearValue=None,):

    return _new('VkClearAttachment', aspectMask=aspectMask,colorAttachment=colorAttachment,clearValue=clearValue)

def VkAttachmentDescription(flags=None,format=None,samples=None,loadOp=None,storeOp=None,stencilLoadOp=None,stencilStoreOp=None,initialLayout=None,finalLayout=None,):

    return _new('VkAttachmentDescription', flags=flags,format=format,samples=samples,loadOp=loadOp,storeOp=storeOp,stencilLoadOp=stencilLoadOp,stencilStoreOp=stencilStoreOp,initialLayout=initialLayout,finalLayout=finalLayout)

def VkAttachmentReference(attachment=None,layout=None,):

    return _new('VkAttachmentReference', attachment=attachment,layout=layout)

def VkSubpassDescription(flags=None,pipelineBindPoint=None,inputAttachmentCount=None,pInputAttachments=None,colorAttachmentCount=None,pColorAttachments=None,pResolveAttachments=None,pDepthStencilAttachment=None,preserveAttachmentCount=None,pPreserveAttachments=None,):
    if inputAttachmentCount is None and pInputAttachments is not None:
        inputAttachmentCount = len(pInputAttachments)
    if colorAttachmentCount is None and pColorAttachments is not None:
        colorAttachmentCount = len(pColorAttachments)
    if colorAttachmentCount is None and pResolveAttachments is not None:
        colorAttachmentCount = len(pResolveAttachments)
    if preserveAttachmentCount is None and pPreserveAttachments is not None:
        preserveAttachmentCount = len(pPreserveAttachments)

    return _new('VkSubpassDescription', flags=flags,pipelineBindPoint=pipelineBindPoint,inputAttachmentCount=inputAttachmentCount,pInputAttachments=pInputAttachments,colorAttachmentCount=colorAttachmentCount,pColorAttachments=pColorAttachments,pResolveAttachments=pResolveAttachments,pDepthStencilAttachment=pDepthStencilAttachment,preserveAttachmentCount=preserveAttachmentCount,pPreserveAttachments=pPreserveAttachments)

def VkSubpassDependency(srcSubpass=None,dstSubpass=None,srcStageMask=None,dstStageMask=None,srcAccessMask=None,dstAccessMask=None,dependencyFlags=None,):

    return _new('VkSubpassDependency', srcSubpass=srcSubpass,dstSubpass=dstSubpass,srcStageMask=srcStageMask,dstStageMask=dstStageMask,srcAccessMask=srcAccessMask,dstAccessMask=dstAccessMask,dependencyFlags=dependencyFlags)

def VkRenderPassCreateInfo(sType=VK_STRUCTURE_TYPE_RENDER_PASS_CREATE_INFO,pNext=None,flags=None,attachmentCount=None,pAttachments=None,subpassCount=None,pSubpasses=None,dependencyCount=None,pDependencies=None,):
    if attachmentCount is None and pAttachments is not None:
        attachmentCount = len(pAttachments)
    if subpassCount is None and pSubpasses is not None:
        subpassCount = len(pSubpasses)
    if dependencyCount is None and pDependencies is not None:
        dependencyCount = len(pDependencies)

    return _new('VkRenderPassCreateInfo', sType=sType,pNext=pNext,flags=flags,attachmentCount=attachmentCount,pAttachments=pAttachments,subpassCount=subpassCount,pSubpasses=pSubpasses,dependencyCount=dependencyCount,pDependencies=pDependencies)

def VkEventCreateInfo(sType=VK_STRUCTURE_TYPE_EVENT_CREATE_INFO,pNext=None,flags=None,):

    return _new('VkEventCreateInfo', sType=sType,pNext=pNext,flags=flags)

def VkFenceCreateInfo(sType=VK_STRUCTURE_TYPE_FENCE_CREATE_INFO,pNext=None,flags=None,):

    return _new('VkFenceCreateInfo', sType=sType,pNext=pNext,flags=flags)

def VkPhysicalDeviceFeatures(robustBufferAccess=None,fullDrawIndexUint32=None,imageCubeArray=None,independentBlend=None,geometryShader=None,tessellationShader=None,sampleRateShading=None,dualSrcBlend=None,logicOp=None,multiDrawIndirect=None,drawIndirectFirstInstance=None,depthClamp=None,depthBiasClamp=None,fillModeNonSolid=None,depthBounds=None,wideLines=None,largePoints=None,alphaToOne=None,multiViewport=None,samplerAnisotropy=None,textureCompressionETC2=None,textureCompressionASTC_LDR=None,textureCompressionBC=None,occlusionQueryPrecise=None,pipelineStatisticsQuery=None,vertexPipelineStoresAndAtomics=None,fragmentStoresAndAtomics=None,shaderTessellationAndGeometryPointSize=None,shaderImageGatherExtended=None,shaderStorageImageExtendedFormats=None,shaderStorageImageMultisample=None,shaderStorageImageReadWithoutFormat=None,shaderStorageImageWriteWithoutFormat=None,shaderUniformBufferArrayDynamicIndexing=None,shaderSampledImageArrayDynamicIndexing=None,shaderStorageBufferArrayDynamicIndexing=None,shaderStorageImageArrayDynamicIndexing=None,shaderClipDistance=None,shaderCullDistance=None,shaderFloat64=None,shaderInt64=None,shaderInt16=None,shaderResourceResidency=None,shaderResourceMinLod=None,sparseBinding=None,sparseResidencyBuffer=None,sparseResidencyImage2D=None,sparseResidencyImage3D=None,sparseResidency2Samples=None,sparseResidency4Samples=None,sparseResidency8Samples=None,sparseResidency16Samples=None,sparseResidencyAliased=None,variableMultisampleRate=None,inheritedQueries=None,):

    return _new('VkPhysicalDeviceFeatures', robustBufferAccess=robustBufferAccess,fullDrawIndexUint32=fullDrawIndexUint32,imageCubeArray=imageCubeArray,independentBlend=independentBlend,geometryShader=geometryShader,tessellationShader=tessellationShader,sampleRateShading=sampleRateShading,dualSrcBlend=dualSrcBlend,logicOp=logicOp,multiDrawIndirect=multiDrawIndirect,drawIndirectFirstInstance=drawIndirectFirstInstance,depthClamp=depthClamp,depthBiasClamp=depthBiasClamp,fillModeNonSolid=fillModeNonSolid,depthBounds=depthBounds,wideLines=wideLines,largePoints=largePoints,alphaToOne=alphaToOne,multiViewport=multiViewport,samplerAnisotropy=samplerAnisotropy,textureCompressionETC2=textureCompressionETC2,textureCompressionASTC_LDR=textureCompressionASTC_LDR,textureCompressionBC=textureCompressionBC,occlusionQueryPrecise=occlusionQueryPrecise,pipelineStatisticsQuery=pipelineStatisticsQuery,vertexPipelineStoresAndAtomics=vertexPipelineStoresAndAtomics,fragmentStoresAndAtomics=fragmentStoresAndAtomics,shaderTessellationAndGeometryPointSize=shaderTessellationAndGeometryPointSize,shaderImageGatherExtended=shaderImageGatherExtended,shaderStorageImageExtendedFormats=shaderStorageImageExtendedFormats,shaderStorageImageMultisample=shaderStorageImageMultisample,shaderStorageImageReadWithoutFormat=shaderStorageImageReadWithoutFormat,shaderStorageImageWriteWithoutFormat=shaderStorageImageWriteWithoutFormat,shaderUniformBufferArrayDynamicIndexing=shaderUniformBufferArrayDynamicIndexing,shaderSampledImageArrayDynamicIndexing=shaderSampledImageArrayDynamicIndexing,shaderStorageBufferArrayDynamicIndexing=shaderStorageBufferArrayDynamicIndexing,shaderStorageImageArrayDynamicIndexing=shaderStorageImageArrayDynamicIndexing,shaderClipDistance=shaderClipDistance,shaderCullDistance=shaderCullDistance,shaderFloat64=shaderFloat64,shaderInt64=shaderInt64,shaderInt16=shaderInt16,shaderResourceResidency=shaderResourceResidency,shaderResourceMinLod=shaderResourceMinLod,sparseBinding=sparseBinding,sparseResidencyBuffer=sparseResidencyBuffer,sparseResidencyImage2D=sparseResidencyImage2D,sparseResidencyImage3D=sparseResidencyImage3D,sparseResidency2Samples=sparseResidency2Samples,sparseResidency4Samples=sparseResidency4Samples,sparseResidency8Samples=sparseResidency8Samples,sparseResidency16Samples=sparseResidency16Samples,sparseResidencyAliased=sparseResidencyAliased,variableMultisampleRate=variableMultisampleRate,inheritedQueries=inheritedQueries)

def VkPhysicalDeviceSparseProperties(residencyStandard2DBlockShape=None,residencyStandard2DMultisampleBlockShape=None,residencyStandard3DBlockShape=None,residencyAlignedMipSize=None,residencyNonResidentStrict=None,):

    return _new('VkPhysicalDeviceSparseProperties', residencyStandard2DBlockShape=residencyStandard2DBlockShape,residencyStandard2DMultisampleBlockShape=residencyStandard2DMultisampleBlockShape,residencyStandard3DBlockShape=residencyStandard3DBlockShape,residencyAlignedMipSize=residencyAlignedMipSize,residencyNonResidentStrict=residencyNonResidentStrict)

def VkPhysicalDeviceLimits(maxImageDimension1D=None,maxImageDimension2D=None,maxImageDimension3D=None,maxImageDimensionCube=None,maxImageArrayLayers=None,maxTexelBufferElements=None,maxUniformBufferRange=None,maxStorageBufferRange=None,maxPushConstantsSize=None,maxMemoryAllocationCount=None,maxSamplerAllocationCount=None,bufferImageGranularity=None,sparseAddressSpaceSize=None,maxBoundDescriptorSets=None,maxPerStageDescriptorSamplers=None,maxPerStageDescriptorUniformBuffers=None,maxPerStageDescriptorStorageBuffers=None,maxPerStageDescriptorSampledImages=None,maxPerStageDescriptorStorageImages=None,maxPerStageDescriptorInputAttachments=None,maxPerStageResources=None,maxDescriptorSetSamplers=None,maxDescriptorSetUniformBuffers=None,maxDescriptorSetUniformBuffersDynamic=None,maxDescriptorSetStorageBuffers=None,maxDescriptorSetStorageBuffersDynamic=None,maxDescriptorSetSampledImages=None,maxDescriptorSetStorageImages=None,maxDescriptorSetInputAttachments=None,maxVertexInputAttributes=None,maxVertexInputBindings=None,maxVertexInputAttributeOffset=None,maxVertexInputBindingStride=None,maxVertexOutputComponents=None,maxTessellationGenerationLevel=None,maxTessellationPatchSize=None,maxTessellationControlPerVertexInputComponents=None,maxTessellationControlPerVertexOutputComponents=None,maxTessellationControlPerPatchOutputComponents=None,maxTessellationControlTotalOutputComponents=None,maxTessellationEvaluationInputComponents=None,maxTessellationEvaluationOutputComponents=None,maxGeometryShaderInvocations=None,maxGeometryInputComponents=None,maxGeometryOutputComponents=None,maxGeometryOutputVertices=None,maxGeometryTotalOutputComponents=None,maxFragmentInputComponents=None,maxFragmentOutputAttachments=None,maxFragmentDualSrcAttachments=None,maxFragmentCombinedOutputResources=None,maxComputeSharedMemorySize=None,maxComputeWorkGroupCount=None,maxComputeWorkGroupInvocations=None,maxComputeWorkGroupSize=None,subPixelPrecisionBits=None,subTexelPrecisionBits=None,mipmapPrecisionBits=None,maxDrawIndexedIndexValue=None,maxDrawIndirectCount=None,maxSamplerLodBias=None,maxSamplerAnisotropy=None,maxViewports=None,maxViewportDimensions=None,viewportBoundsRange=None,viewportSubPixelBits=None,minMemoryMapAlignment=None,minTexelBufferOffsetAlignment=None,minUniformBufferOffsetAlignment=None,minStorageBufferOffsetAlignment=None,minTexelOffset=None,maxTexelOffset=None,minTexelGatherOffset=None,maxTexelGatherOffset=None,minInterpolationOffset=None,maxInterpolationOffset=None,subPixelInterpolationOffsetBits=None,maxFramebufferWidth=None,maxFramebufferHeight=None,maxFramebufferLayers=None,framebufferColorSampleCounts=None,framebufferDepthSampleCounts=None,framebufferStencilSampleCounts=None,framebufferNoAttachmentsSampleCounts=None,maxColorAttachments=None,sampledImageColorSampleCounts=None,sampledImageIntegerSampleCounts=None,sampledImageDepthSampleCounts=None,sampledImageStencilSampleCounts=None,storageImageSampleCounts=None,maxSampleMaskWords=None,timestampComputeAndGraphics=None,timestampPeriod=None,maxClipDistances=None,maxCullDistances=None,maxCombinedClipAndCullDistances=None,discreteQueuePriorities=None,pointSizeRange=None,lineWidthRange=None,pointSizeGranularity=None,lineWidthGranularity=None,strictLines=None,standardSampleLocations=None,optimalBufferCopyOffsetAlignment=None,optimalBufferCopyRowPitchAlignment=None,nonCoherentAtomSize=None,):

    return _new('VkPhysicalDeviceLimits', maxImageDimension1D=maxImageDimension1D,maxImageDimension2D=maxImageDimension2D,maxImageDimension3D=maxImageDimension3D,maxImageDimensionCube=maxImageDimensionCube,maxImageArrayLayers=maxImageArrayLayers,maxTexelBufferElements=maxTexelBufferElements,maxUniformBufferRange=maxUniformBufferRange,maxStorageBufferRange=maxStorageBufferRange,maxPushConstantsSize=maxPushConstantsSize,maxMemoryAllocationCount=maxMemoryAllocationCount,maxSamplerAllocationCount=maxSamplerAllocationCount,bufferImageGranularity=bufferImageGranularity,sparseAddressSpaceSize=sparseAddressSpaceSize,maxBoundDescriptorSets=maxBoundDescriptorSets,maxPerStageDescriptorSamplers=maxPerStageDescriptorSamplers,maxPerStageDescriptorUniformBuffers=maxPerStageDescriptorUniformBuffers,maxPerStageDescriptorStorageBuffers=maxPerStageDescriptorStorageBuffers,maxPerStageDescriptorSampledImages=maxPerStageDescriptorSampledImages,maxPerStageDescriptorStorageImages=maxPerStageDescriptorStorageImages,maxPerStageDescriptorInputAttachments=maxPerStageDescriptorInputAttachments,maxPerStageResources=maxPerStageResources,maxDescriptorSetSamplers=maxDescriptorSetSamplers,maxDescriptorSetUniformBuffers=maxDescriptorSetUniformBuffers,maxDescriptorSetUniformBuffersDynamic=maxDescriptorSetUniformBuffersDynamic,maxDescriptorSetStorageBuffers=maxDescriptorSetStorageBuffers,maxDescriptorSetStorageBuffersDynamic=maxDescriptorSetStorageBuffersDynamic,maxDescriptorSetSampledImages=maxDescriptorSetSampledImages,maxDescriptorSetStorageImages=maxDescriptorSetStorageImages,maxDescriptorSetInputAttachments=maxDescriptorSetInputAttachments,maxVertexInputAttributes=maxVertexInputAttributes,maxVertexInputBindings=maxVertexInputBindings,maxVertexInputAttributeOffset=maxVertexInputAttributeOffset,maxVertexInputBindingStride=maxVertexInputBindingStride,maxVertexOutputComponents=maxVertexOutputComponents,maxTessellationGenerationLevel=maxTessellationGenerationLevel,maxTessellationPatchSize=maxTessellationPatchSize,maxTessellationControlPerVertexInputComponents=maxTessellationControlPerVertexInputComponents,maxTessellationControlPerVertexOutputComponents=maxTessellationControlPerVertexOutputComponents,maxTessellationControlPerPatchOutputComponents=maxTessellationControlPerPatchOutputComponents,maxTessellationControlTotalOutputComponents=maxTessellationControlTotalOutputComponents,maxTessellationEvaluationInputComponents=maxTessellationEvaluationInputComponents,maxTessellationEvaluationOutputComponents=maxTessellationEvaluationOutputComponents,maxGeometryShaderInvocations=maxGeometryShaderInvocations,maxGeometryInputComponents=maxGeometryInputComponents,maxGeometryOutputComponents=maxGeometryOutputComponents,maxGeometryOutputVertices=maxGeometryOutputVertices,maxGeometryTotalOutputComponents=maxGeometryTotalOutputComponents,maxFragmentInputComponents=maxFragmentInputComponents,maxFragmentOutputAttachments=maxFragmentOutputAttachments,maxFragmentDualSrcAttachments=maxFragmentDualSrcAttachments,maxFragmentCombinedOutputResources=maxFragmentCombinedOutputResources,maxComputeSharedMemorySize=maxComputeSharedMemorySize,maxComputeWorkGroupCount=maxComputeWorkGroupCount,maxComputeWorkGroupInvocations=maxComputeWorkGroupInvocations,maxComputeWorkGroupSize=maxComputeWorkGroupSize,subPixelPrecisionBits=subPixelPrecisionBits,subTexelPrecisionBits=subTexelPrecisionBits,mipmapPrecisionBits=mipmapPrecisionBits,maxDrawIndexedIndexValue=maxDrawIndexedIndexValue,maxDrawIndirectCount=maxDrawIndirectCount,maxSamplerLodBias=maxSamplerLodBias,maxSamplerAnisotropy=maxSamplerAnisotropy,maxViewports=maxViewports,maxViewportDimensions=maxViewportDimensions,viewportBoundsRange=viewportBoundsRange,viewportSubPixelBits=viewportSubPixelBits,minMemoryMapAlignment=minMemoryMapAlignment,minTexelBufferOffsetAlignment=minTexelBufferOffsetAlignment,minUniformBufferOffsetAlignment=minUniformBufferOffsetAlignment,minStorageBufferOffsetAlignment=minStorageBufferOffsetAlignment,minTexelOffset=minTexelOffset,maxTexelOffset=maxTexelOffset,minTexelGatherOffset=minTexelGatherOffset,maxTexelGatherOffset=maxTexelGatherOffset,minInterpolationOffset=minInterpolationOffset,maxInterpolationOffset=maxInterpolationOffset,subPixelInterpolationOffsetBits=subPixelInterpolationOffsetBits,maxFramebufferWidth=maxFramebufferWidth,maxFramebufferHeight=maxFramebufferHeight,maxFramebufferLayers=maxFramebufferLayers,framebufferColorSampleCounts=framebufferColorSampleCounts,framebufferDepthSampleCounts=framebufferDepthSampleCounts,framebufferStencilSampleCounts=framebufferStencilSampleCounts,framebufferNoAttachmentsSampleCounts=framebufferNoAttachmentsSampleCounts,maxColorAttachments=maxColorAttachments,sampledImageColorSampleCounts=sampledImageColorSampleCounts,sampledImageIntegerSampleCounts=sampledImageIntegerSampleCounts,sampledImageDepthSampleCounts=sampledImageDepthSampleCounts,sampledImageStencilSampleCounts=sampledImageStencilSampleCounts,storageImageSampleCounts=storageImageSampleCounts,maxSampleMaskWords=maxSampleMaskWords,timestampComputeAndGraphics=timestampComputeAndGraphics,timestampPeriod=timestampPeriod,maxClipDistances=maxClipDistances,maxCullDistances=maxCullDistances,maxCombinedClipAndCullDistances=maxCombinedClipAndCullDistances,discreteQueuePriorities=discreteQueuePriorities,pointSizeRange=pointSizeRange,lineWidthRange=lineWidthRange,pointSizeGranularity=pointSizeGranularity,lineWidthGranularity=lineWidthGranularity,strictLines=strictLines,standardSampleLocations=standardSampleLocations,optimalBufferCopyOffsetAlignment=optimalBufferCopyOffsetAlignment,optimalBufferCopyRowPitchAlignment=optimalBufferCopyRowPitchAlignment,nonCoherentAtomSize=nonCoherentAtomSize)

def VkSemaphoreCreateInfo(sType=VK_STRUCTURE_TYPE_SEMAPHORE_CREATE_INFO,pNext=None,flags=None,):

    return _new('VkSemaphoreCreateInfo', sType=sType,pNext=pNext,flags=flags)

def VkQueryPoolCreateInfo(sType=VK_STRUCTURE_TYPE_QUERY_POOL_CREATE_INFO,pNext=None,flags=None,queryType=None,queryCount=None,pipelineStatistics=None,):

    return _new('VkQueryPoolCreateInfo', sType=sType,pNext=pNext,flags=flags,queryType=queryType,queryCount=queryCount,pipelineStatistics=pipelineStatistics)

def VkFramebufferCreateInfo(sType=VK_STRUCTURE_TYPE_FRAMEBUFFER_CREATE_INFO,pNext=None,flags=None,renderPass=None,attachmentCount=None,pAttachments=None,width=None,height=None,layers=None,):
    if attachmentCount is None and pAttachments is not None:
        attachmentCount = len(pAttachments)

    return _new('VkFramebufferCreateInfo', sType=sType,pNext=pNext,flags=flags,renderPass=renderPass,attachmentCount=attachmentCount,pAttachments=pAttachments,width=width,height=height,layers=layers)

def VkDrawIndirectCommand(vertexCount=None,instanceCount=None,firstVertex=None,firstInstance=None,):

    return _new('VkDrawIndirectCommand', vertexCount=vertexCount,instanceCount=instanceCount,firstVertex=firstVertex,firstInstance=firstInstance)

def VkDrawIndexedIndirectCommand(indexCount=None,instanceCount=None,firstIndex=None,vertexOffset=None,firstInstance=None,):

    return _new('VkDrawIndexedIndirectCommand', indexCount=indexCount,instanceCount=instanceCount,firstIndex=firstIndex,vertexOffset=vertexOffset,firstInstance=firstInstance)

def VkDispatchIndirectCommand(x=None,y=None,z=None,):

    return _new('VkDispatchIndirectCommand', x=x,y=y,z=z)

def VkSubmitInfo(sType=VK_STRUCTURE_TYPE_SUBMIT_INFO,pNext=None,waitSemaphoreCount=None,pWaitSemaphores=None,pWaitDstStageMask=None,commandBufferCount=None,pCommandBuffers=None,signalSemaphoreCount=None,pSignalSemaphores=None,):
    if waitSemaphoreCount is None and pWaitSemaphores is not None:
        waitSemaphoreCount = len(pWaitSemaphores)
    if waitSemaphoreCount is None and pWaitDstStageMask is not None:
        waitSemaphoreCount = len(pWaitDstStageMask)
    if commandBufferCount is None and pCommandBuffers is not None:
        commandBufferCount = len(pCommandBuffers)
    if signalSemaphoreCount is None and pSignalSemaphores is not None:
        signalSemaphoreCount = len(pSignalSemaphores)

    return _new('VkSubmitInfo', sType=sType,pNext=pNext,waitSemaphoreCount=waitSemaphoreCount,pWaitSemaphores=pWaitSemaphores,pWaitDstStageMask=pWaitDstStageMask,commandBufferCount=commandBufferCount,pCommandBuffers=pCommandBuffers,signalSemaphoreCount=signalSemaphoreCount,pSignalSemaphores=pSignalSemaphores)

def VkDisplayPropertiesKHR(display=None,displayName=None,physicalDimensions=None,physicalResolution=None,supportedTransforms=None,planeReorderPossible=None,persistentContent=None,):

    return _new('VkDisplayPropertiesKHR', display=display,displayName=displayName,physicalDimensions=physicalDimensions,physicalResolution=physicalResolution,supportedTransforms=supportedTransforms,planeReorderPossible=planeReorderPossible,persistentContent=persistentContent)

def VkDisplayPlanePropertiesKHR(currentDisplay=None,currentStackIndex=None,):

    return _new('VkDisplayPlanePropertiesKHR', currentDisplay=currentDisplay,currentStackIndex=currentStackIndex)

def VkDisplayModeParametersKHR(visibleRegion=None,refreshRate=None,):

    return _new('VkDisplayModeParametersKHR', visibleRegion=visibleRegion,refreshRate=refreshRate)

def VkDisplayModePropertiesKHR(displayMode=None,parameters=None,):

    return _new('VkDisplayModePropertiesKHR', displayMode=displayMode,parameters=parameters)

def VkDisplayModeCreateInfoKHR(sType=VK_STRUCTURE_TYPE_DISPLAY_MODE_CREATE_INFO_KHR,pNext=None,flags=None,parameters=None,):

    return _new('VkDisplayModeCreateInfoKHR', sType=sType,pNext=pNext,flags=flags,parameters=parameters)

def VkDisplayPlaneCapabilitiesKHR(supportedAlpha=None,minSrcPosition=None,maxSrcPosition=None,minSrcExtent=None,maxSrcExtent=None,minDstPosition=None,maxDstPosition=None,minDstExtent=None,maxDstExtent=None,):

    return _new('VkDisplayPlaneCapabilitiesKHR', supportedAlpha=supportedAlpha,minSrcPosition=minSrcPosition,maxSrcPosition=maxSrcPosition,minSrcExtent=minSrcExtent,maxSrcExtent=maxSrcExtent,minDstPosition=minDstPosition,maxDstPosition=maxDstPosition,minDstExtent=minDstExtent,maxDstExtent=maxDstExtent)

def VkDisplaySurfaceCreateInfoKHR(sType=VK_STRUCTURE_TYPE_DISPLAY_SURFACE_CREATE_INFO_KHR,pNext=None,flags=None,displayMode=None,planeIndex=None,planeStackIndex=None,transform=None,globalAlpha=None,alphaMode=None,imageExtent=None,):

    return _new('VkDisplaySurfaceCreateInfoKHR', sType=sType,pNext=pNext,flags=flags,displayMode=displayMode,planeIndex=planeIndex,planeStackIndex=planeStackIndex,transform=transform,globalAlpha=globalAlpha,alphaMode=alphaMode,imageExtent=imageExtent)

def VkDisplayPresentInfoKHR(sType=VK_STRUCTURE_TYPE_DISPLAY_PRESENT_INFO_KHR,pNext=None,srcRect=None,dstRect=None,persistent=None,):

    return _new('VkDisplayPresentInfoKHR', sType=sType,pNext=pNext,srcRect=srcRect,dstRect=dstRect,persistent=persistent)

def VkSurfaceCapabilitiesKHR(minImageCount=None,maxImageCount=None,currentExtent=None,minImageExtent=None,maxImageExtent=None,maxImageArrayLayers=None,supportedTransforms=None,currentTransform=None,supportedCompositeAlpha=None,supportedUsageFlags=None,):

    return _new('VkSurfaceCapabilitiesKHR', minImageCount=minImageCount,maxImageCount=maxImageCount,currentExtent=currentExtent,minImageExtent=minImageExtent,maxImageExtent=maxImageExtent,maxImageArrayLayers=maxImageArrayLayers,supportedTransforms=supportedTransforms,currentTransform=currentTransform,supportedCompositeAlpha=supportedCompositeAlpha,supportedUsageFlags=supportedUsageFlags)

def VkAndroidSurfaceCreateInfoKHR(sType=VK_STRUCTURE_TYPE_ANDROID_SURFACE_CREATE_INFO_KHR,pNext=None,flags=None,window=None,):

    return _new('VkAndroidSurfaceCreateInfoKHR', sType=sType,pNext=pNext,flags=flags,window=window)

def VkViSurfaceCreateInfoNN(sType=VK_STRUCTURE_TYPE_VI_SURFACE_CREATE_INFO_NN,pNext=None,flags=None,window=None,):

    return _new('VkViSurfaceCreateInfoNN', sType=sType,pNext=pNext,flags=flags,window=window)

def VkWaylandSurfaceCreateInfoKHR(sType=VK_STRUCTURE_TYPE_WAYLAND_SURFACE_CREATE_INFO_KHR,pNext=None,flags=None,display=None,surface=None,):

    return _new('VkWaylandSurfaceCreateInfoKHR', sType=sType,pNext=pNext,flags=flags,display=display,surface=surface)

def VkWin32SurfaceCreateInfoKHR(sType=VK_STRUCTURE_TYPE_WIN32_SURFACE_CREATE_INFO_KHR,pNext=None,flags=None,hinstance=None,hwnd=None,):

    return _new('VkWin32SurfaceCreateInfoKHR', sType=sType,pNext=pNext,flags=flags,hinstance=hinstance,hwnd=hwnd)

def VkXlibSurfaceCreateInfoKHR(sType=VK_STRUCTURE_TYPE_XLIB_SURFACE_CREATE_INFO_KHR,pNext=None,flags=None,dpy=None,window=None,):

    return _new('VkXlibSurfaceCreateInfoKHR', sType=sType,pNext=pNext,flags=flags,dpy=dpy,window=window)

def VkXcbSurfaceCreateInfoKHR(sType=VK_STRUCTURE_TYPE_XCB_SURFACE_CREATE_INFO_KHR,pNext=None,flags=None,connection=None,window=None,):

    return _new('VkXcbSurfaceCreateInfoKHR', sType=sType,pNext=pNext,flags=flags,connection=connection,window=window)

def VkImagePipeSurfaceCreateInfoFUCHSIA(sType=VK_STRUCTURE_TYPE_IMAGEPIPE_SURFACE_CREATE_INFO_FUCHSIA,pNext=None,flags=None,imagePipeHandle=None,):

    return _new('VkImagePipeSurfaceCreateInfoFUCHSIA', sType=sType,pNext=pNext,flags=flags,imagePipeHandle=imagePipeHandle)

def VkSurfaceFormatKHR(format=None,colorSpace=None,):

    return _new('VkSurfaceFormatKHR', format=format,colorSpace=colorSpace)

def VkSwapchainCreateInfoKHR(sType=VK_STRUCTURE_TYPE_SWAPCHAIN_CREATE_INFO_KHR,pNext=None,flags=None,surface=None,minImageCount=None,imageFormat=None,imageColorSpace=None,imageExtent=None,imageArrayLayers=None,imageUsage=None,imageSharingMode=None,queueFamilyIndexCount=None,pQueueFamilyIndices=None,preTransform=None,compositeAlpha=None,presentMode=None,clipped=None,oldSwapchain=None,):
    if queueFamilyIndexCount is None and pQueueFamilyIndices is not None:
        queueFamilyIndexCount = len(pQueueFamilyIndices)

    return _new('VkSwapchainCreateInfoKHR', sType=sType,pNext=pNext,flags=flags,surface=surface,minImageCount=minImageCount,imageFormat=imageFormat,imageColorSpace=imageColorSpace,imageExtent=imageExtent,imageArrayLayers=imageArrayLayers,imageUsage=imageUsage,imageSharingMode=imageSharingMode,queueFamilyIndexCount=queueFamilyIndexCount,pQueueFamilyIndices=pQueueFamilyIndices,preTransform=preTransform,compositeAlpha=compositeAlpha,presentMode=presentMode,clipped=clipped,oldSwapchain=oldSwapchain)

def VkPresentInfoKHR(sType=VK_STRUCTURE_TYPE_PRESENT_INFO_KHR,pNext=None,waitSemaphoreCount=None,pWaitSemaphores=None,swapchainCount=None,pSwapchains=None,pImageIndices=None,pResults=None,):
    if waitSemaphoreCount is None and pWaitSemaphores is not None:
        waitSemaphoreCount = len(pWaitSemaphores)
    if swapchainCount is None and pSwapchains is not None:
        swapchainCount = len(pSwapchains)
    if swapchainCount is None and pImageIndices is not None:
        swapchainCount = len(pImageIndices)
    if swapchainCount is None and pResults is not None:
        swapchainCount = len(pResults)

    return _new('VkPresentInfoKHR', sType=sType,pNext=pNext,waitSemaphoreCount=waitSemaphoreCount,pWaitSemaphores=pWaitSemaphores,swapchainCount=swapchainCount,pSwapchains=pSwapchains,pImageIndices=pImageIndices,pResults=pResults)

def VkDebugReportCallbackCreateInfoEXT(sType=VK_STRUCTURE_TYPE_DEBUG_REPORT_CALLBACK_CREATE_INFO_EXT,pNext=None,flags=None,pfnCallback=None,pUserData=None,):

    return _new('VkDebugReportCallbackCreateInfoEXT', sType=sType,pNext=pNext,flags=flags,pfnCallback=pfnCallback,pUserData=pUserData)

def VkValidationFlagsEXT(sType=VK_STRUCTURE_TYPE_VALIDATION_FLAGS_EXT,pNext=None,disabledValidationCheckCount=None,pDisabledValidationChecks=None,):
    if disabledValidationCheckCount is None and pDisabledValidationChecks is not None:
        disabledValidationCheckCount = len(pDisabledValidationChecks)

    return _new('VkValidationFlagsEXT', sType=sType,pNext=pNext,disabledValidationCheckCount=disabledValidationCheckCount,pDisabledValidationChecks=pDisabledValidationChecks)

def VkValidationFeaturesEXT(sType=VK_STRUCTURE_TYPE_VALIDATION_FEATURES_EXT,pNext=None,enabledValidationFeatureCount=None,pEnabledValidationFeatures=None,disabledValidationFeatureCount=None,pDisabledValidationFeatures=None,):
    if enabledValidationFeatureCount is None and pEnabledValidationFeatures is not None:
        enabledValidationFeatureCount = len(pEnabledValidationFeatures)
    if disabledValidationFeatureCount is None and pDisabledValidationFeatures is not None:
        disabledValidationFeatureCount = len(pDisabledValidationFeatures)

    return _new('VkValidationFeaturesEXT', sType=sType,pNext=pNext,enabledValidationFeatureCount=enabledValidationFeatureCount,pEnabledValidationFeatures=pEnabledValidationFeatures,disabledValidationFeatureCount=disabledValidationFeatureCount,pDisabledValidationFeatures=pDisabledValidationFeatures)

def VkPipelineRasterizationStateRasterizationOrderAMD(sType=VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_RASTERIZATION_ORDER_AMD,pNext=None,rasterizationOrder=None,):

    return _new('VkPipelineRasterizationStateRasterizationOrderAMD', sType=sType,pNext=pNext,rasterizationOrder=rasterizationOrder)

def VkDebugMarkerObjectNameInfoEXT(sType=VK_STRUCTURE_TYPE_DEBUG_MARKER_OBJECT_NAME_INFO_EXT,pNext=None,objectType=None,object=None,pObjectName=None,):

    return _new('VkDebugMarkerObjectNameInfoEXT', sType=sType,pNext=pNext,objectType=objectType,object=object,pObjectName=pObjectName)

def VkDebugMarkerObjectTagInfoEXT(sType=VK_STRUCTURE_TYPE_DEBUG_MARKER_OBJECT_TAG_INFO_EXT,pNext=None,objectType=None,object=None,tagName=None,tagSize=None,pTag=None,):
    if tagSize is None and pTag is not None:
        tagSize = len(pTag)

    return _new('VkDebugMarkerObjectTagInfoEXT', sType=sType,pNext=pNext,objectType=objectType,object=object,tagName=tagName,tagSize=tagSize,pTag=pTag)

def VkDebugMarkerMarkerInfoEXT(sType=VK_STRUCTURE_TYPE_DEBUG_MARKER_MARKER_INFO_EXT,pNext=None,pMarkerName=None,color=None,):

    return _new('VkDebugMarkerMarkerInfoEXT', sType=sType,pNext=pNext,pMarkerName=pMarkerName,color=color)

def VkDedicatedAllocationImageCreateInfoNV(sType=VK_STRUCTURE_TYPE_DEDICATED_ALLOCATION_IMAGE_CREATE_INFO_NV,pNext=None,dedicatedAllocation=None,):

    return _new('VkDedicatedAllocationImageCreateInfoNV', sType=sType,pNext=pNext,dedicatedAllocation=dedicatedAllocation)

def VkDedicatedAllocationBufferCreateInfoNV(sType=VK_STRUCTURE_TYPE_DEDICATED_ALLOCATION_BUFFER_CREATE_INFO_NV,pNext=None,dedicatedAllocation=None,):

    return _new('VkDedicatedAllocationBufferCreateInfoNV', sType=sType,pNext=pNext,dedicatedAllocation=dedicatedAllocation)

def VkDedicatedAllocationMemoryAllocateInfoNV(sType=VK_STRUCTURE_TYPE_DEDICATED_ALLOCATION_MEMORY_ALLOCATE_INFO_NV,pNext=None,image=None,buffer=None,):

    return _new('VkDedicatedAllocationMemoryAllocateInfoNV', sType=sType,pNext=pNext,image=image,buffer=buffer)

def VkExternalImageFormatPropertiesNV(imageFormatProperties=None,externalMemoryFeatures=None,exportFromImportedHandleTypes=None,compatibleHandleTypes=None,):

    return _new('VkExternalImageFormatPropertiesNV', imageFormatProperties=imageFormatProperties,externalMemoryFeatures=externalMemoryFeatures,exportFromImportedHandleTypes=exportFromImportedHandleTypes,compatibleHandleTypes=compatibleHandleTypes)

def VkExternalMemoryImageCreateInfoNV(sType=VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_IMAGE_CREATE_INFO_NV,pNext=None,handleTypes=None,):

    return _new('VkExternalMemoryImageCreateInfoNV', sType=sType,pNext=pNext,handleTypes=handleTypes)

def VkExportMemoryAllocateInfoNV(sType=VK_STRUCTURE_TYPE_EXPORT_MEMORY_ALLOCATE_INFO_NV,pNext=None,handleTypes=None,):

    return _new('VkExportMemoryAllocateInfoNV', sType=sType,pNext=pNext,handleTypes=handleTypes)

def VkImportMemoryWin32HandleInfoNV(sType=VK_STRUCTURE_TYPE_IMPORT_MEMORY_WIN32_HANDLE_INFO_NV,pNext=None,handleType=None,handle=None,):

    return _new('VkImportMemoryWin32HandleInfoNV', sType=sType,pNext=pNext,handleType=handleType,handle=handle)

def VkExportMemoryWin32HandleInfoNV(sType=VK_STRUCTURE_TYPE_EXPORT_MEMORY_WIN32_HANDLE_INFO_NV,pNext=None,pAttributes=None,dwAccess=None,):

    return _new('VkExportMemoryWin32HandleInfoNV', sType=sType,pNext=pNext,pAttributes=pAttributes,dwAccess=dwAccess)

def VkWin32KeyedMutexAcquireReleaseInfoNV(sType=VK_STRUCTURE_TYPE_WIN32_KEYED_MUTEX_ACQUIRE_RELEASE_INFO_NV,pNext=None,acquireCount=None,pAcquireSyncs=None,pAcquireKeys=None,pAcquireTimeoutMilliseconds=None,releaseCount=None,pReleaseSyncs=None,pReleaseKeys=None,):
    if acquireCount is None and pAcquireSyncs is not None:
        acquireCount = len(pAcquireSyncs)
    if acquireCount is None and pAcquireKeys is not None:
        acquireCount = len(pAcquireKeys)
    if acquireCount is None and pAcquireTimeoutMilliseconds is not None:
        acquireCount = len(pAcquireTimeoutMilliseconds)
    if releaseCount is None and pReleaseSyncs is not None:
        releaseCount = len(pReleaseSyncs)
    if releaseCount is None and pReleaseKeys is not None:
        releaseCount = len(pReleaseKeys)

    return _new('VkWin32KeyedMutexAcquireReleaseInfoNV', sType=sType,pNext=pNext,acquireCount=acquireCount,pAcquireSyncs=pAcquireSyncs,pAcquireKeys=pAcquireKeys,pAcquireTimeoutMilliseconds=pAcquireTimeoutMilliseconds,releaseCount=releaseCount,pReleaseSyncs=pReleaseSyncs,pReleaseKeys=pReleaseKeys)

def VkDeviceGeneratedCommandsFeaturesNVX(sType=VK_STRUCTURE_TYPE_DEVICE_GENERATED_COMMANDS_FEATURES_NVX,pNext=None,computeBindingPointSupport=None,):

    return _new('VkDeviceGeneratedCommandsFeaturesNVX', sType=sType,pNext=pNext,computeBindingPointSupport=computeBindingPointSupport)

def VkDeviceGeneratedCommandsLimitsNVX(sType=VK_STRUCTURE_TYPE_DEVICE_GENERATED_COMMANDS_LIMITS_NVX,pNext=None,maxIndirectCommandsLayoutTokenCount=None,maxObjectEntryCounts=None,minSequenceCountBufferOffsetAlignment=None,minSequenceIndexBufferOffsetAlignment=None,minCommandsTokenBufferOffsetAlignment=None,):

    return _new('VkDeviceGeneratedCommandsLimitsNVX', sType=sType,pNext=pNext,maxIndirectCommandsLayoutTokenCount=maxIndirectCommandsLayoutTokenCount,maxObjectEntryCounts=maxObjectEntryCounts,minSequenceCountBufferOffsetAlignment=minSequenceCountBufferOffsetAlignment,minSequenceIndexBufferOffsetAlignment=minSequenceIndexBufferOffsetAlignment,minCommandsTokenBufferOffsetAlignment=minCommandsTokenBufferOffsetAlignment)

def VkIndirectCommandsTokenNVX(tokenType=None,buffer=None,offset=None,):

    return _new('VkIndirectCommandsTokenNVX', tokenType=tokenType,buffer=buffer,offset=offset)

def VkIndirectCommandsLayoutTokenNVX(tokenType=None,bindingUnit=None,dynamicCount=None,divisor=None,):

    return _new('VkIndirectCommandsLayoutTokenNVX', tokenType=tokenType,bindingUnit=bindingUnit,dynamicCount=dynamicCount,divisor=divisor)

def VkIndirectCommandsLayoutCreateInfoNVX(sType=VK_STRUCTURE_TYPE_INDIRECT_COMMANDS_LAYOUT_CREATE_INFO_NVX,pNext=None,pipelineBindPoint=None,flags=None,tokenCount=None,pTokens=None,):
    if tokenCount is None and pTokens is not None:
        tokenCount = len(pTokens)

    return _new('VkIndirectCommandsLayoutCreateInfoNVX', sType=sType,pNext=pNext,pipelineBindPoint=pipelineBindPoint,flags=flags,tokenCount=tokenCount,pTokens=pTokens)

def VkCmdProcessCommandsInfoNVX(sType=VK_STRUCTURE_TYPE_CMD_PROCESS_COMMANDS_INFO_NVX,pNext=None,objectTable=None,indirectCommandsLayout=None,indirectCommandsTokenCount=None,pIndirectCommandsTokens=None,maxSequencesCount=None,targetCommandBuffer=None,sequencesCountBuffer=None,sequencesCountOffset=None,sequencesIndexBuffer=None,sequencesIndexOffset=None,):
    if indirectCommandsTokenCount is None and pIndirectCommandsTokens is not None:
        indirectCommandsTokenCount = len(pIndirectCommandsTokens)

    return _new('VkCmdProcessCommandsInfoNVX', sType=sType,pNext=pNext,objectTable=objectTable,indirectCommandsLayout=indirectCommandsLayout,indirectCommandsTokenCount=indirectCommandsTokenCount,pIndirectCommandsTokens=pIndirectCommandsTokens,maxSequencesCount=maxSequencesCount,targetCommandBuffer=targetCommandBuffer,sequencesCountBuffer=sequencesCountBuffer,sequencesCountOffset=sequencesCountOffset,sequencesIndexBuffer=sequencesIndexBuffer,sequencesIndexOffset=sequencesIndexOffset)

def VkCmdReserveSpaceForCommandsInfoNVX(sType=VK_STRUCTURE_TYPE_CMD_RESERVE_SPACE_FOR_COMMANDS_INFO_NVX,pNext=None,objectTable=None,indirectCommandsLayout=None,maxSequencesCount=None,):

    return _new('VkCmdReserveSpaceForCommandsInfoNVX', sType=sType,pNext=pNext,objectTable=objectTable,indirectCommandsLayout=indirectCommandsLayout,maxSequencesCount=maxSequencesCount)

def VkObjectTableCreateInfoNVX(sType=VK_STRUCTURE_TYPE_OBJECT_TABLE_CREATE_INFO_NVX,pNext=None,objectCount=None,pObjectEntryTypes=None,pObjectEntryCounts=None,pObjectEntryUsageFlags=None,maxUniformBuffersPerDescriptor=None,maxStorageBuffersPerDescriptor=None,maxStorageImagesPerDescriptor=None,maxSampledImagesPerDescriptor=None,maxPipelineLayouts=None,):
    if objectCount is None and pObjectEntryTypes is not None:
        objectCount = len(pObjectEntryTypes)
    if objectCount is None and pObjectEntryCounts is not None:
        objectCount = len(pObjectEntryCounts)
    if objectCount is None and pObjectEntryUsageFlags is not None:
        objectCount = len(pObjectEntryUsageFlags)

    return _new('VkObjectTableCreateInfoNVX', sType=sType,pNext=pNext,objectCount=objectCount,pObjectEntryTypes=pObjectEntryTypes,pObjectEntryCounts=pObjectEntryCounts,pObjectEntryUsageFlags=pObjectEntryUsageFlags,maxUniformBuffersPerDescriptor=maxUniformBuffersPerDescriptor,maxStorageBuffersPerDescriptor=maxStorageBuffersPerDescriptor,maxStorageImagesPerDescriptor=maxStorageImagesPerDescriptor,maxSampledImagesPerDescriptor=maxSampledImagesPerDescriptor,maxPipelineLayouts=maxPipelineLayouts)

def VkObjectTableEntryNVX(type=None,flags=None,):

    return _new('VkObjectTableEntryNVX', type=type,flags=flags)

def VkObjectTablePipelineEntryNVX(type=None,flags=None,pipeline=None,):

    return _new('VkObjectTablePipelineEntryNVX', type=type,flags=flags,pipeline=pipeline)

def VkObjectTableDescriptorSetEntryNVX(type=None,flags=None,pipelineLayout=None,descriptorSet=None,):

    return _new('VkObjectTableDescriptorSetEntryNVX', type=type,flags=flags,pipelineLayout=pipelineLayout,descriptorSet=descriptorSet)

def VkObjectTableVertexBufferEntryNVX(type=None,flags=None,buffer=None,):

    return _new('VkObjectTableVertexBufferEntryNVX', type=type,flags=flags,buffer=buffer)

def VkObjectTableIndexBufferEntryNVX(type=None,flags=None,buffer=None,indexType=None,):

    return _new('VkObjectTableIndexBufferEntryNVX', type=type,flags=flags,buffer=buffer,indexType=indexType)

def VkObjectTablePushConstantEntryNVX(type=None,flags=None,pipelineLayout=None,stageFlags=None,):

    return _new('VkObjectTablePushConstantEntryNVX', type=type,flags=flags,pipelineLayout=pipelineLayout,stageFlags=stageFlags)

def VkPhysicalDeviceFeatures2(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FEATURES_2,pNext=None,features=None,):

    return _new('VkPhysicalDeviceFeatures2', sType=sType,pNext=pNext,features=features)

def VkPhysicalDeviceProperties2(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROPERTIES_2,pNext=None,properties=None,):

    return _new('VkPhysicalDeviceProperties2', sType=sType,pNext=pNext,properties=properties)

def VkFormatProperties2(sType=VK_STRUCTURE_TYPE_FORMAT_PROPERTIES_2,pNext=None,formatProperties=None,):

    return _new('VkFormatProperties2', sType=sType,pNext=pNext,formatProperties=formatProperties)

def VkImageFormatProperties2(sType=VK_STRUCTURE_TYPE_IMAGE_FORMAT_PROPERTIES_2,pNext=None,imageFormatProperties=None,):

    return _new('VkImageFormatProperties2', sType=sType,pNext=pNext,imageFormatProperties=imageFormatProperties)

def VkPhysicalDeviceImageFormatInfo2(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_FORMAT_INFO_2,pNext=None,format=None,type=None,tiling=None,usage=None,flags=None,):

    return _new('VkPhysicalDeviceImageFormatInfo2', sType=sType,pNext=pNext,format=format,type=type,tiling=tiling,usage=usage,flags=flags)

def VkQueueFamilyProperties2(sType=VK_STRUCTURE_TYPE_QUEUE_FAMILY_PROPERTIES_2,pNext=None,queueFamilyProperties=None,):

    return _new('VkQueueFamilyProperties2', sType=sType,pNext=pNext,queueFamilyProperties=queueFamilyProperties)

def VkPhysicalDeviceMemoryProperties2(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_PROPERTIES_2,pNext=None,memoryProperties=None,):

    return _new('VkPhysicalDeviceMemoryProperties2', sType=sType,pNext=pNext,memoryProperties=memoryProperties)

def VkSparseImageFormatProperties2(sType=VK_STRUCTURE_TYPE_SPARSE_IMAGE_FORMAT_PROPERTIES_2,pNext=None,properties=None,):

    return _new('VkSparseImageFormatProperties2', sType=sType,pNext=pNext,properties=properties)

def VkPhysicalDeviceSparseImageFormatInfo2(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SPARSE_IMAGE_FORMAT_INFO_2,pNext=None,format=None,type=None,samples=None,usage=None,tiling=None,):

    return _new('VkPhysicalDeviceSparseImageFormatInfo2', sType=sType,pNext=pNext,format=format,type=type,samples=samples,usage=usage,tiling=tiling)

def VkPhysicalDevicePushDescriptorPropertiesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PUSH_DESCRIPTOR_PROPERTIES_KHR,pNext=None,maxPushDescriptors=None,):

    return _new('VkPhysicalDevicePushDescriptorPropertiesKHR', sType=sType,pNext=pNext,maxPushDescriptors=maxPushDescriptors)

def VkConformanceVersionKHR(major=None,minor=None,subminor=None,patch=None,):

    return _new('VkConformanceVersionKHR', major=major,minor=minor,subminor=subminor,patch=patch)

def VkPhysicalDeviceDriverPropertiesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DRIVER_PROPERTIES_KHR,pNext=None,driverID=None,driverName=None,driverInfo=None,conformanceVersion=None,):

    return _new('VkPhysicalDeviceDriverPropertiesKHR', sType=sType,pNext=pNext,driverID=driverID,driverName=driverName,driverInfo=driverInfo,conformanceVersion=conformanceVersion)

def VkPresentRegionsKHR(sType=VK_STRUCTURE_TYPE_PRESENT_REGIONS_KHR,pNext=None,swapchainCount=None,pRegions=None,):
    if swapchainCount is None and pRegions is not None:
        swapchainCount = len(pRegions)

    return _new('VkPresentRegionsKHR', sType=sType,pNext=pNext,swapchainCount=swapchainCount,pRegions=pRegions)

def VkPresentRegionKHR(rectangleCount=None,pRectangles=None,):
    if rectangleCount is None and pRectangles is not None:
        rectangleCount = len(pRectangles)

    return _new('VkPresentRegionKHR', rectangleCount=rectangleCount,pRectangles=pRectangles)

def VkRectLayerKHR(offset=None,extent=None,layer=None,):

    return _new('VkRectLayerKHR', offset=offset,extent=extent,layer=layer)

def VkPhysicalDeviceVariablePointerFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VARIABLE_POINTER_FEATURES,pNext=None,variablePointersStorageBuffer=None,variablePointers=None,):

    return _new('VkPhysicalDeviceVariablePointerFeatures', sType=sType,pNext=pNext,variablePointersStorageBuffer=variablePointersStorageBuffer,variablePointers=variablePointers)

def VkExternalMemoryProperties(externalMemoryFeatures=None,exportFromImportedHandleTypes=None,compatibleHandleTypes=None,):

    return _new('VkExternalMemoryProperties', externalMemoryFeatures=externalMemoryFeatures,exportFromImportedHandleTypes=exportFromImportedHandleTypes,compatibleHandleTypes=compatibleHandleTypes)

def VkPhysicalDeviceExternalImageFormatInfo(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_IMAGE_FORMAT_INFO,pNext=None,handleType=None,):

    return _new('VkPhysicalDeviceExternalImageFormatInfo', sType=sType,pNext=pNext,handleType=handleType)

def VkExternalImageFormatProperties(sType=VK_STRUCTURE_TYPE_EXTERNAL_IMAGE_FORMAT_PROPERTIES,pNext=None,externalMemoryProperties=None,):

    return _new('VkExternalImageFormatProperties', sType=sType,pNext=pNext,externalMemoryProperties=externalMemoryProperties)

def VkPhysicalDeviceExternalBufferInfo(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_BUFFER_INFO,pNext=None,flags=None,usage=None,handleType=None,):

    return _new('VkPhysicalDeviceExternalBufferInfo', sType=sType,pNext=pNext,flags=flags,usage=usage,handleType=handleType)

def VkExternalBufferProperties(sType=VK_STRUCTURE_TYPE_EXTERNAL_BUFFER_PROPERTIES,pNext=None,externalMemoryProperties=None,):

    return _new('VkExternalBufferProperties', sType=sType,pNext=pNext,externalMemoryProperties=externalMemoryProperties)

def VkPhysicalDeviceIDProperties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ID_PROPERTIES,pNext=None,deviceUUID=None,driverUUID=None,deviceLUID=None,deviceNodeMask=None,deviceLUIDValid=None,):

    return _new('VkPhysicalDeviceIDProperties', sType=sType,pNext=pNext,deviceUUID=deviceUUID,driverUUID=driverUUID,deviceLUID=deviceLUID,deviceNodeMask=deviceNodeMask,deviceLUIDValid=deviceLUIDValid)

def VkExternalMemoryImageCreateInfo(sType=VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_IMAGE_CREATE_INFO,pNext=None,handleTypes=None,):

    return _new('VkExternalMemoryImageCreateInfo', sType=sType,pNext=pNext,handleTypes=handleTypes)

def VkExternalMemoryBufferCreateInfo(sType=VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_BUFFER_CREATE_INFO,pNext=None,handleTypes=None,):

    return _new('VkExternalMemoryBufferCreateInfo', sType=sType,pNext=pNext,handleTypes=handleTypes)

def VkExportMemoryAllocateInfo(sType=VK_STRUCTURE_TYPE_EXPORT_MEMORY_ALLOCATE_INFO,pNext=None,handleTypes=None,):

    return _new('VkExportMemoryAllocateInfo', sType=sType,pNext=pNext,handleTypes=handleTypes)

def VkImportMemoryWin32HandleInfoKHR(sType=VK_STRUCTURE_TYPE_IMPORT_MEMORY_WIN32_HANDLE_INFO_KHR,pNext=None,handleType=None,handle=None,name=None,):

    return _new('VkImportMemoryWin32HandleInfoKHR', sType=sType,pNext=pNext,handleType=handleType,handle=handle,name=name)

def VkExportMemoryWin32HandleInfoKHR(sType=VK_STRUCTURE_TYPE_EXPORT_MEMORY_WIN32_HANDLE_INFO_KHR,pNext=None,pAttributes=None,dwAccess=None,name=None,):

    return _new('VkExportMemoryWin32HandleInfoKHR', sType=sType,pNext=pNext,pAttributes=pAttributes,dwAccess=dwAccess,name=name)

def VkMemoryWin32HandlePropertiesKHR(sType=VK_STRUCTURE_TYPE_MEMORY_WIN32_HANDLE_PROPERTIES_KHR,pNext=None,memoryTypeBits=None,):

    return _new('VkMemoryWin32HandlePropertiesKHR', sType=sType,pNext=pNext,memoryTypeBits=memoryTypeBits)

def VkMemoryGetWin32HandleInfoKHR(sType=VK_STRUCTURE_TYPE_MEMORY_GET_WIN32_HANDLE_INFO_KHR,pNext=None,memory=None,handleType=None,):

    return _new('VkMemoryGetWin32HandleInfoKHR', sType=sType,pNext=pNext,memory=memory,handleType=handleType)

def VkImportMemoryFdInfoKHR(sType=VK_STRUCTURE_TYPE_IMPORT_MEMORY_FD_INFO_KHR,pNext=None,handleType=None,fd=None,):

    return _new('VkImportMemoryFdInfoKHR', sType=sType,pNext=pNext,handleType=handleType,fd=fd)

def VkMemoryFdPropertiesKHR(sType=VK_STRUCTURE_TYPE_MEMORY_FD_PROPERTIES_KHR,pNext=None,memoryTypeBits=None,):

    return _new('VkMemoryFdPropertiesKHR', sType=sType,pNext=pNext,memoryTypeBits=memoryTypeBits)

def VkMemoryGetFdInfoKHR(sType=VK_STRUCTURE_TYPE_MEMORY_GET_FD_INFO_KHR,pNext=None,memory=None,handleType=None,):

    return _new('VkMemoryGetFdInfoKHR', sType=sType,pNext=pNext,memory=memory,handleType=handleType)

def VkWin32KeyedMutexAcquireReleaseInfoKHR(sType=VK_STRUCTURE_TYPE_WIN32_KEYED_MUTEX_ACQUIRE_RELEASE_INFO_KHR,pNext=None,acquireCount=None,pAcquireSyncs=None,pAcquireKeys=None,pAcquireTimeouts=None,releaseCount=None,pReleaseSyncs=None,pReleaseKeys=None,):
    if acquireCount is None and pAcquireSyncs is not None:
        acquireCount = len(pAcquireSyncs)
    if acquireCount is None and pAcquireKeys is not None:
        acquireCount = len(pAcquireKeys)
    if acquireCount is None and pAcquireTimeouts is not None:
        acquireCount = len(pAcquireTimeouts)
    if releaseCount is None and pReleaseSyncs is not None:
        releaseCount = len(pReleaseSyncs)
    if releaseCount is None and pReleaseKeys is not None:
        releaseCount = len(pReleaseKeys)

    return _new('VkWin32KeyedMutexAcquireReleaseInfoKHR', sType=sType,pNext=pNext,acquireCount=acquireCount,pAcquireSyncs=pAcquireSyncs,pAcquireKeys=pAcquireKeys,pAcquireTimeouts=pAcquireTimeouts,releaseCount=releaseCount,pReleaseSyncs=pReleaseSyncs,pReleaseKeys=pReleaseKeys)

def VkPhysicalDeviceExternalSemaphoreInfo(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_SEMAPHORE_INFO,pNext=None,handleType=None,):

    return _new('VkPhysicalDeviceExternalSemaphoreInfo', sType=sType,pNext=pNext,handleType=handleType)

def VkExternalSemaphoreProperties(sType=VK_STRUCTURE_TYPE_EXTERNAL_SEMAPHORE_PROPERTIES,pNext=None,exportFromImportedHandleTypes=None,compatibleHandleTypes=None,externalSemaphoreFeatures=None,):

    return _new('VkExternalSemaphoreProperties', sType=sType,pNext=pNext,exportFromImportedHandleTypes=exportFromImportedHandleTypes,compatibleHandleTypes=compatibleHandleTypes,externalSemaphoreFeatures=externalSemaphoreFeatures)

def VkExportSemaphoreCreateInfo(sType=VK_STRUCTURE_TYPE_EXPORT_SEMAPHORE_CREATE_INFO,pNext=None,handleTypes=None,):

    return _new('VkExportSemaphoreCreateInfo', sType=sType,pNext=pNext,handleTypes=handleTypes)

def VkImportSemaphoreWin32HandleInfoKHR(sType=VK_STRUCTURE_TYPE_IMPORT_SEMAPHORE_WIN32_HANDLE_INFO_KHR,pNext=None,semaphore=None,flags=None,handleType=None,handle=None,name=None,):

    return _new('VkImportSemaphoreWin32HandleInfoKHR', sType=sType,pNext=pNext,semaphore=semaphore,flags=flags,handleType=handleType,handle=handle,name=name)

def VkExportSemaphoreWin32HandleInfoKHR(sType=VK_STRUCTURE_TYPE_EXPORT_SEMAPHORE_WIN32_HANDLE_INFO_KHR,pNext=None,pAttributes=None,dwAccess=None,name=None,):

    return _new('VkExportSemaphoreWin32HandleInfoKHR', sType=sType,pNext=pNext,pAttributes=pAttributes,dwAccess=dwAccess,name=name)

def VkD3D12FenceSubmitInfoKHR(sType=VK_STRUCTURE_TYPE_D3D12_FENCE_SUBMIT_INFO_KHR,pNext=None,waitSemaphoreValuesCount=None,pWaitSemaphoreValues=None,signalSemaphoreValuesCount=None,pSignalSemaphoreValues=None,):
    if waitSemaphoreValuesCount is None and pWaitSemaphoreValues is not None:
        waitSemaphoreValuesCount = len(pWaitSemaphoreValues)
    if signalSemaphoreValuesCount is None and pSignalSemaphoreValues is not None:
        signalSemaphoreValuesCount = len(pSignalSemaphoreValues)

    return _new('VkD3D12FenceSubmitInfoKHR', sType=sType,pNext=pNext,waitSemaphoreValuesCount=waitSemaphoreValuesCount,pWaitSemaphoreValues=pWaitSemaphoreValues,signalSemaphoreValuesCount=signalSemaphoreValuesCount,pSignalSemaphoreValues=pSignalSemaphoreValues)

def VkSemaphoreGetWin32HandleInfoKHR(sType=VK_STRUCTURE_TYPE_SEMAPHORE_GET_WIN32_HANDLE_INFO_KHR,pNext=None,semaphore=None,handleType=None,):

    return _new('VkSemaphoreGetWin32HandleInfoKHR', sType=sType,pNext=pNext,semaphore=semaphore,handleType=handleType)

def VkImportSemaphoreFdInfoKHR(sType=VK_STRUCTURE_TYPE_IMPORT_SEMAPHORE_FD_INFO_KHR,pNext=None,semaphore=None,flags=None,handleType=None,fd=None,):

    return _new('VkImportSemaphoreFdInfoKHR', sType=sType,pNext=pNext,semaphore=semaphore,flags=flags,handleType=handleType,fd=fd)

def VkSemaphoreGetFdInfoKHR(sType=VK_STRUCTURE_TYPE_SEMAPHORE_GET_FD_INFO_KHR,pNext=None,semaphore=None,handleType=None,):

    return _new('VkSemaphoreGetFdInfoKHR', sType=sType,pNext=pNext,semaphore=semaphore,handleType=handleType)

def VkPhysicalDeviceExternalFenceInfo(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_FENCE_INFO,pNext=None,handleType=None,):

    return _new('VkPhysicalDeviceExternalFenceInfo', sType=sType,pNext=pNext,handleType=handleType)

def VkExternalFenceProperties(sType=VK_STRUCTURE_TYPE_EXTERNAL_FENCE_PROPERTIES,pNext=None,exportFromImportedHandleTypes=None,compatibleHandleTypes=None,externalFenceFeatures=None,):

    return _new('VkExternalFenceProperties', sType=sType,pNext=pNext,exportFromImportedHandleTypes=exportFromImportedHandleTypes,compatibleHandleTypes=compatibleHandleTypes,externalFenceFeatures=externalFenceFeatures)

def VkExportFenceCreateInfo(sType=VK_STRUCTURE_TYPE_EXPORT_FENCE_CREATE_INFO,pNext=None,handleTypes=None,):

    return _new('VkExportFenceCreateInfo', sType=sType,pNext=pNext,handleTypes=handleTypes)

def VkImportFenceWin32HandleInfoKHR(sType=VK_STRUCTURE_TYPE_IMPORT_FENCE_WIN32_HANDLE_INFO_KHR,pNext=None,fence=None,flags=None,handleType=None,handle=None,name=None,):

    return _new('VkImportFenceWin32HandleInfoKHR', sType=sType,pNext=pNext,fence=fence,flags=flags,handleType=handleType,handle=handle,name=name)

def VkExportFenceWin32HandleInfoKHR(sType=VK_STRUCTURE_TYPE_EXPORT_FENCE_WIN32_HANDLE_INFO_KHR,pNext=None,pAttributes=None,dwAccess=None,name=None,):

    return _new('VkExportFenceWin32HandleInfoKHR', sType=sType,pNext=pNext,pAttributes=pAttributes,dwAccess=dwAccess,name=name)

def VkFenceGetWin32HandleInfoKHR(sType=VK_STRUCTURE_TYPE_FENCE_GET_WIN32_HANDLE_INFO_KHR,pNext=None,fence=None,handleType=None,):

    return _new('VkFenceGetWin32HandleInfoKHR', sType=sType,pNext=pNext,fence=fence,handleType=handleType)

def VkImportFenceFdInfoKHR(sType=VK_STRUCTURE_TYPE_IMPORT_FENCE_FD_INFO_KHR,pNext=None,fence=None,flags=None,handleType=None,fd=None,):

    return _new('VkImportFenceFdInfoKHR', sType=sType,pNext=pNext,fence=fence,flags=flags,handleType=handleType,fd=fd)

def VkFenceGetFdInfoKHR(sType=VK_STRUCTURE_TYPE_FENCE_GET_FD_INFO_KHR,pNext=None,fence=None,handleType=None,):

    return _new('VkFenceGetFdInfoKHR', sType=sType,pNext=pNext,fence=fence,handleType=handleType)

def VkPhysicalDeviceMultiviewFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_FEATURES,pNext=None,multiview=None,multiviewGeometryShader=None,multiviewTessellationShader=None,):

    return _new('VkPhysicalDeviceMultiviewFeatures', sType=sType,pNext=pNext,multiview=multiview,multiviewGeometryShader=multiviewGeometryShader,multiviewTessellationShader=multiviewTessellationShader)

def VkPhysicalDeviceMultiviewProperties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_PROPERTIES,pNext=None,maxMultiviewViewCount=None,maxMultiviewInstanceIndex=None,):

    return _new('VkPhysicalDeviceMultiviewProperties', sType=sType,pNext=pNext,maxMultiviewViewCount=maxMultiviewViewCount,maxMultiviewInstanceIndex=maxMultiviewInstanceIndex)

def VkRenderPassMultiviewCreateInfo(sType=VK_STRUCTURE_TYPE_RENDER_PASS_MULTIVIEW_CREATE_INFO,pNext=None,subpassCount=None,pViewMasks=None,dependencyCount=None,pViewOffsets=None,correlationMaskCount=None,pCorrelationMasks=None,):
    if subpassCount is None and pViewMasks is not None:
        subpassCount = len(pViewMasks)
    if dependencyCount is None and pViewOffsets is not None:
        dependencyCount = len(pViewOffsets)
    if correlationMaskCount is None and pCorrelationMasks is not None:
        correlationMaskCount = len(pCorrelationMasks)

    return _new('VkRenderPassMultiviewCreateInfo', sType=sType,pNext=pNext,subpassCount=subpassCount,pViewMasks=pViewMasks,dependencyCount=dependencyCount,pViewOffsets=pViewOffsets,correlationMaskCount=correlationMaskCount,pCorrelationMasks=pCorrelationMasks)

def VkSurfaceCapabilities2EXT(sType=VK_STRUCTURE_TYPE_SURFACE_CAPABILITIES_2_EXT,pNext=None,minImageCount=None,maxImageCount=None,currentExtent=None,minImageExtent=None,maxImageExtent=None,maxImageArrayLayers=None,supportedTransforms=None,currentTransform=None,supportedCompositeAlpha=None,supportedUsageFlags=None,supportedSurfaceCounters=None,):

    return _new('VkSurfaceCapabilities2EXT', sType=sType,pNext=pNext,minImageCount=minImageCount,maxImageCount=maxImageCount,currentExtent=currentExtent,minImageExtent=minImageExtent,maxImageExtent=maxImageExtent,maxImageArrayLayers=maxImageArrayLayers,supportedTransforms=supportedTransforms,currentTransform=currentTransform,supportedCompositeAlpha=supportedCompositeAlpha,supportedUsageFlags=supportedUsageFlags,supportedSurfaceCounters=supportedSurfaceCounters)

def VkDisplayPowerInfoEXT(sType=VK_STRUCTURE_TYPE_DISPLAY_POWER_INFO_EXT,pNext=None,powerState=None,):

    return _new('VkDisplayPowerInfoEXT', sType=sType,pNext=pNext,powerState=powerState)

def VkDeviceEventInfoEXT(sType=VK_STRUCTURE_TYPE_DEVICE_EVENT_INFO_EXT,pNext=None,deviceEvent=None,):

    return _new('VkDeviceEventInfoEXT', sType=sType,pNext=pNext,deviceEvent=deviceEvent)

def VkDisplayEventInfoEXT(sType=VK_STRUCTURE_TYPE_DISPLAY_EVENT_INFO_EXT,pNext=None,displayEvent=None,):

    return _new('VkDisplayEventInfoEXT', sType=sType,pNext=pNext,displayEvent=displayEvent)

def VkSwapchainCounterCreateInfoEXT(sType=VK_STRUCTURE_TYPE_SWAPCHAIN_COUNTER_CREATE_INFO_EXT,pNext=None,surfaceCounters=None,):

    return _new('VkSwapchainCounterCreateInfoEXT', sType=sType,pNext=pNext,surfaceCounters=surfaceCounters)

def VkPhysicalDeviceGroupProperties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_GROUP_PROPERTIES,pNext=None,physicalDeviceCount=None,physicalDevices=None,subsetAllocation=None,):

    return _new('VkPhysicalDeviceGroupProperties', sType=sType,pNext=pNext,physicalDeviceCount=physicalDeviceCount,physicalDevices=physicalDevices,subsetAllocation=subsetAllocation)

def VkMemoryAllocateFlagsInfo(sType=VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_FLAGS_INFO,pNext=None,flags=None,deviceMask=None,):

    return _new('VkMemoryAllocateFlagsInfo', sType=sType,pNext=pNext,flags=flags,deviceMask=deviceMask)

def VkBindBufferMemoryInfo(sType=VK_STRUCTURE_TYPE_BIND_BUFFER_MEMORY_INFO,pNext=None,buffer=None,memory=None,memoryOffset=None,):

    return _new('VkBindBufferMemoryInfo', sType=sType,pNext=pNext,buffer=buffer,memory=memory,memoryOffset=memoryOffset)

def VkBindBufferMemoryDeviceGroupInfo(sType=VK_STRUCTURE_TYPE_BIND_BUFFER_MEMORY_DEVICE_GROUP_INFO,pNext=None,deviceIndexCount=None,pDeviceIndices=None,):
    if deviceIndexCount is None and pDeviceIndices is not None:
        deviceIndexCount = len(pDeviceIndices)

    return _new('VkBindBufferMemoryDeviceGroupInfo', sType=sType,pNext=pNext,deviceIndexCount=deviceIndexCount,pDeviceIndices=pDeviceIndices)

def VkBindImageMemoryInfo(sType=VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_INFO,pNext=None,image=None,memory=None,memoryOffset=None,):

    return _new('VkBindImageMemoryInfo', sType=sType,pNext=pNext,image=image,memory=memory,memoryOffset=memoryOffset)

def VkBindImageMemoryDeviceGroupInfo(sType=VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_DEVICE_GROUP_INFO,pNext=None,deviceIndexCount=None,pDeviceIndices=None,splitInstanceBindRegionCount=None,pSplitInstanceBindRegions=None,):
    if deviceIndexCount is None and pDeviceIndices is not None:
        deviceIndexCount = len(pDeviceIndices)
    if splitInstanceBindRegionCount is None and pSplitInstanceBindRegions is not None:
        splitInstanceBindRegionCount = len(pSplitInstanceBindRegions)

    return _new('VkBindImageMemoryDeviceGroupInfo', sType=sType,pNext=pNext,deviceIndexCount=deviceIndexCount,pDeviceIndices=pDeviceIndices,splitInstanceBindRegionCount=splitInstanceBindRegionCount,pSplitInstanceBindRegions=pSplitInstanceBindRegions)

def VkDeviceGroupRenderPassBeginInfo(sType=VK_STRUCTURE_TYPE_DEVICE_GROUP_RENDER_PASS_BEGIN_INFO,pNext=None,deviceMask=None,deviceRenderAreaCount=None,pDeviceRenderAreas=None,):
    if deviceRenderAreaCount is None and pDeviceRenderAreas is not None:
        deviceRenderAreaCount = len(pDeviceRenderAreas)

    return _new('VkDeviceGroupRenderPassBeginInfo', sType=sType,pNext=pNext,deviceMask=deviceMask,deviceRenderAreaCount=deviceRenderAreaCount,pDeviceRenderAreas=pDeviceRenderAreas)

def VkDeviceGroupCommandBufferBeginInfo(sType=VK_STRUCTURE_TYPE_DEVICE_GROUP_COMMAND_BUFFER_BEGIN_INFO,pNext=None,deviceMask=None,):

    return _new('VkDeviceGroupCommandBufferBeginInfo', sType=sType,pNext=pNext,deviceMask=deviceMask)

def VkDeviceGroupSubmitInfo(sType=VK_STRUCTURE_TYPE_DEVICE_GROUP_SUBMIT_INFO,pNext=None,waitSemaphoreCount=None,pWaitSemaphoreDeviceIndices=None,commandBufferCount=None,pCommandBufferDeviceMasks=None,signalSemaphoreCount=None,pSignalSemaphoreDeviceIndices=None,):
    if waitSemaphoreCount is None and pWaitSemaphoreDeviceIndices is not None:
        waitSemaphoreCount = len(pWaitSemaphoreDeviceIndices)
    if commandBufferCount is None and pCommandBufferDeviceMasks is not None:
        commandBufferCount = len(pCommandBufferDeviceMasks)
    if signalSemaphoreCount is None and pSignalSemaphoreDeviceIndices is not None:
        signalSemaphoreCount = len(pSignalSemaphoreDeviceIndices)

    return _new('VkDeviceGroupSubmitInfo', sType=sType,pNext=pNext,waitSemaphoreCount=waitSemaphoreCount,pWaitSemaphoreDeviceIndices=pWaitSemaphoreDeviceIndices,commandBufferCount=commandBufferCount,pCommandBufferDeviceMasks=pCommandBufferDeviceMasks,signalSemaphoreCount=signalSemaphoreCount,pSignalSemaphoreDeviceIndices=pSignalSemaphoreDeviceIndices)

def VkDeviceGroupBindSparseInfo(sType=VK_STRUCTURE_TYPE_DEVICE_GROUP_BIND_SPARSE_INFO,pNext=None,resourceDeviceIndex=None,memoryDeviceIndex=None,):

    return _new('VkDeviceGroupBindSparseInfo', sType=sType,pNext=pNext,resourceDeviceIndex=resourceDeviceIndex,memoryDeviceIndex=memoryDeviceIndex)

def VkDeviceGroupPresentCapabilitiesKHR(sType=VK_STRUCTURE_TYPE_DEVICE_GROUP_PRESENT_CAPABILITIES_KHR,pNext=None,presentMask=None,modes=None,):

    return _new('VkDeviceGroupPresentCapabilitiesKHR', sType=sType,pNext=pNext,presentMask=presentMask,modes=modes)

def VkImageSwapchainCreateInfoKHR(sType=VK_STRUCTURE_TYPE_IMAGE_SWAPCHAIN_CREATE_INFO_KHR,pNext=None,swapchain=None,):

    return _new('VkImageSwapchainCreateInfoKHR', sType=sType,pNext=pNext,swapchain=swapchain)

def VkBindImageMemorySwapchainInfoKHR(sType=VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_SWAPCHAIN_INFO_KHR,pNext=None,swapchain=None,imageIndex=None,):

    return _new('VkBindImageMemorySwapchainInfoKHR', sType=sType,pNext=pNext,swapchain=swapchain,imageIndex=imageIndex)

def VkAcquireNextImageInfoKHR(sType=VK_STRUCTURE_TYPE_ACQUIRE_NEXT_IMAGE_INFO_KHR,pNext=None,swapchain=None,timeout=None,semaphore=None,fence=None,deviceMask=None,):

    return _new('VkAcquireNextImageInfoKHR', sType=sType,pNext=pNext,swapchain=swapchain,timeout=timeout,semaphore=semaphore,fence=fence,deviceMask=deviceMask)

def VkDeviceGroupPresentInfoKHR(sType=VK_STRUCTURE_TYPE_DEVICE_GROUP_PRESENT_INFO_KHR,pNext=None,swapchainCount=None,pDeviceMasks=None,mode=None,):
    if swapchainCount is None and pDeviceMasks is not None:
        swapchainCount = len(pDeviceMasks)

    return _new('VkDeviceGroupPresentInfoKHR', sType=sType,pNext=pNext,swapchainCount=swapchainCount,pDeviceMasks=pDeviceMasks,mode=mode)

def VkDeviceGroupDeviceCreateInfo(sType=VK_STRUCTURE_TYPE_DEVICE_GROUP_DEVICE_CREATE_INFO,pNext=None,physicalDeviceCount=None,pPhysicalDevices=None,):
    if physicalDeviceCount is None and pPhysicalDevices is not None:
        physicalDeviceCount = len(pPhysicalDevices)

    return _new('VkDeviceGroupDeviceCreateInfo', sType=sType,pNext=pNext,physicalDeviceCount=physicalDeviceCount,pPhysicalDevices=pPhysicalDevices)

def VkDeviceGroupSwapchainCreateInfoKHR(sType=VK_STRUCTURE_TYPE_DEVICE_GROUP_SWAPCHAIN_CREATE_INFO_KHR,pNext=None,modes=None,):

    return _new('VkDeviceGroupSwapchainCreateInfoKHR', sType=sType,pNext=pNext,modes=modes)

def VkDescriptorUpdateTemplateEntry(dstBinding=None,dstArrayElement=None,descriptorCount=None,descriptorType=None,offset=None,stride=None,):

    return _new('VkDescriptorUpdateTemplateEntry', dstBinding=dstBinding,dstArrayElement=dstArrayElement,descriptorCount=descriptorCount,descriptorType=descriptorType,offset=offset,stride=stride)

def VkDescriptorUpdateTemplateCreateInfo(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_UPDATE_TEMPLATE_CREATE_INFO,pNext=None,flags=None,descriptorUpdateEntryCount=None,pDescriptorUpdateEntries=None,templateType=None,descriptorSetLayout=None,pipelineBindPoint=None,pipelineLayout=None,set=None,):
    if descriptorUpdateEntryCount is None and pDescriptorUpdateEntries is not None:
        descriptorUpdateEntryCount = len(pDescriptorUpdateEntries)

    return _new('VkDescriptorUpdateTemplateCreateInfo', sType=sType,pNext=pNext,flags=flags,descriptorUpdateEntryCount=descriptorUpdateEntryCount,pDescriptorUpdateEntries=pDescriptorUpdateEntries,templateType=templateType,descriptorSetLayout=descriptorSetLayout,pipelineBindPoint=pipelineBindPoint,pipelineLayout=pipelineLayout,set=set)

def VkXYColorEXT(x=None,y=None,):

    return _new('VkXYColorEXT', x=x,y=y)

def VkHdrMetadataEXT(sType=VK_STRUCTURE_TYPE_HDR_METADATA_EXT,pNext=None,displayPrimaryRed=None,displayPrimaryGreen=None,displayPrimaryBlue=None,whitePoint=None,maxLuminance=None,minLuminance=None,maxContentLightLevel=None,maxFrameAverageLightLevel=None,):

    return _new('VkHdrMetadataEXT', sType=sType,pNext=pNext,displayPrimaryRed=displayPrimaryRed,displayPrimaryGreen=displayPrimaryGreen,displayPrimaryBlue=displayPrimaryBlue,whitePoint=whitePoint,maxLuminance=maxLuminance,minLuminance=minLuminance,maxContentLightLevel=maxContentLightLevel,maxFrameAverageLightLevel=maxFrameAverageLightLevel)

def VkRefreshCycleDurationGOOGLE(refreshDuration=None,):

    return _new('VkRefreshCycleDurationGOOGLE', refreshDuration=refreshDuration)

def VkPastPresentationTimingGOOGLE(presentID=None,desiredPresentTime=None,actualPresentTime=None,earliestPresentTime=None,presentMargin=None,):

    return _new('VkPastPresentationTimingGOOGLE', presentID=presentID,desiredPresentTime=desiredPresentTime,actualPresentTime=actualPresentTime,earliestPresentTime=earliestPresentTime,presentMargin=presentMargin)

def VkPresentTimesInfoGOOGLE(sType=VK_STRUCTURE_TYPE_PRESENT_TIMES_INFO_GOOGLE,pNext=None,swapchainCount=None,pTimes=None,):
    if swapchainCount is None and pTimes is not None:
        swapchainCount = len(pTimes)

    return _new('VkPresentTimesInfoGOOGLE', sType=sType,pNext=pNext,swapchainCount=swapchainCount,pTimes=pTimes)

def VkPresentTimeGOOGLE(presentID=None,desiredPresentTime=None,):

    return _new('VkPresentTimeGOOGLE', presentID=presentID,desiredPresentTime=desiredPresentTime)

def VkIOSSurfaceCreateInfoMVK(sType=VK_STRUCTURE_TYPE_IOS_SURFACE_CREATE_INFO_MVK,pNext=None,flags=None,pView=None,):

    return _new('VkIOSSurfaceCreateInfoMVK', sType=sType,pNext=pNext,flags=flags,pView=pView)

def VkMacOSSurfaceCreateInfoMVK(sType=VK_STRUCTURE_TYPE_MACOS_SURFACE_CREATE_INFO_MVK,pNext=None,flags=None,pView=None,):

    return _new('VkMacOSSurfaceCreateInfoMVK', sType=sType,pNext=pNext,flags=flags,pView=pView)

def VkViewportWScalingNV(xcoeff=None,ycoeff=None,):

    return _new('VkViewportWScalingNV', xcoeff=xcoeff,ycoeff=ycoeff)

def VkPipelineViewportWScalingStateCreateInfoNV(sType=VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_W_SCALING_STATE_CREATE_INFO_NV,pNext=None,viewportWScalingEnable=None,viewportCount=None,pViewportWScalings=None,):
    if viewportCount is None and pViewportWScalings is not None:
        viewportCount = len(pViewportWScalings)

    return _new('VkPipelineViewportWScalingStateCreateInfoNV', sType=sType,pNext=pNext,viewportWScalingEnable=viewportWScalingEnable,viewportCount=viewportCount,pViewportWScalings=pViewportWScalings)

def VkViewportSwizzleNV(x=None,y=None,z=None,w=None,):

    return _new('VkViewportSwizzleNV', x=x,y=y,z=z,w=w)

def VkPipelineViewportSwizzleStateCreateInfoNV(sType=VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_SWIZZLE_STATE_CREATE_INFO_NV,pNext=None,flags=None,viewportCount=None,pViewportSwizzles=None,):
    if viewportCount is None and pViewportSwizzles is not None:
        viewportCount = len(pViewportSwizzles)

    return _new('VkPipelineViewportSwizzleStateCreateInfoNV', sType=sType,pNext=pNext,flags=flags,viewportCount=viewportCount,pViewportSwizzles=pViewportSwizzles)

def VkPhysicalDeviceDiscardRectanglePropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DISCARD_RECTANGLE_PROPERTIES_EXT,pNext=None,maxDiscardRectangles=None,):

    return _new('VkPhysicalDeviceDiscardRectanglePropertiesEXT', sType=sType,pNext=pNext,maxDiscardRectangles=maxDiscardRectangles)

def VkPipelineDiscardRectangleStateCreateInfoEXT(sType=VK_STRUCTURE_TYPE_PIPELINE_DISCARD_RECTANGLE_STATE_CREATE_INFO_EXT,pNext=None,flags=None,discardRectangleMode=None,discardRectangleCount=None,pDiscardRectangles=None,):
    if discardRectangleCount is None and pDiscardRectangles is not None:
        discardRectangleCount = len(pDiscardRectangles)

    return _new('VkPipelineDiscardRectangleStateCreateInfoEXT', sType=sType,pNext=pNext,flags=flags,discardRectangleMode=discardRectangleMode,discardRectangleCount=discardRectangleCount,pDiscardRectangles=pDiscardRectangles)

def VkPhysicalDeviceMultiviewPerViewAttributesPropertiesNVX(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_PER_VIEW_ATTRIBUTES_PROPERTIES_NVX,pNext=None,perViewPositionAllComponents=None,):

    return _new('VkPhysicalDeviceMultiviewPerViewAttributesPropertiesNVX', sType=sType,pNext=pNext,perViewPositionAllComponents=perViewPositionAllComponents)

def VkInputAttachmentAspectReference(subpass=None,inputAttachmentIndex=None,aspectMask=None,):

    return _new('VkInputAttachmentAspectReference', subpass=subpass,inputAttachmentIndex=inputAttachmentIndex,aspectMask=aspectMask)

def VkRenderPassInputAttachmentAspectCreateInfo(sType=VK_STRUCTURE_TYPE_RENDER_PASS_INPUT_ATTACHMENT_ASPECT_CREATE_INFO,pNext=None,aspectReferenceCount=None,pAspectReferences=None,):
    if aspectReferenceCount is None and pAspectReferences is not None:
        aspectReferenceCount = len(pAspectReferences)

    return _new('VkRenderPassInputAttachmentAspectCreateInfo', sType=sType,pNext=pNext,aspectReferenceCount=aspectReferenceCount,pAspectReferences=pAspectReferences)

def VkPhysicalDeviceSurfaceInfo2KHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SURFACE_INFO_2_KHR,pNext=None,surface=None,):

    return _new('VkPhysicalDeviceSurfaceInfo2KHR', sType=sType,pNext=pNext,surface=surface)

def VkSurfaceCapabilities2KHR(sType=VK_STRUCTURE_TYPE_SURFACE_CAPABILITIES_2_KHR,pNext=None,surfaceCapabilities=None,):

    return _new('VkSurfaceCapabilities2KHR', sType=sType,pNext=pNext,surfaceCapabilities=surfaceCapabilities)

def VkSurfaceFormat2KHR(sType=VK_STRUCTURE_TYPE_SURFACE_FORMAT_2_KHR,pNext=None,surfaceFormat=None,):

    return _new('VkSurfaceFormat2KHR', sType=sType,pNext=pNext,surfaceFormat=surfaceFormat)

def VkDisplayProperties2KHR(sType=VK_STRUCTURE_TYPE_DISPLAY_PROPERTIES_2_KHR,pNext=None,displayProperties=None,):

    return _new('VkDisplayProperties2KHR', sType=sType,pNext=pNext,displayProperties=displayProperties)

def VkDisplayPlaneProperties2KHR(sType=VK_STRUCTURE_TYPE_DISPLAY_PLANE_PROPERTIES_2_KHR,pNext=None,displayPlaneProperties=None,):

    return _new('VkDisplayPlaneProperties2KHR', sType=sType,pNext=pNext,displayPlaneProperties=displayPlaneProperties)

def VkDisplayModeProperties2KHR(sType=VK_STRUCTURE_TYPE_DISPLAY_MODE_PROPERTIES_2_KHR,pNext=None,displayModeProperties=None,):

    return _new('VkDisplayModeProperties2KHR', sType=sType,pNext=pNext,displayModeProperties=displayModeProperties)

def VkDisplayPlaneInfo2KHR(sType=VK_STRUCTURE_TYPE_DISPLAY_PLANE_INFO_2_KHR,pNext=None,mode=None,planeIndex=None,):

    return _new('VkDisplayPlaneInfo2KHR', sType=sType,pNext=pNext,mode=mode,planeIndex=planeIndex)

def VkDisplayPlaneCapabilities2KHR(sType=VK_STRUCTURE_TYPE_DISPLAY_PLANE_CAPABILITIES_2_KHR,pNext=None,capabilities=None,):

    return _new('VkDisplayPlaneCapabilities2KHR', sType=sType,pNext=pNext,capabilities=capabilities)

def VkSharedPresentSurfaceCapabilitiesKHR(sType=VK_STRUCTURE_TYPE_SHARED_PRESENT_SURFACE_CAPABILITIES_KHR,pNext=None,sharedPresentSupportedUsageFlags=None,):

    return _new('VkSharedPresentSurfaceCapabilitiesKHR', sType=sType,pNext=pNext,sharedPresentSupportedUsageFlags=sharedPresentSupportedUsageFlags)

def VkPhysicalDevice16BitStorageFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_16BIT_STORAGE_FEATURES,pNext=None,storageBuffer16BitAccess=None,uniformAndStorageBuffer16BitAccess=None,storagePushConstant16=None,storageInputOutput16=None,):

    return _new('VkPhysicalDevice16BitStorageFeatures', sType=sType,pNext=pNext,storageBuffer16BitAccess=storageBuffer16BitAccess,uniformAndStorageBuffer16BitAccess=uniformAndStorageBuffer16BitAccess,storagePushConstant16=storagePushConstant16,storageInputOutput16=storageInputOutput16)

def VkPhysicalDeviceSubgroupProperties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBGROUP_PROPERTIES,pNext=None,subgroupSize=None,supportedStages=None,supportedOperations=None,quadOperationsInAllStages=None,):

    return _new('VkPhysicalDeviceSubgroupProperties', sType=sType,pNext=pNext,subgroupSize=subgroupSize,supportedStages=supportedStages,supportedOperations=supportedOperations,quadOperationsInAllStages=quadOperationsInAllStages)

def VkBufferMemoryRequirementsInfo2(sType=VK_STRUCTURE_TYPE_BUFFER_MEMORY_REQUIREMENTS_INFO_2,pNext=None,buffer=None,):

    return _new('VkBufferMemoryRequirementsInfo2', sType=sType,pNext=pNext,buffer=buffer)

def VkImageMemoryRequirementsInfo2(sType=VK_STRUCTURE_TYPE_IMAGE_MEMORY_REQUIREMENTS_INFO_2,pNext=None,image=None,):

    return _new('VkImageMemoryRequirementsInfo2', sType=sType,pNext=pNext,image=image)

def VkImageSparseMemoryRequirementsInfo2(sType=VK_STRUCTURE_TYPE_IMAGE_SPARSE_MEMORY_REQUIREMENTS_INFO_2,pNext=None,image=None,):

    return _new('VkImageSparseMemoryRequirementsInfo2', sType=sType,pNext=pNext,image=image)

def VkMemoryRequirements2(sType=VK_STRUCTURE_TYPE_MEMORY_REQUIREMENTS_2,pNext=None,memoryRequirements=None,):

    return _new('VkMemoryRequirements2', sType=sType,pNext=pNext,memoryRequirements=memoryRequirements)

def VkSparseImageMemoryRequirements2(sType=VK_STRUCTURE_TYPE_SPARSE_IMAGE_MEMORY_REQUIREMENTS_2,pNext=None,memoryRequirements=None,):

    return _new('VkSparseImageMemoryRequirements2', sType=sType,pNext=pNext,memoryRequirements=memoryRequirements)

def VkPhysicalDevicePointClippingProperties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_POINT_CLIPPING_PROPERTIES,pNext=None,pointClippingBehavior=None,):

    return _new('VkPhysicalDevicePointClippingProperties', sType=sType,pNext=pNext,pointClippingBehavior=pointClippingBehavior)

def VkMemoryDedicatedRequirements(sType=VK_STRUCTURE_TYPE_MEMORY_DEDICATED_REQUIREMENTS,pNext=None,prefersDedicatedAllocation=None,requiresDedicatedAllocation=None,):

    return _new('VkMemoryDedicatedRequirements', sType=sType,pNext=pNext,prefersDedicatedAllocation=prefersDedicatedAllocation,requiresDedicatedAllocation=requiresDedicatedAllocation)

def VkMemoryDedicatedAllocateInfo(sType=VK_STRUCTURE_TYPE_MEMORY_DEDICATED_ALLOCATE_INFO,pNext=None,image=None,buffer=None,):

    return _new('VkMemoryDedicatedAllocateInfo', sType=sType,pNext=pNext,image=image,buffer=buffer)

def VkImageViewUsageCreateInfo(sType=VK_STRUCTURE_TYPE_IMAGE_VIEW_USAGE_CREATE_INFO,pNext=None,usage=None,):

    return _new('VkImageViewUsageCreateInfo', sType=sType,pNext=pNext,usage=usage)

def VkPipelineTessellationDomainOriginStateCreateInfo(sType=VK_STRUCTURE_TYPE_PIPELINE_TESSELLATION_DOMAIN_ORIGIN_STATE_CREATE_INFO,pNext=None,domainOrigin=None,):

    return _new('VkPipelineTessellationDomainOriginStateCreateInfo', sType=sType,pNext=pNext,domainOrigin=domainOrigin)

def VkSamplerYcbcrConversionInfo(sType=VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_INFO,pNext=None,conversion=None,):

    return _new('VkSamplerYcbcrConversionInfo', sType=sType,pNext=pNext,conversion=conversion)

def VkSamplerYcbcrConversionCreateInfo(sType=VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_CREATE_INFO,pNext=None,format=None,ycbcrModel=None,ycbcrRange=None,components=None,xChromaOffset=None,yChromaOffset=None,chromaFilter=None,forceExplicitReconstruction=None,):

    return _new('VkSamplerYcbcrConversionCreateInfo', sType=sType,pNext=pNext,format=format,ycbcrModel=ycbcrModel,ycbcrRange=ycbcrRange,components=components,xChromaOffset=xChromaOffset,yChromaOffset=yChromaOffset,chromaFilter=chromaFilter,forceExplicitReconstruction=forceExplicitReconstruction)

def VkBindImagePlaneMemoryInfo(sType=VK_STRUCTURE_TYPE_BIND_IMAGE_PLANE_MEMORY_INFO,pNext=None,planeAspect=None,):

    return _new('VkBindImagePlaneMemoryInfo', sType=sType,pNext=pNext,planeAspect=planeAspect)

def VkImagePlaneMemoryRequirementsInfo(sType=VK_STRUCTURE_TYPE_IMAGE_PLANE_MEMORY_REQUIREMENTS_INFO,pNext=None,planeAspect=None,):

    return _new('VkImagePlaneMemoryRequirementsInfo', sType=sType,pNext=pNext,planeAspect=planeAspect)

def VkPhysicalDeviceSamplerYcbcrConversionFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLER_YCBCR_CONVERSION_FEATURES,pNext=None,samplerYcbcrConversion=None,):

    return _new('VkPhysicalDeviceSamplerYcbcrConversionFeatures', sType=sType,pNext=pNext,samplerYcbcrConversion=samplerYcbcrConversion)

def VkSamplerYcbcrConversionImageFormatProperties(sType=VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_IMAGE_FORMAT_PROPERTIES,pNext=None,combinedImageSamplerDescriptorCount=None,):

    return _new('VkSamplerYcbcrConversionImageFormatProperties', sType=sType,pNext=pNext,combinedImageSamplerDescriptorCount=combinedImageSamplerDescriptorCount)

def VkTextureLODGatherFormatPropertiesAMD(sType=VK_STRUCTURE_TYPE_TEXTURE_LOD_GATHER_FORMAT_PROPERTIES_AMD,pNext=None,supportsTextureGatherLODBiasAMD=None,):

    return _new('VkTextureLODGatherFormatPropertiesAMD', sType=sType,pNext=pNext,supportsTextureGatherLODBiasAMD=supportsTextureGatherLODBiasAMD)

def VkConditionalRenderingBeginInfoEXT(sType=VK_STRUCTURE_TYPE_CONDITIONAL_RENDERING_BEGIN_INFO_EXT,pNext=None,buffer=None,offset=None,flags=None,):

    return _new('VkConditionalRenderingBeginInfoEXT', sType=sType,pNext=pNext,buffer=buffer,offset=offset,flags=flags)

def VkProtectedSubmitInfo(sType=VK_STRUCTURE_TYPE_PROTECTED_SUBMIT_INFO,pNext=None,protectedSubmit=None,):

    return _new('VkProtectedSubmitInfo', sType=sType,pNext=pNext,protectedSubmit=protectedSubmit)

def VkPhysicalDeviceProtectedMemoryFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROTECTED_MEMORY_FEATURES,pNext=None,protectedMemory=None,):

    return _new('VkPhysicalDeviceProtectedMemoryFeatures', sType=sType,pNext=pNext,protectedMemory=protectedMemory)

def VkPhysicalDeviceProtectedMemoryProperties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROTECTED_MEMORY_PROPERTIES,pNext=None,protectedNoFault=None,):

    return _new('VkPhysicalDeviceProtectedMemoryProperties', sType=sType,pNext=pNext,protectedNoFault=protectedNoFault)

def VkDeviceQueueInfo2(sType=VK_STRUCTURE_TYPE_DEVICE_QUEUE_INFO_2,pNext=None,flags=None,queueFamilyIndex=None,queueIndex=None,):

    return _new('VkDeviceQueueInfo2', sType=sType,pNext=pNext,flags=flags,queueFamilyIndex=queueFamilyIndex,queueIndex=queueIndex)

def VkPipelineCoverageToColorStateCreateInfoNV(sType=VK_STRUCTURE_TYPE_PIPELINE_COVERAGE_TO_COLOR_STATE_CREATE_INFO_NV,pNext=None,flags=None,coverageToColorEnable=None,coverageToColorLocation=None,):

    return _new('VkPipelineCoverageToColorStateCreateInfoNV', sType=sType,pNext=pNext,flags=flags,coverageToColorEnable=coverageToColorEnable,coverageToColorLocation=coverageToColorLocation)

def VkPhysicalDeviceSamplerFilterMinmaxPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLER_FILTER_MINMAX_PROPERTIES_EXT,pNext=None,filterMinmaxSingleComponentFormats=None,filterMinmaxImageComponentMapping=None,):

    return _new('VkPhysicalDeviceSamplerFilterMinmaxPropertiesEXT', sType=sType,pNext=pNext,filterMinmaxSingleComponentFormats=filterMinmaxSingleComponentFormats,filterMinmaxImageComponentMapping=filterMinmaxImageComponentMapping)

def VkSampleLocationEXT(x=None,y=None,):

    return _new('VkSampleLocationEXT', x=x,y=y)

def VkSampleLocationsInfoEXT(sType=VK_STRUCTURE_TYPE_SAMPLE_LOCATIONS_INFO_EXT,pNext=None,sampleLocationsPerPixel=None,sampleLocationGridSize=None,sampleLocationsCount=None,pSampleLocations=None,):
    if sampleLocationsCount is None and pSampleLocations is not None:
        sampleLocationsCount = len(pSampleLocations)

    return _new('VkSampleLocationsInfoEXT', sType=sType,pNext=pNext,sampleLocationsPerPixel=sampleLocationsPerPixel,sampleLocationGridSize=sampleLocationGridSize,sampleLocationsCount=sampleLocationsCount,pSampleLocations=pSampleLocations)

def VkAttachmentSampleLocationsEXT(attachmentIndex=None,sampleLocationsInfo=None,):

    return _new('VkAttachmentSampleLocationsEXT', attachmentIndex=attachmentIndex,sampleLocationsInfo=sampleLocationsInfo)

def VkSubpassSampleLocationsEXT(subpassIndex=None,sampleLocationsInfo=None,):

    return _new('VkSubpassSampleLocationsEXT', subpassIndex=subpassIndex,sampleLocationsInfo=sampleLocationsInfo)

def VkRenderPassSampleLocationsBeginInfoEXT(sType=VK_STRUCTURE_TYPE_RENDER_PASS_SAMPLE_LOCATIONS_BEGIN_INFO_EXT,pNext=None,attachmentInitialSampleLocationsCount=None,pAttachmentInitialSampleLocations=None,postSubpassSampleLocationsCount=None,pPostSubpassSampleLocations=None,):
    if attachmentInitialSampleLocationsCount is None and pAttachmentInitialSampleLocations is not None:
        attachmentInitialSampleLocationsCount = len(pAttachmentInitialSampleLocations)
    if postSubpassSampleLocationsCount is None and pPostSubpassSampleLocations is not None:
        postSubpassSampleLocationsCount = len(pPostSubpassSampleLocations)

    return _new('VkRenderPassSampleLocationsBeginInfoEXT', sType=sType,pNext=pNext,attachmentInitialSampleLocationsCount=attachmentInitialSampleLocationsCount,pAttachmentInitialSampleLocations=pAttachmentInitialSampleLocations,postSubpassSampleLocationsCount=postSubpassSampleLocationsCount,pPostSubpassSampleLocations=pPostSubpassSampleLocations)

def VkPipelineSampleLocationsStateCreateInfoEXT(sType=VK_STRUCTURE_TYPE_PIPELINE_SAMPLE_LOCATIONS_STATE_CREATE_INFO_EXT,pNext=None,sampleLocationsEnable=None,sampleLocationsInfo=None,):

    return _new('VkPipelineSampleLocationsStateCreateInfoEXT', sType=sType,pNext=pNext,sampleLocationsEnable=sampleLocationsEnable,sampleLocationsInfo=sampleLocationsInfo)

def VkPhysicalDeviceSampleLocationsPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLE_LOCATIONS_PROPERTIES_EXT,pNext=None,sampleLocationSampleCounts=None,maxSampleLocationGridSize=None,sampleLocationCoordinateRange=None,sampleLocationSubPixelBits=None,variableSampleLocations=None,):

    return _new('VkPhysicalDeviceSampleLocationsPropertiesEXT', sType=sType,pNext=pNext,sampleLocationSampleCounts=sampleLocationSampleCounts,maxSampleLocationGridSize=maxSampleLocationGridSize,sampleLocationCoordinateRange=sampleLocationCoordinateRange,sampleLocationSubPixelBits=sampleLocationSubPixelBits,variableSampleLocations=variableSampleLocations)

def VkMultisamplePropertiesEXT(sType=VK_STRUCTURE_TYPE_MULTISAMPLE_PROPERTIES_EXT,pNext=None,maxSampleLocationGridSize=None,):

    return _new('VkMultisamplePropertiesEXT', sType=sType,pNext=pNext,maxSampleLocationGridSize=maxSampleLocationGridSize)

def VkSamplerReductionModeCreateInfoEXT(sType=VK_STRUCTURE_TYPE_SAMPLER_REDUCTION_MODE_CREATE_INFO_EXT,pNext=None,reductionMode=None,):

    return _new('VkSamplerReductionModeCreateInfoEXT', sType=sType,pNext=pNext,reductionMode=reductionMode)

def VkPhysicalDeviceBlendOperationAdvancedFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BLEND_OPERATION_ADVANCED_FEATURES_EXT,pNext=None,advancedBlendCoherentOperations=None,):

    return _new('VkPhysicalDeviceBlendOperationAdvancedFeaturesEXT', sType=sType,pNext=pNext,advancedBlendCoherentOperations=advancedBlendCoherentOperations)

def VkPhysicalDeviceBlendOperationAdvancedPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BLEND_OPERATION_ADVANCED_PROPERTIES_EXT,pNext=None,advancedBlendMaxColorAttachments=None,advancedBlendIndependentBlend=None,advancedBlendNonPremultipliedSrcColor=None,advancedBlendNonPremultipliedDstColor=None,advancedBlendCorrelatedOverlap=None,advancedBlendAllOperations=None,):

    return _new('VkPhysicalDeviceBlendOperationAdvancedPropertiesEXT', sType=sType,pNext=pNext,advancedBlendMaxColorAttachments=advancedBlendMaxColorAttachments,advancedBlendIndependentBlend=advancedBlendIndependentBlend,advancedBlendNonPremultipliedSrcColor=advancedBlendNonPremultipliedSrcColor,advancedBlendNonPremultipliedDstColor=advancedBlendNonPremultipliedDstColor,advancedBlendCorrelatedOverlap=advancedBlendCorrelatedOverlap,advancedBlendAllOperations=advancedBlendAllOperations)

def VkPipelineColorBlendAdvancedStateCreateInfoEXT(sType=VK_STRUCTURE_TYPE_PIPELINE_COLOR_BLEND_ADVANCED_STATE_CREATE_INFO_EXT,pNext=None,srcPremultiplied=None,dstPremultiplied=None,blendOverlap=None,):

    return _new('VkPipelineColorBlendAdvancedStateCreateInfoEXT', sType=sType,pNext=pNext,srcPremultiplied=srcPremultiplied,dstPremultiplied=dstPremultiplied,blendOverlap=blendOverlap)

def VkPhysicalDeviceInlineUniformBlockFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INLINE_UNIFORM_BLOCK_FEATURES_EXT,pNext=None,inlineUniformBlock=None,descriptorBindingInlineUniformBlockUpdateAfterBind=None,):

    return _new('VkPhysicalDeviceInlineUniformBlockFeaturesEXT', sType=sType,pNext=pNext,inlineUniformBlock=inlineUniformBlock,descriptorBindingInlineUniformBlockUpdateAfterBind=descriptorBindingInlineUniformBlockUpdateAfterBind)

def VkPhysicalDeviceInlineUniformBlockPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INLINE_UNIFORM_BLOCK_PROPERTIES_EXT,pNext=None,maxInlineUniformBlockSize=None,maxPerStageDescriptorInlineUniformBlocks=None,maxPerStageDescriptorUpdateAfterBindInlineUniformBlocks=None,maxDescriptorSetInlineUniformBlocks=None,maxDescriptorSetUpdateAfterBindInlineUniformBlocks=None,):

    return _new('VkPhysicalDeviceInlineUniformBlockPropertiesEXT', sType=sType,pNext=pNext,maxInlineUniformBlockSize=maxInlineUniformBlockSize,maxPerStageDescriptorInlineUniformBlocks=maxPerStageDescriptorInlineUniformBlocks,maxPerStageDescriptorUpdateAfterBindInlineUniformBlocks=maxPerStageDescriptorUpdateAfterBindInlineUniformBlocks,maxDescriptorSetInlineUniformBlocks=maxDescriptorSetInlineUniformBlocks,maxDescriptorSetUpdateAfterBindInlineUniformBlocks=maxDescriptorSetUpdateAfterBindInlineUniformBlocks)

def VkWriteDescriptorSetInlineUniformBlockEXT(sType=VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET_INLINE_UNIFORM_BLOCK_EXT,pNext=None,dataSize=None,pData=None,):
    if dataSize is None and pData is not None:
        dataSize = len(pData)

    return _new('VkWriteDescriptorSetInlineUniformBlockEXT', sType=sType,pNext=pNext,dataSize=dataSize,pData=pData)

def VkDescriptorPoolInlineUniformBlockCreateInfoEXT(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_POOL_INLINE_UNIFORM_BLOCK_CREATE_INFO_EXT,pNext=None,maxInlineUniformBlockBindings=None,):

    return _new('VkDescriptorPoolInlineUniformBlockCreateInfoEXT', sType=sType,pNext=pNext,maxInlineUniformBlockBindings=maxInlineUniformBlockBindings)

def VkPipelineCoverageModulationStateCreateInfoNV(sType=VK_STRUCTURE_TYPE_PIPELINE_COVERAGE_MODULATION_STATE_CREATE_INFO_NV,pNext=None,flags=None,coverageModulationMode=None,coverageModulationTableEnable=None,coverageModulationTableCount=None,pCoverageModulationTable=None,):
    if coverageModulationTableCount is None and pCoverageModulationTable is not None:
        coverageModulationTableCount = len(pCoverageModulationTable)

    return _new('VkPipelineCoverageModulationStateCreateInfoNV', sType=sType,pNext=pNext,flags=flags,coverageModulationMode=coverageModulationMode,coverageModulationTableEnable=coverageModulationTableEnable,coverageModulationTableCount=coverageModulationTableCount,pCoverageModulationTable=pCoverageModulationTable)

def VkImageFormatListCreateInfoKHR(sType=VK_STRUCTURE_TYPE_IMAGE_FORMAT_LIST_CREATE_INFO_KHR,pNext=None,viewFormatCount=None,pViewFormats=None,):
    if viewFormatCount is None and pViewFormats is not None:
        viewFormatCount = len(pViewFormats)

    return _new('VkImageFormatListCreateInfoKHR', sType=sType,pNext=pNext,viewFormatCount=viewFormatCount,pViewFormats=pViewFormats)

def VkValidationCacheCreateInfoEXT(sType=VK_STRUCTURE_TYPE_VALIDATION_CACHE_CREATE_INFO_EXT,pNext=None,flags=None,initialDataSize=None,pInitialData=None,):
    if initialDataSize is None and pInitialData is not None:
        initialDataSize = len(pInitialData)

    return _new('VkValidationCacheCreateInfoEXT', sType=sType,pNext=pNext,flags=flags,initialDataSize=initialDataSize,pInitialData=pInitialData)

def VkShaderModuleValidationCacheCreateInfoEXT(sType=VK_STRUCTURE_TYPE_SHADER_MODULE_VALIDATION_CACHE_CREATE_INFO_EXT,pNext=None,validationCache=None,):

    return _new('VkShaderModuleValidationCacheCreateInfoEXT', sType=sType,pNext=pNext,validationCache=validationCache)

def VkPhysicalDeviceMaintenance3Properties(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_3_PROPERTIES,pNext=None,maxPerSetDescriptors=None,maxMemoryAllocationSize=None,):

    return _new('VkPhysicalDeviceMaintenance3Properties', sType=sType,pNext=pNext,maxPerSetDescriptors=maxPerSetDescriptors,maxMemoryAllocationSize=maxMemoryAllocationSize)

def VkDescriptorSetLayoutSupport(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_SUPPORT,pNext=None,supported=None,):

    return _new('VkDescriptorSetLayoutSupport', sType=sType,pNext=pNext,supported=supported)

def VkPhysicalDeviceShaderDrawParameterFeatures(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_DRAW_PARAMETER_FEATURES,pNext=None,shaderDrawParameters=None,):

    return _new('VkPhysicalDeviceShaderDrawParameterFeatures', sType=sType,pNext=pNext,shaderDrawParameters=shaderDrawParameters)

def VkPhysicalDeviceFloat16Int8FeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FLOAT16_INT8_FEATURES_KHR,pNext=None,shaderFloat16=None,shaderInt8=None,):

    return _new('VkPhysicalDeviceFloat16Int8FeaturesKHR', sType=sType,pNext=pNext,shaderFloat16=shaderFloat16,shaderInt8=shaderInt8)

def VkPhysicalDeviceFloatControlsPropertiesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FLOAT_CONTROLS_PROPERTIES_KHR,pNext=None,separateDenormSettings=None,separateRoundingModeSettings=None,shaderSignedZeroInfNanPreserveFloat16=None,shaderSignedZeroInfNanPreserveFloat32=None,shaderSignedZeroInfNanPreserveFloat64=None,shaderDenormPreserveFloat16=None,shaderDenormPreserveFloat32=None,shaderDenormPreserveFloat64=None,shaderDenormFlushToZeroFloat16=None,shaderDenormFlushToZeroFloat32=None,shaderDenormFlushToZeroFloat64=None,shaderRoundingModeRTEFloat16=None,shaderRoundingModeRTEFloat32=None,shaderRoundingModeRTEFloat64=None,shaderRoundingModeRTZFloat16=None,shaderRoundingModeRTZFloat32=None,shaderRoundingModeRTZFloat64=None,):

    return _new('VkPhysicalDeviceFloatControlsPropertiesKHR', sType=sType,pNext=pNext,separateDenormSettings=separateDenormSettings,separateRoundingModeSettings=separateRoundingModeSettings,shaderSignedZeroInfNanPreserveFloat16=shaderSignedZeroInfNanPreserveFloat16,shaderSignedZeroInfNanPreserveFloat32=shaderSignedZeroInfNanPreserveFloat32,shaderSignedZeroInfNanPreserveFloat64=shaderSignedZeroInfNanPreserveFloat64,shaderDenormPreserveFloat16=shaderDenormPreserveFloat16,shaderDenormPreserveFloat32=shaderDenormPreserveFloat32,shaderDenormPreserveFloat64=shaderDenormPreserveFloat64,shaderDenormFlushToZeroFloat16=shaderDenormFlushToZeroFloat16,shaderDenormFlushToZeroFloat32=shaderDenormFlushToZeroFloat32,shaderDenormFlushToZeroFloat64=shaderDenormFlushToZeroFloat64,shaderRoundingModeRTEFloat16=shaderRoundingModeRTEFloat16,shaderRoundingModeRTEFloat32=shaderRoundingModeRTEFloat32,shaderRoundingModeRTEFloat64=shaderRoundingModeRTEFloat64,shaderRoundingModeRTZFloat16=shaderRoundingModeRTZFloat16,shaderRoundingModeRTZFloat32=shaderRoundingModeRTZFloat32,shaderRoundingModeRTZFloat64=shaderRoundingModeRTZFloat64)

def VkNativeBufferANDROID(sType=VK_STRUCTURE_TYPE_NATIVE_BUFFER_ANDROID,pNext=None,handle=None,stride=None,format=None,usage=None,):

    return _new('VkNativeBufferANDROID', sType=sType,pNext=pNext,handle=handle,stride=stride,format=format,usage=usage)

def VkShaderResourceUsageAMD(numUsedVgprs=None,numUsedSgprs=None,ldsSizePerLocalWorkGroup=None,ldsUsageSizeInBytes=None,scratchMemUsageInBytes=None,):

    return _new('VkShaderResourceUsageAMD', numUsedVgprs=numUsedVgprs,numUsedSgprs=numUsedSgprs,ldsSizePerLocalWorkGroup=ldsSizePerLocalWorkGroup,ldsUsageSizeInBytes=ldsUsageSizeInBytes,scratchMemUsageInBytes=scratchMemUsageInBytes)

def VkShaderStatisticsInfoAMD(shaderStageMask=None,resourceUsage=None,numPhysicalVgprs=None,numPhysicalSgprs=None,numAvailableVgprs=None,numAvailableSgprs=None,computeWorkGroupSize=None,):

    return _new('VkShaderStatisticsInfoAMD', shaderStageMask=shaderStageMask,resourceUsage=resourceUsage,numPhysicalVgprs=numPhysicalVgprs,numPhysicalSgprs=numPhysicalSgprs,numAvailableVgprs=numAvailableVgprs,numAvailableSgprs=numAvailableSgprs,computeWorkGroupSize=computeWorkGroupSize)

def VkDeviceQueueGlobalPriorityCreateInfoEXT(sType=VK_STRUCTURE_TYPE_DEVICE_QUEUE_GLOBAL_PRIORITY_CREATE_INFO_EXT,pNext=None,globalPriority=None,):

    return _new('VkDeviceQueueGlobalPriorityCreateInfoEXT', sType=sType,pNext=pNext,globalPriority=globalPriority)

def VkDebugUtilsObjectNameInfoEXT(sType=VK_STRUCTURE_TYPE_DEBUG_UTILS_OBJECT_NAME_INFO_EXT,pNext=None,objectType=None,objectHandle=None,pObjectName=None,):

    return _new('VkDebugUtilsObjectNameInfoEXT', sType=sType,pNext=pNext,objectType=objectType,objectHandle=objectHandle,pObjectName=pObjectName)

def VkDebugUtilsObjectTagInfoEXT(sType=VK_STRUCTURE_TYPE_DEBUG_UTILS_OBJECT_TAG_INFO_EXT,pNext=None,objectType=None,objectHandle=None,tagName=None,tagSize=None,pTag=None,):
    if tagSize is None and pTag is not None:
        tagSize = len(pTag)

    return _new('VkDebugUtilsObjectTagInfoEXT', sType=sType,pNext=pNext,objectType=objectType,objectHandle=objectHandle,tagName=tagName,tagSize=tagSize,pTag=pTag)

def VkDebugUtilsLabelEXT(sType=VK_STRUCTURE_TYPE_DEBUG_UTILS_LABEL_EXT,pNext=None,pLabelName=None,color=None,):

    return _new('VkDebugUtilsLabelEXT', sType=sType,pNext=pNext,pLabelName=pLabelName,color=color)

def VkDebugUtilsMessengerCreateInfoEXT(sType=VK_STRUCTURE_TYPE_DEBUG_UTILS_MESSENGER_CREATE_INFO_EXT,pNext=None,flags=None,messageSeverity=None,messageType=None,pfnUserCallback=None,pUserData=None,):

    return _new('VkDebugUtilsMessengerCreateInfoEXT', sType=sType,pNext=pNext,flags=flags,messageSeverity=messageSeverity,messageType=messageType,pfnUserCallback=pfnUserCallback,pUserData=pUserData)

def VkDebugUtilsMessengerCallbackDataEXT(sType=VK_STRUCTURE_TYPE_DEBUG_UTILS_MESSENGER_CALLBACK_DATA_EXT,pNext=None,flags=None,pMessageIdName=None,messageIdNumber=None,pMessage=None,queueLabelCount=None,pQueueLabels=None,cmdBufLabelCount=None,pCmdBufLabels=None,objectCount=None,pObjects=None,):
    if queueLabelCount is None and pQueueLabels is not None:
        queueLabelCount = len(pQueueLabels)
    if cmdBufLabelCount is None and pCmdBufLabels is not None:
        cmdBufLabelCount = len(pCmdBufLabels)
    if objectCount is None and pObjects is not None:
        objectCount = len(pObjects)

    return _new('VkDebugUtilsMessengerCallbackDataEXT', sType=sType,pNext=pNext,flags=flags,pMessageIdName=pMessageIdName,messageIdNumber=messageIdNumber,pMessage=pMessage,queueLabelCount=queueLabelCount,pQueueLabels=pQueueLabels,cmdBufLabelCount=cmdBufLabelCount,pCmdBufLabels=pCmdBufLabels,objectCount=objectCount,pObjects=pObjects)

def VkImportMemoryHostPointerInfoEXT(sType=VK_STRUCTURE_TYPE_IMPORT_MEMORY_HOST_POINTER_INFO_EXT,pNext=None,handleType=None,pHostPointer=None,):

    return _new('VkImportMemoryHostPointerInfoEXT', sType=sType,pNext=pNext,handleType=handleType,pHostPointer=pHostPointer)

def VkMemoryHostPointerPropertiesEXT(sType=VK_STRUCTURE_TYPE_MEMORY_HOST_POINTER_PROPERTIES_EXT,pNext=None,memoryTypeBits=None,):

    return _new('VkMemoryHostPointerPropertiesEXT', sType=sType,pNext=pNext,memoryTypeBits=memoryTypeBits)

def VkPhysicalDeviceExternalMemoryHostPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_MEMORY_HOST_PROPERTIES_EXT,pNext=None,minImportedHostPointerAlignment=None,):

    return _new('VkPhysicalDeviceExternalMemoryHostPropertiesEXT', sType=sType,pNext=pNext,minImportedHostPointerAlignment=minImportedHostPointerAlignment)

def VkPhysicalDeviceConservativeRasterizationPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CONSERVATIVE_RASTERIZATION_PROPERTIES_EXT,pNext=None,primitiveOverestimationSize=None,maxExtraPrimitiveOverestimationSize=None,extraPrimitiveOverestimationSizeGranularity=None,primitiveUnderestimation=None,conservativePointAndLineRasterization=None,degenerateTrianglesRasterized=None,degenerateLinesRasterized=None,fullyCoveredFragmentShaderInputVariable=None,conservativeRasterizationPostDepthCoverage=None,):

    return _new('VkPhysicalDeviceConservativeRasterizationPropertiesEXT', sType=sType,pNext=pNext,primitiveOverestimationSize=primitiveOverestimationSize,maxExtraPrimitiveOverestimationSize=maxExtraPrimitiveOverestimationSize,extraPrimitiveOverestimationSizeGranularity=extraPrimitiveOverestimationSizeGranularity,primitiveUnderestimation=primitiveUnderestimation,conservativePointAndLineRasterization=conservativePointAndLineRasterization,degenerateTrianglesRasterized=degenerateTrianglesRasterized,degenerateLinesRasterized=degenerateLinesRasterized,fullyCoveredFragmentShaderInputVariable=fullyCoveredFragmentShaderInputVariable,conservativeRasterizationPostDepthCoverage=conservativeRasterizationPostDepthCoverage)

def VkCalibratedTimestampInfoEXT(sType=VK_STRUCTURE_TYPE_CALIBRATED_TIMESTAMP_INFO_EXT,pNext=None,timeDomain=None,):

    return _new('VkCalibratedTimestampInfoEXT', sType=sType,pNext=pNext,timeDomain=timeDomain)

def VkPhysicalDeviceShaderCorePropertiesAMD(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_CORE_PROPERTIES_AMD,pNext=None,shaderEngineCount=None,shaderArraysPerEngineCount=None,computeUnitsPerShaderArray=None,simdPerComputeUnit=None,wavefrontsPerSimd=None,wavefrontSize=None,sgprsPerSimd=None,minSgprAllocation=None,maxSgprAllocation=None,sgprAllocationGranularity=None,vgprsPerSimd=None,minVgprAllocation=None,maxVgprAllocation=None,vgprAllocationGranularity=None,):

    return _new('VkPhysicalDeviceShaderCorePropertiesAMD', sType=sType,pNext=pNext,shaderEngineCount=shaderEngineCount,shaderArraysPerEngineCount=shaderArraysPerEngineCount,computeUnitsPerShaderArray=computeUnitsPerShaderArray,simdPerComputeUnit=simdPerComputeUnit,wavefrontsPerSimd=wavefrontsPerSimd,wavefrontSize=wavefrontSize,sgprsPerSimd=sgprsPerSimd,minSgprAllocation=minSgprAllocation,maxSgprAllocation=maxSgprAllocation,sgprAllocationGranularity=sgprAllocationGranularity,vgprsPerSimd=vgprsPerSimd,minVgprAllocation=minVgprAllocation,maxVgprAllocation=maxVgprAllocation,vgprAllocationGranularity=vgprAllocationGranularity)

def VkPipelineRasterizationConservativeStateCreateInfoEXT(sType=VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_CONSERVATIVE_STATE_CREATE_INFO_EXT,pNext=None,flags=None,conservativeRasterizationMode=None,extraPrimitiveOverestimationSize=None,):

    return _new('VkPipelineRasterizationConservativeStateCreateInfoEXT', sType=sType,pNext=pNext,flags=flags,conservativeRasterizationMode=conservativeRasterizationMode,extraPrimitiveOverestimationSize=extraPrimitiveOverestimationSize)

def VkPhysicalDeviceDescriptorIndexingFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_INDEXING_FEATURES_EXT,pNext=None,shaderInputAttachmentArrayDynamicIndexing=None,shaderUniformTexelBufferArrayDynamicIndexing=None,shaderStorageTexelBufferArrayDynamicIndexing=None,shaderUniformBufferArrayNonUniformIndexing=None,shaderSampledImageArrayNonUniformIndexing=None,shaderStorageBufferArrayNonUniformIndexing=None,shaderStorageImageArrayNonUniformIndexing=None,shaderInputAttachmentArrayNonUniformIndexing=None,shaderUniformTexelBufferArrayNonUniformIndexing=None,shaderStorageTexelBufferArrayNonUniformIndexing=None,descriptorBindingUniformBufferUpdateAfterBind=None,descriptorBindingSampledImageUpdateAfterBind=None,descriptorBindingStorageImageUpdateAfterBind=None,descriptorBindingStorageBufferUpdateAfterBind=None,descriptorBindingUniformTexelBufferUpdateAfterBind=None,descriptorBindingStorageTexelBufferUpdateAfterBind=None,descriptorBindingUpdateUnusedWhilePending=None,descriptorBindingPartiallyBound=None,descriptorBindingVariableDescriptorCount=None,runtimeDescriptorArray=None,):

    return _new('VkPhysicalDeviceDescriptorIndexingFeaturesEXT', sType=sType,pNext=pNext,shaderInputAttachmentArrayDynamicIndexing=shaderInputAttachmentArrayDynamicIndexing,shaderUniformTexelBufferArrayDynamicIndexing=shaderUniformTexelBufferArrayDynamicIndexing,shaderStorageTexelBufferArrayDynamicIndexing=shaderStorageTexelBufferArrayDynamicIndexing,shaderUniformBufferArrayNonUniformIndexing=shaderUniformBufferArrayNonUniformIndexing,shaderSampledImageArrayNonUniformIndexing=shaderSampledImageArrayNonUniformIndexing,shaderStorageBufferArrayNonUniformIndexing=shaderStorageBufferArrayNonUniformIndexing,shaderStorageImageArrayNonUniformIndexing=shaderStorageImageArrayNonUniformIndexing,shaderInputAttachmentArrayNonUniformIndexing=shaderInputAttachmentArrayNonUniformIndexing,shaderUniformTexelBufferArrayNonUniformIndexing=shaderUniformTexelBufferArrayNonUniformIndexing,shaderStorageTexelBufferArrayNonUniformIndexing=shaderStorageTexelBufferArrayNonUniformIndexing,descriptorBindingUniformBufferUpdateAfterBind=descriptorBindingUniformBufferUpdateAfterBind,descriptorBindingSampledImageUpdateAfterBind=descriptorBindingSampledImageUpdateAfterBind,descriptorBindingStorageImageUpdateAfterBind=descriptorBindingStorageImageUpdateAfterBind,descriptorBindingStorageBufferUpdateAfterBind=descriptorBindingStorageBufferUpdateAfterBind,descriptorBindingUniformTexelBufferUpdateAfterBind=descriptorBindingUniformTexelBufferUpdateAfterBind,descriptorBindingStorageTexelBufferUpdateAfterBind=descriptorBindingStorageTexelBufferUpdateAfterBind,descriptorBindingUpdateUnusedWhilePending=descriptorBindingUpdateUnusedWhilePending,descriptorBindingPartiallyBound=descriptorBindingPartiallyBound,descriptorBindingVariableDescriptorCount=descriptorBindingVariableDescriptorCount,runtimeDescriptorArray=runtimeDescriptorArray)

def VkPhysicalDeviceDescriptorIndexingPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_INDEXING_PROPERTIES_EXT,pNext=None,maxUpdateAfterBindDescriptorsInAllPools=None,shaderUniformBufferArrayNonUniformIndexingNative=None,shaderSampledImageArrayNonUniformIndexingNative=None,shaderStorageBufferArrayNonUniformIndexingNative=None,shaderStorageImageArrayNonUniformIndexingNative=None,shaderInputAttachmentArrayNonUniformIndexingNative=None,robustBufferAccessUpdateAfterBind=None,quadDivergentImplicitLod=None,maxPerStageDescriptorUpdateAfterBindSamplers=None,maxPerStageDescriptorUpdateAfterBindUniformBuffers=None,maxPerStageDescriptorUpdateAfterBindStorageBuffers=None,maxPerStageDescriptorUpdateAfterBindSampledImages=None,maxPerStageDescriptorUpdateAfterBindStorageImages=None,maxPerStageDescriptorUpdateAfterBindInputAttachments=None,maxPerStageUpdateAfterBindResources=None,maxDescriptorSetUpdateAfterBindSamplers=None,maxDescriptorSetUpdateAfterBindUniformBuffers=None,maxDescriptorSetUpdateAfterBindUniformBuffersDynamic=None,maxDescriptorSetUpdateAfterBindStorageBuffers=None,maxDescriptorSetUpdateAfterBindStorageBuffersDynamic=None,maxDescriptorSetUpdateAfterBindSampledImages=None,maxDescriptorSetUpdateAfterBindStorageImages=None,maxDescriptorSetUpdateAfterBindInputAttachments=None,):

    return _new('VkPhysicalDeviceDescriptorIndexingPropertiesEXT', sType=sType,pNext=pNext,maxUpdateAfterBindDescriptorsInAllPools=maxUpdateAfterBindDescriptorsInAllPools,shaderUniformBufferArrayNonUniformIndexingNative=shaderUniformBufferArrayNonUniformIndexingNative,shaderSampledImageArrayNonUniformIndexingNative=shaderSampledImageArrayNonUniformIndexingNative,shaderStorageBufferArrayNonUniformIndexingNative=shaderStorageBufferArrayNonUniformIndexingNative,shaderStorageImageArrayNonUniformIndexingNative=shaderStorageImageArrayNonUniformIndexingNative,shaderInputAttachmentArrayNonUniformIndexingNative=shaderInputAttachmentArrayNonUniformIndexingNative,robustBufferAccessUpdateAfterBind=robustBufferAccessUpdateAfterBind,quadDivergentImplicitLod=quadDivergentImplicitLod,maxPerStageDescriptorUpdateAfterBindSamplers=maxPerStageDescriptorUpdateAfterBindSamplers,maxPerStageDescriptorUpdateAfterBindUniformBuffers=maxPerStageDescriptorUpdateAfterBindUniformBuffers,maxPerStageDescriptorUpdateAfterBindStorageBuffers=maxPerStageDescriptorUpdateAfterBindStorageBuffers,maxPerStageDescriptorUpdateAfterBindSampledImages=maxPerStageDescriptorUpdateAfterBindSampledImages,maxPerStageDescriptorUpdateAfterBindStorageImages=maxPerStageDescriptorUpdateAfterBindStorageImages,maxPerStageDescriptorUpdateAfterBindInputAttachments=maxPerStageDescriptorUpdateAfterBindInputAttachments,maxPerStageUpdateAfterBindResources=maxPerStageUpdateAfterBindResources,maxDescriptorSetUpdateAfterBindSamplers=maxDescriptorSetUpdateAfterBindSamplers,maxDescriptorSetUpdateAfterBindUniformBuffers=maxDescriptorSetUpdateAfterBindUniformBuffers,maxDescriptorSetUpdateAfterBindUniformBuffersDynamic=maxDescriptorSetUpdateAfterBindUniformBuffersDynamic,maxDescriptorSetUpdateAfterBindStorageBuffers=maxDescriptorSetUpdateAfterBindStorageBuffers,maxDescriptorSetUpdateAfterBindStorageBuffersDynamic=maxDescriptorSetUpdateAfterBindStorageBuffersDynamic,maxDescriptorSetUpdateAfterBindSampledImages=maxDescriptorSetUpdateAfterBindSampledImages,maxDescriptorSetUpdateAfterBindStorageImages=maxDescriptorSetUpdateAfterBindStorageImages,maxDescriptorSetUpdateAfterBindInputAttachments=maxDescriptorSetUpdateAfterBindInputAttachments)

def VkDescriptorSetLayoutBindingFlagsCreateInfoEXT(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_BINDING_FLAGS_CREATE_INFO_EXT,pNext=None,bindingCount=None,pBindingFlags=None,):
    if bindingCount is None and pBindingFlags is not None:
        bindingCount = len(pBindingFlags)

    return _new('VkDescriptorSetLayoutBindingFlagsCreateInfoEXT', sType=sType,pNext=pNext,bindingCount=bindingCount,pBindingFlags=pBindingFlags)

def VkDescriptorSetVariableDescriptorCountAllocateInfoEXT(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_SET_VARIABLE_DESCRIPTOR_COUNT_ALLOCATE_INFO_EXT,pNext=None,descriptorSetCount=None,pDescriptorCounts=None,):
    if descriptorSetCount is None and pDescriptorCounts is not None:
        descriptorSetCount = len(pDescriptorCounts)

    return _new('VkDescriptorSetVariableDescriptorCountAllocateInfoEXT', sType=sType,pNext=pNext,descriptorSetCount=descriptorSetCount,pDescriptorCounts=pDescriptorCounts)

def VkDescriptorSetVariableDescriptorCountLayoutSupportEXT(sType=VK_STRUCTURE_TYPE_DESCRIPTOR_SET_VARIABLE_DESCRIPTOR_COUNT_LAYOUT_SUPPORT_EXT,pNext=None,maxVariableDescriptorCount=None,):

    return _new('VkDescriptorSetVariableDescriptorCountLayoutSupportEXT', sType=sType,pNext=pNext,maxVariableDescriptorCount=maxVariableDescriptorCount)

def VkAttachmentDescription2KHR(sType=VK_STRUCTURE_TYPE_ATTACHMENT_DESCRIPTION_2_KHR,pNext=None,flags=None,format=None,samples=None,loadOp=None,storeOp=None,stencilLoadOp=None,stencilStoreOp=None,initialLayout=None,finalLayout=None,):

    return _new('VkAttachmentDescription2KHR', sType=sType,pNext=pNext,flags=flags,format=format,samples=samples,loadOp=loadOp,storeOp=storeOp,stencilLoadOp=stencilLoadOp,stencilStoreOp=stencilStoreOp,initialLayout=initialLayout,finalLayout=finalLayout)

def VkAttachmentReference2KHR(sType=VK_STRUCTURE_TYPE_ATTACHMENT_REFERENCE_2_KHR,pNext=None,attachment=None,layout=None,aspectMask=None,):

    return _new('VkAttachmentReference2KHR', sType=sType,pNext=pNext,attachment=attachment,layout=layout,aspectMask=aspectMask)

def VkSubpassDescription2KHR(sType=VK_STRUCTURE_TYPE_SUBPASS_DESCRIPTION_2_KHR,pNext=None,flags=None,pipelineBindPoint=None,viewMask=None,inputAttachmentCount=None,pInputAttachments=None,colorAttachmentCount=None,pColorAttachments=None,pResolveAttachments=None,pDepthStencilAttachment=None,preserveAttachmentCount=None,pPreserveAttachments=None,):
    if inputAttachmentCount is None and pInputAttachments is not None:
        inputAttachmentCount = len(pInputAttachments)
    if colorAttachmentCount is None and pColorAttachments is not None:
        colorAttachmentCount = len(pColorAttachments)
    if colorAttachmentCount is None and pResolveAttachments is not None:
        colorAttachmentCount = len(pResolveAttachments)
    if preserveAttachmentCount is None and pPreserveAttachments is not None:
        preserveAttachmentCount = len(pPreserveAttachments)

    return _new('VkSubpassDescription2KHR', sType=sType,pNext=pNext,flags=flags,pipelineBindPoint=pipelineBindPoint,viewMask=viewMask,inputAttachmentCount=inputAttachmentCount,pInputAttachments=pInputAttachments,colorAttachmentCount=colorAttachmentCount,pColorAttachments=pColorAttachments,pResolveAttachments=pResolveAttachments,pDepthStencilAttachment=pDepthStencilAttachment,preserveAttachmentCount=preserveAttachmentCount,pPreserveAttachments=pPreserveAttachments)

def VkSubpassDependency2KHR(sType=VK_STRUCTURE_TYPE_SUBPASS_DEPENDENCY_2_KHR,pNext=None,srcSubpass=None,dstSubpass=None,srcStageMask=None,dstStageMask=None,srcAccessMask=None,dstAccessMask=None,dependencyFlags=None,viewOffset=None,):

    return _new('VkSubpassDependency2KHR', sType=sType,pNext=pNext,srcSubpass=srcSubpass,dstSubpass=dstSubpass,srcStageMask=srcStageMask,dstStageMask=dstStageMask,srcAccessMask=srcAccessMask,dstAccessMask=dstAccessMask,dependencyFlags=dependencyFlags,viewOffset=viewOffset)

def VkRenderPassCreateInfo2KHR(sType=VK_STRUCTURE_TYPE_RENDER_PASS_CREATE_INFO_2_KHR,pNext=None,flags=None,attachmentCount=None,pAttachments=None,subpassCount=None,pSubpasses=None,dependencyCount=None,pDependencies=None,correlatedViewMaskCount=None,pCorrelatedViewMasks=None,):
    if attachmentCount is None and pAttachments is not None:
        attachmentCount = len(pAttachments)
    if subpassCount is None and pSubpasses is not None:
        subpassCount = len(pSubpasses)
    if dependencyCount is None and pDependencies is not None:
        dependencyCount = len(pDependencies)
    if correlatedViewMaskCount is None and pCorrelatedViewMasks is not None:
        correlatedViewMaskCount = len(pCorrelatedViewMasks)

    return _new('VkRenderPassCreateInfo2KHR', sType=sType,pNext=pNext,flags=flags,attachmentCount=attachmentCount,pAttachments=pAttachments,subpassCount=subpassCount,pSubpasses=pSubpasses,dependencyCount=dependencyCount,pDependencies=pDependencies,correlatedViewMaskCount=correlatedViewMaskCount,pCorrelatedViewMasks=pCorrelatedViewMasks)

def VkSubpassBeginInfoKHR(sType=VK_STRUCTURE_TYPE_SUBPASS_BEGIN_INFO_KHR,pNext=None,contents=None,):

    return _new('VkSubpassBeginInfoKHR', sType=sType,pNext=pNext,contents=contents)

def VkSubpassEndInfoKHR(sType=VK_STRUCTURE_TYPE_SUBPASS_END_INFO_KHR,pNext=None,):

    return _new('VkSubpassEndInfoKHR', sType=sType,pNext=pNext)

def VkVertexInputBindingDivisorDescriptionEXT(binding=None,divisor=None,):

    return _new('VkVertexInputBindingDivisorDescriptionEXT', binding=binding,divisor=divisor)

def VkPipelineVertexInputDivisorStateCreateInfoEXT(sType=VK_STRUCTURE_TYPE_PIPELINE_VERTEX_INPUT_DIVISOR_STATE_CREATE_INFO_EXT,pNext=None,vertexBindingDivisorCount=None,pVertexBindingDivisors=None,):
    if vertexBindingDivisorCount is None and pVertexBindingDivisors is not None:
        vertexBindingDivisorCount = len(pVertexBindingDivisors)

    return _new('VkPipelineVertexInputDivisorStateCreateInfoEXT', sType=sType,pNext=pNext,vertexBindingDivisorCount=vertexBindingDivisorCount,pVertexBindingDivisors=pVertexBindingDivisors)

def VkPhysicalDeviceVertexAttributeDivisorPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VERTEX_ATTRIBUTE_DIVISOR_PROPERTIES_EXT,pNext=None,maxVertexAttribDivisor=None,):

    return _new('VkPhysicalDeviceVertexAttributeDivisorPropertiesEXT', sType=sType,pNext=pNext,maxVertexAttribDivisor=maxVertexAttribDivisor)

def VkPhysicalDevicePCIBusInfoPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PCI_BUS_INFO_PROPERTIES_EXT,pNext=None,pciDomain=None,pciBus=None,pciDevice=None,pciFunction=None,):

    return _new('VkPhysicalDevicePCIBusInfoPropertiesEXT', sType=sType,pNext=pNext,pciDomain=pciDomain,pciBus=pciBus,pciDevice=pciDevice,pciFunction=pciFunction)

def VkImportAndroidHardwareBufferInfoANDROID(sType=VK_STRUCTURE_TYPE_IMPORT_ANDROID_HARDWARE_BUFFER_INFO_ANDROID,pNext=None,buffer=None,):

    return _new('VkImportAndroidHardwareBufferInfoANDROID', sType=sType,pNext=pNext,buffer=buffer)

def VkAndroidHardwareBufferUsageANDROID(sType=VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_USAGE_ANDROID,pNext=None,androidHardwareBufferUsage=None,):

    return _new('VkAndroidHardwareBufferUsageANDROID', sType=sType,pNext=pNext,androidHardwareBufferUsage=androidHardwareBufferUsage)

def VkAndroidHardwareBufferPropertiesANDROID(sType=VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_PROPERTIES_ANDROID,pNext=None,allocationSize=None,memoryTypeBits=None,):

    return _new('VkAndroidHardwareBufferPropertiesANDROID', sType=sType,pNext=pNext,allocationSize=allocationSize,memoryTypeBits=memoryTypeBits)

def VkMemoryGetAndroidHardwareBufferInfoANDROID(sType=VK_STRUCTURE_TYPE_MEMORY_GET_ANDROID_HARDWARE_BUFFER_INFO_ANDROID,pNext=None,memory=None,):

    return _new('VkMemoryGetAndroidHardwareBufferInfoANDROID', sType=sType,pNext=pNext,memory=memory)

def VkAndroidHardwareBufferFormatPropertiesANDROID(sType=VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_FORMAT_PROPERTIES_ANDROID,pNext=None,format=None,externalFormat=None,formatFeatures=None,samplerYcbcrConversionComponents=None,suggestedYcbcrModel=None,suggestedYcbcrRange=None,suggestedXChromaOffset=None,suggestedYChromaOffset=None,):

    return _new('VkAndroidHardwareBufferFormatPropertiesANDROID', sType=sType,pNext=pNext,format=format,externalFormat=externalFormat,formatFeatures=formatFeatures,samplerYcbcrConversionComponents=samplerYcbcrConversionComponents,suggestedYcbcrModel=suggestedYcbcrModel,suggestedYcbcrRange=suggestedYcbcrRange,suggestedXChromaOffset=suggestedXChromaOffset,suggestedYChromaOffset=suggestedYChromaOffset)

def VkCommandBufferInheritanceConditionalRenderingInfoEXT(sType=VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_CONDITIONAL_RENDERING_INFO_EXT,pNext=None,conditionalRenderingEnable=None,):

    return _new('VkCommandBufferInheritanceConditionalRenderingInfoEXT', sType=sType,pNext=pNext,conditionalRenderingEnable=conditionalRenderingEnable)

def VkExternalFormatANDROID(sType=VK_STRUCTURE_TYPE_EXTERNAL_FORMAT_ANDROID,pNext=None,externalFormat=None,):

    return _new('VkExternalFormatANDROID', sType=sType,pNext=pNext,externalFormat=externalFormat)

def VkPhysicalDevice8BitStorageFeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_8BIT_STORAGE_FEATURES_KHR,pNext=None,storageBuffer8BitAccess=None,uniformAndStorageBuffer8BitAccess=None,storagePushConstant8=None,):

    return _new('VkPhysicalDevice8BitStorageFeaturesKHR', sType=sType,pNext=pNext,storageBuffer8BitAccess=storageBuffer8BitAccess,uniformAndStorageBuffer8BitAccess=uniformAndStorageBuffer8BitAccess,storagePushConstant8=storagePushConstant8)

def VkPhysicalDeviceConditionalRenderingFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CONDITIONAL_RENDERING_FEATURES_EXT,pNext=None,conditionalRendering=None,inheritedConditionalRendering=None,):

    return _new('VkPhysicalDeviceConditionalRenderingFeaturesEXT', sType=sType,pNext=pNext,conditionalRendering=conditionalRendering,inheritedConditionalRendering=inheritedConditionalRendering)

def VkPhysicalDeviceVulkanMemoryModelFeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_MEMORY_MODEL_FEATURES_KHR,pNext=None,vulkanMemoryModel=None,vulkanMemoryModelDeviceScope=None,vulkanMemoryModelAvailabilityVisibilityChains=None,):

    return _new('VkPhysicalDeviceVulkanMemoryModelFeaturesKHR', sType=sType,pNext=pNext,vulkanMemoryModel=vulkanMemoryModel,vulkanMemoryModelDeviceScope=vulkanMemoryModelDeviceScope,vulkanMemoryModelAvailabilityVisibilityChains=vulkanMemoryModelAvailabilityVisibilityChains)

def VkPhysicalDeviceShaderAtomicInt64FeaturesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_ATOMIC_INT64_FEATURES_KHR,pNext=None,shaderBufferInt64Atomics=None,shaderSharedInt64Atomics=None,):

    return _new('VkPhysicalDeviceShaderAtomicInt64FeaturesKHR', sType=sType,pNext=pNext,shaderBufferInt64Atomics=shaderBufferInt64Atomics,shaderSharedInt64Atomics=shaderSharedInt64Atomics)

def VkPhysicalDeviceVertexAttributeDivisorFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VERTEX_ATTRIBUTE_DIVISOR_FEATURES_EXT,pNext=None,vertexAttributeInstanceRateDivisor=None,vertexAttributeInstanceRateZeroDivisor=None,):

    return _new('VkPhysicalDeviceVertexAttributeDivisorFeaturesEXT', sType=sType,pNext=pNext,vertexAttributeInstanceRateDivisor=vertexAttributeInstanceRateDivisor,vertexAttributeInstanceRateZeroDivisor=vertexAttributeInstanceRateZeroDivisor)

def VkQueueFamilyCheckpointPropertiesNV(sType=VK_STRUCTURE_TYPE_QUEUE_FAMILY_CHECKPOINT_PROPERTIES_NV,pNext=None,checkpointExecutionStageMask=None,):

    return _new('VkQueueFamilyCheckpointPropertiesNV', sType=sType,pNext=pNext,checkpointExecutionStageMask=checkpointExecutionStageMask)

def VkCheckpointDataNV(sType=VK_STRUCTURE_TYPE_CHECKPOINT_DATA_NV,pNext=None,stage=None,pCheckpointMarker=None,):

    return _new('VkCheckpointDataNV', sType=sType,pNext=pNext,stage=stage,pCheckpointMarker=pCheckpointMarker)

def VkPhysicalDeviceDepthStencilResolvePropertiesKHR(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEPTH_STENCIL_RESOLVE_PROPERTIES_KHR,pNext=None,supportedDepthResolveModes=None,supportedStencilResolveModes=None,independentResolveNone=None,independentResolve=None,):

    return _new('VkPhysicalDeviceDepthStencilResolvePropertiesKHR', sType=sType,pNext=pNext,supportedDepthResolveModes=supportedDepthResolveModes,supportedStencilResolveModes=supportedStencilResolveModes,independentResolveNone=independentResolveNone,independentResolve=independentResolve)

def VkSubpassDescriptionDepthStencilResolveKHR(sType=VK_STRUCTURE_TYPE_SUBPASS_DESCRIPTION_DEPTH_STENCIL_RESOLVE_KHR,pNext=None,depthResolveMode=None,stencilResolveMode=None,pDepthStencilResolveAttachment=None,):

    return _new('VkSubpassDescriptionDepthStencilResolveKHR', sType=sType,pNext=pNext,depthResolveMode=depthResolveMode,stencilResolveMode=stencilResolveMode,pDepthStencilResolveAttachment=pDepthStencilResolveAttachment)

def VkImageViewASTCDecodeModeEXT(sType=VK_STRUCTURE_TYPE_IMAGE_VIEW_ASTC_DECODE_MODE_EXT,pNext=None,decodeMode=None,):

    return _new('VkImageViewASTCDecodeModeEXT', sType=sType,pNext=pNext,decodeMode=decodeMode)

def VkPhysicalDeviceASTCDecodeFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ASTC_DECODE_FEATURES_EXT,pNext=None,decodeModeSharedExponent=None,):

    return _new('VkPhysicalDeviceASTCDecodeFeaturesEXT', sType=sType,pNext=pNext,decodeModeSharedExponent=decodeModeSharedExponent)

def VkPhysicalDeviceTransformFeedbackFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TRANSFORM_FEEDBACK_FEATURES_EXT,pNext=None,transformFeedback=None,geometryStreams=None,):

    return _new('VkPhysicalDeviceTransformFeedbackFeaturesEXT', sType=sType,pNext=pNext,transformFeedback=transformFeedback,geometryStreams=geometryStreams)

def VkPhysicalDeviceTransformFeedbackPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TRANSFORM_FEEDBACK_PROPERTIES_EXT,pNext=None,maxTransformFeedbackStreams=None,maxTransformFeedbackBuffers=None,maxTransformFeedbackBufferSize=None,maxTransformFeedbackStreamDataSize=None,maxTransformFeedbackBufferDataSize=None,maxTransformFeedbackBufferDataStride=None,transformFeedbackQueries=None,transformFeedbackStreamsLinesTriangles=None,transformFeedbackRasterizationStreamSelect=None,transformFeedbackDraw=None,):

    return _new('VkPhysicalDeviceTransformFeedbackPropertiesEXT', sType=sType,pNext=pNext,maxTransformFeedbackStreams=maxTransformFeedbackStreams,maxTransformFeedbackBuffers=maxTransformFeedbackBuffers,maxTransformFeedbackBufferSize=maxTransformFeedbackBufferSize,maxTransformFeedbackStreamDataSize=maxTransformFeedbackStreamDataSize,maxTransformFeedbackBufferDataSize=maxTransformFeedbackBufferDataSize,maxTransformFeedbackBufferDataStride=maxTransformFeedbackBufferDataStride,transformFeedbackQueries=transformFeedbackQueries,transformFeedbackStreamsLinesTriangles=transformFeedbackStreamsLinesTriangles,transformFeedbackRasterizationStreamSelect=transformFeedbackRasterizationStreamSelect,transformFeedbackDraw=transformFeedbackDraw)

def VkPipelineRasterizationStateStreamCreateInfoEXT(sType=VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_STREAM_CREATE_INFO_EXT,pNext=None,flags=None,rasterizationStream=None,):

    return _new('VkPipelineRasterizationStateStreamCreateInfoEXT', sType=sType,pNext=pNext,flags=flags,rasterizationStream=rasterizationStream)

def VkPhysicalDeviceRepresentativeFragmentTestFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_REPRESENTATIVE_FRAGMENT_TEST_FEATURES_NV,pNext=None,representativeFragmentTest=None,):

    return _new('VkPhysicalDeviceRepresentativeFragmentTestFeaturesNV', sType=sType,pNext=pNext,representativeFragmentTest=representativeFragmentTest)

def VkPipelineRepresentativeFragmentTestStateCreateInfoNV(sType=VK_STRUCTURE_TYPE_PIPELINE_REPRESENTATIVE_FRAGMENT_TEST_STATE_CREATE_INFO_NV,pNext=None,representativeFragmentTestEnable=None,):

    return _new('VkPipelineRepresentativeFragmentTestStateCreateInfoNV', sType=sType,pNext=pNext,representativeFragmentTestEnable=representativeFragmentTestEnable)

def VkPhysicalDeviceExclusiveScissorFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXCLUSIVE_SCISSOR_FEATURES_NV,pNext=None,exclusiveScissor=None,):

    return _new('VkPhysicalDeviceExclusiveScissorFeaturesNV', sType=sType,pNext=pNext,exclusiveScissor=exclusiveScissor)

def VkPipelineViewportExclusiveScissorStateCreateInfoNV(sType=VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_EXCLUSIVE_SCISSOR_STATE_CREATE_INFO_NV,pNext=None,exclusiveScissorCount=None,pExclusiveScissors=None,):
    if exclusiveScissorCount is None and pExclusiveScissors is not None:
        exclusiveScissorCount = len(pExclusiveScissors)

    return _new('VkPipelineViewportExclusiveScissorStateCreateInfoNV', sType=sType,pNext=pNext,exclusiveScissorCount=exclusiveScissorCount,pExclusiveScissors=pExclusiveScissors)

def VkPhysicalDeviceCornerSampledImageFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CORNER_SAMPLED_IMAGE_FEATURES_NV,pNext=None,cornerSampledImage=None,):

    return _new('VkPhysicalDeviceCornerSampledImageFeaturesNV', sType=sType,pNext=pNext,cornerSampledImage=cornerSampledImage)

def VkPhysicalDeviceComputeShaderDerivativesFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COMPUTE_SHADER_DERIVATIVES_FEATURES_NV,pNext=None,computeDerivativeGroupQuads=None,computeDerivativeGroupLinear=None,):

    return _new('VkPhysicalDeviceComputeShaderDerivativesFeaturesNV', sType=sType,pNext=pNext,computeDerivativeGroupQuads=computeDerivativeGroupQuads,computeDerivativeGroupLinear=computeDerivativeGroupLinear)

def VkPhysicalDeviceFragmentShaderBarycentricFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADER_BARYCENTRIC_FEATURES_NV,pNext=None,fragmentShaderBarycentric=None,):

    return _new('VkPhysicalDeviceFragmentShaderBarycentricFeaturesNV', sType=sType,pNext=pNext,fragmentShaderBarycentric=fragmentShaderBarycentric)

def VkPhysicalDeviceShaderImageFootprintFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_IMAGE_FOOTPRINT_FEATURES_NV,pNext=None,imageFootprint=None,):

    return _new('VkPhysicalDeviceShaderImageFootprintFeaturesNV', sType=sType,pNext=pNext,imageFootprint=imageFootprint)

def VkPhysicalDeviceDedicatedAllocationImageAliasingFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEDICATED_ALLOCATION_IMAGE_ALIASING_FEATURES_NV,pNext=None,dedicatedAllocationImageAliasing=None,):

    return _new('VkPhysicalDeviceDedicatedAllocationImageAliasingFeaturesNV', sType=sType,pNext=pNext,dedicatedAllocationImageAliasing=dedicatedAllocationImageAliasing)

def VkShadingRatePaletteNV(shadingRatePaletteEntryCount=None,pShadingRatePaletteEntries=None,):
    if shadingRatePaletteEntryCount is None and pShadingRatePaletteEntries is not None:
        shadingRatePaletteEntryCount = len(pShadingRatePaletteEntries)

    return _new('VkShadingRatePaletteNV', shadingRatePaletteEntryCount=shadingRatePaletteEntryCount,pShadingRatePaletteEntries=pShadingRatePaletteEntries)

def VkPipelineViewportShadingRateImageStateCreateInfoNV(sType=VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_SHADING_RATE_IMAGE_STATE_CREATE_INFO_NV,pNext=None,shadingRateImageEnable=None,viewportCount=None,pShadingRatePalettes=None,):
    if viewportCount is None and pShadingRatePalettes is not None:
        viewportCount = len(pShadingRatePalettes)

    return _new('VkPipelineViewportShadingRateImageStateCreateInfoNV', sType=sType,pNext=pNext,shadingRateImageEnable=shadingRateImageEnable,viewportCount=viewportCount,pShadingRatePalettes=pShadingRatePalettes)

def VkPhysicalDeviceShadingRateImageFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADING_RATE_IMAGE_FEATURES_NV,pNext=None,shadingRateImage=None,shadingRateCoarseSampleOrder=None,):

    return _new('VkPhysicalDeviceShadingRateImageFeaturesNV', sType=sType,pNext=pNext,shadingRateImage=shadingRateImage,shadingRateCoarseSampleOrder=shadingRateCoarseSampleOrder)

def VkPhysicalDeviceShadingRateImagePropertiesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADING_RATE_IMAGE_PROPERTIES_NV,pNext=None,shadingRateTexelSize=None,shadingRatePaletteSize=None,shadingRateMaxCoarseSamples=None,):

    return _new('VkPhysicalDeviceShadingRateImagePropertiesNV', sType=sType,pNext=pNext,shadingRateTexelSize=shadingRateTexelSize,shadingRatePaletteSize=shadingRatePaletteSize,shadingRateMaxCoarseSamples=shadingRateMaxCoarseSamples)

def VkCoarseSampleLocationNV(pixelX=None,pixelY=None,sample=None,):

    return _new('VkCoarseSampleLocationNV', pixelX=pixelX,pixelY=pixelY,sample=sample)

def VkCoarseSampleOrderCustomNV(shadingRate=None,sampleCount=None,sampleLocationCount=None,pSampleLocations=None,):
    if sampleLocationCount is None and pSampleLocations is not None:
        sampleLocationCount = len(pSampleLocations)

    return _new('VkCoarseSampleOrderCustomNV', shadingRate=shadingRate,sampleCount=sampleCount,sampleLocationCount=sampleLocationCount,pSampleLocations=pSampleLocations)

def VkPipelineViewportCoarseSampleOrderStateCreateInfoNV(sType=VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_COARSE_SAMPLE_ORDER_STATE_CREATE_INFO_NV,pNext=None,sampleOrderType=None,customSampleOrderCount=None,pCustomSampleOrders=None,):
    if customSampleOrderCount is None and pCustomSampleOrders is not None:
        customSampleOrderCount = len(pCustomSampleOrders)

    return _new('VkPipelineViewportCoarseSampleOrderStateCreateInfoNV', sType=sType,pNext=pNext,sampleOrderType=sampleOrderType,customSampleOrderCount=customSampleOrderCount,pCustomSampleOrders=pCustomSampleOrders)

def VkPhysicalDeviceMeshShaderFeaturesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MESH_SHADER_FEATURES_NV,pNext=None,taskShader=None,meshShader=None,):

    return _new('VkPhysicalDeviceMeshShaderFeaturesNV', sType=sType,pNext=pNext,taskShader=taskShader,meshShader=meshShader)

def VkPhysicalDeviceMeshShaderPropertiesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MESH_SHADER_PROPERTIES_NV,pNext=None,maxDrawMeshTasksCount=None,maxTaskWorkGroupInvocations=None,maxTaskWorkGroupSize=None,maxTaskTotalMemorySize=None,maxTaskOutputCount=None,maxMeshWorkGroupInvocations=None,maxMeshWorkGroupSize=None,maxMeshTotalMemorySize=None,maxMeshOutputVertices=None,maxMeshOutputPrimitives=None,maxMeshMultiviewViewCount=None,meshOutputPerVertexGranularity=None,meshOutputPerPrimitiveGranularity=None,):

    return _new('VkPhysicalDeviceMeshShaderPropertiesNV', sType=sType,pNext=pNext,maxDrawMeshTasksCount=maxDrawMeshTasksCount,maxTaskWorkGroupInvocations=maxTaskWorkGroupInvocations,maxTaskWorkGroupSize=maxTaskWorkGroupSize,maxTaskTotalMemorySize=maxTaskTotalMemorySize,maxTaskOutputCount=maxTaskOutputCount,maxMeshWorkGroupInvocations=maxMeshWorkGroupInvocations,maxMeshWorkGroupSize=maxMeshWorkGroupSize,maxMeshTotalMemorySize=maxMeshTotalMemorySize,maxMeshOutputVertices=maxMeshOutputVertices,maxMeshOutputPrimitives=maxMeshOutputPrimitives,maxMeshMultiviewViewCount=maxMeshMultiviewViewCount,meshOutputPerVertexGranularity=meshOutputPerVertexGranularity,meshOutputPerPrimitiveGranularity=meshOutputPerPrimitiveGranularity)

def VkDrawMeshTasksIndirectCommandNV(taskCount=None,firstTask=None,):

    return _new('VkDrawMeshTasksIndirectCommandNV', taskCount=taskCount,firstTask=firstTask)

def VkRayTracingShaderGroupCreateInfoNV(sType=VK_STRUCTURE_TYPE_RAY_TRACING_SHADER_GROUP_CREATE_INFO_NV,pNext=None,type=None,generalShader=None,closestHitShader=None,anyHitShader=None,intersectionShader=None,):

    return _new('VkRayTracingShaderGroupCreateInfoNV', sType=sType,pNext=pNext,type=type,generalShader=generalShader,closestHitShader=closestHitShader,anyHitShader=anyHitShader,intersectionShader=intersectionShader)

def VkRayTracingPipelineCreateInfoNV(sType=VK_STRUCTURE_TYPE_RAY_TRACING_PIPELINE_CREATE_INFO_NV,pNext=None,flags=None,stageCount=None,pStages=None,groupCount=None,pGroups=None,maxRecursionDepth=None,layout=None,basePipelineHandle=None,basePipelineIndex=None,):
    if stageCount is None and pStages is not None:
        stageCount = len(pStages)
    if groupCount is None and pGroups is not None:
        groupCount = len(pGroups)

    return _new('VkRayTracingPipelineCreateInfoNV', sType=sType,pNext=pNext,flags=flags,stageCount=stageCount,pStages=pStages,groupCount=groupCount,pGroups=pGroups,maxRecursionDepth=maxRecursionDepth,layout=layout,basePipelineHandle=basePipelineHandle,basePipelineIndex=basePipelineIndex)

def VkGeometryTrianglesNV(sType=VK_STRUCTURE_TYPE_GEOMETRY_TRIANGLES_NV,pNext=None,vertexData=None,vertexOffset=None,vertexCount=None,vertexStride=None,vertexFormat=None,indexData=None,indexOffset=None,indexCount=None,indexType=None,transformData=None,transformOffset=None,):

    return _new('VkGeometryTrianglesNV', sType=sType,pNext=pNext,vertexData=vertexData,vertexOffset=vertexOffset,vertexCount=vertexCount,vertexStride=vertexStride,vertexFormat=vertexFormat,indexData=indexData,indexOffset=indexOffset,indexCount=indexCount,indexType=indexType,transformData=transformData,transformOffset=transformOffset)

def VkGeometryAABBNV(sType=VK_STRUCTURE_TYPE_GEOMETRY_AABB_NV,pNext=None,aabbData=None,numAABBs=None,stride=None,offset=None,):

    return _new('VkGeometryAABBNV', sType=sType,pNext=pNext,aabbData=aabbData,numAABBs=numAABBs,stride=stride,offset=offset)

def VkGeometryDataNV(triangles=None,aabbs=None,):

    return _new('VkGeometryDataNV', triangles=triangles,aabbs=aabbs)

def VkGeometryNV(sType=VK_STRUCTURE_TYPE_GEOMETRY_NV,pNext=None,geometryType=None,geometry=None,flags=None,):

    return _new('VkGeometryNV', sType=sType,pNext=pNext,geometryType=geometryType,geometry=geometry,flags=flags)

def VkAccelerationStructureInfoNV(sType=VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_INFO_NV,pNext=None,type=None,flags=None,instanceCount=None,geometryCount=None,pGeometries=None,):
    if geometryCount is None and pGeometries is not None:
        geometryCount = len(pGeometries)

    return _new('VkAccelerationStructureInfoNV', sType=sType,pNext=pNext,type=type,flags=flags,instanceCount=instanceCount,geometryCount=geometryCount,pGeometries=pGeometries)

def VkAccelerationStructureCreateInfoNV(sType=VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_CREATE_INFO_NV,pNext=None,compactedSize=None,info=None,):

    return _new('VkAccelerationStructureCreateInfoNV', sType=sType,pNext=pNext,compactedSize=compactedSize,info=info)

def VkBindAccelerationStructureMemoryInfoNV(sType=VK_STRUCTURE_TYPE_BIND_ACCELERATION_STRUCTURE_MEMORY_INFO_NV,pNext=None,accelerationStructure=None,memory=None,memoryOffset=None,deviceIndexCount=None,pDeviceIndices=None,):
    if deviceIndexCount is None and pDeviceIndices is not None:
        deviceIndexCount = len(pDeviceIndices)

    return _new('VkBindAccelerationStructureMemoryInfoNV', sType=sType,pNext=pNext,accelerationStructure=accelerationStructure,memory=memory,memoryOffset=memoryOffset,deviceIndexCount=deviceIndexCount,pDeviceIndices=pDeviceIndices)

def VkWriteDescriptorSetAccelerationStructureNV(sType=VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET_ACCELERATION_STRUCTURE_NV,pNext=None,accelerationStructureCount=None,pAccelerationStructures=None,):
    if accelerationStructureCount is None and pAccelerationStructures is not None:
        accelerationStructureCount = len(pAccelerationStructures)

    return _new('VkWriteDescriptorSetAccelerationStructureNV', sType=sType,pNext=pNext,accelerationStructureCount=accelerationStructureCount,pAccelerationStructures=pAccelerationStructures)

def VkAccelerationStructureMemoryRequirementsInfoNV(sType=VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_INFO_NV,pNext=None,type=None,accelerationStructure=None,):

    return _new('VkAccelerationStructureMemoryRequirementsInfoNV', sType=sType,pNext=pNext,type=type,accelerationStructure=accelerationStructure)

def VkPhysicalDeviceRayTracingPropertiesNV(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_TRACING_PROPERTIES_NV,pNext=None,shaderGroupHandleSize=None,maxRecursionDepth=None,maxShaderGroupStride=None,shaderGroupBaseAlignment=None,maxGeometryCount=None,maxInstanceCount=None,maxTriangleCount=None,maxDescriptorSetAccelerationStructures=None,):

    return _new('VkPhysicalDeviceRayTracingPropertiesNV', sType=sType,pNext=pNext,shaderGroupHandleSize=shaderGroupHandleSize,maxRecursionDepth=maxRecursionDepth,maxShaderGroupStride=maxShaderGroupStride,shaderGroupBaseAlignment=shaderGroupBaseAlignment,maxGeometryCount=maxGeometryCount,maxInstanceCount=maxInstanceCount,maxTriangleCount=maxTriangleCount,maxDescriptorSetAccelerationStructures=maxDescriptorSetAccelerationStructures)

def VkDrmFormatModifierPropertiesListEXT(sType=VK_STRUCTURE_TYPE_DRM_FORMAT_MODIFIER_PROPERTIES_LIST_EXT,pNext=None,drmFormatModifierCount=None,pDrmFormatModifierProperties=None,):
    if drmFormatModifierCount is None and pDrmFormatModifierProperties is not None:
        drmFormatModifierCount = len(pDrmFormatModifierProperties)

    return _new('VkDrmFormatModifierPropertiesListEXT', sType=sType,pNext=pNext,drmFormatModifierCount=drmFormatModifierCount,pDrmFormatModifierProperties=pDrmFormatModifierProperties)

def VkDrmFormatModifierPropertiesEXT(drmFormatModifier=None,drmFormatModifierPlaneCount=None,drmFormatModifierTilingFeatures=None,):

    return _new('VkDrmFormatModifierPropertiesEXT', drmFormatModifier=drmFormatModifier,drmFormatModifierPlaneCount=drmFormatModifierPlaneCount,drmFormatModifierTilingFeatures=drmFormatModifierTilingFeatures)

def VkPhysicalDeviceImageDrmFormatModifierInfoEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_DRM_FORMAT_MODIFIER_INFO_EXT,pNext=None,drmFormatModifier=None,sharingMode=None,queueFamilyIndexCount=None,pQueueFamilyIndices=None,):
    if queueFamilyIndexCount is None and pQueueFamilyIndices is not None:
        queueFamilyIndexCount = len(pQueueFamilyIndices)

    return _new('VkPhysicalDeviceImageDrmFormatModifierInfoEXT', sType=sType,pNext=pNext,drmFormatModifier=drmFormatModifier,sharingMode=sharingMode,queueFamilyIndexCount=queueFamilyIndexCount,pQueueFamilyIndices=pQueueFamilyIndices)

def VkImageDrmFormatModifierListCreateInfoEXT(sType=VK_STRUCTURE_TYPE_IMAGE_DRM_FORMAT_MODIFIER_LIST_CREATE_INFO_EXT,pNext=None,drmFormatModifierCount=None,pDrmFormatModifiers=None,):
    if drmFormatModifierCount is None and pDrmFormatModifiers is not None:
        drmFormatModifierCount = len(pDrmFormatModifiers)

    return _new('VkImageDrmFormatModifierListCreateInfoEXT', sType=sType,pNext=pNext,drmFormatModifierCount=drmFormatModifierCount,pDrmFormatModifiers=pDrmFormatModifiers)

def VkImageDrmFormatModifierExplicitCreateInfoEXT(sType=VK_STRUCTURE_TYPE_IMAGE_DRM_FORMAT_MODIFIER_EXPLICIT_CREATE_INFO_EXT,pNext=None,drmFormatModifier=None,drmFormatModifierPlaneCount=None,pPlaneLayouts=None,):
    if drmFormatModifierPlaneCount is None and pPlaneLayouts is not None:
        drmFormatModifierPlaneCount = len(pPlaneLayouts)

    return _new('VkImageDrmFormatModifierExplicitCreateInfoEXT', sType=sType,pNext=pNext,drmFormatModifier=drmFormatModifier,drmFormatModifierPlaneCount=drmFormatModifierPlaneCount,pPlaneLayouts=pPlaneLayouts)

def VkImageDrmFormatModifierPropertiesEXT(sType=VK_STRUCTURE_TYPE_IMAGE_DRM_FORMAT_MODIFIER_PROPERTIES_EXT,pNext=None,drmFormatModifier=None,):

    return _new('VkImageDrmFormatModifierPropertiesEXT', sType=sType,pNext=pNext,drmFormatModifier=drmFormatModifier)

def VkImageStencilUsageCreateInfoEXT(sType=VK_STRUCTURE_TYPE_IMAGE_STENCIL_USAGE_CREATE_INFO_EXT,pNext=None,stencilUsage=None,):

    return _new('VkImageStencilUsageCreateInfoEXT', sType=sType,pNext=pNext,stencilUsage=stencilUsage)

def VkDeviceMemoryOverallocationCreateInfoAMD(sType=VK_STRUCTURE_TYPE_DEVICE_MEMORY_OVERALLOCATION_CREATE_INFO_AMD,pNext=None,overallocationBehavior=None,):

    return _new('VkDeviceMemoryOverallocationCreateInfoAMD', sType=sType,pNext=pNext,overallocationBehavior=overallocationBehavior)

def VkPhysicalDeviceFragmentDensityMapFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_FEATURES_EXT,pNext=None,fragmentDensityMap=None,fragmentDensityMapDynamic=None,fragmentDensityMapNonSubsampledImages=None,):

    return _new('VkPhysicalDeviceFragmentDensityMapFeaturesEXT', sType=sType,pNext=pNext,fragmentDensityMap=fragmentDensityMap,fragmentDensityMapDynamic=fragmentDensityMapDynamic,fragmentDensityMapNonSubsampledImages=fragmentDensityMapNonSubsampledImages)

def VkPhysicalDeviceFragmentDensityMapPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_PROPERTIES_EXT,pNext=None,minFragmentDensityTexelSize=None,maxFragmentDensityTexelSize=None,fragmentDensityInvocations=None,):

    return _new('VkPhysicalDeviceFragmentDensityMapPropertiesEXT', sType=sType,pNext=pNext,minFragmentDensityTexelSize=minFragmentDensityTexelSize,maxFragmentDensityTexelSize=maxFragmentDensityTexelSize,fragmentDensityInvocations=fragmentDensityInvocations)

def VkRenderPassFragmentDensityMapCreateInfoEXT(sType=VK_STRUCTURE_TYPE_RENDER_PASS_FRAGMENT_DENSITY_MAP_CREATE_INFO_EXT,pNext=None,fragmentDensityMapAttachment=None,):

    return _new('VkRenderPassFragmentDensityMapCreateInfoEXT', sType=sType,pNext=pNext,fragmentDensityMapAttachment=fragmentDensityMapAttachment)

def VkPhysicalDeviceScalarBlockLayoutFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SCALAR_BLOCK_LAYOUT_FEATURES_EXT,pNext=None,scalarBlockLayout=None,):

    return _new('VkPhysicalDeviceScalarBlockLayoutFeaturesEXT', sType=sType,pNext=pNext,scalarBlockLayout=scalarBlockLayout)

def VkPhysicalDeviceMemoryBudgetPropertiesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_BUDGET_PROPERTIES_EXT,pNext=None,heapBudget=None,heapUsage=None,):

    return _new('VkPhysicalDeviceMemoryBudgetPropertiesEXT', sType=sType,pNext=pNext,heapBudget=heapBudget,heapUsage=heapUsage)

def VkPhysicalDeviceMemoryPriorityFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_PRIORITY_FEATURES_EXT,pNext=None,memoryPriority=None,):

    return _new('VkPhysicalDeviceMemoryPriorityFeaturesEXT', sType=sType,pNext=pNext,memoryPriority=memoryPriority)

def VkMemoryPriorityAllocateInfoEXT(sType=VK_STRUCTURE_TYPE_MEMORY_PRIORITY_ALLOCATE_INFO_EXT,pNext=None,priority=None,):

    return _new('VkMemoryPriorityAllocateInfoEXT', sType=sType,pNext=pNext,priority=priority)

def VkPhysicalDeviceBufferAddressFeaturesEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BUFFER_ADDRESS_FEATURES_EXT,pNext=None,bufferDeviceAddress=None,bufferDeviceAddressCaptureReplay=None,bufferDeviceAddressMultiDevice=None,):

    return _new('VkPhysicalDeviceBufferAddressFeaturesEXT', sType=sType,pNext=pNext,bufferDeviceAddress=bufferDeviceAddress,bufferDeviceAddressCaptureReplay=bufferDeviceAddressCaptureReplay,bufferDeviceAddressMultiDevice=bufferDeviceAddressMultiDevice)

def VkBufferDeviceAddressInfoEXT(sType=VK_STRUCTURE_TYPE_BUFFER_DEVICE_ADDRESS_INFO_EXT,pNext=None,buffer=None,):

    return _new('VkBufferDeviceAddressInfoEXT', sType=sType,pNext=pNext,buffer=buffer)

def VkBufferDeviceAddressCreateInfoEXT(sType=VK_STRUCTURE_TYPE_BUFFER_DEVICE_ADDRESS_CREATE_INFO_EXT,pNext=None,deviceAddress=None,):

    return _new('VkBufferDeviceAddressCreateInfoEXT', sType=sType,pNext=pNext,deviceAddress=deviceAddress)

def VkPhysicalDeviceImageViewImageFormatInfoEXT(sType=VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_VIEW_IMAGE_FORMAT_INFO_EXT,pNext=None,imageViewType=None,):

    return _new('VkPhysicalDeviceImageViewImageFormatInfoEXT', sType=sType,pNext=pNext,imageViewType=imageViewType)

def VkFilterCubicImageViewImageFormatPropertiesEXT(sType=VK_STRUCTURE_TYPE_FILTER_CUBIC_IMAGE_VIEW_IMAGE_FORMAT_PROPERTIES_EXT,pNext=None,filterCubic=None,filterCubicMinmax=None,):

    return _new('VkFilterCubicImageViewImageFormatPropertiesEXT', sType=sType,pNext=pNext,filterCubic=filterCubic,filterCubicMinmax=filterCubicMinmax)



def _auto_handle(x, _type):
    if x is None:
        return ffi.NULL
    if _type.kind == 'pointer':
        ptr, _ = _cast_ptr(x, _type)
        return ptr
    return x


def _callApi(fn, *args):
    fn_args = [_auto_handle(i, j) for i, j in zip(args, ffi.typeof(fn).args)]
    return fn(*fn_args)







def vkCreateInstance(
pCreateInfo
        ,pAllocator
        ,pInstance=None
        ,):

    custom_return = True
    if not pInstance:
        pInstance = ffi.new('VkInstance*')
        custom_return = False

    result = _callApi(lib.vkCreateInstance, pCreateInfo,pAllocator,pInstance)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pInstance

    return pInstance[0]



def vkDestroyInstance(
instance
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyInstance, instance,pAllocator)


def vkEnumeratePhysicalDevices(
instance
        ,):

    pPhysicalDeviceCount = ffi.new('uint32_t*')
    pPhysicalDevices = ffi.NULL

    result = _callApi(lib.vkEnumeratePhysicalDevices, instance,pPhysicalDeviceCount,pPhysicalDevices)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    pPhysicalDevices = ffi.new('VkPhysicalDevice[]', pPhysicalDeviceCount[0])
    result = _callApi(lib.vkEnumeratePhysicalDevices, instance,pPhysicalDeviceCount,pPhysicalDevices)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    return pPhysicalDevices


def vkGetPhysicalDeviceProperties(
physicalDevice
        ,pProperties=None
        ,):

    custom_return = True
    if not pProperties:
        pProperties = ffi.new('VkPhysicalDeviceProperties*')
        custom_return = False

    result = _callApi(lib.vkGetPhysicalDeviceProperties, physicalDevice,pProperties)

    if custom_return:
        return pProperties

    return StrWrap(pProperties[0])



def vkGetPhysicalDeviceQueueFamilyProperties(
physicalDevice
        ,):

    pQueueFamilyPropertyCount = ffi.new('uint32_t*')
    pQueueFamilyProperties = ffi.NULL

    result = _callApi(lib.vkGetPhysicalDeviceQueueFamilyProperties, physicalDevice,pQueueFamilyPropertyCount,pQueueFamilyProperties)

    pQueueFamilyProperties = ffi.new('VkQueueFamilyProperties[]', pQueueFamilyPropertyCount[0])
    result = _callApi(lib.vkGetPhysicalDeviceQueueFamilyProperties, physicalDevice,pQueueFamilyPropertyCount,pQueueFamilyProperties)

    return pQueueFamilyProperties


def vkGetPhysicalDeviceMemoryProperties(
physicalDevice
        ,pMemoryProperties=None
        ,):

    custom_return = True
    if not pMemoryProperties:
        pMemoryProperties = ffi.new('VkPhysicalDeviceMemoryProperties*')
        custom_return = False

    result = _callApi(lib.vkGetPhysicalDeviceMemoryProperties, physicalDevice,pMemoryProperties)

    if custom_return:
        return pMemoryProperties

    return pMemoryProperties[0]



def vkGetPhysicalDeviceFeatures(
physicalDevice
        ,pFeatures=None
        ,):

    custom_return = True
    if not pFeatures:
        pFeatures = ffi.new('VkPhysicalDeviceFeatures*')
        custom_return = False

    result = _callApi(lib.vkGetPhysicalDeviceFeatures, physicalDevice,pFeatures)

    if custom_return:
        return pFeatures

    return pFeatures[0]



def vkGetPhysicalDeviceFormatProperties(
physicalDevice
        ,format
        ,pFormatProperties=None
        ,):

    custom_return = True
    if not pFormatProperties:
        pFormatProperties = ffi.new('VkFormatProperties*')
        custom_return = False

    result = _callApi(lib.vkGetPhysicalDeviceFormatProperties, physicalDevice,format,pFormatProperties)

    if custom_return:
        return pFormatProperties

    return pFormatProperties[0]



def vkGetPhysicalDeviceImageFormatProperties(
physicalDevice
        ,format
        ,type
        ,tiling
        ,usage
        ,flags
        ,pImageFormatProperties=None
        ,):

    custom_return = True
    if not pImageFormatProperties:
        pImageFormatProperties = ffi.new('VkImageFormatProperties*')
        custom_return = False

    result = _callApi(lib.vkGetPhysicalDeviceImageFormatProperties, physicalDevice,format,type,tiling,usage,flags,pImageFormatProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pImageFormatProperties

    return pImageFormatProperties[0]



def vkCreateDevice(
physicalDevice
        ,pCreateInfo
        ,pAllocator
        ,pDevice=None
        ,):

    custom_return = True
    if not pDevice:
        pDevice = ffi.new('VkDevice*')
        custom_return = False

    result = _callApi(lib.vkCreateDevice, physicalDevice,pCreateInfo,pAllocator,pDevice)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pDevice

    return pDevice[0]



def vkDestroyDevice(
device
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyDevice, device,pAllocator)


def vkEnumerateInstanceVersion(
pApiVersion=None
        ,):

    custom_return = True
    if not pApiVersion:
        pApiVersion = ffi.new('uint32_t*')
        custom_return = False

    result = _callApi(lib.vkEnumerateInstanceVersion, pApiVersion)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pApiVersion

    return pApiVersion[0]



def vkEnumerateInstanceLayerProperties(
):

    pPropertyCount = ffi.new('uint32_t*')
    pProperties = ffi.NULL

    result = _callApi(lib.vkEnumerateInstanceLayerProperties, pPropertyCount,pProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    pProperties = ffi.new('VkLayerProperties[]', pPropertyCount[0])
    result = _callApi(lib.vkEnumerateInstanceLayerProperties, pPropertyCount,pProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    result = (StrWrap(x) for x in pProperties)
    _weakkey_dict[result] = pProperties
    return result


def vkEnumerateInstanceExtensionProperties(
pLayerName
        ,):

    pPropertyCount = ffi.new('uint32_t*')
    pProperties = ffi.NULL

    result = _callApi(lib.vkEnumerateInstanceExtensionProperties, pLayerName,pPropertyCount,pProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    pProperties = ffi.new('VkExtensionProperties[]', pPropertyCount[0])
    result = _callApi(lib.vkEnumerateInstanceExtensionProperties, pLayerName,pPropertyCount,pProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    result = (StrWrap(x) for x in pProperties)
    _weakkey_dict[result] = pProperties
    return result


def vkEnumerateDeviceLayerProperties(
physicalDevice
        ,):

    pPropertyCount = ffi.new('uint32_t*')
    pProperties = ffi.NULL

    result = _callApi(lib.vkEnumerateDeviceLayerProperties, physicalDevice,pPropertyCount,pProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    pProperties = ffi.new('VkLayerProperties[]', pPropertyCount[0])
    result = _callApi(lib.vkEnumerateDeviceLayerProperties, physicalDevice,pPropertyCount,pProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    result = (StrWrap(x) for x in pProperties)
    _weakkey_dict[result] = pProperties
    return result


def vkEnumerateDeviceExtensionProperties(
physicalDevice
        ,pLayerName
        ,):

    pPropertyCount = ffi.new('uint32_t*')
    pProperties = ffi.NULL

    result = _callApi(lib.vkEnumerateDeviceExtensionProperties, physicalDevice,pLayerName,pPropertyCount,pProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    pProperties = ffi.new('VkExtensionProperties[]', pPropertyCount[0])
    result = _callApi(lib.vkEnumerateDeviceExtensionProperties, physicalDevice,pLayerName,pPropertyCount,pProperties)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    result = (StrWrap(x) for x in pProperties)
    _weakkey_dict[result] = pProperties
    return result


def vkGetDeviceQueue(
device
        ,queueFamilyIndex
        ,queueIndex
        ,pQueue=None
        ,):

    custom_return = True
    if not pQueue:
        pQueue = ffi.new('VkQueue*')
        custom_return = False

    result = _callApi(lib.vkGetDeviceQueue, device,queueFamilyIndex,queueIndex,pQueue)

    if custom_return:
        return pQueue

    return pQueue[0]



def vkQueueSubmit(
queue
        ,submitCount
        ,pSubmits
        ,fence
        ,):
    result = _callApi(lib.vkQueueSubmit, queue,submitCount,pSubmits,fence)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkQueueWaitIdle(
queue
        ,):
    result = _callApi(lib.vkQueueWaitIdle, queue)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkDeviceWaitIdle(
device
        ,):
    result = _callApi(lib.vkDeviceWaitIdle, device)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkAllocateMemory(
device
        ,pAllocateInfo
        ,pAllocator
        ,pMemory=None
        ,):

    custom_return = True
    if not pMemory:
        pMemory = ffi.new('VkDeviceMemory*')
        custom_return = False

    result = _callApi(lib.vkAllocateMemory, device,pAllocateInfo,pAllocator,pMemory)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pMemory

    return pMemory[0]



def vkFreeMemory(
device
        ,memory
        ,pAllocator
        ,):
    result = _callApi(lib.vkFreeMemory, device,memory,pAllocator)


def vkUnmapMemory(
device
        ,memory
        ,):
    result = _callApi(lib.vkUnmapMemory, device,memory)


def vkFlushMappedMemoryRanges(
device
        ,memoryRangeCount
        ,pMemoryRanges
        ,):
    result = _callApi(lib.vkFlushMappedMemoryRanges, device,memoryRangeCount,pMemoryRanges)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkInvalidateMappedMemoryRanges(
device
        ,memoryRangeCount
        ,pMemoryRanges
        ,):
    result = _callApi(lib.vkInvalidateMappedMemoryRanges, device,memoryRangeCount,pMemoryRanges)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkGetDeviceMemoryCommitment(
device
        ,memory
        ,pCommittedMemoryInBytes=None
        ,):

    custom_return = True
    if not pCommittedMemoryInBytes:
        pCommittedMemoryInBytes = ffi.new('VkDeviceSize*')
        custom_return = False

    result = _callApi(lib.vkGetDeviceMemoryCommitment, device,memory,pCommittedMemoryInBytes)

    if custom_return:
        return pCommittedMemoryInBytes

    return pCommittedMemoryInBytes[0]



def vkGetBufferMemoryRequirements(
device
        ,buffer
        ,pMemoryRequirements=None
        ,):

    custom_return = True
    if not pMemoryRequirements:
        pMemoryRequirements = ffi.new('VkMemoryRequirements*')
        custom_return = False

    result = _callApi(lib.vkGetBufferMemoryRequirements, device,buffer,pMemoryRequirements)

    if custom_return:
        return pMemoryRequirements

    return pMemoryRequirements[0]



def vkBindBufferMemory(
device
        ,buffer
        ,memory
        ,memoryOffset
        ,):
    result = _callApi(lib.vkBindBufferMemory, device,buffer,memory,memoryOffset)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkGetImageMemoryRequirements(
device
        ,image
        ,pMemoryRequirements=None
        ,):

    custom_return = True
    if not pMemoryRequirements:
        pMemoryRequirements = ffi.new('VkMemoryRequirements*')
        custom_return = False

    result = _callApi(lib.vkGetImageMemoryRequirements, device,image,pMemoryRequirements)

    if custom_return:
        return pMemoryRequirements

    return pMemoryRequirements[0]



def vkBindImageMemory(
device
        ,image
        ,memory
        ,memoryOffset
        ,):
    result = _callApi(lib.vkBindImageMemory, device,image,memory,memoryOffset)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkGetImageSparseMemoryRequirements(
device
        ,image
        ,):

    pSparseMemoryRequirementCount = ffi.new('uint32_t*')
    pSparseMemoryRequirements = ffi.NULL

    result = _callApi(lib.vkGetImageSparseMemoryRequirements, device,image,pSparseMemoryRequirementCount,pSparseMemoryRequirements)

    pSparseMemoryRequirements = ffi.new('VkSparseImageMemoryRequirements[]', pSparseMemoryRequirementCount[0])
    result = _callApi(lib.vkGetImageSparseMemoryRequirements, device,image,pSparseMemoryRequirementCount,pSparseMemoryRequirements)

    return pSparseMemoryRequirements


def vkGetPhysicalDeviceSparseImageFormatProperties(
physicalDevice
        ,format
        ,type
        ,samples
        ,usage
        ,tiling
        ,):

    pPropertyCount = ffi.new('uint32_t*')
    pProperties = ffi.NULL

    result = _callApi(lib.vkGetPhysicalDeviceSparseImageFormatProperties, physicalDevice,format,type,samples,usage,tiling,pPropertyCount,pProperties)

    pProperties = ffi.new('VkSparseImageFormatProperties[]', pPropertyCount[0])
    result = _callApi(lib.vkGetPhysicalDeviceSparseImageFormatProperties, physicalDevice,format,type,samples,usage,tiling,pPropertyCount,pProperties)

    return pProperties


def vkQueueBindSparse(
queue
        ,bindInfoCount
        ,pBindInfo
        ,fence
        ,):
    result = _callApi(lib.vkQueueBindSparse, queue,bindInfoCount,pBindInfo,fence)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkCreateFence(
device
        ,pCreateInfo
        ,pAllocator
        ,pFence=None
        ,):

    custom_return = True
    if not pFence:
        pFence = ffi.new('VkFence*')
        custom_return = False

    result = _callApi(lib.vkCreateFence, device,pCreateInfo,pAllocator,pFence)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pFence

    return pFence[0]



def vkDestroyFence(
device
        ,fence
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyFence, device,fence,pAllocator)


def vkResetFences(
device
        ,fenceCount
        ,pFences
        ,):
    result = _callApi(lib.vkResetFences, device,fenceCount,pFences)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkGetFenceStatus(
device
        ,fence
        ,):
    result = _callApi(lib.vkGetFenceStatus, device,fence)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkWaitForFences(
device
        ,fenceCount
        ,pFences
        ,waitAll
        ,timeout
        ,):
    result = _callApi(lib.vkWaitForFences, device,fenceCount,pFences,waitAll,timeout)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkCreateSemaphore(
device
        ,pCreateInfo
        ,pAllocator
        ,pSemaphore=None
        ,):

    custom_return = True
    if not pSemaphore:
        pSemaphore = ffi.new('VkSemaphore*')
        custom_return = False

    result = _callApi(lib.vkCreateSemaphore, device,pCreateInfo,pAllocator,pSemaphore)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pSemaphore

    return pSemaphore[0]



def vkDestroySemaphore(
device
        ,semaphore
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroySemaphore, device,semaphore,pAllocator)


def vkCreateEvent(
device
        ,pCreateInfo
        ,pAllocator
        ,pEvent=None
        ,):

    custom_return = True
    if not pEvent:
        pEvent = ffi.new('VkEvent*')
        custom_return = False

    result = _callApi(lib.vkCreateEvent, device,pCreateInfo,pAllocator,pEvent)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pEvent

    return pEvent[0]



def vkDestroyEvent(
device
        ,event
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyEvent, device,event,pAllocator)


def vkGetEventStatus(
device
        ,event
        ,):
    result = _callApi(lib.vkGetEventStatus, device,event)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkSetEvent(
device
        ,event
        ,):
    result = _callApi(lib.vkSetEvent, device,event)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkResetEvent(
device
        ,event
        ,):
    result = _callApi(lib.vkResetEvent, device,event)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkCreateQueryPool(
device
        ,pCreateInfo
        ,pAllocator
        ,pQueryPool=None
        ,):

    custom_return = True
    if not pQueryPool:
        pQueryPool = ffi.new('VkQueryPool*')
        custom_return = False

    result = _callApi(lib.vkCreateQueryPool, device,pCreateInfo,pAllocator,pQueryPool)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pQueryPool

    return pQueryPool[0]



def vkDestroyQueryPool(
device
        ,queryPool
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyQueryPool, device,queryPool,pAllocator)


def vkGetQueryPoolResults(
device
        ,queryPool
        ,firstQuery
        ,queryCount
        ,dataSize
        ,pData
        ,stride
        ,flags
        ,):
    result = _callApi(lib.vkGetQueryPoolResults, device,queryPool,firstQuery,queryCount,dataSize,pData,stride,flags)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkCreateBuffer(
device
        ,pCreateInfo
        ,pAllocator
        ,pBuffer=None
        ,):

    custom_return = True
    if not pBuffer:
        pBuffer = ffi.new('VkBuffer*')
        custom_return = False

    result = _callApi(lib.vkCreateBuffer, device,pCreateInfo,pAllocator,pBuffer)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pBuffer

    return pBuffer[0]



def vkDestroyBuffer(
device
        ,buffer
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyBuffer, device,buffer,pAllocator)


def vkCreateBufferView(
device
        ,pCreateInfo
        ,pAllocator
        ,pView=None
        ,):

    custom_return = True
    if not pView:
        pView = ffi.new('VkBufferView*')
        custom_return = False

    result = _callApi(lib.vkCreateBufferView, device,pCreateInfo,pAllocator,pView)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pView

    return pView[0]



def vkDestroyBufferView(
device
        ,bufferView
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyBufferView, device,bufferView,pAllocator)


def vkCreateImage(
device
        ,pCreateInfo
        ,pAllocator
        ,pImage=None
        ,):

    custom_return = True
    if not pImage:
        pImage = ffi.new('VkImage*')
        custom_return = False

    result = _callApi(lib.vkCreateImage, device,pCreateInfo,pAllocator,pImage)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pImage

    return pImage[0]



def vkDestroyImage(
device
        ,image
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyImage, device,image,pAllocator)


def vkGetImageSubresourceLayout(
device
        ,image
        ,pSubresource
        ,pLayout=None
        ,):

    custom_return = True
    if not pLayout:
        pLayout = ffi.new('VkSubresourceLayout*')
        custom_return = False

    result = _callApi(lib.vkGetImageSubresourceLayout, device,image,pSubresource,pLayout)

    if custom_return:
        return pLayout

    return pLayout[0]



def vkCreateImageView(
device
        ,pCreateInfo
        ,pAllocator
        ,pView=None
        ,):

    custom_return = True
    if not pView:
        pView = ffi.new('VkImageView*')
        custom_return = False

    result = _callApi(lib.vkCreateImageView, device,pCreateInfo,pAllocator,pView)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pView

    return pView[0]



def vkDestroyImageView(
device
        ,imageView
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyImageView, device,imageView,pAllocator)


def vkCreateShaderModule(
device
        ,pCreateInfo
        ,pAllocator
        ,pShaderModule=None
        ,):

    custom_return = True
    if not pShaderModule:
        pShaderModule = ffi.new('VkShaderModule*')
        custom_return = False

    result = _callApi(lib.vkCreateShaderModule, device,pCreateInfo,pAllocator,pShaderModule)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pShaderModule

    return pShaderModule[0]



def vkDestroyShaderModule(
device
        ,shaderModule
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyShaderModule, device,shaderModule,pAllocator)


def vkCreatePipelineCache(
device
        ,pCreateInfo
        ,pAllocator
        ,pPipelineCache=None
        ,):

    custom_return = True
    if not pPipelineCache:
        pPipelineCache = ffi.new('VkPipelineCache*')
        custom_return = False

    result = _callApi(lib.vkCreatePipelineCache, device,pCreateInfo,pAllocator,pPipelineCache)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pPipelineCache

    return pPipelineCache[0]



def vkDestroyPipelineCache(
device
        ,pipelineCache
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyPipelineCache, device,pipelineCache,pAllocator)


def vkMergePipelineCaches(
device
        ,dstCache
        ,srcCacheCount
        ,pSrcCaches
        ,):
    result = _callApi(lib.vkMergePipelineCaches, device,dstCache,srcCacheCount,pSrcCaches)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkCreateGraphicsPipelines(
device
        ,pipelineCache
        ,createInfoCount
        ,pCreateInfos
        ,pAllocator
        ,pPipelines=None
        ,):

    custom_return = True
    if not pPipelines:
        pPipelines = ffi.new('VkPipeline[%d]' % createInfoCount)
        custom_return = False

    result = _callApi(lib.vkCreateGraphicsPipelines, device,pipelineCache,createInfoCount,pCreateInfos,pAllocator,pPipelines)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pPipelines

    return pPipelines



def vkCreateComputePipelines(
device
        ,pipelineCache
        ,createInfoCount
        ,pCreateInfos
        ,pAllocator
        ,pPipelines=None
        ,):

    custom_return = True
    if not pPipelines:
        pPipelines = ffi.new('VkPipeline[%d]' % createInfoCount)
        custom_return = False

    result = _callApi(lib.vkCreateComputePipelines, device,pipelineCache,createInfoCount,pCreateInfos,pAllocator,pPipelines)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pPipelines

    return pPipelines



def vkDestroyPipeline(
device
        ,pipeline
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyPipeline, device,pipeline,pAllocator)


def vkCreatePipelineLayout(
device
        ,pCreateInfo
        ,pAllocator
        ,pPipelineLayout=None
        ,):

    custom_return = True
    if not pPipelineLayout:
        pPipelineLayout = ffi.new('VkPipelineLayout*')
        custom_return = False

    result = _callApi(lib.vkCreatePipelineLayout, device,pCreateInfo,pAllocator,pPipelineLayout)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pPipelineLayout

    return pPipelineLayout[0]



def vkDestroyPipelineLayout(
device
        ,pipelineLayout
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyPipelineLayout, device,pipelineLayout,pAllocator)


def vkCreateSampler(
device
        ,pCreateInfo
        ,pAllocator
        ,pSampler=None
        ,):

    custom_return = True
    if not pSampler:
        pSampler = ffi.new('VkSampler*')
        custom_return = False

    result = _callApi(lib.vkCreateSampler, device,pCreateInfo,pAllocator,pSampler)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pSampler

    return pSampler[0]



def vkDestroySampler(
device
        ,sampler
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroySampler, device,sampler,pAllocator)


def vkCreateDescriptorSetLayout(
device
        ,pCreateInfo
        ,pAllocator
        ,pSetLayout=None
        ,):

    custom_return = True
    if not pSetLayout:
        pSetLayout = ffi.new('VkDescriptorSetLayout*')
        custom_return = False

    result = _callApi(lib.vkCreateDescriptorSetLayout, device,pCreateInfo,pAllocator,pSetLayout)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pSetLayout

    return pSetLayout[0]



def vkDestroyDescriptorSetLayout(
device
        ,descriptorSetLayout
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyDescriptorSetLayout, device,descriptorSetLayout,pAllocator)


def vkCreateDescriptorPool(
device
        ,pCreateInfo
        ,pAllocator
        ,pDescriptorPool=None
        ,):

    custom_return = True
    if not pDescriptorPool:
        pDescriptorPool = ffi.new('VkDescriptorPool*')
        custom_return = False

    result = _callApi(lib.vkCreateDescriptorPool, device,pCreateInfo,pAllocator,pDescriptorPool)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pDescriptorPool

    return pDescriptorPool[0]



def vkDestroyDescriptorPool(
device
        ,descriptorPool
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyDescriptorPool, device,descriptorPool,pAllocator)


def vkResetDescriptorPool(
device
        ,descriptorPool
        ,flags
        ,):
    result = _callApi(lib.vkResetDescriptorPool, device,descriptorPool,flags)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkAllocateDescriptorSets(
device
        ,pAllocateInfo
        ,pDescriptorSets=None
        ,):

    custom_return = True
    if not pDescriptorSets:
        pDescriptorSets = ffi.new('VkDescriptorSet[%d]' % pAllocateInfo.descriptorSetCount)
        custom_return = False

    result = _callApi(lib.vkAllocateDescriptorSets, device,pAllocateInfo,pDescriptorSets)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pDescriptorSets

    return pDescriptorSets



def vkFreeDescriptorSets(
device
        ,descriptorPool
        ,descriptorSetCount
        ,pDescriptorSets
        ,):
    result = _callApi(lib.vkFreeDescriptorSets, device,descriptorPool,descriptorSetCount,pDescriptorSets)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkUpdateDescriptorSets(
device
        ,descriptorWriteCount
        ,pDescriptorWrites
        ,descriptorCopyCount
        ,pDescriptorCopies
        ,):
    result = _callApi(lib.vkUpdateDescriptorSets, device,descriptorWriteCount,pDescriptorWrites,descriptorCopyCount,pDescriptorCopies)


def vkCreateFramebuffer(
device
        ,pCreateInfo
        ,pAllocator
        ,pFramebuffer=None
        ,):

    custom_return = True
    if not pFramebuffer:
        pFramebuffer = ffi.new('VkFramebuffer*')
        custom_return = False

    result = _callApi(lib.vkCreateFramebuffer, device,pCreateInfo,pAllocator,pFramebuffer)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pFramebuffer

    return pFramebuffer[0]



def vkDestroyFramebuffer(
device
        ,framebuffer
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyFramebuffer, device,framebuffer,pAllocator)


def vkCreateRenderPass(
device
        ,pCreateInfo
        ,pAllocator
        ,pRenderPass=None
        ,):

    custom_return = True
    if not pRenderPass:
        pRenderPass = ffi.new('VkRenderPass*')
        custom_return = False

    result = _callApi(lib.vkCreateRenderPass, device,pCreateInfo,pAllocator,pRenderPass)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pRenderPass

    return pRenderPass[0]



def vkDestroyRenderPass(
device
        ,renderPass
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyRenderPass, device,renderPass,pAllocator)


def vkGetRenderAreaGranularity(
device
        ,renderPass
        ,pGranularity=None
        ,):

    custom_return = True
    if not pGranularity:
        pGranularity = ffi.new('VkExtent2D*')
        custom_return = False

    result = _callApi(lib.vkGetRenderAreaGranularity, device,renderPass,pGranularity)

    if custom_return:
        return pGranularity

    return pGranularity[0]



def vkCreateCommandPool(
device
        ,pCreateInfo
        ,pAllocator
        ,pCommandPool=None
        ,):

    custom_return = True
    if not pCommandPool:
        pCommandPool = ffi.new('VkCommandPool*')
        custom_return = False

    result = _callApi(lib.vkCreateCommandPool, device,pCreateInfo,pAllocator,pCommandPool)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pCommandPool

    return pCommandPool[0]



def vkDestroyCommandPool(
device
        ,commandPool
        ,pAllocator
        ,):
    result = _callApi(lib.vkDestroyCommandPool, device,commandPool,pAllocator)


def vkResetCommandPool(
device
        ,commandPool
        ,flags
        ,):
    result = _callApi(lib.vkResetCommandPool, device,commandPool,flags)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkAllocateCommandBuffers(
device
        ,pAllocateInfo
        ,pCommandBuffers=None
        ,):

    custom_return = True
    if not pCommandBuffers:
        pCommandBuffers = ffi.new('VkCommandBuffer[%d]' % pAllocateInfo.commandBufferCount)
        custom_return = False

    result = _callApi(lib.vkAllocateCommandBuffers, device,pAllocateInfo,pCommandBuffers)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    if custom_return:
        return pCommandBuffers

    return pCommandBuffers



def vkFreeCommandBuffers(
device
        ,commandPool
        ,commandBufferCount
        ,pCommandBuffers
        ,):
    result = _callApi(lib.vkFreeCommandBuffers, device,commandPool,commandBufferCount,pCommandBuffers)


def vkBeginCommandBuffer(
commandBuffer
        ,pBeginInfo
        ,):
    result = _callApi(lib.vkBeginCommandBuffer, commandBuffer,pBeginInfo)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkEndCommandBuffer(
commandBuffer
        ,):
    result = _callApi(lib.vkEndCommandBuffer, commandBuffer)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkResetCommandBuffer(
commandBuffer
        ,flags
        ,):
    result = _callApi(lib.vkResetCommandBuffer, commandBuffer,flags)
    if result != VK_SUCCESS:
        raise exception_codes[result]


def vkCmdBindPipeline(
commandBuffer
        ,pipelineBindPoint
        ,pipeline
        ,):
    result = _callApi(lib.vkCmdBindPipeline, commandBuffer,pipelineBindPoint,pipeline)


def vkCmdSetViewport(
commandBuffer
        ,firstViewport
        ,viewportCount
        ,pViewports
        ,):
    result = _callApi(lib.vkCmdSetViewport, commandBuffer,firstViewport,viewportCount,pViewports)


def vkCmdSetScissor(
commandBuffer
        ,firstScissor
        ,scissorCount
        ,pScissors
        ,):
    result = _callApi(lib.vkCmdSetScissor, commandBuffer,firstScissor,scissorCount,pScissors)


def vkCmdSetLineWidth(
commandBuffer
        ,lineWidth
        ,):
    result = _callApi(lib.vkCmdSetLineWidth, commandBuffer,lineWidth)


def vkCmdSetDepthBias(
commandBuffer
        ,depthBiasConstantFactor
        ,depthBiasClamp
        ,depthBiasSlopeFactor
        ,):
    result = _callApi(lib.vkCmdSetDepthBias, commandBuffer,depthBiasConstantFactor,depthBiasClamp,depthBiasSlopeFactor)


def vkCmdSetBlendConstants(
commandBuffer
        ,blendConstants
        ,):
    result = _callApi(lib.vkCmdSetBlendConstants, commandBuffer,blendConstants)


def vkCmdSetDepthBounds(
commandBuffer
        ,minDepthBounds
        ,maxDepthBounds
        ,):
    result = _callApi(lib.vkCmdSetDepthBounds, commandBuffer,minDepthBounds,maxDepthBounds)


def vkCmdSetStencilCompareMask(
commandBuffer
        ,faceMask
        ,compareMask
        ,):
    result = _callApi(lib.vkCmdSetStencilCompareMask, commandBuffer,faceMask,compareMask)


def vkCmdSetStencilWriteMask(
commandBuffer
        ,faceMask
        ,writeMask
        ,):
    result = _callApi(lib.vkCmdSetStencilWriteMask, commandBuffer,faceMask,writeMask)


def vkCmdSetStencilReference(
commandBuffer
        ,faceMask
        ,reference
        ,):
    result = _callApi(lib.vkCmdSetStencilReference, commandBuffer,faceMask,reference)


def vkCmdBindDescriptorSets(
commandBuffer
        ,pipelineBindPoint
        ,layout
        ,firstSet
        ,descriptorSetCount
        ,pDescriptorSets
        ,dynamicOffsetCount
        ,pDynamicOffsets
        ,):
    result = _callApi(lib.vkCmdBindDescriptorSets, commandBuffer,pipelineBindPoint,layout,firstSet,descriptorSetCount,pDescriptorSets,dynamicOffsetCount,pDynamicOffsets)


def vkCmdBindIndexBuffer(
commandBuffer
        ,buffer
        ,offset
        ,indexType
        ,):
    result = _callApi(lib.vkCmdBindIndexBuffer, commandBuffer,buffer,offset,indexType)


def vkCmdBindVertexBuffers(
commandBuffer
        ,firstBinding
        ,bindingCount
        ,pBuffers
        ,pOffsets
        ,):
    result = _callApi(lib.vkCmdBindVertexBuffers, commandBuffer,firstBinding,bindingCount,pBuffers,pOffsets)


def vkCmdDraw(
commandBuffer
        ,vertexCount
        ,instanceCount
        ,firstVertex
        ,firstInstance
        ,):
    result = _callApi(lib.vkCmdDraw, commandBuffer,vertexCount,instanceCount,firstVertex,firstInstance)


def vkCmdDrawIndexed(
commandBuffer
        ,indexCount
        ,instanceCount
        ,firstIndex
        ,vertexOffset
        ,firstInstance
        ,):
    result = _callApi(lib.vkCmdDrawIndexed, commandBuffer,indexCount,instanceCount,firstIndex,vertexOffset,firstInstance)


def vkCmdDrawIndirect(
commandBuffer
        ,buffer
        ,offset
        ,drawCount
        ,stride
        ,):
    result = _callApi(lib.vkCmdDrawIndirect, commandBuffer,buffer,offset,drawCount,stride)


def vkCmdDrawIndexedIndirect(
commandBuffer
        ,buffer
        ,offset
        ,drawCount
        ,stride
        ,):
    result = _callApi(lib.vkCmdDrawIndexedIndirect, commandBuffer,buffer,offset,drawCount,stride)


def vkCmdDispatch(
commandBuffer
        ,groupCountX
        ,groupCountY
        ,groupCountZ
        ,):
    result = _callApi(lib.vkCmdDispatch, commandBuffer,groupCountX,groupCountY,groupCountZ)


def vkCmdDispatchIndirect(
commandBuffer
        ,buffer
        ,offset
        ,):
    result = _callApi(lib.vkCmdDispatchIndirect, commandBuffer,buffer,offset)


def vkCmdCopyBuffer(
commandBuffer
        ,srcBuffer
        ,dstBuffer
        ,regionCount
        ,pRegions
        ,):
    result = _callApi(lib.vkCmdCopyBuffer, commandBuffer,srcBuffer,dstBuffer,regionCount,pRegions)


def vkCmdCopyImage(
commandBuffer
        ,srcImage
        ,srcImageLayout
        ,dstImage
        ,dstImageLayout
        ,regionCount
        ,pRegions
        ,):
    result = _callApi(lib.vkCmdCopyImage, commandBuffer,srcImage,srcImageLayout,dstImage,dstImageLayout,regionCount,pRegions)


def vkCmdBlitImage(
commandBuffer
        ,srcImage
        ,srcImageLayout
        ,dstImage
        ,dstImageLayout
        ,regionCount
        ,pRegions
        ,filter
        ,):
    result = _callApi(lib.vkCmdBlitImage, commandBuffer,srcImage,srcImageLayout,dstImage,dstImageLayout,regionCount,pRegions,filter)


def vkCmdCopyBufferToImage(
commandBuffer
        ,srcBuffer
        ,dstImage
        ,dstImageLayout
        ,regionCount
        ,pRegions
        ,):
    result = _callApi(lib.vkCmdCopyBufferToImage, commandBuffer,srcBuffer,dstImage,dstImageLayout,regionCount,pRegions)


def vkCmdCopyImageToBuffer(
commandBuffer
        ,srcImage
        ,srcImageLayout
        ,dstBuffer
        ,regionCount
        ,pRegions
        ,):
    result = _callApi(lib.vkCmdCopyImageToBuffer, commandBuffer,srcImage,srcImageLayout,dstBuffer,regionCount,pRegions)


def vkCmdUpdateBuffer(
commandBuffer
        ,dstBuffer
        ,dstOffset
        ,dataSize
        ,pData
        ,):
    result = _callApi(lib.vkCmdUpdateBuffer, commandBuffer,dstBuffer,dstOffset,dataSize,pData)


def vkCmdFillBuffer(
commandBuffer
        ,dstBuffer
        ,dstOffset
        ,size
        ,data
        ,):
    result = _callApi(lib.vkCmdFillBuffer, commandBuffer,dstBuffer,dstOffset,size,data)


def vkCmdClearColorImage(
commandBuffer
        ,image
        ,imageLayout
        ,pColor
        ,rangeCount
        ,pRanges
        ,):
    result = _callApi(lib.vkCmdClearColorImage, commandBuffer,image,imageLayout,pColor,rangeCount,pRanges)


def vkCmdClearDepthStencilImage(
commandBuffer
        ,image
        ,imageLayout
        ,pDepthStencil
        ,rangeCount
        ,pRanges
        ,):
    result = _callApi(lib.vkCmdClearDepthStencilImage, commandBuffer,image,imageLayout,pDepthStencil,rangeCount,pRanges)


def vkCmdClearAttachments(
commandBuffer
        ,attachmentCount
        ,pAttachments
        ,rectCount
        ,pRects
        ,):
    result = _callApi(lib.vkCmdClearAttachments, commandBuffer,attachmentCount,pAttachments,rectCount,pRects)


def vkCmdResolveImage(
commandBuffer
        ,srcImage
        ,srcImageLayout
        ,dstImage
        ,dstImageLayout
        ,regionCount
        ,pRegions
        ,):
    result = _callApi(lib.vkCmdResolveImage, commandBuffer,srcImage,srcImageLayout,dstImage,dstImageLayout,regionCount,pRegions)


def vkCmdSetEvent(
commandBuffer
        ,event
        ,stageMask
        ,):
    result = _callApi(lib.vkCmdSetEvent, commandBuffer,event,stageMask)


def vkCmdResetEvent(
commandBuffer
        ,event
        ,stageMask
        ,):
    result = _callApi(lib.vkCmdResetEvent, commandBuffer,event,stageMask)


def vkCmdWaitEvents(
commandBuffer
        ,eventCount
        ,pEvents
        ,srcStageMask
        ,dstStageMask
        ,memoryBarrierCount
        ,pMemoryBarriers
        ,bufferMemoryBarrierCount
        ,pBufferMemoryBarriers
        ,imageMemoryBarrierCount
        ,pImageMemoryBarriers
        ,):
    result = _callApi(lib.vkCmdWaitEvents, commandBuffer,eventCount,pEvents,srcStageMask,dstStageMask,memoryBarrierCount,pMemoryBarriers,bufferMemoryBarrierCount,pBufferMemoryBarriers,imageMemoryBarrierCount,pImageMemoryBarriers)


def vkCmdPipelineBarrier(
commandBuffer
        ,srcStageMask
        ,dstStageMask
        ,dependencyFlags
        ,memoryBarrierCount
        ,pMemoryBarriers
        ,bufferMemoryBarrierCount
        ,pBufferMemoryBarriers
        ,imageMemoryBarrierCount
        ,pImageMemoryBarriers
        ,):
    result = _callApi(lib.vkCmdPipelineBarrier, commandBuffer,srcStageMask,dstStageMask,dependencyFlags,memoryBarrierCount,pMemoryBarriers,bufferMemoryBarrierCount,pBufferMemoryBarriers,imageMemoryBarrierCount,pImageMemoryBarriers)


def vkCmdBeginQuery(
commandBuffer
        ,queryPool
        ,query
        ,flags
        ,):
    result = _callApi(lib.vkCmdBeginQuery, commandBuffer,queryPool,query,flags)


def vkCmdEndQuery(
commandBuffer
        ,queryPool
        ,query
        ,):
    result = _callApi(lib.vkCmdEndQuery, commandBuffer,queryPool,query)


def _wrap_vkCmdBeginConditionalRenderingEXT(fn):
    def vkCmdBeginConditionalRenderingEXT(
    commandBuffer
            ,pConditionalRenderingBegin
            ,):
        result = _callApi(fn, commandBuffer,pConditionalRenderingBegin)


    return vkCmdBeginConditionalRenderingEXT
def _wrap_vkCmdEndConditionalRenderingEXT(fn):
    def vkCmdEndConditionalRenderingEXT(
    commandBuffer
            ,):
        result = _callApi(fn, commandBuffer)


    return vkCmdEndConditionalRenderingEXT
def vkCmdResetQueryPool(
commandBuffer
        ,queryPool
        ,firstQuery
        ,queryCount
        ,):
    result = _callApi(lib.vkCmdResetQueryPool, commandBuffer,queryPool,firstQuery,queryCount)


def vkCmdWriteTimestamp(
commandBuffer
        ,pipelineStage
        ,queryPool
        ,query
        ,):
    result = _callApi(lib.vkCmdWriteTimestamp, commandBuffer,pipelineStage,queryPool,query)


def vkCmdCopyQueryPoolResults(
commandBuffer
        ,queryPool
        ,firstQuery
        ,queryCount
        ,dstBuffer
        ,dstOffset
        ,stride
        ,flags
        ,):
    result = _callApi(lib.vkCmdCopyQueryPoolResults, commandBuffer,queryPool,firstQuery,queryCount,dstBuffer,dstOffset,stride,flags)


def vkCmdPushConstants(
commandBuffer
        ,layout
        ,stageFlags
        ,offset
        ,size
        ,pValues
        ,):
    result = _callApi(lib.vkCmdPushConstants, commandBuffer,layout,stageFlags,offset,size,pValues)


def vkCmdBeginRenderPass(
commandBuffer
        ,pRenderPassBegin
        ,contents
        ,):
    result = _callApi(lib.vkCmdBeginRenderPass, commandBuffer,pRenderPassBegin,contents)


def vkCmdNextSubpass(
commandBuffer
        ,contents
        ,):
    result = _callApi(lib.vkCmdNextSubpass, commandBuffer,contents)


def vkCmdEndRenderPass(
commandBuffer
        ,):
    result = _callApi(lib.vkCmdEndRenderPass, commandBuffer)


def vkCmdExecuteCommands(
commandBuffer
        ,commandBufferCount
        ,pCommandBuffers
        ,):
    result = _callApi(lib.vkCmdExecuteCommands, commandBuffer,commandBufferCount,pCommandBuffers)


def _wrap_vkCreateAndroidSurfaceKHR(fn):
    def vkCreateAndroidSurfaceKHR(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateAndroidSurfaceKHR
def _wrap_vkGetPhysicalDeviceDisplayPropertiesKHR(fn):
    def vkGetPhysicalDeviceDisplayPropertiesKHR(
    physicalDevice
            ,):

        pPropertyCount = ffi.new('uint32_t*')
        pProperties = ffi.NULL

        result = _callApi(fn, physicalDevice,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pProperties = ffi.new('VkDisplayPropertiesKHR[]', pPropertyCount[0])
        result = _callApi(fn, physicalDevice,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        result = (StrWrap(x) for x in pProperties)
        _weakkey_dict[result] = pProperties
        return result


    return vkGetPhysicalDeviceDisplayPropertiesKHR
def _wrap_vkGetPhysicalDeviceDisplayPlanePropertiesKHR(fn):
    def vkGetPhysicalDeviceDisplayPlanePropertiesKHR(
    physicalDevice
            ,):

        pPropertyCount = ffi.new('uint32_t*')
        pProperties = ffi.NULL

        result = _callApi(fn, physicalDevice,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pProperties = ffi.new('VkDisplayPlanePropertiesKHR[]', pPropertyCount[0])
        result = _callApi(fn, physicalDevice,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pProperties


    return vkGetPhysicalDeviceDisplayPlanePropertiesKHR
def _wrap_vkGetDisplayPlaneSupportedDisplaysKHR(fn):
    def vkGetDisplayPlaneSupportedDisplaysKHR(
    physicalDevice
            ,planeIndex
            ,):

        pDisplayCount = ffi.new('uint32_t*')
        pDisplays = ffi.NULL

        result = _callApi(fn, physicalDevice,planeIndex,pDisplayCount,pDisplays)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pDisplays = ffi.new('VkDisplayKHR[]', pDisplayCount[0])
        result = _callApi(fn, physicalDevice,planeIndex,pDisplayCount,pDisplays)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pDisplays


    return vkGetDisplayPlaneSupportedDisplaysKHR
def _wrap_vkGetDisplayModePropertiesKHR(fn):
    def vkGetDisplayModePropertiesKHR(
    physicalDevice
            ,display
            ,):

        pPropertyCount = ffi.new('uint32_t*')
        pProperties = ffi.NULL

        result = _callApi(fn, physicalDevice,display,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pProperties = ffi.new('VkDisplayModePropertiesKHR[]', pPropertyCount[0])
        result = _callApi(fn, physicalDevice,display,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pProperties


    return vkGetDisplayModePropertiesKHR
def _wrap_vkCreateDisplayModeKHR(fn):
    def vkCreateDisplayModeKHR(
    physicalDevice
            ,display
            ,pCreateInfo
            ,pAllocator
            ,pMode=None
            ,):

        custom_return = True
        if not pMode:
            pMode = ffi.new('VkDisplayModeKHR*')
            custom_return = False

        result = _callApi(fn, physicalDevice,display,pCreateInfo,pAllocator,pMode)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pMode

        return pMode[0]



    return vkCreateDisplayModeKHR
def _wrap_vkGetDisplayPlaneCapabilitiesKHR(fn):
    def vkGetDisplayPlaneCapabilitiesKHR(
    physicalDevice
            ,mode
            ,planeIndex
            ,pCapabilities=None
            ,):

        custom_return = True
        if not pCapabilities:
            pCapabilities = ffi.new('VkDisplayPlaneCapabilitiesKHR*')
            custom_return = False

        result = _callApi(fn, physicalDevice,mode,planeIndex,pCapabilities)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pCapabilities

        return pCapabilities[0]



    return vkGetDisplayPlaneCapabilitiesKHR
def _wrap_vkCreateDisplayPlaneSurfaceKHR(fn):
    def vkCreateDisplayPlaneSurfaceKHR(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateDisplayPlaneSurfaceKHR
def _wrap_vkCreateSharedSwapchainsKHR(fn):
    def vkCreateSharedSwapchainsKHR(
    device
            ,swapchainCount
            ,pCreateInfos
            ,pAllocator
            ,pSwapchains=None
            ,):

        custom_return = True
        if not pSwapchains:
            pSwapchains = ffi.new('VkSwapchainKHR[%d]' % swapchainCount)
            custom_return = False

        result = _callApi(fn, device,swapchainCount,pCreateInfos,pAllocator,pSwapchains)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSwapchains

        return pSwapchains



    return vkCreateSharedSwapchainsKHR
def _wrap_vkDestroySurfaceKHR(fn):
    def vkDestroySurfaceKHR(
    instance
            ,surface
            ,pAllocator
            ,):
        result = _callApi(fn, instance,surface,pAllocator)


    return vkDestroySurfaceKHR
def _wrap_vkGetPhysicalDeviceSurfaceSupportKHR(fn):
    def vkGetPhysicalDeviceSurfaceSupportKHR(
    physicalDevice
            ,queueFamilyIndex
            ,surface
            ,pSupported=None
            ,):

        custom_return = True
        if not pSupported:
            pSupported = ffi.new('VkBool32*')
            custom_return = False

        result = _callApi(fn, physicalDevice,queueFamilyIndex,surface,pSupported)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSupported

        return pSupported[0]



    return vkGetPhysicalDeviceSurfaceSupportKHR
def _wrap_vkGetPhysicalDeviceSurfaceCapabilitiesKHR(fn):
    def vkGetPhysicalDeviceSurfaceCapabilitiesKHR(
    physicalDevice
            ,surface
            ,pSurfaceCapabilities=None
            ,):

        custom_return = True
        if not pSurfaceCapabilities:
            pSurfaceCapabilities = ffi.new('VkSurfaceCapabilitiesKHR*')
            custom_return = False

        result = _callApi(fn, physicalDevice,surface,pSurfaceCapabilities)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurfaceCapabilities

        return pSurfaceCapabilities[0]



    return vkGetPhysicalDeviceSurfaceCapabilitiesKHR
def _wrap_vkGetPhysicalDeviceSurfaceFormatsKHR(fn):
    def vkGetPhysicalDeviceSurfaceFormatsKHR(
    physicalDevice
            ,surface
            ,):

        pSurfaceFormatCount = ffi.new('uint32_t*')
        pSurfaceFormats = ffi.NULL

        result = _callApi(fn, physicalDevice,surface,pSurfaceFormatCount,pSurfaceFormats)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pSurfaceFormats = ffi.new('VkSurfaceFormatKHR[]', pSurfaceFormatCount[0])
        result = _callApi(fn, physicalDevice,surface,pSurfaceFormatCount,pSurfaceFormats)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pSurfaceFormats


    return vkGetPhysicalDeviceSurfaceFormatsKHR
def _wrap_vkGetPhysicalDeviceSurfacePresentModesKHR(fn):
    def vkGetPhysicalDeviceSurfacePresentModesKHR(
    physicalDevice
            ,surface
            ,):

        pPresentModeCount = ffi.new('uint32_t*')
        pPresentModes = ffi.NULL

        result = _callApi(fn, physicalDevice,surface,pPresentModeCount,pPresentModes)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pPresentModes = ffi.new('VkPresentModeKHR[]', pPresentModeCount[0])
        result = _callApi(fn, physicalDevice,surface,pPresentModeCount,pPresentModes)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pPresentModes


    return vkGetPhysicalDeviceSurfacePresentModesKHR
def _wrap_vkCreateSwapchainKHR(fn):
    def vkCreateSwapchainKHR(
    device
            ,pCreateInfo
            ,pAllocator
            ,pSwapchain=None
            ,):

        custom_return = True
        if not pSwapchain:
            pSwapchain = ffi.new('VkSwapchainKHR*')
            custom_return = False

        result = _callApi(fn, device,pCreateInfo,pAllocator,pSwapchain)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSwapchain

        return pSwapchain[0]



    return vkCreateSwapchainKHR
def _wrap_vkDestroySwapchainKHR(fn):
    def vkDestroySwapchainKHR(
    device
            ,swapchain
            ,pAllocator
            ,):
        result = _callApi(fn, device,swapchain,pAllocator)


    return vkDestroySwapchainKHR
def _wrap_vkGetSwapchainImagesKHR(fn):
    def vkGetSwapchainImagesKHR(
    device
            ,swapchain
            ,):

        pSwapchainImageCount = ffi.new('uint32_t*')
        pSwapchainImages = ffi.NULL

        result = _callApi(fn, device,swapchain,pSwapchainImageCount,pSwapchainImages)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pSwapchainImages = ffi.new('VkImage[]', pSwapchainImageCount[0])
        result = _callApi(fn, device,swapchain,pSwapchainImageCount,pSwapchainImages)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pSwapchainImages


    return vkGetSwapchainImagesKHR
def _wrap_vkAcquireNextImageKHR(fn):
    def vkAcquireNextImageKHR(
    device
            ,swapchain
            ,timeout
            ,semaphore
            ,fence
            ,pImageIndex=None
            ,):

        custom_return = True
        if not pImageIndex:
            pImageIndex = ffi.new('uint32_t*')
            custom_return = False

        result = _callApi(fn, device,swapchain,timeout,semaphore,fence,pImageIndex)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pImageIndex

        return pImageIndex[0]



    return vkAcquireNextImageKHR
def _wrap_vkQueuePresentKHR(fn):
    def vkQueuePresentKHR(
    queue
            ,pPresentInfo
            ,):
        result = _callApi(fn, queue,pPresentInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkQueuePresentKHR
def _wrap_vkCreateViSurfaceNN(fn):
    def vkCreateViSurfaceNN(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateViSurfaceNN
def _wrap_vkCreateWaylandSurfaceKHR(fn):
    def vkCreateWaylandSurfaceKHR(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateWaylandSurfaceKHR
def _wrap_vkGetPhysicalDeviceWaylandPresentationSupportKHR(fn):
    def vkGetPhysicalDeviceWaylandPresentationSupportKHR(
    physicalDevice
            ,queueFamilyIndex
            ,display
            ,):
        result = _callApi(fn, physicalDevice,queueFamilyIndex,display)


    return vkGetPhysicalDeviceWaylandPresentationSupportKHR
def _wrap_vkCreateWin32SurfaceKHR(fn):
    def vkCreateWin32SurfaceKHR(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateWin32SurfaceKHR
def _wrap_vkGetPhysicalDeviceWin32PresentationSupportKHR(fn):
    def vkGetPhysicalDeviceWin32PresentationSupportKHR(
    physicalDevice
            ,queueFamilyIndex
            ,):
        result = _callApi(fn, physicalDevice,queueFamilyIndex)


    return vkGetPhysicalDeviceWin32PresentationSupportKHR
def _wrap_vkCreateXlibSurfaceKHR(fn):
    def vkCreateXlibSurfaceKHR(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateXlibSurfaceKHR
def _wrap_vkGetPhysicalDeviceXlibPresentationSupportKHR(fn):
    def vkGetPhysicalDeviceXlibPresentationSupportKHR(
    physicalDevice
            ,queueFamilyIndex
            ,dpy
            ,visualID
            ,):
        result = _callApi(fn, physicalDevice,queueFamilyIndex,dpy,visualID)


    return vkGetPhysicalDeviceXlibPresentationSupportKHR
def _wrap_vkCreateXcbSurfaceKHR(fn):
    def vkCreateXcbSurfaceKHR(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateXcbSurfaceKHR
def _wrap_vkGetPhysicalDeviceXcbPresentationSupportKHR(fn):
    def vkGetPhysicalDeviceXcbPresentationSupportKHR(
    physicalDevice
            ,queueFamilyIndex
            ,connection
            ,visual_id
            ,):
        result = _callApi(fn, physicalDevice,queueFamilyIndex,connection,visual_id)


    return vkGetPhysicalDeviceXcbPresentationSupportKHR
def _wrap_vkCreateImagePipeSurfaceFUCHSIA(fn):
    def vkCreateImagePipeSurfaceFUCHSIA(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateImagePipeSurfaceFUCHSIA
def _wrap_vkCreateDebugReportCallbackEXT(fn):
    def vkCreateDebugReportCallbackEXT(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pCallback=None
            ,):

        custom_return = True
        if not pCallback:
            pCallback = ffi.new('VkDebugReportCallbackEXT*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pCallback)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pCallback

        return pCallback[0]



    return vkCreateDebugReportCallbackEXT
def _wrap_vkDestroyDebugReportCallbackEXT(fn):
    def vkDestroyDebugReportCallbackEXT(
    instance
            ,callback
            ,pAllocator
            ,):
        result = _callApi(fn, instance,callback,pAllocator)


    return vkDestroyDebugReportCallbackEXT
def _wrap_vkDebugReportMessageEXT(fn):
    def vkDebugReportMessageEXT(
    instance
            ,flags
            ,objectType
            ,object
            ,location
            ,messageCode
            ,pLayerPrefix
            ,pMessage
            ,):
        result = _callApi(fn, instance,flags,objectType,object,location,messageCode,pLayerPrefix,pMessage)


    return vkDebugReportMessageEXT
def _wrap_vkDebugMarkerSetObjectNameEXT(fn):
    def vkDebugMarkerSetObjectNameEXT(
    device
            ,pNameInfo
            ,):
        result = _callApi(fn, device,pNameInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkDebugMarkerSetObjectNameEXT
def _wrap_vkDebugMarkerSetObjectTagEXT(fn):
    def vkDebugMarkerSetObjectTagEXT(
    device
            ,pTagInfo
            ,):
        result = _callApi(fn, device,pTagInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkDebugMarkerSetObjectTagEXT
def _wrap_vkCmdDebugMarkerBeginEXT(fn):
    def vkCmdDebugMarkerBeginEXT(
    commandBuffer
            ,pMarkerInfo
            ,):
        result = _callApi(fn, commandBuffer,pMarkerInfo)


    return vkCmdDebugMarkerBeginEXT
def _wrap_vkCmdDebugMarkerEndEXT(fn):
    def vkCmdDebugMarkerEndEXT(
    commandBuffer
            ,):
        result = _callApi(fn, commandBuffer)


    return vkCmdDebugMarkerEndEXT
def _wrap_vkCmdDebugMarkerInsertEXT(fn):
    def vkCmdDebugMarkerInsertEXT(
    commandBuffer
            ,pMarkerInfo
            ,):
        result = _callApi(fn, commandBuffer,pMarkerInfo)


    return vkCmdDebugMarkerInsertEXT
def _wrap_vkGetPhysicalDeviceExternalImageFormatPropertiesNV(fn):
    def vkGetPhysicalDeviceExternalImageFormatPropertiesNV(
    physicalDevice
            ,format
            ,type
            ,tiling
            ,usage
            ,flags
            ,externalHandleType
            ,pExternalImageFormatProperties=None
            ,):

        custom_return = True
        if not pExternalImageFormatProperties:
            pExternalImageFormatProperties = ffi.new('VkExternalImageFormatPropertiesNV*')
            custom_return = False

        result = _callApi(fn, physicalDevice,format,type,tiling,usage,flags,externalHandleType,pExternalImageFormatProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pExternalImageFormatProperties

        return pExternalImageFormatProperties[0]



    return vkGetPhysicalDeviceExternalImageFormatPropertiesNV
def _wrap_vkGetMemoryWin32HandleNV(fn):
    def vkGetMemoryWin32HandleNV(
    device
            ,memory
            ,handleType
            ,pHandle=None
            ,):

        custom_return = True
        if not pHandle:
            pHandle = ffi.new('HANDLE*')
            custom_return = False

        result = _callApi(fn, device,memory,handleType,pHandle)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pHandle

        return pHandle[0]



    return vkGetMemoryWin32HandleNV
def _wrap_vkCmdDrawIndirectCountAMD(fn):
    def vkCmdDrawIndirectCountAMD(
    commandBuffer
            ,buffer
            ,offset
            ,countBuffer
            ,countBufferOffset
            ,maxDrawCount
            ,stride
            ,):
        result = _callApi(fn, commandBuffer,buffer,offset,countBuffer,countBufferOffset,maxDrawCount,stride)


    return vkCmdDrawIndirectCountAMD
def _wrap_vkCmdDrawIndexedIndirectCountAMD(fn):
    def vkCmdDrawIndexedIndirectCountAMD(
    commandBuffer
            ,buffer
            ,offset
            ,countBuffer
            ,countBufferOffset
            ,maxDrawCount
            ,stride
            ,):
        result = _callApi(fn, commandBuffer,buffer,offset,countBuffer,countBufferOffset,maxDrawCount,stride)


    return vkCmdDrawIndexedIndirectCountAMD
def _wrap_vkCmdProcessCommandsNVX(fn):
    def vkCmdProcessCommandsNVX(
    commandBuffer
            ,pProcessCommandsInfo
            ,):
        result = _callApi(fn, commandBuffer,pProcessCommandsInfo)


    return vkCmdProcessCommandsNVX
def _wrap_vkCmdReserveSpaceForCommandsNVX(fn):
    def vkCmdReserveSpaceForCommandsNVX(
    commandBuffer
            ,pReserveSpaceInfo
            ,):
        result = _callApi(fn, commandBuffer,pReserveSpaceInfo)


    return vkCmdReserveSpaceForCommandsNVX
def _wrap_vkCreateIndirectCommandsLayoutNVX(fn):
    def vkCreateIndirectCommandsLayoutNVX(
    device
            ,pCreateInfo
            ,pAllocator
            ,pIndirectCommandsLayout=None
            ,):

        custom_return = True
        if not pIndirectCommandsLayout:
            pIndirectCommandsLayout = ffi.new('VkIndirectCommandsLayoutNVX*')
            custom_return = False

        result = _callApi(fn, device,pCreateInfo,pAllocator,pIndirectCommandsLayout)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pIndirectCommandsLayout

        return pIndirectCommandsLayout[0]



    return vkCreateIndirectCommandsLayoutNVX
def _wrap_vkDestroyIndirectCommandsLayoutNVX(fn):
    def vkDestroyIndirectCommandsLayoutNVX(
    device
            ,indirectCommandsLayout
            ,pAllocator
            ,):
        result = _callApi(fn, device,indirectCommandsLayout,pAllocator)


    return vkDestroyIndirectCommandsLayoutNVX
def _wrap_vkCreateObjectTableNVX(fn):
    def vkCreateObjectTableNVX(
    device
            ,pCreateInfo
            ,pAllocator
            ,pObjectTable=None
            ,):

        custom_return = True
        if not pObjectTable:
            pObjectTable = ffi.new('VkObjectTableNVX*')
            custom_return = False

        result = _callApi(fn, device,pCreateInfo,pAllocator,pObjectTable)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pObjectTable

        return pObjectTable[0]



    return vkCreateObjectTableNVX
def _wrap_vkDestroyObjectTableNVX(fn):
    def vkDestroyObjectTableNVX(
    device
            ,objectTable
            ,pAllocator
            ,):
        result = _callApi(fn, device,objectTable,pAllocator)


    return vkDestroyObjectTableNVX
def _wrap_vkRegisterObjectsNVX(fn):
    def vkRegisterObjectsNVX(
    device
            ,objectTable
            ,objectCount
            ,ppObjectTableEntries
            ,pObjectIndices
            ,):
        result = _callApi(fn, device,objectTable,objectCount,ppObjectTableEntries,pObjectIndices)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkRegisterObjectsNVX
def _wrap_vkUnregisterObjectsNVX(fn):
    def vkUnregisterObjectsNVX(
    device
            ,objectTable
            ,objectCount
            ,pObjectEntryTypes
            ,pObjectIndices
            ,):
        result = _callApi(fn, device,objectTable,objectCount,pObjectEntryTypes,pObjectIndices)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkUnregisterObjectsNVX
def _wrap_vkGetPhysicalDeviceGeneratedCommandsPropertiesNVX(fn):
    def vkGetPhysicalDeviceGeneratedCommandsPropertiesNVX(
    physicalDevice
            ,pFeatures
            ,pLimits=None
            ,):

        custom_return = True
        if not pLimits:
            pLimits = ffi.new('VkDeviceGeneratedCommandsLimitsNVX*')
            custom_return = False

        result = _callApi(fn, physicalDevice,pFeatures,pLimits)

        if custom_return:
            return pLimits

        return pLimits[0]



    return vkGetPhysicalDeviceGeneratedCommandsPropertiesNVX
def _wrap_vkGetPhysicalDeviceFeatures2(fn):
    def vkGetPhysicalDeviceFeatures2(
    physicalDevice
            ,pFeatures=None
            ,):

        custom_return = True
        if not pFeatures:
            pFeatures = ffi.new('VkPhysicalDeviceFeatures2*')
            custom_return = False

        result = _callApi(fn, physicalDevice,pFeatures)

        if custom_return:
            return pFeatures

        return pFeatures[0]



    return vkGetPhysicalDeviceFeatures2
def _wrap_vkGetPhysicalDeviceProperties2(fn):
    def vkGetPhysicalDeviceProperties2(
    physicalDevice
            ,pProperties=None
            ,):

        custom_return = True
        if not pProperties:
            pProperties = ffi.new('VkPhysicalDeviceProperties2*')
            custom_return = False

        result = _callApi(fn, physicalDevice,pProperties)

        if custom_return:
            return pProperties

        return pProperties[0]



    return vkGetPhysicalDeviceProperties2
def _wrap_vkGetPhysicalDeviceFormatProperties2(fn):
    def vkGetPhysicalDeviceFormatProperties2(
    physicalDevice
            ,format
            ,pFormatProperties=None
            ,):

        custom_return = True
        if not pFormatProperties:
            pFormatProperties = ffi.new('VkFormatProperties2*')
            custom_return = False

        result = _callApi(fn, physicalDevice,format,pFormatProperties)

        if custom_return:
            return pFormatProperties

        return pFormatProperties[0]



    return vkGetPhysicalDeviceFormatProperties2
def _wrap_vkGetPhysicalDeviceImageFormatProperties2(fn):
    def vkGetPhysicalDeviceImageFormatProperties2(
    physicalDevice
            ,pImageFormatInfo
            ,pImageFormatProperties=None
            ,):

        custom_return = True
        if not pImageFormatProperties:
            pImageFormatProperties = ffi.new('VkImageFormatProperties2*')
            custom_return = False

        result = _callApi(fn, physicalDevice,pImageFormatInfo,pImageFormatProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pImageFormatProperties

        return pImageFormatProperties[0]



    return vkGetPhysicalDeviceImageFormatProperties2
def _wrap_vkGetPhysicalDeviceQueueFamilyProperties2(fn):
    def vkGetPhysicalDeviceQueueFamilyProperties2(
    physicalDevice
            ,):

        pQueueFamilyPropertyCount = ffi.new('uint32_t*')
        pQueueFamilyProperties = ffi.NULL

        result = _callApi(fn, physicalDevice,pQueueFamilyPropertyCount,pQueueFamilyProperties)

        pQueueFamilyProperties = ffi.new('VkQueueFamilyProperties2[]', pQueueFamilyPropertyCount[0])
        result = _callApi(fn, physicalDevice,pQueueFamilyPropertyCount,pQueueFamilyProperties)

        return pQueueFamilyProperties


    return vkGetPhysicalDeviceQueueFamilyProperties2
def _wrap_vkGetPhysicalDeviceMemoryProperties2(fn):
    def vkGetPhysicalDeviceMemoryProperties2(
    physicalDevice
            ,pMemoryProperties=None
            ,):

        custom_return = True
        if not pMemoryProperties:
            pMemoryProperties = ffi.new('VkPhysicalDeviceMemoryProperties2*')
            custom_return = False

        result = _callApi(fn, physicalDevice,pMemoryProperties)

        if custom_return:
            return pMemoryProperties

        return pMemoryProperties[0]



    return vkGetPhysicalDeviceMemoryProperties2
def _wrap_vkGetPhysicalDeviceSparseImageFormatProperties2(fn):
    def vkGetPhysicalDeviceSparseImageFormatProperties2(
    physicalDevice
            ,pFormatInfo
            ,):

        pPropertyCount = ffi.new('uint32_t*')
        pProperties = ffi.NULL

        result = _callApi(fn, physicalDevice,pFormatInfo,pPropertyCount,pProperties)

        pProperties = ffi.new('VkSparseImageFormatProperties2[]', pPropertyCount[0])
        result = _callApi(fn, physicalDevice,pFormatInfo,pPropertyCount,pProperties)

        return pProperties


    return vkGetPhysicalDeviceSparseImageFormatProperties2
def _wrap_vkCmdPushDescriptorSetKHR(fn):
    def vkCmdPushDescriptorSetKHR(
    commandBuffer
            ,pipelineBindPoint
            ,layout
            ,set
            ,descriptorWriteCount
            ,pDescriptorWrites
            ,):
        result = _callApi(fn, commandBuffer,pipelineBindPoint,layout,set,descriptorWriteCount,pDescriptorWrites)


    return vkCmdPushDescriptorSetKHR
def _wrap_vkTrimCommandPool(fn):
    def vkTrimCommandPool(
    device
            ,commandPool
            ,flags
            ,):
        result = _callApi(fn, device,commandPool,flags)


    return vkTrimCommandPool
def _wrap_vkGetPhysicalDeviceExternalBufferProperties(fn):
    def vkGetPhysicalDeviceExternalBufferProperties(
    physicalDevice
            ,pExternalBufferInfo
            ,pExternalBufferProperties=None
            ,):

        custom_return = True
        if not pExternalBufferProperties:
            pExternalBufferProperties = ffi.new('VkExternalBufferProperties*')
            custom_return = False

        result = _callApi(fn, physicalDevice,pExternalBufferInfo,pExternalBufferProperties)

        if custom_return:
            return pExternalBufferProperties

        return pExternalBufferProperties[0]



    return vkGetPhysicalDeviceExternalBufferProperties
def _wrap_vkGetMemoryWin32HandleKHR(fn):
    def vkGetMemoryWin32HandleKHR(
    device
            ,pGetWin32HandleInfo
            ,pHandle=None
            ,):

        custom_return = True
        if not pHandle:
            pHandle = ffi.new('HANDLE*')
            custom_return = False

        result = _callApi(fn, device,pGetWin32HandleInfo,pHandle)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pHandle

        return pHandle[0]



    return vkGetMemoryWin32HandleKHR
def _wrap_vkGetMemoryWin32HandlePropertiesKHR(fn):
    def vkGetMemoryWin32HandlePropertiesKHR(
    device
            ,handleType
            ,handle
            ,pMemoryWin32HandleProperties=None
            ,):

        custom_return = True
        if not pMemoryWin32HandleProperties:
            pMemoryWin32HandleProperties = ffi.new('VkMemoryWin32HandlePropertiesKHR*')
            custom_return = False

        result = _callApi(fn, device,handleType,handle,pMemoryWin32HandleProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pMemoryWin32HandleProperties

        return pMemoryWin32HandleProperties[0]



    return vkGetMemoryWin32HandlePropertiesKHR
def _wrap_vkGetMemoryFdKHR(fn):
    def vkGetMemoryFdKHR(
    device
            ,pGetFdInfo
            ,pFd=None
            ,):

        custom_return = True
        if not pFd:
            pFd = ffi.new('int*')
            custom_return = False

        result = _callApi(fn, device,pGetFdInfo,pFd)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pFd

        return pFd[0]



    return vkGetMemoryFdKHR
def _wrap_vkGetMemoryFdPropertiesKHR(fn):
    def vkGetMemoryFdPropertiesKHR(
    device
            ,handleType
            ,fd
            ,pMemoryFdProperties=None
            ,):

        custom_return = True
        if not pMemoryFdProperties:
            pMemoryFdProperties = ffi.new('VkMemoryFdPropertiesKHR*')
            custom_return = False

        result = _callApi(fn, device,handleType,fd,pMemoryFdProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pMemoryFdProperties

        return pMemoryFdProperties[0]



    return vkGetMemoryFdPropertiesKHR
def _wrap_vkGetPhysicalDeviceExternalSemaphoreProperties(fn):
    def vkGetPhysicalDeviceExternalSemaphoreProperties(
    physicalDevice
            ,pExternalSemaphoreInfo
            ,pExternalSemaphoreProperties=None
            ,):

        custom_return = True
        if not pExternalSemaphoreProperties:
            pExternalSemaphoreProperties = ffi.new('VkExternalSemaphoreProperties*')
            custom_return = False

        result = _callApi(fn, physicalDevice,pExternalSemaphoreInfo,pExternalSemaphoreProperties)

        if custom_return:
            return pExternalSemaphoreProperties

        return pExternalSemaphoreProperties[0]



    return vkGetPhysicalDeviceExternalSemaphoreProperties
def _wrap_vkGetSemaphoreWin32HandleKHR(fn):
    def vkGetSemaphoreWin32HandleKHR(
    device
            ,pGetWin32HandleInfo
            ,pHandle=None
            ,):

        custom_return = True
        if not pHandle:
            pHandle = ffi.new('HANDLE*')
            custom_return = False

        result = _callApi(fn, device,pGetWin32HandleInfo,pHandle)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pHandle

        return pHandle[0]



    return vkGetSemaphoreWin32HandleKHR
def _wrap_vkImportSemaphoreWin32HandleKHR(fn):
    def vkImportSemaphoreWin32HandleKHR(
    device
            ,pImportSemaphoreWin32HandleInfo
            ,):
        result = _callApi(fn, device,pImportSemaphoreWin32HandleInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkImportSemaphoreWin32HandleKHR
def _wrap_vkGetSemaphoreFdKHR(fn):
    def vkGetSemaphoreFdKHR(
    device
            ,pGetFdInfo
            ,pFd=None
            ,):

        custom_return = True
        if not pFd:
            pFd = ffi.new('int*')
            custom_return = False

        result = _callApi(fn, device,pGetFdInfo,pFd)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pFd

        return pFd[0]



    return vkGetSemaphoreFdKHR
def _wrap_vkImportSemaphoreFdKHR(fn):
    def vkImportSemaphoreFdKHR(
    device
            ,pImportSemaphoreFdInfo
            ,):
        result = _callApi(fn, device,pImportSemaphoreFdInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkImportSemaphoreFdKHR
def _wrap_vkGetPhysicalDeviceExternalFenceProperties(fn):
    def vkGetPhysicalDeviceExternalFenceProperties(
    physicalDevice
            ,pExternalFenceInfo
            ,pExternalFenceProperties=None
            ,):

        custom_return = True
        if not pExternalFenceProperties:
            pExternalFenceProperties = ffi.new('VkExternalFenceProperties*')
            custom_return = False

        result = _callApi(fn, physicalDevice,pExternalFenceInfo,pExternalFenceProperties)

        if custom_return:
            return pExternalFenceProperties

        return pExternalFenceProperties[0]



    return vkGetPhysicalDeviceExternalFenceProperties
def _wrap_vkGetFenceWin32HandleKHR(fn):
    def vkGetFenceWin32HandleKHR(
    device
            ,pGetWin32HandleInfo
            ,pHandle=None
            ,):

        custom_return = True
        if not pHandle:
            pHandle = ffi.new('HANDLE*')
            custom_return = False

        result = _callApi(fn, device,pGetWin32HandleInfo,pHandle)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pHandle

        return pHandle[0]



    return vkGetFenceWin32HandleKHR
def _wrap_vkImportFenceWin32HandleKHR(fn):
    def vkImportFenceWin32HandleKHR(
    device
            ,pImportFenceWin32HandleInfo
            ,):
        result = _callApi(fn, device,pImportFenceWin32HandleInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkImportFenceWin32HandleKHR
def _wrap_vkGetFenceFdKHR(fn):
    def vkGetFenceFdKHR(
    device
            ,pGetFdInfo
            ,pFd=None
            ,):

        custom_return = True
        if not pFd:
            pFd = ffi.new('int*')
            custom_return = False

        result = _callApi(fn, device,pGetFdInfo,pFd)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pFd

        return pFd[0]



    return vkGetFenceFdKHR
def _wrap_vkImportFenceFdKHR(fn):
    def vkImportFenceFdKHR(
    device
            ,pImportFenceFdInfo
            ,):
        result = _callApi(fn, device,pImportFenceFdInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkImportFenceFdKHR
def _wrap_vkReleaseDisplayEXT(fn):
    def vkReleaseDisplayEXT(
    physicalDevice
            ,display
            ,):
        result = _callApi(fn, physicalDevice,display)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkReleaseDisplayEXT
def _wrap_vkAcquireXlibDisplayEXT(fn):
    def vkAcquireXlibDisplayEXT(
    physicalDevice
            ,dpy
            ,display=None
            ,):

        custom_return = True
        if not display:
            display = ffi.new('VkDisplayKHR*')
            custom_return = False

        result = _callApi(fn, physicalDevice,dpy,display)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return display

        return display[0]



    return vkAcquireXlibDisplayEXT
def _wrap_vkGetRandROutputDisplayEXT(fn):
    def vkGetRandROutputDisplayEXT(
    physicalDevice
            ,dpy
            ,rrOutput
            ,pDisplay=None
            ,):

        custom_return = True
        if not pDisplay:
            pDisplay = ffi.new('VkDisplayKHR*')
            custom_return = False

        result = _callApi(fn, physicalDevice,dpy,rrOutput,pDisplay)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pDisplay

        return pDisplay[0]



    return vkGetRandROutputDisplayEXT
def _wrap_vkDisplayPowerControlEXT(fn):
    def vkDisplayPowerControlEXT(
    device
            ,display
            ,pDisplayPowerInfo
            ,):
        result = _callApi(fn, device,display,pDisplayPowerInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkDisplayPowerControlEXT
def _wrap_vkRegisterDeviceEventEXT(fn):
    def vkRegisterDeviceEventEXT(
    device
            ,pDeviceEventInfo
            ,pAllocator
            ,pFence
            ,):
        result = _callApi(fn, device,pDeviceEventInfo,pAllocator,pFence)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkRegisterDeviceEventEXT
def _wrap_vkRegisterDisplayEventEXT(fn):
    def vkRegisterDisplayEventEXT(
    device
            ,display
            ,pDisplayEventInfo
            ,pAllocator
            ,pFence
            ,):
        result = _callApi(fn, device,display,pDisplayEventInfo,pAllocator,pFence)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkRegisterDisplayEventEXT
def _wrap_vkGetSwapchainCounterEXT(fn):
    def vkGetSwapchainCounterEXT(
    device
            ,swapchain
            ,counter
            ,pCounterValue=None
            ,):

        custom_return = True
        if not pCounterValue:
            pCounterValue = ffi.new('uint64_t*')
            custom_return = False

        result = _callApi(fn, device,swapchain,counter,pCounterValue)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pCounterValue

        return pCounterValue[0]



    return vkGetSwapchainCounterEXT
def _wrap_vkGetPhysicalDeviceSurfaceCapabilities2EXT(fn):
    def vkGetPhysicalDeviceSurfaceCapabilities2EXT(
    physicalDevice
            ,surface
            ,pSurfaceCapabilities=None
            ,):

        custom_return = True
        if not pSurfaceCapabilities:
            pSurfaceCapabilities = ffi.new('VkSurfaceCapabilities2EXT*')
            custom_return = False

        result = _callApi(fn, physicalDevice,surface,pSurfaceCapabilities)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurfaceCapabilities

        return pSurfaceCapabilities[0]



    return vkGetPhysicalDeviceSurfaceCapabilities2EXT
def _wrap_vkEnumeratePhysicalDeviceGroups(fn):
    def vkEnumeratePhysicalDeviceGroups(
    instance
            ,):

        pPhysicalDeviceGroupCount = ffi.new('uint32_t*')
        pPhysicalDeviceGroupProperties = ffi.NULL

        result = _callApi(fn, instance,pPhysicalDeviceGroupCount,pPhysicalDeviceGroupProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pPhysicalDeviceGroupProperties = ffi.new('VkPhysicalDeviceGroupProperties[]', pPhysicalDeviceGroupCount[0])
        result = _callApi(fn, instance,pPhysicalDeviceGroupCount,pPhysicalDeviceGroupProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pPhysicalDeviceGroupProperties


    return vkEnumeratePhysicalDeviceGroups
def _wrap_vkGetDeviceGroupPeerMemoryFeatures(fn):
    def vkGetDeviceGroupPeerMemoryFeatures(
    device
            ,heapIndex
            ,localDeviceIndex
            ,remoteDeviceIndex
            ,pPeerMemoryFeatures=None
            ,):

        custom_return = True
        if not pPeerMemoryFeatures:
            pPeerMemoryFeatures = ffi.new('VkPeerMemoryFeatureFlags*')
            custom_return = False

        result = _callApi(fn, device,heapIndex,localDeviceIndex,remoteDeviceIndex,pPeerMemoryFeatures)

        if custom_return:
            return pPeerMemoryFeatures

        return pPeerMemoryFeatures[0]



    return vkGetDeviceGroupPeerMemoryFeatures
def _wrap_vkBindBufferMemory2(fn):
    def vkBindBufferMemory2(
    device
            ,bindInfoCount
            ,pBindInfos
            ,):
        result = _callApi(fn, device,bindInfoCount,pBindInfos)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkBindBufferMemory2
def _wrap_vkBindImageMemory2(fn):
    def vkBindImageMemory2(
    device
            ,bindInfoCount
            ,pBindInfos
            ,):
        result = _callApi(fn, device,bindInfoCount,pBindInfos)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkBindImageMemory2
def _wrap_vkCmdSetDeviceMask(fn):
    def vkCmdSetDeviceMask(
    commandBuffer
            ,deviceMask
            ,):
        result = _callApi(fn, commandBuffer,deviceMask)


    return vkCmdSetDeviceMask
def _wrap_vkGetDeviceGroupPresentCapabilitiesKHR(fn):
    def vkGetDeviceGroupPresentCapabilitiesKHR(
    device
            ,pDeviceGroupPresentCapabilities=None
            ,):

        custom_return = True
        if not pDeviceGroupPresentCapabilities:
            pDeviceGroupPresentCapabilities = ffi.new('VkDeviceGroupPresentCapabilitiesKHR*')
            custom_return = False

        result = _callApi(fn, device,pDeviceGroupPresentCapabilities)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pDeviceGroupPresentCapabilities

        return pDeviceGroupPresentCapabilities[0]



    return vkGetDeviceGroupPresentCapabilitiesKHR
def _wrap_vkGetDeviceGroupSurfacePresentModesKHR(fn):
    def vkGetDeviceGroupSurfacePresentModesKHR(
    device
            ,surface
            ,pModes=None
            ,):

        custom_return = True
        if not pModes:
            pModes = ffi.new('VkDeviceGroupPresentModeFlagsKHR*')
            custom_return = False

        result = _callApi(fn, device,surface,pModes)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pModes

        return pModes[0]



    return vkGetDeviceGroupSurfacePresentModesKHR
def _wrap_vkAcquireNextImage2KHR(fn):
    def vkAcquireNextImage2KHR(
    device
            ,):

        pAcquireInfo = ffi.new('VkAcquireNextImageInfoKHR*')
        pImageIndex = ffi.NULL

        result = _callApi(fn, device,pAcquireInfo,pImageIndex)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pImageIndex = ffi.new('uint32_t[]', pAcquireInfo[0])
        result = _callApi(fn, device,pAcquireInfo,pImageIndex)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pImageIndex


    return vkAcquireNextImage2KHR
def _wrap_vkCmdDispatchBase(fn):
    def vkCmdDispatchBase(
    commandBuffer
            ,baseGroupX
            ,baseGroupY
            ,baseGroupZ
            ,groupCountX
            ,groupCountY
            ,groupCountZ
            ,):
        result = _callApi(fn, commandBuffer,baseGroupX,baseGroupY,baseGroupZ,groupCountX,groupCountY,groupCountZ)


    return vkCmdDispatchBase
def _wrap_vkGetPhysicalDevicePresentRectanglesKHR(fn):
    def vkGetPhysicalDevicePresentRectanglesKHR(
    physicalDevice
            ,surface
            ,):

        pRectCount = ffi.new('uint32_t*')
        pRects = ffi.NULL

        result = _callApi(fn, physicalDevice,surface,pRectCount,pRects)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pRects = ffi.new('VkRect2D[]', pRectCount[0])
        result = _callApi(fn, physicalDevice,surface,pRectCount,pRects)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pRects


    return vkGetPhysicalDevicePresentRectanglesKHR
def _wrap_vkCreateDescriptorUpdateTemplate(fn):
    def vkCreateDescriptorUpdateTemplate(
    device
            ,pCreateInfo
            ,pAllocator
            ,pDescriptorUpdateTemplate=None
            ,):

        custom_return = True
        if not pDescriptorUpdateTemplate:
            pDescriptorUpdateTemplate = ffi.new('VkDescriptorUpdateTemplate*')
            custom_return = False

        result = _callApi(fn, device,pCreateInfo,pAllocator,pDescriptorUpdateTemplate)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pDescriptorUpdateTemplate

        return pDescriptorUpdateTemplate[0]



    return vkCreateDescriptorUpdateTemplate
def _wrap_vkDestroyDescriptorUpdateTemplate(fn):
    def vkDestroyDescriptorUpdateTemplate(
    device
            ,descriptorUpdateTemplate
            ,pAllocator
            ,):
        result = _callApi(fn, device,descriptorUpdateTemplate,pAllocator)


    return vkDestroyDescriptorUpdateTemplate
def _wrap_vkUpdateDescriptorSetWithTemplate(fn):
    def vkUpdateDescriptorSetWithTemplate(
    device
            ,descriptorSet
            ,descriptorUpdateTemplate
            ,pData
            ,):
        result = _callApi(fn, device,descriptorSet,descriptorUpdateTemplate,pData)


    return vkUpdateDescriptorSetWithTemplate
def _wrap_vkCmdPushDescriptorSetWithTemplateKHR(fn):
    def vkCmdPushDescriptorSetWithTemplateKHR(
    commandBuffer
            ,descriptorUpdateTemplate
            ,layout
            ,set
            ,pData
            ,):
        result = _callApi(fn, commandBuffer,descriptorUpdateTemplate,layout,set,pData)


    return vkCmdPushDescriptorSetWithTemplateKHR
def _wrap_vkSetHdrMetadataEXT(fn):
    def vkSetHdrMetadataEXT(
    device
            ,swapchainCount
            ,pSwapchains
            ,pMetadata
            ,):
        result = _callApi(fn, device,swapchainCount,pSwapchains,pMetadata)


    return vkSetHdrMetadataEXT
def _wrap_vkGetSwapchainStatusKHR(fn):
    def vkGetSwapchainStatusKHR(
    device
            ,swapchain=None
            ,):

        custom_return = True
        if not swapchain:
            swapchain = ffi.new('VkSwapchainKHR*')
            custom_return = False

        result = _callApi(fn, device,swapchain)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return swapchain

        return swapchain[0]



    return vkGetSwapchainStatusKHR
def _wrap_vkGetRefreshCycleDurationGOOGLE(fn):
    def vkGetRefreshCycleDurationGOOGLE(
    device
            ,swapchain
            ,pDisplayTimingProperties=None
            ,):

        custom_return = True
        if not pDisplayTimingProperties:
            pDisplayTimingProperties = ffi.new('VkRefreshCycleDurationGOOGLE*')
            custom_return = False

        result = _callApi(fn, device,swapchain,pDisplayTimingProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pDisplayTimingProperties

        return pDisplayTimingProperties[0]



    return vkGetRefreshCycleDurationGOOGLE
def _wrap_vkGetPastPresentationTimingGOOGLE(fn):
    def vkGetPastPresentationTimingGOOGLE(
    device
            ,swapchain
            ,):

        pPresentationTimingCount = ffi.new('uint32_t*')
        pPresentationTimings = ffi.NULL

        result = _callApi(fn, device,swapchain,pPresentationTimingCount,pPresentationTimings)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pPresentationTimings = ffi.new('VkPastPresentationTimingGOOGLE[]', pPresentationTimingCount[0])
        result = _callApi(fn, device,swapchain,pPresentationTimingCount,pPresentationTimings)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pPresentationTimings


    return vkGetPastPresentationTimingGOOGLE
def _wrap_vkCreateIOSSurfaceMVK(fn):
    def vkCreateIOSSurfaceMVK(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateIOSSurfaceMVK
def _wrap_vkCreateMacOSSurfaceMVK(fn):
    def vkCreateMacOSSurfaceMVK(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pSurface=None
            ,):

        custom_return = True
        if not pSurface:
            pSurface = ffi.new('VkSurfaceKHR*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pSurface)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurface

        return pSurface[0]



    return vkCreateMacOSSurfaceMVK
def _wrap_vkCmdSetViewportWScalingNV(fn):
    def vkCmdSetViewportWScalingNV(
    commandBuffer
            ,firstViewport
            ,viewportCount
            ,pViewportWScalings
            ,):
        result = _callApi(fn, commandBuffer,firstViewport,viewportCount,pViewportWScalings)


    return vkCmdSetViewportWScalingNV
def _wrap_vkCmdSetDiscardRectangleEXT(fn):
    def vkCmdSetDiscardRectangleEXT(
    commandBuffer
            ,firstDiscardRectangle
            ,discardRectangleCount
            ,pDiscardRectangles
            ,):
        result = _callApi(fn, commandBuffer,firstDiscardRectangle,discardRectangleCount,pDiscardRectangles)


    return vkCmdSetDiscardRectangleEXT
def _wrap_vkCmdSetSampleLocationsEXT(fn):
    def vkCmdSetSampleLocationsEXT(
    commandBuffer
            ,pSampleLocationsInfo
            ,):
        result = _callApi(fn, commandBuffer,pSampleLocationsInfo)


    return vkCmdSetSampleLocationsEXT
def _wrap_vkGetPhysicalDeviceMultisamplePropertiesEXT(fn):
    def vkGetPhysicalDeviceMultisamplePropertiesEXT(
    physicalDevice
            ,samples
            ,pMultisampleProperties=None
            ,):

        custom_return = True
        if not pMultisampleProperties:
            pMultisampleProperties = ffi.new('VkMultisamplePropertiesEXT*')
            custom_return = False

        result = _callApi(fn, physicalDevice,samples,pMultisampleProperties)

        if custom_return:
            return pMultisampleProperties

        return pMultisampleProperties[0]



    return vkGetPhysicalDeviceMultisamplePropertiesEXT
def _wrap_vkGetPhysicalDeviceSurfaceCapabilities2KHR(fn):
    def vkGetPhysicalDeviceSurfaceCapabilities2KHR(
    physicalDevice
            ,pSurfaceInfo
            ,pSurfaceCapabilities=None
            ,):

        custom_return = True
        if not pSurfaceCapabilities:
            pSurfaceCapabilities = ffi.new('VkSurfaceCapabilities2KHR*')
            custom_return = False

        result = _callApi(fn, physicalDevice,pSurfaceInfo,pSurfaceCapabilities)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pSurfaceCapabilities

        return pSurfaceCapabilities[0]



    return vkGetPhysicalDeviceSurfaceCapabilities2KHR
def _wrap_vkGetPhysicalDeviceSurfaceFormats2KHR(fn):
    def vkGetPhysicalDeviceSurfaceFormats2KHR(
    physicalDevice
            ,pSurfaceInfo
            ,):

        pSurfaceFormatCount = ffi.new('uint32_t*')
        pSurfaceFormats = ffi.NULL

        result = _callApi(fn, physicalDevice,pSurfaceInfo,pSurfaceFormatCount,pSurfaceFormats)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pSurfaceFormats = ffi.new('VkSurfaceFormat2KHR[]', pSurfaceFormatCount[0])
        result = _callApi(fn, physicalDevice,pSurfaceInfo,pSurfaceFormatCount,pSurfaceFormats)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pSurfaceFormats


    return vkGetPhysicalDeviceSurfaceFormats2KHR
def _wrap_vkGetPhysicalDeviceDisplayProperties2KHR(fn):
    def vkGetPhysicalDeviceDisplayProperties2KHR(
    physicalDevice
            ,):

        pPropertyCount = ffi.new('uint32_t*')
        pProperties = ffi.NULL

        result = _callApi(fn, physicalDevice,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pProperties = ffi.new('VkDisplayProperties2KHR[]', pPropertyCount[0])
        result = _callApi(fn, physicalDevice,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pProperties


    return vkGetPhysicalDeviceDisplayProperties2KHR
def _wrap_vkGetPhysicalDeviceDisplayPlaneProperties2KHR(fn):
    def vkGetPhysicalDeviceDisplayPlaneProperties2KHR(
    physicalDevice
            ,):

        pPropertyCount = ffi.new('uint32_t*')
        pProperties = ffi.NULL

        result = _callApi(fn, physicalDevice,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pProperties = ffi.new('VkDisplayPlaneProperties2KHR[]', pPropertyCount[0])
        result = _callApi(fn, physicalDevice,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pProperties


    return vkGetPhysicalDeviceDisplayPlaneProperties2KHR
def _wrap_vkGetDisplayModeProperties2KHR(fn):
    def vkGetDisplayModeProperties2KHR(
    physicalDevice
            ,display
            ,):

        pPropertyCount = ffi.new('uint32_t*')
        pProperties = ffi.NULL

        result = _callApi(fn, physicalDevice,display,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pProperties = ffi.new('VkDisplayModeProperties2KHR[]', pPropertyCount[0])
        result = _callApi(fn, physicalDevice,display,pPropertyCount,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pProperties


    return vkGetDisplayModeProperties2KHR
def _wrap_vkGetDisplayPlaneCapabilities2KHR(fn):
    def vkGetDisplayPlaneCapabilities2KHR(
    physicalDevice
            ,pDisplayPlaneInfo
            ,pCapabilities=None
            ,):

        custom_return = True
        if not pCapabilities:
            pCapabilities = ffi.new('VkDisplayPlaneCapabilities2KHR*')
            custom_return = False

        result = _callApi(fn, physicalDevice,pDisplayPlaneInfo,pCapabilities)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pCapabilities

        return pCapabilities[0]



    return vkGetDisplayPlaneCapabilities2KHR
def _wrap_vkGetBufferMemoryRequirements2(fn):
    def vkGetBufferMemoryRequirements2(
    device
            ,pInfo
            ,pMemoryRequirements=None
            ,):

        custom_return = True
        if not pMemoryRequirements:
            pMemoryRequirements = ffi.new('VkMemoryRequirements2*')
            custom_return = False

        result = _callApi(fn, device,pInfo,pMemoryRequirements)

        if custom_return:
            return pMemoryRequirements

        return pMemoryRequirements[0]



    return vkGetBufferMemoryRequirements2
def _wrap_vkGetImageMemoryRequirements2(fn):
    def vkGetImageMemoryRequirements2(
    device
            ,pInfo
            ,pMemoryRequirements=None
            ,):

        custom_return = True
        if not pMemoryRequirements:
            pMemoryRequirements = ffi.new('VkMemoryRequirements2*')
            custom_return = False

        result = _callApi(fn, device,pInfo,pMemoryRequirements)

        if custom_return:
            return pMemoryRequirements

        return pMemoryRequirements[0]



    return vkGetImageMemoryRequirements2
def _wrap_vkGetImageSparseMemoryRequirements2(fn):
    def vkGetImageSparseMemoryRequirements2(
    device
            ,pInfo
            ,):

        pSparseMemoryRequirementCount = ffi.new('uint32_t*')
        pSparseMemoryRequirements = ffi.NULL

        result = _callApi(fn, device,pInfo,pSparseMemoryRequirementCount,pSparseMemoryRequirements)

        pSparseMemoryRequirements = ffi.new('VkSparseImageMemoryRequirements2[]', pSparseMemoryRequirementCount[0])
        result = _callApi(fn, device,pInfo,pSparseMemoryRequirementCount,pSparseMemoryRequirements)

        return pSparseMemoryRequirements


    return vkGetImageSparseMemoryRequirements2
def _wrap_vkCreateSamplerYcbcrConversion(fn):
    def vkCreateSamplerYcbcrConversion(
    device
            ,pCreateInfo
            ,pAllocator
            ,pYcbcrConversion=None
            ,):

        custom_return = True
        if not pYcbcrConversion:
            pYcbcrConversion = ffi.new('VkSamplerYcbcrConversion*')
            custom_return = False

        result = _callApi(fn, device,pCreateInfo,pAllocator,pYcbcrConversion)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pYcbcrConversion

        return pYcbcrConversion[0]



    return vkCreateSamplerYcbcrConversion
def _wrap_vkDestroySamplerYcbcrConversion(fn):
    def vkDestroySamplerYcbcrConversion(
    device
            ,ycbcrConversion
            ,pAllocator
            ,):
        result = _callApi(fn, device,ycbcrConversion,pAllocator)


    return vkDestroySamplerYcbcrConversion
def vkGetDeviceQueue2(
device
        ,pQueueInfo
        ,pQueue=None
        ,):

    custom_return = True
    if not pQueue:
        pQueue = ffi.new('VkQueue*')
        custom_return = False

    result = _callApi(lib.vkGetDeviceQueue2, device,pQueueInfo,pQueue)

    if custom_return:
        return pQueue

    return pQueue[0]



def _wrap_vkCreateValidationCacheEXT(fn):
    def vkCreateValidationCacheEXT(
    device
            ,pCreateInfo
            ,pAllocator
            ,pValidationCache=None
            ,):

        custom_return = True
        if not pValidationCache:
            pValidationCache = ffi.new('VkValidationCacheEXT*')
            custom_return = False

        result = _callApi(fn, device,pCreateInfo,pAllocator,pValidationCache)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pValidationCache

        return pValidationCache[0]



    return vkCreateValidationCacheEXT
def _wrap_vkDestroyValidationCacheEXT(fn):
    def vkDestroyValidationCacheEXT(
    device
            ,validationCache
            ,pAllocator
            ,):
        result = _callApi(fn, device,validationCache,pAllocator)


    return vkDestroyValidationCacheEXT
def _wrap_vkGetValidationCacheDataEXT(fn):
    def vkGetValidationCacheDataEXT(
    device
            ,validationCache
            ,pDataSize
            ,pData=None
            ,):

        custom_return = True
        if not pData:
            pData = ffi.new('void[%d]' % pDataSize)
            custom_return = False

        result = _callApi(fn, device,validationCache,pDataSize,pData)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pData

        return pData



    return vkGetValidationCacheDataEXT
def _wrap_vkMergeValidationCachesEXT(fn):
    def vkMergeValidationCachesEXT(
    device
            ,dstCache
            ,srcCacheCount
            ,pSrcCaches
            ,):
        result = _callApi(fn, device,dstCache,srcCacheCount,pSrcCaches)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkMergeValidationCachesEXT
def _wrap_vkGetDescriptorSetLayoutSupport(fn):
    def vkGetDescriptorSetLayoutSupport(
    device
            ,pCreateInfo
            ,pSupport=None
            ,):

        custom_return = True
        if not pSupport:
            pSupport = ffi.new('VkDescriptorSetLayoutSupport*')
            custom_return = False

        result = _callApi(fn, device,pCreateInfo,pSupport)

        if custom_return:
            return pSupport

        return pSupport[0]



    return vkGetDescriptorSetLayoutSupport
def _wrap_vkGetSwapchainGrallocUsageANDROID(fn):
    def vkGetSwapchainGrallocUsageANDROID(
    device
            ,format
            ,imageUsage
            ,grallocUsage=None
            ,):

        custom_return = True
        if not grallocUsage:
            grallocUsage = ffi.new('int*')
            custom_return = False

        result = _callApi(fn, device,format,imageUsage,grallocUsage)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return grallocUsage

        return grallocUsage[0]



    return vkGetSwapchainGrallocUsageANDROID
def _wrap_vkAcquireImageANDROID(fn):
    def vkAcquireImageANDROID(
    device
            ,image
            ,nativeFenceFd
            ,semaphore
            ,fence=None
            ,):

        custom_return = True
        if not fence:
            fence = ffi.new('VkFence*')
            custom_return = False

        result = _callApi(fn, device,image,nativeFenceFd,semaphore,fence)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return fence

        return fence[0]



    return vkAcquireImageANDROID
def _wrap_vkQueueSignalReleaseImageANDROID(fn):
    def vkQueueSignalReleaseImageANDROID(
    queue
            ,waitSemaphoreCount
            ,pWaitSemaphores
            ,image
            ,pNativeFenceFd
            ,):
        result = _callApi(fn, queue,waitSemaphoreCount,pWaitSemaphores,image,pNativeFenceFd)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkQueueSignalReleaseImageANDROID
def _wrap_vkGetShaderInfoAMD(fn):
    def vkGetShaderInfoAMD(
    device
            ,pipeline
            ,shaderStage
            ,infoType
            ,pInfoSize
            ,pInfo=None
            ,):

        custom_return = True
        if not pInfo:
            pInfo = ffi.new('void[%d]' % pInfoSize)
            custom_return = False

        result = _callApi(fn, device,pipeline,shaderStage,infoType,pInfoSize,pInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pInfo

        return pInfo



    return vkGetShaderInfoAMD
def _wrap_vkGetPhysicalDeviceCalibrateableTimeDomainsEXT(fn):
    def vkGetPhysicalDeviceCalibrateableTimeDomainsEXT(
    physicalDevice
            ,):

        pTimeDomainCount = ffi.new('uint32_t*')
        pTimeDomains = ffi.NULL

        result = _callApi(fn, physicalDevice,pTimeDomainCount,pTimeDomains)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        pTimeDomains = ffi.new('VkTimeDomainEXT[]', pTimeDomainCount[0])
        result = _callApi(fn, physicalDevice,pTimeDomainCount,pTimeDomains)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        return pTimeDomains


    return vkGetPhysicalDeviceCalibrateableTimeDomainsEXT
def _wrap_vkGetCalibratedTimestampsEXT(fn):
    def vkGetCalibratedTimestampsEXT(
    device
            ,timestampCount
            ,pTimestampInfos
            ,pTimestamps
            ,pMaxDeviation=None
            ,):

        custom_return = True
        if not pMaxDeviation:
            pMaxDeviation = ffi.new('uint64_t*')
            custom_return = False

        result = _callApi(fn, device,timestampCount,pTimestampInfos,pTimestamps,pMaxDeviation)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pMaxDeviation

        return pMaxDeviation[0]



    return vkGetCalibratedTimestampsEXT
def _wrap_vkSetDebugUtilsObjectNameEXT(fn):
    def vkSetDebugUtilsObjectNameEXT(
    device
            ,pNameInfo
            ,):
        result = _callApi(fn, device,pNameInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkSetDebugUtilsObjectNameEXT
def _wrap_vkSetDebugUtilsObjectTagEXT(fn):
    def vkSetDebugUtilsObjectTagEXT(
    device
            ,pTagInfo
            ,):
        result = _callApi(fn, device,pTagInfo)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkSetDebugUtilsObjectTagEXT
def _wrap_vkQueueBeginDebugUtilsLabelEXT(fn):
    def vkQueueBeginDebugUtilsLabelEXT(
    queue
            ,pLabelInfo
            ,):
        result = _callApi(fn, queue,pLabelInfo)


    return vkQueueBeginDebugUtilsLabelEXT
def _wrap_vkQueueEndDebugUtilsLabelEXT(fn):
    def vkQueueEndDebugUtilsLabelEXT(
    queue
            ,):
        result = _callApi(fn, queue)


    return vkQueueEndDebugUtilsLabelEXT
def _wrap_vkQueueInsertDebugUtilsLabelEXT(fn):
    def vkQueueInsertDebugUtilsLabelEXT(
    queue
            ,pLabelInfo
            ,):
        result = _callApi(fn, queue,pLabelInfo)


    return vkQueueInsertDebugUtilsLabelEXT
def _wrap_vkCmdBeginDebugUtilsLabelEXT(fn):
    def vkCmdBeginDebugUtilsLabelEXT(
    commandBuffer
            ,pLabelInfo
            ,):
        result = _callApi(fn, commandBuffer,pLabelInfo)


    return vkCmdBeginDebugUtilsLabelEXT
def _wrap_vkCmdEndDebugUtilsLabelEXT(fn):
    def vkCmdEndDebugUtilsLabelEXT(
    commandBuffer
            ,):
        result = _callApi(fn, commandBuffer)


    return vkCmdEndDebugUtilsLabelEXT
def _wrap_vkCmdInsertDebugUtilsLabelEXT(fn):
    def vkCmdInsertDebugUtilsLabelEXT(
    commandBuffer
            ,pLabelInfo
            ,):
        result = _callApi(fn, commandBuffer,pLabelInfo)


    return vkCmdInsertDebugUtilsLabelEXT
def _wrap_vkCreateDebugUtilsMessengerEXT(fn):
    def vkCreateDebugUtilsMessengerEXT(
    instance
            ,pCreateInfo
            ,pAllocator
            ,pMessenger=None
            ,):

        custom_return = True
        if not pMessenger:
            pMessenger = ffi.new('VkDebugUtilsMessengerEXT*')
            custom_return = False

        result = _callApi(fn, instance,pCreateInfo,pAllocator,pMessenger)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pMessenger

        return pMessenger[0]



    return vkCreateDebugUtilsMessengerEXT
def _wrap_vkDestroyDebugUtilsMessengerEXT(fn):
    def vkDestroyDebugUtilsMessengerEXT(
    instance
            ,messenger
            ,pAllocator
            ,):
        result = _callApi(fn, instance,messenger,pAllocator)


    return vkDestroyDebugUtilsMessengerEXT
def _wrap_vkSubmitDebugUtilsMessageEXT(fn):
    def vkSubmitDebugUtilsMessageEXT(
    instance
            ,messageSeverity
            ,messageTypes
            ,pCallbackData
            ,):
        result = _callApi(fn, instance,messageSeverity,messageTypes,pCallbackData)


    return vkSubmitDebugUtilsMessageEXT
def _wrap_vkGetMemoryHostPointerPropertiesEXT(fn):
    def vkGetMemoryHostPointerPropertiesEXT(
    device
            ,handleType
            ,pHostPointer
            ,pMemoryHostPointerProperties=None
            ,):

        custom_return = True
        if not pMemoryHostPointerProperties:
            pMemoryHostPointerProperties = ffi.new('VkMemoryHostPointerPropertiesEXT*')
            custom_return = False

        result = _callApi(fn, device,handleType,pHostPointer,pMemoryHostPointerProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pMemoryHostPointerProperties

        return pMemoryHostPointerProperties[0]



    return vkGetMemoryHostPointerPropertiesEXT
def _wrap_vkCmdWriteBufferMarkerAMD(fn):
    def vkCmdWriteBufferMarkerAMD(
    commandBuffer
            ,pipelineStage
            ,dstBuffer
            ,dstOffset
            ,marker
            ,):
        result = _callApi(fn, commandBuffer,pipelineStage,dstBuffer,dstOffset,marker)


    return vkCmdWriteBufferMarkerAMD
def _wrap_vkCreateRenderPass2KHR(fn):
    def vkCreateRenderPass2KHR(
    device
            ,pCreateInfo
            ,pAllocator
            ,pRenderPass=None
            ,):

        custom_return = True
        if not pRenderPass:
            pRenderPass = ffi.new('VkRenderPass*')
            custom_return = False

        result = _callApi(fn, device,pCreateInfo,pAllocator,pRenderPass)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pRenderPass

        return pRenderPass[0]



    return vkCreateRenderPass2KHR
def _wrap_vkCmdBeginRenderPass2KHR(fn):
    def vkCmdBeginRenderPass2KHR(
    commandBuffer
            ,pRenderPassBegin
            ,pSubpassBeginInfo
            ,):
        result = _callApi(fn, commandBuffer,pRenderPassBegin,pSubpassBeginInfo)


    return vkCmdBeginRenderPass2KHR
def _wrap_vkCmdNextSubpass2KHR(fn):
    def vkCmdNextSubpass2KHR(
    commandBuffer
            ,pSubpassBeginInfo
            ,pSubpassEndInfo
            ,):
        result = _callApi(fn, commandBuffer,pSubpassBeginInfo,pSubpassEndInfo)


    return vkCmdNextSubpass2KHR
def _wrap_vkCmdEndRenderPass2KHR(fn):
    def vkCmdEndRenderPass2KHR(
    commandBuffer
            ,pSubpassEndInfo
            ,):
        result = _callApi(fn, commandBuffer,pSubpassEndInfo)


    return vkCmdEndRenderPass2KHR
def _wrap_vkGetAndroidHardwareBufferPropertiesANDROID(fn):
    def vkGetAndroidHardwareBufferPropertiesANDROID(
    device
            ,buffer
            ,pProperties=None
            ,):

        custom_return = True
        if not pProperties:
            pProperties = ffi.new('VkAndroidHardwareBufferPropertiesANDROID*')
            custom_return = False

        result = _callApi(fn, device,buffer,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pProperties

        return pProperties[0]



    return vkGetAndroidHardwareBufferPropertiesANDROID
def _wrap_vkGetMemoryAndroidHardwareBufferANDROID(fn):
    def vkGetMemoryAndroidHardwareBufferANDROID(
    device
            ,pInfo
            ,pBuffer=None
            ,):

        custom_return = True
        if not pBuffer:
            pBuffer = ffi.new('AHardwareBuffer*')
            custom_return = False

        result = _callApi(fn, device,pInfo,pBuffer)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pBuffer

        return pBuffer[0]



    return vkGetMemoryAndroidHardwareBufferANDROID
def _wrap_vkCmdDrawIndirectCountKHR(fn):
    def vkCmdDrawIndirectCountKHR(
    commandBuffer
            ,buffer
            ,offset
            ,countBuffer
            ,countBufferOffset
            ,maxDrawCount
            ,stride
            ,):
        result = _callApi(fn, commandBuffer,buffer,offset,countBuffer,countBufferOffset,maxDrawCount,stride)


    return vkCmdDrawIndirectCountKHR
def _wrap_vkCmdDrawIndexedIndirectCountKHR(fn):
    def vkCmdDrawIndexedIndirectCountKHR(
    commandBuffer
            ,buffer
            ,offset
            ,countBuffer
            ,countBufferOffset
            ,maxDrawCount
            ,stride
            ,):
        result = _callApi(fn, commandBuffer,buffer,offset,countBuffer,countBufferOffset,maxDrawCount,stride)


    return vkCmdDrawIndexedIndirectCountKHR
def _wrap_vkCmdSetCheckpointNV(fn):
    def vkCmdSetCheckpointNV(
    commandBuffer
            ,pCheckpointMarker
            ,):
        result = _callApi(fn, commandBuffer,pCheckpointMarker)


    return vkCmdSetCheckpointNV
def _wrap_vkGetQueueCheckpointDataNV(fn):
    def vkGetQueueCheckpointDataNV(
    queue
            ,):

        pCheckpointDataCount = ffi.new('uint32_t*')
        pCheckpointData = ffi.NULL

        result = _callApi(fn, queue,pCheckpointDataCount,pCheckpointData)

        pCheckpointData = ffi.new('VkCheckpointDataNV[]', pCheckpointDataCount[0])
        result = _callApi(fn, queue,pCheckpointDataCount,pCheckpointData)

        return pCheckpointData


    return vkGetQueueCheckpointDataNV
def _wrap_vkCmdBindTransformFeedbackBuffersEXT(fn):
    def vkCmdBindTransformFeedbackBuffersEXT(
    commandBuffer
            ,firstBinding
            ,bindingCount
            ,pBuffers
            ,pOffsets
            ,pSizes
            ,):
        result = _callApi(fn, commandBuffer,firstBinding,bindingCount,pBuffers,pOffsets,pSizes)


    return vkCmdBindTransformFeedbackBuffersEXT
def _wrap_vkCmdBeginTransformFeedbackEXT(fn):
    def vkCmdBeginTransformFeedbackEXT(
    commandBuffer
            ,firstCounterBuffer
            ,counterBufferCount
            ,pCounterBuffers
            ,pCounterBufferOffsets
            ,):
        result = _callApi(fn, commandBuffer,firstCounterBuffer,counterBufferCount,pCounterBuffers,pCounterBufferOffsets)


    return vkCmdBeginTransformFeedbackEXT
def _wrap_vkCmdEndTransformFeedbackEXT(fn):
    def vkCmdEndTransformFeedbackEXT(
    commandBuffer
            ,firstCounterBuffer
            ,counterBufferCount
            ,pCounterBuffers
            ,pCounterBufferOffsets
            ,):
        result = _callApi(fn, commandBuffer,firstCounterBuffer,counterBufferCount,pCounterBuffers,pCounterBufferOffsets)


    return vkCmdEndTransformFeedbackEXT
def _wrap_vkCmdBeginQueryIndexedEXT(fn):
    def vkCmdBeginQueryIndexedEXT(
    commandBuffer
            ,queryPool
            ,query
            ,flags
            ,index
            ,):
        result = _callApi(fn, commandBuffer,queryPool,query,flags,index)


    return vkCmdBeginQueryIndexedEXT
def _wrap_vkCmdEndQueryIndexedEXT(fn):
    def vkCmdEndQueryIndexedEXT(
    commandBuffer
            ,queryPool
            ,query
            ,index
            ,):
        result = _callApi(fn, commandBuffer,queryPool,query,index)


    return vkCmdEndQueryIndexedEXT
def _wrap_vkCmdDrawIndirectByteCountEXT(fn):
    def vkCmdDrawIndirectByteCountEXT(
    commandBuffer
            ,instanceCount
            ,firstInstance
            ,counterBuffer
            ,counterBufferOffset
            ,counterOffset
            ,vertexStride
            ,):
        result = _callApi(fn, commandBuffer,instanceCount,firstInstance,counterBuffer,counterBufferOffset,counterOffset,vertexStride)


    return vkCmdDrawIndirectByteCountEXT
def _wrap_vkCmdSetExclusiveScissorNV(fn):
    def vkCmdSetExclusiveScissorNV(
    commandBuffer
            ,firstExclusiveScissor
            ,exclusiveScissorCount
            ,pExclusiveScissors
            ,):
        result = _callApi(fn, commandBuffer,firstExclusiveScissor,exclusiveScissorCount,pExclusiveScissors)


    return vkCmdSetExclusiveScissorNV
def _wrap_vkCmdBindShadingRateImageNV(fn):
    def vkCmdBindShadingRateImageNV(
    commandBuffer
            ,imageView
            ,imageLayout
            ,):
        result = _callApi(fn, commandBuffer,imageView,imageLayout)


    return vkCmdBindShadingRateImageNV
def _wrap_vkCmdSetViewportShadingRatePaletteNV(fn):
    def vkCmdSetViewportShadingRatePaletteNV(
    commandBuffer
            ,firstViewport
            ,viewportCount
            ,pShadingRatePalettes
            ,):
        result = _callApi(fn, commandBuffer,firstViewport,viewportCount,pShadingRatePalettes)


    return vkCmdSetViewportShadingRatePaletteNV
def _wrap_vkCmdSetCoarseSampleOrderNV(fn):
    def vkCmdSetCoarseSampleOrderNV(
    commandBuffer
            ,sampleOrderType
            ,customSampleOrderCount
            ,pCustomSampleOrders
            ,):
        result = _callApi(fn, commandBuffer,sampleOrderType,customSampleOrderCount,pCustomSampleOrders)


    return vkCmdSetCoarseSampleOrderNV
def _wrap_vkCmdDrawMeshTasksNV(fn):
    def vkCmdDrawMeshTasksNV(
    commandBuffer
            ,taskCount
            ,firstTask
            ,):
        result = _callApi(fn, commandBuffer,taskCount,firstTask)


    return vkCmdDrawMeshTasksNV
def _wrap_vkCmdDrawMeshTasksIndirectNV(fn):
    def vkCmdDrawMeshTasksIndirectNV(
    commandBuffer
            ,buffer
            ,offset
            ,drawCount
            ,stride
            ,):
        result = _callApi(fn, commandBuffer,buffer,offset,drawCount,stride)


    return vkCmdDrawMeshTasksIndirectNV
def _wrap_vkCmdDrawMeshTasksIndirectCountNV(fn):
    def vkCmdDrawMeshTasksIndirectCountNV(
    commandBuffer
            ,buffer
            ,offset
            ,countBuffer
            ,countBufferOffset
            ,maxDrawCount
            ,stride
            ,):
        result = _callApi(fn, commandBuffer,buffer,offset,countBuffer,countBufferOffset,maxDrawCount,stride)


    return vkCmdDrawMeshTasksIndirectCountNV
def _wrap_vkCompileDeferredNV(fn):
    def vkCompileDeferredNV(
    device
            ,pipeline
            ,shader
            ,):
        result = _callApi(fn, device,pipeline,shader)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkCompileDeferredNV
def _wrap_vkCreateAccelerationStructureNV(fn):
    def vkCreateAccelerationStructureNV(
    device
            ,pCreateInfo
            ,pAllocator
            ,pAccelerationStructure=None
            ,):

        custom_return = True
        if not pAccelerationStructure:
            pAccelerationStructure = ffi.new('VkAccelerationStructureNV*')
            custom_return = False

        result = _callApi(fn, device,pCreateInfo,pAllocator,pAccelerationStructure)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pAccelerationStructure

        return pAccelerationStructure[0]



    return vkCreateAccelerationStructureNV
def _wrap_vkDestroyAccelerationStructureNV(fn):
    def vkDestroyAccelerationStructureNV(
    device
            ,accelerationStructure
            ,pAllocator
            ,):
        result = _callApi(fn, device,accelerationStructure,pAllocator)


    return vkDestroyAccelerationStructureNV
def _wrap_vkGetAccelerationStructureMemoryRequirementsNV(fn):
    def vkGetAccelerationStructureMemoryRequirementsNV(
    device
            ,pInfo
            ,pMemoryRequirements=None
            ,):

        custom_return = True
        if not pMemoryRequirements:
            pMemoryRequirements = ffi.new('VkMemoryRequirements2KHR*')
            custom_return = False

        result = _callApi(fn, device,pInfo,pMemoryRequirements)

        if custom_return:
            return pMemoryRequirements

        return pMemoryRequirements[0]



    return vkGetAccelerationStructureMemoryRequirementsNV
def _wrap_vkBindAccelerationStructureMemoryNV(fn):
    def vkBindAccelerationStructureMemoryNV(
    device
            ,bindInfoCount
            ,pBindInfos
            ,):
        result = _callApi(fn, device,bindInfoCount,pBindInfos)
        if result != VK_SUCCESS:
            raise exception_codes[result]


    return vkBindAccelerationStructureMemoryNV
def _wrap_vkCmdCopyAccelerationStructureNV(fn):
    def vkCmdCopyAccelerationStructureNV(
    commandBuffer
            ,dst
            ,src
            ,mode
            ,):
        result = _callApi(fn, commandBuffer,dst,src,mode)


    return vkCmdCopyAccelerationStructureNV
def _wrap_vkCmdWriteAccelerationStructuresPropertiesNV(fn):
    def vkCmdWriteAccelerationStructuresPropertiesNV(
    commandBuffer
            ,accelerationStructureCount
            ,pAccelerationStructures
            ,queryType
            ,queryPool
            ,firstQuery
            ,):
        result = _callApi(fn, commandBuffer,accelerationStructureCount,pAccelerationStructures,queryType,queryPool,firstQuery)


    return vkCmdWriteAccelerationStructuresPropertiesNV
def _wrap_vkCmdBuildAccelerationStructureNV(fn):
    def vkCmdBuildAccelerationStructureNV(
    commandBuffer
            ,pInfo
            ,instanceData
            ,instanceOffset
            ,update
            ,dst
            ,src
            ,scratch
            ,scratchOffset
            ,):
        result = _callApi(fn, commandBuffer,pInfo,instanceData,instanceOffset,update,dst,src,scratch,scratchOffset)


    return vkCmdBuildAccelerationStructureNV
def _wrap_vkCmdTraceRaysNV(fn):
    def vkCmdTraceRaysNV(
    commandBuffer
            ,raygenShaderBindingTableBuffer
            ,raygenShaderBindingOffset
            ,missShaderBindingTableBuffer
            ,missShaderBindingOffset
            ,missShaderBindingStride
            ,hitShaderBindingTableBuffer
            ,hitShaderBindingOffset
            ,hitShaderBindingStride
            ,callableShaderBindingTableBuffer
            ,callableShaderBindingOffset
            ,callableShaderBindingStride
            ,width
            ,height
            ,depth
            ,):
        result = _callApi(fn, commandBuffer,raygenShaderBindingTableBuffer,raygenShaderBindingOffset,missShaderBindingTableBuffer,missShaderBindingOffset,missShaderBindingStride,hitShaderBindingTableBuffer,hitShaderBindingOffset,hitShaderBindingStride,callableShaderBindingTableBuffer,callableShaderBindingOffset,callableShaderBindingStride,width,height,depth)


    return vkCmdTraceRaysNV
def _wrap_vkGetRayTracingShaderGroupHandlesNV(fn):
    def vkGetRayTracingShaderGroupHandlesNV(
    device
            ,pipeline
            ,firstGroup
            ,groupCount
            ,dataSize
            ,pData=None
            ,):

        custom_return = True
        if not pData:
            pData = ffi.new('uint64_t[%d]' % dataSize)
            custom_return = False

        result = _callApi(fn, device,pipeline,firstGroup,groupCount,dataSize,pData)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pData

        return pData



    return vkGetRayTracingShaderGroupHandlesNV
def _wrap_vkGetAccelerationStructureHandleNV(fn):
    def vkGetAccelerationStructureHandleNV(
    device
            ,accelerationStructure
            ,dataSize
            ,pData=None
            ,):

        custom_return = True
        if not pData:
            pData = ffi.new('uint64_t[%d]' % dataSize)
            custom_return = False

        result = _callApi(fn, device,accelerationStructure,dataSize,pData)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pData

        return pData



    return vkGetAccelerationStructureHandleNV
def _wrap_vkCreateRayTracingPipelinesNV(fn):
    def vkCreateRayTracingPipelinesNV(
    device
            ,pipelineCache
            ,createInfoCount
            ,pCreateInfos
            ,pAllocator
            ,pPipelines=None
            ,):

        custom_return = True
        if not pPipelines:
            pPipelines = ffi.new('VkPipeline[%d]' % createInfoCount)
            custom_return = False

        result = _callApi(fn, device,pipelineCache,createInfoCount,pCreateInfos,pAllocator,pPipelines)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pPipelines

        return pPipelines



    return vkCreateRayTracingPipelinesNV
def _wrap_vkGetImageDrmFormatModifierPropertiesEXT(fn):
    def vkGetImageDrmFormatModifierPropertiesEXT(
    device
            ,image
            ,pProperties=None
            ,):

        custom_return = True
        if not pProperties:
            pProperties = ffi.new('VkImageDrmFormatModifierPropertiesEXT*')
            custom_return = False

        result = _callApi(fn, device,image,pProperties)
        if result != VK_SUCCESS:
            raise exception_codes[result]

        if custom_return:
            return pProperties

        return pProperties[0]



    return vkGetImageDrmFormatModifierPropertiesEXT
def _wrap_vkGetBufferDeviceAddressEXT(fn):
    def vkGetBufferDeviceAddressEXT(
    device
            ,pInfo=None
            ,):

        custom_return = True
        if not pInfo:
            pInfo = ffi.new('VkBufferDeviceAddressInfoEXT*')
            custom_return = False

        result = _callApi(fn, device,pInfo)

        if custom_return:
            return pInfo

        return pInfo[0]



    return vkGetBufferDeviceAddressEXT

_instance_ext_funcs = {
    'vkDestroySurfaceKHR':_wrap_vkDestroySurfaceKHR,
    'vkGetPhysicalDeviceSurfaceSupportKHR':_wrap_vkGetPhysicalDeviceSurfaceSupportKHR,
    'vkGetPhysicalDeviceSurfaceCapabilitiesKHR':_wrap_vkGetPhysicalDeviceSurfaceCapabilitiesKHR,
    'vkGetPhysicalDeviceSurfaceFormatsKHR':_wrap_vkGetPhysicalDeviceSurfaceFormatsKHR,
    'vkGetPhysicalDeviceSurfacePresentModesKHR':_wrap_vkGetPhysicalDeviceSurfacePresentModesKHR,
    'vkGetPhysicalDeviceDisplayPropertiesKHR':_wrap_vkGetPhysicalDeviceDisplayPropertiesKHR,
    'vkGetPhysicalDeviceDisplayPlanePropertiesKHR':_wrap_vkGetPhysicalDeviceDisplayPlanePropertiesKHR,
    'vkGetDisplayPlaneSupportedDisplaysKHR':_wrap_vkGetDisplayPlaneSupportedDisplaysKHR,
    'vkGetDisplayModePropertiesKHR':_wrap_vkGetDisplayModePropertiesKHR,
    'vkCreateDisplayModeKHR':_wrap_vkCreateDisplayModeKHR,
    'vkGetDisplayPlaneCapabilitiesKHR':_wrap_vkGetDisplayPlaneCapabilitiesKHR,
    'vkCreateDisplayPlaneSurfaceKHR':_wrap_vkCreateDisplayPlaneSurfaceKHR,
    'vkCreateXlibSurfaceKHR':_wrap_vkCreateXlibSurfaceKHR,
    'vkGetPhysicalDeviceXlibPresentationSupportKHR':_wrap_vkGetPhysicalDeviceXlibPresentationSupportKHR,
    'vkCreateXcbSurfaceKHR':_wrap_vkCreateXcbSurfaceKHR,
    'vkGetPhysicalDeviceXcbPresentationSupportKHR':_wrap_vkGetPhysicalDeviceXcbPresentationSupportKHR,
    'vkCreateWaylandSurfaceKHR':_wrap_vkCreateWaylandSurfaceKHR,
    'vkGetPhysicalDeviceWaylandPresentationSupportKHR':_wrap_vkGetPhysicalDeviceWaylandPresentationSupportKHR,
    'vkCreateAndroidSurfaceKHR':_wrap_vkCreateAndroidSurfaceKHR,
    'vkCreateWin32SurfaceKHR':_wrap_vkCreateWin32SurfaceKHR,
    'vkGetPhysicalDeviceWin32PresentationSupportKHR':_wrap_vkGetPhysicalDeviceWin32PresentationSupportKHR,
    'vkCreateDebugReportCallbackEXT':_wrap_vkCreateDebugReportCallbackEXT,
    'vkDestroyDebugReportCallbackEXT':_wrap_vkDestroyDebugReportCallbackEXT,
    'vkDebugReportMessageEXT':_wrap_vkDebugReportMessageEXT,
    'vkGetPhysicalDeviceExternalImageFormatPropertiesNV':_wrap_vkGetPhysicalDeviceExternalImageFormatPropertiesNV,
    'vkGetPhysicalDeviceFeatures2KHR':_wrap_vkGetPhysicalDeviceFeatures2,
    'vkGetPhysicalDeviceProperties2KHR':_wrap_vkGetPhysicalDeviceProperties2,
    'vkGetPhysicalDeviceFormatProperties2KHR':_wrap_vkGetPhysicalDeviceFormatProperties2,
    'vkGetPhysicalDeviceImageFormatProperties2KHR':_wrap_vkGetPhysicalDeviceImageFormatProperties2,
    'vkGetPhysicalDeviceQueueFamilyProperties2KHR':_wrap_vkGetPhysicalDeviceQueueFamilyProperties2,
    'vkGetPhysicalDeviceMemoryProperties2KHR':_wrap_vkGetPhysicalDeviceMemoryProperties2,
    'vkGetPhysicalDeviceSparseImageFormatProperties2KHR':_wrap_vkGetPhysicalDeviceSparseImageFormatProperties2,
    'vkCreateViSurfaceNN':_wrap_vkCreateViSurfaceNN,
    'vkEnumeratePhysicalDeviceGroupsKHR':_wrap_vkEnumeratePhysicalDeviceGroups,
    'vkGetPhysicalDeviceExternalBufferPropertiesKHR':_wrap_vkGetPhysicalDeviceExternalBufferProperties,
    'vkGetPhysicalDeviceExternalSemaphorePropertiesKHR':_wrap_vkGetPhysicalDeviceExternalSemaphoreProperties,
    'vkReleaseDisplayEXT':_wrap_vkReleaseDisplayEXT,
    'vkAcquireXlibDisplayEXT':_wrap_vkAcquireXlibDisplayEXT,
    'vkGetRandROutputDisplayEXT':_wrap_vkGetRandROutputDisplayEXT,
    'vkGetPhysicalDeviceSurfaceCapabilities2EXT':_wrap_vkGetPhysicalDeviceSurfaceCapabilities2EXT,
    'vkGetPhysicalDeviceExternalFencePropertiesKHR':_wrap_vkGetPhysicalDeviceExternalFenceProperties,
    'vkGetPhysicalDeviceSurfaceCapabilities2KHR':_wrap_vkGetPhysicalDeviceSurfaceCapabilities2KHR,
    'vkGetPhysicalDeviceSurfaceFormats2KHR':_wrap_vkGetPhysicalDeviceSurfaceFormats2KHR,
    'vkGetPhysicalDeviceDisplayProperties2KHR':_wrap_vkGetPhysicalDeviceDisplayProperties2KHR,
    'vkGetPhysicalDeviceDisplayPlaneProperties2KHR':_wrap_vkGetPhysicalDeviceDisplayPlaneProperties2KHR,
    'vkGetDisplayModeProperties2KHR':_wrap_vkGetDisplayModeProperties2KHR,
    'vkGetDisplayPlaneCapabilities2KHR':_wrap_vkGetDisplayPlaneCapabilities2KHR,
    'vkCreateIOSSurfaceMVK':_wrap_vkCreateIOSSurfaceMVK,
    'vkCreateMacOSSurfaceMVK':_wrap_vkCreateMacOSSurfaceMVK,
    'vkSetDebugUtilsObjectNameEXT':_wrap_vkSetDebugUtilsObjectNameEXT,
    'vkSetDebugUtilsObjectTagEXT':_wrap_vkSetDebugUtilsObjectTagEXT,
    'vkQueueBeginDebugUtilsLabelEXT':_wrap_vkQueueBeginDebugUtilsLabelEXT,
    'vkQueueEndDebugUtilsLabelEXT':_wrap_vkQueueEndDebugUtilsLabelEXT,
    'vkQueueInsertDebugUtilsLabelEXT':_wrap_vkQueueInsertDebugUtilsLabelEXT,
    'vkCmdBeginDebugUtilsLabelEXT':_wrap_vkCmdBeginDebugUtilsLabelEXT,
    'vkCmdEndDebugUtilsLabelEXT':_wrap_vkCmdEndDebugUtilsLabelEXT,
    'vkCmdInsertDebugUtilsLabelEXT':_wrap_vkCmdInsertDebugUtilsLabelEXT,
    'vkCreateDebugUtilsMessengerEXT':_wrap_vkCreateDebugUtilsMessengerEXT,
    'vkDestroyDebugUtilsMessengerEXT':_wrap_vkDestroyDebugUtilsMessengerEXT,
    'vkSubmitDebugUtilsMessageEXT':_wrap_vkSubmitDebugUtilsMessageEXT,
    'vkCreateImagePipeSurfaceFUCHSIA':_wrap_vkCreateImagePipeSurfaceFUCHSIA,
    'vkCreateSwapchainKHR':_wrap_vkCreateSwapchainKHR,
    'vkDestroySwapchainKHR':_wrap_vkDestroySwapchainKHR,
    'vkGetSwapchainImagesKHR':_wrap_vkGetSwapchainImagesKHR,
    'vkAcquireNextImageKHR':_wrap_vkAcquireNextImageKHR,
    'vkQueuePresentKHR':_wrap_vkQueuePresentKHR,
    'vkGetDeviceGroupPresentCapabilitiesKHR':_wrap_vkGetDeviceGroupPresentCapabilitiesKHR,
    'vkGetDeviceGroupSurfacePresentModesKHR':_wrap_vkGetDeviceGroupSurfacePresentModesKHR,
    'vkGetPhysicalDevicePresentRectanglesKHR':_wrap_vkGetPhysicalDevicePresentRectanglesKHR,
    'vkAcquireNextImage2KHR':_wrap_vkAcquireNextImage2KHR,
    'vkCreateSharedSwapchainsKHR':_wrap_vkCreateSharedSwapchainsKHR,
    'vkGetSwapchainGrallocUsageANDROID':_wrap_vkGetSwapchainGrallocUsageANDROID,
    'vkAcquireImageANDROID':_wrap_vkAcquireImageANDROID,
    'vkQueueSignalReleaseImageANDROID':_wrap_vkQueueSignalReleaseImageANDROID,
    'vkDebugMarkerSetObjectTagEXT':_wrap_vkDebugMarkerSetObjectTagEXT,
    'vkDebugMarkerSetObjectNameEXT':_wrap_vkDebugMarkerSetObjectNameEXT,
    'vkCmdDebugMarkerBeginEXT':_wrap_vkCmdDebugMarkerBeginEXT,
    'vkCmdDebugMarkerEndEXT':_wrap_vkCmdDebugMarkerEndEXT,
    'vkCmdDebugMarkerInsertEXT':_wrap_vkCmdDebugMarkerInsertEXT,
    'vkCmdBindTransformFeedbackBuffersEXT':_wrap_vkCmdBindTransformFeedbackBuffersEXT,
    'vkCmdBeginTransformFeedbackEXT':_wrap_vkCmdBeginTransformFeedbackEXT,
    'vkCmdEndTransformFeedbackEXT':_wrap_vkCmdEndTransformFeedbackEXT,
    'vkCmdBeginQueryIndexedEXT':_wrap_vkCmdBeginQueryIndexedEXT,
    'vkCmdEndQueryIndexedEXT':_wrap_vkCmdEndQueryIndexedEXT,
    'vkCmdDrawIndirectByteCountEXT':_wrap_vkCmdDrawIndirectByteCountEXT,
    'vkCmdDrawIndirectCountAMD':_wrap_vkCmdDrawIndirectCountAMD,
    'vkCmdDrawIndexedIndirectCountAMD':_wrap_vkCmdDrawIndexedIndirectCountAMD,
    'vkGetShaderInfoAMD':_wrap_vkGetShaderInfoAMD,
    'vkGetMemoryWin32HandleNV':_wrap_vkGetMemoryWin32HandleNV,
    'vkGetDeviceGroupPeerMemoryFeaturesKHR':_wrap_vkGetDeviceGroupPeerMemoryFeatures,
    'vkCmdSetDeviceMaskKHR':_wrap_vkCmdSetDeviceMask,
    'vkCmdDispatchBaseKHR':_wrap_vkCmdDispatchBase,
    'vkTrimCommandPoolKHR':_wrap_vkTrimCommandPool,
    'vkGetMemoryWin32HandleKHR':_wrap_vkGetMemoryWin32HandleKHR,
    'vkGetMemoryWin32HandlePropertiesKHR':_wrap_vkGetMemoryWin32HandlePropertiesKHR,
    'vkGetMemoryFdKHR':_wrap_vkGetMemoryFdKHR,
    'vkGetMemoryFdPropertiesKHR':_wrap_vkGetMemoryFdPropertiesKHR,
    'vkImportSemaphoreWin32HandleKHR':_wrap_vkImportSemaphoreWin32HandleKHR,
    'vkGetSemaphoreWin32HandleKHR':_wrap_vkGetSemaphoreWin32HandleKHR,
    'vkImportSemaphoreFdKHR':_wrap_vkImportSemaphoreFdKHR,
    'vkGetSemaphoreFdKHR':_wrap_vkGetSemaphoreFdKHR,
    'vkCmdPushDescriptorSetKHR':_wrap_vkCmdPushDescriptorSetKHR,
    'vkCmdPushDescriptorSetWithTemplateKHR':_wrap_vkCmdPushDescriptorSetWithTemplateKHR,
    'vkCmdBeginConditionalRenderingEXT':_wrap_vkCmdBeginConditionalRenderingEXT,
    'vkCmdEndConditionalRenderingEXT':_wrap_vkCmdEndConditionalRenderingEXT,
    'vkCreateDescriptorUpdateTemplateKHR':_wrap_vkCreateDescriptorUpdateTemplate,
    'vkDestroyDescriptorUpdateTemplateKHR':_wrap_vkDestroyDescriptorUpdateTemplate,
    'vkUpdateDescriptorSetWithTemplateKHR':_wrap_vkUpdateDescriptorSetWithTemplate,
    'vkCmdProcessCommandsNVX':_wrap_vkCmdProcessCommandsNVX,
    'vkCmdReserveSpaceForCommandsNVX':_wrap_vkCmdReserveSpaceForCommandsNVX,
    'vkCreateIndirectCommandsLayoutNVX':_wrap_vkCreateIndirectCommandsLayoutNVX,
    'vkDestroyIndirectCommandsLayoutNVX':_wrap_vkDestroyIndirectCommandsLayoutNVX,
    'vkCreateObjectTableNVX':_wrap_vkCreateObjectTableNVX,
    'vkDestroyObjectTableNVX':_wrap_vkDestroyObjectTableNVX,
    'vkRegisterObjectsNVX':_wrap_vkRegisterObjectsNVX,
    'vkUnregisterObjectsNVX':_wrap_vkUnregisterObjectsNVX,
    'vkGetPhysicalDeviceGeneratedCommandsPropertiesNVX':_wrap_vkGetPhysicalDeviceGeneratedCommandsPropertiesNVX,
    'vkCmdSetViewportWScalingNV':_wrap_vkCmdSetViewportWScalingNV,
    'vkDisplayPowerControlEXT':_wrap_vkDisplayPowerControlEXT,
    'vkRegisterDeviceEventEXT':_wrap_vkRegisterDeviceEventEXT,
    'vkRegisterDisplayEventEXT':_wrap_vkRegisterDisplayEventEXT,
    'vkGetSwapchainCounterEXT':_wrap_vkGetSwapchainCounterEXT,
    'vkGetRefreshCycleDurationGOOGLE':_wrap_vkGetRefreshCycleDurationGOOGLE,
    'vkGetPastPresentationTimingGOOGLE':_wrap_vkGetPastPresentationTimingGOOGLE,
    'vkCmdSetDiscardRectangleEXT':_wrap_vkCmdSetDiscardRectangleEXT,
    'vkSetHdrMetadataEXT':_wrap_vkSetHdrMetadataEXT,
    'vkCreateRenderPass2KHR':_wrap_vkCreateRenderPass2KHR,
    'vkCmdBeginRenderPass2KHR':_wrap_vkCmdBeginRenderPass2KHR,
    'vkCmdNextSubpass2KHR':_wrap_vkCmdNextSubpass2KHR,
    'vkCmdEndRenderPass2KHR':_wrap_vkCmdEndRenderPass2KHR,
    'vkGetSwapchainStatusKHR':_wrap_vkGetSwapchainStatusKHR,
    'vkImportFenceWin32HandleKHR':_wrap_vkImportFenceWin32HandleKHR,
    'vkGetFenceWin32HandleKHR':_wrap_vkGetFenceWin32HandleKHR,
    'vkImportFenceFdKHR':_wrap_vkImportFenceFdKHR,
    'vkGetFenceFdKHR':_wrap_vkGetFenceFdKHR,
    'vkGetAndroidHardwareBufferPropertiesANDROID':_wrap_vkGetAndroidHardwareBufferPropertiesANDROID,
    'vkGetMemoryAndroidHardwareBufferANDROID':_wrap_vkGetMemoryAndroidHardwareBufferANDROID,
    'vkCmdSetSampleLocationsEXT':_wrap_vkCmdSetSampleLocationsEXT,
    'vkGetPhysicalDeviceMultisamplePropertiesEXT':_wrap_vkGetPhysicalDeviceMultisamplePropertiesEXT,
    'vkGetImageMemoryRequirements2KHR':_wrap_vkGetImageMemoryRequirements2,
    'vkGetBufferMemoryRequirements2KHR':_wrap_vkGetBufferMemoryRequirements2,
    'vkGetImageSparseMemoryRequirements2KHR':_wrap_vkGetImageSparseMemoryRequirements2,
    'vkCreateSamplerYcbcrConversionKHR':_wrap_vkCreateSamplerYcbcrConversion,
    'vkDestroySamplerYcbcrConversionKHR':_wrap_vkDestroySamplerYcbcrConversion,
    'vkBindBufferMemory2KHR':_wrap_vkBindBufferMemory2,
    'vkBindImageMemory2KHR':_wrap_vkBindImageMemory2,
    'vkGetImageDrmFormatModifierPropertiesEXT':_wrap_vkGetImageDrmFormatModifierPropertiesEXT,
    'vkCreateValidationCacheEXT':_wrap_vkCreateValidationCacheEXT,
    'vkDestroyValidationCacheEXT':_wrap_vkDestroyValidationCacheEXT,
    'vkMergeValidationCachesEXT':_wrap_vkMergeValidationCachesEXT,
    'vkGetValidationCacheDataEXT':_wrap_vkGetValidationCacheDataEXT,
    'vkCmdBindShadingRateImageNV':_wrap_vkCmdBindShadingRateImageNV,
    'vkCmdSetViewportShadingRatePaletteNV':_wrap_vkCmdSetViewportShadingRatePaletteNV,
    'vkCmdSetCoarseSampleOrderNV':_wrap_vkCmdSetCoarseSampleOrderNV,
    'vkCreateAccelerationStructureNV':_wrap_vkCreateAccelerationStructureNV,
    'vkDestroyAccelerationStructureNV':_wrap_vkDestroyAccelerationStructureNV,
    'vkGetAccelerationStructureMemoryRequirementsNV':_wrap_vkGetAccelerationStructureMemoryRequirementsNV,
    'vkBindAccelerationStructureMemoryNV':_wrap_vkBindAccelerationStructureMemoryNV,
    'vkCmdBuildAccelerationStructureNV':_wrap_vkCmdBuildAccelerationStructureNV,
    'vkCmdCopyAccelerationStructureNV':_wrap_vkCmdCopyAccelerationStructureNV,
    'vkCmdTraceRaysNV':_wrap_vkCmdTraceRaysNV,
    'vkCreateRayTracingPipelinesNV':_wrap_vkCreateRayTracingPipelinesNV,
    'vkGetRayTracingShaderGroupHandlesNV':_wrap_vkGetRayTracingShaderGroupHandlesNV,
    'vkGetAccelerationStructureHandleNV':_wrap_vkGetAccelerationStructureHandleNV,
    'vkCmdWriteAccelerationStructuresPropertiesNV':_wrap_vkCmdWriteAccelerationStructuresPropertiesNV,
    'vkCompileDeferredNV':_wrap_vkCompileDeferredNV,
    'vkGetDescriptorSetLayoutSupportKHR':_wrap_vkGetDescriptorSetLayoutSupport,
    'vkCmdDrawIndirectCountKHR':_wrap_vkCmdDrawIndirectCountKHR,
    'vkCmdDrawIndexedIndirectCountKHR':_wrap_vkCmdDrawIndexedIndirectCountKHR,
    'vkGetMemoryHostPointerPropertiesEXT':_wrap_vkGetMemoryHostPointerPropertiesEXT,
    'vkCmdWriteBufferMarkerAMD':_wrap_vkCmdWriteBufferMarkerAMD,
    'vkGetPhysicalDeviceCalibrateableTimeDomainsEXT':_wrap_vkGetPhysicalDeviceCalibrateableTimeDomainsEXT,
    'vkGetCalibratedTimestampsEXT':_wrap_vkGetCalibratedTimestampsEXT,
    'vkCmdDrawMeshTasksNV':_wrap_vkCmdDrawMeshTasksNV,
    'vkCmdDrawMeshTasksIndirectNV':_wrap_vkCmdDrawMeshTasksIndirectNV,
    'vkCmdDrawMeshTasksIndirectCountNV':_wrap_vkCmdDrawMeshTasksIndirectCountNV,
    'vkCmdSetExclusiveScissorNV':_wrap_vkCmdSetExclusiveScissorNV,
    'vkCmdSetCheckpointNV':_wrap_vkCmdSetCheckpointNV,
    'vkGetQueueCheckpointDataNV':_wrap_vkGetQueueCheckpointDataNV,
    'vkGetBufferDeviceAddressEXT':_wrap_vkGetBufferDeviceAddressEXT,
}


_device_ext_funcs = {
    'vkCreateSwapchainKHR':_wrap_vkCreateSwapchainKHR,
    'vkDestroySwapchainKHR':_wrap_vkDestroySwapchainKHR,
    'vkGetSwapchainImagesKHR':_wrap_vkGetSwapchainImagesKHR,
    'vkAcquireNextImageKHR':_wrap_vkAcquireNextImageKHR,
    'vkQueuePresentKHR':_wrap_vkQueuePresentKHR,
    'vkGetDeviceGroupPresentCapabilitiesKHR':_wrap_vkGetDeviceGroupPresentCapabilitiesKHR,
    'vkGetDeviceGroupSurfacePresentModesKHR':_wrap_vkGetDeviceGroupSurfacePresentModesKHR,
    'vkGetPhysicalDevicePresentRectanglesKHR':_wrap_vkGetPhysicalDevicePresentRectanglesKHR,
    'vkAcquireNextImage2KHR':_wrap_vkAcquireNextImage2KHR,
    'vkCreateSharedSwapchainsKHR':_wrap_vkCreateSharedSwapchainsKHR,
    'vkGetSwapchainGrallocUsageANDROID':_wrap_vkGetSwapchainGrallocUsageANDROID,
    'vkAcquireImageANDROID':_wrap_vkAcquireImageANDROID,
    'vkQueueSignalReleaseImageANDROID':_wrap_vkQueueSignalReleaseImageANDROID,
    'vkDebugMarkerSetObjectTagEXT':_wrap_vkDebugMarkerSetObjectTagEXT,
    'vkDebugMarkerSetObjectNameEXT':_wrap_vkDebugMarkerSetObjectNameEXT,
    'vkCmdDebugMarkerBeginEXT':_wrap_vkCmdDebugMarkerBeginEXT,
    'vkCmdDebugMarkerEndEXT':_wrap_vkCmdDebugMarkerEndEXT,
    'vkCmdDebugMarkerInsertEXT':_wrap_vkCmdDebugMarkerInsertEXT,
    'vkCmdBindTransformFeedbackBuffersEXT':_wrap_vkCmdBindTransformFeedbackBuffersEXT,
    'vkCmdBeginTransformFeedbackEXT':_wrap_vkCmdBeginTransformFeedbackEXT,
    'vkCmdEndTransformFeedbackEXT':_wrap_vkCmdEndTransformFeedbackEXT,
    'vkCmdBeginQueryIndexedEXT':_wrap_vkCmdBeginQueryIndexedEXT,
    'vkCmdEndQueryIndexedEXT':_wrap_vkCmdEndQueryIndexedEXT,
    'vkCmdDrawIndirectByteCountEXT':_wrap_vkCmdDrawIndirectByteCountEXT,
    'vkCmdDrawIndirectCountAMD':_wrap_vkCmdDrawIndirectCountAMD,
    'vkCmdDrawIndexedIndirectCountAMD':_wrap_vkCmdDrawIndexedIndirectCountAMD,
    'vkGetShaderInfoAMD':_wrap_vkGetShaderInfoAMD,
    'vkGetMemoryWin32HandleNV':_wrap_vkGetMemoryWin32HandleNV,
    'vkGetDeviceGroupPeerMemoryFeaturesKHR':_wrap_vkGetDeviceGroupPeerMemoryFeatures,
    'vkCmdSetDeviceMaskKHR':_wrap_vkCmdSetDeviceMask,
    'vkCmdDispatchBaseKHR':_wrap_vkCmdDispatchBase,
    'vkTrimCommandPoolKHR':_wrap_vkTrimCommandPool,
    'vkGetMemoryWin32HandleKHR':_wrap_vkGetMemoryWin32HandleKHR,
    'vkGetMemoryWin32HandlePropertiesKHR':_wrap_vkGetMemoryWin32HandlePropertiesKHR,
    'vkGetMemoryFdKHR':_wrap_vkGetMemoryFdKHR,
    'vkGetMemoryFdPropertiesKHR':_wrap_vkGetMemoryFdPropertiesKHR,
    'vkImportSemaphoreWin32HandleKHR':_wrap_vkImportSemaphoreWin32HandleKHR,
    'vkGetSemaphoreWin32HandleKHR':_wrap_vkGetSemaphoreWin32HandleKHR,
    'vkImportSemaphoreFdKHR':_wrap_vkImportSemaphoreFdKHR,
    'vkGetSemaphoreFdKHR':_wrap_vkGetSemaphoreFdKHR,
    'vkCmdPushDescriptorSetKHR':_wrap_vkCmdPushDescriptorSetKHR,
    'vkCmdPushDescriptorSetWithTemplateKHR':_wrap_vkCmdPushDescriptorSetWithTemplateKHR,
    'vkCmdBeginConditionalRenderingEXT':_wrap_vkCmdBeginConditionalRenderingEXT,
    'vkCmdEndConditionalRenderingEXT':_wrap_vkCmdEndConditionalRenderingEXT,
    'vkCreateDescriptorUpdateTemplateKHR':_wrap_vkCreateDescriptorUpdateTemplate,
    'vkDestroyDescriptorUpdateTemplateKHR':_wrap_vkDestroyDescriptorUpdateTemplate,
    'vkUpdateDescriptorSetWithTemplateKHR':_wrap_vkUpdateDescriptorSetWithTemplate,
    'vkCmdProcessCommandsNVX':_wrap_vkCmdProcessCommandsNVX,
    'vkCmdReserveSpaceForCommandsNVX':_wrap_vkCmdReserveSpaceForCommandsNVX,
    'vkCreateIndirectCommandsLayoutNVX':_wrap_vkCreateIndirectCommandsLayoutNVX,
    'vkDestroyIndirectCommandsLayoutNVX':_wrap_vkDestroyIndirectCommandsLayoutNVX,
    'vkCreateObjectTableNVX':_wrap_vkCreateObjectTableNVX,
    'vkDestroyObjectTableNVX':_wrap_vkDestroyObjectTableNVX,
    'vkRegisterObjectsNVX':_wrap_vkRegisterObjectsNVX,
    'vkUnregisterObjectsNVX':_wrap_vkUnregisterObjectsNVX,
    'vkGetPhysicalDeviceGeneratedCommandsPropertiesNVX':_wrap_vkGetPhysicalDeviceGeneratedCommandsPropertiesNVX,
    'vkCmdSetViewportWScalingNV':_wrap_vkCmdSetViewportWScalingNV,
    'vkDisplayPowerControlEXT':_wrap_vkDisplayPowerControlEXT,
    'vkRegisterDeviceEventEXT':_wrap_vkRegisterDeviceEventEXT,
    'vkRegisterDisplayEventEXT':_wrap_vkRegisterDisplayEventEXT,
    'vkGetSwapchainCounterEXT':_wrap_vkGetSwapchainCounterEXT,
    'vkGetRefreshCycleDurationGOOGLE':_wrap_vkGetRefreshCycleDurationGOOGLE,
    'vkGetPastPresentationTimingGOOGLE':_wrap_vkGetPastPresentationTimingGOOGLE,
    'vkCmdSetDiscardRectangleEXT':_wrap_vkCmdSetDiscardRectangleEXT,
    'vkSetHdrMetadataEXT':_wrap_vkSetHdrMetadataEXT,
    'vkCreateRenderPass2KHR':_wrap_vkCreateRenderPass2KHR,
    'vkCmdBeginRenderPass2KHR':_wrap_vkCmdBeginRenderPass2KHR,
    'vkCmdNextSubpass2KHR':_wrap_vkCmdNextSubpass2KHR,
    'vkCmdEndRenderPass2KHR':_wrap_vkCmdEndRenderPass2KHR,
    'vkGetSwapchainStatusKHR':_wrap_vkGetSwapchainStatusKHR,
    'vkImportFenceWin32HandleKHR':_wrap_vkImportFenceWin32HandleKHR,
    'vkGetFenceWin32HandleKHR':_wrap_vkGetFenceWin32HandleKHR,
    'vkImportFenceFdKHR':_wrap_vkImportFenceFdKHR,
    'vkGetFenceFdKHR':_wrap_vkGetFenceFdKHR,
    'vkGetAndroidHardwareBufferPropertiesANDROID':_wrap_vkGetAndroidHardwareBufferPropertiesANDROID,
    'vkGetMemoryAndroidHardwareBufferANDROID':_wrap_vkGetMemoryAndroidHardwareBufferANDROID,
    'vkCmdSetSampleLocationsEXT':_wrap_vkCmdSetSampleLocationsEXT,
    'vkGetPhysicalDeviceMultisamplePropertiesEXT':_wrap_vkGetPhysicalDeviceMultisamplePropertiesEXT,
    'vkGetImageMemoryRequirements2KHR':_wrap_vkGetImageMemoryRequirements2,
    'vkGetBufferMemoryRequirements2KHR':_wrap_vkGetBufferMemoryRequirements2,
    'vkGetImageSparseMemoryRequirements2KHR':_wrap_vkGetImageSparseMemoryRequirements2,
    'vkCreateSamplerYcbcrConversionKHR':_wrap_vkCreateSamplerYcbcrConversion,
    'vkDestroySamplerYcbcrConversionKHR':_wrap_vkDestroySamplerYcbcrConversion,
    'vkBindBufferMemory2KHR':_wrap_vkBindBufferMemory2,
    'vkBindImageMemory2KHR':_wrap_vkBindImageMemory2,
    'vkGetImageDrmFormatModifierPropertiesEXT':_wrap_vkGetImageDrmFormatModifierPropertiesEXT,
    'vkCreateValidationCacheEXT':_wrap_vkCreateValidationCacheEXT,
    'vkDestroyValidationCacheEXT':_wrap_vkDestroyValidationCacheEXT,
    'vkMergeValidationCachesEXT':_wrap_vkMergeValidationCachesEXT,
    'vkGetValidationCacheDataEXT':_wrap_vkGetValidationCacheDataEXT,
    'vkCmdBindShadingRateImageNV':_wrap_vkCmdBindShadingRateImageNV,
    'vkCmdSetViewportShadingRatePaletteNV':_wrap_vkCmdSetViewportShadingRatePaletteNV,
    'vkCmdSetCoarseSampleOrderNV':_wrap_vkCmdSetCoarseSampleOrderNV,
    'vkCreateAccelerationStructureNV':_wrap_vkCreateAccelerationStructureNV,
    'vkDestroyAccelerationStructureNV':_wrap_vkDestroyAccelerationStructureNV,
    'vkGetAccelerationStructureMemoryRequirementsNV':_wrap_vkGetAccelerationStructureMemoryRequirementsNV,
    'vkBindAccelerationStructureMemoryNV':_wrap_vkBindAccelerationStructureMemoryNV,
    'vkCmdBuildAccelerationStructureNV':_wrap_vkCmdBuildAccelerationStructureNV,
    'vkCmdCopyAccelerationStructureNV':_wrap_vkCmdCopyAccelerationStructureNV,
    'vkCmdTraceRaysNV':_wrap_vkCmdTraceRaysNV,
    'vkCreateRayTracingPipelinesNV':_wrap_vkCreateRayTracingPipelinesNV,
    'vkGetRayTracingShaderGroupHandlesNV':_wrap_vkGetRayTracingShaderGroupHandlesNV,
    'vkGetAccelerationStructureHandleNV':_wrap_vkGetAccelerationStructureHandleNV,
    'vkCmdWriteAccelerationStructuresPropertiesNV':_wrap_vkCmdWriteAccelerationStructuresPropertiesNV,
    'vkCompileDeferredNV':_wrap_vkCompileDeferredNV,
    'vkGetDescriptorSetLayoutSupportKHR':_wrap_vkGetDescriptorSetLayoutSupport,
    'vkCmdDrawIndirectCountKHR':_wrap_vkCmdDrawIndirectCountKHR,
    'vkCmdDrawIndexedIndirectCountKHR':_wrap_vkCmdDrawIndexedIndirectCountKHR,
    'vkGetMemoryHostPointerPropertiesEXT':_wrap_vkGetMemoryHostPointerPropertiesEXT,
    'vkCmdWriteBufferMarkerAMD':_wrap_vkCmdWriteBufferMarkerAMD,
    'vkGetPhysicalDeviceCalibrateableTimeDomainsEXT':_wrap_vkGetPhysicalDeviceCalibrateableTimeDomainsEXT,
    'vkGetCalibratedTimestampsEXT':_wrap_vkGetCalibratedTimestampsEXT,
    'vkCmdDrawMeshTasksNV':_wrap_vkCmdDrawMeshTasksNV,
    'vkCmdDrawMeshTasksIndirectNV':_wrap_vkCmdDrawMeshTasksIndirectNV,
    'vkCmdDrawMeshTasksIndirectCountNV':_wrap_vkCmdDrawMeshTasksIndirectCountNV,
    'vkCmdSetExclusiveScissorNV':_wrap_vkCmdSetExclusiveScissorNV,
    'vkCmdSetCheckpointNV':_wrap_vkCmdSetCheckpointNV,
    'vkGetQueueCheckpointDataNV':_wrap_vkGetQueueCheckpointDataNV,
    'vkGetBufferDeviceAddressEXT':_wrap_vkGetBufferDeviceAddressEXT,
}


def vkGetInstanceProcAddr(instance, pName):
    fn = _callApi(lib.vkGetInstanceProcAddr, instance, pName)
    if fn == ffi.NULL:
        raise ProcedureNotFoundError()
    if not pName in _instance_ext_funcs:
        raise ExtensionNotSupportedError()
    fn = ffi.cast('PFN_' + pName, fn)
    return _instance_ext_funcs[pName](fn)


def vkGetDeviceProcAddr(device, pName):
    fn = _callApi(lib.vkGetDeviceProcAddr, device, pName)
    if fn == ffi.NULL:
        raise ProcedureNotFoundError()
    if not pName in _device_ext_funcs:
        raise ExtensionNotSupportedError()
    fn = ffi.cast('PFN_'+pName, fn)
    return _device_ext_funcs[pName](fn)


def vkMapMemory(device, memory, offset, size, flags):
    ppData = ffi.new('void**')

    result = _callApi(lib.vkMapMemory, device,memory,offset,size,flags,ppData)
    if result != VK_SUCCESS:
        raise exception_codes[result]

    return ffi.buffer(ppData[0], size)


#VkDescriptorUpdateTemplateKHR = VkDescriptorUpdateTemplate
#VkSamplerYcbcrConversionKHR = VkSamplerYcbcrConversion
#vkGetPhysicalDeviceFeatures2KHR = vkGetPhysicalDeviceFeatures2
#vkGetPhysicalDeviceProperties2KHR = vkGetPhysicalDeviceProperties2
#vkGetPhysicalDeviceFormatProperties2KHR = vkGetPhysicalDeviceFormatProperties2
#vkGetPhysicalDeviceImageFormatProperties2KHR = vkGetPhysicalDeviceImageFormatProperties2
#vkGetPhysicalDeviceQueueFamilyProperties2KHR = vkGetPhysicalDeviceQueueFamilyProperties2
#vkGetPhysicalDeviceMemoryProperties2KHR = vkGetPhysicalDeviceMemoryProperties2
#vkGetPhysicalDeviceSparseImageFormatProperties2KHR = vkGetPhysicalDeviceSparseImageFormatProperties2
#vkTrimCommandPoolKHR = vkTrimCommandPool
#vkGetPhysicalDeviceExternalBufferPropertiesKHR = vkGetPhysicalDeviceExternalBufferProperties
#vkGetPhysicalDeviceExternalSemaphorePropertiesKHR = vkGetPhysicalDeviceExternalSemaphoreProperties
#vkGetPhysicalDeviceExternalFencePropertiesKHR = vkGetPhysicalDeviceExternalFenceProperties
#vkEnumeratePhysicalDeviceGroupsKHR = vkEnumeratePhysicalDeviceGroups
#vkGetDeviceGroupPeerMemoryFeaturesKHR = vkGetDeviceGroupPeerMemoryFeatures
#vkBindBufferMemory2KHR = vkBindBufferMemory2
#vkBindImageMemory2KHR = vkBindImageMemory2
#vkCmdSetDeviceMaskKHR = vkCmdSetDeviceMask
#vkCmdDispatchBaseKHR = vkCmdDispatchBase
#vkCreateDescriptorUpdateTemplateKHR = vkCreateDescriptorUpdateTemplate
#vkDestroyDescriptorUpdateTemplateKHR = vkDestroyDescriptorUpdateTemplate
#vkUpdateDescriptorSetWithTemplateKHR = vkUpdateDescriptorSetWithTemplate
#vkGetBufferMemoryRequirements2KHR = vkGetBufferMemoryRequirements2
#vkGetImageMemoryRequirements2KHR = vkGetImageMemoryRequirements2
#vkGetImageSparseMemoryRequirements2KHR = vkGetImageSparseMemoryRequirements2
#vkCreateSamplerYcbcrConversionKHR = vkCreateSamplerYcbcrConversion
#vkDestroySamplerYcbcrConversionKHR = vkDestroySamplerYcbcrConversion
#vkGetDescriptorSetLayoutSupportKHR = vkGetDescriptorSetLayoutSupport
