# AUTOGENERATED! DO NOT EDIT! File to edit: nbdev_nbs/peptide/precursor.ipynb (unless otherwise specified).

__all__ = ['refine_precursor_df', 'is_precursor_refined', 'update_precursor_mz', 'reset_precursor_df',
           'is_precursor_sorted', 'calc_precursor_mz', 'get_mod_seq_hash', 'get_mod_seq_charge_hash', 'hash_mod_seq_df',
           'hash_mod_seq_charge_df', 'hash_precursor_df']

# Cell
import pandas as pd
import numpy as np

from alphabase.constants.element import (
    MASS_PROTON
)
from alphabase.peptide.mass_calc import (
    calc_peptide_masses_for_same_len_seqs
)

def refine_precursor_df(
    df:pd.DataFrame,
    drop_frag_idx = True,
    ensure_data_validity = False,
)->pd.DataFrame:
    """
    Refine df inplace for faster precursor/fragment calculation.
    """
    if ensure_data_validity:
        df.fillna('', inplace=True)
        if 'charge' in df.columns:
            if df.charge.dtype not in [
                'int','int8','int64','int32',
                # np.int64, np.int32, np.int8,
            ]:
                df['charge'] = df['charge'].astype(np.int8)
        if 'mod_sites' in df.columns:
            if df.mod_sites.dtype not in ['O','U']:
                df['mod_sites'] = df.mod_sites.astype('U')

    if 'nAA' not in df.columns:
        df['nAA']= df.sequence.str.len().astype(np.int32)
    df.sort_values('nAA', inplace=True)
    df.reset_index(drop=True, inplace=True)

    if drop_frag_idx and 'frag_start_idx' in df.columns:
        df.drop(columns=[
            'frag_start_idx','frag_end_idx'
        ], inplace=True)

    return df

reset_precursor_df = refine_precursor_df

def is_precursor_refined(precursor_df: pd.DataFrame):
    return (
        (len(precursor_df) == 0) or (
            (precursor_df.index.values[0] == 0) and
            precursor_df.nAA.is_monotonic and
            np.all(
                np.diff(precursor_df.index.values)==1
            )
        )
    )

is_precursor_sorted = is_precursor_refined

def update_precursor_mz(
    precursor_df: pd.DataFrame,
    batch_size = 500000,
)->pd.DataFrame:
    """
    Calculate precursor_mz for the precursor_df
    Args:
        precursor_df (pd.DataFrame):
          precursor_df with the 'charge' column

    Returns:
        pd.DataFrame: precursor_df with 'precursor_mz'
    """

    if 'nAA' not in precursor_df:
        reset_precursor_df(precursor_df)
        _calc_in_order = True
    elif is_precursor_sorted(precursor_df):
        _calc_in_order = True
    else:
        _calc_in_order = False
    precursor_df['precursor_mz'] = 0.
    _grouped = precursor_df.groupby('nAA')
    for nAA, big_df_group in _grouped:
        for i in range(0, len(big_df_group), batch_size):
            batch_end = i+batch_size

            df_group = big_df_group.iloc[i:batch_end,:]

            pep_mzs = calc_peptide_masses_for_same_len_seqs(
                df_group.sequence.values.astype('U'),
                df_group.mods.values,
                df_group.mod_deltas.values if
                'mod_deltas' in df_group.columns else None
            )/df_group.charge + MASS_PROTON
            if _calc_in_order:
                precursor_df.loc[:,'precursor_mz'].values[
                    df_group.index.values[0]:
                    df_group.index.values[-1]+1
                ] = pep_mzs
            else:
                precursor_df.loc[
                    df_group.index, 'precursor_mz'
                ] = pep_mzs
    return precursor_df

calc_precursor_mz = update_precursor_mz

# Cell
from mmh3 import hash64

def get_mod_seq_hash(
    sequence:str, mods:str,
    mod_sites:str,
    *, seed:int=0
)->np.int64:
    """Get hash code value for a peptide:
      (sequence, mods, mod_sites)

    Args:
        sequence (str): amino acid sequence
        mods (str): modification names in AlphaBase format
        mod_sites (str): modification sites in AlphaBase format
        seed (int, optional): seed for hashing
          Defaults to 0.

    Returns:
        np.int64: 64-bit hash code value
    """
    return np.sum([
        hash64(sequence, seed=seed)[0],
        hash64(mods, seed=seed)[0],
        hash64(mod_sites, seed=seed)[0],
    ],dtype=np.int64) # use np.sum to prevent overflow

def get_mod_seq_charge_hash(
    sequence:str, mods:str,
    mod_sites:str, charge:int,
    *, seed=0
):
    """Get hash code value for a precursor:
      (sequence, mods, mod_sites, charge)

    Args:
        sequence (str): amino acid sequence
        mods (str): modification names in AlphaBase format
        mod_sites (str): modification sites in AlphaBase format
        charge (int): precursor charge state
        seed (int, optional): seed for hashing
          Defaults to 0.

    Returns:
        np.int64: 64-bit hash code value
    """
    return np.sum([
        get_mod_seq_hash(
            sequence, mods, mod_sites,
            seed=seed
        ),
        charge,
    ],dtype=np.int64) # use np.sum to prevent overflow

def hash_mod_seq_df(
    precursor_df:pd.DataFrame,
    *, seed=0
):
    """ Internal function """
    hash_vals = precursor_df.sequence.apply(
        lambda x: hash64(x, seed=seed)[0]
    ).astype(np.int64).values
    hash_vals += precursor_df.mods.apply(
        lambda x: hash64(x, seed=seed)[0]
    ).values
    hash_vals += precursor_df.mod_sites.apply(
        lambda x: hash64(x, seed=seed)[0]
    ).values

    precursor_df[
        "mod_seq_hash"
    ] = hash_vals

def hash_mod_seq_charge_df(
    precursor_df:pd.DataFrame,
    *, seed=0
):
    """ Internal function """
    if "mod_seq_hash" not in precursor_df.columns:
        hash_mod_seq_df(precursor_df, seed=seed)
    if "charge" not in precursor_df.columns:
        raise ValueError(
            "DataFrame must contain 'charge' column"
        )

    precursor_df["mod_seq_charge_hash"] = (
        precursor_df["mod_seq_hash"].values
        + precursor_df["charge"].values
    )

def hash_precursor_df(
    precursor_df:pd.DataFrame,
    *, seed:int=0
)->pd.DataFrame:
    """Add columns 'mod_seq_hash' and 'mod_seq_charge_hash'
    into precursor_df (inplace).
    The 64-bit hash function is from mmh3 (mmh3.hash64).

    Args:
        precursor_df (pd.DataFrame): precursor_df
        seed (int, optional): seed for mmh3.hash64.
          Defaults to 0
    """
    hash_mod_seq_df(precursor_df, seed=seed)

    if 'charge' in precursor_df.columns:
        hash_mod_seq_charge_df(precursor_df, seed=seed)
    return precursor_df