"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DestinationPattern = exports.DestinationIdentifier = exports.DockerImageManifestEntry = exports.FileManifestEntry = exports.AssetManifestReader = void 0;
// FIXME: copied from `ckd-assets`, because this tool needs to read the asset manifest aswell.
const fs = require("fs");
const path = require("path");
const cloud_assembly_schema_1 = require("../../../cloud-assembly-schema");
/**
 * A manifest of assets
 */
class AssetManifestReader {
    constructor(directory, manifest) {
        this.manifest = manifest;
        this.directory = directory;
    }
    /**
     * Load an asset manifest from the given file
     */
    static fromFile(fileName) {
        try {
            const obj = cloud_assembly_schema_1.Manifest.loadAssetManifest(fileName);
            return new AssetManifestReader(path.dirname(fileName), obj);
        }
        catch (e) {
            throw new Error(`Cannot read asset manifest '${fileName}': ${e.message}`);
        }
    }
    /**
     * Load an asset manifest from the given file or directory
     *
     * If the argument given is a directoy, the default asset file name will be used.
     */
    static fromPath(filePath) {
        let st;
        try {
            st = fs.statSync(filePath);
        }
        catch (e) {
            throw new Error(`Cannot read asset manifest at '${filePath}': ${e.message}`);
        }
        if (st.isDirectory()) {
            return AssetManifestReader.fromFile(path.join(filePath, AssetManifestReader.DEFAULT_FILENAME));
        }
        return AssetManifestReader.fromFile(filePath);
    }
    /**
     * Select a subset of assets and destinations from this manifest.
     *
     * Only assets with at least 1 selected destination are retained.
     *
     * If selection is not given, everything is returned.
     */
    select(selection) {
        if (selection === undefined) {
            return this;
        }
        const ret = { version: this.manifest.version, dockerImages: {}, files: {} };
        for (const assetType of ASSET_TYPES) {
            for (const [assetId, asset] of Object.entries(this.manifest[assetType] || {})) {
                const filteredDestinations = filterDict(asset.destinations, (_, destId) => selection.some(sel => sel.matches(new DestinationIdentifier(assetId, destId))));
                if (Object.keys(filteredDestinations).length > 0) {
                    ret[assetType][assetId] = {
                        ...asset,
                        destinations: filteredDestinations,
                    };
                }
            }
        }
        return new AssetManifestReader(this.directory, ret);
    }
    /**
     * Describe the asset manifest as a list of strings
     */
    list() {
        return [
            ...describeAssets('file', this.manifest.files || {}),
            ...describeAssets('docker-image', this.manifest.dockerImages || {}),
        ];
        function describeAssets(type, assets) {
            const ret = new Array();
            for (const [assetId, asset] of Object.entries(assets || {})) {
                ret.push(`${assetId} ${type} ${JSON.stringify(asset.source)}`);
                const destStrings = Object.entries(asset.destinations).map(([destId, dest]) => ` ${assetId}:${destId} ${JSON.stringify(dest)}`);
                ret.push(...prefixTreeChars(destStrings, '  '));
            }
            return ret;
        }
    }
    /**
     * List of assets, splat out to destinations
     */
    get entries() {
        return [
            ...makeEntries(this.manifest.files || {}, FileManifestEntry),
            ...makeEntries(this.manifest.dockerImages || {}, DockerImageManifestEntry),
        ];
        function makeEntries(assets, ctor) {
            const ret = new Array();
            for (const [assetId, asset] of Object.entries(assets)) {
                for (const [destId, destination] of Object.entries(asset.destinations)) {
                    ret.push(new ctor(new DestinationIdentifier(assetId, destId), asset.source, destination));
                }
            }
            return ret;
        }
    }
}
exports.AssetManifestReader = AssetManifestReader;
/**
 * The default name of the asset manifest in a cdk.out directory
 */
AssetManifestReader.DEFAULT_FILENAME = 'assets.json';
const ASSET_TYPES = ['files', 'dockerImages'];
/**
 * A manifest entry for a file asset
 */
class FileManifestEntry {
    constructor(
    /** Identifier for this asset */
    id, 
    /** Source of the file asset */
    source, 
    /** Destination for the file asset */
    destination) {
        this.id = id;
        this.source = source;
        this.destination = destination;
        this.type = 'file';
        this.genericSource = source;
    }
}
exports.FileManifestEntry = FileManifestEntry;
/**
 * A manifest entry for a docker image asset
 */
class DockerImageManifestEntry {
    constructor(
    /** Identifier for this asset */
    id, 
    /** Source of the file asset */
    source, 
    /** Destination for the file asset */
    destination) {
        this.id = id;
        this.source = source;
        this.destination = destination;
        this.type = 'docker-image';
        this.genericSource = source;
    }
}
exports.DockerImageManifestEntry = DockerImageManifestEntry;
/**
 * Identify an asset destination in an asset manifest
 */
class DestinationIdentifier {
    constructor(assetId, destinationId) {
        this.assetId = assetId;
        this.destinationId = destinationId;
    }
    /**
     * Return a string representation for this asset identifier
     */
    toString() {
        return this.destinationId ? `${this.assetId}:${this.destinationId}` : this.assetId;
    }
}
exports.DestinationIdentifier = DestinationIdentifier;
function filterDict(xs, pred) {
    const ret = {};
    for (const [key, value] of Object.entries(xs)) {
        if (pred(value, key)) {
            ret[key] = value;
        }
    }
    return ret;
}
/**
 * A filter pattern for an destination identifier
 */
class DestinationPattern {
    constructor(assetId, destinationId) {
        this.assetId = assetId;
        this.destinationId = destinationId;
    }
    /**
     * Parse a ':'-separated string into an asset/destination identifier
     */
    static parse(s) {
        if (!s) {
            throw new Error('Empty string is not a valid destination identifier');
        }
        const parts = s.split(':').map(x => x !== '*' ? x : undefined);
        if (parts.length === 1) {
            return new DestinationPattern(parts[0]);
        }
        if (parts.length === 2) {
            return new DestinationPattern(parts[0] || undefined, parts[1] || undefined);
        }
        throw new Error(`Asset identifier must contain at most 2 ':'-separated parts, got '${s}'`);
    }
    /**
     * Whether or not this pattern matches the given identifier
     */
    matches(id) {
        return (this.assetId === undefined || this.assetId === id.assetId)
            && (this.destinationId === undefined || this.destinationId === id.destinationId);
    }
    /**
     * Return a string representation for this asset identifier
     */
    toString() {
        return `${this.assetId ?? '*'}:${this.destinationId ?? '*'}`;
    }
}
exports.DestinationPattern = DestinationPattern;
/**
 * Prefix box-drawing characters to make lines look like a hanging tree
 */
function prefixTreeChars(xs, prefix = '') {
    const ret = new Array();
    for (let i = 0; i < xs.length; i++) {
        const isLast = i === xs.length - 1;
        const boxChar = isLast ? '└' : '├';
        ret.push(`${prefix}${boxChar}${xs[i]}`);
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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