"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationSecurityCheck = void 0;
const path = require("path");
const codebuild = require("../../../aws-codebuild");
const iam = require("../../../aws-iam");
const lambda = require("../../../aws-lambda");
const core_1 = require("../../../core");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("../../../core");
/**
 * A construct containing both the Lambda and CodeBuild Project
 * needed to conduct a security check on any given application stage.
 *
 * The Lambda acts as an auto approving mechanism that should only be
 * triggered when the CodeBuild Project registers no security changes.
 *
 * The CodeBuild Project runs a security diff on the application stage,
 * and exports the link to the console of the project.
 */
class ApplicationSecurityCheck extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        core_1.Tags.of(props.codePipeline).add('SECURITY_CHECK', 'ALLOW_APPROVE', {
            includeResourceTypes: ['AWS::CodePipeline::Pipeline'],
        });
        this.preApproveLambda = new lambda.Function(this, 'CDKPipelinesAutoApprove', {
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_14_X,
            code: lambda.Code.fromAsset(path.resolve(__dirname, 'approve-lambda')),
            timeout: core_1.Duration.minutes(5),
        });
        this.preApproveLambda.addToRolePolicy(new iam.PolicyStatement({
            actions: ['codepipeline:GetPipelineState', 'codepipeline:PutApprovalResult'],
            conditions: {
                StringEquals: {
                    'aws:ResourceTag/SECURITY_CHECK': 'ALLOW_APPROVE',
                },
            },
            resources: ['*'],
        }));
        const invokeLambda = 'aws lambda invoke' +
            ` --function-name ${this.preApproveLambda.functionName}` +
            ' --invocation-type Event' +
            ' --payload "$payload"' +
            ' lambda.out';
        const message = [
            'An upcoming change would broaden security changes in $PIPELINE_NAME.',
            'Review and approve the changes in CodePipeline to proceed with the deployment.',
            '',
            'Review the changes in CodeBuild:',
            '',
            '$LINK',
            '',
            'Approve the changes in CodePipeline (stage $STAGE_NAME, action $ACTION_NAME):',
            '',
            '$PIPELINE_LINK',
        ];
        const publishNotification = 'aws sns publish' +
            ' --topic-arn $NOTIFICATION_ARN' +
            ' --subject "$NOTIFICATION_SUBJECT"' +
            ` --message "${message.join('\n')}"`;
        this.cdkDiffProject = new codebuild.Project(this, 'CDKSecurityCheck', {
            buildSpec: codebuild.BuildSpec.fromObject({
                version: 0.2,
                phases: {
                    build: {
                        commands: [
                            'npm install -g aws-cdk',
                            // $CODEBUILD_INITIATOR will always be Code Pipeline and in the form of:
                            // "codepipeline/example-pipeline-name-Xxx"
                            'export PIPELINE_NAME="$(node -pe \'`${process.env.CODEBUILD_INITIATOR}`.split("/")[1]\')"',
                            'payload="$(node -pe \'JSON.stringify({ "PipelineName": process.env.PIPELINE_NAME, "StageName": process.env.STAGE_NAME, "ActionName": process.env.ACTION_NAME })\' )"',
                            // ARN: "arn:aws:codebuild:$region:$account_id:build/$project_name:$project_execution_id$"
                            'ARN=$CODEBUILD_BUILD_ARN',
                            'REGION="$(node -pe \'`${process.env.ARN}`.split(":")[3]\')"',
                            'ACCOUNT_ID="$(node -pe \'`${process.env.ARN}`.split(":")[4]\')"',
                            'PROJECT_NAME="$(node -pe \'`${process.env.ARN}`.split(":")[5].split("/")[1]\')"',
                            'PROJECT_ID="$(node -pe \'`${process.env.ARN}`.split(":")[6]\')"',
                            // Manual Approval adds 'http/https' to the resolved link
                            'export LINK="https://$REGION.console.aws.amazon.com/codesuite/codebuild/$ACCOUNT_ID/projects/$PROJECT_NAME/build/$PROJECT_NAME:$PROJECT_ID/?region=$REGION"',
                            'export PIPELINE_LINK="https://$REGION.console.aws.amazon.com/codesuite/codepipeline/pipelines/$PIPELINE_NAME/view?region=$REGION"',
                            // Run invoke only if cdk diff passes (returns exit code 0)
                            // 0 -> true, 1 -> false
                            ifElse({
                                condition: 'cdk diff -a . --security-only --fail $STAGE_PATH/\\*',
                                thenStatements: [
                                    invokeLambda,
                                    'export MESSAGE="No security-impacting changes detected."',
                                ],
                                elseStatements: [
                                    `[ -z "\${NOTIFICATION_ARN}" ] || ${publishNotification}`,
                                    'export MESSAGE="Deployment would make security-impacting changes. Click the link below to inspect them, then click Approve if all changes are expected."',
                                ],
                            }),
                        ],
                    },
                },
                env: {
                    'exported-variables': [
                        'LINK',
                        'MESSAGE',
                    ],
                },
            }),
        });
        // this is needed to check the status the stacks when doing `cdk diff`
        this.cdkDiffProject.addToRolePolicy(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: ['*'],
            conditions: {
                'ForAnyValue:StringEquals': {
                    'iam:ResourceTag/aws-cdk:bootstrap-role': ['deploy'],
                },
            },
        }));
        this.preApproveLambda.grantInvoke(this.cdkDiffProject);
    }
}
exports.ApplicationSecurityCheck = ApplicationSecurityCheck;
const ifElse = ({ condition, thenStatements, elseStatements }) => {
    let statement = thenStatements.reduce((acc, ifTrue) => {
        return `${acc} ${ifTrue};`;
    }, `if ${condition}; then`);
    if (elseStatements) {
        statement = elseStatements.reduce((acc, ifFalse) => {
            return `${acc} ${ifFalse};`;
        }, `${statement} else`);
    }
    return `${statement} fi`;
};
//# sourceMappingURL=data:application/json;base64,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