"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StepOutput = void 0;
const core_1 = require("../../../core");
const STEP_OUTPUT_SYM = Symbol.for('@aws-cdk/pipelines.StepOutput');
const PRODUCED_OUTPUTS_SYM = Symbol.for('@aws-cdk/pipelines.outputs');
/**
 * A symbolic reference to a value produced by another step
 *
 * Generating and consuming outputs is engine-specific. Many engines will be
 * able to support a feature like "outputs", but it's not guaranteed that
 * all of them will.
 *
 * Outputs can only be generated by engine-specific steps (CodeBuildStep instead
 * of ShellStep, etc), but can (currently) be consumed anywhere(*). When
 * an engine-specific step generates an Output, it should put a well-known
 * string and arbitrary data that is useful to the engine into the engine-specific
 * fields on the StepOutput.
 *
 * The graph blueprint will take care of dependencies and ordering, the engine
 * is responsible interpreting and rendering StepOutputs. The engine should call
 * `defineResolution()` on all outputs.
 *
 * StepOutputs currently purposely aren't part of the public API because users
 * shouldn't see the innards poking out. So, instead of keeping state on `Step`,
 * we keep side-state here in a WeakMap which can be accessed via static members
 * on `StepOutput`.
 *
 * (*) If we need to restrict this, we add the checking and erroring in the engine.
 */
class StepOutput {
    constructor(step, engineName, engineSpecificInformation) {
        this.creationStack = [];
        this.resolution = undefined;
        this.step = step;
        this.engineName = engineName;
        this.engineSpecificInformation = engineSpecificInformation;
        Object.defineProperty(this, STEP_OUTPUT_SYM, { value: true });
    }
    /**
     * Return true if the given IResolvable is a StepOutput
     */
    static isStepOutput(resolvable) {
        return !!resolvable[STEP_OUTPUT_SYM];
    }
    /**
     * Find all StepOutputs referenced in the given structure
     */
    static findAll(structure) {
        return findAllStepOutputs(structure);
    }
    /**
     * Return the produced outputs for the given step
     */
    static producedStepOutputs(step) {
        return step[PRODUCED_OUTPUTS_SYM] ?? [];
    }
    /**
     * Add produced outputs for the given step
     */
    static recordProducer(...outputs) {
        for (const output of outputs) {
            const step = output.step;
            let list = step[PRODUCED_OUTPUTS_SYM];
            if (!list) {
                list = [];
                step[PRODUCED_OUTPUTS_SYM] = list;
            }
            list.push(...outputs);
        }
    }
    /**
     * Define the resolved value for this StepOutput.
     *
     * Should be called by the engine.
     */
    defineResolution(value) {
        this.resolution = value;
    }
    resolve(_context) {
        if (this.resolution === undefined) {
            throw new Error(`Output for step ${this.step} not configured. Either the step is not in the pipeline, the step implementation did not call 'this.discoverReferencedOutputs()', or this engine does not support Outputs for this step.`);
        }
        return this.resolution;
    }
    toString() {
        return core_1.Token.asString(this);
    }
}
exports.StepOutput = StepOutput;
function findAllStepOutputs(structure) {
    const ret = new Set();
    recurse(structure);
    return Array.from(ret);
    function checkToken(x) {
        if (x && StepOutput.isStepOutput(x)) {
            ret.add(x);
            return true;
        }
        // Return false if it wasn't a Token in the first place (in which case we recurse)
        return x !== undefined;
    }
    function recurse(x) {
        if (!x) {
            return;
        }
        if (core_1.Tokenization.isResolvable(x)) {
            checkToken(x);
            return;
        }
        if (Array.isArray(x)) {
            if (!checkToken(core_1.Tokenization.reverseList(x))) {
                x.forEach(recurse);
            }
            return;
        }
        if (typeof x === 'number') {
            checkToken(core_1.Tokenization.reverseNumber(x));
            return;
        }
        if (typeof x === 'string') {
            core_1.Tokenization.reverseString(x).tokens.forEach(checkToken);
            return;
        }
        if (typeof x === 'object') {
            for (const [k, v] of Object.entries(x)) {
                recurse(k);
                recurse(v);
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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