"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isGraph = exports.GraphNodeCollection = exports.Graph = exports.DependencyBuilders = exports.DependencyBuilder = exports.GraphNode = void 0;
/**
 * A library for nested graphs
 */
const javascript_1 = require("../private/javascript");
const toposort_1 = require("./toposort");
class GraphNode {
    constructor(id, props = {}) {
        this.id = id;
        this.dependencies = [];
        this.data = props.data;
    }
    static of(id, data) {
        return new GraphNode(id, { data });
    }
    /**
     * A graph-wide unique identifier for this node. Rendered by joining the IDs
     * of all ancestors with hyphens.
     */
    get uniqueId() {
        return this.ancestorPath(this.root).map(x => x.id).join('-');
    }
    /**
     * The union of all dependencies of this node and the dependencies of all
     * parent graphs.
     */
    get allDeps() {
        const fromParent = this.parentGraph?.allDeps ?? [];
        return Array.from(new Set([...this.dependencies, ...fromParent]));
    }
    dependOn(...dependencies) {
        if (dependencies.includes(this)) {
            throw new Error(`Cannot add dependency on self: ${this}`);
        }
        this.dependencies.push(...dependencies.filter(javascript_1.isDefined));
    }
    ancestorPath(upTo) {
        let x = this;
        const ret = [x];
        while (x.parentGraph && x.parentGraph !== upTo) {
            x = x.parentGraph;
            ret.unshift(x);
        }
        return ret;
    }
    rootPath() {
        let x = this;
        const ret = [x];
        while (x.parentGraph) {
            x = x.parentGraph;
            ret.unshift(x);
        }
        return ret;
    }
    get root() {
        let x = this;
        while (x.parentGraph) {
            x = x.parentGraph;
        }
        return x;
    }
    get parentGraph() {
        return this._parentGraph;
    }
    /**
     * @internal
     */
    _setParentGraph(parentGraph) {
        if (this._parentGraph) {
            throw new Error('Node already has a parent');
        }
        this._parentGraph = parentGraph;
    }
    toString() {
        return `${this.constructor.name}(${this.id})`;
    }
}
exports.GraphNode = GraphNode;
/**
 * A dependency set that can be constructed partially and later finished
 *
 * It doesn't matter in what order sources and targets for the dependency
 * relationship(s) get added. This class can serve as a synchronization
 * point if the order in which graph nodes get added to the graph is not
 * well-defined.
 *
 * Useful utility during graph building.
 */
class DependencyBuilder {
    constructor() {
        this.targets = [];
        this.sources = [];
    }
    dependOn(...targets) {
        for (const target of targets) {
            for (const source of this.sources) {
                source.dependOn(target);
            }
            this.targets.push(target);
        }
        return this;
    }
    dependBy(...sources) {
        for (const source of sources) {
            for (const target of this.targets) {
                source.dependOn(target);
            }
            this.sources.push(source);
        }
        return this;
    }
}
exports.DependencyBuilder = DependencyBuilder;
class DependencyBuilders {
    constructor() {
        this.builders = new Map();
    }
    get(key) {
        const b = this.builders.get(key);
        if (b) {
            return b;
        }
        const ret = new DependencyBuilder();
        this.builders.set(key, ret);
        return ret;
    }
}
exports.DependencyBuilders = DependencyBuilders;
class Graph extends GraphNode {
    constructor(name, props = {}) {
        super(name, props);
        this.children = new Map();
        if (props.nodes) {
            this.add(...props.nodes);
        }
    }
    static of(id, data, nodes) {
        return new Graph(id, { data, nodes });
    }
    get nodes() {
        return new Set(this.children.values());
    }
    tryGetChild(name) {
        return this.children.get(name);
    }
    contains(node) {
        return this.nodes.has(node);
    }
    add(...nodes) {
        for (const node of nodes) {
            node._setParentGraph(this);
            if (this.children.has(node.id)) {
                throw new Error(`Node with duplicate id: ${node.id}`);
            }
            this.children.set(node.id, node);
        }
    }
    absorb(other) {
        this.add(...other.nodes);
    }
    /**
     * Return topologically sorted tranches of nodes at this graph level
     */
    sortedChildren() {
        // Project dependencies to current children
        const nodes = this.nodes;
        const projectedDependencies = projectDependencies(this.deepDependencies(), (node) => {
            while (!nodes.has(node) && node.parentGraph) {
                node = node.parentGraph;
            }
            return nodes.has(node) ? [node] : [];
        });
        return toposort_1.topoSort(nodes, projectedDependencies);
    }
    /**
     * Return a topologically sorted list of non-Graph nodes in the entire subgraph
     */
    sortedLeaves() {
        // Project dependencies to leaf nodes
        const descendantsMap = new Map();
        findDescendants(this);
        function findDescendants(node) {
            const ret = [];
            if (node instanceof Graph) {
                for (const child of node.nodes) {
                    ret.push(...findDescendants(child));
                }
            }
            else {
                ret.push(node);
            }
            descendantsMap.set(node, ret);
            return ret;
        }
        const projectedDependencies = projectDependencies(this.deepDependencies(), (node) => descendantsMap.get(node) ?? []);
        return toposort_1.topoSort(new Set(projectedDependencies.keys()), projectedDependencies);
    }
    consoleLog(indent = 0) {
        process.stdout.write(' '.repeat(indent) + this + depString(this) + '\n');
        for (const node of this.nodes) {
            if (node instanceof Graph) {
                node.consoleLog(indent + 2);
            }
            else {
                process.stdout.write(' '.repeat(indent + 2) + node + depString(node) + '\n');
            }
        }
        function depString(node) {
            if (node.dependencies.length > 0) {
                return ` -> ${Array.from(node.dependencies).join(', ')}`;
            }
            return '';
        }
    }
    /**
     * Return the union of all dependencies of the descendants of this graph
     */
    deepDependencies() {
        const ret = new Map();
        for (const node of this.nodes) {
            recurse(node);
        }
        return ret;
        function recurse(node) {
            let deps = ret.get(node);
            if (!deps) {
                ret.set(node, deps = new Set());
            }
            for (let dep of node.dependencies) {
                deps.add(dep);
            }
            if (node instanceof Graph) {
                for (const child of node.nodes) {
                    recurse(child);
                }
            }
        }
    }
    /**
     * Return all non-Graph nodes
     */
    allLeaves() {
        const ret = [];
        recurse(this);
        return new GraphNodeCollection(ret);
        function recurse(node) {
            if (node instanceof Graph) {
                for (const child of node.nodes) {
                    recurse(child);
                }
            }
            else {
                ret.push(node);
            }
        }
    }
}
exports.Graph = Graph;
/**
 * A collection of graph nodes
 */
class GraphNodeCollection {
    constructor(nodes) {
        this.nodes = Array.from(nodes);
    }
    dependOn(...dependencies) {
        for (const node of this.nodes) {
            node.dependOn(...dependencies.filter(javascript_1.isDefined));
        }
    }
    /**
    * Returns the graph node that's shared between these nodes
    */
    commonAncestor() {
        const paths = new Array();
        for (const x of this.nodes) {
            paths.push(x.rootPath());
        }
        if (paths.length === 0) {
            throw new Error('Cannot find common ancestor between an empty set of nodes');
        }
        if (paths.length === 1) {
            const path = paths[0];
            if (path.length < 2) {
                throw new Error(`Cannot find ancestor of node without ancestor: ${path[0]}`);
            }
            return path[path.length - 2];
        }
        const originalPaths = [...paths];
        // Remove the first element of every path as long as the 2nd elements are all
        // the same -- this leaves the shared element in first place.
        //
        //   A, B, C, 1, 2    }---> C
        //   A, B, C, 3       }
        while (paths.every(path => paths[0].length >= 2 && path.length >= 2 && path[1] === paths[0][1])) {
            for (const path of paths) {
                path.shift();
            }
        }
        // If any of the paths are left with 1 element, there's no shared parent.
        if (paths.some(path => path.length < 2)) {
            throw new Error(`Could not determine a shared parent between nodes: ${originalPaths.map(nodes => nodes.map(n => n.id).join('/'))}`);
        }
        return paths[0][0];
    }
}
exports.GraphNodeCollection = GraphNodeCollection;
/**
 * Dependency map of nodes in this graph, taking into account dependencies between nodes in subgraphs
 *
 * Guaranteed to return an entry in the map for every node in the current graph.
 */
function projectDependencies(dependencies, project) {
    // Project keys
    for (const node of dependencies.keys()) {
        const projectedNodes = project(node);
        if (projectedNodes.length === 1 && projectedNodes[0] === node) {
            continue;
        } // Nothing to do, just for efficiency
        const deps = javascript_1.extract(dependencies, node);
        for (const projectedNode of projectedNodes) {
            javascript_1.addAll(dependencies.get(projectedNode), deps);
        }
    }
    // Project values. Ignore self-dependencies, they were just between nodes that were collapsed into the same node.
    for (const [node, deps] of dependencies.entries()) {
        const depset = new Set(javascript_1.flatMap(deps, project));
        depset.delete(node);
        dependencies.set(node, depset);
    }
    return dependencies;
}
function isGraph(x) {
    return x instanceof Graph;
}
exports.isGraph = isGraph;
//# sourceMappingURL=data:application/json;base64,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