"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodePipelineFileSet = exports.ArtifactMap = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const cp = require("../../../aws-codepipeline");
const blueprint_1 = require("../blueprint");
const helpers_internal_1 = require("../helpers-internal");
/**
 * Translate FileSets to CodePipeline Artifacts
 */
class ArtifactMap {
    constructor() {
        this.artifacts = new Map();
        this.usedNames = new Set();
    }
    /**
     * Return the matching CodePipeline artifact for a FileSet
     */
    toCodePipeline(x) {
        try {
            jsiiDeprecationWarnings.monocdk_pipelines_FileSet(x);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.toCodePipeline);
            }
            throw error;
        }
        if (x instanceof CodePipelineFileSet) {
            return x._artifact;
        }
        let ret = this.artifacts.get(x);
        if (!ret) {
            // They all need a name
            const artifactName = this.makeUniqueName(`${x.producer.id}.${x.id}`);
            this.usedNames.add(artifactName);
            this.artifacts.set(x, ret = new cp.Artifact(artifactName));
        }
        return ret;
    }
    makeUniqueName(baseName) {
        let i = 1;
        baseName = sanitizeArtifactName(baseName);
        let name = baseName;
        while (this.usedNames.has(name)) {
            name = `${baseName}${++i}`;
        }
        return name;
    }
}
exports.ArtifactMap = ArtifactMap;
_a = JSII_RTTI_SYMBOL_1;
ArtifactMap[_a] = { fqn: "monocdk.pipelines.ArtifactMap", version: "1.185.0" };
/**
 * Sanitize a string to be a valid artifact name
 *
 * This must comport to both the rules of artifacts in CodePipeline, as well
 * as the names of Source Identifiers in CodeBuild.
 *
 * Artifact Name limits aren't documented.
 *
 * Source Identifier limits are documented here:
 * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ProjectSource.html#CodeBuild-Type-ProjectSource-sourceIdentifier
 */
function sanitizeArtifactName(x) {
    let sani = x.replace(/[^A-Za-z0-9_]/g, '_'); // Charset requirement is imposed by CodeBuild
    const maxLength = 100; // Max length of 100 is imposed by CodePipeline library
    if (sani.length > maxLength) {
        const fingerprint = crypto.createHash('sha256').update(sani).digest('hex').slice(0, 8);
        sani = sani.slice(0, maxLength - fingerprint.length) + fingerprint;
    }
    return sani;
}
/**
 * A FileSet created from a CodePipeline artifact
 *
 * You only need to use this if you want to add CDK Pipeline stages
 * add the end of an existing CodePipeline, which should be very rare.
 */
class CodePipelineFileSet extends blueprint_1.FileSet {
    constructor(artifact) {
        super(artifact.artifactName ?? 'Imported', helpers_internal_1.PipelineGraph.NO_STEP);
        this._artifact = artifact;
    }
    /**
     * Turn a CodePipeline Artifact into a FileSet
     */
    static fromArtifact(artifact) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_codepipeline_Artifact(artifact);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromArtifact);
            }
            throw error;
        }
        return new CodePipelineFileSet(artifact);
    }
}
exports.CodePipelineFileSet = CodePipelineFileSet;
_b = JSII_RTTI_SYMBOL_1;
CodePipelineFileSet[_b] = { fqn: "monocdk.pipelines.CodePipelineFileSet", version: "1.185.0" };
//# sourceMappingURL=data:application/json;base64,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