"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StageDeployment = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloud_assembly_internals_1 = require("../private/cloud-assembly-internals");
const construct_internals_1 = require("../private/construct-internals");
const stack_deployment_1 = require("./stack-deployment");
/**
 * Deployment of a single `Stage`
 *
 * A `Stage` consists of one or more `Stacks`, which will be
 * deployed in dependency order.
 */
class StageDeployment {
    constructor(
    /** The stacks deployed in this stage */
    stacks, props = {}) {
        this.stacks = stacks;
        this.stageName = props.stageName ?? '';
        this.pre = props.pre ?? [];
        this.post = props.post ?? [];
        this.stackSteps = props.stackSteps ?? [];
    }
    /**
     * Create a new `StageDeployment` from a `Stage`
     *
     * Synthesizes the target stage, and deployes the stacks found inside
     * in dependency order.
     */
    static fromStage(stage, props = {}) {
        try {
            jsiiDeprecationWarnings.monocdk_Stage(stage);
            jsiiDeprecationWarnings.monocdk_pipelines_StageDeploymentProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromStage);
            }
            throw error;
        }
        const assembly = construct_internals_1.pipelineSynth(stage);
        if (assembly.stacks.length === 0) {
            // If we don't check here, a more puzzling "stage contains no actions"
            // error will be thrown come deployment time.
            throw new Error(`The given Stage construct ('${stage.node.path}') should contain at least one Stack`);
        }
        const stepFromArtifact = new Map();
        for (const artifact of assembly.stacks) {
            const step = stack_deployment_1.StackDeployment.fromArtifact(artifact);
            stepFromArtifact.set(artifact, step);
        }
        if (props.stackSteps) {
            for (const stackstep of props.stackSteps) {
                const stackArtifact = assembly.getStackArtifact(stackstep.stack.artifactId);
                const thisStep = stepFromArtifact.get(stackArtifact);
                if (!thisStep) {
                    throw new Error('Logic error: we just added a step for this artifact but it disappeared.');
                }
                thisStep.addStackSteps(stackstep.pre ?? [], stackstep.changeSet ?? [], stackstep.post ?? []);
            }
        }
        for (const artifact of assembly.stacks) {
            const thisStep = stepFromArtifact.get(artifact);
            if (!thisStep) {
                throw new Error('Logic error: we just added a step for this artifact but it disappeared.');
            }
            const stackDependencies = artifact.dependencies.filter(cloud_assembly_internals_1.isStackArtifact);
            for (const dep of stackDependencies) {
                const depStep = stepFromArtifact.get(dep);
                if (!depStep) {
                    throw new Error(`Stack '${artifact.id}' depends on stack not found in same Stage: '${dep.id}'`);
                }
                thisStep.addStackDependency(depStep);
            }
        }
        return new StageDeployment(Array.from(stepFromArtifact.values()), {
            stageName: stage.stageName,
            ...props,
        });
    }
    /**
     * Add an additional step to run before any of the stacks in this stage
     */
    addPre(...steps) {
        try {
            jsiiDeprecationWarnings.monocdk_pipelines_Step(steps);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addPre);
            }
            throw error;
        }
        this.pre.push(...steps);
    }
    /**
     * Add an additional step to run after all of the stacks in this stage
     */
    addPost(...steps) {
        try {
            jsiiDeprecationWarnings.monocdk_pipelines_Step(steps);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addPost);
            }
            throw error;
        }
        this.post.push(...steps);
    }
}
exports.StageDeployment = StageDeployment;
_a = JSII_RTTI_SYMBOL_1;
StageDeployment[_a] = { fqn: "monocdk.pipelines.StageDeployment", version: "1.185.0" };
//# sourceMappingURL=data:application/json;base64,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