"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudArtifact = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxschema = require("../../cloud-assembly-schema");
const metadata_1 = require("./metadata");
/**
 * Represents an artifact within a cloud assembly.
 */
class CloudArtifact {
    constructor(assembly, id, manifest) {
        this.assembly = assembly;
        this.id = id;
        try {
            jsiiDeprecationWarnings.monocdk_cx_api_CloudAssembly(assembly);
            jsiiDeprecationWarnings.monocdk_cloud_assembly_schema_ArtifactManifest(manifest);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CloudArtifact);
            }
            throw error;
        }
        this.manifest = manifest;
        this.messages = this.renderMessages();
        this._dependencyIDs = manifest.dependencies || [];
    }
    /**
     * Returns a subclass of `CloudArtifact` based on the artifact type defined in the artifact manifest.
     *
     * @param assembly The cloud assembly from which to load the artifact
     * @param id The artifact ID
     * @param artifact The artifact manifest
     * @returns the `CloudArtifact` that matches the artifact type or `undefined` if it's an artifact type that is unrecognized by this module.
     */
    static fromManifest(assembly, id, artifact) {
        try {
            jsiiDeprecationWarnings.monocdk_cx_api_CloudAssembly(assembly);
            jsiiDeprecationWarnings.monocdk_cloud_assembly_schema_ArtifactManifest(artifact);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromManifest);
            }
            throw error;
        }
        // Implementation is defined in a separate file to break cyclic dependencies
        void (assembly), void (id), void (artifact);
        throw new Error('Implementation not overridden yet');
    }
    /**
     * Returns all the artifacts that this artifact depends on.
     */
    get dependencies() {
        if (this._deps) {
            return this._deps;
        }
        this._deps = this._dependencyIDs.map(id => {
            const dep = this.assembly.tryGetArtifact(id);
            if (!dep) {
                throw new Error(`Artifact ${this.id} depends on non-existing artifact ${id}`);
            }
            return dep;
        });
        return this._deps;
    }
    /**
     * @returns all the metadata entries of a specific type in this artifact.
     * @param type
     */
    findMetadataByType(type) {
        const result = new Array();
        for (const path of Object.keys(this.manifest.metadata || {})) {
            for (const entry of (this.manifest.metadata || {})[path]) {
                if (entry.type === type) {
                    result.push({ path, ...entry });
                }
            }
        }
        return result;
    }
    renderMessages() {
        const messages = new Array();
        for (const [id, metadata] of Object.entries(this.manifest.metadata || {})) {
            for (const entry of metadata) {
                let level;
                switch (entry.type) {
                    case cxschema.ArtifactMetadataEntryType.WARN:
                        level = metadata_1.SynthesisMessageLevel.WARNING;
                        break;
                    case cxschema.ArtifactMetadataEntryType.ERROR:
                        level = metadata_1.SynthesisMessageLevel.ERROR;
                        break;
                    case cxschema.ArtifactMetadataEntryType.INFO:
                        level = metadata_1.SynthesisMessageLevel.INFO;
                        break;
                    default:
                        continue;
                }
                messages.push({ level, entry, id });
            }
        }
        return messages;
    }
    /**
     * An identifier that shows where this artifact is located in the tree
     * of nested assemblies, based on their manifests. Defaults to the normal
     * id. Should only be used in user interfaces.
     */
    get hierarchicalId() {
        return this.manifest.displayName ?? this.id;
    }
}
exports.CloudArtifact = CloudArtifact;
_a = JSII_RTTI_SYMBOL_1;
CloudArtifact[_a] = { fqn: "monocdk.cx_api.CloudArtifact", version: "1.185.0" };
//# sourceMappingURL=data:application/json;base64,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