"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CallApiGatewayEndpointBase = void 0;
const iam = require("../../../aws-iam");
const sfn = require("../../../aws-stepfunctions");
const task_utils_1 = require("../private/task-utils");
const base_types_1 = require("./base-types");
/**
 * Base CallApiGatewayEndpoint Task
 * @internal
 */
class CallApiGatewayEndpointBase extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.baseProps = props;
        this.integrationPattern = props.integrationPattern ?? sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, CallApiGatewayEndpointBase.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN) {
            if (!sfn.FieldUtils.containsTaskToken(this.baseProps.headers)) {
                throw new Error('Task Token is required in `headers` for WAIT_FOR_TASK_TOKEN pattern. Use JsonPath.taskToken to set the token.');
            }
        }
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('apigateway', 'invoke', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                ApiEndpoint: this.apiEndpoint,
                Method: this.baseProps.method,
                Headers: this.baseProps.headers?.value,
                Stage: this.stageName,
                Path: this.baseProps.apiPath,
                QueryParameters: this.baseProps.queryParameters?.value,
                RequestBody: this.baseProps.requestBody?.value,
                AuthType: this.baseProps.authType ? this.baseProps.authType : 'NO_AUTH',
            }),
        };
    }
    createPolicyStatements() {
        if (this.baseProps.authType === base_types_1.AuthType.NO_AUTH) {
            return [];
        }
        return [
            new iam.PolicyStatement({
                resources: [this.arnForExecuteApi],
                actions: ['execute-api:Invoke'],
            }),
        ];
    }
}
exports.CallApiGatewayEndpointBase = CallApiGatewayEndpointBase;
CallApiGatewayEndpointBase.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
//# sourceMappingURL=data:application/json;base64,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