"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Asset = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const kms = require("../../aws-kms");
const s3 = require("../../aws-s3");
const cdk = require("../../core");
const cxapi = require("../../cx-api");
const compat_1 = require("./compat");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("../../core");
/**
 * An asset represents a local file or directory, which is automatically uploaded to S3
 * and then can be referenced within a CDK application.
 */
class Asset extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.monocdk_aws_s3_assets_AssetProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Asset);
            }
            throw error;
        }
        this.isBundled = props.bundling != null;
        // stage the asset source (conditionally).
        const staging = new cdk.AssetStaging(this, 'Stage', {
            ...props,
            sourcePath: path.resolve(props.path),
            follow: props.followSymlinks ?? compat_1.toSymlinkFollow(props.follow),
            assetHash: props.assetHash ?? props.sourceHash,
        });
        this.assetHash = staging.assetHash;
        this.sourceHash = this.assetHash;
        const stack = cdk.Stack.of(this);
        this.assetPath = staging.relativeStagedPath(stack);
        this.isFile = staging.packaging === cdk.FileAssetPackaging.FILE;
        this.isZipArchive = staging.isArchive;
        const location = stack.synthesizer.addFileAsset({
            packaging: staging.packaging,
            sourceHash: this.sourceHash,
            fileName: this.assetPath,
        });
        this.s3BucketName = location.bucketName;
        this.s3ObjectKey = location.objectKey;
        this.s3ObjectUrl = location.s3ObjectUrl;
        this.httpUrl = location.httpUrl;
        this.s3Url = location.httpUrl; // for backwards compatibility
        const kmsKey = location.kmsKeyArn ? kms.Key.fromKeyArn(this, 'Key', location.kmsKeyArn) : undefined;
        this.bucket = s3.Bucket.fromBucketAttributes(this, 'AssetBucket', {
            bucketName: this.s3BucketName,
            encryptionKey: kmsKey,
        });
        for (const reader of (props.readers ?? [])) {
            this.grantRead(reader);
        }
    }
    /**
     * Adds CloudFormation template metadata to the specified resource with
     * information that indicates which resource property is mapped to this local
     * asset. This can be used by tools such as SAM CLI to provide local
     * experience such as local invocation and debugging of Lambda functions.
     *
     * Asset metadata will only be included if the stack is synthesized with the
     * "aws:cdk:enable-asset-metadata" context key defined, which is the default
     * behavior when synthesizing via the CDK Toolkit.
     *
     * @see https://github.com/aws/aws-cdk/issues/1432
     *
     * @param resource The CloudFormation resource which is using this asset [disable-awslint:ref-via-interface]
     * @param resourceProperty The property name where this asset is referenced
     * (e.g. "Code" for AWS::Lambda::Function)
     */
    addResourceMetadata(resource, resourceProperty) {
        try {
            jsiiDeprecationWarnings.monocdk_CfnResource(resource);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addResourceMetadata);
            }
            throw error;
        }
        if (!this.node.tryGetContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT)) {
            return; // not enabled
        }
        // tell tools such as SAM CLI that the "Code" property of this resource
        // points to a local path in order to enable local invocation of this function.
        resource.cfnOptions.metadata = resource.cfnOptions.metadata || {};
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PATH_KEY] = this.assetPath;
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_IS_BUNDLED_KEY] = this.isBundled;
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY] = resourceProperty;
    }
    /**
     * Grants read permissions to the principal on the assets bucket.
     */
    grantRead(grantee) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_iam_IGrantable(grantee);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantRead);
            }
            throw error;
        }
        // we give permissions on all files in the bucket since we don't want to
        // accidentally revoke permission on old versions when deploying a new
        // version (for example, when using Lambda traffic shifting).
        this.bucket.grantRead(grantee);
    }
}
exports.Asset = Asset;
_a = JSII_RTTI_SYMBOL_1;
Asset[_a] = { fqn: "monocdk.aws_s3_assets.Asset", version: "1.185.0" };
//# sourceMappingURL=data:application/json;base64,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