"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LogRetention = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const iam = require("../../aws-iam");
const s3_assets = require("../../aws-s3-assets");
const cdk = require("../../core");
const log_group_1 = require("./log-group");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("../../core");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("../../core");
/**
 * Creates a custom resource to control the retention policy of a CloudWatch Logs
 * log group. The log group is created if it doesn't already exist. The policy
 * is removed when `retentionDays` is `undefined` or equal to `Infinity`.
 * Log group can be created in the region that is different from stack region by
 * specifying `logGroupRegion`
 */
class LogRetention extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.monocdk_aws_logs_LogRetentionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, LogRetention);
            }
            throw error;
        }
        // Custom resource provider
        const provider = this.ensureSingletonLogRetentionFunction(props);
        // Need to use a CfnResource here to prevent lerna dependency cycles
        // @aws-cdk/aws-cloudformation -> @aws-cdk/aws-lambda -> @aws-cdk/aws-cloudformation
        const retryOptions = props.logRetentionRetryOptions;
        const resource = new cdk.CfnResource(this, 'Resource', {
            type: 'Custom::LogRetention',
            properties: {
                ServiceToken: provider.functionArn,
                LogGroupName: props.logGroupName,
                LogGroupRegion: props.logGroupRegion,
                SdkRetry: retryOptions ? {
                    maxRetries: retryOptions.maxRetries,
                    base: retryOptions.base?.toMilliseconds(),
                } : undefined,
                RetentionInDays: props.retention === log_group_1.RetentionDays.INFINITE ? undefined : props.retention,
            },
        });
        const logGroupName = resource.getAtt('LogGroupName').toString();
        // Append ':*' at the end of the ARN to match with how CloudFormation does this for LogGroup ARNs
        // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-logs-loggroup.html#aws-resource-logs-loggroup-return-values
        this.logGroupArn = cdk.Stack.of(this).formatArn({
            region: props.logGroupRegion,
            service: 'logs',
            resource: 'log-group',
            resourceName: `${logGroupName}:*`,
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
        });
    }
    /**
     * Helper method to ensure that only one instance of LogRetentionFunction resources are in the stack mimicking the
     * behaviour of @aws-cdk/aws-lambda's SingletonFunction to prevent circular dependencies
     */
    ensureSingletonLogRetentionFunction(props) {
        const functionLogicalId = 'LogRetentionaae0aa3c5b4d4f87b02d85b201efdd8a';
        const existing = cdk.Stack.of(this).node.tryFindChild(functionLogicalId);
        if (existing) {
            return existing;
        }
        return new LogRetentionFunction(cdk.Stack.of(this), functionLogicalId, props);
    }
}
exports.LogRetention = LogRetention;
_a = JSII_RTTI_SYMBOL_1;
LogRetention[_a] = { fqn: "monocdk.aws_logs.LogRetention", version: "1.185.0" };
/**
 * Private provider Lambda function to support the log retention custom resource.
 */
class LogRetentionFunction extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.tags = new cdk.TagManager(cdk.TagType.KEY_VALUE, 'AWS::Lambda::Function');
        // Code
        const asset = new s3_assets.Asset(this, 'Code', {
            path: path.join(__dirname, 'log-retention-provider'),
        });
        // Role
        const role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole')],
        });
        // Duplicate statements will be deduplicated by `PolicyDocument`
        role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:PutRetentionPolicy', 'logs:DeleteRetentionPolicy'],
            // We need '*' here because we will also put a retention policy on
            // the log group of the provider function. Referencing its name
            // creates a CF circular dependency.
            resources: ['*'],
        }));
        // Lambda function
        const resource = new cdk.CfnResource(this, 'Resource', {
            type: 'AWS::Lambda::Function',
            properties: {
                Handler: 'index.handler',
                Runtime: 'nodejs14.x',
                Code: {
                    S3Bucket: asset.s3BucketName,
                    S3Key: asset.s3ObjectKey,
                },
                Role: role.roleArn,
                Tags: this.tags.renderedTags,
            },
        });
        this.functionArn = resource.getAtt('Arn');
        asset.addResourceMetadata(resource, 'Code');
        // Function dependencies
        role.node.children.forEach((child) => {
            if (cdk.CfnResource.isCfnResource(child)) {
                resource.addDependsOn(child);
            }
            if (cdk.Construct.isConstruct(child) && child.node.defaultChild && cdk.CfnResource.isCfnResource(child.node.defaultChild)) {
                resource.addDependsOn(child.node.defaultChild);
            }
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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