"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EventField = exports.RuleTargetInput = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
/**
 * The input to send to the event target
 */
class RuleTargetInput {
    constructor() {
    }
    /**
     * Pass text to the event target
     *
     * May contain strings returned by `EventField.from()` to substitute in parts of the
     * matched event.
     *
     * The Rule Target input value will be a single string: the string you pass
     * here.  Do not use this method to pass a complex value like a JSON object to
     * a Rule Target.  Use `RuleTargetInput.fromObject()` instead.
     */
    static fromText(text) {
        return new FieldAwareEventInput(text, InputType.Text);
    }
    /**
     * Pass text to the event target, splitting on newlines.
     *
     * This is only useful when passing to a target that does not
     * take a single argument.
     *
     * May contain strings returned by `EventField.from()` to substitute in parts
     * of the matched event.
     */
    static fromMultilineText(text) {
        return new FieldAwareEventInput(text, InputType.Multiline);
    }
    /**
     * Pass a JSON object to the event target
     *
     * May contain strings returned by `EventField.from()` to substitute in parts of the
     * matched event.
     */
    static fromObject(obj) {
        return new FieldAwareEventInput(obj, InputType.Object);
    }
    /**
     * Take the event target input from a path in the event JSON
     */
    static fromEventPath(path) {
        return new LiteralEventInput({ inputPath: path });
    }
}
exports.RuleTargetInput = RuleTargetInput;
_a = JSII_RTTI_SYMBOL_1;
RuleTargetInput[_a] = { fqn: "monocdk.aws_events.RuleTargetInput", version: "1.185.0" };
/**
 * Event Input that is directly derived from the construct
 */
class LiteralEventInput extends RuleTargetInput {
    constructor(props) {
        super();
        this.props = props;
    }
    /**
     * Return the input properties for this input object
     */
    bind(_rule) {
        return this.props;
    }
}
/**
 * Input object that can contain field replacements
 *
 * Evaluation is done in the bind() method because token resolution
 * requires access to the construct tree.
 *
 * Multiple tokens that use the same path will use the same substitution
 * key.
 *
 * One weird exception: if we're in object context, we MUST skip the quotes
 * around the placeholder. I assume this is so once a trivial string replace is
 * done later on by EventBridge, numbers are still numbers.
 *
 * So in string context:
 *
 *    "this is a string with a <field>"
 *
 * But in object context:
 *
 *    "{ \"this is the\": <field> }"
 *
 * To achieve the latter, we postprocess the JSON string to remove the surrounding
 * quotes by using a string replace.
 */
class FieldAwareEventInput extends RuleTargetInput {
    constructor(input, inputType) {
        super();
        this.input = input;
        this.inputType = inputType;
    }
    bind(rule) {
        let fieldCounter = 0;
        const pathToKey = new Map();
        const inputPathsMap = {};
        function keyForField(f) {
            const existing = pathToKey.get(f.path);
            if (existing !== undefined) {
                return existing;
            }
            fieldCounter += 1;
            const key = f.displayHint || `f${fieldCounter}`;
            pathToKey.set(f.path, key);
            return key;
        }
        class EventFieldReplacer extends core_1.DefaultTokenResolver {
            constructor() {
                super(new core_1.StringConcat());
            }
            resolveToken(t, _context) {
                if (!isEventField(t)) {
                    return core_1.Token.asString(t);
                }
                const key = keyForField(t);
                if (inputPathsMap[key] && inputPathsMap[key] !== t.path) {
                    throw new Error(`Single key '${key}' is used for two different JSON paths: '${t.path}' and '${inputPathsMap[key]}'`);
                }
                inputPathsMap[key] = t.path;
                return `<${key}>`;
            }
        }
        const stack = core_1.Stack.of(rule);
        let resolved;
        if (this.inputType === InputType.Multiline) {
            // JSONify individual lines
            resolved = core_1.Tokenization.resolve(this.input, {
                scope: rule,
                resolver: new EventFieldReplacer(),
            });
            resolved = resolved.split('\n').map(stack.toJsonString).join('\n');
        }
        else {
            resolved = stack.toJsonString(core_1.Tokenization.resolve(this.input, {
                scope: rule,
                resolver: new EventFieldReplacer(),
            }));
        }
        const keys = Object.keys(inputPathsMap);
        if (keys.length === 0) {
            // Nothing special, just return 'input'
            return { input: resolved };
        }
        return {
            inputTemplate: this.unquoteKeyPlaceholders(resolved, keys),
            inputPathsMap,
        };
    }
    /**
     * Removing surrounding quotes from any object placeholders
     * when key is the lone value.
     *
     * Those have been put there by JSON.stringify(), but we need to
     * remove them.
     *
     * Do not remove quotes when the key is part of a larger string.
     *
     * Valid: { "data": "Some string with \"quotes\"<key>" } // key will be string
     * Valid: { "data": <key> } // Key could be number, bool, obj, or string
     */
    unquoteKeyPlaceholders(sub, keys) {
        if (this.inputType !== InputType.Object) {
            return sub;
        }
        return core_1.Lazy.uncachedString({ produce: (ctx) => core_1.Token.asString(deepUnquote(ctx.resolve(sub))) });
        function deepUnquote(resolved) {
            if (Array.isArray(resolved)) {
                return resolved.map(deepUnquote);
            }
            else if (typeof (resolved) === 'object' && resolved !== null) {
                for (const [key, value] of Object.entries(resolved)) {
                    resolved[key] = deepUnquote(value);
                }
                return resolved;
            }
            else if (typeof (resolved) === 'string') {
                return keys.reduce((r, key) => r.replace(new RegExp(`(?<!\\\\)\"\<${key}\>\"`, 'g'), `<${key}>`), resolved);
            }
            return resolved;
        }
    }
}
/**
 * Represents a field in the event pattern
 */
class EventField {
    /**
     *
     * @param path the path to a field in the event pattern
     */
    constructor(path) {
        this.path = path;
        this.displayHint = this.path.replace(/^[^a-zA-Z0-9_-]+/, '').replace(/[^a-zA-Z0-9_-]/g, '-');
        Object.defineProperty(this, EVENT_FIELD_SYMBOL, { value: true });
        this.creationStack = core_1.captureStackTrace();
    }
    /**
     * Extract the event ID from the event
     */
    static get eventId() {
        return this.fromPath('$.id');
    }
    /**
     * Extract the detail type from the event
     */
    static get detailType() {
        return this.fromPath('$.detail-type');
    }
    /**
     * Extract the source from the event
     */
    static get source() {
        return this.fromPath('$.source');
    }
    /**
     * Extract the account from the event
     */
    static get account() {
        return this.fromPath('$.account');
    }
    /**
     * Extract the time from the event
     */
    static get time() {
        return this.fromPath('$.time');
    }
    /**
     * Extract the region from the event
     */
    static get region() {
        return this.fromPath('$.region');
    }
    /**
     * Extract a custom JSON path from the event
     */
    static fromPath(path) {
        return new EventField(path).toString();
    }
    resolve(_ctx) {
        try {
            jsiiDeprecationWarnings.monocdk_IResolveContext(_ctx);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.resolve);
            }
            throw error;
        }
        return this.path;
    }
    toString() {
        return core_1.Token.asString(this, { displayHint: this.displayHint });
    }
    /**
     * Convert the path to the field in the event pattern to JSON
     */
    toJSON() {
        return `<path:${this.path}>`;
    }
}
exports.EventField = EventField;
_b = JSII_RTTI_SYMBOL_1;
EventField[_b] = { fqn: "monocdk.aws_events.EventField", version: "1.185.0" };
var InputType;
(function (InputType) {
    InputType[InputType["Object"] = 0] = "Object";
    InputType[InputType["Text"] = 1] = "Text";
    InputType[InputType["Multiline"] = 2] = "Multiline";
})(InputType || (InputType = {}));
function isEventField(x) {
    return EVENT_FIELD_SYMBOL in x;
}
const EVENT_FIELD_SYMBOL = Symbol.for('@aws-cdk/aws-events.EventField');
//# sourceMappingURL=data:application/json;base64,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