"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsTask = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2");
const ecs = require("../../aws-ecs");
const events = require("../../aws-events");
const iam = require("../../aws-iam");
const cdk = require("../../core");
const util_1 = require("./util");
/**
 * Start a task on an ECS cluster
 */
class EcsTask {
    constructor(props) {
        this.props = props;
        try {
            jsiiDeprecationWarnings.monocdk_aws_events_targets_EcsTaskProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, EcsTask);
            }
            throw error;
        }
        if (props.securityGroup !== undefined && props.securityGroups !== undefined) {
            throw new Error('Only one of SecurityGroup or SecurityGroups can be populated.');
        }
        this.cluster = props.cluster;
        this.taskDefinition = props.taskDefinition;
        this.taskCount = props.taskCount ?? 1;
        this.platformVersion = props.platformVersion;
        this.role = props.role ?? util_1.singletonEventRole(this.taskDefinition);
        for (const stmt of this.createEventRolePolicyStatements()) {
            this.role.addToPrincipalPolicy(stmt);
        }
        // Security groups are only configurable with the "awsvpc" network mode.
        if (this.taskDefinition.networkMode !== ecs.NetworkMode.AWS_VPC) {
            if (props.securityGroup !== undefined || props.securityGroups !== undefined) {
                cdk.Annotations.of(this.taskDefinition).addWarning('security groups are ignored when network mode is not awsvpc');
            }
            return;
        }
        if (props.securityGroups) {
            this.securityGroups = props.securityGroups;
            return;
        }
        if (!cdk.Construct.isConstruct(this.taskDefinition)) {
            throw new Error('Cannot create a security group for ECS task. ' +
                'The task definition in ECS task is not a Construct. ' +
                'Please pass a taskDefinition as a Construct in EcsTaskProps.');
        }
        let securityGroup = props.securityGroup || this.taskDefinition.node.tryFindChild('SecurityGroup');
        securityGroup = securityGroup || new ec2.SecurityGroup(this.taskDefinition, 'SecurityGroup', { vpc: this.props.cluster.vpc });
        this.securityGroup = securityGroup; // Maintain backwards-compatibility for customers that read the generated security group.
        this.securityGroups = [securityGroup];
    }
    /**
     * Allows using tasks as target of EventBridge events
     */
    bind(_rule, _id) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_events_IRule(_rule);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        const arn = this.cluster.clusterArn;
        const role = this.role;
        const containerOverrides = this.props.containerOverrides && this.props.containerOverrides
            .map(({ containerName, ...overrides }) => ({ name: containerName, ...overrides }));
        const input = { containerOverrides };
        const taskCount = this.taskCount;
        const taskDefinitionArn = this.taskDefinition.taskDefinitionArn;
        const subnetSelection = this.props.subnetSelection || { subnetType: ec2.SubnetType.PRIVATE_WITH_NAT };
        const assignPublicIp = subnetSelection.subnetType === ec2.SubnetType.PUBLIC ? 'ENABLED' : 'DISABLED';
        const baseEcsParameters = { taskCount, taskDefinitionArn };
        const ecsParameters = this.taskDefinition.networkMode === ecs.NetworkMode.AWS_VPC
            ? {
                ...baseEcsParameters,
                launchType: this.taskDefinition.isEc2Compatible ? 'EC2' : 'FARGATE',
                platformVersion: this.platformVersion,
                networkConfiguration: {
                    awsVpcConfiguration: {
                        subnets: this.props.cluster.vpc.selectSubnets(subnetSelection).subnetIds,
                        assignPublicIp,
                        securityGroups: this.securityGroups && this.securityGroups.map(sg => sg.securityGroupId),
                    },
                },
            }
            : baseEcsParameters;
        return {
            arn,
            role,
            ecsParameters,
            input: events.RuleTargetInput.fromObject(input),
            targetResource: this.taskDefinition,
        };
    }
    createEventRolePolicyStatements() {
        const policyStatements = [new iam.PolicyStatement({
                actions: ['ecs:RunTask'],
                resources: [this.taskDefinition.taskDefinitionArn],
                conditions: {
                    ArnEquals: { 'ecs:cluster': this.cluster.clusterArn },
                },
            })];
        // If it so happens that a Task Execution Role was created for the TaskDefinition,
        // then the EventBridge Role must have permissions to pass it (otherwise it doesn't).
        if (this.taskDefinition.executionRole !== undefined) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.taskDefinition.executionRole.roleArn],
            }));
        }
        // For Fargate task we need permission to pass the task role.
        if (this.taskDefinition.isFargateCompatible) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.taskDefinition.taskRole.roleArn],
            }));
        }
        return policyStatements;
    }
}
exports.EcsTask = EcsTask;
_a = JSII_RTTI_SYMBOL_1;
EcsTask[_a] = { fqn: "monocdk.aws_events_targets.EcsTask", version: "1.185.0" };
//# sourceMappingURL=data:application/json;base64,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