"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationTargetGroup = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("../../../aws-cloudwatch");
const ec2 = require("../../../aws-ec2");
const core_1 = require("../../../core");
const elasticloadbalancingv2_canned_metrics_generated_1 = require("../elasticloadbalancingv2-canned-metrics.generated");
const base_target_group_1 = require("../shared/base-target-group");
const enums_1 = require("../shared/enums");
const imported_1 = require("../shared/imported");
const util_1 = require("../shared/util");
/**
 * Define an Application Target Group
 */
class ApplicationTargetGroup extends base_target_group_1.TargetGroupBase {
    constructor(scope, id, props = {}) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_ApplicationTargetGroupProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ApplicationTargetGroup);
            }
            throw error;
        }
        const [protocol, port] = util_1.determineProtocolAndPort(props.protocol, props.port);
        const { protocolVersion } = props;
        super(scope, id, { ...props }, {
            protocol,
            protocolVersion,
            port,
        });
        this.protocol = protocol;
        this.port = port;
        // this.targetType is lazy
        this.node.addValidation({
            validate: () => {
                if (this.targetType === enums_1.TargetType.LAMBDA && (this.port || this.protocol)) {
                    return ['port/protocol should not be specified for Lambda targets'];
                }
                else {
                    return [];
                }
            },
        });
        this.connectableMembers = [];
        this.listeners = [];
        if (props) {
            if (props.slowStart !== undefined) {
                if (props.slowStart.toSeconds() < 30 || props.slowStart.toSeconds() > 900) {
                    throw new Error('Slow start duration value must be between 30 and 900 seconds.');
                }
                this.setAttribute('slow_start.duration_seconds', props.slowStart.toSeconds().toString());
            }
            if (props.stickinessCookieDuration) {
                this.enableCookieStickiness(props.stickinessCookieDuration, props.stickinessCookieName);
            }
            else {
                this.setAttribute('stickiness.enabled', 'false');
            }
            if (props.loadBalancingAlgorithmType) {
                this.setAttribute('load_balancing.algorithm.type', props.loadBalancingAlgorithmType);
            }
            this.addTarget(...(props.targets || []));
        }
    }
    /**
     * Import an existing target group
     */
    static fromTargetGroupAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_TargetGroupAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromTargetGroupAttributes);
            }
            throw error;
        }
        return new ImportedApplicationTargetGroup(scope, id, attrs);
    }
    /**
     * Import an existing target group
     *
     * @deprecated Use `fromTargetGroupAttributes` instead
     */
    static import(scope, id, props) {
        try {
            jsiiDeprecationWarnings.print("monocdk.aws_elasticloadbalancingv2.ApplicationTargetGroup#import", "Use `fromTargetGroupAttributes` instead");
            jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_TargetGroupImportProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.import);
            }
            throw error;
        }
        return ApplicationTargetGroup.fromTargetGroupAttributes(scope, id, props);
    }
    /**
     * Add a load balancing target to this target group
     */
    addTarget(...targets) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_IApplicationLoadBalancerTarget(targets);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addTarget);
            }
            throw error;
        }
        for (const target of targets) {
            const result = target.attachToApplicationTargetGroup(this);
            this.addLoadBalancerTarget(result);
        }
        if (this.targetType === enums_1.TargetType.LAMBDA) {
            this.setAttribute('stickiness.enabled', undefined);
        }
    }
    /**
     * Enable sticky routing via a cookie to members of this target group.
     *
     * Note: If the `cookieName` parameter is set, application-based stickiness will be applied,
     * otherwise it defaults to duration-based stickiness attributes (`lb_cookie`).
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/sticky-sessions.html
     */
    enableCookieStickiness(duration, cookieName) {
        try {
            jsiiDeprecationWarnings.monocdk_Duration(duration);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.enableCookieStickiness);
            }
            throw error;
        }
        if (duration.toSeconds() < 1 || duration.toSeconds() > 604800) {
            throw new Error('Stickiness cookie duration value must be between 1 second and 7 days (604800 seconds).');
        }
        if (cookieName !== undefined) {
            if (!core_1.Token.isUnresolved(cookieName) && (cookieName.startsWith('AWSALB') || cookieName.startsWith('AWSALBAPP') || cookieName.startsWith('AWSALBTG'))) {
                throw new Error('App cookie names that start with the following prefixes are not allowed: AWSALB, AWSALBAPP, and AWSALBTG; they\'re reserved for use by the load balancer.');
            }
            if (cookieName === '') {
                throw new Error('App cookie name cannot be an empty string.');
            }
        }
        this.setAttribute('stickiness.enabled', 'true');
        if (cookieName) {
            this.setAttribute('stickiness.type', 'app_cookie');
            this.setAttribute('stickiness.app_cookie.cookie_name', cookieName);
            this.setAttribute('stickiness.app_cookie.duration_seconds', duration.toSeconds().toString());
        }
        else {
            this.setAttribute('stickiness.type', 'lb_cookie');
            this.setAttribute('stickiness.lb_cookie.duration_seconds', duration.toSeconds().toString());
        }
    }
    /**
     * Register a connectable as a member of this target group.
     *
     * Don't call this directly. It will be called by load balancing targets.
     */
    registerConnectable(connectable, portRange) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ec2_IConnectable(connectable);
            jsiiDeprecationWarnings.monocdk_aws_ec2_Port(portRange);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.registerConnectable);
            }
            throw error;
        }
        portRange = portRange || ec2.Port.tcp(this.defaultPort);
        // Notify all listeners that we already know about of this new connectable.
        // Then remember for new listeners that might get added later.
        this.connectableMembers.push({ connectable, portRange });
        for (const listener of this.listeners) {
            listener.registerConnectable(connectable, portRange);
        }
    }
    /**
     * Register a listener that is load balancing to this target group.
     *
     * Don't call this directly. It will be called by listeners.
     */
    registerListener(listener, associatingConstruct) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_IApplicationListener(listener);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.registerListener);
            }
            throw error;
        }
        // Notify this listener of all connectables that we know about.
        // Then remember for new connectables that might get added later.
        for (const member of this.connectableMembers) {
            listener.registerConnectable(member.connectable, member.portRange);
        }
        this.listeners.push(listener);
        this.loadBalancerAttachedDependencies.add((associatingConstruct || listener));
    }
    /**
     * Full name of first load balancer
     */
    get firstLoadBalancerFullName() {
        if (this.listeners.length === 0) {
            throw new Error('The TargetGroup needs to be attached to a LoadBalancer before you can call this method');
        }
        return base_target_group_1.loadBalancerNameFromListenerArn(this.listeners[0].listenerArn);
    }
    /**
     * Return the given named metric for this Application Load Balancer Target Group
     *
     * Returns the metric for this target group from the point of view of the first
     * load balancer load balancing to it. If you have multiple load balancers load
     * sending traffic to the same target group, you will have to override the dimensions
     * on this metric.
     *
     * @default Average over 5 minutes
     */
    metric(metricName, props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metric);
            }
            throw error;
        }
        return new cloudwatch.Metric({
            namespace: 'AWS/ApplicationELB',
            metricName,
            dimensionsMap: {
                TargetGroup: this.targetGroupFullName,
                LoadBalancer: this.firstLoadBalancerFullName,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * The number of IPv6 requests received by the target group
     *
     * @default Sum over 5 minutes
     */
    metricIpv6RequestCount(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricIpv6RequestCount);
            }
            throw error;
        }
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.ApplicationELBMetrics.iPv6RequestCountSum, props);
    }
    /**
     * The number of requests processed over IPv4 and IPv6.
     *
     * This count includes only the requests with a response generated by a target of the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricRequestCount(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricRequestCount);
            }
            throw error;
        }
        return this.cannedMetric(elasticloadbalancingv2_canned_metrics_generated_1.ApplicationELBMetrics.requestCountSum, props);
    }
    /**
     * The number of healthy hosts in the target group
     *
     * @default Average over 5 minutes
     */
    metricHealthyHostCount(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricHealthyHostCount);
            }
            throw error;
        }
        return this.metric('HealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of unhealthy hosts in the target group
     *
     * @default Average over 5 minutes
     */
    metricUnhealthyHostCount(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricUnhealthyHostCount);
            }
            throw error;
        }
        return this.metric('UnHealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of HTTP 2xx/3xx/4xx/5xx response codes generated by all targets in this target group.
     *
     * This does not include any response codes generated by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricHttpCodeTarget(code, props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_HttpCodeTarget(code);
            jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricHttpCodeTarget);
            }
            throw error;
        }
        return this.metric(code, {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The average number of requests received by each target in a target group.
     *
     * The only valid statistic is Sum. Note that this represents the average not the sum.
     *
     * @default Sum over 5 minutes
     */
    metricRequestCountPerTarget(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricRequestCountPerTarget);
            }
            throw error;
        }
        return this.metric('RequestCountPerTarget', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of connections that were not successfully established between the load balancer and target.
     *
     * @default Sum over 5 minutes
     */
    metricTargetConnectionErrorCount(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricTargetConnectionErrorCount);
            }
            throw error;
        }
        return this.metric('TargetConnectionErrorCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The time elapsed, in seconds, after the request leaves the load balancer until a response from the target is received.
     *
     * @default Average over 5 minutes
     */
    metricTargetResponseTime(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricTargetResponseTime);
            }
            throw error;
        }
        return this.metric('TargetResponseTime', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of TLS connections initiated by the load balancer that did not establish a session with the target.
     *
     * Possible causes include a mismatch of ciphers or protocols.
     *
     * @default Sum over 5 minutes
     */
    metricTargetTLSNegotiationErrorCount(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cloudwatch_MetricOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.metricTargetTLSNegotiationErrorCount);
            }
            throw error;
        }
        return this.metric('TargetTLSNegotiationErrorCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    validate() {
        const ret = super.validate();
        if (this.targetType !== undefined && this.targetType !== enums_1.TargetType.LAMBDA
            && (this.protocol === undefined || this.port === undefined)) {
            ret.push('At least one of \'port\' or \'protocol\' is required for a non-Lambda TargetGroup');
        }
        if (this.healthCheck && this.healthCheck.protocol) {
            if (ALB_HEALTH_CHECK_PROTOCOLS.includes(this.healthCheck.protocol)) {
                if (this.healthCheck.interval && this.healthCheck.timeout &&
                    this.healthCheck.interval.toMilliseconds() <= this.healthCheck.timeout.toMilliseconds()) {
                    ret.push(`Healthcheck interval ${this.healthCheck.interval.toHumanString()} must be greater than the timeout ${this.healthCheck.timeout.toHumanString()}`);
                }
            }
            if (!ALB_HEALTH_CHECK_PROTOCOLS.includes(this.healthCheck.protocol)) {
                ret.push([
                    `Health check protocol '${this.healthCheck.protocol}' is not supported. `,
                    `Must be one of [${ALB_HEALTH_CHECK_PROTOCOLS.join(', ')}]`,
                ].join(''));
            }
        }
        return ret;
    }
    cannedMetric(fn, props) {
        return new cloudwatch.Metric({
            ...fn({
                LoadBalancer: this.firstLoadBalancerFullName,
                TargetGroup: this.targetGroupFullName,
            }),
            ...props,
        }).attachTo(this);
    }
}
exports.ApplicationTargetGroup = ApplicationTargetGroup;
_a = JSII_RTTI_SYMBOL_1;
ApplicationTargetGroup[_a] = { fqn: "monocdk.aws_elasticloadbalancingv2.ApplicationTargetGroup", version: "1.185.0" };
/**
 * An imported application target group
 */
class ImportedApplicationTargetGroup extends imported_1.ImportedTargetGroupBase {
    registerListener(_listener, _associatingConstruct) {
        // Nothing to do, we know nothing of our members
        core_1.Annotations.of(this).addWarning('Cannot register listener on imported target group -- security groups might need to be updated manually');
    }
    registerConnectable(_connectable, _portRange) {
        core_1.Annotations.of(this).addWarning('Cannot register connectable on imported target group -- security groups might need to be updated manually');
    }
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToApplicationTargetGroup(this);
            if (result.targetJson !== undefined) {
                throw new Error('Cannot add a non-self registering target to an imported TargetGroup. Create a new TargetGroup instead.');
            }
        }
    }
}
const ALB_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS];
//# sourceMappingURL=data:application/json;base64,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