"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceAccount = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("../../aws-iam");
const core_1 = require("../../core");
const k8s_manifest_1 = require("./k8s-manifest");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("../../core");
/**
 * Service Account
 */
class ServiceAccount extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.monocdk_aws_eks_ServiceAccountProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ServiceAccount);
            }
            throw error;
        }
        const { cluster } = props;
        this.serviceAccountName = props.name ?? core_1.Names.uniqueId(this).toLowerCase();
        this.serviceAccountNamespace = props.namespace ?? 'default';
        // From K8s docs: https://kubernetes.io/docs/tasks/configure-pod-container/configure-service-account/
        if (!this.isValidDnsSubdomainName(this.serviceAccountName)) {
            throw RangeError('The name of a ServiceAccount object must be a valid DNS subdomain name.');
        }
        // From K8s docs: https://kubernetes.io/docs/concepts/overview/working-with-objects/namespaces/#namespaces-and-dns
        if (!this.isValidDnsLabelName(this.serviceAccountNamespace)) {
            throw RangeError('All namespace names must be valid RFC 1123 DNS labels.');
        }
        /* Add conditions to the role to improve security. This prevents other pods in the same namespace to assume the role.
        * See documentation: https://docs.aws.amazon.com/eks/latest/userguide/create-service-account-iam-policy-and-role.html
        */
        const conditions = new core_1.CfnJson(this, 'ConditionJson', {
            value: {
                [`${cluster.openIdConnectProvider.openIdConnectProviderIssuer}:aud`]: 'sts.amazonaws.com',
                [`${cluster.openIdConnectProvider.openIdConnectProviderIssuer}:sub`]: `system:serviceaccount:${this.serviceAccountNamespace}:${this.serviceAccountName}`,
            },
        });
        const principal = new aws_iam_1.OpenIdConnectPrincipal(cluster.openIdConnectProvider).withConditions({
            StringEquals: conditions,
        });
        this.role = new aws_iam_1.Role(this, 'Role', { assumedBy: principal });
        this.assumeRoleAction = this.role.assumeRoleAction;
        this.grantPrincipal = this.role.grantPrincipal;
        this.policyFragment = this.role.policyFragment;
        // Note that we cannot use `cluster.addManifest` here because that would create the manifest
        // constrct in the scope of the cluster stack, which might be a different stack than this one.
        // This means that the cluster stack would depend on this stack because of the role,
        // and since this stack inherintely depends on the cluster stack, we will have a circular dependency.
        new k8s_manifest_1.KubernetesManifest(this, `manifest-${id}ServiceAccountResource`, {
            cluster,
            manifest: [{
                    apiVersion: 'v1',
                    kind: 'ServiceAccount',
                    metadata: {
                        name: this.serviceAccountName,
                        namespace: this.serviceAccountNamespace,
                        labels: {
                            'app.kubernetes.io/name': this.serviceAccountName,
                            ...props.labels,
                        },
                        annotations: {
                            'eks.amazonaws.com/role-arn': this.role.roleArn,
                            ...props.annotations,
                        },
                    },
                }],
        });
    }
    /**
     * @deprecated use `addToPrincipalPolicy()`
     */
    addToPolicy(statement) {
        try {
            jsiiDeprecationWarnings.print("monocdk.aws_eks.ServiceAccount#addToPolicy", "use `addToPrincipalPolicy()`");
            jsiiDeprecationWarnings.monocdk_aws_iam_PolicyStatement(statement);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToPolicy);
            }
            throw error;
        }
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    addToPrincipalPolicy(statement) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_iam_PolicyStatement(statement);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToPrincipalPolicy);
            }
            throw error;
        }
        return this.role.addToPrincipalPolicy(statement);
    }
    /**
     * If the value is a DNS subdomain name as defined in RFC 1123, from K8s docs.
     *
     * https://kubernetes.io/docs/concepts/overview/working-with-objects/names/#dns-subdomain-names
     */
    isValidDnsSubdomainName(value) {
        return value.length <= 253 && /^[a-z0-9]+[a-z0-9-.]*[a-z0-9]+$/.test(value);
    }
    /**
     * If the value follows DNS label standard as defined in RFC 1123, from K8s docs.
     *
     * https://kubernetes.io/docs/concepts/overview/working-with-objects/names/#dns-label-names
     */
    isValidDnsLabelName(value) {
        return value.length <= 63 && /^[a-z0-9]+[a-z0-9-]*[a-z0-9]+$/.test(value);
    }
}
exports.ServiceAccount = ServiceAccount;
_a = JSII_RTTI_SYMBOL_1;
ServiceAccount[_a] = { fqn: "monocdk.aws_eks.ServiceAccount", version: "1.185.0" };
//# sourceMappingURL=data:application/json;base64,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