"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClusterResourceProvider = void 0;
const path = require("path");
const lambda = require("../../aws-lambda");
const core_1 = require("../../core");
const cr = require("../../custom-resources");
const lambda_layer_node_proxy_agent_1 = require("../../lambda-layer-node-proxy-agent");
const HANDLER_DIR = path.join(__dirname, 'cluster-resource-handler');
const HANDLER_RUNTIME = lambda.Runtime.NODEJS_14_X;
/**
 * A custom resource provider that handles cluster operations. It serves
 * multiple custom resources such as the cluster resource and the fargate
 * resource.
 *
 * @internal
 */
class ClusterResourceProvider extends core_1.NestedStack {
    constructor(scope, id, props) {
        super(scope, id);
        // The NPM dependency proxy-agent is required in order to support proxy routing with the AWS JS SDK.
        const nodeProxyAgentLayer = new lambda_layer_node_proxy_agent_1.NodeProxyAgentLayer(this, 'NodeProxyAgentLayer');
        const onEvent = new lambda.Function(this, 'OnEventHandler', {
            code: lambda.Code.fromAsset(HANDLER_DIR),
            description: 'onEvent handler for EKS cluster resource provider',
            runtime: HANDLER_RUNTIME,
            environment: {
                AWS_STS_REGIONAL_ENDPOINTS: 'regional',
                ...props.environment,
            },
            handler: 'index.onEvent',
            timeout: core_1.Duration.minutes(1),
            vpc: props.subnets ? props.vpc : undefined,
            vpcSubnets: props.subnets ? { subnets: props.subnets } : undefined,
            securityGroups: props.securityGroup ? [props.securityGroup] : undefined,
            // Allow user to override the layer.
            layers: props.onEventLayer ? [props.onEventLayer] : [nodeProxyAgentLayer],
        });
        const isComplete = new lambda.Function(this, 'IsCompleteHandler', {
            code: lambda.Code.fromAsset(HANDLER_DIR),
            description: 'isComplete handler for EKS cluster resource provider',
            runtime: HANDLER_RUNTIME,
            environment: {
                AWS_STS_REGIONAL_ENDPOINTS: 'regional',
                ...props.environment,
            },
            handler: 'index.isComplete',
            timeout: core_1.Duration.minutes(1),
            vpc: props.subnets ? props.vpc : undefined,
            vpcSubnets: props.subnets ? { subnets: props.subnets } : undefined,
            securityGroups: props.securityGroup ? [props.securityGroup] : undefined,
            layers: [nodeProxyAgentLayer],
        });
        this.provider = new cr.Provider(this, 'Provider', {
            onEventHandler: onEvent,
            isCompleteHandler: isComplete,
            totalTimeout: core_1.Duration.hours(1),
            queryInterval: core_1.Duration.minutes(1),
            vpc: props.subnets ? props.vpc : undefined,
            vpcSubnets: props.subnets ? { subnets: props.subnets } : undefined,
            securityGroups: props.securityGroup ? [props.securityGroup] : undefined,
        });
        props.adminRole.grant(onEvent.role, 'sts:AssumeRole');
        props.adminRole.grant(isComplete.role, 'sts:AssumeRole');
    }
    static getOrCreate(scope, props) {
        const stack = core_1.Stack.of(scope);
        const uid = '@aws-cdk/aws-eks.ClusterResourceProvider';
        return stack.node.tryFindChild(uid) ?? new ClusterResourceProvider(stack, uid, props);
    }
    /**
     * The custom resource service token for this provider.
     */
    get serviceToken() { return this.provider.serviceToken; }
}
exports.ClusterResourceProvider = ClusterResourceProvider;
//# sourceMappingURL=data:application/json;base64,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