"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClusterResource = void 0;
const path = require("path");
const iam = require("../../aws-iam");
const lambda = require("../../aws-lambda");
const core_1 = require("../../core");
const kubectl_layer_1 = require("./kubectl-layer");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("../../core");
/**
 * A low-level CFN resource Amazon EKS cluster implemented through a custom
 * resource.
 *
 * Implements EKS create/update/delete through a CloudFormation custom resource
 * in order to allow us to control the IAM role which creates the cluster. This
 * is required in order to be able to allow CloudFormation to interact with the
 * cluster via `kubectl` to enable Kubernetes management capabilities like apply
 * manifest and IAM role/user RBAC mapping.
 */
class ClusterResource extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // each cluster resource will have it's own lambda handler since permissions
        // are scoped to this cluster and related resources like it's role
        const handler = new lambda.Function(this, 'ResourceHandler', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'cluster-resource')),
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'index.handler',
            timeout: core_1.Duration.minutes(15),
            memorySize: 512,
            layers: [kubectl_layer_1.KubectlLayer.getOrCreate(this)],
        });
        if (!props.roleArn) {
            throw new Error('"roleArn" is required');
        }
        // since we don't know the cluster name at this point, we must give this role star resource permissions
        handler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['eks:CreateCluster', 'eks:DescribeCluster', 'eks:DeleteCluster', 'eks:UpdateClusterVersion'],
            resources: ['*'],
        }));
        // the CreateCluster API will allow the cluster to assume this role, so we
        // need to allow the lambda execution role to pass it.
        handler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [props.roleArn],
        }));
        const resource = new core_1.CustomResource(this, 'Resource', {
            resourceType: ClusterResource.RESOURCE_TYPE,
            serviceToken: handler.functionArn,
            properties: {
                Config: props,
            },
        });
        this.ref = resource.ref;
        this.attrEndpoint = core_1.Token.asString(resource.getAtt('Endpoint'));
        this.attrArn = core_1.Token.asString(resource.getAtt('Arn'));
        this.attrCertificateAuthorityData = core_1.Token.asString(resource.getAtt('CertificateAuthorityData'));
        this.creationRole = handler.role;
    }
}
exports.ClusterResource = ClusterResource;
/**
 * The AWS CloudFormation resource type used for this resource.
 */
ClusterResource.RESOURCE_TYPE = 'Custom::AWSCDK-EKS-Cluster';
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2x1c3Rlci1yZXNvdXJjZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImNsdXN0ZXItcmVzb3VyY2UudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsNkJBQTZCO0FBQzdCLHFDQUFxQztBQUNyQywyQ0FBMkM7QUFDM0MscUNBQTZEO0FBRTdELG1EQUErQztBQUUvQyxpR0FBaUc7QUFDakcsOERBQThEO0FBQzlELHFDQUF1QztBQUV2Qzs7Ozs7Ozs7O0dBU0c7QUFDSCxNQUFhLGVBQWdCLFNBQVEsZ0JBQVM7SUFrQjVDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBc0I7UUFDOUQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQiw0RUFBNEU7UUFDNUUsa0VBQWtFO1FBQ2xFLE1BQU0sT0FBTyxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsaUJBQWlCLEVBQUU7WUFDM0QsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLGtCQUFrQixDQUFDLENBQUM7WUFDckUsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsVUFBVTtZQUNsQyxPQUFPLEVBQUUsZUFBZTtZQUN4QixPQUFPLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7WUFDN0IsVUFBVSxFQUFFLEdBQUc7WUFDZixNQUFNLEVBQUUsQ0FBQyw0QkFBWSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUN6QyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRTtZQUNsQixNQUFNLElBQUksS0FBSyxDQUFDLHVCQUF1QixDQUFDLENBQUM7U0FDMUM7UUFFRCx1R0FBdUc7UUFDdkcsT0FBTyxDQUFDLGVBQWUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDOUMsT0FBTyxFQUFFLENBQUMsbUJBQW1CLEVBQUUscUJBQXFCLEVBQUUsbUJBQW1CLEVBQUUsMEJBQTBCLENBQUM7WUFDdEcsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO1NBQ2pCLENBQUMsQ0FBQyxDQUFDO1FBRUosMEVBQTBFO1FBQzFFLHNEQUFzRDtRQUN0RCxPQUFPLENBQUMsZUFBZSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUM5QyxPQUFPLEVBQUUsQ0FBQyxjQUFjLENBQUM7WUFDekIsU0FBUyxFQUFFLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQztTQUMzQixDQUFDLENBQUMsQ0FBQztRQUVKLE1BQU0sUUFBUSxHQUFHLElBQUkscUJBQWMsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ3BELFlBQVksRUFBRSxlQUFlLENBQUMsYUFBYTtZQUMzQyxZQUFZLEVBQUUsT0FBTyxDQUFDLFdBQVc7WUFDakMsVUFBVSxFQUFFO2dCQUNWLE1BQU0sRUFBRSxLQUFLO2FBQ2Q7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsR0FBRyxHQUFHLFFBQVEsQ0FBQyxHQUFHLENBQUM7UUFDeEIsSUFBSSxDQUFDLFlBQVksR0FBRyxZQUFLLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQztRQUNoRSxJQUFJLENBQUMsT0FBTyxHQUFHLFlBQUssQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBQ3RELElBQUksQ0FBQyw0QkFBNEIsR0FBRyxZQUFLLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsMEJBQTBCLENBQUMsQ0FBQyxDQUFDO1FBQ2hHLElBQUksQ0FBQyxZQUFZLEdBQUcsT0FBTyxDQUFDLElBQUssQ0FBQztLQUNuQzs7QUE5REgsMENBK0RDO0FBOURDOztHQUVHO0FBQ29CLDZCQUFhLEdBQUcsNEJBQTRCLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJy4uLy4uL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJy4uLy4uL2F3cy1sYW1iZGEnO1xuaW1wb3J0IHsgQ3VzdG9tUmVzb3VyY2UsIER1cmF0aW9uLCBUb2tlbiB9IGZyb20gJy4uLy4uL2NvcmUnO1xuaW1wb3J0IHsgQ2ZuQ2x1c3RlclByb3BzIH0gZnJvbSAnLi9la3MuZ2VuZXJhdGVkJztcbmltcG9ydCB7IEt1YmVjdGxMYXllciB9IGZyb20gJy4va3ViZWN0bC1sYXllcic7XG5cbi8vIGtlZXAgdGhpcyBpbXBvcnQgc2VwYXJhdGUgZnJvbSBvdGhlciBpbXBvcnRzIHRvIHJlZHVjZSBjaGFuY2UgZm9yIG1lcmdlIGNvbmZsaWN0cyB3aXRoIHYyLW1haW5cbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1kdXBsaWNhdGUtaW1wb3J0cywgaW1wb3J0L29yZGVyXG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICcuLi8uLi9jb3JlJztcblxuLyoqXG4gKiBBIGxvdy1sZXZlbCBDRk4gcmVzb3VyY2UgQW1hem9uIEVLUyBjbHVzdGVyIGltcGxlbWVudGVkIHRocm91Z2ggYSBjdXN0b21cbiAqIHJlc291cmNlLlxuICpcbiAqIEltcGxlbWVudHMgRUtTIGNyZWF0ZS91cGRhdGUvZGVsZXRlIHRocm91Z2ggYSBDbG91ZEZvcm1hdGlvbiBjdXN0b20gcmVzb3VyY2VcbiAqIGluIG9yZGVyIHRvIGFsbG93IHVzIHRvIGNvbnRyb2wgdGhlIElBTSByb2xlIHdoaWNoIGNyZWF0ZXMgdGhlIGNsdXN0ZXIuIFRoaXNcbiAqIGlzIHJlcXVpcmVkIGluIG9yZGVyIHRvIGJlIGFibGUgdG8gYWxsb3cgQ2xvdWRGb3JtYXRpb24gdG8gaW50ZXJhY3Qgd2l0aCB0aGVcbiAqIGNsdXN0ZXIgdmlhIGBrdWJlY3RsYCB0byBlbmFibGUgS3ViZXJuZXRlcyBtYW5hZ2VtZW50IGNhcGFiaWxpdGllcyBsaWtlIGFwcGx5XG4gKiBtYW5pZmVzdCBhbmQgSUFNIHJvbGUvdXNlciBSQkFDIG1hcHBpbmcuXG4gKi9cbmV4cG9ydCBjbGFzcyBDbHVzdGVyUmVzb3VyY2UgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICAvKipcbiAgICogVGhlIEFXUyBDbG91ZEZvcm1hdGlvbiByZXNvdXJjZSB0eXBlIHVzZWQgZm9yIHRoaXMgcmVzb3VyY2UuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFJFU09VUkNFX1RZUEUgPSAnQ3VzdG9tOjpBV1NDREstRUtTLUNsdXN0ZXInO1xuXG4gIHB1YmxpYyByZWFkb25seSBhdHRyRW5kcG9pbnQ6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGF0dHJBcm46IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGF0dHJDZXJ0aWZpY2F0ZUF1dGhvcml0eURhdGE6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IHJlZjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgSUFNIHJvbGUgd2hpY2ggY3JlYXRlZCB0aGUgY2x1c3Rlci4gSW5pdGlhbGx5IHRoaXMgaXMgdGhlIG9ubHkgSUFNIHJvbGVcbiAgICogdGhhdCBnZXRzIGFkbWluaXN0cmF0b3IgcHJpdmlsYWdlcyBvbiB0aGUgY2x1c3RlciAoYHN5c3RlbTptYXN0ZXJzYCksIGFuZFxuICAgKiB3aWxsIGJlIGFibGUgdG8gaXNzdWUgYGt1YmVjdGxgIGNvbW1hbmRzIGFnYWluc3QgaXQuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgY3JlYXRpb25Sb2xlOiBpYW0uSVJvbGU7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IENmbkNsdXN0ZXJQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICAvLyBlYWNoIGNsdXN0ZXIgcmVzb3VyY2Ugd2lsbCBoYXZlIGl0J3Mgb3duIGxhbWJkYSBoYW5kbGVyIHNpbmNlIHBlcm1pc3Npb25zXG4gICAgLy8gYXJlIHNjb3BlZCB0byB0aGlzIGNsdXN0ZXIgYW5kIHJlbGF0ZWQgcmVzb3VyY2VzIGxpa2UgaXQncyByb2xlXG4gICAgY29uc3QgaGFuZGxlciA9IG5ldyBsYW1iZGEuRnVuY3Rpb24odGhpcywgJ1Jlc291cmNlSGFuZGxlcicsIHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnY2x1c3Rlci1yZXNvdXJjZScpKSxcbiAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLlBZVEhPTl8zXzcsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICB0aW1lb3V0OiBEdXJhdGlvbi5taW51dGVzKDE1KSxcbiAgICAgIG1lbW9yeVNpemU6IDUxMixcbiAgICAgIGxheWVyczogW0t1YmVjdGxMYXllci5nZXRPckNyZWF0ZSh0aGlzKV0sXG4gICAgfSk7XG5cbiAgICBpZiAoIXByb3BzLnJvbGVBcm4pIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignXCJyb2xlQXJuXCIgaXMgcmVxdWlyZWQnKTtcbiAgICB9XG5cbiAgICAvLyBzaW5jZSB3ZSBkb24ndCBrbm93IHRoZSBjbHVzdGVyIG5hbWUgYXQgdGhpcyBwb2ludCwgd2UgbXVzdCBnaXZlIHRoaXMgcm9sZSBzdGFyIHJlc291cmNlIHBlcm1pc3Npb25zXG4gICAgaGFuZGxlci5hZGRUb1JvbGVQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgYWN0aW9uczogWydla3M6Q3JlYXRlQ2x1c3RlcicsICdla3M6RGVzY3JpYmVDbHVzdGVyJywgJ2VrczpEZWxldGVDbHVzdGVyJywgJ2VrczpVcGRhdGVDbHVzdGVyVmVyc2lvbiddLFxuICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICB9KSk7XG5cbiAgICAvLyB0aGUgQ3JlYXRlQ2x1c3RlciBBUEkgd2lsbCBhbGxvdyB0aGUgY2x1c3RlciB0byBhc3N1bWUgdGhpcyByb2xlLCBzbyB3ZVxuICAgIC8vIG5lZWQgdG8gYWxsb3cgdGhlIGxhbWJkYSBleGVjdXRpb24gcm9sZSB0byBwYXNzIGl0LlxuICAgIGhhbmRsZXIuYWRkVG9Sb2xlUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFsnaWFtOlBhc3NSb2xlJ10sXG4gICAgICByZXNvdXJjZXM6IFtwcm9wcy5yb2xlQXJuXSxcbiAgICB9KSk7XG5cbiAgICBjb25zdCByZXNvdXJjZSA9IG5ldyBDdXN0b21SZXNvdXJjZSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICByZXNvdXJjZVR5cGU6IENsdXN0ZXJSZXNvdXJjZS5SRVNPVVJDRV9UWVBFLFxuICAgICAgc2VydmljZVRva2VuOiBoYW5kbGVyLmZ1bmN0aW9uQXJuLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICBDb25maWc6IHByb3BzLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIHRoaXMucmVmID0gcmVzb3VyY2UucmVmO1xuICAgIHRoaXMuYXR0ckVuZHBvaW50ID0gVG9rZW4uYXNTdHJpbmcocmVzb3VyY2UuZ2V0QXR0KCdFbmRwb2ludCcpKTtcbiAgICB0aGlzLmF0dHJBcm4gPSBUb2tlbi5hc1N0cmluZyhyZXNvdXJjZS5nZXRBdHQoJ0FybicpKTtcbiAgICB0aGlzLmF0dHJDZXJ0aWZpY2F0ZUF1dGhvcml0eURhdGEgPSBUb2tlbi5hc1N0cmluZyhyZXNvdXJjZS5nZXRBdHQoJ0NlcnRpZmljYXRlQXV0aG9yaXR5RGF0YScpKTtcbiAgICB0aGlzLmNyZWF0aW9uUm9sZSA9IGhhbmRsZXIucm9sZSE7XG4gIH1cbn1cbiJdfQ==