"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileSystem = exports.ThroughputMode = exports.PerformanceMode = exports.OutOfInfrequentAccessPolicy = exports.LifecyclePolicy = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports
const core_2 = require("../../core");
const cxapi = require("../../cx-api");
const access_point_1 = require("./access-point");
const efs_generated_1 = require("./efs.generated");
/**
 * EFS Lifecycle Policy, if a file is not accessed for given days, it will move to EFS Infrequent Access.
 *
 * @see http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html#cfn-elasticfilesystem-filesystem-lifecyclepolicies
 */
var LifecyclePolicy;
(function (LifecyclePolicy) {
    /**
     * After 7 days of not being accessed.
     */
    LifecyclePolicy["AFTER_7_DAYS"] = "AFTER_7_DAYS";
    /**
     * After 14 days of not being accessed.
     */
    LifecyclePolicy["AFTER_14_DAYS"] = "AFTER_14_DAYS";
    /**
     * After 30 days of not being accessed.
     */
    LifecyclePolicy["AFTER_30_DAYS"] = "AFTER_30_DAYS";
    /**
     * After 60 days of not being accessed.
     */
    LifecyclePolicy["AFTER_60_DAYS"] = "AFTER_60_DAYS";
    /**
     * After 90 days of not being accessed.
     */
    LifecyclePolicy["AFTER_90_DAYS"] = "AFTER_90_DAYS";
})(LifecyclePolicy = exports.LifecyclePolicy || (exports.LifecyclePolicy = {}));
/**
 * EFS Out Of Infrequent Access Policy, if a file is accessed given times, it will move back to primary
 * storage class.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-efs-filesystem-lifecyclepolicy.html#cfn-efs-filesystem-lifecyclepolicy-transitiontoprimarystorageclass
 */
var OutOfInfrequentAccessPolicy;
(function (OutOfInfrequentAccessPolicy) {
    /**
     * After 1 access
     */
    OutOfInfrequentAccessPolicy["AFTER_1_ACCESS"] = "AFTER_1_ACCESS";
})(OutOfInfrequentAccessPolicy = exports.OutOfInfrequentAccessPolicy || (exports.OutOfInfrequentAccessPolicy = {}));
/**
 * EFS Performance mode.
 *
 * @see https://docs.aws.amazon.com/efs/latest/ug/performance.html#performancemodes
 */
var PerformanceMode;
(function (PerformanceMode) {
    /**
     * General Purpose is ideal for latency-sensitive use cases, like web serving
     * environments, content management systems, home directories, and general file serving.
     * Recommended for the majority of Amazon EFS file systems.
     */
    PerformanceMode["GENERAL_PURPOSE"] = "generalPurpose";
    /**
     * File systems in the Max I/O mode can scale to higher levels of aggregate
     * throughput and operations per second. This scaling is done with a tradeoff
     * of slightly higher latencies for file metadata operations.
     * Highly parallelized applications and workloads, such as big data analysis,
     * media processing, and genomics analysis, can benefit from this mode.
     */
    PerformanceMode["MAX_IO"] = "maxIO";
})(PerformanceMode = exports.PerformanceMode || (exports.PerformanceMode = {}));
/**
 * EFS Throughput mode.
 *
 * @see https://docs.aws.amazon.com/efs/latest/ug/performance.html#throughput-modes
 */
var ThroughputMode;
(function (ThroughputMode) {
    /**
     * This mode on Amazon EFS scales as the size of the file system in the standard storage class grows.
     */
    ThroughputMode["BURSTING"] = "bursting";
    /**
     * This mode can instantly provision the throughput of the file system (in MiB/s) independent of the amount of data stored.
     */
    ThroughputMode["PROVISIONED"] = "provisioned";
})(ThroughputMode = exports.ThroughputMode || (exports.ThroughputMode = {}));
class FileSystemBase extends core_1.Resource {
    /**
     * Grant the actions defined in actions to the given grantee
     * on this File System resource.
     *
     * @param grantee Principal to grant right to
     * @param actions The actions to grant
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee: grantee,
            actions: actions,
            resourceArns: [this.fileSystemArn],
        });
    }
}
/**
 * The Elastic File System implementation of IFileSystem.
 * It creates a new, empty file system in Amazon Elastic File System (Amazon EFS).
 * It also creates mount target (AWS::EFS::MountTarget) implicitly to mount the
 * EFS file system on an Amazon Elastic Compute Cloud (Amazon EC2) instance or another resource.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-efs-filesystem.html
 *
 * @resource AWS::EFS::FileSystem
 */
class FileSystem extends FileSystemBase {
    /**
     * Constructor for creating a new EFS FileSystem.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this._mountTargetsAvailable = new core_1.ConcreteDependable();
        try {
            jsiiDeprecationWarnings.monocdk_aws_efs_FileSystemProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, FileSystem);
            }
            throw error;
        }
        if (props.throughputMode === ThroughputMode.PROVISIONED && props.provisionedThroughputPerSecond === undefined) {
            throw new Error('Property provisionedThroughputPerSecond is required when throughputMode is PROVISIONED');
        }
        // we explictly use 'undefined' to represent 'false' to maintain backwards compatibility since
        // its considered an actual change in CloudFormations eyes, even though they have the same meaning.
        const encrypted = props.encrypted ?? (core_2.FeatureFlags.of(this).isEnabled(cxapi.EFS_DEFAULT_ENCRYPTION_AT_REST) ? true : undefined);
        // LifecyclePolicies is an array of lists containing a single policy
        let lifecyclePolicies = [];
        if (props.lifecyclePolicy) {
            lifecyclePolicies.push({ transitionToIa: props.lifecyclePolicy });
        }
        if (props.outOfInfrequentAccessPolicy) {
            lifecyclePolicies.push({ transitionToPrimaryStorageClass: props.outOfInfrequentAccessPolicy });
        }
        const filesystem = new efs_generated_1.CfnFileSystem(this, 'Resource', {
            encrypted: encrypted,
            kmsKeyId: props.kmsKey?.keyArn,
            lifecyclePolicies: lifecyclePolicies.length > 0 ? lifecyclePolicies : undefined,
            performanceMode: props.performanceMode,
            throughputMode: props.throughputMode,
            provisionedThroughputInMibps: props.provisionedThroughputPerSecond?.toMebibytes(),
            backupPolicy: props.enableAutomaticBackups ? { status: 'ENABLED' } : undefined,
        });
        filesystem.applyRemovalPolicy(props.removalPolicy);
        this.fileSystemId = filesystem.ref;
        this.fileSystemArn = filesystem.attrArn;
        core_1.Tags.of(this).add('Name', props.fileSystemName || this.node.path);
        const securityGroup = (props.securityGroup || new ec2.SecurityGroup(this, 'EfsSecurityGroup', {
            vpc: props.vpc,
        }));
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort: ec2.Port.tcp(FileSystem.DEFAULT_PORT),
        });
        const subnets = props.vpc.selectSubnets(props.vpcSubnets ?? { onePerAz: true });
        // We now have to create the mount target for each of the mentioned subnet
        let mountTargetCount = 0;
        this.mountTargetsAvailable = [];
        subnets.subnetIds.forEach((subnetId) => {
            const mountTarget = new efs_generated_1.CfnMountTarget(this, 'EfsMountTarget' + (++mountTargetCount), {
                fileSystemId: this.fileSystemId,
                securityGroups: Array.of(securityGroup.securityGroupId),
                subnetId,
            });
            this._mountTargetsAvailable.add(mountTarget);
        });
        this.mountTargetsAvailable = this._mountTargetsAvailable;
    }
    /**
     * Import an existing File System from the given properties.
     */
    static fromFileSystemAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_efs_FileSystemAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromFileSystemAttributes);
            }
            throw error;
        }
        return new ImportedFileSystem(scope, id, attrs);
    }
    /**
     * create access point from this filesystem
     */
    addAccessPoint(id, accessPointOptions = {}) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_efs_AccessPointOptions(accessPointOptions);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addAccessPoint);
            }
            throw error;
        }
        return new access_point_1.AccessPoint(this, id, {
            fileSystem: this,
            ...accessPointOptions,
        });
    }
}
exports.FileSystem = FileSystem;
_a = JSII_RTTI_SYMBOL_1;
FileSystem[_a] = { fqn: "monocdk.aws_efs.FileSystem", version: "1.185.0" };
/**
 * The default port File System listens on.
 */
FileSystem.DEFAULT_PORT = 2049;
class ImportedFileSystem extends FileSystemBase {
    constructor(scope, id, attrs) {
        super(scope, id);
        if (!!attrs.fileSystemId === !!attrs.fileSystemArn) {
            throw new Error('One of fileSystemId or fileSystemArn, but not both, must be provided.');
        }
        this.fileSystemArn = attrs.fileSystemArn ?? core_1.Stack.of(scope).formatArn({
            service: 'elasticfilesystem',
            resource: 'file-system',
            resourceName: attrs.fileSystemId,
        });
        const parsedArn = core_1.Stack.of(scope).splitArn(this.fileSystemArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        if (!parsedArn.resourceName) {
            throw new Error(`Invalid FileSystem Arn ${this.fileSystemArn}`);
        }
        this.fileSystemId = attrs.fileSystemId ?? parsedArn.resourceName;
        this.connections = new ec2.Connections({
            securityGroups: [attrs.securityGroup],
            defaultPort: ec2.Port.tcp(FileSystem.DEFAULT_PORT),
        });
        this.mountTargetsAvailable = new core_1.ConcreteDependable();
    }
}
//# sourceMappingURL=data:application/json;base64,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