"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Protocol = exports.ContainerDependencyCondition = exports.UlimitName = exports.ContainerDefinition = exports.Secret = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("../../core");
const task_definition_1 = require("./base/task-definition");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("../../core");
/**
 * A secret environment variable.
 */
class Secret {
    /**
     * Creates an environment variable value from a parameter stored in AWS
     * Systems Manager Parameter Store.
     */
    static fromSsmParameter(parameter) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ssm_IParameter(parameter);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromSsmParameter);
            }
            throw error;
        }
        return {
            arn: parameter.parameterArn,
            grantRead: grantee => parameter.grantRead(grantee),
        };
    }
    /**
     * Creates a environment variable value from a secret stored in AWS Secrets
     * Manager.
     *
     * @param secret the secret stored in AWS Secrets Manager
     * @param field the name of the field with the value that you want to set as
     * the environment variable value. Only values in JSON format are supported.
     * If you do not specify a JSON field, then the full content of the secret is
     * used.
     */
    static fromSecretsManager(secret, field) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_secretsmanager_ISecret(secret);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromSecretsManager);
            }
            throw error;
        }
        return {
            arn: field ? `${secret.secretArn}:${field}::` : secret.secretArn,
            hasField: !!field,
            grantRead: grantee => secret.grantRead(grantee),
        };
    }
    /**
     * Creates a environment variable value from a secret stored in AWS Secrets
     * Manager.
     *
     * @param secret the secret stored in AWS Secrets Manager
     * @param versionInfo the version information to reference the secret
     * @param field the name of the field with the value that you want to set as
     * the environment variable value. Only values in JSON format are supported.
     * If you do not specify a JSON field, then the full content of the secret is
     * used.
     */
    static fromSecretsManagerVersion(secret, versionInfo, field) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_secretsmanager_ISecret(secret);
            jsiiDeprecationWarnings.monocdk_aws_ecs_SecretVersionInfo(versionInfo);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromSecretsManagerVersion);
            }
            throw error;
        }
        return {
            arn: `${secret.secretArn}:${field ?? ''}:${versionInfo.versionStage ?? ''}:${versionInfo.versionId ?? ''}`,
            hasField: !!field,
            grantRead: grantee => secret.grantRead(grantee),
        };
    }
}
exports.Secret = Secret;
_a = JSII_RTTI_SYMBOL_1;
Secret[_a] = { fqn: "monocdk.aws_ecs.Secret", version: "1.185.0" };
/**
 * A container definition is used in a task definition to describe the containers that are launched as part of a task.
 */
class ContainerDefinition extends core_1.Construct {
    /**
     * Constructs a new instance of the ContainerDefinition class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        /**
         * The mount points for data volumes in your container.
         */
        this.mountPoints = new Array();
        /**
         * The list of port mappings for the container. Port mappings allow containers to access ports
         * on the host container instance to send or receive traffic.
         */
        this.portMappings = new Array();
        /**
         * The data volumes to mount from another container in the same task definition.
         */
        this.volumesFrom = new Array();
        /**
         * An array of ulimits to set in the container.
         */
        this.ulimits = new Array();
        /**
         * An array dependencies defined for container startup and shutdown.
         */
        this.containerDependencies = new Array();
        /**
         * The inference accelerators referenced by this container.
         */
        this.inferenceAcceleratorResources = [];
        /**
         * The configured container links
         */
        this.links = new Array();
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_ContainerDefinitionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, ContainerDefinition);
            }
            throw error;
        }
        if (props.memoryLimitMiB !== undefined && props.memoryReservationMiB !== undefined) {
            if (props.memoryLimitMiB < props.memoryReservationMiB) {
                throw new Error('MemoryLimitMiB should not be less than MemoryReservationMiB.');
            }
        }
        this.essential = props.essential ?? true;
        this.taskDefinition = props.taskDefinition;
        this.memoryLimitSpecified = props.memoryLimitMiB !== undefined || props.memoryReservationMiB !== undefined;
        this.linuxParameters = props.linuxParameters;
        this.containerName = props.containerName ?? this.node.id;
        this.imageConfig = props.image.bind(this, this);
        this.imageName = this.imageConfig.imageName;
        if (props.logging) {
            this.logDriverConfig = props.logging.bind(this, this);
        }
        if (props.secrets) {
            this.secrets = [];
            for (const [name, secret] of Object.entries(props.secrets)) {
                if (secret.hasField) {
                    this.referencesSecretJsonField = true;
                }
                secret.grantRead(this.taskDefinition.obtainExecutionRole());
                this.secrets.push({
                    name,
                    valueFrom: secret.arn,
                });
            }
        }
        if (props.environment) {
            this.environment = { ...props.environment };
        }
        else {
            this.environment = {};
        }
        if (props.environmentFiles) {
            this.environmentFiles = [];
            for (const environmentFile of props.environmentFiles) {
                this.environmentFiles.push(environmentFile.bind(this));
            }
        }
        props.taskDefinition._linkContainer(this);
        if (props.portMappings) {
            this.addPortMappings(...props.portMappings);
        }
        if (props.inferenceAcceleratorResources) {
            this.addInferenceAcceleratorResource(...props.inferenceAcceleratorResources);
        }
    }
    /**
     * This method adds a link which allows containers to communicate with each other without the need for port mappings.
     *
     * This parameter is only supported if the task definition is using the bridge network mode.
     * Warning: The --link flag is a legacy feature of Docker. It may eventually be removed.
     */
    addLink(container, alias) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_ContainerDefinition(container);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addLink);
            }
            throw error;
        }
        if (this.taskDefinition.networkMode !== task_definition_1.NetworkMode.BRIDGE) {
            throw new Error('You must use network mode Bridge to add container links.');
        }
        if (alias !== undefined) {
            this.links.push(`${container.containerName}:${alias}`);
        }
        else {
            this.links.push(`${container.containerName}`);
        }
    }
    /**
     * This method adds one or more mount points for data volumes to the container.
     */
    addMountPoints(...mountPoints) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_MountPoint(mountPoints);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addMountPoints);
            }
            throw error;
        }
        this.mountPoints.push(...mountPoints);
    }
    /**
     * This method mounts temporary disk space to the container.
     *
     * This adds the correct container mountPoint and task definition volume.
     */
    addScratch(scratch) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_ScratchSpace(scratch);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addScratch);
            }
            throw error;
        }
        const mountPoint = {
            containerPath: scratch.containerPath,
            readOnly: scratch.readOnly,
            sourceVolume: scratch.name,
        };
        const volume = {
            host: {
                sourcePath: scratch.sourcePath,
            },
            name: scratch.name,
        };
        this.taskDefinition.addVolume(volume);
        this.addMountPoints(mountPoint);
    }
    /**
     * This method adds one or more port mappings to the container.
     */
    addPortMappings(...portMappings) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_PortMapping(portMappings);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addPortMappings);
            }
            throw error;
        }
        this.portMappings.push(...portMappings.map(pm => {
            if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.AWS_VPC || this.taskDefinition.networkMode === task_definition_1.NetworkMode.HOST) {
                if (pm.containerPort !== pm.hostPort && pm.hostPort !== undefined) {
                    throw new Error(`Host port (${pm.hostPort}) must be left out or equal to container port ${pm.containerPort} for network mode ${this.taskDefinition.networkMode}`);
                }
            }
            if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.BRIDGE) {
                if (pm.hostPort === undefined) {
                    pm = {
                        ...pm,
                        hostPort: 0,
                    };
                }
            }
            return pm;
        }));
    }
    /**
     * This method adds an environment variable to the container.
     */
    addEnvironment(name, value) {
        this.environment[name] = value;
    }
    /**
     * This method adds one or more resources to the container.
     */
    addInferenceAcceleratorResource(...inferenceAcceleratorResources) {
        this.inferenceAcceleratorResources.push(...inferenceAcceleratorResources.map(resource => {
            for (const inferenceAccelerator of this.taskDefinition.inferenceAccelerators) {
                if (resource === inferenceAccelerator.deviceName) {
                    return resource;
                }
            }
            throw new Error(`Resource value ${resource} in container definition doesn't match any inference accelerator device name in the task definition.`);
        }));
    }
    /**
     * This method adds one or more ulimits to the container.
     */
    addUlimits(...ulimits) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_Ulimit(ulimits);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addUlimits);
            }
            throw error;
        }
        this.ulimits.push(...ulimits);
    }
    /**
     * This method adds one or more container dependencies to the container.
     */
    addContainerDependencies(...containerDependencies) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_ContainerDependency(containerDependencies);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addContainerDependencies);
            }
            throw error;
        }
        this.containerDependencies.push(...containerDependencies);
    }
    /**
     * This method adds one or more volumes to the container.
     */
    addVolumesFrom(...volumesFrom) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_VolumeFrom(volumesFrom);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addVolumesFrom);
            }
            throw error;
        }
        this.volumesFrom.push(...volumesFrom);
    }
    /**
     * This method adds the specified statement to the IAM task execution policy in the task definition.
     */
    addToExecutionPolicy(statement) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_iam_PolicyStatement(statement);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToExecutionPolicy);
            }
            throw error;
        }
        this.taskDefinition.addToExecutionRolePolicy(statement);
    }
    /**
     * Returns the host port for the requested container port if it exists
     */
    findPortMapping(containerPort, protocol) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_Protocol(protocol);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.findPortMapping);
            }
            throw error;
        }
        for (const portMapping of this.portMappings) {
            const p = portMapping.protocol || Protocol.TCP;
            const c = portMapping.containerPort;
            if (c === containerPort && p === protocol) {
                return portMapping;
            }
        }
        return undefined;
    }
    /**
     * The inbound rules associated with the security group the task or service will use.
     *
     * This property is only used for tasks that use the awsvpc network mode.
     */
    get ingressPort() {
        if (this.portMappings.length === 0) {
            throw new Error(`Container ${this.containerName} hasn't defined any ports. Call addPortMappings().`);
        }
        const defaultPortMapping = this.portMappings[0];
        if (defaultPortMapping.hostPort !== undefined && defaultPortMapping.hostPort !== 0) {
            return defaultPortMapping.hostPort;
        }
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.BRIDGE) {
            return 0;
        }
        return defaultPortMapping.containerPort;
    }
    /**
     * The port the container will listen on.
     */
    get containerPort() {
        if (this.portMappings.length === 0) {
            throw new Error(`Container ${this.containerName} hasn't defined any ports. Call addPortMappings().`);
        }
        const defaultPortMapping = this.portMappings[0];
        return defaultPortMapping.containerPort;
    }
    /**
     * Render this container definition to a CloudFormation object
     *
     * @param _taskDefinition [disable-awslint:ref-via-interface] (unused but kept to avoid breaking change)
     */
    renderContainerDefinition(_taskDefinition) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_TaskDefinition(_taskDefinition);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.renderContainerDefinition);
            }
            throw error;
        }
        return {
            command: this.props.command,
            cpu: this.props.cpu,
            disableNetworking: this.props.disableNetworking,
            dependsOn: cdk.Lazy.any({ produce: () => this.containerDependencies.map(renderContainerDependency) }, { omitEmptyArray: true }),
            dnsSearchDomains: this.props.dnsSearchDomains,
            dnsServers: this.props.dnsServers,
            dockerLabels: this.props.dockerLabels,
            dockerSecurityOptions: this.props.dockerSecurityOptions,
            entryPoint: this.props.entryPoint,
            essential: this.essential,
            hostname: this.props.hostname,
            image: this.imageConfig.imageName,
            memory: this.props.memoryLimitMiB,
            memoryReservation: this.props.memoryReservationMiB,
            mountPoints: cdk.Lazy.any({ produce: () => this.mountPoints.map(renderMountPoint) }, { omitEmptyArray: true }),
            name: this.containerName,
            portMappings: cdk.Lazy.any({ produce: () => this.portMappings.map(renderPortMapping) }, { omitEmptyArray: true }),
            privileged: this.props.privileged,
            readonlyRootFilesystem: this.props.readonlyRootFilesystem,
            repositoryCredentials: this.imageConfig.repositoryCredentials,
            startTimeout: this.props.startTimeout && this.props.startTimeout.toSeconds(),
            stopTimeout: this.props.stopTimeout && this.props.stopTimeout.toSeconds(),
            ulimits: cdk.Lazy.any({ produce: () => this.ulimits.map(renderUlimit) }, { omitEmptyArray: true }),
            user: this.props.user,
            volumesFrom: cdk.Lazy.any({ produce: () => this.volumesFrom.map(renderVolumeFrom) }, { omitEmptyArray: true }),
            workingDirectory: this.props.workingDirectory,
            logConfiguration: this.logDriverConfig,
            environment: this.environment && Object.keys(this.environment).length ? renderKV(this.environment, 'name', 'value') : undefined,
            environmentFiles: this.environmentFiles && renderEnvironmentFiles(cdk.Stack.of(this).partition, this.environmentFiles),
            secrets: this.secrets,
            extraHosts: this.props.extraHosts && renderKV(this.props.extraHosts, 'hostname', 'ipAddress'),
            healthCheck: this.props.healthCheck && renderHealthCheck(this.props.healthCheck),
            links: cdk.Lazy.list({ produce: () => this.links }, { omitEmpty: true }),
            linuxParameters: this.linuxParameters && this.linuxParameters.renderLinuxParameters(),
            resourceRequirements: (!this.props.gpuCount && this.inferenceAcceleratorResources.length == 0) ? undefined :
                renderResourceRequirements(this.props.gpuCount, this.inferenceAcceleratorResources),
            systemControls: this.props.systemControls && renderSystemControls(this.props.systemControls),
        };
    }
}
exports.ContainerDefinition = ContainerDefinition;
_b = JSII_RTTI_SYMBOL_1;
ContainerDefinition[_b] = { fqn: "monocdk.aws_ecs.ContainerDefinition", version: "1.185.0" };
function renderKV(env, keyName, valueName) {
    const ret = [];
    for (const [key, value] of Object.entries(env)) {
        ret.push({ [keyName]: key, [valueName]: value });
    }
    return ret;
}
function renderEnvironmentFiles(partition, environmentFiles) {
    const ret = [];
    for (const environmentFile of environmentFiles) {
        const s3Location = environmentFile.s3Location;
        if (!s3Location) {
            throw Error('Environment file must specify an S3 location');
        }
        ret.push({
            type: environmentFile.fileType,
            value: `arn:${partition}:s3:::${s3Location.bucketName}/${s3Location.objectKey}`,
        });
    }
    return ret;
}
function renderHealthCheck(hc) {
    return {
        command: getHealthCheckCommand(hc),
        interval: hc.interval?.toSeconds() ?? 30,
        retries: hc.retries ?? 3,
        startPeriod: hc.startPeriod?.toSeconds(),
        timeout: hc.timeout?.toSeconds() ?? 5,
    };
}
function getHealthCheckCommand(hc) {
    const cmd = hc.command;
    const hcCommand = new Array();
    if (cmd.length === 0) {
        throw new Error('At least one argument must be supplied for health check command.');
    }
    if (cmd.length === 1) {
        hcCommand.push('CMD-SHELL', cmd[0]);
        return hcCommand;
    }
    if (cmd[0] !== 'CMD' && cmd[0] !== 'CMD-SHELL') {
        hcCommand.push('CMD');
    }
    return hcCommand.concat(cmd);
}
function renderResourceRequirements(gpuCount = 0, inferenceAcceleratorResources = []) {
    const ret = [];
    for (const resource of inferenceAcceleratorResources) {
        ret.push({
            type: 'InferenceAccelerator',
            value: resource,
        });
    }
    if (gpuCount > 0) {
        ret.push({
            type: 'GPU',
            value: gpuCount.toString(),
        });
    }
    return ret;
}
/**
 * Type of resource to set a limit on
 */
var UlimitName;
(function (UlimitName) {
    UlimitName["CORE"] = "core";
    UlimitName["CPU"] = "cpu";
    UlimitName["DATA"] = "data";
    UlimitName["FSIZE"] = "fsize";
    UlimitName["LOCKS"] = "locks";
    UlimitName["MEMLOCK"] = "memlock";
    UlimitName["MSGQUEUE"] = "msgqueue";
    UlimitName["NICE"] = "nice";
    UlimitName["NOFILE"] = "nofile";
    UlimitName["NPROC"] = "nproc";
    UlimitName["RSS"] = "rss";
    UlimitName["RTPRIO"] = "rtprio";
    UlimitName["RTTIME"] = "rttime";
    UlimitName["SIGPENDING"] = "sigpending";
    UlimitName["STACK"] = "stack";
})(UlimitName = exports.UlimitName || (exports.UlimitName = {}));
function renderUlimit(ulimit) {
    return {
        name: ulimit.name,
        softLimit: ulimit.softLimit,
        hardLimit: ulimit.hardLimit,
    };
}
var ContainerDependencyCondition;
(function (ContainerDependencyCondition) {
    /**
     * This condition emulates the behavior of links and volumes today.
     * It validates that a dependent container is started before permitting other containers to start.
     */
    ContainerDependencyCondition["START"] = "START";
    /**
     * This condition validates that a dependent container runs to completion (exits) before permitting other containers to start.
     * This can be useful for nonessential containers that run a script and then exit.
     */
    ContainerDependencyCondition["COMPLETE"] = "COMPLETE";
    /**
     * This condition is the same as COMPLETE, but it also requires that the container exits with a zero status.
     */
    ContainerDependencyCondition["SUCCESS"] = "SUCCESS";
    /**
     * This condition validates that the dependent container passes its Docker health check before permitting other containers to start.
     * This requires that the dependent container has health checks configured. This condition is confirmed only at task startup.
     */
    ContainerDependencyCondition["HEALTHY"] = "HEALTHY";
})(ContainerDependencyCondition = exports.ContainerDependencyCondition || (exports.ContainerDependencyCondition = {}));
function renderContainerDependency(containerDependency) {
    return {
        containerName: containerDependency.container.containerName,
        condition: containerDependency.condition || ContainerDependencyCondition.HEALTHY,
    };
}
/**
 * Network protocol
 */
var Protocol;
(function (Protocol) {
    /**
     * TCP
     */
    Protocol["TCP"] = "tcp";
    /**
     * UDP
     */
    Protocol["UDP"] = "udp";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
function renderPortMapping(pm) {
    return {
        containerPort: pm.containerPort,
        hostPort: pm.hostPort,
        protocol: pm.protocol || Protocol.TCP,
    };
}
function renderMountPoint(mp) {
    return {
        containerPath: mp.containerPath,
        readOnly: mp.readOnly,
        sourceVolume: mp.sourceVolume,
    };
}
function renderVolumeFrom(vf) {
    return {
        sourceContainer: vf.sourceContainer,
        readOnly: vf.readOnly,
    };
}
function renderSystemControls(systemControls) {
    return systemControls.map(sc => ({
        namespace: sc.namespace,
        value: sc.value,
    }));
}
//# sourceMappingURL=data:application/json;base64,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