"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkLoadBalancedServiceBase = exports.NetworkLoadBalancedServiceRecordType = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ecs_1 = require("../../../aws-ecs");
const aws_elasticloadbalancingv2_1 = require("../../../aws-elasticloadbalancingv2");
const aws_route53_1 = require("../../../aws-route53");
const aws_route53_targets_1 = require("../../../aws-route53-targets");
const cdk = require("../../../core");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("../../../core");
/**
 * Describes the type of DNS record the service should create
 */
var NetworkLoadBalancedServiceRecordType;
(function (NetworkLoadBalancedServiceRecordType) {
    /**
     * Create Route53 A Alias record
     */
    NetworkLoadBalancedServiceRecordType[NetworkLoadBalancedServiceRecordType["ALIAS"] = 0] = "ALIAS";
    /**
     * Create a CNAME record
     */
    NetworkLoadBalancedServiceRecordType[NetworkLoadBalancedServiceRecordType["CNAME"] = 1] = "CNAME";
    /**
     * Do not create any DNS records
     */
    NetworkLoadBalancedServiceRecordType[NetworkLoadBalancedServiceRecordType["NONE"] = 2] = "NONE";
})(NetworkLoadBalancedServiceRecordType = exports.NetworkLoadBalancedServiceRecordType || (exports.NetworkLoadBalancedServiceRecordType = {}));
/**
 * The base class for NetworkLoadBalancedEc2Service and NetworkLoadBalancedFargateService services.
 */
class NetworkLoadBalancedServiceBase extends core_1.Construct {
    /**
     * Constructs a new instance of the NetworkLoadBalancedServiceBase class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_patterns_NetworkLoadBalancedServiceBaseProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, NetworkLoadBalancedServiceBase);
            }
            throw error;
        }
        if (props.cluster && props.vpc) {
            throw new Error('You can only specify either vpc or cluster. Alternatively, you can leave both blank');
        }
        this.cluster = props.cluster || this.getDefaultCluster(this, props.vpc);
        if (props.desiredCount !== undefined && props.desiredCount < 1) {
            throw new Error('You must specify a desiredCount greater than 0');
        }
        this.desiredCount = props.desiredCount || 1;
        this.internalDesiredCount = props.desiredCount;
        const internetFacing = props.publicLoadBalancer ?? true;
        const lbProps = {
            vpc: this.cluster.vpc,
            internetFacing,
        };
        const loadBalancer = props.loadBalancer ?? new aws_elasticloadbalancingv2_1.NetworkLoadBalancer(this, 'LB', lbProps);
        const listenerPort = props.listenerPort ?? 80;
        const targetProps = {
            port: props.taskImageOptions?.containerPort ?? 80,
        };
        this.listener = loadBalancer.addListener('PublicListener', { port: listenerPort });
        this.targetGroup = this.listener.addTargets('ECS', targetProps);
        if (typeof props.domainName !== 'undefined') {
            if (typeof props.domainZone === 'undefined') {
                throw new Error('A Route53 hosted domain zone name is required to configure the specified domain name');
            }
            switch (props.recordType ?? NetworkLoadBalancedServiceRecordType.ALIAS) {
                case NetworkLoadBalancedServiceRecordType.ALIAS:
                    new aws_route53_1.ARecord(this, 'DNS', {
                        zone: props.domainZone,
                        recordName: props.domainName,
                        target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.LoadBalancerTarget(loadBalancer)),
                    });
                    break;
                case NetworkLoadBalancedServiceRecordType.CNAME:
                    new aws_route53_1.CnameRecord(this, 'DNS', {
                        zone: props.domainZone,
                        recordName: props.domainName,
                        domainName: loadBalancer.loadBalancerDnsName,
                    });
                    break;
                case NetworkLoadBalancedServiceRecordType.NONE:
                    // Do not create a DNS record
                    break;
            }
        }
        if (loadBalancer instanceof aws_elasticloadbalancingv2_1.NetworkLoadBalancer) {
            this._networkLoadBalancer = loadBalancer;
        }
        if (props.loadBalancer === undefined) {
            new cdk.CfnOutput(this, 'LoadBalancerDNS', { value: this.loadBalancer.loadBalancerDnsName });
        }
    }
    /**
     * The Network Load Balancer for the service.
     */
    get loadBalancer() {
        if (!this._networkLoadBalancer) {
            throw new Error('.loadBalancer can only be accessed if the class was constructed with an owned, not imported, load balancer');
        }
        return this._networkLoadBalancer;
    }
    /**
     * Returns the default cluster.
     */
    getDefaultCluster(scope, vpc) {
        try {
            jsiiDeprecationWarnings.monocdk_Construct(scope);
            jsiiDeprecationWarnings.monocdk_aws_ec2_IVpc(vpc);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.getDefaultCluster);
            }
            throw error;
        }
        // magic string to avoid collision with user-defined constructs
        const DEFAULT_CLUSTER_ID = `EcsDefaultClusterMnL3mNNYN${vpc ? vpc.node.id : ''}`;
        const stack = cdk.Stack.of(scope);
        return stack.node.tryFindChild(DEFAULT_CLUSTER_ID) || new aws_ecs_1.Cluster(stack, DEFAULT_CLUSTER_ID, { vpc });
    }
    /**
     * Adds service as a target of the target group.
     */
    addServiceAsTarget(service) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ecs_BaseService(service);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addServiceAsTarget);
            }
            throw error;
        }
        this.targetGroup.addTarget(service);
    }
    createAWSLogDriver(prefix) {
        return new aws_ecs_1.AwsLogDriver({ streamPrefix: prefix });
    }
}
exports.NetworkLoadBalancedServiceBase = NetworkLoadBalancedServiceBase;
_a = JSII_RTTI_SYMBOL_1;
NetworkLoadBalancedServiceBase[_a] = { fqn: "monocdk.aws_ecs_patterns.NetworkLoadBalancedServiceBase", version: "1.185.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibmV0d29yay1sb2FkLWJhbGFuY2VkLXNlcnZpY2UtYmFzZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIm5ldHdvcmstbG9hZC1iYWxhbmNlZC1zZXJ2aWNlLWJhc2UudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBQ0EsOENBRzBCO0FBQzFCLG9GQUFxSTtBQUVySSxzREFBdUY7QUFDdkYsc0VBQWtFO0FBQ2xFLHFDQUFxQztBQUdyQyxpR0FBaUc7QUFDakcsOERBQThEO0FBQzlELHdDQUEyRDtBQUUzRDs7R0FFRztBQUNILElBQVksb0NBYVg7QUFiRCxXQUFZLG9DQUFvQztJQUM5Qzs7T0FFRztJQUNILGlHQUFLLENBQUE7SUFDTDs7T0FFRztJQUNILGlHQUFLLENBQUE7SUFDTDs7T0FFRztJQUNILCtGQUFJLENBQUE7QUFDTixDQUFDLEVBYlcsb0NBQW9DLEdBQXBDLDRDQUFvQyxLQUFwQyw0Q0FBb0MsUUFhL0M7QUFzUEQ7O0dBRUc7QUFDSCxNQUFzQiw4QkFBK0IsU0FBUSxnQkFBYTtJQXdDeEU7O09BRUc7SUFDSCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFFBQTZDLEVBQUU7UUFDdkYsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQzs7Ozs7OytDQTVDQyw4QkFBOEI7Ozs7UUE4Q2hELElBQUksS0FBSyxDQUFDLE9BQU8sSUFBSSxLQUFLLENBQUMsR0FBRyxFQUFFO1lBQzlCLE1BQU0sSUFBSSxLQUFLLENBQUMscUZBQXFGLENBQUMsQ0FBQztTQUN4RztRQUNELElBQUksQ0FBQyxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sSUFBSSxJQUFJLENBQUMsaUJBQWlCLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUV4RSxJQUFJLEtBQUssQ0FBQyxZQUFZLEtBQUssU0FBUyxJQUFJLEtBQUssQ0FBQyxZQUFZLEdBQUcsQ0FBQyxFQUFFO1lBQzlELE1BQU0sSUFBSSxLQUFLLENBQUMsZ0RBQWdELENBQUMsQ0FBQztTQUNuRTtRQUVELElBQUksQ0FBQyxZQUFZLEdBQUcsS0FBSyxDQUFDLFlBQVksSUFBSSxDQUFDLENBQUM7UUFDNUMsSUFBSSxDQUFDLG9CQUFvQixHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUM7UUFFL0MsTUFBTSxjQUFjLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixJQUFJLElBQUksQ0FBQztRQUV4RCxNQUFNLE9BQU8sR0FBRztZQUNkLEdBQUcsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUc7WUFDckIsY0FBYztTQUNmLENBQUM7UUFFRixNQUFNLFlBQVksR0FBRyxLQUFLLENBQUMsWUFBWSxJQUFJLElBQUksZ0RBQW1CLENBQUMsSUFBSSxFQUFFLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztRQUN4RixNQUFNLFlBQVksR0FBRyxLQUFLLENBQUMsWUFBWSxJQUFJLEVBQUUsQ0FBQztRQUM5QyxNQUFNLFdBQVcsR0FBRztZQUNsQixJQUFJLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixFQUFFLGFBQWEsSUFBSSxFQUFFO1NBQ2xELENBQUM7UUFFRixJQUFJLENBQUMsUUFBUSxHQUFHLFlBQVksQ0FBQyxXQUFXLENBQUMsZ0JBQWdCLEVBQUUsRUFBRSxJQUFJLEVBQUUsWUFBWSxFQUFFLENBQUMsQ0FBQztRQUNuRixJQUFJLENBQUMsV0FBVyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxXQUFXLENBQUMsQ0FBQztRQUVoRSxJQUFJLE9BQU8sS0FBSyxDQUFDLFVBQVUsS0FBSyxXQUFXLEVBQUU7WUFDM0MsSUFBSSxPQUFPLEtBQUssQ0FBQyxVQUFVLEtBQUssV0FBVyxFQUFFO2dCQUMzQyxNQUFNLElBQUksS0FBSyxDQUFDLHNGQUFzRixDQUFDLENBQUM7YUFDekc7WUFFRCxRQUFRLEtBQUssQ0FBQyxVQUFVLElBQUksb0NBQW9DLENBQUMsS0FBSyxFQUFFO2dCQUN0RSxLQUFLLG9DQUFvQyxDQUFDLEtBQUs7b0JBQzdDLElBQUkscUJBQU8sQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFO3dCQUN2QixJQUFJLEVBQUUsS0FBSyxDQUFDLFVBQVU7d0JBQ3RCLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTt3QkFDNUIsTUFBTSxFQUFFLDBCQUFZLENBQUMsU0FBUyxDQUFDLElBQUksd0NBQWtCLENBQUMsWUFBWSxDQUFDLENBQUM7cUJBQ3JFLENBQUMsQ0FBQztvQkFDSCxNQUFNO2dCQUNSLEtBQUssb0NBQW9DLENBQUMsS0FBSztvQkFDN0MsSUFBSSx5QkFBVyxDQUFDLElBQUksRUFBRSxLQUFLLEVBQUU7d0JBQzNCLElBQUksRUFBRSxLQUFLLENBQUMsVUFBVTt3QkFDdEIsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVO3dCQUM1QixVQUFVLEVBQUUsWUFBWSxDQUFDLG1CQUFtQjtxQkFDN0MsQ0FBQyxDQUFDO29CQUNILE1BQU07Z0JBQ1IsS0FBSyxvQ0FBb0MsQ0FBQyxJQUFJO29CQUM1Qyw2QkFBNkI7b0JBQzdCLE1BQU07YUFDVDtTQUNGO1FBRUQsSUFBSSxZQUFZLFlBQVksZ0RBQW1CLEVBQUU7WUFDL0MsSUFBSSxDQUFDLG9CQUFvQixHQUFHLFlBQVksQ0FBQztTQUMxQztRQUVELElBQUksS0FBSyxDQUFDLFlBQVksS0FBSyxTQUFTLEVBQUU7WUFDcEMsSUFBSSxHQUFHLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxpQkFBaUIsRUFBRSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsWUFBWSxDQUFDLG1CQUFtQixFQUFFLENBQUMsQ0FBQztTQUM5RjtLQUNGO0lBN0ZEOztPQUVHO0lBQ0gsSUFBVyxZQUFZO1FBQ3JCLElBQUksQ0FBQyxJQUFJLENBQUMsb0JBQW9CLEVBQUU7WUFDOUIsTUFBTSxJQUFJLEtBQUssQ0FBQyw0R0FBNEcsQ0FBQyxDQUFDO1NBQy9IO1FBQ0QsT0FBTyxJQUFJLENBQUMsb0JBQW9CLENBQUM7S0FDbEM7SUF1RkQ7O09BRUc7SUFDTyxpQkFBaUIsQ0FBQyxLQUFvQixFQUFFLEdBQVU7Ozs7Ozs7Ozs7O1FBQzFELCtEQUErRDtRQUMvRCxNQUFNLGtCQUFrQixHQUFHLDZCQUE2QixHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQztRQUNqRixNQUFNLEtBQUssR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNsQyxPQUFPLEtBQUssQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLGtCQUFrQixDQUFZLElBQUksSUFBSSxpQkFBTyxDQUFDLEtBQUssRUFBRSxrQkFBa0IsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7S0FDbEg7SUFFRDs7T0FFRztJQUNPLGtCQUFrQixDQUFDLE9BQW9COzs7Ozs7Ozs7O1FBQy9DLElBQUksQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxDQUFDO0tBQ3JDO0lBRVMsa0JBQWtCLENBQUMsTUFBYztRQUN6QyxPQUFPLElBQUksc0JBQVksQ0FBQyxFQUFFLFlBQVksRUFBRSxNQUFNLEVBQUUsQ0FBQyxDQUFDO0tBQ25EOztBQWhJSCx3RUFpSUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBJVnBjIH0gZnJvbSAnLi4vLi4vLi4vYXdzLWVjMic7XG5pbXBvcnQge1xuICBBd3NMb2dEcml2ZXIsIEJhc2VTZXJ2aWNlLCBDbG91ZE1hcE9wdGlvbnMsIENsdXN0ZXIsIENvbnRhaW5lckltYWdlLCBEZXBsb3ltZW50Q29udHJvbGxlciwgRGVwbG95bWVudENpcmN1aXRCcmVha2VyLFxuICBJQ2x1c3RlciwgTG9nRHJpdmVyLCBQcm9wYWdhdGVkVGFnU291cmNlLCBTZWNyZXQsXG59IGZyb20gJy4uLy4uLy4uL2F3cy1lY3MnO1xuaW1wb3J0IHsgSU5ldHdvcmtMb2FkQmFsYW5jZXIsIE5ldHdvcmtMaXN0ZW5lciwgTmV0d29ya0xvYWRCYWxhbmNlciwgTmV0d29ya1RhcmdldEdyb3VwIH0gZnJvbSAnLi4vLi4vLi4vYXdzLWVsYXN0aWNsb2FkYmFsYW5jaW5ndjInO1xuaW1wb3J0IHsgSVJvbGUgfSBmcm9tICcuLi8uLi8uLi9hd3MtaWFtJztcbmltcG9ydCB7IEFSZWNvcmQsIENuYW1lUmVjb3JkLCBJSG9zdGVkWm9uZSwgUmVjb3JkVGFyZ2V0IH0gZnJvbSAnLi4vLi4vLi4vYXdzLXJvdXRlNTMnO1xuaW1wb3J0IHsgTG9hZEJhbGFuY2VyVGFyZ2V0IH0gZnJvbSAnLi4vLi4vLi4vYXdzLXJvdXRlNTMtdGFyZ2V0cyc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnLi4vLi4vLi4vY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcblxuLy8ga2VlcCB0aGlzIGltcG9ydCBzZXBhcmF0ZSBmcm9tIG90aGVyIGltcG9ydHMgdG8gcmVkdWNlIGNoYW5jZSBmb3IgbWVyZ2UgY29uZmxpY3RzIHdpdGggdjItbWFpblxuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLWR1cGxpY2F0ZS1pbXBvcnRzLCBpbXBvcnQvb3JkZXJcbmltcG9ydCB7IENvbnN0cnVjdCBhcyBDb3JlQ29uc3RydWN0IH0gZnJvbSAnLi4vLi4vLi4vY29yZSc7XG5cbi8qKlxuICogRGVzY3JpYmVzIHRoZSB0eXBlIG9mIEROUyByZWNvcmQgdGhlIHNlcnZpY2Ugc2hvdWxkIGNyZWF0ZVxuICovXG5leHBvcnQgZW51bSBOZXR3b3JrTG9hZEJhbGFuY2VkU2VydmljZVJlY29yZFR5cGUge1xuICAvKipcbiAgICogQ3JlYXRlIFJvdXRlNTMgQSBBbGlhcyByZWNvcmRcbiAgICovXG4gIEFMSUFTLFxuICAvKipcbiAgICogQ3JlYXRlIGEgQ05BTUUgcmVjb3JkXG4gICAqL1xuICBDTkFNRSxcbiAgLyoqXG4gICAqIERvIG5vdCBjcmVhdGUgYW55IEROUyByZWNvcmRzXG4gICAqL1xuICBOT05FXG59XG5cbi8qKlxuICogVGhlIHByb3BlcnRpZXMgZm9yIHRoZSBiYXNlIE5ldHdvcmtMb2FkQmFsYW5jZWRFYzJTZXJ2aWNlIG9yIE5ldHdvcmtMb2FkQmFsYW5jZWRGYXJnYXRlU2VydmljZSBzZXJ2aWNlLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIE5ldHdvcmtMb2FkQmFsYW5jZWRTZXJ2aWNlQmFzZVByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBjbHVzdGVyIHRoYXQgaG9zdHMgdGhlIHNlcnZpY2UuXG4gICAqXG4gICAqIElmIGEgY2x1c3RlciBpcyBzcGVjaWZpZWQsIHRoZSB2cGMgY29uc3RydWN0IHNob3VsZCBiZSBvbWl0dGVkLiBBbHRlcm5hdGl2ZWx5LCB5b3UgY2FuIG9taXQgYm90aCBjbHVzdGVyIGFuZCB2cGMuXG4gICAqIEBkZWZhdWx0IC0gY3JlYXRlIGEgbmV3IGNsdXN0ZXI7IGlmIGJvdGggY2x1c3RlciBhbmQgdnBjIGFyZSBvbWl0dGVkLCBhIG5ldyBWUEMgd2lsbCBiZSBjcmVhdGVkIGZvciB5b3UuXG4gICAqL1xuICByZWFkb25seSBjbHVzdGVyPzogSUNsdXN0ZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBWUEMgd2hlcmUgdGhlIGNvbnRhaW5lciBpbnN0YW5jZXMgd2lsbCBiZSBsYXVuY2hlZCBvciB0aGUgZWxhc3RpYyBuZXR3b3JrIGludGVyZmFjZXMgKEVOSXMpIHdpbGwgYmUgZGVwbG95ZWQuXG4gICAqXG4gICAqIElmIGEgdnBjIGlzIHNwZWNpZmllZCwgdGhlIGNsdXN0ZXIgY29uc3RydWN0IHNob3VsZCBiZSBvbWl0dGVkLiBBbHRlcm5hdGl2ZWx5LCB5b3UgY2FuIG9taXQgYm90aCB2cGMgYW5kIGNsdXN0ZXIuXG4gICAqIEBkZWZhdWx0IC0gdXNlcyB0aGUgVlBDIGRlZmluZWQgaW4gdGhlIGNsdXN0ZXIgb3IgY3JlYXRlcyBhIG5ldyBWUEMuXG4gICAqL1xuICByZWFkb25seSB2cGM/OiBJVnBjO1xuXG4gIC8qKlxuICAgKiBUaGUgcHJvcGVydGllcyByZXF1aXJlZCB0byBjcmVhdGUgYSBuZXcgdGFzayBkZWZpbml0aW9uLiBPbmUgb2YgdGFza0ltYWdlT3B0aW9ucyBvciB0YXNrRGVmaW5pdGlvbiBtdXN0IGJlIHNwZWNpZmllZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBub25lXG4gICAqL1xuICByZWFkb25seSB0YXNrSW1hZ2VPcHRpb25zPzogTmV0d29ya0xvYWRCYWxhbmNlZFRhc2tJbWFnZU9wdGlvbnM7XG5cbiAgLyoqXG4gICAqIERldGVybWluZXMgd2hldGhlciB0aGUgTG9hZCBCYWxhbmNlciB3aWxsIGJlIGludGVybmV0LWZhY2luZy5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgcHVibGljTG9hZEJhbGFuY2VyPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIGRlc2lyZWQgbnVtYmVyIG9mIGluc3RhbnRpYXRpb25zIG9mIHRoZSB0YXNrIGRlZmluaXRpb24gdG8ga2VlcCBydW5uaW5nIG9uIHRoZSBzZXJ2aWNlLlxuICAgKiBUaGUgbWluaW11bSB2YWx1ZSBpcyAxXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gSWYgdGhlIGZlYXR1cmUgZmxhZywgRUNTX1JFTU9WRV9ERUZBVUxUX0RFU0lSRURfQ09VTlQgaXMgZmFsc2UsIHRoZSBkZWZhdWx0IGlzIDE7XG4gICAqIGlmIHRydWUsIHRoZSBkZWZhdWx0IGlzIDEgZm9yIGFsbCBuZXcgc2VydmljZXMgYW5kIHVzZXMgdGhlIGV4aXN0aW5nIHNlcnZpY2VzIGRlc2lyZWQgY291bnRcbiAgICogd2hlbiB1cGRhdGluZyBhbiBleGlzdGluZyBzZXJ2aWNlLlxuICAgKi9cbiAgcmVhZG9ubHkgZGVzaXJlZENvdW50PzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgZG9tYWluIG5hbWUgZm9yIHRoZSBzZXJ2aWNlLCBlLmcuIFwiYXBpLmV4YW1wbGUuY29tLlwiXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gZG9tYWluIG5hbWUuXG4gICAqL1xuICByZWFkb25seSBkb21haW5OYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgUm91dGU1MyBob3N0ZWQgem9uZSBmb3IgdGhlIGRvbWFpbiwgZS5nLiBcImV4YW1wbGUuY29tLlwiXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gUm91dGU1MyBob3N0ZWQgZG9tYWluIHpvbmUuXG4gICAqL1xuICByZWFkb25seSBkb21haW5ab25lPzogSUhvc3RlZFpvbmU7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBzZXJ2aWNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIENsb3VkRm9ybWF0aW9uLWdlbmVyYXRlZCBuYW1lLlxuICAgKi9cbiAgcmVhZG9ubHkgc2VydmljZU5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBwZXJpb2Qgb2YgdGltZSwgaW4gc2Vjb25kcywgdGhhdCB0aGUgQW1hem9uIEVDUyBzZXJ2aWNlIHNjaGVkdWxlciBpZ25vcmVzIHVuaGVhbHRoeVxuICAgKiBFbGFzdGljIExvYWQgQmFsYW5jaW5nIHRhcmdldCBoZWFsdGggY2hlY2tzIGFmdGVyIGEgdGFzayBoYXMgZmlyc3Qgc3RhcnRlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBkZWZhdWx0cyB0byA2MCBzZWNvbmRzIGlmIGF0IGxlYXN0IG9uZSBsb2FkIGJhbGFuY2VyIGlzIGluLXVzZSBhbmQgaXQgaXMgbm90IGFscmVhZHkgc2V0XG4gICAqL1xuICByZWFkb25seSBoZWFsdGhDaGVja0dyYWNlUGVyaW9kPzogY2RrLkR1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiBUaGUgbWF4aW11bSBudW1iZXIgb2YgdGFza3MsIHNwZWNpZmllZCBhcyBhIHBlcmNlbnRhZ2Ugb2YgdGhlIEFtYXpvbiBFQ1NcbiAgICogc2VydmljZSdzIERlc2lyZWRDb3VudCB2YWx1ZSwgdGhhdCBjYW4gcnVuIGluIGEgc2VydmljZSBkdXJpbmcgYVxuICAgKiBkZXBsb3ltZW50LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIDEwMCBpZiBkYWVtb24sIG90aGVyd2lzZSAyMDBcbiAgICovXG4gIHJlYWRvbmx5IG1heEhlYWx0aHlQZXJjZW50PzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgbWluaW11bSBudW1iZXIgb2YgdGFza3MsIHNwZWNpZmllZCBhcyBhIHBlcmNlbnRhZ2Ugb2ZcbiAgICogdGhlIEFtYXpvbiBFQ1Mgc2VydmljZSdzIERlc2lyZWRDb3VudCB2YWx1ZSwgdGhhdCBtdXN0XG4gICAqIGNvbnRpbnVlIHRvIHJ1biBhbmQgcmVtYWluIGhlYWx0aHkgZHVyaW5nIGEgZGVwbG95bWVudC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSAwIGlmIGRhZW1vbiwgb3RoZXJ3aXNlIDUwXG4gICAqL1xuICByZWFkb25seSBtaW5IZWFsdGh5UGVyY2VudD86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIG5ldHdvcmsgbG9hZCBiYWxhbmNlciB0aGF0IHdpbGwgc2VydmUgdHJhZmZpYyB0byB0aGUgc2VydmljZS5cbiAgICogSWYgdGhlIGxvYWQgYmFsYW5jZXIgaGFzIGJlZW4gaW1wb3J0ZWQsIHRoZSB2cGMgYXR0cmlidXRlIG11c3QgYmUgc3BlY2lmaWVkXG4gICAqIGluIHRoZSBjYWxsIHRvIGZyb21OZXR3b3JrTG9hZEJhbGFuY2VyQXR0cmlidXRlcygpLlxuICAgKlxuICAgKiBbZGlzYWJsZS1hd3NsaW50OnJlZi12aWEtaW50ZXJmYWNlXVxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGEgbmV3IGxvYWQgYmFsYW5jZXIgd2lsbCBiZSBjcmVhdGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgbG9hZEJhbGFuY2VyPzogSU5ldHdvcmtMb2FkQmFsYW5jZXI7XG5cbiAgLyoqXG4gICAqIExpc3RlbmVyIHBvcnQgb2YgdGhlIG5ldHdvcmsgbG9hZCBiYWxhbmNlciB0aGF0IHdpbGwgc2VydmUgdHJhZmZpYyB0byB0aGUgc2VydmljZS5cbiAgICpcbiAgICogQGRlZmF1bHQgODBcbiAgICovXG4gIHJlYWRvbmx5IGxpc3RlbmVyUG9ydD86IG51bWJlcjtcblxuICAvKipcbiAgICogU3BlY2lmaWVzIHdoZXRoZXIgdG8gcHJvcGFnYXRlIHRoZSB0YWdzIGZyb20gdGhlIHRhc2sgZGVmaW5pdGlvbiBvciB0aGUgc2VydmljZSB0byB0aGUgdGFza3MgaW4gdGhlIHNlcnZpY2UuXG4gICAqIFRhZ3MgY2FuIG9ubHkgYmUgcHJvcGFnYXRlZCB0byB0aGUgdGFza3Mgd2l0aGluIHRoZSBzZXJ2aWNlIGR1cmluZyBzZXJ2aWNlIGNyZWF0aW9uLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vbmVcbiAgICovXG4gIHJlYWRvbmx5IHByb3BhZ2F0ZVRhZ3M/OiBQcm9wYWdhdGVkVGFnU291cmNlO1xuXG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgd2hldGhlciB0byBlbmFibGUgQW1hem9uIEVDUyBtYW5hZ2VkIHRhZ3MgZm9yIHRoZSB0YXNrcyB3aXRoaW4gdGhlIHNlcnZpY2UuIEZvciBtb3JlIGluZm9ybWF0aW9uLCBzZWVcbiAgICogW1RhZ2dpbmcgWW91ciBBbWF6b24gRUNTIFJlc291cmNlc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkVDUy9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvZWNzLXVzaW5nLXRhZ3MuaHRtbClcbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGVuYWJsZUVDU01hbmFnZWRUYWdzPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIG9wdGlvbnMgZm9yIGNvbmZpZ3VyaW5nIGFuIEFtYXpvbiBFQ1Mgc2VydmljZSB0byB1c2Ugc2VydmljZSBkaXNjb3ZlcnkuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQVdTIENsb3VkIE1hcCBzZXJ2aWNlIGRpc2NvdmVyeSBpcyBub3QgZW5hYmxlZC5cbiAgICovXG4gIHJlYWRvbmx5IGNsb3VkTWFwT3B0aW9ucz86IENsb3VkTWFwT3B0aW9ucztcblxuICAvKipcbiAgICogU3BlY2lmaWVzIHdoZXRoZXIgdGhlIFJvdXRlNTMgcmVjb3JkIHNob3VsZCBiZSBhIENOQU1FLCBhbiBBIHJlY29yZCB1c2luZyB0aGUgQWxpYXMgZmVhdHVyZSBvciBubyByZWNvcmQgYXQgYWxsLlxuICAgKiBUaGlzIGlzIHVzZWZ1bCBpZiB5b3UgbmVlZCB0byB3b3JrIHdpdGggRE5TIHN5c3RlbXMgdGhhdCBkbyBub3Qgc3VwcG9ydCBhbGlhcyByZWNvcmRzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBOZXR3b3JrTG9hZEJhbGFuY2VkU2VydmljZVJlY29yZFR5cGUuQUxJQVNcbiAgICovXG4gIHJlYWRvbmx5IHJlY29yZFR5cGU/OiBOZXR3b3JrTG9hZEJhbGFuY2VkU2VydmljZVJlY29yZFR5cGU7XG5cbiAgLyoqXG4gKiBTcGVjaWZpZXMgd2hpY2ggZGVwbG95bWVudCBjb250cm9sbGVyIHRvIHVzZSBmb3IgdGhlIHNlcnZpY2UuIEZvciBtb3JlIGluZm9ybWF0aW9uLCBzZWVcbiAqIFtBbWF6b24gRUNTIERlcGxveW1lbnQgVHlwZXNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2RlcGxveW1lbnQtdHlwZXMuaHRtbClcbiAqXG4gKiBAZGVmYXVsdCAtIFJvbGxpbmcgdXBkYXRlIChFQ1MpXG4gKi9cbiAgcmVhZG9ubHkgZGVwbG95bWVudENvbnRyb2xsZXI/OiBEZXBsb3ltZW50Q29udHJvbGxlcjtcblxuICAvKipcbiAgICogV2hldGhlciB0byBlbmFibGUgdGhlIGRlcGxveW1lbnQgY2lyY3VpdCBicmVha2VyLiBJZiB0aGlzIHByb3BlcnR5IGlzIGRlZmluZWQsIGNpcmN1aXQgYnJlYWtlciB3aWxsIGJlIGltcGxpY2l0bHlcbiAgICogZW5hYmxlZC5cbiAgICogQGRlZmF1bHQgLSBkaXNhYmxlZFxuICAgKi9cbiAgcmVhZG9ubHkgY2lyY3VpdEJyZWFrZXI/OiBEZXBsb3ltZW50Q2lyY3VpdEJyZWFrZXI7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgTmV0d29ya0xvYWRCYWxhbmNlZFRhc2tJbWFnZU9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIGltYWdlIHVzZWQgdG8gc3RhcnQgYSBjb250YWluZXIuIEltYWdlIG9yIHRhc2tEZWZpbml0aW9uIG11c3QgYmUgc3BlY2lmaWVkLCBidXQgbm90IGJvdGguXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgaW1hZ2U6IENvbnRhaW5lckltYWdlO1xuXG4gIC8qKlxuICAgKiBUaGUgZW52aXJvbm1lbnQgdmFyaWFibGVzIHRvIHBhc3MgdG8gdGhlIGNvbnRhaW5lci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBlbnZpcm9ubWVudCB2YXJpYWJsZXMuXG4gICAqL1xuICByZWFkb25seSBlbnZpcm9ubWVudD86IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH07XG5cbiAgLyoqXG4gICAqIFRoZSBzZWNyZXQgdG8gZXhwb3NlIHRvIHRoZSBjb250YWluZXIgYXMgYW4gZW52aXJvbm1lbnQgdmFyaWFibGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gc2VjcmV0IGVudmlyb25tZW50IHZhcmlhYmxlcy5cbiAgICovXG4gIHJlYWRvbmx5IHNlY3JldHM/OiB7IFtrZXk6IHN0cmluZ106IFNlY3JldCB9O1xuXG4gIC8qKlxuICAgKiBGbGFnIHRvIGluZGljYXRlIHdoZXRoZXIgdG8gZW5hYmxlIGxvZ2dpbmcuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGVuYWJsZUxvZ2dpbmc/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgbG9nIGRyaXZlciB0byB1c2UuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQXdzTG9nRHJpdmVyIGlmIGVuYWJsZUxvZ2dpbmcgaXMgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgbG9nRHJpdmVyPzogTG9nRHJpdmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgdGFzayBleGVjdXRpb24gSUFNIHJvbGUgdGhhdCBncmFudHMgdGhlIEFtYXpvbiBFQ1MgY29udGFpbmVyIGFnZW50IHBlcm1pc3Npb24gdG8gY2FsbCBBV1MgQVBJcyBvbiB5b3VyIGJlaGFsZi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyB2YWx1ZVxuICAgKi9cbiAgcmVhZG9ubHkgZXhlY3V0aW9uUm9sZT86IElSb2xlO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgdGFzayBJQU0gcm9sZSB0aGF0IGdyYW50cyBjb250YWluZXJzIGluIHRoZSB0YXNrIHBlcm1pc3Npb24gdG8gY2FsbCBBV1MgQVBJcyBvbiB5b3VyIGJlaGFsZi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBIHRhc2sgcm9sZSBpcyBhdXRvbWF0aWNhbGx5IGNyZWF0ZWQgZm9yIHlvdS5cbiAgICovXG4gIHJlYWRvbmx5IHRhc2tSb2xlPzogSVJvbGU7XG5cbiAgLyoqXG4gICAqIFRoZSBjb250YWluZXIgbmFtZSB2YWx1ZSB0byBiZSBzcGVjaWZpZWQgaW4gdGhlIHRhc2sgZGVmaW5pdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBub25lXG4gICAqL1xuICByZWFkb25seSBjb250YWluZXJOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcG9ydCBudW1iZXIgb24gdGhlIGNvbnRhaW5lciB0aGF0IGlzIGJvdW5kIHRvIHRoZSB1c2VyLXNwZWNpZmllZCBvciBhdXRvbWF0aWNhbGx5IGFzc2lnbmVkIGhvc3QgcG9ydC5cbiAgICpcbiAgICogSWYgeW91IGFyZSB1c2luZyBjb250YWluZXJzIGluIGEgdGFzayB3aXRoIHRoZSBhd3N2cGMgb3IgaG9zdCBuZXR3b3JrIG1vZGUsIGV4cG9zZWQgcG9ydHMgc2hvdWxkIGJlIHNwZWNpZmllZCB1c2luZyBjb250YWluZXJQb3J0LlxuICAgKiBJZiB5b3UgYXJlIHVzaW5nIGNvbnRhaW5lcnMgaW4gYSB0YXNrIHdpdGggdGhlIGJyaWRnZSBuZXR3b3JrIG1vZGUgYW5kIHlvdSBzcGVjaWZ5IGEgY29udGFpbmVyIHBvcnQgYW5kIG5vdCBhIGhvc3QgcG9ydCxcbiAgICogeW91ciBjb250YWluZXIgYXV0b21hdGljYWxseSByZWNlaXZlcyBhIGhvc3QgcG9ydCBpbiB0aGUgZXBoZW1lcmFsIHBvcnQgcmFuZ2UuXG4gICAqXG4gICAqIFBvcnQgbWFwcGluZ3MgdGhhdCBhcmUgYXV0b21hdGljYWxseSBhc3NpZ25lZCBpbiB0aGlzIHdheSBkbyBub3QgY291bnQgdG93YXJkIHRoZSAxMDAgcmVzZXJ2ZWQgcG9ydHMgbGltaXQgb2YgYSBjb250YWluZXIgaW5zdGFuY2UuXG4gICAqXG4gICAqIEZvciBtb3JlIGluZm9ybWF0aW9uLCBzZWVcbiAgICogW2hvc3RQb3J0XShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uRUNTL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1BvcnRNYXBwaW5nLmh0bWwjRUNTLVR5cGUtUG9ydE1hcHBpbmctaG9zdFBvcnQpLlxuICAgKlxuICAgKiBAZGVmYXVsdCA4MFxuICAgKi9cbiAgcmVhZG9ubHkgY29udGFpbmVyUG9ydD86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgYSBmYW1pbHkgdGhhdCB0aGlzIHRhc2sgZGVmaW5pdGlvbiBpcyByZWdpc3RlcmVkIHRvLiBBIGZhbWlseSBncm91cHMgbXVsdGlwbGUgdmVyc2lvbnMgb2YgYSB0YXNrIGRlZmluaXRpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQXV0b21hdGljYWxseSBnZW5lcmF0ZWQgbmFtZS5cbiAgICovXG4gIHJlYWRvbmx5IGZhbWlseT86IHN0cmluZztcblxuICAvKipcbiAgICogQSBrZXkvdmFsdWUgbWFwIG9mIGxhYmVscyB0byBhZGQgdG8gdGhlIGNvbnRhaW5lci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBsYWJlbHMuXG4gICAqL1xuICByZWFkb25seSBkb2NrZXJMYWJlbHM/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9O1xufVxuXG4vKipcbiAqIFRoZSBiYXNlIGNsYXNzIGZvciBOZXR3b3JrTG9hZEJhbGFuY2VkRWMyU2VydmljZSBhbmQgTmV0d29ya0xvYWRCYWxhbmNlZEZhcmdhdGVTZXJ2aWNlIHNlcnZpY2VzLlxuICovXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgTmV0d29ya0xvYWRCYWxhbmNlZFNlcnZpY2VCYXNlIGV4dGVuZHMgQ29yZUNvbnN0cnVjdCB7XG4gIC8qKlxuICAgKiBUaGUgZGVzaXJlZCBudW1iZXIgb2YgaW5zdGFudGlhdGlvbnMgb2YgdGhlIHRhc2sgZGVmaW5pdGlvbiB0byBrZWVwIHJ1bm5pbmcgb24gdGhlIHNlcnZpY2UuXG4gICAqIEBkZXByZWNhdGVkIC0gVXNlIGBpbnRlcm5hbERlc2lyZWRDb3VudGAgaW5zdGVhZC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBkZXNpcmVkQ291bnQ6IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIGRlc2lyZWQgbnVtYmVyIG9mIGluc3RhbnRpYXRpb25zIG9mIHRoZSB0YXNrIGRlZmluaXRpb24gdG8ga2VlcCBydW5uaW5nIG9uIHRoZSBzZXJ2aWNlLlxuICAgKiBUaGUgZGVmYXVsdCBpcyAxIGZvciBhbGwgbmV3IHNlcnZpY2VzIGFuZCB1c2VzIHRoZSBleGlzdGluZyBzZXJ2aWNlcyBkZXNpcmVkIGNvdW50XG4gICAqIHdoZW4gdXBkYXRpbmcgYW4gZXhpc3Rpbmcgc2VydmljZSwgaWYgb25lIGlzIG5vdCBwcm92aWRlZC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBpbnRlcm5hbERlc2lyZWRDb3VudD86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIE5ldHdvcmsgTG9hZCBCYWxhbmNlciBmb3IgdGhlIHNlcnZpY2UuXG4gICAqL1xuICBwdWJsaWMgZ2V0IGxvYWRCYWxhbmNlcigpOiBOZXR3b3JrTG9hZEJhbGFuY2VyIHtcbiAgICBpZiAoIXRoaXMuX25ldHdvcmtMb2FkQmFsYW5jZXIpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignLmxvYWRCYWxhbmNlciBjYW4gb25seSBiZSBhY2Nlc3NlZCBpZiB0aGUgY2xhc3Mgd2FzIGNvbnN0cnVjdGVkIHdpdGggYW4gb3duZWQsIG5vdCBpbXBvcnRlZCwgbG9hZCBiYWxhbmNlcicpO1xuICAgIH1cbiAgICByZXR1cm4gdGhpcy5fbmV0d29ya0xvYWRCYWxhbmNlcjtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgbGlzdGVuZXIgZm9yIHRoZSBzZXJ2aWNlLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGxpc3RlbmVyOiBOZXR3b3JrTGlzdGVuZXI7XG5cbiAgLyoqXG4gICAqIFRoZSB0YXJnZXQgZ3JvdXAgZm9yIHRoZSBzZXJ2aWNlLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHRhcmdldEdyb3VwOiBOZXR3b3JrVGFyZ2V0R3JvdXA7XG5cbiAgLyoqXG4gICAqIFRoZSBjbHVzdGVyIHRoYXQgaG9zdHMgdGhlIHNlcnZpY2UuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgY2x1c3RlcjogSUNsdXN0ZXI7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBfbmV0d29ya0xvYWRCYWxhbmNlcj86IE5ldHdvcmtMb2FkQmFsYW5jZXI7XG4gIC8qKlxuICAgKiBDb25zdHJ1Y3RzIGEgbmV3IGluc3RhbmNlIG9mIHRoZSBOZXR3b3JrTG9hZEJhbGFuY2VkU2VydmljZUJhc2UgY2xhc3MuXG4gICAqL1xuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogTmV0d29ya0xvYWRCYWxhbmNlZFNlcnZpY2VCYXNlUHJvcHMgPSB7fSkge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBpZiAocHJvcHMuY2x1c3RlciAmJiBwcm9wcy52cGMpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignWW91IGNhbiBvbmx5IHNwZWNpZnkgZWl0aGVyIHZwYyBvciBjbHVzdGVyLiBBbHRlcm5hdGl2ZWx5LCB5b3UgY2FuIGxlYXZlIGJvdGggYmxhbmsnKTtcbiAgICB9XG4gICAgdGhpcy5jbHVzdGVyID0gcHJvcHMuY2x1c3RlciB8fCB0aGlzLmdldERlZmF1bHRDbHVzdGVyKHRoaXMsIHByb3BzLnZwYyk7XG5cbiAgICBpZiAocHJvcHMuZGVzaXJlZENvdW50ICE9PSB1bmRlZmluZWQgJiYgcHJvcHMuZGVzaXJlZENvdW50IDwgMSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdZb3UgbXVzdCBzcGVjaWZ5IGEgZGVzaXJlZENvdW50IGdyZWF0ZXIgdGhhbiAwJyk7XG4gICAgfVxuXG4gICAgdGhpcy5kZXNpcmVkQ291bnQgPSBwcm9wcy5kZXNpcmVkQ291bnQgfHwgMTtcbiAgICB0aGlzLmludGVybmFsRGVzaXJlZENvdW50ID0gcHJvcHMuZGVzaXJlZENvdW50O1xuXG4gICAgY29uc3QgaW50ZXJuZXRGYWNpbmcgPSBwcm9wcy5wdWJsaWNMb2FkQmFsYW5jZXIgPz8gdHJ1ZTtcblxuICAgIGNvbnN0IGxiUHJvcHMgPSB7XG4gICAgICB2cGM6IHRoaXMuY2x1c3Rlci52cGMsXG4gICAgICBpbnRlcm5ldEZhY2luZyxcbiAgICB9O1xuXG4gICAgY29uc3QgbG9hZEJhbGFuY2VyID0gcHJvcHMubG9hZEJhbGFuY2VyID8/IG5ldyBOZXR3b3JrTG9hZEJhbGFuY2VyKHRoaXMsICdMQicsIGxiUHJvcHMpO1xuICAgIGNvbnN0IGxpc3RlbmVyUG9ydCA9IHByb3BzLmxpc3RlbmVyUG9ydCA/PyA4MDtcbiAgICBjb25zdCB0YXJnZXRQcm9wcyA9IHtcbiAgICAgIHBvcnQ6IHByb3BzLnRhc2tJbWFnZU9wdGlvbnM/LmNvbnRhaW5lclBvcnQgPz8gODAsXG4gICAgfTtcblxuICAgIHRoaXMubGlzdGVuZXIgPSBsb2FkQmFsYW5jZXIuYWRkTGlzdGVuZXIoJ1B1YmxpY0xpc3RlbmVyJywgeyBwb3J0OiBsaXN0ZW5lclBvcnQgfSk7XG4gICAgdGhpcy50YXJnZXRHcm91cCA9IHRoaXMubGlzdGVuZXIuYWRkVGFyZ2V0cygnRUNTJywgdGFyZ2V0UHJvcHMpO1xuXG4gICAgaWYgKHR5cGVvZiBwcm9wcy5kb21haW5OYW1lICE9PSAndW5kZWZpbmVkJykge1xuICAgICAgaWYgKHR5cGVvZiBwcm9wcy5kb21haW5ab25lID09PSAndW5kZWZpbmVkJykge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0EgUm91dGU1MyBob3N0ZWQgZG9tYWluIHpvbmUgbmFtZSBpcyByZXF1aXJlZCB0byBjb25maWd1cmUgdGhlIHNwZWNpZmllZCBkb21haW4gbmFtZScpO1xuICAgICAgfVxuXG4gICAgICBzd2l0Y2ggKHByb3BzLnJlY29yZFR5cGUgPz8gTmV0d29ya0xvYWRCYWxhbmNlZFNlcnZpY2VSZWNvcmRUeXBlLkFMSUFTKSB7XG4gICAgICAgIGNhc2UgTmV0d29ya0xvYWRCYWxhbmNlZFNlcnZpY2VSZWNvcmRUeXBlLkFMSUFTOlxuICAgICAgICAgIG5ldyBBUmVjb3JkKHRoaXMsICdETlMnLCB7XG4gICAgICAgICAgICB6b25lOiBwcm9wcy5kb21haW5ab25lLFxuICAgICAgICAgICAgcmVjb3JkTmFtZTogcHJvcHMuZG9tYWluTmFtZSxcbiAgICAgICAgICAgIHRhcmdldDogUmVjb3JkVGFyZ2V0LmZyb21BbGlhcyhuZXcgTG9hZEJhbGFuY2VyVGFyZ2V0KGxvYWRCYWxhbmNlcikpLFxuICAgICAgICAgIH0pO1xuICAgICAgICAgIGJyZWFrO1xuICAgICAgICBjYXNlIE5ldHdvcmtMb2FkQmFsYW5jZWRTZXJ2aWNlUmVjb3JkVHlwZS5DTkFNRTpcbiAgICAgICAgICBuZXcgQ25hbWVSZWNvcmQodGhpcywgJ0ROUycsIHtcbiAgICAgICAgICAgIHpvbmU6IHByb3BzLmRvbWFpblpvbmUsXG4gICAgICAgICAgICByZWNvcmROYW1lOiBwcm9wcy5kb21haW5OYW1lLFxuICAgICAgICAgICAgZG9tYWluTmFtZTogbG9hZEJhbGFuY2VyLmxvYWRCYWxhbmNlckRuc05hbWUsXG4gICAgICAgICAgfSk7XG4gICAgICAgICAgYnJlYWs7XG4gICAgICAgIGNhc2UgTmV0d29ya0xvYWRCYWxhbmNlZFNlcnZpY2VSZWNvcmRUeXBlLk5PTkU6XG4gICAgICAgICAgLy8gRG8gbm90IGNyZWF0ZSBhIEROUyByZWNvcmRcbiAgICAgICAgICBicmVhaztcbiAgICAgIH1cbiAgICB9XG5cbiAgICBpZiAobG9hZEJhbGFuY2VyIGluc3RhbmNlb2YgTmV0d29ya0xvYWRCYWxhbmNlcikge1xuICAgICAgdGhpcy5fbmV0d29ya0xvYWRCYWxhbmNlciA9IGxvYWRCYWxhbmNlcjtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMubG9hZEJhbGFuY2VyID09PSB1bmRlZmluZWQpIHtcbiAgICAgIG5ldyBjZGsuQ2ZuT3V0cHV0KHRoaXMsICdMb2FkQmFsYW5jZXJETlMnLCB7IHZhbHVlOiB0aGlzLmxvYWRCYWxhbmNlci5sb2FkQmFsYW5jZXJEbnNOYW1lIH0pO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBkZWZhdWx0IGNsdXN0ZXIuXG4gICAqL1xuICBwcm90ZWN0ZWQgZ2V0RGVmYXVsdENsdXN0ZXIoc2NvcGU6IENvcmVDb25zdHJ1Y3QsIHZwYz86IElWcGMpOiBDbHVzdGVyIHtcbiAgICAvLyBtYWdpYyBzdHJpbmcgdG8gYXZvaWQgY29sbGlzaW9uIHdpdGggdXNlci1kZWZpbmVkIGNvbnN0cnVjdHNcbiAgICBjb25zdCBERUZBVUxUX0NMVVNURVJfSUQgPSBgRWNzRGVmYXVsdENsdXN0ZXJNbkwzbU5OWU4ke3ZwYyA/IHZwYy5ub2RlLmlkIDogJyd9YDtcbiAgICBjb25zdCBzdGFjayA9IGNkay5TdGFjay5vZihzY29wZSk7XG4gICAgcmV0dXJuIHN0YWNrLm5vZGUudHJ5RmluZENoaWxkKERFRkFVTFRfQ0xVU1RFUl9JRCkgYXMgQ2x1c3RlciB8fCBuZXcgQ2x1c3RlcihzdGFjaywgREVGQVVMVF9DTFVTVEVSX0lELCB7IHZwYyB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIHNlcnZpY2UgYXMgYSB0YXJnZXQgb2YgdGhlIHRhcmdldCBncm91cC5cbiAgICovXG4gIHByb3RlY3RlZCBhZGRTZXJ2aWNlQXNUYXJnZXQoc2VydmljZTogQmFzZVNlcnZpY2UpIHtcbiAgICB0aGlzLnRhcmdldEdyb3VwLmFkZFRhcmdldChzZXJ2aWNlKTtcbiAgfVxuXG4gIHByb3RlY3RlZCBjcmVhdGVBV1NMb2dEcml2ZXIocHJlZml4OiBzdHJpbmcpOiBBd3NMb2dEcml2ZXIge1xuICAgIHJldHVybiBuZXcgQXdzTG9nRHJpdmVyKHsgc3RyZWFtUHJlZml4OiBwcmVmaXggfSk7XG4gIH1cbn1cbiJdfQ==