"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FlowLog = exports.FlowLogDestination = exports.FlowLogResourceType = exports.FlowLogDestinationType = exports.FlowLogTrafficType = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const logs = require("../../aws-logs");
const s3 = require("../../aws-s3");
const core_1 = require("../../core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * The type of VPC traffic to log
 *
 *
 */
var FlowLogTrafficType;
(function (FlowLogTrafficType) {
    /**
     * Only log accepts
     */
    FlowLogTrafficType["ACCEPT"] = "ACCEPT";
    /**
     * Log all requests
     */
    FlowLogTrafficType["ALL"] = "ALL";
    /**
     * Only log rejects
     */
    FlowLogTrafficType["REJECT"] = "REJECT";
})(FlowLogTrafficType = exports.FlowLogTrafficType || (exports.FlowLogTrafficType = {}));
/**
 * The available destination types for Flow Logs
 *
 */
var FlowLogDestinationType;
(function (FlowLogDestinationType) {
    /**
     * Send flow logs to CloudWatch Logs Group
     */
    FlowLogDestinationType["CLOUD_WATCH_LOGS"] = "cloud-watch-logs";
    /**
     * Send flow logs to S3 Bucket
     */
    FlowLogDestinationType["S3"] = "s3";
})(FlowLogDestinationType = exports.FlowLogDestinationType || (exports.FlowLogDestinationType = {}));
/**
 * The type of resource to create the flow log for
 *
 *
 */
class FlowLogResourceType {
    /**
     * The subnet to attach the Flow Log to
     */
    static fromSubnet(subnet) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ec2_ISubnet(subnet);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromSubnet);
            }
            throw error;
        }
        return {
            resourceType: 'Subnet',
            resourceId: subnet.subnetId,
        };
    }
    /**
     * The VPC to attach the Flow Log to
     */
    static fromVpc(vpc) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ec2_IVpc(vpc);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromVpc);
            }
            throw error;
        }
        return {
            resourceType: 'VPC',
            resourceId: vpc.vpcId,
        };
    }
    /**
     * The Network Interface to attach the Flow Log to
     */
    static fromNetworkInterfaceId(id) {
        return {
            resourceType: 'NetworkInterface',
            resourceId: id,
        };
    }
}
exports.FlowLogResourceType = FlowLogResourceType;
_a = JSII_RTTI_SYMBOL_1;
FlowLogResourceType[_a] = { fqn: "monocdk.aws_ec2.FlowLogResourceType", version: "1.185.0" };
/**
 * The destination type for the flow log
 *
 *
 */
class FlowLogDestination {
    /**
     * Use CloudWatch logs as the destination
     */
    static toCloudWatchLogs(logGroup, iamRole) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_logs_ILogGroup(logGroup);
            jsiiDeprecationWarnings.monocdk_aws_iam_IRole(iamRole);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.toCloudWatchLogs);
            }
            throw error;
        }
        return new CloudWatchLogsDestination({
            logDestinationType: FlowLogDestinationType.CLOUD_WATCH_LOGS,
            logGroup,
            iamRole,
        });
    }
    /**
     * Use S3 as the destination
     */
    static toS3(bucket, keyPrefix) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_s3_IBucket(bucket);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.toS3);
            }
            throw error;
        }
        return new S3Destination({
            logDestinationType: FlowLogDestinationType.S3,
            s3Bucket: bucket,
            keyPrefix,
        });
    }
}
exports.FlowLogDestination = FlowLogDestination;
_b = JSII_RTTI_SYMBOL_1;
FlowLogDestination[_b] = { fqn: "monocdk.aws_ec2.FlowLogDestination", version: "1.185.0" };
/**
 *
 */
class S3Destination extends FlowLogDestination {
    constructor(props) {
        super();
        this.props = props;
    }
    bind(scope, _flowLog) {
        let s3Bucket;
        if (this.props.s3Bucket === undefined) {
            s3Bucket = new s3.Bucket(scope, 'Bucket', {
                encryption: s3.BucketEncryption.UNENCRYPTED,
                removalPolicy: core_1.RemovalPolicy.RETAIN,
            });
        }
        else {
            s3Bucket = this.props.s3Bucket;
        }
        return {
            logDestinationType: FlowLogDestinationType.S3,
            s3Bucket,
            keyPrefix: this.props.keyPrefix,
        };
    }
}
/**
 *
 */
class CloudWatchLogsDestination extends FlowLogDestination {
    constructor(props) {
        super();
        this.props = props;
    }
    bind(scope, _flowLog) {
        let iamRole;
        let logGroup;
        if (this.props.iamRole === undefined) {
            iamRole = new iam.Role(scope, 'IAMRole', {
                roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
            });
        }
        else {
            iamRole = this.props.iamRole;
        }
        if (this.props.logGroup === undefined) {
            logGroup = new logs.LogGroup(scope, 'LogGroup');
        }
        else {
            logGroup = this.props.logGroup;
        }
        iamRole.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: [
                'logs:CreateLogStream',
                'logs:PutLogEvents',
                'logs:DescribeLogStreams',
            ],
            effect: iam.Effect.ALLOW,
            resources: [logGroup.logGroupArn],
        }));
        iamRole.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            effect: iam.Effect.ALLOW,
            resources: [iamRole.roleArn],
        }));
        return {
            logDestinationType: FlowLogDestinationType.CLOUD_WATCH_LOGS,
            logGroup,
            iamRole,
        };
    }
}
/**
 * The base class for a Flow Log
 *
 *
 */
class FlowLogBase extends core_1.Resource {
}
/**
 * A VPC flow log.
 * @resource AWS::EC2::FlowLog
 *
 *
 */
class FlowLog extends FlowLogBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.flowLogName,
        });
        try {
            jsiiDeprecationWarnings.monocdk_aws_ec2_FlowLogProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, FlowLog);
            }
            throw error;
        }
        const destination = props.destination || FlowLogDestination.toCloudWatchLogs();
        const destinationConfig = destination.bind(this, this);
        this.logGroup = destinationConfig.logGroup;
        this.bucket = destinationConfig.s3Bucket;
        this.iamRole = destinationConfig.iamRole;
        this.keyPrefix = destinationConfig.keyPrefix;
        let logDestination = undefined;
        if (this.bucket) {
            logDestination = this.keyPrefix ? this.bucket.arnForObjects(this.keyPrefix) : this.bucket.bucketArn;
        }
        const flowLog = new ec2_generated_1.CfnFlowLog(this, 'FlowLog', {
            deliverLogsPermissionArn: this.iamRole ? this.iamRole.roleArn : undefined,
            logDestinationType: destinationConfig.logDestinationType,
            logGroupName: this.logGroup ? this.logGroup.logGroupName : undefined,
            resourceId: props.resourceType.resourceId,
            resourceType: props.resourceType.resourceType,
            trafficType: props.trafficType
                ? props.trafficType
                : FlowLogTrafficType.ALL,
            logDestination,
        });
        this.flowLogId = flowLog.ref;
    }
    /**
     * Import a Flow Log by it's Id
     */
    static fromFlowLogId(scope, id, flowLogId) {
        class Import extends FlowLogBase {
            constructor() {
                super(...arguments);
                this.flowLogId = flowLogId;
            }
        }
        return new Import(scope, id);
    }
}
exports.FlowLog = FlowLog;
_c = JSII_RTTI_SYMBOL_1;
FlowLog[_c] = { fqn: "monocdk.aws_ec2.FlowLog", version: "1.185.0" };
//# sourceMappingURL=data:application/json;base64,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