"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Volume = exports.EbsDeviceVolumeType = exports.BlockDeviceVolume = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const aws_iam_1 = require("../../aws-iam");
const aws_kms_1 = require("../../aws-kms");
const core_1 = require("../../core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * Describes a block device mapping for an EC2 instance or Auto Scaling group.
 */
class BlockDeviceVolume {
    /**
     * @param ebsDevice EBS device info
     * @param virtualName Virtual device name
     */
    constructor(ebsDevice, virtualName) {
        this.ebsDevice = ebsDevice;
        this.virtualName = virtualName;
        try {
            jsiiDeprecationWarnings.monocdk_aws_ec2_EbsDeviceProps(ebsDevice);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, BlockDeviceVolume);
            }
            throw error;
        }
    }
    /**
     * Creates a new Elastic Block Storage device
     *
     * @param volumeSize The volume size, in Gibibytes (GiB)
     * @param options additional device options
     */
    static ebs(volumeSize, options = {}) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ec2_EbsDeviceOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.ebs);
            }
            throw error;
        }
        return new this({ ...options, volumeSize });
    }
    /**
     * Creates a new Elastic Block Storage device from an existing snapshot
     *
     * @param snapshotId The snapshot ID of the volume to use
     * @param options additional device options
     */
    static ebsFromSnapshot(snapshotId, options = {}) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ec2_EbsDeviceSnapshotOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.ebsFromSnapshot);
            }
            throw error;
        }
        return new this({ ...options, snapshotId });
    }
    /**
     * Creates a virtual, ephemeral device.
     * The name will be in the form ephemeral{volumeIndex}.
     *
     * @param volumeIndex the volume index. Must be equal or greater than 0
     */
    static ephemeral(volumeIndex) {
        if (volumeIndex < 0) {
            throw new Error(`volumeIndex must be a number starting from 0, got "${volumeIndex}"`);
        }
        return new this(undefined, `ephemeral${volumeIndex}`);
    }
}
exports.BlockDeviceVolume = BlockDeviceVolume;
_a = JSII_RTTI_SYMBOL_1;
BlockDeviceVolume[_a] = { fqn: "monocdk.aws_ec2.BlockDeviceVolume", version: "1.185.0" };
/**
 * Supported EBS volume types for blockDevices
 */
var EbsDeviceVolumeType;
(function (EbsDeviceVolumeType) {
    /**
     * Magnetic
     */
    EbsDeviceVolumeType["STANDARD"] = "standard";
    /**
     *  Provisioned IOPS SSD - IO1
     */
    EbsDeviceVolumeType["IO1"] = "io1";
    /**
     *  Provisioned IOPS SSD - IO2
     */
    EbsDeviceVolumeType["IO2"] = "io2";
    /**
     * General Purpose SSD - GP2
     */
    EbsDeviceVolumeType["GP2"] = "gp2";
    /**
     * General Purpose SSD - GP3
     */
    EbsDeviceVolumeType["GP3"] = "gp3";
    /**
     * Throughput Optimized HDD
     */
    EbsDeviceVolumeType["ST1"] = "st1";
    /**
     * Cold HDD
     */
    EbsDeviceVolumeType["SC1"] = "sc1";
    /**
     * General purpose SSD volume (GP2) that balances price and performance for a wide variety of workloads.
     */
    EbsDeviceVolumeType["GENERAL_PURPOSE_SSD"] = "gp2";
    /**
     * General purpose SSD volume (GP3) that balances price and performance for a wide variety of workloads.
     */
    EbsDeviceVolumeType["GENERAL_PURPOSE_SSD_GP3"] = "gp3";
    /**
     * Highest-performance SSD volume (IO1) for mission-critical low-latency or high-throughput workloads.
     */
    EbsDeviceVolumeType["PROVISIONED_IOPS_SSD"] = "io1";
    /**
     * Highest-performance SSD volume (IO2) for mission-critical low-latency or high-throughput workloads.
     */
    EbsDeviceVolumeType["PROVISIONED_IOPS_SSD_IO2"] = "io2";
    /**
     * Low-cost HDD volume designed for frequently accessed, throughput-intensive workloads.
     */
    EbsDeviceVolumeType["THROUGHPUT_OPTIMIZED_HDD"] = "st1";
    /**
     * Lowest cost HDD volume designed for less frequently accessed workloads.
     */
    EbsDeviceVolumeType["COLD_HDD"] = "sc1";
    /**
     * Magnetic volumes are backed by magnetic drives and are suited for workloads where data is accessed infrequently, and scenarios where low-cost
     * storage for small volume sizes is important.
     */
    EbsDeviceVolumeType["MAGNETIC"] = "standard";
})(EbsDeviceVolumeType = exports.EbsDeviceVolumeType || (exports.EbsDeviceVolumeType = {}));
/**
 * Common behavior of Volumes. Users should not use this class directly, and instead use ``Volume``.
 */
class VolumeBase extends core_1.Resource {
    grantAttachVolume(grantee, instances) {
        const result = aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['ec2:AttachVolume'],
            resourceArns: this.collectGrantResourceArns(instances),
        });
        if (this.encryptionKey) {
            // When attaching a volume, the EC2 Service will need to grant to itself permission
            // to be able to decrypt the encryption key. We restrict the CreateGrant for principle
            // of least privilege, in accordance with best practices.
            // See: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#ebs-encryption-permissions
            const kmsGrant = this.encryptionKey.grant(grantee, 'kms:CreateGrant');
            kmsGrant.principalStatement.addConditions({
                Bool: { 'kms:GrantIsForAWSResource': true },
                StringEquals: {
                    'kms:ViaService': `ec2.${core_1.Stack.of(this).region}.amazonaws.com`,
                    'kms:GrantConstraintType': 'EncryptionContextSubset',
                },
            });
        }
        return result;
    }
    grantAttachVolumeByResourceTag(grantee, constructs, tagKeySuffix) {
        const tagValue = this.calculateResourceTagValue([this, ...constructs]);
        const tagKey = `VolumeGrantAttach-${tagKeySuffix ?? tagValue.slice(0, 10).toUpperCase()}`;
        const grantCondition = {};
        grantCondition[`ec2:ResourceTag/${tagKey}`] = tagValue;
        const result = this.grantAttachVolume(grantee);
        result.principalStatement.addCondition('ForAnyValue:StringEquals', grantCondition);
        // The ResourceTag condition requires that all resources involved in the operation have
        // the given tag, so we tag this and all constructs given.
        core_1.Tags.of(this).add(tagKey, tagValue);
        constructs.forEach(construct => core_1.Tags.of(construct).add(tagKey, tagValue));
        return result;
    }
    grantDetachVolume(grantee, instances) {
        const result = aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['ec2:DetachVolume'],
            resourceArns: this.collectGrantResourceArns(instances),
        });
        // Note: No encryption key permissions are required to detach an encrypted volume.
        return result;
    }
    grantDetachVolumeByResourceTag(grantee, constructs, tagKeySuffix) {
        const tagValue = this.calculateResourceTagValue([this, ...constructs]);
        const tagKey = `VolumeGrantDetach-${tagKeySuffix ?? tagValue.slice(0, 10).toUpperCase()}`;
        const grantCondition = {};
        grantCondition[`ec2:ResourceTag/${tagKey}`] = tagValue;
        const result = this.grantDetachVolume(grantee);
        result.principalStatement.addCondition('ForAnyValue:StringEquals', grantCondition);
        // The ResourceTag condition requires that all resources involved in the operation have
        // the given tag, so we tag this and all constructs given.
        core_1.Tags.of(this).add(tagKey, tagValue);
        constructs.forEach(construct => core_1.Tags.of(construct).add(tagKey, tagValue));
        return result;
    }
    collectGrantResourceArns(instances) {
        const stack = core_1.Stack.of(this);
        const resourceArns = [
            `arn:${stack.partition}:ec2:${stack.region}:${stack.account}:volume/${this.volumeId}`,
        ];
        const instanceArnPrefix = `arn:${stack.partition}:ec2:${stack.region}:${stack.account}:instance`;
        if (instances) {
            instances.forEach(instance => resourceArns.push(`${instanceArnPrefix}/${instance?.instanceId}`));
        }
        else {
            resourceArns.push(`${instanceArnPrefix}/*`);
        }
        return resourceArns;
    }
    calculateResourceTagValue(constructs) {
        const md5 = crypto.createHash('md5');
        constructs.forEach(construct => md5.update(core_1.Names.uniqueId(construct)));
        return md5.digest('hex');
    }
}
/**
 * Creates a new EBS Volume in AWS EC2.
 */
class Volume extends VolumeBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.volumeName,
        });
        try {
            jsiiDeprecationWarnings.monocdk_aws_ec2_VolumeProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Volume);
            }
            throw error;
        }
        this.validateProps(props);
        const resource = new ec2_generated_1.CfnVolume(this, 'Resource', {
            availabilityZone: props.availabilityZone,
            autoEnableIo: props.autoEnableIo,
            encrypted: props.encrypted,
            kmsKeyId: props.encryptionKey?.keyArn,
            iops: props.iops,
            multiAttachEnabled: props.enableMultiAttach ?? false,
            size: props.size?.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL }),
            snapshotId: props.snapshotId,
            volumeType: props.volumeType ?? EbsDeviceVolumeType.GENERAL_PURPOSE_SSD,
        });
        resource.applyRemovalPolicy(props.removalPolicy);
        if (props.volumeName)
            core_1.Tags.of(resource).add('Name', props.volumeName);
        this.volumeId = resource.ref;
        this.availabilityZone = props.availabilityZone;
        this.encryptionKey = props.encryptionKey;
        if (this.encryptionKey) {
            // Per: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#ebs-encryption-requirements
            const principal = new aws_kms_1.ViaServicePrincipal(`ec2.${core_1.Stack.of(this).region}.amazonaws.com`, new aws_iam_1.AccountRootPrincipal()).withConditions({
                StringEquals: {
                    'kms:CallerAccount': core_1.Stack.of(this).account,
                },
            });
            const grant = this.encryptionKey.grant(principal, 
            // Describe & Generate are required to be able to create the CMK-encrypted Volume.
            'kms:DescribeKey', 'kms:GenerateDataKeyWithoutPlainText');
            if (props.snapshotId) {
                // ReEncrypt is required for when re-encrypting from an encrypted snapshot.
                grant.principalStatement?.addActions('kms:ReEncrypt*');
            }
        }
    }
    /**
     * Import an existing EBS Volume into the Stack.
     *
     * @param scope the scope of the import.
     * @param id    the ID of the imported Volume in the construct tree.
     * @param attrs the attributes of the imported Volume
     */
    static fromVolumeAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ec2_VolumeAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromVolumeAttributes);
            }
            throw error;
        }
        class Import extends VolumeBase {
            constructor() {
                super(...arguments);
                this.volumeId = attrs.volumeId;
                this.availabilityZone = attrs.availabilityZone;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        // Check that the provided volumeId looks like it could be valid.
        if (!core_1.Token.isUnresolved(attrs.volumeId) && !/^vol-[0-9a-fA-F]+$/.test(attrs.volumeId)) {
            throw new Error('`volumeId` does not match expected pattern. Expected `vol-<hexadecmial value>` (ex: `vol-05abe246af`) or a Token');
        }
        return new Import(scope, id);
    }
    validateProps(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ec2_VolumeProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.validateProps);
            }
            throw error;
        }
        if (!(props.size || props.snapshotId)) {
            throw new Error('Must provide at least one of `size` or `snapshotId`');
        }
        if (props.snapshotId && !core_1.Token.isUnresolved(props.snapshotId) && !/^snap-[0-9a-fA-F]+$/.test(props.snapshotId)) {
            throw new Error('`snapshotId` does match expected pattern. Expected `snap-<hexadecmial value>` (ex: `snap-05abe246af`) or Token');
        }
        if (props.encryptionKey && !props.encrypted) {
            throw new Error('`encrypted` must be true when providing an `encryptionKey`.');
        }
        if (props.volumeType &&
            [
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD,
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2,
            ].includes(props.volumeType) &&
            !props.iops) {
            throw new Error('`iops` must be specified if the `volumeType` is `PROVISIONED_IOPS_SSD` or `PROVISIONED_IOPS_SSD_IO2`.');
        }
        if (props.iops) {
            const volumeType = props.volumeType ?? EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
            if (![
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD,
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2,
                EbsDeviceVolumeType.GENERAL_PURPOSE_SSD_GP3,
            ].includes(volumeType)) {
                throw new Error('`iops` may only be specified if the `volumeType` is `PROVISIONED_IOPS_SSD`, `PROVISIONED_IOPS_SSD_IO2` or `GENERAL_PURPOSE_SSD_GP3`.');
            }
            // Enforce minimum & maximum IOPS:
            // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ebs-volume.html
            const iopsRanges = {};
            iopsRanges[EbsDeviceVolumeType.GENERAL_PURPOSE_SSD_GP3] = { Min: 3000, Max: 16000 };
            iopsRanges[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD] = { Min: 100, Max: 64000 };
            iopsRanges[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2] = { Min: 100, Max: 64000 };
            const { Min, Max } = iopsRanges[volumeType];
            if (props.iops < Min || props.iops > Max) {
                throw new Error(`\`${volumeType}\` volumes iops must be between ${Min} and ${Max}.`);
            }
            // Enforce maximum ratio of IOPS/GiB:
            // https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volume-types.html
            const maximumRatios = {};
            maximumRatios[EbsDeviceVolumeType.GENERAL_PURPOSE_SSD_GP3] = 500;
            maximumRatios[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD] = 50;
            maximumRatios[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2] = 500;
            const maximumRatio = maximumRatios[volumeType];
            if (props.size && (props.iops > maximumRatio * props.size.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL }))) {
                throw new Error(`\`${volumeType}\` volumes iops has a maximum ratio of ${maximumRatio} IOPS/GiB.`);
            }
        }
        if (props.enableMultiAttach) {
            const volumeType = props.volumeType ?? EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
            if (![
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD,
                EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2,
            ].includes(volumeType)) {
                throw new Error('multi-attach is supported exclusively on `PROVISIONED_IOPS_SSD` and `PROVISIONED_IOPS_SSD_IO2` volumes.');
            }
        }
        if (props.size) {
            const size = props.size.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL });
            // Enforce minimum & maximum volume size:
            // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ebs-volume.html
            const sizeRanges = {};
            sizeRanges[EbsDeviceVolumeType.GENERAL_PURPOSE_SSD] = { Min: 1, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.GENERAL_PURPOSE_SSD_GP3] = { Min: 1, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD] = { Min: 4, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD_IO2] = { Min: 4, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.THROUGHPUT_OPTIMIZED_HDD] = { Min: 125, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.COLD_HDD] = { Min: 125, Max: 16384 };
            sizeRanges[EbsDeviceVolumeType.MAGNETIC] = { Min: 1, Max: 1024 };
            const volumeType = props.volumeType ?? EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
            const { Min, Max } = sizeRanges[volumeType];
            if (size < Min || size > Max) {
                throw new Error(`\`${volumeType}\` volumes must be between ${Min} GiB and ${Max} GiB in size.`);
            }
        }
    }
}
exports.Volume = Volume;
_b = JSII_RTTI_SYMBOL_1;
Volume[_b] = { fqn: "monocdk.aws_ec2.Volume", version: "1.185.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidm9sdW1lLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidm9sdW1lLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7OztBQUFBLGlDQUFpQztBQUVqQywyQ0FBd0U7QUFDeEUsMkNBQTBEO0FBQzFELHFDQUF1SDtBQUV2SCxtREFBNEM7QUE0SDVDOztHQUVHO0FBQ0gsTUFBYSxpQkFBaUI7SUFtQzVCOzs7T0FHRztJQUNILFlBQXNDLFNBQTBCLEVBQWtCLFdBQW9CO1FBQWhFLGNBQVMsR0FBVCxTQUFTLENBQWlCO1FBQWtCLGdCQUFXLEdBQVgsV0FBVyxDQUFTOzs7Ozs7K0NBdkMzRixpQkFBaUI7Ozs7S0F3QzNCO0lBdkNEOzs7OztPQUtHO0lBQ0ksTUFBTSxDQUFDLEdBQUcsQ0FBQyxVQUFrQixFQUFFLFVBQTRCLEVBQUU7Ozs7Ozs7Ozs7UUFDbEUsT0FBTyxJQUFJLElBQUksQ0FBQyxFQUFFLEdBQUcsT0FBTyxFQUFFLFVBQVUsRUFBRSxDQUFDLENBQUM7S0FDN0M7SUFFRDs7Ozs7T0FLRztJQUNJLE1BQU0sQ0FBQyxlQUFlLENBQUMsVUFBa0IsRUFBRSxVQUFvQyxFQUFFOzs7Ozs7Ozs7O1FBQ3RGLE9BQU8sSUFBSSxJQUFJLENBQUMsRUFBRSxHQUFHLE9BQU8sRUFBRSxVQUFVLEVBQUUsQ0FBQyxDQUFDO0tBQzdDO0lBRUQ7Ozs7O09BS0c7SUFDSSxNQUFNLENBQUMsU0FBUyxDQUFDLFdBQW1CO1FBQ3pDLElBQUksV0FBVyxHQUFHLENBQUMsRUFBRTtZQUNuQixNQUFNLElBQUksS0FBSyxDQUFDLHNEQUFzRCxXQUFXLEdBQUcsQ0FBQyxDQUFDO1NBQ3ZGO1FBRUQsT0FBTyxJQUFJLElBQUksQ0FBQyxTQUFTLEVBQUUsWUFBWSxXQUFXLEVBQUUsQ0FBQyxDQUFDO0tBQ3ZEOztBQWpDSCw4Q0F5Q0M7OztBQUVEOztHQUVHO0FBQ0gsSUFBWSxtQkF1RVg7QUF2RUQsV0FBWSxtQkFBbUI7SUFDN0I7O09BRUc7SUFDSCw0Q0FBcUIsQ0FBQTtJQUVyQjs7T0FFRztJQUNILGtDQUFXLENBQUE7SUFFWDs7T0FFRztJQUNILGtDQUFXLENBQUE7SUFFWDs7T0FFRztJQUNILGtDQUFXLENBQUE7SUFFWDs7T0FFRztJQUNILGtDQUFXLENBQUE7SUFFWDs7T0FFRztJQUNILGtDQUFXLENBQUE7SUFFWDs7T0FFRztJQUNILGtDQUFXLENBQUE7SUFFWDs7T0FFRztJQUNILGtEQUF5QixDQUFBO0lBRXpCOztPQUVHO0lBQ0gsc0RBQTZCLENBQUE7SUFFN0I7O09BRUc7SUFDSCxtREFBMEIsQ0FBQTtJQUUxQjs7T0FFRztJQUNILHVEQUE4QixDQUFBO0lBRTlCOztPQUVHO0lBQ0gsdURBQThCLENBQUE7SUFFOUI7O09BRUc7SUFDSCx1Q0FBYyxDQUFBO0lBRWQ7OztPQUdHO0lBQ0gsNENBQW1CLENBQUE7QUFDckIsQ0FBQyxFQXZFVyxtQkFBbUIsR0FBbkIsMkJBQW1CLEtBQW5CLDJCQUFtQixRQXVFOUI7QUFnT0Q7O0dBRUc7QUFDSCxNQUFlLFVBQVcsU0FBUSxlQUFRO0lBS2pDLGlCQUFpQixDQUFDLE9BQW1CLEVBQUUsU0FBdUI7UUFDbkUsTUFBTSxNQUFNLEdBQUcsZUFBSyxDQUFDLGNBQWMsQ0FBQztZQUNsQyxPQUFPO1lBQ1AsT0FBTyxFQUFFLENBQUMsa0JBQWtCLENBQUM7WUFDN0IsWUFBWSxFQUFFLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxTQUFTLENBQUM7U0FDdkQsQ0FBQyxDQUFDO1FBRUgsSUFBSSxJQUFJLENBQUMsYUFBYSxFQUFFO1lBQ3RCLG1GQUFtRjtZQUNuRixzRkFBc0Y7WUFDdEYseURBQXlEO1lBQ3pELHlHQUF5RztZQUN6RyxNQUFNLFFBQVEsR0FBVSxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUUsaUJBQWlCLENBQUMsQ0FBQztZQUM3RSxRQUFRLENBQUMsa0JBQW1CLENBQUMsYUFBYSxDQUN4QztnQkFDRSxJQUFJLEVBQUUsRUFBRSwyQkFBMkIsRUFBRSxJQUFJLEVBQUU7Z0JBQzNDLFlBQVksRUFBRTtvQkFDWixnQkFBZ0IsRUFBRSxPQUFPLFlBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsTUFBTSxnQkFBZ0I7b0JBQzlELHlCQUF5QixFQUFFLHlCQUF5QjtpQkFDckQ7YUFDRixDQUNGLENBQUM7U0FDSDtRQUVELE9BQU8sTUFBTSxDQUFDO0tBQ2Y7SUFFTSw4QkFBOEIsQ0FBQyxPQUFtQixFQUFFLFVBQXVCLEVBQUUsWUFBcUI7UUFDdkcsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLHlCQUF5QixDQUFDLENBQUMsSUFBSSxFQUFFLEdBQUcsVUFBVSxDQUFDLENBQUMsQ0FBQztRQUN2RSxNQUFNLE1BQU0sR0FBRyxxQkFBcUIsWUFBWSxJQUFJLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLFdBQVcsRUFBRSxFQUFFLENBQUM7UUFDMUYsTUFBTSxjQUFjLEdBQThCLEVBQUUsQ0FBQztRQUNyRCxjQUFjLENBQUMsbUJBQW1CLE1BQU0sRUFBRSxDQUFDLEdBQUcsUUFBUSxDQUFDO1FBRXZELE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUMvQyxNQUFNLENBQUMsa0JBQW1CLENBQUMsWUFBWSxDQUNyQywwQkFBMEIsRUFBRSxjQUFjLENBQzNDLENBQUM7UUFFRix1RkFBdUY7UUFDdkYsMERBQTBEO1FBQzFELFdBQUksQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsR0FBRyxDQUFDLE1BQU0sRUFBRSxRQUFRLENBQUMsQ0FBQztRQUNwQyxVQUFVLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxFQUFFLENBQUMsV0FBSSxDQUFDLEVBQUUsQ0FBQyxTQUEwQixDQUFDLENBQUMsR0FBRyxDQUFDLE1BQU0sRUFBRSxRQUFRLENBQUMsQ0FBQyxDQUFDO1FBRTNGLE9BQU8sTUFBTSxDQUFDO0tBQ2Y7SUFFTSxpQkFBaUIsQ0FBQyxPQUFtQixFQUFFLFNBQXVCO1FBQ25FLE1BQU0sTUFBTSxHQUFHLGVBQUssQ0FBQyxjQUFjLENBQUM7WUFDbEMsT0FBTztZQUNQLE9BQU8sRUFBRSxDQUFDLGtCQUFrQixDQUFDO1lBQzdCLFlBQVksRUFBRSxJQUFJLENBQUMsd0JBQXdCLENBQUMsU0FBUyxDQUFDO1NBQ3ZELENBQUMsQ0FBQztRQUNILGtGQUFrRjtRQUNsRixPQUFPLE1BQU0sQ0FBQztLQUNmO0lBRU0sOEJBQThCLENBQUMsT0FBbUIsRUFBRSxVQUF1QixFQUFFLFlBQXFCO1FBQ3ZHLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDLElBQUksRUFBRSxHQUFHLFVBQVUsQ0FBQyxDQUFDLENBQUM7UUFDdkUsTUFBTSxNQUFNLEdBQUcscUJBQXFCLFlBQVksSUFBSSxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxXQUFXLEVBQUUsRUFBRSxDQUFDO1FBQzFGLE1BQU0sY0FBYyxHQUE4QixFQUFFLENBQUM7UUFDckQsY0FBYyxDQUFDLG1CQUFtQixNQUFNLEVBQUUsQ0FBQyxHQUFHLFFBQVEsQ0FBQztRQUV2RCxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsaUJBQWlCLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDL0MsTUFBTSxDQUFDLGtCQUFtQixDQUFDLFlBQVksQ0FDckMsMEJBQTBCLEVBQUUsY0FBYyxDQUMzQyxDQUFDO1FBRUYsdUZBQXVGO1FBQ3ZGLDBEQUEwRDtRQUMxRCxXQUFJLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLEdBQUcsQ0FBQyxNQUFNLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDcEMsVUFBVSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsRUFBRSxDQUFDLFdBQUksQ0FBQyxFQUFFLENBQUMsU0FBMEIsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxNQUFNLEVBQUUsUUFBUSxDQUFDLENBQUMsQ0FBQztRQUUzRixPQUFPLE1BQU0sQ0FBQztLQUNmO0lBRU8sd0JBQXdCLENBQUMsU0FBdUI7UUFDdEQsTUFBTSxLQUFLLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUM3QixNQUFNLFlBQVksR0FBYTtZQUM3QixPQUFPLEtBQUssQ0FBQyxTQUFTLFFBQVEsS0FBSyxDQUFDLE1BQU0sSUFBSSxLQUFLLENBQUMsT0FBTyxXQUFXLElBQUksQ0FBQyxRQUFRLEVBQUU7U0FDdEYsQ0FBQztRQUNGLE1BQU0saUJBQWlCLEdBQUcsT0FBTyxLQUFLLENBQUMsU0FBUyxRQUFRLEtBQUssQ0FBQyxNQUFNLElBQUksS0FBSyxDQUFDLE9BQU8sV0FBVyxDQUFDO1FBQ2pHLElBQUksU0FBUyxFQUFFO1lBQ2IsU0FBUyxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsRUFBRSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsR0FBRyxpQkFBaUIsSUFBSSxRQUFRLEVBQUUsVUFBVSxFQUFFLENBQUMsQ0FBQyxDQUFDO1NBQ2xHO2FBQU07WUFDTCxZQUFZLENBQUMsSUFBSSxDQUFDLEdBQUcsaUJBQWlCLElBQUksQ0FBQyxDQUFDO1NBQzdDO1FBQ0QsT0FBTyxZQUFZLENBQUM7S0FDckI7SUFFTyx5QkFBeUIsQ0FBQyxVQUF1QjtRQUN2RCxNQUFNLEdBQUcsR0FBRyxNQUFNLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ3JDLFVBQVUsQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLFlBQUssQ0FBQyxRQUFRLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3ZFLE9BQU8sR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQztLQUMxQjtDQUNGO0FBRUQ7O0dBRUc7QUFDSCxNQUFhLE1BQU8sU0FBUSxVQUFVO0lBeUJwQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQWtCO1FBQzFELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsWUFBWSxFQUFFLEtBQUssQ0FBQyxVQUFVO1NBQy9CLENBQUMsQ0FBQzs7Ozs7OytDQTVCTSxNQUFNOzs7O1FBOEJmLElBQUksQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFMUIsTUFBTSxRQUFRLEdBQUcsSUFBSSx5QkFBUyxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDL0MsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLGdCQUFnQjtZQUN4QyxZQUFZLEVBQUUsS0FBSyxDQUFDLFlBQVk7WUFDaEMsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTO1lBQzFCLFFBQVEsRUFBRSxLQUFLLENBQUMsYUFBYSxFQUFFLE1BQU07WUFDckMsSUFBSSxFQUFFLEtBQUssQ0FBQyxJQUFJO1lBQ2hCLGtCQUFrQixFQUFFLEtBQUssQ0FBQyxpQkFBaUIsSUFBSSxLQUFLO1lBQ3BELElBQUksRUFBRSxLQUFLLENBQUMsSUFBSSxFQUFFLFdBQVcsQ0FBQyxFQUFFLFFBQVEsRUFBRSwyQkFBb0IsQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUN0RSxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVU7WUFDNUIsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVLElBQUksbUJBQW1CLENBQUMsbUJBQW1CO1NBQ3hFLENBQUMsQ0FBQztRQUNILFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUM7UUFFakQsSUFBSSxLQUFLLENBQUMsVUFBVTtZQUFFLFdBQUksQ0FBQyxFQUFFLENBQUMsUUFBUSxDQUFDLENBQUMsR0FBRyxDQUFDLE1BQU0sRUFBRSxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUM7UUFFdEUsSUFBSSxDQUFDLFFBQVEsR0FBRyxRQUFRLENBQUMsR0FBRyxDQUFDO1FBQzdCLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxLQUFLLENBQUMsZ0JBQWdCLENBQUM7UUFDL0MsSUFBSSxDQUFDLGFBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDO1FBRXpDLElBQUksSUFBSSxDQUFDLGFBQWEsRUFBRTtZQUN0QiwwR0FBMEc7WUFDMUcsTUFBTSxTQUFTLEdBQ2IsSUFBSSw2QkFBbUIsQ0FBQyxPQUFPLFlBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsTUFBTSxnQkFBZ0IsRUFBRSxJQUFJLDhCQUFvQixFQUFFLENBQUMsQ0FBQyxjQUFjLENBQUM7Z0JBQy9HLFlBQVksRUFBRTtvQkFDWixtQkFBbUIsRUFBRSxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU87aUJBQzVDO2FBQ0YsQ0FBQyxDQUFDO1lBQ0wsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLGFBQWEsQ0FBQyxLQUFLLENBQUMsU0FBUztZQUM5QyxrRkFBa0Y7WUFDbEYsaUJBQWlCLEVBQ2pCLHFDQUFxQyxDQUN0QyxDQUFDO1lBQ0YsSUFBSSxLQUFLLENBQUMsVUFBVSxFQUFFO2dCQUNwQiwyRUFBMkU7Z0JBQzNFLEtBQUssQ0FBQyxrQkFBa0IsRUFBRSxVQUFVLENBQUMsZ0JBQWdCLENBQUMsQ0FBQzthQUN4RDtTQUNGO0tBQ0Y7SUFwRUQ7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLG9CQUFvQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXVCOzs7Ozs7Ozs7O1FBQ3RGLE1BQU0sTUFBTyxTQUFRLFVBQVU7WUFBL0I7O2dCQUNrQixhQUFRLEdBQUcsS0FBSyxDQUFDLFFBQVEsQ0FBQztnQkFDMUIscUJBQWdCLEdBQUcsS0FBSyxDQUFDLGdCQUFnQixDQUFDO2dCQUMxQyxrQkFBYSxHQUFHLEtBQUssQ0FBQyxhQUFhLENBQUM7WUFDdEQsQ0FBQztTQUFBO1FBQ0QsaUVBQWlFO1FBQ2pFLElBQUksQ0FBQyxZQUFLLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLEVBQUU7WUFDckYsTUFBTSxJQUFJLEtBQUssQ0FBQyxrSEFBa0gsQ0FBQyxDQUFDO1NBQ3JJO1FBQ0QsT0FBTyxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7S0FDOUI7SUFvRFMsYUFBYSxDQUFDLEtBQWtCOzs7Ozs7Ozs7O1FBQ3hDLElBQUksQ0FBQyxDQUFDLEtBQUssQ0FBQyxJQUFJLElBQUksS0FBSyxDQUFDLFVBQVUsQ0FBQyxFQUFFO1lBQ3JDLE1BQU0sSUFBSSxLQUFLLENBQUMscURBQXFELENBQUMsQ0FBQztTQUN4RTtRQUVELElBQUksS0FBSyxDQUFDLFVBQVUsSUFBSSxDQUFDLFlBQUssQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMscUJBQXFCLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsRUFBRTtZQUM5RyxNQUFNLElBQUksS0FBSyxDQUFDLGdIQUFnSCxDQUFDLENBQUM7U0FDbkk7UUFFRCxJQUFJLEtBQUssQ0FBQyxhQUFhLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFO1lBQzNDLE1BQU0sSUFBSSxLQUFLLENBQUMsNkRBQTZELENBQUMsQ0FBQztTQUNoRjtRQUVELElBQ0UsS0FBSyxDQUFDLFVBQVU7WUFDaEI7Z0JBQ0UsbUJBQW1CLENBQUMsb0JBQW9CO2dCQUN4QyxtQkFBbUIsQ0FBQyx3QkFBd0I7YUFDN0MsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQztZQUM1QixDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQ1g7WUFDQSxNQUFNLElBQUksS0FBSyxDQUNiLHVHQUF1RyxDQUN4RyxDQUFDO1NBQ0g7UUFFRCxJQUFJLEtBQUssQ0FBQyxJQUFJLEVBQUU7WUFDZCxNQUFNLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxJQUFJLG1CQUFtQixDQUFDLG1CQUFtQixDQUFDO1lBQy9FLElBQ0UsQ0FBQztnQkFDQyxtQkFBbUIsQ0FBQyxvQkFBb0I7Z0JBQ3hDLG1CQUFtQixDQUFDLHdCQUF3QjtnQkFDNUMsbUJBQW1CLENBQUMsdUJBQXVCO2FBQzVDLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxFQUN0QjtnQkFDQSxNQUFNLElBQUksS0FBSyxDQUNiLHNJQUFzSSxDQUN2SSxDQUFDO2FBQ0g7WUFDRCxrQ0FBa0M7WUFDbEMsb0dBQW9HO1lBQ3BHLE1BQU0sVUFBVSxHQUFvRCxFQUFFLENBQUM7WUFDdkUsVUFBVSxDQUFDLG1CQUFtQixDQUFDLHVCQUF1QixDQUFDLEdBQUcsRUFBRSxHQUFHLEVBQUUsSUFBSSxFQUFFLEdBQUcsRUFBRSxLQUFLLEVBQUUsQ0FBQztZQUNwRixVQUFVLENBQUMsbUJBQW1CLENBQUMsb0JBQW9CLENBQUMsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEtBQUssRUFBRSxDQUFDO1lBQ2hGLFVBQVUsQ0FBQyxtQkFBbUIsQ0FBQyx3QkFBd0IsQ0FBQyxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsS0FBSyxFQUFFLENBQUM7WUFDcEYsTUFBTSxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxVQUFVLENBQUMsVUFBVSxDQUFDLENBQUM7WUFDNUMsSUFBSSxLQUFLLENBQUMsSUFBSSxHQUFHLEdBQUcsSUFBSSxLQUFLLENBQUMsSUFBSSxHQUFHLEdBQUcsRUFBRTtnQkFDeEMsTUFBTSxJQUFJLEtBQUssQ0FBQyxLQUFLLFVBQVUsbUNBQW1DLEdBQUcsUUFBUSxHQUFHLEdBQUcsQ0FBQyxDQUFDO2FBQ3RGO1lBRUQscUNBQXFDO1lBQ3JDLDRFQUE0RTtZQUM1RSxNQUFNLGFBQWEsR0FBOEIsRUFBRSxDQUFDO1lBQ3BELGFBQWEsQ0FBQyxtQkFBbUIsQ0FBQyx1QkFBdUIsQ0FBQyxHQUFHLEdBQUcsQ0FBQztZQUNqRSxhQUFhLENBQUMsbUJBQW1CLENBQUMsb0JBQW9CLENBQUMsR0FBRyxFQUFFLENBQUM7WUFDN0QsYUFBYSxDQUFDLG1CQUFtQixDQUFDLHdCQUF3QixDQUFDLEdBQUcsR0FBRyxDQUFDO1lBQ2xFLE1BQU0sWUFBWSxHQUFHLGFBQWEsQ0FBQyxVQUFVLENBQUMsQ0FBQztZQUMvQyxJQUFJLEtBQUssQ0FBQyxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxHQUFHLFlBQVksR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFLFFBQVEsRUFBRSwyQkFBb0IsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLEVBQUU7Z0JBQy9HLE1BQU0sSUFBSSxLQUFLLENBQUMsS0FBSyxVQUFVLDBDQUEwQyxZQUFZLFlBQVksQ0FBQyxDQUFDO2FBQ3BHO1NBQ0Y7UUFFRCxJQUFJLEtBQUssQ0FBQyxpQkFBaUIsRUFBRTtZQUMzQixNQUFNLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxJQUFJLG1CQUFtQixDQUFDLG1CQUFtQixDQUFDO1lBQy9FLElBQ0UsQ0FBQztnQkFDQyxtQkFBbUIsQ0FBQyxvQkFBb0I7Z0JBQ3hDLG1CQUFtQixDQUFDLHdCQUF3QjthQUM3QyxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsRUFDdEI7Z0JBQ0EsTUFBTSxJQUFJLEtBQUssQ0FBQyx5R0FBeUcsQ0FBQyxDQUFDO2FBQzVIO1NBQ0Y7UUFFRCxJQUFJLEtBQUssQ0FBQyxJQUFJLEVBQUU7WUFDZCxNQUFNLElBQUksR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxFQUFFLFFBQVEsRUFBRSwyQkFBb0IsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO1lBQzdFLHlDQUF5QztZQUN6QyxvR0FBb0c7WUFDcEcsTUFBTSxVQUFVLEdBQW9ELEVBQUUsQ0FBQztZQUN2RSxVQUFVLENBQUMsbUJBQW1CLENBQUMsbUJBQW1CLENBQUMsR0FBRyxFQUFFLEdBQUcsRUFBRSxDQUFDLEVBQUUsR0FBRyxFQUFFLEtBQUssRUFBRSxDQUFDO1lBQzdFLFVBQVUsQ0FBQyxtQkFBbUIsQ0FBQyx1QkFBdUIsQ0FBQyxHQUFHLEVBQUUsR0FBRyxFQUFFLENBQUMsRUFBRSxHQUFHLEVBQUUsS0FBSyxFQUFFLENBQUM7WUFDakYsVUFBVSxDQUFDLG1CQUFtQixDQUFDLG9CQUFvQixDQUFDLEdBQUcsRUFBRSxHQUFHLEVBQUUsQ0FBQyxFQUFFLEdBQUcsRUFBRSxLQUFLLEVBQUUsQ0FBQztZQUM5RSxVQUFVLENBQUMsbUJBQW1CLENBQUMsd0JBQXdCLENBQUMsR0FBRyxFQUFFLEdBQUcsRUFBRSxDQUFDLEVBQUUsR0FBRyxFQUFFLEtBQUssRUFBRSxDQUFDO1lBQ2xGLFVBQVUsQ0FBQyxtQkFBbUIsQ0FBQyx3QkFBd0IsQ0FBQyxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsS0FBSyxFQUFFLENBQUM7WUFDcEYsVUFBVSxDQUFDLG1CQUFtQixDQUFDLFFBQVEsQ0FBQyxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLEVBQUUsS0FBSyxFQUFFLENBQUM7WUFDcEUsVUFBVSxDQUFDLG1CQUFtQixDQUFDLFFBQVEsQ0FBQyxHQUFHLEVBQUUsR0FBRyxFQUFFLENBQUMsRUFBRSxHQUFHLEVBQUUsSUFBSSxFQUFFLENBQUM7WUFDakUsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsSUFBSSxtQkFBbUIsQ0FBQyxtQkFBbUIsQ0FBQztZQUMvRSxNQUFNLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLFVBQVUsQ0FBQyxVQUFVLENBQUMsQ0FBQztZQUM1QyxJQUFJLElBQUksR0FBRyxHQUFHLElBQUksSUFBSSxHQUFHLEdBQUcsRUFBRTtnQkFDNUIsTUFBTSxJQUFJLEtBQUssQ0FBQyxLQUFLLFVBQVUsOEJBQThCLEdBQUcsWUFBWSxHQUFHLGVBQWUsQ0FBQyxDQUFDO2FBQ2pHO1NBQ0Y7S0FDRjs7QUFuS0gsd0JBb0tDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY3J5cHRvIGZyb20gJ2NyeXB0byc7XG5cbmltcG9ydCB7IEFjY291bnRSb290UHJpbmNpcGFsLCBHcmFudCwgSUdyYW50YWJsZSB9IGZyb20gJy4uLy4uL2F3cy1pYW0nO1xuaW1wb3J0IHsgSUtleSwgVmlhU2VydmljZVByaW5jaXBhbCB9IGZyb20gJy4uLy4uL2F3cy1rbXMnO1xuaW1wb3J0IHsgSVJlc291cmNlLCBSZXNvdXJjZSwgU2l6ZSwgU2l6ZVJvdW5kaW5nQmVoYXZpb3IsIFN0YWNrLCBUb2tlbiwgVGFncywgTmFtZXMsIFJlbW92YWxQb2xpY3kgfSBmcm9tICcuLi8uLi9jb3JlJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgQ2ZuVm9sdW1lIH0gZnJvbSAnLi9lYzIuZ2VuZXJhdGVkJztcbmltcG9ydCB7IElJbnN0YW5jZSB9IGZyb20gJy4vaW5zdGFuY2UnO1xuXG4vLyB2MiAtIGtlZXAgdGhpcyBpbXBvcnQgYXMgYSBzZXBhcmF0ZSBzZWN0aW9uIHRvIHJlZHVjZSBtZXJnZSBjb25mbGljdCB3aGVuIGZvcndhcmQgbWVyZ2luZyB3aXRoIHRoZSB2MiBicmFuY2guXG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmVcbmltcG9ydCB7IENvbnN0cnVjdCBhcyBDb3JlQ29uc3RydWN0IH0gZnJvbSAnLi4vLi4vY29yZSc7XG5cblxuLyoqXG4gKiBCbG9jayBkZXZpY2VcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBCbG9ja0RldmljZSB7XG4gIC8qKlxuICAgKiBUaGUgZGV2aWNlIG5hbWUgZXhwb3NlZCB0byB0aGUgRUMyIGluc3RhbmNlXG4gICAqXG4gICAqIEZvciBleGFtcGxlLCBhIHZhbHVlIGxpa2UgYC9kZXYvc2RoYCwgYHh2ZGhgLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NFQzIvbGF0ZXN0L1VzZXJHdWlkZS9kZXZpY2VfbmFtaW5nLmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IGRldmljZU5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogRGVmaW5lcyB0aGUgYmxvY2sgZGV2aWNlIHZvbHVtZSwgdG8gYmUgZWl0aGVyIGFuIEFtYXpvbiBFQlMgdm9sdW1lIG9yIGFuIGVwaGVtZXJhbCBpbnN0YW5jZSBzdG9yZSB2b2x1bWVcbiAgICpcbiAgICogRm9yIGV4YW1wbGUsIGEgdmFsdWUgbGlrZSBgQmxvY2tEZXZpY2VWb2x1bWUuZWJzKDE1KWAsIGBCbG9ja0RldmljZVZvbHVtZS5lcGhlbWVyYWwoMClgLlxuICAgKi9cbiAgcmVhZG9ubHkgdm9sdW1lOiBCbG9ja0RldmljZVZvbHVtZTtcblxuICAvKipcbiAgICogSWYgZmFsc2UsIHRoZSBkZXZpY2UgbWFwcGluZyB3aWxsIGJlIHN1cHByZXNzZWQuXG4gICAqIElmIHNldCB0byBmYWxzZSBmb3IgdGhlIHJvb3QgZGV2aWNlLCB0aGUgaW5zdGFuY2UgbWlnaHQgZmFpbCB0aGUgQW1hem9uIEVDMiBoZWFsdGggY2hlY2suXG4gICAqIEFtYXpvbiBFQzIgQXV0byBTY2FsaW5nIGxhdW5jaGVzIGEgcmVwbGFjZW1lbnQgaW5zdGFuY2UgaWYgdGhlIGluc3RhbmNlIGZhaWxzIHRoZSBoZWFsdGggY2hlY2suXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWUgLSBkZXZpY2UgbWFwcGluZyBpcyBsZWZ0IHVudG91Y2hlZFxuICAgKi9cbiAgcmVhZG9ubHkgbWFwcGluZ0VuYWJsZWQ/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIEJhc2UgYmxvY2sgZGV2aWNlIG9wdGlvbnMgZm9yIGFuIEVCUyB2b2x1bWVcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFYnNEZXZpY2VPcHRpb25zQmFzZSB7XG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgd2hldGhlciB0byBkZWxldGUgdGhlIHZvbHVtZSB3aGVuIHRoZSBpbnN0YW5jZSBpcyB0ZXJtaW5hdGVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHRydWUgZm9yIEFtYXpvbiBFQzIgQXV0byBTY2FsaW5nLCBmYWxzZSBvdGhlcndpc2UgKGUuZy4gRUJTKVxuICAgKi9cbiAgcmVhZG9ubHkgZGVsZXRlT25UZXJtaW5hdGlvbj86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgSS9PIG9wZXJhdGlvbnMgcGVyIHNlY29uZCAoSU9QUykgdG8gcHJvdmlzaW9uIGZvciB0aGUgdm9sdW1lLlxuICAgKlxuICAgKiBNdXN0IG9ubHkgYmUgc2V0IGZvciB7QGxpbmsgdm9sdW1lVHlwZX06IHtAbGluayBFYnNEZXZpY2VWb2x1bWVUeXBlLklPMX1cbiAgICpcbiAgICogVGhlIG1heGltdW0gcmF0aW8gb2YgSU9QUyB0byB2b2x1bWUgc2l6ZSAoaW4gR2lCKSBpcyA1MDoxLCBzbyBmb3IgNSwwMDAgcHJvdmlzaW9uZWQgSU9QUyxcbiAgICogeW91IG5lZWQgYXQgbGVhc3QgMTAwIEdpQiBzdG9yYWdlIG9uIHRoZSB2b2x1bWUuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0VDMi9sYXRlc3QvVXNlckd1aWRlL0VCU1ZvbHVtZVR5cGVzLmh0bWxcbiAgICpcbiAgICogQGRlZmF1bHQgLSBub25lLCByZXF1aXJlZCBmb3Ige0BsaW5rIEVic0RldmljZVZvbHVtZVR5cGUuSU8xfVxuICAgKi9cbiAgcmVhZG9ubHkgaW9wcz86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIEVCUyB2b2x1bWUgdHlwZVxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NFQzIvbGF0ZXN0L1VzZXJHdWlkZS9FQlNWb2x1bWVUeXBlcy5odG1sXG4gICAqXG4gICAqIEBkZWZhdWx0IHtAbGluayBFYnNEZXZpY2VWb2x1bWVUeXBlLkdQMn1cbiAgICovXG4gIHJlYWRvbmx5IHZvbHVtZVR5cGU/OiBFYnNEZXZpY2VWb2x1bWVUeXBlO1xufVxuXG4vKipcbiAqIEJsb2NrIGRldmljZSBvcHRpb25zIGZvciBhbiBFQlMgdm9sdW1lXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRWJzRGV2aWNlT3B0aW9ucyBleHRlbmRzIEVic0RldmljZU9wdGlvbnNCYXNlIHtcbiAgLyoqXG4gICAqIFNwZWNpZmllcyB3aGV0aGVyIHRoZSBFQlMgdm9sdW1lIGlzIGVuY3J5cHRlZC5cbiAgICogRW5jcnlwdGVkIEVCUyB2b2x1bWVzIGNhbiBvbmx5IGJlIGF0dGFjaGVkIHRvIGluc3RhbmNlcyB0aGF0IHN1cHBvcnQgQW1hem9uIEVCUyBlbmNyeXB0aW9uXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0VDMi9sYXRlc3QvVXNlckd1aWRlL0VCU0VuY3J5cHRpb24uaHRtbCNFQlNFbmNyeXB0aW9uX3N1cHBvcnRlZF9pbnN0YW5jZXNcbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGVuY3J5cHRlZD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBBUk4gb2YgdGhlIEFXUyBLZXkgTWFuYWdlbWVudCBTZXJ2aWNlIChBV1MgS01TKSBDTUsgdXNlZCBmb3IgZW5jcnlwdGlvbi5cbiAgICpcbiAgICogWW91IGhhdmUgdG8gZW5zdXJlIHRoYXQgdGhlIEtNUyBDTUsgaGFzIHRoZSBjb3JyZWN0IHBlcm1pc3Npb25zIHRvIGJlIHVzZWQgYnkgdGhlIHNlcnZpY2UgbGF1bmNoaW5nIHRoZSBlYzIgaW5zdGFuY2VzLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NFQzIvbGF0ZXN0L1VzZXJHdWlkZS9FQlNFbmNyeXB0aW9uLmh0bWwjZWJzLWVuY3J5cHRpb24tcmVxdWlyZW1lbnRzXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gSWYgZW5jcnlwdGVkIGlzIHRydWUsIHRoZSBkZWZhdWx0IGF3cy9lYnMgS01TIGtleSB3aWxsIGJlIHVzZWQuXG4gICAqL1xuICByZWFkb25seSBrbXNLZXk/OiBJS2V5O1xufVxuXG4vKipcbiAqIEJsb2NrIGRldmljZSBvcHRpb25zIGZvciBhbiBFQlMgdm9sdW1lIGNyZWF0ZWQgZnJvbSBhIHNuYXBzaG90XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRWJzRGV2aWNlU25hcHNob3RPcHRpb25zIGV4dGVuZHMgRWJzRGV2aWNlT3B0aW9uc0Jhc2Uge1xuICAvKipcbiAgICogVGhlIHZvbHVtZSBzaXplLCBpbiBHaWJpYnl0ZXMgKEdpQilcbiAgICpcbiAgICogSWYgeW91IHNwZWNpZnkgdm9sdW1lU2l6ZSwgaXQgbXVzdCBiZSBlcXVhbCBvciBncmVhdGVyIHRoYW4gdGhlIHNpemUgb2YgdGhlIHNuYXBzaG90LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFRoZSBzbmFwc2hvdCBzaXplXG4gICAqL1xuICByZWFkb25seSB2b2x1bWVTaXplPzogbnVtYmVyO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgb2YgYW4gRUJTIGJsb2NrIGRldmljZVxuICovXG5leHBvcnQgaW50ZXJmYWNlIEVic0RldmljZVByb3BzIGV4dGVuZHMgRWJzRGV2aWNlU25hcHNob3RPcHRpb25zLCBFYnNEZXZpY2VPcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBzbmFwc2hvdCBJRCBvZiB0aGUgdm9sdW1lIHRvIHVzZVxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIHNuYXBzaG90IHdpbGwgYmUgdXNlZFxuICAgKi9cbiAgcmVhZG9ubHkgc25hcHNob3RJZD86IHN0cmluZztcbn1cblxuLyoqXG4gKiBEZXNjcmliZXMgYSBibG9jayBkZXZpY2UgbWFwcGluZyBmb3IgYW4gRUMyIGluc3RhbmNlIG9yIEF1dG8gU2NhbGluZyBncm91cC5cbiAqL1xuZXhwb3J0IGNsYXNzIEJsb2NrRGV2aWNlVm9sdW1lIHtcbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBuZXcgRWxhc3RpYyBCbG9jayBTdG9yYWdlIGRldmljZVxuICAgKlxuICAgKiBAcGFyYW0gdm9sdW1lU2l6ZSBUaGUgdm9sdW1lIHNpemUsIGluIEdpYmlieXRlcyAoR2lCKVxuICAgKiBAcGFyYW0gb3B0aW9ucyBhZGRpdGlvbmFsIGRldmljZSBvcHRpb25zXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGVicyh2b2x1bWVTaXplOiBudW1iZXIsIG9wdGlvbnM6IEVic0RldmljZU9wdGlvbnMgPSB7fSk6IEJsb2NrRGV2aWNlVm9sdW1lIHtcbiAgICByZXR1cm4gbmV3IHRoaXMoeyAuLi5vcHRpb25zLCB2b2x1bWVTaXplIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBuZXcgRWxhc3RpYyBCbG9jayBTdG9yYWdlIGRldmljZSBmcm9tIGFuIGV4aXN0aW5nIHNuYXBzaG90XG4gICAqXG4gICAqIEBwYXJhbSBzbmFwc2hvdElkIFRoZSBzbmFwc2hvdCBJRCBvZiB0aGUgdm9sdW1lIHRvIHVzZVxuICAgKiBAcGFyYW0gb3B0aW9ucyBhZGRpdGlvbmFsIGRldmljZSBvcHRpb25zXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGVic0Zyb21TbmFwc2hvdChzbmFwc2hvdElkOiBzdHJpbmcsIG9wdGlvbnM6IEVic0RldmljZVNuYXBzaG90T3B0aW9ucyA9IHt9KTogQmxvY2tEZXZpY2VWb2x1bWUge1xuICAgIHJldHVybiBuZXcgdGhpcyh7IC4uLm9wdGlvbnMsIHNuYXBzaG90SWQgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIHZpcnR1YWwsIGVwaGVtZXJhbCBkZXZpY2UuXG4gICAqIFRoZSBuYW1lIHdpbGwgYmUgaW4gdGhlIGZvcm0gZXBoZW1lcmFse3ZvbHVtZUluZGV4fS5cbiAgICpcbiAgICogQHBhcmFtIHZvbHVtZUluZGV4IHRoZSB2b2x1bWUgaW5kZXguIE11c3QgYmUgZXF1YWwgb3IgZ3JlYXRlciB0aGFuIDBcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZXBoZW1lcmFsKHZvbHVtZUluZGV4OiBudW1iZXIpIHtcbiAgICBpZiAodm9sdW1lSW5kZXggPCAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYHZvbHVtZUluZGV4IG11c3QgYmUgYSBudW1iZXIgc3RhcnRpbmcgZnJvbSAwLCBnb3QgXCIke3ZvbHVtZUluZGV4fVwiYCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyB0aGlzKHVuZGVmaW5lZCwgYGVwaGVtZXJhbCR7dm9sdW1lSW5kZXh9YCk7XG4gIH1cblxuICAvKipcbiAgICogQHBhcmFtIGVic0RldmljZSBFQlMgZGV2aWNlIGluZm9cbiAgICogQHBhcmFtIHZpcnR1YWxOYW1lIFZpcnR1YWwgZGV2aWNlIG5hbWVcbiAgICovXG4gIHByb3RlY3RlZCBjb25zdHJ1Y3RvcihwdWJsaWMgcmVhZG9ubHkgZWJzRGV2aWNlPzogRWJzRGV2aWNlUHJvcHMsIHB1YmxpYyByZWFkb25seSB2aXJ0dWFsTmFtZT86IHN0cmluZykge1xuICB9XG59XG5cbi8qKlxuICogU3VwcG9ydGVkIEVCUyB2b2x1bWUgdHlwZXMgZm9yIGJsb2NrRGV2aWNlc1xuICovXG5leHBvcnQgZW51bSBFYnNEZXZpY2VWb2x1bWVUeXBlIHtcbiAgLyoqXG4gICAqIE1hZ25ldGljXG4gICAqL1xuICBTVEFOREFSRCA9ICdzdGFuZGFyZCcsXG5cbiAgLyoqXG4gICAqICBQcm92aXNpb25lZCBJT1BTIFNTRCAtIElPMVxuICAgKi9cbiAgSU8xID0gJ2lvMScsXG5cbiAgLyoqXG4gICAqICBQcm92aXNpb25lZCBJT1BTIFNTRCAtIElPMlxuICAgKi9cbiAgSU8yID0gJ2lvMicsXG5cbiAgLyoqXG4gICAqIEdlbmVyYWwgUHVycG9zZSBTU0QgLSBHUDJcbiAgICovXG4gIEdQMiA9ICdncDInLFxuXG4gIC8qKlxuICAgKiBHZW5lcmFsIFB1cnBvc2UgU1NEIC0gR1AzXG4gICAqL1xuICBHUDMgPSAnZ3AzJyxcblxuICAvKipcbiAgICogVGhyb3VnaHB1dCBPcHRpbWl6ZWQgSEREXG4gICAqL1xuICBTVDEgPSAnc3QxJyxcblxuICAvKipcbiAgICogQ29sZCBIRERcbiAgICovXG4gIFNDMSA9ICdzYzEnLFxuXG4gIC8qKlxuICAgKiBHZW5lcmFsIHB1cnBvc2UgU1NEIHZvbHVtZSAoR1AyKSB0aGF0IGJhbGFuY2VzIHByaWNlIGFuZCBwZXJmb3JtYW5jZSBmb3IgYSB3aWRlIHZhcmlldHkgb2Ygd29ya2xvYWRzLlxuICAgKi9cbiAgR0VORVJBTF9QVVJQT1NFX1NTRCA9IEdQMixcblxuICAvKipcbiAgICogR2VuZXJhbCBwdXJwb3NlIFNTRCB2b2x1bWUgKEdQMykgdGhhdCBiYWxhbmNlcyBwcmljZSBhbmQgcGVyZm9ybWFuY2UgZm9yIGEgd2lkZSB2YXJpZXR5IG9mIHdvcmtsb2Fkcy5cbiAgICovXG4gIEdFTkVSQUxfUFVSUE9TRV9TU0RfR1AzID0gR1AzLFxuXG4gIC8qKlxuICAgKiBIaWdoZXN0LXBlcmZvcm1hbmNlIFNTRCB2b2x1bWUgKElPMSkgZm9yIG1pc3Npb24tY3JpdGljYWwgbG93LWxhdGVuY3kgb3IgaGlnaC10aHJvdWdocHV0IHdvcmtsb2Fkcy5cbiAgICovXG4gIFBST1ZJU0lPTkVEX0lPUFNfU1NEID0gSU8xLFxuXG4gIC8qKlxuICAgKiBIaWdoZXN0LXBlcmZvcm1hbmNlIFNTRCB2b2x1bWUgKElPMikgZm9yIG1pc3Npb24tY3JpdGljYWwgbG93LWxhdGVuY3kgb3IgaGlnaC10aHJvdWdocHV0IHdvcmtsb2Fkcy5cbiAgICovXG4gIFBST1ZJU0lPTkVEX0lPUFNfU1NEX0lPMiA9IElPMixcblxuICAvKipcbiAgICogTG93LWNvc3QgSEREIHZvbHVtZSBkZXNpZ25lZCBmb3IgZnJlcXVlbnRseSBhY2Nlc3NlZCwgdGhyb3VnaHB1dC1pbnRlbnNpdmUgd29ya2xvYWRzLlxuICAgKi9cbiAgVEhST1VHSFBVVF9PUFRJTUlaRURfSEREID0gU1QxLFxuXG4gIC8qKlxuICAgKiBMb3dlc3QgY29zdCBIREQgdm9sdW1lIGRlc2lnbmVkIGZvciBsZXNzIGZyZXF1ZW50bHkgYWNjZXNzZWQgd29ya2xvYWRzLlxuICAgKi9cbiAgQ09MRF9IREQgPSBTQzEsXG5cbiAgLyoqXG4gICAqIE1hZ25ldGljIHZvbHVtZXMgYXJlIGJhY2tlZCBieSBtYWduZXRpYyBkcml2ZXMgYW5kIGFyZSBzdWl0ZWQgZm9yIHdvcmtsb2FkcyB3aGVyZSBkYXRhIGlzIGFjY2Vzc2VkIGluZnJlcXVlbnRseSwgYW5kIHNjZW5hcmlvcyB3aGVyZSBsb3ctY29zdFxuICAgKiBzdG9yYWdlIGZvciBzbWFsbCB2b2x1bWUgc2l6ZXMgaXMgaW1wb3J0YW50LlxuICAgKi9cbiAgTUFHTkVUSUMgPSBTVEFOREFSRCxcbn1cblxuLyoqXG4gKiBBbiBFQlMgVm9sdW1lIGluIEFXUyBFQzIuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSVZvbHVtZSBleHRlbmRzIElSZXNvdXJjZSB7XG4gIC8qKlxuICAgKiBUaGUgRUJTIFZvbHVtZSdzIElEXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IHZvbHVtZUlkOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBhdmFpbGFiaWxpdHkgem9uZSB0aGF0IHRoZSBFQlMgVm9sdW1lIGlzIGNvbnRhaW5lZCB3aXRoaW4gKGV4OiB1cy13ZXN0LTJhKVxuICAgKi9cbiAgcmVhZG9ubHkgYXZhaWxhYmlsaXR5Wm9uZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgY3VzdG9tZXItbWFuYWdlZCBlbmNyeXB0aW9uIGtleSB0aGF0IGlzIHVzZWQgdG8gZW5jcnlwdCB0aGUgVm9sdW1lLlxuICAgKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSBlbmNyeXB0aW9uS2V5PzogSUtleTtcblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYXR0YWNoIHRoaXMgVm9sdW1lIHRvIGFuIGluc3RhbmNlLlxuICAgKiBDQVVUSU9OOiBHcmFudGluZyBhbiBpbnN0YW5jZSBwZXJtaXNzaW9uIHRvIGF0dGFjaCB0byBpdHNlbGYgdXNpbmcgdGhpcyBtZXRob2Qgd2lsbCBsZWFkIHRvXG4gICAqIGFuIHVucmVzb2x2YWJsZSBjaXJjdWxhciByZWZlcmVuY2UgYmV0d2VlbiB0aGUgaW5zdGFuY2Ugcm9sZSBhbmQgdGhlIGluc3RhbmNlLlxuICAgKiBVc2Uge0BsaW5rIElWb2x1bWUuZ3JhbnRBdHRhY2hWb2x1bWVUb1NlbGZ9IHRvIGdyYW50IGFuIGluc3RhbmNlIHBlcm1pc3Npb24gdG8gYXR0YWNoIHRoaXNcbiAgICogdm9sdW1lIHRvIGl0c2VsZi5cbiAgICpcbiAgICogQHBhcmFtIGdyYW50ZWUgIHRoZSBwcmluY2lwYWwgYmVpbmcgZ3JhbnRlZCBwZXJtaXNzaW9uLlxuICAgKiBAcGFyYW0gaW5zdGFuY2VzIHRoZSBpbnN0YW5jZXMgdG8gd2hpY2ggcGVybWlzc2lvbiBpcyBiZWluZyBncmFudGVkIHRvIGF0dGFjaCB0aGlzXG4gICAqICAgICAgICAgICAgICAgICB2b2x1bWUgdG8uIElmIG5vdCBzcGVjaWZpZWQsIHRoZW4gcGVybWlzc2lvbiBpcyBncmFudGVkIHRvIGF0dGFjaFxuICAgKiAgICAgICAgICAgICAgICAgdG8gYWxsIGluc3RhbmNlcyBpbiB0aGlzIGFjY291bnQuXG4gICAqL1xuICBncmFudEF0dGFjaFZvbHVtZShncmFudGVlOiBJR3JhbnRhYmxlLCBpbnN0YW5jZXM/OiBJSW5zdGFuY2VbXSk6IEdyYW50O1xuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhdHRhY2ggdGhlIFZvbHVtZSBieSBhIFJlc291cmNlVGFnIGNvbmRpdGlvbi4gSWYgeW91IGFyZSBsb29raW5nIHRvXG4gICAqIGdyYW50IGFuIEluc3RhbmNlLCBBdXRvU2NhbGluZ0dyb3VwLCBFQzItRmxlZXQsIFNwb3RGbGVldCwgRUNTIGhvc3QsIGV0YyB0aGUgYWJpbGl0eSB0byBhdHRhY2hcbiAgICogdGhpcyB2b2x1bWUgdG8gKippdHNlbGYqKiB0aGVuIHRoaXMgaXMgdGhlIG1ldGhvZCB5b3Ugd2FudCB0byB1c2UuXG4gICAqXG4gICAqIFRoaXMgaXMgaW1wbGVtZW50ZWQgYnkgYWRkaW5nIGEgVGFnIHdpdGgga2V5IGBWb2x1bWVHcmFudEF0dGFjaC08c3VmZml4PmAgdG8gdGhlIGdpdmVuXG4gICAqIGNvbnN0cnVjdHMgYW5kIHRoaXMgVm9sdW1lLCBhbmQgdGhlbiBjb25kaXRpb25pbmcgdGhlIEdyYW50IHN1Y2ggdGhhdCB0aGUgZ3JhbnRlZSBpcyBvbmx5XG4gICAqIGdpdmVuIHRoZSBhYmlsaXR5IHRvIEF0dGFjaFZvbHVtZSBpZiBib3RoIHRoZSBWb2x1bWUgYW5kIHRoZSBkZXN0aW5hdGlvbiBJbnN0YW5jZSBoYXZlIHRoYXRcbiAgICogdGFnIGFwcGxpZWQgdG8gdGhlbS5cbiAgICpcbiAgICogQHBhcmFtIGdyYW50ZWUgICAgdGhlIHByaW5jaXBhbCBiZWluZyBncmFudGVkIHBlcm1pc3Npb24uXG4gICAqIEBwYXJhbSBjb25zdHJ1Y3RzIFRoZSBsaXN0IG9mIGNvbnN0cnVjdHMgdGhhdCB3aWxsIGhhdmUgdGhlIGdlbmVyYXRlZCByZXNvdXJjZSB0YWcgYXBwbGllZCB0byB0aGVtLlxuICAgKiBAcGFyYW0gdGFnS2V5U3VmZml4IEEgc3VmZml4IHRvIHVzZSBvbiB0aGUgZ2VuZXJhdGVkIFRhZyBrZXkgaW4gcGxhY2Ugb2YgdGhlIGdlbmVyYXRlZCBoYXNoIHZhbHVlLlxuICAgKiAgICAgICAgICAgICAgICAgICAgIERlZmF1bHRzIHRvIGEgaGFzaCBjYWxjdWxhdGVkIGZyb20gdGhpcyB2b2x1bWUgYW5kIGxpc3Qgb2YgY29uc3RydWN0cy4gKERFUFJFQ0FURUQpXG4gICAqL1xuICBncmFudEF0dGFjaFZvbHVtZUJ5UmVzb3VyY2VUYWcoZ3JhbnRlZTogSUdyYW50YWJsZSwgY29uc3RydWN0czogQ29uc3RydWN0W10sIHRhZ0tleVN1ZmZpeD86IHN0cmluZyk6IEdyYW50O1xuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZXRhY2ggdGhpcyBWb2x1bWUgZnJvbSBhbiBpbnN0YW5jZVxuICAgKiBDQVVUSU9OOiBHcmFudGluZyBhbiBpbnN0YW5jZSBwZXJtaXNzaW9uIHRvIGRldGFjaCBmcm9tIGl0c2VsZiB1c2luZyB0aGlzIG1ldGhvZCB3aWxsIGxlYWQgdG9cbiAgICogYW4gdW5yZXNvbHZhYmxlIGNpcmN1bGFyIHJlZmVyZW5jZSBiZXR3ZWVuIHRoZSBpbnN0YW5jZSByb2xlIGFuZCB0aGUgaW5zdGFuY2UuXG4gICAqIFVzZSB7QGxpbmsgSVZvbHVtZS5ncmFudERldGFjaFZvbHVtZUZyb21TZWxmfSB0byBncmFudCBhbiBpbnN0YW5jZSBwZXJtaXNzaW9uIHRvIGRldGFjaCB0aGlzXG4gICAqIHZvbHVtZSBmcm9tIGl0c2VsZi5cbiAgICpcbiAgICogQHBhcmFtIGdyYW50ZWUgIHRoZSBwcmluY2lwYWwgYmVpbmcgZ3JhbnRlZCBwZXJtaXNzaW9uLlxuICAgKiBAcGFyYW0gaW5zdGFuY2VzIHRoZSBpbnN0YW5jZXMgdG8gd2hpY2ggcGVybWlzc2lvbiBpcyBiZWluZyBncmFudGVkIHRvIGRldGFjaCB0aGlzXG4gICAqICAgICAgICAgICAgICAgICB2b2x1bWUgZnJvbS4gSWYgbm90IHNwZWNpZmllZCwgdGhlbiBwZXJtaXNzaW9uIGlzIGdyYW50ZWQgdG8gZGV0YWNoXG4gICAqICAgICAgICAgICAgICAgICBmcm9tIGFsbCBpbnN0YW5jZXMgaW4gdGhpcyBhY2NvdW50LlxuICAgKi9cbiAgZ3JhbnREZXRhY2hWb2x1bWUoZ3JhbnRlZTogSUdyYW50YWJsZSwgaW5zdGFuY2VzPzogSUluc3RhbmNlW10pOiBHcmFudDtcblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGV0YWNoIHRoZSBWb2x1bWUgYnkgYSBSZXNvdXJjZVRhZyBjb25kaXRpb24uXG4gICAqXG4gICAqIFRoaXMgaXMgaW1wbGVtZW50ZWQgdmlhIHRoZSBzYW1lIG1lY2hhbmlzbSBhcyB7QGxpbmsgSVZvbHVtZS5ncmFudEF0dGFjaFZvbHVtZUJ5UmVzb3VyY2VUYWd9LFxuICAgKiBhbmQgaXMgc3ViamVjdCB0byB0aGUgc2FtZSBjb25kaXRpb25zLlxuICAgKlxuICAgKiBAcGFyYW0gZ3JhbnRlZSAgICB0aGUgcHJpbmNpcGFsIGJlaW5nIGdyYW50ZWQgcGVybWlzc2lvbi5cbiAgICogQHBhcmFtIGNvbnN0cnVjdHMgVGhlIGxpc3Qgb2YgY29uc3RydWN0cyB0aGF0IHdpbGwgaGF2ZSB0aGUgZ2VuZXJhdGVkIHJlc291cmNlIHRhZyBhcHBsaWVkIHRvIHRoZW0uXG4gICAqIEBwYXJhbSB0YWdLZXlTdWZmaXggQSBzdWZmaXggdG8gdXNlIG9uIHRoZSBnZW5lcmF0ZWQgVGFnIGtleSBpbiBwbGFjZSBvZiB0aGUgZ2VuZXJhdGVkIGhhc2ggdmFsdWUuXG4gICAqICAgICAgICAgICAgICAgICAgICAgRGVmYXVsdHMgdG8gYSBoYXNoIGNhbGN1bGF0ZWQgZnJvbSB0aGlzIHZvbHVtZSBhbmQgbGlzdCBvZiBjb25zdHJ1Y3RzLiAoREVQUkVDQVRFRClcbiAgICovXG4gIGdyYW50RGV0YWNoVm9sdW1lQnlSZXNvdXJjZVRhZyhncmFudGVlOiBJR3JhbnRhYmxlLCBjb25zdHJ1Y3RzOiBDb25zdHJ1Y3RbXSwgdGFnS2V5U3VmZml4Pzogc3RyaW5nKTogR3JhbnQ7XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBvZiBhbiBFQlMgVm9sdW1lXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVm9sdW1lUHJvcHMge1xuICAvKipcbiAgICogVGhlIHZhbHVlIG9mIHRoZSBwaHlzaWNhbE5hbWUgcHJvcGVydHkgb2YgdGhpcyByZXNvdXJjZS5cbiAgICpcbiAgICogQGRlZmF1bHQgVGhlIHBoeXNpY2FsIG5hbWUgd2lsbCBiZSBhbGxvY2F0ZWQgYnkgQ2xvdWRGb3JtYXRpb24gYXQgZGVwbG95bWVudCB0aW1lXG4gICAqL1xuICByZWFkb25seSB2b2x1bWVOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgQXZhaWxhYmlsaXR5IFpvbmUgaW4gd2hpY2ggdG8gY3JlYXRlIHRoZSB2b2x1bWUuXG4gICAqL1xuICByZWFkb25seSBhdmFpbGFiaWxpdHlab25lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBzaXplIG9mIHRoZSB2b2x1bWUsIGluIEdpQnMuIFlvdSBtdXN0IHNwZWNpZnkgZWl0aGVyIGEgc25hcHNob3QgSUQgb3IgYSB2b2x1bWUgc2l6ZS5cbiAgICogU2VlIHtAbGluayBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTQ2xvdWRGb3JtYXRpb24vbGF0ZXN0L1VzZXJHdWlkZS9hd3MtcHJvcGVydGllcy1lYzItZWJzLXZvbHVtZS5odG1sfVxuICAgKiBmb3IgZGV0YWlscyBvbiB0aGUgYWxsb3dhYmxlIHNpemUgZm9yIGVhY2ggdHlwZSBvZiB2b2x1bWUuXG4gICAqXG4gICAqIEBkZWZhdWx0IElmIHlvdSdyZSBjcmVhdGluZyB0aGUgdm9sdW1lIGZyb20gYSBzbmFwc2hvdCBhbmQgZG9uJ3Qgc3BlY2lmeSBhIHZvbHVtZSBzaXplLCB0aGUgZGVmYXVsdCBpcyB0aGUgc25hcHNob3Qgc2l6ZS5cbiAgICovXG4gIHJlYWRvbmx5IHNpemU/OiBTaXplO1xuXG4gIC8qKlxuICAgKiBUaGUgc25hcHNob3QgZnJvbSB3aGljaCB0byBjcmVhdGUgdGhlIHZvbHVtZS4gWW91IG11c3Qgc3BlY2lmeSBlaXRoZXIgYSBzbmFwc2hvdCBJRCBvciBhIHZvbHVtZSBzaXplLlxuICAgKlxuICAgKiBAZGVmYXVsdCBUaGUgRUJTIHZvbHVtZSBpcyBub3QgY3JlYXRlZCBmcm9tIGEgc25hcHNob3QuXG4gICAqL1xuICByZWFkb25seSBzbmFwc2hvdElkPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgd2hldGhlciBBbWF6b24gRUJTIE11bHRpLUF0dGFjaCBpcyBlbmFibGVkLlxuICAgKiBTZWUge0BsaW5rIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NFQzIvbGF0ZXN0L1VzZXJHdWlkZS9lYnMtdm9sdW1lcy1tdWx0aS5odG1sI2NvbnNpZGVyYXRpb25zfENvbnNpZGVyYXRpb25zIGFuZCBsaW1pdGF0aW9uc31cbiAgICogZm9yIHRoZSBjb25zdHJhaW50cyBvZiBtdWx0aS1hdHRhY2guXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBlbmFibGVNdWx0aUF0dGFjaD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFNwZWNpZmllcyB3aGV0aGVyIHRoZSB2b2x1bWUgc2hvdWxkIGJlIGVuY3J5cHRlZC4gVGhlIGVmZmVjdCBvZiBzZXR0aW5nIHRoZSBlbmNyeXB0aW9uIHN0YXRlIHRvIHRydWUgZGVwZW5kcyBvbiB0aGUgdm9sdW1lIG9yaWdpblxuICAgKiAobmV3IG9yIGZyb20gYSBzbmFwc2hvdCksIHN0YXJ0aW5nIGVuY3J5cHRpb24gc3RhdGUsIG93bmVyc2hpcCwgYW5kIHdoZXRoZXIgZW5jcnlwdGlvbiBieSBkZWZhdWx0IGlzIGVuYWJsZWQuIEZvciBtb3JlIGluZm9ybWF0aW9uLFxuICAgKiBzZWUge0BsaW5rIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NFQzIvbGF0ZXN0L1VzZXJHdWlkZS9FQlNFbmNyeXB0aW9uLmh0bWwjZW5jcnlwdGlvbi1ieS1kZWZhdWx0fEVuY3J5cHRpb24gYnkgRGVmYXVsdH1cbiAgICogaW4gdGhlIEFtYXpvbiBFbGFzdGljIENvbXB1dGUgQ2xvdWQgVXNlciBHdWlkZS5cbiAgICpcbiAgICogRW5jcnlwdGVkIEFtYXpvbiBFQlMgdm9sdW1lcyBtdXN0IGJlIGF0dGFjaGVkIHRvIGluc3RhbmNlcyB0aGF0IHN1cHBvcnQgQW1hem9uIEVCUyBlbmNyeXB0aW9uLiBGb3IgbW9yZSBpbmZvcm1hdGlvbiwgc2VlXG4gICAqIHtAbGluayBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTRUMyL2xhdGVzdC9Vc2VyR3VpZGUvRUJTRW5jcnlwdGlvbi5odG1sI0VCU0VuY3J5cHRpb25fc3VwcG9ydGVkX2luc3RhbmNlc3xTdXBwb3J0ZWQgSW5zdGFuY2UgVHlwZXMufVxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgZW5jcnlwdGVkPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIGN1c3RvbWVyLW1hbmFnZWQgZW5jcnlwdGlvbiBrZXkgdGhhdCBpcyB1c2VkIHRvIGVuY3J5cHQgdGhlIFZvbHVtZS4gVGhlIGVuY3J5cHRlZCBwcm9wZXJ0eSBtdXN0XG4gICAqIGJlIHRydWUgaWYgdGhpcyBpcyBwcm92aWRlZC5cbiAgICpcbiAgICogTm90ZTogSWYgdXNpbmcgYW4ge0BsaW5rIGF3cy1rbXMuSUtleX0gY3JlYXRlZCBmcm9tIGEge0BsaW5rIGF3cy1rbXMuS2V5LmZyb21LZXlBcm4oKX0gaGVyZSxcbiAgICogdGhlbiB0aGUgS01TIGtleSAqKm11c3QqKiBoYXZlIHRoZSBmb2xsb3dpbmcgaW4gaXRzIEtleSBwb2xpY3k7IG90aGVyd2lzZSwgdGhlIFZvbHVtZVxuICAgKiB3aWxsIGZhaWwgdG8gY3JlYXRlLlxuICAgKlxuICAgKiAgICAge1xuICAgKiAgICAgICBcIkVmZmVjdFwiOiBcIkFsbG93XCIsXG4gICAqICAgICAgIFwiUHJpbmNpcGFsXCI6IHsgXCJBV1NcIjogXCI8YXJuIGZvciB5b3VyIGFjY291bnQtdXNlcj4gZXg6IGFybjphd3M6aWFtOjowMDAwMDAwMDAwMDpyb290XCIgfSxcbiAgICogICAgICAgXCJSZXNvdXJjZVwiOiBcIipcIixcbiAgICogICAgICAgXCJBY3Rpb25cIjogW1xuICAgKiAgICAgICAgIFwia21zOkRlc2NyaWJlS2V5XCIsXG4gICAqICAgICAgICAgXCJrbXM6R2VuZXJhdGVEYXRhS2V5V2l0aG91dFBsYWluVGV4dFwiLFxuICAgKiAgICAgICBdLFxuICAgKiAgICAgICBcIkNvbmRpdGlvblwiOiB7XG4gICAqICAgICAgICAgXCJTdHJpbmdFcXVhbHNcIjoge1xuICAgKiAgICAgICAgICAgXCJrbXM6VmlhU2VydmljZVwiOiBcImVjMi48UmVnaW9uPi5hbWF6b25hd3MuY29tXCIsIChlZzogZWMyLnVzLWVhc3QtMS5hbWF6b25hd3MuY29tKVxuICAgKiAgICAgICAgICAgXCJrbXM6Q2FsbGVyQWNjb3VudFwiOiBcIjAwMDAwMDAwMDBcIiAoeW91ciBhY2NvdW50IElEKVxuICAgKiAgICAgICAgIH1cbiAgICogICAgICAgfVxuICAgKiAgICAgfVxuICAgKlxuICAgKiBAZGVmYXVsdCBUaGUgZGVmYXVsdCBLTVMga2V5IGZvciB0aGUgYWNjb3VudCwgcmVnaW9uLCBhbmQgRUMyIHNlcnZpY2UgaXMgdXNlZC5cbiAgICovXG4gIHJlYWRvbmx5IGVuY3J5cHRpb25LZXk/OiBJS2V5O1xuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgd2hldGhlciB0aGUgdm9sdW1lIGlzIGF1dG8tZW5hYmxlZCBmb3IgSS9PIG9wZXJhdGlvbnMuIEJ5IGRlZmF1bHQsIEFtYXpvbiBFQlMgZGlzYWJsZXMgSS9PIHRvIHRoZSB2b2x1bWUgZnJvbSBhdHRhY2hlZCBFQzJcbiAgICogaW5zdGFuY2VzIHdoZW4gaXQgZGV0ZXJtaW5lcyB0aGF0IGEgdm9sdW1lJ3MgZGF0YSBpcyBwb3RlbnRpYWxseSBpbmNvbnNpc3RlbnQuIElmIHRoZSBjb25zaXN0ZW5jeSBvZiB0aGUgdm9sdW1lIGlzIG5vdCBhIGNvbmNlcm4sIGFuZFxuICAgKiB5b3UgcHJlZmVyIHRoYXQgdGhlIHZvbHVtZSBiZSBtYWRlIGF2YWlsYWJsZSBpbW1lZGlhdGVseSBpZiBpdCdzIGltcGFpcmVkLCB5b3UgY2FuIGNvbmZpZ3VyZSB0aGUgdm9sdW1lIHRvIGF1dG9tYXRpY2FsbHkgZW5hYmxlIEkvTy5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGF1dG9FbmFibGVJbz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSB0eXBlIG9mIHRoZSB2b2x1bWU7IHdoYXQgdHlwZSBvZiBzdG9yYWdlIHRvIHVzZSB0byBmb3JtIHRoZSBFQlMgVm9sdW1lLlxuICAgKlxuICAgKiBAZGVmYXVsdCB7QGxpbmsgRWJzRGV2aWNlVm9sdW1lVHlwZS5HRU5FUkFMX1BVUlBPU0VfU1NEfVxuICAgKi9cbiAgcmVhZG9ubHkgdm9sdW1lVHlwZT86IEVic0RldmljZVZvbHVtZVR5cGU7XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgSS9PIG9wZXJhdGlvbnMgcGVyIHNlY29uZCAoSU9QUykgdG8gcHJvdmlzaW9uIGZvciB0aGUgdm9sdW1lLiBUaGUgbWF4aW11bSByYXRpbyBpcyA1MCBJT1BTL0dpQiBmb3IgUFJPVklTSU9ORURfSU9QU19TU0QsXG4gICAqIGFuZCA1MDAgSU9QUy9HaUIgZm9yIGJvdGggUFJPVklTSU9ORURfSU9QU19TU0RfSU8yIGFuZCBHRU5FUkFMX1BVUlBPU0VfU1NEX0dQMy5cbiAgICogU2VlIHtAbGluayBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTQ2xvdWRGb3JtYXRpb24vbGF0ZXN0L1VzZXJHdWlkZS9hd3MtcHJvcGVydGllcy1lYzItZWJzLXZvbHVtZS5odG1sfVxuICAgKiBmb3IgbW9yZSBpbmZvcm1hdGlvbi5cbiAgICpcbiAgICogVGhpcyBwYXJhbWV0ZXIgaXMgdmFsaWQgb25seSBmb3IgUFJPVklTSU9ORURfSU9QU19TU0QsIFBST1ZJU0lPTkVEX0lPUFNfU1NEX0lPMiBhbmQgR0VORVJBTF9QVVJQT1NFX1NTRF9HUDMgdm9sdW1lcy5cbiAgICpcbiAgICogQGRlZmF1bHQgTm9uZSAtLSBSZXF1aXJlZCBmb3IgaW8xIGFuZCBpbzIgdm9sdW1lcy4gVGhlIGRlZmF1bHQgZm9yIGdwMyB2b2x1bWVzIGlzIDMsMDAwIElPUFMgaWYgb21pdHRlZC5cbiAgICovXG4gIHJlYWRvbmx5IGlvcHM/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFBvbGljeSB0byBhcHBseSB3aGVuIHRoZSB2b2x1bWUgaXMgcmVtb3ZlZCBmcm9tIHRoZSBzdGFja1xuICAgKlxuICAgKiBAZGVmYXVsdCBSZW1vdmFsUG9saWN5LlJFVEFJTlxuICAgKi9cbiAgcmVhZG9ubHkgcmVtb3ZhbFBvbGljeT86IFJlbW92YWxQb2xpY3k7XG59XG5cbi8qKlxuICogQXR0cmlidXRlcyByZXF1aXJlZCB0byBpbXBvcnQgYW4gZXhpc3RpbmcgRUJTIFZvbHVtZSBpbnRvIHRoZSBTdGFjay5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBWb2x1bWVBdHRyaWJ1dGVzIHtcbiAgLyoqXG4gICAqIFRoZSBFQlMgVm9sdW1lJ3MgSURcbiAgICovXG4gIHJlYWRvbmx5IHZvbHVtZUlkOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBhdmFpbGFiaWxpdHkgem9uZSB0aGF0IHRoZSBFQlMgVm9sdW1lIGlzIGNvbnRhaW5lZCB3aXRoaW4gKGV4OiB1cy13ZXN0LTJhKVxuICAgKi9cbiAgcmVhZG9ubHkgYXZhaWxhYmlsaXR5Wm9uZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgY3VzdG9tZXItbWFuYWdlZCBlbmNyeXB0aW9uIGtleSB0aGF0IGlzIHVzZWQgdG8gZW5jcnlwdCB0aGUgVm9sdW1lLlxuICAgKlxuICAgKiBAZGVmYXVsdCBOb25lIC0tIFRoZSBFQlMgVm9sdW1lIGlzIG5vdCB1c2luZyBhIGN1c3RvbWVyLW1hbmFnZWQgS01TIGtleSBmb3IgZW5jcnlwdGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IGVuY3J5cHRpb25LZXk/OiBJS2V5O1xufVxuXG4vKipcbiAqIENvbW1vbiBiZWhhdmlvciBvZiBWb2x1bWVzLiBVc2VycyBzaG91bGQgbm90IHVzZSB0aGlzIGNsYXNzIGRpcmVjdGx5LCBhbmQgaW5zdGVhZCB1c2UgYGBWb2x1bWVgYC5cbiAqL1xuYWJzdHJhY3QgY2xhc3MgVm9sdW1lQmFzZSBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSVZvbHVtZSB7XG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSB2b2x1bWVJZDogc3RyaW5nO1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgYXZhaWxhYmlsaXR5Wm9uZTogc3RyaW5nO1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgZW5jcnlwdGlvbktleT86IElLZXk7XG5cbiAgcHVibGljIGdyYW50QXR0YWNoVm9sdW1lKGdyYW50ZWU6IElHcmFudGFibGUsIGluc3RhbmNlcz86IElJbnN0YW5jZVtdKTogR3JhbnQge1xuICAgIGNvbnN0IHJlc3VsdCA9IEdyYW50LmFkZFRvUHJpbmNpcGFsKHtcbiAgICAgIGdyYW50ZWUsXG4gICAgICBhY3Rpb25zOiBbJ2VjMjpBdHRhY2hWb2x1bWUnXSxcbiAgICAgIHJlc291cmNlQXJuczogdGhpcy5jb2xsZWN0R3JhbnRSZXNvdXJjZUFybnMoaW5zdGFuY2VzKSxcbiAgICB9KTtcblxuICAgIGlmICh0aGlzLmVuY3J5cHRpb25LZXkpIHtcbiAgICAgIC8vIFdoZW4gYXR0YWNoaW5nIGEgdm9sdW1lLCB0aGUgRUMyIFNlcnZpY2Ugd2lsbCBuZWVkIHRvIGdyYW50IHRvIGl0c2VsZiBwZXJtaXNzaW9uXG4gICAgICAvLyB0byBiZSBhYmxlIHRvIGRlY3J5cHQgdGhlIGVuY3J5cHRpb24ga2V5LiBXZSByZXN0cmljdCB0aGUgQ3JlYXRlR3JhbnQgZm9yIHByaW5jaXBsZVxuICAgICAgLy8gb2YgbGVhc3QgcHJpdmlsZWdlLCBpbiBhY2NvcmRhbmNlIHdpdGggYmVzdCBwcmFjdGljZXMuXG4gICAgICAvLyBTZWU6IGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NFQzIvbGF0ZXN0L1VzZXJHdWlkZS9FQlNFbmNyeXB0aW9uLmh0bWwjZWJzLWVuY3J5cHRpb24tcGVybWlzc2lvbnNcbiAgICAgIGNvbnN0IGttc0dyYW50OiBHcmFudCA9IHRoaXMuZW5jcnlwdGlvbktleS5ncmFudChncmFudGVlLCAna21zOkNyZWF0ZUdyYW50Jyk7XG4gICAgICBrbXNHcmFudC5wcmluY2lwYWxTdGF0ZW1lbnQhLmFkZENvbmRpdGlvbnMoXG4gICAgICAgIHtcbiAgICAgICAgICBCb29sOiB7ICdrbXM6R3JhbnRJc0ZvckFXU1Jlc291cmNlJzogdHJ1ZSB9LFxuICAgICAgICAgIFN0cmluZ0VxdWFsczoge1xuICAgICAgICAgICAgJ2ttczpWaWFTZXJ2aWNlJzogYGVjMi4ke1N0YWNrLm9mKHRoaXMpLnJlZ2lvbn0uYW1hem9uYXdzLmNvbWAsXG4gICAgICAgICAgICAna21zOkdyYW50Q29uc3RyYWludFR5cGUnOiAnRW5jcnlwdGlvbkNvbnRleHRTdWJzZXQnLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICApO1xuICAgIH1cblxuICAgIHJldHVybiByZXN1bHQ7XG4gIH1cblxuICBwdWJsaWMgZ3JhbnRBdHRhY2hWb2x1bWVCeVJlc291cmNlVGFnKGdyYW50ZWU6IElHcmFudGFibGUsIGNvbnN0cnVjdHM6IENvbnN0cnVjdFtdLCB0YWdLZXlTdWZmaXg/OiBzdHJpbmcpOiBHcmFudCB7XG4gICAgY29uc3QgdGFnVmFsdWUgPSB0aGlzLmNhbGN1bGF0ZVJlc291cmNlVGFnVmFsdWUoW3RoaXMsIC4uLmNvbnN0cnVjdHNdKTtcbiAgICBjb25zdCB0YWdLZXkgPSBgVm9sdW1lR3JhbnRBdHRhY2gtJHt0YWdLZXlTdWZmaXggPz8gdGFnVmFsdWUuc2xpY2UoMCwgMTApLnRvVXBwZXJDYXNlKCl9YDtcbiAgICBjb25zdCBncmFudENvbmRpdGlvbjogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfSA9IHt9O1xuICAgIGdyYW50Q29uZGl0aW9uW2BlYzI6UmVzb3VyY2VUYWcvJHt0YWdLZXl9YF0gPSB0YWdWYWx1ZTtcblxuICAgIGNvbnN0IHJlc3VsdCA9IHRoaXMuZ3JhbnRBdHRhY2hWb2x1bWUoZ3JhbnRlZSk7XG4gICAgcmVzdWx0LnByaW5jaXBhbFN0YXRlbWVudCEuYWRkQ29uZGl0aW9uKFxuICAgICAgJ0ZvckFueVZhbHVlOlN0cmluZ0VxdWFscycsIGdyYW50Q29uZGl0aW9uLFxuICAgICk7XG5cbiAgICAvLyBUaGUgUmVzb3VyY2VUYWcgY29uZGl0aW9uIHJlcXVpcmVzIHRoYXQgYWxsIHJlc291cmNlcyBpbnZvbHZlZCBpbiB0aGUgb3BlcmF0aW9uIGhhdmVcbiAgICAvLyB0aGUgZ2l2ZW4gdGFnLCBzbyB3ZSB0YWcgdGhpcyBhbmQgYWxsIGNvbnN0cnVjdHMgZ2l2ZW4uXG4gICAgVGFncy5vZih0aGlzKS5hZGQodGFnS2V5LCB0YWdWYWx1ZSk7XG4gICAgY29uc3RydWN0cy5mb3JFYWNoKGNvbnN0cnVjdCA9PiBUYWdzLm9mKGNvbnN0cnVjdCBhcyBDb3JlQ29uc3RydWN0KS5hZGQodGFnS2V5LCB0YWdWYWx1ZSkpO1xuXG4gICAgcmV0dXJuIHJlc3VsdDtcbiAgfVxuXG4gIHB1YmxpYyBncmFudERldGFjaFZvbHVtZShncmFudGVlOiBJR3JhbnRhYmxlLCBpbnN0YW5jZXM/OiBJSW5zdGFuY2VbXSk6IEdyYW50IHtcbiAgICBjb25zdCByZXN1bHQgPSBHcmFudC5hZGRUb1ByaW5jaXBhbCh7XG4gICAgICBncmFudGVlLFxuICAgICAgYWN0aW9uczogWydlYzI6RGV0YWNoVm9sdW1lJ10sXG4gICAgICByZXNvdXJjZUFybnM6IHRoaXMuY29sbGVjdEdyYW50UmVzb3VyY2VBcm5zKGluc3RhbmNlcyksXG4gICAgfSk7XG4gICAgLy8gTm90ZTogTm8gZW5jcnlwdGlvbiBrZXkgcGVybWlzc2lvbnMgYXJlIHJlcXVpcmVkIHRvIGRldGFjaCBhbiBlbmNyeXB0ZWQgdm9sdW1lLlxuICAgIHJldHVybiByZXN1bHQ7XG4gIH1cblxuICBwdWJsaWMgZ3JhbnREZXRhY2hWb2x1bWVCeVJlc291cmNlVGFnKGdyYW50ZWU6IElHcmFudGFibGUsIGNvbnN0cnVjdHM6IENvbnN0cnVjdFtdLCB0YWdLZXlTdWZmaXg/OiBzdHJpbmcpOiBHcmFudCB7XG4gICAgY29uc3QgdGFnVmFsdWUgPSB0aGlzLmNhbGN1bGF0ZVJlc291cmNlVGFnVmFsdWUoW3RoaXMsIC4uLmNvbnN0cnVjdHNdKTtcbiAgICBjb25zdCB0YWdLZXkgPSBgVm9sdW1lR3JhbnREZXRhY2gtJHt0YWdLZXlTdWZmaXggPz8gdGFnVmFsdWUuc2xpY2UoMCwgMTApLnRvVXBwZXJDYXNlKCl9YDtcbiAgICBjb25zdCBncmFudENvbmRpdGlvbjogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfSA9IHt9O1xuICAgIGdyYW50Q29uZGl0aW9uW2BlYzI6UmVzb3VyY2VUYWcvJHt0YWdLZXl9YF0gPSB0YWdWYWx1ZTtcblxuICAgIGNvbnN0IHJlc3VsdCA9IHRoaXMuZ3JhbnREZXRhY2hWb2x1bWUoZ3JhbnRlZSk7XG4gICAgcmVzdWx0LnByaW5jaXBhbFN0YXRlbWVudCEuYWRkQ29uZGl0aW9uKFxuICAgICAgJ0ZvckFueVZhbHVlOlN0cmluZ0VxdWFscycsIGdyYW50Q29uZGl0aW9uLFxuICAgICk7XG5cbiAgICAvLyBUaGUgUmVzb3VyY2VUYWcgY29uZGl0aW9uIHJlcXVpcmVzIHRoYXQgYWxsIHJlc291cmNlcyBpbnZvbHZlZCBpbiB0aGUgb3BlcmF0aW9uIGhhdmVcbiAgICAvLyB0aGUgZ2l2ZW4gdGFnLCBzbyB3ZSB0YWcgdGhpcyBhbmQgYWxsIGNvbnN0cnVjdHMgZ2l2ZW4uXG4gICAgVGFncy5vZih0aGlzKS5hZGQodGFnS2V5LCB0YWdWYWx1ZSk7XG4gICAgY29uc3RydWN0cy5mb3JFYWNoKGNvbnN0cnVjdCA9PiBUYWdzLm9mKGNvbnN0cnVjdCBhcyBDb3JlQ29uc3RydWN0KS5hZGQodGFnS2V5LCB0YWdWYWx1ZSkpO1xuXG4gICAgcmV0dXJuIHJlc3VsdDtcbiAgfVxuXG4gIHByaXZhdGUgY29sbGVjdEdyYW50UmVzb3VyY2VBcm5zKGluc3RhbmNlcz86IElJbnN0YW5jZVtdKTogc3RyaW5nW10ge1xuICAgIGNvbnN0IHN0YWNrID0gU3RhY2sub2YodGhpcyk7XG4gICAgY29uc3QgcmVzb3VyY2VBcm5zOiBzdHJpbmdbXSA9IFtcbiAgICAgIGBhcm46JHtzdGFjay5wYXJ0aXRpb259OmVjMjoke3N0YWNrLnJlZ2lvbn06JHtzdGFjay5hY2NvdW50fTp2b2x1bWUvJHt0aGlzLnZvbHVtZUlkfWAsXG4gICAgXTtcbiAgICBjb25zdCBpbnN0YW5jZUFyblByZWZpeCA9IGBhcm46JHtzdGFjay5wYXJ0aXRpb259OmVjMjoke3N0YWNrLnJlZ2lvbn06JHtzdGFjay5hY2NvdW50fTppbnN0YW5jZWA7XG4gICAgaWYgKGluc3RhbmNlcykge1xuICAgICAgaW5zdGFuY2VzLmZvckVhY2goaW5zdGFuY2UgPT4gcmVzb3VyY2VBcm5zLnB1c2goYCR7aW5zdGFuY2VBcm5QcmVmaXh9LyR7aW5zdGFuY2U/Lmluc3RhbmNlSWR9YCkpO1xuICAgIH0gZWxzZSB7XG4gICAgICByZXNvdXJjZUFybnMucHVzaChgJHtpbnN0YW5jZUFyblByZWZpeH0vKmApO1xuICAgIH1cbiAgICByZXR1cm4gcmVzb3VyY2VBcm5zO1xuICB9XG5cbiAgcHJpdmF0ZSBjYWxjdWxhdGVSZXNvdXJjZVRhZ1ZhbHVlKGNvbnN0cnVjdHM6IENvbnN0cnVjdFtdKTogc3RyaW5nIHtcbiAgICBjb25zdCBtZDUgPSBjcnlwdG8uY3JlYXRlSGFzaCgnbWQ1Jyk7XG4gICAgY29uc3RydWN0cy5mb3JFYWNoKGNvbnN0cnVjdCA9PiBtZDUudXBkYXRlKE5hbWVzLnVuaXF1ZUlkKGNvbnN0cnVjdCkpKTtcbiAgICByZXR1cm4gbWQ1LmRpZ2VzdCgnaGV4Jyk7XG4gIH1cbn1cblxuLyoqXG4gKiBDcmVhdGVzIGEgbmV3IEVCUyBWb2x1bWUgaW4gQVdTIEVDMi5cbiAqL1xuZXhwb3J0IGNsYXNzIFZvbHVtZSBleHRlbmRzIFZvbHVtZUJhc2Uge1xuICAvKipcbiAgICogSW1wb3J0IGFuIGV4aXN0aW5nIEVCUyBWb2x1bWUgaW50byB0aGUgU3RhY2suXG4gICAqXG4gICAqIEBwYXJhbSBzY29wZSB0aGUgc2NvcGUgb2YgdGhlIGltcG9ydC5cbiAgICogQHBhcmFtIGlkICAgIHRoZSBJRCBvZiB0aGUgaW1wb3J0ZWQgVm9sdW1lIGluIHRoZSBjb25zdHJ1Y3QgdHJlZS5cbiAgICogQHBhcmFtIGF0dHJzIHRoZSBhdHRyaWJ1dGVzIG9mIHRoZSBpbXBvcnRlZCBWb2x1bWVcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVZvbHVtZUF0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IFZvbHVtZUF0dHJpYnV0ZXMpOiBJVm9sdW1lIHtcbiAgICBjbGFzcyBJbXBvcnQgZXh0ZW5kcyBWb2x1bWVCYXNlIHtcbiAgICAgIHB1YmxpYyByZWFkb25seSB2b2x1bWVJZCA9IGF0dHJzLnZvbHVtZUlkO1xuICAgICAgcHVibGljIHJlYWRvbmx5IGF2YWlsYWJpbGl0eVpvbmUgPSBhdHRycy5hdmFpbGFiaWxpdHlab25lO1xuICAgICAgcHVibGljIHJlYWRvbmx5IGVuY3J5cHRpb25LZXkgPSBhdHRycy5lbmNyeXB0aW9uS2V5O1xuICAgIH1cbiAgICAvLyBDaGVjayB0aGF0IHRoZSBwcm92aWRlZCB2b2x1bWVJZCBsb29rcyBsaWtlIGl0IGNvdWxkIGJlIHZhbGlkLlxuICAgIGlmICghVG9rZW4uaXNVbnJlc29sdmVkKGF0dHJzLnZvbHVtZUlkKSAmJiAhL152b2wtWzAtOWEtZkEtRl0rJC8udGVzdChhdHRycy52b2x1bWVJZCkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignYHZvbHVtZUlkYCBkb2VzIG5vdCBtYXRjaCBleHBlY3RlZCBwYXR0ZXJuLiBFeHBlY3RlZCBgdm9sLTxoZXhhZGVjbWlhbCB2YWx1ZT5gIChleDogYHZvbC0wNWFiZTI0NmFmYCkgb3IgYSBUb2tlbicpO1xuICAgIH1cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgcHVibGljIHJlYWRvbmx5IHZvbHVtZUlkOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBhdmFpbGFiaWxpdHlab25lOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBlbmNyeXB0aW9uS2V5PzogSUtleTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogVm9sdW1lUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIHBoeXNpY2FsTmFtZTogcHJvcHMudm9sdW1lTmFtZSxcbiAgICB9KTtcblxuICAgIHRoaXMudmFsaWRhdGVQcm9wcyhwcm9wcyk7XG5cbiAgICBjb25zdCByZXNvdXJjZSA9IG5ldyBDZm5Wb2x1bWUodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgYXZhaWxhYmlsaXR5Wm9uZTogcHJvcHMuYXZhaWxhYmlsaXR5Wm9uZSxcbiAgICAgIGF1dG9FbmFibGVJbzogcHJvcHMuYXV0b0VuYWJsZUlvLFxuICAgICAgZW5jcnlwdGVkOiBwcm9wcy5lbmNyeXB0ZWQsXG4gICAgICBrbXNLZXlJZDogcHJvcHMuZW5jcnlwdGlvbktleT8ua2V5QXJuLFxuICAgICAgaW9wczogcHJvcHMuaW9wcyxcbiAgICAgIG11bHRpQXR0YWNoRW5hYmxlZDogcHJvcHMuZW5hYmxlTXVsdGlBdHRhY2ggPz8gZmFsc2UsXG4gICAgICBzaXplOiBwcm9wcy5zaXplPy50b0dpYmlieXRlcyh7IHJvdW5kaW5nOiBTaXplUm91bmRpbmdCZWhhdmlvci5GQUlMIH0pLFxuICAgICAgc25hcHNob3RJZDogcHJvcHMuc25hcHNob3RJZCxcbiAgICAgIHZvbHVtZVR5cGU6IHByb3BzLnZvbHVtZVR5cGUgPz8gRWJzRGV2aWNlVm9sdW1lVHlwZS5HRU5FUkFMX1BVUlBPU0VfU1NELFxuICAgIH0pO1xuICAgIHJlc291cmNlLmFwcGx5UmVtb3ZhbFBvbGljeShwcm9wcy5yZW1vdmFsUG9saWN5KTtcblxuICAgIGlmIChwcm9wcy52b2x1bWVOYW1lKSBUYWdzLm9mKHJlc291cmNlKS5hZGQoJ05hbWUnLCBwcm9wcy52b2x1bWVOYW1lKTtcblxuICAgIHRoaXMudm9sdW1lSWQgPSByZXNvdXJjZS5yZWY7XG4gICAgdGhpcy5hdmFpbGFiaWxpdHlab25lID0gcHJvcHMuYXZhaWxhYmlsaXR5Wm9uZTtcbiAgICB0aGlzLmVuY3J5cHRpb25LZXkgPSBwcm9wcy5lbmNyeXB0aW9uS2V5O1xuXG4gICAgaWYgKHRoaXMuZW5jcnlwdGlvbktleSkge1xuICAgICAgLy8gUGVyOiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTRUMyL2xhdGVzdC9Vc2VyR3VpZGUvRUJTRW5jcnlwdGlvbi5odG1sI2Vicy1lbmNyeXB0aW9uLXJlcXVpcmVtZW50c1xuICAgICAgY29uc3QgcHJpbmNpcGFsID1cbiAgICAgICAgbmV3IFZpYVNlcnZpY2VQcmluY2lwYWwoYGVjMi4ke1N0YWNrLm9mKHRoaXMpLnJlZ2lvbn0uYW1hem9uYXdzLmNvbWAsIG5ldyBBY2NvdW50Um9vdFByaW5jaXBhbCgpKS53aXRoQ29uZGl0aW9ucyh7XG4gICAgICAgICAgU3RyaW5nRXF1YWxzOiB7XG4gICAgICAgICAgICAna21zOkNhbGxlckFjY291bnQnOiBTdGFjay5vZih0aGlzKS5hY2NvdW50LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuICAgICAgY29uc3QgZ3JhbnQgPSB0aGlzLmVuY3J5cHRpb25LZXkuZ3JhbnQocHJpbmNpcGFsLFxuICAgICAgICAvLyBEZXNjcmliZSAmIEdlbmVyYXRlIGFyZSByZXF1aXJlZCB0byBiZSBhYmxlIHRvIGNyZWF0ZSB0aGUgQ01LLWVuY3J5cHRlZCBWb2x1bWUuXG4gICAgICAgICdrbXM6RGVzY3JpYmVLZXknLFxuICAgICAgICAna21zOkdlbmVyYXRlRGF0YUtleVdpdGhvdXRQbGFpblRleHQnLFxuICAgICAgKTtcbiAgICAgIGlmIChwcm9wcy5zbmFwc2hvdElkKSB7XG4gICAgICAgIC8vIFJlRW5jcnlwdCBpcyByZXF1aXJlZCBmb3Igd2hlbiByZS1lbmNyeXB0aW5nIGZyb20gYW4gZW5jcnlwdGVkIHNuYXBzaG90LlxuICAgICAgICBncmFudC5wcmluY2lwYWxTdGF0ZW1lbnQ/LmFkZEFjdGlvbnMoJ2ttczpSZUVuY3J5cHQqJyk7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgcHJvdGVjdGVkIHZhbGlkYXRlUHJvcHMocHJvcHM6IFZvbHVtZVByb3BzKSB7XG4gICAgaWYgKCEocHJvcHMuc2l6ZSB8fCBwcm9wcy5zbmFwc2hvdElkKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdNdXN0IHByb3ZpZGUgYXQgbGVhc3Qgb25lIG9mIGBzaXplYCBvciBgc25hcHNob3RJZGAnKTtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMuc25hcHNob3RJZCAmJiAhVG9rZW4uaXNVbnJlc29sdmVkKHByb3BzLnNuYXBzaG90SWQpICYmICEvXnNuYXAtWzAtOWEtZkEtRl0rJC8udGVzdChwcm9wcy5zbmFwc2hvdElkKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdgc25hcHNob3RJZGAgZG9lcyBtYXRjaCBleHBlY3RlZCBwYXR0ZXJuLiBFeHBlY3RlZCBgc25hcC08aGV4YWRlY21pYWwgdmFsdWU+YCAoZXg6IGBzbmFwLTA1YWJlMjQ2YWZgKSBvciBUb2tlbicpO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy5lbmNyeXB0aW9uS2V5ICYmICFwcm9wcy5lbmNyeXB0ZWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignYGVuY3J5cHRlZGAgbXVzdCBiZSB0cnVlIHdoZW4gcHJvdmlkaW5nIGFuIGBlbmNyeXB0aW9uS2V5YC4nKTtcbiAgICB9XG5cbiAgICBpZiAoXG4gICAgICBwcm9wcy52b2x1bWVUeXBlICYmXG4gICAgICBbXG4gICAgICAgIEVic0RldmljZVZvbHVtZVR5cGUuUFJPVklTSU9ORURfSU9QU19TU0QsXG4gICAgICAgIEVic0RldmljZVZvbHVtZVR5cGUuUFJPVklTSU9ORURfSU9QU19TU0RfSU8yLFxuICAgICAgXS5pbmNsdWRlcyhwcm9wcy52b2x1bWVUeXBlKSAmJlxuICAgICAgIXByb3BzLmlvcHNcbiAgICApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgJ2Bpb3BzYCBtdXN0IGJlIHNwZWNpZmllZCBpZiB0aGUgYHZvbHVtZVR5cGVgIGlzIGBQUk9WSVNJT05FRF9JT1BTX1NTRGAgb3IgYFBST1ZJU0lPTkVEX0lPUFNfU1NEX0lPMmAuJyxcbiAgICAgICk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLmlvcHMpIHtcbiAgICAgIGNvbnN0IHZvbHVtZVR5cGUgPSBwcm9wcy52b2x1bWVUeXBlID8/IEVic0RldmljZVZvbHVtZVR5cGUuR0VORVJBTF9QVVJQT1NFX1NTRDtcbiAgICAgIGlmIChcbiAgICAgICAgIVtcbiAgICAgICAgICBFYnNEZXZpY2VWb2x1bWVUeXBlLlBST1ZJU0lPTkVEX0lPUFNfU1NELFxuICAgICAgICAgIEVic0RldmljZVZvbHVtZVR5cGUuUFJPVklTSU9ORURfSU9QU19TU0RfSU8yLFxuICAgICAgICAgIEVic0RldmljZVZvbHVtZVR5cGUuR0VORVJBTF9QVVJQT1NFX1NTRF9HUDMsXG4gICAgICAgIF0uaW5jbHVkZXModm9sdW1lVHlwZSlcbiAgICAgICkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgJ2Bpb3BzYCBtYXkgb25seSBiZSBzcGVjaWZpZWQgaWYgdGhlIGB2b2x1bWVUeXBlYCBpcyBgUFJPVklTSU9ORURfSU9QU19TU0RgLCBgUFJPVklTSU9ORURfSU9QU19TU0RfSU8yYCBvciBgR0VORVJBTF9QVVJQT1NFX1NTRF9HUDNgLicsXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgICAvLyBFbmZvcmNlIG1pbmltdW0gJiBtYXhpbXVtIElPUFM6XG4gICAgICAvLyBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTQ2xvdWRGb3JtYXRpb24vbGF0ZXN0L1VzZXJHdWlkZS9hd3MtcHJvcGVydGllcy1lYzItZWJzLXZvbHVtZS5odG1sXG4gICAgICBjb25zdCBpb3BzUmFuZ2VzOiB7IFtrZXk6IHN0cmluZ106IHsgTWluOiBudW1iZXIsIE1heDogbnVtYmVyIH0gfSA9IHt9O1xuICAgICAgaW9wc1Jhbmdlc1tFYnNEZXZpY2VWb2x1bWVUeXBlLkdFTkVSQUxfUFVSUE9TRV9TU0RfR1AzXSA9IHsgTWluOiAzMDAwLCBNYXg6IDE2MDAwIH07XG4gICAgICBpb3BzUmFuZ2VzW0Vic0RldmljZVZvbHVtZVR5cGUuUFJPVklTSU9ORURfSU9QU19TU0RdID0geyBNaW46IDEwMCwgTWF4OiA2NDAwMCB9O1xuICAgICAgaW9wc1Jhbmdlc1tFYnNEZXZpY2VWb2x1bWVUeXBlLlBST1ZJU0lPTkVEX0lPUFNfU1NEX0lPMl0gPSB7IE1pbjogMTAwLCBNYXg6IDY0MDAwIH07XG4gICAgICBjb25zdCB7IE1pbiwgTWF4IH0gPSBpb3BzUmFuZ2VzW3ZvbHVtZVR5cGVdO1xuICAgICAgaWYgKHByb3BzLmlvcHMgPCBNaW4gfHwgcHJvcHMuaW9wcyA+IE1heCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFxcYCR7dm9sdW1lVHlwZX1cXGAgdm9sdW1lcyBpb3BzIG11c3QgYmUgYmV0d2VlbiAke01pbn0gYW5kICR7TWF4fS5gKTtcbiAgICAgIH1cblxuICAgICAgLy8gRW5mb3JjZSBtYXhpbXVtIHJhdGlvIG9mIElPUFMvR2lCOlxuICAgICAgLy8gaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0VDMi9sYXRlc3QvVXNlckd1aWRlL2Vicy12b2x1bWUtdHlwZXMuaHRtbFxuICAgICAgY29uc3QgbWF4aW11bVJhdGlvczogeyBba2V5OiBzdHJpbmddOiBudW1iZXIgfSA9IHt9O1xuICAgICAgbWF4aW11bVJhdGlvc1tFYnNEZXZpY2VWb2x1bWVUeXBlLkdFTkVSQUxfUFVSUE9TRV9TU0RfR1AzXSA9IDUwMDtcbiAgICAgIG1heGltdW1SYXRpb3NbRWJzRGV2aWNlVm9sdW1lVHlwZS5QUk9WSVNJT05FRF9JT1BTX1NTRF0gPSA1MDtcbiAgICAgIG1heGltdW1SYXRpb3NbRWJzRGV2aWNlVm9sdW1lVHlwZS5QUk9WSVNJT05FRF9JT1BTX1NTRF9JTzJdID0gNTAwO1xuICAgICAgY29uc3QgbWF4aW11bVJhdGlvID0gbWF4aW11bVJhdGlvc1t2b2x1bWVUeXBlXTtcbiAgICAgIGlmIChwcm9wcy5zaXplICYmIChwcm9wcy5pb3BzID4gbWF4aW11bVJhdGlvICogcHJvcHMuc2l6ZS50b0dpYmlieXRlcyh7IHJvdW5kaW5nOiBTaXplUm91bmRpbmdCZWhhdmlvci5GQUlMIH0pKSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFxcYCR7dm9sdW1lVHlwZX1cXGAgdm9sdW1lcyBpb3BzIGhhcyBhIG1heGltdW0gcmF0aW8gb2YgJHttYXhpbXVtUmF0aW99IElPUFMvR2lCLmApO1xuICAgICAgfVxuICAgIH1cblxuICAgIGlmIChwcm9wcy5lbmFibGVNdWx0aUF0dGFjaCkge1xuICAgICAgY29uc3Qgdm9sdW1lVHlwZSA9IHByb3BzLnZvbHVtZVR5cGUgPz8gRWJzRGV2aWNlVm9sdW1lVHlwZS5HRU5FUkFMX1BVUlBPU0VfU1NEO1xuICAgICAgaWYgKFxuICAgICAgICAhW1xuICAgICAgICAgIEVic0RldmljZVZvbHVtZVR5cGUuUFJPVklTSU9ORURfSU9QU19TU0QsXG4gICAgICAgICAgRWJzRGV2aWNlVm9sdW1lVHlwZS5QUk9WSVNJT05FRF9JT1BTX1NTRF9JTzIsXG4gICAgICAgIF0uaW5jbHVkZXModm9sdW1lVHlwZSlcbiAgICAgICkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ211bHRpLWF0dGFjaCBpcyBzdXBwb3J0ZWQgZXhjbHVzaXZlbHkgb24gYFBST1ZJU0lPTkVEX0lPUFNfU1NEYCBhbmQgYFBST1ZJU0lPTkVEX0lPUFNfU1NEX0lPMmAgdm9sdW1lcy4nKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBpZiAocHJvcHMuc2l6ZSkge1xuICAgICAgY29uc3Qgc2l6ZSA9IHByb3BzLnNpemUudG9HaWJpYnl0ZXMoeyByb3VuZGluZzogU2l6ZVJvdW5kaW5nQmVoYXZpb3IuRkFJTCB9KTtcbiAgICAgIC8vIEVuZm9yY2UgbWluaW11bSAmIG1heGltdW0gdm9sdW1lIHNpemU6XG4gICAgICAvLyBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTQ2xvdWRGb3JtYXRpb24vbGF0ZXN0L1VzZXJHdWlkZS9hd3MtcHJvcGVydGllcy1lYzItZWJzLXZvbHVtZS5odG1sXG4gICAgICBjb25zdCBzaXplUmFuZ2VzOiB7IFtrZXk6IHN0cmluZ106IHsgTWluOiBudW1iZXIsIE1heDogbnVtYmVyIH0gfSA9IHt9O1xuICAgICAgc2l6ZVJhbmdlc1tFYnNEZXZpY2VWb2x1bWVUeXBlLkdFTkVSQUxfUFVSUE9TRV9TU0RdID0geyBNaW46IDEsIE1heDogMTYzODQgfTtcbiAgICAgIHNpemVSYW5nZXNbRWJzRGV2aWNlVm9sdW1lVHlwZS5HRU5FUkFMX1BVUlBPU0VfU1NEX0dQM10gPSB7IE1pbjogMSwgTWF4OiAxNjM4NCB9O1xuICAgICAgc2l6ZVJhbmdlc1tFYnNEZXZpY2VWb2x1bWVUeXBlLlBST1ZJU0lPTkVEX0lPUFNfU1NEXSA9IHsgTWluOiA0LCBNYXg6IDE2Mzg0IH07XG4gICAgICBzaXplUmFuZ2VzW0Vic0RldmljZVZvbHVtZVR5cGUuUFJPVklTSU9ORURfSU9QU19TU0RfSU8yXSA9IHsgTWluOiA0LCBNYXg6IDE2Mzg0IH07XG4gICAgICBzaXplUmFuZ2VzW0Vic0RldmljZVZvbHVtZVR5cGUuVEhST1VHSFBVVF9PUFRJTUlaRURfSEREXSA9IHsgTWluOiAxMjUsIE1heDogMTYzODQgfTtcbiAgICAgIHNpemVSYW5nZXNbRWJzRGV2aWNlVm9sdW1lVHlwZS5DT0xEX0hERF0gPSB7IE1pbjogMTI1LCBNYXg6IDE2Mzg0IH07XG4gICAgICBzaXplUmFuZ2VzW0Vic0RldmljZVZvbHVtZVR5cGUuTUFHTkVUSUNdID0geyBNaW46IDEsIE1heDogMTAyNCB9O1xuICAgICAgY29uc3Qgdm9sdW1lVHlwZSA9IHByb3BzLnZvbHVtZVR5cGUgPz8gRWJzRGV2aWNlVm9sdW1lVHlwZS5HRU5FUkFMX1BVUlBPU0VfU1NEO1xuICAgICAgY29uc3QgeyBNaW4sIE1heCB9ID0gc2l6ZVJhbmdlc1t2b2x1bWVUeXBlXTtcbiAgICAgIGlmIChzaXplIDwgTWluIHx8IHNpemUgPiBNYXgpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBcXGAke3ZvbHVtZVR5cGV9XFxgIHZvbHVtZXMgbXVzdCBiZSBiZXR3ZWVuICR7TWlufSBHaUIgYW5kICR7TWF4fSBHaUIgaW4gc2l6ZS5gKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cbn1cbiJdfQ==