"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.InitConfig = exports.CloudFormationInit = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
const machine_image_1 = require("./machine-image");
const cfn_init_internal_1 = require("./private/cfn-init-internal");
/**
 * A CloudFormation-init configuration
 */
class CloudFormationInit {
    constructor(configSets, configs) {
        this._configSets = {};
        this._configs = {};
        Object.assign(this._configSets, configSets);
        Object.assign(this._configs, configs);
    }
    /**
     * Build a new config from a set of Init Elements
     */
    static fromElements(...elements) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ec2_InitElement(elements);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromElements);
            }
            throw error;
        }
        return CloudFormationInit.fromConfig(new InitConfig(elements));
    }
    /**
     * Use an existing InitConfig object as the default and only config
     */
    static fromConfig(config) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ec2_InitConfig(config);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromConfig);
            }
            throw error;
        }
        return CloudFormationInit.fromConfigSets({
            configSets: {
                default: ['config'],
            },
            configs: { config },
        });
    }
    /**
     * Build a CloudFormationInit from config sets
     */
    static fromConfigSets(props) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ec2_ConfigSetProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromConfigSets);
            }
            throw error;
        }
        return new CloudFormationInit(props.configSets, props.configs);
    }
    /**
     * Add a config with the given name to this CloudFormationInit object
     */
    addConfig(configName, config) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ec2_InitConfig(config);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addConfig);
            }
            throw error;
        }
        if (this._configs[configName]) {
            throw new Error(`CloudFormationInit already contains a config named '${configName}'`);
        }
        this._configs[configName] = config;
    }
    /**
     * Add a config set with the given name to this CloudFormationInit object
     *
     * The new configset will reference the given configs in the given order.
     */
    addConfigSet(configSetName, configNames = []) {
        if (this._configSets[configSetName]) {
            throw new Error(`CloudFormationInit already contains a configSet named '${configSetName}'`);
        }
        const unk = configNames.filter(c => !this._configs[c]);
        if (unk.length > 0) {
            throw new Error(`Unknown configs referenced in definition of '${configSetName}': ${unk}`);
        }
        this._configSets[configSetName] = [...configNames];
    }
    /**
     * Attach the CloudFormation Init config to the given resource
     *
     * As an app builder, use `instance.applyCloudFormationInit()` or
     * `autoScalingGroup.applyCloudFormationInit()` to trigger this method.
     *
     * This method does the following:
     *
     * - Renders the `AWS::CloudFormation::Init` object to the given resource's
     *   metadata, potentially adding a `AWS::CloudFormation::Authentication` object
     *   next to it if required.
     * - Updates the instance role policy to be able to call the APIs required for
     *   `cfn-init` and `cfn-signal` to work, and potentially add permissions to download
     *   referenced asset and bucket resources.
     * - Updates the given UserData with commands to execute the `cfn-init` script.
     */
    attach(attachedResource, attachOptions) {
        try {
            jsiiDeprecationWarnings.monocdk_CfnResource(attachedResource);
            jsiiDeprecationWarnings.monocdk_aws_ec2_AttachInitOptions(attachOptions);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.attach);
            }
            throw error;
        }
        if (attachOptions.platform === machine_image_1.OperatingSystemType.UNKNOWN) {
            throw new Error('Cannot attach CloudFormationInit to an unknown OS type');
        }
        const CFN_INIT_METADATA_KEY = 'AWS::CloudFormation::Init';
        if (attachedResource.getMetadata(CFN_INIT_METADATA_KEY) !== undefined) {
            throw new Error(`Cannot bind CfnInit: resource '${attachedResource.node.path}' already has '${CFN_INIT_METADATA_KEY}' attached`);
        }
        // Note: This will not reflect mutations made after attaching.
        const bindResult = this.bind(attachedResource.stack, attachOptions);
        attachedResource.addMetadata(CFN_INIT_METADATA_KEY, bindResult.configData);
        // Need to resolve the various tokens from assets in the config,
        // as well as include any asset hashes provided so the fingerprint is accurate.
        const resolvedConfig = attachedResource.stack.resolve(bindResult.configData);
        const fingerprintInput = { config: resolvedConfig, assetHash: bindResult.assetHash };
        const fingerprint = contentHash(JSON.stringify(fingerprintInput)).slice(0, 16);
        attachOptions.instanceRole.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['cloudformation:DescribeStackResource', 'cloudformation:SignalResource'],
            resources: [core_1.Aws.STACK_ID],
        }));
        if (bindResult.authData) {
            attachedResource.addMetadata('AWS::CloudFormation::Authentication', bindResult.authData);
        }
        // To identify the resources that have the metadata and where the signal
        // needs to be sent, we need { region, stackName, logicalId }
        let resourceLocator = `--region ${core_1.Aws.REGION} --stack ${core_1.Aws.STACK_NAME} --resource ${attachedResource.logicalId}`;
        const signalResource = attachOptions.signalResource?.logicalId ?? attachedResource.logicalId;
        let notifyResourceLocator = `--region ${core_1.Aws.REGION} --stack ${core_1.Aws.STACK_NAME} --resource ${signalResource}`;
        // If specified in attachOptions, include arguments in cfn-init/cfn-signal commands
        if (attachOptions.includeUrl) {
            resourceLocator = `${resourceLocator} --url https://cloudformation.${core_1.Aws.REGION}.${core_1.Aws.URL_SUFFIX}`;
            notifyResourceLocator = `${notifyResourceLocator} --url https://cloudformation.${core_1.Aws.REGION}.${core_1.Aws.URL_SUFFIX}`;
        }
        if (attachOptions.includeRole) {
            resourceLocator = `${resourceLocator} --role ${attachOptions.instanceRole.roleName}`;
            notifyResourceLocator = `${notifyResourceLocator} --role ${attachOptions.instanceRole.roleName}`;
        }
        const configSets = (attachOptions.configSets ?? ['default']).join(',');
        const printLog = attachOptions.printLog ?? true;
        if (attachOptions.embedFingerprint ?? true) {
            // It just so happens that the comment char is '#' for both bash and PowerShell
            attachOptions.userData.addCommands(`# fingerprint: ${fingerprint}`);
        }
        if (attachOptions.platform === machine_image_1.OperatingSystemType.WINDOWS) {
            const errCode = attachOptions.ignoreFailures ? '0' : '$LASTEXITCODE';
            attachOptions.userData.addCommands(...[
                `cfn-init.exe -v ${resourceLocator} -c ${configSets}`,
                `cfn-signal.exe -e ${errCode} ${notifyResourceLocator}`,
                ...(printLog ? ['type C:\\cfn\\log\\cfn-init.log'] : []),
            ]);
        }
        else {
            const errCode = attachOptions.ignoreFailures ? '0' : '$?';
            attachOptions.userData.addCommands(...[
                // Run a subshell without 'errexit', so we can signal using the exit code of cfn-init
                '(',
                '  set +e',
                `  /opt/aws/bin/cfn-init -v ${resourceLocator} -c ${configSets}`,
                `  /opt/aws/bin/cfn-signal -e ${errCode} ${notifyResourceLocator}`,
                ...(printLog ? ['  cat /var/log/cfn-init.log >&2'] : []),
                ')',
            ]);
        }
    }
    bind(scope, options) {
        const nonEmptyConfigs = mapValues(this._configs, c => c.isEmpty() ? undefined : c);
        const configNameToBindResult = mapValues(nonEmptyConfigs, c => c._bind(scope, options));
        return {
            configData: {
                configSets: mapValues(this._configSets, configNames => configNames.filter(name => nonEmptyConfigs[name] !== undefined)),
                ...mapValues(configNameToBindResult, c => c.config),
            },
            authData: Object.values(configNameToBindResult).map(c => c.authentication).reduce(deepMerge, undefined),
            assetHash: combineAssetHashesOrUndefined(Object.values(configNameToBindResult).map(c => c.assetHash)),
        };
    }
}
exports.CloudFormationInit = CloudFormationInit;
_a = JSII_RTTI_SYMBOL_1;
CloudFormationInit[_a] = { fqn: "monocdk.aws_ec2.CloudFormationInit", version: "1.185.0" };
/**
 * A collection of configuration elements
 */
class InitConfig {
    constructor(elements) {
        this.elements = new Array();
        this.add(...elements);
    }
    /**
     * Whether this configset has elements or not
     */
    isEmpty() {
        return this.elements.length === 0;
    }
    /**
     * Add one or more elements to the config
     */
    add(...elements) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_ec2_InitElement(elements);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.add);
            }
            throw error;
        }
        this.elements.push(...elements);
    }
    /**
     * Called when the config is applied to an instance.
     * Creates the CloudFormation representation of the Init config and handles any permissions and assets.
     * @internal
     */
    _bind(scope, options) {
        const bindOptions = {
            instanceRole: options.instanceRole,
            platform: this.initPlatformFromOSType(options.platform),
            scope,
        };
        const packageConfig = this.bindForType(cfn_init_internal_1.InitElementType.PACKAGE, bindOptions);
        const groupsConfig = this.bindForType(cfn_init_internal_1.InitElementType.GROUP, bindOptions);
        const usersConfig = this.bindForType(cfn_init_internal_1.InitElementType.USER, bindOptions);
        const sourcesConfig = this.bindForType(cfn_init_internal_1.InitElementType.SOURCE, bindOptions);
        const filesConfig = this.bindForType(cfn_init_internal_1.InitElementType.FILE, bindOptions);
        const commandsConfig = this.bindForType(cfn_init_internal_1.InitElementType.COMMAND, bindOptions);
        // Must be last!
        const servicesConfig = this.bindForType(cfn_init_internal_1.InitElementType.SERVICE, bindOptions);
        const allConfig = [packageConfig, groupsConfig, usersConfig, sourcesConfig, filesConfig, commandsConfig, servicesConfig];
        const authentication = allConfig.map(c => c?.authentication).reduce(deepMerge, undefined);
        const assetHash = combineAssetHashesOrUndefined(allConfig.map(c => c?.assetHash));
        return {
            config: {
                packages: packageConfig?.config,
                groups: groupsConfig?.config,
                users: usersConfig?.config,
                sources: sourcesConfig?.config,
                files: filesConfig?.config,
                commands: commandsConfig?.config,
                services: servicesConfig?.config,
            },
            authentication,
            assetHash,
        };
    }
    bindForType(elementType, renderOptions) {
        const elements = this.elements.filter(elem => elem.elementType === elementType);
        if (elements.length === 0) {
            return undefined;
        }
        const bindResults = elements.map((e, index) => e._bind({ index, ...renderOptions }));
        return {
            config: bindResults.map(r => r.config).reduce(deepMerge, undefined) ?? {},
            authentication: bindResults.map(r => r.authentication).reduce(deepMerge, undefined),
            assetHash: combineAssetHashesOrUndefined(bindResults.map(r => r.assetHash)),
        };
    }
    initPlatformFromOSType(osType) {
        switch (osType) {
            case machine_image_1.OperatingSystemType.LINUX: {
                return cfn_init_internal_1.InitPlatform.LINUX;
            }
            case machine_image_1.OperatingSystemType.WINDOWS: {
                return cfn_init_internal_1.InitPlatform.WINDOWS;
            }
            default: {
                throw new Error('Cannot attach CloudFormationInit to an unknown OS type');
            }
        }
    }
}
exports.InitConfig = InitConfig;
_b = JSII_RTTI_SYMBOL_1;
InitConfig[_b] = { fqn: "monocdk.aws_ec2.InitConfig", version: "1.185.0" };
/**
 * Deep-merge objects and arrays
 *
 * Treat arrays as sets, removing duplicates. This is acceptable for rendering
 * cfn-inits, not applicable elsewhere.
 */
function deepMerge(target, src) {
    if (target == null) {
        return src;
    }
    if (src == null) {
        return target;
    }
    for (const [key, value] of Object.entries(src)) {
        if (Array.isArray(value)) {
            if (target[key] && !Array.isArray(target[key])) {
                throw new Error(`Trying to merge array [${value}] into a non-array '${target[key]}'`);
            }
            target[key] = Array.from(new Set([
                ...target[key] ?? [],
                ...value,
            ]));
            continue;
        }
        if (typeof value === 'object' && value) {
            target[key] = deepMerge(target[key] ?? {}, value);
            continue;
        }
        if (value !== undefined) {
            target[key] = value;
        }
    }
    return target;
}
/**
 * Map a function over values of an object
 *
 * If the mapping function returns undefined, remove the key
 */
function mapValues(xs, fn) {
    const ret = {};
    for (const [k, v] of Object.entries(xs)) {
        const mapped = fn(v);
        if (mapped !== undefined) {
            ret[k] = mapped;
        }
    }
    return ret;
}
// Combines all input asset hashes into one, or if no hashes are present, returns undefined.
function combineAssetHashesOrUndefined(hashes) {
    const hashArray = hashes.filter((x) => x !== undefined);
    return hashArray.length > 0 ? hashArray.join('') : undefined;
}
function contentHash(content) {
    return crypto.createHash('sha256').update(content).digest('hex');
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2ZuLWluaXQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjZm4taW5pdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFBQSxpQ0FBaUM7QUFDakMscUNBQXFDO0FBQ3JDLHFDQUE4QztBQUU5QyxtREFBc0Q7QUFDdEQsbUVBQWdIO0FBT2hIOztHQUVHO0FBQ0gsTUFBYSxrQkFBa0I7SUE4QjdCLFlBQW9CLFVBQW9DLEVBQUUsT0FBbUM7UUFINUUsZ0JBQVcsR0FBNkIsRUFBRSxDQUFDO1FBQzNDLGFBQVEsR0FBK0IsRUFBRSxDQUFDO1FBR3pELE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUM1QyxNQUFNLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsT0FBTyxDQUFDLENBQUM7S0FDdkM7SUFoQ0Q7O09BRUc7SUFDSSxNQUFNLENBQUMsWUFBWSxDQUFDLEdBQUcsUUFBdUI7Ozs7Ozs7Ozs7UUFDbkQsT0FBTyxrQkFBa0IsQ0FBQyxVQUFVLENBQUMsSUFBSSxVQUFVLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQztLQUNoRTtJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLFVBQVUsQ0FBQyxNQUFrQjs7Ozs7Ozs7OztRQUN6QyxPQUFPLGtCQUFrQixDQUFDLGNBQWMsQ0FBQztZQUN2QyxVQUFVLEVBQUU7Z0JBQ1YsT0FBTyxFQUFFLENBQUMsUUFBUSxDQUFDO2FBQ3BCO1lBQ0QsT0FBTyxFQUFFLEVBQUUsTUFBTSxFQUFFO1NBQ3BCLENBQUMsQ0FBQztLQUNKO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsY0FBYyxDQUFDLEtBQXFCOzs7Ozs7Ozs7O1FBQ2hELE9BQU8sSUFBSSxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsVUFBVSxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztLQUNoRTtJQVVEOztPQUVHO0lBQ0ksU0FBUyxDQUFDLFVBQWtCLEVBQUUsTUFBa0I7Ozs7Ozs7Ozs7UUFDckQsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxFQUFFO1lBQzdCLE1BQU0sSUFBSSxLQUFLLENBQUMsdURBQXVELFVBQVUsR0FBRyxDQUFDLENBQUM7U0FDdkY7UUFDRCxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxHQUFHLE1BQU0sQ0FBQztLQUNwQztJQUVEOzs7O09BSUc7SUFDSSxZQUFZLENBQUMsYUFBcUIsRUFBRSxjQUF3QixFQUFFO1FBQ25FLElBQUksSUFBSSxDQUFDLFdBQVcsQ0FBQyxhQUFhLENBQUMsRUFBRTtZQUNuQyxNQUFNLElBQUksS0FBSyxDQUFDLDBEQUEwRCxhQUFhLEdBQUcsQ0FBQyxDQUFDO1NBQzdGO1FBRUQsTUFBTSxHQUFHLEdBQUcsV0FBVyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3ZELElBQUksR0FBRyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDbEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxnREFBZ0QsYUFBYSxNQUFNLEdBQUcsRUFBRSxDQUFDLENBQUM7U0FDM0Y7UUFFRCxJQUFJLENBQUMsV0FBVyxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsR0FBRyxXQUFXLENBQUMsQ0FBQztLQUNwRDtJQUVEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLE1BQU0sQ0FBQyxnQkFBNkIsRUFBRSxhQUFnQzs7Ozs7Ozs7Ozs7UUFDM0UsSUFBSSxhQUFhLENBQUMsUUFBUSxLQUFLLG1DQUFtQixDQUFDLE9BQU8sRUFBRTtZQUMxRCxNQUFNLElBQUksS0FBSyxDQUFDLHdEQUF3RCxDQUFDLENBQUM7U0FDM0U7UUFFRCxNQUFNLHFCQUFxQixHQUFHLDJCQUEyQixDQUFDO1FBRTFELElBQUksZ0JBQWdCLENBQUMsV0FBVyxDQUFDLHFCQUFxQixDQUFDLEtBQUssU0FBUyxFQUFFO1lBQ3JFLE1BQU0sSUFBSSxLQUFLLENBQUMsa0NBQWtDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxJQUFJLGtCQUFrQixxQkFBcUIsWUFBWSxDQUFDLENBQUM7U0FDbEk7UUFFRCw4REFBOEQ7UUFDOUQsTUFBTSxVQUFVLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFDcEUsZ0JBQWdCLENBQUMsV0FBVyxDQUFDLHFCQUFxQixFQUFFLFVBQVUsQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUUzRSxnRUFBZ0U7UUFDaEUsK0VBQStFO1FBQy9FLE1BQU0sY0FBYyxHQUFHLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQzdFLE1BQU0sZ0JBQWdCLEdBQUcsRUFBRSxNQUFNLEVBQUUsY0FBYyxFQUFFLFNBQVMsRUFBRSxVQUFVLENBQUMsU0FBUyxFQUFFLENBQUM7UUFDckYsTUFBTSxXQUFXLEdBQUcsV0FBVyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFL0UsYUFBYSxDQUFDLFlBQVksQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDdEUsT0FBTyxFQUFFLENBQUMsc0NBQXNDLEVBQUUsK0JBQStCLENBQUM7WUFDbEYsU0FBUyxFQUFFLENBQUMsVUFBRyxDQUFDLFFBQVEsQ0FBQztTQUMxQixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksVUFBVSxDQUFDLFFBQVEsRUFBRTtZQUN2QixnQkFBZ0IsQ0FBQyxXQUFXLENBQUMscUNBQXFDLEVBQUUsVUFBVSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1NBQzFGO1FBRUQsd0VBQXdFO1FBQ3hFLDZEQUE2RDtRQUM3RCxJQUFJLGVBQWUsR0FBRyxZQUFZLFVBQUcsQ0FBQyxNQUFNLFlBQVksVUFBRyxDQUFDLFVBQVUsZUFBZSxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUUsQ0FBQztRQUNsSCxNQUFNLGNBQWMsR0FBRyxhQUFhLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxnQkFBZ0IsQ0FBQyxTQUFTLENBQUM7UUFDN0YsSUFBSSxxQkFBcUIsR0FBRyxZQUFZLFVBQUcsQ0FBQyxNQUFNLFlBQVksVUFBRyxDQUFDLFVBQVUsZUFBZSxjQUFjLEVBQUUsQ0FBQztRQUU1RyxtRkFBbUY7UUFDbkYsSUFBSSxhQUFhLENBQUMsVUFBVSxFQUFFO1lBQzVCLGVBQWUsR0FBRyxHQUFHLGVBQWUsaUNBQWlDLFVBQUcsQ0FBQyxNQUFNLElBQUksVUFBRyxDQUFDLFVBQVUsRUFBRSxDQUFDO1lBQ3BHLHFCQUFxQixHQUFHLEdBQUcscUJBQXFCLGlDQUFpQyxVQUFHLENBQUMsTUFBTSxJQUFJLFVBQUcsQ0FBQyxVQUFVLEVBQUUsQ0FBQztTQUNqSDtRQUNELElBQUksYUFBYSxDQUFDLFdBQVcsRUFBRTtZQUM3QixlQUFlLEdBQUcsR0FBRyxlQUFlLFdBQVcsYUFBYSxDQUFDLFlBQVksQ0FBQyxRQUFRLEVBQUUsQ0FBQztZQUNyRixxQkFBcUIsR0FBRyxHQUFHLHFCQUFxQixXQUFXLGFBQWEsQ0FBQyxZQUFZLENBQUMsUUFBUSxFQUFFLENBQUM7U0FDbEc7UUFFRCxNQUFNLFVBQVUsR0FBRyxDQUFDLGFBQWEsQ0FBQyxVQUFVLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUN2RSxNQUFNLFFBQVEsR0FBRyxhQUFhLENBQUMsUUFBUSxJQUFJLElBQUksQ0FBQztRQUVoRCxJQUFJLGFBQWEsQ0FBQyxnQkFBZ0IsSUFBSSxJQUFJLEVBQUU7WUFDMUMsK0VBQStFO1lBQy9FLGFBQWEsQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLGtCQUFrQixXQUFXLEVBQUUsQ0FBQyxDQUFDO1NBQ3JFO1FBRUQsSUFBSSxhQUFhLENBQUMsUUFBUSxLQUFLLG1DQUFtQixDQUFDLE9BQU8sRUFBRTtZQUMxRCxNQUFNLE9BQU8sR0FBRyxhQUFhLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLGVBQWUsQ0FBQztZQUNyRSxhQUFhLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FDaEMsR0FBRztnQkFDRCxtQkFBbUIsZUFBZSxPQUFPLFVBQVUsRUFBRTtnQkFDckQscUJBQXFCLE9BQU8sSUFBSSxxQkFBcUIsRUFBRTtnQkFDdkQsR0FBRyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7YUFDekQsQ0FDRixDQUFDO1NBQ0g7YUFBTTtZQUNMLE1BQU0sT0FBTyxHQUFHLGFBQWEsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDO1lBQzFELGFBQWEsQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUNoQyxHQUFHO2dCQUNELHFGQUFxRjtnQkFDckYsR0FBRztnQkFDSCxVQUFVO2dCQUNWLDhCQUE4QixlQUFlLE9BQU8sVUFBVSxFQUFFO2dCQUNoRSxnQ0FBZ0MsT0FBTyxJQUFJLHFCQUFxQixFQUFFO2dCQUNsRSxHQUFHLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLGlDQUFpQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztnQkFDeEQsR0FBRzthQUNKLENBQ0YsQ0FBQztTQUNIO0tBQ0Y7SUFFTyxJQUFJLENBQUMsS0FBZ0IsRUFBRSxPQUEwQjtRQUN2RCxNQUFNLGVBQWUsR0FBRyxTQUFTLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUVuRixNQUFNLHNCQUFzQixHQUFHLFNBQVMsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQyxDQUFDO1FBRXhGLE9BQU87WUFDTCxVQUFVLEVBQUU7Z0JBQ1YsVUFBVSxFQUFFLFNBQVMsQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLFdBQVcsQ0FBQyxFQUFFLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsS0FBSyxTQUFTLENBQUMsQ0FBQztnQkFDdkgsR0FBRyxTQUFTLENBQUMsc0JBQXNCLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDO2FBQ3BEO1lBQ0QsUUFBUSxFQUFFLE1BQU0sQ0FBQyxNQUFNLENBQUMsc0JBQXNCLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLENBQUMsTUFBTSxDQUFDLFNBQVMsRUFBRSxTQUFTLENBQUM7WUFDdkcsU0FBUyxFQUFFLDZCQUE2QixDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsc0JBQXNCLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUM7U0FDdEcsQ0FBQztLQUNIOztBQTNLSCxnREE2S0M7OztBQUVEOztHQUVHO0FBQ0gsTUFBYSxVQUFVO0lBR3JCLFlBQVksUUFBdUI7UUFGbEIsYUFBUSxHQUFHLElBQUksS0FBSyxFQUFlLENBQUM7UUFHbkQsSUFBSSxDQUFDLEdBQUcsQ0FBQyxHQUFHLFFBQVEsQ0FBQyxDQUFDO0tBQ3ZCO0lBRUQ7O09BRUc7SUFDSSxPQUFPO1FBQ1osT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sS0FBSyxDQUFDLENBQUM7S0FDbkM7SUFFRDs7T0FFRztJQUNJLEdBQUcsQ0FBQyxHQUFHLFFBQXVCOzs7Ozs7Ozs7O1FBQ25DLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEdBQUcsUUFBUSxDQUFDLENBQUM7S0FDakM7SUFFRDs7OztPQUlHO0lBQ0ksS0FBSyxDQUFDLEtBQWdCLEVBQUUsT0FBMEI7UUFDdkQsTUFBTSxXQUFXLEdBQUc7WUFDbEIsWUFBWSxFQUFFLE9BQU8sQ0FBQyxZQUFZO1lBQ2xDLFFBQVEsRUFBRSxJQUFJLENBQUMsc0JBQXNCLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQztZQUN2RCxLQUFLO1NBQ04sQ0FBQztRQUVGLE1BQU0sYUFBYSxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsbUNBQWUsQ0FBQyxPQUFPLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDN0UsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxtQ0FBZSxDQUFDLEtBQUssRUFBRSxXQUFXLENBQUMsQ0FBQztRQUMxRSxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLG1DQUFlLENBQUMsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQ3hFLE1BQU0sYUFBYSxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsbUNBQWUsQ0FBQyxNQUFNLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDNUUsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxtQ0FBZSxDQUFDLElBQUksRUFBRSxXQUFXLENBQUMsQ0FBQztRQUN4RSxNQUFNLGNBQWMsR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLG1DQUFlLENBQUMsT0FBTyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQzlFLGdCQUFnQjtRQUNoQixNQUFNLGNBQWMsR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLG1DQUFlLENBQUMsT0FBTyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBRTlFLE1BQU0sU0FBUyxHQUFHLENBQUMsYUFBYSxFQUFFLFlBQVksRUFBRSxXQUFXLEVBQUUsYUFBYSxFQUFFLFdBQVcsRUFBRSxjQUFjLEVBQUUsY0FBYyxDQUFDLENBQUM7UUFDekgsTUFBTSxjQUFjLEdBQUcsU0FBUyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsRUFBRSxjQUFjLENBQUMsQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzFGLE1BQU0sU0FBUyxHQUFHLDZCQUE2QixDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLEVBQUUsU0FBUyxDQUFDLENBQUMsQ0FBQztRQUVsRixPQUFPO1lBQ0wsTUFBTSxFQUFFO2dCQUNOLFFBQVEsRUFBRSxhQUFhLEVBQUUsTUFBTTtnQkFDL0IsTUFBTSxFQUFFLFlBQVksRUFBRSxNQUFNO2dCQUM1QixLQUFLLEVBQUUsV0FBVyxFQUFFLE1BQU07Z0JBQzFCLE9BQU8sRUFBRSxhQUFhLEVBQUUsTUFBTTtnQkFDOUIsS0FBSyxFQUFFLFdBQVcsRUFBRSxNQUFNO2dCQUMxQixRQUFRLEVBQUUsY0FBYyxFQUFFLE1BQU07Z0JBQ2hDLFFBQVEsRUFBRSxjQUFjLEVBQUUsTUFBTTthQUNqQztZQUNELGNBQWM7WUFDZCxTQUFTO1NBQ1YsQ0FBQztLQUNIO0lBRU8sV0FBVyxDQUFDLFdBQTRCLEVBQUUsYUFBNkM7UUFDN0YsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsV0FBVyxLQUFLLFdBQVcsQ0FBQyxDQUFDO1FBQ2hGLElBQUksUUFBUSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFBRSxPQUFPLFNBQVMsQ0FBQztTQUFFO1FBRWhELE1BQU0sV0FBVyxHQUFHLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLEVBQUUsS0FBSyxFQUFFLEdBQUcsYUFBYSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBRXJGLE9BQU87WUFDTCxNQUFNLEVBQUUsV0FBVyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQyxJQUFJLEVBQUU7WUFDekUsY0FBYyxFQUFFLFdBQVcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLENBQUMsTUFBTSxDQUFDLFNBQVMsRUFBRSxTQUFTLENBQUM7WUFDbkYsU0FBUyxFQUFFLDZCQUE2QixDQUFDLFdBQVcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUM7U0FDNUUsQ0FBQztLQUNIO0lBRU8sc0JBQXNCLENBQUMsTUFBMkI7UUFDeEQsUUFBUSxNQUFNLEVBQUU7WUFDZCxLQUFLLG1DQUFtQixDQUFDLEtBQUssQ0FBQyxDQUFDO2dCQUM5QixPQUFPLGdDQUFZLENBQUMsS0FBSyxDQUFDO2FBQzNCO1lBQ0QsS0FBSyxtQ0FBbUIsQ0FBQyxPQUFPLENBQUMsQ0FBQztnQkFDaEMsT0FBTyxnQ0FBWSxDQUFDLE9BQU8sQ0FBQzthQUM3QjtZQUNELE9BQU8sQ0FBQyxDQUFDO2dCQUNQLE1BQU0sSUFBSSxLQUFLLENBQUMsd0RBQXdELENBQUMsQ0FBQzthQUMzRTtTQUNGO0tBQ0Y7O0FBdEZILGdDQXVGQzs7O0FBaUJEOzs7OztHQUtHO0FBQ0gsU0FBUyxTQUFTLENBQUMsTUFBNEIsRUFBRSxHQUF5QjtJQUN4RSxJQUFJLE1BQU0sSUFBSSxJQUFJLEVBQUU7UUFBRSxPQUFPLEdBQUcsQ0FBQztLQUFFO0lBQ25DLElBQUksR0FBRyxJQUFJLElBQUksRUFBRTtRQUFFLE9BQU8sTUFBTSxDQUFDO0tBQUU7SUFFbkMsS0FBSyxNQUFNLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEVBQUU7UUFDOUMsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxFQUFFO1lBQ3hCLElBQUksTUFBTSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsRUFBRTtnQkFDOUMsTUFBTSxJQUFJLEtBQUssQ0FBQywwQkFBMEIsS0FBSyx1QkFBdUIsTUFBTSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQzthQUN2RjtZQUNELE1BQU0sQ0FBQyxHQUFHLENBQUMsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksR0FBRyxDQUFDO2dCQUMvQixHQUFHLE1BQU0sQ0FBQyxHQUFHLENBQUMsSUFBSSxFQUFFO2dCQUNwQixHQUFHLEtBQUs7YUFDVCxDQUFDLENBQUMsQ0FBQztZQUNKLFNBQVM7U0FDVjtRQUNELElBQUksT0FBTyxLQUFLLEtBQUssUUFBUSxJQUFJLEtBQUssRUFBRTtZQUN0QyxNQUFNLENBQUMsR0FBRyxDQUFDLEdBQUcsU0FBUyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsSUFBSSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7WUFDbEQsU0FBUztTQUNWO1FBQ0QsSUFBSSxLQUFLLEtBQUssU0FBUyxFQUFFO1lBQ3ZCLE1BQU0sQ0FBQyxHQUFHLENBQUMsR0FBRyxLQUFLLENBQUM7U0FDckI7S0FDRjtJQUVELE9BQU8sTUFBTSxDQUFDO0FBQ2hCLENBQUM7QUFFRDs7OztHQUlHO0FBQ0gsU0FBUyxTQUFTLENBQU8sRUFBcUIsRUFBRSxFQUEyQjtJQUN6RSxNQUFNLEdBQUcsR0FBc0IsRUFBRSxDQUFDO0lBQ2xDLEtBQUssTUFBTSxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxFQUFFO1FBQ3ZDLE1BQU0sTUFBTSxHQUFHLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNyQixJQUFJLE1BQU0sS0FBSyxTQUFTLEVBQUU7WUFDeEIsR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLE1BQU0sQ0FBQztTQUNqQjtLQUNGO0lBQ0QsT0FBTyxHQUFHLENBQUM7QUFDYixDQUFDO0FBRUQsNEZBQTRGO0FBQzVGLFNBQVMsNkJBQTZCLENBQUMsTUFBOEI7SUFDbkUsTUFBTSxTQUFTLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBZSxFQUFFLENBQUMsQ0FBQyxLQUFLLFNBQVMsQ0FBQyxDQUFDO0lBQ3JFLE9BQU8sU0FBUyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztBQUMvRCxDQUFDO0FBRUQsU0FBUyxXQUFXLENBQUMsT0FBZTtJQUNsQyxPQUFPLE1BQU0sQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQztBQUNuRSxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY3J5cHRvIGZyb20gJ2NyeXB0byc7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnLi4vLi4vYXdzLWlhbSc7XG5pbXBvcnQgeyBBd3MsIENmblJlc291cmNlIH0gZnJvbSAnLi4vLi4vY29yZSc7XG5pbXBvcnQgeyBJbml0RWxlbWVudCB9IGZyb20gJy4vY2ZuLWluaXQtZWxlbWVudHMnO1xuaW1wb3J0IHsgT3BlcmF0aW5nU3lzdGVtVHlwZSB9IGZyb20gJy4vbWFjaGluZS1pbWFnZSc7XG5pbXBvcnQgeyBJbml0QmluZE9wdGlvbnMsIEluaXRFbGVtZW50Q29uZmlnLCBJbml0RWxlbWVudFR5cGUsIEluaXRQbGF0Zm9ybSB9IGZyb20gJy4vcHJpdmF0ZS9jZm4taW5pdC1pbnRlcm5hbCc7XG5pbXBvcnQgeyBVc2VyRGF0YSB9IGZyb20gJy4vdXNlci1kYXRhJztcblxuLy8ga2VlcCB0aGlzIGltcG9ydCBzZXBhcmF0ZSBmcm9tIG90aGVyIGltcG9ydHMgdG8gcmVkdWNlIGNoYW5jZSBmb3IgbWVyZ2UgY29uZmxpY3RzIHdpdGggdjItbWFpblxuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLWR1cGxpY2F0ZS1pbXBvcnRzLCBpbXBvcnQvb3JkZXJcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJy4uLy4uL2NvcmUnO1xuXG4vKipcbiAqIEEgQ2xvdWRGb3JtYXRpb24taW5pdCBjb25maWd1cmF0aW9uXG4gKi9cbmV4cG9ydCBjbGFzcyBDbG91ZEZvcm1hdGlvbkluaXQge1xuICAvKipcbiAgICogQnVpbGQgYSBuZXcgY29uZmlnIGZyb20gYSBzZXQgb2YgSW5pdCBFbGVtZW50c1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tRWxlbWVudHMoLi4uZWxlbWVudHM6IEluaXRFbGVtZW50W10pOiBDbG91ZEZvcm1hdGlvbkluaXQge1xuICAgIHJldHVybiBDbG91ZEZvcm1hdGlvbkluaXQuZnJvbUNvbmZpZyhuZXcgSW5pdENvbmZpZyhlbGVtZW50cykpO1xuICB9XG5cbiAgLyoqXG4gICAqIFVzZSBhbiBleGlzdGluZyBJbml0Q29uZmlnIG9iamVjdCBhcyB0aGUgZGVmYXVsdCBhbmQgb25seSBjb25maWdcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUNvbmZpZyhjb25maWc6IEluaXRDb25maWcpOiBDbG91ZEZvcm1hdGlvbkluaXQge1xuICAgIHJldHVybiBDbG91ZEZvcm1hdGlvbkluaXQuZnJvbUNvbmZpZ1NldHMoe1xuICAgICAgY29uZmlnU2V0czoge1xuICAgICAgICBkZWZhdWx0OiBbJ2NvbmZpZyddLFxuICAgICAgfSxcbiAgICAgIGNvbmZpZ3M6IHsgY29uZmlnIH0sXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQnVpbGQgYSBDbG91ZEZvcm1hdGlvbkluaXQgZnJvbSBjb25maWcgc2V0c1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tQ29uZmlnU2V0cyhwcm9wczogQ29uZmlnU2V0UHJvcHMpOiBDbG91ZEZvcm1hdGlvbkluaXQge1xuICAgIHJldHVybiBuZXcgQ2xvdWRGb3JtYXRpb25Jbml0KHByb3BzLmNvbmZpZ1NldHMsIHByb3BzLmNvbmZpZ3MpO1xuICB9XG5cbiAgcHJpdmF0ZSByZWFkb25seSBfY29uZmlnU2V0czogUmVjb3JkPHN0cmluZywgc3RyaW5nW10+ID0ge307XG4gIHByaXZhdGUgcmVhZG9ubHkgX2NvbmZpZ3M6IFJlY29yZDxzdHJpbmcsIEluaXRDb25maWc+ID0ge307XG5cbiAgcHJpdmF0ZSBjb25zdHJ1Y3Rvcihjb25maWdTZXRzOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmdbXT4sIGNvbmZpZ3M6IFJlY29yZDxzdHJpbmcsIEluaXRDb25maWc+KSB7XG4gICAgT2JqZWN0LmFzc2lnbih0aGlzLl9jb25maWdTZXRzLCBjb25maWdTZXRzKTtcbiAgICBPYmplY3QuYXNzaWduKHRoaXMuX2NvbmZpZ3MsIGNvbmZpZ3MpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhIGNvbmZpZyB3aXRoIHRoZSBnaXZlbiBuYW1lIHRvIHRoaXMgQ2xvdWRGb3JtYXRpb25Jbml0IG9iamVjdFxuICAgKi9cbiAgcHVibGljIGFkZENvbmZpZyhjb25maWdOYW1lOiBzdHJpbmcsIGNvbmZpZzogSW5pdENvbmZpZykge1xuICAgIGlmICh0aGlzLl9jb25maWdzW2NvbmZpZ05hbWVdKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYENsb3VkRm9ybWF0aW9uSW5pdCBhbHJlYWR5IGNvbnRhaW5zIGEgY29uZmlnIG5hbWVkICcke2NvbmZpZ05hbWV9J2ApO1xuICAgIH1cbiAgICB0aGlzLl9jb25maWdzW2NvbmZpZ05hbWVdID0gY29uZmlnO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhIGNvbmZpZyBzZXQgd2l0aCB0aGUgZ2l2ZW4gbmFtZSB0byB0aGlzIENsb3VkRm9ybWF0aW9uSW5pdCBvYmplY3RcbiAgICpcbiAgICogVGhlIG5ldyBjb25maWdzZXQgd2lsbCByZWZlcmVuY2UgdGhlIGdpdmVuIGNvbmZpZ3MgaW4gdGhlIGdpdmVuIG9yZGVyLlxuICAgKi9cbiAgcHVibGljIGFkZENvbmZpZ1NldChjb25maWdTZXROYW1lOiBzdHJpbmcsIGNvbmZpZ05hbWVzOiBzdHJpbmdbXSA9IFtdKSB7XG4gICAgaWYgKHRoaXMuX2NvbmZpZ1NldHNbY29uZmlnU2V0TmFtZV0pIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgQ2xvdWRGb3JtYXRpb25Jbml0IGFscmVhZHkgY29udGFpbnMgYSBjb25maWdTZXQgbmFtZWQgJyR7Y29uZmlnU2V0TmFtZX0nYCk7XG4gICAgfVxuXG4gICAgY29uc3QgdW5rID0gY29uZmlnTmFtZXMuZmlsdGVyKGMgPT4gIXRoaXMuX2NvbmZpZ3NbY10pO1xuICAgIGlmICh1bmsubGVuZ3RoID4gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbmtub3duIGNvbmZpZ3MgcmVmZXJlbmNlZCBpbiBkZWZpbml0aW9uIG9mICcke2NvbmZpZ1NldE5hbWV9JzogJHt1bmt9YCk7XG4gICAgfVxuXG4gICAgdGhpcy5fY29uZmlnU2V0c1tjb25maWdTZXROYW1lXSA9IFsuLi5jb25maWdOYW1lc107XG4gIH1cblxuICAvKipcbiAgICogQXR0YWNoIHRoZSBDbG91ZEZvcm1hdGlvbiBJbml0IGNvbmZpZyB0byB0aGUgZ2l2ZW4gcmVzb3VyY2VcbiAgICpcbiAgICogQXMgYW4gYXBwIGJ1aWxkZXIsIHVzZSBgaW5zdGFuY2UuYXBwbHlDbG91ZEZvcm1hdGlvbkluaXQoKWAgb3JcbiAgICogYGF1dG9TY2FsaW5nR3JvdXAuYXBwbHlDbG91ZEZvcm1hdGlvbkluaXQoKWAgdG8gdHJpZ2dlciB0aGlzIG1ldGhvZC5cbiAgICpcbiAgICogVGhpcyBtZXRob2QgZG9lcyB0aGUgZm9sbG93aW5nOlxuICAgKlxuICAgKiAtIFJlbmRlcnMgdGhlIGBBV1M6OkNsb3VkRm9ybWF0aW9uOjpJbml0YCBvYmplY3QgdG8gdGhlIGdpdmVuIHJlc291cmNlJ3NcbiAgICogICBtZXRhZGF0YSwgcG90ZW50aWFsbHkgYWRkaW5nIGEgYEFXUzo6Q2xvdWRGb3JtYXRpb246OkF1dGhlbnRpY2F0aW9uYCBvYmplY3RcbiAgICogICBuZXh0IHRvIGl0IGlmIHJlcXVpcmVkLlxuICAgKiAtIFVwZGF0ZXMgdGhlIGluc3RhbmNlIHJvbGUgcG9saWN5IHRvIGJlIGFibGUgdG8gY2FsbCB0aGUgQVBJcyByZXF1aXJlZCBmb3JcbiAgICogICBgY2ZuLWluaXRgIGFuZCBgY2ZuLXNpZ25hbGAgdG8gd29yaywgYW5kIHBvdGVudGlhbGx5IGFkZCBwZXJtaXNzaW9ucyB0byBkb3dubG9hZFxuICAgKiAgIHJlZmVyZW5jZWQgYXNzZXQgYW5kIGJ1Y2tldCByZXNvdXJjZXMuXG4gICAqIC0gVXBkYXRlcyB0aGUgZ2l2ZW4gVXNlckRhdGEgd2l0aCBjb21tYW5kcyB0byBleGVjdXRlIHRoZSBgY2ZuLWluaXRgIHNjcmlwdC5cbiAgICovXG4gIHB1YmxpYyBhdHRhY2goYXR0YWNoZWRSZXNvdXJjZTogQ2ZuUmVzb3VyY2UsIGF0dGFjaE9wdGlvbnM6IEF0dGFjaEluaXRPcHRpb25zKSB7XG4gICAgaWYgKGF0dGFjaE9wdGlvbnMucGxhdGZvcm0gPT09IE9wZXJhdGluZ1N5c3RlbVR5cGUuVU5LTk9XTikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3QgYXR0YWNoIENsb3VkRm9ybWF0aW9uSW5pdCB0byBhbiB1bmtub3duIE9TIHR5cGUnKTtcbiAgICB9XG5cbiAgICBjb25zdCBDRk5fSU5JVF9NRVRBREFUQV9LRVkgPSAnQVdTOjpDbG91ZEZvcm1hdGlvbjo6SW5pdCc7XG5cbiAgICBpZiAoYXR0YWNoZWRSZXNvdXJjZS5nZXRNZXRhZGF0YShDRk5fSU5JVF9NRVRBREFUQV9LRVkpICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgQ2Fubm90IGJpbmQgQ2ZuSW5pdDogcmVzb3VyY2UgJyR7YXR0YWNoZWRSZXNvdXJjZS5ub2RlLnBhdGh9JyBhbHJlYWR5IGhhcyAnJHtDRk5fSU5JVF9NRVRBREFUQV9LRVl9JyBhdHRhY2hlZGApO1xuICAgIH1cblxuICAgIC8vIE5vdGU6IFRoaXMgd2lsbCBub3QgcmVmbGVjdCBtdXRhdGlvbnMgbWFkZSBhZnRlciBhdHRhY2hpbmcuXG4gICAgY29uc3QgYmluZFJlc3VsdCA9IHRoaXMuYmluZChhdHRhY2hlZFJlc291cmNlLnN0YWNrLCBhdHRhY2hPcHRpb25zKTtcbiAgICBhdHRhY2hlZFJlc291cmNlLmFkZE1ldGFkYXRhKENGTl9JTklUX01FVEFEQVRBX0tFWSwgYmluZFJlc3VsdC5jb25maWdEYXRhKTtcblxuICAgIC8vIE5lZWQgdG8gcmVzb2x2ZSB0aGUgdmFyaW91cyB0b2tlbnMgZnJvbSBhc3NldHMgaW4gdGhlIGNvbmZpZyxcbiAgICAvLyBhcyB3ZWxsIGFzIGluY2x1ZGUgYW55IGFzc2V0IGhhc2hlcyBwcm92aWRlZCBzbyB0aGUgZmluZ2VycHJpbnQgaXMgYWNjdXJhdGUuXG4gICAgY29uc3QgcmVzb2x2ZWRDb25maWcgPSBhdHRhY2hlZFJlc291cmNlLnN0YWNrLnJlc29sdmUoYmluZFJlc3VsdC5jb25maWdEYXRhKTtcbiAgICBjb25zdCBmaW5nZXJwcmludElucHV0ID0geyBjb25maWc6IHJlc29sdmVkQ29uZmlnLCBhc3NldEhhc2g6IGJpbmRSZXN1bHQuYXNzZXRIYXNoIH07XG4gICAgY29uc3QgZmluZ2VycHJpbnQgPSBjb250ZW50SGFzaChKU09OLnN0cmluZ2lmeShmaW5nZXJwcmludElucHV0KSkuc2xpY2UoMCwgMTYpO1xuXG4gICAgYXR0YWNoT3B0aW9ucy5pbnN0YW5jZVJvbGUuYWRkVG9QcmluY2lwYWxQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgYWN0aW9uczogWydjbG91ZGZvcm1hdGlvbjpEZXNjcmliZVN0YWNrUmVzb3VyY2UnLCAnY2xvdWRmb3JtYXRpb246U2lnbmFsUmVzb3VyY2UnXSxcbiAgICAgIHJlc291cmNlczogW0F3cy5TVEFDS19JRF0sXG4gICAgfSkpO1xuXG4gICAgaWYgKGJpbmRSZXN1bHQuYXV0aERhdGEpIHtcbiAgICAgIGF0dGFjaGVkUmVzb3VyY2UuYWRkTWV0YWRhdGEoJ0FXUzo6Q2xvdWRGb3JtYXRpb246OkF1dGhlbnRpY2F0aW9uJywgYmluZFJlc3VsdC5hdXRoRGF0YSk7XG4gICAgfVxuXG4gICAgLy8gVG8gaWRlbnRpZnkgdGhlIHJlc291cmNlcyB0aGF0IGhhdmUgdGhlIG1ldGFkYXRhIGFuZCB3aGVyZSB0aGUgc2lnbmFsXG4gICAgLy8gbmVlZHMgdG8gYmUgc2VudCwgd2UgbmVlZCB7IHJlZ2lvbiwgc3RhY2tOYW1lLCBsb2dpY2FsSWQgfVxuICAgIGxldCByZXNvdXJjZUxvY2F0b3IgPSBgLS1yZWdpb24gJHtBd3MuUkVHSU9OfSAtLXN0YWNrICR7QXdzLlNUQUNLX05BTUV9IC0tcmVzb3VyY2UgJHthdHRhY2hlZFJlc291cmNlLmxvZ2ljYWxJZH1gO1xuICAgIGNvbnN0IHNpZ25hbFJlc291cmNlID0gYXR0YWNoT3B0aW9ucy5zaWduYWxSZXNvdXJjZT8ubG9naWNhbElkID8/IGF0dGFjaGVkUmVzb3VyY2UubG9naWNhbElkO1xuICAgIGxldCBub3RpZnlSZXNvdXJjZUxvY2F0b3IgPSBgLS1yZWdpb24gJHtBd3MuUkVHSU9OfSAtLXN0YWNrICR7QXdzLlNUQUNLX05BTUV9IC0tcmVzb3VyY2UgJHtzaWduYWxSZXNvdXJjZX1gO1xuXG4gICAgLy8gSWYgc3BlY2lmaWVkIGluIGF0dGFjaE9wdGlvbnMsIGluY2x1ZGUgYXJndW1lbnRzIGluIGNmbi1pbml0L2Nmbi1zaWduYWwgY29tbWFuZHNcbiAgICBpZiAoYXR0YWNoT3B0aW9ucy5pbmNsdWRlVXJsKSB7XG4gICAgICByZXNvdXJjZUxvY2F0b3IgPSBgJHtyZXNvdXJjZUxvY2F0b3J9IC0tdXJsIGh0dHBzOi8vY2xvdWRmb3JtYXRpb24uJHtBd3MuUkVHSU9OfS4ke0F3cy5VUkxfU1VGRklYfWA7XG4gICAgICBub3RpZnlSZXNvdXJjZUxvY2F0b3IgPSBgJHtub3RpZnlSZXNvdXJjZUxvY2F0b3J9IC0tdXJsIGh0dHBzOi8vY2xvdWRmb3JtYXRpb24uJHtBd3MuUkVHSU9OfS4ke0F3cy5VUkxfU1VGRklYfWA7XG4gICAgfVxuICAgIGlmIChhdHRhY2hPcHRpb25zLmluY2x1ZGVSb2xlKSB7XG4gICAgICByZXNvdXJjZUxvY2F0b3IgPSBgJHtyZXNvdXJjZUxvY2F0b3J9IC0tcm9sZSAke2F0dGFjaE9wdGlvbnMuaW5zdGFuY2VSb2xlLnJvbGVOYW1lfWA7XG4gICAgICBub3RpZnlSZXNvdXJjZUxvY2F0b3IgPSBgJHtub3RpZnlSZXNvdXJjZUxvY2F0b3J9IC0tcm9sZSAke2F0dGFjaE9wdGlvbnMuaW5zdGFuY2VSb2xlLnJvbGVOYW1lfWA7XG4gICAgfVxuXG4gICAgY29uc3QgY29uZmlnU2V0cyA9IChhdHRhY2hPcHRpb25zLmNvbmZpZ1NldHMgPz8gWydkZWZhdWx0J10pLmpvaW4oJywnKTtcbiAgICBjb25zdCBwcmludExvZyA9IGF0dGFjaE9wdGlvbnMucHJpbnRMb2cgPz8gdHJ1ZTtcblxuICAgIGlmIChhdHRhY2hPcHRpb25zLmVtYmVkRmluZ2VycHJpbnQgPz8gdHJ1ZSkge1xuICAgICAgLy8gSXQganVzdCBzbyBoYXBwZW5zIHRoYXQgdGhlIGNvbW1lbnQgY2hhciBpcyAnIycgZm9yIGJvdGggYmFzaCBhbmQgUG93ZXJTaGVsbFxuICAgICAgYXR0YWNoT3B0aW9ucy51c2VyRGF0YS5hZGRDb21tYW5kcyhgIyBmaW5nZXJwcmludDogJHtmaW5nZXJwcmludH1gKTtcbiAgICB9XG5cbiAgICBpZiAoYXR0YWNoT3B0aW9ucy5wbGF0Zm9ybSA9PT0gT3BlcmF0aW5nU3lzdGVtVHlwZS5XSU5ET1dTKSB7XG4gICAgICBjb25zdCBlcnJDb2RlID0gYXR0YWNoT3B0aW9ucy5pZ25vcmVGYWlsdXJlcyA/ICcwJyA6ICckTEFTVEVYSVRDT0RFJztcbiAgICAgIGF0dGFjaE9wdGlvbnMudXNlckRhdGEuYWRkQ29tbWFuZHMoXG4gICAgICAgIC4uLltcbiAgICAgICAgICBgY2ZuLWluaXQuZXhlIC12ICR7cmVzb3VyY2VMb2NhdG9yfSAtYyAke2NvbmZpZ1NldHN9YCxcbiAgICAgICAgICBgY2ZuLXNpZ25hbC5leGUgLWUgJHtlcnJDb2RlfSAke25vdGlmeVJlc291cmNlTG9jYXRvcn1gLFxuICAgICAgICAgIC4uLihwcmludExvZyA/IFsndHlwZSBDOlxcXFxjZm5cXFxcbG9nXFxcXGNmbi1pbml0LmxvZyddIDogW10pLFxuICAgICAgICBdLFxuICAgICAgKTtcbiAgICB9IGVsc2Uge1xuICAgICAgY29uc3QgZXJyQ29kZSA9IGF0dGFjaE9wdGlvbnMuaWdub3JlRmFpbHVyZXMgPyAnMCcgOiAnJD8nO1xuICAgICAgYXR0YWNoT3B0aW9ucy51c2VyRGF0YS5hZGRDb21tYW5kcyhcbiAgICAgICAgLi4uW1xuICAgICAgICAgIC8vIFJ1biBhIHN1YnNoZWxsIHdpdGhvdXQgJ2VycmV4aXQnLCBzbyB3ZSBjYW4gc2lnbmFsIHVzaW5nIHRoZSBleGl0IGNvZGUgb2YgY2ZuLWluaXRcbiAgICAgICAgICAnKCcsXG4gICAgICAgICAgJyAgc2V0ICtlJyxcbiAgICAgICAgICBgICAvb3B0L2F3cy9iaW4vY2ZuLWluaXQgLXYgJHtyZXNvdXJjZUxvY2F0b3J9IC1jICR7Y29uZmlnU2V0c31gLFxuICAgICAgICAgIGAgIC9vcHQvYXdzL2Jpbi9jZm4tc2lnbmFsIC1lICR7ZXJyQ29kZX0gJHtub3RpZnlSZXNvdXJjZUxvY2F0b3J9YCxcbiAgICAgICAgICAuLi4ocHJpbnRMb2cgPyBbJyAgY2F0IC92YXIvbG9nL2Nmbi1pbml0LmxvZyA+JjInXSA6IFtdKSxcbiAgICAgICAgICAnKScsXG4gICAgICAgIF0sXG4gICAgICApO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgYmluZChzY29wZTogQ29uc3RydWN0LCBvcHRpb25zOiBBdHRhY2hJbml0T3B0aW9ucyk6IHsgY29uZmlnRGF0YTogYW55LCBhdXRoRGF0YTogYW55LCBhc3NldEhhc2g/OiBhbnkgfSB7XG4gICAgY29uc3Qgbm9uRW1wdHlDb25maWdzID0gbWFwVmFsdWVzKHRoaXMuX2NvbmZpZ3MsIGMgPT4gYy5pc0VtcHR5KCkgPyB1bmRlZmluZWQgOiBjKTtcblxuICAgIGNvbnN0IGNvbmZpZ05hbWVUb0JpbmRSZXN1bHQgPSBtYXBWYWx1ZXMobm9uRW1wdHlDb25maWdzLCBjID0+IGMuX2JpbmQoc2NvcGUsIG9wdGlvbnMpKTtcblxuICAgIHJldHVybiB7XG4gICAgICBjb25maWdEYXRhOiB7XG4gICAgICAgIGNvbmZpZ1NldHM6IG1hcFZhbHVlcyh0aGlzLl9jb25maWdTZXRzLCBjb25maWdOYW1lcyA9PiBjb25maWdOYW1lcy5maWx0ZXIobmFtZSA9PiBub25FbXB0eUNvbmZpZ3NbbmFtZV0gIT09IHVuZGVmaW5lZCkpLFxuICAgICAgICAuLi5tYXBWYWx1ZXMoY29uZmlnTmFtZVRvQmluZFJlc3VsdCwgYyA9PiBjLmNvbmZpZyksXG4gICAgICB9LFxuICAgICAgYXV0aERhdGE6IE9iamVjdC52YWx1ZXMoY29uZmlnTmFtZVRvQmluZFJlc3VsdCkubWFwKGMgPT4gYy5hdXRoZW50aWNhdGlvbikucmVkdWNlKGRlZXBNZXJnZSwgdW5kZWZpbmVkKSxcbiAgICAgIGFzc2V0SGFzaDogY29tYmluZUFzc2V0SGFzaGVzT3JVbmRlZmluZWQoT2JqZWN0LnZhbHVlcyhjb25maWdOYW1lVG9CaW5kUmVzdWx0KS5tYXAoYyA9PiBjLmFzc2V0SGFzaCkpLFxuICAgIH07XG4gIH1cblxufVxuXG4vKipcbiAqIEEgY29sbGVjdGlvbiBvZiBjb25maWd1cmF0aW9uIGVsZW1lbnRzXG4gKi9cbmV4cG9ydCBjbGFzcyBJbml0Q29uZmlnIHtcbiAgcHJpdmF0ZSByZWFkb25seSBlbGVtZW50cyA9IG5ldyBBcnJheTxJbml0RWxlbWVudD4oKTtcblxuICBjb25zdHJ1Y3RvcihlbGVtZW50czogSW5pdEVsZW1lbnRbXSkge1xuICAgIHRoaXMuYWRkKC4uLmVsZW1lbnRzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoaXMgY29uZmlnc2V0IGhhcyBlbGVtZW50cyBvciBub3RcbiAgICovXG4gIHB1YmxpYyBpc0VtcHR5KCkge1xuICAgIHJldHVybiB0aGlzLmVsZW1lbnRzLmxlbmd0aCA9PT0gMDtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgb25lIG9yIG1vcmUgZWxlbWVudHMgdG8gdGhlIGNvbmZpZ1xuICAgKi9cbiAgcHVibGljIGFkZCguLi5lbGVtZW50czogSW5pdEVsZW1lbnRbXSkge1xuICAgIHRoaXMuZWxlbWVudHMucHVzaCguLi5lbGVtZW50cyk7XG4gIH1cblxuICAvKipcbiAgICogQ2FsbGVkIHdoZW4gdGhlIGNvbmZpZyBpcyBhcHBsaWVkIHRvIGFuIGluc3RhbmNlLlxuICAgKiBDcmVhdGVzIHRoZSBDbG91ZEZvcm1hdGlvbiByZXByZXNlbnRhdGlvbiBvZiB0aGUgSW5pdCBjb25maWcgYW5kIGhhbmRsZXMgYW55IHBlcm1pc3Npb25zIGFuZCBhc3NldHMuXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHVibGljIF9iaW5kKHNjb3BlOiBDb25zdHJ1Y3QsIG9wdGlvbnM6IEF0dGFjaEluaXRPcHRpb25zKTogSW5pdEVsZW1lbnRDb25maWcge1xuICAgIGNvbnN0IGJpbmRPcHRpb25zID0ge1xuICAgICAgaW5zdGFuY2VSb2xlOiBvcHRpb25zLmluc3RhbmNlUm9sZSxcbiAgICAgIHBsYXRmb3JtOiB0aGlzLmluaXRQbGF0Zm9ybUZyb21PU1R5cGUob3B0aW9ucy5wbGF0Zm9ybSksXG4gICAgICBzY29wZSxcbiAgICB9O1xuXG4gICAgY29uc3QgcGFja2FnZUNvbmZpZyA9IHRoaXMuYmluZEZvclR5cGUoSW5pdEVsZW1lbnRUeXBlLlBBQ0tBR0UsIGJpbmRPcHRpb25zKTtcbiAgICBjb25zdCBncm91cHNDb25maWcgPSB0aGlzLmJpbmRGb3JUeXBlKEluaXRFbGVtZW50VHlwZS5HUk9VUCwgYmluZE9wdGlvbnMpO1xuICAgIGNvbnN0IHVzZXJzQ29uZmlnID0gdGhpcy5iaW5kRm9yVHlwZShJbml0RWxlbWVudFR5cGUuVVNFUiwgYmluZE9wdGlvbnMpO1xuICAgIGNvbnN0IHNvdXJjZXNDb25maWcgPSB0aGlzLmJpbmRGb3JUeXBlKEluaXRFbGVtZW50VHlwZS5TT1VSQ0UsIGJpbmRPcHRpb25zKTtcbiAgICBjb25zdCBmaWxlc0NvbmZpZyA9IHRoaXMuYmluZEZvclR5cGUoSW5pdEVsZW1lbnRUeXBlLkZJTEUsIGJpbmRPcHRpb25zKTtcbiAgICBjb25zdCBjb21tYW5kc0NvbmZpZyA9IHRoaXMuYmluZEZvclR5cGUoSW5pdEVsZW1lbnRUeXBlLkNPTU1BTkQsIGJpbmRPcHRpb25zKTtcbiAgICAvLyBNdXN0IGJlIGxhc3QhXG4gICAgY29uc3Qgc2VydmljZXNDb25maWcgPSB0aGlzLmJpbmRGb3JUeXBlKEluaXRFbGVtZW50VHlwZS5TRVJWSUNFLCBiaW5kT3B0aW9ucyk7XG5cbiAgICBjb25zdCBhbGxDb25maWcgPSBbcGFja2FnZUNvbmZpZywgZ3JvdXBzQ29uZmlnLCB1c2Vyc0NvbmZpZywgc291cmNlc0NvbmZpZywgZmlsZXNDb25maWcsIGNvbW1hbmRzQ29uZmlnLCBzZXJ2aWNlc0NvbmZpZ107XG4gICAgY29uc3QgYXV0aGVudGljYXRpb24gPSBhbGxDb25maWcubWFwKGMgPT4gYz8uYXV0aGVudGljYXRpb24pLnJlZHVjZShkZWVwTWVyZ2UsIHVuZGVmaW5lZCk7XG4gICAgY29uc3QgYXNzZXRIYXNoID0gY29tYmluZUFzc2V0SGFzaGVzT3JVbmRlZmluZWQoYWxsQ29uZmlnLm1hcChjID0+IGM/LmFzc2V0SGFzaCkpO1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIGNvbmZpZzoge1xuICAgICAgICBwYWNrYWdlczogcGFja2FnZUNvbmZpZz8uY29uZmlnLFxuICAgICAgICBncm91cHM6IGdyb3Vwc0NvbmZpZz8uY29uZmlnLFxuICAgICAgICB1c2VyczogdXNlcnNDb25maWc/LmNvbmZpZyxcbiAgICAgICAgc291cmNlczogc291cmNlc0NvbmZpZz8uY29uZmlnLFxuICAgICAgICBmaWxlczogZmlsZXNDb25maWc/LmNvbmZpZyxcbiAgICAgICAgY29tbWFuZHM6IGNvbW1hbmRzQ29uZmlnPy5jb25maWcsXG4gICAgICAgIHNlcnZpY2VzOiBzZXJ2aWNlc0NvbmZpZz8uY29uZmlnLFxuICAgICAgfSxcbiAgICAgIGF1dGhlbnRpY2F0aW9uLFxuICAgICAgYXNzZXRIYXNoLFxuICAgIH07XG4gIH1cblxuICBwcml2YXRlIGJpbmRGb3JUeXBlKGVsZW1lbnRUeXBlOiBJbml0RWxlbWVudFR5cGUsIHJlbmRlck9wdGlvbnM6IE9taXQ8SW5pdEJpbmRPcHRpb25zLCAnaW5kZXgnPik6IEluaXRFbGVtZW50Q29uZmlnIHwgdW5kZWZpbmVkIHtcbiAgICBjb25zdCBlbGVtZW50cyA9IHRoaXMuZWxlbWVudHMuZmlsdGVyKGVsZW0gPT4gZWxlbS5lbGVtZW50VHlwZSA9PT0gZWxlbWVudFR5cGUpO1xuICAgIGlmIChlbGVtZW50cy5sZW5ndGggPT09IDApIHsgcmV0dXJuIHVuZGVmaW5lZDsgfVxuXG4gICAgY29uc3QgYmluZFJlc3VsdHMgPSBlbGVtZW50cy5tYXAoKGUsIGluZGV4KSA9PiBlLl9iaW5kKHsgaW5kZXgsIC4uLnJlbmRlck9wdGlvbnMgfSkpO1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIGNvbmZpZzogYmluZFJlc3VsdHMubWFwKHIgPT4gci5jb25maWcpLnJlZHVjZShkZWVwTWVyZ2UsIHVuZGVmaW5lZCkgPz8ge30sXG4gICAgICBhdXRoZW50aWNhdGlvbjogYmluZFJlc3VsdHMubWFwKHIgPT4gci5hdXRoZW50aWNhdGlvbikucmVkdWNlKGRlZXBNZXJnZSwgdW5kZWZpbmVkKSxcbiAgICAgIGFzc2V0SGFzaDogY29tYmluZUFzc2V0SGFzaGVzT3JVbmRlZmluZWQoYmluZFJlc3VsdHMubWFwKHIgPT4gci5hc3NldEhhc2gpKSxcbiAgICB9O1xuICB9XG5cbiAgcHJpdmF0ZSBpbml0UGxhdGZvcm1Gcm9tT1NUeXBlKG9zVHlwZTogT3BlcmF0aW5nU3lzdGVtVHlwZSk6IEluaXRQbGF0Zm9ybSB7XG4gICAgc3dpdGNoIChvc1R5cGUpIHtcbiAgICAgIGNhc2UgT3BlcmF0aW5nU3lzdGVtVHlwZS5MSU5VWDoge1xuICAgICAgICByZXR1cm4gSW5pdFBsYXRmb3JtLkxJTlVYO1xuICAgICAgfVxuICAgICAgY2FzZSBPcGVyYXRpbmdTeXN0ZW1UeXBlLldJTkRPV1M6IHtcbiAgICAgICAgcmV0dXJuIEluaXRQbGF0Zm9ybS5XSU5ET1dTO1xuICAgICAgfVxuICAgICAgZGVmYXVsdDoge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0Nhbm5vdCBhdHRhY2ggQ2xvdWRGb3JtYXRpb25Jbml0IHRvIGFuIHVua25vd24gT1MgdHlwZScpO1xuICAgICAgfVxuICAgIH1cbiAgfVxufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIENsb3VkRm9ybWF0aW9uSW5pdC53aXRoQ29uZmlnU2V0c1xuICovXG5leHBvcnQgaW50ZXJmYWNlIENvbmZpZ1NldFByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBkZWZpbml0aW9ucyBvZiBlYWNoIGNvbmZpZyBzZXRcbiAgICovXG4gIHJlYWRvbmx5IGNvbmZpZ1NldHM6IFJlY29yZDxzdHJpbmcsIHN0cmluZ1tdPjtcblxuICAvKipcbiAgICogVGhlIHNldHMgb2YgY29uZmlncyB0byBwaWNrIGZyb21cbiAgICovXG4gIHJlYWRvbmx5IGNvbmZpZ3M6IFJlY29yZDxzdHJpbmcsIEluaXRDb25maWc+O1xufVxuXG4vKipcbiAqIERlZXAtbWVyZ2Ugb2JqZWN0cyBhbmQgYXJyYXlzXG4gKlxuICogVHJlYXQgYXJyYXlzIGFzIHNldHMsIHJlbW92aW5nIGR1cGxpY2F0ZXMuIFRoaXMgaXMgYWNjZXB0YWJsZSBmb3IgcmVuZGVyaW5nXG4gKiBjZm4taW5pdHMsIG5vdCBhcHBsaWNhYmxlIGVsc2V3aGVyZS5cbiAqL1xuZnVuY3Rpb24gZGVlcE1lcmdlKHRhcmdldD86IFJlY29yZDxzdHJpbmcsIGFueT4sIHNyYz86IFJlY29yZDxzdHJpbmcsIGFueT4pIHtcbiAgaWYgKHRhcmdldCA9PSBudWxsKSB7IHJldHVybiBzcmM7IH1cbiAgaWYgKHNyYyA9PSBudWxsKSB7IHJldHVybiB0YXJnZXQ7IH1cblxuICBmb3IgKGNvbnN0IFtrZXksIHZhbHVlXSBvZiBPYmplY3QuZW50cmllcyhzcmMpKSB7XG4gICAgaWYgKEFycmF5LmlzQXJyYXkodmFsdWUpKSB7XG4gICAgICBpZiAodGFyZ2V0W2tleV0gJiYgIUFycmF5LmlzQXJyYXkodGFyZ2V0W2tleV0pKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgVHJ5aW5nIHRvIG1lcmdlIGFycmF5IFske3ZhbHVlfV0gaW50byBhIG5vbi1hcnJheSAnJHt0YXJnZXRba2V5XX0nYCk7XG4gICAgICB9XG4gICAgICB0YXJnZXRba2V5XSA9IEFycmF5LmZyb20obmV3IFNldChbXG4gICAgICAgIC4uLnRhcmdldFtrZXldID8/IFtdLFxuICAgICAgICAuLi52YWx1ZSxcbiAgICAgIF0pKTtcbiAgICAgIGNvbnRpbnVlO1xuICAgIH1cbiAgICBpZiAodHlwZW9mIHZhbHVlID09PSAnb2JqZWN0JyAmJiB2YWx1ZSkge1xuICAgICAgdGFyZ2V0W2tleV0gPSBkZWVwTWVyZ2UodGFyZ2V0W2tleV0gPz8ge30sIHZhbHVlKTtcbiAgICAgIGNvbnRpbnVlO1xuICAgIH1cbiAgICBpZiAodmFsdWUgIT09IHVuZGVmaW5lZCkge1xuICAgICAgdGFyZ2V0W2tleV0gPSB2YWx1ZTtcbiAgICB9XG4gIH1cblxuICByZXR1cm4gdGFyZ2V0O1xufVxuXG4vKipcbiAqIE1hcCBhIGZ1bmN0aW9uIG92ZXIgdmFsdWVzIG9mIGFuIG9iamVjdFxuICpcbiAqIElmIHRoZSBtYXBwaW5nIGZ1bmN0aW9uIHJldHVybnMgdW5kZWZpbmVkLCByZW1vdmUgdGhlIGtleVxuICovXG5mdW5jdGlvbiBtYXBWYWx1ZXM8QSwgQj4oeHM6IFJlY29yZDxzdHJpbmcsIEE+LCBmbjogKHg6IEEpID0+IEIgfCB1bmRlZmluZWQpOiBSZWNvcmQ8c3RyaW5nLCBCPiB7XG4gIGNvbnN0IHJldDogUmVjb3JkPHN0cmluZywgQj4gPSB7fTtcbiAgZm9yIChjb25zdCBbaywgdl0gb2YgT2JqZWN0LmVudHJpZXMoeHMpKSB7XG4gICAgY29uc3QgbWFwcGVkID0gZm4odik7XG4gICAgaWYgKG1hcHBlZCAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICByZXRba10gPSBtYXBwZWQ7XG4gICAgfVxuICB9XG4gIHJldHVybiByZXQ7XG59XG5cbi8vIENvbWJpbmVzIGFsbCBpbnB1dCBhc3NldCBoYXNoZXMgaW50byBvbmUsIG9yIGlmIG5vIGhhc2hlcyBhcmUgcHJlc2VudCwgcmV0dXJucyB1bmRlZmluZWQuXG5mdW5jdGlvbiBjb21iaW5lQXNzZXRIYXNoZXNPclVuZGVmaW5lZChoYXNoZXM6IChzdHJpbmcgfCB1bmRlZmluZWQpW10pOiBzdHJpbmcgfCB1bmRlZmluZWQge1xuICBjb25zdCBoYXNoQXJyYXkgPSBoYXNoZXMuZmlsdGVyKCh4KTogeCBpcyBzdHJpbmcgPT4geCAhPT0gdW5kZWZpbmVkKTtcbiAgcmV0dXJuIGhhc2hBcnJheS5sZW5ndGggPiAwID8gaGFzaEFycmF5LmpvaW4oJycpIDogdW5kZWZpbmVkO1xufVxuXG5mdW5jdGlvbiBjb250ZW50SGFzaChjb250ZW50OiBzdHJpbmcpIHtcbiAgcmV0dXJuIGNyeXB0by5jcmVhdGVIYXNoKCdzaGEyNTYnKS51cGRhdGUoY29udGVudCkuZGlnZXN0KCdoZXgnKTtcbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBhdHRhY2hpbmcgYSBDbG91ZEZvcm1hdGlvbkluaXQgdG8gYSByZXNvdXJjZVxuICovXG5leHBvcnQgaW50ZXJmYWNlIEF0dGFjaEluaXRPcHRpb25zIHtcbiAgLyoqXG4gICAqIEluc3RhbmNlIHJvbGUgb2YgdGhlIGNvbnN1bWluZyBpbnN0YW5jZSBvciBmbGVldFxuICAgKi9cbiAgcmVhZG9ubHkgaW5zdGFuY2VSb2xlOiBpYW0uSVJvbGU7XG5cbiAgLyoqXG4gICAqIEluY2x1ZGUgLS11cmwgYXJndW1lbnQgd2hlbiBydW5uaW5nIGNmbi1pbml0IGFuZCBjZm4tc2lnbmFsIGNvbW1hbmRzXG4gICAqXG4gICAqIFRoaXMgd2lsbCBiZSB0aGUgY2xvdWRmb3JtYXRpb24gZW5kcG9pbnQgaW4gdGhlIGRlcGxveWVkIHJlZ2lvblxuICAgKiBlLmcuIGh0dHBzOi8vY2xvdWRmb3JtYXRpb24udXMtZWFzdC0xLmFtYXpvbmF3cy5jb21cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGluY2x1ZGVVcmw/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBJbmNsdWRlIC0tcm9sZSBhcmd1bWVudCB3aGVuIHJ1bm5pbmcgY2ZuLWluaXQgYW5kIGNmbi1zaWduYWwgY29tbWFuZHNcbiAgICpcbiAgICogVGhpcyB3aWxsIGJlIHRoZSBJQU0gaW5zdGFuY2UgcHJvZmlsZSBhdHRhY2hlZCB0byB0aGUgRUMyIGluc3RhbmNlXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBpbmNsdWRlUm9sZT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIE9TIFBsYXRmb3JtIHRoZSBpbml0IGNvbmZpZyB3aWxsIGJlIHVzZWQgZm9yXG4gICAqL1xuICByZWFkb25seSBwbGF0Zm9ybTogT3BlcmF0aW5nU3lzdGVtVHlwZTtcblxuICAvKipcbiAgICogVXNlckRhdGEgdG8gYWRkIGNvbW1hbmRzIHRvXG4gICAqL1xuICByZWFkb25seSB1c2VyRGF0YTogVXNlckRhdGE7XG5cbiAgLyoqXG4gICAqIENvbmZpZ1NldCB0byBhY3RpdmF0ZVxuICAgKlxuICAgKiBAZGVmYXVsdCBbJ2RlZmF1bHQnXVxuICAgKi9cbiAgcmVhZG9ubHkgY29uZmlnU2V0cz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRvIGVtYmVkIGEgaGFzaCBpbnRvIHRoZSB1c2VyRGF0YVxuICAgKlxuICAgKiBJZiBgdHJ1ZWAgKHRoZSBkZWZhdWx0KSwgYSBoYXNoIG9mIHRoZSBjb25maWcgd2lsbCBiZSBlbWJlZGRlZCBpbnRvIHRoZVxuICAgKiBVc2VyRGF0YSwgc28gdGhhdCBpZiB0aGUgY29uZmlnIGNoYW5nZXMsIHRoZSBVc2VyRGF0YSBjaGFuZ2VzIGFuZFxuICAgKiB0aGUgaW5zdGFuY2Ugd2lsbCBiZSByZXBsYWNlZC5cbiAgICpcbiAgICogSWYgYGZhbHNlYCwgbm8gc3VjaCBoYXNoIHdpbGwgYmUgZW1iZWRkZWQsIGFuZCBpZiB0aGUgQ2xvdWRGb3JtYXRpb24gSW5pdFxuICAgKiBjb25maWcgY2hhbmdlcyBub3RoaW5nIHdpbGwgaGFwcGVuIHRvIHRoZSBydW5uaW5nIGluc3RhbmNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBlbWJlZEZpbmdlcnByaW50PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogUHJpbnQgdGhlIHJlc3VsdHMgb2YgcnVubmluZyBjZm4taW5pdCB0byB0aGUgSW5zdGFuY2UgU3lzdGVtIExvZ1xuICAgKlxuICAgKiBCeSBkZWZhdWx0LCB0aGUgb3V0cHV0IG9mIHJ1bm5pbmcgY2ZuLWluaXQgaXMgd3JpdHRlbiB0byBhIGxvZyBmaWxlXG4gICAqIG9uIHRoZSBpbnN0YW5jZS4gU2V0IHRoaXMgdG8gYHRydWVgIHRvIHByaW50IGl0IHRvIHRoZSBTeXN0ZW0gTG9nXG4gICAqICh2aXNpYmxlIGZyb20gdGhlIEVDMiBDb25zb2xlKSwgYGZhbHNlYCB0byBub3QgcHJpbnQgaXQuXG4gICAqXG4gICAqIChCZSBhd2FyZSB0aGF0IHRoZSBzeXN0ZW0gbG9nIGlzIHJlZnJlc2hlZCBhdCBjZXJ0YWluIHBvaW50cyBpblxuICAgKiB0aW1lIG9mIHRoZSBpbnN0YW5jZSBsaWZlIGN5Y2xlLCBhbmQgc3VjY2Vzc2Z1bCBleGVjdXRpb24gbWF5XG4gICAqIG5vdCBhbHdheXMgc2hvdyB1cCkuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IHByaW50TG9nPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogRG9uJ3QgZmFpbCB0aGUgaW5zdGFuY2UgY3JlYXRpb24gd2hlbiBjZm4taW5pdCBmYWlsc1xuICAgKlxuICAgKiBZb3UgY2FuIHVzZSB0aGlzIHRvIHByZXZlbnQgQ2xvdWRGb3JtYXRpb24gZnJvbSByb2xsaW5nIGJhY2sgd2hlblxuICAgKiBpbnN0YW5jZXMgZmFpbCB0byBzdGFydCB1cCwgdG8gaGVscCBpbiBkZWJ1Z2dpbmcuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBpZ25vcmVGYWlsdXJlcz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFdoZW4gcHJvdmlkZWQsIHNpZ25hbHMgdGhpcyByZXNvdXJjZSBpbnN0ZWFkIG9mIHRoZSBhdHRhY2hlZCByZXNvdXJjZVxuICAgKlxuICAgKiBZb3UgY2FuIHVzZSB0aGlzIHRvIHN1cHBvcnQgc2lnbmFsaW5nIExhdW5jaFRlbXBsYXRlIHdoaWxlIGF0dGFjaGluZyBBdXRvU2NhbGluZ0dyb3VwXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gaWYgdGhpcyBwcm9wZXJ0eSBpcyB1bmRlZmluZWQgY2ZuLXNpZ25hbCBzaWduYWxzIHRoZSBhdHRhY2hlZCByZXNvdXJjZVxuICAgKi9cbiAgcmVhZG9ubHkgc2lnbmFsUmVzb3VyY2U/OiBDZm5SZXNvdXJjZTtcbn1cbiJdfQ==