"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseInstance = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2");
const cdk = require("../../core");
const core_1 = require("../../core");
const docdb_generated_1 = require("./docdb.generated");
const endpoint_1 = require("./endpoint");
/**
 * A new or imported database instance.
 */
class DatabaseInstanceBase extends cdk.Resource {
    /**
     * Import an existing database instance.
     */
    static fromDatabaseInstanceAttributes(scope, id, attrs) {
        class Import extends DatabaseInstanceBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.instanceIdentifier = attrs.instanceIdentifier;
                this.dbInstanceEndpointAddress = attrs.instanceEndpointAddress;
                this.dbInstanceEndpointPort = attrs.port.toString();
                this.instanceEndpoint = new endpoint_1.Endpoint(attrs.instanceEndpointAddress, attrs.port);
            }
        }
        return new Import(scope, id);
    }
    /**
     * The instance arn.
     */
    get instanceArn() {
        return cdk.Stack.of(this).formatArn({
            service: 'rds',
            resource: 'db',
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
            resourceName: this.instanceIdentifier,
        });
    }
}
/**
 * A database instance
 *
 * @resource AWS::DocDB::DBInstance
 */
class DatabaseInstance extends DatabaseInstanceBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings.monocdk_aws_docdb_DatabaseInstanceProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, DatabaseInstance);
            }
            throw error;
        }
        const instance = new docdb_generated_1.CfnDBInstance(this, 'Resource', {
            dbClusterIdentifier: props.cluster.clusterIdentifier,
            dbInstanceClass: `db.${props.instanceType}`,
            autoMinorVersionUpgrade: props.autoMinorVersionUpgrade ?? true,
            availabilityZone: props.availabilityZone,
            dbInstanceIdentifier: props.dbInstanceName,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
        });
        this.cluster = props.cluster;
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpoint;
        this.dbInstanceEndpointPort = instance.attrPort;
        // create a number token that represents the port of the instance
        const portAttribute = cdk.Token.asNumber(instance.attrPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpoint, portAttribute);
        instance.applyRemovalPolicy(props.removalPolicy, {
            applyToUpdateReplacePolicy: true,
        });
    }
}
exports.DatabaseInstance = DatabaseInstance;
_a = JSII_RTTI_SYMBOL_1;
DatabaseInstance[_a] = { fqn: "monocdk.aws_docdb.DatabaseInstance", version: "1.185.0" };
//# sourceMappingURL=data:application/json;base64,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