"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Endpoint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
/**
 * Connection endpoint of a database cluster or instance
 *
 * Consists of a combination of hostname and port.
 */
class Endpoint {
    /**
     * Constructs an Endpoint instance.
     *
     * @param address - The hostname or address of the endpoint
     * @param port - The port number of the endpoint
     */
    constructor(address, port) {
        if (!core_1.Token.isUnresolved(port) && !Endpoint.isValidPort(port)) {
            throw new Error(`Port must be an integer between [${Endpoint.MIN_PORT}, ${Endpoint.MAX_PORT}] but got: ${port}`);
        }
        this.hostname = address;
        this.port = port;
        const portDesc = core_1.Token.isUnresolved(port) ? core_1.Token.asString(port) : port;
        this.socketAddress = `${address}:${portDesc}`;
    }
    /**
     * Determines if a port is valid
     *
     * @param port: The port number
     * @returns boolean whether the port is valid
     */
    static isValidPort(port) {
        return Number.isInteger(port) && port >= Endpoint.MIN_PORT && port <= Endpoint.MAX_PORT;
    }
    /**
     * Returns the port number as a string representation that can be used for embedding within other strings.
     *
     * This is intended to deal with CDK's token system. Numeric CDK tokens are not expanded when their string
     * representation is embedded in a string. This function returns the port either as an unresolved string token or
     * as a resolved string representation of the port value.
     *
     * @returns {string} An (un)resolved string representation of the endpoint's port number
     */
    portAsString() {
        if (core_1.Token.isUnresolved(this.port)) {
            return core_1.Token.asString(this.port);
        }
        else {
            return this.port.toString();
        }
    }
}
exports.Endpoint = Endpoint;
_a = JSII_RTTI_SYMBOL_1;
Endpoint[_a] = { fqn: "monocdk.aws_docdb.Endpoint", version: "1.185.0" };
/**
 * The minimum port value
 */
Endpoint.MIN_PORT = 1;
/**
 * The maximum port value
 */
Endpoint.MAX_PORT = 65535;
//# sourceMappingURL=data:application/json;base64,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