"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPool = exports.AccountRecovery = exports.Mfa = exports.VerificationEmailStyle = exports.UserPoolOperation = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("../../aws-iam");
const core_1 = require("../../core");
const punycode_1 = require("punycode/");
const cognito_generated_1 = require("./cognito.generated");
const attr_names_1 = require("./private/attr-names");
const user_pool_client_1 = require("./user-pool-client");
const user_pool_domain_1 = require("./user-pool-domain");
const user_pool_resource_server_1 = require("./user-pool-resource-server");
/**
 * User pool operations to which lambda triggers can be attached.
 */
class UserPoolOperation {
    constructor(operationName) {
        this.operationName = operationName;
    }
    /** A custom user pool operation */
    static of(name) {
        const lowerCamelCase = name.charAt(0).toLowerCase() + name.slice(1);
        return new UserPoolOperation(lowerCamelCase);
    }
}
exports.UserPoolOperation = UserPoolOperation;
_a = JSII_RTTI_SYMBOL_1;
UserPoolOperation[_a] = { fqn: "monocdk.aws_cognito.UserPoolOperation", version: "1.185.0" };
/**
 * Creates a challenge in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-create-auth-challenge.html
 */
UserPoolOperation.CREATE_AUTH_CHALLENGE = new UserPoolOperation('createAuthChallenge');
/**
 * Advanced customization and localization of messages
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-message.html
 */
UserPoolOperation.CUSTOM_MESSAGE = new UserPoolOperation('customMessage');
/**
 * Determines the next challenge in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-define-auth-challenge.html
 */
UserPoolOperation.DEFINE_AUTH_CHALLENGE = new UserPoolOperation('defineAuthChallenge');
/**
 * Event logging for custom analytics
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-authentication.html
 */
UserPoolOperation.POST_AUTHENTICATION = new UserPoolOperation('postAuthentication');
/**
 * Custom welcome messages or event logging for custom analytics
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-confirmation.html
 */
UserPoolOperation.POST_CONFIRMATION = new UserPoolOperation('postConfirmation');
/**
 * Custom validation to accept or deny the sign-in request
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-authentication.html
 */
UserPoolOperation.PRE_AUTHENTICATION = new UserPoolOperation('preAuthentication');
/**
 * Custom validation to accept or deny the sign-up request
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html
 */
UserPoolOperation.PRE_SIGN_UP = new UserPoolOperation('preSignUp');
/**
 * Add or remove attributes in Id tokens
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-token-generation.html
 */
UserPoolOperation.PRE_TOKEN_GENERATION = new UserPoolOperation('preTokenGeneration');
/**
 * Migrate a user from an existing user directory to user pools
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-migrate-user.html
 */
UserPoolOperation.USER_MIGRATION = new UserPoolOperation('userMigration');
/**
 * Determines if a response is correct in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-verify-auth-challenge-response.html
 */
UserPoolOperation.VERIFY_AUTH_CHALLENGE_RESPONSE = new UserPoolOperation('verifyAuthChallengeResponse');
/**
 * Amazon Cognito invokes this trigger to send email notifications to users.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-email-sender.html
 */
UserPoolOperation.CUSTOM_EMAIL_SENDER = new UserPoolOperation('customEmailSender');
/**
 * Amazon Cognito invokes this trigger to send email notifications to users.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-sms-sender.html
 */
UserPoolOperation.CUSTOM_SMS_SENDER = new UserPoolOperation('customSmsSender');
/**
 * The email verification style
 */
var VerificationEmailStyle;
(function (VerificationEmailStyle) {
    /** Verify email via code */
    VerificationEmailStyle["CODE"] = "CONFIRM_WITH_CODE";
    /** Verify email via link */
    VerificationEmailStyle["LINK"] = "CONFIRM_WITH_LINK";
})(VerificationEmailStyle = exports.VerificationEmailStyle || (exports.VerificationEmailStyle = {}));
/**
 * The different ways in which a user pool's MFA enforcement can be configured.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa.html
 */
var Mfa;
(function (Mfa) {
    /** Users are not required to use MFA for sign in, and cannot configure one. */
    Mfa["OFF"] = "OFF";
    /** Users are not required to use MFA for sign in, but can configure one if they so choose to. */
    Mfa["OPTIONAL"] = "OPTIONAL";
    /** Users are required to configure an MFA, and have to use it to sign in. */
    Mfa["REQUIRED"] = "ON";
})(Mfa = exports.Mfa || (exports.Mfa = {}));
/**
 * How will a user be able to recover their account?
 *
 * When a user forgets their password, they can have a code sent to their verified email or verified phone to recover their account.
 * You can choose the preferred way to send codes below.
 * We recommend not allowing phone to be used for both password resets and multi-factor authentication (MFA).
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/how-to-recover-a-user-account.html
 */
var AccountRecovery;
(function (AccountRecovery) {
    /**
     * Email if available, otherwise phone, but don’t allow a user to reset their password via phone if they are also using it for MFA
     */
    AccountRecovery[AccountRecovery["EMAIL_AND_PHONE_WITHOUT_MFA"] = 0] = "EMAIL_AND_PHONE_WITHOUT_MFA";
    /**
     * Phone if available, otherwise email, but don’t allow a user to reset their password via phone if they are also using it for MFA
     */
    AccountRecovery[AccountRecovery["PHONE_WITHOUT_MFA_AND_EMAIL"] = 1] = "PHONE_WITHOUT_MFA_AND_EMAIL";
    /**
     * Email only
     */
    AccountRecovery[AccountRecovery["EMAIL_ONLY"] = 2] = "EMAIL_ONLY";
    /**
     * Phone only, but don’t allow a user to reset their password via phone if they are also using it for MFA
     */
    AccountRecovery[AccountRecovery["PHONE_ONLY_WITHOUT_MFA"] = 3] = "PHONE_ONLY_WITHOUT_MFA";
    /**
     * (Not Recommended) Phone if available, otherwise email, and do allow a user to reset their password via phone if they are also using it for MFA.
     */
    AccountRecovery[AccountRecovery["PHONE_AND_EMAIL"] = 4] = "PHONE_AND_EMAIL";
    /**
     * None – users will have to contact an administrator to reset their passwords
     */
    AccountRecovery[AccountRecovery["NONE"] = 5] = "NONE";
})(AccountRecovery = exports.AccountRecovery || (exports.AccountRecovery = {}));
class UserPoolBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        this.identityProviders = [];
    }
    addClient(id, options) {
        return new user_pool_client_1.UserPoolClient(this, id, {
            userPool: this,
            ...options,
        });
    }
    addDomain(id, options) {
        return new user_pool_domain_1.UserPoolDomain(this, id, {
            userPool: this,
            ...options,
        });
    }
    addResourceServer(id, options) {
        return new user_pool_resource_server_1.UserPoolResourceServer(this, id, {
            userPool: this,
            ...options,
        });
    }
    registerIdentityProvider(provider) {
        this.identityProviders.push(provider);
    }
    /**
     * Adds an IAM policy statement associated with this user pool to an
     * IAM principal's policy.
     */
    grant(grantee, ...actions) {
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.userPoolArn],
            scope: this,
        });
    }
}
/**
 * Define a Cognito User Pool
 */
class UserPool extends UserPoolBase {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.triggers = {};
        try {
            jsiiDeprecationWarnings.monocdk_aws_cognito_UserPoolProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, UserPool);
            }
            throw error;
        }
        const signIn = this.signInConfiguration(props);
        if (props.customSenderKmsKey) {
            const kmsKey = props.customSenderKmsKey;
            this.triggers.kmsKeyId = kmsKey.keyArn;
        }
        if (props.lambdaTriggers) {
            for (const t of Object.keys(props.lambdaTriggers)) {
                let trigger;
                switch (t) {
                    case 'customSmsSender':
                    case 'customEmailSender':
                        if (!this.triggers.kmsKeyId) {
                            throw new Error('you must specify a KMS key if you are using customSmsSender or customEmailSender.');
                        }
                        trigger = props.lambdaTriggers[t];
                        const version = 'V1_0';
                        if (trigger !== undefined) {
                            this.addLambdaPermission(trigger, t);
                            this.triggers[t] = {
                                lambdaArn: trigger.functionArn,
                                lambdaVersion: version,
                            };
                        }
                        break;
                    default:
                        trigger = props.lambdaTriggers[t];
                        if (trigger !== undefined) {
                            this.addLambdaPermission(trigger, t);
                            this.triggers[t] = trigger.functionArn;
                        }
                        break;
                }
            }
        }
        const verificationMessageTemplate = this.verificationMessageConfiguration(props);
        let emailVerificationMessage;
        let emailVerificationSubject;
        if (verificationMessageTemplate.defaultEmailOption === VerificationEmailStyle.CODE) {
            emailVerificationMessage = verificationMessageTemplate.emailMessage;
            emailVerificationSubject = verificationMessageTemplate.emailSubject;
        }
        const smsVerificationMessage = verificationMessageTemplate.smsMessage;
        const inviteMessageTemplate = {
            emailMessage: props.userInvitation?.emailBody,
            emailSubject: props.userInvitation?.emailSubject,
            smsMessage: props.userInvitation?.smsMessage,
        };
        const selfSignUpEnabled = props.selfSignUpEnabled ?? false;
        const adminCreateUserConfig = {
            allowAdminCreateUserOnly: !selfSignUpEnabled,
            inviteMessageTemplate: props.userInvitation !== undefined ? inviteMessageTemplate : undefined,
        };
        const passwordPolicy = this.configurePasswordPolicy(props);
        if (props.email && props.emailSettings) {
            throw new Error('you must either provide "email" or "emailSettings", but not both');
        }
        const emailConfiguration = props.email ? props.email._bind(this) : undefinedIfNoKeys({
            from: encodePuny(props.emailSettings?.from),
            replyToEmailAddress: encodePuny(props.emailSettings?.replyTo),
        });
        const userPool = new cognito_generated_1.CfnUserPool(this, 'Resource', {
            userPoolName: props.userPoolName,
            usernameAttributes: signIn.usernameAttrs,
            aliasAttributes: signIn.aliasAttrs,
            autoVerifiedAttributes: signIn.autoVerifyAttrs,
            lambdaConfig: core_1.Lazy.any({ produce: () => undefinedIfNoKeys(this.triggers) }),
            smsAuthenticationMessage: this.mfaMessage(props),
            smsConfiguration: this.smsConfiguration(props),
            adminCreateUserConfig,
            emailVerificationMessage,
            emailVerificationSubject,
            smsVerificationMessage,
            verificationMessageTemplate,
            schema: this.schemaConfiguration(props),
            mfaConfiguration: props.mfa,
            enabledMfas: this.mfaConfiguration(props),
            policies: passwordPolicy !== undefined ? { passwordPolicy } : undefined,
            emailConfiguration,
            usernameConfiguration: undefinedIfNoKeys({
                caseSensitive: props.signInCaseSensitive,
            }),
            accountRecoverySetting: this.accountRecovery(props),
            deviceConfiguration: props.deviceTracking,
        });
        userPool.applyRemovalPolicy(props.removalPolicy);
        this.userPoolId = userPool.ref;
        this.userPoolArn = userPool.attrArn;
        this.userPoolProviderName = userPool.attrProviderName;
        this.userPoolProviderUrl = userPool.attrProviderUrl;
    }
    /**
     * Import an existing user pool based on its id.
     */
    static fromUserPoolId(scope, id, userPoolId) {
        let userPoolArn = core_1.Stack.of(scope).formatArn({
            service: 'cognito-idp',
            resource: 'userpool',
            resourceName: userPoolId,
        });
        return UserPool.fromUserPoolArn(scope, id, userPoolArn);
    }
    /**
     * Import an existing user pool based on its ARN.
     */
    static fromUserPoolArn(scope, id, userPoolArn) {
        const arnParts = core_1.Stack.of(scope).splitArn(userPoolArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        if (!arnParts.resourceName) {
            throw new Error('invalid user pool ARN');
        }
        const userPoolId = arnParts.resourceName;
        class ImportedUserPool extends UserPoolBase {
            constructor() {
                super(scope, id, {
                    account: arnParts.account,
                    region: arnParts.region,
                });
                this.userPoolArn = userPoolArn;
                this.userPoolId = userPoolId;
            }
        }
        return new ImportedUserPool();
    }
    /**
     * Add a lambda trigger to a user pool operation
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html
     */
    addTrigger(operation, fn) {
        try {
            jsiiDeprecationWarnings.monocdk_aws_cognito_UserPoolOperation(operation);
            jsiiDeprecationWarnings.monocdk_aws_lambda_IFunction(fn);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addTrigger);
            }
            throw error;
        }
        if (operation.operationName in this.triggers) {
            throw new Error(`A trigger for the operation ${operation.operationName} already exists.`);
        }
        this.addLambdaPermission(fn, operation.operationName);
        switch (operation.operationName) {
            case 'customEmailSender':
            case 'customSmsSender':
                if (!this.triggers.kmsKeyId) {
                    throw new Error('you must specify a KMS key if you are using customSmsSender or customEmailSender.');
                }
                this.triggers[operation.operationName] = {
                    lambdaArn: fn.functionArn,
                    lambdaVersion: 'V1_0',
                };
                break;
            default:
                this.triggers[operation.operationName] = fn.functionArn;
        }
    }
    addLambdaPermission(fn, name) {
        const capitalize = name.charAt(0).toUpperCase() + name.slice(1);
        fn.addPermission(`${capitalize}Cognito`, {
            principal: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com'),
            sourceArn: core_1.Lazy.string({ produce: () => this.userPoolArn }),
        });
    }
    mfaMessage(props) {
        const CODE_TEMPLATE = '{####}';
        const MAX_LENGTH = 140;
        const message = props.mfaMessage;
        if (message && !core_1.Token.isUnresolved(message)) {
            if (!message.includes(CODE_TEMPLATE)) {
                throw new Error(`MFA message must contain the template string '${CODE_TEMPLATE}'`);
            }
            if (message.length > MAX_LENGTH) {
                throw new Error(`MFA message must be between ${CODE_TEMPLATE.length} and ${MAX_LENGTH} characters`);
            }
        }
        return message;
    }
    verificationMessageConfiguration(props) {
        const CODE_TEMPLATE = '{####}';
        const VERIFY_EMAIL_TEMPLATE = '{##Verify Email##}';
        const emailStyle = props.userVerification?.emailStyle ?? VerificationEmailStyle.CODE;
        const emailSubject = props.userVerification?.emailSubject ?? 'Verify your new account';
        const smsMessage = props.userVerification?.smsMessage ?? `The verification code to your new account is ${CODE_TEMPLATE}`;
        if (emailStyle === VerificationEmailStyle.CODE) {
            const emailMessage = props.userVerification?.emailBody ?? `The verification code to your new account is ${CODE_TEMPLATE}`;
            if (!core_1.Token.isUnresolved(emailMessage) && emailMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${CODE_TEMPLATE}'`);
            }
            if (!core_1.Token.isUnresolved(smsMessage) && smsMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`SMS message must contain the template string '${CODE_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.CODE,
                emailMessage,
                emailSubject,
                smsMessage,
            };
        }
        else {
            const emailMessage = props.userVerification?.emailBody ??
                `Verify your account by clicking on ${VERIFY_EMAIL_TEMPLATE}`;
            if (!core_1.Token.isUnresolved(emailMessage) && emailMessage.indexOf(VERIFY_EMAIL_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${VERIFY_EMAIL_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.LINK,
                emailMessageByLink: emailMessage,
                emailSubjectByLink: emailSubject,
                smsMessage,
            };
        }
    }
    signInConfiguration(props) {
        let aliasAttrs;
        let usernameAttrs;
        let autoVerifyAttrs;
        const signIn = props.signInAliases ?? { username: true };
        if (signIn.preferredUsername && !signIn.username) {
            throw new Error('username signIn must be enabled if preferredUsername is enabled');
        }
        if (signIn.username) {
            aliasAttrs = [];
            if (signIn.email) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
            if (signIn.preferredUsername) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.preferredUsername);
            }
            if (aliasAttrs.length === 0) {
                aliasAttrs = undefined;
            }
        }
        else {
            usernameAttrs = [];
            if (signIn.email) {
                usernameAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                usernameAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        if (props.autoVerify) {
            autoVerifyAttrs = [];
            if (props.autoVerify.email) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (props.autoVerify.phone) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        else if (signIn.email || signIn.phone) {
            autoVerifyAttrs = [];
            if (signIn.email) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        return { usernameAttrs, aliasAttrs, autoVerifyAttrs };
    }
    smsConfiguration(props) {
        if (props.enableSmsRole === false && props.smsRole) {
            throw new Error('enableSmsRole cannot be disabled when smsRole is specified');
        }
        if (props.smsRole) {
            return {
                snsCallerArn: props.smsRole.roleArn,
                externalId: props.smsRoleExternalId,
                snsRegion: props.snsRegion,
            };
        }
        if (props.enableSmsRole === false) {
            return undefined;
        }
        const mfaConfiguration = this.mfaConfiguration(props);
        const phoneVerification = props.signInAliases?.phone === true || props.autoVerify?.phone === true;
        const roleRequired = mfaConfiguration?.includes('SMS_MFA') || phoneVerification;
        if (!roleRequired && props.enableSmsRole === undefined) {
            return undefined;
        }
        const smsRoleExternalId = core_1.Names.uniqueId(this).slice(0, 1223); // sts:ExternalId max length of 1224
        const smsRole = props.smsRole ?? new aws_iam_1.Role(this, 'smsRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com', {
                conditions: {
                    StringEquals: { 'sts:ExternalId': smsRoleExternalId },
                },
            }),
            inlinePolicies: {
                /*
                  * The UserPool is very particular that it must contain an 'sns:Publish' action as an inline policy.
                  * Ideally, a conditional that restricts this action to 'sms' protocol needs to be attached, but the UserPool deployment fails validation.
                  * Seems like a case of being excessively strict.
                  */
                'sns-publish': new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: ['sns:Publish'],
                            resources: ['*'],
                        }),
                    ],
                }),
            },
        });
        return {
            externalId: smsRoleExternalId,
            snsCallerArn: smsRole.roleArn,
            snsRegion: props.snsRegion,
        };
    }
    mfaConfiguration(props) {
        if (props.mfa === undefined || props.mfa === Mfa.OFF) {
            // since default is OFF, treat undefined and OFF the same way
            return undefined;
        }
        else if (props.mfaSecondFactor === undefined &&
            (props.mfa === Mfa.OPTIONAL || props.mfa === Mfa.REQUIRED)) {
            return ['SMS_MFA'];
        }
        else {
            const enabledMfas = [];
            if (props.mfaSecondFactor.sms) {
                enabledMfas.push('SMS_MFA');
            }
            if (props.mfaSecondFactor.otp) {
                enabledMfas.push('SOFTWARE_TOKEN_MFA');
            }
            return enabledMfas;
        }
    }
    configurePasswordPolicy(props) {
        const tempPasswordValidity = props.passwordPolicy?.tempPasswordValidity;
        if (tempPasswordValidity !== undefined && tempPasswordValidity.toDays() > core_1.Duration.days(365).toDays()) {
            throw new Error(`tempPasswordValidity cannot be greater than 365 days (received: ${tempPasswordValidity.toDays()})`);
        }
        const minLength = props.passwordPolicy ? props.passwordPolicy.minLength ?? 8 : undefined;
        if (minLength !== undefined && (minLength < 6 || minLength > 99)) {
            throw new Error(`minLength for password must be between 6 and 99 (received: ${minLength})`);
        }
        return undefinedIfNoKeys({
            temporaryPasswordValidityDays: tempPasswordValidity?.toDays({ integral: true }),
            minimumLength: minLength,
            requireLowercase: props.passwordPolicy?.requireLowercase,
            requireUppercase: props.passwordPolicy?.requireUppercase,
            requireNumbers: props.passwordPolicy?.requireDigits,
            requireSymbols: props.passwordPolicy?.requireSymbols,
        });
    }
    schemaConfiguration(props) {
        const schema = [];
        if (props.standardAttributes) {
            const stdAttributes = Object.entries(props.standardAttributes)
                .filter(([, attr]) => !!attr)
                .map(([attrName, attr]) => ({
                name: attr_names_1.StandardAttributeNames[attrName],
                mutable: attr.mutable ?? true,
                required: attr.required ?? false,
            }));
            schema.push(...stdAttributes);
        }
        if (props.customAttributes) {
            const customAttrs = Object.keys(props.customAttributes).map((attrName) => {
                const attrConfig = props.customAttributes[attrName].bind();
                const numberConstraints = {
                    minValue: attrConfig.numberConstraints?.min?.toString(),
                    maxValue: attrConfig.numberConstraints?.max?.toString(),
                };
                const stringConstraints = {
                    minLength: attrConfig.stringConstraints?.minLen?.toString(),
                    maxLength: attrConfig.stringConstraints?.maxLen?.toString(),
                };
                return {
                    name: attrName,
                    attributeDataType: attrConfig.dataType,
                    numberAttributeConstraints: attrConfig.numberConstraints
                        ? numberConstraints
                        : undefined,
                    stringAttributeConstraints: attrConfig.stringConstraints
                        ? stringConstraints
                        : undefined,
                    mutable: attrConfig.mutable,
                };
            });
            schema.push(...customAttrs);
        }
        if (schema.length === 0) {
            return undefined;
        }
        return schema;
    }
    accountRecovery(props) {
        const accountRecovery = props.accountRecovery ?? AccountRecovery.PHONE_WITHOUT_MFA_AND_EMAIL;
        switch (accountRecovery) {
            case AccountRecovery.EMAIL_AND_PHONE_WITHOUT_MFA:
                return {
                    recoveryMechanisms: [
                        { name: 'verified_email', priority: 1 },
                        { name: 'verified_phone_number', priority: 2 },
                    ],
                };
            case AccountRecovery.PHONE_WITHOUT_MFA_AND_EMAIL:
                return {
                    recoveryMechanisms: [
                        { name: 'verified_phone_number', priority: 1 },
                        { name: 'verified_email', priority: 2 },
                    ],
                };
            case AccountRecovery.EMAIL_ONLY:
                return {
                    recoveryMechanisms: [{ name: 'verified_email', priority: 1 }],
                };
            case AccountRecovery.PHONE_ONLY_WITHOUT_MFA:
                return {
                    recoveryMechanisms: [{ name: 'verified_phone_number', priority: 1 }],
                };
            case AccountRecovery.NONE:
                return {
                    recoveryMechanisms: [{ name: 'admin_only', priority: 1 }],
                };
            case AccountRecovery.PHONE_AND_EMAIL:
                return undefined;
            default:
                throw new Error(`Unsupported AccountRecovery type - ${accountRecovery}`);
        }
    }
}
exports.UserPool = UserPool;
_b = JSII_RTTI_SYMBOL_1;
UserPool[_b] = { fqn: "monocdk.aws_cognito.UserPool", version: "1.185.0" };
function undefinedIfNoKeys(struct) {
    const allUndefined = Object.values(struct).every(val => val === undefined);
    return allUndefined ? undefined : struct;
}
function encodePuny(input) {
    return input !== undefined ? punycode_1.toASCII(input) : input;
}
//# sourceMappingURL=data:application/json;base64,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